/*
 * Wohnraumuhr 2, Sensor-Option fr 1-Wire-Sensoren und DHT22/AM2302, Scott-Falk Hhn, "main.c", v1.00, 02.12.2017
 * AtmelStudio 7.x Editor-Einstellung: Tools / Options / Text Editor / GCC / Tabs / Smart, Tab size: 2, Indent size: 2, Insert spaces
 *
 * Controller: ATtiny84A
 * Taktfrequenz 8,0MHz
 * Betriebsspannung: 5,0V
 *
 * Verwendung der I/O-Ports:
 * PA0 - Eingang Jumper 1, Verkrzung der RS-232-Periodendauer, mit Pull-up
 * PA1 - Eingang Jumper 2, Verlngerung der RS-232-Periodendauer, mit Pull-up
 * PA2 - frei
 * PA3 - frei
 * PA4 - Eingang SPI-SCK
 * PA5 - Ausgang SPI-MISO
 * PA6 - Eingang SPI-MOSI
 * PA7 - frei
 * PB0 - Ein-/Ausgang 1-Wire-Bus 1, Low-aktiv
 * PB1 - Ein-/Ausgang 1-Wire-Bus 2, Low-aktiv
 * PB2 - Ausgang TXD zum Hauptcontroller
 * PB3 - Eingang SPI-RES
 *
 * Einstellung der Fuse-Bits:
 * SELFPRGEN = [ ]
 * RSTDISBL =  [ ]
 * DWEN =      [ ]
 * SPIEN =     [X]
 * WDTON =     [ ]
 * EESAVE =    [ ]
 * BODLEVEL =  2V7
 * CKDIV8 =    [ ]
 * CKOUT =     [ ]
 * SUT_CKSEL = INTRCOSC_8MHZ_6CK_14CK_64MS_DEFAULT
 *
 * EXTENDED = 0xFF
 * HIGH =     0xDD
 * LOW =      0xE2
 */ 

// Definitionen und Bibliotheken

#define F_CPU 8000000UL           // Taktfrequenz 8,0MHz
#include <avr/io.h>
#include <avr/pgmspace.h>
#include <util/delay.h>

// Konstanten

#define RSBITP 104                // RS-232-Bit-Periodendauer 104s bei 9600Bit/s
#define BUS1_DDR DDRB             // Port-Parameter fr 1-Wire-Bus-Anschluss 1
#define BUS1_PRT PORTB
#define BUS1_PIN PINB
#define BUS1_SIG PINB0
#define BUS2_DDR DDRB             // Port-Parameter fr 1-Wire-Bus-Anschluss 2
#define BUS2_PRT PORTB
#define BUS2_PIN PINB
#define BUS2_SIG PINB1
#define RSTX_DDR DDRB             // Port-Parameter fr TXD-Anschluss
#define RSTX_PRT PORTB
#define RSTX_SIG PINB2

// Variablen

static uint8_t rsbdiff;           // RS-232-Bit-Periodendauer Differenz (0-2), zur Korrektur von starken Abweichungen des Taktoszillators
                                  //   0 = keine Differenz
                                  //   1 = Periodendauer um 3s verkrzen (RSBITP - 3)
                                  //   2 = Periodendauer um 3s verlngern (RSBITP + 3)
static uint8_t senmode;           // Sensormodus (0-4):
                                  //   0 = kein interner Sensor angeschlossen
                                  //   1 = DHT22/AM2302 (Anzeige von Temperatur und Luftfeuchtigkeit)
                                  //   2 = DS2438 (Anzeige von Temperatur und Luftfeuchtigkeit)
                                  //   3 = DS18S20/DS18B20/DS1822 (Anzeige der Temperatur)
                                  //   4 = 2x DS18S20/DS18B20/DS1822 (Anzeige der Temperaturen)
static uint8_t rom_id[2][9];      // 1-Wire-ROM-ID-Speicher fr 2 Sensoren, Bytes 0-7 enthalten den 64-Bit-ROM-Code, Byte 8 enthlt die Busnummer (0-1)
static uint8_t owbuff[9];         // 1-Wire-Puffer fr empfangene 64-Bit-ROM-ID und fr den Empfang des Scratchpad, wird auch als Speicher fr die Daten des DHT22 verwendet
static uint8_t ascbuff[4];        // ASCII-Puffer fr Umrechnung Integer -> ASCII
static int16_t ds2438b1[8];       // DS2438-Puffer fr die letzten 8 Temperaturwerte
static int16_t ds2438b2[8];       // DS2438-Puffer fr die letzten 8 Sensorbetriebsspannungswerte
static int16_t ds2438b3[8];       // DS2438-Puffer fr die letzten 8 Sensorausgangsspannungswerte

#include "tables.h"               // CRC8-Tabelle fr 1-Wire
#include "functions.h"            // verschiedene Funktionen

int main(void)
{
  int8_t i;                                                           // universelle Zhlervariable
  uint8_t x;                                                          // universelle 8-Bit-Variable
  uint8_t e;                                                          // Fehlerzhler
  uint8_t s;                                                          // Sensorzhler
  int16_t t;                                                          // Zwischenspeicher fr Temperaturwert
  int16_t h;                                                          // Zwischenspeicher fr Luftfeuchtigkeitswert
  int16_t b;                                                          // Zwischenspeicher fr Sensorbetriebsspannung (DS2438)
  int16_t a;                                                          // Zwischenspeicher fr Sensorausgangsspannung (DS2438)
  int32_t w;                                                          // Zwischenspeicher fr Luftfeuchtigkeitsberechnung (DS2438)

  /*
  OSCCAL = 0xAA;                                                      // Der Oszillator auf dem Testsystem wurde kalibriert, um optimale Timings fr die Sensor- und RS-232-Kommunikation zu ermitteln.
                                                                      // Dieser Schritt ist fr den Nachbau nicht erforderlich, gegebenenfalls knnen Fehler bei der RS-232-bertragung mit den Jumpern
                                                                      // J201 und J202 (an PA0 und PA1) beseitigt werden.
  */
  DDRA = 0;                                                           // PortA alles Eingnge
  PORTA = 0b11111111;                                                 // PortA alle Eingnge mit Pull-up
  BUS1_DDR &= ~_BV(BUS1_SIG);                                         // Bus 1 Port auf Eingang
  BUS2_DDR &= ~_BV(BUS2_SIG);                                         // Bus 2 Port auf Eingang
  RSTX_DDR |= _BV(RSTX_SIG);                                          // RS-232-TX Port auf Ausgang
  BUS1_PRT |= _BV(BUS1_SIG);                                          // Bus 1 Eingang mit Pull-up
  BUS2_PRT |= _BV(BUS2_SIG);                                          // Bus 2 Eingang mit Pull-up
  RSTX_PRT |= _BV(RSTX_SIG);                                          // RS-232-TX Port auf High

  ds2438b1[0] = 0x8000;                                               // ersten Pufferplatz mit ungltigem Temperaturwert beschreiben
  ds2438b2[0] = 0x8000;                                               // ersten Pufferplatz mit ungltigem Betriebsspannungswert beschreiben
  ds2438b3[0] = 0x8000;                                               // ersten Pufferplatz mit ungltigem Sensorspannungswert beschreiben
  _delay_ms(2000);                                                    // 2s warten (erforderlich fr DHT22/AM2302)

  senmode = dht_read();                                               // DHT22/AM2302 versuchen
  if (!senmode) {                                                     // wenn kein DHT22/AM2302 erkannt
    _delay_ms(10);                                                    // 10ms warten
    ow_search();                                                      // 1-Wire-Suche starten
    if (rom_id[0][0] == 0x26) senmode = 2;                            // wenn DS2438 gefunden -> Sensormodus 2 setzen (DS2438)
    else {                                                            // wenn kein DS2438 gefunden
      if (rom_id[0][0]) senmode = 3;                                  // wenn ein anderer Sensor gefunden -> Sensormodus 3 setzen
      if (rom_id[1][0]) senmode = 4;                                  // wenn zwei andere Sensoren gefunden -> Sensormodus 4 setzen
    }
  }
                                                                      // Sensorinformationen als Konfigurations-Datensatz ber RS-232 senden:

                                                                      // .-------- Datensatzkennzeichen "S"
                                                                      // |
                                                                      // |.------- Trennzeichen ":"
                                                                      // ||
                                                                      // ||.------ Kennung von Sensor 1: 0 = kein Sensor
                                                                      // |||                             1 = Temperatursensor (Sensorkennung 1 vom Temperaturmesssystem)
                                                                      // |||
                                                                      // |||.----- Kennung von Sensor 2: 0 = kein Sensor
                                                                      // ||||                            1 = Temperatursensor (Sensorkennung 1 vom Temperaturmesssystem)
                                                                      // ||||                            2 = Temperatursensor (Sensorkennung 2 vom Temperaturmesssystem)
                                                                      // ||||                            x = Luftfeuchtigkeitssensor (Sensorkennung 29 vom Temperaturmesssystem)
                                                                      // ||||
                                                                      // ||||.---- Typ von Sensor 1: 0 = kein Sensor
                                                                      // |||||                       1 = DHT22/AM2302 (Kombisensor T/H, Typ von Sensor 2 ist in diesem Fall 0)
                                                                      // |||||                       2 = DS2438 (Kombisensor T/H, Typ von Sensor 2 ist in diesem Fall 0)
                                                                      // |||||                       3 = DS18B20
                                                                      // |||||                       4 = DS1822
                                                                      // |||||                       5 = DS18S20
                                                                      // |||||                       9 = nicht untersttzter Sensor
                                                                      // |||||.--- Typ von Sensor 2: 0 = kein Sensor
                                                                      // ||||||                      3 = DS18B20
                                                                      // ||||||                      4 = DS1822
                                                                      // ||||||                      5 = DS18S20
                                                                      // ||||||                      9 = nicht untersttzter Sensor
                                                                      // S:abcd
                                                                      
  rs232_send('\r');                                                   // CR senden (Synchronisierung)
  rs232_send('S');                                                    // Kennung "S" senden (Sensorkonfiguration)
  rs232_send(':');                                                    // Trennzeichen senden
  switch (senmode) {                                                  // Sensormodus auswerten und verzweigen
    case 0:                                                           // wenn kein Sensor erkannt
      rs232_send('0');                                                // Sensorkennung 1 "0" senden
      rs232_send('0');                                                // Sensorkennung 2 "0" senden
      rs232_send('0');                                                // Sensortyp 1 "0" senden
      rs232_send('0');                                                // Sensortyp 2 "0" senden
      break;
    case 1:                                                           // wenn DHT22/AM2302
      rs232_send('1');                                                // Sensorkennung 1 "1" senden
      rs232_send('x');                                                // Sensorkennung 2 "x" senden
      rs232_send('1');                                                // Sensortyp 1 "DHT22" senden
      rs232_send('0');                                                // Sensortyp 2 "0" senden
      break;
    case 2:                                                           // wenn DS2438
      rs232_send('1');                                                // Sensorkennung 1 "1" senden
      rs232_send('x');                                                // Sensorkennung 2 "x" senden
      rs232_send('2');                                                // Sensortyp 1 "DS2438" senden
      rs232_send('0');                                                // Sensortyp 2 "0" senden
      break;
    case 3:                                                           // wenn ein anderer 1-Wire-Sensor
      rs232_send('1');                                                // Sensorkennung 1 "1" senden
      rs232_send('0');                                                // Sensorkennung 2 "0" senden
      switch (rom_id[0][0]) {                                         // Family-Code auswerten und verzweigen
        case 0x28: rs232_send('3'); break;                            // wenn DS18B20 -> Sensortyp 1 "DS18B20" senden
        case 0x22: rs232_send('4'); break;                            // wenn DS1822 -> Sensortyp 1 "DS1822" senden
        case 0x10: rs232_send('5'); break;                            // wenn DS18S20 -> Sensortyp 1 "DS18S20" senden
        default:   rs232_send('9'); break;                            // wenn nicht untersttzt -> Sensortyp 1 "nicht untersttzt" senden
      }
      rs232_send('0');                                                // Sensortyp 2 "0" senden
      break;
    case 4:                                                           // wenn 2 andere 1-Wire-Sensoren
      rs232_send('1');                                                // Sensorkennung 1 "1" senden
      rs232_send('2');                                                // Sensorkennung 2 "2" senden
      switch (rom_id[0][0]) {                                         // Family-Code von Sensor 1 auswerten und verzweigen
        case 0x28: rs232_send('3'); break;                            // wenn DS18B20 -> Sensortyp 1 "DS18B20" senden
        case 0x22: rs232_send('4'); break;                            // wenn DS1822 -> Sensortyp 1 "DS1822" senden
        case 0x10: rs232_send('5'); break;                            // wenn DS18S20 -> Sensortyp 1 "DS18S20" senden
        default:   rs232_send('9'); break;                            // wenn nicht untersttzt -> Sensortyp 1 "nicht untersttzt" senden
      }
      switch (rom_id[1][0]) {                                         // Family-Code von Sensor 2 auswerten und verzweigen
        case 0x28: rs232_send('3'); break;                            // wenn DS18B20 -> Sensortyp 2 "DS18B20" senden
        case 0x22: rs232_send('4'); break;                            // wenn DS1822 -> Sensortyp 2 "DS1822" senden
        case 0x10: rs232_send('5'); break;                            // wenn DS18S20 -> Sensortyp 2 "DS18S20" senden
        default:   rs232_send('9'); break;                            // wenn nicht untersttzt -> Sensortyp 2 "nicht untersttzt" senden
      }
      break;
  }
  rs232_send('\r');                                                   // CR senden
  _delay_ms(500);                                                     // 500ms warten

  while (1) {                                                         // Hauptprogrammschleife
    switch (senmode) {                                                // Sensormodus auswerten und verzweigen
      case 0: break;                                                  // wenn kein Sensor angeschlossen -> nichts tun
      case 1:                                                         // wenn DHT22/AM2302 angeschlossen
        e = 0;                                                        // Fehlerzhler lschen
        t = 0;                                                        // Temperaturwert zunchst lschen
        h = 0;                                                        // Luftfeuchtigkeitswert zunchst lschen
        x = dht_read();                                               // DHT22/AM2302 lesen
        if (x) {                                                      // wenn DHT22/AM2302 erfolgreich gelesen
          t = ((owbuff[2] & 0x7f) << 8) + owbuff[1];                  // Temperaturwert ermitteln
          if (t > 1250) e ++;                                         // wenn Temperaturwert > 125C -> Fehlerzhler erhhen
          else {                                                      // wenn Temperaturwert <= 125C
            if (owbuff[2] & 0x80) t = -t;                             // wenn Temperaturwert negativ -> in Zweierkomplement wandeln
          }
          h = (owbuff[4] << 8) + owbuff[3];                           // Luftfeuchtigkeitswert ermitteln
          if (h > 1000) e ++;                                         // wenn Luftfeuchtigkeitswert > 100% -> Fehlerzhler erhhen
        }
        else e ++;                                                    // wenn DHT22/AM2302 nicht erfolgreich gelesen -> Fehlerzhler erhhen
        if (!e) {                                                     // wenn kein Fehler
          int_to_asc(t, 2);                                           // Temperaturwert in ASCII wandeln (eine Nachkommastelle)
          rs232_send('1');                                            // Sensorkennung "1" senden
          rs232_send(':');                                            // Trennzeichen senden
          rs232_send(ascbuff[0]);                                     // Hunderterstelle senden
          rs232_send(ascbuff[1]);                                     // Zehnerstelle senden
          rs232_send(ascbuff[2]);                                     // Einerstelle senden
          rs232_send('.');                                            // Komma senden
          rs232_send(ascbuff[3]);                                     // Zehntelstelle senden
          rs232_send('\r');                                           // CR senden
          _delay_ms(1000);                                            // 1s warten
          int_to_asc(h / 10, 3);                                      // Luftfeuchtigkeitswert ganzzahlig in ASCII wandeln (ohne Nachkommastelle)
          rs232_send('x');                                            // Sensorkennung "x" senden
          rs232_send(':');                                            // Trennzeichen senden
          rs232_send(' ');                                            // Tausenderstelle senden
          rs232_send(ascbuff[0]);                                     // Hunderterstelle senden
          rs232_send(ascbuff[1]);                                     // Zehnerstelle senden
          rs232_send(ascbuff[2]);                                     // Einerstelle senden
          rs232_send(ascbuff[3]);                                     // Zehnerstelle senden
          rs232_send('\r');                                           // CR senden
        }
        _delay_ms(4000);                                              // 4s warten
        break;

      case 2:                                                         // wenn DS2438/HIH4000
        e = 0;                                                        // Fehlerzhler lschen
        x = ow_presence(rom_id[0][8]);                                // 1-Wire Reset und Presence
        if (!x) e ++;                                                 // wenn Presence nicht erkannt -> Fehlerzhler erhhen
        if (!e) {                                                     // wenn kein Fehler
          ow_write_byte(rom_id[0][8], 0x55);                          // 1-Wire Match-ROM senden
          ow_write_romid(rom_id[0][8], 0);                            // 1-Wire ROM-ID senden (logischer Sensor 0)
          ow_write_byte(rom_id[0][8], 0x44);                          // 1-Wire Convert T senden
          ow_str_pullup(rom_id[0][8]);                                // 1-Wire Strong Pull-up aktivieren
        }
        _delay_ms(11);                                                // 11ms warten
        x = ow_presence(rom_id[0][8]);                                // 1-Wire Reset und Presence
        if (!x) e ++;                                                 // wenn Presence nicht erkannt -> Fehlerzhler erhhen
        if (!e) {                                                     // wenn kein Fehler
          ow_write_byte(rom_id[0][8], 0x55);                          // 1-Wire Match-ROM senden
          ow_write_romid(rom_id[0][8], 0);                            // 1-Wire ROM-ID senden (logischer Sensor 0)
          ow_write_byte(rom_id[0][8], 0x4e);                          // 1-Wire Write Scratchpad senden
          ow_write_byte(rom_id[0][8], 0x00);                          // 1-Wire Page 0 senden
          ow_write_byte(rom_id[0][8], 0x0f);                          // 1-Wire Configuration senden (AD = 1, Sensorbetriebsspannung messen)
        }
        x = ow_presence(rom_id[0][8]);                                // 1-Wire Reset und Presence
        if (!x) e ++;                                                 // wenn Presence nicht erkannt -> Fehlerzhler erhhen
        if (!e) {                                                     // wenn kein Fehler
          ow_write_byte(rom_id[0][8], 0x55);                          // 1-Wire Match-ROM senden
          ow_write_romid(rom_id[0][8], 0);                            // 1-Wire ROM-ID senden (logischer Sensor 0)
          ow_write_byte(rom_id[0][8], 0xb4);                          // 1-Wire Convert V senden
          ow_str_pullup(rom_id[0][8]);                                // 1-Wire Strong Pull-up aktivieren
        }
        _delay_ms(11);                                                // 11ms warten
        x = ow_presence(rom_id[0][8]);                                // 1-Wire Reset und Presence
        if (!x) e ++;                                                 // wenn Presence nicht erkannt -> Fehlerzhler erhhen
        if (!e) {                                                     // wenn kein Fehler
          ow_write_byte(rom_id[0][8], 0x55);                          // 1-Wire Match-ROM senden
          ow_write_romid(rom_id[0][8], 0);                            // 1-Wire ROM-ID senden (logischer Sensor 0)
          ow_write_byte(rom_id[0][8], 0xb8);                          // 1-Wire Recall Memory senden
          ow_write_byte(rom_id[0][8], 0x00);                          // 1-Wire Page 0 senden
        }
        x = ow_presence(rom_id[0][8]);                                // 1-Wire Reset und Presence
        if (!x) e ++;                                                 // wenn Presence nicht erkannt -> Fehlerzhler erhhen
        if (!e) {                                                     // wenn kein Fehler
          ow_write_byte(rom_id[0][8], 0x55);                          // 1-Wire Match-ROM senden
          ow_write_romid(rom_id[0][8], 0);                            // 1-Wire ROM-ID senden (logischer Sensor 0)
          ow_write_byte(rom_id[0][8], 0xbe);                          // 1-Wire Read Scratchpad senden
          ow_write_byte(rom_id[0][8], 0x00);                          // 1-Wire Page 0 senden
          x = ow_read_scrp(rom_id[0][8]);                             // 1-Wire Scratchpad lesen
          if (!x) e ++;                                               // wenn CRC-Fehler -> Fehlerzhler erhhen
        }
        if (!e) {                                                     // wenn kein Fehler
          t = ((owbuff[2] << 8) + owbuff[1]) >> 3;                    // Temperaturwert aus 1-Wire-Puffer holen, als 13-Bit-Wert zusammenfassen und an richtige Bit-Position schieben
          if (ds2438b1[0] == 0x8000) {                                // wenn DS2438-Temperaturpuffer noch leer
            for (i = 0; i < 8; i ++) ds2438b1[i] = t;                 // kompletten Puffer mit Temperaturwert fllen
          }
          else {                                                      // wenn DS2438-Temperaturpuffer bereits Daten enthlt
            for (i = 0; i < 7; i ++) ds2438b1[i] = ds2438b1[i + 1];   // alle Werte um eine Pufferposition verschieben
            ds2438b1[7] = t;                                          // aktuellen Wert in letzte Pufferposition schreiben
          }
          b = (owbuff[4] << 8) + owbuff[3];                           // Sensorbetriebsspannungswert aus 1-Wire-Puffer holen und als 10-Bit-Wert zusammenfassen
          if (ds2438b2[0] == 0x8000) {                                // wenn DS2438-Sensorbetriebsspannungspuffer noch leer
            for (i = 0; i < 8; i ++) ds2438b2[i] = b;                 // kompletten Puffer mit Spannungswert fllen
          }
          else {                                                      // wenn DS2438-Sensorbetriebsspannungspuffer bereits Daten enthlt
            for (i = 0; i < 7; i ++) ds2438b2[i] = ds2438b2[i + 1];   // alle Werte um eine Pufferposition verschieben
            ds2438b2[7] = b;                                          // aktuellen Wert in letzte Pufferposition schreiben
          }
        }
        x = ow_presence(rom_id[0][8]);                                // 1-Wire Reset und Presence
        if (!x) e ++;                                                 // wenn Presence nicht erkannt -> Fehlerzhler erhhen
        if (!e) {                                                     // wenn kein Fehler
          ow_write_byte(rom_id[0][8], 0x55);                          // 1-Wire Match-ROM senden
          ow_write_romid(rom_id[0][8], 0);                            // 1-Wire ROM-ID senden (logischer Sensor 0)
          ow_write_byte(rom_id[0][8], 0x4e);                          // 1-Wire Write Scratchpad senden
          ow_write_byte(rom_id[0][8], 0x00);                          // 1-Wire Page 0 senden
          ow_write_byte(rom_id[0][8], 0x07);                          // 1-Wire Configuration senden (AD = 0, Sensorausgangsspannung messen)
        }
        x = ow_presence(rom_id[0][8]);                                // 1-Wire Reset und Presence
        if (!x) e ++;                                                 // wenn Presence nicht erkannt -> Fehlerzhler erhhen
        if (!e) {                                                     // wenn kein Fehler
          ow_write_byte(rom_id[0][8], 0x55);                          // 1-Wire Match-ROM senden
          ow_write_romid(rom_id[0][8], 0);                            // 1-Wire ROM-ID senden (logischer Sensor 0)
          ow_write_byte(rom_id[0][8], 0xb4);                          // 1-Wire Convert V senden
          ow_str_pullup(rom_id[0][8]);                                // 1-Wire Strong Pull-up aktivieren
        }
        _delay_ms(11);                                                // 11ms warten
        x = ow_presence(rom_id[0][8]);                                // 1-Wire Reset und Presence
        if (!x) e ++;                                                 // wenn Presence nicht erkannt -> Fehlerzhler erhhen
        if (!e) {                                                     // wenn kein Fehler
          ow_write_byte(rom_id[0][8], 0x55);                          // 1-Wire Match-ROM senden
          ow_write_romid(rom_id[0][8], 0);                            // 1-Wire ROM-ID senden (logischer Sensor 0)
          ow_write_byte(rom_id[0][8], 0xb8);                          // 1-Wire Recall Memory senden
          ow_write_byte(rom_id[0][8], 0x00);                          // 1-Wire Page 0 senden
        }
        x = ow_presence(rom_id[0][8]);                                // 1-Wire Reset und Presence
        if (!x) e ++;                                                 // wenn Presence nicht erkannt -> Fehlerzhler erhhen
        if (!e) {                                                     // wenn kein Fehler
          ow_write_byte(rom_id[0][8], 0x55);                          // 1-Wire Match-ROM senden
          ow_write_romid(rom_id[0][8], 0);                            // 1-Wire ROM-ID senden (logischer Sensor 0)
          ow_write_byte(rom_id[0][8], 0xbe);                          // 1-Wire Read Scratchpad senden
          ow_write_byte(rom_id[0][8], 0x00);                          // 1-Wire Page 0 senden
          x = ow_read_scrp(rom_id[0][8]);                             // 1-Wire Scratchpad lesen
          if (!x) e ++;                                               // wenn CRC-Fehler -> Fehlerzhler erhhen
        }
        if (!e) {                                                     // wenn kein Fehler
          a = (owbuff[4] << 8) + owbuff[3];                           // Sensorausgangsspannungswert aus 1-Wire-Puffer holen und als 10-Bit-Wert zusammenfassen
          if (ds2438b3[0] == 0x8000) {                                // wenn DS2438-Sensorausgangsspannungspuffer noch leer
            for (i = 0; i < 8; i ++) ds2438b3[i] = a;                 // kompletten Puffer mit Spannungswert fllen
          }
          else {                                                      // wenn DS2438-Sensorausgangsspannungspuffer bereits Daten enthlt
            for (i = 0; i < 7; i ++) ds2438b3[i] = ds2438b3[i + 1];   // alle Werte um eine Pufferposition verschieben
            ds2438b3[7] = a;                                          // aktuellen Wert in letzte Pufferposition schreiben
          }
          t = 0;                                                      // Mittelwert fr Temperatur ermitteln, zunchst Wert lschen
          for (i = 0; i < 8; i ++) t += ds2438b1[i];                  // 8 Pufferpositionen addieren
          t >>= 3;                                                    // Ergebnis / 8
          t = t * 5 / 16;                                             // Wert in Zehntel-Grad umrechnen
          if ((t < -550) || (t > 1250)) e ++;                         // wenn Temperatur < -55C oder > 125C -> Fehlerzhler erhhen
          b = 0;                                                      // Mittelwert fr Sensorbetriebsspannung ermitteln, zunchst Wert lschen
          for (i = 0; i < 8; i ++) b += ds2438b2[i];                  // 8 Pufferpositionen addieren
          b >>= 3;                                                    // Ergebnis / 8
          if ((b < 200) || (b > 600)) e ++;                           // wenn Sensorbetriebsspannung < 2,0V oder > 6,0V -> Fehlerzhler erhhen
          a = 0;                                                      // Mittelwert fr Sensorausgangsspannung ermitteln, zunchst Wert lschen
          for (i = 0; i < 8; i ++) a += ds2438b3[i];                  // 8 Pufferpositionen addieren
          a >>= 3;                                                    // Ergebnis / 8
          if ((a < 50) || (a > 500)) e ++;                            // wenn Sensorausgangsspannung < 0,5V oder > 5,0V -> Fehlerzhler erhhen
        }
        if (!e) {                                                     // wenn kein Fehler
                                                                      // Umrechnen der Sensormittelwerte von Ausgangsspannung und Betriebsspannung nach der Formel aus dem Datenblatt des HIH-4000:
                                                                      // Vout = Vs * (0.0062 * RH + 0.16)              Vout = Sensorausgangsspannung, Vs = Betriebsspannung, RH = Relative Feuchte (%)
                                                                      // RH soll ermittelt werden, gegeben sind die Messwerte Vout und Vs:
                                                                      // RH = (Vout / Vs - 0.16) / 0.0062              Modifikation der Formel zu Vereinfachung der Berechnung:
                                                                      // RH = (Vout / Vs) / 0.0062 - 0.16 / 0.0062     Weitere Vereinfachung:
                                                                      // RH = (Vout / Vs) / 0.0062 - 25,8
                                                                      // Nochmals Vereinfachung: Division durch 0.0062 wird durch eine Multiplikation mit 1 / 0.0062 = 161,3 ersetzt:
                                                                      // RH = Vout / Vs * 161,3 - 25,8
                                                                      // Rechnen mit den zehnfachen Werten, um mit ganzen Zahlen auszukommen und ein Ergebnis in Zehntel % zu erhalten:
                                                                      // RH = Vout / Vs * 1613 - 258
                                                                      // RH = Vout * 1613 / Vs - 258
          w = (int32_t) a * 1613 / b - 258;                           // Luftfeuchtigkeitswert mit 32-Bit-Genauigkeit nach optimierter Formel ermitteln

                                                                      // Temperaturkompensation des ermittelten Wertes nach der Formal aus dem Datenblatt des HIH-4000:
                                                                      // RHT = RH / (1.0546 - 0.00216 * T)             RHT = Temperatur-kompensierte Luftfeuchte, RH = Luftfeuchte, T = Temperatur
                                                                      // Rechnen mit den 10000-fachen Werten, um mit ganzen Zahlen auszukommen, die Abweichung durch das Runden ist minimal:
                                                                      // RHT = RH * 10000 / (10546 - 22 * T)
          w = w * 10000 / (10546 - 22 * t / 10);                      // Temperaturkompensierten Luftfeuchtigkeitswert mit 32-Bit-Genauigkeit nach optimierter Formel ermitteln

          if (w > 1000) w = 1000;                                     // wenn Ergebnis > 100% -> auf 100% begrenzen
          if (w < 0) w = 0;                                           // wenn Ergebnis < 0% -> auf 0% begrenzen

          int_to_asc(t, 2);                                           // Temperaturwert in ASCII wandeln (eine Nachkommastelle)
          rs232_send('1');                                            // Sensorkennung "1" senden
          rs232_send(':');                                            // Trennzeichen senden
          rs232_send(ascbuff[0]);                                     // Hunderterstelle senden
          rs232_send(ascbuff[1]);                                     // Zehnerstelle senden
          rs232_send(ascbuff[2]);                                     // Einerstelle senden
          rs232_send('.');                                            // Komma senden
          rs232_send(ascbuff[3]);                                     // Zehntelstelle senden
          rs232_send('\r');                                           // CR senden
          _delay_ms(1000);                                            // 1s warten
          int_to_asc(w / 10, 3);                                      // Luftfeuchtigkeitswert in ASCII wandeln (ohne Nachkommastelle)
          rs232_send('x');                                            // Sensorkennung "x" senden
          rs232_send(':');                                            // Trennzeichen senden
          rs232_send(' ');                                            // Tausenderstelle senden
          rs232_send(ascbuff[0]);                                     // Hunderterstelle senden
          rs232_send(ascbuff[1]);                                     // Zehnerstelle senden
          rs232_send(ascbuff[2]);                                     // Einerstelle senden
          rs232_send(ascbuff[3]);                                     // Zehnerstelle senden
          rs232_send('\r');                                           // CR senden
        }
        _delay_ms(4000);                                              // 4s warten
        break;

      case 3 ... 4:                                                   // wenn ein oder zwei 1-Wire-Sensor(en)
        s = 0;                                                        // Sensorzhler lschen
        while (s < (senmode - 2)) {                                   // solange noch Sensoren zu bearbeiten sind
          e = 0;                                                      // Fehlerzhler lschen
          x = ow_presence(rom_id[s][8]);                              // 1-Wire Reset und Presence
          if (!x) e ++;                                               // wenn Presence nicht erkannt -> Fehlerzhler erhhen
          if (!e) {                                                   // wenn kein Fehler
            ow_write_byte(rom_id[s][8], 0x55);                        // 1-Wire Match-ROM senden
            ow_write_romid(rom_id[s][8], s);                          // 1-Wire ROM-ID senden
            ow_write_byte(rom_id[s][8], 0x44);                        // 1-Wire Convert T senden
            ow_str_pullup(rom_id[s][8]);                              // 1-Wire Strong Pull-up aktivieren
          }
          _delay_ms(800);                                             // 800ms warten
          x = ow_presence(rom_id[s][8]);                              // 1-Wire Reset und Presence
          if (!x) e ++;                                               // wenn Presence nicht erkannt -> Fehlerzhler erhhen
          if (!e) {                                                   // wenn kein Fehler
            ow_write_byte(rom_id[s][8], 0x55);                        // 1-Wire Match-ROM senden
            ow_write_romid(rom_id[s][8], s);                          // 1-Wire ROM-ID senden
            ow_write_byte(rom_id[s][8], 0xbe);                        // 1-Wire Read Scratchpad senden
            x = ow_read_scrp(rom_id[s][8]);                           // 1-Wire Scratchpad lesen
            if (!x) e ++;                                             // wenn CRC-Fehler -> Fehlerzhler erhhen
          }
          if (!e) {                                                   // wenn kein Fehler
            if (owbuff[7] != 0x10) e ++;                              // wenn Scratchpad Byte 7 nicht 0x10 enthlt -> Fehlerzhler erhhen
          }

          if ((rom_id[s][0] == 0x28) || (rom_id[s][0] == 0x22)) {     // wenn Family-Code 0x28 (DS18B20) oder 0x22 (DS1822)
            if (!e) {                                                 // wenn kein Fehler
              t = ((owbuff[1] << 8) + owbuff[0]) * 5 / 8;             // Temperaturwert aus 1-Wire-Puffer holen, als 12-Bit-Wert zusammenfassen und in Zehntel-Grad umrechnen
              if ((t < -550) || (t > 1250)) e ++;                     // wenn Temperatur < -55C oder > 125C -> Fehlerzhler erhhen
            }
            if (!e) {                                                 // wenn kein Fehler
              int_to_asc(t, 2);                                       // Temperaturwert in ASCII wandeln (eine Nachkommastelle)
              rs232_send(s + '1');                                    // Sensorkennung "1" oder "2" senden
              rs232_send(':');                                        // Trennzeichen senden
              rs232_send(ascbuff[0]);                                 // Hunderterstelle senden
              rs232_send(ascbuff[1]);                                 // Zehnerstelle senden
              rs232_send(ascbuff[2]);                                 // Einerstelle senden
              rs232_send('.');                                        // Komma senden
              rs232_send(ascbuff[3]);                                 // Zehntelstelle senden
              rs232_send('\r');                                       // CR senden
            }
          }

          if (rom_id[s][0] == 0x10) {                                 // wenn Family-Code 0x10 (DS18S20)
            if (!e) {                                                 // wenn kein Fehler
                                                                      // Erhhung der Genauigkeit durch Algorithmus nach Datenblatt; es wird mit 160-fachen Werten gerechnet, so dass ganzzahlig
                                                                      // gerechnet werden kann: Temp = (160 * Temp_Read - 40 + 10 * (Count_Per_C - Count_Remain)) / 16 (Ergebnis in Zehntel-Grad)
              t = (((owbuff[1] << 8) + owbuff[0]) / 2 * 160 - 40 + 10 * (16 - owbuff[6])) / 16;
              if ((t < -550) || (t > 1250)) e ++;                     // wenn Temperatur < -55C oder > 125C -> Fehlerzhler erhhen
            }
            if (!e) {                                                 // wenn kein Fehler
              int_to_asc(t, 2);                                       // Temperaturwert in ASCII wandeln (eine Nachkommastelle)
              rs232_send(s + '1');                                    // Sensorkennung "1" oder "2" senden
              rs232_send(':');                                        // Trennzeichen senden
              rs232_send(ascbuff[0]);                                 // Hunderterstelle senden
              rs232_send(ascbuff[1]);                                 // Zehnerstelle senden
              rs232_send(ascbuff[2]);                                 // Einerstelle senden
              rs232_send('.');                                        // Komma senden
              rs232_send(ascbuff[3]);                                 // Zehntelstelle senden
              rs232_send('\r');                                       // CR senden
            }
          }
          s ++;                                                       // Sensorzhler erhhen
        }
        if (senmode == 3) _delay_ms(4200);                            // wenn nur ein Sensor -> 4,2s warten
        else _delay_ms(3400);                                         // wenn zwei Sensoren -> 3,4s warten
        break;
    }
  }
}
