/*
 * Wohnraumuhr 2, Sensor-Option fr 1-Wire-Sensoren und DHT22/AM2302, Scott-Falk Hhn, "functions.h"
 * Verschiedene Funktionen
 */

#ifndef FUNCTIONS_H_
#define FUNCTIONS_H_

void rs232_send(char c)                                               // Ein Zeichen ber RS-232 senden
{                                                                     // c: zu sendendes Zeichen
  uint8_t i;                                                          // Zhlervariable

  rsbdiff = 0;                                                        // normale Periodendauer setzen
  if ((PINA & _BV(PINA0)) == 0) rsbdiff = 1;                          // wenn Jumper an PINA0 gesteckt (low) -> Periodendauer um 3s verkrzen
  if ((PINA & _BV(PINA1)) == 0) rsbdiff = 2;                          // wenn Jumper an PINA1 gesteckt (low) -> Periodendauer um 3s verlngern
  RSTX_PRT &= ~_BV(RSTX_SIG);                                         // Startbit setzen (Low-Pegel)
  switch(rsbdiff) {                                                   // Periodendauer-Differenz auswerten und verzweigen
    case 0: _delay_us(RSBITP); break;                                 // eine normale Taktperiode warten
    case 1: _delay_us(RSBITP - 3); break;                             // eine verkrzte Taktperiode warten
    case 2: _delay_us(RSBITP + 3); break;                             // eine verlngerte Taktperiode warten
  }
  for (i = 0; i < 8; i ++) {                                          // 8 Datenbits bearbeiten
    if (c & 0x01) RSTX_PRT |= _BV(RSTX_SIG);                          // wenn LSB gesetzt -> High-Pegel setzen
    else RSTX_PRT &= ~_BV(RSTX_SIG);                                  // wenn LSB nicht gesetzt -> Low-Pegel setzen
    switch(rsbdiff) {                                                 // Periodendauer-Differenz auswerten und verzweigen
      case 0: _delay_us(RSBITP); break;                               // eine normale Taktperiode warten
      case 1: _delay_us(RSBITP - 3); break;                           // eine verkrzte Taktperiode warten
      case 2: _delay_us(RSBITP + 3); break;                           // eine verlngerte Taktperiode warten
    }
    c >>= 1;                                                          // nchstes Bit an LSB schieben
  }
  RSTX_PRT |= _BV(RSTX_SIG);                                          // Stoppbit setzen (High-Pegel)
  switch(rsbdiff) {                                                   // Periodendauer-Differenz auswerten und verzweigen
    case 0: _delay_us(RSBITP); break;                                 // eine normale Taktperiode warten
    case 1: _delay_us(RSBITP - 3); break;                             // eine verkrzte Taktperiode warten
    case 2: _delay_us(RSBITP + 3); break;                             // eine verlngerte Taktperiode warten
  }
}

void rs232_sendhex(uint8_t n)                                         // Ein Byte hexadezimal 2-stellig ber RS-232 senden (fr Testzwecke, wird ansonsten nicht bentigt)
{                                                                     // n: zu sendendes Byte
  uint8_t h, l;                                                       // h: Zwischenspeicher fr oberes Nibble, l: Zwischenspeicher fr unteres Nibble
  h = n / 16;                                                         // oberes Nibble ermitteln
  l = n % 16;                                                         // unteres Nibble ermitteln
  if (h > 9) rs232_send(h + 55);                                      // wenn Wert ber 9 -> A-F ausgeben
  else rs232_send(h + 48);                                            // sonst 0-9 ausgeben
  if (l > 9) rs232_send(l + 55);                                      // wenn Wert ber 9 -> A-F ausgeben
  else rs232_send(l + 48);                                            // sonst 0-9 ausgeben
}

void int_to_asc(int16_t n, uint8_t k)                                 // Eine 16-Bit-Integer-Zahl in ASCII konvertieren, Ergebnis wird im ASCII-Puffer gespeichert
{                                                                     // n: Temperatur- oder Luftfeuchtigkeitswert, k: Komma-Position (zur Unterdrckung fhrender Nullen)
  uint8_t i;                                                          // Zhlervariable
  uint8_t s = 0;                                                      // Vorzeichen-Flag
  
  if (n < 0) {                                                        // wenn Wert negativ
    n = -n;                                                           // in Zweierkomplement wandeln
    s = 1;                                                            // Vorzeichen-Flag setzen
  }
  for (i = 0; i < 4; i ++) ascbuff[i] = 0;                            // zunchst ASCII-Puffer mit Nullwerten fllen
  if (n > 9999) n = 9999;                                             // Zahl auf 9999 begrenzen
  if (n > 999) {                                                      // wenn Zahl > 999
    ascbuff[0] = n / 1000;                                            // Tausenderstelle speichern
    n -= ascbuff[0] * 1000;                                           // Tausenderstelle von Restwert subtrahieren
  }
  if (n > 99) {                                                       // wenn Restwert > 99
    ascbuff[1] = n / 100;                                             // Hunderterstelle speichern
    n -= ascbuff[1] * 100;                                            // Hunderterstelle von Restwert subtrahieren
  }
  if (n > 9) {                                                        // wenn Restwert > 9
    ascbuff[2] = n / 10;                                              // Zehnerstelle speichern
    n -= ascbuff[2] * 10;                                             // Zehnerstelle von Restwert subtrahieren
  }
  ascbuff[3] = n;                                                     // Einerstelle speichern
  for (i = 0; i < 4; i ++) ascbuff[i] += '0';                         // alle Stellen in ASCII wandeln
  for (i = 0; i < k; i ++) {                                          // fhrende Nullen von links bis zur Komma-Position prfen
    if (ascbuff[i] == '0') ascbuff[i] = ' ';                          // wenn fhrende Null -> durch Leerzeichen ersetzen
    else break;                                                       // wenn keine fhrende Null -> Schleife abbrechen
  }
  if (s) {                                                            // wenn Vorzeichen-Flag gesetzt
    for (i = 2; i >= 0; i --) {                                       // alle Anzeigestellen von Komma-Position aus nach links prfen
      if (ascbuff[i] == ' ') {                                        // wenn Leerzeichen gefunden
        ascbuff[i] = '-';                                             // Minuszeichen direkt vor den Temperaturwert setzen
        break;                                                        // Schleife abbrechen
      }
    }
  }
}

uint8_t ow_presence(uint8_t bus)                                      // 1-Wire, Reset senden und Presence lesen
{                                                                     // bus: logische Busnummer (0-1), Ergebnis: 0 = Fehler, 1 = Presence ok
  uint8_t r;                                                          // Zwischenspeicher fr Rckgabewert

  if (bus) {                                                          // wenn Busnummer 2 (logisch 1)
    BUS2_PRT &= ~_BV(BUS2_SIG);                                       // Strong-Pull-Up ausschalten
    BUS2_DDR |= _BV(BUS2_SIG);                                        // Port auf Ausgang setzen (Low)
    _delay_us(480);                                                   // 480s warten
    BUS2_DDR &= ~_BV(BUS2_SIG);                                       // Port auf Eingang setzen
    _delay_us(60);                                                    // 60s warten
    r = ~BUS2_PIN & _BV(BUS2_SIG);                                    // Portpin lesen und als Presence zurckgeben
  }
  else {                                                              // wenn Busnummer 1 (logisch 0)
    BUS1_PRT &= ~_BV(BUS1_SIG);                                       // Strong-Pull-Up ausschalten
    BUS1_DDR |= _BV(BUS1_SIG);                                        // Port auf Ausgang setzen (Low)
    _delay_us(480);                                                   // 480s warten
    BUS1_DDR &= ~_BV(BUS1_SIG);                                       // Port auf Eingang setzen
    _delay_us(60);                                                    // 60s warten
    r = ~BUS1_PIN & _BV(BUS1_SIG);                                    // Portpin lesen und als Presence zurckgeben
  }
  _delay_us(420);                                                     // 420s warten
  return r;                                                           // Presence zurckgeben (1 = ok)
}

void ow_str_pullup(uint8_t bus)                                       // 1-Wire, Strong-Pull-up einschalten (1-Wire-Bus auf High setzen)
{                                                                     // bus: logische Busnummer (0-1)
  if (bus) {                                                          // wenn Busnummer 2 (logisch 1)
    BUS2_PRT |= _BV(BUS2_SIG);                                        // Port auf High setzen
    BUS2_DDR |= _BV(BUS2_SIG);                                        // Port auf Ausgang setzen
  }
  else {                                                              // wenn Busnummer 1 (logisch 0)
    BUS1_PRT |= _BV(BUS1_SIG);                                        // Port auf High setzen
    BUS1_DDR |= _BV(BUS1_SIG);                                        // Port auf Ausgang setzen
  }
}

void ow_write_bit(uint8_t bus, uint8_t bit)                           // 1-Wire, Schreiben eines Bits
{                                                                     // bus: logische Busnummer (0-1), bit: zu schreibendes Bit
  if (bus) {                                                          // wenn Busnummer 2 (logisch 1)
    BUS2_DDR |= _BV(BUS2_SIG);                                        // Port auf Ausgang setzen (Low)
    if (bit) {                                                        // wenn 1-Bit
      _delay_us(2);                                                   // 2s warten
      BUS2_DDR &= ~_BV(BUS2_SIG);                                     // Port auf Eingang setzen
      _delay_us(60);                                                  // 60s warten
    }
    else {                                                            // wenn 0-Bit
      _delay_us(60);                                                  // 60s warten
      BUS2_DDR &= ~_BV(BUS2_SIG);                                     // Port auf Eingang setzen
      _delay_us(5);                                                   // 5s warten
    }
  }
  else {                                                              // wenn Busnummer 1 (logisch 0)
    BUS1_DDR |= _BV(BUS1_SIG);                                        // Port auf Ausgang setzen (Low)
    if (bit) {                                                        // wenn 1-Bit
      _delay_us(2);                                                   // 2s warten
      BUS1_DDR &= ~_BV(BUS1_SIG);                                     // Port auf Eingang setzen
      _delay_us(60);                                                  // 60s warten
    }
    else {                                                            // wenn 0-Bit
      _delay_us(60);                                                  // 60s warten
      BUS1_DDR &= ~_BV(BUS1_SIG);                                     // Port auf Eingang setzen
      _delay_us(5);                                                   // 5s warten
    }
  }
}

void ow_write_byte(uint8_t bus, uint8_t byte)                         // 1-Wire, Schreiben eines Bytes
{                                                                     // bus: logische Busnummer (0-1), byte: zu schreibendes Byte
  uint8_t i;                                                          // Zhlervariable

  for (i = 0; i < 8; i ++) {                                          // 8 Bits bearbeiten
    ow_write_bit(bus, byte & 0x01);                                   // LSB schreiben
    byte >>= 1;                                                       // nchstes Bit an LSB-Position schieben
  }
}

uint8_t ow_read_bit(uint8_t bus)                                      // 1-Wire, Lesen eines Bytes
{                                                                     // bus: logische Busnummer (0-1), Ergebnis: gelesenes Bit (0-1)
  uint8_t r;                                                          // Zwischenspeicher fr Rckgabewert

  if (bus) {                                                          // wenn Busnummer 2 (logisch 1)
    BUS2_DDR |= _BV(BUS2_SIG);                                        // Port auf Ausgang setzen (Low)
    _delay_us(2);                                                     // 2s warten
    BUS2_DDR &= ~_BV(BUS2_SIG);                                       // Port auf Eingang setzen
    _delay_us(9);                                                     // 9s warten
    r = BUS2_PIN & _BV(BUS2_SIG);                                     // Portpin lesen und Bitwert zurckgeben
  }
  else {                                                              // wenn Busnummer 1 (logisch 0)
    BUS1_DDR |= _BV(BUS1_SIG);                                        // Port auf Ausgang setzen (Low)
    _delay_us(2);                                                     // 2s warten
    BUS1_DDR &= ~_BV(BUS1_SIG);                                       // Port auf Eingang setzen
    _delay_us(9);                                                     // 9s warten
    r = BUS1_PIN & _BV(BUS1_SIG);                                     // Portpin lesen und Bitwert zurckgeben
  }
  _delay_us(50);                                                      // 50s warten
  if (r) r = 1;                                                       // wenn Bitwert > 0 -> auf 1 setzen
  return r;                                                           // Bitwert zurckgeben
}

uint8_t ow_read_byte(uint8_t bus)                                     // 1-Wire, Lesen eines Bits
{                                                                     // bus: logische Busnummer (0-1), Ergebnis: gelesenes Byte
  uint8_t i;                                                          // Zhlervariable
  uint8_t byte = 0;                                                   // Zwischenspeicher fr gelesenes Byte

  for (i = 0; i < 8; i ++) {                                          // 8 Bits bearbeiten
    byte >>= 1;                                                       // MSB-Position fr nchstes Bit vorbereiten
    byte |= ow_read_bit(bus) * 0x80;                                  // Bit lesen und in Byte an MSB einfgen
  }
  return byte;                                                        // Byte zurckgeben
}

uint8_t ow_crc(uint8_t n)                                             // 1-Wire, CRC8-Berechnung (nach Application Note 27 von https://www.maximintegrated.com)
{                                                                     // n: Anzahl der Bytes (1-9), Ergebnis: ermittelter CRC-Wert
  uint8_t i;                                                          // Zhlervariable
  uint8_t c = 0;                                                      // CRC-Anfangswert

  for (i = 0; i < n; i ++) {                                          // Schleifen entsprechend der bergebenen Anzahl
    c = pgm_read_byte(&crctab[c ^ owbuff[i]]);                        // Pufferbyte mit bisherigem CRC-Wert verknpfen (XOR) und neuen CRC-Wert aus Tabelle holen
  }
  return c;                                                           // CRC-Wert zurckgeben
}

void ow_save_bit(uint8_t bitnr, uint8_t bitval)                       // 1-Wire, Bit im 1-Wire Puffer speichern
{                                                                     // bitnr: Bitnummer (1-64), bitval: Bitwert (0-1)
  uint8_t p;                                                          // Pufferposition

  bitnr --;                                                           // Bitnummer korrigieren (0-63)
  p = bitnr / 8;                                                      // Pufferposition ermitteln
  if (bitval) owbuff[p] |= _BV(bitnr % 8);                            // wenn 1-Bit -> in Puffer setzen
  else owbuff[p] &= ~_BV(bitnr % 8);                                  // wenn 0-Bit -> in Puffer lschen
}

uint8_t ow_load_bit(uint8_t bitnr)                                    // 1-Wire, Bit aus 1-Wire Puffer holen
{                                                                     // bitnr: Bitnummer (1-64), bitval: Bitwert (0-1)
  uint8_t p;                                                          // Pufferposition
  uint8_t b;                                                          // Zwischenspeicher fr Bit

  bitnr --;                                                           // Bitnummer korrigieren (0-63)
  p = bitnr / 8;                                                      // Pufferposition ermitteln
  b = owbuff[p] & _BV(bitnr % 8);                                     // Bit aus Puffer holen und filtern
  if (b) b = 1;                                                       // wenn Bitwert > 0 -> auf 1 setzen
  return b;                                                           // Bitwert zurckgeben
}

void ow_search(void)                                                  // 1-Wire, Sensorsuche (nach Application Note 187 von https://www.maximintegrated.com)
{                                                                     // sucht nach 1-Wire-Sensoren und speichert maximal 2 ROM-IDs
  uint8_t i;                                                          // Zhlervariable
  uint8_t bus;                                                        // Busnummer (0-1)
  uint8_t sensor;                                                     // Sensornummer (0-1)
  uint8_t id_bit;                                                     // gelesenes Original-Bit (0-1)
  uint8_t cmp_id_bit;                                                 // gelesenes Komplementr-Bit (0-1)
  uint8_t id_bit_number;                                              // laufende Bitnummer (1-64)
  uint8_t LastDeviceFlag;
  uint8_t LastDiscrepancy;
  uint8_t last_zero;
  uint8_t search_direction;
  uint8_t return_value;

  bus = 0;                                                            // Busnummer = 0 setzen
  sensor = 0;                                                         // Sensornummer = 0 setzen
  LastDeviceFlag = 0;
  LastDiscrepancy = 0;
  return_value = 0;                                                   // Rckgabewert lschen

  while ((sensor < 2) && (bus < 2)) {                                 // Schleife solange noch nicht 2 Sensoren gefunden und 2 Busse durchsucht worden sind
    if (ow_presence(bus)) {                                           // wenn Sensor present
      if (!LastDeviceFlag) {                                          // wenn nicht letzter Sensor
        id_bit_number = 1;                                            // Bitnummer auf 1 setzen
        last_zero = 0;                                                // Last_Zero lschen
        ow_write_byte(bus, 0xf0);                                     // 1-Wire-Kommando "Search ROM" senden
        do {                                                          // Schleife
          id_bit = ow_read_bit(bus);                                  // Original-Bit lesen
          cmp_id_bit = ow_read_bit(bus);                              // Komplementr-Bit lesen
          if ((id_bit == 1) && (cmp_id_bit == 1)) break;              // wenn Original- und Komplementr-Bit = 1 (keine Sensorantwort) -> Abbruch
          else {                                                      // wenn Sensorantwort
            if ((id_bit == 0) && (cmp_id_bit == 0)) {                       // wenn Original- und Komplementr-Bit = 0
              if (id_bit_number == LastDiscrepancy) search_direction = 1;
              else {
                if (id_bit_number > LastDiscrepancy) search_direction = 0;
                else search_direction = ow_load_bit(id_bit_number);
              }
              if (search_direction == 0) last_zero = id_bit_number;
            }
            else search_direction = id_bit;                           // wenn Original- und Komplementr-Bit <> 0
            ow_save_bit(id_bit_number, search_direction);             // ermitteltes Bit im 1-Wire-Puffer speichern
            ow_write_bit(bus, search_direction);                      // ermitteltes Bit senden
            id_bit_number ++;                                         // Bitnummer erhhen
          }
        } while (id_bit_number < 65);                                 // solange noch nicht 64 Bits bearbeitet worden sind

        if (id_bit_number > 64) {                                     // wenn kein Abbruch
          LastDiscrepancy = last_zero;
          if (LastDiscrepancy == 0) LastDeviceFlag = 1;
          if (ow_crc(8) == 0) return_value = 1;                       // wenn CRC-Wert ok -> Rckgabewert setzen
        }
      }
      else {                                                          // wenn letzter Sensor
        LastDiscrepancy = 0;
        LastDeviceFlag = 0;
        return_value = 0;                                             // Rckgabewert lschen
        bus ++;                                                       // nchste Busnummer
      }
    }
    else {                                                            // wenn kein Sensor present
      LastDiscrepancy = 0;
      LastDeviceFlag = 0;
      return_value = 0;                                               // Rckgabewert lschen
      bus ++;                                                         // nchste Busnummer
    }
    if (return_value) {                                               // wenn Rckgabewert vorhanden
      for (i = 0; i < 8; i ++) rom_id[sensor][i] = owbuff[i];         // 8 Bytes aus dem 1-Wire-Puffer als ROM-ID speichern
      rom_id[sensor][8] = bus;                                        // Busnummer speichern
      sensor ++;                                                      // nchster Sensor
    }
  }
}

void ow_write_romid(uint8_t bus, uint8_t s)                           // 1-Wire, Schreiben der ROM-ID (8 Bytes)
{                                                                     // bus: logische Busnummer (0-1), s: logische Sensornummer (0-1)
  uint8_t i;                                                          // Zhlervariable

  for (i = 0; i < 8; i ++) ow_write_byte(bus, rom_id[s][i]);          // 8 Bytes ROM-ID schreiben
}

uint8_t ow_read_scrp(uint8_t bus)                                     // 1-Wire, Lesen des Scratchpads (8 Bytes + CRC)
{                                                                     // bus: logische Busnummer (0-1)
  uint8_t i;                                                          // Zhlervariable

  for (i = 0; i < 9; i ++) owbuff[i] = ow_read_byte(bus);             // 9 Bytes Scratchpad lesen
  return (ow_crc(8) == owbuff[8]);                                    // Ergebnis der CRC-Prfung zurckgeben (1 = ok)
}

uint8_t dht_read(void)                                                // DHT22/AM2302-Sensor erkennen und lesen
{                                                                     // Rckgabe: 0 = Fehler, 1 = ok
  uint8_t i;                                                          // Zhlervariable
  uint8_t bitnum;                                                     // Bitnummer (1-40)
  uint8_t bitval;                                                     // Bitwert (0-1)
  uint8_t ok;                                                         // ok-Flag (0-1)
  uint8_t csum;                                                       // Prfsumme

  ok = 1;                                                             // zunchst ok-Flag setzen
  BUS1_PRT &= ~_BV(BUS1_SIG);                                         // Port auf Low setzen
  BUS1_DDR |= _BV(BUS1_SIG);                                          // Port auf Ausgang setzen (Low)
  _delay_us(800);                                                     // 800s warten
  BUS1_DDR &= ~_BV(BUS1_SIG);                                         // Port auf Eingang setzen (High)
  _delay_us(20);                                                      // 20s warten

  for (i = 180; i > 0; i --) {                                        // Port 180 mal im Abstand von 1s abfragen (Sollwert 0-180s)
    if ((BUS1_PIN & _BV(BUS1_SIG)) == 0) break;                       // wenn Pegelwechsel auf Low erkannt -> Schleife abbrechen
    _delay_us(1);                                                     // 1s warten
  }
  if (i == 0) ok = 0;                                                 // wenn Schleife nicht abgebrochen wurde -> Fehler (Timeout)
  for (i = 85; i > 0; i --) {                                         // Port 90 mal im Abstand von 1s abfragen (Sollwert 85s)
    if ((BUS1_PIN & _BV(BUS1_SIG)) != 0) break;                       // wenn Pegelwechsel auf High erkannt -> Schleife abbrechen
    _delay_us(1);                                                     // 1s warten
  }
  if (i == 0) ok = 0;                                                 // wenn Schleife nicht abgebrochen wurde -> Fehler (Timeout)
  for (i = 85; i > 0; i --) {                                         // Port 90 mal im Abstand von 1s abfragen (Sollwert 85s)
    if ((BUS1_PIN & _BV(BUS1_SIG)) == 0) break;                       // wenn Pegelwechsel auf Low erkannt -> Schleife abbrechen
    _delay_us(1);                                                     // 1s warten
  }
  if (i == 0) ok = 0;                                                 // wenn Schleife nicht abgebrochen wurde -> Fehler (Timeout)
  if (ok) {                                                           // wenn bisher kein Fehler aufgetreten ist
    for (bitnum = 40; bitnum > 0; bitnum --) {                        // 40 Bits einlesen; weil MSB zuerst kommt, rckwrts zhlen
      for (i = 55; i > 0; i --) {                                     // Port 55 mal im Abstand von 1s abfragen (Sollwert 50s)
        if ((BUS1_PIN & _BV(BUS1_SIG)) != 0) break;                   // wenn Pegelwechsel auf High erkannt -> Schleife abbrechen
        _delay_us(1);                                                 // 1s warten
      }
      if (i == 0) ok = 0;                                             // wenn Schleife nicht abgebrochen wurde -> Fehler (Timeout)
      for (i = 80; i > 0; i --) {                                     // Port 80 mal im Abstand von 1s abfragen (Sollwert 27s oder 70s)
        if ((BUS1_PIN & _BV(BUS1_SIG)) == 0) break;                   // wenn Pegelwechsel auf Low erkannt -> Schleife abbrechen
        _delay_us(1);                                                 // 1s warten
      }
      if (i == 0) ok = 0;                                             // wenn Schleife nicht abgebrochen wurde -> Fehler (Timeout)
      if (i < 51) bitval = 1; else bitval = 0;                        // wenn Restwert des Zhlers < 51 -> 1-Bit, sonst 0-Bit
      ow_save_bit(bitnum, bitval);                                    // Bitwert im 1-Wire-Puffer speichern
    }
  }
  if (ok) {                                                           // wenn bisher kein Fehler aufgetreten ist
    csum = 0;                                                         // Prfsumme lschen
    for (i = 1; i < 5; i ++) csum += owbuff[i];                       // gelesene Bytes an Pufferposition 1-5 addieren (nur 8 Bit)
    if (csum != owbuff[0]) ok = 0;                                    // wenn errechnete Prfsumme nicht mit der gelesenen Prfsumme bereinstimmt -> Fehler
  }
  return ok;                                                          // Fehlerwert zurckgeben
}

#endif /* FUNCTIONS_H_ */
