/*
 * Wohnraumuhr 2 mit 4-stelliger 7-Segment-Anzeige und optionaler 2-stelliger 16-Segment-Anzeige, Scott-Falk Hhn, "main.c", v1.01, 02.04.2020
 * AtmelStudio 7.x Editor-Einstellung: Tools / Options / Text Editor / GCC / Tabs / Smart, Tab size: 2, Indent size: 2, Insert spaces
 *
 * Controller: ATmega128A
 * Taktfrequenz 14,7456MHz
 * Betriebsspannung: 5,0V
 *
 * Verwendung der I/O-Ports:
 * PA0 - Ausgang Anzeige 1, Segment A, Low-aktiv
 * PA1 - Ausgang Anzeige 1, Segment B, Low-aktiv
 * PA2 - Ausgang Anzeige 1, Segment C, Low-aktiv
 * PA3 - Ausgang Anzeige 1, Segment D, Low-aktiv
 * PA4 - Ausgang Anzeige 1, Segment E, Low-aktiv
 * PA5 - Ausgang Anzeige 1, Segment F, Low-aktiv
 * PA6 - Ausgang Anzeige 1, Segment G, Low-aktiv
 * PA7 - Ausgang Anzeige 1, Segment DP, Low-aktiv
 * PB0 - Ausgang Doppelpunkt, Low-aktiv
 * PB1 - frei (ISP-SCK)
 * PB2 - Ausgang Spannungsregler fr LED, Low-aktiv
 * PB3 - Ausgang Anzeige 5+6, ENA (Schieberegister Ausgnge freigeben)
 * PB4 - Ausgang Anzeige 5+6, SI (Schieberegister Datenleitung)
 * PB5 - Ausgang Anzeige 5+6, SCK (Schieberegister Takt)
 * PB6 - Ausgang Anzeige 5+6, RCK (Schieberegister Datenbernahme)
 * PB7 - Ausgang Reset fr Sensor-Controller, Low-aktiv
 * PC0 - Ausgang Anzeige 2, Segment A, Low-aktiv
 * PC1 - Ausgang Anzeige 2, Segment B, Low-aktiv
 * PC2 - Ausgang Anzeige 2, Segment C, Low-aktiv
 * PC3 - Ausgang Anzeige 2, Segment D, Low-aktiv
 * PC4 - Ausgang Anzeige 2, Segment E, Low-aktiv
 * PC5 - Ausgang Anzeige 2, Segment F, Low-aktiv
 * PC6 - Ausgang Anzeige 2, Segment G, Low-aktiv
 * PC7 - Ausgang Anzeige 2, Segment DP, Low-aktiv
 * PD0 - Ausgang Anzeige 3, Segment A, Low-aktiv
 * PD1 - Ausgang Anzeige 3, Segment B, Low-aktiv
 * PD2 - Eingang RXD1 vom Sensormodul
 * PD3 - Ausgang Anzeige 3, Segment D, Low-aktiv
 * PD4 - Ausgang Anzeige 3, Segment E, Low-aktiv
 * PD5 - Ausgang Anzeige 3, Segment F, Low-aktiv
 * PD6 - Ausgang Anzeige 3, Segment G, Low-aktiv
 * PD7 - Ausgang Anzeige 3, Segment DP, Low-aktiv
 * PE0 - Eingang RXD0 vom USB (SPI-MOSI)
 * PE1 - Ausgang TXD0 zum USB (SPI-MISO)
 * PE2 - Ausgang Anzeige 4, Segment C, Low-aktiv
 * PE3 - Ausgang Anzeige 4, Segment D, Low-aktiv
 * PE4 - Ausgang Anzeige 4, Segment E, Low-aktiv
 * PE5 - Ausgang Anzeige 4, Segment F, Low-aktiv
 * PE6 - Ausgang Anzeige 4, Segment G, Low-aktiv
 * PE7 - Ausgang Anzeige 4, Segment DP, Low-aktiv
 * PF0 - Eingang Lichtsensor analog
 * PF1 - Eingang Taster 1 mit Pull-up
 * PF2 - Eingang Taster 2 mit Pull-up
 * PF3 - Eingang DCF77-Empfnger, Pull-up schaltbar
 * PF4 - Eingang Anzeige-Invertierung 1-4 mit Pull-up
 * PF5 - Eingang Anzeige-Invertierung 5-6 mit Pull-up
 * PF6 - Ausgang Anzeige 5, Segment DP, Low-aktiv
 * PF7 - Ausgang Anzeige 6, Segment DP, Low-aktiv
 * PG0 - Ausgang Anzeige 4, Segment A, Low-aktiv
 * PG1 - Ausgang Anzeige 4, Segment B, Low-aktiv
 * PG2 - Ausgang Anzeige 3, Segment C, Low-aktiv
 *
 * Verwendung der Timer:
 * Timer0  - Steuerung der Anzeige-Helligkeit ber PWM, wird von Timer1 gestartet
 * Timer1A - Ausgabe der Segmente an die Anzeigeelemente und starten von Timer0 zum genauen Ausschalten (PWM), bei niedriger Helligkeit schaltet Timer1 die Anzeige direkt wieder aus (1024Hz / 977s)
 * Timer3A - Abfrage von DCF77-Signal, Lichtsensor, Taster und Timeout-Zhler; Zeitzhlung (50Hz / 20ms)
 *
 * Verwendung der USART:
 * USART0 - TXD/RXD im Asynchron-Modus 8N1, Baudrate 230,4k, Empfang ber Interrupt, Erkennung einer Bootloader-Anfrage
 * USART1 - RXD im Asynchron-Modus 8N1, Baudrate 9600, Empfang ber Interrupt
 *
 * Verwendung des ADC:
 * Abfrage des Lichtsensors (PF0), Vorteiler 128, ADC-Takt: 115,2kHz
 *
 * Einstellung der Fuse-Bits:
 * M103C =     [ ]
 * WDTON =     [ ]
 * OCDEN =     [ ]
 * JTAGEN =    [ ]
 * SPIEN =     [X]
 * EESAVE =    [ ]                          <- Ist im Auslieferungszustand des Crumb128 nicht gesetzt und kann so bleiben, wenn die Software immer ber den Bootloader installiert wird. Soll die
 * BOOTSZ =    1024W_FC00                      Software ber ISP installiert werden, dann muss dieses Bit gesetzt werden. Dies verhindert einen EEPROM-Datenverlust bei spteren Updates.
 * BOOTRST =   [X]
 * CKOPT =     [ ]
 * BODLEVEL =  2V7
 * BODEN =     [X]                          <- Ist im Auslieferungszustand des Crumb128 nicht gesetzt; sollte aber gesetzt werden, um Fehlfunktionen bei Unterspannung zu vermeiden.
 * SUT_CKSEL = EXTHIFXTALRES_16KCK_64MS
 *
 * EXTENDED =  0xFF
 * HIGH =      0xDC
 * LOW =       0xBF
 */ 

// Definitionen und Bibliotheken

#define F_CPU 14745600UL          // Taktfrequenz 14,7456MHz
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include <util/delay.h>

// Konstanten

const char strd00[] PROGMEM = " 101V "; // Versionsnummer
#define USBBAUD 230400            // USB-Geschwindigkeit 230,4kBit/s
#define RSBAUD 9600               // RS-232-Geschwindigkeit 9600Bit/s
#define TIM1_CLK 1024             // Timer1 Taktfrequenz 1024Hz
#define TIM3_CLK 50               // Timer3 Taktfrequenz 50Hz
#define DCFTOUT 30                // Zeit in Minuten bis zur Wiedereinschaltung des Displays bei fester DCF77-Synchronisierungszeit
#define SYNCTOUT 25               // Zeit in Stunden bis eine DCF77-Neusynchronisierung erforderlich ist
#define CONFTOUT 120              // Zeit in Sekunden bis zur automatischen Rckkehr aus der Konfiguration in die normale Anzeige
#define SENTOUT 120               // Zeit in Sekunden bis ein Sensorwert ungltig wird
#define DCFHOU 3                  // Voreinstellung fr die feste Synchronisierungszeit (3:00 Uhr)
#define YEARH 20                  // Jahrhundert
#define YEAR 20                   // Jahr Anfangswert und Minimalwert fr manuelle Zeiteinstellung
#define VERTOUT 5                 // Anzeigezeit in Sekunden fr die Versionsnummer beim Systemstart
#define USBTOUT 120               // Zeit in Sekunden bis zur Rcksetzung in den USB-Monitormodus
#define BOOTTOUT 5                // Zeit in Sekunden bis zur Rcksetzung des Bootloader-Zhlers
#define KEYLONG 25                // Zeit bis zum Erkennen eines langen Tastendrucks in Einheiten von 40 ms
#define KEYARPT 4                 // Wiederholungszeit fr Auto-Repeat in Einheiten von 40 ms (muss immer kleiner sein als KEYLONG)
#define SCROLLT 12                // Scroll-Text-Zeit in Einheiten von 20ms

// EEPROM-Adressen (werden als Konstanten definiert, um eine feste Adresse zu erhalten)

const uint16_t e_lang =     0;    // feste EEPROM-Adresse 000 - Lnge 001 Byte - Sprache (siehe lang)
const uint16_t e_brmode =   1;    // feste EEPROM-Adresse 001 - Lnge 001 Byte - Helligkeitsmodus (siehe brmode)
const uint16_t e_brithr =   2;    // feste EEPROM-Adresse 002 - Lnge 001 Byte - Helligkeitsschwelle (siehe brithr)
const uint16_t e_dcfsyn =   3;    // feste EEPROM-Adresse 003 - Lnge 001 Byte - Feste DCF-Synchronisierungzeit (siehe dcfsyn)
const uint16_t e_dcfhou =   4;    // feste EEPROM-Adresse 004 - Lnge 001 Byte - DCF-Synchronisierungsstunde (siehe dcfhou)
const uint16_t e_dcfinv =   5;    // feste EEPROM-Adresse 005 - Lnge 001 Byte - DCF-Invertierung (siehe dcfinv)
const uint16_t e_dcfpup =   6;    // feste EEPROM-Adresse 006 - Lnge 001 Byte - DCF-Pull-Up (siehe dcfpup)
const uint16_t e_autost =   7;    // feste EEPROM-Adresse 007 - Lnge 001 Byte - Automatische Sommerzeit (siehe autosumt)
const uint16_t e_timezo =   8;    // feste EEPROM-Adresse 008 - Lnge 001 Byte - Zeitzone (siehe timezone)
const uint16_t e_sensn1 =   9;    // feste EEPROM-Adresse 009 - Lnge 001 Byte - Sensornummer 1 (siehe sensorn1)
const uint16_t e_sensn2 =  10;    // feste EEPROM-Adresse 010 - Lnge 001 Byte - Sensornummer 2 (siehe sensorn2)
const uint16_t e_senst1 =  11;    // feste EEPROM-Adresse 011 - Lnge 002 Byte - Sensortext 1 (siehe sensort1)
const uint16_t e_senst2 =  13;    // feste EEPROM-Adresse 013 - Lnge 002 Byte - Sensortext 2 (siehe sensort2)
const uint16_t e_senso1 =  15;    // feste EEPROM-Adresse 015 - Lnge 001 Byte - Sensoroffset 1 (siehe sensoro1)
const uint16_t e_senso2 =  16;    // feste EEPROM-Adresse 016 - Lnge 001 Byte - Sensoroffset 2 (siehe sensoro2)
const uint16_t e_sensac =  17;    // feste EEPROM-Adresse 017 - Lnge 001 Byte - Automatische Sensor-Konfiguration (siehe sensacfg)
const uint16_t e_dateor =  18;    // feste EEPROM-Adresse 018 - Lnge 001 Byte - Datum-Reihenfolge (siehe dateordr)
const uint16_t e_datemo =  19;    // feste EEPROM-Adresse 019 - Lnge 001 Byte - Datum-Modus (siehe datemode)
const uint16_t e_senmo1 =  20;    // feste EEPROM-Adresse 020 - Lnge 001 Byte - Sensor-1-Modus (siehe senmode1)
const uint16_t e_senmo2 =  21;    // feste EEPROM-Adresse 021 - Lnge 001 Byte - Sensor-2-Modus (siehe senmode2)
const uint16_t e_inflag =  22;    // feste EEPROM-Adresse 022 - Lnge 001 Byte - Initialisierungs-Flag (siehe initflag)

// Variablen

static uint8_t usbbuff[256];      // USB-Empfangspuffer fr 256 Zeichen
static uint8_t rxbuff[256];       // RS-232-Empfangspuffer fr 256 Zeichen
static uint8_t usbwrite;          // Schreib-Zeiger fr USB-Empfangspuffer
static uint8_t usbread;           // Lese-Zeiger fr USB-Empfangspuffer
static uint8_t rxwrite;           // Schreib-Zeiger fr RS-232-Empfangspuffer
static uint8_t rxread;            // Lese-Zeiger fr RS-232-Empfangspuffer
static char usbdata[3];           // USB-Datenpuffer
static uint8_t usbpos;            // USB-Datenpuffer-Position
static char rxdata[8];            // RS-232-Datenpuffer
static uint8_t rxpos;             // RS-232-Datenpuffer-Position
static uint8_t usbmode;           // USB-Modus (0-1), 0 = USB im Monitormodus (Anzeige der Sensordaten), 1 = USB im Konfigurationsmodus (Men-gesteuerte Konfiguration)
static uint8_t usbtout;           // USB-Timeout-Zhler, wird beim Wechsel auf USB-Konfigurationsmodus auf USBTOUT gesetzt und im Sekundentakt auf 0 gezhlt
static char digits[6];            // LED-Zeichenspeicher fr die Anzeigestellen 0-5 (ASCII)
static uint8_t dots[7];           // LED-Dezimalpunktspeicher fr die Anzeigenstellen 0-5 und den Doppelpunkt (0 = aus, 1 = ein)
static uint8_t segbuf[8];         // LED-Segment-Puffer fr 7-Segment-Anzeigen (0-3) und 16-Segment-Anzeigen (4-7)
static uint8_t lang;              // Sprache (0-1), 0 = Deutsch, 1 = Englisch
static uint8_t intcnt;            // Interrupt-Zhler (0-49), reduziert die Frequenz von Timer3: 50Hz -> 1Hz (20ms -> 1s)
static uint8_t t1sold;            // Taster 1, Status vom vorherigen Interrupt (0-255)
static uint8_t t1stat;            // Taster 1, Status (0-1), 0 = Losgelassen, 1 = gedrckt
static uint8_t t1quit;            // Taster 1, Quittierungsstatus (0-1), 1 = Taster-Bettigung wurde quittiert
static uint8_t t1long;            // Taster 1, Zhler fr langen Tastendruck (0-c_keyl1)
static uint8_t t2sold;            // Taster 2, Status vom vorherigen Interrupt (0-255)
static uint8_t t2stat;            // Taster 2, Status (0-1), 0 = Losgelassen, 1 = gedrckt
static uint8_t t2quit;            // Taster 2, Quittierungsstatus (0-1), 1 = Taster-Bettigung wurde quittiert
static uint8_t t2long;            // Taster 2, Zhler fr langen Tastendruck (0-c_keyl1)
static uint8_t dcfsyn;            // Feste DCF77-Synchronisierungszeit (0-1), 0 = aus, 1 = feste Zeit mit Abschaltung des Displays zur sicheren DCF-Synchronisierung
static uint8_t dcfhou;            // DCF77-Synchronisierungsstunde (0-23)
static uint8_t dcfinv;            // DCF77-Invertierung (0-1), 0 = DCF-Empfnger liefert Low-Impulse, 1 = DCF-Empfnger liefert High-Impulse
static uint8_t dcfpup;            // DCF77-Pull-up (0-1), 0 = Pull-up ausgeschaltet, 1 = Pull-up eingeschaltet
static uint8_t dcflstat;          // DCF77-Signal-Status beim vorherigen Interrupt (0-1)
static uint8_t dcfflag;           // DCF77-Flag, Signalisierung des Minutenwechsels (0-1)
static uint8_t dcfcnt;            // DCF77-Zhler zur Ermittlung der Impulslnge
static uint8_t dcfimp;            // DCF77-Zhler zur Ermittlung der Impulsanzahl
static uint8_t dcftab[42];        // DCF77-Tabelle, enthlt empfangene Bits, wobei jedes Bit ein Byte der Tabelle belegt. Eingehende Bits werden in das letzte Byte eingeschoben und alle weiteren
                                  // Bytes um eine Position weitergerckt; die Bytes enthalten nach einem kompletten Minutenzyklus folgende Daten:
                                  //   Byte  0:    Zeitzone, 0 = Winterzeit, 1 = Sommerzeit
                                  //   Byte  1:    Zeitzone, 1 = Winterzeit, 0 = Sommerzeit
                                  //   Byte  2:    Schaltsekunde (nicht ausgewertet)
                                  //   Byte  3:    Startbit (nicht ausgewertet)
                                  //   Byte  4- 7: Minute Einer
                                  //   Byte  8-10: Minute Zehner
                                  //   Byte 11:    Prfbit 1 (Minuten)
                                  //   Byte 12-15: Stunde Einer
                                  //   Byte 16-17: Stunde Zehner
                                  //   Byte 18:    Prfbit 2 (Stunden)
                                  //   Byte 19-22: Kalendertag Einer
                                  //   Byte 23-24: Kalendertag Zehner
                                  //   Byte 25-27: Wochentag
                                  //   Byte 28-31: Monat Einer
                                  //   Byte 32:    Monat Zehner
                                  //   Byte 33-36: Jahr Einer
                                  //   Byte 37-40: Jahr Zehner
                                  //   Byte 41:    Prfbit 3 (Datum)
static uint8_t dcfbuf1[7];        // DCF-Puffer 1, enthlt die aktuelle dekodierte Uhrzeit:
                                  //   Byte 0: Zeitzone (0-1), 0 = MEZ, 1 = MESZ
                                  //   Byte 1: Minute (0-59)
                                  //   Byte 2: Stunde (0-23)
                                  //   Byte 3: Kalendertag (1-31)
                                  //   Byte 4: Wochentag (1-7)
                                  //   Byte 5: Monat (1-12)
                                  //   Byte 6: Jahr (0-99)
static uint8_t dcfbuf2[7];        // DCF-Puffer 2, enthlt die dekodierte Uhrzeit eine Minute zuvor (korrigiert um eine Minute), gleicher Aufbau wie dcbuf1
static uint8_t dcfbuf3[7];        // DCF-Puffer 3, enthlt die dekodierte Uhrzeit eine Minute zuvor (korrigiert um eine Minute), gleicher Aufbau wie dcbuf1
static uint8_t syncstat;          // Status der DCF77-Synchronsierung (0-4):
                                  //   0 = Uhr wurde noch nie synchronisert
                                  //   1 = Uhr wurde manuell gestellt
                                  //   2 = Uhr wurde mindestens 24h nicht synchronisiert
                                  //   3 = Uhr wurde innerhalb 24h synchronisiert
static uint8_t autosumt;          // Sommerzeit-Automatik (0-1), 0 = aus, 1 = ein
static uint8_t timezone;          // Zeitzone (0-1), 0 = MEZ (Normalzeit), 1 = MESZ (Sommerzeit)
static uint8_t second;            // Sekunden-Zhler (0-59)
static uint8_t minute;            // Minuten-Zhler (0-59)
static uint8_t hour;              // Stunden-Zhler (0-23)
static uint8_t day;               // Tages-Zhler (1-31)
static uint8_t wday;              // Wochentags-Zhler (1-7)
static uint8_t month;             // Monatszhler (1-12)
static uint8_t year;              // Jahres-Zhler (0-99)
static uint8_t xminute;           // Minutenwert fr Zeiteinstellung (0-59)
static uint8_t xhour;             // Stundenwert fr Zeiteinstellung (0-23)
static uint8_t xday;              // Tageswert fr Zeiteinstellung (1-31)
static uint8_t xmonth;            // Monatswert fr Zeiteinstellung (1-12)
static uint8_t xyear;             // Jahreswert fr Zeiteinstellung (0-99)
static uint8_t timeflag;          // Zeit-nderungs-Flag, wird bei der manuellen Zeiteingabe bei der nderung eines Wertes gesetzt
static uint8_t t10sflag;          // 10s-Flag (0-1), wird alle 10s von Interrupt gesetzt und vom Hauptprogramm nach Bearbeitung gelscht
static uint8_t dispmode;          // Anzeigemodus (0-n)
                                  //   0: Normale Zeitanzeige mit Stunden, Minuten, Sekunden
                                  //   1: Datumsanzeige mit Tag, Monat, Wochentag
                                  //   2: Sensoranzeige 1 mit Wert und Einheit von Sensor 1
                                  //   3: Sensoranzeige 2 mit Wert und Einheit von Sensor 2
                                  //   4: Anzeige der Versionsnummer
static uint8_t dispactv;          // Anzeige ist aktiv (1) oder ausgeschaltet (0); wird vom Hauptprogramm gesetzt, die Ausgabe ber Interrupt erledigt den Rest
static uint8_t confmode;          // Konfigurationsmodus (0-25), 0 = aus, 1...25 = aktueller Konfigurationsmodus:
                                  //   1: Sprache
                                  //   2: Helligkeits-Modus
                                  //   3: Helligkeits-Schwelle
                                  //   4: Feste DCF77-Synchronisierungszeit
                                  //   5: DCF77-Synchronisierungsstunde
                                  //   6: DCF77-Invertierung
                                  //   7: DCF77-Pull-Up
                                  //   8: Sensor-Nummer 1
                                  //   9: Sensor-Nummer 2
                                  //  10: Sensor-Text 1
                                  //  11: Sensor-Text 2
                                  //  12: Sensor-Offset 1
                                  //  13: Sensor-Offset 2
                                  //  14: Automatische Sensor-Konfiguration
                                  //  15: Datum-Reihenfolge
                                  //  16: Datum anzeigen
                                  //  17: Sensor 1 anzeigen
                                  //  18: Sensor 2 anzeigen
                                  //  19: Automatische Sommerzeit
                                  //  20: Tag einstellen
                                  //  21: Monat einstellen
                                  //  22: Jahr einstellen
                                  //  23: Stunde einstellen
                                  //  24: Minute einstellen
                                  //  25: Zeit bernehmen
static uint8_t confsubm;          // Konfigurations-Submodus (0-1), wird bei confmode 10 und 11 verwendet, um die Zeichenpositionen fr den Sensortext zu selektieren
static uint8_t conftout;          // Konfigurations-Timeout-Zhler, wird beim Aufruf des Konfigurationsmens auf CONFTOUT gesetzt und im Sekundentakt auf 0 gezhlt, anschlieend wird der Zhler
                                  // gestoppt und wieder zur normalen Zeitanzeige geschaltet
static uint8_t count1;            // freilaufender Zhler 1 fr Timeouts (Vorteiler)
static uint8_t count2;            // freilaufender Zhler 2 fr Timeouts (Sekunden)
static uint8_t count3;            // freilaufender Zhler 3 fr Timeouts (Minuten)
static uint8_t vertout;           // Versions-Timeout-Zhler, wird beim Systemstart auf VERTOUT gesetzt und im Sekundentakt auf 0 gezhlt
static uint8_t dcftout;           // DCF77-Timeout-Zhler, wird bei fester DCF77-Synchronisierungszeit (dcfsyn = 1) nach dem Systemstart und zur Synchronisierungsstunde gestartet; whrend der Lauf-
                                  // zeit wird das Display ausgeschaltet; nach erfolgreicher Synchronisierung oder nach Ablauf der Zeit wird der Zhler gestoppt und das Display wieder eingeschaltet
static uint8_t synctout;          // Synchronisierungs-Timeout-Zhler, nach Ablauf der Zeit wird der Synchronisierungs-Status auf 2 gesetzt; wird vom Hauptprogramm gesetzt und vom Timer-Interrupt im
                                  // Stundentakt auf 0 gezhlt, das Hauptprogramm setzt nach Bearbeitung den Wert auf 255 (Stopp)
static uint8_t bribuf[128];       // Puffer fr die letzten 128 vom ADC gelesenen Helligkeitswerte
static uint8_t bripos;            // aktuelle Schreibposition des Helligkeitspuffers
static uint8_t bright;            // berechneter Helligkeitsmittelwert
static uint8_t brmode;            // Helligkeitsmodus (0-1), 0 = Direkt (stndige nderung), 1 = Verzgert (nderung einmal pro Sekunde)
static uint8_t brithr;            // Helligkeitsschwelle (0-10); eine Helligkeit ab diesem Tabellenwert gilt als Tag, eine Helligkeit unter diesem Wert gilt als Nacht
static uint8_t saveflag;          // Speicher-Flag (0-1); wird gesetzt, wenn Daten im EEPROM gespeichert werden mssen
static uint8_t initflag;          // Initialisierungs-Flag; wird bei jedem Systemstart zusammen mit sensoro1 und sensoro2 geprft, dient zur Korrektur der Sensor-Offsets beim ersten Systemstart
static uint8_t sensorn1;          // Sensornummer 1 (0-31): 0 = aus, 1-31 = Sensornummer vom Temperatur-Messsystem
static uint8_t sensorn2;          // Sensornummer 2 (0-31): 0 = aus, 1-31 = Sensornummer vom Temperatur-Messsystem
static char sensort1[2];          // Sensortext 1 (2 ASCII-Zeichen)
static char sensort2[2];          // Sensortext 2 (2 ASCII-Zeichen)
static int8_t sensoro1;           // Sensoroffset 1, Korrekturwert fr Sensor 1 (-30 bis 30)
static int8_t sensoro2;           // Sensoroffset 2, Korrekturwert fr Sensor 2 (-30 bis 30)
static uint8_t sensacfg;          // automatische Sensor-Konfiguration (0-1), 0 = aus, 1 = ein
static char senval1[4];           // Sensorwert 1 vierstellig im ASCII-Format
static char senval2[4];           // Sensorwert 2 vierstellig im ASCII-Format
static char senbuf1[4];           // Sensorpuffer 1 vierstellig im ASCII-Format, Zwischenspeicher
static char senbuf2[4];           // Sensorpuffer 2 vierstellig im ASCII-Format, Zwischenspeicher
static uint8_t senmemf1;          // Zwischenspeicher-Flag 1; wird gesetzt, wenn Sensor 1 gerade angezeigt wird, die bertragung in den Sensorwert erfolgt dann spter
static uint8_t senmemf2;          // Zwischenspeicher-Flag 2; wird gesetzt, wenn Sensor 2 gerade angezeigt wird, die bertragung in den Sensorwert erfolgt dann spter
static uint8_t sendot1;           // Position des Dezimalpunktes im Sensorwert 1 (0-4), 4 = Dezimalpunkt aus
static uint8_t sendot2;           // Position des Dezimalpunktes im Sensorwert 2 (0-4), 4 = Dezimalpunkt aus
static uint8_t sentout1;          // Sensor-1-Timeout-Zhler, wird beim Empfang eines Wertes auf SENTOUT gesetzt und im Sekundentakt auf 0 gezhlt
static uint8_t sentout2;          // Sensor-2-Timeout-Zhler, wird beim Empfang eines Wertes auf SENTOUT gesetzt und im Sekundentakt auf 0 gezhlt
static uint8_t ascbuff[4];        // ASCII-Puffer fr die Konvertierung Integer <-> ASCII
static uint8_t dateordr;          // Datum-Reihenfolge (0-1): 0 = Tag/Monat, 1 = Monat/Tag
static uint8_t datemode;          // Datum-Modus (0-2): 0 = Datum immer anzeigen, 1 = Datum nur am Tag anzeigen, 2 = Datum nie anzeigen
static uint8_t senmode1;          // Sensor-1-Modus (0-1): 0 = Sensor 1 immer anzeigen, 1 = Sensor 1 nur am Tag anzeigen
static uint8_t senmode2;          // Sensor-2-Modus (0-1): 0 = Sensor 2 immer anzeigen, 1 = Sensor 2 nur am Tag anzeigen
static uint8_t dateflag;          // Datum-Flag (0-1): 0 = Datum nicht anzeigen, 1 = Datum anzeigen; wird zu Beginn jedes 10s-Intervalls fr die Anzeigesteuerung neu gesetzt
static uint8_t senflag1;          // Sensor-1-Flag (0-1): 0 = Sensorwert 1 nicht anzeigen, 1 = Sensorwert 1 anzeigen; wird zu Beginn jedes 10s-Intervalls fr die Anzeigesteuerung neu gesetzt
static uint8_t senflag2;          // Sensor-2-Flag (0-1): 0 = Sensorwert 2 nicht anzeigen, 1 = Sensorwert 2 anzeigen; wird zu Beginn jedes 10s-Intervalls fr die Anzeigesteuerung neu gesetzt
static uint8_t dayflag;           // Tag-Flag (0-1): 0 = Nachtmodus, 1 = Tagmodus; fr Anzeigesteuerung von Datum und Sensorwerten
static uint8_t bootlcnt;          // Bootloader-Zeichen-Zhler (0-2), wird bei jedem empfangenen 'U' erhht, bei 2 wird der Bootloader gestartet
static uint8_t boottout;          // Bootloader-Timeout-Zhler, wird beim USB-Empfang eines 'U' auf BOOTTOUT gesetzt und im Sekundentakt auf 0 gezhlt, anschlieend wird der Bootloader-Zeichen-Zhler
                                  // gelscht (nur mehrere kurz nacheinander empfangene 'U' lsen den Bootloader-Start aus)
static uint8_t menufn;            // aktuelle Men-Funktion (0, "a"-"x")
static uint8_t scrtxt;            // auszugebender Scrolltext, 0 = aus oder String-Nummer
static uint8_t scrcnt;            // Scrollzhler, zhlt alle 20ms von 0 bis zum Wert SCROLLT, anschlieend wird die Zeichenposition erhht
static uint8_t scrpos;            // Scrollposition, zeigt auf die aktuelle Zeichenposition im auszugebenden Scrolltext, besondere Positionen:
                                  //    0: String-Anfang 1, Ausgabe von 2 Leerzeichen
                                  //    1: String-Anfang 2, Ausgabe eines Leerzeichens und des ersten Zeichens
                                  //  100: String-Ende 1, Ausgabe des letzten Zeichens und eines Leerzeichens
                                  //  101: String-Ende 2, Ausgabe von 2 Leerzeichen
static char scrbuf[2];            // Scrollpuffer, enthlt die beiden auszugebenden Zeichen des Scrolltextes

#include "tables.h"               // Segmenttabellen fr die Anzeigen und weitere Tabellen einfgen
#include "strings.h"              // verschiedene Texte einfgen
#include "interrupt.h"            // Interrupt-Routinen einfgen
#include "functions.h"            // verschiedene Funktionen einfgen

// Hauptprogramm - Initialisierung --------------------------------------------------------------------------------------------------------------------------------------------------------------------

int main(void)
{
  int8_t i;                                                           // universelle Zhlervariable
  uint8_t x;                                                          // universelle 8-Bit-Variable
  uint8_t e;                                                          // Fehlerzhler
  char c;                                                             // universelle Zeichenvariable
  int8_t m;                                                           // universelle 8-Bit-Variable fr Eingaben
  int16_t n;                                                          // universelle 16-Bit-Variable fr Sensorwerte
  PGM_P p;                                                            // universeller Adress-Zeiger auf Strings
  
  DDRA = 0b11111111;                                                  // PortA alles Ausgnge
  DDRB = 0b11111101;                                                  // PortB Pins 0 und 2-7 als Ausgang
  DDRC = 0b11111111;                                                  // PortC alles Ausgnge
  DDRD = 0b11111011;                                                  // PortD Pins 0-1 und 3-7 als Ausgang
  DDRE = 0b11111100;                                                  // PortE Pins 2-7 als Ausgang
  DDRF = 0b11000000;                                                  // PortF Pins 6-7 als Ausgang
  DDRG = 0b00000111;                                                  // PortG Pins 0-2 als Ausgang
  PORTA = 0b11111111;                                                 // PortA alle Ausgnge auf High
  PORTB = 0b10001011;                                                 // PortB Ausgnge 0, 3 und 7 auf High, Eingange 1 mit Pull-up
  PORTC = 0b11111111;                                                 // PortC alle Ausgnge auf High
  PORTD = 0b11111111;                                                 // PortD Ausgnge 0-1 und 3-7 auf High, Eingang 2 mit Pull-up
  PORTE = 0b11111100;                                                 // PortE Ausgnge 2-7 auf High
  PORTF = 0b11110110;                                                 // PortF Ausgnge 6-7 auf High, Eingnge 1-2 und 4-5 mit Pull-up
  PORTG = 0b00000111;                                                 // PortG Ausgnge 0-2 auf High

  UCSR0A = 0;                                                         // USART0 Status-Register zurcksetzen
  UCSR0B = _BV(RXCIE0) | _BV(RXEN0) | _BV(TXEN0);                     // USART0 Empfnger mit Interrupt und Sender aktivieren
  UCSR0C = _BV(UCSZ01) | _BV(UCSZ00);                                 // USART0 asynchron, 8-Bit, keine Paritt, 1 Stopp-Bit
  UBRR0H = (F_CPU / 16 / USBBAUD - 1) >> 8;                           // USART0 Baudrate (High) setzen
  UBRR0L = F_CPU / 16 / USBBAUD - 1;                                  // USART0 Baudrate (Low) setzen
  UCSR1A = 0;                                                         // USART1 Status-Register zurcksetzen
  UCSR1B = _BV(RXCIE1) | _BV(RXEN1);                                  // USART1 RXD mit Interrupt aktivieren
  UCSR1C = _BV(UCSZ11) | _BV(UCSZ10);                                 // USART1 asynchron, 8-Bit, keine Paritt, 1 Stopp-Bit
  UBRR1H = (F_CPU / 16 / RSBAUD - 1) >> 8;                            // USART1 Baudrate (High) setzen
  UBRR1L = F_CPU / 16 / RSBAUD - 1;                                   // USART1 Baudrate (Low) setzen

  TCCR0 = _BV(CS02);                                                  // Timer0 wird fr die Helligkeitssteuerung verwendet, Mode 0 (Normal), Vorteiler = 64
  TCCR1A = 0;                                                         // Timer1 wird fr die Ausgabe der Segmentbelegung an die Anzeige vewendet
  TCCR1B = _BV(WGM12) | _BV(CS11) | _BV(CS10);                        // Timer1 Mode 4 (CTC/OCR1A), Vorteiler = 64
  TCCR1C = 0;
  OCR1A = F_CPU / 64 / TIM1_CLK - 1;                                  // Timer1 Teilerwert setzen, Frequenz = 1024Hz
  TIMSK = _BV(OCIE1A);                                                // Timer1 Interrupt fr Output Compare 1A Match aktivieren
  TCCR3A = 0;                                                         // Timer3 wird fr die Zeitzhlung, Tasterabfrage und DCF77 verwendet
  TCCR3B = _BV(WGM32) | _BV(CS31) | _BV(CS30);                        // Timer3 Mode 4 (CTC/OCR3A), Vorteiler = 64
  TCCR3C = 0;
  OCR3A = F_CPU / 64 / TIM3_CLK - 1;                                  // Timer3 Teilerwert setzen, Frequenz = 50Hz
  ETIMSK = _BV(OCIE3A);                                               // Timer3 Interrupt fr Output Compare 3A Match aktivieren

  ADMUX = _BV(REFS1) | _BV(REFS0) | _BV(ADLAR);                       // ADC mit interner Referenz und Kondensator an AREF, Multiplexer auf Lichtsensor setzen
  ADCSRA = _BV(ADEN) | _BV(ADSC) | _BV(ADFR) |
    _BV(ADPS2) | _BV(ADPS1) | _BV(ADPS0);                             // ADC aktivieren und starten, freilaufend, Vorteiler = 128

  PORTB &= ~_BV(PINB7);                                               // Reset am Sensor-Controller auslsen (Low)
  _delay_us(10);                                                      // 10s warten
  PORTB |= _BV(PINB7);                                                // Reset wieder auf High setzen

  lang = eep_read_byte(e_lang);                                       // Sprache aus EEPROM lesen
  if (lang > 1) lang = 0;                                             // wenn Bereich berschritten -> Deutsch einstellen
  brmode = eep_read_byte(e_brmode);                                   // Helligkeitsmodus aus EEPROM lesen
  if (brmode > 1) brmode = 0;                                         // wenn Bereich berschritten -> auf Normal setzen
  dcfsyn = eep_read_byte(e_dcfsyn);                                   // feste DCF77-Synchronisierungszeit aus EEPROM lesen
  if (dcfsyn > 1) dcfsyn = 1;                                         // wenn Bereich berschritten -> einschalten
  dcfhou = eep_read_byte(e_dcfhou);                                   // DCF77-Synchronisierungsstunde aus EEPROM lesen
  if (dcfhou > 23) dcfhou = DCFHOU;                                   // wenn Bereich berschritten -> DCF-Synchronisierungstunde auf Voreinstellung setzen
  dcfinv = eep_read_byte(e_dcfinv);                                   // DCF77-Invertierung aus EEPROM lesen
  if (dcfinv > 1) dcfinv = 1;                                         // wenn Bereich berschritten -> einschalten (Grundeinstellung fr Pollin-Empfnger)
  dcfpup = eep_read_byte(e_dcfpup);                                   // DCF77-Pull-up aus EEPROM lesen
  if (dcfpup > 1) dcfpup = 0;                                         // wenn Bereich berschritten -> ausschalten (Grundeinstellung fr Pollin-Empfnger)
  if (dcfpup) PORTF |= _BV(PINF3);                                    // wenn Pull-up aktiviert -> beim DCF77-Eingang Pull-up einschalten
  autosumt = eep_read_byte(e_autost);                                 // automatische Sommerzeit aus EEPROM lesen
  if (autosumt > 1) autosumt = 1;                                     // wenn Bereich berschritten -> einschalten
  timezone = eep_read_byte(e_timezo);                                 // Zeitzone aus EEPROM lesen
  if (timezone > 1) timezone = 0;                                     // wenn Bereich berschritten -> Normalzeit einstellen
  sensorn1 = eep_read_byte(e_sensn1);                                 // Sensornummer 1 aus EEPROM lesen
  if (sensorn1 > 31) sensorn1 = 0;                                    // wenn Bereich berschritten -> Sensor 1 ausschalten
  sensorn2 = eep_read_byte(e_sensn2);                                 // Sensornummer 2 aus EEPROM lesen
  if (sensorn2 > 31) sensorn2 = 0;                                    // wenn Bereich berschritten -> Sensor 2 ausschalten
  sensort1[0] = eep_read_byte(e_senst1);                              // Sensortext 1, Zeichen 1 aus EEPROM lesen
  if ((sensort1[0] < ' ') || (sensort1[0] > 127)) sensort1[0] = ' ';  // wenn Bereich berschritten -> Leerzeichen setzen
  sensort1[1] = eep_read_byte(e_senst1 + 1);                          // Sensortext 1, Zeichen 2 aus EEPROM lesen
  if ((sensort1[1] < ' ') || (sensort1[1] > 127)) sensort1[1] = ' ';  // wenn Bereich berschritten -> Leerzeichen setzen
  sensort2[0] = eep_read_byte(e_senst2);                              // Sensortext 1, Zeichen 1 aus EEPROM lesen
  if ((sensort2[0] < ' ') || (sensort2[0] > 127)) sensort2[0] = ' ';  // wenn Bereich berschritten -> Leerzeichen setzen
  sensort2[1] = eep_read_byte(e_senst2 + 1);                          // Sensortext 1, Zeichen 2 aus EEPROM lesen
  if ((sensort2[1] < ' ') || (sensort2[1] > 127)) sensort2[1] = ' ';  // wenn Bereich berschritten -> Leerzeichen setzen
  sensoro1 = eep_read_byte(e_senso1);                                 // Sensoroffset 1 aus EEPROM lesen
  if ((sensoro1 < -30) || (sensoro1 > 30)) sensoro1 = 0;              // wenn Bereich berschritten -> auf 0 setzen
  sensoro2 = eep_read_byte(e_senso2);                                 // Sensoroffset 2 aus EEPROM lesen
  if ((sensoro2 < -30) || (sensoro2 > 30)) sensoro2 = 0;              // wenn Bereich berschritten -> auf 0 setzen
  sensacfg = eep_read_byte(e_sensac);                                 // automatische Sensor-Konfiguration aus EEPROM lesen
  if (sensacfg > 1) sensacfg = 1;                                     // wenn Bereich berschritten -> einschalten
  dateordr = eep_read_byte(e_dateor);                                 // Datum-Reihenfolge aus EEPROM lesen
  if (dateordr > 2) dateordr = 0;                                     // wenn Bereich berschritten -> auf Tag/Monat setzen
  datemode = eep_read_byte(e_datemo);                                 // Datum-Modus aus EEPROM lesen
  if (datemode > 2) datemode = 0;                                     // wenn Bereich berschritten -> auf 0 setzen
  senmode1 = eep_read_byte(e_senmo1);                                 // Sensor-1-Modus aus EEPROM lesen
  if (senmode1 > 1) senmode1 = 0;                                     // wenn Bereich berschritten -> auf 0 setzen
  senmode2 = eep_read_byte(e_senmo2);                                 // Sensor-2-Modus aus EEPROM lesen
  if (senmode2 > 1) senmode2 = 0;                                     // wenn Bereich berschritten -> auf 0 setzen
  brithr = eep_read_byte(e_brithr);                                   // Helligkeitsschwelle aus EEPROM lesen
  if (brithr > 10) brithr = 5;                                        // wenn Bereich berschritten -> auf 5 setzen
  initflag = eep_read_byte(e_inflag);                                 // Initialisierungs-Flag aus EEPROM lesen
  if (initflag == 0xff) {                                             // wenn Initialisierung-Flag gesetzt (erster Systemstart)
    initflag = 0;                                                     // Initialisierungs-Flag lschen
    saveflag = 1;                                                     // genderte Konfiguration speichern
    if ((sensoro1 == -1) && (sensoro2 == -1)) {                       // wenn beide Sensoroffsets = -1
      sensoro1 = 0;                                                   // Sensoroffset 1 auf 0.0 setzen
      sensoro2 = 0;                                                   // Sensoroffset 2 auf 0.0 setzen
    }
  }

  for (i = 0; i < 6; i ++) digits[i] = ' ';                           // Leerzeichen auf allen Anzeigestellen ausgeben
  asc_to_seg();                                                       // ASCII in Segment-Daten konvertieren
  day = 0;                                                            // Tag voreinstellen
  month = 1;                                                          // Monat voreinstellen
  year = YEAR;                                                        // Jahr voreinstellen
  wday = calc_wday(day, month, year);                                 // Wochentag berechnen
  timezone = chk_summtime();                                          // Zeitzone voreinstellen
  hour = 23;                                                          // Stunde voreinstellen
  minute = 59;                                                        // Minute voreinstellen
  second = 60 - VERTOUT;                                              // Sekunde voreinstellen (Anzeige der Versionsnummer bercksichtigen)
  vertout = VERTOUT;                                                  // Versions-Timeout-Zhler setzen
  dispmode = 4;                                                       // Anzeigemodus auf Versionsnummer setzen
  for (i = 0; i < 4; i ++) {                                          // Sensorwerte lschen
    senval1[i] = ' ';                                                 // Sensorwert 1 mit Leerzeichen berschreiben
    senval2[i] = ' ';                                                 // Sensorwert 2 mit Leerzeichen berschreiben
  }
  if (sensacfg) {                                                     // wenn automatische Sensor-Konfiguration eingeschaltet ist
    sensorn1 = 0;                                                     // Sensornummer 1 lschen
    sensorn2 = 0;                                                     // Sensornummer 2 lschen
  }
  else {                                                              // wenn automatische Sensor-Konfiguration ausgeschaltet ist
    switch (sensorn1) {                                               // Sensornummer 1 auswerten und verzweigen
      case 0:         sendot1 = 4; break;                             // wenn kein Sensor -> Dezimalpunkt ausschalten
      case 1 ... 27:  sendot1 = 2; break;                             // wenn Temperatursensor -> Dezimalpunkt an Minutenzehner einschalten (eine Nachkommastelle)
      case 28 ... 31: sendot1 = 3; break;                             // wenn Luftdruck- oder Luftfeuchtigkeitssensor -> Dezimalpunkt an Minuteneiner einschalten (keine Nachkommastelle)
    }
    switch (sensorn2) {                                               // Sensornummer 2 auswerten und verzweigen
      case 0:         sendot2 = 4; break;                             // wenn kein Sensor -> Dezimalpunkt ausschalten
      case 1 ... 27:  sendot2 = 2; break;                             // wenn Temperatursensor -> Dezimalpunkt an Minutenzehner einschalten (eine Nachkommastelle)
      case 28 ... 31: sendot2 = 3; break;                             // wenn Luftdruck- oder Luftfeuchtigkeitssensor -> Dezimalpunkt an Minuteneiner einschalten (keine Nachkommastelle)
    }
  }
  sei();                                                              // Interrupts aktivieren

// Hauptprogrammschleife ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

  while(1) {
                                                                      // --- Steuerung der Anzeigeautomatik ---
    if (t10sflag) {                                                   // wenn neues 10s-Intervall beginnt
      t10sflag = 0;                                                   // 10s-Flag wieder lschen
      dateflag = 0;                                                   // Datum-Flag zunchst lschen
      senflag1 = 0;                                                   // Sensor-1-Flag zunchst lschen
      senflag2 = 0;                                                   // Sensor-2-Flag zunchst lschen
      if (syncstat) {                                                 // wenn die Uhr eine gltige Uhrzeit hat
        if (datemode == 0) dateflag = 1;                              // wenn Datum stndig angezeigt werden soll -> Datum-Flag setzen
        if ((datemode == 1) && dayflag) dateflag = 1;                 // wenn Datum nur am Tag angezeigt werden soll und Tag-Modus -> Datum-Flag setzen
      }
      if ((sensorn1 > 0) && (sentout1 < 255)) {                       // wenn Sensor 1 vorhanden und Timeout-Zhler aktiv (Wert ist gltig)
        if (senmode1 == 0) senflag1 = 1;                              // wenn Sensor 1 stndig angezeigt werden soll -> Sensor-1-Flag setzen
        if ((senmode1 == 1) && dayflag) senflag1 = 1;                 // wenn Sensor 1 nur am Tag angezeigt werden soll und Tag-Modus -> Sensor-1-Flag setzen
      }
      if ((sensorn2 > 0) && (sentout2 < 255)) {                       // wenn Sensor 2 vorhanden und Timeout-Zhler aktiv (Wert ist gltig)
        if (senmode2 == 0) senflag2 = 1;                              // wenn Sensor 2 stndig angezeigt werden soll -> Sensor-2-Flag setzen
        if ((senmode2 == 1) && dayflag) senflag2 = 1;                 // wenn Sensor 2 nur am Tag angezeigt werden soll und Tag-Modus -> Sensor-2-Flag setzen
      }
    }
    if (dispmode < 4) {                                               // wenn normaler Anzeigemodus (0-3)
      i = second % 10;                                                // 10-Sekunden-Intervall erzeugen
      x = 0;                                                          // Tabellenindex = 0 setzen (kein Datum und keine Sensorwerte anzeigen)
      if (senflag1 && !senflag2) x = 10;                              // wenn Sensorwert 1 angezeigt werden soll -> Tabellenindex = 10 setzen
      if (!senflag1 && senflag2) x = 20;                              // wenn Sensorwert 2 angezeigt werden soll -> Tabellenindex = 20 setzen
      if (senflag1 && senflag2) x = 30;                               // wenn beide Sensorwerte angezeigt werden sollen -> Tabellenindex = 30 setzen
      if (dateflag) x+= 40;                                           // wenn das Datum angezeigt werden soll -> Tabellenindex um 40 erhhen
      dispmode = pgm_read_byte(&dispmatr[x + i]);                     // Tabellenindex um den Sekundenwert erhhen und Anzeigemodus aus Tabelle holen
    }
                                                                      // --- LED-Anzeige entsprechend Modus zunchst als ASCII-Daten vorbereiten und dann als Segmente ausgeben ---
    if (!confmode) {                                                  // wenn nicht im Konfigurationsmodus
      switch (dispmode) {                                             // Anzeigemodus auswerten und verzweigen
        case 0:                                                       // wenn Anzeigemodus 0 (normale Zeitanzeige)
          if (hour > 9) digits[0] = hour / 10 + '0';                  // wenn Stundenwert > 9 -> Stundenzehner ausgeben
          else digits[0] = ' ';                                       // sonst Leerzeichen ausgeben
          digits[1] = hour % 10 + '0';                                // Stundeneiner ausgeben
          digits[2] = minute / 10 + '0';                              // Minutenzehner ausgeben
          digits[3] = minute % 10 + '0';                              // Minuteneiner ausgeben
          digits[4] = second / 10 + '0';                              // Sekundenzehner ausgeben
          digits[5] = second % 10 + '0';                              // Sekundeneiner ausgeben
          dots[0] = 0;                                                // Dezimalpunkt an Stundenzehner ausschalten
          dots[1] = 0;                                                // Dezimalpunkt an Stundeneiner ausschalten
          dots[2] = 0;                                                // Dezimalpunkt an Minutenzehner ausschalten
          dots[4] = 0;                                                // Dezimalpunkt an Sekundenzehner ausschalten
          dots[5] = 0;                                                // Dezimalpunkt an Sekundeneiner ausschalten
          if (((syncstat == 0) && (intcnt < 25)) || (syncstat == 2)) { // wenn noch keine gltige Zeit vorhanden und Interruptzhler < 25 oder DCF77-Synchronisierung erforderlich
            dots[3] = 1;                                              // Dezimalpunkt an Minuteneiner einschalten
          }
          else dots[3] = 0;                                           // sonst Dezimalpunkt an Minuteneiner ausschalten
          if ((intcnt > 0) && (intcnt < 25)) dots[6] = 1;             // wenn Interruptzhler zwischen 1 und 25 (erste Sekundenhlfte) -> Doppelpunkt einschalten
          else dots[6] = 0;                                           // sonst Doppelpunkt ausschalten
          break;

        case 1:                                                       // wenn Anzeigemodus 1 (Datumsanzeige)
          if (dateordr) {                                             // wenn Datum-Reihenfolge Monat/Tag
            if (month > 9) digits[0] = month / 10 + '0';              // wenn Monatswert > 9 -> Monatszehner ausgeben
            else digits[0] = ' ';                                     // sonst Leerzeichen ausgeben
            digits[1] = month % 10 + '0';                             // Monatseiner ausgeben
            if (day > 9) digits[2] = day / 10 + '0';                  // wenn Tageswert > 9 -> Tageszehner ausgeben
            else digits[2] = ' ';                                     // sonst Leerzeichen ausgeben
            digits[3] = day % 10 + '0';                               // Tageseiner ausgeben
          }
          else {                                                      // wenn Datum-Reihenfolge Tag/Monat
            if (day > 9) digits[0] = day / 10 + '0';                  // wenn Tageswert > 9 -> Tageszehner ausgeben
            else digits[0] = ' ';                                     // sonst Leerzeichen ausgeben
            digits[1] = day % 10 + '0';                               // Tageseiner ausgeben
            if (month > 9) digits[2] = month / 10 + '0';              // wenn Monatswert > 9 -> Monatszehner ausgeben
            else digits[2] = ' ';                                     // sonst Leerzeichen ausgeben
            digits[3] = month % 10 + '0';                             // Monatseiner ausgeben
          }
          str_digit(wday, 4);                                         // Wochentag an Sekundenzehner und -Einer ausgeben
          dots[0] = 0;                                                // Dezimalpunkt an Tageszehner ausschalten
          dots[1] = 1;                                                // Dezimalpunkt an Tageseiner einschalten
          dots[2] = 0;                                                // Dezimalpunkt an Monatszehner ausschalten
          dots[3] = 1;                                                // Dezimalpunkt an Monatseiner einschalten
          dots[4] = 0;                                                // Dezimalpunkt an Sekundenzehner ausschalten
          dots[5] = 1;                                                // Dezimalpunkt an Sekundeneiner einschalten
          dots[6] = 0;                                                // Doppelpunkt ausschalten
          break;
        
        case 2:                                                       // wenn Anzeigemodus 2 (Sensoranzeige 1)
          for (i = 0; i < 4; i ++) digits[i] = senval1[i];            // Sensorwert 1 in den Anzeigespeicher kopieren
          digits[4] = sensort1[0];                                    // Sensortext 1, Zeichen 1 in Anzeigespeicher kopieren
          digits[5] = sensort1[1];                                    // Sensortext 1, Zeichen 2 in Anzeigespeicher kopieren
          for (i = 0; i < 4; i ++) {                                  // 4 Dezimalpunkte bearbeiten
            if (sendot1 == i) dots[i] = 1;                            // wenn richtige Position erreicht -> Dezimalpunkt setzen
            else dots[i] = 0;                                         // sonst Dezimalpunkt lschen
          }
          dots[4] = 0;                                                // Dezimalpunkt an Sekundenzehner ausschalten
          dots[5] = 0;                                                // Dezimalpunkt an Sekundeneiner ausschalten
          dots[6] = 0;                                                // Doppelpunkt ausschalten
          break;

        case 3:                                                       // wenn Anzeigemodus 3 (Sensoranzeige 2)
          for (i = 0; i < 4; i ++) digits[i] = senval2[i];            // Sensorwert 2 in den Anzeigespeicher kopieren
          digits[4] = sensort2[0];                                    // Sensortext 2, Zeichen 1 in Anzeigespeicher kopieren
          digits[5] = sensort2[1];                                    // Sensortext 2, Zeichen 2 in Anzeigespeicher kopieren
          for (i = 0; i < 4; i ++) {                                  // 4 Dezimalpunkte bearbeiten
            if (sendot2 == i) dots[i] = 1;                            // wenn richtige Position erreicht -> Dezimalpunkt setzen
            else dots[i] = 0;                                         // sonst Dezimalpunkt lschen
          }
          dots[4] = 0;                                                // Dezimalpunkt an Sekundenzehner ausschalten
          dots[5] = 0;                                                // Dezimalpunkt an Sekundeneiner ausschalten
          dots[6] = 0;                                                // Doppelpunkt ausschalten
          break;

        case 4:                                                       // wenn Anzeigemodus 4 (Versionsnummer)
          p = string(0);                                              // Versionsnummer-String holen
          for (i = 0; i < 6; i ++) digits[i] = pgm_read_byte(p ++);   // Versionsnummer in den Anzeigespeicher kopieren
          dots[0] = 0;                                                // Dezimalpunkt an Stundenzehner ausschalten
          dots[1] = 1;                                                // Dezimalpunkt an Stundeneiner einschalten
          dots[2] = 0;                                                // Dezimalpunkt an Minutenzehner ausschalten
          dots[3] = 0;                                                // Dezimalpunkt an Minuteneiner ausschalten
          dots[4] = 1;                                                // Dezimalpunkt an Sekundenzehner einsschalten
          dots[5] = 0;                                                // Dezimalpunkt an Sekundeneiner ausschalten
          break;
      }
    }
    else {                                                            // wenn im Konfigurationsmodus
      dots[0] = 0;                                                    // Dezimalpunkt an Stundenzehner ausschalten
      dots[2] = 0;                                                    // Dezimalpunkt an Minutenzehner ausschalten
      if (confmode < 25) {                                            // wenn Konfigurationsmodus < 25 (alle Einstellungen auer "Zeit bernehmen")
        dots[1] = 1;                                                  // Dezimalpunkt an Stundeneiner einschalten
        dots[6] = 0;                                                  // Doppelpunkt ausschalten
        if ((confmode == 10) || (confmode == 11)) {                   // wenn Konfigurationsmodus 10 oder 11 (Sensortext eingeben)
          if (!confsubm) dots[5] = 0;                                 // wenn Konfigurations-Submodus 0 -> nicht blinkende Stelle (Sekunden-Einer) ausschalten
          else dots[4] = 0;                                           // wenn Konfigurations-Submodus 1 -> nicht blinkende Stelle (Sekunden-Zehner) ausschalten
          dots[3] = 0;                                                // Dezimalpunkt an Minuteneiner ausschalten
        }
        else {                                                        // wenn Konfigurations auer 10 und 11
          dots[3] = 1;                                                // Dezimalpunkt an Minuteneiner einschalten
          dots[4] = 0;                                                // Dezimalpunkt an Sekundenzehner einschalten
          dots[5] = 0;                                                // Dezimalpunkt an Sekundeneiner einschalten
        }
        if (confmode < 20) {                                          // wenn Konfigurationsmodus < 20 (System-Einstellungen)
          digits[0] = confmode / 10 + '0';                            // aktuellen Konfigurationsmodus-Zehner ausgeben
          digits[1] = confmode % 10 + '0';                            // aktuellen Konfigurationsmodus-Einer ausgeben
        }
        else {                                                        // wenn Konfigurationsmodus > 19 (Zeit-Einstellungen)
          str_digit(67 + confmode - 20, 0);                           // Kurztext an Stunden-Zehner und -Einer ausgeben
        }
      }
      else {                                                          // wenn Konfigurationsmodus 25 (Zeit bernehmen)
        dots[1] = 0;                                                  // Dezimalpunkt am Stundeneiner einschalten
        dots[3] = 0;                                                  // Dezimalpunkt am Minuteneiner ausschalten
        dots[4] = 0;                                                  // Dezimalpunkt am Sekundenzehner ausschalten
        dots[5] = 0;                                                  // Dezimalpunkt am Sekundeneiner ausschalten
      }

      if (scrtxt) {                                                   // wenn ein Scrolltext ausgegeben werden soll
        p = string(scrtxt);                                           // String-Adresse im Flash ermitteln
        if (scrpos == 0) {                                            // wenn Scrollposition 0 (String-Anfang 1)
          scrbuf[0] = ' ';                                            // Leerzeichen auf ersten Pufferplatz legen
          scrbuf[1] = ' ';                                            // Leerzeichen auf zweiten Pufferplatz legen
        }
        if (scrpos == 1) {                                            // wenn Scrollposition 1 (String-Anfang 2)
          scrbuf[0] = ' ';                                            // Leerzeichen auf ersten Pufferplatz legen
          scrbuf[1] = pgm_read_byte(p);                               // erstes String-Zeichen auf zweiten Pufferplatz legen
        }
        if ((scrpos > 1) && (scrpos < 100)) {                         // wenn Scrollposition innerhalb des Strings
          scrbuf[0] = pgm_read_byte(p + scrpos - 2);                  // Zeichen zwei Stellen vor aktueller Position auf ersten Pufferplatz legen
          if (pgm_read_byte(p + scrpos - 1) == 0) {                   // wenn Ende-Zeichen vor der aktuellen Scrollposition
            scrbuf[1] = ' ';                                          // Leerzeichen auf zweiten Pufferplatz legen
            scrpos = 100;                                             // Scrollposition auf 100 setzen (String-Ende 1)
          }
          else scrbuf[1] = pgm_read_byte(p + scrpos - 1);             // wenn keine Ende-Zeichen -> Zeichen vor aktueller Position auf zweiten Pufferplatz legen
        }
        if (scrpos == 101) {                                          // wenn Scrollposition hinter dem String-Ende (String-Ende 2)
          scrbuf[0] = ' ';                                            // Leerzeichen auf ersten Pufferplatz legen
          scrbuf[1] = ' ';                                            // Leerzeichen auf zweiten Pufferplatz legen
        }
        if (scrpos > 101) scrpos = 0;                                 // wenn String-Ende 2 berschritten -> Scrollposition zurcksetzen
        digits[4] = scrbuf[0];                                        // ersten Pufferplatz an Sekundenzehner ausgeben
        digits[5] = scrbuf[1];                                        // zweiten Pufferplatz an Sekundeneiner ausgeben
      }

      if ((intcnt < 21) || ((intcnt > 24) && (intcnt < 46))) {        // wenn Interrupt-Zhler im Bereich 0-400ms oder 500-900ms -> Daten ausgeben (Blinken, aktive Phase)
        switch (confmode) {                                           // Konfigurationsmodus auswerten und verzweigen
          case 1:                                                     // wenn Konfigurationsmodus 1 (Sprache)
            digits[2] = ' ';                                          // Minutenzehner werden nicht bentigt
            digits[3] = lang + '0';                                   // aktuelle Sprache an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 2:                                                     // wenn Konfigurationsmodus 2 (Helligkeitsmodus)
            digits[2] = ' ';                                          // Minutenzehner werden nicht bentigt
            digits[3] = brmode + '0';                                 // aktuellen Helligkeitsmodus an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 3:                                                     // wenn Konfigurationsmodus 3 (Helligkeitsschwelle)
            digits[2] = brithr / 10 + '0';                            // aktuelle Helligkeitsschwelle an Minutenzehner ausgeben
            if (digits[2] == '0') digits[2] = ' ';                    // fhrende Null ausblenden
            digits[3] = brithr % 10 + '0';                            // aktuelle Helligkeitsschwelle an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 4:                                                     // wenn Konfigurationsmodus 4 (feste DCF77-Synchronisierungszeit)
            digits[2] = ' ';                                          // Minutenzehner werden nicht bentigt
            digits[3] = dcfsyn + '0';                                 // aktuelle feste DCF77-Synchroniserungszeit an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 5:                                                     // wenn Konfigurationsmodus 5 (DCF77-Synchronisierungsstunde)
            digits[2] = dcfhou / 10 + '0';                            // DCF77-Synchronisierungsstunde an Minutenzehner ausgeben
            if (digits[2] == '0') digits[2] = ' ';                    // fhrende Null ausblenden
            digits[3] = dcfhou % 10 + '0';                            // DCF77-Synchronisierungsstunde an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 6:                                                     // wenn Konfigurationsmodus 6 (DCF77-Invertierung)
            digits[2] = ' ';                                          // Minutenzehner werden nicht bentigt
            digits[3] = dcfinv + '0';                                 // DCF77-Invertierung an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 7:                                                     // wenn Konfigurationsmodus 7 (DCF77-Pull-up)
            digits[2] = ' ';                                          // Minutenzehner werden nicht bentigt
            digits[3] = dcfpup + '0';                                 // DCF77-Pull-up an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 8:                                                     // wenn Konfigurationsmodus 8 (Sensornummer 1)
            digits[2] = sensorn1 / 10 + '0';                          // Sensornummer 1 an Minutenzehner ausgeben
            if (digits[2] == '0') digits[2] = ' ';                    // fhrende Null ausblenden
            digits[3] = sensorn1 % 10 + '0';                          // Sensornummer 1 an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 9:                                                     // wenn Konfigurationsmodus 9 (Sensornummer 2)
            digits[2] = sensorn2 / 10 + '0';                          // Sensornummer 2 an Minutenzehner ausgeben
            if (digits[2] == '0') digits[2] = ' ';                    // fhrende Null ausblenden
            digits[3] = sensorn2 % 10 + '0';                          // Sensornummer 2 an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 10:                                                    // wenn Konfigurationsmodus 10 (Sensortext 1)
            digits[2] = '-';                                          // Minuszeichen an Minutenzehner ausgeben
            digits[3] = '-';                                          // Minuszeichen an Minuteneiner ausgeben
            digits[4] = sensort1[0];                                  // Sensortext 1, Zeichen 1 ausgeben
            digits[5] = sensort1[1];                                  // Sensortext 1, Zeichen 2 ausgeben
            if (!confsubm) dots[4] = 1;                               // wenn Konfigurations-Submodus 0 -> Dezimalpunkt am Sekundenzehner einschalten
            else dots[5] = 1;                                         // wenn Konfigurations-Submodus 1 -> Dezimalpunkt am Sekundeneiner einschalten
            break;
          case 11:                                                    // wenn Konfigurationsmodus 11 (Sensortext 2)
            digits[2] = '-';                                          // Minuszeichen an Minutenzehner ausgeben
            digits[3] = '-';                                          // Minuszeichen an Minuteneiner ausgeben
            digits[4] = sensort2[0];                                  // Sensortext 2, Zeichen 1 ausgeben
            digits[5] = sensort2[1];                                  // Sensortext 2, Zeichen 2 ausgeben
            if (!confsubm) dots[4] = 1;                               // wenn Konfigurations-Submodus 0 -> Dezimalpunkt am Sekundenzehner einschalten
            else dots[5] = 1;                                         // wenn Konfigurations-Submodus 1 -> Dezimalpunkt am Sekundeneiner einschalten
            break;
          case 12 ... 13:                                             // wenn Konfigurationsmodus 12 oder 13 (Sensoroffset 1 oder 2)
            if (confmode == 12) m = sensoro1;                         // wenn Konfigurationsmodus 12 -> Sensoroffset 1 kopieren
            else m = sensoro2;                                        // wenn Konfigurationsmodus 13 -> Sensoroffset 2 kopieren
            if (m < 0) {                                              // wenn Wert negativ
              m = -m;                                                 // Wert negieren
              if (m > 9) {                                            // wenn Betrag > 9
                if (intcnt < 25) {                                    // wenn erste Sekundenhlfte
                  digits[2] = '-';                                    // Minuszeichen an Minutenzehner ausgeben
                  digits[3] = ' ';                                    // Minuteneiner lschen
                }
                else {                                                // wenn zweite Sekundenhlfte
                  digits[2] = m / 10 + '0';                           // Zehner an Minutenzehner ausgeben
                  digits[3] = m % 10 + '0';                           // Einer an Minuteneiner ausgeben
                }
              }
              else {                                                  // wenn Betrag < 10
                digits[2] = '-';                                      // Minuszeichen Minutenzehner ausgeben
                digits[3] = m + '0';                                  // Einer an Minuteneiner ausgeben
              }
            }
            else {                                                    // wenn Sensoroffset positiv
              digits[2] = m / 10 + '0';                               // Zehner an Minutenzehner ausgeben
              if (digits[2] == '0') digits[2] = ' ';                  // fhrende Null ausblenden
              digits[3] = m % 10 + '0';                               // Einer an Minuteneiner ausgeben
            }
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 14:                                                    // wenn Konfigurationsmodus 14 (Automatische Sensorkonfiguration)
            digits[2] = ' ';                                          // Minutenzehner werden nicht bentigt
            digits[3] = sensacfg + '0';                               // Automatische Sensorkonfiguration an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 15:                                                    // wenn Konfigurationsmodus 15 (Datum-Reihenfolge)
            digits[2] = ' ';                                          // Minutenzehner werden nicht bentigt
            digits[3] = dateordr + '0';                               // Datum-Reihenfolge an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 16:                                                    // wenn Konfigurationsmodus 16 (Datum-Modus)
            digits[2] = ' ';                                          // Minutenzehner werden nicht bentigt
            digits[3] = datemode + '0';                               // Datum-Modus an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 17:                                                    // wenn Konfigurationsmodus 17 (Sensor-1-Modus)
            digits[2] = ' ';                                          // Minutenzehner werden nicht bentigt
            digits[3] = senmode1 + '0';                               // Sensor-1-Modus an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 18:                                                    // wenn Konfigurationsmodus 18 (Sensor-2-Modus)
            digits[2] = ' ';                                          // Minutenzehner werden nicht bentigt
            digits[3] = senmode2 + '0';                               // Sensor-2-Modus an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 19:                                                    // wenn Konfigurationsmodus 19 (Automatische Sommerzeit)
            digits[2] = ' ';                                          // Minutenzehner werden nicht bentigt
            digits[3] = autosumt + '0';                               // Automatische Sommerzeit an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 20:                                                    // wenn Konfigurationsmodus 20 (Tag einstellen)
            digits[2] = xday / 10 + '0';                              // Tag an Minutenzehner ausgeben
            if (digits[2] == '0') digits[2] = ' ';                    // fhrende Null ausblenden
            digits[3] = xday % 10 + '0';                              // Tag an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 21:                                                    // wenn Konfigurationsmodus 21 (Monat einstellen)
            digits[2] = xmonth / 10 + '0';                            // Monat an Minutenzehner ausgeben
            if (digits[2] == '0') digits[2] = ' ';                    // fhrende Null ausblenden
            digits[3] = xmonth % 10 + '0';                            // Monat an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 22:                                                    // wenn Konfigurationsmodus 22 (Jahr einstellen)
            digits[2] = xyear / 10 + '0';                             // Jahr an Minutenzehner ausgeben
            if (digits[2] == '0') digits[2] = ' ';                    // fhrende Null ausblenden
            digits[3] = xyear % 10 + '0';                             // Jahr an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 23:                                                    // wenn Konfigurationsmodus 23 (Stunde einstellen)
            digits[2] = xhour / 10 + '0';                             // Stunde an Minutenzehner ausgeben
            if (digits[2] == '0') digits[2] = ' ';                    // fhrende Null ausblenden
            digits[3] = xhour % 10 + '0';                             // Stunde an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 24:                                                    // wenn Konfigurationsmodus 24 (Minute einstellen)
            digits[2] = xminute / 10 + '0';                           // Minute an Minutenzehner ausgeben
            if (digits[2] == '0') digits[2] = ' ';                    // fhrende Null ausblenden
            digits[3] = xminute % 10 + '0';                           // Minute an Minuteneiner ausgeben
            dots[3] = 1;                                              // Dezimalpunkt am Minuteneiner einschalten
            break;
          case 25:                                                    // wenn Konfigurationsmodus 25 (Zeit bernehmen)
            digits[0] = xhour / 10 + '0';                             // Stundenzehner ausgeben
            digits[1] = xhour % 10 + '0';                             // Stundeneiner ausgeben
            digits[2] = xminute / 10 + '0';                           // Minutenzehner ausgeben
            digits[3] = xminute % 10 + '0';                           // Minuteneiner ausgeben
            dots[6] = 1;                                              // Doppelpunkt einschalten
            break;
        }
      }      
      else {                                                          // wenn Interrupt-Zhler im Bereich 400-500ms oder 900-1000ms -> Leerzeichen ausgeben (Blinken, passive Phase)
        if ((confmode == 10) || (confmode == 11)) {                   // wenn Konfigurationsmodus 10 oder 11 -> zustzlich Blinken an Sekundenanzeige steuern
          if (!confsubm) digits[2] = ' ';                             // Leerzeichen an Minutenzehner ausgeben
          else digits[3] = ' ';                                       // Leerzeichen an Minuteneiner ausgeben
        }
        else {                                                        // wenn Konfigurationsmodus nicht 10 oder 11
          digits[2] = ' ';                                            // Leerzeichen an Minutenzehner ausgeben
          digits[3] = ' ';                                            // Leerzeichen an Minuteneiner ausgeben
        }

        dots[3] = 0;                                                  // Dezimalpunkt am Minuteneiner ausschalten
        if ((confmode == 10) || (confmode == 11)) {                   // wenn Konfigurationsmodus 10 oder 11 -> zustzlich Blinken an Sekundenanzeige steuern
          if (!confsubm) digits[4] = ' ';                             // wenn Konfigurations-Submodus 0 -> Leerzeichen an Sekundenzehner ausgeben
          else digits[5] = ' ';                                       // wenn Konfigurations-Submodus 1 -> Leerzeichen an Sekundeneiner ausgeben
          dots[4] = 0;                                                // Dezimalpunkt an Sekundenzehner ausschalten
          dots[5] = 0;                                                // Dezimalpunkt an Sekundeneiner ausschalten
        }
        if (confmode == 25) {                                         // wenn Konfigurationsmodus 25 (Zeit bernehmen)
          digits[0] = ' ';                                            // Leerzeichen am Stundenzehner ausgeben
          digits[1] = ' ';                                            // Leerzeichen am Stundeneiner ausgeben
          dots[6] = 0;                                                // Doppelpunkt ausschalten
        }
      }
    }
    asc_to_seg();                                                     // ASCII-Daten vom Anzeigespeicher in Segment-Daten konvertieren

                                                                      // --- DCF77 dekodieren und Uhrzeit setzen ---
    uint8_t errcnt;                                                   // Fehlerzhler fr DCF77-Auswertung
    uint8_t parcnt;                                                   // Zhler fr die Parittsprfung der Daten
    uint8_t dcftmp;                                                   // Zwischenspeicher fr den DCF77-Impulszhler
    uint8_t zeroflg;                                                  // Nullbyte-Flag (wenn DCF77-Puffer nur 0-Werte enthlt)

    if (dcfflag) {                                                    // wenn ein DCF77-Minutenzyklus komplett ist
      cli();                                                          // Interrupts sperren
      dcfflag = 0;                                                    // Flag fr Minutenzyklus lschen
      dcftmp = dcfimp;                                                // DCF77-Impulszhler kopieren
      dcfimp = 0;                                                     // DCF77-Impulszhler lschen
      sei();                                                          // Interrupts wieder freigeben
      if (dcftmp == 59) {                                             // wenn 59 Impulse empfangen worden sind
        errcnt = 0;                                                   // Fehlerzhler lschen
        parcnt = 0;                                                   // Parittszhler lschen
        for (i = 4; i < 12; i ++) {                                   // Minuten-Bits (im Byte 4-11) prfen
          if (dcftab[i]) parcnt ++;                                   // wenn Bit = 1 -> Parittszhler erhhen
        }
        if (parcnt & 1) errcnt ++;                                    // wenn Minuten-Prfsumme ungeradzahlig -> Fehler
        parcnt = 0;                                                   // Parittszhler lschen
        for (i = 12; i < 19; i ++) {                                  // Stunden-Bits (im Byte 12-18) prfen
          if (dcftab[i]) parcnt ++;                                   // wenn Bit = 1 -> Parittszhler erhhen
        }
        if (parcnt & 1) errcnt ++;                                    // wenn Stunden-Prfsumme ungeradzahlig -> Fehler
        parcnt = 0;                                                   // Parittszhler lschen
        for (i = 19; i < 42; i ++) {                                  // Datum-Bits (im Byte 19-41) prfen
          if (dcftab[i]) parcnt ++;                                   // wenn Bit = 1 -> Parittszhler erhhen
        }
        if (parcnt & 1) errcnt ++;                                    // wenn Datum-Prfsumme ungeradzahlig -> Fehler

        if (errcnt == 0) {                                            // wenn alle Prfbits ok -> Zeit ermitteln
          dcfbuf1[0] = dcftab[0];                                     // Zeitzone speichern
          dcfbuf1[1] = dcftab[4] + (dcftab[5] * 2) +                  // Minuten-Bits zusammenfgen
          (dcftab[6] * 4) + (dcftab[7] * 8) + (dcftab[8] * 10) +
            (dcftab[9] * 20) + (dcftab[10] * 40);                     // und Minutenwert speichern
          dcfbuf1[2] = dcftab[12] + (dcftab[13] * 2) +                // Stunden-Bits zusammenfgen
          (dcftab[14] * 4) + (dcftab[15] * 8) +
            (dcftab[16] * 10) + (dcftab[17] * 20);                    // und Stundenwert speichern
          dcfbuf1[3] = dcftab[19] + (dcftab[20] * 2) +                // Kalendertag-Bits zusammenfgen
          (dcftab[21] * 4) + (dcftab[22] * 8) +
            (dcftab[23] * 10) + (dcftab[24] * 20);                    // und Kalendertag speichern
          dcfbuf1[4] = dcftab[25] + (dcftab[26] * 2) +                // Wochentag-Bits zusammenfgen
            (dcftab[27] * 4);                                         // und Wochentag speichern
          dcfbuf1[5] = dcftab[28] + (dcftab[29] * 2) +                // Monat-Bits zusammenfgen
            (dcftab[30] * 4) + (dcftab[31] * 8) + (dcftab[32] * 10);  // und Monat speichern
          dcfbuf1[6] = dcftab[33] + (dcftab[34] * 2) +                // Jahr-Bits zusammenfgen
          (dcftab[35] * 4) + (dcftab[36] * 8) +
            (dcftab[37] * 10) + (dcftab[38] * 20) +
            (dcftab[39] * 40) + (dcftab[40] * 80);                    // und Jahr speichern
          zeroflg = 0;                                                // Nullbyte-Flag lschen
          for (i = 0; i < 7; i ++) {                                  // 7 Puffer-Bytes vergleichen
            zeroflg = zeroflg | dcfbuf1[i];                           // Puffer 1 mit Flag verknpfen
            if (dcfbuf1[i] != dcfbuf2[i]) errcnt ++;                  // Vergleich Puffer 1 mit Puffer 2
            if (dcfbuf1[i] != dcfbuf3[i]) errcnt ++;                  // Vergleich Puffer 1 mit Puffer 3
            dcfbuf3[i] = dcfbuf2[i];                                  // Kopieren Puffer 2 -> Puffer 3
            dcfbuf2[i] = dcfbuf1[i];                                  // Kopieren Puffer 1 -> Puffer 2
          }
          if (zeroflg == 0) errcnt ++;                                // wenn alle Puffer-Bytes = 0 -> Fehler
          dcfbuf2[1] ++;                                              // Puffer 2 um eine Minute erhhen
          if (dcfbuf2[1] > 59) {                                      // wenn berlauf
            dcfbuf2[1] = 0;                                           // Minuten zurcksetzen
            dcfbuf2[2] ++;                                            // Stunde erhhen
          }
          dcfbuf3[1] ++;                                              // Puffer 3 um eine Minute erhhen
          if (dcfbuf3[1] > 59) {                                      // wenn berlauf
            dcfbuf3[1] = 0;                                           // Minuten zurcksetzen
            dcfbuf3[2] ++;                                            // Stunde erhhen
          }
        }

        if (errcnt == 0) {                                            // wenn gesamte Prozedur fehlerfrei -> Uhr stellen
          cli();                                                      // Interrupts sperren
          intcnt = 0;                                                 // Interrupt-Zhler zurcksetzen
          second = 0;                                                 // Sekunden zurcksetzen
          timezone = dcfbuf1[0];                                      // Zeitzone setzen
          minute = dcfbuf1[1];                                        // Minute setzen
          hour = dcfbuf1[2];                                          // Stunde setzen
          day = dcfbuf1[3];                                           // Kalendertag setzen
          wday = dcfbuf1[4];                                          // Wochentag setzen
          month = dcfbuf1[5];                                         // Monat setzen
          year = dcfbuf1[6];                                          // Jahr setzen
          sei();                                                      // Interrupts wieder freigeben
          syncstat = 3;                                               // Uhr ist jetzt mit DCF77 synchronisiert
          dcftout = 255;                                              // DCF77-Synchronisierungs-Timeout-Zhler stoppen
          synctout = SYNCTOUT;                                        // Synchronisierungs-Timeout-Zhler neu starten
          dispactv = 1;                                               // Anzeige aktivieren
          saveflag = 1;                                               // genderte Zeitzone speichern
          t10sflag = 1;                                               // 10s-Flag setzen, damit die Anzeigeautomatik auf den neuen Zustand reagiert
        }
      }
    }
                                                                      // --- Sensordaten einlesen und auswerten, automatische Sensorkonfiguration steuern ---
    while (rxwrite != rxread) {                                       // solange Schreib- und Lese-Zeiger des RS-232-Puffers unterschiedlich sind
      c = rxbuff[rxread ++];                                          // empfangenes Zeichen aus RS-232-Puffer holen und Lese-Zeiger erhhen
      if (usbmode == 0) {                                             // wenn USB im Monitormodus
        usb_send(c);                                                  // Zeichen sofort ber USB wieder senden
        if (c == '\r') usb_send('\n');                                // wenn Zeichen = CR -> zustzlich LF senden
      }
      if ((rxpos < sizeof(rxdata)) && (c >= 32) && (c < 128)) {       // wenn Pufferende noch nicht erreicht und kein Steuerzeichen
        rxdata[rxpos ++] = c;                                         // empfangenes Zeichen in RX-Datenpuffer legen und Zeichenzhler erhhen
      }
      if ((c == '\r') || (c == '\n')) {                               // wenn empfangenes Zeichen Carriage Return (ASCII 13) oder New Line (ASCII 10)
        if (rxdata[1] == ':') {                                       // wenn Trennzeichen an Position 2 -> Datensatz empfangen
          if ((rxpos == 6) && sensacfg) {                             // wenn 6 Zeichen im Datenpuffer (Sensorkonfiguration)
            if (rxdata[0] == 'S') {                                   // wenn Datensatzkennzeichen "S"
              if ((rxdata[2] == '0') || (rxdata[2] == '1')) {         // wenn Kennung von Sensor 1 "0" oder "1"
                if ((rxdata[3] == '0') || (rxdata[3] == '1') || (rxdata[3] == '2') || (rxdata[3] == 'x')) { // wenn Kennung von Sensor 2 "0"-"2" oder "x"
                  if ((rxdata[4] >= '0') && (rxdata[4] <= '9')) {     // wenn Typ von Sensor 1 zwischen "0" und "9"
                    if ((rxdata[5] >= '0') && (rxdata[5] <= '9')) {   // wenn Typ von Sensor 2 zwischen "0" und "9"
                      sensorn1 = rxdata[2] - '0';                     // Sensornummer 1 entsprechend der empfangenen Kennung ausschalten oder auf Temperatursensor setzen
                      if (rxdata[3] == 'x') sensorn2 = 29;            // Wenn empfangene Kennung von Sensor 2 "x" -> Sensornummer 2 auf Luftfeuchtigkeitssensor (29) setzen
                      else sensorn2 = rxdata[3] - '0';                // sonst Sensornummer 2 entsprechend der empfangenen Kennung ausschalten oder auf Temperatursensor setzen
                      if (sensorn1 > 0) {                             // wenn Sensor 1 vorhanden
                        sendot1 = 2;                                  // Dezimalpunkt auf Minuten-Zehner setzen
                        sensort1[0] = '#';                            // Sensortext 1 "C"
                        sensort1[1] = 'C';
                      }
                      if (sensorn2 > 0) {                             // wenn Sensor 2 vorhanden
                        if (sensorn2 == 29) {                         // wenn Luftfeuchtigkeitssensor
                          sendot2 = 3;                                // Dezimalpunkt auf Minuten-Einer setzen
                          sensort2[0] = '%';                          // Sensortext 2 "% "
                          sensort2[1] = ' ';
                        }
                        else {                                        // wenn Temperatursensor
                          sendot2 = 2;                                // Dezimalpunkt auf Minuten-Zehner setzen
                          sensort2[0] = '#';                          // Sensortext 2 "C"
                          sensort2[1] = 'C';
                        }
                      }
                      if (!usbmode) {                                 // wenn USB im Monitormodus
                        usb_sendstr(8);                               // String "Sensor " ber USB senden
                        usb_send('1');                                // Zeichen "1" ber USB senden
                        usb_send(':');                                // Trennzeichen ber USB senden
                        usb_send(' ');                                // Leerzeichen ber USB senden
                        switch (rxdata[4] - '0') {                    // Typ von Sensor 1 auswerten und verzweigen
                          case 0: usb_sendstr(9); break;              // wenn Sensortyp 0 -> String "nicht vorhanden" ber USB senden
                          case 1: usb_sendstr(11); break;             // wenn Sensortyp 1 -> String "DHT22/AM2302" ber USB senden
                          case 2: usb_sendstr(12); break;             // wenn Sensortyp 2 -> String "DS2438/HIH4000" ber USB senden
                          case 3: usb_sendstr(13); break;             // wenn Sensortyp 3 -> String "DS18B20" ber USB senden
                          case 4: usb_sendstr(14); break;             // wenn Sensortyp 4 -> String "DS1822" ber USB senden
                          case 5: usb_sendstr(15); break;             // wenn Sensortyp 5 -> String "DS18S20" ber USB senden
                          default: usb_sendstr(10); break;            // wenn Sensortyp 9 -> String "nicht untersttzt" ber USB senden
                        }
                        usb_sendcrlf();                               // CR/LF ber USB senden
                        usb_sendstr(8);                               // String "Sensor " ber USB senden
                        usb_send('2');                                // Zeichen "2" ber USB senden
                        usb_send(':');                                // Trennzeichen ber USB senden
                        usb_send(' ');                                // Leerzeichen ber USB senden
                        switch (rxdata[5] - '0') {                    // Typ von Sensor 2 auswerten und verzweigen
                          case 0: usb_sendstr(9); break;              // wenn Sensortyp 0 -> String "nicht vorhanden" ber USB senden
                          case 3: usb_sendstr(13); break;             // wenn Sensortyp 3 -> String "DS18B20" ber USB senden
                          case 4: usb_sendstr(14); break;             // wenn Sensortyp 4 -> String "DS1822" ber USB senden
                          case 5: usb_sendstr(15); break;             // wenn Sensortyp 5 -> String "DS18S20" ber USB senden
                          default: usb_sendstr(10); break;            // wenn Sensortyp 9 -> String "nicht untersttzt" ber USB senden
                        }
                        usb_sendcrlf();                               // CR ber USB senden
                      }
                      saveflag = 1;                                   // neue Sensorkonfiguration speichern
                    }
                  }
                }
              }
            }
          }
          if (rxpos == 7) {                                           // wenn 7 Zeichen im Datenpuffer (Sensorwert)
            x = 0;                                                    // Sensorkennung zunchst lschen
            if ((rxdata[0] >= '1') && (rxdata[0] <= '8')) x = rxdata[0] - '0';      // wenn Sensorkennung im Bereich 1-8 -> in Wert 1-8 umrechnen 
            if ((rxdata[0] >= 'a') && (rxdata[0] <= 's')) x = rxdata[0] - 'a' + 9;  // wenn Sensorkennung im Bereich a-s -> in Wert 9-27 umrechnen
            if ((rxdata[0] >= 'w') && (rxdata[0] <= 'z')) x = rxdata[0] - 'a' + 6;  // wenn Sensorkennung im Bereich w-z -> in Wert 28-31 umrechnen
            if (x) {                                                  // wenn gltige Sensorkennung
              if ((x == sensorn1) || (x == sensorn2)) {               // wenn Sensorkennung fr die Uhr bestimmt ist
                e = 0;                                                // Fehlerzhlen lschen
                switch (x) {                                          // Sensorkennung auswerten und verzweigen
                  case 1 ... 27:                                      // wenn Sensorkennung 1-27 (Temperatursensor)
                    ascbuff[0] = rxdata[2];                           // Temperaturwert in ASCII-Puffer legen
                    ascbuff[1] = rxdata[3];
                    ascbuff[2] = rxdata[4];
                    ascbuff[3] = rxdata[6];
                    n = asc_to_int();                                 // Temperaturwert in Integer wandeln
                    if (n == 0x8000) e ++;                            // wenn ungltiger Wert -> Fehlerzhler erhhen
                    if (!e) {                                         // wenn kein Fehler
                      if ((n < -550) || (n > 1250)) e ++;             // wenn Temperaturwert auerhalb des zulssigen Bereichs -> Fehlerzhler erhhen
                    }
                    if (!e) {                                         // wenn kein Fehler
                      if (x == sensorn1) n += sensoro1;               // wenn Wert fr Sensor 1 bestimmt -> Sensoroffset 1 addieren
                      else n += sensoro2;                             // wenn Wert fr Sensor 2 bestimmt -> Sensoroffset 2 addieren
                      int_to_asc(n, 2);                               // Temperaturwert wieder in ASCII wandeln (eine Nachkommastelle)
                    }
                    break;
                  case 28:                                            // wenn Sensorkennung 28 (Luftdruck)
                    ascbuff[0] = rxdata[3];                           // Luftdruckwert in ASCII-Puffer legen
                    ascbuff[1] = rxdata[4];
                    ascbuff[2] = rxdata[5];
                    ascbuff[3] = rxdata[6];
                    n = asc_to_int();                                 // Luftdruckwert in Integer wandeln
                    if (n == 0x8000) e ++;                            // wenn ungltiger Wert -> Fehlerzhler erhhen
                    if (!e) {                                         // wenn kein Fehler
                      if ((n < 200) || (n > 1150)) e ++;              // wenn Luftdruckwert auerhalb des zulssigen Bereichs -> Fehlerzhler erhhen
                    }
                    if (!e) {                                         // wenn kein Fehler
                      if (x == sensorn1) n += sensoro1;               // wenn Wert fr Sensor 1 bestimmt -> Sensoroffset 1 addieren
                      else n += sensoro2;                             // wenn Wert fr Sensor 2 bestimmt -> Sensoroffset 2 addieren
                      int_to_asc(n, 3);                               // Luftdruckwert wieder in ASCII wandeln (keine Nachkommastelle)
                    }
                    break;
                  case 29 ... 31:                                     // wenn Sensorkennung 29-31 (Luftfeuchtigkeit)
                    ascbuff[0] = rxdata[3];                           // Luftfeuchtigkeitswert in ASCII-Puffer legen
                    ascbuff[1] = rxdata[4];
                    ascbuff[2] = rxdata[5];
                    ascbuff[3] = rxdata[6];
                    n = asc_to_int();                                 // Luftfeuchtigkeitswert in Integer wandeln
                    if (n == 0x8000) e ++;                            // wenn ungltiger Wert -> Fehlerzhler erhhen
                    if (!e) {                                         // wenn kein Fehler
                      if ((n < 0) || (n > 100)) e ++;                 // wenn Luftfeuchtigkeitswert auerhalb des zulssigen Bereichs -> Fehlerzhler erhhen
                    }
                    if (!e) {                                         // wenn kein Fehler
                      if (x == sensorn1) n += sensoro1;               // wenn Wert fr Sensor 1 bestimmt -> Sensoroffset 1 addieren
                      else n += sensoro2;                             // wenn Wert fr Sensor 2 bestimmt -> Sensoroffset 2 addieren
                      if (n < 0) n = 0;                               // wenn nach Offset-Addition Luftfeuchtigkeitswert < 0 -> auf 0 setzen
                      if (n > 100) n = 100;                           // wenn nach Offset-Addition Luftfeuchtigkeitswert > 100 -> auf 100 setzen
                      int_to_asc(n, 3);                               // Luftfeuchtigkeitswert wieder in ASCII wandeln (keine Nachkommastelle)
                    }
                    break;
                }
                if (!e) {                                             // wenn kein Fehler
                  if (x == sensorn1) {                                // wenn Wert fr Sensor 1 bestimmt ist
                    sentout1 = SENTOUT;                               // Sensor-1-Timeout-Zhler starten
                    if (dispmode == 2) {                                // wenn Sensorwert 1 gerade angezeigt wird
                      for (i = 0; i < 4; i ++) senbuf1[i] = ascbuff[i]; // neuen Sensorwert zunchst in den Zwischenspeicher schreiben
                      senmemf1 = 1;                                     // Zwischenspeicher-Flag 1 setzen
                    }
                    else {                                              // wenn Sensorwert 1 gerade nicht angezeigt wird
                      for (i = 0; i < 4; i ++) senval1[i] = ascbuff[i]; // neuen Sensorwert direkt verwenden
                    }
                  }
                  else {                                              // wenn Wert fr Sensor 2 bestimmt ist
                    sentout2 = SENTOUT;                               // Sensor-2-Timeout-Zhler starten
                    if (dispmode == 3) {                                // wenn Sensorwert 2 gerade angezeigt wird
                      for (i = 0; i < 4; i ++) senbuf2[i] = ascbuff[i]; // neuen Sensorwert zunchst in den Zwischenspeicher schreiben
                      senmemf2 = 1;                                     // Zwischenspeicher-Flag 2 setzen
                    }
                    else {                                              // wenn Sensorwert 1 gerade nicht angezeigt wird
                      for (i = 0; i < 4; i ++) senval2[i] = ascbuff[i]; // neuen Sensorwert direkt verwenden
                    }
                  }
                }
              }
            }
          }
        }
        rxpos = 0;                                                    // RX-Datenpuffer wieder lschen
      }
    }
                                                                      // --- USB-Empfangsdaten einlesen und auswerten, Men-Konfiguration und Bootloaderstart steuern ---
    while (usbwrite != usbread) {                                     // solange Schreib- und Lese-Zeiger des USB-Puffers unterschiedlich sind
      c = usbbuff[usbread ++];                                        // empfangenes Zeichen aus RS-232-Puffer holen und Lese-Zeiger erhhen
      if (c == 'U') {                                                 // wenn empfangenes Zeichen = 'U'
        boottout = BOOTTOUT;                                          // Bootloader-Timeout-Zhler neu setzen
        bootlcnt ++;                                                  // Bootloader-Zeichen-Zhler erhhen
        if (bootlcnt == 2) {                                          // wenn Bootloader-Zeichen-Zhler = 2
          WDTCR = _BV(WDCE) | _BV(WDE);                               // Watchdog-Reset aktivieren, Timeout 16ms
          while(1);                                                   // Endlosschleife, Watchdog-Reset unterbricht die Schleife und startet anschlieend den Bootloader
        }
      }
      else {                                                          // wenn empfangenes Zeichen <> 'U'
        usbtout = USBTOUT;                                            // USB-Timeout-Zhler neu starten
        if (usbmode) {                                                // wenn USB im Konfigurationsmodus
          if (menufn == 0) {                                          // wenn keine Men-Funktion gesetzt
            switch(c) {                                               // empfangenes Zeichen auswerten und verzweigen
              case 'a':                                               // wenn 'a' empfangen (Sprache)
                lang ++;                                              // Sprache erhhen
                if (lang > 1) lang = 0;                               // wenn Endwert berschritten -> auf Anfangswert setzen
                saveflag = 1;                                         // Speicher-Flag setzen
                usb_sendmenu();                                       // Konfigurations-Men senden
                break;
              case 'b':                                               // wenn 'b' empfangen (Helligkeitsmodus)
                brmode ++;                                            // Helligkeitsmodus erhhen
                if (brmode > 1) brmode = 0;                           // wenn Endwert berschritten -> auf Anfangswert setzen
                saveflag = 1;                                         // Speicher-Flag setzen
                usb_sendmenu();                                       // Konfigurations-Men senden
                break;
              case 'c':                                               // wenn 'c' empfangen (Helligkeitsschwelle)
                menufn = 'c';                                         // Men-Funktion 'c' setzen
                usb_sendstr(51);                                      // Text "Helligkeitsschwelle (0...10): " senden
                break;
              case 'd':                                               // wenn 'd' empfangen (feste Synchronisierungszeit)
                dcfsyn ++;                                            // feste Synchronisierungszeit erhhen
                if (dcfsyn > 1) dcfsyn = 0;                           // wenn Endwert berschritten -> auf Anfangswert setzen
                saveflag = 1;                                         // Speicher-Flag setzen
                usb_sendmenu();                                       // Konfigurations-Men senden
                break;
              case 'e':                                               // wenn 'e' empfangen (DCF-Synchronisierungsstunde)
                menufn = 'e';                                         // Men-Funktion 'e' setzen
                usb_sendstr(52);                                      // Text "DCF77-Synchronisierungsstunde (0...23): " senden
                break;
              case 'f':                                               // wenn 'f' empfangen (DCF-Invertierung)
                dcfinv ++;                                            // DCF-Invertierung erhhen
                if (dcfinv > 1) dcfinv = 0;                           // wenn Endwert berschritten -> auf Anfangswert setzen
                saveflag = 1;                                         // Speicher-Flag setzen
                usb_sendmenu();                                       // Konfigurations-Men senden
                break;
              case 'g':                                               // wenn 'g' empfangen (DCF-Pullup)
                dcfpup ++;                                            // DCF-Pullup erhhen
                if (dcfpup > 1) dcfpup = 0;                           // wenn Endwert berschritten -> auf Anfangswert setzen
                if (dcfpup) PORTF |= _BV(PINF3);                      // wenn Pull-up aktiviert -> beim DCF77-Eingang Pull-up einschalten
                else PORTF &= ~_BV(PINF3);                            // wenn Pull-up deaktiviert -> beim DCF77-Eingang Pull-up ausschalten
                saveflag = 1;                                         // Speicher-Flag setzen
                usb_sendmenu();                                       // Konfigurations-Men senden
                break;
              case 'h':                                               // wenn 'h' empfangen (Sensornummer 1)
                menufn = 'h';                                         // Men-Funktion 'h' setzen
                usb_sendstr(53);                                      // Text "Sensornummer 1 (0...31): " senden
                break;
              case 'i':                                               // wenn 'i' empfangen (Sensornummer 2)
                menufn = 'i';                                         // Men-Funktion 'i' setzen
                usb_sendstr(54);                                      // Text "Sensornummer 2 (0...31): " senden
                break;
              case 'j':                                               // wenn 'j' empfangen (Sensortext 1)
                menufn = 'j';                                         // Men-Funktion 'j' setzen
                usb_sendstr(55);                                      // Text "Sensortext 1 (2 Zeichen): " senden
                break;
              case 'k':                                               // wenn 'k' empfangen (Sensortext 2)
                menufn = 'k';                                         // Men-Funktion 'k' setzen
                usb_sendstr(56);                                      // Text "Sensortext 2 (2 Zeichen): " senden
                break;
              case 'l':                                               // wenn 'l' empfangen (Sensoroffset 1)
                menufn = 'l';                                         // Men-Funktion 'l' setzen
                usb_sendstr(57);                                      // Text "Sensoroffset 1 (-30...30): " senden
                break;
              case 'm':                                               // wenn 'm' empfangen (Sensoroffset 2)
                menufn = 'm';                                         // Men-Funktion 'm' setzen
                usb_sendstr(58);                                      // Text "Sensoroffset 2 (-30...30): " senden
                break;
              case 'n':                                               // wenn 'n' empfangen (automatische Sensorkonfiguration)
                sensacfg ++;                                          // automatische Sensorkonfiguration erhhen
                if (sensacfg > 1) sensacfg = 0;                       // wenn Endwert berschritten -> auf Anfangswert setzen
                saveflag = 1;                                         // Speicher-Flag setzen
                usb_sendmenu();                                       // Konfigurations-Men senden
                break;
              case 'o':                                               // wenn 'o' empfangen (Datum-Reihenfolge)
                dateordr ++;                                          // Datum-Reihenfolge erhhen
                if (dateordr > 1) dateordr = 0;                       // wenn Endwert berschritten -> auf Anfangswert setzen
                saveflag = 1;                                         // Speicher-Flag setzen
                usb_sendmenu();                                       // Konfigurations-Men senden
                break;
              case 'p':                                               // wenn 'p' empfangen (Datum anzeigen)
                datemode ++;                                          // Datum-Modus erhhen
                if (datemode > 2) datemode = 0;                       // wenn Endwert berschritten -> auf Anfangswert setzen
                saveflag = 1;                                         // Speicher-Flag setzen
                usb_sendmenu();                                       // Konfigurations-Men senden
                break;
              case 'q':                                               // wenn 'q' empfangen (Sensor 1 anzeigen)
                senmode1 ++;                                          // Sensor-1-Modus erhhen
                if (senmode1 > 1) senmode1 = 0;                       // wenn Endwert berschritten -> auf Anfangswert setzen
                saveflag = 1;                                         // Speicher-Flag setzen
                usb_sendmenu();                                       // Konfigurations-Men senden
                break;
              case 'r':                                               // wenn 'r' empfangen (Sensor 2 anzeigen)
                senmode2 ++;                                          // Sensor-2-Modus erhhen
                if (senmode2 > 1) senmode2 = 0;                       // wenn Endwert berschritten -> auf Anfangswert setzen
                saveflag = 1;                                         // Speicher-Flag setzen
                usb_sendmenu();                                       // Konfigurations-Men senden
                break;
              case 's':                                               // wenn 's' empfangen (automatische Sommerzeit)
                autosumt ++;                                          // automatische Sommerzeit erhhen
                if (autosumt > 1) autosumt = 0;                       // wenn Endwert berschritten -> auf Anfangswert setzen
                saveflag = 1;                                         // Speicher-Flag setzen
                usb_sendmenu();                                       // Konfigurations-Men senden
                break;
              case 't':                                               // wenn 't' empfangen (Tag einstellen)
                menufn = 't';                                         // Men-Funktion 't' setzen
                usb_sendstr(60);                                      // Text "Tag (1...31): " senden
                break;
              case 'u':                                               // wenn 'u' empfangen (Monat einstellen)
                menufn = 'u';                                         // Men-Funktion 'u' setzen
                usb_sendstr(61);                                      // Text "Monat (1...12): " senden
                break;
              case 'v':                                               // wenn 'v' empfangen (Jahr einstellen)
                menufn = 'v';                                         // Men-Funktion 'v' setzen
                usb_sendstr(62);                                      // Text "Jahr (0...99): " senden
                break;
              case 'w':                                               // wenn 'w' empfangen (Stunde einstellen)
                menufn = 'w';                                         // Men-Funktion 'w' setzen
                usb_sendstr(63);                                      // Text "Stunde (0...23): " senden
                break;
              case 'x':                                               // wenn 'x' empfangen (Minute einstellen)
                menufn = 'x';                                         // Men-Funktion 'x' setzen
                usb_sendstr(64);                                      // Text "Minute (0...59): " senden
                break;
              case 'y':                                               // wenn 'y' empfangen (Zeit bernehmen)
                if (timeflag) {                                       // wenn Zeit-nderungs-Flag gesetzt
                  cli();                                              // Interrupts sperren
                  intcnt = 0;                                         // Interrupt-Zhler zurcksetzen
                  second = 0;                                         // Sekunden zurcksetzen
                  minute = xminute;                                   // Minute bernehmen
                  hour = xhour;                                       // Stunde bernehmen
                  sei();                                              // Interrupts wieder freigeben
                  if ((xyear % 4) == 0) {                               // wenn Schaltjahr
                    if (xday > pgm_read_byte_far(&daytab2[xmonth - 1])) // wenn Tag > Schaltjahr-Tabelle
                    xday = pgm_read_byte_far(&daytab2[xmonth - 1]);     // auf letzten Tag im Monat setzen
                  }
                  else {                                                // wenn kein Schaltjahr
                    if (xday > pgm_read_byte_far(&daytab1[xmonth - 1])) // wenn Tag > Normaljahr-Tabelle
                    xday = pgm_read_byte_far(&daytab2[xmonth - 1]);     // auf letzten Tag im Monat setzen
                  }
                  day = xday;                                         // Kalendertag bernehmen
                  month = xmonth;                                     // Monat bernehmen
                  year = xyear;                                       // Jahr bernehmen
                  wday = calc_wday(day, month, year);                 // Wochentag ermitteln
                  if (autosumt) timezone = chk_summtime();            // wenn Sommerzeit-Automatik eingeschaltet -> Zeitzone einstellen
                  else timezone = 0;                                  // sonst Zeitzone auf MEZ setzen
                  syncstat = 1;                                       // Uhr wurde manuell gestellt
                  synctout = 255;                                     // Synchronisierungs-Timeout-Zhler stoppen
                  dcftout = 255;                                      // Timeout-Zhler stoppen
                  dispactv = 1;                                       // Display aktivieren
                  saveflag = 1;                                       // genderte Sommerzeit-Automatik und Zeitzone speichern
                  t10sflag = 1;                                       // 10s-Flag setzen, damit die Anzeigeautomatik auf den neuen Zustand reagiert
                }
                break;
              case 'z':                                               // wenn 'z' empfangen (Konfiguration beenden)
                usb_sendstr(16);                                      // Text "Konfiguration beendet" senden
                usbtout = 255;                                        // USB-Timeout-Zhler stoppen
                usbmode = 0;                                          // USB in Monitormodus umschalten
                usbpos = 0;                                           // USB-Datenpuffer lschen
                break;
            }
          }
          else {                                                      // wenn eine Men-Funktion gesetzt
            if ((c == '\r') || (c == '\n')) {                         // wenn CR oder LF empfangen
              switch(menufn) {                                        // Men-Funktion auswerten und verzweigen
                case 'c':                                             // wenn 'c' gesetzt
                  m = asc_to_int8();                                  // eingegebenen Wert ermitteln
                  if ((m >= 0) && (m <= 10)) {                        // wenn Wert im gltigen Bereich
                    brithr = m;                                       // neuen Wert bernehmen
                    saveflag = 1;                                     // Speicher-Flag setzen
                  }
                  break;
                case 'e':                                             // wenn 'e' gesetzt (DCF-Synchronisierungsstunde)
                  m = asc_to_int8();                                  // eingegebenen Wert ermitteln
                  if ((m >= 0) && (m <= 23)) {                        // wenn Wert im gltigen Bereich
                    dcfhou = m;                                       // neuen Wert bernehmen
                    saveflag = 1;                                     // Speicher-Flag setzen
                  }
                  break;
                case 'h':                                             // wenn 'h' gesetzt (Sensornummer 1)
                  m = asc_to_int8();                                  // eingegebenen Wert ermitteln
                  if ((m >= 0) && (m <= 31)) {                        // wenn Wert im gltigen Bereich
                    sensorn1 = m;                                     // neuen Wert bernehmen
                    saveflag = 1;                                     // Speicher-Flag setzen
                    switch (sensorn1) {                               // Sensornummer 1 auswerten und verzweigen
                      case 0:         sendot1 = 4; break;             // wenn kein Sensor -> Dezimalpunkt ausschalten
                      case 1 ... 27:  sendot1 = 2; break;             // wenn Temperatursensor -> Dezimalpunkt an Minutenzehner einschalten (eine Nachkommastelle)
                      case 28 ... 31: sendot1 = 3; break;             // wenn Luftdruck- oder Luftfeuchtigkeitssensor -> Dezimalpunkt an Minuteneiner einschalten (keine Nachkommastelle)
                    }
                  }
                  break;
                case 'i':                                             // wenn 'i' gesetzt (Sensornummer 2)
                  m = asc_to_int8();                                  // eingegebenen Wert ermitteln
                  if ((m >= 0) && (m <= 31)) {                        // wenn Wert im gltigen Bereich
                    sensorn2 = m;                                     // neuen Wert bernehmen
                    saveflag = 1;                                     // Speicher-Flag setzen
                    switch (sensorn2) {                               // Sensornummer 2 auswerten und verzweigen
                      case 0:         sendot2 = 4; break;             // wenn kein Sensor -> Dezimalpunkt ausschalten
                      case 1 ... 27:  sendot2 = 2; break;             // wenn Temperatursensor -> Dezimalpunkt an Minutenzehner einschalten (eine Nachkommastelle)
                      case 28 ... 31: sendot2 = 3; break;             // wenn Luftdruck- oder Luftfeuchtigkeitssensor -> Dezimalpunkt an Minuteneiner einschalten (keine Nachkommastelle)
                    }
                  }
                  break;
                case 'j':                                             // wenn 'j' gesetzt (Sensortext 1)
                  if (usbpos) {                                       // wenn Zeichen im USB-Datenpuffer
                    sensort1[0] = usbdata[0];                         // erstes Zeichen aus USB-Datenpuffer holen
                    switch(usbpos) {                                  // Anzahl der Zeichen im USB-Datenpuffer auswerten und verzweigen
                      case 1: sensort1[1] = ' '; break;               // wenn 1 Zeichen im Puffer -> Leerzeichen als zweites Zeichen verwenden
                      case 2: sensort1[1] = usbdata[1]; break;        // wenn 2 Zeichen im Puffer -> zweites Zeichen holen
                      case 3: sensort1[1] = conv_char(usbdata[1], usbdata[2]); break; // wenn 3 Zeichen im Puffer -> zweites Zeichen holen und konvertieren
                    }
                    saveflag = 1;                                     // Speicher-Flag setzen
                  }
                  break;
                case 'k':                                             // wenn 'k' gesetzt (Sensortext 2)
                  if (usbpos) {                                       // wenn Zeichen im USB-Datenpuffer
                    sensort2[0] = (usbdata[0]);                       // erstes Zeichen aus USB-Datenpuffer holen
                    switch(usbpos) {                                  // Anzahl der Zeichen im USB-Datenpuffer auswerten und verzweigen
                      case 1: sensort2[1] = ' '; break;               // wenn 1 Zeichen im Puffer -> Leerzeichen als zweites Zeichen verwenden
                      case 2: sensort2[1] = usbdata[1]; break;        // wenn 2 Zeichen im Puffer -> zweites Zeichen holen
                      case 3: sensort2[1] = conv_char(usbdata[1], usbdata[2]); break; // wenn 3 Zeichen im Puffer -> zweites Zeichen holen und konvertieren
                    }
                    saveflag = 1;                                     // Speicher-Flag setzen
                  }
                  break;
                case 'l':                                             // wenn 'l' gesetzt (Sensoroffset 1)
                  m = asc_to_int8();                                  // eingegebenen Wert ermitteln
                  if ((m >= -30) && (m <= 30)) {                      // wenn Wert im gltigen Bereich
                    sensoro1 = m;                                     // neuen Wert bernehmen
                    saveflag = 1;                                     // Speicher-Flag setzen
                  }
                  break;
                case 'm':                                             // wenn 'm' gesetzt (Sensoroffset 2)
                  m = asc_to_int8();                                  // eingegebenen Wert ermitteln
                  if ((m >= -30) && (m <= 30)) {                      // wenn Wert im gltigen Bereich
                    sensoro2 = m;                                     // neuen Wert bernehmen
                    saveflag = 1;                                     // Speicher-Flag setzen
                  }
                  break;
                case 't':                                             // wenn 't' gesetzt (Tag einstellen)
                  m = asc_to_int8();                                  // eingegebenen Wert ermitteln
                  if ((m > 0) && (m < 32)) {                          // wenn Wert im gltigen Bereich
                    xday = m;                                         // neuen Wert bernehmen
                    timeflag = 1;                                     // Zeit-nderungs-Flag setzen
                  }
                case 'u':                                             // wenn 'u' gesetzt (Monat einstellen)
                  m = asc_to_int8();                                  // eingegebenen Wert ermitteln
                  if ((m > 0) && (m < 13)) {                          // wenn Wert im gltigen Bereich
                    xmonth = m;                                       // neuen Wert bernehmen
                    timeflag = 1;                                     // Zeit-nderungs-Flag setzen
                  }
                case 'v':                                             // wenn 'v' gesetzt (Jahr einstellen)
                  m = asc_to_int8();                                  // eingegebenen Wert ermitteln
                  if ((m >= 0) && (m <= 99)) {                        // wenn Wert im gltigen Bereich
                    xyear = m;                                        // neuen Wert bernehmen
                    timeflag = 1;                                     // Zeit-nderungs-Flag setzen
                  }
                case 'w':                                             // wenn 'w' gesetzt (Stunde einstellen)
                  m = asc_to_int8();                                  // eingegebenen Wert ermitteln
                  if ((m >= 0) && (m <= 23)) {                        // wenn Wert im gltigen Bereich
                    xhour = m;                                        // neuen Wert bernehmen
                    timeflag = 1;                                     // Zeit-nderungs-Flag setzen
                  }
                case 'x':                                             // wenn 'x' gesetzt (Minute einstellen)
                  m = asc_to_int8();                                  // eingegebenen Wert ermitteln
                  if ((m >= 0) && (m <= 59)) {                        // wenn Wert im gltigen Bereich
                    xminute = m;                                      // neuen Wert bernehmen
                    timeflag = 1;                                     // Zeit-nderungs-Flag setzen
                  }
              }
              usb_sendcrlf();                                         // CR/LF senden
              usb_sendmenu();                                         // Konfigurations-Men senden
              menufn = 0;                                             // Men-Funktion wieder lschen
              usbpos = 0;                                             // USB-Datenpuffer wieder lschen
            }
            else {                                                    // wenn nicht CR empfangen
              if ((usbpos < sizeof(usbdata)) && (c >= 32) && (c < 127)) { // wenn Pufferende noch nicht erreicht und ASCII-Zeichen
                usbdata[usbpos ++] = c;                               // empfangenes Zeichen in USB-Datenpuffer legen und Zeichenzhler erhhen
                usb_send(c);                                          // Zeichen als Echo wieder senden
              }
            }
          }
        }
        else {                                                        // wenn USB im Monitormodus
          if ((c == '\r') || (c == '\n')) {                           // wenn CR oder LF (Enter) empfangen
            usbmode = 1;                                              // USB in Konfigurationsmodus umschalten
            usbtout = USBTOUT;                                        // USB-Timeout-Zhler neu starten
            xminute = minute;                                         // Minute fr Zeiteinstellung kopieren
            xhour = hour;                                             // Stunde fr Zeiteinstellung kopieren
            xday = day;                                               // Tag fr Zeiteinstellung kopieren
            xmonth = month;                                           // Monat fr Zeiteinstellung kopieren
            xyear = year;                                             // Jahr fr Zeiteinstellung kopieren
            timeflag = 0;                                             // Zeit-nderungs-Flag lschen
            usb_sendmenu();                                           // Konfigurations-Men senden
            menufn = 0;                                               // Men-Funktion lschen
            usbpos = 0;                                               // USB-Datenpuffer lschen
          }
        }
      }
    }
                                                                      // --- Kurzen Druck von Taster 1 auswerten ---
    if (t1stat && !t1quit) {                                          // wenn Taster 1 gedrckt und noch nicht quittiert wurde
      t1quit = 1;                                                     // Taster-Bedienung quittieren
      if (dcftout < 255) {                                            // wenn DCF77-Synchronisierung aktiv
        dcftout = 255;                                                // DCF77-Synchronisierungs-Timeout-Zhler stoppen
        dispactv = 1;                                                 // Display aktivieren
      }
      else {                                                          // wenn keine DCF77-Synchronisierung aktiv
        if ((confmode == 10) || (confmode == 11)) {                   // wenn Konfigurationsmodus 10 oder 11
          if (!confsubm) confsubm ++;                                 // wenn Konfigurations-Submodus 0 -> auf 1 setzen
          else {                                                      // wenn Konfigurations-Submodus 1
            confsubm = 0;                                             // wieder auf Konfigurations-Submodus 0 setzen
            confmode ++;                                              // Konfigurationsmodus erhhen
          }
        }
        else {                                                        // wenn Konfigurationsmodus auer 10 und 11
          confmode ++;                                                // Konfigurationsmodus erhhen
          if (confmode == 10) confsubm = 0;                           // wenn Konfigurationsmodus jetzt 10 -> Konfigurations-Submodus 0 setzen
        }
        if (!timeflag && (confmode == 25)) confmode = 0;              // wenn Zeit-nderungs-Flag nicht gesetzt und Konfigurationsmodus auf "Zeit bernehmen" -> Konfigurationsmodus wieder ausschalten
        if (confmode > 25) confmode = 0;                              // wenn Bereich berschritten -> Konfigurationsmodus wieder ausschalten
        if (confmode == 20) {                                         // wenn Konfigurationsmodus 20 (Beginn der Zeit-Einstellungen)
          xminute = minute;                                           // Minute fr Zeiteinstellung kopieren
          xhour = hour;                                               // Stunde fr Zeiteinstellung kopieren
          xday = day;                                                 // Tag fr Zeiteinstellung kopieren
          xmonth = month;                                             // Monat fr Zeiteinstellung kopieren
          xyear = year;                                               // Jahr fr Zeiteinstellung kopieren
          timeflag = 0;                                               // Zeit-nderungs-Flag lschen
        }
        if (confmode) {                                               // wenn Konfigurationsmodus aktiv
          conftout = CONFTOUT;                                        // Konfigurations-Timeout-Zhler setzen
          scroll_text();                                              // Scrolltext entsprechend Konfigurationsmodus starten
        }
      }
    }
                                                                      // --- Kurzen Druck von Taster 2 auswerten ---
    if (t2stat && !t2quit) {                                          // wenn Taster 2 gedrckt und noch nicht quittiert wurde
      t2quit = 1;                                                     // Taster-Bedienung quittieren
      if (dcftout < 255) {                                            // wenn DCF77-Synchronisierung aktiv
        dcftout = 255;                                                // DCF77-Synchronisierungs-Timeout-Zhler stoppen
        dispactv = 1;                                                 // Display aktivieren
      }
      else {                                                          // wenn keine DCF77-Synchronisierung aktiv
        if (confmode) conftout = CONFTOUT;                            // wenn Konfigurationsmodus aktiv -> Konfigurations-Timeout-Zhler setzen
        switch (confmode) {                                           // Konfigurationsmodus auswerten und verzweigen
          case 1:                                                     // wenn Konfigurationsmodus 1 (Sprache)
            lang ++;                                                  // Sprache erhhen
            if (lang > 1) lang = 0;                                   // wenn Bereich berschritten -> auf Anfangswert setzen
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 2:                                                     // wenn Konfigurationsmodus 2 (Helligkeitsmodus)
            brmode ++;                                                // Helligkeitsmodus erhhen
            if (brmode > 1) brmode = 0;                               // wenn Bereich berschritten -> auf Anfangswert setzen
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 3:                                                     // wenn Konfigurationsmodus 3 (Helligkeitsschwelle)
            brithr ++;                                                // Helligkeitsschwelle erhhen
            if (brithr > 10) brithr = 0;                              // wenn Bereich berschritten -> auf Anfangswert setzen
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 4:                                                     // wenn Konfigurationsmodus 4 (feste Synchronisierungszeit)
            dcfsyn ++;                                                // feste Synchronisierungszeit erhhen
            if (dcfsyn > 1) dcfsyn = 0;                               // wenn Bereich berschritten -> auf Anfangswert setzen
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 5:                                                     // wenn Konfigurationsmodus 5 (Synchronisierungsstunde)
            dcfhou ++;                                                // Synchronisierungsstunde erhhen
            if (dcfhou > 23) dcfhou = 0;                              // wenn Bereich berschritten -> auf Anfangswert setzen
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 6:                                                     // wenn Konfigurationsmodus 6 (DCF77-Invertierung)
            dcfinv ++;                                                // DCF77-Invertierung erhhen
            if (dcfinv > 1) dcfinv = 0;                               // wenn Bereich berschritten -> auf Anfangswert setzen
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 7:                                                     // wenn Konfigurationsmodus 7 (DCF77-Pull-up)
            dcfpup ++;                                                // DCF77-Pull-up erhhen
            if (dcfpup > 1) dcfpup = 0;                               // wenn Bereich berschritten -> auf Anfangswert setzen
            if (dcfpup) PORTF |= _BV(PINF3);                          // wenn Pull-up aktiviert -> beim DCF77-Eingang Pull-up einschalten
            else PORTF &= ~_BV(PINF3);                                // wenn Pull-up deaktiviert -> beim DCF77-Eingang Pull-up ausschalten
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 8:                                                     // wenn Konfigurationsmodus 8 (Sensornummer 1)
            sensorn1 ++;                                              // Sensornummer 1 erhhen
            if (sensorn1 > 31) sensorn1 = 0;                          // wenn Bereich berschritten -> auf Anfangswert setzen
            switch (sensorn1) {                                       // Sensornummer 1 auswerten und verzweigen
              case 0:         sendot1 = 4; break;                     // wenn kein Sensor -> Dezimalpunkt ausschalten
              case 1 ... 27:  sendot1 = 2; break;                     // wenn Temperatursensor -> Dezimalpunkt an Minutenzehner einschalten (eine Nachkommastelle)
              case 28 ... 31: sendot1 = 3; break;                     // wenn Luftdruck- oder Luftfeuchtigkeitssensor -> Dezimalpunkt an Minuteneiner einschalten (keine Nachkommastelle)
            }
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 9:                                                     // wenn Konfigurationsmodus 9 (Sensornummer 2)
            sensorn2 ++;                                              // Sensornummer 2 erhhen
            if (sensorn2 > 31) sensorn2 = 0;                          // wenn Bereich berschritten -> auf Anfangswert setzen
            switch (sensorn2) {                                       // Sensornummer 2 auswerten und verzweigen
              case 0:         sendot2 = 4; break;                     // wenn kein Sensor -> Dezimalpunkt ausschalten
              case 1 ... 27:  sendot2 = 2; break;                     // wenn Temperatursensor -> Dezimalpunkt an Minutenzehner einschalten (eine Nachkommastelle)
              case 28 ... 31: sendot2 = 3; break;                     // wenn Luftdruck- oder Luftfeuchtigkeitssensor -> Dezimalpunkt an Minuteneiner einschalten (keine Nachkommastelle)
            }
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 10:                                                    // wenn Konfigurationsmodus 10 (Sensortext 1)
            if (!confsubm) sensort1[0] = next_char(sensort1[0]);      // wenn Konfigurations-Submen = 0 -> Zeichen 1 erhhen
            else sensort1[1] = next_char(sensort1[1]);                // wenn Konfigurations-Submen = 1 -> Zeichen 2 erhhen
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 11:                                                    // wenn Konfigurationsmodus 11 (Sensortext 2)
            if (!confsubm) sensort2[0] = next_char(sensort2[0]);      // wenn Konfigurations-Submen = 0 -> Zeichen 1 erhhen
            else sensort2[1] = next_char(sensort2[1]);                // wenn Konfigurations-Submen = 1 -> Zeichen 2 erhhen
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 12:                                                    // wenn Konfigurationsmodus 12 (Sensoroffset 1)
            sensoro1 ++;                                              // Sensoroffset 1 erhhen
            if (sensoro1 > 30) sensoro1 = -30;                        // wenn Bereich berschritten -> auf Anfangswert setzen
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 13:                                                    // wenn Konfigurationsmodus 13 (Sensoroffset 2)
            sensoro2 ++;                                              // Sensoroffset 2 erhhen
            if (sensoro2 > 30) sensoro2 = -30;                        // wenn Bereich berschritten -> auf Anfangswert setzen
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 14:                                                    // wenn Konfigurationsmodus 14 (Automatische Sensorkonfiguration)
            sensacfg ++;                                              // Automatische Sensorkonfiguration erhhen
            if (sensacfg > 1) sensacfg = 0;                           // wenn Bereich berschritten -> auf Anfangswert setzen
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 15:                                                    // wenn Konfigurationsmodus 15 (Datum-Reihenfolge)
            dateordr ++;                                              // Datum-Reihenfolge erhhen
            if (dateordr > 1) dateordr = 0;                           // wenn Bereich berschritten -> auf Anfangswert setzen
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 16:                                                    // wenn Konfigurationsmodus 15 (Datum-Modus)
            datemode ++;                                              // Datum-Modus erhhen
            if (datemode > 2) datemode = 0;                           // wenn Bereich berschritten -> auf Anfangswert setzen
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 17:                                                    // wenn Konfigurationsmodus 17 (Sensor-1-Modus)
            senmode1 ++;                                              // Sensor-1-Modus erhhen
            if (senmode1 > 1) senmode1 = 0;                           // wenn Bereich berschritten -> auf Anfangswert setzen
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 18:                                                    // wenn Konfigurationsmodus 18 (Sensor-2-Modus)
            senmode2 ++;                                              // Sensor-2-Modus erhhen
            if (senmode2 > 1) senmode2 = 0;                           // wenn Bereich berschritten -> auf Anfangswert setzen
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 19:                                                    // wenn Konfigurationsmodus 19 (Automatische Sommerzeit)
            autosumt ++;                                              // Automatische Sommerzeit erhhen
            if (autosumt > 1) autosumt = 0;                           // wenn Bereich berschritten -> auf Anfangswert setzen
            saveflag = 1;                                             // Speicher-Flag setzen
            break;
          case 20:                                                    // wenn Konfigurationsmodus 20 (Tag einstellen)
            xday ++;                                                  // Tag erhhen
            if (xday > 31) xday = 1;                                  // wenn Bereich berschritten -> auf Anfangswert setzen
            timeflag = 1;                                             // Zeit-nderungs-Flag setzen
            break;
          case 21:                                                    // wenn Konfigurationsmodus 21 (Monat einstellen)
            xmonth ++;                                                // Monat erhhen
            if (xmonth > 12) xmonth = 1;                              // wenn Bereich berschritten -> auf Anfangswert setzen
            timeflag = 1;                                             // Zeit-nderungs-Flag setzen
            break;
          case 22:                                                    // wenn Konfigurationsmodus 22 (Jahr einstellen)
            xyear ++;                                                 // Jahr erhhen
            if (xyear > 99) xyear = YEAR;                             // wenn Bereich berschritten -> auf Anfangswert setzen
            timeflag = 1;                                             // Zeit-nderungs-Flag setzen
            break;
          case 23:                                                    // wenn Konfigurationsmodus 23 (Stunde einstellen)
            xhour ++;                                                 // Stunde erhhen
            if (xhour > 23) xhour = 0;                                // wenn Bereich berschritten -> auf Anfangswert setzen
            timeflag = 1;                                             // Zeit-nderungs-Flag setzen
            break;
          case 24:                                                    // wenn Konfigurationsmodus 24 (Minute einstellen)
            xminute ++;                                               // Minute erhhen
            if (xminute > 59) xminute = 0;                            // wenn Bereich berschritten -> auf Anfangswert setzen
            timeflag = 1;                                             // Zeit-nderungs-Flag setzen
            break;
          case 25:                                                    // wenn Konfigurationsmodus 25 (Zeit bernehmen)
            if (timeflag) {                                           // wenn Zeit-nderungs-Flag gesetzt
              cli();                                                  // Interrupts sperren
              intcnt = 0;                                             // Interrupt-Zhler zurcksetzen
              second = 0;                                             // Sekunden zurcksetzen
              minute = xminute;                                       // Minute bernehmen
              hour = xhour;                                           // Stunde bernehmen
              sei();                                                  // Interrupts wieder freigeben
              if ((xyear % 4) == 0) {                                 // wenn Schaltjahr
                if (xday > pgm_read_byte_far(&daytab2[xmonth - 1]))   // wenn Tag > Schaltjahr-Tabelle
                xday = pgm_read_byte_far(&daytab2[xmonth - 1]);       // auf letzten Tag im Monat setzen
              }
              else {                                                  // wenn kein Schaltjahr
                if (xday > pgm_read_byte_far(&daytab1[xmonth - 1]))   // wenn Tag > Normaljahr-Tabelle
                xday = pgm_read_byte_far(&daytab2[xmonth - 1]);       // auf letzten Tag im Monat setzen
              }
              day = xday;                                             // Kalendertag bernehmen
              month = xmonth;                                         // Monat bernehmen
              year = xyear;                                           // Jahr bernehmen
              wday = calc_wday(day, month, year);                     // Wochentag ermitteln
              if (autosumt) timezone = chk_summtime();                // wenn Sommerzeit-Automatik eingeschaltet -> Zeitzone einstellen
              else timezone = 0;                                      // sonst Zeitzone auf MEZ setzen
              syncstat = 1;                                           // Uhr wurde manuell gestellt
              synctout = 255;                                         // Synchronisierungs-Timeout-Zhler stoppen
              saveflag = 1;                                           // genderte Sommerzeit-Automatik und Zeitzone speichern
              t10sflag = 1;                                           // 10s-Flag setzen, damit die Anzeigeautomatik auf den neuen Zustand reagiert
              confmode = 0;                                           // Konfiguration beenden
            }
            break;
        }
      }
    }
                                                                      // --- Langen Druck von Taster 1 auswerten ---
    if (t1long == KEYLONG) {                                          // wenn Taster 1 lange gedrckt
      if (confmode != 20) {                                           // wenn nicht Konfigurationsmodus 20
        confmode = 20;                                                // auf Konfigurationsmodus 20 setzen
        xminute = minute;                                             // Minute fr Zeiteinstellung kopieren
        xhour = hour;                                                 // Stunde fr Zeiteinstellung kopieren
        xday = day;                                                   // Tag fr Zeiteinstellung kopieren
        xmonth = month;                                               // Monat fr Zeiteinstellung kopieren
        xyear = year;                                                 // Jahr fr Zeiteinstellung kopieren
        timeflag = 0;                                                 // Zeit-nderungs-Flag lschen
        conftout = CONFTOUT;                                          // Konfigurations-Timeout-Zhler setzen
        scroll_text();                                                // Scrolltext entsprechend Konfigurationsmodus starten
      }
    }
                                                                      // --- Langen Druck von Taster 2 und Auto-Repeat auswerten ---
    if (t2long == KEYLONG) {                                          // wenn Taster 2 lange gedrckt
      t2long = KEYLONG - KEYARPT;                                     // Zhler auf Auto-Repeat-Wert einstellen
      conftout = CONFTOUT;                                            // Konfigurations-Timeout-Zhler setzen
      switch (confmode) {                                             // Konfigurationsmodus auswerten und verzweigen
        case 3:                                                       // wenn Konfigurationsmodus 3 (Helligkeitsschwelle)
          brithr ++;                                                  // Helligkeitsschwelle erhhen
          if (brithr > 10) brithr = 0;                                // wenn Bereich berschritten -> auf Anfangswert setzen
          saveflag = 1;                                               // Speicher-Flag setzen
          break;
        case 5:                                                       // wenn Konfigurationsmodus 5 (Synchronisierungsstunde)
          dcfhou ++;                                                  // Synchronisierungsstunde erhhen
          if (dcfhou > 23) dcfhou = 0;                                // wenn Bereich berschritten -> auf Anfangswert setzen
          saveflag = 1;                                               // Speicher-Flag setzen
          break;
        case 8:                                                       // wenn Konfigurationsmodus 8 (Sensornummer 1)
          sensorn1 ++;                                                // Sensornummer 1 erhhen
          if (sensorn1 > 31) sensorn1 = 0;                            // wenn Bereich berschritten -> auf Anfangswert setzen
          switch (sensorn1) {                                         // Sensornummer 1 auswerten und verzweigen
            case 0:         sendot1 = 4; break;                       // wenn kein Sensor -> Dezimalpunkt ausschalten
            case 1 ... 27:  sendot1 = 2; break;                       // wenn Temperatursensor -> Dezimalpunkt an Minutenzehner einschalten (eine Nachkommastelle)
            case 28 ... 31: sendot1 = 3; break;                       // wenn Luftdruck- oder Luftfeuchtigkeitssensor -> Dezimalpunkt an Minuteneiner einschalten (keine Nachkommastelle)
          }
          saveflag = 1;                                               // Speicher-Flag setzen
          break;
        case 9:                                                       // wenn Konfigurationsmodus 9 (Sensornummer 2)
          sensorn2 ++;                                                // Sensornummer 2 erhhen
          if (sensorn2 > 31) sensorn2 = 0;                            // wenn Bereich berschritten -> auf Anfangswert setzen
          switch (sensorn2) {                                         // Sensornummer 2 auswerten und verzweigen
            case 0:         sendot2 = 4; break;                       // wenn kein Sensor -> Dezimalpunkt ausschalten
            case 1 ... 27:  sendot2 = 2; break;                       // wenn Temperatursensor -> Dezimalpunkt an Minutenzehner einschalten (eine Nachkommastelle)
            case 28 ... 31: sendot2 = 3; break;                       // wenn Luftdruck- oder Luftfeuchtigkeitssensor -> Dezimalpunkt an Minuteneiner einschalten (keine Nachkommastelle)
          }
          saveflag = 1;                                               // Speicher-Flag setzen
          break;
        case 10:                                                      // wenn Konfigurationsmodus 10 (Sensortext 1)
          if (!confsubm) sensort1[0] = next_char(sensort1[0]);        // wenn Konfigurations-Submen = 0 -> Zeichen 1 erhhen
          else sensort1[1] = next_char(sensort1[1]);                  // wenn Konfigurations-Submen = 1 -> Zeichen 2 erhhen
          saveflag = 1;                                               // Speicher-Flag setzen
          break;
        case 11:                                                      // wenn Konfigurationsmodus 11 (Sensortext 2)
          if (!confsubm) sensort2[0] = next_char(sensort2[0]);        // wenn Konfigurations-Submen = 0 -> Zeichen 1 erhhen
          else sensort2[1] = next_char(sensort2[1]);                  // wenn Konfigurations-Submen = 1 -> Zeichen 2 erhhen
          saveflag = 1;                                               // Speicher-Flag setzen
          break;
        case 12:                                                      // wenn Konfigurationsmodus 12 (Sensoroffset 1)
          sensoro1 ++;                                                // Sensoroffset 1 erhhen
          if (sensoro1 > 30) sensoro1 = -30;                          // wenn Bereich berschritten -> auf Anfangswert setzen
          saveflag = 1;                                               // Speicher-Flag setzen
          break;
        case 13:                                                      // wenn Konfigurationsmodus 13 (Sensoroffset 2)
          sensoro2 ++;                                                // Sensoroffset 2 erhhen
          if (sensoro2 > 30) sensoro2 = -30;                          // wenn Bereich berschritten -> auf Anfangswert setzen
          saveflag = 1;                                               // Speicher-Flag setzen
          break;
        case 20:                                                      // wenn Konfigurationsmodus 20 (Tag einstellen)
          xday ++;                                                    // Tag erhhen
          if (xday > 31) xday = 1;                                    // wenn Bereich berschritten -> auf Anfangswert setzen
          timeflag = 1;                                               // Zeit-nderungs-Flag setzen
          break;
        case 21:                                                      // wenn Konfigurationsmodus 21 (Monat einstellen)
          xmonth ++;                                                  // Monat erhhen
          if (xmonth > 12) xmonth = 1;                                // wenn Bereich berschritten -> auf Anfangswert setzen
          timeflag = 1;                                               // Zeit-nderungs-Flag setzen
          break;
        case 22:                                                      // wenn Konfigurationsmodus 22 (Jahr einstellen)
          xyear ++;                                                   // Jahr erhhen
          if (xyear > 99) xyear = YEAR;                               // wenn Bereich berschritten -> auf Anfangswert setzen
          timeflag = 1;                                               // Zeit-nderungs-Flag setzen
          break;
        case 23:                                                      // wenn Konfigurationsmodus 23 (Stunde einstellen)
          xhour ++;                                                   // Stunde erhhen
          if (xhour > 23) xhour = 0;                                  // wenn Bereich berschritten -> auf Anfangswert setzen
          timeflag = 1;                                               // Zeit-nderungs-Flag setzen
          break;
        case 24:                                                      // wenn Konfigurationsmodus 24 (Minute einstellen)
          xminute ++;                                                 // Minute erhhen
          if (xminute > 59) xminute = 0;                              // wenn Bereich berschritten -> auf Anfangswert setzen
          timeflag = 1;                                               // Zeit-nderungs-Flag setzen
          break;
      }
    }
                                                                      // --- verzgerte bertragung der Sensorwerte steuern ---
    if (senmemf1 && (dispmode != 2)) {                                // wenn Zwischenspeicher-Flag 1 gesetzt und aktuell keine Anzeige von Sensorwert 1
      senmemf1 = 0;                                                   // Zwischenspeicher-Flag 1 lschen
      for (i = 0; i < 4; i ++) senval1[i] = senbuf1[i];               // zwischengespeicherten Sensorwert 1 bernehmen
    }
    if (senmemf2 && (dispmode != 3)) {                                // wenn Zwischenspeicher-Flag 2 gesetzt und aktuell keine Anzeige von Sensorwert 2
      senmemf2 = 0;                                                   // Zwischenspeicher-Flag 1 lschen
      for (i = 0; i < 4; i ++) senval2[i] = senbuf2[i];               // zwischengespeicherten Sensorwert 2 bernehmen
    }
                                                                      // --- DCF77-Synchronisierung bei fester Synchronisierungszeit
    if (dcftout == 0) {                                               // wenn DCF77-Synchronisierungs-Timeout-Zhler abgelaufen
      dcftout = 255;                                                  // Timeout-Zhler stoppen
      dispactv = 1;                                                   // Display aktivieren
    }
    if (dcfsyn) {                                                     // wenn feste DCF-Synchronisierungszeit eingeschaltet
      if ((dispmode == 0) && (hour == 0) && (minute == 2) &&          // wenn Uhr noch nie synchronisiert, normale Anzeige, Zeit = 2 min nach Systemstart und nicht im Konfigurationsmodus
      (second == 0) && (syncstat == 0) && !confmode) {
        dcftout = DCFTOUT;                                            // DCF77-Synchronisierungs-Timeout setzen
        dispactv = 0;                                                 // Display deaktivieren
      }
      if ((dispmode == 0) && (hour == dcfhou) && (minute == 0) &&     // wenn normale Anzeige, eingestellte Synchronisierungsstunde erreicht und nicht im Konfigurationsmodus
      (second == 0) && !confmode) {
        dcftout = DCFTOUT;                                            // DCF77-Synchronisierungs-Timeout setzen
        dispactv = 0;                                                 // Display deaktivieren
      }
    }
                                                                      // --- verschiedene Timeout bearbeiten ---
    if (synctout == 0) {                                              // wenn Synchronisierungs-Timeout-Zhler abgelaufen
      synctout = 255;                                                 // Timeout-Zhler stoppen
      if (syncstat > 2) syncstat = 2;                                 // wenn Uhr synchronisiert war -> Status auf "Sync erforderlich" setzen
    }
    if (vertout == 0) {                                               // wenn Versions-Timeout-Zhler abgelaufen
      vertout = 255;                                                  // Versions-Timeout-Zhler stoppen
      dispmode = 0;                                                   // zur normalen Anzeige bergehen
    }
    if (sentout1 == 0) {                                              // wenn Sensor-1-Timeout-Zhler abgelaufen
      sentout1 = 255;                                                 // Sensor-1-Timeout-Zhler stoppen
    }
    if (sentout2 == 0) {                                              // wenn Sensor-2-Timeout-Zhler abgelaufen
      sentout2 = 255;                                                 // Sensor-2-Timeout-Zhler stoppen
    }
    if (usbtout == 0) {                                               // wenn USB-Timeout-Zhler abgelaufen
      if (usbmode) usb_sendstr(16);                                   // wenn USB im Konfigurationsmodus -> Text "Konfiguration beendet" senden
      usbtout = 255;                                                  // USB-Timeout-Zhler stoppen
      usbmode = 0;                                                    // USB in Monitormodus umschalten
      usbpos = 0;                                                     // USB-Datenpuffer lschen
    }
    if (boottout == 0) {                                              // wenn Bootloader-Timeout-Zhler abgelaufen
      boottout = 255;                                                 // Bootloader-Timeout-Zhler stoppen
      bootlcnt = 0;                                                   // Bootloader-Zeichen-Zhler lschen
    }
    if (conftout == 0) {                                              // wenn Konfigurations-Timeout-Zhler abgelaufen
      conftout = 255;                                                 // Konfigurations-Timeout-Zhler stoppen
      confmode = 0;                                                   // Konfigurationsmodus ausschalten
    }

    if (saveflag) {                                                   // wenn Daten im EEPROM gespeichert werden mssen
      saveflag = 0;                                                   // Speicher-Flag wieder lschen
      eep_write_byte(e_lang, lang);                                   // Sprache im EEPROM speichern
      eep_write_byte(e_brmode, brmode);                               // Helligkeitsmodus im EEPROM speichern
      eep_write_byte(e_brithr, brithr);                               // Helligkeitsschwelle im EEPROM speichern
      eep_write_byte(e_dcfsyn, dcfsyn);                               // Feste DCF77-Synchronisierungszeit im EEPROM speichern
      eep_write_byte(e_dcfhou, dcfhou);                               // DCF77-Synchronisierungsstunde im EEPROM speichern
      eep_write_byte(e_dcfinv, dcfinv);                               // DCF77-Invertierung im EEPROM speichern
      eep_write_byte(e_dcfpup, dcfpup);                               // DCF77-Pull-up im EEPROM speichern
      eep_write_byte(e_autost, autosumt);                             // Automatische Sommerzeit im EEPROM speichern
      eep_write_byte(e_timezo, timezone);                             // Zeitzone im EEPROM speichern
      eep_write_byte(e_sensn1, sensorn1);                             // Sensornummer 1 im EEPROM speichern
      eep_write_byte(e_sensn2, sensorn2);                             // Sensornummer 2 im EEPROM speichern
      eep_write_byte(e_senst1, sensort1[0]);                          // Sensortext 1, Zeichen 1 im EEPROM speichern
      eep_write_byte(e_senst1 + 1, sensort1[1]);                      // Sensortext 1, Zeichen 2 im EEPROM speichern
      eep_write_byte(e_senst2, sensort2[0]);                          // Sensortext 2, Zeichen 1 im EEPROM speichern
      eep_write_byte(e_senst2 + 1, sensort2[1]);                      // Sensortext 2, Zeichen 2 im EEPROM speichern
      eep_write_byte(e_senso1, sensoro1);                             // Sensoroffset 1 im EEPROM speichern
      eep_write_byte(e_senso2, sensoro2);                             // Sensoroffset 2 im EEPROM speichern
      eep_write_byte(e_sensac, sensacfg);                             // Automatische Sensor-Konfiguration im EEPROM speichern
      eep_write_byte(e_dateor, dateordr);                             // Datum-Reihenfolge im EEPROM speichern
      eep_write_byte(e_datemo, datemode);                             // Datum-Modus im EEPROM speichern
      eep_write_byte(e_senmo1, senmode1);                             // Sensor-1-Modus im EEPROM speichern
      eep_write_byte(e_senmo2, senmode2);                             // Sensor-2-Modus im EEPROM speichern
      eep_write_byte(e_inflag, initflag);                             // Initialisierungs-Flag im EEPROM speichern
    }
  }
}
