/*
 * Wohnraumuhr 2 mit 4-stelliger 7-Segment-Anzeige und optionaler 2-stelliger 16-Segment-Anzeige, Scott-Falk Hhn, "functions.h"
 * Verschiedene Funktionen
 */

#ifndef FUNCTIONS_H_
#define FUNCTIONS_H_

uint8_t eep_read_byte(uint16_t addr)                                  // Ein Byte von einer bestimmten EEPROM-Adresse lesen (aus AVR-GCC-Tutorial von http://www.mikrocontroller.net)
{                                                                     // addr: EEPROM-Adresse, Ergebnis: gelesenes Byte
  return eeprom_read_byte((uint8_t *) addr);                          // Byte ber die interne EEPROM-Routine lesen
}

void eep_write_byte(uint16_t addr, uint8_t val)                       // Ein Byte auf eine bestimmte EEPROM-Adresse schreiben (aus AVR-GCC-Tutorial von http://www.mikrocontroller.net)
{                                                                     // addr: EEPROM-Adresse, val: zu schreibendes Byte
  if (eeprom_read_byte((uint8_t *) addr) != val) {                    // wenn das Byte nicht mit dem EEPROM-Inhalt identisch ist
    eeprom_write_byte((uint8_t *) addr, val);                         // Byte ber die interne EEPROM-Routine schreiben
  }
}

PGM_P string(uint8_t strnr)                                           // Speicheradresse eines Strings ermitteln
{                                                                     // strnr: String-Nummer im Flash-Speicher, Ergebnis: Speicheradresse
  PGM_P p;                                                            // Speicheradresse auf Stringdaten im Flash
  if (lang) p = (PGM_P) pgm_read_word(&stre[strnr]);                  // String-Adresse ermitteln (wenn Englisch gewhlt)
  else p = (PGM_P) pgm_read_word(&strd[strnr]);                       // String-Adresse ermitteln (wenn Deutsch gewhlt)
  return p;                                                           // ermittelte Speicheradresse zurckgeben
}

void str_digit(uint8_t strnr, uint8_t digit)                          // Einen String direkt in den Anzeigespeicher kopieren (2 Zeichen)
{                                                                     // strnr: String-Nummer im Flash-Speicher, digit: Anzeigestelle (0-5)
  PGM_P p;                                                            // Speicheradresse auf Stringdaten im Flash
  p = string(strnr);                                                  // Speicheradresse vom bermittelten String ermitteln
  digits[digit] = pgm_read_byte(p ++);                                // erstes Zeichen des Strings in Anzeigespeicher kopieren
  digits[digit + 1] = pgm_read_byte(p);                               // zweites Zeichen des Strings in Anzeigespeicher kopieren
}

char next_char(char c)                                                // Nchstes Zeichen aus dem Zeichenvorrat ermitteln
{                                                                     // c: aktuelles Zeichen, Ergebnis: nchstes Zeichen
  uint8_t i;                                                          // Zhlervariable
  uint8_t p = 255;                                                    // Zeichenposition zunchst auf ungltigen Wert setzen
  for (i = 0; i < sizeof(charlist); i ++) {                           // Zeichen in der Liste suchen
    if (pgm_read_byte(&charlist[i]) == c) p = i;                      // wenn gefunden -> Position setzen
  }
  if (p < 255) p ++;                                                  // wenn Position gefunden -> auf nchste Position setzen
  else p = 0;                                                         // sonst auf Anfang setzen
  if (p >= sizeof(charlist) - 1) p = 0;                               // wenn Listenende erreicht -> Position auf Anfang setzen
  c = pgm_read_byte(&charlist[p]);                                    // nchstes Zeichen aus der Liste holen
  return c;
}

char conv_char(char c1, char c2)                                      // Ein Zeichen vom externen Zeichensatz in den internen Zeichensatz konvertieren
{                                                                     // c1: erstes Zeichen, c2: zweites Zeichen, Ergebnis: konvertiertes Zeichen
  uint8_t i;                                                          // Zhlervariable
  uint8_t cc = c1;                                                    // konvertiertes Zeichen auf erstes Zeichen voreinstellen
  for (i = 0; i < sizeof(conv_tab); i += 3) {                         // Konvertierungstabelle in Schritten von 3 Bytes durchsuchen
    if (cc == c1) {                                                   // wenn noch kein Zeichen gefunden
      if (pgm_read_byte(&conv_tab[i + 1]) == c1) {                    // wenn erstes Zeichen gefunden
        if (pgm_read_byte(&conv_tab[i + 2]) == c2) {                  // wenn zweites Zeichen gefunden
          cc = pgm_read_byte(&conv_tab[i]);                           // internes Zeichen lesen
        }
      }
    }
  }
  return cc;                                                          // konvertiertes Zeichen zurckgeben
}

void usb_send(char c)                                                 // Ein Zeichen ber die USB-Schnittstelle senden
{                                                                     // c: zu sendendes Zeichen
  while (!(UCSR0A & _BV(UDRE0)));                                     // warten bis Sendepuffer leer ist
  UDR0 = c;                                                           // Zeichen in Sendepuffer legen
}

void usb_sendcrlf(void)                                               // Zeilenende (CR + LF) ber die USB-Schnittstelle senden
{
  usb_send('\r');                                                     // CR senden
  usb_send('\n');                                                     // LF senden
}

void usb_sendconv(char c)                                             // Ein Zeichen vom internen in den externen Zeichensatz konvertieren und ber die USB-Schnittstelle senden
{                                                                     // c: zu sendendes Zeichen
  uint8_t i;                                                          // Zhlervariable
  uint8_t f = 0;                                                      // Konvertierungs-Flag
  for (i = 0; i < sizeof(conv_tab); i += 3) {                         // Konvertierungstabelle in Schritten von 3 Bytes durchsuchen
    if (pgm_read_byte(&conv_tab[i]) == c) {                           // wenn Zeichen gefunden
      usb_send(pgm_read_byte(&conv_tab[i + 1]));                      // durch externen Kode (erstes Zeichen) ersetzen
      usb_send(pgm_read_byte(&conv_tab[i + 2]));                      // durch externen Kode (zweites Zeichen) ersetzen
      f = 1;                                                          // Konvertierungs-Flag setzen
    }
  }
  if (!f) usb_send(c);                                                // wenn keine Konvertierung -> Zeichen direkt senden
}

void usb_sendnum(int8_t n)                                            // Eine 8-Bit-Integer-Zahl (-99 bis 99) ber die USB-Schnittstelle senden
{                                                                     // n: auszugebende Zahl
  if (n > 99) n = 99;                                                 // positive Zahl auf 99 begrenzen
  if (n < -99) n = -99;                                               // negative Zahl auf -99 begrenzen
  if (n < 0) {                                                        // wenn Zahl negativ
    usb_send('-');                                                    // Minuszeichen senden
    n = -n;                                                           // Zahl negieren
  }
  if (n > 9) usb_send(n / 10 + '0');                                  // Zehner ermitteln und senden
  usb_send(n % 10 + '0');                                             // Einer ermitteln und senden
}

void usb_sendnum2(int8_t n)                                           // Eine 8-Bit-Integer-Zahl (0 bis 99) zweistellig ber die USB-Schnittstelle senden
{                                                                     // n: auszugebende Zahl
  if (n > 99) n = 99;                                                 // positive Zahl auf 99 begrenzen
  if (n < 0) n = 0;                                                   // negative Zahl auf 0 setzen
  if (n > 9) usb_send(n / 10 + '0');                                  // wenn Zahl > 9 -> Zehner ermitteln und senden
  else usb_send('0');                                                 // wenn Zahl < 10 -> '0' senden
  usb_send(n % 10 + '0');                                             // Einer ermitteln und senden
}

void usb_sendhex(uint8_t n)                                           // Ein Byte hexadezimal 2-stellig ber die USB-Schnittstelle senden (fr Testzwecke, wird ansonsten nicht bentigt)
{                                                                     // n: auszugebender Wert
  uint8_t h, l;                                                       // h: Zwischenspeicher fr oberes Nibble, l: Zwischenspeicher fr unteres Nibble
  h = n / 16;                                                         // oberes Nibble ermitteln
  l = n % 16;                                                         // unteres Nibble ermitteln
  if (h > 9) usb_send(h + 55);                                        // wenn Wert ber 9 -> A-F ausgeben
  else usb_send(h + 48);                                              // sonst 0-9 ausgeben
  if (l > 9) usb_send(l + 55);                                        // wenn Wert ber 9 -> A-F ausgeben
  else usb_send(l + 48);                                              // sonst 0-9 ausgeben
}

void usb_sendstr(uint8_t strnr)                                       // Einen String ber die USB-Schnittstelle senden
{                                                                     // strnr: String-Nummer
  PGM_P p;                                                            // Speicheradresse auf Stringdaten im Flash
  p = string(strnr);                                                  // Speicheradresse vom bermittelten String ermitteln
  while (pgm_read_byte(p) > 0) {                                      // Schleife bis zum Endezeichen
    usb_send(pgm_read_byte(p));                                       // Zeichen senden
    if (pgm_read_byte(p) == '\r') usb_send('\n');                     // wenn Zeichen CR -> zustzlich LF senden
    p ++;                                                             // Zhler erhhen
  }
}

void usb_sendmenu(void)                                               // Konfigurations-Men ber die USB-Schnittstelle senden
{
  usb_sendstr(17);                                                    // Text "Konfiguration-Men" senden
  usb_sendstr(18);                                                    // Text "a - Sprache/Language: Deutsch" senden
  usb_sendstr(19);                                                    // Text "b - Helligkeitsmodus: " senden
  usb_sendstr(20 + brmode);                                           // aktuellen Helligkeitsmodus senden
  usb_sendstr(22);                                                    // Text "c - Helligkeitsschwelle: " senden
  usb_sendnum(brithr);                                                // aktuellen Schwellwert senden
  usb_sendcrlf();                                                     // CR/LF senden
  usb_sendstr(23);                                                    // Text "d - DCF77-Synchronisierungszeit: " senden
  usb_sendstr(24 + dcfsyn);                                           // aktuelle Synchronisierungszeit-Einstellung senden
  usb_sendstr(26);                                                    // Text "e - DCF77-Synchronisierungsstunde: " senden
  usb_sendnum(dcfhou);                                                // aktuelle Stunden-Einstellung senden
  usb_sendstr(59);                                                    // Text ":00" senden
  usb_sendstr(27);                                                    // Text "f - DCF77-Invertierung: " senden
  usb_sendstr(24 + dcfinv);                                           // aktuelle Invertierungs-Einstellung senden
  usb_sendstr(28);                                                    // Text "g - DCF77-Pullup: " senden
  usb_sendstr(24 + dcfpup);                                           // aktuelle Pullup-Einstellung senden
  usb_sendstr(29);                                                    // Text "h - Sensornummer 1: " senden
  if (sensorn1) {                                                     // wenn Sensornummer 1 gltig
    usb_sendnum(sensorn1);                                            // aktuelle Sensornummer 1 senden
    usb_sendcrlf();                                                   // CR/LF senden
  }
  else usb_sendstr(24);                                               // wenn Sensornummer 1 = 0 -> // Text "aus" senden
  usb_sendstr(30);                                                    // Text "i - Sensornummer 2: " senden
  if (sensorn2) {                                                     // wenn Sensornummer 2 gltig
    usb_sendnum(sensorn2);                                            // aktuelle Sensornummer 2 senden
    usb_sendcrlf();                                                   // CR/LF senden
  }
  else usb_sendstr(24);                                               // wenn Sensornummer 2 = 0 -> // Text "aus" senden
  usb_sendstr(31);                                                    // Text "j - Sensortext 1: " senden
  usb_sendconv(sensort1[0]);                                          // aktuellen Sensortext 1, erstes Zeichen konvertieren und senden
  usb_sendconv(sensort1[1]);                                          // aktuellen Sensortext 1, zweites Zeichen konvertieren und senden
  usb_sendcrlf();                                                     // CR/LF senden
  usb_sendstr(32);                                                    // Text "k - Sensortext 2: " senden
  usb_sendconv(sensort2[0]);                                          // aktuellen Sensortext 2, erstes Zeichen konvertieren und senden
  usb_sendconv(sensort2[1]);                                          // aktuellen Sensortext 2, zweites Zeichen konvertieren und senden
  usb_sendcrlf();                                                     // CR/LF senden
  usb_sendstr(33);                                                    // Text "l - Sensoroffset 1: " senden
  usb_sendnum(sensoro1);                                              // aktuellen Sensoroffset 1 senden
  usb_sendcrlf();                                                     // CR/LF senden
  usb_sendstr(34);                                                    // Text "m - Sensoroffset 2: " senden
  usb_sendnum(sensoro2);                                              // aktuellen Sensoroffset 2 senden
  usb_sendcrlf();                                                     // CR/LF senden
  usb_sendstr(35);                                                    // Text "n - Automatische Sensorkonfiguration: " senden
  usb_sendstr(24 + sensacfg);                                         // aktuelle Sensorkonfigurations-Einstellung senden
  usb_sendstr(36);                                                    // Text "o - Datum-Reihenfolge: " senden
  usb_sendstr(65 + dateordr);                                         // aktuelle Datum-Reihenfolge senden
  usb_sendstr(37);                                                    // Text "p - Datum anzeigen: " senden
  usb_sendstr(38 + datemode);                                         // aktuellen Datum-Modus senden
  usb_sendstr(41);                                                    // Text "q - Sensor 1 anzeigen: " senden
  usb_sendstr(38 + senmode1);                                         // aktuellen Sensor-1-Modus senden
  usb_sendstr(42);                                                    // Text "r - Sensor 2 anzeigen: " senden
  usb_sendstr(38 + senmode2);                                         // aktuellen Sensor-2-Modus senden
  usb_sendstr(43);                                                    // Text "s - Automatische Sommerzeit: " senden
  usb_sendstr(24 + autosumt);                                         // aktuelle Sommerzeit-Einstellung senden
  usb_sendstr(44);                                                    // Text "t - Tag: " senden
  usb_sendnum(xday);                                                  // zwischengespeicherten Tag senden
  usb_sendcrlf();                                                     // CR/LF senden
  usb_sendstr(45);                                                    // Text "u - Monat: " senden
  usb_sendnum(xmonth);                                                // zwischengespeicherten Monat senden
  usb_sendcrlf();                                                     // CR/LF senden
  usb_sendstr(46);                                                    // Text "v - Jahr: " senden
  usb_sendnum2(xyear);                                                // zwischengespeichertes Jahr senden
  usb_sendcrlf();                                                     // CR/LF senden
  usb_sendstr(47);                                                    // Text "w - Stunde: " senden
  usb_sendnum(xhour);                                                 // zwischengespeicherte Stunde senden
  usb_sendcrlf();                                                     // CR/LF senden
  usb_sendstr(48);                                                    // Text "x - Minute: " senden
  usb_sendnum2(xminute);                                              // zwischengespeicherte Minute senden
  usb_sendcrlf();                                                     // CR/LF senden
  if (timeflag) usb_sendstr(49);                                      // wenn Zeit-nderungs-Flag gesetzt -> Text "y - Zeit bernehmen" senden
  usb_sendstr(50);                                                    // Text "z - Konfiguration beenden" und Prompt senden
}

uint8_t calc_wday(uint8_t d, uint8_t m, uint8_t y)                    // Wochentags-Berechnung nach Gauss
{                                                                     // d: Tag, m: Monat, y: Jahr, Ergebnis: Wochentag (1-7)
  int8_t w;                                                           // Wochtag
  if (m <= 2) {                                                       // wenn Monat Januar oder Februar
    m += 12;                                                          // Monat ans Jahresende verschieben
    y --;                                                             // Jahr vermindern
  }
  w = (d + (13 * (m + 1)) / 5 + y + y / 4 + YEARH / 4 - 2 * YEARH) % 7; // Wochtagsformel, kleinster Wert entspricht Samstag
  if (w < 0) w += 7;                                                  // falls Ergebnis negativ war
  w --;                                                               // Wochentag korrigieren (kleinster Wert ist Montag)
  if (w < 1) w += 7;                                                  // und in den Bereich 1-7 legen
  return w;                                                           // Wochentag zurckgeben
}

uint8_t chk_summtime(void)                                            // Prfen, ob aktuelles Datum in der Sommerzeit liegt
{                                                                     // Ergebnis: 0 = Normalzeit, 1 = Sommerzeit
  uint8_t stime = 0;                                                  // Anfangswert auf Normalzeit setzen
  if ((month > 3) && (month < 10)) stime = 1;                         // wenn Monat April bis September -> Sommerzeit setzen
  if ((month == 3) && (day > 24)) {                                   // wenn Datum im Zeitraum vom 25. bis 31. Mrz
    if (calc_wday(day, month, year) == 7) stime = 1;                  // wenn Sonntag -> Sommerzeit setzen (Uhrzeit wird hier nicht bercksichtigt)
    else if ((day - calc_wday(day, month, year)) > 24) stime = 1;     // wenn nicht Sonntag -> wenn Tag nach der Umstellung -> Sommerzeit setzen
  }
  if ((month == 10) && (day > 24)) {                                  // wenn Datum im Zeitraum vom 25. bis 31. Oktober
    if (calc_wday(day, month, year) == 7) stime = 0;                  // wenn Sonntag -> Normalzeit setzen (Uhrzeit wird hier nicht bercksichtigt)
    else if ((day - calc_wday(day, month, year)) < 25) stime = 1;     // wenn nicht Sonntag -> wenn Tag vor der Umstellung -> Sommerzeit setzen
  }
  return stime;                                                       // Sommerzeit-Status zurckgeben
}

void asc_to_seg(void)                                                 // ASCII in Segment-Daten konvertieren
{
  uint8_t i;                                                          // Zhlervariable
  for (i = 0; i < 4; i ++) {                                          // 4 Stellen der 7-Segment-Anzeige bearbeiten
    segbuf[i] = 0x7f ^ pgm_read_byte(&seg7_list[digits[i] - 32]);     // Segment-Belegung a-g aus Liste holen, invertieren und im Segment-Puffer ablegen
  }
  for (i = 0; i < 2; i ++) {                                          // 2 Stellen der 16-Segment-Anzeigen bearbeiten
    segbuf[i * 2 + 4] = ~pgm_read_byte(&seg16_list[(digits[i + 4] - 32) * 2 + 1]); // Segment-Belegung a-h aus Liste holen, invertieren und im Segment-Puffer ablegen
    segbuf[i * 2 + 5] = ~pgm_read_byte(&seg16_list[(digits[i + 4] - 32) * 2]);     // Segment-Belegung k-u aus Liste holen, invertieren und im Segment-Puffer ablegen
  }
}

void int_to_asc(int16_t n, uint8_t k)                                 // Eine 16-Bit-Integer-Zahl in 4-stelligen ASCII-Wert konvertieren, Ergebnis wird im ASCII-Puffer gespeichert
{                                                                     // n: Temperatur-, Luftdruck- oder Luftfeuchtigkeitswert, k: Komma-Position (zur Unterdrckung fhrender Nullen)
  int8_t i;                                                           // Zhlervariable
  uint8_t s = 0;                                                      // Vorzeichen-Flag
  if (n < 0) {                                                        // wenn Wert negativ
    n = -n;                                                           // in Zweierkomplement wandeln
    s = 1;                                                            // Vorzeichen-Flag setzen
  }
  for (i = 0; i < 4; i ++) ascbuff[i] = 0;                            // zunchst ASCII-Puffer mit Nullwerten fllen
  if (n > 9999) n = 9999;                                             // Zahl auf 9999 begrenzen
  if (n > 999) {                                                      // wenn Zahl > 999
    ascbuff[0] = n / 1000;                                            // Tausenderstelle speichern
    n -= ascbuff[0] * 1000;                                           // Tausenderstelle von Restwert subtrahieren
  }
  if (n > 99) {                                                       // wenn Restwert > 99
    ascbuff[1] = n / 100;                                             // Hunderterstelle speichern
    n -= ascbuff[1] * 100;                                            // Hunderterstelle von Restwert subtrahieren
  }
  if (n > 9) {                                                        // wenn Restwert > 9
    ascbuff[2] = n / 10;                                              // Zehnerstelle speichern
    n -= ascbuff[2] * 10;                                             // Zehnerstelle von Restwert subtrahieren
  }
  ascbuff[3] = n;                                                     // Einerstelle speichern
  for (i = 0; i < 4; i ++) ascbuff[i] += '0';                         // alle Stellen in ASCII wandeln
  for (i = 0; i < k; i ++) {                                          // fhrende Nullen von links bis zur Komma-Position prfen
    if (ascbuff[i] == '0') ascbuff[i] = ' ';                          // wenn fhrende Null -> durch Leerzeichen ersetzen
    else break;                                                       // wenn keine fhrende Null -> Schleife abbrechen
  }
  if (s) {                                                            // wenn Vorzeichen-Flag gesetzt
    for (i = 2; i >= 0; i --) {                                       // alle Stellen von Komma-Position aus nach links prfen
      if (ascbuff[i] == ' ') {                                        // wenn Leerzeichen gefunden
        ascbuff[i] = '-';                                             // Minuszeichen direkt vor den Temperaturwert setzen
        break;                                                        // Schleife abbrechen
      }
    }
  }
}

int16_t asc_to_int(void)                                              // Einen 4-stelligen ASCII-Wert in eine 16-Bit-Integer-Zahl konvertieren, Wert wird aus dem Konvertierungspuffer entnommen
{                                                                     // Ergebnis: Temperatur- oder Luftfeuchtigkeitswert
  int8_t i;                                                           // Zhlervariable
  uint8_t s = 0;                                                      // Vorzeichen-Flag
  uint8_t e = 0;                                                      // Fehler-Flag
  int16_t n;                                                          // Ergebnis
  for (i = 0; i < 4; i ++) {                                          // alle Stellen nach Minuszeichen durchsuchen und in Zahlenwerte konvertieren
    if (ascbuff[i] == '-') {                                          // wenn Minuszeichen gefunden
      s = 1;                                                          // Vorzeichen-Flag setzen
      ascbuff[i] = '0';                                               // Vorzeichen durch "0" ersetzen
    }
    if (ascbuff[i] == ' ') ascbuff[i] = '0';                          // wenn Leerzeichen gefunden -> durch "0" ersetzen
    if ((ascbuff[i] >= '0') && (ascbuff[i] <= '9')) ascbuff[i] -= '0'; // wenn Ziffer gefunden -> in Zahlenwert konvertieren
    else e = 1;                                                       // wenn keine Ziffer gefunden -> Fehler-Flag setzen
  }
  if (!e) {                                                           // wenn kein Fehler
    n = 1000 * ascbuff[0] + 100 * ascbuff[1] + 10 * ascbuff[2] + ascbuff[3]; // gesamten Zahlenwert ermitteln
    if (s) n = -n;                                                    // wenn Vorzeichen-Flag -> Zweierkomplement bilden
  }
  else n = 0x8000;                                                    // wenn Fehler -> Fehlerwert setzen
  return n;                                                           // Integer-Wert zurckgeben
}

int8_t asc_to_int8(void)                                              // Einen 3-stelligen ASCII-Wert in eine 8-Bit-Integer-Zahl konvertieren, Wert wird aus USB-Puffer entnommen
{                                                                     // Ergebnis: 8-Bit-Integer-Wert
  int8_t i;                                                           // Zhlervariable
  uint8_t s = 0;                                                      // Vorzeichen-Flag
  uint8_t e = 0;                                                      // Fehler-Flag
  int8_t n = 0;                                                       // Ergebnis
  if (usbpos) {                                                       // wenn Zeichen im USB-Datenpuffer
    for (i = 0; i < usbpos; i ++) {                                   // Puffer vom Anfang bis zur aktuellen Pufferposition durchsuchen
      if (usbdata[i] == '-') {                                        // wenn Minuszeichen gefunden
        s = 1;                                                        // Vorzeichen-Flag setzen
        usbdata[i] = '0';                                             // Vorzeichen durch "0" ersetzen
      }
      if ((usbdata[i] >= '0') && (usbdata[i] <= '9')) usbdata[i] -= '0'; // wenn Ziffer gefunden -> in Zahlenwert konvertieren
      else e = 1;                                                     // wenn keine Ziffer gefunden -> Fehler-Flag setzen
    }
  }
  else e = 1;                                                         // wenn keine Zeichen im USB-Datenpuffer -> Fehler-Flag setzen
  if (!e) {                                                           // wenn kein Fehler
    switch (usbpos) {                                                 // Anzahl der Zeichen im USB-Datenpuffer auswerten und verzweigen
      case 1:                                                         // wenn 1 Zeichen im USB-Datenpuffer
        n = usbdata[0];                                               // Zahlenwert ermitteln
        break;
      case 2:                                                         // wenn 2 Zeichen im USB-Datenpuffer
        n = 10 * usbdata[0] + usbdata[1];                             // gesamten Zahlenwert ermitteln
        break;
      case 3:                                                         // wenn 3 Zeichen im USB-Datenpuffer
        n = 100 * usbdata[0] + 10 * usbdata[1] + usbdata[2];          // gesamten Zahlenwert ermitteln
        break;
    }
    if (s) n = -n;                                                    // wenn Vorzeichen-Flag -> Zweierkomplement bilden
  }
  else n = 0x80;                                                      // wenn Fehler -> Fehlerwert setzen
  return n;                                                           // Ergebnis zurckgeben
}

void byte_to_hex(uint8_t n, uint8_t d)                                // Ein Byte Hexadezimal 2-stellig auf Stundenanzeige ausgeben (fr Testzwecke, wird ansonsten nicht bentigt)
{                                                                     // n: auszugebender Wert, d: auszugebende Stelle (0, 2, 4)
  uint8_t h, l;                                                       // h: Zwischenspeicher fr oberes Nibble, l: Zwischenspeicher fr unteres Nibble
  h = n / 16;                                                         // oberes Nibble ermitteln
  l = n % 16;                                                         // unteres Nibble ermitteln
  if (h > 9) digits[d] = h + 55;                                      // wenn Wert ber 9 -> A-F ausgeben
  else digits[d] = h + 48;                                            // sonst 0-9 ausgeben
  if (l > 9) digits[d + 1] = l + 55;                                  // wenn Wert ber 9 -> A-F ausgeben
  else digits[d + 1] = l + 48;                                        // sonst 0-9 ausgeben
}

void scroll_text(void)                                                // Scrolltext starten
{
  scrcnt = 0;                                                         // Scrollzhler zurcksetzen
  scrpos = 0;                                                         // Scrollposition auf String-Anfang setzen
  scrtxt = pgm_read_byte(&scrtab[confmode - 1]);                      // Scrolltext-Nummer aus Tabelle holen
}

#endif /* FUNCTIONS_H_ */
