; ================================================
; WeckUhr mit AT90S4434, AT90S8535 oder ATmega8535
; ================================================
;
; Version 1.05, letzte Bearbeitung am 04.03.2007
;		- Verbesserung der DCF/HBG-Dekodierung durch zustzli-
;		  chen Zwischenspeicher, jetzt mssen 3 (anstatt 2 wie
;		  bisher) Minutenzyklen genau stimmen, damit die Uhr
;		  synchronisiert wird
;
; Version 1.04, letzte Bearbeitung am 06.05.2005
;		- Anpassung an das HBG75 Signal (Schweizer Zeitzeichen-
;		  sender), die Uhr kann jetzt wahlweise mit DCF77 oder
;		  HBG75 Empfnger betrieben werden
;		- Verbesserung der DCF/HBG-Auswertung
;
; Version 1.03, letzte Bearbeitung am 06.05.2005
;		- Normal- und Inversbetrieb der Ausgnge ist jetzt in
;		  einer Software vereint, der Inversbetrieb wird durch
;		  setzen des Adresse invfl1 auf 0xff eingeschaltet
;
; Version 1.02, letzte Bearbeitung am 17.03.2004
;
; ----------------------
; Belegung der I/O-Ports
; ----------------------
;
; A0: Ausgang LED 2 - Segment A (0=ein)
; A1: Ausgang LED 2 - Segment B (0=ein)
; A2: Ausgang LED 2 - Segment C (0=ein)
; A3: Ausgang LED 2 - Segment D (0=ein)
; A4: Ausgang LED 2 - Segment E (0=ein)
; A5: Ausgang LED 2 - Segment F (0=ein)
; A6: Ausgang LED 2 - Segment G (0=ein)
; A7: Ausgang LED 2 - Segment P (0=ein)
;
; B0: Ausgang LED 1 - Segment B (0=ein)
; B1: Ausgang LED 1 - Segment C (0=ein)
; B2: Ausgang LED 1 - Segment A,D,E,G (0=ein)
; B3: Eingang DCF77-Empfnger (mit Pullup)
; B4: Eingang Taste 1 - Stunden (mit Pullup)
; B5: Eingang Taste 2 - Minuten (mit Pullup)
; B6: Eingang Taste 3 - Alarm 1 (mit Pullup)
; B7: Eingang Taste 4 - Alarm 2 (mit Pullup)
;
; C0: Ausgang LED 3 - Segment A (0=ein)
; C1: Ausgang LED 3 - Segment B (0=ein)
; C2: Ausgang LED 3 - Segment C (0=ein)
; C3: Ausgang LED 3 - Segment D (0=ein)
; C4: Ausgang LED 3 - Segment E (0=ein)
; C5: Ausgang LED 3 - Segment F (0=ein)
; C6: Ausgang LED 3 - Segment G (0=ein)
; C7: Ausgang Tongenerator (1=ein)
;
; D0: Ausgang LED 4 - Segment A (0=ein)
; D1: Ausgang LED 4 - Segment B (0=ein)
; D2: Ausgang LED 4 - Segment C (0=ein)
; D3: Ausgang LED 4 - Segment D (0=ein)
; D4: Ausgang LED 4 - Segment E (0=ein)
; D5: Ausgang LED 4 - Segment F (0=ein)
; D6: Ausgang LED 4 - Segment G (0=ein)
; D7: Ausgang LED 4 - Segment P (0=ein)
;
; ------------------
; Belegung der Timer
; ------------------
;
; Timer1:	Zeitzhlung 10ms (IRQ Compare A)
;
; -----------------------------------
; Standard-Definitionen fr AT90S4434
; -----------------------------------
;
.include "4434def.inc"
;
; ------------------------
; Speicheradressen im SRAM
; ------------------------
;
.dseg
.org	0x60
;
pcount:	.byte	1	  ;Vorteiler (0-99) zhlt 10ms-Interrupts
second:	.byte	1	  ;Sekundenzhler (0-59)
minute:	.byte	1	  ;Minutenzhler (0-59)
hour:	.byte	1	  ;Stundenzhler (0-23)
flags1:	.byte	1	  ;letzter Tastenstatus
			  ;Bit 0: Taste Std gedrckt  \
			  ;Bit 1: Taste Min gedrckt   \ wird von Timer
			  ;Bit 2: Taste Al1 gedrckt   / Int gesteuert
			  ;Bit 3: Taste Al2 gedrckt  /
flags2:	.byte	1	  ;gltiger Tastenstatus
			  ;Bit 0: Taste Std entprellt \
			  ;Bit 1: Taste Min entprellt  \ wird von Timer
			  ;Bit 2: Taste Al1 entprellt  / Int gesteuert
			  ;Bit 3: Taste Al2 entprellt /
flags3:	.byte	1	  ;Bearbeitungszustand der Tasten
			  ;Bit 0: Taste Std bearbeitet \
			  ;Bit 1: Taste Min bearbeitet  \wird von Timer
			  ;Bit 2: Taste Al1 bearbeitet  /Int gelscht
			  ;Bit 3: Taste Al2 bearbeitet /
flags4:	.byte	1	  ;DCF77-Flags
			  ;Bit 0: akueller DCF77-Zustand
			  ;Bit 1: vorheriger DCF77-Zustand
			  ;Bit 2: Uhr wurde mind. einmal synchronisiert
			  ;Bit 3: Beginn einer neuen Minute (wird durch
			  ;	  Timer gesetzt und vom Hauptprogramm
			  ;	  zurckgesetzt)
flags5:	.byte	1	  ;Alarm-Flags
			  ;Bit 0: Alarm 1 aktiviert
			  ;Bit 1: Alarm 2 aktiviert
dcfcnt:	.byte	1	  ;DCF77-Zhler, Impuls-Zhlung (Zyklen 10ms)
dcftab:	.byte	6	  ;DCF77-Tabelle (6 Byte), eingehende Bits
			  ;werden in Byte 5, Bit 7 eingeschoben und
			  ;alle weiteren Bits um eine Position weiter-
			  ;gerckt, Byte/Bit:
			  ;0/6-7: Zeitzone (6=1 Sommer, 7=1 Winter)
			  ;1/2-5: Minute Einer
			  ;1/6-7: Minute Zehner LSB
			  ;2/0:   Minute Zehner MSB
			  ;2/1:   Prfbit 1 (Minuten)
			  ;2/2-5: Stunde Einer
			  ;2/6-7: Stunde Zehner
			  ;3/0:   Prfbit 2 (Stunden)
			  ;3/1-4: Kalendertag Einer
			  ;3/5-6: Kalendertag Zehner
			  ;3/7:   Wochentag LSB
			  ;4/0-1: Wochentag MSB
			  ;4/2-5: Monat Einer
			  ;4/6:   Monat Zehner
			  ;4/7:   Jahr Einer LSB
			  ;5/0-2: Jahr Einer MSB
			  ;5/3-6: Jahr Zehner
			  ;5/7:   Prfbit 3 (Datum)
dcbuf1:	.byte	12	  ;DCF-Puffer 1, aktuelle decodierte Uhrzeit
			  ;(12 Bytes)
			  ;Byte  0: Zeitzone
			  ;Byte  1: Minute Einer
			  ;Byte  2: Minute Zehner
			  ;Byte  3: Stunde Einer
			  ;Byte  4: Stunde Zehner
			  ;Byte  5: Kalendertag Einer
			  ;Byte  6: Kalendertag Zehner
			  ;Byte  7: Wochentag
			  ;Byte  8: Monat Einer
			  ;Byte  9: Monat Zehner
			  ;Byte 10: Jahr Einer
			  ;Byte 11: Jahr Zehner
dcbuf2:	.byte	12	  ;DCF-Puffer 2, vorherige decodierte Uhrzeit
			  ;+1 Minute, Aufbau wie dcbuf1 (12 Bytes)
dcbuf3:	.byte	12	  ;DCF-Puffer 3, vorherige decodierte Uhrzeit
			  ;+2 Minuten, Aufbau wie dcbuf1 (12 Bytes)
mincnt:	.byte	1	  ;Minuten-Eingabe (1 Byte)
stdcnt:	.byte	1	  ;Stunden-Eingabe (1 Byte)
al1min:	.byte	1	  ;Alarm 1 - Minuten (0-59)
al1hou:	.byte	1	  ;Alarm 1 - Stunden (0-23)
al2min:	.byte	1	  ;Alarm 2 - Minuten (0-59)
al2hou:	.byte	1	  ;Alarm 2 - Stunden (0-23)
dimode:	.byte	1	  ;Anzeige-Modus:
			  ;0: normale Zeitanzeige
			  ;1: Sekundenanzeige
			  ;2: Alarmzeit 1
			  ;3: Alarmzeit 2
			  ;4: Versionsnummer-Anzeige
ditime:	.byte	1	  ;Zhler fr Anzeige-Timeout, zhlt rckwrts
			  ;auf 0 im Sekundentakt
;
; ---------------------
; Register-Definitionen
; ---------------------
;
.def	invflg=	r5	;Invers-Flag fr die LED-Ausgnge,
			;0= Normalbetrieb, 0xff= Inversbetrieb
.def	dcfimp=	r6	;DCF77-Zhler, zhlt die empfangenen Impulse
			;innerhalb einer Minute (0-59)
;
; --------------------
; Konstanten festlegen
; --------------------
;
.equ	tm1end=	40000	  ;Timer1 Endwert (entspricht 100 Interrupts/s
			  ;oder Zykluszeit 10ms)
;
; ---------------------------------
; Prgrammbeginn, Interrupt-Vektoren
; ---------------------------------
;
.cseg
.org	0x0000
;
reset:	rjmp	start			;Programmstart
exint0:	reti				;nicht genutzt
exint1:	reti				;nicht genutzt
t2comp:	reti				;nicht genutzt
t2ovfl:	reti				;nicht genutzt
t1icap:	reti				;nicht genutzt
t1coma:	rjmp	timint			;Multiplexer und Zeitzhler
t1comb:	reti				;nicht genutzt
t1ovfl:	reti				;nicht genutzt
t0ovfl:	reti				;nicht genutzt
spiint:	reti				;nicht genutzt
uartrx:	reti				;nicht genutzt
uartem:	reti				;nicht genutzt
uarttx:	reti				;nicht genutzt
adccom:	reti				;nicht genutzt
eeprom:	reti				;nicht genutzt
ancomp:	reti				;nicht genutzt
;
;	Initialisierung von Watchdog und RAM
;
start:	cli				;Interrupts sperren
	wdr				;Watchdog zurcksetzen
	ldi	r16,1<<wde		;Watchdog aktivieren, 15ms
	out	wdtcr,r16		;ausgeben
	ldi	r16,low(ramend)		;Stackpointer setzen
	out	spl,r16
	ldi	r16,high(ramend)
	out	sph,r16
	clr	r16
	ldi	yl,0x60			;Beginn des RAM-Bereiches Low
	ldi	yh,0			;Beginn des RAM-Bereiches High
ramclr:	st	y+,r16			;Speicherzelle lschen
	cpi	yl,low(ramend+1)	;RAM-Ende (Low) erreicht?
	brne	ramclr			;nein -> Schleife
	cpi	yh,high(ramend+1)	;RAM-Ende (High) erreicht?
	brne	ramclr			;nein -> Schleife
	clr	dcfimp			;DCF-Impulszhler lschen
;
;	- Flag fr Inversbetrieb lesen
;
	ldi	zl,low(2*invfl1)
	ldi	zh,high(2*invfl1)	;Zeiger auf Invers-Flag
	lpm				;Flag lesen
	mov	invflg,r0		;und in Flag-Register kopieren
;
;	- Initialisierung von I/O und Interrupt
;
	ldi	r16,0b00000111		;PortB 3-7 als Eingang
	out	ddrb,r16		;ausgeben
	ldi	r16,0b11111111		;PortB 3-7 mit Pullup, LEDs aus
	mov	r17,invflg		;Invers-Flag holen
	andi	r17,0x07		;LED-Ports filtern
	eor	r16,r17			;LED-Ports invertieren
	out	portb,r16
	ser	r16
	out	ddra,r16		;PortA alles Ausgnge
	out	ddrc,r16		;PortC alles Ausgnge
	out	ddrd,r16		;PortD alles Ausgnge
	eor	r16,invflg		;LED-Ports invertieren
	out	porta,r16		;PortA alle LEDs aus
	out	portd,r16		;PortD alle LEDs aus
	ldi	r16,0b01111111
	mov	r17,invflg		;Invers-Flag holen
	andi	r17,0x7f		;LED-Ports filtern
	eor	r16,r17			;LED-Ports invertieren
	out	portc,r16		;PortC alle LEDs aus, Lsp aus
	ldi	r16,high(tm1end)	;Timer1 Endwert1 H
	out	ocr1ah,r16		;ausgeben
	ldi	r16,low(tm1end)		;Timer1 Endwert1 L
	out	ocr1al,r16		;ausgeben
	clr	r16			;Timer1 kein PWM,
	out	tccr1a,r16		;kein Compare-Out
	ldi	r16,(1<<ctc1)+(1<<cs10)	;Timer1 Lschen nach Vergleich
	out	tccr1b,r16		;und Vorteiler=1 setzen
;
;	- gespeicherte Werte aus EEPROM holen
;
	ldi	r17,high(ealsta)
	ldi	r16,low(ealsta)		;EEPROM-Adresse Alarm-Status
	rcall	eeread			;EEPROM lesen
	sts	flags5,r0		;Alarmstatus speichern
	ldi	r16,low(ea1min)		;EEPROM-Adresse Alarm 1 Minuten
	rcall	eeread			;EEPROM lesen
	sts	al1min,r0		;Alarm 1 Minuten speichern
	ldi	r16,low(ea1hou)		;EEPROM-Adresse Alarm 1 Stunden
	rcall	eeread			;EEPROM lesen
	sts	al1hou,r0		;Alarm 1 Stunden speichern
	ldi	r16,low(ea2min)		;EEPROM-Adresse Alarm 2 Minuten
	rcall	eeread			;EEPROM lesen
	sts	al2min,r0		;Alarm 2 Minuten speichern
	ldi	r16,low(ea2hou)		;EEPROM-Adresse Alarm 1 Stunden
	rcall	eeread			;EEPROM lesen
	sts	al2hou,r0		;Alarm 2 Stunden speichern
;
;	- Interrupt aktivieren
;
	clr	r16
	out	gimsk,r16		;externe Interrupts sperren
	ldi	r16,1<<ocie1a
	out	timsk,r16		;Timer1 Vergleich1 freigeben
	sei
;
;	- Versionsnummer anzeigen
;
	ldi	r16,4			;Anzeigemodus fr Versions-
	sts	dimode,r16		;nummer setzen
	ldi	r16,2			;fr 2 Sekunden
	sts	ditime,r16		;Versionsnummer anzeigen
;
;	-------------------------------------------------
;	Hauptprogramm, DCF77 auswerten und Uhrzeit setzen
;	-------------------------------------------------
;
main:	lds	r16,flags4		;DCF77-Flags holen
	bst	r16,3			;Beginn einer neuen Minute?
	brtc	ma0100			;nein -> weiter
	andi	r16,0xf7		;Minuten-Flag lschen
	sts	flags4,r16		;DCF77-Flags wieder speichern
	rcall	mdcprc			;DCF77-Auswertung
	tst	r19			;DCF77 ok?
	brne	ma0100			;nein -> weiter
	rcall	mtsync			;sonst Uhr synchronisieren
;
;	-----------------------------------------
;	Hauptprogramm, Anzeige-Timeout bearbeiten
;	-----------------------------------------
;
ma0100:	lds	r16,ditime		;Anzeige-Timeout-Zhler holen
	tst	r16			;Zeit abgelaufen?
	brne	ma0200			;nein -> weiter
	lds	r16,dimode		;sonst Anzeigemodus holen
	cpi	r16,2			;Modus>=2?
	brcs	ma0160			;ja -> berspringen
	brne	ma0120			;Modus=2? nein -> weiter testen
	ldi	r17,high(ea1min)	;sonst gespeicherte
	ldi	r16,low(ea1min)		;Al1-Minuten
	rcall	eeread			;aus EEPROM holen
	lds	r18,al1min		;aktuelle Al1-Minuten holen
	cp	r0,r18			;wurden Werte gendert?
	breq	ma0110			;nein -> weiter
	rcall	eewrit			;neuen Wert in EEPROM schreiben
ma0110:	ldi	r16,low(ea1hou)		;Al1-Stunden
	rcall	eeread			;aus EEPROM holen
	lds	r18,al1hou		;aktuelle Al1-Stunden holen
	cp	r0,r18			;wurden Werte gendert?
	breq	ma0150			;nein -> Mode=0 (Zeitanzeige)
	rjmp	ma0140			;neuen Wert in EEPROM schreiben
ma0120:	cpi	r16,3			;Modus=3 (Al2)?
	brne	ma0150			;nein -> weiter
	ldi	r17,high(ea2min)	;sonst gespeicherte
	ldi	r16,low(ea2min)		;Al2-Minuten
	rcall	eeread			;aus EEPROM holen
	lds	r18,al2min		;aktuelle Al2-Minuten holen
	cp	r0,r18			;wurden Werte gendert?
	breq	ma0130			;nein -> weiter
	rcall	eewrit			;neuen Wert in EEPROM schreiben
ma0130:	ldi	r16,low(ea2hou)		;Al2-Stunden
	rcall	eeread			;aus EEPROM holen
	lds	r18,al2hou		;aktuelle Al2-Stunden holen
	cp	r0,r18			;wurden Werte gendert?
	breq	ma0150			;nein -> weiter
ma0140:	rcall	eewrit			;neuen Wert in EEPROM schreiben
ma0150:	clr	r16			;auf normale Zeitanzeige
	sts	dimode,r16		;zurcksetzen
;
ma0160:	ldi	r17,high(ealsta)
	ldi	r16,low(ealsta)		;gespeicherten Alarm-Status
	rcall	eeread			;holen
	lds	r18,flags5		;aktuellen Alarmstatus holen
	cp	r0,r18			;wurde Wert gendert?
	breq	ma0200			;nein -> weiter
	rcall	eewrit			;neuen Wert in EEPROM schreiben
;
;	-------------------------------------
;	Hauptprogramm, Weckzeiten vergleichen
;	-------------------------------------
;
ma0200:	lds	r16,flags4		;DCF77-Flags holen
	bst	r16,2			;Uhr schon synchronisiert?
	brts	ma0210			;ja -> Alarm testen
	rjmp	ma1000			;sonst weiter zur Anzeige
ma0210:	lds	r16,dimode		;Anzeige-Modus holen
	cpi	r16,2			;Modus<2?
	brcs	ma0220			;ja -> Alarm testen
	rjmp	ma1000			;sonst weiter zur Anzeige
ma0220:	lds	r16,flags5		;Alarm-Flags holen
	bst	r16,0			;Alarm 1 aktiviert?
	brtc	ma0240			;nein -> Alarm 2 testen
	lds	r16,minute		;Minuten holen
	lds	r17,al1min		;Alarm 1 Minuten holen
	cp	r16,r17			;Alarmzeit erreicht?
	breq	ma0230			;ja -> Stunden vergleichen
	rjmp	ma1000			;sonst weiter zur Anzeige
ma0230:	lds	r16,hour		;Stunden holen
	lds	r17,al1hou		;Alarm 1 Stunden holen
	cp	r16,r17			;Alarmzeit erreicht?
	breq	ma0300			;ja -> Signal einschalten
	rjmp	ma1000			;sonst weiter zur Anzeige
;
ma0240:	bst	r16,1			;Alarm 2 aktiviert?
	brts	ma0250			;ja -> Alarm testen
	rjmp	ma1000			;sonst weiter zur Anzeige
ma0250:	lds	r16,minute		;Minuten holen
	lds	r17,al2min		;Alarm 2 Minuten holen
	cp	r16,r17			;Alarmzeit erreicht?
	breq	ma0260			;ja -> Stunden vergleichen
	rjmp	ma1000			;sonst weiter zur Anzeige
ma0260:	lds	r16,hour		;Stunden holen
	lds	r17,al2hou		;Alarm 2 Stunden holen
	cp	r16,r17			;Alarmzeit erreicht?
	brne	ma1000			;nein -> weiter zur Anzeige
;
;	----------------------------------------
;	Hauptprogramme, Soundausgabe: Tonfolge
;	3 Tne mit 80ms Ton, 80ms Pause
;	----------------------------------------
;
ma0300:	clr	r16			;Anzeige-Modus auf normale
	sts	dimode,r16		;Zeitanzeige setzen
	rcall	clkout			;Uhrzeit anzeigen
	ldi	r21,6			;6 Zyklen
	ldi	r20,0x80		;Bitmaske fr Tongenerator-Bit
	ldi	r19,8			;Zeitabstand 80ms
	clr	r18			;Anfangswert fr Vergleich
;
ma0310:	add	r18,r19			;neuen Vergleichswert setzen
	in	r16,portc		;PortC lesen
	eor	r16,r20			;Tongenerator-Bit wechseln
	out	portc,r16		;PortC aktualisieren
;
ma0320:	lds	r16,flags2		;Tastenflags holen
	andi	r16,0x0f		;wurde eine Taste gedrckt?
	brne	ma0400			;ja - Sound abbrechen
	lds	r16,pcount		;Vorteiler holen
	cp	r16,r18			;Vergleichwert erreicht?
	brne	ma0320			;nein -> warten
	dec	r21			;alle Zyklen abgearbeitet?
	brne	ma0310			;nein -> Schleife
;
ma0330:	rcall	clkout			;Uhrzeit anzeigen
	lds	r16,flags2		;Tastenflags holen
	andi	r16,0x0f		;wurde eine Taste gedrckt?
	brne	ma0400			;ja - Sound abbrechen
	lds	r16,pcount		;Vorteiler holen
	tst	r16			;beginnt neue Sekunde?
	brne	ma0330			;nein -> warten
	rjmp	main
;
ma0400:	cbi	portc,7			;Tongenerator-Bit lschen
	ldi	r16,0x0f		;alle Tasten als bearbeitet
	sts	flags3,r16		;markieren
	clr	r16			;alle Alarme
	sts	flags5,r16		;deaktivieren
;
;	----------------------------------------------------
;	Hauptprogramm, Anzeige entsprechend Mode,
;	Tastenabfrage und Ausfhrung entsprechender Aktionen
;	----------------------------------------------------
;
ma1000:	lds	r16,dimode		;Anzeigemodus holen
	tst	r16			;normale Zeitanzeige?
	brne	ma2000			;nein -> weiter testen
	rcall	clkout			;aktuelle Zeitanzeige
	rcall	ktest1			;Taste 1 (Std) gedrckt?
	breq	ma1100			;nein -> weiter
	ldi	r16,1			;sonst zur Sekundenanzeige
	sts	dimode,r16		;wechseln
	rjmp	main
ma1100:	rcall	ktest2			;Taste 2 (Min) gedrckt?
	breq	ma1200			;nein -> weiter
	ldi	r16,1			;sonst zur Sekundenanzeige
	sts	dimode,r16		;wechseln
	rjmp	main
ma1200:	rcall	ktest3			;Taste 3 (Al1) gedrckt?
	breq	ma1300			;nein -> weiter
ma1210:	lds	r16,flags5		;Alarm-Flags holen
	tst	r16			;Alarm aktiviert?
	breq	ma1220			;nein -> weiter
	clr	r16			;sonst Alarme deaktivieren
	sts	flags5,r16		;und speichern
	rjmp	main
ma1220:	inc	r16			;Alarm 1 aktivieren
	sts	flags5,r16		;und speichern
	ldi	r16,2			;zur Alarm 1 Anzeige
	sts	dimode,r16		;wechseln
	ldi	r16,6			;Timeout-Zhler auf 6
	sts	ditime,r16		;Sekunden setzen
	rjmp	main
ma1300:	rcall	ktest4			;Taste 4 (Al2) gedrckt?
	breq	ma1400			;nein -> weiter
ma1310:	lds	r16,flags5		;Alarm-Flags holen
	tst	r16			;Alarm aktiviert?
	breq	ma1320			;nein -> weiter
	clr	r16			;sonst Alarme deaktivieren
	sts	flags5,r16		;und speichern
	rjmp	main
ma1320:	ldi	r16,2			;Alarm 2 aktivieren
	sts	flags5,r16		;und speichern
	ldi	r16,3			;zur Alarm 1 Anzeige
	sts	dimode,r16		;wechseln
	ldi	r16,6			;Timeout-Zhler auf 6
	sts	ditime,r16		;Sekunden setzen
ma1400:	rjmp	main
;
ma2000:	dec	r16			;Sekundenanzeige?
	brne	ma3000			;nein -> weiter testen
	rcall	secout			;Sekundenanzeige
	rcall	ktest1			;Taste 1 (Std) gedrckt?
	breq	ma2100			;nein -> weiter
	clr	r16			;sonst in normale Zeitanzeige
	sts	dimode,r16		;wechseln
ma2100:	rcall	ktest2			;Taste 2 (Min) gedrckt?
	breq	ma2200			;nein -> weiter
	clr	r16			;sonst in normale Zeitanzeige
	sts	dimode,r16		;wechseln
	rjmp	main
ma2200:	rcall	ktest3			;Taste 3 (Al1) gedrckt?
	breq	ma2300			;nein -> weiter
	clr	r16			;zur normalen Zeitanzeige
	sts	dimode,r16		;wechseln
	rjmp	ma1210			;weiter wie bei Modus 0, Taste3
ma2300:	rcall	ktest4			;Taste 4 (Al2) gedrckt?
	breq	ma2400			;nein -> weiter
	clr	r16			;zur normalen Zeitanzeige
	sts	dimode,r16		;wechseln
	rjmp	ma1310			;weiter wie bei Modus 0, Taste4
ma2400:	rjmp	main
;
ma3000:	dec	r16			;Alarm 1 Anzeige?
	breq	ma3010			;ja -> bearbeiten
	rjmp	ma4000			;sonst weiter testen
ma3010:	rcall	al1out			;Alarm 1 anzeigen
	rcall	ktest1			;Taste 1 (Std) gedrckt?
	breq	ma3100			;nein -> weiter
	lds	r16,flags3		;bearbeitete Tasten holen
	bst	r16,1			;Taste 2 (Min) zustzlich gedr?
	brtc	ma3030			;nein -> weiter
	lds	r16,al1min		;sonst Alarm 1 Minuten holen
	subi	r16,5			;um 5 Minuten verringern
	brcc	ma3020			;berlauf?
	ldi	r16,55			;ja -> korrigieren
ma3020:	sts	al1min,r16		;Alarm 1 Minuten speichern
	ldi	r16,6			;Timeout-Zhler auf 6
	sts	ditime,r16		;Sekunden setzen
	rjmp	main
ma3030:	lds	r16,al1hou		;sonst Alarm 1 Stunden holen
	inc	r16			;erhhen
	cpi	r16,24			;Endwert erreicht?
	brcs	ma3040			;nein -> weiter
	clr	r16			;sonst zurcksetzen
ma3040:	sts	al1hou,r16		;und wieder speichern
	ldi	r16,6			;Timeout-Zhler auf 6
	sts	ditime,r16		;Sekunden setzen
	rjmp	main
ma3100:	rcall	ktest2			;Taste 2 (Min) gedrckt?
	breq	ma3200			;nein -> weiter
	lds	r16,flags3		;bearbeitete Tasten holen
	bst	r16,0			;Taste 1 (Std) zustzlich gedr?
	brtc	ma3120			;nein -> weiter
	lds	r16,al1hou		;sonst Alarm 1 Stunden holen
	subi	r16,1			;um 1 Stunde verringern
	brcc	ma3110			;berlauf?
	ldi	r16,23			;ja -> korrigieren
ma3110:	sts	al1hou,r16		;Alarm 1 Minuten speichern
	ldi	r16,6			;Timeout-Zhler auf 6
	sts	ditime,r16		;Sekunden setzen
	rjmp	main
ma3120:	lds	r16,al1min		;sonst Alarm 1 Minuten holen
	ldi	r17,5			;um 5 Minuten
	add	r16,r17			;erhhen
	cpi	r16,60			;Endwert erreicht?
	brcs	ma3130			;nein -> weiter
	clr	r16			;sonst zurcksetzen
ma3130:	sts	al1min,r16		;und wieder speichern
	ldi	r16,6			;Timeout-Zhler auf 6
	sts	ditime,r16		;Sekunden setzen
	rjmp	main
ma3200:	rcall	ktest3			;Taste 3 (Al1) gedrckt?
	breq	ma3300			;nein -> weiter
	clr	r16			;sonst Timeout-Zhler
	sts	ditime,r16		;lschen
	rjmp	main	
ma3300:	rcall	ktest4			;Taste 4 (Al2) gedrckt?
	breq	ma3400			;nein -> weiter
	clr	r16			;sonst Timeout-Zhler
	sts	ditime,r16		;lschen
ma3400:	rjmp	main
;
ma4000:	dec	r16			;Alarm 2 Anzeige?
	breq	ma4010			;ja - bearbeiten
	rjmp	ma5000			;sonst weiter testen
ma4010:	rcall	al2out			;Alarm 2 anzeigen
	rcall	ktest1			;Taste 1 (Std) gedrckt?
	breq	ma4100			;nein -> weiter
	lds	r16,flags3		;bearbeitete Tasten holen
	bst	r16,1			;Taste 2 (Min) zustzlich gedr?
	brtc	ma4030			;nein -> weiter
	lds	r16,al2min		;sonst Alarm 2 Minuten holen
	subi	r16,5			;um 5 Minuten verringern
	brcc	ma4020			;berlauf?
	ldi	r16,55			;ja -> korrigieren
ma4020:	sts	al2min,r16		;Alarm 2 Minuten speichern
	ldi	r16,6			;Timeout-Zhler auf 6
	sts	ditime,r16		;Sekunden setzen
	rjmp	main
ma4030:	lds	r16,al2hou		;sonst Alarm 2 Stunden holen
	inc	r16			;erhhen
	cpi	r16,24			;Endwert erreicht?
	brcs	ma4040			;nein -> weiter
	clr	r16			;sonst zurcksetzen
ma4040:	sts	al2hou,r16		;und wieder speichern
	ldi	r16,6			;Timeout-Zhler auf 6
	sts	ditime,r16		;Sekunden setzen
	rjmp	main
ma4100:	rcall	ktest2			;Taste 2 (Min) gedrckt?
	breq	ma4200			;nein -> weiter
	lds	r16,flags3		;bearbeitete Tasten holen
	bst	r16,0			;Taste 1 (Std) zustzlich gedr?
	brtc	ma4120			;nein -> weiter
	lds	r16,al2hou		;sonst Alarm 2 Stunden holen
	subi	r16,1			;um 1 Stunde verringern
	brcc	ma4110			;berlauf?
	ldi	r16,23			;ja -> korrigieren
ma4110:	sts	al2hou,r16		;Alarm 2 Minuten speichern
	ldi	r16,6			;Timeout-Zhler auf 6
	sts	ditime,r16		;Sekunden setzen
	rjmp	main
ma4120:	lds	r16,al2min		;sonst Alarm 2 Minuten holen
	ldi	r17,5			;um 5 Minuten
	add	r16,r17			;erhhen
	cpi	r16,60			;Endwert erreicht?
	brcs	ma4130			;nein -> weiter
	clr	r16			;sonst zurcksetzen
ma4130:	sts	al2min,r16		;und wieder speichern
	ldi	r16,6			;Timeout-Zhler auf 6
	sts	ditime,r16		;Sekunden setzen
	rjmp	main
ma4200:	rcall	ktest3			;Taste 3 (Al1) gedrckt?
	breq	ma4300			;nein -> weiter
	clr	r16			;sonst Timeout-Zhler
	sts	ditime,r16		;lschen
	rjmp	main	
ma4300:	rcall	ktest4			;Taste 4 (Al2) gedrckt?
	breq	ma4400			;nein -> weiter
	clr	r16			;sonst Timeout-Zhler
	sts	ditime,r16		;lschen
ma4400:	rjmp	main
ma5000:	rcall	verout			;sonst Versionsnummer anzeigen
	rjmp	main
;
;	-----------------------------------------------
;	Tastentests
;	Register: r16,r17
;	Ausgang:  Z-Flag - 0= keine Taste zu bearbeiten
;			   1= Taste bearbeiten
;	-----------------------------------------------
;
ktest1:	ldi	r16,0x01		;Maske fr Taste 1 (Std)
	rjmp	ktestx
;
ktest2:	ldi	r16,0x02		;Maske fr Taste 2 (Min)
	rjmp	ktestx
;
ktest3:	ldi	r16,0x04		;Maske fr Taste 3 (Al1)
	rjmp	ktestx
;
ktest4:	ldi	r16,0x08		;Maske fr Taste 4 (Al2)
;
ktestx:	lds	r17,flags3		;Bearbeitungs-Flags holen
	and	r17,r16			;Taste bereits bearbeitet?
	breq	ktes10			;nein -> Taste bearbeiten
	clr	r17			;sonst Ende mit Z=0
	ret
;
ktes10:	lds	r17,flags2		;gedrckte Tasten holen
	and	r17,r16			;gewnschte Taste filtern
	brne	ktes20			;Taste gedrckt? ja -> weiter
	ret				;sonst Ende mit Z=0
;
ktes20:	lds	r17,flags3		;Bearbeitungs-Flags holen
	or	r17,r16			;gewnschte Taste setzen
	sts	flags3,r17		;Flags wieder speichern
	ret
;
;	---------------------------------------------
;	DCF77-Auswertung, Kontrolle der Prfbits 1-3,
;	Ablegen der Zeitinformation und Vergleich mit
;	vorheriger Zeitinformation
;	Ausgang: r19=0 - alles ok, r19>0 - Fehler
;	---------------------------------------------
;
;	- Kontrolle des Impulszhlers und der Prfbits fr Minuten,
;	  Stunden und Datum
;
mdcprc:	clr	r19			;Fehlerzhler lschen
	mov	r21,dcfimp		;DCF77-Impulszhler kopieren
	clr	dcfimp			;DCF77-Impulszhler lschen
	cpi	r21,59			;wurden 59 Impulse empfangen?
	breq	mdc010			;ja -> weiter
	inc	r19			;sonst Fehlerzhler erhhen
	rjmp	mdc200			;Auswertung vorzeitig beenden
;
mdc010:	clr	r21			;wird fr Parittsber. bentigt
	clr	r22			;Zhler fr Parittsberechnung
	lds	r23,dcftab+1		;Byte 1 aus Tabelle holen
	ldi	r20,6			;6 Minuten-Bits auswerten
mdc020:	rol	r23			;relevantes Bit in Carry
	adc	r22,r21			;Eins-Bit addieren
	dec	r20			;alle Bits ausgewertet?
	brne	mdc020			;nein -> Schleife
	lds	r23,dcftab+2		;Byte 2 aus Tabelle holen
	ror	r23			;relevantes Bit in Carry
	adc	r22,r21			;Eins-Bit addieren
	ror	r23			;relevantes Bit in Carry
	adc	r22,r21			;Eins-Bit addieren
	bst	r22,0			;war das Ergebnis geradzahlig?
	brtc	mdc030			;ja -> weiter
	inc	r19			;sonst Fehlerzhler erhhen
;
mdc030:	clr	r22			;Zhler fr Parittsberechnung
	ldi	r20,6			;6 Stunden-Bits auswerten
mdc040:	ror	r23			;relevantes Bit in Carry
	adc	r22,r21			;Eins-Bit addieren
	dec	r20			;alle Bits ausgewertet?
	brne	mdc040			;nein -> Schleife
	lds	r23,dcftab+3		;Byte 3 aus Tabelle holen
	ror	r23			;relevantes Bit in Carry
	adc	r22,r21			;Eins-Bit addieren
	bst	r22,0			;war das Ergebnis geradzahlig?
	brtc	mdc050			;ja -> weiter
	inc	r19			;sonst Fehlerzhler erhhen
;
mdc050:	clr	r22			;Zhler fr Parittsberechnung
	ldi	r20,7			;7 Datum-Bits auswerten
mdc060:	ror	r23			;relevantes Bit in Carry
	adc	r22,r21			;Eins-Bit addieren
	dec	r20			;alle Bits ausgewertet?
	brne	mdc060			;nein -> Schleife
	lds	r23,dcftab+4		;Byte 4 aus Tabelle holen
	ldi	r20,8			;8 Datum-Bits auswerten
mdc070:	ror	r23			;relevantes Bit in Carry
	adc	r22,r21			;Eins-Bit addieren
	dec	r20			;alle Bits ausgewertet?
	brne	mdc070			;nein -> Schleife
	lds	r23,dcftab+5		;Byte 5 aus Tabelle holen
	ldi	r20,8			;8 Datum-Bits auswerten
mdc080:	ror	r23			;relevantes Bit in Carry
	adc	r22,r21			;Eins-Bit addieren
	dec	r20			;alle Bits ausgewertet?
	brne	mdc080			;nein -> Schleife
	bst	r22,0			;war das Ergebnis geradzahlig?
	brtc	mdc090			;ja -> weiter
	inc	r19			;sonst Fehlerzhler erhhen
mdc090:	tst	r19			;Parittsfehler aufgetreten?
	breq	mdc100			;nein -> weiter
	ret				;sonst Ende
;
;	- Ablegen der Zeitinformation in Puffer 1
;
mdc100:	ldi	zl,low(dcbuf1)		;Zeiger auf DCF77-Puffer 1
	ldi	zh,high(dcbuf1)
	lds	r23,dcftab		;Byte 0 aus Tabelle holen
	rol	r23			;Bit 6 auf Bit 0 schieben
	rol	r23
	rol	r23
	andi	r23,0x01		;relevantes Bit filtern
	st	z+,r23			;Zeitzone speichern
	lds	r23,dcftab+1		;Byte 1 aus Tabelle holen
	ror	r23			;Bits 2-5 auf Bits 0-3 schieben
	ror	r23
	andi	r23,0x0f		;relevante Bits filtern
	st	z+,r23			;Minuten Einer speichern
	lds	r23,dcftab+1		;Byte 1 aus Tabelle holen
	lds	r22,dcftab+2		;Byte 2 aus Tabelle holen
	rol	r23			;Bits 6-7 aus r23 und 0 aus r22
	rol	r22			;auf Bits 0-2 von r22 schieben
	rol	r23
	rol	r22
	andi	r22,0x07		;relevante Bits filtern
	st	z+,r22			;Minuten Zehner speichern
	lds	r23,dcftab+2		;Byte 2 aus Tabelle holen
	ror	r23			;Bits 2-5 auf Bits 0-3 schieben
	ror	r23
	andi	r23,0x0f		;relevante Bits filtern
	st	z+,r23			;Stunden Einer speichern
	lds	r23,dcftab+2		;Byte 2 aus Tabelle holen
	rol	r23			;Bits 6-7 auf Bits 0-1 schieben
	rol	r23
	rol	r23
	andi	r23,0x03		;relevante Bits filtern
	st	z+,r23			;Stunden Zehner speichern
	lds	r23,dcftab+3		;Byte 3 aus Tabelle holen
	ror	r23			;Bits 1-4 auf Bits 0-3 schieben
	andi	r23,0x0f		;relevante Bits filtern
	st	z+,r23			;Kalendertag Einer speichern
	lds	r23,dcftab+3		;Byte 3 aus Tabelle holen
	ror	r23			;Bits 5-6 auf Bits 4-5 schieben
	swap	r23			;Nibbles tauschen
	andi	r23,0x03		;relevante Bits filtern
	st	z+,r23			;Kalendertag Zehner speichern
	lds	r23,dcftab+3		;Byte 3 aus Tabelle holen
	rol	r23			;Bit 7 in Carry schieben
	lds	r23,dcftab+4		;Byte 4 aus Tabelle holen
	rol	r23			;Carry in Bit 0 schieben
	andi	r23,0x07		;relevante Bits filtern
	st	z+,r23			;Wochentag speichern
	lds	r23,dcftab+4		;Byte 4 aus Tabelle holen
	ror	r23			;Bits 2-5 in Bits 0-3 schieben
	ror	r23
	andi	r23,0x0f		;relevante Bits filtern
	st	z+,r23			;Monat Einer speichern
	lds	r23,dcftab+4		;Byte 4 aus Tabelle holen
	rol	r23			;Bit 6 in Bit 0 schieben
	rol	r23
	rol	r23
	andi	r23,0x01		;relevantes Bit filtern
	st	z+,r23			;Monat Zehner speichern
	lds	r23,dcftab+4		;Byte 4 aus Tabelle holen
	rol	r23			;Bit 7 in Carry schieben
	lds	r23,dcftab+5		;Byte 5 aus Tabelle holen
	rol	r23			;Carry in Bit 0 schieben
	andi	r23,0x0f		;relevante Bits filtern
	st	z+,r23			;Jahr Einer speichern
	lds	r23,dcftab+5		;Byte 5 aus Tabelle holen
	rol	r23			;Bits 3-6 in Bits 4-7 schieben
	swap	r23			;Nibbles tauschen
	andi	r23,0x0f		;relevante Bits filtern
	st	z+,r23			;Jahr Zehner speichern
;
;	- Vergleich des aktuellen Puffers mit der vorherigen Zeit-
;	  information und Puffer 2->3 und Puffer 1->2 kopieren
;
	ldi	yl,low(dcbuf1)
	ldi	yh,high(dcbuf1)		;Zeiger auf DCF77-Puffer 1
	ldi	zl,low(dcbuf2)
	ldi	zh,high(dcbuf2)		;Zeiger auf DCF77-Puffer 2
	ldi	xl,low(dcbuf3)
	ldi	xh,high(dcbuf3)		;Zeiger auf DCF77-Puffer 3
	clr	r18			;Nullbyte-Detektor lschen
	ldi	r23,12			;12 Bytes vergleichen
mdc110:	ld	r22,y+			;Puffer 1 Inhalt holen
	or	r18,r22			;Nullbyte-Detektor bearbeiten
	ld	r21,z			;Puffer 2 Inhalt holen
	ld	r20,x			;Puffer 3 Inhalt holen
	st	x+,r21			;Puffer 2 nach 3 kopieren
	st	z+,r22			;Puffer 1 nach 2 kopieren
	cp	r21,r22			;Inhalt P1/P2 identisch?
	brne	mdc120			;nein -> Fehler
	cp	r20,r21			;Inhalt P2/P3 identisch?
	breq	mdc130			;ja -> weiter
mdc120:	inc	r19			;sonst Fehlerzhler erhhen
mdc130:	dec	r23			;alle Bytes bearbeitet?
	brne	mdc110			;nein -> Schleife
	tst	r18			;alles Nullbytes im Puffer?
	brne	mdc140			;nein -> weiter
	inc	r19			;sonst Fehlerzhler erhhen
;
;	- Puffer 2 und 3 um eine Minute erhhen, es werden nur Minuten
;	  und Stunden bercksichtigt
;
mdc140:	ldi	zl,low(dcbuf2)
	ldi	zh,high(dcbuf2)		;Zeiger auf DCF77-Puffer 2
	rcall	incdcb			;um eine Minute erhhen
	ldi	zl,low(dcbuf3)
	ldi	zh,high(dcbuf3)		;Zeiger auf DCF77-Puffer 3
	rcall	incdcb			;um eine Minute erhhen
;
;	- Ende, DCF-Tabelle lschen
;
mdc200:	ldi	zl,low(dcftab)		;Zeiger auf DCF77-Tabelle
	ldi	zh,high(dcftab)
	ldi	r23,6			;6 Bytes bearbeiten
	clr	r22			;Nullwert
mdc210:	st	z+,r22			;Tabellenplatz lschen
	dec	r23			;alle Pltze bearbeitet?
	brne	mdc210			;nein -> Schleife
	ret
;
;	------------------------------------------
;	Zeitinformationen in Uhrenbereich kopieren
;	und Uhrzeit synchronisieren
;	------------------------------------------
;
mtsync:	cli				;Interrupts sperren
	lds	r0,dcbuf1+1		;Minuten Einer holen
	lds	r1,dcbuf1+2		;Minuten Zehner holen
	rcall	bcdhex			;in Hexcode wandeln
	sts	minute,r16		;und speichern
	lds	r0,dcbuf1+3		;Stunden Einer holen
	lds	r1,dcbuf1+4		;Stunden Zehner holen
	rcall	bcdhex			;in Hexcode wandeln
	sts	hour,r16		;und speichern
	clr	r16
	sts	second,r16		;Sekundenzhler lschen
	sts	pcount,r16		;Vorteiler lschen
	lds	r16,flags4		;DCF-Flags holen
	ori	r16,0b00000100		;Synchron-Bit setzen
	sts	flags4,r16		;und wieder speichern
	sei				;Interrupts wieder aktivieren
	ret
;
;	----------------------
;	Aktuelle Zeit ausgeben
;	Register: r16,r17
;	----------------------
;
clkout:	lds	r16,hour		;Stunden holen
	rcall	hexbcd			;in BCD-Zahlen wandeln
	mov	r16,r1			;Zehner holen
	rcall	bcdxse			;in 7-Segment-Code wandeln
	mov	r17,invflg		;Invers-Flag holen
	andi	r17,0x07		;LED-Ports filtern
	eor	r16,r17			;LED-Ports invertieren
	out	portb,r16		;und ausgeben
	mov	r16,r0			;Einer holen
	rcall	bcd7se			;in 7-Segment-Code wandeln
	lds	r17,flags4		;DCF77-Flags holen
	bst	r17,2			;Uhr schon synchronisiert?
	brtc	clko10			;nein -> Punkt 2 zeigt DCF
	lds	r17,pcount		;Vorteiler holen
	cpi	r17,50			;Wert<50 (Punkt 2 ein)?
	brcs	clko20			;ja -> weiter
	ori	r16,0x80		;sonst Punkt 2 ausschalten
	rjmp	clko20			;ausgeben
clko10:	bst	r17,0			;DCF77-Status holen
	bld	r16,7			;als Dezimalpunkt setzen
	ldi	r17,0x80
	eor	r16,r17			;invertieren
clko20:	eor	r16,invflg		;LED-Ports invertieren
	out	porta,r16		;ausgeben
	lds	r16,minute		;Minuten holen
	rcall	hexbcd			;in BCD-Zahlen wandeln
	mov	r18,r0			;Einer sichern
	mov	r16,r1			;Zehner holen
	rcall	bcd7se			;in 7-Segment-Code wandeln
	mov	r17,invflg		;Invers-Flag holen
	andi	r17,0x7f		;LED-Ports filtern
	eor	r16,r17			;LED-Ports invertieren
	out	portc,r16		;und ausgeben
	mov	r16,r18			;Einer holen
	rcall	bcd7se			;in 7-Segment-Code wandeln
	ori	r16,0x80		;Punkt 4 ausschalten
	lds	r17,flags5		;Alarm-Flags holen
	bst	r17,0			;Alarm 1 aktiviert?
	brtc	clko30			;nein -> weiter
	andi	r16,0x7f		;sonst Punkt 4 einschalten
	rjmp	clko40			;ausgeben
clko30:	bst	r17,1			;Alarm 2 aktiviert?
	brtc	clko40			;nein -> weiter
	lds	r17,pcount		;Vorteiler holen
	cpi	r17,50			;Wert<50 (Punkt 4 ein)?
	brcc	clko40			;nein -> weiter
	andi	r16,0x7f		;sonst Punkt 4 einschalten
clko40:	eor	r16,invflg		;LED-Ports invertieren
	out	portd,r16
	ret
;
;	----------------------------------
;	Sekunden und DCF77-Signal ausgeben
;	Register: r16,r17
;	----------------------------------
;
secout:	ser	r16			;alle Segmente aus
	mov	r17,invflg		;Invers-Flag holen
	andi	r17,0x07		;LED-Ports filtern
	eor	r16,r17			;LED-Ports invertieren
	out	portb,r16		;Ziffer 1 ausgeben
	ser	r16			;alle Segmente aus
	lds	r17,flags4		;DCF77-Flags holen
	bst	r17,0			;DCF77-Status holen
	bld	r16,7			;als Dezimalpunkt setzen
	ldi	r17,0x80
	eor	r16,r17			;invertieren
	eor	r16,invflg		;LED-Ports invertieren
	out	porta,r16		;Ziffer 2 ausgeben
	lds	r16,second		;Sekunden holen
	rcall	hexbcd			;in BCD-Zahlen wandeln
	mov	r18,r0			;Einer sichern
	mov	r16,r1			;Zehner holen
	rcall	bcd7se			;in 7-Segment-Code wandeln
	mov	r17,invflg		;Invers-Flag holen
	andi	r17,0x7f		;LED-Ports filtern
	eor	r16,r17			;LED-Ports invertieren
	out	portc,r16		;und ausgeben
	mov	r16,r18			;Einer holen
	rcall	bcd7se			;in 7-Segment-Code wandeln
	ori	r16,0x80		;Dezimalpunkt lschen
	eor	r16,invflg		;LED-Ports invertieren
	out	portd,r16		;ausgeben
	ret
;
;	---------------------------------------------------
;	Alarmzeit 1 blinkend anzeigen, 450ms ein / 50ms aus
;	Register: r16,r17
;	---------------------------------------------------
;
al1out:	lds	r16,pcount		;Vorteiler holen
	cpi	r16,50			;Wert<50?
	brcs	al1o10			;ja -> weiter
	subi	r16,50			;sonst 50 subtrahieren
al1o10:	cpi	r16,45			;Wert<45 (Anzeige ein)
	brcs	al1o30			;ja -> Anzeige
al1o20:	ser	r16			;sonst alle Segmente lschen
	eor	r16,invflg		;LED-Ports invertieren
	out	porta,r16		;ausgeben
	out	portd,r16		;ausgeben
	ser	r16			;alle Segmente lschen
	mov	r17,invflg		;Invers-Flag holen
	andi	r17,0x07		;LED-Ports filtern
	eor	r16,r17			;LED-Ports invertieren
	out	portb,r16		;ausgeben
	ldi	r16,0x7f		;Tongenerator-Bit=0
	mov	r17,invflg		;Invers-Flag holen
	andi	r17,0x7f		;LED-Ports filtern
	eor	r16,r17			;LED-Ports invertieren
	out	portc,r16		;ausgeben
	ret
al1o30:	lds	r16,al1hou		;Alarm 1 Stunden holen
	rcall	hexbcd			;in BCD-Zahlen wandeln
	mov	r16,r1			;Zehner holen
	rcall	bcdxse			;in 7-Segment-Code wandeln
	mov	r17,invflg		;Invers-Flag holen
	andi	r17,0x07		;LED-Ports filtern
	eor	r16,r17			;LED-Ports invertieren
	out	portb,r16		;und ausgeben
	mov	r16,r0			;Einer holen
	rcall	bcd7se			;in 7-Segment-Code wandeln
	eor	r16,invflg		;LED-Ports invertieren
	out	porta,r16		;ausgeben
	lds	r16,al1min		;Alarm 1 Minuten holen
	rcall	hexbcd			;in BCD-Zahlen wandeln
	mov	r18,r0			;Einer sichern
	mov	r16,r1			;Zehner holen
	rcall	bcd7se			;in 7-Segment-Code wandeln
	mov	r17,invflg		;Invers-Flag holen
	andi	r17,0x7f		;LED-Ports filtern
	eor	r16,r17			;LED-Ports invertieren
	out	portc,r16		;und ausgeben
	mov	r16,r18			;Einer holen
	rcall	bcd7se			;in 7-Segment-Code wandeln
	ori	r16,0x80		;Dezimalpunkt lschen
	eor	r16,invflg		;LED-Ports invertieren
	out	portd,r16		;ausgeben
	ret
;
;	---------------------------------------------------
;	Alarmzeit 2 blinkend anzeigen, 450ms ein / 50ms aus
;	Register: r16,r17
;	---------------------------------------------------
;
al2out:	lds	r16,pcount		;Vorteiler holen
	cpi	r16,50			;Wert<50?
	brcs	al2o10			;ja -> weiter
	subi	r16,50			;sonst 50 subtrahieren
al2o10:	cpi	r16,45			;Wert<45 (Anzeige ein)
	brcc	al1o20			;nein -> keine Anzeige
	lds	r16,al2hou		;Alarm 2 Stunden holen
	rcall	hexbcd			;in BCD-Zahlen wandeln
	mov	r16,r1			;Zehner holen
	rcall	bcdxse			;in 7-Segment-Code wandeln
	mov	r17,invflg		;Invers-Flag holen
	andi	r17,0x07		;LED-Ports filtern
	eor	r16,r17			;LED-Ports invertieren
	out	portb,r16		;und ausgeben
	mov	r16,r0			;Einer holen
	rcall	bcd7se			;in 7-Segment-Code wandeln
	eor	r16,invflg		;LED-Ports invertieren
	out	porta,r16		;ausgeben
	lds	r16,al2min		;Alarm 2 Minuten holen
	rcall	hexbcd			;in BCD-Zahlen wandeln
	mov	r18,r0			;Einer sichern
	mov	r16,r1			;Zehner holen
	rcall	bcd7se			;in 7-Segment-Code wandeln
	mov	r17,invflg		;Invers-Flag holen
	andi	r17,0x7f		;LED-Ports filtern
	eor	r16,r17			;LED-Ports invertieren
	out	portc,r16		;und ausgeben
	mov	r16,r18			;Einer holen
	rcall	bcd7se			;in 7-Segment-Code wandeln
	ori	r16,0x80		;sonst Punkt 4 ausschalten
	eor	r16,invflg		;LED-Ports invertieren
	out	portd,r16
	ret
;
;	-----------------------
;	Versionsnummer ausgeben
;	Register: r16,r17,zl,zh
;	-----------------------
;
verout:	ser	r16			;Stunden-Zehner Stelle lschen
	mov	r17,invflg		;Invers-Flag holen
	andi	r17,0x07		;LED-Ports filtern
	eor	r16,r17			;LED-Ports invertieren
	out	portb,r16		;ausgeben
	ldi	zl,low(versnr*2)
	ldi	zh,high(versnr*2)	;Zeiger auf Versionsnummer
	lpm				;erste Stelle holen
	mov	r2,r0			;zwischenspeichern
	adiw	zl,1			;Zeiger auf nchste Stelle
	lpm				;nchste Stelle holen
	mov	r3,r0			;zwischenspeichern
	adiw	zl,1			;Zeiger auf nchste Stelle
	lpm				;nchste Stelle holen
	mov	r4,r0			;zwischenspeichern
	mov	r16,r2			;erste Stelle holen
	rcall	bcd7se			;in 7-Segment-Code wandeln
	eor	r16,invflg		;LED-Ports invertieren
	out	porta,r16		;ausgeben
	mov	r16,r3			;zweite Stelle holen
	rcall	bcd7se			;in 7-Segment-Code wandeln
	mov	r17,invflg		;Invers-Flag holen
	andi	r17,0x7f		;LED-Ports filtern
	eor	r16,r17			;LED-Ports invertieren
	out	portc,r16		;ausgeben
	mov	r16,r4			;dritte Stelle holen
	rcall	bcd7se			;in 7-Segment-Code wandeln
	ori	r16,0x80		;Dezimalpunkt 4 ausschalten
	eor	r16,invflg		;LED-Ports invertieren
	out	portd,r16		;ausgeben
	ret
;
;	--------------------------------
;	Konvertierung BCD -> Hexadezimal
;	Register: r0  - BCD-Zahl L (0-9)
;		  r1  - BCD-Zahl H (0-9)
;		  r16 - Hexzahl (0-99)
;		  r17 - intern
;	--------------------------------
;
bcdhex:	clr	r16			;Ergebnis=0
	ldi	r17,10
bcdh10:	tst	r1			;H-Wert=0?
	breq	bcdh20			;ja -> L-Wert bearbeiten
	add	r16,r17			;sonst H-Wert +10
	dec	r1
	rjmp	bcdh10			;Schleife
bcdh20:	add	r16,r0			;L-Wert addieren
	ret
;
;	--------------------------------
;	Konvertierung Hexadezimal -> BCD
;	Register: r16 - Hexzahl (0-99)
;		  r0  - BCD-Zahl L (0-9)
;		  r1  - BCD-Zahl H (0-9)
;	--------------------------------
;
hexbcd:	clr	r1			;High-Zahl lschen
hexb10:	cpi	r16,10			;Hexwert>9?
	brcs	hexb20			;nein -> Low-Zahl bearbeiten
	inc	r1			;sonst High-Zahl erhhen
	subi	r16,10			;Hexwert-10
	rjmp	hexb10			;Schleife
hexb20:	mov	r0,r16			;Low-Zahl speichern
	ret
;
;	----------------------------------------------
;	Konvertierung BCD -> Siebensegment (Standard)
;	Register: r16 - Ziffer (0-9) -> 7-Segment-Code
;		  r0,r1,r30,r31 - intern
;	----------------------------------------------
;
bcd7se:	ldi	zl,low(segtab*2)	;Zeiger auf Segment-Tabelle
	ldi	zh,high(segtab*2)
	clr	r1			;Wert fr bertragsberechnung
	add	zl,r16			;Tabellenoffset berechnen
	adc	zh,r1			;eventuell bertrag addieren
	lpm				;Segment-Code holen
	mov	r16,r0			;und ablegen
	ret
;
;	-----------------------------------------------
;	Konvertierung BCD -> Siebensegment (Std-Zehner)
;	Register: r16 - Ziffer (0-2) -> 7-Segment-Code
;	-----------------------------------------------
;
bcdxse:	tst	r16			;Ziffer=0?
	brne	bcdx10			;nein -> weiter testen
	ldi	r16,0xff		;sonst alle Segmente aus
	ret
bcdx10:	dec	r16			;Ziffer=1?
	brne	bcdx20			;nein -> weiter testen
	ldi	r16,0b11111100		;Segmentcode fr "1"
	ret
bcdx20:	ldi	r16,0b11111010		;sonst Segmentcode fr "2"
	ret
;
;	-----------------------------------------
;	Unterprogramm zum Lesen vom EEPROM
;	Register: r16 - EEPROM-Adresse Low
;		  r17 - EEPROM-Adresse High
;		  r0  - EEPROM-Daten
;		  r1  - interner Zwischenspeicher
;	-----------------------------------------
;
eeread:	sbic	eecr,eewe		;luft Schreibzyklus?
	rjmp	eeread			;ja - warten
	out 	eearh,r17		;EEPROM-Adresse High setzen
	out	eearl,r16		;EEPROM-Adresse Low setzen
	sbi	eecr,eere		;EEPROM lesen aktivieren
	in	r0,eedr			;Daten lesen
	clr	r1
	out	eearh,r1		;EEPROM-Adresse auf 0 setzen
	out	eearl,r1
	ret
;
;	-----------------------------------------
;	Unterprogramm zum Schreiben in den EEPROM
;	Register: r16 - EEPROM-Adresse Low
;		  r17 - EEPROM-Adresse High
;		  r18 - EEPROM-Daten
;		  r1  - interner Zwischenspeicher
;	-----------------------------------------
;
eewrit:	sbic	eecr,eewe		;luft Schreibzyklus?
	rjmp	eewrit			;ja -> warten
	out	eearh,r17		;EEPROM-Adresse H setzen
	out	eearl,r16		;EEPROM-Adresse L setzen
	out	eedr,r18		;EEPROM-Datenbyte setzen
	sbi	eecr,eemwe		;Masterwrite-Bit setzen
	sbi	eecr,eewe		;Write-Bit setzen
	ret
;
;	------------------------------------------------
;	Erhhen des DCF77-Pufferinhaltes um eine Minute,
;	es werden nur Minuten und Stunden bercksichtigt
;	Register: r22,r23
;		  z <- Zeiger auf DCF77-Puffer
;	------------------------------------------------
;
incdcb:	ldd	r23,z+1			;Minuten Einer holen
	inc	r23			;erhhen
	std	z+1,r23			;und wieder speichern
	cpi	r23,10			;bertrag?
	brcs	incd20			;nein -> weiter
	clr	r23			;sonst zurcksetzen
	std	z+1,r23			;und speichern
	ldd	r23,z+2			;Minuten Zehner holen
	inc	r23			;erhhen
	std	z+2,r23			;und wieder speichern
	cpi	r23,6			;bertrag?
	brcs	incd20			;nein -> weiter
	clr	r23			;sonst zurcksetzen
	std	z+2,r23			;und speichern
	ldd	r22,z+4			;Stunden Zehner holen
	ldd	r23,z+3			;Stunden Einer holen
	inc	r23			;erhhen
	std	z+3,r23			;und wieder speichern
	cpi	r23,10			;bertrag?
	brcs	incd10			;nein -> Test auf "24"
	clr	r23			;sonst zurcksetzen
	std	z+3,r23			;und speichern
	inc	r22			;Stunden Zehner erhhen
	std	z+4,r22			;und speichern
incd10:	cpi	r22,2			;Stunden Zehner=2?
	brne	incd20			;nein -> weiter
	cpi	r23,4			;Stunden Einer=4 (=24) ?
	brne	incd20			;nein -> weiter
	clr	r23			;sonst Stunden zurcksetzen
	std	z+3,r23			;Stunden Einer speichern
	std	z+4,r23			;Stunden Zehner speichern
incd20:	ret
;
;	---------------------------------------------
;	Interrupt-Routine Timer1 Compare A, alle 10ms
;	liest DCF77 und Tasten ein, zhlt die Uhrzeit
;	Register: r13,14,r15,r25,xl,xh
;	---------------------------------------------
;
timint:	in	r15,sreg		;CPU-Flags sichern
	wdr				;Watchdog zurcksetzen
	lds	r25,pcount		;Vorteiler holen
	ror	r25			;Vorteiler geradzahlig?
	brcc	timi10			;ja -> DCF77/Tasten bearbeiten
	rjmp	timi60			;sonst berspringen
;
timi10:	lds	r14,flags4		;DCF77-Flags holen
	in	r25,pinb		;Tasten und DCF77 einlesen
	bst	r25,3			;DCF77-Status holen
	bld	r14,0			;und in Flags speichern
	sts	flags4,r14		;DCF77-Flags speichern
;
	swap	r25			;Tastenbits von 7-4 auf 3-0
	com	r25			;Tastenbits invertieren
	andi	r25,0x0f		;relevante Bits filten
	mov	r13,r25			;und kopieren
	lds	r14,flags1		;letzten Tastenstatus holen
	eor	r13,r14			;Statusnderungs-Maske erzeugen
	sts	flags1,r25		;neuen Tastenstatus speichern
	lds	r14,flags2		;gltige Tasten holen
	and	r14,r13			;unvernderte Tasten filtern
	com	r13			;Maske invertieren
	and	r25,r13			;vernderte Tasten filtern
	or	r14,r25			;gltige Tasten kombinieren
	sts	flags2,r14		;und speichern
	lds	r13,flags3		;Bearbeitungs-Flags holen
	and	r13,r14			;losgelassene Tasten lschen
	sts	flags3,r13		;Bearbeitungs-Flags speichern
;
	lds	r25,dcfcnt		;DCF77-Zhler holen
	inc	r25			;Zhler erhhen
	sts	dcfcnt,r25		;und wieder speichern
	lds	r14,flags4		;DCF-Flags holen
	bst	r14,0			;akuellen DCF77-Status prfen
	brts	timi30			;ist DCF77 High? ja -> weiter
	bst	r14,1			;sonst vorh. DCF-Status prfen
	brtc	timi50			;war DCF77 vorher H? n-> weiter
	cpi	r25,75			;Zhler<75 (1,5s) ?
	brcs	timi20			;nein -> weiter
	bld	r14,3			;sonst Rahmenende setzen
timi20:	cpi	r25,45			;Zhler<45 (900ms) ?
	brcs	timi50			;ja -> ignorieren
	clr	r25			;sonst
	sts	dcfcnt,r25		;DCF77-Zhler lschen
	rjmp	timi50			;Ende
timi30:	bst	r14,1			;vorherigen DCF-Status prfen
	brts	timi50			;war DCF77 vorher H? ja -> Ende
	cpi	r25,13			;DCF-Zhler<13 (260ms)?
	brcc	timi50			;nein -> ignorieren
	cpi	r25,3			;DCF-Zhler<3 (60ms)?
	brcs	timi50			;ja -> ignorieren
	inc	dcfimp			;sonst Impuls zhlen
	cpi	r25,7			;Zhler<7? Ergebnis in Carry
	ldi	r25,6			;6 Pufferbytes bearbeiten
	ldi	xl,low(dcftab+6)	;Zeiger auf Pufferende setzen
	ldi	xh,high(dcftab+6)
	rol	r13			;Carry in Register legen
	com	r13			;alle Bits invertieren
	ror	r13			;wieder in Carry schieben
timi40:	ld	r13,-x			;Pufferbyte holen
	ror	r13			;neues Bit einschieben als MSB
	st	x,r13			;LSB in Carry und wieder spei.
	dec	r25			;alle Pufferbytes bearbeitet?
	brne	timi40			;nein -> Schleife
timi50:	bst	r14,0			;aktuellen DCF77-Status holen
	bld	r14,1			;und als neuen Status speichern
	sts	flags4,r14		;DCF77-Flags speichern
;
timi60:	lds	r25,pcount		;Vorteiler holen
	inc	r25			;erhhen
	sts	pcount,r25		;und wieder speichern
	cpi	r25,100			;Endwert erreicht?
	brne	timend			;nein -> Ende
	clr	r25			;sonst zurcksetzen
	sts	pcount,r25		;und speichern
;
	lds	r25,ditime		;Timeout-Zhler holen
	tst	r25			;ist Zhler bereits auf 0?
	breq	timi70			;ja -> Ende
	dec	r25			;sonst vermindern
	sts	ditime,r25		;und wieder speichern
;
timi70:	lds	r25,second		;Sekunden holen
	inc	r25			;erhhen
	sts	second,r25		;und wieder speichern
	cpi	r25,60			;Endwert erreicht?
	brne	timend			;nein -> Ende
	clr	r25			;sonst zurcksetzen
	sts	second,r25		;und speichern
;
	lds	r25,minute		;Minuten holen
	inc	r25			;erhhen
	sts	minute,r25		;und wieder speichern
	cpi	r25,60			;Endwert erreicht?
	brne	timend			;nein -> Ende
	clr	r25			;sonst zurcksetzen
	sts	minute,r25		;und speichern
;
	lds	r25,hour		;Stunden holen
	inc	r25			;erhhen
	sts	hour,r25		;und wieder speichern
	cpi	r25,24			;Endwert erreicht?
	brne	timend			;nein -> Ende
	clr	r25			;sonst zurcksetzen
	sts	hour,r25		;und speichern
;
timend:	out	sreg,r15		;CPU-Flags wiederherstellen
	reti				;Ende
;
segtab:	.db	0b01000000,0b01111001	;Ziffer "0", "1"
	.db	0b00100100,0b00110000	;Ziffer "2", "3"
	.db	0b00011001,0b00010010	;Ziffer "4", "5"
	.db	0b00000010,0b01111000	;Ziffer "6", "7"
	.db	0b00000000,0b00010000	;Ziffer "8", "9"
;
versnr:	.db	1,0,5,0
;
invfl1:	.db	0xff,0		;Flag fr die Invertierung der Ausgnge
				;nur das erste Byte ist relevant
				;0= Normalbetrieb, 0xff = Inversbetrieb
;
.eseg
;
	.db	0xff			;erstes Byte wird nicht genutzt
ealsta:	.db	0			;Alarmstatus
					;Bit0: 1= Alarm 1 ein
					;Bit1: 1= Alarm 2 ein
ea1min:	.db	0			;Alarm 1 Minuten (0-59)
ea1hou:	.db	6			;Alarm 1 Stunden (0-23)
ea2min:	.db	0			;Alarm 2 Minuten (0-59)
ea2hou:	.db	5			;Alarm 2 Stunden (0-23)
