// ====================================================================================================================================================================================================
// Tischuhr mit ATmega128 und grafischem LCD (240x128 Pixel) und T6963C Controller
// Version 1.100, 06.07.2013
// ====================================================================================================================================================================================================

// Belegung der I/O-Ports -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

// A0: Ein-/Ausgang Datenleitung D0 fr LCD
// A1: Ein-/Ausgang Datenleitung D1 fr LCD
// A2: Ein-/Ausgang Datenleitung D2 fr LCD
// A3: Ein-/Ausgang Datenleitung D3 fr LCD
// A4: Ein-/Ausgang Datenleitung D4 fr LCD
// A5: Ein-/Ausgang Datenleitung D5 fr LCD
// A6: Ein-/Ausgang Datenleitung D6 fr LCD
// A7: Ein-/Ausgang Datenleitung D7 fr LCD

// B0: Ausgang LCD-Betriebsspannung (L-aktiv)
// B1: Eingang SCK/ISP (Pull-Up)
// B2: Eingang Jumper 1, Drehimpulsgeber-Modus (Pull-Up): offen= 2-Schritt-Raster, geschlossen= 4-Schritt-Raster
// B3: Eingang Jumper 2, Drehimpulsgeber-Richtung in Mens (Pull-Up): offen= Rechtsdrehung bewegt Cursor nach unten, geschlossen: Rechtsdrehung bewegt Cursor nach oben
// B4: Eingang Jumper 3, Drehimpulsgeber-Richtung im Editor (Pull-Up): offen= Rechtsdrehung vergrert Wert, geschlossen: Rechtsdrehung verkleinert Wert
// B6: Ausgang Lautstrke PWM (OC1B)
// B7: Ausgang LCD-Hintergrundbeleuchtung PWM (OC2)

// C0: Ausgang Datenleitung /WR fr LCD
// C1: Ausgang Datenleitung /RD fr LCD
// C2: Ausgang Datenleitung /CE fr LCD
// C3: Ausgang Datenleitung C/D fr LCD
// C4: Eingang Drehimpulsgeber-A (Pull-Up)
// C5: Eingang Drehimpulsgeber-B (Pull-Up)
// C6: Eingang Taster 1 (Pull-Up)
// C7: Eingang Taster 2 (Pull-Up)

// D2: Eingang RXD1 Daten vom USB-Wandler
// D3: Ausgang TXD1 Daten zum USB-Wandler
// D4: Ausgang LED, L-aktiv

// E0: Eingang RXD0 Datenleitung MOSI
// E1: Ausgang TXD0 Datenleitung MISO
// E2: Eingang DCF77-Empfnger (Pull-Up, schaltbar)
// E3: Ausgang Tonfrequenz (OC3A)

// F4: Eingang (Pull-Up) JTAG-TCK }
// F5: Eingang (Pull-Up) JTAG-TMS } AVR-JTAG
// F6: Eingang (Pull-Up) JTAG-TDO } Interface
// F7: Eingang (Pull-Up) JTAG-TMI }

// Belegung der Timer ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

// Timer0:  nicht genutzt
// Timer1A: Haupttimer fr alle zeitgesteuerten Vorgnge ber Interrupt; der Uhrenzhler wird alle 78,125s ausgelst und ber 3 Software-Vorteiler auf 625s (/8), 20ms (/32) und 1s (/50)
//          herabgesetzt; eine weitere Zhlkette mit der gleichen Abstufung steuert unabhngig von der Uhr verschiedene Timeout-Zhler; eine dritte Zhlkette arbeitet mit 625s (/8) und steuert die
//          Sound-Ausgabe
// Timer1B: Steuert ber PWM die Lautstrke der Ton-Ausgabe mit 78,125s (12800Hz)
// Timer2:  Steuert die Helligkeit der LCD-Hintergrundbeleuchtung
// Timer3A: Bestimmt die Ton-Frequenz und kann Tne im gesamten hrbaren Bereich erzeugen (Vorteiler 8)

// Verwendete Bibliotheken ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

#include <avr/io.h>                    // I/O-Definitionen
#include <stdint.h>                    // Standard Integer Typen
#include <util/delay.h>                // Warteschleifen-Funktionen
#include <avr/pgmspace.h>              // Zugriff auf Daten im Flash-Speicher
#include <avr/eeprom.h>                // Zugriff auf Daten im EEPROM-Speicher
#include <avr/interrupt.h>             // Interrupt-Routinen

// Versionsnummer -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

const char strd00[] PROGMEM = "v1.100, 06.07.2013";

// Eigene Header-Files --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

#include "clock.h"                     // Bitmap-Daten fr die Analoguhr (im Flash-Speicher)
#include "circle.h"                    // Kreis-Daten (im Flash-Speicher)
#include "charset.h"                   // Zeichensatz-Daten (im Flash-Speicher)
#include "strings.h"                   // Texte Deutsch und Englisch (im Flash-Speicher)
#include "sound.h"                     // Sound-Daten (im Flash-Speicher)

// Konstanten -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

const uint8_t c_yearh = 20;            // Jahrhundert festlegen: 20xx
const uint8_t c_year = 13;             // frhestes Jahr festlegen (fr ersten Start und manuelle Zeiteingabe): 2013
const uint8_t c_hhlen = 40;            // Lnge des Stundenzeigers in Pixel (maximal 55)
const uint8_t c_mhlen = 52;            // Lnge des Minutenzeigers in Pixel (maximal 55)
const uint8_t c_shlen1 = 55;           // Lnge des Sekundenzeigers in Pixel (maximal 55)
const uint8_t c_shlen2 = 12;           // Lnge des Sekundenzeiger-Endes in Pixel (maximal 55)
const uint8_t c_disptout = 120;        // Anzeige-Timeout fr die Rckkehr in die normale Uhrenanzeige: 120s
const uint8_t c_usbtout = 5;           // Timeout fr die Anzeige des USB-Symbols: 5s
const uint8_t c_syntout = 30;          // Timeout fr Synchronisierung: nach 30min LCD wieder einschalten
const uint8_t c_dcftout = 24;          // Timeout fr die Anzeige "DCF77-Signal ok": 24h
const uint8_t c_nightout = 50;         // Timeout fr die Einstellung der Nacht-Helligkeit: 50 x 20ms = 1s

// EEPROM-Adressen (werden als Konstanten definiert, um feste Adresse zu erhalten) --------------------------------------------------------------------------------------------------------------------

const uint16_t e_lang = 1;             // feste EEPROM-Adresse 0001 - Sprache (siehe lang)
const uint16_t e_lcdinv = 2;           // feste EEPROM-Adresse 0002 - LCD-Invertierung (siehe lcdinv)
const uint16_t e_dcfinv = 3;           // feste EEPROM-Adresse 0003 - DCF-Invertierung (siehe dcfinv)
const uint16_t e_dcfpup = 4;           // feste EEPROM-Adresse 0004 - DCF-Pull-Up (siehe dcfpup)
const uint16_t e_nightsyn = 5;         // feste EEPROM-Adresse 0005 - LCD zur Synchronisierung ausschalten (siehe lcdsyn)
const uint16_t e_lcdnight = 6;         // feste EEPROM-Adresse 0006 - LCD-Nacht-Modus eingeschaltet (siehe lcdnight)
const uint16_t e_lcdnigbr = 7;         // feste EEPROM-Adresse 0007 - LCD-Nacht-Helligkeit (siehe lcdnigbr)
const uint16_t e_lcdnigm1 = 8;         // feste EEPROM-Adresse 0008 - Minuten, LCD-Nacht-Modus Beginn (siehe lcdnigm1)
const uint16_t e_lcdnigh1 = 9;         // feste EEPROM-Adresse 0009 - Stunden, LCD-Nacht-Modus Beginn (siehe lcdnigh1)
const uint16_t e_lcdnigm2 = 10;        // feste EEPROM-Adresse 0010 - Minuten, LCD-Nacht-Modus Ende (siehe lcdnigm2)
const uint16_t e_lcdnigh2 = 11;        // feste EEPROM-Adresse 0011 - Stunden, LCD-Nacht-Modus Ende (siehe lcdnigh2)
const uint16_t e_alarms = 12;          // feste EEPROM-Adresse 0012 - Alarmdaten (4 x 10 Bytes, siehe alarms[], ohne Alarm-Status)
const uint16_t e_senable = 52;         // feste EEPROM-Adresse 0052 - Sound-Freigabe (siehe s_enable)
const uint16_t e_aclkpos = 53;         // feste EEPROM-Adresse 0053 - Analog-Uhr-Position (siehe aclkpos)
const uint16_t e_bdordr = 54;          // feste EEPROM-Adresse 0054 - Geburtstags-Namen-Reihenfolge (siehe bdordr)
const uint16_t e_bdagem = 55;          // feste EEPROM-Adresse 0055 - Geburtstags-Alters-Anzeigemodus (siehe bdagem)
const uint16_t e_bdlsort = 56;         // feste EEPROM-Adresse 0056 - Geburtstagslisten-Sortierung (siehe bdlsort)
const uint16_t e_bdslist = 57;         // feste EEPROM-Adresse 0057 - Geburtstagsindex-Liste (3 Bytes, siehe bdslist[])
const uint16_t e_lastday = 60;         // feste EEPROM-Adresse 0060 - letzter Tag (siehe lastday)
const uint16_t e_lastmonth = 61;       // feste EEPROM-Adresse 0061 - letzter Monat (siehe lastmonth)
const uint16_t e_lastyear = 62;        // feste EEPROM-Adresse 0062 - letztes Jahr (siehe lastmonth)
const uint16_t e_gong = 63;            // feste EEPROM-Adresse 0063 - Gong-Status (siehe gong)
const uint16_t e_gongh1 = 64;          // feste EEPROM-Adresse 0064 - Gongzeit-Beginn (siehe gongh1)
const uint16_t e_gongh2 = 65;          // feste EERPOM-Adresse 0065 - Gongzeit-Ende (siehe gongh2)
const uint16_t e_gong_snd = 66;        // feste EEPROM-Adresse 0066 - Gong-Sound (siehe gong_snd)
const uint16_t e_gong_vol = 67;        // feste EERPOM-Adresse 0067 - Gong-Lautstrke (siehe gong_vol)
                                       // freier Bereich fr Erweiterungen
const uint16_t e_bddata = 186;         // feste EEPROM-Adresse 0186 - Geburtstagsdaten (115 Eintrge x 34 Bytes mit folgender Struktur):
                                       // Byte 0:     Tag (1-31, oder 255 wenn Eintrag leer)
                                       // Byte 1:     Monat (1-12)
                                       // Byte 2:     Jahr (0-99)
                                       // Byte 3:     Jahrhundert (0-99, oder 255 wenn Jahr unbekannt)
                                       // Byte 4-18:  Vorname (0-15 Zeichen), ungenutzte Zeichen = 255
                                       // Byte 19-33: Nachname (0-15 Zeichen), ungenutzte Zeichen = 255

// Globale Variablen im RAM ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------

static uint8_t lcd_bitmap[2048];       // Bitmap-Speicher fr das LCD, alle Ausgaben an das LCD werden zunchst hier abgelegt und dann per Blocktransfer direkt in den LCD-Speicher geschrieben
static uint8_t rx0buf[256];            // USART0-Empfangspuffer (fr die USB-Kommunikation mit Crumb128)
static uint8_t tx0buf[256];            // USART0-Sendepuffer (fr die USB-Kommunikation mit Crumb128)
static uint8_t rx1buf[256];            // USART1-Empfangspuffer (fr die USB-Kommunikation mit IC7)
static uint8_t tx1buf[256];            // USART1-Sendepuffer (fr die USB-Kommunikation mit IC7)
static char rx0data[50];               // USART0-Empfangsdaten (vom Crumb128)
static char rx1data[50];               // USART1-Empfangsdaten (von IC7)
static char cmdbuf[50];                // Kommandopuffer
volatile uint8_t rx0write;             // Schreib-Zeiger fr USART0-Empfangspuffer
static uint8_t rx0read;                // Lese-Zeiger fr USART0-Empfangspuffer
static uint8_t tx0write;               // Schreib-Zeiger fr USART0-Sendepuffer
volatile uint8_t tx0read;              // Lese-Zeiger fr USART0-Sendepuffer
volatile uint8_t rx1write;             // Schreib-Zeiger fr USART1-Empfangspuffer
static uint8_t rx1read;                // Lese-Zeiger fr USART1-Empfangspuffer
static uint8_t tx1write;               // Schreib-Zeiger fr USART1-Sendepuffer
volatile uint8_t tx1read;              // Lese-Zeiger fr USART1-Sendepuffer
static uint8_t rx0pos;                 // USART0-Empfangsdaten-Position
static uint8_t rx1pos;                 // USART1-Empfangsdaten-Position
static uint8_t cmdport;                // 0-2, Status des Kommandopuffers:
                                       // 0= kein Kommando zu bearbeiten
                                       // 1= Kommando von USB-Port 1 im Puffer (USART0)
                                       // 2= Kommando von USB-Port 2 im Puffer (USART1)
static uint8_t scrshot;                // Screenshot-Steuerung:
                                       // 0= kein Screenshot bearbeiten
                                       // 1= Daten der Analoguhr senden (2048 Bytes, 128 x 16)
                                       // 2= Daten vom Info-Bereich senden (2048 Bytes, 128 x 16)
static char strbuf[30];                // Stringpuffer fr die LCD-Textausgabe
static uint8_t lcd_is_on;              // 0-1, 1= LCD ist eingeschaltet
static uint8_t lang;                   // 0-1, Sprache, 0= deutsch, 1= englisch
static uint8_t lcdinv;                 // 0-1, LCD-Invertierung, 1= LCD komplett invertieren
static uint8_t aclkpos;                // 0-1, Position der Analog-Uhr, 0= links, 1= rechts
static uint8_t nightsyn;               // 0-1, Nachtsynchronisierung, 1= LCD zur DCF-Nacht-Synchronisierung ausschalten
static uint8_t lcdnight;               // 0-1, LCD-Nacht-Modus, 1= LCD-Nacht-Modus eingeschaltet (Hintergrundbeleuchtung aus oder gedimmt)
static uint8_t lcdn_act;               // 0-1, LCD-Nacht-Modus, 1= LCD-Nacht-Modus ist aktiv (Uhrzeit im Nachtzeitraum)
static uint8_t lcdnigm1;               // 0-59, Minuten, LCD-Nacht-Modus Beginn
static uint8_t lcdnigh1;               // 0-23, Stunden, LCD-Nacht-Modus Beginn
static uint8_t lcdnigm2;               // 0-59, Minuten, LCD-Nacht-Modus Ende
static uint8_t lcdnigh2;               // 0-23, Stunden, LCD-Nacht-Modus Ende
static uint8_t lcdnigbr;               // 0-255, Helligkeit der Hintergrundbeleuchtung im Nachtmodus
static uint8_t gong;                   // 0-1, Gong-Status, 1= Gong ist eingeschaltet
static uint8_t gongh1;                 // 0-23, Stunden, Gongzeit-Beginn (Minuten sind immer 0)
static uint8_t gongh2;                 // 0-23, Stunden, Gongzeit-Ende (Minuten sind immer 0)
static uint8_t gong_snd;               // 0-12, Gong-Sound-Nummer
static uint8_t gong_vol;               // 0-10, Gong-Lautstrke (+5 * 17 ergibt Wert fr PWM 102-255)
static uint8_t text_x;                 // 0-127, X-Koordinate fr Textausgabe
static uint8_t text_y;                 // 0-127, Y-Koordinate fr Textausgabe
static uint8_t xlimit;                 // 0-109, X-Endposition fr die Namensausgabe bei Geburtstagen
static uint8_t text_inv;               // 0-1, 1= Textausgabe invertiert
static uint8_t gr_x;                   // 0-127, X-Koordinate fr Zeigerberechnung
static uint8_t gr_y;                   // 0-127, Y-Koordinate fr Zeigerberechnung
static uint8_t dotsize;                // 0-2, Punktgre fr das Zeichnen der Zeiger: 0= 1 Pixel, 1= 3 Pixel, 2= 5 Pixel
static uint8_t timezone;               // 0-1, Zeitzone, 0= MEZ, 1= MESZ
static uint8_t syncstat;               // 0-4, Status der DCF77-Synchronsierung
                                       // 0= Uhr wurde noch nie synchronisert
                                       // 1= Uhr wurde noch nie synchronisert, DCF-Sync-Versuch abgebrochen
                                       // 2= Uhr wurde manuell gestellt
                                       // 3= Uhr wurde mindestens 24h nicht synchronisiert
                                       // 4= Uhr wurde innerhalb 24h synchronisiert
static uint8_t dispmode;               // 0-10, Anzeigemodus im Informationsbereich:
                                       // 0=  Normale Uhrenanzeige
                                       // 1=  Hauptmen
                                       // 2=  Men "Uhr stellen"
                                       // 3=  Men "Einstellungen 1"
                                       // 4=  Men "Informationen"
                                       // 5=  Men "Alarme einstellen"
                                       // 6=  Men "Einstellungen 2"
                                       // 7=  Men "Geburtstage" - Anzeige von 8 Eintrgen in Kurzform
                                       // 8=  Men "Geburtstage" - Anzeige von 4 Eintrgen in Langform
                                       // 9=  Men "Geburtstag eingeben"
                                       // 10= Men "Einstellungen 3"
static uint8_t chg_flag;               // 0-1, nderungsflag; wird gesetzt, wenn der Informationsbereich neu ausgegeben werden muss
static uint8_t alm_flag;               // 0-1, Alarm-Flag; wird gesetzt, wenn die Alarmdaten gendert worden sind und die Check- und Speicher-Routine aufgerufen werden muss
static uint8_t bd_flag;                // 0-1, Geburtstags-Flag; wird gesetzt, wenn die Geburtstagsdaten gendert worden sind und die Check-Routine aufgerufen werden muss
static uint8_t reorg_flag;             // 0-2, Reorganisations-Flag; wird tglich um 0:00 Uhr gesetzt; nach Durchlauf einer Hauptschleife wird der Alarm-Check ausgefhrt und nach einem weiteren
                                       //      Durchlauf der Geburtstags-Check; diese Sequenz verhindert Verzgerungen bei der Zeitausgabe um 0:00 Uhr:
                                       // 0= Ruhezustand
                                       // 1= Reorganisation der Alarme starten
                                       // 2= Reorganisation der Geburtstage starten
static uint8_t s_enable;               // 0-1, Sound-Freigabe, 0= Sound aus, 1= Sound freigegeben
static uint8_t soundstat;              // 0-1, Status der Tonausgabe, 0= aus, 1= ein
static uint8_t dcfbuf1[7];             // DCF-Puffer 1, aktuelle decodierte Uhrzeit:
                                       // Byte 0: 0-1, 0= Winterzeit, 1= Sommerzeit
                                       // Byte 1: 0-59, Minute
                                       // Byte 2: 0-23, Stunde
                                       // Byte 3: 1-31, Kalendertag
                                       // Byte 4: 1-7,  Wochentag
                                       // Byte 5: 1-12, Monat
                                       // Byte 6: 0-99, Jahr
static uint8_t dcfbuf2[7];             // DCF-Puffer 2, decodierte Uhrzeit eine Minute zuvor (korrigiert um eine Minute), gleicher Aufbau wie dcbuf1
static uint8_t dcfbuf3[7];             // DCF-Puffer 3, decodierte Uhrzeit eine Minute zuvor (korrigiert um eine Minute), gleicher Aufbau wie dcbuf1
static uint8_t lastsync[7];            // Zeitpunkt der letzten DCF77-Synchronisierung, gleicher Aufbau wie dcbuf1 bis auf Byte 0: 0-1, 0= keine Daten, 1= gltige Zeit
static uint8_t alarms[5][11];          // Alarm-Parameter, 5 x 11 Bytes fr 3 Alarme, den Geburtstags-Alarm und einen Zwischenspeicher fr das Setzen von Alarmen ber USB:
                                       // Byte  0: 0= aus, 1= einmal, 2= tglich, 3= wchentlich, 4= monatlich; bei Geburtstags-Alarm: 0= aus, 1= ein
                                       // Byte  1: 0-59, Minute; bei Geburtstags-Alarm: 0-59, Minute
                                       // Byte  2: 0-23, Stunde; bei Geburtstags-Alarm: 0-23, Stunde
                                       // Byte  3: 1-31, Kalendertag (nchster Alarm); bei Geburtstags-Alarm: 1-31, Tag
                                       // Byte  4: 1-7, Wochentag (nchster Alarm, wenn Alarm wchentlich)
                                       // Byte  5: 1-12, Monat (nchster Alarm); bei Geburtstags-Alarm: 1-12, Monat
                                       // Byte  6: 0-99, Jahr (nchster Alarm)
                                       // Byte  7: 0-9, Wochentagsprogramm (wenn Alarm wchentlich): 0: Mo, 1: Di, 2: Mi, 3: Do, 4: Fr, 5: Sa, 6: So, 7: Mo-Fr, 8: Mo-Sa, 9: Sa-So
                                       // Byte  8: 0-12, Sound-Nummer
                                       // Byte  9: 0-10, Lautstrke (+5 * 17 ergibt Wert fr PWM 102-255)
                                       // Byte 10: 0-2, Alarm-Status, 1= Alarm ausgelst, 2= Alarm quittiert
static uint8_t alm;                    // 0-2, aktuell editierte Alarmnummer
static uint16_t daychg;                // Tageswechsel-Erkennung; der Wert von daychg wird stndig mit einem Tageswert (Monat * 31 + Tag) verglichen, bei Differenzen wird eine Reorganisation
                                       // durchgefhrt (Alarme, Geburtstage Normal-/Sommerzeit)
static uint8_t usb_act;                // 0-1, USB-Kommunikation aktiv, 0= aus, 1= ein
static int8_t encmem;                  // letzte bearbeitete Position des Drehimpulsgebers
static int8_t mainopt;                 // gewhlte Hauptmen-Option
static int8_t submopt;                 // gewhlte Untermen-Option
static int8_t editmode;                // 0-1, Editormodus, 1= Eingaben vom Drehimpulsgeber ndern den Wert von editval
static int16_t editval;                // aktueller Eingabewert im Editormodus
static int16_t editmin;                // unterer Grenzwert im Editormodus
static int16_t editmax;                // oberer Grenzwert im Editormodus
static uint8_t m_minute;               // 0-59, Minuten-Wert bei manueller Einstellung
static uint8_t m_hour;                 // 0-23, Stunden-Wert bei manueller Einstellung
static uint8_t m_day;                  // 1-31, Tages-Wert bei manueller Einstellung
static uint8_t m_month;                // 1-12, Monats-Wert bei manueller Einstellung
static uint8_t m_year;                 // 0-99, Jahres-Wert bei manueller Einstellung
static uint8_t t_second;               // 0-59, Sekunden-Wert beim Setzen der Uhr ber USB
static uint8_t t_minute;               // 0-59, Minuten-Wert beim Setzen der Uhr ber USB
static uint8_t t_hour;                 // 0-23, Stunden-Wert beim Setzen der Uhr ber USB, Jahrhundert-Wert beim Setzen eines Geburtstages ber USB
static uint8_t t_day;                  // 1-31, Tages-Wert beim Setzen der Uhr ber USB, Tages-Wert beim Setzen eines Geburtstages ber USB
static uint8_t t_month;                // 1-12, Monats-Wert beim Setzen der Uhr ber USB, Monats-Wert beim Setzen eines Geburtstages ber USB
static uint8_t t_year;                 // 0-99, Jahres-Wert beim Setzen der Uhr ber USB, Jahres-Wert beim Setzen eines Geburtstages ber USB
static uint8_t t_zone;                 // 0-1, Zeitzone beim Setzen der Uhr ber USB
static uint8_t intc78us;               // 0-7, Interrupt-Zhler 1, 78,125s, 12800Hz
static uint8_t intc625us;              // 0-31, Interrupt-Zhler 2, 625s, 1600Hz
static uint8_t intc20ms;               // 0-49, Interrupt-Zhler 3, 20ms, 50Hz
static uint8_t second;                 // 0-59, Sekunden-Zhler
static uint8_t minute;                 // 0-59, Minuten-Zhler
static uint8_t hour;                   // 0-23, Stunden-Zhler
static uint8_t wday;                   // 1-7,  Wochentag-Zhler
static uint8_t day;                    // 1-31, Tages-Zhler
static uint8_t month;                  // 1-12, Monats-Zhler
static uint8_t year;                   // 0-99, Jahres-Zhler
static uint8_t intc20ms_2;             // 0-49, zustzlicher Interruptzhler (20ms) fr Timeout-Zhler
static uint8_t intcsec;                // 0-59, zustzlicher Interruptzhler (1s) fr Timeout-Zhler
static uint8_t intcmin;                // 0-59, zustzlicher Interruptzhler (1min) fr Timeout-Zhler
static uint8_t sec_flag;               // 0-1, 1= ISR Timer1 signalisiert Sekundenwechsel
static uint8_t encstat1;               // Drehimpulsgeber, Status beim letzten Interrupt
static uint8_t encstat2;               // Drehimpulsgeber, letzter stabiler Status
static int16_t encstep;                // Drehimpulsgeber, aktuelle Schritt-Position im Zweierkomplement
static int8_t encpos;                  // Drehimpulsgeber, aktueller Wert im Zweierkomplement, abhngig vom Modus des Gebers
static uint8_t key1old;                // Taster 1, Zustand beim letzten Interrupt
static uint8_t key1stat;               // Taster 1, aktueller stabiler Zustand (0/1)
static uint8_t key1quit;               // Taster 1, Quittierungs-Status (0/1)
static uint8_t key2old;                // Taster 2, Zustand beim letzten Interrupt
static uint8_t key2stat;               // Taster 2, aktueller stabiler Zustand (0/1)
static uint8_t key2quit;               // Taster 2, Quittierungs-Status (0/1)
static uint8_t dcfstat1;               // DCF77, Signal-Status beim vorherigen Interrupt
static uint8_t dcfstat2;               // DCF77, Signalisierung des Minutenwechsels (0/1)
static uint8_t dcfcnt;                 // DCF77-Zhler zur Ermittlung der Impulslnge
static uint8_t dcfimp;                 // DCF77-Zhler zur Ermittlung der Impulsanzahl
static uint8_t dcfinv;                 // DCF77-Invertierung, wird von EEPROM gelesen:
                                       // 0= DCF-Empfnger liefert Low-Impulse
                                       // >0= DCF-Empfnger liefert High-Impulse
static uint8_t dcfpup;                 // DCF77-Pull-Up, wird von EEPROM gelesen: 1= Pull-Up Widerstand am DCF-Eingang eingeschaltet
static uint8_t disptout;               // Anzeige-Timeout-Zhler, nach Ablauf der Zeit springt die Anzeige ins Hauptmen; wird vom Hauptprogramm gesetzt und vom Timer1-Interrupt im Sekundentakt auf 0
                                       // gezhlt, Hauptprogramm setzt nach Bearbeitung auf 255 (Stopp)
static uint8_t usbtout;                // USB-Symbol-Timeout-Zhler, nach Ablauf der Zeit wird das USB-Symbol deaktiviert; wird vom Hauptprogramm gesetzt und vom Timer1-Interrupt im Sekundentakt auf
                                       // 0 gezhlt, Hauptprogramm setzt nach Bearbeitung auf 255 (Stopp)
static uint8_t syntout;                // DCF77-Synchronisierungs-Timeout, nach Ablauf der Zeit wird das zur Synchronisierung abgeschaltete LCD wieder eingeschaltet; wird vom Hauptprogramm gesetzt
                                       // und vom Timer1-Interrupt im Minutentakt auf 0 gezhlt, Hauptprogramm setzt nach Bearbeitung auf 255 (Stopp)
static uint8_t dcftout;                // DCF77-Timeout-Zhler, nach Ablauf der Zeit wird der Status "DCF77-Ausfall" gesetzt; wird vom Hauptprogramm gesetzt und vom Timer1-Interrupt im Stundentakt
                                       // auf 0 gezhlt, Hauptprogramm setzt nach Bearbeitung auf 255 (Stopp)
static uint8_t nightout;               // Nacht-Helligkeits-Timeout, bei der Einstellung der Nacht-Helligkeit wird kurzzeitig die Hintergrundbeleuchtung auf den eingestellten Wert gesetzt; wird vom
                                       // Hauptprogramm gesetzt und vom Timer1-Interrupt im Stundentakt auf 0 gezhlt, Hauptprogramm setzt nach Bearbeitung auf 255 (Stopp)
static uint8_t snstat;                 // Status der Sound-Erzeugung:
                                       // 0= aus (Ruhezustand)
                                       // 1= Start (wird vom Hauptprogramm gesetzt)
                                       // 2= aktiv (wird von Interrupt-Routine nach dem Start gesetzt, nach Ablauf des Sound-Programms wird in den Stopp-Zustand gewechselt)
                                       // 3= Stopp (wird vom Hauptprogramm oder der Sound-Erzeugung selbst zur Alarm-Abschaltung gesetzt, die Sound-Erzeugung wird sofort gestoppt und anschlieend in
                                       // den Ruhezustand gesetzt)
static PGM_P sndptr;                   // Zeiger auf ausgewhlte Soundnummer
static uint8_t sndnot;                 // 0-72, Notenwert, 254= Pause, 255= Ende
static uint8_t sndlen;                 // 0-255, Tonlnge in 20ms-Einheiten
static uint8_t snddec;                 // 0-255, Decay-Wert in Schritten je 20ms
static uint8_t curvol;                 // 0-255, aktueller Lautstrkewert
static uint8_t maxvol;                 // 0-255, maximaler Lautstrkewert
static uint8_t bdordr;                 // 0-1, Anzeige-Reihenfolge der Geburtstagsdaten: 0= Vorname Nachname, 1= Nachname Vorname
static uint8_t bdindx;                 // 0-114, Wert aus dem Geburtstagsindex
static uint8_t bddata;                 // Wert oder Zeichen aus den Geburtstagsdaten
static int8_t bdipos;                  // 0-114, aktuelle Position in der Geburtstagsliste
static int8_t bd1pos;                  // 0-7, aktuelle Position von bdlpos im Men "Geburtstage 1"
static int8_t bd2pos;                  // 0-3, aktuelle Position von bdlpos im Men "Geburtstage 2"
static uint8_t bdagem;                 // 0-1, Anzeigemodus fr das Alter bei Geburtstagen, 0= aus, 1= ein
static uint8_t bdlsort;                // 0-2, Sortierungs-Modus der Geburtstagsliste: 0= Datum, 1= Vorname, 2= Nachname
static uint8_t bd_change;              // 0-1, Flag fr nderung eines Geburtstags-Eintrages
static uint8_t bdeday;                 // 0-31, Geburtstag eingeben: Geburtstag (0= Eintrag lschen)
static uint8_t bdemonth;               // 1-12, Geburtstag eingeben: Geburtsmonat
static uint8_t bdeyear;                // 0-99, Geburtstag eingeben: Geburtsjahr
static uint8_t bdeyearh;               // 0-20, Geburtstag eingeben: Geburtsjahrhundert
static char bde1buf[15];               // Geburtstag eingeben: Vorname
static char bde2buf[15];               // Geburtstag eingeben: Nachname
static char bdtxbuf[15];               // Eingabepuffer fr Vor- oder Nachname
static uint8_t bdtxpos;                // 0-29, aktuelle Cursorposition im Eingabepuffer
static uint8_t bdtxlen;                // 0-109, Lnge des Eingabepuffers in Pixel
static uint8_t bdslist[3];             // Geburtstagsindex-Kurzliste (3 Bytes); enthlt die nchsten 3 Geburtstage, diese werden zustzlich ins EEPROM kopiert, um sie nach einem Reset sofort anzeigen
                                       // zu knnnen
static uint8_t bdilist[115];           // Geburtstagsindex-Liste (115 Bytes); diese wird tglich um 0:00 Uhr und nach nderungen neu angelegt; die Sortierung ist abhngig vom Wert bdlsort:
                                       // 0= Datum; der erste Eintrag enthlt die Datensatznummer des nchsten Geburtstages, der 2. Eintrag die Nummer des bernchsten Geburtstages usw.; ungenutzte
                                       // Eintrge enthalten den Wert 255
                                       // 1= Vorname; Sortierung nach den ersten 2 Buchstaben des Vornamens, bei gleichen Vornamen wird zustzlich der 1. Buchstabe das Nachnamens verglichen
                                       // 2= Nachname; Sortierung nach den ersten 2 Buchstaben des Nachnamens, bei gleichen Nachnamen wird zustzlich der 1. Buchstabe das Vornamens verglichen
static uint8_t dcftab[42];             // DCF77-Tabelle, enthlt empfangene Bits, eingehende Bits werden in das letzte Byte eingeschoben und alle weiteren Bytes um eine Position weitergerckt; die
                                       // Bytes enthalten nach einem kompletten Minutenzyklus folgende Daten:
                                       // Byte 00-00: Zeitzone, 0= Winterzeit, 1= Sommerzeit
                                       // Byte 01-01: Zeitzone, 1= Winterzeit, 0= Sommerzeit
                                       // Byte 02-02: Schaltsekunde (nicht ausgewertet)
                                       // Byte 03-03: Startbit (nicht ausgewertet)
                                       // Byte 04-07: Minute Einer
                                       // Byte 08-10: Minute Zehner
                                       // Byte 11-11: Prfbit 1 (Minuten)
                                       // Byte 12-15: Stunde Einer
                                       // Byte 16-17: Stunde Zehner
                                       // Byte 18-18: Prfbit 2 (Stunden)
                                       // Byte 19-22: Kalendertag Einer
                                       // Byte 23-24: Kalendertag Zehner
                                       // Byte 25-27: Wochentag
                                       // Byte 28-31: Monat Einer
                                       // Byte 32-32: Monat Zehner
                                       // Byte 33-36: Jahr Einer
                                       // Byte 37-40: Jahr Zehner
                                       // Byte 41-41: Prfbit 3 (Datum)
static uint8_t lastday;                // letzter Tag, wird beim Tageswechsel im EEPROM gespeichert
static uint8_t lastmonth;              // letzter Monat, wird beim Tageswechsel im EEPROM gespeichert
static uint8_t lastyear;               // letztes Jahr, wird beim Tageswechsel im EEPROM gespeichert


// Funktionen =========================================================================================================================================================================================

// Ein Byte von einer bestimmten EEPROM-Adresse lesen (aus AVR-GCC-Tutorial von http://www.mikrocontroller.net) ---------------------------------------------------------------------------------------

uint8_t eep_read_byte(uint16_t addr)                                                               // addr: EEPROM-Adresse (0-4095)
{
   return eeprom_read_byte((uint8_t *) addr);                                                      // Byte ber die interne EEPROM-Routine lesen
}

// Ein Byte auf eine bestimmte EEPROM-Adresse schreiben (aus AVR-GCC-Tutorial von http://www.mikrocontroller.net) -------------------------------------------------------------------------------------

void eep_write_byte(uint16_t addr, uint8_t val)                                                    // addr: EEPROM-Adresse (0-4095), val: zu schreibendes Byte
{
   eeprom_write_byte((uint8_t *) addr, val);                                                       // Byte ber die interne EEPROM-Routine schreiben
}

// Ein Byte (Datenbyte oder Steuerzeichen) auf dem LCD ausgeben ---------------------------------------------------------------------------------------------------------------------------------------

void lcd_out (uint8_t databyte, uint8_t datamode)                                                  // databyte: auszugebendes Datenbyte
                                                                                                   // datamode: 0= Daten, 1= Command
{
   #define lcd_wr 0                                                                                // LCD Steuerleitung /WR
   #define lcd_rd 1                                                                                // LCD Steuerleitung /RD
   #define lcd_ce 2                                                                                // LCD Steuerleitung /CE
   #define lcd_cd 3                                                                                // LCD Steuerleitung C/D

   if (lcd_is_on)                                                                                  // wenn LCD eingeschaltet ist
   {
      PORTC = PORTC & ~ (1 << lcd_ce);                                                             // CE auf Low
      PORTC = PORTC & ~ (1 << lcd_rd);                                                             // RD auf Low
      asm volatile ("nop");
      asm volatile ("nop");                                                                        // mindestens 150ns warten
      while ((PINA & 0x03) != 0x03);                                                               // Warten bis STA0 und STA1 High sind
      PORTC = PORTC | (1 << lcd_rd);                                                               // RD wieder auf High
      if (!datamode) PORTC = PORTC & ~ (1 << lcd_cd);                                              // Datenmodus -> C/D auf Low
      PORTA = databyte;                                                                            // Datenbyte ausgeben
      DDRA = 0xff;                                                                                 // Datenleitungen fr LCD auf Ausgang
      PORTC = PORTC & ~ (1 << lcd_wr);                                                             // WR auf Low
      asm volatile ("nop");                                                                        // mindestens 80ns warten
      PORTC = PORTC | (1 << lcd_wr);                                                               // WR wieder auf High
      DDRA = 0;                                                                                    // Datenleitungen fr LCD auf Eingang
      PORTA = 0xff;                                                                                // Pull-Up auf allen Datenleitungen
      PORTC = PORTC | (1 << lcd_ce) | (1 << lcd_cd);                                               // Steuerleitungen wieder auf High
   }
}

// Bitmap-Daten fr die Analoguhr aus dem Flash-Speicher in das RAM kopieren --------------------------------------------------------------------------------------------------------------------------

void get_clkdata (void)
{
   uint16_t i;                                                                                     // Bytezhler (0-2047)
   for (i = 0; i < 2048; i ++)                                                                     // Bitmap-Daten aus Flash-Speicher holen
      lcd_bitmap[i] = pgm_read_byte (&clockdata_array[i]);                                         // und im RAM ablegen
}

// Bitmap-Speicher lschen ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

void ram_clear (void)
{
   uint16_t i;                                                                                     // Bytezhler (0-2047)
   for (i = 0; i < 2048; i ++) lcd_bitmap[i] = 0;                                                  // Speicherzelle lschen
}

// Ein Pixel in der LCD-Bitmap im RAM setzen ----------------------------------------------------------------------------------------------------------------------------------------------------------

void lcd_pixel (uint8_t x, uint8_t y)                                                              // x: X-Position
                                                                                                   // y: Y-Position
{
   uint16_t adr;                                                                                   // Adresse in der LCD-Bitmap
   uint8_t  data;                                                                                  // Datenbyte

   adr = y * 16 + (x >> 3);                                                                        // Adresse des Datenbytes berechnen
   data = lcd_bitmap[adr];                                                                         // Datenbyte holen
   data = data | (1 << (7 - (x & 0x07)));                                                          // entsprechendes Pixel-Bit setzen
   lcd_bitmap[adr] = data;                                                                         // Datenbyte wieder speichern
}

// Ein Pixel in der LCD-Bitmap entsprechend der gewnschten Linienstrke setzen -----------------------------------------------------------------------------------------------------------------------

void lcd_pixel2 (uint8_t x, uint8_t y)                                                             // x: X-Position
                                                                                                   // y: Y-Position
{
   lcd_pixel (x, y);                                                                               // Pixel an der gewnschten Stelle setzen
   if (dotsize > 0)                                                                                // wenn Punktgre > 0
   {
      lcd_pixel (x, y - 1);                                                                        // zustzlich folgende Pixel setzen:
      lcd_pixel (x + 1, y);                                                                        // ....o....
      lcd_pixel (x, y + 1);                                                                        // ...o.o...
      lcd_pixel (x - 1, y);                                                                        // ....o....
   }
   if (dotsize > 1)                                                                                // wenn Punktgre > 1
   {
      lcd_pixel (x, y - 2);                                                                        // zustzlich folgende Pixel setzen:
      lcd_pixel (x + 1, y - 2);
      lcd_pixel (x + 1, y - 1);
      lcd_pixel (x + 2, y - 1);
      lcd_pixel (x + 2, y);
      lcd_pixel (x + 2, y + 1);
      lcd_pixel (x + 1, y + 1);                                                                    // ...ooo...
      lcd_pixel (x + 1, y + 2);                                                                    // ..oo.oo..
      lcd_pixel (x, y + 2);                                                                        // ..o...o..
      lcd_pixel (x - 1, y + 2);                                                                    // ..oo.oo..
      lcd_pixel (x - 1, y + 1);                                                                    // ...ooo...
      lcd_pixel (x - 2, y + 1);
      lcd_pixel (x - 2, y);
      lcd_pixel (x - 2, y - 1);
      lcd_pixel (x - 1, y - 1);
      lcd_pixel (x - 1, y - 2);
   }
}

// Eine Linie in der LCD-Bitmap nach dem Bresenham-Algorithmus zeichnen, Quelle: Wikipedia ------------------------------------------------------------------------------------------------------------

void lcd_draw (uint8_t x1, uint8_t y1, uint8_t x2, uint8_t y2)                                     // x1, y1: Koordinate Startpunkt
                                                                                                   // x2, y2: Koordinate Zielpunkt
{
   uint8_t x, y, t;
   int8_t  dx, dy, incx, incy, pdx, pdy, ddx, ddy, es, el, err;

   if ((x1 | y1 | x2 | y2) < 128)                                                                  // nur gltige Koordinaten verwenden
   {
      dx = x2 - x1;                                                                                // X Differenz ermitteln
      dy = y2 - y1;                                                                                // Y Differenz ermitteln
      incx = 0;
      if (dx > 0) incx = 1;
      if (dx < 0) incx = -1;                                                                       // X Richtung ermitteln
      incy = 0;
      if (dy > 0) incy = 1;
      if (dy < 0) incy = -1;                                                                       // Y Richtung ermitteln
      if (dx < 0) dx = -dx;                                                                        // X Differenz in positiven Wert wandeln
      if (dy < 0) dy = -dy;                                                                        // Y Differenz in positiven Wert wandeln
      if (dx > dy)                                                                                 // wenn X Differenz grer
      {
         pdx = incx;
         pdy = 0;                                                                                  // pdx, pdy ist ein Parallelschritt
         ddx = incx;
         ddy = incy;                                                                               // ddx, ddy ist ein Diagonalschritt
         es = dy;
         el = dx;                                                                                  // Fehlerschritte
      }
      else                                                                                         // wenn Y Differenz grer
      {
         pdx = 0;
         pdy = incy;                                                                               // pdx, pdy ist ein Parallelschritt
         ddx = incx;
         ddy = incy;                                                                               // ddx, ddy ist ein Diagonalschritt
         es = dx;
         el = dy;                                                                                  // Fehlerschritte
      }
      x = x1;
      y = y1;
      err = el / 2;                                                                                // Initialisierungen vor Schleifenbeginn
      lcd_pixel2 (x, y);                                                                           // Anfangsposition ausgeben
      for (t = 0; t < el; t ++)                                                                    // t = Pixelzhler
      {
         err = err - es;                                                                           // Fehler aktualisieren
         if (err < 0)                                                                              // wenn Fehler < 0
         {
            err = err + el;                                                                        // Fehlerterm wieder positiv machen
            x = x + ddx;                                                                           // Schritt in langsame Richtung,
            y = y + ddy;                                                                           // Diagonalschritt
         }
         else
         {
            x = x + pdx;                                                                           // Schritt in schnelle Richtung,
            y = y + pdy;                                                                           // Parallelschritt
         }
         lcd_pixel2 (x, y);                                                                        // berechnetes Pixel ausgeben
      }
   }
}

// Ein ASCII-Zeichen in die LCD-Bitmap schreiben ------------------------------------------------------------------------------------------------------------------------------------------------------

void lcd_char (char c)                                                                             // c: auszugebendes Zeichen
{
   uint16_t offset;                                                                                // Offset in der Zeichensatz-Tabelle
   uint8_t  cwidth;                                                                                // Pixelbreite des Zeichens
   uint8_t  x;                                                                                     // Zhler fr die Zeichenausgabe, X-Position
   uint8_t  y;                                                                                     // Zhler fr die Zeichenausgabe, Y-Position
   uint8_t  b;                                                                                     // Zwischenspeicher fr Bytes aus der Zeichensatz-Tabelle

   if (c >= 32)                                                                                    // Ausgabe nur bei zulssigem Zeichencode
   {
      offset = (c - 32) * 12;                                                                      // Tabellen-Offset berechnen
      cwidth = pgm_read_byte (&charset_array[offset]);                                             // Zeichenbreite holen
      if (cwidth > 8) cwidth = 8;                                                                  // Zeichenbreite auf 8 begrenzen
      for (y = 0; y < 11; y ++)                                                                    // 11 Zeilenbytes ausgeben
      {
         if (text_inv) lcd_pixel (text_x, text_y + y);                                             // ggf. linkes Pixel setzen
         b = pgm_read_byte (&charset_array[offset + y + 1]);                                       // Zeilenbyte holen
         for (x = cwidth; x > 0; x --)                                                             // Zeilenbyte bitweise ausgeben
         {
            if ((b ^ text_inv) & 0x01)                                                             // wenn Pixel gesetzt werden muss
               lcd_pixel (text_x + x, text_y + y);                                                 // LSB ausgeben
            b = b >> 1;                                                                            // nchste Bit-Position an LSB
         }
      }
      text_x = text_x + cwidth + 1;                                                                // X-Text-Position fr nchstes Zeichen
   }
}

// Eine Kleinziffer (Tag & Monat) in die LCD-Bitmap schreiben -----------------------------------------------------------------------------------------------------------------------------------------

void lcd_digit0 (uint8_t d)                                                                        // d: auszugebende Ziffer (0-9)
{
   uint16_t offset;                                                                                // Offset in der Zeichensatz-Tabelle
   uint8_t  x;                                                                                     // Zhler fr die Zeichenausgabe, X-Position
   uint8_t  y;                                                                                     // Zhler fr die Zeichenausgabe, Y-Position
   uint8_t  b;                                                                                     // Zwischenspeicher fr Bytes aus der Zeichensatz-Tabelle

   offset = d * 5;                                                                                 // Tabellen-Offset berechnen
   for (y = 0; y < 5; y ++)                                                                        // 5 Zeilen mit je einem Byte ausgeben
   {
      b = pgm_read_byte (&digit0_array[offset ++]);                                                // Zeilenbyte holen
      for (x = 4; x > 0; x --)                                                                     // Zeilenbyte bitweise ausgeben
      {
         if ((b ^ text_inv) & 0x01)                                                                // wenn Pixel gesetzt werden muss
         lcd_pixel (text_x + x - 1, text_y + y);                                                   // LSB ausgeben
         b = b >> 1;                                                                               // nchste Bit-Position an LSB
      }
   }
   text_x = text_x + 5;                                                                            // X-Text-Position fr nchstes Zeichen
}

// Eine Groziffer (Stunden & Minuten) in die LCD-Bitmap schreiben ------------------------------------------------------------------------------------------------------------------------------------

void lcd_digit1 (uint8_t d)                                                                        // d: auszugebende Ziffer (0-9)
{
   uint16_t offset;                                                                                // Offset in der Zeichensatz-Tabelle
   uint8_t  x;                                                                                     // Zhler fr die Zeichenausgabe, X-Position
   uint8_t  y;                                                                                     // Zhler fr die Zeichenausgabe, Y-Position
   uint8_t  b;                                                                                     // Zwischenspeicher fr Bytes aus der Zeichensatz-Tabelle

   offset = d * 52;                                                                                // Tabellen-Offset berechnen
   for (y = 0; y < 26; y ++)                                                                       // 27 Zeilen mit je 2 Bytes ausgeben
   {
      b = pgm_read_byte (&digit1_array[offset ++]);                                                // Zeilenbyte 1 holen
      for (x = 6; x > 0; x --)                                                                     // Zeilenbyte bitweise ausgeben
      {
         if ((b ^ text_inv) & 0x01)                                                                // wenn Pixel gesetzt werden muss
         lcd_pixel (text_x + x - 1, text_y + y);                                                   // LSB ausgeben
         b = b >> 1;                                                                               // nchste Bit-Position an LSB
      }
      b = pgm_read_byte (&digit1_array[offset ++]);                                                // Zeilenbyte 2 holen
      for (x = 8; x > 0; x --)                                                                     // Zeilenbyte bitweise ausgeben
      {
         if ((b ^ text_inv) & 0x01)                                                                // wenn Pixel gesetzt werden muss
         lcd_pixel (text_x + x + 5, text_y + y);                                                   // LSB ausgeben
         b = b >> 1;                                                                               // nchste Bit-Position an LSB
      }
   }
}

// Eine Groziffer (Sekunden) in die LCD-Bitmap schreiben ---------------------------------------------------------------------------------------------------------------------------------------------

void lcd_digit2 (uint8_t d)                                                                        // d: auszugebende Ziffer (0-9)
{
   uint16_t offset;                                                                                // Offset in der Zeichensatz-Tabelle
   uint8_t  x;                                                                                     // Zhler fr die Zeichenausgabe, X-Position
   uint8_t  y;                                                                                     // Zhler fr die Zeichenausgabe, Y-Position
   uint8_t  b;                                                                                     // Zwischenspeicher fr Bytes aus der Zeichensatz-Tabelle

   offset = d * 36;                                                                                // Tabellen-Offset berechnen
   for (y = 0; y < 18; y ++)                                                                       // 18 Zeilen mit je 2 Bytes ausgeben
   {
      b = pgm_read_byte (&digit2_array[offset ++]);                                                // Zeilenbyte 1 holen
      for (x = 2; x > 0; x --)                                                                     // Zeilenbyte bitweise ausgeben
      {
         if ((b ^ text_inv) & 0x01)                                                                // wenn Pixel gesetzt werden muss
         lcd_pixel (text_x + x - 1, text_y + y);                                                   // LSB ausgeben
         b = b >> 1;                                                                               // nchste Bit-Position an LSB
      }
      b = pgm_read_byte (&digit2_array[offset ++]);                                                // Zeilenbyte holen
      for (x = 8; x > 0; x --)                                                                     // Zeilenbyte bitweise ausgeben
      {
         if ((b ^ text_inv) & 0x01)                                                                // wenn Pixel gesetzt werden muss
         lcd_pixel (text_x + x + 1, text_y + y);                                                   // LSB ausgeben
         b = b >> 1;                                                                               // nchste Bit-Position an LSB
      }
   }
}

// Ein Gro-Symbol in die LCD-Bitmap schreiben --------------------------------------------------------------------------------------------------------------------------------------------------------

void lcd_symbol (uint8_t d)                                                                        // d: auszugebende Symbolnummer (0-7)
{
   uint16_t offset;                                                                                // Offset in der Zeichensatz-Tabelle
   uint8_t  x;                                                                                     // Zhler fr die Zeichenausgabe, X-Position
   uint8_t  y;                                                                                     // Zhler fr die Zeichenausgabe, Y-Position
   uint8_t  b;                                                                                     // Zwischenspeicher fr Bytes aus der Zeichensatz-Tabelle

   offset = d * 32;                                                                                // Tabellen-Offset berechnen
   for (y = 0; y < 16; y ++)                                                                       // 16 Zeilen mit je 2 Bytes ausgeben
   {
      b = pgm_read_byte (&symbol_array[offset ++]);                                                // Zeilenbyte 1 holen
      for (x = 8; x > 0; x --)                                                                     // Zeilenbyte bitweise ausgeben
      {
         if ((b ^ text_inv) & 0x01)                                                                // wenn Pixel gesetzt werden muss
         lcd_pixel (text_x + x - 1, text_y + y);                                                   // LSB ausgeben
         b = b >> 1;                                                                               // nchste Bit-Position an LSB
      }
      b = pgm_read_byte (&symbol_array[offset ++]);                                                // Zeilenbyte 2 holen
      for (x = 8; x > 0; x --)                                                                     // Zeilenbyte bitweise ausgeben
      {
         if ((b ^ text_inv) & 0x01)                                                                // wenn Pixel gesetzt werden muss
         lcd_pixel (text_x + x + 7, text_y + y);                                                   // LSB ausgeben
         b = b >> 1;                                                                               // nchste Bit-Position an LSB
      }
   }
}

// Einen String aus dem Flash-Speicher in die LCD-Bitmap schreiben ------------------------------------------------------------------------------------------------------------------------------------

void lcd_str (uint8_t strnr, uint8_t len)                                                          // strnr: String-Nummer im Flash-Speicher
                                                                                                   // len: Anzahl der auszugebenden Zeichen (0= alle Zeichen ausgeben)
{
   uint8_t i;                                                                                      // Zeichenzhler
   PGM_P   p;                                                                                      // Pointer auf Stringdaten im Flash

   if (lang) p = (PGM_P) pgm_read_word (&stre[strnr]);                                             // falls Englisch gewhlt
   else p = (PGM_P) pgm_read_word (&strd[strnr]);                                                  // sonst Deutsch
   strcpy_P (strbuf, p);                                                                           // String in den RAM-Puffer kopieren
   i = 0;                                                                                          // Zeichenzhler auf erstes Zeichen
   while (strbuf[i] > 0)                                                                           // Schleife bis zum Endezeichen (0)
   {
      if (!len) lcd_char (strbuf[i]);                                                              // wenn Lnge=0 -> alle Zeichen ausgeben
      else if (i < len) lcd_char (strbuf[i]);                                                      // sonst nur bis zur vorgegebenen Lnge
      i ++;                                                                                        // nchstes Zeichen
   }
}

// Eine zweistellige Zahl mit kleinem Zeichensatz (0-99) in die LCD-Bitmap schreiben ------------------------------------------------------------------------------------------------------------------

void lcd_numbs (uint8_t n)                                                                         // n: auszugebende Zahl
{
   lcd_digit0 (n / 10);                                                                            // Zehnerstelle als Kleinziffer ausgeben
   lcd_digit0 (n % 10);                                                                            // Einersteller als Kleinziffer ausgeben
}

// Eine zweistellige Zahl (0-99) in die LCD-Bitmap schreiben ------------------------------------------------------------------------------------------------------------------------------------------

void lcd_numb (uint8_t n)                                                                          // n: auszugebende Zahl
{
   lcd_char (n / 10 + 48);                                                                         // Zehnerstelle ausgeben
   lcd_char (n % 10 + 48);                                                                         // Einersteller ausgeben
}

// Eine dreistellige Zahl (0-999) in die LCD-Bitmap schreiben -----------------------------------------------------------------------------------------------------------------------------------------

void lcd_numb3 (uint16_t n)                                                                        // n: auszugebende Zahl
{
   lcd_char (n / 100 + 48);                                                                        // Hunderterstelle ausgeben
   lcd_numb (n % 100);                                                                             // Restwert zweistellig ausgeben
}

// Eine vierstellige Zahl (0-9999) in die LCD-Bitmap schreiben ----------------------------------------------------------------------------------------------------------------------------------------

void lcd_numb4 (uint16_t n)                                                                        // n: auszugebende Zahl
{
   lcd_char (n / 1000 + 48);                                                                       // Tausenderstelle ausgeben
   n = n % 1000;                                                                                   // Restwert berechnen
   lcd_char (n / 100 + 48);                                                                        // Hunderterstelle ausgeben
   lcd_numb (n % 100);                                                                             // Restwert zweistellig ausgeben
}

// Zeit im Format hh:mm in die LCD-Bitmap schreiben ---------------------------------------------------------------------------------------------------------------------------------------------------

void lcd_time (uint8_t h, uint8_t m)                                                               // h: auszugebender Stundenwert
                                                                                                   // m: auszugebender Minutenwert
{
   lcd_numb (h);                                                                                   // Stunden ausgeben
   lcd_char (':');                                                                                 // Trennzeichen ausgeben
   lcd_numb (m);                                                                                   // Minuten ausgeben
}

// Datum im Format dd.mm.yy in die LCD-Bitmap schreiben -----------------------------------------------------------------------------------------------------------------------------------------------

void lcd_date (uint8_t d, uint8_t m, uint8_t y)                                                    // d: auszugebender Tag
                                                                                                   // m: auszugebender Monat
                                                                                                   // y: auszugebendes Jahr
{
   lcd_numb (d);                                                                                   // Tag ausgeben
   lcd_char ('.');                                                                                 // Trennzeichen ausgeben
   lcd_numb (m);                                                                                   // Monat ausgeben
   lcd_char ('.');                                                                                 // Trennzeichen ausgeben
   lcd_numb (y);                                                                                   // Jahr ausgeben
}

// Eine Byte hexadezimal ausgeben (fr Testzwecke) ----------------------------------------------------------------------------------------------------------------------------------------------------

void lcd_hex (uint8_t n)                                                                           // n: auszugebender Wert
{
   uint8_t h, l;                                                                                   // h: Zwischenspeicher fr oberes Nibble, l: Zwischenspeicher fr unteres Nibble
   h = n / 16;                                                                                     // oberes Nibble ermitteln
   l = n % 16;                                                                                     // unteres Nibble ermitteln
   if (h > 9) lcd_char (h + 55);                                                                   // wenn Wert ber 9 -> A-F ausgeben
   else lcd_char (h + 48);                                                                         // sonst 0-9 ausgeben
   if (l > 9) lcd_char (l + 55);                                                                   // wenn Wert ber 9 -> A-F ausgeben
   else lcd_char (l + 48);                                                                         // sonst 0-9 ausgeben
}

// Eine waagerechte Linie in der LCD-Bitmap zeichnen --------------------------------------------------------------------------------------------------------------------------------------------------

void lcd_line (uint8_t x1, uint8_t y1, uint8_t x2)                                                 // x1: X-Start-Koordinate (0-127)
                                                                                                   // y1: Y-Start-Koordinate (0-127)
                                                                                                   // x2: X-Ende-Koordinate (0-127)
{
   uint8_t i;                                                                                      // Zhlervariable

   for (i = x1; i <= x2; i ++) lcd_pixel (i, y1);                                                  // waagerechte Linie von x1 nach x2
}

// Einen Rahmen in der LCD-Bitmap zeichnen ------------------------------------------------------------------------------------------------------------------------------------------------------------

void lcd_frame (uint8_t x1, uint8_t y1, uint8_t x2, uint8_t y2)                                    // x1: X-Start-Koordinate (0-127)
                                                                                                   // y1: Y-Start-Koordinate (0-127)
                                                                                                   // x2: X-Ende-Koordinate (0-127)
                                                                                                   // y2: Y-Ende-Koordinate (0-127)
{
   uint8_t i;                                                                                      // Zhlervariable

   for (i = x1; i <= x2; i ++)                                                                     // waagerechte Linien von x1 nach x2
   {
      lcd_pixel (i, y1);                                                                           // obere Linie zeichnen
      lcd_pixel (i, y2);                                                                           // untere Linie zeichnen
   }
   for (i = y1; i <= y2; i ++)                                                                     // senkrechte Linien von Y1 nach y2
   {
      lcd_pixel (x1, i);                                                                           // linke Linie zeichnen
      lcd_pixel (x2, i);                                                                           // rechte Linie zeichnen
   }
}

// Pixelposition auf dem Ziffernblatt fr die Zeigerdarstellung ermitteln -----------------------------------------------------------------------------------------------------------------------------

void get_pixelpos (uint8_t r, int16_t a)                                                           // r = Radius (1-55)
                                                                                                   // a = Winkel in Halbgrad (-719 - 719)
{
   while (a < 0) a = a + 720;                                                                      // falls Winkel < 0 -> korrigieren
   while (a > 719) a = a - 720;                                                                    // falls Winkel > 360 -> korrigieren

   if (a < 181)                                                                                    // wenn Winkel im Bereich 0 - 90
   {
      gr_x = pgm_read_byte (&circle_array[(r - 1) * 362 + a * 2]) + 63;
      gr_y = pgm_read_byte (&circle_array[(r - 1) * 362 + a * 2 + 1]) + 63;
   }

   if ((a > 180) && (a < 360))                                                                     // wenn Winkel im Bereich 90 - 179,5
   {
      a = 360 - a;
      gr_x = pgm_read_byte (&circle_array[(r - 1) * 362 + a * 2]) + 63;
      gr_y = - pgm_read_byte (&circle_array[(r - 1) * 362 + a * 2 + 1]) + 63;
   }

   if ((a > 359) && (a < 540))                                                                     // wenn Winkel im Bereich 180 - 269,5
   {
      a = a - 360;
      gr_x = - pgm_read_byte (&circle_array[(r - 1) * 362 + a * 2]) + 63;
      gr_y = - pgm_read_byte (&circle_array[(r - 1) * 362 + a * 2 + 1]) + 63;
   }

   if ((a > 539) && (a < 720))                                                                     // Winkel im Bereich 270 - 359,5
   {
      a = 720 - a;
      gr_x = - pgm_read_byte (&circle_array[(r - 1) * 362 + a * 2]) + 63;
      gr_y = pgm_read_byte (&circle_array[(r - 1) * 362 + a * 2 + 1]) + 63;
   }
}

// Schnelles Lschen des gesamten LCD-Grafikspeichers -------------------------------------------------------------------------------------------------------------------------------------------------

void lcd_clear (void)
{
   uint8_t pixel = 0;                                                                              // normalerweise alle Pixel lschen
   if (lcdinv) pixel = 255;                                                                        // bei LCD-Invertierung alle Pixel setzen
   lcd_out (0, 0);                                                                                 // Anfangsadresse Low-Byte ausgeben
   lcd_out (0, 0);                                                                                 // Anfangsadresse High-Byte ausgeben
   lcd_out (0x24, 1);                                                                              // LCD Command Set Address Pointer
   for (uint16_t adr=0; adr < (30 * 128); adr ++)                                                  // Schleife fr 128 Zeilen mit 30 Spalten
   {
      lcd_out (pixel, 0);                                                                          // Pixel-Daten schreiben
      lcd_out (0xc0, 1);                                                                           // LCD Command Datawrite Auto Increment
   }
}

// LCD einschalten und initialisieren -----------------------------------------------------------------------------------------------------------------------------------------------------------------

void lcd_on (void)
{
   PORTB = PORTB & ~ (1 << PB0);                                                                   // LCD Spannung einschalten
   PORTC = PORTC | 0x0f;                                                                           // alle LCD-Steuerleitungen auf High setzen
   PORTA = 0xff;                                                                                   // PortA, alle Pull-Ups einschalten
   lcd_is_on = 1;                                                                                  // Flag fr eingeschaltetes LCD setzen
                                                                                                   // eventuell weitere Flags fr Neuausgabe der Daten
   _delay_ms (50);                                                                                 // 50ms auf LCD warten
   lcd_out (0, 0);                                                                                 // LCD Grafik-Adresse=0 setzen, Low-Byte
   lcd_out (0, 0);                                                                                 // LCD Grafik-Adresse=0 setzen, High-Byte
   lcd_out (0x42, 1);                                                                              // LCD Command Set Graphic Home Address
   lcd_out (30, 0);                                                                                // LCD Grafik-Area=30 Spalten setzen
   lcd_out (0, 0);                                                                                 // leeres Datenbyte 2 setzen
   lcd_out (0x43, 1);                                                                              // LCD Command Set Graphic Area
   lcd_clear ();                                                                                   // LCD-Speicher schnell lschen
   lcd_out (0x98, 1);                                                                              // LCD Command LCD Grafik on, Text off setzen
   lcd_out (0x80, 1);                                                                              // LCD Command Text/Grafik in OR-Mode setzen
}

// LCD ausschalten ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

void lcd_off (void)
{
   lcd_is_on = 0;                                                                                  // Flag fr eingeschaltetes LCD lschen
   PORTA = 0;                                                                                      // PortA, alle Pull-Ups ausschalten
   PORTC = PORTC & 0xf0;                                                                           // alle LCD-Steuerleitungen auf Low setzen
   PORTB = PORTB | (1 << PB0);                                                                     // LCD Spannung ausschalten
}

// Bitmap-Daten fr die Analoguhr aus dem RAM in den LCD-Speicher kopieren ----------------------------------------------------------------------------------------------------------------------------

void copy_clkdata (void)
{
   uint16_t i = 0;                                                                                 // Bytezhler (0-2047)
   uint16_t lcd_adr;                                                                               // LCD-Speicheradresse
   uint8_t  line;                                                                                  // Zeilenzhler (0-127)
   uint8_t  column;                                                                                // Spaltenzhler (0-15)
   uint8_t  data;                                                                                  // Bitmap-Datenbyte
   uint8_t  data1;                                                                                 // Zwischenspeicher fr bertrag (1 Pixel) in Nachbarbyte

   lcd_adr = 0;                                                                                    // Adresse = 0, entspricht Pixelposition 0 (links)

   if (aclkpos)                                                                                    // wenn Analog-Uhr-Position rechts ->
   {                                                                                               // gesamtes Bild um 1 Pixel nach rechts verschieben
      lcd_adr = 14;                                                                                // Adresse = 14, entspricht Pixelposition 112 (Mitte)
      i = 0;                                                                                       // Bytezhler auf Anfangsposition setzen
      for (line = 0; line < 128; line ++)                                                          // 128 Zeilen verschieben
      {
         data1 = 0;                                                                                // bertrag zunchst lschen
         for (column = 0; column < 16; column ++)                                                  // 16 Zeilenbytes von links beginnend bearbeiten
         {
            data = lcd_bitmap[i];                                                                  // Pixel-Daten aus RAM holen
            lcd_bitmap[i ++] = data1 | (data >> 1);                                                // um 1 Pixel nach rechts verschieben und bertrag ergnzen
            data1 = data << 7;                                                                     // bertrag fr nchstes Byte (1 Pixel von rechts)
         }
      }
   }

   i = 0;                                                                                          // Bytezhler auf Anfangsposition setzen
   for (line = 0; line < 128; line ++)                                                             // 128 Zeilen kopieren
   {
      lcd_out (lcd_adr, 0);                                                                        // LCD-Adresse Low-Byte ausgeben
      lcd_out ((lcd_adr >> 8), 0);                                                                 // LCD-Adresse High-Byte ausgeben
      lcd_out (0x24, 1);                                                                           // LCD Command Set Address Pointer
      for (column = 0; column < 16; column ++)                                                     // 16 Spalten-Bytes kopieren
      {
         data = lcd_bitmap[i ++];                                                                  // Bitmap-Daten aus RAM holen
         if (lcdinv) data = ~ data;                                                                // LCD-Invertierung wenn aktiviert
         lcd_out (data, 0);                                                                        // Pixel-Daten schreiben
         lcd_out (0xc0, 1);                                                                        // LCD Command Datawrite Auto Increment
      }
      lcd_adr = lcd_adr + 30;                                                                      // LCD-Adresse auf nchsten Zeilenbeginn
   }
}

// Bitmap-Daten fr den Infobereich aus dem RAM in den LCD-Speicher kopieren --------------------------------------------------------------------------------------------------------------------------

void copy_infodata (void)
{
   uint16_t i;                                                                                     // Bytezhler (0-2047)
   uint16_t lcd_adr;                                                                               // LCD-Speicheradresse
   uint8_t  line;                                                                                  // Zeilenzhler (0-127)
   int8_t   column;                                                                                // Spaltenzhler (0-13)
   uint8_t  data;                                                                                  // Bitmap-Datenbyte
   uint8_t  data1;                                                                                 // Zwischenspeicher fr bertrag (2 Pixel) in Nachbarbyte

   lcd_adr = 16;                                                                                   // Adresse = 16, entspricht Pixelposition 128 (Mitte)

   if (aclkpos)                                                                                    // wenn Analog-Uhr-Position rechts ->
   {                                                                                               // gesamtes Bild um 2 Pixel nach links verschieben
      lcd_adr = 0;                                                                                 // Adresse = 0, entspricht Pixelposition 0 (links)
      i = 13;                                                                                      // Bytezhler auf Ende von Zeile 0 setzen
      for (line = 0; line < 128; line ++)                                                          // 128 Zeilen verschieben
      {
         data1 = 0;                                                                                // bertrag zunchst lschen
         for (column = 13; column >= 0; column --)                                                 // 14 Zeilenbytes von rechts beginnend bearbeiten
         {
            data = lcd_bitmap[i];                                                                  // Pixel-Daten aus RAM holen
            lcd_bitmap[i --] = data1 | (data << 2);                                                // um 2 Pixel nach links verschieben und bertrag ergnzen
            data1 = data >> 6;                                                                     // bertrag fr nchstes Byte (2 Pixel von links)
         }
         i = i + 30;                                                                               // Bytezhler auf Ende der nchsten Zeile setzen
      }
   }

   i = 0;                                                                                          // Bytezhler auf Anfangsposition setzen
   for (line = 0; line < 128; line ++)                                                             // 128 Zeilen kopieren
   {
      lcd_out (lcd_adr, 0);                                                                        // LCD-Adresse Low-Byte ausgeben
      lcd_out ((lcd_adr >> 8), 0);                                                                 // LCD-Adresse High-Byte ausgeben
      lcd_out (0x24, 1);                                                                           // LCD Command Set Address Pointer
      for (column = 0; column < 14; column ++)                                                     // 14 Spalten-Bytes kopieren
      {
         data = lcd_bitmap[i ++];                                                                  // Bitmap-Daten aus RAM holen
         if (lcdinv) data = ~ data;                                                                // LCD-Invertierung wenn aktiviert
         lcd_out (data, 0);                                                                        // Pixel-Daten schreiben
         lcd_out (0xc0, 1);                                                                        // LCD Command Datawrite Auto Increment
      }
      lcd_adr = lcd_adr + 30;                                                                      // LCD-Adresse auf nchste Zeile
      i = i + 2;                                                                                   // 2 Bytes berspringen
   }
}

// Die Pixelbreite eines Zeichens ermitteln -----------------------------------------------------------------------------------------------------------------------------------------------------------

uint8_t chrl (uint8_t c)                                                                           // Ergebnis: Breite des Zeichens in Pixel
{                                                                                                  // c: Zeichen

   uint16_t offset;                                                                                // Offset in der Zeichensatz-Tabelle
   uint8_t charw;                                                                                  // Pixelbreite des Zeichens

   charw = 0;                                                                                      // zunchst Lnge = 0 setzen
   if (c >= 32)                                                                                    // wenn Zeichen zulssig ist
   {
      offset = (c - 32) * 12;                                                                      // Tabellen-Offset berechnen
      charw = pgm_read_byte (&charset_array[offset]);                                              // Zeichenbreite
      if (charw > 8) charw = 8;                                                                    // Zeichenbreite auf 8 begrenzen
   }
   return charw;
}

// Die Pixellnge eines Strings im Flash-Speicher ermitteln -------------------------------------------------------------------------------------------------------------------------------------------

uint8_t strl (uint8_t strnr, uint8_t len)                                                          // Ergebnis: Lnge des Strings in Pixel
                                                                                                   // strnr: String-Nummer im Flash-Speicher
                                                                                                   // len: Anzahl der auszugebenden Zeichen (0= alle Zeichen ausgeben)
{
   uint8_t i;                                                                                      // Zeichenzhler
   PGM_P p;                                                                                        // Pointer auf Stringdaten im Flash
   uint8_t plen;                                                                                   // Speicher fr Pixellnge

   if (lang) p = (PGM_P) pgm_read_word (&stre[strnr]);                                             // wenn Englisch gewhlt
   else p = (PGM_P) pgm_read_word (&strd[strnr]);                                                  // sonst Deutsch

   plen = 0;                                                                                       // zunchst Pixellnge = 0 setzen
   strcpy_P (strbuf, p);                                                                           // String in den RAM-Puffer kopieren
   i = 0;                                                                                          // Zeichenzhler auf erstes Zeichen
   while (strbuf[i] > 0)                                                                           // alle Zeichen im Puffer bearbeiten
   {
      if (!len) plen = plen + chrl (strbuf[i]) + 1;                                                // wenn Zeichenanzahl = 0 -> gesamten String bearbeiten
      else if (i < len) plen = plen + chrl (strbuf[i]) + 1;                                        // sonst nur bis zur vorgegebenen Zeichenanzahl
      i ++;                                                                                        // nchstes Zeichen
   }
   return plen;
}

// Uhrzeit in groen Ziffern und Datum ausgeben -------------------------------------------------------------------------------------------------------------------------------------------------------

void lcd_timedate (void)
{
   uint8_t temp_x;                                                                                 // Zwischenspeicher fr Text-X-Position

   text_y = 2;                                                                                     // Digitale Groanzeige, Y-Position Stunden
   temp_x = 13;                                                                                    // X-Position bei einstelligen Stunden
   if (hour > 9)                                                                                   // wenn Stunden zweistellig
   {
      temp_x = 22;                                                                                 // X-Position korrigieren
      text_x = 4;                                                                                  // Text-Koordinaten auf Stunden-Zehner
      lcd_digit1 (hour / 10);                                                                      // Stunden-Zehner als Groziffer ausgeben
   }
   text_x = temp_x;                                                                                // Text-Koordinaten auf Stunden-Einer
   lcd_digit1 (hour % 10);                                                                         // Stunden-Einer als Groziffer ausgeben
   text_x = temp_x + 25;                                                                           // Text-Koordinaten auf Minuten-Zehner
   lcd_digit1 (minute / 10);                                                                       // Minuten-Einer als Groziffer ausgeben
   text_x = temp_x + 43;                                                                           // Text-Koordinaten auf Minuten-Einer
   lcd_digit1 (minute % 10);                                                                       // Minuten-Einer als Groziffer ausgeben
   text_y = 10;                                                                                    // Digitale Groanzeige, Y-Position Sekunden
   text_x = temp_x + 65;                                                                           // Text-Koordinaten auf Sekunden-Zehner
   lcd_digit2 (second / 10);                                                                       // Sekunden-Zehner als Groziffer ausgeben
   text_x = temp_x + 78;                                                                           // Text-Koordinaten auf Sekunden-Einer
   lcd_digit2 (second % 10);                                                                       // Sekunden-Einer als Groziffer ausgeben
   if (!(second & 0x01))                                                                           // wenn Sekunde gerade -> Doppelpunkt ausgeben
   {
      dotsize = 2;                                                                                 // Punktgre "gro" setzen
      lcd_pixel2 (temp_x + 19, 10);                                                                // oberen Punkt ausgeben
      lcd_pixel2 (temp_x + 19, 19);                                                                // unteren Punkt ausgeben
   }

   text_x = 56 - ((49 + strl (wday, 0)) / 2);                                                      // Datum zentrieren
   text_y = 31;                                                                                    // Text-Koordinaten auf Datum-Zeile
   lcd_date (day, month, year);                                                                    // Datum ausgeben
   lcd_char (',');                                                                                 // Komma ausgeben
   lcd_char (0x90);                                                                                // schmales Leerzeichen ausgeben
   lcd_str (wday, 0);                                                                              // Wochentag ausgeben
}

// Eine Zeile mit Alarmdaten ausgeben -----------------------------------------------------------------------------------------------------------------------------------------------------------------

void lcd_alarm (uint8_t n)                                                                         // n: Alarmnummer (0-2)
{
   text_x = 3;                                                                                     // X-Position fr Alarmsymbol

   if (!((alarms[n][10] == 1) && (second % 2)))                                                    // wenn Alarm ausgelst und Sekunde ungerade -> nicht anzeigen
                                                                                                   // (Alarmsymbol blinkt bei ausgelstem Alarm)
   {
      switch (alarms[n][0])                                                                        // Symbole entsprechend des Alarm-Modus ausgeben
      {
         case 0:       lcd_char (0x91);                                                            // Alarm aus
                       break;

         case 1:       lcd_char (0x92);                                                            // Alarm Einmalig
                       break;

         case 2 ... 4: lcd_char (0x93);                                                            // Alarm Tglich, Wchentlich oder Monatlich
                       text_x --;                                                                  // X-Position korrigieren (zweites Symbol direkt anhngen)
                       lcd_char (0x94);                                                            // zweites Symbol
      }
   }

   text_x = 17;                                                                                    // X-Position fr Datum
   lcd_date (alarms[n][3], alarms[n][5], alarms[n][6]);                                            // Datum ausgeben
   lcd_char (' ');                                                                                 // Leerzeichen ausgeben
   lcd_time (alarms[n][2], alarms[n][1]);                                                          // Uhrzeit ausgeben
   text_x = 110 - strl (alarms[n][4], 2);                                                          // Position fr rechtsbndige Anzeige des Wochentages ermitteln
   lcd_str (alarms[n][4], 2);                                                                      // 2 Stellen des Wochentages ausgeben
}

// Ausgabe eines Zeichens ber USB --------------------------------------------------------------------------------------------------------------------------------------------------------------------

void usb_char (char data, uint8_t port)                                                            // data: zu sendendes Zeichen, port: 1= USART0, 2= USART1
{
   switch (port)                                                                                   // weiterer Ablauf abhngig von <port>
   {
      case 1:  while ((uint8_t) (tx0write + 1) == tx0read);                                        // warten, bis mindestens ein Platz im Puffer frei ist
               tx0buf[tx0write] = data;                                                            // Zeichen in Puffer schreiben
               tx0write ++;                                                                        // Schreib-Zeiger erhhen
               UCSR0B = UCSR0B | (1 << UDRIE0);                                                    // Interrupt einschalten
               break;

      case 2:  while ((uint8_t) (tx1write + 1) == tx1read);                                        // warten, bis mindestens ein Platz im Puffer frei ist
               tx1buf[tx1write] = data;                                                            // Zeichen in Puffer schreiben
               tx1write ++;                                                                        // Schreib-Zeiger erhhen
               UCSR1B = UCSR1B | (1 << UDRIE1);                                                    // Interrupt einschalten
               break;
   }   
}

// Ausgabe einer zweistelligen Zahl ber USB ----------------------------------------------------------------------------------------------------------------------------------------------------------

void usb_dec2 (uint8_t data, uint8_t port)                                                         // data: zu sendende Zahl (0-99), port: 1 = USART0, 2 = USART1
{
   usb_char (data / 10 + 48, port);                                                                // Zehnerstelle ausgeben
   usb_char (data % 10 + 48, port);                                                                // Einerstelle ausgeben
}

// Ausgabe einer dreistelligen Zahl ber USB ----------------------------------------------------------------------------------------------------------------------------------------------------------

void usb_dec3 (uint8_t data, uint8_t port)                                                         // data: zu sendende Zahl (0-255), port: 1 = USART0, 2 = USART1
{
   usb_char (data / 100 + 48, port);                                                               // Hunderterstelle ausgeben
   usb_char ((data % 100) / 10 + 48, port);                                                        // Zehnerstelle ausgeben
   usb_char ((data % 100) % 10 + 48, port);                                                        // Einerstelle ausgeben
}

// Ausgabe eines dezimalen Byte-Wertes ber USB -------------------------------------------------------------------------------------------------------------------------------------------------------

void usb_byte (uint8_t data, uint8_t port)                                                         // data: zu sendendes Byte (0-255), port: 1 = USART0, 2 = USART1
{
   if (data > 99) usb_char (data / 100 + 48, port);                                                // wenn Wert > 99 -> Hunderterstelle ausgeben
   if (data > 9) usb_char ((data % 100) / 10 + 48, port);                                          // wenn Wert > 9 -> Zehnerstelle ausgeben
   usb_char ((data % 100) % 10 + 48, port);                                                        // Einersteller ausgeben
}

// Ausgabe eines Strings ber USB ---------------------------------------------------------------------------------------------------------------------------------------------------------------------

void usb_str (uint8_t strnr, uint8_t len, uint8_t port)                                            // strnr: String-Nummer im Flash-Speicher
                                                                                                   // len: Anzahl der auszugebenden Zeichen (0= alle Zeichen ausgeben)
                                                                                                   // port: 1= USART0, 2= USART1
{
   uint8_t i;                                                                                      // Zeichenzhler
   PGM_P p;                                                                                        // Pointer auf Stringdaten im Flash

   if (lang) p = (PGM_P) pgm_read_word (&stre[strnr]);                                             // falls Englisch gewhlt
   else p = (PGM_P) pgm_read_word (&strd[strnr]);                                                  // sonst Deutsch
   strcpy_P (strbuf, p);                                                                           // String in den RAM-Puffer kopieren
   i = 0;                                                                                          // Zeichenzhler auf erstes Zeichen
   while (strbuf[i] > 0)                                                                           // Schleife bis zum Endezeichen (0)
   {
      if (! len) usb_char (strbuf[i], port);                                                       // wenn Lnge= 0 -> alle Zeichen ausgeben
      else if (i < len) usb_char (strbuf[i], port);                                                // sonst nur bis zur vorgegebenen Lnge
      i ++;                                                                                        // nchstes Zeichen
   }
}

// Berechnung und Ausgabe des Alters eines Geburtstagseintrages ---------------------------------------------------------------------------------------------------------------------------------------

uint8_t lcd_age (void)                                                                             // Ergebnis: X-Endposition fr die Namensausgabe
{                                                                                                  // bdindx muss vor Aufruf gesetzt sein

   int16_t cur_day;                                                                                // aktueller Tag im Jahr (Monat * 31 + Tag)
   int16_t cur_year;                                                                               // aktuelles Jahr (4-stellig)
   int16_t bd_day;                                                                                 // Geburtstag im Jahr (Geburtsmonat * 31 + Geburtstag)
   int16_t bd_year;                                                                                // Geburtsjahr (4-stellig)
   int16_t bd_age;                                                                                 // ermitteltes Alter
   uint8_t xlim;                                                                                   // X-Endposition fr Namensausgabe

   cur_day = month * 31 + day;                                                                     // aktuellen Tag im Jahr ermitteln (einfache Methode)
   cur_year = c_yearh * 100 + year;                                                                // aktuelles Jahr ermitteln
   bd_day = eep_read_byte (e_bddata + (bdindx * 34 + 1)) * 31
            + eep_read_byte (e_bddata + (bdindx * 34));                                            // Geburtstag im Jahr ermitteln (einfache Methode)
   bd_year = eep_read_byte (e_bddata + (bdindx * 34 + 3)) * 100
            + eep_read_byte (e_bddata + (bdindx * 34 + 2));                                        // Geburtsjahr ermitteln
   bd_age = cur_year - bd_year;                                                                    // Alter berechen
   if (bd_day < cur_day) bd_age ++;                                                                // wenn Geburtstag vor dem aktuellen Tag -> Alter + 1

   if (((year % 4) != 0) && (day == 1) && (month == 3))                                            // wenn kein Schaltjahr und Datum 01.03.
   {
      if ((eep_read_byte (e_bddata + (bdindx * 34)) == 29) &&                                      // wenn Geburtstag am 29.02. ->
         (eep_read_byte (e_bddata + (bdindx * 34 + 1)) == 2)) bd_age --;                           // Alter korrigieren
   }

   text_x = 104;                                                                                   // Text-Position fr 1-stelliges Alter
   xlim = 101;                                                                                     // X-Endposition fr Name und 1-stelliges Alter

   switch (bd_age)                                                                                 // Alters-Ausgabe abhngig von der Anzahl der Stellen
   {
      case 1000 ... 9999:                                                                          // falls Alter 4-stellig
         text_x = 86;                                                                              // Text-Position fr Alter setzen
         xlim = 83;                                                                                // X-Endposition fr Name setzen
         lcd_char (bd_age / 1000 + 48);                                                            // Tausender-Stelle ausgeben
         bd_age = bd_age % 1000;                                                                   // Restwert berechnen

      case 100 ... 999:
         text_x = 92;                                                                              // Text-Position fr Alter setzen
         xlim = 89;                                                                                // X-Endposition fr Name setzen
         lcd_char (bd_age / 100 + 48);                                                             // Hunderter-Stelle ausgeben
         bd_age = bd_age % 100;                                                                    // Restwert berechnen

      case 10 ... 99:
         text_x = 98;                                                                              // Text-Position fr Alter setzen
         xlim = 95;                                                                                // X-Endposition fr Name setzen
         lcd_char (bd_age / 10 + 48);                                                              // Zehner-Stelle ausgeben
   }
   lcd_char (bd_age % 10 + 48);                                                                    // Einer-Stelle ausgeben
   return xlim;
}

// Eine Zeile mit Geburtstagsdaten (Datum, Name, Alter) ausgeben --------------------------------------------------------------------------------------------------------------------------------------

void lcd_bdname (void)                                                                             // bdindx muss vor Aufruf gesetzt sein
{
   uint8_t i;                                                                                      // Zhlervariable
   uint8_t bday;                                                                                   // Geburtstag
   uint8_t bmonth;                                                                                 // Geburtsmonat

   xlimit = 109;                                                                                   // X-Endposition fr Name ohne Alters-Anzeige
   bday = eep_read_byte (e_bddata + (bdindx * 34));                                                // Geburtstag holen
   bmonth = eep_read_byte (e_bddata + (bdindx * 34 + 1));                                          // Geburtsmonat holen
   text_x = 4;

   if (((year % 4) != 0) && (day == 1) && (month == 3))                                            // wenn kein Schaltjahr und Datum 01.03.
   {
      if ((bday == 29) && (bmonth == 2))                                                           // wenn Geburtstag am 29.02.
      {
         bday = 1;
         bmonth = 3;                                                                               // Tag auf den 01.03. setzen
      }
   }

   if ((syncstat > 1) && (bday == day) && (bmonth == month))                                       // wenn Uhr gltige Zeit hat und heute Geburtstag ist
   {
      if (!(second % 2))                                                                           // wenn Sekunde ungeradzahlig
      {
         text_x = 4;
         lcd_char (0x9a);                                                                          // Pfeilsymbol im Sekundentakt blinken lassen
      }
   }
   else                                                                                            // sonst (wenn heute nicht Geburtstag ist)
   {
      lcd_numbs (eep_read_byte (e_bddata + (bdindx * 34)));                                        // Geburtstag holen und ausgeben
      text_x = 4;
      text_y = text_y + 6;                                                                         // Text-Position fr Geburtsmonat festlegen
      lcd_numbs (eep_read_byte (e_bddata + (bdindx * 34 + 1)));                                    // Geburtsmonat holen und ausgeben
      text_y = text_y - 6;                                                                         // Text-Position fr Name festlegen
   }

   if ((syncstat > 1) && bdagem)                                                                   // wenn Uhr eine gltige Zeit hat und Alters-Anzeige aktiviert
   {
      if (eep_read_byte (e_bddata + (bdindx * 34 + 2)) < 100) xlimit = lcd_age ();                 // Alter ausgeben und X-Endposition fr Namen ermitteln
   }

   text_x = 17;
   if (bdordr == 0)                                                                                // wenn Reihenfolge Vorname-Nachname eingestellt ist
   {
      for (i = 4; i < 19; i ++)                                                                    // Vorname ausgeben (max. 15 Zeichen)
      {
         bddata = eep_read_byte (e_bddata + (bdindx * 34 + i));                                    // ein Zeichen vom Vornamen holen
         if ((bddata == 255) || ((chrl (bddata) + text_x) > xlimit)) break;                        // wenn Textende oder rechter Bildrand -> Abbruch
         lcd_char (bddata);                                                                        // Zeichen ausgeben
      }
      if (text_x > 17) text_x = text_x + 4;                                                        // wenn Vorname ausgegeben wurde -> Leerzeichen einfgen
      for (i = 19; i < 34; i ++)                                                                   // Nachname ausgeben (max. 15 Zeichen)
      {
         bddata = eep_read_byte (e_bddata + (bdindx * 34 + i));                                    // ein Zeichen vom Nachnamen holen
         if ((bddata == 255) || ((chrl (bddata) + text_x) > xlimit)) break;                        // wenn Textende oder rechter Bildrand -> Abbruch
         lcd_char (bddata);                                                                        // Zeichen ausgeben
      }
   }
   else                                                                                            // wenn Reihenfolge Nachname-Vorname eingestellt ist
   {
      for (i = 19; i < 34; i ++)                                                                   // Nachname ausgeben (max. 15 Zeichen)
      {
         bddata = eep_read_byte (e_bddata + (bdindx * 34 + i));                                    // ein Zeichen vom Nachnamen holen
         if ((bddata == 255) || ((chrl (bddata) + text_x) > xlimit)) break;                        // wenn Textende oder rechter Bildrand -> Abbruch
         lcd_char (bddata);                                                                        // Zeichen ausgeben
      }
      if (text_x > 17) text_x = text_x + 4;                                                        // wenn Nachname ausgegeben wurde -> Leerzeichen einfgen
      for (i = 4; i < 19; i ++)                                                                    // Vorname ausgeben (max. 15 Zeichen)
      {
         bddata = eep_read_byte (e_bddata + (bdindx * 34 + i));                                    // ein Zeichen vom Vornamen holen
         if ((bddata == 255) || ((chrl (bddata) + text_x) > xlimit)) break;                        // wenn Textende oder rechter Bildrand -> Abbruch
         lcd_char (bddata);                                                                        // Zeichen ausgeben
      }
   }
}

// Wochentags-Berechnung nach Gauss (nach http://www.paswiki.de/index.php/Ewiger_Kalender, existiert leider nicht mehr) -------------------------------------------------------------------------------

uint8_t calc_wday (uint8_t d, uint8_t m, uint8_t yh, int8_t y)                                     // Ergebnis: Wochentag (1-7)
                                                                                                   // d: Tag, m: Monat, yh: Jahrhundert, y: Jahr
{
   int8_t w;                                                                                       // Wochtag

   if (m <= 2)                                                                                     // wenn Monat Januar oder Februar
   {
      m = m + 12;                                                                                  // Monat ans Jahresende verschieben
      y = y - 1;                                                                                   // Jahr vermindern
      if (y < 0)                                                                                   // wenn bertrag
      {
         y = 99;                                                                                   // Jahr auf 99 setzen
         yh = yh - 1;                                                                              // Jahrhundert vermindern
      }
   }
   w = (d + (13 * (m + 1)) / 5 + y + y / 4 + yh / 4 - 2 * yh) % 7;                                 // Wochtagsformel, kleinster Wert entspricht Samstag
   if (w < 0) w = w + 7;                                                                           // falls Ergebnis negativ war
   w = w - 1;                                                                                      // Wochentag korrigieren (kleinster Wert ist Montag)
   if (w < 1) w = w + 7;                                                                           // und in den Bereich 1-7 legen
   return w;
}

// Alarm-Check, wird tglich um 0:00 Uhr und bei nderungen an den Alarmeinstellungen aufgerufen, prft wiederholende Alarme und passt das Datum an den nchsten Alarm an -----------------------------

void alarm_check (uint8_t s)                                                                       // s: selektives Speichern (0= nur die Daten des gerade bearbeiteten Alarms (alm) speichern und
                                                                                                   // einmalige Alarme nicht prfen, 1= alle Daten speichern)
{
   uint16_t cur_date;                                                                              // Zeitstempel fr aktuelles Datum (Jahr*384 + Monat*32 + Tag)
   uint16_t alm_date;                                                                              // Zeitstempel fr Alarmdatum (Jahr*384 + Monat*32 + Tag)
   uint8_t  okflag;                                                                                // Flag fr die Ermittlung des nchsten passenden Wochentages
   uint8_t  moflag;                                                                                // Flag fr Monatswechsel
   uint8_t  a;                                                                                     // Zhlervariable fr die Alarme
   uint8_t  i;                                                                                     // Zhlervariable fr Datenspeicherung im EEPROM

   cur_date = year * 384 + (month - 1) * 32 + day;                                                 // Zeitstempel fr aktuelles Datum ermitteln
   for (a = 0; a < 3; a ++)                                                                        // 3 Alarme bearbeiten
   {
      if (alarms[a][0] > 0)                                                                        // wenn Alarm aktiv ist
      {
         alm_date = alarms[a][6] * 384 + (alarms[a][5] - 1) * 32 + alarms[a][3];                   // Zeitstempel fr Alarmdatum ermitteln

         switch (alarms[a][0])                                                                     // weiterer Ablauf abhngig vom eingestellten Alarmtyp
         {
            case 1:                                                                                // falls Alarm einmalig
               if (s)                                                                              // wenn alle Alarme gespeichert werden sollen
               {
                  if (alm_date < cur_date) alarms[a][0] = 0;                                       // wenn Alarmdatum in Vergangenheit -> Alarm deaktivieren
               }
               break;

            case 2:                                                                                // falls Alarm tglich
               if (alm_date < cur_date)                                                            // wenn Alarmdatum in der Vergangenheit
               {
                  alarms[a][3] = day;
                  alarms[a][4] = wday;
                  alarms[a][5] = month;
                  alarms[a][6] = year;                                                             // Alarmdatum auf aktuelles Datum setzen
               }
               break;

            case 3:                                                                                // falls Alarm wchentlich
               if (alm_date <= cur_date)                                                           // wenn Alarmdatum heute oder in der Vergangenheit
               {
                  alarms[a][3] = day;
                  alarms[a][4] = wday;
                  alarms[a][5] = month;
                  alarms[a][6] = year;                                                             // Alarmdatum zunchst auf aktuelles Datum setzen
                  okflag = 0;                                                                      // Flag zunchst lschen
                  while (!okflag)                                                                  // Schleife bis nchster passender Wochentag gefunden wird
                  {
                     if ((alarms[a][7] + 1) == alarms[a][4]) okflag = 1;                           // einzelner Wochentag
                     if ((alarms[a][7] == 7) && (alarms[a][4] < 6)) okflag = 1;                    // Wochentage Mo-Fr
                     if ((alarms[a][7] == 8) && (alarms[a][4] < 7)) okflag = 1;                    // Wochentage Mo-Sa
                     if ((alarms[a][7] == 9) && (alarms[a][4] > 5)) okflag = 1;                    // Wochentage Sa-So

                     if (!okflag)                                                                  // wenn Wochentag nicht passt
                     {
                        alarms[a][4] ++;                                                           // Wochentag erhhen
                        if (alarms[a][4] > 7) alarms[a][4] = 1;                                    // wenn Wochentagberlauf -> Wochentag zurcksetzen
                        alarms[a][3] ++;                                                           // Tag erhhen
                        moflag = 0;                                                                // Monatswechselflag lschen

                        if ((alarms[a][6] % 4) == 0)                                               // wenn Schaltjahr
                        {
                           if (alarms[a][3] > pgm_read_byte (&daytab2[alarms[a][5] - 1])) moflag = 1; // wenn Tag > Schaltjahr-Tabelle -> Monatswechsel
                        }
                        else                                                                       // wenn kein Schaltjahr
                        {
                           if (alarms[a][3] > pgm_read_byte (&daytab1[alarms[a][5] - 1])) moflag = 1; // wenn Tag > Normaljahr-Tabelle -> Monatswechsel
                        }

                        if (moflag)                                                                // wenn Monatswechsel erforderlich
                        {
                           alarms[a][3] = 1;                                                       // Tag zurcksetzen
                           alarms[a][5] ++;                                                        // Monat erhhen
                           if (alarms[a][5] > 12)                                                  // wenn berlauf
                           {
                              alarms[a][5] = 1;                                                    // Monat zurcksetzen
                              alarms[a][6] ++;                                                     // Jahr erhhen
                           }
                        }
                     }
                  }
               }
               break;

            case 4:                                                                                // falls Alarm monatlich
               alarms[a][5] = month;
               alarms[a][6] = year;                                                                // aktuellen Monat und aktuelles Jahr verwenden

               if (alarms[a][3] > 27)                                                              // wenn Alarmtag ber 27 -> letzten Tag im Monat setzen
               {
                  if ((alarms[a][6] % 4) == 0)                                                     // wenn Schaltjahr
                  {
                     alarms[a][3] = pgm_read_byte (&daytab2[alarms[a][5] - 1]);                    // letzter Tag aus Schaltjahr-Tabelle
                  }
                  else                                                                             // wenn kein Schaltjahr
                  {
                     alarms[a][3] = pgm_read_byte (&daytab1[alarms[a][5] - 1]);                    // letzter Tag aus Normaljahr-Tabelle
                  }                     
               }
               else                                                                                // wenn Alarmtag im Bereich 1-27
               {
                  if (alarms[a][3] < day)                                                          // wenn Tag in der Vergangenheit
                  {
                     alarms[a][5] ++;                                                              // nchsten Monat verwenden
                     if (alarms[a][5] > 12)                                                        // wenn berlauf
                     {
                        alarms[a][5] = 1;                                                          // Monat zurcksetzen
                        alarms[a][6] ++;                                                           // Jahr erhhen
                     }
                  }
               }
               alarms[a][4] = calc_wday (alarms[a][3], alarms[a][5], c_yearh, alarms[a][6]);       // Wochentag
         }
      }
      if (s || (a == alm))                                                                         // wenn alle Daten oder der gerade bearbeitete Alarm aktiv ist
      {
         for (i = 0; i < 8; i ++)                                                                  // 8 Alarmparameter im EEPROM speichern (ohne Laust. und Sound)
            eep_write_byte (e_alarms + (a * 10 + i), alarms[a][i]);                                // Parameter im EEPROM speichern
      }
   }
}

// Geburtstags-Check, wird tglich um 0:00 Uhr und bei nderungen an den Geburtstagen aufgerufen, Indexliste nach dem aktuellen Datum sortieren, Datum des nchsten Geburtstages in die Alarmtabelle
// schreiben, die ersten 3 Eintrge in das EEPROM kopieren --------------------------------------------------------------------------------------------------------------------------------------------

void bd_check (void)
{
   uint8_t i;                                                                                      // Zhler fr die Indexliste
   uint8_t e;                                                                                      // Zhler fr die Geburtstags-Eintrge und die Sortierung
   uint8_t z;                                                                                      // Endwert fr Schleifen
   uint8_t f;                                                                                      // Flag fr den Eintrags-Tausch bei der Sortierung
   uint8_t t;                                                                                      // Zwischenspeicher fr die Sortierung (Datenfeld-Tausch)
   uint8_t bd_day;                                                                                 // laufender Geburtstag
   uint8_t bd_month;                                                                               // laufender Geburtsmonat
   uint8_t end_day;                                                                                // letzter Geburtstag fr die Suche
   uint8_t end_month;                                                                              // letzter Geburtsmonat fr die Suche

                                                                                                   // Sortierung nach Datum, wird immer ausgefhrt; ist die Sortierung nach Name eingestellt, werden
                                                                                                   // nur die ersten 3 Datenstze bearbeitet ----------------------------------------------------------

   i = 0;                                                                                          // Index-Zhler = 0 setzen
   bd_day = day;
   bd_month = month;                                                                               // aktuelles Datum fr die Suche voreinstellen
   end_day = bd_day;
   end_month = bd_month;                                                                           // aktuelles Datum fr die Suche voreinstellen

   if (((year % 4) != 0) && (bd_day == 1) && (bd_month == 3))                                      // wenn kein Schaltjahr und Datum 01.03.
   {
      bd_day = 29;
      bd_month = 2;                                                                                // Suche bereits am 29.02. beginnen
      end_day = 29;
      end_month = 2;                                                                               // und am 29.02. beenden
   }

   z = 115;                                                                                        // Schleifen-Endwert auf 115 setzen (gesamte Liste)
   if (bdlsort) z = 3;                                                                             // wenn Sortierung nach Name -> nur 3 Listeneintrge bearbeiten
   do
   {
      for (e = 0; e < 115; e ++)                                                                   // alle Geburtstagsdatenstze prfen
      {
         if ((bd_day == eep_read_byte (e_bddata + (e * 34))) && (bd_month == eep_read_byte (e_bddata + (e * 34 + 1)))) // wenn Tag und Monat bereinstimmt
         {
            if (i < z) bdilist[i ++] = e;                                                          // wenn Listenzhler im zulssigen Bereich -> Nummer des Datensatzes in die Indexliste schreiben
         }
      }

      bd_day ++;                                                                                   // nchsten Tag prfen
      if (bd_day > pgm_read_byte (&daytab2[bd_month - 1]))                                         // wenn Tageswert berschritten
      {
         bd_day = 1;                                                                               // Tag zurcksetzen
         bd_month ++;                                                                              // Monat erhhen
         if (bd_month > 12) bd_month = 1;                                                          // wenn berlauf -> Monat zurcksetzen
      }
      if ((bd_day == end_day) && (bd_month == end_month)) break;                                   // wenn ein Jahresdurchlauf komplett ist -> Abbruch
   }
   while (i < z);                                                                                  // Schleifen bis zum vorgegebenen Endwert ausfhren

   while (i < 115)                                                                                 // Schleife bis zum Ende der Indexliste
   {
      bdilist[i ++] = 255;                                                                         // restliche Datenstze in der Indexliste lschen
   }

   i = bdilist[0];                                                                                 // Nummer des ersten Eintrags aus dem Geburtstags-Index lesen
   alarms[3][3] = eep_read_byte (e_bddata + (i * 34));                                             // Tag vom Geburtstags-Eintrag in die Alarmtabelle bertragen
   alarms[3][5] = eep_read_byte (e_bddata + (i * 34 + 1));                                         // Monat vom Geburtstags-Eintrag in die Alarmtabelle bertragen
   alarms[3][6] = year;                                                                            // aktuelles Jahr in die Alarmtabelle bertragen

   if ((alarms[3][3] == 29) && (alarms[3][5] == 2))                                                // wenn nchster Geburtstag am 29.02.
   {
      if (((year % 4) != 0) && (day == 1) && (month == 3))                                         // wenn kein Schaltjahr und Datum 01.03.
      {
         alarms[3][3] = 1;
         alarms[3][5] = 3;                                                                         // Alarmeintrag auf den 01.03. ndern
      }
   }
   for (i = 0; i < 3; i ++)                                                                        // die ersten 3 Eintrge der Indexliste bearbeiten
   {
      bdslist[i] = bdilist[i];                                                                     // in die Kurzliste bertragen
      eep_write_byte (e_bdslist + i, bdslist[i]);                                                  // und ins EEPROM schreiben
   }
                                                                                                   // Sortierung nach Vor- oder Nachname --------------------------------------------------------------
                                                                                                   // zunchst wird eine Rohliste erstellt

   if (bdlsort)                                                                                    // wenn Sortierung nach Vor- oder Nachname
   {
      i = 0;                                                                                       // Indexlisten-Zhler auf ersten Eintrag
      for (e = 0; e < 115; e ++)                                                                   // alle Geburtstagseintrge bearbeiten
      {
         if (eep_read_byte (e_bddata + (e * 34)) < 255)                                            // wenn Geburtstagseintrag gltig
         {
            bdilist[i ++] = e;                                                                     // Eintragsnummer in der Indexliste speichern
         }
      }
      z = i;                                                                                       // ermittelten Listen-Endwert fr sptere Sortierung speichern
      while (i < 115)                                                                              // Schleife bis zum Ende der Indexliste
      {
         bdilist[i ++] = 255;                                                                      // restliche Datenstze in der Indexliste lschen
      }

      f = 0;                                                                                       // Flag fr Datensatz-Tausch lschen
      if (z > 1)                                                                                   // wenn mindestens 2 Geburtstagseintrge vorhanden
      {
         if (bdlsort == 1)                                                                         // wenn Sortierung nach Vorname
         {
            for (e = z - 1; e > 0; e --)                                                           // uere Schleife, luft rckwrts vom Listen-Endwert-1 bis 1
            {
               for (i = 0; i < e; i ++)                                                            // innere Schleife, luft von 0 bis zum aktuellen Wert der ueren Schleife
               {
                  if (eep_read_byte (e_bddata + (bdilist[i] * 34 + 4)) >
                     eep_read_byte (e_bddata + (bdilist[i + 1] * 34 + 4)))                         // wenn 1. Vornamenzeichen vom aktuellen Eintrage grer als folgender Eintrag
                  {
                     f = 1;                                                                        // Eintrge in der Indexliste tauschen
                  }
                  if (eep_read_byte (e_bddata + (bdilist[i] * 34 + 4)) ==
                     eep_read_byte (e_bddata + (bdilist[i + 1] * 34 + 4)))                         // wenn 1. Vornamenzeichen vom aktuellen Eintrage gleich dem folgenden Eintrag
                  {
                     if (eep_read_byte (e_bddata + (bdilist[i] * 34 + 5)) >
                        eep_read_byte (e_bddata + (bdilist[i + 1] * 34 + 5)))                      // wenn 2. Vornamenzeichen vom aktuellen Eintrag grer als folgender Eintrag
                     {
                        f = 1;                                                                     // Eintrge in der Indexliste tauschen
                     }
                     if (eep_read_byte (e_bddata + (bdilist[i] * 34 + 5)) ==
                        eep_read_byte (e_bddata + (bdilist[i + 1] * 34 + 5)))                      // wenn 2. Vornamenzeichen vom aktuellen Eintrag gleich dem folgenden Eintrag
                     {
                        if (eep_read_byte (e_bddata + (bdilist[i] * 34 + 19)) >
                           eep_read_byte (e_bddata + (bdilist[i + 1] * 34 + 19)))                  // wenn 1. Nachnamenzeichen vom aktuellen Eintrages grer als folgender Eintrag
                        {
                           f = 1;                                                                  // Eintrge in der Indexliste tauschen
                        }
                     }
                  }
                  if (f)                                                                           // wenn Eintrge getauscht werden mssen
                  {
                     f = 0;                                                                        // Tausch-Flag wieder lschen
                     t = bdilist[i];                                                               // Eintrag vom aktuellen Listenplatz zwischenspeichern
                     bdilist[i] = bdilist[i + 1];                                                  // Eintrag vom folgenden Listenplatz in den aktuellen kopieren
                     bdilist[i + 1] = t;                                                           // zwischengespeicherten Eintrag in den folgenden Listenplatz
                  }
               }
            }
         }
         else                                                                                      // wenn Sortierung nach Nachname
         {
            for (e = z - 1; e > 0; e --)                                                           // uere Schleife
            {
               for (i = 0; i < e; i ++)                                                            // innere Schleife
               {
                  if (eep_read_byte (e_bddata + (bdilist[i] * 34 + 19)) >
                     eep_read_byte (e_bddata + (bdilist[i + 1] * 34 + 19)))                        // wenn 1. Nachnamenzeichen vom aktuellen Eintrag grer als folgender Eintrag
                  {
                     f = 1;                                                                        // Eintrge in der Indexliste tauschen
                  }
                  if (eep_read_byte (e_bddata + (bdilist[i] * 34 + 19)) ==
                     eep_read_byte (e_bddata + (bdilist[i + 1] * 34 + 19)))                        // wenn 1. Nachnamenzeichen vom aktuellen Eintrag gleich dem folgenden Eintrag
                  {
                     if (eep_read_byte (e_bddata + (bdilist[i] * 34 + 20)) >
                        eep_read_byte (e_bddata + (bdilist[i + 1] * 34 + 20)))                     // wenn 2. Nachnamenzeichen vom aktuellen Eintrag grer als folgender Eintrag
                     {
                        f = 1;                                                                     // Eintrge in der Indexliste tauschen
                     }
                     if (eep_read_byte (e_bddata + (bdilist[i] * 34 + 20)) ==
                        eep_read_byte (e_bddata + (bdilist[i + 1] * 34 + 20)))                     // wenn 2. Nachnamenzeichen vom aktuellen Eintrag gleich dem folgenden Eintrag
                     {
                        if (eep_read_byte (e_bddata + (bdilist[i] * 34 + 4)) >
                           eep_read_byte (e_bddata + (bdilist[i + 1] * 34 + 4)))                   // wenn 1. Vornamenzeichen vom aktuellen Eintrag grer als folgender Eintrag
                        {
                           f = 1;                                                                  // Eintrge in der Indexliste tauschen
                        }
                     }
                  }
                  if (f)                                                                           // wenn Eintrge getauscht werden mssen
                  {
                     f = 0;                                                                        // Tausch-Flag wieder lschen
                     t = bdilist[i];                                                               // Eintrag vom aktuellen Listenplatz zwischenspeichern
                     bdilist[i] = bdilist[i + 1];                                                  // Eintrag vom folgenden Listenplatz in den aktuellen kopieren
                     bdilist[i + 1] = t;                                                           // zwischengespeicherten Eintrag in den folgenden Listenplatz
                  }
               }
            }
         }
      }
   }
}

// LCD-Hintergrundbeleuchtung einschalten und Helligkeit setzen ---------------------------------------------------------------------------------------------------------------------------------------

void blt_on (uint8_t b)                                                                            // b: Helligkeit (0-255)
{
   OCR2 = b;                                                                                       // LCD-Hintergrundbeleuchtung einschalten und Helligkeit setzen
}

// LCD-Hintergrundbeleuchtung ausschalten -------------------------------------------------------------------------------------------------------------------------------------------------------------

void blt_off (void)
{
   OCR2 = 0;                                                                                       // LCD-Hintergrundbeleuchtung ausschalten
}

// LED einschalten ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

void led_on (void)
{
   PORTD = PORTD & ~ (1 << PD4);                                                                   // LED-Port auf Low (LED ein)
}

// LED ausschalten ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

void led_off (void)
{
   PORTD = PORTD | (1 << PD4);                                                                     // LED-Port auf High (LED aus)
}

// USB-Aktivitt anzeigen -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

void usb_on (void)
{
   usb_act = 1;                                                                                    // USB-Kommunikation ist aktiv
   usbtout = c_usbtout;                                                                            // Timeout-Zhler fr USB-Symbol setzen
   sec_flag = 1;                                                                                   // Analoguhr aktualisieren
}

// Zeichen in Kleinbuchstaben umwandeln ---------------------------------------------------------------------------------------------------------------------------------------------------------------

char lowcase (char c)                                                                              // Ergebnis Kleinbuchstabe
{
   if ((c >= 'A') && (c <= 'Z')) c = c + 32;                                                       // wenn Grobuchstabe -> wandeln
   return c;
}

// Einstelligen Dezimalwert aus Kommandopuffer lesen und in Zahl umwandeln ----------------------------------------------------------------------------------------------------------------------------

uint8_t read_dec1 (uint8_t pos)                                                                    // Ergebnis Zahl 0-9 oder 255 bei Fehler
{                                                                                                  // pos: Position im Kommandopuffer
   uint8_t res;
   res = 0;                                                                                        // Ergebniswert lschen
   if ((cmdbuf[pos] < '0') || (cmdbuf[pos] > '9')) res = 255;                                      // wenn Ziffern-Bereich berschritten -> Fehler
   if (res == 0) res = cmdbuf[pos] - 48;                                                           // ASCII-Daten in Zahl umwandeln
   return res;
}

// Zweistelligen Dezimalwert aus Kommandopuffer lesen und in Zahl umwandeln ---------------------------------------------------------------------------------------------------------------------------

uint8_t read_dec2 (uint8_t pos)                                                                    // Ergebnis Zahl 0-99 oder 255 bei Fehler
{                                                                                                  // pos: Position im Kommandopuffer
   uint8_t res;
   res = 0;                                                                                        // Ergebniswert lschen
   if ((cmdbuf[pos] < '0') || (cmdbuf[pos] > '9')) res = 255;                                      // wenn Zehner-Bereich berschritten -> Fehler
   if ((cmdbuf[pos + 1] < '0') || (cmdbuf[pos + 1] > '9')) res = 255;                              // wenn Einer-Bereich berschritten -> Fehler
   if (res == 0)                                                                                   // wenn Daten fehlerfrei
   {
      res = (cmdbuf[pos] - 48) * 10 + cmdbuf[pos + 1] - 48;                                        // ASCII-Daten in Zahl umwandeln
   }
   return res;
}

// Dreistelligen Dezimalwert aus Kommandopuffer lesen und in Zahl umwandeln ---------------------------------------------------------------------------------------------------------------------------

uint16_t read_dec3 (uint8_t pos)                                                                   // Ergebnis Zahl 0-255 oder > 255 bei Fehler
{                                                                                                  // pos: Position im Kommandopuffer
   uint16_t res;
   res = 0;                                                                                        // Ergebniswert lschen
   if ((cmdbuf[pos] < '0') || (cmdbuf[pos] > '2')) res = 256;                                      // wenn Hunderter-Bereich berschritten -> Fehler
   if ((cmdbuf[pos + 1] < '0') || (cmdbuf[pos + 1] > '9')) res = 256;                              // wenn Zehner-Bereich berschritten -> Fehler
   if ((cmdbuf[pos + 2] < '0') || (cmdbuf[pos + 2] > '9')) res = 256;                              // wenn Einer-Bereich berschritten -> Fehler
   if (res == 0)                                                                                   // wenn Daten fehlerfrei
   {
      res = (cmdbuf[pos] - 48) * 100 + (cmdbuf[pos + 1] - 48) * 10 + cmdbuf[pos + 2] - 48;         // ASCII-Daten in Zahl umwandeln
   }
   return res;
}

// Interrupt-Service-Routinen =========================================================================================================================================================================

ISR (TIMER1_COMPA_vect)                                                                            // wird alle 78,125s (12800Hz) aufgerufen und ist fr das Lesen des Drehimpulsgebers und der
                                                                                                   // Taster, sowie fr DCF und Uhrenzhler zustndig, weiterhin werden verschiedene Timeout-Zhler und
                                                                                                   // Sound-Funktionen gesteuert
{
                                                                                                   // Zhler zur Teilung auf 625s (1600Hz) bearbeiten ------------------------------------------------

   intc78us ++;                                                                                    // Interrupt-Zhler 1 (78s) erhhen
   if (intc78us > 7)                                                                               // wenn berlauf
   {
      intc78us = 0;                                                                                // Interrupt-Zhler wieder zurcksetzen

                                                                                                   // Der folgende Abschnitt wird alle 625s (1600Hz) aufgerufen

                                                                                                   // Drehimpulsgeber bearbeiten ----------------------------------------------------------------------

      uint8_t oldstat, newstat;                                                                    // Zwischenspeicher fr Status des Drehimpulsgebers

      newstat = PINC & 0x30;                                                                       // aktuellen Status des Drehimpulsgebers einlesen
      oldstat = encstat1;                                                                          // letzten Status des Drehimpulsgebers holen
      encstat1 = newstat;                                                                          // aktuellen Status speichern
      if (newstat == oldstat)                                                                      // wenn sich der Status nicht gendert hat (prellfrei)
      {
         oldstat = encstat2;                                                                       // letzten stabilen Status des Drehimpulsgebers holen
         encstat2 = newstat;                                                                       // neuen stabilen Status speichern
         if (newstat != oldstat)                                                                   // wenn sich der Status gendert hat
         {
            if (oldstat == 0x00)                                                                   // wenn vorheriger Status = 0x00
            {
               if (newstat == 0x10) encstep ++;                                                    // wenn neuer Status = 0x10 -> Rechtsdrehung -> Position + 1
               if (newstat == 0x20) encstep --;                                                    // wenn neuer Status = 0x20 -> Linksdrehung  -> Position - 1
            }
            if (oldstat == 0x10)                                                                   // wenn vorheriger Status = 0x10
            {
               if (newstat == 0x30) encstep ++;                                                    // wenn neuer Status = 0x30 -> Rechtsdrehung -> Position + 1
               if (newstat == 0x00)                                                                // wenn neuer Status = 0x00
               {
                  encstep --;                                                                      // Linksdrehung  -> Position - 1
                  if (!(PINB & (1 << PB2)))                                                        // wenn Jumper 1 geschlossen (Drehimpulsgeber mit 4 Schritten)
                  {
                     encstep = encstep & 0xfffc;                                                   // Schaltposition auf Zwischenschritt legen, Korrektur fr ungenaue Rasterposition des verwendeten
                  }                                                                                // Drehimpulsgebers
               }
            }
            if (oldstat == 0x20)                                                                   // wenn vorheriger Status = 0x20
            {
               if (newstat == 0x00)                                                                // wenn neuer Status = 0x00
               {
                  encstep ++;                                                                      // Rechtsdrehung -> Position + 1
                  if (!(PINB & (1 << PB2)))                                                        // wenn Jumper 1 geschlossen (Drehimpulsgeber mit 4 Schritten)
                  {
                     encstep = encstep & 0xfffc;                                                   // Schaltposition auf Zwischenschritt legen, Korrektur fr ungenaue Rasterposition des verwendeten
                  }                                                                                // Drehimpulsgebers
               }
               if (newstat == 0x30) encstep --;                                                    // wenn neuer Status = 0x30 -> Linksdrehung  -> Position - 1
            }
            if (oldstat == 0x30)                                                                   // wenn vorheriger Status = 0x30
            {
               if (newstat == 0x20) encstep ++;                                                    // wenn neuer Status = 0x20 -> Rechtsdrehung -> Position + 1
               if (newstat == 0x10) encstep --;                                                    // wenn neuer Status = 0x10 -> Linksdrehung  -> Position - 1
            }
         }
      }

      if (PINB & (1 << PB2))                                                                       // wenn Jumper 1 offen (Drehimpulsgeber mit 2 Schritten)
      {
         encpos = encstep >> 1;                                                                    // Schrittwert / 2
      }
      else                                                                                         // wenn Jumper 1 geschlossen (Drehimpulsgeber mit 4 Schritten)
      {
         encpos = encstep >> 2;                                                                    // Schrittwert / 4
      }
                                                                                                   // Zhler zur Teilung auf 20ms (50Hz) bearbeiten ---------------------------------------------------

      intc625us ++;                                                                                // Interrupt-Zhler 2 (625s) erhhen
      if (intc625us > 31)                                                                          // wenn berlauf
      {
         intc625us = 0;                                                                            // Interrupt-Zhler 2 wieder zurcksetzen

                                                                                                   // Der folgende Abschnitt wird alle 20ms (50Hz) aufgerufen und bearbeitet den DCF77-Eingang und die
                                                                                                   // beiden Taster

                                                                                                   // DCF77-Signal lesen und Daten speichern ----------------------------------------------------------

         uint8_t bitval;                                                                           // Wert eines empfangenen DCF77-Bits (0 oder 1)
         uint8_t dcfstat;                                                                          // Zwischenspeicher fr DCF77-Status
         uint8_t dcftemp;                                                                          // Zwischenspeicher fr die Bearbeitung der DCF77-Tabelle
         int8_t i;                                                                                 // Schleifenzhler

         dcfcnt ++;                                                                                // DCF77-Zhler erhhen
         dcfstat = PINE & (1 << PE2);                                                              // aktuellen DCF77-Status lesen und filtern
         if (dcfinv) dcfstat ^= 0x04;                                                              // falls Invertierung aktiviert -> invertieren
         if (!dcfstat1)                                                                            // wenn vorheriges DCF77-Signal Low
         {
            if (dcfstat)                                                                           // wenn aktuelles DCF77-Signal High (= Impuls-Start)
            {
               if ((dcfcnt > 2) && (dcfcnt < 13))                                                  // wenn Impulslnge 60-260ms -> gltiger Impuls
               {
                  dcfimp ++;                                                                       // Impulsanzahl erhhen
                  if (dcfcnt > 7) bitval = 1;                                                      // wenn Impulslnge > 140ms -> 1-Bit
                  else bitval = 0;                                                                 // sonst -> 0-Bit

                  for (i = 41; i >= 0; i --)                                                       // 42 Pufferbytes bearbeiten (von hinten beginnend)
                  {
                     dcftemp = dcftab[i];                                                          // Pufferbyte holen und merken
                     dcftab[i] = bitval;                                                           // neues Pufferbyte setzen
                     bitval = dcftemp;                                                             // bertrag fr nchstes Pufferbyte
                  }
               }
            }
         }
         else                                                                                      // wenn vorheriges DCF77-Signal High
         {
            if (!dcfstat)                                                                          // wenn aktuelles DCF77-Signal Low (= Impuls-Ende)
            {
               if (dcfcnt > 75) dcfstat2 = 1;                                                      // wenn Impulslnge > 1,5s -> neuer Minutenbeginn
               if (dcfcnt > 45) dcfcnt = 0;                                                        // wenn Imuplslnge > 900ms -> Zhler auf 0 setzen
            }
         }
         dcfstat1 = dcfstat;                                                                       // neuen DCF77-Status speichern

                                                                                                   // Taster 1 (Drehimpulsgeber) lesen ----------------------------------------------------------------

         uint8_t keytemp;                                                                          // Zwischenspeicher fr aktuellen Tasterstatus

         keytemp = ~PINC & 0x40;                                                                   // Tasterstatus 1 lesen
         if (keytemp == key1old)                                                                   // wenn aktueller = vorheriger Status
         {
            if (keytemp)                                                                           // wenn Taster gedrckt
            {
               key1stat = 1;                                                                       // aktuellen Status auf "Gedrckt" setzen
            }
            else                                                                                   // wenn Taster losgelassen
            {
               key1stat = 0;                                                                       // aktuellen Status auf "Losgelassen" setzen
               key1quit = 0;                                                                       // Quittierungs-Status lschen
            }
         }
         key1old = keytemp;                                                                        // aktuellen Status auf letzten Status verschieben

                                                                                                   // Taster 2 (Einzeltaster) lesen -------------------------------------------------------------------

         keytemp = ~PINC & 0x80;                                                                   // Tasterstatus 2 lesen
         if (keytemp == key2old)                                                                   // wenn aktueller = vorheriger Status
         {
            if (keytemp)                                                                           // wenn Taster gedrckt
            {
               key2stat = 1;                                                                       // aktuellen Status auf "Gedrckt" setzen
            }
            else                                                                                   // wenn Taster losgelassen
            {
               key2stat = 0;                                                                       // aktuellen Status auf "Losgelassen" setzen
               key2quit = 0;                                                                       // Quittierungs-Status lschen
            }
         }
         key2old = keytemp;                                                                        // aktuellen Status auf letzten Status verschieben

                                                                                                   // Zhler zur Teilung auf 1s (1Hz) bearbeiten ------------------------------------------------------

         intc20ms ++;                                                                              // Interrupt-Zhler 3 (20ms) erhhen
         if (intc20ms > 49)                                                                        // wenn berlauf
         {
            intc20ms = 0;                                                                          // Interrupt-Zhler 3 wieder zurcksetzen

                                                                                                   // Der folgende Abschnitt wird jede Sekunde aufgerufen

                                                                                                   // Uhrenzhler bearbeiten --------------------------------------------------------------------------

            uint8_t mflag = 0;                                                                     // Merker fr Monatswechsel

            sec_flag = 1;                                                                          // Sekunden-Flag setzen, Analoguhr und Symbole neu ausgeben
            chg_flag = 1;                                                                          // nderungs-Flag setzen, Informationen neu ausgeben
            second ++;                                                                             // Sekunden erhhen
            if (second > 59)                                                                       // wenn berlauf
            {
               second = 0;                                                                         // Sekunden zurcksetzen
               minute ++;                                                                          // Minuten erhhen
               if (minute > 59)                                                                    // wenn berlauf
               {
                  minute = 0;                                                                      // Minuten zurcksetzen
                  hour ++;                                                                         // Stunden erhhen
                  if (hour > 23)                                                                   // wenn berlauf
                  {
                     hour = 0;                                                                     // Stunden zurcksetzen
                     wday ++;                                                                      // Wochentag erhhen
                     if (wday > 7) wday = 1;                                                       // wenn Wochentagberlauf -> Wochentag zurcksetzen
                     day ++;                                                                       // Tag erhhen
                     if ((year % 4) == 0)                                                          // wenn Schaltjahr
                     {
                        if (day > pgm_read_byte (&daytab2[month - 1])) mflag = 1;                  // wenn Tag > Schaltjahr-Tabelle -> Monatswechsel
                     }
                     else                                                                          // wenn kein Schaltjahr
                     {
                        if (day > pgm_read_byte (&daytab1[month - 1])) mflag = 1;                  // wenn Tag > Normaljahr-Tabelle -> Monatswechsel
                     }

                     if (mflag)                                                                    // wenn Monatswechsel erforderlich
                     {
                        day = 1;                                                                   // Tag zurcksetzen
                        month ++;                                                                  // Monat erhhen
                        if (month > 12)                                                            // wenn berlauf
                        {
                           month = 1;                                                              // Monat zurcksetzen
                           year ++;                                                                // Jahr erhhen
                        }
                     }
                  }
               }
            }
         }
      }
                                                                                                   // zustzliche Zhlkette fr Timeouts und andere Funktionen bearbeiten -----------------------------
                                                                                                   // diese Funktionen sind damit unabhngig von der normalen Uhren-Zhlung, welche vom DCF-Signal
                                                                                                   // beeinflusst wird

      if (intc625us == 10)                                                                         // wenn Interruptzhler 1 (625s) den Wert 10 hat, werden die Timeout-Zhler bearbeitet; somit
                                                                                                   // erfolgt die Bearbeitung der Uhrenzhler und der Timeouts nie im gleichen Interruptzyklus
      {
                                                                                                   // Timeout-Zhler im 20ms-Raster (50Hz) bearbeiten -------------------------------------------------

         if (nightout < 255)                                                                       // wenn Nacht-Helligkeits-Timeout-Zhler aktiv
         {
            if (nightout > 0)                                                                      // wenn Nacht-Helligkeits-Timeout-Zhler noch nicht abgelaufen
            {
               nightout --;                                                                        // Timeout-Zhler vermindern
            }
         }

         intc20ms_2 ++;                                                                            // zustzlichen Interruptzhler 2 erhhen
         if (intc20ms_2 > 49)                                                                      // wenn berlauf
         {
            intc20ms_2 = 0;                                                                        // zustzlichen Interruptzhler 2 wieder zurcksetzen

                                                                                                   // Timeout-Zhler im Sekundenraster bearbeiten -----------------------------------------------------

            if (disptout < 255)                                                                    // wenn Anzeige-Timeout-Zhler aktiv
            {
               if (disptout > 0)                                                                   // wenn Anzeige-Timeout-Zhler noch nicht abgelaufen
               {
                  disptout --;                                                                     // Timeout-Zhler vermindern
               }
            }

            if (usbtout < 255)                                                                     // wenn USB-Symbol-Timeout-Zhler aktiv
            {
               if (usbtout > 0)                                                                    // wenn USB-Symbol-Timeout-Zhler noch nicht abgelaufen
               {
                  usbtout --;                                                                      // Timeout-Zhler vermindern
               }
            }

            intcsec ++;                                                                            // zustzlichen Sekundenzhler erhhen
            if (intcsec > 59)                                                                      // wenn berlauf
            {
               intcsec = 0;                                                                        // zustzlichen Sekundenzhler wieder zurcksetzen

                                                                                                   // Timeout-Zhler im Minutenraster bearbeiten ------------------------------------------------------

               if (syntout < 255)                                                                  // wenn Synchronisierungs-Timeout-Zhler aktiv
               {
                  if (syntout > 0)                                                                 // wenn Synchronisierungs-Timeout-Zhler noch nicht abgelaufen
                  {
                     syntout --;                                                                   // Timeout-Zhler vermindern
                  }
               }

               intcmin ++;                                                                         // zustzlichen Minutenzhlen erhhen
               if (intcmin > 59)                                                                   // wenn berlauf
               {
                  intcmin = 0;                                                                     // zustzlichen Minutenzhler wieder zurcksetzen

                                                                                                   // Timeout-Zhler im Stundenraster bearbeiten ------------------------------------------------------

                  if (dcftout < 255)                                                               // wenn DCF-Timeout-Zhler aktiv
                  {
                     if (dcftout > 0)                                                              // wenn DCF-Timeout-Zhler noch nicht abgelaufen
                     {
                        dcftout --;                                                                // Timeout-Zhler vermindern
                     }
                  }
               }
            }
         }
      }
                                                                                                   // zustzliche Zhlkette fr die Sound-Erzeugung bearbeiten ----------------------------------------
                                                                                                   // diese Funktionen sind damit unabhngig von der normalen Uhren-Zhlung, welche vom DCF-Signal
                                                                                                   // beeinflusst wird

      if (intc625us == 20)                                                                         // wenn Interruptzhler 1 (625s) den Wert 20 hat, werden der Sound bearbeitet; somit erfolgt die
      {                                                                                            // Bearbeitung nie im gleichen Interruptzyklus

         if (snstat == 1)                                                                          // wenn Sound-Erzeugung gestartet
         {
            snstat = 2;                                                                            // Sound-Erzeugung auf aktiv setzen
            curvol = 0;                                                                            // aktuelle Lautstrke auf 0 setzen
            sndnot = 0;                                                                            // Notenwert = 0 setzen
            sndlen = 0;                                                                            // Tonlnge = 0 setzen
            if (soundstat)                                                                         // wenn Soundausgabe eingeschaltet
            {
               TCCR3B = (1 << WGM32) | (1 << CS31);                                                // Timer3 im CTC-Mode und Vorteiler 8 aktivieren
            }
            else maxvol = 0;                                                                       // wenn Soundausgabe ausgeschaltet -> Lautstrke = 0 setzen
         }

         if (snstat == 2)                                                                          // wenn Sound-Erzeugung aktiv
         {
            if (sndlen == 0)                                                                       // wenn Tonlnge = 0 (komplett abgespielt)
            {
               sndnot = pgm_read_byte (sndptr ++);                                                 // nchsten Notenwert holen
               if (sndnot < 255)                                                                   // wenn weitere Tne folgen
               {
                  sndlen = pgm_read_byte (sndptr ++);                                              // nchste Tonlnge holen
                  snddec = pgm_read_byte (sndptr ++);                                              // nchsten Decay-Wert holen

                  if (sndnot < 254)                                                                // wenn keine Pause
                  {
                     OCR3A = pgm_read_word (&freqdata_array[sndnot]);                              // Timer3-OCR3A mit Teilerwert laden
                     if (TCNT3 >= OCR3A) TCNT3 = OCR3A - 1;                                        // wenn aktueller Zhlerwert von Timer3 zu gro -> Zhlerwert in zulssigen Bereich setzen
                     curvol = maxvol;                                                              // (Tonaussetzer verhindern) auf voreingestellte Lautstrke setzen (Ton einschalten)
                  }
               }
               else snstat = 3;                                                                    // wenn keine weiteren Tne folgen -> Sound-Erzeugung beenden
            }
         }

         if (snstat == 3)                                                                          // wenn Sound-Erzeugung beendet werden soll
         {
            snstat = 0;                                                                            // Sound-Erzeugung in Ruhezustand setzen
            curvol = 0;                                                                            // aktuelle Lautstrke auf 0 setzen
            TCCR3B = 1 << WGM32;                                                                   // Timer3 stoppen (Ton ausschalten)
         }

         if (curvol >= snddec) curvol -= snddec;                                                   // Lautstrke vermindern, wenn Ergebnis im positiven Bereich
         else curvol = 0;                                                                          // sonst Lautstrke = 0 setzen
         OCR1B  = curvol;                                                                          // neuen Lautstrkewert an PWM-Register ausgeben
         if (sndlen > 0) sndlen --;                                                                // wenn Tonlnge nicht komplett abgespielt -> 20ms abziehen
      }
   }
}

ISR (USART0_RX_vect)                                                                               // wird nach Empfang eines Zeichens ber USART0 aufgerufen, empfangenes Zeichen wird im RX0-Puffer
                                                                                                   // abgelegt und der Schreib-Zeiger erhht
{
   rx0buf[rx0write] = UDR0;                                                                        // empfangenes Zeichen auf Zeigerposition in den RX-Puffer schreiben
   rx0write ++;                                                                                    // Schreib-Zeiger erhhen
}

ISR (USART0_UDRE_vect)                                                                             // wird aufgerufen, wenn nach dem Senden eines Zeichens ber USART0 das Sende-Register leer ist
{
   if (tx0write != tx0read)                                                                        // wenn sich Zeichen im Puffer befinden
   {
      UDR0 = tx0buf[tx0read];                                                                      // nchstes Zeichen aus dem TX-Puffer holen und senden
      tx0read ++;                                                                                  // Lese-Zeiger erhhen
   }
   else UCSR0B = UCSR0B & ~ (1 << UDRIE0);                                                         // sonst Interrupt deaktivieren
}

ISR (USART1_RX_vect)                                                                               // wird nach Empfang eines Zeichens ber USART1 aufgerufen, empfangenes Zeichen wird im RX1-Puffer
                                                                                                   // abgelegt und der Schreib-Zeiger erhht
{
   rx1buf[rx1write] = UDR1;                                                                        // empfangenes Zeichen auf Zeigerposition in den RX-Puffer schreiben
   rx1write ++;                                                                                    // Schreib-Zeiger erhhen
}

ISR (USART1_UDRE_vect)                                                                             // wird aufgerufen, wenn nach dem Senden eines Zeichens ber USART1 das Sende-Register leer ist
{
   if (tx1write != tx1read)                                                                        // wenn sich Zeichen im Puffer befinden
   {
      UDR1 = tx1buf[tx1read];                                                                      // nchstes Zeichen aus dem TX-Puffer holen und senden
      tx1read ++;                                                                                  // Lese-Zeiger erhhen
   }
   else UCSR1B = UCSR1B & ~ (1 << UDRIE1);                                                         // sonst Interrupt deaktivieren
}

// Hauptprogramm - Initialisierung ====================================================================================================================================================================

int main (void)
{
   int8_t i;                                                                                       // universelle Schleifenvariable
   int8_t j;                                                                                       // universelle Schleifenvariable
   uint16_t k;                                                                                     // universelle 16-Bit-Variable
   uint8_t m;                                                                                      // universeller Zwischenspeicher
   uint8_t n;                                                                                      // universeller Zwischenspeicher
   char c;                                                                                         // universelle Zeichenvariable
   uint8_t okflag;                                                                                 // Flag fr Plausibilittstests

   PORTA = 0b00000000;                                                                             // PortA: alle Pins ohne Pull-Up
   DDRA  = 0b00000000;                                                                             // PortA: alle Pins auf Eingang
   PORTB = 0b00111110;                                                                             // PortB: Pins 1-5 mit Pull-Up
   DDRB  = 0b11000001;                                                                             // PortB: Pins 0, 6-7 auf Ausgang
   PORTC = 0b11111111;                                                                             // PortC: Pins 0-3 auf High, Pull-Up auf Pins 4-7
   DDRC  = 0b00001111;                                                                             // PortC: Pins 0-3 auf Ausgang
   PORTD = 0b11111000;                                                                             // PortD: Pins 3-4 auf High, Pull-Up auf Pins 5-7
   DDRD  = 0b00011000;                                                                             // PortD: Pins 3-4 auf Ausgang
   PORTE = 0b11110011;                                                                             // PortE: Pin 1 auf High, Pull-Up auf Pins 0, 4-7
   DDRE  = 0b00001010;                                                                             // PortE: Pins 1, 3 auf Ausgang
   PORTF = 0b11111111;                                                                             // PortF: alle Pins mit Pull-Up
   DDRF  = 0b00000000;                                                                             // PortF: alle Pins auf Eingang
   PORTG = 0b00011111;                                                                             // PortG: alle Pins mit Pull-Up
   DDRG  = 0b00000000;                                                                             // PortG: alle Pins auf Eingang

   led_on ();                                                                                      // zuerst LED einschalten

   lang = eep_read_byte (e_lang);                                                                  // Sprache aus EEPROM lesen
   if (lang > 1) lang = 1;                                                                         // gegebenenfalls Wert korrigieren
   lcdinv = eep_read_byte (e_lcdinv);                                                              // LCD-Invertierung aus EEPROM lesen
   if (lcdinv > 1) lcdinv = 1;                                                                     // gegebenenfalls Wert korrigieren
   dcfinv = eep_read_byte (e_dcfinv);                                                              // DCF-Invertierung aus EEPROM lesen
   if (dcfinv > 1) dcfinv = 1;                                                                     // gegebenenfalls Wert korrigieren
   dcfpup = eep_read_byte (e_dcfpup);                                                              // DCF-Pull-Up aus EPROM lesen
   if (dcfpup > 1) dcfpup = 1;                                                                     // gegebenenfalls Wert korrigieren
   nightsyn = eep_read_byte (e_nightsyn);                                                          // LCD-Nacht-Sync-Abschaltung aus EEPROM lesen
   if (nightsyn > 1) nightsyn = 1;                                                                 // gegebenenfalls Wert korrigieren
   lcdnight = eep_read_byte (e_lcdnight);                                                          // LCD-Nachtschaltung aus EEPROM lesen
   if (lcdnight > 1) lcdnight = 1;                                                                 // gegebenenfalls Wert korrigieren
   lcdnigbr = eep_read_byte (e_lcdnigbr);                                                          // LCD-Nacht-Helligkeit aus EEPROM lesen
   lcdnigm1 = eep_read_byte (e_lcdnigm1);                                                          // LCD-Nachtschaltung Beginn Minuten aus EEPROM lesen
   if (lcdnigm1 > 59) lcdnigm1 = 59;                                                               // gegebenenfalls Wert korrigieren
   lcdnigh1 = eep_read_byte (e_lcdnigh1);                                                          // LCD-Nachtschaltung Beginn Stunden aus EEPROM lesen
   if (lcdnigh1 > 23) lcdnigh1 = 23;                                                               // gegebenenfalls Wert korrigieren
   lcdnigm2 = eep_read_byte (e_lcdnigm2);                                                          // LCD-Nachtschaltung Ende Minuten aus EEPROM lesen
   if (lcdnigm2 > 59) lcdnigm2 = 59;                                                               // gegebenenfalls Wert korrigieren
   lcdnigh2 = eep_read_byte (e_lcdnigh2);                                                          // LCD-Nachtschaltung Ende Stunden aus EEPROM lesen
   if (lcdnigh2 > 23) lcdnigh2 = 23;                                                               // gegebenenfalls Wert korrigieren
   gong = eep_read_byte (e_gong);                                                                  // Gong-Status aus EEPROM lesen
   if (gong > 1) gong = 1;                                                                         // gegebenenfalls Wert korrigieren
   gongh1 = eep_read_byte (e_gongh1);                                                              // Gong-Beginn Stunden aus EEPROM lesen
   if (gongh1 > 23) gongh1 = 23;                                                                   // gegebenenfalls Wert korrigieren
   gongh2 = eep_read_byte (e_gongh2);                                                              // Gong-Ende Stunden aus EEPROM lesen
   if (gongh2 > 23) gongh2 = 23;                                                                   // gegebenenfalls Wert korrigieren
   gong_snd = eep_read_byte (e_gong_snd);                                                          // Gong-Soundnummer aus EEPROM lesen
   if (gong_snd > 12) gong_snd = 12;                                                               // gegebenenfalls Wert korrigieren
   gong_vol = eep_read_byte (e_gong_vol);                                                          // Gong-Lautstrke aus EEPROM lesen
   if (gong_vol > 10) gong_vol = 10;                                                               // gegebenenfalls Wert korrigieren

   for (i = 0; i < 4; i ++)                                                                        // Daten von 3 Alarmen und Geburtstags-Alarm aus EEPROM lesen
   {
      alarms[i][0] = eep_read_byte (e_alarms + (i * 10));                                          // Alarm-Modus aus EEPROM lesen
      if (alarms[i][0] > 4) alarms[i][0] = 4;                                                      // gegebenenfalls Wert korrigieren
      alarms[i][1] = eep_read_byte (e_alarms + (i * 10 + 1));                                      // Minute aus EEPROM lesen
      if (alarms[i][1] > 59) alarms[i][1] = 59;                                                    // gegebenenfalls Wert korrigieren
      alarms[i][2] = eep_read_byte (e_alarms + (i * 10 + 2));                                      // Stunde aus EEPROM lesen
      if (alarms[i][2] > 23) alarms[i][2] = 23;                                                    // gegebenenfalls Wert korrigieren
      alarms[i][3] = eep_read_byte (e_alarms + (i * 10 + 3));                                      // Kalendertag aus EEPROM lesen
      if (alarms[i][3] < 1) alarms[i][3] = 1;                                                      // gegebenenfalls Wert nach oben korrigieren
      if (alarms[i][3] > 31) alarms[i][3] = 31;                                                    // gegebenenfalls Wert nach unten korrigieren
      alarms[i][4] = eep_read_byte (e_alarms + (i * 10 + 4));                                      // Wochentag aus EEPROM lesen
      if (alarms[i][4] < 1) alarms[i][4] = 1;                                                      // gegebenenfalls Wert nach oben korrigieren
      if (alarms[i][4] > 7) alarms[i][4] = 7;                                                      // gegebenenfalls Wert nach unten korrigieren
      alarms[i][5] = eep_read_byte (e_alarms + (i * 10 + 5));                                      // Monat aus EEPROM lesen
      if (alarms[i][5] < 1) alarms[i][5] = 1;                                                      // gegebenenfalls Wert nach oben korrigieren
      if (alarms[i][5] > 12) alarms[i][5] = 12;                                                    // gegebenenfalls Wert nach unten korrigieren
      alarms[i][6] = eep_read_byte (e_alarms + (i * 10 + 6));                                      // Jahr aus EEPROM lesen
      if (alarms[i][6] > 99) alarms[i][6] = 99;                                                    // gegebenenfalls Wert korrigieren
      alarms[i][7] = eep_read_byte (e_alarms + (i * 10 + 7));                                      // Wochentagsprogramm aus EEPROM lesen
      if (alarms[i][7] > 9) alarms[i][7] = 9;                                                      // gegebenenfalls Wert korrigieren
      alarms[i][8] = eep_read_byte (e_alarms + (i * 10 + 8));                                      // Soundnummer aus EEPROM lesen
      if (alarms[i][8] > 12) alarms[i][8] = 12;                                                    // gegebenenfalls Wert korrigieren
      alarms[i][9] = eep_read_byte (e_alarms + (i * 10 + 9));                                      // Lautstrke aus EEPROM lesen
      if (alarms[i][9] > 10) alarms[i][9] = 10;                                                    // gegebenenfalls Wert korrigieren
      alarms[i][10] = 0;                                                                           // Alarmstatus lschen
   }

   s_enable = eep_read_byte (e_senable);                                                           // Sound-Freigabe aus EEPROM lesen
   if (s_enable > 1) s_enable = 1;                                                                 // gegebenenfalls Wert korrigieren
   bdordr = eep_read_byte (e_bdordr);                                                              // Geburtstags-Reihenfolge aus EEPROM lesen
   if (bdordr > 1) bdordr = 1;                                                                     // gegebenenfalls Wert korrigieren
   aclkpos = eep_read_byte (e_aclkpos);                                                            // Analog-Uhr-Position aus EEPROM lesen
   if (aclkpos > 1) aclkpos = 1;                                                                   // gegebenenfalls Wert korrigieren
   bdagem = eep_read_byte (e_bdagem);                                                              // Alters-Anzeigemodus aus EEPROM lesen
   if (bdagem > 1) bdagem = 1;                                                                     // gegebenenfalls Wert korrigieren
   bdlsort = eep_read_byte (e_bdlsort);                                                            // Geburtstagslisten-Sortierung aus EEPROM lesen
   if (bdlsort > 2) bdlsort = 2;                                                                   // gegebenenfalls Wert korrigieren
   for (i = 0; i < 3; i ++)                                                                        // 3 Eintrge aus der Geburtstags-Indexliste im EEPROM lesen
   {
      bdslist[i] = eep_read_byte (e_bdslist + i);                                                  // Eintrag aus EEPROM lesen
   }
   lastday = eep_read_byte (e_lastday);                                                            // letzten Tag aus EEPROM lesen
   lastmonth = eep_read_byte (e_lastmonth);                                                        // letzten Monat aus EEPROM lesen
   lastyear = eep_read_byte (e_lastyear);                                                          // letztes Jahr aus EEPROM lesen

   if (dcfpup) PORTE |= 1 << PE2;                                                                  // wenn konfiguriert -> DCF Pull-Up an PortE2 einschalten

   lcd_on ();                                                                                      // LCD einschalten und initialisieren
   disptout = c_disptout;                                                                          // Anzeige-Timeout starten
   usbtout = 255;                                                                                  // USB-Symbol-Timeout stoppen

   intc78us = 0;                                                                                   // Interrupt-Zhler 1 (78s) lschen
   intc625us = 0;                                                                                  // Interrupt-Zhler 2 (625s) lschen
   intc20ms = 0;                                                                                   // Interrupt-Zhler 3 (20ms) lschen
   sec_flag = 1;                                                                                   // Sekunden-Flag setzen, Analoguhr und Symbole neu ausgeben
   chg_flag = 1;                                                                                   // nderungs-Flag setzen, Informationen neu ausgeben
   alm_flag = 0;                                                                                   // Alarm-Flag lschen
   bd_flag = 0;                                                                                    // Geburtstags-Flag lschen
   bd_change = 0;                                                                                  // Geburtstags-Eintrags-nderungs-Flag lschen
   reorg_flag = 0;                                                                                 // Reorganisations-Flag lschen
   bdipos = 0;                                                                                     // ersten Eintrag 0 in der Geburtstagsliste auswhlen
   bd1pos = 0;                                                                                     // erste Zeile in der Geburtstage 1 Liste auswhlen
   bd2pos = 0;                                                                                     // erste Zeile in der Geburtstage 2 Liste auswhlen

   syncstat = 0;                                                                                   // Uhr wurde noch nicht synchronisiert
   second = 0;                                                                                     // Sekunden = 0
   minute = 0;                                                                                     // Minuten = 0
   hour = 12;                                                                                      // Stunden = 12
   day = 1;                                                                                        // Tag = 1
   month = 1;                                                                                      // Monat = 1
   year = c_year;                                                                                  // frhestes Jahr setzen

   i = 0;                                                                                          // Merker lschen
   if (lastyear < 100)                                                                             // wenn letztes gespeichertes Jahr gltig
   {
      if ((lastmonth > 0) && (lastmonth < 13))                                                     // wenn letzter gespeicherter Monat gltig
      {
         if ((lastyear % 4) == 0)                                                                  // wenn letztes gespeichertes Jahr ein Schaltjahr ist
         {
            if ((lastday > 0) && (lastday <= pgm_read_byte (&daytab2[lastmonth - 1]))) i = 1;      // wenn letzter gespeicherter Tag gltig ist -> Merker setzen
         }
         else                                                                                      // wenn letztes gespeichertes Jahr kein Schaltjahr ist
         {
            if ((lastday > 0) && (lastday <= pgm_read_byte (&daytab1[lastmonth - 1]))) i = 1;      // wenn letzter gespeicherter Tag gltig ist -> Merker setzen
         }
      }
   }
   if (i)                                                                                          // wenn Merker gesetzt (Tag, Monat und Jahr ok)
   {
      day = lastday;                                                                               // Tag auf letzten gespeicherten Tag setzen
      month = lastmonth;                                                                           // Monat auf letzten gespeicherten Monat setzen
      year = lastyear;                                                                             // Jahr auf letztes gespeichertes Jahr setzen
   }
   wday = calc_wday (day, month, c_yearh, year);                                                   // Wochentag ermitteln und setzen

   daychg = 0;                                                                                     // Tageswechsel-Erkennung zunchst auf 0 setzen
   dispmode = 0;                                                                                   // Anzeigemodus auf Uhr setzen
   encstep = 2;                                                                                    // Drehimpulsgeber auf Schrittstellung 2 bringen
   encpos = 0;                                                                                     // Drehimpulsgeber auf Position 0 bringen
   soundstat = 0;                                                                                  // Tonausgabe ausschalten
   usb_act = 0;                                                                                    // USB inaktiv
   mainopt = 0;                                                                                    // Hauptmen-Option auf 0 setzen
   submopt = 0;                                                                                    // Untermen-Option auf 0 setzen
   editmode = 0;                                                                                   // Editor-Option ausschalten
   editval = 0;                                                                                    // Editorwert auf 0 setzen
   editmin = 0;                                                                                    // unteren Editorgrenzwert auf 0 setzen
   editmax = 0;                                                                                    // oberen Editorgrenzwert auf 0 setzen
   lastsync[0] = 0;                                                                                // Letzte DCF-Synchronisierung auf ungltig setzen

   rx0write = 0;                                                                                   // Schreib-Zeiger auf RX0-Puffer initialisieren
   rx0read = 0;                                                                                    // Lese-Zeiger auf RX0-Puffer initialiseren
   tx0write = 0;                                                                                   // Schreib-Zeiger auf TX0-Puffer initialisieren
   tx0read = 0;                                                                                    // Lese-Zeiger auf TX0-Puffer initialiseren
   rx1write = 0;                                                                                   // Schreib-Zeiger auf RX1-Puffer initialisieren
   rx1read = 0;                                                                                    // Lese-Teiger auf RX1-Puffer initialiseren
   tx1write = 0;                                                                                   // Schreib-Zeiger auf TX1-Puffer initialisieren
   tx1read = 0;                                                                                    // Lese-Zeiger auf TX1-Puffer initialiseren
   rx0pos = 0;                                                                                     // USART0-Empfangsdaten-Position initialisieren
   rx1pos = 0;                                                                                     // USART1-Empfangsdaten-Position initialisieren
   cmdport = 0;                                                                                    // Status des Kommandopuffers initialisieren
   
   UCSR0A = 0;                                                                                     // USART0, Status-Register zurcksetzen
   UCSR0B = (1 << RXCIE0) | (1 << RXEN0) | (1 << TXEN0);                                           // USART0, RXD mit Interrupt und TXD aktivieren
   UCSR0C = (1 << UCSZ01) | (1 << UCSZ00);                                                         // USART0, Asynchron, 8-Bit, keine Paritt
   UBRR0H = 0;                                                                                     // USART0, Geschwindigkeit 921600 Baud einstellen
   UBRR0L = 0;
   UCSR1A = 0;                                                                                     // USART1, Status-Register zurcksetzen
   UCSR1B = (1 << RXCIE1) | (1 << RXEN1) | (1 << TXEN1);                                           // USART1, RXD mit Interrupt und TXD aktivieren
   UCSR1C = (1 << UCSZ11) | (1 << UCSZ10);                                                         // USART1, Asynchron, 8-Bit, keine Paritt
   UBRR1H = 0;                                                                                     // USART1, Geschwindigkeit 921600 Baud einstellen
   UBRR1L = 0;

   TCCR1A = (1 << COM1B1) | (3 << WGM10);                                                          // Timer1 Fast PWM mit CTC, OC1B aktiv
   TCCR1B = (3 << WGM12) | (1 << CS10);                                                            // Timer1 Fast PWM mit CTC, Vorteiler=1
   TIMSK = 1 << OCIE1A;                                                                            // Interrupt fr Output Compare 1A aktivieren
   OCR1A = F_CPU / 12800 - 1;                                                                      // Teilerwert fr 12800Hz festlegen
   OCR1B = 0;                                                                                      // PWM an OC1B = 0 setzen (Lautstrke 0)

   TCCR2 = (1 << WGM20) | (1 << COM21) | (1 << CS21) | (1 << CS20);                                // Timer2 PhaseCorrect PWM, OC2 aktivieren, Vorteiler = 64
   OCR2 = 0;                                                                                       // PWM-Wert=0 setzen

   TCCR3A = 1 << COM3A0;                                                                           // Timer3 Kanal A, Ausgang OCRA in Toggle Mode
   TCCR3B = 1 << WGM32;                                                                            // CTC-Mode einstellen, kein Takt

   sei ();                                                                                         // Interrupts aktivieren


// Hauptprogrammschleife ==============================================================================================================================================================================

   while (1)                                                                                       // Hauptprogramm luft als Endlosschleife
   {

// Anzeige steuern ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

                                                                                                   // Analoguhr anzeigen ------------------------------------------------------------------------------

      if (sec_flag)                                                                                // wenn Sekunden-Flag -> Analoguhr und Symbole neu anzeigen
      {
         sec_flag = 0;                                                                             // Sekunden-Flag wieder lschen
         get_clkdata ();                                                                           // Ziffernblatt in den RAM-Puffer laden
         dotsize = 2;                                                                              // Punktgre fr Stundenzeiger setzen
         get_pixelpos (c_hhlen, ((hour % 12) * 60) + minute);                                      // Position der Zeigerspitze ermitteln
         lcd_draw (63, 63, gr_x, gr_y);                                                            // Zeiger zeichnen
         dotsize = 1;                                                                              // Punktgre fr Minutenzeiger setzen
         get_pixelpos (c_mhlen, (minute * 12) + (second / 5));                                     // Position der Zeigerspitze ermitteln
         lcd_draw (63, 63, gr_x, gr_y);                                                            // Zeiger zeichnen
         dotsize = 0;                                                                              // Punktgre fr Sekundenzeiger setzen
         get_pixelpos (c_shlen1, second * 12);                                                     // Position der Zeigerspitze ermitteln
         lcd_draw (63, 63, gr_x, gr_y);                                                            // Zeiger zeichnen
         get_pixelpos (c_shlen2, second * 12 + 360);                                               // Position des Zeigerendes ermitteln
         lcd_draw (63, 63, gr_x, gr_y);                                                            // Zeiger zeichnen

         text_inv = 0;                                                                             // Textinvertierung ausschalten
         if (syncstat < 2)                                                                         // wenn die Uhr noch keine gltige Zeit hat
         {
            text_x = 1;
            text_y = 1;
            lcd_symbol (7);                                                                        // Symbol "Uhr nicht gestellt" (Ausrufezeichen) ausgeben
         }
         if (syncstat > 2)                                                                         // wenn die Uhr ber DCF77 synchronisiert wurde
         {
            text_x = 1;
            text_y = 1;                                                                            // Koordinaten fr Antennen-Symbol
            lcd_symbol (syncstat - 3);                                                             // Symbol entsprechend DCF77-Status ausgeben
            text_x = 110;
            text_y = 1;                                                                            // Koordinaten fr Winterzeit-/Sommerzeit-Symbol
            lcd_symbol (timezone + 2);                                                             // Winterzeit-/Sommerzeit-Symbol ausgeben
         }
         text_x = 1;
         text_y = 111;                                                                             // Koordinaten fr Lautsprecher-Symbol
         lcd_symbol (soundstat + 4);                                                               // Lautsprechersymbol ausgeben
         text_x = 110;
         text_y = 111;                                                                             // Koordinaten fr USB-Symbol
         if (usb_act) lcd_symbol (6);                                                              // gegebenenfalls USB-Symbol ausgeben

         copy_clkdata ();                                                                          // komplette Analoguhr in den LCD-Speicher kopieren

         if (scrshot == 1)                                                                         // wenn ein Screenshot (Analoguhr) angefordert wird
         {
            for (k = 0; k < 2048; k ++)                                                            // 2048 Bytes senden (128 x 16 Bytes)
            {
               if ((k % 16) == 0)                                                                  // wenn Zeilenanfang
               {
                  usb_char ('s', cmdport);                                                         // Antwort 's' senden
                  usb_char (':', cmdport);                                                         // Trennzeichen senden
               }
               usb_byte (lcd_bitmap[k], cmdport);                                                  // Datenbyte senden
               if ((k % 16) == 15)                                                                 // wenn Zeilen ende
               {
                  usb_char ('\r', cmdport);                                                        // Carriage Return senden
                  usb_char ('\n', cmdport);                                                        // Line Feed senden
               }               
               else usb_char (',', cmdport);                                                       // sonst Komma senden
            }
            scrshot = 2;                                                                           // Screenshot-Steuerung auf Info-Bereich setzen
         }
      }
                                                                                                   // Infobereich anzeigen ----------------------------------------------------------------------------

      if (chg_flag)                                                                                // wenn Anzeige aktualisiert werden muss
      {
         chg_flag = 0;                                                                             // nderungs-Flag wieder lschen
         ram_clear ();                                                                             // LCD-Bitmap lschen
         text_inv = 0;                                                                             // alle Texte normal anzeigen

         if (dispmode == 0)                                                                        // Anzeigemodus 0, normale Uhrenanzeige ------------------------------------------------------------
         {
            lcd_timedate ();                                                                       // Zeit und Datum ausgeben
            lcd_line (4, 44, 109);                                                                 // Trennlinie zwischen Uhr und Alarmen

            for (i = 0; i < 3; i ++)                                                               // 3 Alarmzeilen ausgeben
            {
               text_x = 3;
               text_y = i * 13 + 47;                                                               // Text-Position festlegen
               lcd_alarm (i);                                                                      // Alarmzeile ausgeben
            }

            lcd_line (4, 86, 109);                                                                 // Trennlinie zwischen Alarmen und Geburtstagen

            for (i = 0; i < 3; i ++)                                                               // 3 Geburtstagszeilen ausgeben
            {
               bdindx = bdslist[i];                                                                // Wert aus dem Geburtstagsindex lesen (Eintrge 0-2)
               if (bdindx < 115)                                                                   // wenn gltiger Eintrag vorhanden
               {
                  text_y = i * 13 + 89;                                                            // Text-Zeilen-Position fr Geburtstag festlegen
                  lcd_bdname ();                                                                   // Geburtstagsdaten ausgeben
               }
            }
         }

         if (dispmode == 1)                                                                        // Anzeigemodus 1, Hauptmen -----------------------------------------------------------------------
         {
            lcd_timedate ();                                                                       // Zeit und Datum ausgeben

            for (i = 0; i < 3; i ++)                                                               // 3 Alarmzeiten ausgeben
            {
               text_x = 3;
               text_y = i * 14 + 45;                                                               // Text-Position festlegen
               lcd_alarm (i);                                                                      // Alarmzeile ausgeben
            }

            text_x = 3;
            text_y = 87;                                                                           // Text-Koordinaten fr Mentext "Geburtstage"
            lcd_str (8, 0);                                                                        // Text ausgeben
            text_x = 3;
            text_y = 101;                                                                          // Text-Koordinaten fr Mentext "Informationen"
            lcd_str (9, 0);                                                                        // Text ausgeben
            text_x = 3;
            text_y = 115;                                                                          // Text-Koordinaten fr Mentext "Einstellungen"
            lcd_str (10, 0);                                                                       // Text ausgeben

            text_x = strl (10, 0) + 9;                                                             // Text-Koordinaten fr Mentext "1"
            lcd_char ('1');                                                                        // Zeichen ausgeben
            text_x = strl (10, 0) + 21;                                                            // Text-Koordinaten fr Mentext "2"
            lcd_char ('2');                                                                        // Zeichen ausgeben
            text_x = strl (10, 0) + 33;                                                            // Text-Koordinaten fr Mentext "3"
            lcd_char ('3');                                                                        // Zeichen ausgeben

            switch (mainopt)                                                                       // gewhlte Hauptmen-Option umrahmen
            {
               case 0: lcd_frame (2, 0, 111, 43); break;
               case 1: lcd_frame (2, 43, 111, 57); break;
               case 2: lcd_frame (2, 57, 111, 71); break;
               case 3: lcd_frame (2, 71, 111, 85); break;
               case 4: lcd_frame (2, 85, strl (8, 0) + 4, 99); break;
               case 5: lcd_frame (2, 99, strl (9, 0) + 4, 113); break;
               case 6: lcd_frame (strl (10, 0) + 8, 113, strl (10, 0) + 16, 127); break;
               case 7: lcd_frame (strl (10, 0) + 20, 113, strl (10, 0) + 28, 127); break;
               case 8: lcd_frame (strl (10, 0) + 32, 113, strl (10, 0) + 40, 127);
            }
            if (mainopt > 5) lcd_frame (2, 113, strl (10, 0) + 4, 127);                            // zustzlich "Einstellungen" umrahmen
         }

         if (dispmode == 2)                                                                        // Men "Uhr stellen" ------------------------------------------------------------------------------
         {
            text_inv = 1;                                                                          // berschrift invertieren
            text_x = 56 - (strl (11, 0) / 2);                                                      // berschrift zentriert ausgeben
            text_y = 0;
            lcd_str (11, 0);                                                                       // berschrift "Uhr stellen"
            lcd_char (127);                                                                        // Endezeichen ausgeben
            text_inv = 0;                                                                          // weitere Texte normal ausgeben

            text_x = 3;
            text_y = 45;                                                                           // Text-Koordinaten fr Zeit
            lcd_str (12, 0);                                                                       // Text "Zeit" ausgeben
            text_x = 82;
            if (editmode && (submopt == 0))                                                        // wenn Zeit editiert wird
            {
               text_inv = 1;                                                                       // Zeit invers ausgeben
               lcd_time (editval / 60, editval % 60);                                              // Editorwert ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else lcd_time (m_hour, m_minute);                                                      // sonst Zeit ausgeben

            text_x = 3;
            text_y = 59;                                                                           // Text-Koordinaten fr Tag
            lcd_str (13, 0);                                                                       // Text "Tag" ausgeben
            text_x = 97;
            if (editmode && (submopt == 1))                                                        // wenn Tag editiert wird
            {
               text_inv = 1;                                                                       // Tag invers ausgeben
               lcd_numb (editval);                                                                 // Editorwert ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else lcd_numb (m_day);                                                                 // sonst Tag ausgeben

            text_x = 3;
            text_y = 73;                                                                           // Text-Koordinaten fr Monat
            lcd_str (14, 0);                                                                       // Text "Monat" ausgeben
            if (editmode && (submopt == 2))                                                        // wenn Monat editiert wird
            {
               text_inv = 1;                                                                       // Monat invers ausgeben
               text_x = 109 - strl (editval + 15, 0);                                              // Editorwert rechtsbndig ausgeben
               lcd_str (editval + 15, 0);                                                          // Editorwert als Monatsname ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else
            {
               text_x = 109 - strl (m_month + 15, 0);                                              // sonst Monatsname rechtsbndig ausgeben
               lcd_str (m_month + 15, 0);                                                          // Monatsname ausgeben
            }

            text_x = 3;
            text_y = 87;                                                                           // Text-Koordinaten fr Jahr
            lcd_str (15, 0);                                                                       // Text "Jahr" ausgeben
            text_x = 85;
            lcd_numb (c_yearh);                                                                    // Jahrhundert ausgeben
            if (editmode && (submopt == 3))                                                        // wenn Jahr editiert wird
            {
               text_inv = 1;                                                                       // Jahr invers ausgeben
               lcd_numb (editval);                                                                 // Editorwert ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else lcd_numb (m_year);                                                                // sonst Jahr ausgeben

            text_x = 3;
            text_y = 101;                                                                          // Text-Koordinaten fr "Zeit bernehmen"
            lcd_str (28, 0);                                                                       // Text ausgeben
            text_x = 3;
            text_y = 115;                                                                          // Text-Koordinaten fr "DCF77-Sync starten"
            lcd_str (29, 0);                                                                       // Text ausgeben

            switch (submopt)                                                                       // gewhlte Untermen-Option umrahmen
            {
               case 0: lcd_frame (2, 43, 111, 57); break;
               case 1: lcd_frame (2, 57, 111, 71); break;
               case 2: lcd_frame (2, 71, 111, 85); break;
               case 3: lcd_frame (2, 85, 111, 99); break;
               case 4: lcd_frame (2, 99, 111, 113); break;
               case 5: lcd_frame (2, 113, 111, 127);
            }

         }

         if (dispmode == 3)                                                                        // Men "Einstellungen 1" --------------------------------------------------------------------------
         {
            text_inv = 1;                                                                          // berschrift invertieren
            text_x = 56 - ((strl (10, 0) + 11) / 2);                                               // berschrift zentriert ausgeben
            text_y = 0;
            lcd_str (10, 0);                                                                       // berschrift "Einstellungen"
            lcd_char (' ');                                                                        // Leerzeichen ausgeben
            lcd_char ('1');                                                                        // "1" ausgeben
            lcd_char (127);                                                                        // Endezeichen ausgeben
            text_inv = 0;                                                                          // weitere Texte normal ausgeben

            text_x = 3;
            text_y = 17;                                                                           // Text-Koordinaten fr Sprache
            lcd_str (30, 0);                                                                       // Text "Sprache" ausgeben
            if (editmode && (submopt == 0))                                                        // wenn Sprache editiert wird
            {
               text_inv = 1;                                                                       // Sprache invers ausgeben
               text_x = 109 - strl (editval + 31, 0);                                              // Editorwert rechtsbndig ausgeben
               lcd_str (editval + 31, 0);                                                          // Editorwert als String ausgeben
               lcd_char (127);                                                                     // Endezeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else
            {
               text_x = 109 - strl (lang + 31, 0);                                                 // sonst Sprache rechtsbndig ausgeben
               lcd_str (lang + 31, 0);                                                             // Sprache ausgeben
            }

            text_x = 3;
            text_y = 31;                                                                           // Text-Koordinaten fr Analog-Uhr-Position
            lcd_str (77, 0);                                                                       // Text "Analog-Uhr" ausgeben
            if (editmode && (submopt == 1))                                                        // wenn Analog-Uhr-Position editiert wird
            {
               text_inv = 1;                                                                       // Analog-Uhr-Position invers ausgeben
               text_x = 109 - strl (editval + 78, 0);                                              // Editorwert rechtsbndig ausgeben
               lcd_str (editval + 78, 0);                                                          // Editorwert als String ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else
            {
               text_x = 109 - strl (aclkpos + 78, 0);                                              // sonst Analog-Uhr-Position rechtsbndig ausgeben
               lcd_str (aclkpos + 78, 0);                                                          // Analog-Uhr-Position ausgeben
            }

            text_x = 3;
            text_y = 45;                                                                           // Text-Koordinaten fr Display-Invertierung
            lcd_str (33, 0);                                                                       // Text "Display" ausgeben
            if (editmode && (submopt == 2))                                                        // wenn Display-Invertierung editiert wird
            {
               text_inv = 1;                                                                       // Display-Invertierung invers ausgeben
               text_x = 109 - strl (editval + 34, 0);                                              // Editorwert rechtsbndig ausgeben
               lcd_str (editval + 34, 0);                                                          // Editorwert als String ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else
            {
               text_x = 109 - strl (lcdinv + 34, 0);                                               // sonst Display-Invertierung rechtsbndig ausgeben
               lcd_str (lcdinv + 34, 0);                                                           // Display-Invertierung ausgeben
            }

            text_x = 3;
            text_y = 59;                                                                           // Text-Koordinaten fr DCF-Nacht-Synchronisierung
            lcd_str (41, 0);                                                                       // Text "Nacht-Sync" ausgeben
            if (editmode && (submopt == 3))                                                        // wenn DCF-Nacht-Synchronisierung editiert wird
            {
               text_inv = 1;                                                                       // DCF-Nacht-Synchronisierung invers ausgeben
               text_x = 109 - strl (editval + 39, 0);                                              // Editorwert rechtsbndig ausgeben
               lcd_str (editval + 39, 0);                                                          // Editorwert als String ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else
            {
               text_x = 109 - strl (nightsyn + 39, 0);                                             // sonst DCF-Nacht-Synchronisierung rechtsbndig ausgeben
               lcd_str (nightsyn + 39, 0);                                                         // DCF-Nacht-Synchronisierung ausgeben
            }

            text_x = 3;
            text_y = 73;                                                                           // Text-Koordinaten fr LCD-Nacht-Modus
            lcd_str (42, 0);                                                                       // Text "Nacht-Modus" ausgeben
            if (editmode && (submopt == 4))                                                        // wenn LCD-Nacht-Modus editiert wird
            {
               text_inv = 1;                                                                       // LCD-Nacht-Modus invers ausgeben
               text_x = 109 - strl (editval + 39, 0);                                              // Editorwert rechtsbndig ausgeben
               lcd_str (editval + 39, 0);                                                          // Editorwert als String ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else
            {
               text_x = 109 - strl (lcdnight + 39, 0);                                             // sonst LCD-Nacht-Modus rechtsbndig ausgeben
               lcd_str (lcdnight + 39, 0);                                                         // LCD-Nacht-Modus ausgeben
            }

            text_x = 3;
            text_y = 87;                                                                           // Text-Koordinaten fr LCD-Nacht-Helligkeit
            lcd_str (43, 0);                                                                       // Text "Nacht-Helligkeit" ausgeben
            text_x = 91;
            if (editmode && (submopt == 5))                                                        // wenn LCD-Nacht-Helligkeit editiert wird
            {
               text_inv = 1;                                                                       // LCD-Nacht-Helligkeit invers ausgeben
               lcd_numb3 (editval);                                                                // Editorwert ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else lcd_numb3 (lcdnigbr);                                                             // sonst LCD-Nacht-Helligkeit ausgeben

            text_x = 3;
            text_y = 101;                                                                          // Text-Koordinaten fr Nacht-Modus-Beginn
            lcd_str (44, 0);                                                                       // Text "Nacht von" ausgeben
            text_x = 82;
            if (editmode && (submopt == 6))                                                        // wenn Nacht-Modus-Beginn editiert wird
            {
               text_inv = 1;                                                                       // Nacht-Modus-Beginn invers ausgeben
               lcd_time (editval / 60, editval % 60);                                              // Editorwert ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else lcd_time (lcdnigh1, lcdnigm1);                                                    // sonst Nacht-Modus-Beginnzeit ausgeben

            text_x = 3;
            text_y = 115;                                                                          // Text-Koordinaten fr Nacht-Modus-Ende
            lcd_str (45, 0);                                                                       // Text "Nacht bis" ausgeben
            text_x = 82;
            if (editmode && (submopt == 7))                                                        // wenn Nacht-Modus-Ende editiert wird
            {
               text_inv = 1;                                                                       // Nacht-Modus-Ende invers ausgeben
               lcd_time (editval / 60, editval % 60);                                              // Editorwert ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else lcd_time (lcdnigh2, lcdnigm2);                                                    // sonst Nacht-Modus-Endezeit ausgeben

            switch (submopt)                                                                       // gewhlte Untermen-Option umrahmen
            {
               case 0: lcd_frame (2, 15, 111, 29); break;
               case 1: lcd_frame (2, 29, 111, 43); break;
               case 2: lcd_frame (2, 43, 111, 57); break;
               case 3: lcd_frame (2, 57, 111, 71); break;
               case 4: lcd_frame (2, 71, 111, 85); break;
               case 5: lcd_frame (2, 85, 111, 99); break;
               case 6: lcd_frame (2, 99, 111, 113); break;
               case 7: lcd_frame (2, 113, 111, 127);
            }
         }

         if (dispmode == 4)                                                                        // Men "Informationen" ----------------------------------------------------------------------------
         {
            text_inv = 1;                                                                          // berschrift invertieren
            text_x = 56 - (strl (9, 0) / 2);                                                       // berschrift zentriert ausgeben
            text_y = 0;
            lcd_str (9, 0);                                                                        // berschrift "Informationen"
            lcd_char (127);                                                                        // Endezeichen ausgeben
            text_inv = 0;                                                                          // weitere Texte normal ausgeben

            text_x = 56 - (strl (46, 0) / 2);                                                      // Text fr Software-Version zentriert ausgeben
            text_y = 17;                                                                           // Text-Koordinaten fr Software-Version
            lcd_str (46, 0);                                                                       // Text "Software-Version" ausgeben

            text_x = 56 - (strl (0, 0) / 2);                                                       // Versionsnummer zentriert ausgeben
            text_y = 31;                                                                           // Text-Koordinaten fr Versionsnummer
            lcd_str (0, 0);                                                                        // Text ausgeben

            lcd_line (4, 47, 109);                                                                 // Trennlinie zwischen Version und letzter Synchronisierung

            if (lastsync[0])                                                                       // wenn eine gltige Zeit vorhanden ist
            {
               text_x = 56 - ((35 + strl (47, 0)) / 2);                                            // Text fr DCF-Sync und Uhrzeit zentriert ausgeben
               text_y = 52;                                                                        // Text-Koordinaten fr DCF-Sync und Uhrzeit
               lcd_str (47, 0);                                                                    // Text ausgeben
               lcd_char (':');                                                                     // Trennzeichen ausgeben
               lcd_char (' ');                                                                     // Leerzeichen ausgeben
               lcd_time (lastsync[2], lastsync[1]);                                                // Uhrzeit ausgeben
               text_x = 56 - ((49 + strl (lastsync[4], 0)) / 2);                                   // Datum zentriert ausgeben
               text_y = 66;                                                                        // Text-Koordinaten fr Datum
               lcd_date (lastsync[3], lastsync[5], lastsync[6]);                                   // Datum ausgeben
               lcd_char (',');                                                                     // Komma ausgeben
               lcd_char (0x90);                                                                    // schmales Leerzeichen ausgeben
               lcd_str (lastsync[4], 0);                                                           // Wochentag ausgeben
            }
            else                                                                                   // wenn keine gltige Zeit vorhanden ist
            {
               text_x = 56 - (strl (47, 0) / 2);                                                   // Text fr DCF77-Sync zentriert ausgeben
               text_y = 52;                                                                        // Text-Koordinaten fr DCF77-Sync und Uhrzeit
               lcd_str (47, 0);                                                                    // Text ausgeben
               text_x = 56 - (strl (48, 0) / 2);                                                   // Text fr "keine Daten" zentriert ausgeben
               text_y = 66;                                                                        // Text-Koordinaten fr "keine Daten"
               lcd_str (48, 0);                                                                    // Text ausgeben
            }

            lcd_line (4, 82, 109);                                                                 // Trennlinie zwischen letzter Sync und Geburtstagsstatus

            text_x = 56 - (strl (8, 0) / 2);                                                       // Text fr Geburtstage zentriert ausgeben
            text_y = 87;                                                                           // Text-Koordinaten fr Geburtstags-Speicher
            lcd_str (8, 0);                                                                        // Text ausgeben
            j = 0;                                                                                 // Zhler fr belegte Geburtstags-Eintrge lschen
            for (i = 0; i < 115; i ++)                                                             // 115 Geburtstags-Eintrge prfen
            {
               if (eep_read_byte (e_bddata + (i * 34)) < 255) j ++;                                // wenn Eintrag belegt, dann Zhler erhhen
            }

            text_x = 56 - ((24 + strl (49, 0)) / 2);                                               // Text fr belegte Eintrge zentriert ausgeben
            text_y = 101;                                                                          // Text-Koordinaten fr "belegte Eintrge"
            lcd_numb3 (j);                                                                         // Anzahl der belegten Eintrge im Geburtstagsspeicher ausgeben
            lcd_char (' ');                                                                        // Leerzeichen ausgeben
            lcd_str (49, 0);                                                                       // Text "belegte Eintrge" ausgeben
            text_x = 56 - ((24 + strl (50, 0)) / 2);                                               // Text fr freie Eintrge zentriert ausgeben
            text_y = 115;                                                                          // Text-Koordinaten fr "freie Eintrge"
            lcd_numb3 (115 - j);                                                                   // Anzahl der freien Eintrge im Geburtstagsspeicher ausgeben
            lcd_char (' ');                                                                        // Leerzeichen ausgeben
            lcd_str (50, 0);                                                                       // Text "freie Eintrge" ausgeben
         }

         if (dispmode == 5)                                                                        // Men "Alarme einstellen" ------------------------------------------------------------------------
         {
            text_inv = 1;                                                                          // berschrift invertieren
            text_x = 56 - ((strl (51, 0) + 20) / 2);                                               // berschrift zentriert ausgeben
            text_y = 0;
            lcd_str (51, 0);                                                                       // berschrift "Alarm einstellen"
            lcd_char (' ');                                                                        // Leerzeichen ausgeben
            lcd_char ('(');                                                                        // Klammerzeichen ausgeben
            lcd_char (alm + 49);                                                                   // Alarmnummer ausgeben
            lcd_char (')');                                                                        // Klammerzeichen ausgeben
            lcd_char (127);                                                                        // Endezeichen ausgeben
            text_inv = 0;                                                                          // weitere Texte normal ausgeben

            text_x = 3;
            text_y = 17;                                                                           // Text-Koordinaten fr Alarm
            lcd_str (52, 0);                                                                       // Text "Alarm" ausgeben
            if (editmode && (submopt == 0))                                                        // wenn Alarm-Modus editiert wird
            {
               text_inv = 1;                                                                       // Aus/Einmal/Tglich/Wchentlich/Monatlich invers ausgeben
               text_x = 109 - strl (editval + 53, 0);                                              // Editorwert rechtsbndig ausgeben
               lcd_str (editval + 53, 0);                                                          // Editorwert als String ausgeben
               lcd_char (127);                                                                     // Endezeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else
            {
               text_x = 109 - strl (alarms[alm][0] + 53, 0);                                       // sonst Alarm-Modus rechtsbndig ausgeben
               lcd_str (alarms[alm][0] + 53, 0);                                                   // Aus/Einmal/Tglich/Wchentlich/Monatlich ausgeben
            }

            text_x = 3;
            text_y = 31;                                                                           // Text-Koordinaten fr Zeit
            lcd_str (12, 0);                                                                       // Text "Zeit" ausgeben
            text_x = 82;
            if (editmode && (submopt == 1))                                                        // wenn Zeit editiert wird
            {
               text_inv = 1;                                                                       // Zeit invers ausgeben
               lcd_time (editval / 60, editval % 60);                                              // Editorwert ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else lcd_time (alarms[alm][2], alarms[alm][1]);                                        // sonst Zeit ausgeben

            text_x = 3;
            text_y = 45;                                                                           // Text-Koordinaten fr Tag
            lcd_str (13, 0);                                                                       // Text "Tag" ausgeben
            if ((alarms[alm][0] == 0) || (alarms[alm][0] == 1) || (alarms[alm][0] == 4))
            {                                                                                      // wenn Alarm aus, einmalig oder monatlich
               text_x = 97;
               if (editmode && (submopt == 2))                                                     // wenn Tag editiert wird
               {
                  text_inv = 1;                                                                    // Tag invers ausgeben
                  lcd_numb (editval);                                                              // Editorwert ausgeben
                  lcd_char (127);                                                                  // Ende-Zeichen ausgeben
                  text_inv = 0;                                                                    // weitere Texte normal anzeigen
               }
               else lcd_numb (alarms[alm][3]);                                                     // sonst Tag ausgeben
            }

            text_x = 3;
            text_y = 59;                                                                           // Text-Koordinaten fr Monat
            lcd_str (14, 0);                                                                       // Text "Monat" ausgeben
            if (alarms[alm][0] < 2)                                                                // wenn Alarm aus oder einmalig
            {
               if (editmode && (submopt == 3))                                                     // wenn Monat editiert wird
               {
                  text_inv = 1;                                                                    // Monat invers ausgeben
                  text_x = 109 - strl (editval + 15, 0);                                           // Editorwert rechtsbndig ausgeben
                  lcd_str (editval + 15, 0);                                                       // Editorwert als Monatsname ausgeben
                  lcd_char (127);                                                                  // Ende-Zeichen ausgeben
                  text_inv = 0;                                                                    // weitere Texte normal anzeigen
               }
               else
               {
                  text_x = 109 - strl (alarms[alm][5] + 15, 0);                                    // sonst Monatsname rechtsbndig ausgeben
                  lcd_str (alarms[alm][5] + 15, 0);                                                // Monatsname ausgeben
               }
            }

            text_x = 3;
            text_y = 73;                                                                           // Text-Koordinaten fr Jahr
            lcd_str (15, 0);                                                                       // Text "Jahr" ausgeben
            if (alarms[alm][0] < 2)                                                                // wenn Alarm aus oder einmalig
            {
               text_x = 85;
               lcd_numb (c_yearh);                                                                 // Jahrhundert ausgeben
               if (editmode && (submopt == 4))                                                     // wenn Jahr editiert wird
               {
                  text_inv = 1;                                                                    // Jahr invers ausgeben
                  lcd_numb (editval);                                                              // Editorwert ausgeben
                  lcd_char (127);                                                                  // Ende-Zeichen ausgeben
                  text_inv = 0;                                                                    // weitere Texte normal anzeigen
               }
               else lcd_numb (alarms[alm][6]);                                                     // sonst Jahr ausgeben
            }

            text_x = 3;
            text_y = 87;                                                                           // Text-Koordinaten fr Wochentag
            lcd_str (58, 0);                                                                       // Text "Wochen-Programm" ausgeben
            if (alarms[alm][0] == 3)                                                               // wenn Alarm wchentlich eingestellt ist
            {
               if (editmode && (submopt == 5))                                                     // wenn Wochentag editiert wird
               {
                  text_inv = 1;                                                                    // Wochenprogramm invers ausgeben
                  switch (editval)                                                                 // Tage entsprechend Wochenprogramm ausgeben
                  {
                     case 0: text_x = 109 - strl (1, 2);
                             lcd_str (1, 2); break;                                                // Montag rechtsbndig ausgeben
                     case 1: text_x = 109 - strl (2, 2);
                             lcd_str (2, 2); break;                                                // Dienstag rechtsbndig ausgeben
                     case 2: text_x = 109 - strl (3, 2);
                             lcd_str (3, 2); break;                                                // Mittwoch rechtsbndig ausgeben
                     case 3: text_x = 109 - strl (4, 2);
                             lcd_str (4, 2); break;                                                // Donnerstag rechtsbndig ausgeben
                     case 4: text_x = 109 - strl (5, 2);
                             lcd_str (5, 2); break;                                                // Freitag rechtsbndig ausgeben
                     case 5: text_x = 109 - strl (6, 2);
                             lcd_str (6, 2); break;                                                // Samstag rechtsbndig ausgeben
                     case 6: text_x = 109 - strl (7, 2);
                             lcd_str (7, 2); break;                                                // Sonntag rechtsbndig ausgeben
                     case 7: text_x = 103 - strl (1, 2) - strl (5, 2);
                             lcd_str (1, 2); lcd_char ('-');
                             lcd_str (5, 2); break;                                                // Montag - Freitag rechtsbndig ausgeben
                     case 8: text_x = 103 - strl (1, 2) - strl (6, 2);
                             lcd_str (1, 2); lcd_char ('-');
                             lcd_str (6, 2); break;                                                // Montag - Samstag rechtsbndig ausgeben
                     case 9: text_x = 103 - strl (6, 2) - strl (7, 2);
                             lcd_str (6, 2); lcd_char ('-');
                             lcd_str (7, 2); break;                                                // Samstag - Sonntag rechtsbndig ausgeben
                  }
                  lcd_char (127);                                                                  // Ende-Zeichen ausgeben
                  text_inv = 0;                                                                    // weitere Texte normal anzeigen
               }
               else
               {
                  switch (alarms[alm][7])                                                          // Tage entsprechend Wochenprogramm ausgeben
                  {
                     case 0: text_x = 109 - strl (1, 2);
                             lcd_str (1, 2); break;                                                // Montag rechtsbndig ausgeben
                     case 1: text_x = 109 - strl (2, 2);
                             lcd_str (2, 2); break;                                                // Dienstag rechtsbndig ausgeben
                     case 2: text_x = 109 - strl (3, 2);
                             lcd_str (3, 2); break;                                                // Mittwoch rechtsbndig ausgeben
                     case 3: text_x = 109 - strl (4, 2);
                             lcd_str (4, 2); break;                                                // Donnerstag rechtsbndig ausgeben
                     case 4: text_x = 109 - strl (5, 2);
                             lcd_str (5, 2); break;                                                // Freitag rechtsbndig ausgeben
                     case 5: text_x = 109 - strl (6, 2);
                             lcd_str (6, 2); break;                                                // Samstag rechtsbndig ausgeben
                     case 6: text_x = 109 - strl (7, 2);
                             lcd_str (7, 2); break;                                                // Sonntag rechtsbndig ausgeben
                     case 7: text_x = 103 - strl (1, 2) - strl (5, 2);
                             lcd_str (1, 2); lcd_char ('-'); lcd_str (5, 2); break;                // Montag - Freitag  rechtsbndig
                     case 8: text_x = 103 - strl (1, 2) - strl (6, 2);
                             lcd_str (1, 2); lcd_char ('-'); lcd_str (6, 2); break;                // Montag - Samstag  rechtsbndig
                     case 9: text_x = 103 - strl (6, 2) - strl (7, 2);
                             lcd_str (6, 2); lcd_char ('-'); lcd_str (7, 2); break;                // Samstag - Sonntag  rechtsbndig
                  }
               }
            }

            text_x = 3;
            text_y = 101;                                                                          // Text-Koordinaten fr Sound
            lcd_char (0x95);                                                                       // Symbol "Note" ausgeben
            text_x = 15;
            if (editmode && (submopt == 6))                                                        // wenn Sound editiert wird
            {
               text_inv = 1;                                                                       // Soundname invers ausgeben
               lcd_str (editval + 60, 0);                                                          // Editorwert ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else lcd_str (alarms[alm][8] + 60, 0);                                                 // Soundname ausgeben

            text_x = 3;
            text_y = 115;                                                                          // Text-Koordinaten fr Lautstrke
            lcd_char (0x96);                                                                       // Symbol "Lautsprecher" ausgeben
            text_x = 15;
            if (editmode && (submopt == 7))                                                        // wenn Lautstrke editiert wird
            {
               text_inv = 1;                                                                       // Lautstrke invers ausgeben
               lcd_numb (editval);                                                                 // Editorwert ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else lcd_numb (alarms[alm][9]);                                                        // sonst Lautstrke ausgeben

            text_x = 97;
            text_y = 115;                                                                          // Text-Koordinaten fr Sound-Test
            lcd_char (0x96);                                                                       // Symbol "Lautsprecher" ausgeben
            if (!snstat || (second % 2))                                                           // wenn Sound-Ausgabe aktiv oder Sekunde ungerade
               lcd_char (0x97);                                                                    // Symbol "ein" ausgeben, blinkt whrend Sound-Ausgabe

            switch (submopt)                                                                       // gewhlte Untermen-Option umrahmen
            {
               case 0: lcd_frame (2, 15, 111, 29); break;
               case 1: lcd_frame (2, 29, 111, 43); break;
               case 2: lcd_frame (2, 43, 111, 57); break;
               case 3: lcd_frame (2, 57, 111, 71); break;
               case 4: lcd_frame (2, 71, 111, 85); break;
               case 5: lcd_frame (2, 85, 111, 99); break;
               case 6: lcd_frame (2, 99, 111, 113); break;
               case 7: lcd_frame (2, 113, 29, 127); break;
               case 8: lcd_frame (96, 113, 111, 127);
            }
         }

         if (dispmode == 6)                                                                        // Men "Einstellungen 2" --------------------------------------------------------------------------
         {
            text_inv = 1;                                                                          // berschrift invertieren
            text_x = 56 - ((strl (10, 0) + 11) / 2);                                               // berschrift zentriert ausgeben
            text_y = 0;
            lcd_str (10, 0);                                                                       // berschrift "Einstellungen"
            lcd_char (' ');                                                                        // Leerzeichen ausgeben
            lcd_char ('2');                                                                        // "2" ausgeben
            lcd_char (127);                                                                        // Endezeichen ausgeben
            text_inv = 0;                                                                          // weitere Texte normal ausgeben

            text_x = 3;
            text_y = 17;                                                                           // Text-Koordinaten fr DCF-Signal-Invertierung
            lcd_str (37, 0);                                                                       // Text "DCF-Signal" ausgeben
            if (editmode && (submopt == 0))                                                        // wenn DCF-Signal-Invertierung editiert wird
            {
               text_inv = 1;                                                                       // DCF-Signal-Invertierung invers ausgeben
               text_x = 109 - strl (editval + 34, 0);                                              // Editorwert rechtsbndig ausgeben
               lcd_str (editval + 34, 0);                                                          // Editorwert als String ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else
            {
               text_x = 109 - strl (dcfinv + 34, 0);                                               // sonst DCF-Signal-Invertierung rechtsbndig ausgeben
               lcd_str (dcfinv + 34, 0);                                                           // DCF-Signal-Invertierung ausgeben
            }

            text_x = 3;
            text_y = 31;                                                                           // Text-Koordinaten fr DCF-Pull-up
            lcd_str (38, 0);                                                                       // Text "DCF-Pull-up" ausgeben
            if (editmode && (submopt == 1))                                                        // wenn DCF-Pull-up editiert wird
            {
               text_inv = 1;                                                                       // DCF-Pull-up invers ausgeben
               text_x = 109 - strl (editval + 39, 0);                                              // Editorwert rechtsbndig ausgeben
               lcd_str (editval + 39, 0);                                                          // Editorwert als String ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else
            {
               text_x = 109 - strl (dcfpup + 39, 0);                                               // sonst DCF-Pull-up rechtsbndig ausgeben
               lcd_str (dcfpup + 39, 0);                                                           // DCF-Pull-up ausgeben
            }

            text_x = 3;
            text_y = 45;                                                                           // Text-Koordinaten fr Geburtstagsnamen-Reihenfolge
            lcd_str (84, 0);                                                                       // Text "Geb-Sort" ausgeben
            if (editmode && (submopt == 2))                                                        // wenn Geburtstagslisten-Sortierung editiert wird
            {
               text_inv = 1;                                                                       // Geburtstagsnamen-Reihenfolge invers ausgeben
               text_x = 109 - strl (editval + 85, 0);                                              // Editorwert rechtsbndig ausgeben
               lcd_str (editval + 85, 0);                                                          // Editorwert als String ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else
            {
               text_x = 109 - strl (bdlsort + 85, 0);                                              // sonst Geburtstagslisten-Sortierung rechtsbndig ausgeben
               lcd_str (bdlsort + 85, 0);                                                          // Geburtstagsnamen-Reihenfolge ausgeben
            }

            text_x = 3;
            text_y = 59;                                                                           // Text-Koordinaten fr Geburtstagsnamen-Reihenfolge
            lcd_str (81, 0);                                                                       // Text "Geb-Namen" ausgeben
            if (editmode && (submopt == 3))                                                        // wenn Geburtstagsnamen-Reihenfolge editiert wird
            {
               text_inv = 1;                                                                       // Geburtstagsnamen-Reihenfolge invers ausgeben
               text_x = 109 - strl (editval + 74, 0);                                              // Editorwert rechtsbndig ausgeben
               lcd_str (editval + 74, 0);                                                          // Editorwert als String ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else
            {
               text_x = 109 - strl (bdordr + 74, 0);                                               // sonst Geburtstagsnamen-Reihenfolge rechtsbndig ausgeben
               lcd_str (bdordr + 74, 0);                                                           // Geburtstagsnamen-Reihenfolge ausgeben
            }

            text_x = 3;
            text_y = 73;                                                                           // Text-Koordinaten fr Alters-Anzeige
            lcd_str (80, 0);                                                                       // Text "Geb-Alter" ausgeben
            if (editmode && (submopt == 4))                                                        // wenn Alters-Anzeige editiert wird
            {
               text_inv = 1;                                                                       // Alters-Anzeige invers ausgeben
               text_x = 109 - strl (editval + 39, 0);                                              // Editorwert rechtsbndig ausgeben
               lcd_str (editval + 39, 0);                                                          // Editorwert als String ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else
            {
               text_x = 109 - strl (bdagem + 39, 0);                                               // sonst Alters-Anzeige rechtsbndig ausgeben
               lcd_str (bdagem + 39, 0);                                                           // Alters-Anzeige ausgeben
            }

            text_x = 3;
            text_y = 87;                                                                           // Text-Koordinaten fr Geburtstags-Alarm
            lcd_str (82, 0);                                                                       // Text "Geb-Alarm" ausgeben
            if (editmode && (submopt == 5))                                                        // wenn Geburtstags-Alarm editiert wird
            {
               text_inv = 1;                                                                       // Zeit invers ausgeben
               if (editval == -1)                                                                  // wenn Zeit = -1 (aus)
               {
                  text_x = 109 - strl (39, 0);
                  lcd_str (39, 0);                                                                 // Text "aus" rechtsbndig ausgeben
               }
               else                                                                                // wenn Zeit im gltigen Bereich
               {
                  text_x = 82;
                  lcd_time (editval / 60, editval % 60);                                           // Editorwert ausgeben
               }
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else
            {
               if (alarms[3][0] == 0)                                                              // wenn Geburtstags-Alarms aus
               {
                  text_x = 109 - strl (39, 0);
                  lcd_str (39, 0);                                                                 // Text "aus" rechtsbndig ausgeben
               }
               else                                                                                // wenn Zeit im gltigen Bereich
               {
                  text_x = 82;
                  lcd_time (alarms[3][2], alarms[3][1]);                                           // sonst Zeit ausgeben
               }
            }

            text_x = 3;
            text_y = 101;                                                                          // Text-Koordinaten fr Sound
            lcd_char (0x95);                                                                       // Symbol "Note" ausgeben
            text_x = 15;
            if (editmode && (submopt == 6))                                                        // wenn Sound editiert wird
            {
               text_inv = 1;                                                                       // Soundname invers ausgeben
               lcd_str (editval + 60, 0);                                                          // Editorwert ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else lcd_str (alarms[3][8] + 60, 0);                                                   // Soundname ausgeben

            text_x = 3;
            text_y = 115;                                                                          // Text-Koordinaten fr Lautstrke
            lcd_char (0x96);                                                                       // Symbol "Lautsprecher" ausgeben
            text_x = 15;
            if (editmode && (submopt == 7))                                                        // wenn Lautstrke editiert wird
            {
               text_inv = 1;                                                                       // Lautstrke invers ausgeben
               lcd_numb (editval);                                                                 // Editorwert ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else lcd_numb (alarms[3][9]);                                                          // sonst Lautstrke ausgeben

            text_x = 97;
            text_y = 115;                                                                          // Text-Koordinaten fr Sound-Test
            lcd_char (0x96);                                                                       // Symbol "Lautsprecher" ausgeben
            if (!snstat || (second % 2))                                                           // wenn Sound-Ausgabe aktiv oder Sekunde ungerade
               lcd_char (0x97);                                                                    // Symbol "ein" ausgeben, blinkt whrend Sound-Ausgabe

            switch (submopt)                                                                       // gewhlte Untermen-Option umrahmen
            {
               case 0: lcd_frame (2, 15, 111, 29); break;
               case 1: lcd_frame (2, 29, 111, 43); break;
               case 2: lcd_frame (2, 43, 111, 57); break;
               case 3: lcd_frame (2, 57, 111, 71); break;
               case 4: lcd_frame (2, 71, 111, 85); break;
               case 5: lcd_frame (2, 85, 111, 99); break;
               case 6: lcd_frame (2, 99, 111, 113); break;
               case 7: lcd_frame (2, 113, 29, 127); break;
               case 8: lcd_frame (96, 113, 111, 127);
            }
         }

         if (dispmode == 7)                                                                        // Men "Geburtstage 1" (8 Eintrge) ---------------------------------------------------------------
         {
            text_inv = 1;                                                                          // berschrift invertieren
            text_x = 56 - ((strl (8, 0) + 32) / 2);                                                // berschrift zentriert ausgeben
            text_y = 0;
            lcd_str (8, 0);                                                                        // berschrift "Geburtstage"
            lcd_char (' ');                                                                        // Leerzeichen ausgeben
            lcd_char ('(');                                                                        // Klammerzeichen ausgeben
            lcd_numb3 (bdipos + 1);                                                                // Indexlisten-Position ausgeben
            lcd_char (')');                                                                        // Klammerzeichen ausgeben
            lcd_char (127);                                                                        // Endezeichen ausgeben
            text_inv = 0;                                                                          // weitere Texte normal ausgeben

            for (i = 0; i < 8; i ++)                                                               // 8 Geburtstage anzeigen
            {
               bdindx = bdilist[i + bdipos - bd1pos];                                              // Wert aus dem Geburtstagsindex lesen
               if (bdindx < 115)                                                                   // wenn gltiger Eintrag vorhanden ist
               {
                  text_y = i * 14 + 17;                                                            // Text-Zeilen-Position fr Geburtstag festlegen
                  lcd_bdname ();                                                                   // Geburtstagsdaten ausgeben
               }
               else                                                                                // wenn kein gltiger Eintrag vorhanden ist
               {
                  text_x = 56 - (strl (92, 0) / 2);
                  text_y = i * 14 + 17;                                                            // Text-Position fr "leer" festlegen
                  lcd_str (92, 0);                                                                 // Text "leer" ausgeben
               }
            }

            switch (bd1pos)                                                                        // gewhlte Geburtstagszeile umrahmen
            {
               case 0: lcd_frame (2, 15, 111, 29); break;
               case 1: lcd_frame (2, 29, 111, 43); break;
               case 2: lcd_frame (2, 43, 111, 57); break;
               case 3: lcd_frame (2, 57, 111, 71); break;
               case 4: lcd_frame (2, 71, 111, 85); break;
               case 5: lcd_frame (2, 85, 111, 99); break;
               case 6: lcd_frame (2, 99, 111, 113); break;
               case 7: lcd_frame (2, 113, 111, 127);
            }
         }

         if (dispmode == 8)                                                                        // Men "Geburtstage 2" (4 Eintrge) ---------------------------------------------------------------
         {
            text_inv = 1;                                                                          // berschrift invertieren
            text_x = 56 - ((strl (8, 0) + 32) / 2);                                                // berschrift zentriert ausgeben
            text_y = 0;
            lcd_str (8, 0);                                                                        // berschrift "Geburtstage"
            lcd_char (' ');                                                                        // Leerzeichen ausgeben
            lcd_char ('(');                                                                        // Klammerzeichen ausgeben
            lcd_numb3 (bdipos + 1);                                                                // Indexlisten-Position ausgeben
            lcd_char (')');                                                                        // Klammerzeichen ausgeben
            lcd_char (127);                                                                        // Endezeichen ausgeben
            text_inv = 0;                                                                          // weitere Texte normal ausgeben

            for (i = 0; i < 4; i ++)                                                               // 4 Geburtstage anzeigen
            {
               bdindx = bdilist[i + bdipos - bd2pos];                                              // Wert aus dem Geburtstagsindex lesen
               if (bdindx < 115)                                                                   // wenn gltiger Eintrag vorhanden ist
               {
                  xlimit = 109;
                  text_y = i * 28 + 17;                                                            // Text-Position fr Geburtstag
                  if (eep_read_byte (e_bddata + (bdindx * 34 + 2)) < 100)                          // wenn gltiges Geburtsjahr vorhanden
                  {
                     xlimit = lcd_age ();                                                          // Alter ausgeben und X-Endposition fr Namen ermitteln
                  }
                  text_x = 4;
                  lcd_numbs (eep_read_byte (e_bddata + (bdindx * 34)));                            // Geburtstag holen und ausgeben
                  text_x = 4;
                  text_y = i * 28 + 23;                                                            // Text-Position fr Geburtsmonat
                  lcd_numbs (eep_read_byte (e_bddata + (bdindx * 34 + 1)));                        // Geburtsmonat holen und ausgeben
                  text_x = 17;
                  text_y = i * 28 + 17;                                                            // Text-Position fr Vorname
                  for (j = 4; j < 19; j ++)                                                        // Vorname ausgeben (max. 15 Zeichen)
                  {
                     bddata = eep_read_byte (e_bddata + (bdindx * 34 + j));                        // ein Zeichen vom Vornamen holen
                     if ((bddata == 255) || ((chrl(bddata) + text_x) > xlimit)) break;             // Textende oder Bildrand -> Abbruch
                     lcd_char (bddata);                                                            // Zeichen ausgeben
                  }

                  text_x = 4;
                  text_y = i * 28 + 31;                                                            // Text-Position fr Geburtsjahrhundert
                  if (eep_read_byte (e_bddata + (bdindx * 34 + 2)) < 100)                          // wenn gltiges Geburtsjahr vorhanden
                  {
                     lcd_numbs (eep_read_byte (e_bddata + (bdindx * 34 + 3)));                     // Geburtsjahrhundert holen und ausgeben
                     text_x = 4;
                     text_y = i * 28 + 37;                                                         // Text-Position fr Geburtsjahr
                     lcd_numbs (eep_read_byte (e_bddata + (bdindx * 34 + 2)));                     // Geburtsjahr holen und ausgeben
                  }
                  else                                                                             // wenn kein gltiges Geburtsjahr vorhanden
                  {
                     text_x = 5;                                                                   // Textposition fr Fragezeichen
                     lcd_char ('?');                                                               // Fragezeichen ausgeben
                  }

                  text_x = 17;
                  text_y = i * 28 + 31;                                                            // Text-Position fr Nachname
                  for (j = 19; j < 34; j ++)                                                       // Nachname ausgeben (max. 15 Zeichen)
                  {
                     bddata = eep_read_byte (e_bddata + (bdindx * 34 + j));                        // ein Zeichen vom Nachnamen holen
                     if ((bddata == 255) || ((chrl(bddata) + text_x) > 109)) break;                // Textende oder Bildrand -> Abbruch
                     lcd_char (bddata);                                                            // Zeichen ausgeben
                  }
               }
               else                                                                                // wenn kein gltiger Eintrag vorhanden ist
               {
                  text_x = 56 - (strl (92, 0) / 2);
                  text_y = i * 28 + 24;                                                            // Text-Position fr "leer"
                  lcd_str (92, 0);                                                                 // Text "leer" ausgeben
               }
            }

            switch (bd2pos)                                                                        // gewhlte Geburtstagszeile umrahmen
            {
               case 0: lcd_frame (2, 15, 111, 43); break;
               case 1: lcd_frame (2, 43, 111, 71); break;
               case 2: lcd_frame (2, 71, 111, 99); break;
               case 3: lcd_frame (2, 99, 111, 127);
            }
         }

         if (dispmode == 9)                                                                        // Men "Geburtstag eingeben" ----------------------------------------------------------------------
         {
            text_inv = 1;                                                                          // berschrift invertieren
            text_x = 56 - (strl (91, 0) / 2);                                                      // berschrift zentriert ausgeben
            text_y = 0;
            lcd_str (91, 0);                                                                       // berschrift "Geburtstag eingeben"
            lcd_char (127);                                                                        // Endezeichen ausgeben
            text_inv = 0;                                                                          // weitere Texte normal ausgeben

            text_x = 3;
            text_y = 31;                                                                           // Text-Koordinaten fr Geburtstag
            lcd_str (13, 0);                                                                       // Text "Tag" ausgeben
            if (editmode && (submopt == 0))                                                        // wenn Geburtstag editiert wird
            {
               text_inv = 1;                                                                       // Geburtstag invers ausgeben
               if (editval == 0)                                                                   // wenn Editorwert = 0
               {
                  text_x = 109 - strl (76, 0);                                                     // Text rechtsbndig ausgeben
                  lcd_str (76, 0);                                                                 // Text "Lschen" ausgeben
               }
               else                                                                                // wenn Editorwert > 0
               {
                  text_x = 97;
                  lcd_numb (editval);                                                              // Editorwert ausgeben
               }
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else                                                                                   // wenn nicht editiert wird
            {
               text_x = 97;
               lcd_numb (bdeday);                                                                  // Geburtstag ausgeben
            }

            text_x = 3;
            text_y = 45;                                                                           // Text-Koordinaten fr Geburtsmonat
            lcd_str (14, 0);                                                                       // Text "Monat" ausgeben
            text_x = 50;
            if (editmode && (submopt == 1))                                                        // wenn Monat editiert wird
            {
               text_inv = 1;                                                                       // Monat invers ausgeben
               text_x = 109 - strl (editval + 15, 0);                                              // Editorwert rechtsbndig ausgeben
               lcd_str (editval + 15, 0);                                                          // Editorwert als Monatsname ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else                                                                                   // wenn nicht editiert wird
            {
               text_x = 109 - strl (bdemonth + 15, 0);                                             // sonst Monatsname rechtsbndig ausgeben
               lcd_str (bdemonth + 15, 0);                                                         // Monatsname ausgeben
            }

            text_x = 3;
            text_y = 59;                                                                           // Text-Koordinaten fr Jahr
            lcd_str (15, 0);                                                                       // Text "Jahr" ausgeben
            if (editmode && (submopt == 2))                                                        // wenn Jahr (4-stellig) editiert wird
            {
               text_inv = 1;                                                                       // Jahr invers ausgeben
               if (editval == -1)                                                                  // wenn Editorwert = -1
               {
                  text_x = 109 - strl (83, 0);                                                     // Text rechtsbndig ausgeben
                  lcd_str (83, 0);                                                                 // Text "Unbekannt" ausgeben
               }
               else                                                                                // wenn Editorwert >= 0
               {
                  text_x = 85;
                  lcd_numb4 (editval);                                                             // Editorwert ausgeben
               }
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else                                                                                   // wenn nicht editiert wird
            {
               if (bdeyear > 99)                                                                   // wenn ungltiges Geburtsjahr
               {
                  text_x = 109 - strl (83, 0);                                                     // Text rechtsbndig ausgeben
                  lcd_str (83, 0);                                                                 // Text "Unbekannt" ausgeben
               }
               else                                                                                // wenn gltiges Geburtsjahr
               {
                  text_x = 85;
                  lcd_numb4 (bdeyearh * 100 + bdeyear);                                            // Jahr ausgeben
               }
            }

            text_x = 3;
            text_y = 73;                                                                           // Text-Koordinaten fr Vorname
            lcd_str (93, 0);                                                                       // Text "Vorname" ausgeben
            text_x = 3;
            text_y = 87;                                                                           // Text-Koordinaten fr Vorname
            if (editmode && (submopt == 3))                                                        // wenn Vorname editiert wird
            {
               for (i = 0; i < bdtxpos; i ++)                                                      // Name bis zur Cursorposition normal ausgeben
               {
                  lcd_char (bdtxbuf[i]);                                                           // Zeichen ausgeben
               }
               if (bdtxpos == 0) text_x ++;                                                        // wenn Puffer noch leer -> erstes Zeichen um ein Pixel versch.
               text_inv = 1;                                                                       // weitere Zeichen invers ausgeben
               if (editval < sizeof (charlist))                                                    // wenn gltiges Zeichen
               {
                  lcd_char (pgm_read_byte (&charlist[editval]));                                   // Zeichen an Cursorposition ausgeben
               }
               if (editval == sizeof (charlist))                                                   // wenn Lschzeichen
               {
                  lcd_char (0x98);                                                                 // Lschzeichen ausgeben
               }
               if (editval == sizeof (charlist) + 1)                                               // wenn OK-Zeichen
               {
                  lcd_char (0x99);                                                                 // OK-Zeichen ausgeben
               }
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else                                                                                   // sonst Vorname normal ausgeben
            {
               text_x = 109;                                                                       // rechtsbndige Ausgabe, letzte Pixelposition
               for (i = 0; i < 15; i ++)                                                           // 15 Zeichen bearbeiten
               {
                  if (bde1buf[i] == 255) break;                                                    // wenn ungltiges Zeichen -> Abbruch
                  text_x = text_x - chrl (bde1buf[i]) - 1;                                         // sonst Position entsprechend Zeichenbreite verschieben
               }
               for (i = 0; i < 15; i ++)                                                           // 15 Zeichen ausgeben
               {
                  if (bde1buf[i] == 255) break;                                                    // wenn ungltiges Zeichen -> Ausgabe abbrechen
                  lcd_char (bde1buf[i]);                                                           // sonst Zeichen ausgeben
               }
            }

            text_x = 3;
            text_y = 101;                                                                          // Text-Koordinaten fr Nachname
            lcd_str (73, 0);                                                                       // Text "Nachname" ausgeben
            text_x = 3;
            text_y = 115;                                                                          // Text-Koordinaten fr Nachname
            if (editmode && (submopt == 4))                                                        // wenn Nachname editiert wird
            {
               for (i = 0; i < bdtxpos; i ++)                                                      // Name bis zur Cursorposition normal ausgeben
               {
                  lcd_char (bdtxbuf[i]);                                                           // Zeichen ausgeben
               }
               if (bdtxpos == 0) text_x ++;                                                        // wenn Puffer noch leer -> erstes Zeichen um ein Pixel verschieben
               text_inv = 1;                                                                       // weitere Zeichen invers ausgeben
               if (editval < sizeof (charlist))                                                    // wenn gltiges Zeichen
               {
                  lcd_char (pgm_read_byte (&charlist[editval]));                                   // Zeichen an Cursorposition ausgeben
               }
               if (editval == sizeof (charlist))                                                   // wenn Lschzeichen
               {
                  lcd_char (0x98);                                                                 // Lschzeichen ausgeben
               }
               if (editval == sizeof (charlist) + 1)                                               // wenn OK-Zeichen
               {
                  lcd_char (0x99);                                                                 // OK-Zeichen ausgeben
               }
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else                                                                                   // sonst Nachname normal ausgeben
            {
               text_x = 109;                                                                       // rechtsbndige Ausgabe, letzte Pixelposition
               for (i = 0; i < 15; i ++)                                                           // 15 Zeichen bearbeiten
               {
                  if (bde2buf[i] == 255) break;                                                    // wenn ungltiges Zeichen -> Ausgabe abbrechen
                  text_x = text_x - chrl (bde2buf[i]) - 1;                                         // sonst Position entsprechend Zeichenbreite verschieben
               }
               for (i = 0; i < 15; i ++)                                                           // 15 Zeichen ausgeben
               {
                  if (bde2buf[i] == 255) break;                                                    // wenn ungltiges Zeichen -> Ausgabe abbrechen
                  lcd_char (bde2buf[i]);                                                           // sonst Zeichen ausgeben
               }
            }

            switch (submopt)                                                                       // gewhlte Untermen-Option umrahmen
            {
               case 0: lcd_frame (2, 29, 111, 43); break;
               case 1: lcd_frame (2, 43, 111, 57); break;
               case 2: lcd_frame (2, 57, 111, 71); break;
               case 3: lcd_frame (2, 71, 111, 99); break;
               case 4: lcd_frame (2, 99, 111, 127);
            }
         }

         if (dispmode == 10)                                                                       // Men "Einstellungen 3" --------------------------------------------------------------------------
         {
            text_inv = 1;                                                                          // berschrift invertieren
            text_x = 56 - ((strl (10, 0) + 11) / 2);                                               // berschrift zentriert ausgeben
            text_y = 0;
            lcd_str (10, 0);                                                                       // berschrift "Einstellungen"
            lcd_char (' ');                                                                        // Leerzeichen ausgeben
            lcd_char ('3');                                                                        // "3" ausgeben
            lcd_char (127);                                                                        // Endezeichen ausgeben
            text_inv = 0;                                                                          // weitere Texte normal ausgeben

            text_x = 3;
            text_y = 17;                                                                           // Text-Koordinaten fr Sound-Freigabe
            lcd_str (36, 0);                                                                       // Text "Sound" ausgeben
            if (editmode && (submopt == 0))                                                        // wenn Sound-Freigabe editiert wird
            {
               text_inv = 1;                                                                       // Sound-Freigabe invers ausgeben
               text_x = 109 - strl (editval + 39, 0);                                              // Editorwert rechtsbndig ausgeben
               lcd_str (editval + 39, 0);                                                          // Editorwert als String ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else
            {
               text_x = 109 - strl (s_enable + 39, 0);                                             // sonst Sound-Freigabe rechtsbndig ausgeben
               lcd_str (s_enable + 39, 0);                                                         // Sound-Freigabe ausgeben
            }

            text_x = 3;
            text_y = 31;                                                                           // Text-Koordinaten fr Gong ein/aus
            lcd_str (88, 0);                                                                       // Text "Gong" ausgeben
            if (editmode && (submopt == 1))                                                        // wenn Gong ein/aus editiert wird
            {
               text_inv = 1;                                                                       // Gong invers ausgeben
               text_x = 109 - strl (editval + 39, 0);                                              // Editorwert rechtsbndig ausgeben
               lcd_str (editval + 39, 0);                                                          // Editorwert als String ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else
            {
               text_x = 109 - strl (gong + 39, 0);                                                 // sonst Gong-Einstellung rechtsbndig ausgeben
               lcd_str (gong + 39, 0);                                                             // Gong-Einstellung ausgeben
            }

            text_x = 3;
            text_y = 45;                                                                           // Text-Koordinaten fr Gong-Beginn
            lcd_str (89, 0);                                                                       // Text "Gong von" ausgeben
            text_x = 82;
            if (editmode && (submopt == 2))                                                        // wenn Gong-Beginn editiert wird
            {
               text_inv = 1;                                                                       // Gong-Beginn invers ausgeben
                text_x --;                                                                         // X-Position korrigieren
               lcd_numb (editval);                                                                 // Editorwert ausgeben (Stunden)
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            lcd_char (':');                                                                        // Doppelpunkt ausgeben
            lcd_numb (0);                                                                          // 0 Minuten ausgeben
            }
            else lcd_time (gongh1, 0);                                                             // sonst Gong-Beginnzeit ausgeben

            text_x = 3;
            text_y = 59;                                                                           // Text-Koordinaten fr Gong-Ende
            lcd_str (90, 0);                                                                       // Text "Gong bis" ausgeben
            text_x = 82;
            if (editmode && (submopt == 3))                                                        // wenn Gong-Ende editiert wird
            {
               text_inv = 1;                                                                       // Gong-Ende invers ausgeben
                text_x --;                                                                         // X-Position korrigieren
               lcd_numb (editval);                                                                 // Editorwert ausgeben (Stunden)
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            lcd_char (':');                                                                        // Doppelpunkt ausgeben
            lcd_numb (0);                                                                          // 0 Minuten ausgeben
            }
            else lcd_time (gongh2, 0);                                                             // sonst Gong-Endezeit ausgeben

            text_x = 3;
            text_y = 73;                                                                           // Text-Koordinaten fr Gong-Sound
            lcd_char (0x95);                                                                       // Symbol "Note" ausgeben
            text_x = 15;
            if (editmode && (submopt == 4))                                                        // wenn Gong-Sound editiert wird
            {
               text_inv = 1;                                                                       // Gong-Soundname invers ausgeben
               lcd_str (editval + 60, 0);                                                          // Editorwert ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else lcd_str (gong_snd + 60, 0);                                                       // Gong-Soundname ausgeben

            text_x = 3;
            text_y = 87;                                                                           // Text-Koordinaten fr Gong-Lautstrke
            lcd_char (0x96);                                                                       // Symbol "Lautsprecher" ausgeben
            text_x = 15;
            if (editmode && (submopt == 5))                                                        // wenn GongLautstrke editiert wird
            {
               text_inv = 1;                                                                       // Gong-Lautstrke invers ausgeben
               lcd_numb (editval);                                                                 // Editorwert ausgeben
               lcd_char (127);                                                                     // Ende-Zeichen ausgeben
               text_inv = 0;                                                                       // weitere Texte normal anzeigen
            }
            else lcd_numb (gong_vol);                                                              // sonst Gong-Lautstrke ausgeben

            text_x = 97;
            text_y = 87;                                                                           // Text-Koordinaten fr Sound-Test
            lcd_char (0x96);                                                                       // Symbol "Lautsprecher" ausgeben
            if (!snstat || (second % 2))                                                           // wenn Sound-Ausgabe aktiv oder Sekunde ungerade
            lcd_char (0x97);                                                                       // Symbol "ein" ausgeben, blinkt whrend Sound-Ausgabe

            switch (submopt)                                                                       // gewhlte Untermen-Option umrahmen
            {
               case 0: lcd_frame (2, 15, 111, 29); break;
               case 1: lcd_frame (2, 29, 111, 43); break;
               case 2: lcd_frame (2, 43, 111, 57); break;
               case 3: lcd_frame (2, 57, 111, 71); break;
               case 4: lcd_frame (2, 71, 111, 85); break;
               case 5: lcd_frame (2, 85, 29, 99); break;
               case 6: lcd_frame (96, 85, 111, 99);
            }
         }

         copy_infodata ();                                                                         // kompletten Info-Bereich in den LCD-Speicher kopieren

         if (scrshot == 2)                                                                         // wenn ein Screenshot (Info-Bereich) angefordert wird
         {
            for (k = 0; k < 2048; k ++)                                                            // 2048 Bytes senden (128 x 16 Bytes)
            {
               if ((k % 16) == 0)                                                                  // wenn Zeilenanfang
               {
                  usb_char ('s', cmdport);                                                         // Antwort 's' senden
                  usb_char (':', cmdport);                                                         // Trennzeichen senden
               }
               usb_byte (lcd_bitmap[k], cmdport);                                                  // Datenbyte senden
               if ((k % 16) == 15)                                                                 // wenn Zeilenende
               {
                  usb_char ('\r', cmdport);                                                        // Carriage Return senden
                  usb_char ('\n', cmdport);                                                        // Line Feed senden
               }               
               else usb_char (',', cmdport);                                                       // sonst Komma senden
            }
            scrshot = 0;                                                                           // Screenshot-Steuerung wieder abschalten
            cmdport = 0;                                                                           // Kommando-Bearbeitung beendet
         }         
      }

// Taster 1 abfragen und Funktionen steuern -----------------------------------------------------------------------------------------------------------------------------------------------------------

      cli ();                                                                                      // Interrupts sperren
      if (key1stat)                                                                                // wenn Taster 1 gedrckt
      {
         if (!key1quit)                                                                            // wenn Tastendruck noch nicht quittiert
         {
            key1quit = 1;                                                                          // Quittierung setzen
            sei ();                                                                                // Interrupt wieder freigeben
            chg_flag = 1;                                                                          // nderungs-Flag setzen, Informationen neu ausgeben
            disptout = c_disptout;                                                                 // Anzeige-Timeout neu starten
            if (lcd_is_on)                                                                         // wenn LCD eingeschaltet
            {
               if ((alarms[0][10] == 1) || (alarms[1][10] == 1) || (alarms[2][10] == 1) || (alarms[3][10] == 1)) // wenn mindestens ein Alarm ausgelst wurde
               {
                  if (alarms[0][10] == 1) alarms[0][10] = 2;                                       // Alarm 1 quittieren
                  if (alarms[1][10] == 1) alarms[1][10] = 2;                                       // Alarm 2 quittieren
                  if (alarms[2][10] == 1) alarms[2][10] = 2;                                       // Alarm 3 quittieren
                  if (alarms[3][10] == 1) alarms[3][10] = 2;                                       // Geburtstags-Alarm quittieren
                  cli ();                                                                          // Interrupts sperren
                  snstat = 3;                                                                      // Sound-Erzeugung stoppen
                  sei ();                                                                          // Interrupts wieder freigeben
               }
               else                                                                                // wenn Alarm-frei
               {
                  switch (dispmode)                                                                // weitere Aktionen entsprechend Anzeigemodus
                  {
                     case 0:                                                                       // falls Anzeigemodus 0 (normale Anzeige)
                        dispmode = 1;                                                              // Anzeigemodus auf 1 setzen (Hauptmen)
                     break;

                     case 1:                                                                       // falls Anzeigemodus 1 (Hauptmen)
                        switch (mainopt)                                                           // weitere Aktionen ensprechend Men-Option
                        {
                           case 0:                                                                 // wenn Men-Option 0 (Uhrzeit) -> Uhr stellen
                              submopt = 0;                                                         // Untermen-Option 0 setzen
                              dispmode = 2;                                                        // Anzeigemodus auf 2 setzen (Uhr stellen)
                              m_minute = minute;                                                   // Minute in Zwischenspeicher legen
                              m_hour = hour;                                                       // Stunde in Zwischenspeicher legen
                              m_day = day;                                                         // Kalendertag in Zwischenspeicher legen
                              m_month = month;                                                     // Monat in Zwischenspeicher legen
                              m_year = year;                                                       // Jahr in Zwischenspeicher legen
                              break;

                           case 1:                                                                 // falls Men-Option 1 (Alarm 1) -> Alarm 1 einstellen
                              submopt = 0;                                                         // Untermen-Option 0 setzen
                              dispmode = 5;                                                        // Anzeigemodus auf 5 setzen (Alarme einstellen)
                              alm = 0;                                                             // Alarm 1 wird editiert
                              break;

                           case 2:                                                                 // falls Men-Option 2 (Alarm 2) -> Alarm 2 einstellen
                              submopt = 0;                                                         // Untermen-Option 0 setzen
                              dispmode = 5;                                                        // Anzeigemodus auf 5 setzen (Alarme einstellen)
                              alm = 1;                                                             // Alarm 2 wird editiert
                              break;

                           case 3:                                                                 // falls Men-Option 3 (Alarm 3) -> Alarm 3 einstellen
                              submopt = 0;                                                         // Untermen-Option 0 setzen
                              dispmode = 5;                                                        // Anzeigemodus auf 5 setzen (Alarme einstellen)
                              alm = 2;                                                             // Alarm 3 wird editiert
                              break;

                           case 4:                                                                 // falls Men-Option 4 (Geburtstage)
                              dispmode = 7;                                                        // Anzeigemodus auf 7 setzen (Geburtstage 1)
                              break;

                           case 5:                                                                 // falls Men-Option 5 (Informationen)
                              submopt = 0;                                                         // Untermen-Option 0 setzen
                              dispmode = 4;                                                        // Anzeigemodus auf 4 setzen (Informationen)
                              break;

                           case 6:                                                                 // falls Men-Option 6 (Einstellungen 1)
                              submopt = 0;                                                         // Untermen-Option 0 setzen
                              dispmode = 3;                                                        // Anzeigemodus auf 3 setzen (Einstellungen 1)
                              break;

                           case 7:                                                                 // falls Men-Option 7 (Einstellungen 2)
                              submopt = 0;                                                         // Untermen-Option 0 setzen
                              dispmode = 6;                                                        // Anzeigemodus auf 6 setzen (Einstellungen 2)
                              break;

                           case 8:                                                                 // falls Men-Option 8 (Einstellungen 3)
                              submopt = 0;                                                         // Untermen-Option 0 setzen
                              dispmode = 10;                                                       // Anzeigemodus auf 10 setzen (Einstellungen 3)
                              break;
                        }
                        break;

                     case 2:                                                                       // falls Anzeigemodus 2 (Uhr stellen)
                        switch (submopt)                                                           // weitere Aktionen entsprechend Untermen-Option
                        {
                           case 0:                                                                 // falls Zeit ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 m_hour = editval / 60;                                            // Stundenwert aus Editorwert holen
                                 m_minute = editval % 60;                                          // Minutenwert aus Editorwert holen
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = m_hour * 60 + m_minute;                                 // Zeit als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 1439;                                                   // Editor-Maximalwert auf 24*60-1 (23:59) setzen
                              }
                              break;

                           case 1:                                                                 // falls Tag ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 m_day = editval;                                                  // Tag aus Editor kopieren
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = m_day;                                                  // Tag als Editorwert verwenden
                                 editmin = 1;                                                      // Editor-Minimalwert auf 1 setzen
                                 editmax = 31;                                                     // Editor-Maximalwert auf 31 setzen
                              }
                              break;

                           case 2:                                                                 // falls Monat ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 m_month = editval;                                                // Monat aus Editor kopieren
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = m_month;                                                // Monat als Editorwert verwenden
                                 editmin = 1;                                                      // Editor-Minimalwert auf 1 setzen
                                 editmax = 12;                                                     // Editor-Maximalwert auf 12 setzen
                              }
                              break;

                           case 3:                                                                 // falls Jahr ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 m_year = editval;                                                 // Jahr aus Editor kopieren
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = m_year;                                                 // Jahr als Editorwert verwenden
                                 editmin = c_year;                                                 // Editor-Minimalwert auf frhestes Jahr setzen
                                 editmax = 99;                                                     // Editor-Maximalwert auf 99 setzen
                              }
                              break;

                           case 4:                                                                 // falls "Zeit bernehmen" ausgewhlt
                              okflag = 1;                                                          // Plausibilitt zunchst auf ok setzen
                              if ((m_year % 4) == 0)                                               // wenn Schaltjahr
                              {
                                 if (m_day > pgm_read_byte (&daytab2[m_month - 1])) okflag = 0;    // wenn Tag grer als
                              }                                                                    // Schaltjahr-Tabelle -> Datum nicht plausibel
                              else                                                                 // wenn kein Schaltjahr
                              {
                                 if (m_day > pgm_read_byte (&daytab1[m_month - 1])) okflag = 0;    // wenn Tag grer als
                              }                                                                    // Normaljahr-Tabelle -> Datum nicht plausibel

                              if (okflag)                                                          // wenn Monat und Tag plausibel
                              {
                                 dispmode = 0;                                                     // Anzeigemodus auf 0 setzen (normale Anzeige)
                                 sec_flag = 1;                                                     // Sekunden-Flag setzen, Analoguhr neu ausgeben
                                 cli ();                                                           // Interrupts sperren
                                 intc20ms = 0;                                                     // Interrupt-Zhler 3 (20ms) zurcksetzen
                                 second = 0;                                                       // Sekunden zurcksetzen
                                 minute = m_minute;                                                // Minute aus Zwischenspeicher holen
                                 hour = m_hour;                                                    // Stunde aus Zwischenspeicher holen
                                 day = m_day;                                                      // Kalendertag aus Zwischenspeicher holen
                                 month = m_month;                                                  // Monat aus Zwischenspeicher holen
                                 year = m_year;                                                    // Jahr aus Zwischenspeicher holen
                                 timezone = 0;                                                     // Zeitzone ist bei manueller Eingabe immer 0
                                 syncstat = 2;                                                     // Uhr ist jetzt manuell synchronisiert
                                 sei ();                                                           // Interrupts wieder freigeben
                                 wday = calc_wday (day, month, c_yearh, year);                     // Wochentag berechnen
                              }
                              else                                                                 // wenn Monat und Tag nicht plausibel
                              {
                                 submopt = 1;                                                      // Untermen auf Tag-Einstellung setzen
                                 editmode = 1;                                                     // Editormodus aktivieren (Tag neu abfragen)
                                 if ((m_year % 4) == 0)                                            // wenn Schaltjahr
                                 {
                                    editval = pgm_read_byte (&daytab2[m_month - 1]);               // Editorwert auf letzten Tag des
                                 }                                                                 // Monats setzen
                                 else                                                              // wenn kein Schaltjahr
                                 {
                                    editval = pgm_read_byte (&daytab1[m_month - 1]);               // Editorwert auf letzten Tag des
                                 }                                                                 // Monats setzen
                                 editmin = 1;                                                      // Editor-Minimalwert auf 1 setzen
                                 editmax = 31;                                                     // Editor-Maximalwert auf 31 setzen
                              }
                              break;

                           case 5:                                                                 // falls "DCF77-Sync starten" ausgewhlt
                              dispmode = 0;                                                        // Anzeigemodus auf 0 setzen (normale Anzeige)
                              if (syncstat > 2) syncstat = 2;                                      // gegebenenfalls Sync-Status herabsetzen (wegen LED)
                              if (syncstat == 0) syncstat = 1;                                     // oder erhhen (kein neuer Sync-Versuch nach Abbruch)
                              lcd_off ();                                                          // LCD ausschalten
                              syntout = c_syntout;                                                 // Synchronisierungs-Timeout setzen
                              break;
                        }
                        break;

                     case 3:                                                                       // falls Anzeigemodus 3 (Einstellungen 1)
                        switch (submopt)                                                           // weitere Aktionen entsprechend Untermen-Option
                        {
                           case 0:                                                                 // falls Sprache ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 lang = editval;                                                   // Sprache aus Editorwert holen
                                 eep_write_byte (e_lang, lang);                                    // Einstellung im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = lang;                                                   // Sprache als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 1;                                                      // Editor-Maximalwert auf 1 setzen
                              }
                              break;

                           case 1:                                                                 // falls Analog-Uhr-Position ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 aclkpos = editval;                                                // Analog-Uhr-Position aus Editorwert holen
                                 eep_write_byte (e_aclkpos, aclkpos);                              // Einstellung im EEPROM speichern
                                 sec_flag = 1;                                                     // Sekunden-Flag setzen, Analoguhr neu ausgeben
                              }
                              else
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = aclkpos;                                                // Analog-Uhr-Position als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 1;                                                      // Editor-Maximalwert auf 1 setzen
                              }
                              break;

                           case 2:                                                                 // falls Display-Invertierung ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 lcdinv = editval;                                                 // LCD-Invertierung aus Editor kopieren
                                 eep_write_byte (e_lcdinv, lcdinv);                                // Einstellung im EEPROM speichern
                                 sec_flag = 1;                                                     // Sekunden-Flag setzen, Analoguhr neu ausgeben
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = lcdinv;                                                 // LCD-Invertierung als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 1;                                                      // Editor-Maximalwert auf 1 setzen
                              }
                              break;

                           case 3:                                                                 // falls DCF-Nacht-Synchronisierung ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 nightsyn = editval;                                               // DCF-Nacht-Synchronisierung aus Editor kopieren
                                 eep_write_byte (e_nightsyn, nightsyn);                            // Einstellung im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = nightsyn;                                               // DCF-Nacht-Synchronisierung als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 1;                                                      // Editor-Maximalwert auf 1 setzen
                              }
                              break;

                           case 4:                                                                 // falls Nacht-Modus ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 lcdnight = editval;                                               // Nacht-Modus aus Editor kopieren
                                 eep_write_byte (e_lcdnight, lcdnight);                            // Einstellung im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = lcdnight;                                               // Nacht-Modus als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 1;                                                      // Editor-Maximalwert auf 1 setzen
                              }
                              break;

                           case 5:                                                                 // falls Nacht-Helligkeit ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 lcdnigbr = editval;                                               // LCD-Nacht-Helligkeit aus Editor kopieren
                                 eep_write_byte (e_lcdnigbr, lcdnigbr);                            // Einstellung im EEPROM speichern
                                 nightout = c_nightout;                                            // Timeout fr kurzzeitige Nacht-Helligkeit setzen
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = lcdnigbr;                                               // LCD-Nacht-Helligkeit als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 255;                                                    // Editor-Maximalwert auf 255 setzen
                              }
                              break;

                           case 6:                                                                 // falls Nacht-Modus-Beginnzeit ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 lcdnigm1 = editval % 60;                                          // Minutenwert aus Editorwert holen
                                 lcdnigh1 = editval / 60;                                          // Stundenwert aus Editorwert holen
                                 eep_write_byte (e_lcdnigm1, lcdnigm1);                            // Minutenwert im EEPROM speichern
                                 eep_write_byte (e_lcdnigh1, lcdnigh1);                            // Stundenwert im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = lcdnigh1 * 60 + lcdnigm1;                               // Nacht-Modus-Beginnzeit als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 1439;                                                   // Editor-Maximalwert auf 24*60-1 (23:59) setzen
                              }
                              break;

                           case 7:                                                                 // falls Nacht-Modus-Endezeit ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 lcdnigm2 = editval % 60;                                          // Minutenwert aus Editorwert holen
                                 lcdnigh2 = editval / 60;                                          // Stundenwert aus Editorwert holen
                                 eep_write_byte (e_lcdnigm2, lcdnigm2);                            // Minutenwert im EEPROM speichern
                                 eep_write_byte (e_lcdnigh2, lcdnigh2);                            // Stundenwert im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = lcdnigh2 * 60 + lcdnigm2;                               // Nacht-Modus-Endezeit als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 1439;                                                   // Editor-Maximalwert auf 24*60-1 (23:59) setzen
                              }
                              break;
                        }
                        break;

                     case 4:                                                                       // falls Anzeigemodus 4 (Information)
                        dispmode = 1;                                                              // Anzeigemodus auf 1 setzen (Hauptmen)
                     break;

                     case 5:                                                                       // falls Anzeigemodus 5 (Alarme einstellen)
                        switch (submopt)                                                           // weitere Aktionen entsprechend Untermen-Option
                        {
                           case 0:                                                                 // falls Alarm-Modus ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 alm_flag = 1;
                                 if (alarms[alm][0] != editval)                                    // wenn Alarm-Modus gendert wurde
                                 {
                                    alarms[alm][3] = day;                                          // aktuellen Tag voreinstellen
                                    alarms[alm][5] = month;                                        // aktuellen Monat voreinstellen
                                    alarms[alm][6] = year;                                         // aktuelles Jahr voreinstellen
                                    alarms[alm][4] = calc_wday (alarms[alm][3], alarms[alm][5], c_yearh, alarms[alm][6]); // aktuellen Wochentag ermitteln und voreinstellen
                                    alarms[alm][7] = alarms[alm][4] - 1;                           // Wochentagsprogramm voreinstellen
                                    alarms[alm][10] = 0;                                           // Alarm-Auslsung lschen
                                 }
                                 alarms[alm][0] = editval;                                         // Alarm-Modus aus Editorwert holen
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = alarms[alm][0];                                         // Alarmaktivierung als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 4;                                                      // Editor-Maximalwert auf 4 setzen
                              }
                              break;

                           case 1:                                                                 // falls Zeit ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 alarms[alm][2] = editval / 60;                                    // Stundenwert aus Editorwert holen
                                 alarms[alm][1] = editval % 60;                                    // Minutenwert aus Editorwert holen
                                 alm_flag = 1;
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = alarms[alm][2] * 60 + alarms[alm][1];                   // Zeit als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 1439;                                                   // Editor-Maximalwert auf 24*60-1 (23:59) setzen
                              }
                              break;

                           case 2:                                                                 // falls Tag ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 alarms[alm][3] = editval;                                         // Tag aus Editor holen
                                 alarms[alm][4] = calc_wday (alarms[alm][3], alarms[alm][5], c_yearh, alarms[alm][6]); // Wochentag ermitteln
                                 alarms[alm][7] = alarms[alm][4] - 1;                              // Wochentagsprogramm voreinstellen
                                 alm_flag = 1;
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = alarms[alm][3];                                         // Tag als Editorwert verwenden
                                 editmin = 1;                                                      // Editor-Minimalwert auf 1 setzen
                                 editmax = 31;                                                     // Editor-Maximalwert auf 31 setzen
                              }
                              break;

                           case 3:                                                                 // falls Monat ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 alarms[alm][5] = editval;                                         // Monat aus Editor holen
                                 alarms[alm][4] = calc_wday (alarms[alm][3], alarms[alm][5], c_yearh, alarms[alm][6]); // Wochentag ermitteln
                                 alarms[alm][7] = alarms[alm][4] - 1;                              // Wochentagsprogramm voreinstellen
                                 alm_flag = 1;
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = alarms[alm][5];                                         // Monat als Editorwert verwenden
                                 editmin = 1;                                                      // Editor-Minimalwert auf 1 setzen
                                 editmax = 12;                                                     // Editor-Maximalwert auf 12 setzen
                              }
                              break;

                           case 4:                                                                 // falls Jahr ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 alarms[alm][6] = editval;                                         // Jahr aus Editor holen
                                 alarms[alm][4] = calc_wday (alarms[alm][3], alarms[alm][5], c_yearh, alarms[alm][6]); // Wochentag ermitteln
                                 alarms[alm][7] = alarms[alm][4] - 1;                              // Wochentagsprogramm voreinstellen
                                 alm_flag = 1;
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = alarms[alm][6];                                         // Jahr als Editorwert verwenden
                                 editmin = year;                                                   // Editor-Minimalwert auf aktuelles Jahr setzen
                                 editmax = 99;                                                     // Editor-Maximalwert auf 99 setzen
                              }
                              break;

                           case 5:                                                                 // falls Wochentagsprogramm ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 alarms[alm][7] = editval;                                         // Wochentagsprogramm aus Editor holen
                                 alm_flag = 1;
                              }
                              else
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = alarms[alm][7];                                         // Wochentagsprogramm als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 9;                                                      // Editor-Maximalwert auf 9 setzen
                              }
                              break;

                           case 6:                                                                 // falls Sound-Nummer ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 alarms[alm][8] = editval;                                         // Sound-Nummer aus Editor holen
                                 eep_write_byte (e_alarms + (alm * 10 + 8), alarms[alm][8]);       // Sound-Nummer im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = alarms[alm][8];                                         // Sound-Nummer als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 12;                                                     // Editor-Maximalwert auf 12 setzen
                              }
                              break;

                           case 7:                                                                 // falls Lautstrke ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 alarms[alm][9] = editval;                                         // Lautstrkewert aus Editor holen
                                 eep_write_byte (e_alarms + (alm * 10 + 9), alarms[alm][9]);       // Lautstrke im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = alarms[alm][9];                                         // Lautstrke als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 10;                                                     // Editor-Maximalwert auf 10 setzen
                              }
                              break;

                           case 8:                                                                 // falls Sound-Test ausgewhlt
                              cli ();                                                              // Interrupts sperren
                              if (snstat)                                                          // wenn Sound-Erzeugung aktiv
                              {
                                 snstat = 3;                                                       // Sound-Erzeugung stoppen
                              }
                              else                                                                 // wenn Sound-Erzeugung inaktiv
                              {
                                 sndptr = (PGM_P) pgm_read_word (&sounds[alarms[alm][8]]);         // Pointer auf Alarm-Sound
                                 if (alarms[alm][9] == 0) maxvol = 0;                              // wenn Lautstrke 0 eingestellt -> aus
                                 else maxvol = (alarms[alm][9] + 5) * 17;                          // sonst Sound-Lautstrke setzen (102-255)
                                 snstat = 1;                                                       // Sound-Erzeugung aktivieren
                              }
                              sei ();                                                              // Interrupts wieder freigeben
                              break;
                        }
                        break;

                     case 6:                                                                       // falls Anzeigemodus 6 (Einstellungen 2)
                        switch (submopt)                                                           // weitere Aktionen entsprechend Untermen-Option
                        {
                           case 0:                                                                 // falls DCF-Signal-Invertierung ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 dcfinv = editval;                                                 // DCF-Signal-Invertierung aus Editor kopieren
                                 eep_write_byte (e_dcfinv, dcfinv);                                // Invertierung im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = dcfinv;                                                 // DCF-Signal-Invertierung als Editorwert
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 1;                                                      // Editor-Maximalwert auf 1 setzen
                              }
                              break;

                           case 1:                                                                 // falls DCF-Pull-up ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 dcfpup = editval;                                                 // DCF-Pull-up aus Editor kopieren
                                 eep_write_byte (e_dcfpup, dcfpup);                                // DCF-Pull-up im EEPROM speichern
                                 if (dcfpup) PORTE |= 1 << PE2;                                    // wenn konfiguriert -> Pull-Up an PortE2 einschalten
                                 else PORTE &= ~(1 << PE2);                                        // oder auschalten
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = dcfpup;                                                 // DCF-Pull-up als Editorwert
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 1;                                                      // Editor-Maximalwert auf 1 setzen
                              }
                              break;

                           case 2:                                                                 // falls Geburtstagslisten-Sortierung ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 bdlsort = editval;                                                // Geburtstagslisten-Sortierung aus Editorwert holen
                                 eep_write_byte (e_bdlsort, bdlsort);                              // Einstellung im EEPROM speichern
                                 bd_flag = 1;                                                      // Geburtstage reorganisieren
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = bdlsort;                                                // Geburtstagslisten-Sort. als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 2;                                                      // Editor-Maximalwert auf 2 setzen
                              }
                              break;

                           case 3:                                                                 // falls Geburtstagsnamen-Reihenfolge ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 bdordr = editval;                                                 // Namen-Reihenfolge aus Editorwert holen
                                 eep_write_byte (e_bdordr, bdordr);                                // Einstellung im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = bdordr;                                                 // Namen-Reihenfolge als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 1;                                                      // Editor-Maximalwert auf 1 setzen
                              }
                              break;

                           case 4:                                                                 // falls Alters-Anzeige ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 bdagem = editval;                                                 // Alters-Anzeige aus Editorwert holen
                                 eep_write_byte (e_bdagem, bdagem);                                // Einstellung im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = bdagem;                                                 // Alters-Anzeige als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 1;                                                      // Editor-Maximalwert auf 1 setzen
                              }
                              break;

                           case 5:                                                                 // falls Geburtstags-Alarm ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 if (editval == -1)                                                // wenn Editorwert = -1 (aus)
                                 {
                                    alarms[3][0] = 0;                                              // Alarm ausschalten
                                    alarms[3][1] = 0;                                              // Minuten = 0 setzen
                                    alarms[3][2] = 0;                                              // Stunden = 0 setzen
                                 }
                                 else
                                 {
                                    alarms[3][0] = 1;                                              // Alarm einschalten
                                    alarms[3][2] = editval / 60;                                   // Stundenwert aus Editorwert holen
                                    alarms[3][1] = editval % 60;                                   // Minutenwert aus Editorwert holen
                                 }
                                 eep_write_byte (e_alarms + 30, alarms[3][0]);                     // Alarm-Modus im EEPROM speichern
                                 eep_write_byte (e_alarms + 31, alarms[3][1]);                     // Minuten im EEPROM speichern
                                 eep_write_byte (e_alarms + 32, alarms[3][2]);                     // Stunden im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 if (alarms[3][0])                                                 // wenn Geburtstags-Alarm eingeschaltet
                                 {
                                    editval = alarms[3][2] * 60 + alarms[3][1];                    // Zeit als Editorwert verwenden
                                 }
                                 else editval = -1;                                                // wenn Alarm ausgeschaltet -> Editorwert auf "aus"
                                 editmin = -1;                                                     // Editor-Minimalwert auf -1 setzen
                                 editmax = 1439;                                                   // Editor-Maximalwert auf 24*60-1 (24:00) setzen
                              }
                              break;

                           case 6:                                                                 // falls Sound-Nummer ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 alarms[3][8] = editval;                                           // Sound-Nummer aus Editor holen
                                 eep_write_byte (e_alarms + 38, alarms[3][8]);                     // Sound-Nummer im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = alarms[3][8];                                           // Sound-Nummer als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 12;                                                     // Editor-Maximalwert auf 12 setzen
                              }
                              break;

                           case 7:                                                                 // falls Lautstrke ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 alarms[3][9] = editval;                                           // Lautstrkewert aus Editor holen
                                 eep_write_byte (e_alarms + 39, alarms[3][9]);                     // Lautstrke im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = alarms[3][9];                                           // Lautstrke als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 10;                                                     // Editor-Maximalwert auf 10 setzen
                              }
                              break;

                           case 8:                                                                 // falls Sound-Test ausgewhlt
                              cli ();                                                              // Interrupts sperren
                              if (snstat)                                                          // wenn Sound-Erzeugung aktiv
                              {
                                 snstat = 3;                                                       // Sound-Erzeugung stoppen
                              }
                              else                                                                 // wenn Sound-Erzeugung inaktiv
                              {
                                 sndptr = (PGM_P) pgm_read_word (&sounds[alarms[3][8]]);           // Pointer auf Alarm-Sound
                                 if (alarms[3][9] == 0) maxvol = 0;                                // wenn Lautstrke 0 eingestellt -> aus
                                 else maxvol = (alarms[3][9] + 5) * 17;                            // sonst Sound-Lautstrke setzen (102-255)
                                 snstat = 1;                                                       // Sound-Erzeugung aktivieren
                              }
                              sei ();                                                              // Interrupts wieder freigeben
                              break;
                        }
                        break;

                     case 7:                                                                       // falls Anzeigemodus 7 (Geburtstage 1 mit 8 Eintrgen)
                        dispmode = 8;                                                              // Anzeigemodus auf 8 setzen (Geburtstage 2 mit 4 Eintrgen)
                        bd2pos = bd1pos / 2;                                                       // Anzeigeposition halbieren
                        break;

                     case 8:                                                                       // falls Anzeigemodus 7 (Geburtstage 1 mit 8 Eintrgen)
                        dispmode = 9;                                                              // Anzeigemodus auf 8 setzen (Geburtstage 2 mit 4 Eintrgen)
                        submopt = 0;                                                               // Untermen-Option 0 setzen
                        bd_change = 0;                                                             // Eintrags-nderungs-Flag lschen
                        bdindx = bdilist[bdipos];                                                  // Wert aus dem Geburtstagsindex lesen
                        if (bdindx < 115)                                                          // wenn gltiger Eintrag vorhanden ist
                        {
                           bdeday = eep_read_byte (e_bddata + (bdindx * 34));                      // Geburtstag kopieren
                           bdemonth = eep_read_byte (e_bddata + (bdindx * 34 + 1));                // Geburtsmonat kopieren
                           bdeyear = eep_read_byte (e_bddata + (bdindx * 34 + 2));                 // Geburtsjahr kopieren
                           bdeyearh = eep_read_byte (e_bddata + (bdindx * 34 + 3));                // Geburtsjahrhundert kopieren
                           for (j = 0; j < 15; j ++)                                               // 15 Textzeichen bearbeiten
                           {
                              bde1buf[j] = eep_read_byte (e_bddata + (bdindx * 34 + 4 + j));       // Vorname kopieren
                              bde2buf[j] = eep_read_byte (e_bddata + (bdindx * 34 + 19 + j));      // Nachname kopieren
                           }
                        }
                        else                                                                       // wenn kein gltiger Eintrag vorhanden ist
                        {
                           for (i = 0; i < 115; i ++)                                              // leeren Platz in der Indexliste suchen
                           {
                              if (eep_read_byte (e_bddata + (i * 34)) > 31) break;                 // leeren Platz gefunden -> Ende
                           }
                           bdindx = i;                                                             // Index setzen
                           bdeday = day;                                                           // Tag setzen
                           bdemonth = month;                                                       // Monat setzen
                           bdeyear = year;                                                         // Jahr setzen
                           bdeyearh = c_yearh;                                                     // Jahrhundert setzen
                           for (j = 0; j < 15; j ++)                                               // 15 Textzeichen bearbeiten
                           {
                              bde1buf[j] = 255;                                                    // Vorname lschen
                              bde2buf[j] = 255;                                                    // Nachname lschen
                           }
                        }
                        break;

                     case 9:                                                                       // falls Anzeigemodus 9 (Geburtstag eingeben)
                        switch (submopt)                                                           // weitere Aktionen entsprechend Untermen-Option
                        {
                           case 0:                                                                 // falls Geburtstag ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 bdeday = editval;                                                 // Geburtstag aus Editor holen
                                 if (bdeday == 0)                                                  // wenn Geburtstag = 0 (lschen)
                                 {
                                    dispmode = 8;                                                  // Anzeigemodus auf 8 setzen (Geburtstage 2 mit 4 Eintrgen)
                                    bd_change = 0;                                                 // Eintrags-nderungs-Flag lschen
                                    bdilist[bdipos] = 255;                                         // Indexeintrag lschen
                                    for (j = 0; j < 34; j ++)                                      // 34 Bytes lschen
                                    {
                                       eep_write_byte (e_bddata + (bdindx * 34 + j), 255);         // Geburtstags-Eintrag lschen
                                    }
                                    bd_flag = 1;                                                   // Geburtstage reorganisieren
                                 }
                                 else bd_change = 1;                                               // Eintrags-nderungs-Flag setzen
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = bdeday;                                                 // Geburtstag als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 31;                                                     // Editor-Maximalwert auf 31 setzen
                              }
                              break;

                           case 1:                                                                 // falls Geburtsmonat ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 bdemonth = editval;                                               // Geburtsmonat aus Editor holen
                                 bd_change = 1;                                                    // Eintrags-nderungs-Flag setzen
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = bdemonth;                                               // Geburtsmonat als Editorwert verwenden
                                 editmin = 1;                                                      // Editor-Minimalwert auf 1 setzen
                                 editmax = 12;                                                     // Editor-Maximalwert auf 12 setzen
                              }
                              break;

                           case 2:                                                                 // falls Geburtsjahr ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 if (editval == -1)                                                // wenn Editorwert = -1
                                 {
                                    bdeyearh = 255;                                                // Geburtsjahrhundert auf 255 setzen
                                    bdeyear = 255;                                                 // Geburtsjahr auf 255 setzen
                                 }
                                 else                                                              // wenn Editorwert >= 0
                                 {
                                    bdeyearh = editval / 100;                                      // Geburtsjahrhundert aus Editor holen
                                    bdeyear = editval % 100;                                       // Geburtsjahr aus Editor holen
                                 }
                                 bd_change = 1;                                                    // Eintrags-nderungs-Flag setzen
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 if (bdeyear > 99)                                                 // wenn ungltiges Geburtsjahr
                                 {
                                    editval = -1;                                                  // Editwert auf -1 (unbekannt) setzen
                                 }
                                 else                                                              // wenn gltiges Geburtsjahr
                                 {
                                    editval = bdeyearh * 100 + bdeyear;                            // Jahr als Editorwert verwenden
                                 }
                                 editmin = -1;                                                     // Editor-Minimalwert auf -1 setzen
                                 editmax = 2000 + year;                                            // Editor-Maximalwert auf aktuelles Jahr setzen
                              }
                              break;

                           case 3 ... 4:                                                           // falls Vorname oder Nachname ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 bdtxlen = 0;                                                      // Anfangswert fr Lnge des Eingabepuffers
                                 for (i = 0; i < bdtxpos; i ++)                                    // alle Zeichen bis zur Cursorposition bearbeiten
                                 {
                                    bdtxlen += chrl (bdtxbuf[i]) + 1;                              // Zeichenbreite und Zwischenraum addieren
                                 }
                                 if (editval < sizeof (charlist))                                  // wenn Editorwert = gltiges Zeichen
                                 {
                                    if (((bdtxlen + chrl (pgm_read_byte (&charlist[editval]))) < 99) && (bdtxpos < 15)) // wenn genug Platz und kein Pufferende
                                    {
                                       bdtxbuf[bdtxpos] = pgm_read_byte (&charlist[editval]);      // neues Zeichen addieren
                                       bdtxpos ++;                                                 // Cursorposition erhhen
                                       if (bdtxpos == 1)                                           // wenn zweite Zeichenposition -> "a" setzen
                                       {
                                          for (i = 0; i < sizeof (charlist); i ++)                 // Zeichenliste nach "a" durchsuchen
                                          {
                                             if (pgm_read_byte (&charlist[i]) == 'a')              // Position als Editorwert
                                                editval = i;                                       // verwenden
                                          }
                                       }
                                    }
                                 }
                                 if (editval == sizeof (charlist))                                 // wenn Editorwert = Lschzeichen
                                 {
                                    if (bdtxpos > 0) bdtxpos --;                                   // wenn nicht Anfangsposition -> Position - 1
                                 }

                                 if (editval == sizeof (charlist) + 1)                             // wenn Editorwert = OK-Zeichen
                                 {
                                    editmode = 0;                                                  // Editormodus deaktivieren
                                    for (i = 0; i < 15; i ++)                                      // 15 Zeichen kopieren
                                    {
                                       if (i < bdtxpos)                                            // wenn Zeichen vor der Cursorposition
                                       {
                                          if (submopt == 3)                                        // wenn Vorname ausgewhlt
                                          {
                                             bde1buf[i] = bdtxbuf[i];                              // Editorpuffer in Vornamepuffer kopieren
                                          }
                                          else                                                     // wenn Nachname ausgewhlt
                                          {
                                             bde2buf[i] = bdtxbuf[i];                              // Editorpuffer in Nachnamepuffer kopieren
                                          }
                                       }
                                       else                                                        // wenn Zeichen ab der Cursorposition
                                       {
                                          if (submopt == 3)                                        // wenn Vorname ausgewhlt
                                          {
                                             bde1buf[i] = 255;                                     // restlichen Vornamepuffer mit 0xff auffllen
                                          }
                                          else                                                     // wenn Nachname ausgewhlt
                                          {
                                             bde2buf[i] = 255;                                     // restlichen Nachnamepuffer mit 0xff auffllen
                                          }
                                       }
                                    }
                                    bd_change = 1;                                                 // Eintrags-nderungs-Flag setzen
                                 }
                                 if (bdtxpos > 14)                                                 // wenn Pufferende erreicht ist
                                 {
                                    editval = sizeof (charlist) + 1;                               // OK-Zeichen als Editorwert verwenden
                                 }
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 for (bdtxpos = 0; bdtxpos < 15; bdtxpos ++)                       // 15 Zeichen kopieren
                                 {
                                    if (submopt == 3)                                              // wenn Vorname ausgewhlt
                                    {
                                       bdtxbuf[bdtxpos] = bde1buf[bdtxpos];                        // Vornamepuffer in Editorpuffer kopieren
                                    }
                                    else                                                           // wenn Nachname ausgewhlt
                                    {
                                       bdtxbuf[bdtxpos] = bde2buf[bdtxpos];                        // Nachnamepuffer in Editorpuffer kopieren
                                    }
                                    if (bdtxbuf[bdtxpos] == 255) break;                            // wenn ungltiges Zeichen -> Ende
                                 }
                                 editval = sizeof (charlist) + 1;                                  // Cursorzeichen OK als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = sizeof (charlist) + 1;                                  // Editor-Maximalwert auf OK-Zeichen setzen
                                 if (bdtxpos == 0)                                                 // wenn noch kein Name vorhanden -> "A" setzen
                                 {
                                    for (i = 0; i < sizeof (charlist); i ++)                       // Zeichenliste nach "A" durchsuchen
                                    {
                                       if (pgm_read_byte (&charlist[i]) == 'A') editval = i;       // Position als Editorwert verwenden
                                    }
                                 }
                              }
                              break;
                        }
                        break;

                     case 10:                                                                      // falls Anzeigemodus 10 (Einstellungen 3)
                        switch (submopt)                                                           // weitere Aktionen entsprechend Untermen-Option
                        {
                           case 0:                                                                 // falls Sound-Freigabe ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 s_enable = editval;                                               // Sound-Freigabe aus Editorwert holen
                                 eep_write_byte (e_senable, s_enable);                             // Einstellung im EEPROM speichern
                                 sec_flag = 1;                                                     // Sekunden-Flag setzen, Analoguhr neu ausgeben
                              }
                              else
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = s_enable;                                               // Sound-Freigabe als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 1;                                                      // Editor-Maximalwert auf 1 setzen
                              }
                              break;

                           case 1:                                                                 // falls Gong-Status ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 gong = editval;                                                   // Gong-Status aus Editor kopieren
                                 eep_write_byte (e_gong, gong);                                    // Gong-Status im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = gong;                                                   // Gong-Status als Editorwert
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 1;                                                      // Editor-Maximalwert auf 1 setzen
                              }
                              break;

                           case 2:                                                                 // falls Gong-Beginnzeit ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 gongh1 = editval;                                                 // Gong-Beginnzeit aus Editor kopieren
                                 eep_write_byte (e_gongh1, gongh1);                                // Gong-Beginnzeit im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = gongh1;                                                 // Gong-Beginnzeit als Editorwert
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 23;                                                     // Editor-Maximalwert auf 23 (Stunden) setzen
                              }
                              break;

                           case 3:                                                                 // falls Gong-Endezeit ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 gongh2 = editval;                                                 // Gong-Endezeit aus Editor kopieren
                                 eep_write_byte (e_gongh2, gongh2);                                // Gong-Endezeit im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = gongh2;                                                 // Gong-Endezeit als Editorwert
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 23;                                                     // Editor-Maximalwert auf 23 (Stunden) setzen
                              }
                              break;

                           case 4:                                                                 // falls Sound-Nummer ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 gong_snd = editval;                                               // Sound-Nummer aus Editor holen
                                 eep_write_byte (e_gong_snd, gong_snd);                            // Sound-Nummer im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = gong_snd;                                               // Sound-Nummer als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 12;                                                     // Editor-Maximalwert auf 12 setzen
                              }
                              break;

                           case 5:                                                                 // falls Lautstrke ausgewhlt
                              if (editmode)                                                        // wenn Editor aktiv
                              {
                                 editmode = 0;                                                     // Editormodus deaktivieren
                                 gong_vol = editval;                                               // Lautstrkewert aus Editor holen
                                 eep_write_byte (e_gong_vol, gong_vol);                            // Lautstrke im EEPROM speichern
                              }
                              else                                                                 // wenn Editor inaktiv
                              {
                                 editmode = 1;                                                     // Editormodus aktivieren
                                 editval = gong_vol;                                               // Lautstrke als Editorwert verwenden
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = 10;                                                     // Editor-Maximalwert auf 10 setzen
                              }
                              break;

                           case 6:                                                                 // falls Sound-Test ausgewhlt
                              cli ();                                                              // Interrupts sperren
                              if (snstat)                                                          // wenn Sound-Erzeugung aktiv
                              {
                                 snstat = 3;                                                       // Sound-Erzeugung stoppen
                              }
                              else                                                                 // wenn Sound-Erzeugung inaktiv
                              {
                                 sndptr = (PGM_P) pgm_read_word (&sounds[gong_snd]);               // Pointer auf Gong-Sound
                                 if (gong_vol == 0) maxvol = 0;                                    // wenn Lautstrke 0 eingestellt -> aus
                                 else maxvol = (gong_vol + 5) * 17;                                // sonst Sound-Lautstrke setzen (102-255)
                                 snstat = 1;                                                       // Sound-Erzeugung aktivieren
                              }
                              sei ();                                                              // Interrupts wieder freigeben
                              break;

                        }
                  }
               }
            }
            else                                                                                   // wenn LCD ausgeschaltet
            {
               lcd_on ();                                                                          // LCD einschalten
               sec_flag = 1;                                                                       // Sekunden-Flag setzen, Analoguhr und Symbole neu ausgeben
               chg_flag = 1;                                                                       // nderungs-Flag setzen, Informationen neu ausgeben
            }
         }
      }
      sei ();                                                                                      // Interrupts wieder freigeben

// Taster 2 abfragen und Funktionen steuern -----------------------------------------------------------------------------------------------------------------------------------------------------------

      cli ();                                                                                      // Interrupts sperren
      if (key2stat)                                                                                // wenn Taster 2 gedrckt
      {
         if (!key2quit)                                                                            // wenn Tastendruck noch nicht quittiert
         {
            key2quit = 1;                                                                          // Quittierung setzen
            sei ();                                                                                // Interrupt wieder freigeben
            chg_flag = 1;                                                                          // nderungs-Flag setzen, Informationen neu ausgeben
            disptout = c_disptout;                                                                 // Anzeige-Timeout neu starten
            if (lcd_is_on)                                                                         // wenn LCD eingeschaltet
            {

               if ((alarms[0][10] == 1) || (alarms[1][10] == 1) || (alarms[2][10] == 1) || (alarms[3][10] == 1)) // wenn mindestens ein Alarm ausgelst wurde
               {
                  if (alarms[0][10] == 1) alarms[0][10] = 2;                                       // Alarm 1 quittieren
                  if (alarms[1][10] == 1) alarms[1][10] = 2;                                       // Alarm 2 quittieren
                  if (alarms[2][10] == 1) alarms[2][10] = 2;                                       // Alarm 3 quittieren
                  if (alarms[3][10] == 1) alarms[3][10] = 2;                                       // Geburtstags-Alarm quittieren
                  cli ();                                                                          // Interrupts sperren
                  snstat = 3;                                                                      // Sound-Erzeugung stoppen
                  sei ();                                                                          // Interrupts wieder freigeben
               }
               else                                                                                // wenn Alarm-frei
               {
                  switch (dispmode)                                                                // weitere Aktionen entsprechend Anzeigemodus
                  {
                     case 0:                                                                       // falls Anzeigemodus 0 (normale Anzeige)
                        break;                                                                     // hier knnte man noch eine Funktion einbauen

                     case 1:                                                                       // falls Anzeigemodus 1 (Hauptmen)
                        dispmode = 0;                                                              // Anzeigemodus auf 0 (normale Anzeige)
                        break;

                     case 2 ... 7:                                                                 // falls Anzeigemodus > 1
                        if (editmode)                                                              // wenn Editor aktiv
                        {
                           editmode = 0;                                                           // Editor deaktivieren
                        }
                        else                                                                       // wenn Editor inaktiv
                        {
                           dispmode = 1;                                                           // Anzeigemodus auf 1 (Hauptmen)
                        }
                        break;

                     case 8:                                                                       // falls Anzeigemodus 8 (Geburtstage 2 mit 4 Eintrgen)
                        dispmode = 7;                                                              // Anzeigemodus auf 7 setzen (Geburtstage 1 mit 8 Eintrgen)
                        if (bd1pos > bdipos) bd1pos = bdipos;                                      // Anzeigeposition am Listenanfang korrigieren
                        if ((bdipos - bd1pos) > 107) bdipos = 107;                                 // Anzeigeposition am Listenende korrigieren
                        break;

                     case 9:                                                                       // falls Anzeigemodus 9 (Geburtstag eingeben)
                        if (editmode)                                                              // wenn Editor aktiv
                        {
                           editmode = 0;                                                           // Editor deaktivieren
                        }
                        else                                                                       // wenn Editor inaktiv
                        {
                           if ((bde1buf[0] == 255) && (bde2buf[0] == 255))                         // wenn kein Vor- und Nachname eingegeben wurde
                           {
                              submopt = 3;                                                         // Untermen auf Vorname-Eingabe setzen
                              editmode = 1;                                                        // Editormodus wieder aktivieren
                              for (bdtxpos = 0; bdtxpos < 15; bdtxpos ++)                          // 15 Zeichen kopieren
                              {
                                 bdtxbuf[bdtxpos] = bde1buf[bdtxpos];                              // Vornamepuffer in Editorpuffer kopieren
                                 if (bdtxbuf[bdtxpos] == 255) break;                               // wenn ungltiges Zeichen -> Ende
                              }
                              editval = sizeof (charlist) + 1;                                     // Cursorzeichen OK als Editorwert verwenden
                              editmin = 0;                                                         // Editor-Minimalwert auf 0 setzen
                              editmax = sizeof (charlist) + 1;                                     // Editor-Maximalwert auf OK-Zeichen setzen
                              if (bdtxpos == 0)                                                    // wenn noch kein Name vorhanden -> "A" setzen
                              {
                                 for (i = 0; i < sizeof (charlist); i ++)                          // Zeichenliste nach "A" durchsuchen
                                 {
                                    if (pgm_read_byte (&charlist[i]) == 'A') editval = i;          // Position als Editorwert verwenden
                                 }
                              }
                           }                           

                           if ((bdeyear % 4) == 0)                                                 // wenn Schaltjahr
                           {
                              if (bdeday > pgm_read_byte (&daytab2[bdemonth - 1]))                 // wenn ungltiger Tag
                              {
                                 submopt = 0;                                                      // Untermen auf Tag-Einstellung setzen
                                 editmode = 1;                                                     // Editormodus wieder aktivieren
                                 editval = pgm_read_byte (&daytab2[bdemonth - 1]);                 // letzten Tag des Monats voreinstellen
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = editval;                                                // Editor-Maximalwert auf letzten Tag des Monats setzen
                              }
                           }                           
                           else                                                                    // wenn kein Schaltjahr
                           {
                              if (bdeday > pgm_read_byte (&daytab1[bdemonth - 1]))                 // wenn ungltiger Tag
                              {
                                 submopt = 0;                                                      // Untermen auf Tag-Einstellung setzen
                                 editmode = 1;                                                     // Editormodus wieder aktivieren
                                 editval = pgm_read_byte (&daytab1[bdemonth - 1]);                 // letzten Tag des Monats voreinstellen
                                 editmin = 0;                                                      // Editor-Minimalwert auf 0 setzen
                                 editmax = editval;                                                // Editor-Maximalwert auf letzten Tag des Monats setzen
                              }                              
                           }

                           if (editmode == 0)                                                      // wenn Editormodus nicht wieder aktiviert wurde
                           {
                              dispmode = 8;                                                        // Anzeigemodus auf 8 setzen (Geburtstage 2 mit 4 Eintrgen)
                              if (bd_change)                                                       // wenn Geburtstagseintrag gendert wurde
                              {
                                 bd_change = 0;                                                    // Eintrags-nderungs-Flag wieder lschen
                                 bdilist[bdipos] = bdindx;                                         // Geburtstagsindex speichern
                                 eep_write_byte (e_bddata + (bdindx * 34), bdeday);                // Geburtstag speichern
                                 eep_write_byte (e_bddata + (bdindx * 34 + 1), bdemonth);          // Geburtsmonat speichern
                                 eep_write_byte (e_bddata + (bdindx * 34 + 2), bdeyear);           // Geburtsjahr speichern
                                 eep_write_byte (e_bddata + (bdindx * 34 + 3), bdeyearh);          // Geburtsjahrhundert speichern
                                 for (j = 0; j < 15; j ++)                                         // 15 Textzeichen bearbeiten
                                 {
                                    eep_write_byte (e_bddata + (bdindx * 34 + 4 + j), bde1buf[j]);  // Vorname und
                                    eep_write_byte (e_bddata + (bdindx * 34 + 19 + j), bde2buf[j]); // Nachname speichern
                                 }
                                 bd_flag = 1;                                                      // Geburtstage reorganisieren
                              }
                           }
                        }
                        break;

                     case 10:                                                                      // falls Anzeigemodus 10 (Einstellungen 3)
                        if (editmode)                                                              // wenn Editor aktiv
                        {
                           editmode = 0;                                                           // Editor deaktivieren
                        }
                        else                                                                       // wenn Editor inaktiv
                        {
                           dispmode = 1;                                                           // Anzeigemodus auf 1 (Hauptmen)
                        }
                        break;

                  }
               }
            }
            else                                                                                   // wenn LCD ausgeschaltet
            {
               lcd_on ();                                                                          // LCD einschalten
               sec_flag = 1;                                                                       // Sekunden-Flag setzen, Analoguhr und Symbole neu ausgeben
            }
         }
      }
      sei ();                                                                                      // Interrupts wieder freigeben

// Drehimpulsgeber abfragen und Funktionen steuern ----------------------------------------------------------------------------------------------------------------------------------------------------

      int8_t encdiff;                                                                              // Zwischenspeicher fr Drehimpulsgeber-Differenz

      cli ();                                                                                      // Interrupts sperren
      if (encpos != encmem)                                                                        // wenn Positions-nderung des Drehimpulsgebers
      {
         encdiff = encpos - encmem;                                                                // Positions-Differenz ermitteln
         encmem = encpos;                                                                          // neue Position speichern
         sei ();                                                                                   // Interrupt wieder freigeben

         if (editmode)                                                                             // wenn Editor aktiv
         {
            if (!(PINB & (1 << PB4))) encdiff = -encdiff;                                          // wenn Jumper 3 gesteckt -> Positions-Differenz negieren
         }
         else                                                                                      // wenn Editor nicht aktiv (Men-Modus)
         {
            if (!(PINB & (1 << PB3))) encdiff = -encdiff;                                          // wenn Jumper 2 gesteckt -> Positions-Differenz negieren
         }

         chg_flag = 1;                                                                             // nderungs-Flag setzen, Informationen neu ausgeben
         disptout = c_disptout;                                                                    // Anzeige-Timeout starten
         if (lcd_is_on)                                                                            // wenn LCD eingeschaltet
         {
            switch (dispmode)                                                                      // weitere Aktionen entsprechend Anzeigemodus
            {
               case 1:                                                                             // falls Anzeigemodus 1 (Hauptmen)
                  mainopt = mainopt + encdiff;                                                     // Differenz zur Menoption addieren
                  if (mainopt < 0) mainopt = 0;                                                    // kleinster Wert = 0
                  if (mainopt > 8) mainopt = 8;                                                    // grter Wert = 8
                  if ((encdiff > 0) && (syncstat < 2) && (mainopt > 0) && (mainopt < 5)) mainopt = 5;
                  if ((encdiff < 0) && (syncstat < 2) && (mainopt > 0) && (mainopt < 5)) mainopt = 0; // Alarme berspringen, wenn Uhr noch nicht gestellt ist
                  break;

               case 2:                                                                             // falls Anzeigemodus 2 (Uhr stellen)
                  if (!editmode)                                                                   // wenn Editor inaktiv
                  {
                     submopt = submopt + encdiff;                                                  // Differenz zur Menoption addieren
                     if (submopt < 0) submopt = 0;                                                 // bei Unterschreitung -> auf kleinsten Wert 0 setzen
                     if (submopt > 5) submopt = 5;                                                 // bei berschreitung -> auf grten Wert 5 setzen
                  }
                  else                                                                             // wenn Editor aktiv
                  {
                     if ((encdiff < -2) || (encdiff > 2)) encdiff = encdiff * 10;                  // wenn Impulsgeber schnell gedreht wird -> Wert verzehnfachen (Beschleunigung)
                     editval = editval + encdiff;                                                  // Differenz zum Editorwert addieren
                     if (editval > editmax) editval = editmin;                                     // wenn Maximalwert berschritten -> auf Minimalwert setzen (berlauf)
                     if (editval < editmin) editval = editmax;                                     // wenn Minimalwert unterschritten -> auf Maximalwert setzen (berlauf)
                  }
                  break;

               case 3:                                                                             // falls Anzeigemodus 3 (Einstellungen 1)
                  if (!editmode)                                                                   // wenn Editor inaktiv
                  {
                     submopt = submopt + encdiff;                                                  // Differenz zur Menoption addieren
                     if (submopt < 0) submopt = 0;                                                 // bei Unterschreitung -> auf kleinsten Wert 0 setzen
                     if (submopt > 7) submopt = 7;                                                 // bei berschreitung -> auf grten Wert 7 setzen
                  }
                  else                                                                             // wenn Editor aktiv
                  {
                     if ((encdiff < -2) || (encdiff > 2)) encdiff = encdiff * 10;                  // wenn Impulsgeber schnell gedreht wird -> Wert verzehnfachen (Beschleunigung)
                     editval = editval + encdiff;                                                  // Differenz zum Editorwert addieren
                     if (editval > editmax) editval = editmin;                                     // wenn Maximalwert berschritten -> auf Minimalwert setzen (berlauf)
                     if (editval < editmin) editval = editmax;                                     // wenn Minimalwert unterschritten -> auf Maximalwert setzen (berlauf)
                  }
                  break;

               case 5:                                                                             // falls Anzeigemodus 5 (Alarme einstellen)
                  if (!editmode)                                                                   // wenn Editor inaktiv
                  {
                     submopt = submopt + encdiff;                                                  // Differenz zur Menoption addieren
                     if (submopt < 0) submopt = 0;                                                 // bei Unterschreitung -> auf kleinsten Wert 0 setzen
                     if (submopt > 8) submopt = 8;                                                 // bei berschreitung -> auf grten Wert 8 setzen

                     if (alarms[alm][0] == 0) submopt = 0;                                         // wenn Alarm nicht aktiv -> nur Option 0 mglich
                     if (alarms[alm][0] == 1)                                                      // wenn Alarm einmalig
                     {
                        if ((encdiff > 0) && (submopt == 5)) submopt = 6;                          // Wochenprogramm-Option berspringen
                        if ((encdiff < 0) && (submopt == 5)) submopt = 4;                          // Wochenprogramm-Option berspringen
                     }
                     if (alarms[alm][0] == 2)                                                      // wenn Alarm tglich
                     {
                        if ((encdiff > 0) && (submopt > 1) && (submopt < 6)) submopt = 6;          // Optionen berspringen
                        if ((encdiff < 0) && (submopt > 1) && (submopt < 6)) submopt = 1;          // Optionen berspringen
                     }
                     if (alarms[alm][0] == 3)                                                      // wenn Alarm wchentlich
                     {
                        if ((encdiff > 0) && (submopt > 1) && (submopt < 5)) submopt = 5;          // Optionen berspringen
                        if ((encdiff < 0) && (submopt > 1) && (submopt < 5)) submopt = 1;          // Optionen berspringen
                     }
                     if (alarms[alm][0] == 4)                                                      // wenn Alarm monatlich
                     {
                        if ((encdiff > 0) && (submopt > 2) && (submopt < 6)) submopt = 6;          // Optionen berspringen
                        if ((encdiff < 0) && (submopt > 2) && (submopt < 6)) submopt = 2;          // Optionen berspringen
                     }
                  }
                  else                                                                             // wenn Editor aktiv
                  {
                     if ((encdiff < -2) || (encdiff > 2)) encdiff = encdiff * 10;                  // wenn Impulsgeber schnell gedreht wird -> Wert verzehnfachen (Beschleunigung)
                     editval = editval + encdiff;                                                  // Differenz zum Editorwert addieren
                     if (editval > editmax) editval = editmin;                                     // wenn Maximalwert berschritten -> auf Minimalwert setzen (berlauf)
                     if (editval < editmin) editval = editmax;                                     // wenn Minimalwert unterschritten -> auf Maximalwert setzen (berlauf)
                  }
                  break;

               case 6:                                                                             // falls Anzeigemodus 6 (Einstellungen 2)
                  if (!editmode)                                                                   // wenn Editor inaktiv
                  {
                     submopt = submopt + encdiff;                                                  // Differenz zur Menoption addieren
                     if (submopt < 0) submopt = 0;                                                 // bei Unterschreitung -> auf kleinsten Wert 0 setzen
                     if (submopt > 8) submopt = 8;                                                 // bei berschreitung -> auf grten Wert 8 setzen
                  }
                  else                                                                             // wenn Editor aktiv
                  {
                     if ((encdiff < -2) || (encdiff > 2)) encdiff = encdiff * 10;                  // wenn Impulsgeber schnell gedreht wird -> Wert verzehnfachen (Beschleunigung)
                     editval = editval + encdiff;                                                  // Differenz zum Editorwert addieren
                     if (editval > editmax) editval = editmin;                                     // wenn Maximalwert berschritten -> auf Minimalwert setzen (berlauf)
                     if (editval < editmin) editval = editmax;                                     // wenn Minimalwert unterschritten -> auf Maximalwert setzen (berlauf)
                  }
                  break;

               case 7:                                                                             // falls Anzeigemodus 6 (Geburtstage 1 mit 8 Eintrgen)
                  bdipos = bdipos + encdiff;                                                       // Differenz zur Geburtstagslisten-Position addieren
                  bd1pos = bd1pos + encdiff;                                                       // Differenz zur Anzeige-Position addieren
                  if (bdipos < 0) bdipos = 0;                                                      // bei Unterschreitung -> auf kleinsten Wert 0 setzen
                  if (bdipos > 114) bdipos = 114;                                                  // bei berschreitung -> auf grten Wert 114 setzen
                  if (bd1pos < 0) bd1pos = 0;                                                      // bei Unterschreitung -> auf kleinsten Wert 0 setzen
                  if (bd1pos > 7) bd1pos = 7;                                                      // bei berschreitung -> auf grten Wert 7 setzen
                  break;

               case 8:                                                                             // falls Anzeigemodus 7 (Geburtstage 2 mit 4 Eintrgen)
                  bdipos = bdipos + encdiff;                                                       // Differenz zur Geburtstagslisten-Position addieren
                  bd2pos = bd2pos + encdiff;                                                       // Differenz zur Anzeige-Position addieren
                  if (bdipos < 0) bdipos = 0;                                                      // bei Unterschreitung -> auf kleinsten Wert 0 setzen
                  if (bdipos > 114) bdipos = 114;                                                  // bei berschreitung -> auf grten Wert 114 setzen
                  if (bd2pos < 0) bd2pos = 0;                                                      // bei Unterschreitung -> auf kleinsten Wert 0 setzen
                  if (bd2pos > 3) bd2pos = 3;                                                      // bei berschreitung -> auf grten Wert 3 setzen
                  break;

               case 9:                                                                             // falls Anzeigemodus 9 (Geburtstag eingeben)
                  if (!editmode)                                                                   // wenn Editor inaktiv
                  {
                     submopt = submopt + encdiff;                                                  // Differenz zur Menoption addieren
                     if (submopt < 0) submopt = 0;                                                 // bei Unterschreitung -> auf kleinsten Wert 0 setzen
                     if (submopt > 4) submopt = 4;                                                 // bei berschreitung -> auf grten Wert 4 setzen
                  }
                  else                                                                             // wenn Editor aktiv
                  {
                     if ((encdiff < -2) || (encdiff > 2)) encdiff = encdiff * 10;                  // wenn Impulsgeber schnell gedreht wird -> Wert verzehnfachen (Beschleunigung)
                     editval = editval + encdiff;                                                  // Differenz zum Editorwert addieren
                     if (editval > editmax) editval = editmin;                                     // wenn Maximalwert berschritten -> auf Minimalwert setzen (berlauf)
                     if (editval < editmin) editval = editmax;                                     // wenn Minimalwert unterschritten -> auf Maximalwert setzen (berlauf)
                  }
                  break;

               case 10:                                                                            // falls Anzeigemodus 10 (Einstellungen 3)
                  if (!editmode)                                                                   // wenn Editor inaktiv
                  {
                     submopt = submopt + encdiff;                                                  // Differenz zur Menoption addieren
                     if (submopt < 0) submopt = 0;                                                 // bei Unterschreitung -> auf kleinsten Wert 0 setzen
                     if (submopt > 6) submopt = 6;                                                 // bei berschreitung -> auf grten Wert 6 setzen
                  }
                  else                                                                             // wenn Editor aktiv
                  {
                     if ((encdiff < -2) || (encdiff > 2)) encdiff = encdiff * 10;                  // wenn Impulsgeber schnell gedreht wird -> Wert verzehnfachen (Beschleunigung)
                     editval = editval + encdiff;                                                  // Differenz zum Editorwert addieren
                     if (editval > editmax) editval = editmin;                                     // wenn Maximalwert berschritten -> auf Minimalwert setzen (berlauf)
                     if (editval < editmin) editval = editmax;                                     // wenn Minimalwert unterschritten -> auf Maximalwert setzen (berlauf)
                  }
                  break;
            }
         }
         else                                                                                      // wenn LCD ausgeschaltet
         {
            lcd_on ();                                                                             // LCD einschalten
            sec_flag = 1;                                                                          // Sekunden-Flag setzen, Analoguhr und Symbole neu ausgeben
         }
      }
      sei ();                                                                                      // Interrupt wieder freigeben

// DCF77-Auswertung -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

      uint8_t errcnt;                                                                              // Fehlerzhler
      uint8_t parcnt;                                                                              // Zhler fr die Parittsprfung der Daten
      uint8_t dcftmp;                                                                              // Zwischenspeicher fr den DCF77-Impulszhler
      uint8_t zeroflg;                                                                             // Nullbyte-Flag (wenn DCF77-Puffer nur 0-Werte enthlt)

      if (dcfstat2)                                                                                // wenn ein DCF77-Minutenzyklus komplett ist
      {
         cli ();                                                                                   // Interrupts sperren
         dcfstat2 = 0;                                                                             // Flag fr Minutenzyklus lschen
         dcftmp = dcfimp;                                                                          // DCF77-Impulszhler kopieren
         dcfimp = 0;                                                                               // DCF77-Impulszhler lschen
         sei ();                                                                                   // Interrupts wieder freigeben

         if (dcftmp == 59)                                                                         // wenn 59 Impulse empfangen worden sind
         {

            errcnt = 0;                                                                            // Fehlerzhler lschen
            parcnt = 0;                                                                            // Parittszhler lschen
            for (i = 4; i < 12; i ++)                                                              // Minuten-Bits (im Byte 04-11) prfen
            {
               if (dcftab[i]) parcnt ++;                                                           // wenn Bit=1 -> Parittszhler erhhen
            }
            if (parcnt & 0x01) errcnt ++;                                                          // wenn Minuten-Prfsumme ungeradzahlig -> Fehler

            parcnt = 0;                                                                            // Parittszhler lschen
            for (i = 12; i < 19; i ++)                                                             // Stunden-Bits (im Byte 12-18) prfen
            {
               if (dcftab[i]) parcnt ++;                                                           // wenn Bit=1 -> Parittszhler erhhen
            }
            if (parcnt & 0x01) errcnt ++;                                                          // wenn Stunden-Prfsumme ungeradzahlig -> Fehler

            parcnt = 0;                                                                            // Parittszhler lschen
            for (i = 19; i < 42; i ++)                                                             // Datum-Bits (im Byte 19-41) prfen
            {
               if (dcftab[i]) parcnt ++;                                                           // wenn Bit=1 -> Parittszhler erhhen
            }
            if (parcnt & 0x01) errcnt ++;                                                          // wenn Datum-Prfsumme ungeradzahlig -> Fehler

            if (errcnt == 0)                                                                       // wenn alle Prfbits ok -> Zeit ermitteln
            {
               dcfbuf1[0] = dcftab[0];                                                             // Zeitzone speichern

               dcfbuf1[1] = dcftab[4] +                                                            // Minuten-Bits zusammenfgen
                            (dcftab[5] * 2) +
                            (dcftab[6] * 4) +
                            (dcftab[7] * 8) +
                            (dcftab[8] * 10) +
                            (dcftab[9] * 20) +
                            (dcftab[10] * 40);                                                     // Minuten speichern

               dcfbuf1[2] = dcftab[12] +                                                           // Stunden-Bits zusammenfgen
                            (dcftab[13] * 2) +
                            (dcftab[14] * 4) +
                            (dcftab[15] * 8) +
                            (dcftab[16] * 10) +
                            (dcftab[17] * 20);                                                     // Stunden speichern

               dcfbuf1[3] = dcftab[19] +                                                           // Kalendertag-Bits zusammenfgen
                            (dcftab[20] * 2) +
                            (dcftab[21] * 4) +
                            (dcftab[22] * 8) +
                            (dcftab[23] * 10) +
                            (dcftab[24] * 20);                                                     // Kalendertag speichern

               dcfbuf1[4] = dcftab[25] +                                                           // Wochentag-Bits zusammenfgen
                            (dcftab[26] * 2) +
                            (dcftab[27] * 4);                                                      // Wochentag speichern

               dcfbuf1[5] = dcftab[28] +                                                           // Monat-Bits zusammenfgen
                            (dcftab[29] * 2) +
                            (dcftab[30] * 4) +
                            (dcftab[31] * 8) +
                            (dcftab[32] * 10);                                                     // Monat speichern

               dcfbuf1[6] = dcftab[33] +                                                           // Jahr-Bits zusammenfgen
                            (dcftab[34] * 2) +
                            (dcftab[35] * 4) +
                            (dcftab[36] * 8) +
                            (dcftab[37] * 10) +
                            (dcftab[38] * 20) +
                            (dcftab[39] * 40) +
                            (dcftab[40] * 80);                                                     // Jahr speichern

               zeroflg = 0;                                                                        // Nullbyte-Flag lschen
               for (i = 0; i < 7; i ++)                                                            // 7 Puffer-Bytes vergleichen
               {
                  zeroflg = zeroflg | dcfbuf1[i];                                                  // Puffer 1 mit Flag verknpfen
                  if (dcfbuf1[i] != dcfbuf2[i]) errcnt ++;                                         // Vergleich Puffer 1 mit Puffer 2
                  if (dcfbuf1[i] != dcfbuf3[i]) errcnt ++;                                         // Vergleich Puffer 1 mit Puffer 3
                  dcfbuf3[i] = dcfbuf2[i];                                                         // Kopieren Puffer 2 -> Puffer 3
                  dcfbuf2[i] = dcfbuf1[i];                                                         // Kopieren Puffer 1 -> Puffer 2
               }
               if (zeroflg == 0) errcnt ++;                                                        // wenn alle Puffer-Bytes = 0 -> Fehler

               dcfbuf2[1] ++;                                                                      // Puffer 2 um eine Minute erhhen
               if (dcfbuf2[1] > 59)                                                                // wenn berlauf
               {
                  dcfbuf2[1] = 0;                                                                  // Minuten zurcksetzen
                  dcfbuf2[2] ++;                                                                   // Stunde erhhen
               }

               dcfbuf3[1] ++;                                                                      // Puffer 3 um eine Minute erhhen
               if (dcfbuf3[1] > 59)                                                                // wenn berlauf
               {
                  dcfbuf3[1] = 0;                                                                  // Minuten zurcksetzen
                  dcfbuf3[2] ++;                                                                   // Stunde erhhen
               }
            }

            if (errcnt == 0)                                                                       // wenn gesamte Prozedur fehlerfrei -> Uhr stellen
            {
               cli ();                                                                             // Interrupts sperren
               timezone = dcfbuf1[0];                                                              // Zeitzone setzen
               minute = dcfbuf1[1];                                                                // Minute setzen
               hour = dcfbuf1[2];                                                                  // Stunde setzen
               day = dcfbuf1[3];                                                                   // Kalendertag setzen
               wday = dcfbuf1[4];                                                                  // Wochentag setzen
               month = dcfbuf1[5];                                                                 // Monat setzen
               year = dcfbuf1[6];                                                                  // Jahr setzen
               second = 0;                                                                         // Sekunden zurcksetzen
               intc20ms = 0;                                                                       // Interrupt-Zhler 3 (20ms) zurcksetzen
               syncstat = 4;                                                                       // Uhr ist jetzt synchronisiert
               sec_flag = 1;                                                                       // Sekunden-Flag setzen, Analoguhr und Symbole neu ausgeben
               chg_flag = 1;                                                                       // nderungs-Flag setzen, Informationen neu ausgeben
               dcftout = c_dcftout;                                                                // DCF-Timeout neu starten
               sei ();                                                                             // Interrupts wieder freigeben

               lastsync[0] = 1;                                                                    // letzte DCF-Synchronisierungszeit ist jetzt gltig
               for (i = 1; i < 7; i ++)                                                            // 6 Pufferbytes kopieren
               {
                  lastsync[i] = dcfbuf1[i];                                                        // aktuelle Zeit als letzte Synchronisierungszeit speichern
               }
               if (!lcd_is_on) lcd_on ();                                                          // wenn LCD ausgeschaltet war -> LCD wieder einschalten
            }
         }
      }

// USB-Kommunikation ber USART0 und USART1 -----------------------------------------------------------------------------------------------------------------------------------------------------------

      char rx_char;                                                                                // Zwischenspeicher fr empfangenes Zeichen

                                                                                                   // Datenempfang auf USART0 -------------------------------------------------------------------------

      while (rx0write != rx0read)                                                                  // solange Schreib- und Lese-Zeiger unterschiedlich sind
      {
         rx_char = rx0buf[rx0read];                                                                // empfangenes Zeichen aus USART0-Puffer holen
         rx0read ++;                                                                               // Lese-Zeiger erhhen
         if ((rx0pos < sizeof (rx0data)) && (rx_char >= 32))                                       // wenn Pufferende noch nicht erreicht und kein Steuerzeichen
         {
            rx0data[rx0pos] = rx_char;                                                             // empfangenes Zeichen in RX1-Datenpuffer legen
            rx0pos ++;                                                                             // Zeichenzhler erhhen
         }
         if (rx_char == '\r')                                                                      // wenn empfangenes Zeichen = Carriage Return
         {
            if ((cmdport == 0) && (rx0pos > 1))                                                    // wenn Kommandopuffer im Ruhezustand und mindestens zwei Zeichen empfangen wurden
            {
               for (i = 0; i < sizeof (rx0data); i ++)                                             // alle Zeichen vom RX1-Datenpuffer kopieren
               {
                  if (i < rx0pos) cmdbuf[i] = rx0data[i];                                          // wenn gltiges Zeichen -> in Kommandopuffer kopieren
                  else cmdbuf[i] = 255;                                                            // sonst Kommandopuffer mit 0xff auffllen
               }
               usb_on ();                                                                          // USB-Kommunikation ist jetzt aktiv
               cmdport = 1;                                                                        // neues Kommando von Port 1 (USART0) bearbeiten
            }
            rx0pos = 0;                                                                            // RX0-Datenpuffer wieder lschen
         }
      }
                                                                                                   // Datenempfang auf USART1 -------------------------------------------------------------------------

      while (rx1write != rx1read)                                                                  // solange Schreib- und Lese-Zeiger unterschiedlich sind
      {
         rx_char = rx1buf[rx1read];                                                                // empfangenes Zeichen aus USART1-Puffer holen
         rx1read ++;                                                                               // Lese-Zeiger erhhen
         if ((rx1pos < sizeof (rx1data)) && (rx_char >= 32))                                       // wenn Pufferende noch nicht erreicht und kein Steuerzeichen
         {
            rx1data[rx1pos] = rx_char;                                                             // empfangenes Zeichen in RX1-Datenpuffer legen
            rx1pos ++;                                                                             // Zeichenzhler erhhen
         }
         if (rx_char == '\r')                                                                      // wenn empfangenes Zeichen = Carriage Return
         {
            if ((cmdport == 0) && (rx1pos > 1))                                                    // wenn Kommandopuffer im Ruhezustand und mindestens zwei Zeichen empfangen wurden
            {
               for (i = 0; i < sizeof (rx1data); i ++)                                             // alle Zeichen vom RX1-Datenpuffer kopieren
               {
                  if (i < rx1pos) cmdbuf[i] = rx1data[i];                                          // wenn gltiges Zeichen -> in Kommandopuffer kopieren
                  else cmdbuf[i] = 255;                                                            // sonst Kommandopuffer mit 0xff auffllen
               }
               usb_on ();                                                                          // USB-Kommunikation ist jetzt aktiv
               cmdport = 2;                                                                        // neues Kommando von Port 2 (USART1) bearbeiten
            }
            rx1pos = 0;                                                                            // RX1-Datenpuffer wieder lschen
         }
      }
                                                                                                   // Empfangenes Kommando bearbeiten -----------------------------------------------------------------

      if (cmdport > 0)                                                                             // wenn neues Kommando bearbeitet werden muss
      {
         switch (lowcase (cmdbuf[0]))                                                              // weitere Bearbeitung abhngig vom Kommando
         {
            case 'v':                                                                              // wenn Kommando 'v' erkannt (Version)
               if (cmdbuf[1] == '?')                                                               // wenn anschlieendes Fragezeichen erkannt (Info anfordern)
               {
                  usb_char ('v', cmdport);                                                         // Antwort 'v' senden
                  usb_char (':', cmdport);                                                         // Trennzeichen ':' senden
                  usb_str (0, 0, cmdport);                                                         // Versionsnummer und -Datum senden
                  usb_char ('\r', cmdport);                                                        // Carriage Return senden
                  usb_char ('\n', cmdport);                                                        // Line Feed senden
               }               
               cmdport = 0;                                                                        // Kommando beenden
               break;
                                                                                                   // Kommando 't' (Zeit) -----------------------------------------------------------------------------

            case 't':                                                                              // wenn Kommando 't' erkannt (Zeit)
               if (cmdbuf[1] == '?')                                                               // wenn anschlieendes Fragezeichen erkannt (Zeit anfordern)
               {
                  usb_char ('t', cmdport);                                                         // Antwort 't' senden
                  usb_char (':', cmdport);                                                         // Trennzeichen ':' senden
                  usb_dec2 (hour, cmdport);                                                        // Stunden senden
                  usb_char (':', cmdport);                                                         // Trennzeichen ':' senden
                  usb_dec2 (minute, cmdport);                                                      // Minuten senden
                  usb_char (':', cmdport);                                                         // Trennzeichen ':' senden
                  usb_dec2 (second, cmdport);                                                      // Sekunden senden
                  usb_char (',', cmdport);                                                         // Trennzeichen ',' senden
                  usb_dec2 (c_yearh, cmdport);                                                     // Jahrhundert senden
                  usb_dec2 (year, cmdport);                                                        // Jahr senden
                  usb_char ('-', cmdport);                                                         // Trennzeichen '-' senden
                  usb_dec2 (month, cmdport);                                                       // Monat senden
                  usb_char ('-', cmdport);                                                         // Trennzeichen '-' senden
                  usb_dec2 (day, cmdport);                                                         // Tag senden
                  usb_char (',', cmdport);                                                         // Trennzeichen ',' senden
                  usb_char (timezone + 48, cmdport);                                               // Zeitzone senden
                  usb_char ('\r', cmdport);                                                        // Carriage Return senden
                  usb_char ('\n', cmdport);                                                        // Line Feed senden
               }
               if (cmdbuf[1] == ':')                                                               // wenn anschlieendes Trennzeichen erkannt (Zeit setzen)
               {
                  okflag = 1;                                                                      // Plausibilitt zunchst auf ok setzen
                  t_hour = read_dec2 (2);                                                          // Stunden aus Puffer lesen
                  if (t_hour > 23) okflag = 0;                                                     // wenn Stunden fehlerhaft -> nicht plausibel
                  if (cmdbuf[4] != ':') okflag = 0;                                                // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  t_minute = read_dec2 (5);                                                        // Minuten aus Puffer lesen
                  if (t_minute > 59) okflag = 0;                                                   // wenn Minuten fehlerhaft -> nicht plausibel
                  if (cmdbuf[7] != ':') okflag = 0;                                                // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  t_second = read_dec2 (8);                                                        // Sekunden aus Puffer lesen
                  if (t_second > 59) okflag = 0;                                                   // wenn Sekunden fehlerhaft -> nicht plausibel
                  if (cmdbuf[10] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  if (read_dec2 (11) != c_yearh) okflag = 0;                                       // wenn Jahrhundert fehlerhaft -> nicht plausibel
                  t_year = read_dec2 (13);                                                         // Jahr aus Puffer lesen
                  if (t_year > 99) okflag = 0;                                                     // wenn Jahr fehlerhaft -> nicht plausibel
                  if (cmdbuf[15] != '-') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  t_month = read_dec2 (16);                                                        // Monat aus Puffer lesen
                  if ((t_month < 1) || (t_month > 12)) okflag = 0;                                 // wenn Monat fehlerhaft -> nicht plausibel
                  if (cmdbuf[18] != '-') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  t_day = read_dec2 (19);                                                          // Tag aus Puffer lesen
                  if (t_day < 1) okflag = 0;                                                       // wenn Tag kleiner 1 -> nicht plausibel
                  if (okflag)                                                                      // wenn bisher alles plausibel
                  {
                     if ((t_year % 4) == 0)                                                        // wenn Schaltjahr
                     {
                        if (t_day > pgm_read_byte (&daytab2[t_month - 1])) okflag = 0;             // wenn Tag grer als Schaltjahr-Tabelle -> nicht plausibel
                     }
                     else                                                                          // wenn kein Schaltjahr
                     {
                        if (t_day > pgm_read_byte (&daytab1[t_month - 1])) okflag = 0;             // wenn Tag grer als Normaljahr-Tabelle -> nicht plausibel
                     }
                  }
                  if (cmdbuf[21] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  t_zone = read_dec1 (22);                                                         // Zeitzone aus Puffer lesen
                  if (t_zone > 1) okflag = 0;                                                      // wenn Zeitzone fehlerhaft -> nicht plausibel
                  if (cmdbuf[23] < 255) okflag = 0;                                                // wenn weitere Zeichen folgen -> nicht plausibel

                  if (okflag)                                                                      // wenn alle Daten plausibel
                  {
                     sec_flag = 1;                                                                 // Sekunden-Flag setzen, Analoguhr neu ausgeben
                     chg_flag = 1;                                                                 // nderungs-Flag setzen, Informationsbereich neu ausgeben
                     cli ();                                                                       // Interrupts sperren
                     intc20ms = 0;                                                                 // Interrupt-Zhler 3 (20ms) zurcksetzen
                     second = t_second;                                                            // Sekunden aus Zwischenspeicher holen
                     minute = t_minute;                                                            // Minuten aus Zwischenspeicher holen
                     hour = t_hour;                                                                // Stunden aus Zwischenspeicher holen
                     day = t_day;                                                                  // Kalendertag aus Zwischenspeicher holen
                     month = t_month;                                                              // Monat aus Zwischenspeicher holen
                     year = t_year;                                                                // Jahr aus Zwischenspeicher holen
                     timezone = t_zone;                                                            // Zeitzone aus Zwischenspeicher holen
                     syncstat = 4;                                                                 // Uhr ist jetzt synchronisiert
                     dcftout = c_dcftout;                                                          // DCF-Timeout neu starten
                     sei ();                                                                       // Interrupts wieder freigeben
                     wday = calc_wday (day, month, c_yearh, year);                                 // Wochentag berechnen
                     if (!lcd_is_on) lcd_on ();                                                    // wenn LCD ausgeschaltet war -> LCD wieder einschalten
                  }
                  usb_char ('t', cmdport);                                                         // Antwort 't' senden
                  if (okflag) usb_char ('0', cmdport);                                             // wenn Zeit plausibel -> Zeichen '0' senden
                  else usb_char ('1', cmdport);                                                    // sonst Zeichen '1' senden
                  usb_char ('\r', cmdport);                                                        // Carriage Return senden
                  usb_char ('\n', cmdport);                                                        // Line Feed senden
               }
               cmdport = 0;                                                                        // Kommando beenden
               break;
                                                                                                   // Kommando '1', '2' und '3' (Alarm) ---------------------------------------------------------------

            case '1' ... '3':                                                                      // wenn Kommando '1', '2' oder '3' erkannt (Alarm)
               if (cmdbuf[1] == '?')                                                               // wenn anschlieendes Fragezeichen erkannt (Info anfordern)
               {
                  usb_char (cmdbuf[0], cmdport);                                                   // Antwort '1', '2' oder '3' senden
                  usb_char (':', cmdport);                                                         // Trennzeichen ':' senden
                  usb_char (alarms[cmdbuf[0] - 49][0] + 48, cmdport);                              // Alarm-Typ senden
                  usb_char (',', cmdport);                                                         // Trennzeichen ',' senden
                  usb_dec2 (alarms[cmdbuf[0] - 49][2], cmdport);                                   // Alarm-Stunde senden
                  usb_char (':', cmdport);                                                         // Trennzeichen ':' senden
                  usb_dec2 (alarms[cmdbuf[0] - 49][1], cmdport);                                   // Alarm-Minute senden
                  usb_char (',', cmdport);                                                         // Trennzeichen ',' senden
                  usb_dec2 (c_yearh, cmdport);                                                     // Alarm-Jahrhundert senden
                  usb_dec2 (alarms[cmdbuf[0] - 49][6], cmdport);                                   // Alarm-Jahr senden
                  usb_char ('-', cmdport);                                                         // Trennzeichen '-' senden
                  usb_dec2 (alarms[cmdbuf[0] - 49][5], cmdport);                                   // Alarm-Monat senden
                  usb_char ('-', cmdport);                                                         // Trennzeichen '-' senden
                  usb_dec2 (alarms[cmdbuf[0] - 49][3], cmdport);                                   // Alarm-Tag senden
                  usb_char (',', cmdport);                                                         // Trennzeichen ',' senden
                  usb_char (alarms[cmdbuf[0] - 49][4] + 48, cmdport);                              // Alarm-Wochentag senden
                  usb_char (',', cmdport);                                                         // Trennzeichen ',' senden
                  usb_char (alarms[cmdbuf[0] - 49][7] + 48, cmdport);                              // Alarm-Wochentagsprogramm senden
                  usb_char (',', cmdport);                                                         // Trennzeichen ',' senden
                  usb_dec2 (alarms[cmdbuf[0] - 49][8], cmdport);                                   // Alarm-Soundnummer senden
                  usb_char (',', cmdport);                                                         // Trennzeichen ',' senden
                  usb_dec2 (alarms[cmdbuf[0] - 49][9], cmdport);                                   // Alarm-Lautstrke senden
                  usb_char ('\r', cmdport);                                                        // Carriage Return senden
                  usb_char ('\n', cmdport);                                                        // Line Feed senden
               }
               if (cmdbuf[1] == ':')                                                               // wenn anschlieendes Trennzeichen erkannt (Alarm setzen)
               {
                  okflag = 1;                                                                      // Plausibilitt zunchst auf ok setzen
                  alarms[4][0] = read_dec1 (2);                                                    // Alarm-Typ aus Puffer lesen
                  if (alarms[4][0] > 4) okflag = 0;                                                // wenn Alarm-Typ fehlerhaft -> nicht plausibel
                  if (cmdbuf[3] != ',') okflag = 0;                                                // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  alarms[4][2] = read_dec2 (4);                                                    // Stunde aus Puffer lesen
                  if (alarms[4][2] > 23) okflag = 0;                                               // wenn Stunde fehlerhaft -> nicht plausibel
                  if (cmdbuf[6] != ':') okflag = 0;                                                // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  alarms[4][1] = read_dec2 (7);                                                    // Minute aus Puffer lesen
                  if (alarms[4][1] > 59) okflag = 0;                                               // wenn Minute fehlerhaft -> nicht plausibel
                  if (cmdbuf[9] != ',') okflag = 0;                                                // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  if (read_dec2 (10) != c_yearh) okflag = 0;                                       // wenn Jahrhundert fehlerhaft -> nicht plausibel
                  alarms[4][6] = read_dec2 (12);                                                   // Jahr aus Puffer lesen
                  if (alarms[4][6] > 99) okflag = 0;                                               // wenn Jahr fehlerhaft -> nicht plausibel
                  if (cmdbuf[14] != '-') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  alarms[4][5] = read_dec2 (15);                                                   // Monat aus Puffer lesen
                  if ((alarms[4][5] < 1) || (alarms[4][5] > 12)) okflag = 0;                       // wenn Monat fehlerhaft -> nicht plausibel
                  if (cmdbuf[17] != '-') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  alarms[4][3] = read_dec2 (18);                                                   // Tag aus Puffer lesen
                  if (alarms[4][3] < 1) okflag = 0;                                                // wenn Tag kleiner 1 -> nicht plausibel
                  if (okflag)                                                                      // wenn bisher alles plausibel
                  {
                     if ((alarms[4][6] % 4) == 0)                                                  // wenn Schaltjahr
                     {
                        if (alarms[4][3] > pgm_read_byte (&daytab2[alarms[4][5] - 1])) okflag = 0; // wenn Tag grer als Schaltjahr-Tabelle -> nicht plausibel
                     }
                     else                                                                          // wenn kein Schaltjahr
                     {
                        if (alarms[4][3] > pgm_read_byte (&daytab1[alarms[4][5] - 1])) okflag = 0; // wenn Tag grer als Normaljahr-Tabelle -> nicht plausibel
                     }
                  }
                  if (cmdbuf[20] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  alarms[4][4] = read_dec1 (21);                                                   // Wochentag aus Puffer lesen
                  if ((alarms[4][4] < 1) || (alarms[4][4] > 7)) okflag = 0;                        // wenn Wochentag fehlerhaft -> nicht plausibel
                  if (cmdbuf[22] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  alarms[4][7] = read_dec1 (23);                                                   // Wochentagsprogramm aus Puffer lesen
                  if (alarms[4][7] > 9) okflag = 0;                                                // wenn Wochentagsprogramm fehlerhaft -> nicht plausibel
                  if (cmdbuf[24] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  alarms[4][8] = read_dec2 (25);                                                   // Sound-Nummer aus Puffer lesen
                  if (alarms[4][8] > 12) okflag = 0;                                               // wenn Sound-Nummer fehlerhaft -> nicht plausibel
                  if (cmdbuf[27] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  alarms[4][9] = read_dec2 (28);                                                   // Lautstrke aus Puffer lesen
                  if (alarms[4][9] > 10) okflag = 0;                                               // wenn Lautstrke fehlerhaft -> nicht plausibel
                  if (cmdbuf[30] < 255) okflag = 0;                                                // wenn weitere Zeichen folgen -> nicht plausibel

                  if (okflag)                                                                      // wenn alle Daten plausibel
                  {
                     alarms[4][4] = calc_wday (alarms[4][3], alarms[4][5], c_yearh, alarms[4][6]); // Wochentag berschreiben
                     for (i = 0; i < 10; i ++)                                                     // Alarm-Parameter 0-9 speichern
                     {
                        alarms[cmdbuf[0] - 49][i] = alarms[4][i];                                  // Parameter in Tabelle kopieren
                        eep_write_byte (e_alarms + ((cmdbuf[0] - 49) * 10 + i), alarms[4][i]);     // Parameter im EEPROM speichern
                     }
                     alarms[cmdbuf[0] - 49][10] = 0;                                               // Alarm nicht ausgelst oder quittiert
                  }                  
                  usb_char (cmdbuf[0], cmdport);                                                   // Antwort '1', '2' oder '3' senden
                  if (okflag) usb_char ('0', cmdport);                                             // wenn Alarm plausibel -> Zeichen '0' senden
                  else usb_char ('1', cmdport);                                                    // sonst Zeichen '1' senden
                  usb_char ('\r', cmdport);                                                        // Carriage Return senden
                  usb_char ('\n', cmdport);                                                        // Line Feed senden
               }
               cmdport = 0;                                                                        // Kommando beenden
               break;

            case 's':                                                                              // wenn Kommando 's' erkannt (Screenshot)
               if (cmdbuf[1] == '?')                                                               // wenn anschlieendes Fragezeichen erkannt (Screenshot anfordern)
               {
                  scrshot = 1;                                                                     // Screenshot starten
                  sec_flag = 1;                                                                    // Sekunden-Flag setzen (Analoguhr neu ausgeben)
                  chg_flag = 1;                                                                    // nderungs-Flag setzen (Info-Bereich neu ausgeben)
               }
               else cmdport = 0;                                                                   // wenn kein Fragezeichen -> Kommando beenden
               break;

            case 'b':                                                                              // wenn Kommando 'b' erkannt (Geburtstage)
               if (cmdbuf[1] == '?')                                                               // wenn anschlieendes Fragezeichen erkannt (Geburtstage anfordern)
               {
                  k = 1;                                                                           // Merker fr leere Geburtstagsliste setzen
                  for (i = 0; i < 115; i ++)                                                       // Schleife, alle Geburtstagseintrge aus EEPROM lesen
                  {
                     m = eep_read_byte (e_bddata + (i * 34));                                      // Tag lesen
                     if (m < 255)                                                                  // wenn gltiger Eintrag
                     {
                        k = 0;                                                                     // Merker fr leere Geburtstagsliste lschen
                        usb_char ('b', cmdport);                                                   // Antwort 'b' senden
                        usb_char (':', cmdport);                                                   // Trennzeichen ':' senden
                        n = eep_read_byte (e_bddata + (i * 34) + 3);                               // Jahrhundert lesen
                        if (n < 100)                                                               // wenn gltiges Jahrhundert
                        {
                           usb_dec2 (n, cmdport);                                                  // Jahrhundert senden
                           usb_dec2 (eep_read_byte(e_bddata + (i * 34) + 2), cmdport);             // Jahr senden
                        }                        
                        else                                                                       // sonst
                        {
                           usb_char ('?', cmdport);
                           usb_char ('?', cmdport);
                           usb_char ('?', cmdport);
                           usb_char ('?', cmdport);                                                // Jahrhundert und Jahr als Fragezeichen senden
                        }
                        usb_char ('-', cmdport);                                                   // Trennzeichen '-' senden
                        usb_dec2 (eep_read_byte (e_bddata + (i * 34 + 1)), cmdport);               // Monat senden
                        usb_char ('-', cmdport);                                                   // Trennzeichen '-' senden
                        usb_dec2 (m, cmdport);                                                     // Tag senden
                        usb_char (',', cmdport);                                                   // Trennzeichen ',' senden
                        for (j = 4; j < 19; j ++)                                                  // Schleife fr Vorname, 15 Zeichen lesen
                        {
                           n = eep_read_byte(e_bddata + (i * 34 + j));                             // Zeichen lesen
                           if (n < 255) usb_char (n, cmdport);                                     // wenn gltiges Zeichen -> senden
                        }                        
                        usb_char (',', cmdport);                                                   // Trennzeichen ',' senden
                        for (j = 19; j < 34; j ++)                                                 // Schleife fr Nachname, 15 Zeichen lesen
                        {
                           n = eep_read_byte(e_bddata + (i * 34 + j));                             // Zeichen lesen
                           if (n < 255) usb_char (n, cmdport);                                     // wenn gltiges Zeichen -> senden
                        }
                        usb_char ('\r', cmdport);                                                  // Carriage Return senden
                        usb_char ('\n', cmdport);                                                  // Line Feed senden
                     }
                  }
                  if (k)                                                                           // wenn leere Geburtsliste
                  {
                     usb_char ('b', cmdport);                                                      // Antwort 'b' senden
                     usb_char (':', cmdport);                                                      // Trennzeichen ':' senden
                     usb_char ('\r', cmdport);                                                     // Carriage Return senden
                     usb_char ('\n', cmdport);                                                     // Line Feed senden
                  }
               }

               if (cmdbuf[1] == ':')                                                               // wenn anschlieendes Trennzeichen erkannt (Geburtstag setzen)
               {
                  okflag = 1;                                                                      // Plausibilitt zunchst auf ok setzen
                  t_hour = read_dec2 (2);                                                          // Jahrhundert aus Puffer lesen
                  t_year = read_dec2 (4);                                                          // Jahr aus Puffer lesen

                  if ((t_hour > 99) || (t_year > 99))                                              // wenn Jahrhundert oder Jahr fehlerhaft
                  {
                     if ((cmdbuf[2] == '?') && (cmdbuf[3] == '?') && (cmdbuf[4] == '?') && (cmdbuf[5] == '?')) // wenn unbekanntes Jahr
                     {
                        t_hour = 255;                                                              // Jahrhundert auf unbekannt setzen
                     }
                     else okflag = 0;                                                              // sonst Jahr nicht plausibel
                  }
                  if (cmdbuf[6] != '-') okflag = 0;                                                // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  t_month = read_dec2 (7);                                                         // Monat aus Puffer lesen
                  if ((t_month < 1) || (t_month > 12)) okflag = 0;                                 // wenn Monat fehlerhaft -> nicht plausibel
                  if (cmdbuf[9] != '-') okflag = 0;                                                // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  t_day = read_dec2 (10);                                                          // Tag aus Puffer lesen

                  if (okflag)                                                                      // wenn bisher alles plausibel
                  {
                     if ((t_year % 4) == 0)                                                        // wenn Schaltjahr
                     {
                        if (t_day > pgm_read_byte (&daytab2[t_month - 1])) okflag = 0;             // wenn Tag grer als Schaltjahr-Tabelle -> nicht plausibel
                     }
                     else                                                                          // wenn kein Schaltjahr
                     {
                        if (t_day > pgm_read_byte (&daytab1[t_month - 1])) okflag = 0;             // wenn Tag grer als Normaljahr-Tabelle -> nicht plausibel
                     }
                  }
                  if (cmdbuf[12] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  m = 0;                                                                           // Anfangswert fr Vorname-Ende-Position lschen
                  n = 0;                                                                           // Anfangswert fr Nachname-Ende-Position lschen
                  for (i = 13; i < 29; i ++)                                                       // Vornamen-Bereich prfen
                  {
                     if ((cmdbuf[i] > 0x90) || (cmdbuf[i] < 0x21))                                 // wenn Zeichencode > 0x90 oder Steuerzeichen (ungltig)
                     {
                        okflag = 0;                                                                // nicht plausibel
                        break;                                                                     // Schleife abbrechen
                     }
                     if (cmdbuf[i] == ',')                                                         // wenn Komma gefunden
                     {
                        m = i - 1;                                                                 // Ende-Position setzen
                        break;                                                                     // Schleife abbrechen
                     }
                  }
                  if (m == 0) okflag = 0;                                                          // wenn keine Ende-Position gefunden
                  if (okflag)                                                                      // wenn bisher alles plausibel
                  {
                     for (i = m + 2; i < m + 17; i ++)                                             // Nachnamen-Bereich prfen
                     {
                        if ((cmdbuf[i] > 0x90) || (cmdbuf[i] < 0x21))                              // wenn Zeichencode > 0x90 oder Steuerzeichen (ungltig)
                        {
                           n = i - 1;                                                              // Ende-Position setzen
                           break;                                                                  // Schleife abbrechen
                        }
                     }
                     if (n == 0) n = m + 16;                                                       // wenn zu viele Zeichen -> Ende auf 15 Zeichen setzen
                  }
                  if (okflag)                                                                      // wenn bisher alles plausibel
                  {
                     if ((m < 13) && (n < m + 2)) okflag = 0;                                      // wenn kein Vor- und kein Nachname vorhanden -> nicht plausibel
                  }

                  if (okflag)                                                                      // wenn alle Daten plausibel
                  {
                     for (i = 0; i < 115; i ++)                                                    // Geburtstagsliste durchsuchen
                     {
                        for (j = 0; j < 15; j ++)                                                  // zuerst 15 Zeichen des Vornamens vergleichen
                        {
                           if ((j + 13) > m) c = 255;                                              // wenn Ende des Vornamens erreicht -> Zeichen = 255 setzen
                           else c = cmdbuf[j + 13];                                                // sonst Zeichen aus Kommandopuffer holen
                           if (c != eep_read_byte(e_bddata + (i * 34 + j + 4))) break;             // wenn Zeichen nicht mit Geburtstagseintrag bereinstimmt -> Abbruch
                        }
                        if (j == 15)                                                               // wenn Schleife ohne Abbruch durchgelaufen -> Vorname ok
                        {
                           for (j = 0; j < 15; j ++)                                               // jetzt 15 Zeichen des Nachnamens vergleichen
                           {
                              if ((j + m + 2) > n) c = 255;                                        // wenn Ende des Nachnamens erreicht -> Zeichen = 255 setzen
                              else c = cmdbuf[j + m + 2];                                          // sonst Zeichen aus Kommandopuffer holen
                              if (c != eep_read_byte(e_bddata + (i * 34 + j + 19))) break;         // wenn Zeichen nicht mit Geburtstagseintrag bereinstimmt -> Abbruch
                           }
                           if (j == 15) break;                                                     // wenn Schleife ohne Abbruch durchgelaufen -> Nachname ebenfalls ok
                        }                                                                          // i-Schleife wird abgebrochen und i enthlt Nummer des Eintrages
                     }

                     if (i < 115)                                                                  // wenn Eintrag gefunden -> berschreiben oder lschen
                     {
                        if (t_day > 0)                                                             // wenn Tag > 0 (Eintrag berschreiben)
                        {
                           eep_write_byte (e_bddata + (i * 34), t_day);                            // Geburtstag speichern
                           eep_write_byte (e_bddata + (i * 34 + 1), t_month);                      // Geburtsmonat speichern
                           eep_write_byte (e_bddata + (i * 34 + 2), t_year);                       // Geburtsjahr speichern
                           eep_write_byte (e_bddata + (i * 34 + 3), t_hour);                       // Geburtsjahrhundert speichern
                        }
                        else                                                                       // wenn Tag = 0 (Eintrag lschen)
                        {
                           for (j = 0; j < 34; j ++)                                               // 34 Bytes lschen
                           {
                              eep_write_byte (e_bddata + (i * 34 + j), 255);                       // Geburtstags-Eintrag lschen
                           }
                        }
                     }
                     else                                                                          // wenn Eintrag nicht gefunden -> neu anlegen
                     {
                        if (t_day > 0)                                                             // wenn Tag > 0 (neuen Eintrag anlegen)
                        {
                           for (i = 0; i < 115; i ++)                                              // Geburtstagsliste durchsuchen
                           {
                              if (eep_read_byte (e_bddata + (i * 34)) > 31)                        // wenn freier Eintrag gefunden
                              {
                                 eep_write_byte (e_bddata + (i * 34), t_day);                      // Geburtstag speichern
                                 eep_write_byte (e_bddata + (i * 34 + 1), t_month);                // Geburtsmonat speichern
                                 eep_write_byte (e_bddata + (i * 34 + 2), t_year);                 // Geburtsjahr speichern
                                 eep_write_byte (e_bddata + (i * 34 + 3), t_hour);                 // Geburtsjahrhundert speichern
                                 for (j = 0; j < 15; j ++)                                         // 15 Zeichen Vor- und Nachname bearbeiten
                                 {
                                    if ((j + 13) > m) c = 255;                                     // wenn Ende des Vornamens erreicht -> Zeichen = 255 setzen
                                    else c = cmdbuf[j + 13];                                       // sonst Zeichen aus Kommandopuffer holen
                                    eep_write_byte (e_bddata + (i * 34 + j + 4), c);               // Zeichen vom Vornamen speichern
                                    if ((j + m + 2) > n) c = 255;                                  // wenn Ende des Nachnamens erreicht -> Zeichen = 255 setzen
                                    else c = cmdbuf[j + m + 2];                                    // sonst Zeichen aus Kommandopuffer holen
                                    eep_write_byte (e_bddata + (i * 34 + j + 19), c);              // Zeichen vom Nachnamen speichern
                                 }
                                 break;                                                            // Suche abbrechen
                              }
                           }
                           if (i == 115) okflag = 0;                                               // wenn kein freier Eintrag gefunden -> Fehler
                        }
                        else okflag = 0;                                                           // wenn Tag = 0 -> zu lschender Eintrag wurde nicht gefunden
                     }
                  }

                  bd_flag = 1;                                                                     // Geburtstage reorganisieren
                  usb_char ('b', cmdport);                                                         // Antwort 'b' senden
                  if (okflag) usb_char ('0', cmdport);                                             // wenn Geburtstag plausibel -> Zeichen '0' senden
                  else usb_char ('1', cmdport);                                                    // sonst Zeichen '1' senden
                  usb_char ('\r', cmdport);                                                        // Carriage Return senden
                  usb_char ('\n', cmdport);                                                        // Line Feed senden
               }
               cmdport = 0;                                                                        // Kommando beenden
               break;

            case 'c':                                                                              // wenn Kommando 'c' erkannt (Konfiguration)
               if (cmdbuf[1] == '?')                                                               // wenn anschlieendes Fragezeichen erkannt (Konfiguration anfordern)
               {
                  usb_char ('c', cmdport);                                                         // Antwort 'c' senden
                  usb_char (':', cmdport);                                                         // Trennzeichen ':' senden
                  usb_dec3 ((lang << 7) | (lcdinv << 6) | (aclkpos << 5) | (dcfinv << 4) |
                           (dcfpup << 3) | (nightsyn << 2) | (lcdnight << 1) | s_enable, cmdport); // Konfigurationsbyte 1 zusammensetzen und senden
                  usb_char (',', cmdport);                                                         // Trennzeichen senden
                  usb_dec2 (lcdnigm1, cmdport);                                                    // LCD-Nacht-Modus-Beginn Minuten senden
                  usb_char (',', cmdport);                                                         // Trennzeichen senden
                  usb_dec2 (lcdnigh1, cmdport);                                                    // LCD-Nacht-Modus-Beginn Stunden senden
                  usb_char (',', cmdport);                                                         // Trennzeichen senden
                  usb_dec2 (lcdnigm2, cmdport);                                                    // LCD-Nacht-Modus-Ende Minuten senden
                  usb_char (',', cmdport);                                                         // Trennzeichen senden
                  usb_dec2 (lcdnigh2, cmdport);                                                    // LCD-Nacht-Modus-Ende Stunden senden
                  usb_char (',', cmdport);                                                         // Trennzeichen senden
                  usb_dec3 (lcdnigbr, cmdport);                                                    // Helligkeit der Hintergrundbeleuchtung im Nachtmodus senden
                  usb_char (',', cmdport);                                                         // Trennzeichen senden
                  usb_dec2 (gongh1, cmdport);                                                      // Gongzeit-Beginn Stunden senden
                  usb_char (',', cmdport);                                                         // Trennzeichen senden
                  usb_dec2 (gongh2, cmdport);                                                      // Gongzeit-Ende Stunden senden
                  usb_char (',', cmdport);                                                         // Trennzeichen senden
                  usb_dec2 (gong_snd, cmdport);                                                    // Gong-Sound-Nummer senden
                  usb_char (',', cmdport);                                                         // Trennzeichen senden
                  usb_dec2 (gong_vol, cmdport);                                                    // Gong-Lautstrke senden
                  usb_char (',', cmdport);                                                         // Trennzeichen senden
                  usb_dec2 ((gong << 5) | (bdordr << 4) | (bdagem << 3) | (bdlsort << 1) |
                           alarms[3][0], cmdport);                                                 // Konfigurationsbyte 2 zusammensetzen und senden
                  usb_char (',', cmdport);                                                         // Trennzeichen senden
                  usb_dec2 (alarms[3][1], cmdport);                                                // Geburtstags-Alarm Minuten senden
                  usb_char (',', cmdport);                                                         // Trennzeichen senden
                  usb_dec2 (alarms[3][2], cmdport);                                                // Geburtstags-Alarm Stunden senden
                  usb_char (',', cmdport);                                                         // Trennzeichen senden
                  usb_dec2 (alarms[3][8], cmdport);                                                // Geburtstags-Alarm Sound-Nummer senden
                  usb_char (',', cmdport);                                                         // Trennzeichen senden
                  usb_dec2 (alarms[3][9], cmdport);                                                // Geburtstags-Alarm Lautstrke senden
                  usb_char ('\r', cmdport);
                  usb_char ('\n', cmdport);                                                        // Line Feed senden
               }

               if (cmdbuf[1] == ':')                                                               // wenn anschlieendes Trennzeichen erkannt (Konfiguration setzen)
               {
                  okflag = 1;                                                                      // Plausibilitt zunchst auf ok setzen
                  if (read_dec3 (2) > 255) okflag = 0;                                             // wenn Konfigurationsbyte 1 fehlerhaft -> nicht plausibel
                  if (cmdbuf[5] != ',') okflag = 0;                                                // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  if (read_dec2 (6) > 59) okflag = 0;                                              // wenn LCD-Nacht-Modus-Beginn Minuten fehlerhaft -> nicht plausibel
                  if (cmdbuf[8] != ',') okflag = 0;                                                // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  if (read_dec2 (9) > 23) okflag = 0;                                              // wenn LCD-Nacht-Modus-Beginn Stunden fehlerhaft -> nicht plausibel
                  if (cmdbuf[11] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  if (read_dec2 (12) > 59) okflag = 0;                                             // wenn LCD-Nacht-Modus-Ende Minuten fehlerhaft -> nicht plausibel
                  if (cmdbuf[14] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  if (read_dec2 (15) > 23) okflag = 0;                                             // wenn LCD-Nacht-Modus-Beginn Stunden fehlerhaft -> nicht plausibel
                  if (cmdbuf[17] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  if (read_dec3 (18) > 255) okflag = 0;                                            // wenn Helligkeit im Nachtmodus fehlerhaft -> nicht plausibel
                  if (cmdbuf[21] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  if (read_dec2 (22) > 23) okflag = 0;                                             // wenn Gongzeit-Beginn Stunden fehlerhaft -> nicht plausibel
                  if (cmdbuf[24] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  if (read_dec2 (25) > 23) okflag = 0;                                             // wenn Gongzeit-Ende Stunden fehlerhaft -> nicht plausibel
                  if (cmdbuf[27] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  if (read_dec2 (28) > 12) okflag = 0;                                             // wenn Gong-Sound-Nummer fehlerhaft -> nicht plausibel
                  if (cmdbuf[30] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  if (read_dec2 (31) > 10) okflag = 0;                                             // wenn Gong-Lautstrke fehlerhaft -> nicht plausibel
                  if (cmdbuf[33] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  if (read_dec2 (34) > 63) okflag = 0;                                             // wenn Konfigurationsbyte 2 fehlerhaft -> nicht plausibel
                  if (cmdbuf[36] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  if (read_dec2 (37) > 59) okflag = 0;                                             // wenn Geburtstags-Alarm Minuten fehlerhaft -> nicht plausibel
                  if (cmdbuf[39] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  if (read_dec2 (40) > 23) okflag = 0;                                             // wenn Geburtstags-Alarm Stunden fehlerhaft -> nicht plausibel
                  if (cmdbuf[42] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  if (read_dec2 (43) > 12) okflag = 0;                                             // wenn Geburtstags-Alarm Sound-Nummer fehlerhaft -> nicht plausibel
                  if (cmdbuf[45] != ',') okflag = 0;                                               // wenn Trennzeichen fehlerhaft -> nicht plausibel
                  if (read_dec2 (46) > 10) okflag = 0;                                             // wenn Geburtstags-Alarm Lautstrke fehlerhaft -> nicht plausibel

                  if (okflag)                                                                      // wenn alle Daten plausibel
                  {
                     m = read_dec3 (2);                                                            // Konfigurationsbyte 1 lesen
                     lang = (m >> 7) & 1;                                                          // Sprache setzen
                     lcdinv = (m >> 6) & 1;                                                        // LCD-Invertierung setzen
                     aclkpos = (m >> 5) & 1;                                                       // Position der Analog-Uhr setzen
                     dcfinv = (m >> 4) & 1;                                                        // DCF77-Invertierung setzen
                     dcfpup = (m >> 3) & 1;                                                        // DCF77-Pull-Up setzen
                     nightsyn = (m >> 2) & 1;                                                      // Nachtsynchronisierung setzen
                     lcdnight = (m >> 1) & 1;                                                      // LCD-Nachtmodus setzen
                     s_enable = (m >> 0) & 1;                                                      // Sound-Freigabe setzen

                     lcdnigm1 = read_dec2 (6);                                                     // LCD-Nacht-Modus-Beginn Minuten setzen
                     lcdnigh1 = read_dec2 (9);                                                     // LCD-Nacht-Modus-Beginn Stunden setzen
                     lcdnigm2 = read_dec2 (12);                                                    // LCD-Nacht-Modus-Ende Minuten setzen
                     lcdnigh2 = read_dec2 (15);                                                    // LCD-Nacht-Modus-Ende Stunden setzen
                     lcdnigbr = read_dec3 (18);                                                    // Helligkeit der Hintergrundbeleuchtung im Nachtmodus setzen
                     gongh1 = read_dec2 (22);                                                      // Gongzeit-Beginn Stunden setzen
                     gongh2 = read_dec2 (25);                                                      // Gongzeit-Ende Stunden setzen
                     gong_snd = read_dec2 (28);                                                    // Gong-Sound-Nummer setzen
                     gong_vol = read_dec2 (31);                                                    // Gong-Lautstrke setzen

                     n = read_dec2 (34);                                                           // Konfigurationsbyte 2 lesen
                     gong = (n >> 5) & 1;                                                          // Gong-Status setzen
                     bdordr = (n >> 4) & 1;                                                        // Anzeige-Reihenfolge der Geburtstagsdaten setzen
                     bdagem = (n >> 3) & 1;                                                        // Anzeigemodus fr das Alter bei Geburtstagen setzen
                     bdlsort = (n >> 1) & 3;                                                       // Sortierungs-Modus der Geburtstagsliste setzen
                     if (bdlsort > 2) bdlsort = 2;                                                 // Wert begrenzen
                     alarms[3][0] = (n >> 0) & 1;                                                  // Geburtstags-Alarm-Status setzen
                     alarms[3][1] = read_dec2 (37);                                                // Geburtstags-Alarm Minuten setzen
                     alarms[3][2] = read_dec2 (40);                                                // Geburtstags-Alarm Stunden setzen
                     alarms[3][8] = read_dec2 (43);                                                // Geburtstags-Alarm Sound-Nummer setzen
                     alarms[3][9] = read_dec2 (46);                                                // Geburtstags-Alarm Lautstrke setzen

                     eep_write_byte (e_lang, lang);                                                // Sprache im EEPROM speichern
                     eep_write_byte (e_lcdinv, lcdinv);                                            // LCD-Invertierung im EEPROM speichern
                     eep_write_byte (e_aclkpos, aclkpos);                                          // Analog-Uhr-Position im EEPROM speichern
                     eep_write_byte (e_dcfinv, dcfinv);                                            // DCF-Invertierung im EEPROM speichern
                     eep_write_byte (e_dcfpup, dcfpup);                                            // DCF-Pull-up im EEPROM speichern
                     eep_write_byte (e_nightsyn, nightsyn);                                        // LCD-Nacht-Sync-Abschaltung im EEPROM speichern
                     eep_write_byte (e_lcdnight, lcdnight);                                        // LCD-Nachtschaltung im EEPROM speichern
                     eep_write_byte (e_lcdnigm1, lcdnigm1);                                        // LCD-Nachtschaltung Beginn Minuten im EEPROM speichern
                     eep_write_byte (e_lcdnigh1, lcdnigh1);                                        // LCD-Nachtschaltung Beginn Stunden im EEPROM speichern
                     eep_write_byte (e_lcdnigm2, lcdnigm2);                                        // LCD-Nachtschaltung Ende Minuten im EEPROM speichern
                     eep_write_byte (e_lcdnigh2, lcdnigh2);                                        // LCD-Nachtschaltung Ende Stunden im EEPROM speichern
                     eep_write_byte (e_lcdnigbr, lcdnigbr);                                        // LCD-Nacht-Helligkeit im EEPROM speichern
                     eep_write_byte (e_gong, gong);                                                // Gong-Status im EEPROM speichern
                     eep_write_byte (e_gongh1, gongh1);                                            // Gong-Beginnzeit im EEPROM speichern
                     eep_write_byte (e_gongh2, gongh2);                                            // Gong-Endezeit im EEPROM speichern
                     eep_write_byte (e_gong_snd, gong_snd);                                        // Gong-Sound-Nummer im EEPROM speichern
                     eep_write_byte (e_gong_vol, gong_vol);                                        // Gong-Lautstrke im EEPROM speichern
                     eep_write_byte (e_bdlsort, bdlsort);                                          // Geburtstagslisten-Sortierung im EEPROM speichern
                     eep_write_byte (e_bdordr, bdordr);                                            // Geburtstags-Reihenfolge im EEPROM speichern
                     eep_write_byte (e_bdagem, bdagem);                                            // Alters-Anzeigemodus im EEPROM speichern
                     eep_write_byte (e_alarms + 30, alarms[3][0]);                                 // Geburtstags-Alarm Status im EEPROM speichern
                     eep_write_byte (e_alarms + 31, alarms[3][1]);                                 // Geburtstags-Alarm Minuten im EEPROM speichern
                     eep_write_byte (e_alarms + 32, alarms[3][2]);                                 // Geburtstags-Alarm Stunden im EEPROM speichern
                     eep_write_byte (e_alarms + 38, alarms[3][8]);                                 // Geburtstags-Alarm Sound-Nummer im EEPROM speichern
                     eep_write_byte (e_alarms + 39, alarms[3][9]);                                 // Geburtstags-Alarm Lautstrke im EEPROM speichern
                     eep_write_byte (e_senable, s_enable);                                         // Sound-Freigabe im EEPROM speichern
                     if (dcfpup) PORTE |= 1 << PE2; else PORTE &= ~ (1 << PE2);                    // DCF-Pull-Up an PortE2 ein- oder ausschalten
                  }
                  usb_char ('c', cmdport);                                                         // Antwort 'c' senden
                  if (okflag) usb_char ('0', cmdport);                                             // wenn Konfiguration plausibel -> Zeichen '0' senden
                  else usb_char ('1', cmdport);                                                    // sonst Zeichen '1' senden
                  usb_char ('\r', cmdport);                                                        // Carriage Return senden
                  usb_char ('\n', cmdport);                                                        // Line Feed senden
               }
               cmdport = 0;
               break;

            default: cmdport = 0;                                                                  // wenn kein Kommando erkannt -> Kommando beenden
         }
      }

// Timeouts bearbeiten --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

      if (disptout == 0)                                                                           // wenn Anzeige-Timeout-Zhler abgelaufen
      {
         disptout = 255;                                                                           // Timeout-Zhler stoppen
         dispmode = 0;                                                                             // Anzeigemodus auf 0 (normale Anzeige)
         editmode = 0;                                                                             // Editormodus deaktivieren
         chg_flag = 1;                                                                             // nderungs-Flag setzen, Informationen neu ausgeben
      }

      if (usbtout == 0)                                                                            // wenn USB-Symbol-Timeout-Zhler abgelaufen
      {
         usbtout = 255;                                                                            // Timeout-Zhler stoppen
         usb_act = 0;                                                                              // USB-Symbol ausschalten
      }

      if (syntout == 0)                                                                            // wenn Synchronisierungs-Timeout abgelaufen
      {
         syntout = 255;                                                                            // Timeout-Zhler stoppen
         if (!lcd_is_on)                                                                           // wenn LCD ausgeschaltet
         {
            lcd_on ();                                                                             // LCD einschalten
            sec_flag = 1;                                                                          // Sekunden-Flag setzen, Analoguhr und Symbole neu ausgeben
            chg_flag = 1;                                                                          // nderungs-Flag setzen, Informationen neu ausgeben
            if (syncstat==0) syncstat = 1;                                                         // wenn Uhr noch nie synchronisiert -> Sync-Status auf Abbruch
         }
      }

      if (dcftout == 0)                                                                            // wenn DCF-Timeout-Zhler abgelaufen
      {
         dcftout = 255;                                                                            // Timeout-Zhler stoppen
         chg_flag = 1;                                                                             // nderungs-Flag setzen, Informationen neu ausgeben
         if (syncstat == 4) syncstat = 3;                                                          // Status auf "DCF mindestens 24h nicht synchronisiert" setzen
      }

      if (nightout == 0)                                                                           // wenn Nacht-Helligkeits-Timeout-Zhler abgelaufen
      {
         nightout = 255;                                                                           // Timeout-Zhler stoppen, Aktionen steuert and. Programmteil
      }

// Zeitabhngige Funktionen steuern -------------------------------------------------------------------------------------------------------------------------------------------------------------------

                                                                                                   // DCF-Nacht-Synchronisierung steuern --------------------------------------------------------------

      if (lcd_is_on && nightsyn && (dispmode == 0) && (hour == 3) && (minute == 0) && (second == 0)) // wenn LCD und Nacht-Synchronisierung eingeschaltet, normale Anzeige und Zeit = 3:00 Uhr
      {
         lcd_off ();                                                                               // LCD ausschalten
         syntout = c_syntout;                                                                      // Synchronisierungs-Timeout setzen
      }
                                                                                                   // Nachtzeiten prfen und Aktiv-Flag steuern -------------------------------------------------------

      int16_t ntime1, ntime2, curtime;                                                             // Zwischenspeicher fr die Zeitrechnung

      lcdn_act = 0;                                                                                // Nachtzeit zunchst inaktiv setzen
      if (lcdnight)                                                                                // wenn LCD-Nacht-Modus eingeschaltet
      {
         ntime1 = lcdnigh1 * 60 + lcdnigm1;                                                        // Nacht-Beginnzeit in einen Minutenwert umrechnen
         ntime2 = lcdnigh2 * 60 + lcdnigm2;                                                        // Nacht-Endezeit in einen Minutenwert umrechnen
         curtime = hour * 60 + minute;                                                             // aktuelle Zeit in einen Minutenwert umrechnen
         if (ntime1 < ntime2)                                                                      // wenn Startzeit kleiner als Endezeit
         {
            if ((curtime >= ntime1) && (curtime < ntime2)) lcdn_act = 1;                           // wenn aktuelle Zeit zwischen Start- und Endezeit -> Nacht-Modus aktiv setzen
         }
         else                                                                                      // wenn Startzeit grer als Endezeit
         {
            if ((curtime >= ntime1) || (curtime < ntime2)) lcdn_act = 1;                           // wenn aktuelle Zeit grer als Startzeit oder kleiner als Endezeit -> Nacht-Modus aktiv setzen
         }
      }
                                                                                                   // Alarmzeiten prfen ------------------------------------------------------------------------------

      uint8_t tminute;                                                                             // temporrer Zwischenspeicher fr den aktuellen Minutenwert
      uint8_t thour;                                                                               // temporrer Zwischenspeicher fr den aktuellen Stundenwert
      uint8_t aflag;                                                                               // Merker fr bereits ausgelsten Alarm

      if (syncstat > 1)                                                                            // wenn die Uhr eine gltige Zeit hat
      {
         tminute = minute;                                                                         // Minutenwert kopieren \ nderungen whrend der Alarmprfung
         thour = hour;                                                                             // Sundenwert kopieren  / knnen zu einem falschen Sound fhren
         aflag = 0;                                                                                // Merker lschen
         for (i = 0; i < 4; i ++)                                                                  // 3 Alarmzeiten und Geburtstagsalarm prfen
         {
            if (alarms[i][0] > 0)                                                                  // wenn Alarm aktiv ist
            {
               if ((alarms[i][3] == day) && (alarms[i][5] == month) && (alarms[i][6] == year) &&
                  (alarms[i][1] == tminute) && (alarms[i][2] == thour))                            // wenn Datum und Uhrzeit stimmt
               {
                  if (alarms[i][10] < 2)                                                           // wenn Alarm noch nicht quittiert wurde
                  {
                     if (!lcd_is_on)                                                               // wenn LCD wegen Synchronisierung ausgeschaltet ist
                     {
                        lcd_on ();                                                                 // LCD einschalten
                        sec_flag = 1;                                                              // Sekunden-Flag setzen, Analoguhr und Symbole neu ausgeben
                        chg_flag = 1;                                                              // nderungs-Flag setzen, Informationen neu ausgeben
                        disptout = c_disptout;                                                     // Anzeige-Timeout starten
                     }

                     if (lcdn_act)                                                                 // wenn Nachtzeit aktiv
                     {
                        disptout = c_disptout;                                                     // Anzeige-Timeout starten (Anzeige hell schalten)
                     }

                     alarms[i][10] = 1;                                                            // Alarm auslsen
                     if (!aflag)                                                                   // wenn noch kein Alarm vermerkt ist
                     {
                        if (snstat == 0)                                                           // wenn Sound-Erzeugung im Ruhezustand
                        {
                           cli ();                                                                 // Interrupts sperren
                           sndptr = (PGM_P) pgm_read_word (&sounds[alarms[i][8]]);                 // Pointer auf Alarm-Sound setzen
                           if (alarms[i][9] == 0) maxvol = 0;                                      // wenn Lautstrke 0 eingestellt
                           else maxvol = (alarms[i][9] + 5) * 17;                                  // sonst Alarm-Sound-Lautstrke setzen (102-255)
                           snstat = 1;                                                             // Sound-Erzeugung aktivieren
                           sei ();                                                                 // Interrupts wieder freigeben
                        }
                     }
                     aflag = 1;                                                                    // Merker setzen, weitere zur gleichen Zeit aktive Alarme
                  }                                                                                // werden bei der Sound-Ausgabe nicht bercksichtigt
               }
               else                                                                                // wenn Datum und Uhrzeit nicht stimmt
               {
                  if (alarms[i][10] == 2) alarms[i][10] = 0;                                       // wenn Alarm quittiert wurde -> Auslsung lschen
               }
            }
         }
      }
                                                                                                   // Stunden-Gong steuern ----------------------------------------------------------------------------

      uint8_t tsecond;                                                                             // temporrer Zwischenspeicher fr den aktuellen Sekundenwert

      if ((syncstat > 1) && gong)                                                                  // wenn die Uhr eine gltige Zeit hat und der Gong aktiviert ist
      {
         tsecond = second;                                                                         // Sekundenwert kopieren
         tminute = minute;                                                                         // Minutenwert kopieren
         thour = hour;                                                                             // Sundenwert kopieren
         aflag = 0;                                                                                // Merker lschen

         if (gongh1 < gongh2)                                                                      // wenn Gong-Beginnzeit kleiner als Endezeit
         {
            if ((thour >= gongh1) && (thour <= gongh2)) aflag = 1;                                 // wenn Zeit zwischen Beginn und Ende
         }
         if (gongh1 > gongh2)                                                                      // wenn Gong-Beginnzeit grer als Endezeit
         {
            if ((thour >= gongh1) || (thour <= gongh2)) aflag = 1;                                 // wenn Zeit grer als Beginn oder kleiner als Ende
         }
         if (gongh1 == gongh2) aflag = 1;                                                          // wenn Beginn = Ende -> immer Gong auslsen

         if ((tsecond == 0) && (tminute == 0) && aflag)                                            // wenn Sekunden / Minuten auf 0 und Gongzeitraum
         {
            aflag = 0;                                                                             // Merker lschen
            for (i = 0; i < 4; i ++)                                                               // 3 Alarmzeiten und Geburtstagsalarm prfen
            {
               if (alarms[i][0] > 0)                                                               // wenn Alarm aktiv ist
               {
                  if ((alarms[i][3] == day) && (alarms[i][5] == month) && (alarms[i][6] == year) &&
                  (alarms[i][1] == tminute) && (alarms[i][2] == thour)) aflag = 1;                 // wenn Datum und Uhrzeit stimmt -> Merker setzen
               }
            }
            if (!aflag)                                                                            // wenn kein anderer Alarm aktiv ist
            {
               if (snstat == 0)                                                                    // wenn Sound-Erzeugung im Ruhezustand
               {
                  cli ();                                                                          // Interrupts sperren
                  sndptr = (PGM_P) pgm_read_word (&sounds[gong_snd]);                              // Pointer auf Gong-Sound setzen
                  if (gong_vol == 0) maxvol = 0;                                                   // wenn Lautstrke 0 eingestellt
                  else maxvol = (gong_vol + 5) * 17;                                               // sonst Gong-Lautstrke setzen (102-255)
                  snstat = 1;                                                                      // Sound-Erzeugung aktivieren
                  sei ();                                                                          // Interrupts wieder freigeben
               }
            }
         }
     }

// Statusabhngige Funktionen steuern -----------------------------------------------------------------------------------------------------------------------------------------------------------------

                                                                                                   // Hintergrundbeleuchtung steuern ------------------------------------------------------------------

      if (lcd_is_on)                                                                               // wenn das LCD eingeschaltet ist
      {
         if (lcdn_act)                                                                             // wenn Nachtzeit ist
         {
            if ((disptout < 255) && (nightout == 255)) blt_on (255);                               // wenn Anzeige-Timeout-Zhler aktiv und Nacht-Helligkeits-Timeout-Zhler inaktiv ist -> Beleuchtung
            else blt_on (lcdnigbr);                                                                // auf volle Helligkeit, sonst Nacht-Helligkeit einstellen
         }
         else                                                                                      // wenn nicht Nachtzeit ist
         {
            if (nightout == 255) blt_on (255);                                                     // wenn Nacht-Helligkeits-Timeout inaktiv -> Beleuchtung auf volle Helligkeit einstellen
            else blt_on (lcdnigbr);                                                                // sonst Nacht-Helligkeit einstellen
         }
      }
      else blt_off ();                                                                             // wenn das LCD ausgeschaltet ist -> Beleuchtung ausschalten

                                                                                                   // Automatische DCF-Synchronisierung nach Reset ----------------------------------------------------

      if (lcd_is_on && nightsyn && (disptout == 255) && (syncstat == 0))                           // wenn LCD eingeschaltet, Nacht-Sync aktiviert, Anzeige-Timeout abgelaufen und Uhr noch nie
      {                                                                                            // synchronisiert wurde
         syncstat = 1;                                                                             // Sync-Status erhhen (kein neuer Sync-Versuch nach Abbruch)
         lcd_off ();                                                                               // LCD ausschalten
         syntout = c_syntout;                                                                      // Synchronisierungs-Timeout setzen
      }
                                                                                                   // LED-Funktion steuern ----------------------------------------------------------------------------

      if (!lcd_is_on)                                                                              // wenn LCD ausgeschaltet ist
      {
         if (syncstat < 3)                                                                         // wenn Uhr noch nie ber DCF77 synchronisiert wurde
         {
            if (dcfstat1) led_off ();                                                              // LED entsprechend Pegel des DCF77-Signals aus-
            else led_on ();                                                                        // oder einschalten
         }
      }
      else                                                                                         // wenn das LCD eingeschaltet ist
      {
         if (syncstat > 1)                                                                         // wenn die Uhr eine gltige Zeit hat
         {
            if ((alarms[0][10] == 1) || (alarms[1][10] == 1) || (alarms[2][10] == 1) || (alarms[3][10] == 1)) // wenn ein Alarm ausgelst wurde
            {
               if ((intc20ms % 25) < 12) led_on ();                                                // wenn 20ms-Zhler im Bereich 0-12 oder 25-37 -> LED ein
               else led_off ();                                                                    // sonst LED ausschalten (Blinken mit 2Hz)
            }
            else led_off ();                                                                       // wenn kein Alarm ausgelst wurde -> LED aus
         }
         else led_off ();                                                                          // wenn die Uhr keine gltige Zeit hat -> LED aus
      }
                                                                                                   // Alarm-Check und Geburtstags-Check ---------------------------------------------------------------

      if (syncstat > 1)                                                                            // wenn die Uhr eine gltige Zeit hat
      {
         if (alm_flag)                                                                             // wenn das Alarm-Flag gesetzt ist
         {
            alm_flag = 0;                                                                          // Alarm-Flag wieder lschen
            alarm_check (0);                                                                       // Alarm-Check aufrufen und Daten des gerade bearbeiteten Alarms speichern
         }

         if (dispmode < 2)                                                                         // wenn normale Anzeige oder Hauptmen aktiv ist
         {
            if (bd_flag)                                                                           // wenn Geburtstags-Flag gesetzt ist
            {
               bd_flag = 0;                                                                        // Geburtstags-Flag wieder lschen
               bd_check ();                                                                        // Geburtstags-Check aufrufen und Indexliste reorganisieren
            }
         }

         if (dispmode < 1)                                                                         // wenn normale Anzeige aktiv ist
         {
            if (reorg_flag == 2)                                                                   // wenn Reorganisation der Geburtstage erforderlich
            {
               bd_check ();                                                                        // Geburtstags-Check aufrufen und Indexliste reorganisieren
               reorg_flag = 0;                                                                     // Reorganisation abgeschlossen
            }
            if (reorg_flag == 1)                                                                   // wenn Reorganisation der Alarme erforderlich
            {
               alarm_check (1);                                                                    // Alarme prfen, korrigieren und alle Alarmdaten speichern
               reorg_flag = 2;                                                                     // Reorganisation der Geburtstage im nchsten Zyklus
            }
         }

         k = month * 31 + day;                                                                     // aktuellen Tag im Jahr ermitteln (einfache Methode)
         if (daychg != k)                                                                          // wenn Tageswechsel erkannt wurde
         {
            daychg = k;                                                                            // neuen Tageswert speichern
            eep_write_byte (e_lastday, day);                                                       // neuen Tag im EEPROM speichern
            eep_write_byte (e_lastmonth, month);                                                   // neuen Monat im EEPROM speichern
            eep_write_byte (e_lastyear, year);                                                     // neues Jahr im EEPROM speichern
            reorg_flag = 1;                                                                        // Reorganisation starten
         }
      }
                                                                                                   // Sound-Freigabe prfen und setzen ----------------------------------------------------------------

      if ((syncstat > 1) && s_enable) soundstat = 1;                                               // wenn gltige Zeit und Sound-Freigabe -> Sound einschalten
      else soundstat = 0;                                                                          // sonst Sound-Ausgabe ausschalten

   }
   return 0;
}
