; =====================================================================
; Sensormodul zur Temperatur- und Alarm-Erfassung mit ATmega8
; von Scott-Falk Hhn (s.huehn@soemtron.de)
; Version 1.07, letzte Bearbeitung am 12.06.2005
; =====================================================================
;
; Dieses Sensormodul kommuniziert mit bis zu 16 Temperatursensoren, die
; an bis zu 8 1-Wire-Busse angeschlossen werden knnen. Verwendbar sind
; folgende Typen: DS1820, DS18S20, DS18B20, DS1822, DS1920. Zustzlich
; sind noch 4 Alarmeingnge vorhanden. Ein optionales LCD mit I2C
; Anschluss (DSM-0822A) zeigt zyklisch die Temperaturwerte aller Sen-
; soren an. Die ermittelten Daten werden folgendermaen ber RS232 mit
; 9600 Baud im ASCII Format gesendet:
;
; * Sensorbelegung:
;
; 33000200#     (Beispiel: 3 Sensoren an Bus1, 3 Sensoren an Bus2 und
;		2 Sensoren an Bus6)
;	  - Enter-Zeichen CR (13)
; --------  Anzahl der Sensoren an jedem 1-Wire Bus, von Bus1 bis Bus8;
;	    bei mehr als 9 Sensoren an einem Bus wird ein Buchstabe
;	    ausgegeben: aus 10 wird "A", aus 11 wird "B" usw.
;
; Diese Meldung wird nur einmal in der Startphase des Sensormoduls
; gesendet (ca. 11 Sekunden nach dem Einschalten).
;
; * Temperaturwerte:
;
; 2: 23.5#	(Temperatur 2 = 23.5C)
;        - Enter-Zeichen CR (13)
;   -----  Temperaturwert im Bereich -55.0 bis 125.0
;  -       Trennzeichen (Doppelpunkt)
; -        Sensornummer im Bereich 1-8 und a-h (Sensor 9-16)
;
; Das Minuszeichen kann an 3. oder 4. Stelle stehen. Die Daten werden
; fortlaufend im Abstand von einer Sekunde gesendet. Ein kompletter
; Zyklus dauert dann im Normalbetrieb 18 Sekunden bei Verwendung von 16
; Sensoren.
;
; * Alarmmeldungen:
;
; C:1#		(Alarm 3 ein)
;    - Enter-Zeichen CR (13)
;   -  Alarmwert 0 =Alarm aus (Kontakt geffnet)
;		 1 =Alarm ein (Kontakt geschlossen)
;  -   Trennzeichen (Doppelpunkt)
; -    Alarmnummer, A entspricht Alarm 1, B entspreicht Alarm 2 usw.
;
; Ein Alarmmeldung wird bei jeder Statusnderung eines Alarmeingangs
; gesendet. Auerdem wird zu Beginn eines neuen Zyklus der aktuelle
; Status aller 4 Alarme gesendet.
;
; Die beiden Jumper bzw. DIP-Schalter werden einmalig beim Einschalten
; des Sensormoduls gelesen und haben folgende Bedeutung:
;
; * Jumper 1 - Speichermodus: Bei gestecktem Jumper werden beim Ein-
;   schalten des Sensormoduls die ermittelten Sensordaten im EEPROM
;   abgelegt. Bei allen nachfolgenden Neustarts (mit abgezogenem Jum-
;   per) werden dann die Sensordaten nicht neu ermittelt und stattdes-
;   sen aus dem EERPOM gelesen. Beim Entfernen oder Hinzufgen von
;   Sensoren muss ein Neustart mit gestecktem Jumper durchgefhrt wer-
;   den um die nderungen zu speichern. Der aktivierte Speichermodus
;   wird durch das "Stereo"-Symbol auf dem LCD angezeigt.
;
; * Jumper 2 - Spezialmodus: Dieser Modus ist fr spezielle Anwendungen
;   wie z.B. Heizungssteuerungen gedacht. Hier werden im Abstand von 2
;   Sekunden alle Temperaturdaten komplett ausgegeben. In diesem Modus
;   werden keine Messwerte auf dem LCD angezeigt, stattdessen erscheint
;   der Text "SENDEN". Der aktivierte Spezialmodus wird durch 3 Feld-
;   strkebalken auf dem LCD angezeigt.
;
; =====================================================================
;
; Einstellung der Konfigurationsbits (Fuse-Bits):
;
; CKOPT				 = 1
; CKSEL3, CKSEL2, CKSEL1, CKSEL0 = 1111  (Quarz 3-8 MHz)
; SUT1, SUT0			 = 11    (Quarz, langsame Betriebsspg.)
; BODLEVEL, BODEN		 = 10    (Brown-Out ein, 2,7 V)
;
; =====================================================================
;
; ----------------------
; Belegung der I/O-Ports
; ----------------------
;
; PortB0: Eingang Alarm 1 mit Pull-up, offen= Alarm aus, Low= Alarm ein
; PortB1: Eingang Alarm 2 mit Pull-up, offen= Alarm aus, Low= Alarm ein
; PortB2: Eingang Jumper 1 (Speicher) mit Pull-up
; PortB3: (MOSI)
; PortB4: (MISO)
; PortB5: (SCK)
;
; PortC0: Ein/Ausgang Sensorbus 1
; PortC1: Ein/Ausgang Sensorbus 2
; PortC2: Ein/Ausgang Sensorbus 3
; PortC3: Ein/Ausgang Sensorbus 4
; PortC4: SDA \ TWI (I2C) fr LCD    mit Pull-up
; PortC5: SCL / DSM-0822A (Pollin)   mit Pull-up
;
; PortD0: Eingang Jumper 2 (Spezial) mit Pull-up
; PortD1: Ausgang RS-232, 9600 Baud
; PortD2: Eingang Alarm 3 mit Pull-up, offen= Alarm aus, Low= Alarm ein
; PortD3: Eingang Alarm 4 mit Pull-up, offen= Alarm aus, Low= Alarm ein
; PortD4: Ein/Ausgang Sensorbus 5
; PortD5: Ein/Ausgang Sensorbus 6
; PortD6: Ein/Ausgang Sensorbus 7
; PortD7: Ein/Ausgang Sensorbus 8
;
; ------------------
; Belegung der Timer
; ------------------
;
; Timer0: nicht genutzt
; Timer1: Interrupt alle 100ms
; Timer2: nicht genutzt
;
; -----------------------------------
; Konstanten und Registerdefinitionen
; -----------------------------------
;
.nolist
.include "m8def.inc"
.list
;
.equ	clock=	4000000	;Controller Taktfrequenz= 4,0MHz
.equ	twifr=	100000	;TWI Taktfrequenz= 100kHz
.equ	tim1fr=	10	;Timer1 Frequenz= 10Hz (100ms)
.equ	baud=	9600	;RS-232 Baudrate= 9600
.equ	cr=	13	;ASCII-Code fr CR
.equ	almtio=	7	;Alarm-Timeout (in 100ms-Schritten)
.equ	maxsen=	16	;maximale Anzahl an Temperatursensoren (16)
;
.def	isreg=	r12	;Zwischenspeicher fr SREG bei Interruptbearb.
.def	itemp1=	r13	;Zwischenspeicher 1 fr Interruptroutine
.def	flags=	r24	;verschiedene Flags
			;Bit0: Temperaturmess-Flag, wird vom Timer1-
			;      Interrupt zyklisch zur Sekunde 0 gesetzt
			;      und vom Hauptprogramm wieder gelscht;
			;      zu dieser Zeit werden ebenfalls die
			;      Alarmmeldungen neu gesendet.
			;Bit1: Messende-Flag, wird von Timer1-Interrupt
			;      zyklisch zur Sekunde 1 gesetzt und vom
			;      Hauptprogramm wieder gelscht
			;Bit2: Sensor-Flag, wird im Normalmodus vom
			;      Timer1-Interrupt an den Sekunden
			;      2,3,4,5,6,...,17 gesetzt
			;      und vom Hauptprogramm wieder gelscht
			;Bit3: Speichermodus (Sensor-Daten wurden aus
			;      EEPROM gelesen)
			;Bit4: Spezial-Flag, wird bei aktiviertem Spe-
			;      zialmodus zur Sekunde 1.2 gesetzt und
			;      vom Hauptprogramm wieder gelscht
.def	itemp2=	r25	;Zwischenspeicher 2 fr Interruptroutine
;
; --------------
; SRAM Variablen
; --------------
;
.dseg
.org	0x60
;
lcdbuf:	.byte	8	;LCD-Puffer fr 8 Textzeichen (ASCII)
alstat:	.byte	4	;Alarmstatus fr 4 Alarme, wird vollstndig vom
			;Interrupt gesteuert:
			;Bit0: aktuell gelesener Alarmstatus
			;Bit1: letzter stabiler Alarmstatus
			;Bit2: Bearbeitungs-Flag (zeigt an, dass Sta-
			;      tuswechsel schon bearbeitet wird)
			;Bit3: Melde-Flag, Info fr Hauptprogramm zum
			;      sofortigen Senden des Statuswechsels,
			;      wird dann vom Hauptprogramm gelscht
acount:	.byte	4	;Alarmzeitzhler fr 4 Alarme, wird von Inter-
			;rupt bei Statusnderung auf <almtio> gesetzt,
			;auf 0 gezhlt, danach wird Melde-Flag gesetzt
sensid:	.byte	16*8	;8-Byte-ROM-IDs der 16 Temperatursensoren
senstb:	.byte	16	;Sensor-Tabelle, enthlt fr jeden der 16 Sen-
			;soren die Nummer des 1-Wire-Busses (0-7), an
			;dem er angeschlossen ist, fr nicht vorhandene
			;Sensoren wird 0xff eingetragen
senflg:	.byte	16	;Sensor-Flags der 16 Temperatursensoren
			;Bit0: Temperaturwert 85,0 Grad
owbuff:	.byte	8	;1-Wire-Puffer fr empfangene 64-Bit-ROM-ID und
			;fr den Empfang des Scratchpad
icoval:	.byte	1	;Basiswert fr den Interruptzhler; der Wert
			;wird bei der Sensorsuche ermittelt und liegt
			;zwischen 2 (kein Sensor) und 18 (16 Sensoren)
icount:	.byte	1	;Interrupt-Zhler, wird bei jedem 10. Timer1-
			;Interrupt aufwrts bis <icoval> gezhlt und
			;dann wieder zurckgesetzt
icoun2:	.byte	1	;Interrupt-Zhler 2, dient als Vorteiler fr
			;<icount>
jflags:	.byte	1	;Jumper-Flags:
			;Bit0: Jumper 1 (Speicher) ist gesetzt
			;Bit1: Jumper 2 (Spezial) ist gesetzt
scount:	.byte	1	;Sensorzhler im Spezialmodus, zhlt von 2 bis
			;<icoval>; wird nach einem Durchlauf auf 0xff
			;gesetzt
;
; ===============
; Programmbereich
; ===============
;
.cseg
.org	0
;
; Interrupt-Vektoren
;
reset:	rjmp	start		;Programm-Start
intr0:	reti			;External Interrupt Request 0
intr1:	reti			;External Interrupt Request 1
t2comp:	reti			;Timer/Counter2 Compare Match
t2ovf:	reti			;Timer/Counter2 Overflow
t1capt:	reti			;Timer/Counter1 Capture Event
t1coma:	rjmp	timer1		;Timer/Counter1 Compare Match A
t1comb:	reti			;Timer/Counter1 Compare Match B
t1ovf:	reti			;Timer/Counter1 Overflow
t0ovf:	reti			;Timer/Counter0 Overflow
spistc:	reti			;SPI, STC Serial Transfer Complete
usrxc:	reti			;RXC USART, Rx Complete
usudre:	reti			;UDRE USART Data Register Empty
ustxc:	reti			;TXC USART, Tx Complete
adc_cc:	reti			;ADC Conversion Complete
ee_rdy:	reti			;EEPROM Ready
anacom:	reti			;Analog Comparator
twi:	reti			;Two-wire Serial Interface
spmrdy:	reti			;Store Program Memory Ready
;
; Initialisierung SRAM, I/O-Ports, Hardware
;
start:	ldi	zl,0x60
	ldi	zh,0			;Anfangsadresse SRAM
	clr	r16
sramcl:	st	z+,r16			;Speicherzelle lschen
	cpi	zl,low(ramend+1)	;letzte Speicherzelle erreicht?
	brne	sramcl			;nein -> Schleife
	cpi	zh,high(ramend+1)	;letzte Speicherzelle erreicht?
	brne	sramcl			;nein -> Schleife
;
	ldi	r16,low(ramend)
	ldi	r17,high(ramend)
	out	spl,r16
	out	sph,r17			;Stackpointer setzen
	clr	r16
	out	ddrb,r16		;PortB, alles Eingnge
	ldi	r16,0x3f
	out	portb,r16		;PortB, alle Pullups ein
	clr	r16
	out	ddrc,r16		;PortC, alles Eingnge
	ldi	r16,0x30
	out	portc,r16		;PortC, Pullups fr TWI ein
	ldi	r16,2
	out	ddrd,r16		;PortD, nur TXD als Ausgang
	ldi	r16,0x0d
	out	portd,r16		;PortD, Pullups fr Alarm-Eing.
;
	ldi	r16,low((clock/(16*baud))-1)
	ldi	r17,high((clock/(16*baud))-1)
	out	ubrrh,r17		;USART Baudratenregister auf
	out	ubrrl,r16		;9600 Baud setzen
	ldi	r16,(1<<ursel)|(1<<ucsz1)|(1<<ucsz0)
	out	ucsrc,r16		;8 Datenbits, 1 Stoppbit
	ldi	r16,1<<txen
	out	ucsrb,r16		;Sender aktivieren
;
	ldi	r16,low((clock/(8*10))-1)
	ldi	r17,high((clock/(8*10))-1)
	out	ocr1ah,r17		;Timer1, Output Compare A
	out	ocr1al,r16		;fr 10 Hz setzen
	ldi	r16,(1<<wgm12)|(1<<cs11)
	out	tccr1b,r16		;Timer1, CTC-Modus, Vorteiler=8
	ldi	r16,1<<ocie1a
	out	timsk,r16		;OC1A-Interrupt aktivieren
;
	ldi	r16,(clock/(2*twifr))-8	;Wert fr Baudratenregister
	out	twbr,r16		;fr TWI setzen
	clr	r16
	out	twsr,r16		;Vorteiler=1 setzen
	ldi	r16,1<<twen
	out	twcr,r16		;TWI aktivieren
;
	clr	flags			;alle Flags lschen
	rcall	wait50			;50 ms warten
;
	clr	r17
	in	r16,pinb		;PortB Eingnge lesen
	bst	r16,2			;Jumper 1 gesetzt?
	brts	testj1			;nein -> weiter
	sbr	r17,0x01		;sonst Flag fr Jumper 1 setzen
testj1:	in	r16,pind		;PortD Eingnge lesen
	bst	r16,0			;Jumper 2 gesetzt?
	brts	testj2			;nein -> weiter
	sbr	r17,0x02		;sonst Flag fr Jumper 2 setzen
testj2:	sts	jflags,r17		;Jumper-Flags speichern
;
; LCD initialisieren
;
	rcall	twista			;TWI Start Condition
	ldi	r17,0x70		;LCD-Adresse und W-Bit
	rcall	twisnd			;Byte senden
	ldi	r17,0xe0		;LCD Device Select 0
	rcall	twisnd			;Byte senden
	ldi	r17,0xc8		;LCD Mode set, 1:4, 1/3 Bias
	rcall	twisnd			;Byte senden
	ldi	r17,0			;LCD Data Pointer, Adresse 0
	rcall	twisnd			;Byte senden
	ldi	r18,20			;20 Datenbytes an LCD ausgeben
clrlcd:	ldi	r17,0			;LCD-Segmente lschen
	rcall	twisnd			;Byte senden
	dec	r18			;alle Bytes gesendet?
	brne	clrlcd			;nein -> Schleife
	rcall	twisto			;TWI Stop Condition
;
; Versiontext ausgeben
;
	ldi	zl,low(2*verstx)
	ldi	zh,high(2*verstx)	;Zeiger auf Versiontext
	ldi	xl,low(lcdbuf)
	ldi	xh,high(lcdbuf)		;Zeiger auf LCD-Textpuffer
	ldi	r17,8			;8 Zeichen kopieren
copyvt:	lpm	r16,z+			;Zeichen aus Flash holen
	st	x+,r16			;und in Textpuffer kopieren
	dec	r17			;alle Zeichen kopiert?
	brne	copyvt			;nein -> Schleife
	rcall	lctout			;Text auf LCD ausgeben
;
; Sonderzeichen ausgeben (Dezimalpunkt 6)
;
	rcall	twista			;TWI Start Condition
	ldi	r17,0x70		;LCD-Adresse und W-Bit
	rcall	twisnd			;Byte senden
	ldi	r17,0xe0		;LCD Device Select 0
	rcall	twisnd			;Byte senden
	ldi	r17,32			;LCD Data Pointer, Adresse 32
	rcall	twisnd			;Byte senden
	ldi	r17,0x10		;Sonderzeichen Dezimalpunkt 6
	rcall	twisnd			;auf LCD ausgeben
	clr	r17			;restliche Segmente aus
	rcall	twisnd			;auf LCD ausgeben
	rcall	twisto			;TWI Stop Condition
;
	ldi	r16,60			;60*50ms (3 sek) warten
delay1:	rcall	wait50			;50ms warten
	dec	r16
	brne	delay1
	rcall	clrlcb			;LCD-Puffer lschen
	rcall	lctout			;LCD-Puffer ausgeben
	rcall	lcaout			;Alarme und Sonderz. ausgeben
;
; Temperatursensor-Tabelle lschen
;
	ldi	xl,low(senstb)
	ldi	xh,high(senstb)		;Zeiger auf Sensortabelle
	ldi	r16,0xff		;Startwert 0xff
initab:	st	x+,r16			;in Puffer schreiben
	cpi	xl,low(senstb+maxsen)	;Tabellenende erreicht?
	brne	initab			;nein -> Schleife
;
	lds	r16,jflags		;Jumper-Flags holen
	bst	r16,0			;Jumper 1 (Speicher) gesetzt?
	brts	search			;ja -> Sensor-Suche starten
	ldi	r16,low(eprflg)
	ldi	r17,high(eprflg)	;EEPROM-Adresse fr EEPROM-Flag
	rcall	eeread			;EEPROM lesen
	tst	r18			;sind EEPROM-Daten vorhanden?
	brne	search			;nein -> Sensor-Suche starten
;
; Sensordaten aus EEPROM auslesen
;
	ldi	r20,maxsen		;Anzahl der Sensoren holen (16)
	ldi	r16,low(esenid)
	ldi	r17,high(esenid)	;EEPROM-Adresse der Sensor-IDs
	ldi	xl,low(sensid)
	ldi	xh,high(sensid)		;Zeiger auf Sensor-IDs im RAM
loadi1:	ldi	r19,8			;jeweils 8 ID-Bytes
loadi2:	rcall	eeread			;EEPROM lesen
	st	x+,r18			;gelesenes Byte speichern
	inc	r16			;nchste EEPROM-Adresse L
	brne	loadi3			;bertrag? nein -> weiter
	inc	r17			;nchste EEPROM-Adresse H
loadi3:	dec	r19			;alle 8 Bytes gelesen?
	brne	loadi2			;nein -> Schleife
	dec	r20			;alle Sensoren gelesen?
	brne	loadi1			;nein -> Schleife
;
	ldi	r20,maxsen		;Anzahl der Sensoren holen (16)
	ldi	r16,low(esentb)
	ldi	r17,high(esentb)	;EEPROM-Adresse der Sensor-Tab.
	ldi	xl,low(senstb)
	ldi	xh,high(senstb)		;Zeiger auf Sensor-Tab. im RAM
loadt1:	rcall	eeread			;EEPROM lesen
	st	x+,r18			;gelesenes Byte speichern
	inc	r16			;nchste EEPROM-Adresse L
	brne	loadt2			;bertrag? nein -> weiter
	inc	r17			;nchste EEPROM-Adresse H
loadt2:	dec	r20			;alle Sensoren gelesen?
	brne	loadt1			;nein -> Schleife
	sbr	flags,0x08		;Speichermodus-Flag setzen
	rjmp	sea910			;Daten ausgeben
;
; 1-Wire Temperatursensoren suchen und ROM-IDs in SRAM schreiben,
; Algorithmus nach Application Note 187 von Dallas/Maxim
;
search:	ldi	r22,0			;Sensornummer=0 setzen
	ldi	r20,0			;Busnummer=0 setzen
	ldi	zl,low(sensid)
	ldi	zh,high(sensid)		;Zeiger auf Sensor-IDs
seanew:	clr	r21			;LastDiscrepancy lschen
	clr	r9			;LastDeviceFlag lschen
	clr	r8			;ReturnValue lschen
searom:	rcall	owres			;Reset senden
	brcs	sea700			;Presence? nein -> Suchende
	tst	r9			;LastDeviceFlag gesetzt?
	brne	sea700			;ja -> Suchende
	clr	r11
	inc	r11			;Bit-Number=1 setzen
	clr	r10			;Last-Zero=0 setzen
	ldi	r19,0xf0		;Search-ROM-Kommando (0xf0)
	rcall	owwrit			;senden
sea100:	rcall	owbrea			;einzelnes Bit lesen (Bit A)
	rol	r18			;Bit A sichern
	andi	r18,0x01		;restliche Bits lschen
	rcall	owbrea			;einzelnes Bit lesen (Bit B)
	rol	r19			;Bit B sichern
	andi	r19,0x01		;restliche Bits lschen
	mov	r17,r19			;Bit B kopieren
	and	r17,r18			;Beide Bits=1?
	brne	sea700			;ja -> weiter
	mov	r17,r19			;Bit B kopieren
	or	r17,r18			;Beide Bits=0?
	breq	sea400			;ja -> weiter
	mov	r16,r18			;Bit A speichern
sea200:	bst	r16,0			;Bit sichern
	rcall	bitsto			;in Puffer einschieben
	bld	r16,0			;gesichertes Bit holen
	ror	r16			;und in Carry schieben
	rcall	owbwri			;Bit in Carry senden
	inc	r11			;Bit-Number erhhen
	mov	r16,r11			;Bit-Index kopieren
	cpi	r16,65			;Endwert erreicht (>64)?
	brcs	sea100			;nein -> Schleife
	mov	r21,r10			;LastDiscrepancy=Last-Zero
	tst	r21			;LastDiscrepancy=0?
	brne	sea300			;nein -> weiter
	inc	r9			;sonst LastDeviceFlag setzen
sea300:	rcall	crc			;CRC der ROM-ID prfen
	tst	r17			;CRC-Wert=0 (ok)?
	brne	sea700			;nein -> nchster Sensor
	inc	r8			;Return-Value setzen
	rjmp	sea800			;Ende
;
sea400:	cp	r11,r21			;Bit-Number=Last-Discrepancy?
	brne	sea500			;nein -> weiter
	ldi	r16,1			;sonst "1"-Bit setzen
sea450:	cpi	r16,0			;Search-Direction=0?
	brne	sea200			;nein -> Bit speichern/senden
	mov	r10,r11			;Last-Zero=Bit-Number
	rjmp	sea200			;Bit speichern/senden
;
sea500:	brcs	sea600			;Bit-Number<Last-Discrepancy?->
	clr	r16			;sonst "0"-Bit setzen
	rjmp	sea450			;Bit speichern und senden
;
sea600:	lds	r16,owbuff+7		;entprechendes Bit von letzter
	andi	r16,0x01		;Suche holen und maskieren
	rjmp	sea450			;Bit speichern und senden
;
sea700:	clr	r21			;LastDiscrepancy=0 setzen
	clr	r9			;LastDeviceFlag=0 setzen
	clr	r8			;ReturnValue=0 setzen
;
sea800:	tst	r8			;wurde ein Sensor gefunden?
	breq	sea900			;nein -> nchsten Bus testen
;
	lds	r16,owbuff+7		;sonst Family-Code holen
	cpi	r16,0			;gltiger Family-Code (>0)?
	breq	sea900			;nein -> nchster Bus
	cpi	r16,0x10		;Family-Code=10h (DS18S20)?
	breq	sea810			;ja -> bearbeiten
	cpi	r16,0x28		;Family-Code=28h (DS18B20)?
	breq	sea810			;ja -> bearbeiten
	cpi	r16,0x22		;Family-Code=22h (DS1822)?
	brne	sea830			;nein -> nchster Sensor
;
sea810:	ldi	xl,low(owbuff)
	ldi	xh,high(owbuff)		;Zeiger auf 1-Wire-Puffer
	ldi	r19,8			;8 Bytes kopieren
sea820:	ld	r18,x+			;Bytes aus 1-Wire-Puffer holen
	st	z+,r18			;und ROM-ID-Feld speichern
	dec	r19			;alle Bytes bearbeitet?
	brne	sea820			;nein -> Schleife
	ldi	xl,low(senstb)
	ldi	xh,high(senstb)		;Zeiger auf Sensortabelle
	add	xl,r22			;Sensornummer addieren
	adc	xh,r19			;Tabellenplatz berechnen
	st	x,r20			;Busnummer speichern
	inc	r22			;Sensornummer erhhen
	cpi	r22,maxsen		;Maximum 16 erreicht?
	breq	sea910			;ja -> Ende
;
sea830:	tst	r9			;weitere Sensoren am Bus?
	brne	sea900			;nein -> nchsten Bus testen
	rjmp	searom			;sonst weitere Sensoren lesen
;
sea900:	inc	r20			;Busnummer erhhen
	cpi	r20,8			;Maximum 8 erreicht?
	breq	sea910			;ja -> Ende
	rjmp	seanew			;neue Suche beginnen
;
; Sensordaten bei gesetztem Jumper 1 im EEPROM speichern
;
sea910:	lds	r16,jflags		;Jumper-Flags holen
	bst	r16,0			;Jumper 1 (Speicher) gesetzt?
	brtc	senlcd			;nein -> Daten ausgeben
;
	ldi	r20,maxsen		;Anzahl der Sensoren holen (16)
	ldi	r16,low(esenid)
	ldi	r17,high(esenid)	;EEPROM-Adresse der Sensor-IDs
	ldi	xl,low(sensid)
	ldi	xh,high(sensid)		;Zeiger auf Sensor-IDs im RAM
savei1:	ldi	r19,8			;jeweils 8 ID-Bytes
savei2:	ld	r18,x+			;Byte aus RAM holen
	rcall	eewrit			;und im EEPROM speichern
	inc	r16			;nchste EEPROM-Adresse L
	brne	savei3			;bertrag? nein -> weiter
	inc	r17			;nchste EEPROM-Adresse H
savei3:	dec	r19			;alle 8 Bytes gelesen?
	brne	savei2			;nein -> Schleife
	dec	r20			;alle Sensoren gelesen?
	brne	savei1			;nein -> Schleife
;
	ldi	r20,maxsen		;Anzahl der Sensoren holen (16)
	ldi	r16,low(esentb)
	ldi	r17,high(esentb)	;EEPROM-Adresse der Sensor-Tab.
	ldi	xl,low(senstb)
	ldi	xh,high(senstb)		;Zeiger auf Sensor-Tab. im RAM
savet1:	ld	r18,x+			;Byte aus RAM holen
	rcall	eewrit			;und im EEPROM speichern
	inc	r16			;nchste EEPROM-Adresse L
	brne	savet2			;bertrag? nein -> weiter
	inc	r17			;nchste EEPROM-Adresse H
savet2:	dec	r20			;alle Sensoren gelesen?
	brne	savet1			;nein -> Schleife
;
	ldi	r16,low(eprflg)
	ldi	r17,high(eprflg)	;EEPROM-Adresse fr EEPROM-Flag
	clr	r18			;neuer Flag-Wert
	rcall	eewrit			;Flag im EEPROM speichern
	sbr	flags,0x08		;Speichermodus-Flag setzen
;
; Sensorenbersicht auf LCD anzeigen und nach Wartezeit von 8 Sekunden
; ber RS-232 senden, Anzeigeposition entspricht Busnummer, angezeigter
; Wert entspricht der Anzahl der gefundenen Sensoren, Beispiel:
; 33000200 -> Bus1 hat 3 Sensoren, Bus2 hat ebenfalls 3 Sensoren und
;	      Bus6 hat 2 Sensoren gefunden, restliche Busse ohne Sensor
;
senlcd:	ldi	xl,low(lcdbuf)
	ldi	xh,high(lcdbuf)		;Zeiger auf LCD-Textpuffer
	ldi	r19,0			;Busnummer=0 setzen
sensor:	ldi	zl,low(senstb)
	ldi	zh,high(senstb)		;Zeiger auf Sensortabelle
	ldi	r18,maxsen		;16 Werte durchsuchen
	ldi	r17,'0'			;Sensorzhler=0 setzen (ASCII)
sen100:	ld	r16,z+			;Wert aus Tabelle lesen
	cp	r16,r19			;Sensor am aktuellen Bus?
	brne	sen200			;nein -> weiter
	inc	r17			;sonst Sensorzhler erhhen
sen200:	dec	r18			;alle Tabellenpltze gelesen?
	brne	sen100			;nein -> Schleife
	cpi	r17,'9'+1		;mehr als 9 Sensoren am Bus?
	brcs	sen210			;nein -> weiter
	ldi	r16,7			;sonst Zahl in Buchstaben
	add	r17,r16			;wandeln: 10..16 -> A..G
sen210:	st	x+,r17			;Sensoranzahl speichern
	inc	r19			;nchsten Bus testen
	cpi	r19,8			;alle Busse getestet?
	brne	sensor			;nein -> Schleife
;
	rcall	lctout			;LCD-Puffer ausgeben
;
	ldi	xl,low(senstb)
	ldi	xh,high(senstb)		;Zeiger auf Sensortabelle
	clr	r17			;Sensor-Zhler=0 setzen
sen300:	ld	r16,x+			;Wert aus Tabelle lesen
	cpi	r16,0xff		;Sensor vorhanden?
	breq	sen310			;nein -> berspringen
	inc	r17			;sonst Sensor zhlen
sen310:	cpi	xl,low(senstb+maxsen)	;alle Tabellenpltze bearb.?
	brne	sen300			;nein -> Schleife
	ldi	r16,2			;Offset fr Interruptzhler
	add	r17,r16			;addieren
	sts	icoval,r17		;Wert speichern
;
	ldi	r16,160			;160*50ms (8 sek) warten
delay2:	rcall	wait50
	dec	r16
	brne	delay2
;
	ldi	r16,cr			;CR laden und senden zur Syn-
	rcall	rssend			;chronisation mit Empfnger
	ldi	xl,low(lcdbuf)
	ldi	xh,high(lcdbuf)		;Zeiger auf LCD-Puffer
	ldi	r17,8			;8 Zeichen senden
sen400:	ld	r16,x+			;Zeichen holen
	rcall	rssend			;Zeichen senden
	dec	r17			;alle Zeichen gesendet?
	brne	sen400			;nein -> Schleife
	ldi	r16,cr			;CR-Code laden
	rcall	rssend			;und senden
;
	rcall	clrlcb			;LCD-Puffer lschen
	rcall	lctout			;LCD-Puffer ausgeben
	clr	r16
	sts	icoun2,r16		;Vorteiler lschen
	ldi	r16,-4			;Startwert fr Interrupt-
	sts	icount,r16		;Zhler setzen (-4 Sekunden)
	ldi	r16,0xff		;Stoppwert fr Sensorzhler
	sts	scount,r16		;setzen (fr Spezialmodus)
	sei				;Interrupts freigeben
;
;
; =====================================================================
; Hauptprogrammschleife
; =====================================================================
;
; Status-Anzeige aktualisieren, aktuelle Alarmmeldungen ausgeben
;
main:	ldi	zl,low(alstat)
	ldi	zh,high(alstat)		;Zeiger auf Alarmstatus
	ldi	r20,0			;aktuelle Alarmnummer
ma0100:	ld	r19,z			;Alarmstatus holen
	bst	r19,3			;Melde-Flag gesetzt?
	brtc	ma0110			;nein -> weiter
	cli				;Interrupts sperren
	ld	r19,z			;Alarmstatus nochmals holen
	cbr	r19,0b1000		;Melde-Flag lschen
	st	z,r19			;Alarmstatus wieder speichern
	sei				;Interrupts wieder aktivieren
	ror	r19			;Alarmstatus an Bitposition 0
	andi	r19,0x01		;restliche Bits entfernen
	rcall	rsaout			;Alarmmeldung senden
	rcall	lcaout			;Alarmstatus auf LCD anzeigen
ma0110:	adiw	zl,1			;Zeiger auf nchsten Alarm
	inc	r20			;Alarmnummer erhhen
	cpi	r20,4			;alle Alarme bearbeitet?
	brne	ma0100			;nein -> Schleife
;
; Temperaturmess-Flag prfen und bei Bedarf eine neue Messung starten
;
	bst	flags,0			;Temperaturmess-Flag gesetzt?
	brtc	ma0400			;nein -> berspringen
	cbr	flags,0b001		;sonst TempMess-Flag lschen
	clr	r20			;Busnummer=0 setzen
ma0200:	ldi	xl,low(senstb)
	ldi	xh,high(senstb)		;Zeiger auf Sensor-Tabelle
ma0210:	ld	r19,x+			;verwendeten Bus des Sensors
	cp	r19,r20			;holen, =aktueller Bus?
	breq	ma0220			;ja -> Messung auf Bus starten
	cpi	xl,low(senstb+maxsen)	;Tabellenende erreicht?
	brne	ma0210			;nein -> nchster Tabellenplatz
	rjmp	ma0230			;sonst nchster Bus
;
ma0220:	cli				;Interrupts sperren
	rcall	owres			;1-Wire-Reset senden
	brcs	ma0230			;Presence? nein -> Fehler
	ldi	r19,0xcc		;ROM-Kommando "Skip ROM" senden
	rcall	owwrit			;(alle Sensoren ansprechen)
	ldi	r19,0x44		;Funktion-Kommando "Convert T"
	rcall	owwrit			;senden und fr sichere Messung
	rcall	owstpu			;Strong-Pullup aktivieren
ma0230:	sei				;Interrupts wieder freigeben
	inc	r20			;nchsten Bus bearbeiten
	cpi	r20,8			;letzter Bus erreicht?
	brne	ma0200			;nein -> Schleife
;
; Alarmmeldungen erneut senden
;
	ldi	zl,low(alstat)
	ldi	zh,high(alstat)		;Zeiger auf Alarmstatus
	ldi	r20,0			;aktuelle Alarmnummer
ma0300:	ld	r19,z+			;Alarmstatus holen
	ror	r19			;Alarmstatus an Bitposition 0
	andi	r19,0x01		;restliche Bits entfernen
	rcall	rsaout			;Alarmmeldung senden
	inc	r20			;Alarmnummer erhhen
	cpi	r20,4			;alle Alarme bearbeitet?
	brne	ma0300			;nein -> Schleife
;
; Text "Messung" oder "k.Sensor" (kein Sensor) auf LCD ausgeben
;
	lds	r16,icoval		;Endwert fr Interrupt-Zhler
	cpi	r16,2			;holen, Wert=2 (kein Sensor)?
	brne	ma0310			;nein -> "Messung" ausgeben
	ldi	zl,low(2*senstx)	;sonst "k.Sensor" ausgeben
	ldi	zh,high(2*senstx)	;Zeiger auf Text
	rjmp	ma0320
ma0310:	ldi	zl,low(2*meastx)
	ldi	zh,high(2*meastx)	;Zeiger auf Alarmtext
ma0320:	ldi	xl,low(lcdbuf)
	ldi	xh,high(lcdbuf)		;Zeiger auf LCD-Puffer
	ldi	r17,8			;8 Zeichen kopieren
ma0330:	lpm	r16,z+			;Zeichen aus Flash holen
	st	x+,r16			;und in LCD-Puffer schreiben
	dec	r17			;alle Zeichen bearbeitet?
	brne	ma0330			;nein -> Schleife
	rcall	lctout			;LCD-Puffer ausgeben
;
; Temperaturmessende-Flag prfen und bei Bedarf Messungen stoppen
;
ma0400:	bst	flags,1			;Messende-Flag gesetzt?
	brtc	ma1000			;nein -> berspringen
	cbr	flags,0b010		;sonst Ende-Flag wieder lschen
	clr	r20			;Busnummer=0 setzen
ma0410:	ldi	xl,low(senstb)
	ldi	xh,high(senstb)		;Zeiger auf Sensor-Tabelle
ma0420:	ld	r19,x+			;verwendeten Bus des Sensors
	cp	r19,r20			;holen, =aktueller Bus?
	breq	ma0430			;ja -> Messung auf Bus stoppen
	cpi	xl,low(senstb+maxsen)	;Tabellenende erreicht?
	brne	ma0420			;nein -> nchster Tabellenplatz
	rjmp	ma0440			;sonst nchster Bus
;
ma0430:	cli				;Interrupts sperren
	rcall	owres			;1-Wire-Reset,Strong-Pullup aus
	sei				;Interrupts wieder freigeben
ma0440:	inc	r20			;nchsten Bus bearbeiten
	cpi	r20,8			;letzter Bus erreicht?
	brne	ma0410			;nein -> Schleife
;
; Sensor-Flag prfen und bei Bedarf den entsprechenden Sensor auslesen
;
ma1000:	bst	flags,2			;Sensor-Flag gesetzt?
	brts	ma1010			;ja -> bearbeiten
	rjmp	ma1030			;sonst Spezial-Flag prfen
;
ma1010:	cbr	flags,0b100		;Sensor-Flag wieder lschen
	lds	r16,icount		;Interrupt-Zhler holen
	subi	r16,2			;2 Sekunden abziehen, Negativ?
	brcc	ma1020			;nein -> weiter
	clr	r16			;sonst Zhler=0 setzen
ma1020:	cpi	r16,maxsen		;zulssiger Wert berschritten?
	brcs	ma1100			;nein -> Sensornummer verwenden
	rjmp	ma2000
;
; Spezial-Flag prfen und alle Sensoren ausgeben (jeweils einen pro
; Hauptschleifenzyklus)
;
ma1030:	bst	flags,4			;Spezial-Flag gesetzt?
	brts	ma1040			;ja -> bearbeiten
	rjmp	ma2000			;sonst Ende
;
ma1040:	lds	r16,scount		;Sensorzhler holen
	cpi	r16,0xff		;Stopp-Zustand?
	brne	ma1050			;nein -> nchster Sensor
	clr	r16			;sonst ersten Sensor whlen
	rjmp	ma1060
ma1050:	inc	r16			;nchsten Sensor whlen
ma1060:	sts	scount,r16		;Sensorzhler speichern
	lds	r17,icoval		;Basiswert fr Zhler holen
	subi	r17,2			;korrigieren
	cp	r16,r17			;Endwert berschritten?
	brcs	ma1080			;nein -> Sensor bearbeiten
ma1070:	ldi	r16,0xff		;sonst Zhler in Stopp-Zustand
	sts	scount,r16		;Sensorzhler speichern
	cbr	flags,0x10		;Spezial-Flag wieder lschen
	rjmp	ma2000
;
ma1080:	tst	r16			;erster Sensor?
	brne	ma1100			;nein -> ausgeben
;
	ldi	zl,low(2*sentxt)	;sonst
	ldi	zh,high(2*sentxt)	;Zeiger auf Senden-Text
	ldi	xl,low(lcdbuf)
	ldi	xh,high(lcdbuf)		;Zeiger auf LCD-Puffer
	ldi	r17,8			;8 Zeichen kopieren
ma1090:	lpm	r16,z+			;Zeichen aus Flash holen
	st	x+,r16			;und in LCD-Puffer schreiben
	dec	r17			;alle Zeichen bearbeitet?
	brne	ma1090			;nein -> Schleife
	rcall	lctout			;LCD-Puffer ausgeben
	lds	r16,scount		;Sensorzhler nochmals holen
;
; Scratchpad aus Sensor auslesen
;
ma1100:	mov	r17,r16			;Sensor-Nummer kopieren
	mov	r22,r16			;Sonsor-Nummer sichern
	cpi	r17,8			;Sensor-Nummer>8?
	brcc	ma1110			;ja -> als Buchstabe ausgeben
	ldi	r17,49
	add	r17,r16			;in ASCII (1-8) wandeln
	rjmp	ma1120			;ausgeben
ma1110:	ldi	r17,89
	add	r17,r16			;in ASCII (a-h) wandeln
ma1120:	sts	lcdbuf,r17		;in LCD-Puffer schreiben
	ldi	r17,':'			;Doppelpunkt
	sts	lcdbuf+1,r17		;in LCD-Puffer schreiben
	ldi	r17,' '			;Leerzeichen
	sts	lcdbuf+2,r17		;in LCD-Puffer schreiben
;
	ldi	xl,low(senstb)
	ldi	xh,high(senstb)		;Zeiger auf Sensor-Tabelle
	clr	r17			;H-Byte=0 fr Tabellenplatz
	add	xl,r16			;Tabellenplatz berechnen
	adc	xh,r17			;bertrag addieren
	ld	r20,x			;zugehrige Busnummer holen
	cpi	r20,0xff		;Sensor vorhanden?
	brne	ma1130			;ja -> weiter
	rjmp	ma1500			;sonst Fehler
;
ma1130:	ldi	xl,low(sensid)
	ldi	xh,high(sensid)		;Zeiger auf Sensor-ROM-IDs
	add	r16,r16
	add	r16,r16
	add	r16,r16			;Sensor-Nummer*8 als Offset
	add	xl,r16			;Tabellenplatz berechnen
	adc	xh,r17			;bertrag addieren
	adiw	xl,7			;Zeiger auf Family-Code
	ld	r7,x+			;Code holen, Zeiger auf Ende
;
	cli				;Interrupts sperren
	rcall	owres			;1-Wire-Reset senden
	sei				;Interrupts wieder freigeben
	brcc	ma1140			;Presence? ja -> weiter
	rjmp	ma1500			;sonst Fehler
;
ma1140:	cli				;Interrupts wieder sperren
	ldi	r19,0x55		;ROM-Kommando "Match ROM"
	rcall	owwrit			;senden
	ldi	r21,8			;8 Bytes ROM-ID senden
ma1150:	ld	r19,-x			;Byte aus Tabelle holen
	rcall	owwrit			;Byte senden
	dec	r21			;alle Bytes gesendet?
	brne	ma1150			;nein -> Schleife
	ldi	r19,0xbe		;Funktion "Read Scratchpad"
	rcall	owwrit			;senden
	ldi	xl,low(owbuff+8)
	ldi	xh,high(owbuff+8)	;Zeiger auf 1-Wire-Pufferende
	ldi	r21,8			;8 Bytes Scratchpad einlesen
ma1160:	rcall	owread			;Byte lesen
	st	-x,r19			;Byte in Puffer speichern
	dec	r21			;alle Bytes gelesen?
	brne	ma1160			;nein -> Schleife
	rcall	owread			;CRC-Byte lesen
	sei				;Interrupts wieder freigeben
	mov	r21,r19			;gelesenen CRC-Wert sichern
	rcall	crc			;CRC vom Scratchpad ermitteln
	cp	r17,r21			;war CRC-Wert ok?
	brne	ma1170			;nein -> Fehler
	lds	r16,owbuff		;zustzlich Count_Per_C holen
	cpi	r16,0x10		;und prfen,Count_Per_C=10h?
	breq	ma1180			;ja -> weiter
ma1170:	rjmp	ma1500			;sonst Fehler
;
ma1180:	ldi	r16,0x10		;Family-Code 0x10 (DS18S20)
	cp	r7,r16			;Family-Code ok?
	brne	ma1250			;nein -> and. Family bearbeiten
;
; DS18S20: Gelesene Daten aufbereiten, Erhhung der Mess-Genauigkeit
; durch Algorithmus nach Datenblatt; die Formel wurde etwas modifi-
; ziert, so dass mit ganzzahligen Werten gerechnet werden kann und
; keine Divisionsroutine ntig ist:
; Temp = (160 * Temp_Read - 40 + 10*(Count_Per_C - Count_Remain)) / 16
; Das Ergebnis liegt dann in Zehntel Grad vor.
;
ma1200:	lds	r17,owbuff+6		;Temperaturwert H holen
	lds	r16,owbuff+7		;Temperaturwert L holen
	ror	r17
	ror	r16			;LSB abschneiden (Temp_Read)
	ldi	r17,160			;Wert fr hhere Genauigkeit
	mulsu	r16,r17			;mit 160 multiplizieren
	movw	xl,r0			;Ergebnis kopieren
	sbiw	xl,40			;40 subtrahieren
	ldi	r16,16			;Count_Per_C laden (immer 16)
	lds	r17,owbuff+1		;Count_Remain holen
	sub	r16,r17			;Count_Per_C - Count_Remain
	add	r16,r16			;Wert *2
	mov	r17,r16			;kopieren
	add	r16,r16			;Wert nochmals *2 (*4)
	add	r16,r16			;Wert nochmals *2 (*8)
	add	r16,r17			;*2-Wert addieren (*10)
	clr	r17			;H-Byte=0
	add	xl,r16			;zur vorherigen Rechnung add.
	adc	xh,r17			;bertrag addieren
	asr	xh
	ror	xl			;Wert /2
	asr	xh
	ror	xl			;Wert /2 (/4)
	asr	xh
	ror	xl			;Wert /2 (/8)
	asr	xh
	ror	xl			;Wert /2 (/16)
	brcc	ma1210			;runden? nein -> weiter
	adiw	xl,1			;sonst Ergebnis aufrunden
ma1210:	mov	r20,xh			;Vorzeichen sichern
	bst	xh,7			;Ergebnis negativ?
	brtc	ma1300			;nein -> berspringen
	clr	r16			;Wert 0x10000 zur Wandlung ins
	clr	r17			;Zweierkomplement laden
	sub	r16,xl			;L subtrahieren
	sbc	r17,xh			;H mit bertrag subtrahieren
	movw	xl,r16			;ins Ergebnis-Register kopieren
	rjmp	ma1300			;in ASCII wandeln
;
; DS18B20/DS1822: Gelesene Daten aufbereiten, Umrechnung des 12-Bit
; Binrwertes in einen Dezimalwert in Zehntel Grad nach folgender
; Formel: Temp = Temp_Register * 5 / 8
;
ma1250:	lds	r16,owbuff+3		;Configuration Register holen
	cpi	r16,0x7f		;Sensor auf 12-Bit eingestellt?
	breq	ma1260			;ja -> weiter
	rjmp	ma1500			;sonst Fehler
ma1260:	lds	xh,owbuff+6		;Temperaturwert H holen
	lds	xl,owbuff+7		;Temperaturwert L holen
	mov	r20,xh			;Vorzeichen sichern
	bst	xh,7			;Temperaturwert negativ?
	brtc	ma1270			;nein -> weiter
	clr	r16			;Wert 0x10000 zur Wandlung ins
	clr	r17			;Zweierkomplement laden
	sub	r16,xl			;L subtrahieren
	sbc	r17,xh			;H mit bertrag subtrahieren
	movw	xl,r16			;ins Ergebnis-Register kopieren
ma1270:	movw	r16,xl			;Temperaturwert kopieren
	add	xl,xl
	adc	xh,xh			;Temperaturwert verdoppeln
	add	xl,xl
	adc	xh,xh			;nochmals verdoppeln (*4)
	add	xl,r16
	adc	xh,r17			;und Originalwert addieren (*5)
	asr	xh
	ror	xl			;Wert /2
	asr	xh
	ror	xl			;Wert /2 (/4)
	asr	xh
	ror	xl			;Wert /2 (/8)
	brcc	ma1300			;runden? nein -> weiter
	adiw	xl,1			;sonst Ergebnis aufrunden
;
; Ergebnis in Dezimal-ASCII umwandeln
;
ma1300:	clr	r16			;0-Wert fr 16-Bit Subtraktion
	clr	r18			;ASCII Stelle 2 lschen
	clr	r19			;ASCII Stelle 1 lschen
ma1310:	mov	r17,xl			;Binrzahl zwischenspeichern L
	subi	xl,100			;100 subtrahieren
	sbc	xh,r16			;bertrag subtrahieren
	brcs	ma1320			;war Zahl>99? nein -> weiter
	inc	r18			;ASCII Stelle 2 erhhen
	cpi	r18,10			;bertrag?
	brcs	ma1310			;nein -> Schleife
	clr	r18			;sonst ASCII Stelle 2 lschen
	inc	r19			;und ASCII Stelle 1 erhhen
	rjmp	ma1310			;Schleife
;
ma1320:	mov	xl,r17			;letzten Binrwert L holen
	clr	r17			;ASCII Stelle 3 lschen
ma1330:	cpi	xl,10			;Wert>9?
	brcs	ma1340			;nein -> Rest auswerten
	inc	r17			;sonst ASCII Stelle 3 erhhen
	subi	xl,10			;Wert um 10 vermindern
	rjmp	ma1330			;Schleife
ma1340:	mov	r16,xl			;Rest in ASCII Stelle 4 legen
	ori	r16,0x30		;Ergebnis in ASCII wandeln
	ori	r17,0x30
	ori	r18,0x30
	ori	r19,0x30
;
; Daten fr LCD und RS-232-Ausgabe vorbereiten und senden
;
	cpi	r19,'0'			;ASCII Stelle 1 =0?
	brne	ma1350			;nein -> weiter
	ldi	r19,' '			;sonst durch Leerzeichen ers.
	cpi	r18,'0'			;hchste Ziffer=0?
	brne	ma1350			;nein -> berspringen
	ldi	r18,' '			;sonst durch Leerzeichen ers.
ma1350:	bst	r20,7			;ist Temperaturwert negativ?
	brtc	ma1370			;nein -> berspringen
	cpi	r18,' '			;Stelle 2 = Leerzeichen?
	breq	ma1360			;ja -> durch Minus ersetzen
	ldi	r19,'-'			;sonst Stelle 1 = Minuszeichen
	rjmp	ma1370
ma1360:	ldi	r18,'-'			;Stelle 2 = Minuszeichen
ma1370:	sts	lcdbuf+2,r19		;ASCII Stelle 1 in LCD-Puffer
	sts	lcdbuf+3,r18		;ASCII Stelle 2 in LCD-Puffer
	sts	lcdbuf+4,r17		;ASCII Stelle 3 in LCD-Puffer
	ldi	r20,'.'			;Dezimalpunkt
	sts	lcdbuf+5,r20		;in LCD-Puffer schreiben
	sts	lcdbuf+6,r16		;ASCII Stelle 4 in LCD-Puffer
	ldi	r20,0x23		;Grad-Zeichen
	sts	lcdbuf+7,r20		;in LCD-Puffer schreiben
;
	lds	r16,jflags		;Jumper-Flags holen
	bst	r16,1			;Spezial-Jumper gesetzt?
	brts	ma1380			;ja -> Temp-Wert nicht anzeigen
	rcall	lctout			;sonst LCD-Puffer ausgeben
;
ma1380:	ldi	xl,low(senflg)
	ldi	xh,high(senflg)		;Zeiger auf Sensorflags
	clr	r16
	add	xl,r22			;gesicherte Sensornummer
	adc	xh,r16			;addieren, bertrag
	ld	r17,x			;Sensorflag holen
;
	lds	r16,lcdbuf+2		;ASCII Stelle 1 holen
	cpi	r16,' '			;Wert=0?
	brne	ma1390			;nein -> Flag lschen
	lds	r16,lcdbuf+3		;ASCII Stelle 2 holen
	cpi	r16,'8'			;Wert=8?
	brne	ma1390			;nein -> Flag lschen
	lds	r16,lcdbuf+4		;ASCII Stelle 3 holen
	cpi	r16,'5'			;Wert=5?
	brne	ma1390			;nein -> Flag lschen
	lds	r16,lcdbuf+6		;ASCII Stelle 4 holen
	cpi	r16,'0'			;Wert=0?
	brne	ma1390			;nein -> Flag lschen
;
	bst	r17,0			;85-Grad-Flag gesetzt?
	brts	ma1400			;ja -> Daten ausgeben
	sbr	r17,0x01		;sonst 85-Grad-Flag setzen
	st	x,r17			;Sensorflags wieder speichern
	rjmp	ma2000			;Datenausgabe berspringen
;
ma1390:	cbr	r17,0x01		;85-Grad-Flag lschen
	st	x,r17			;Sensorflags wieder speichern
;
ma1400:	ldi	xl,low(lcdbuf)
	ldi	xh,high(lcdbuf)		;Zeiger auf LCD-Puffer
	ldi	r17,7			;7 Zeichen senden
ma1410:	ld	r16,x+			;Zeichen holen
	rcall	rssend			;Zeichen senden
	dec	r17			;alle Zeichen gesendet?
	brne	ma1410			;nein -> Schleife
	ldi	r16,cr			;CR-Code laden
	rcall	rssend			;und senden
	rjmp	ma2000
;
ma1500:	ldi	zl,low(2*errtxt)
	ldi	zh,high(2*errtxt)	;Zeiger auf Fehlertext
	ldi	xl,low(lcdbuf+2)
	ldi	xh,high(lcdbuf+2)	;Zeiger auf LCD-Puffer
	ldi	r17,6			;6 Zeichen kopieren
ma1510:	lpm	r16,z+			;Zeichen aus Flash holen
	st	x+,r16			;und in LCD-Puffer schreiben
	dec	r17			;alle Zeichen bearbeitet?
	brne	ma1510			;nein -> Schleife
;
	lds	r16,jflags		;Jumper-Flags holen
	bst	r16,1			;Spezial-Jumper gesetzt?
	brts	ma2000			;ja -> Temp-Wert nicht anzeigen
	rcall	lctout			;sonst LCD-Puffer ausgeben
;
ma2000:	rjmp	main
;
;
; ----------------------
; Unterprogramm-Routinen
; ----------------------
;
; Einzelnes Bit in 1-Wire-Puffer einschieben
; Register r16,r17,x
;
bitsto:	ldi	xl,low(owbuff)
	ldi	xh,high(owbuff)		;Zeiger auf 1-Wire-Puffer
	ldi	r17,8			;8 Pufferbytes bearbeiten
	ror	r16			;neues Bit in Carry schieben
bits10:	ld	r16,x			;Pufferbyte holen
	ror	r16			;bertrag-Bit einschieben
	st	x+,r16			;und wieder speichern
	dec	r17			;alle Pufferbytes bearbeitet?
	brne	bits10			;nein -> Schleife
	ret
;
; LCD-Puffer lschen
; Register r16,x
;
clrlcb:	ldi	xl,low(lcdbuf)
	ldi	xh,high(lcdbuf)		;Zeiger auf LCD-Puffer
	ldi	r16,' '			;Leerzeichen
clrl10:	st	x+,r16			;in Puffer schreiben
	cpi	xl,low(lcdbuf+8)	;Pufferende erreicht?
	brne	clrl10			;nein -> Schleife
	ret
;
; 1-Wire, Reset und Presence-Test
; Register r16
;	   r20 <- Busnummer
;	   Carry -> 0=ok, 1=Fehler
;
owres:	cpi	r20,0			;Busnummer=0?
	breq	ow0res			;ja -> bearbeiten
	cpi	r20,1			;Busnummer=1?
	breq	ow1res			;ja -> bearbeiten
	cpi	r20,2			;Busnummer=2?
	breq	ow2res			;ja -> bearbeiten
	cpi	r20,3			;Busnummer=3?
	breq	ow3res			;ja -> bearbeiten
	cpi	r20,4			;Busnummer=4?
	breq	ow4res			;ja -> bearbeiten
	cpi	r20,5			;Busnummer=5?
	breq	owrs10			;ja -> bearbeiten
	cpi	r20,6			;Busnummer=6?
	breq	owrs20			;ja -> bearbeiten
	rjmp	ow7res			;sonst Busnummer 7 bearbeiten
owrs10:	rjmp	ow5res
owrs20:	rjmp	ow6res
;
ow0res:	cbi	portc,0			;Strong-pullup auschalten
	sbi	ddrc,0			;Port auf Ausgang Low setzen
	rcall	w500us			;500s warten
	cbi	ddrc,0			;Port auf Eingang setzen
	rcall	w060us			;60s warten (Presence Start)
	in	r16,pinc		;Port lesen
	andi	r16,0x01		;Portpin selektieren
	breq	ow0s10			;Presence ok? ja -> weiter
	sec				;sonst Fehler
	ret
ow0s10:	rcall	w240us			;240s warten (Presence Ende)
	ret				;Reset ok
;
ow1res:	cbi	portc,1			;Strong-pullup auschalten
	sbi	ddrc,1			;Port auf Ausgang Low setzen
	rcall	w500us			;500s warten
	cbi	ddrc,1			;Port auf Eingang setzen
	rcall	w060us			;60s warten (Presence Start)
	in	r16,pinc		;Port lesen
	andi	r16,0x02		;Portpin selektieren
	breq	ow1s10			;Presence ok? ja -> weiter
	sec				;sonst Fehler
	ret
ow1s10:	rcall	w240us			;240s warten (Presence Ende)
	ret				;Reset ok
;
ow2res:	cbi	portc,2			;Strong-pullup auschalten
	sbi	ddrc,2			;Port auf Ausgang Low setzen
	rcall	w500us			;500s warten
	cbi	ddrc,2			;Port auf Eingang setzen
	rcall	w060us			;60s warten (Presence Start)
	in	r16,pinc		;Port lesen
	andi	r16,0x04		;Portpin selektieren
	breq	ow2s10			;Presence ok? ja -> weiter
	sec				;sonst Fehler
	ret
ow2s10:	rcall	w240us			;240s warten (Presence Ende)
	ret				;Reset ok
;
ow3res:	cbi	portc,3			;Strong-pullup auschalten
	sbi	ddrc,3			;Port auf Ausgang Low setzen
	rcall	w500us			;500s warten
	cbi	ddrc,3			;Port auf Eingang setzen
	rcall	w060us			;60s warten (Presence Start)
	in	r16,pinc		;Port lesen
	andi	r16,0x08		;Portpin selektieren
	breq	ow3s10			;Presence ok? ja -> weiter
	sec				;sonst Fehler
	ret
ow3s10:	rcall	w240us			;240s warten (Presence Ende)
	ret				;Reset ok
;
ow4res:	cbi	portd,4			;Strong-pullup auschalten
	sbi	ddrd,4			;Port auf Ausgang Low setzen
	rcall	w500us			;500s warten
	cbi	ddrd,4			;Port auf Eingang setzen
	rcall	w060us			;60s warten (Presence Start)
	in	r16,pind		;Port lesen
	andi	r16,0x10		;Portpin selektieren
	breq	ow4s10			;Presence ok? ja -> weiter
	sec				;sonst Fehler
	ret
ow4s10:	rcall	w240us			;240s warten (Presence Ende)
	ret				;Reset ok
;
ow5res:	cbi	portd,5			;Strong-pullup auschalten
	sbi	ddrd,5			;Port auf Ausgang Low setzen
	rcall	w500us			;500s warten
	cbi	ddrd,5			;Port auf Eingang setzen
	rcall	w060us			;60s warten (Presence Start)
	in	r16,pind		;Port lesen
	andi	r16,0x20		;Portpin selektieren
	breq	ow5s10			;Presence ok? ja -> weiter
	sec				;sonst Fehler
	ret
ow5s10:	rcall	w240us			;240s warten (Presence Ende)
	ret				;Reset ok
;
ow6res:	cbi	portd,6			;Strong-pullup auschalten
	sbi	ddrd,6			;Port auf Ausgang Low setzen
	rcall	w500us			;500s warten
	cbi	ddrd,6			;Port auf Eingang setzen
	rcall	w060us			;60s warten (Presence Start)
	in	r16,pind		;Port lesen
	andi	r16,0x40		;Portpin selektieren
	breq	ow6s10			;Presence ok? ja -> weiter
	sec				;sonst Fehler
	ret
ow6s10:	rcall	w240us			;240s warten (Presence Ende)
	ret				;Reset ok
;
ow7res:	cbi	portd,7			;Strong-pullup auschalten
	sbi	ddrd,7			;Port auf Ausgang Low setzen
	rcall	w500us			;500s warten
	cbi	ddrd,7			;Port auf Eingang setzen
	rcall	w060us			;60s warten (Presence Start)
	in	r16,pind		;Port lesen
	andi	r16,0x80		;Portpin selektieren
	breq	ow7s10			;Presence ok? ja -> weiter
	sec				;sonst Fehler
	ret
ow7s10:	rcall	w240us			;240s warten (Presence Ende)
	ret				;Reset ok
;
; 1-Wire, Schreiben eines Byte
; Register r16,r18
;	   r19 zu sendendes Byte
;	   r20 Busnummer (0-7)
;
owwrit:	ldi	r18,8			;8 Bits ausgeben
owwr10:	ror	r19			;LSB in Carry schieben
	rcall	owbwri			;Bit ausgeben
	dec	r18			;alle Bits gesendet?
	brne	owwr10			;nein -> Schleife
	ret
;
; 1-Wire, Lesen eines Bytes
; Register r16,r17,r18
;	   r19 -> gelesenes Byte
;	   r20 <- Busnummer (0-7)
;
owread:	ldi	r18,8			;8 Bits ausgeben
owre10:	rcall	owbrea			;Bit einlesen
	ror	r19			;Bit aus Carry einschieben
	dec	r18			;alle Bits gelesen?
	brne	owre10			;nein -> Schleife
	ret
;
; 1-Wire, Schreiben eines Bits
; Register r16
;	   r20 <- Busnummer (0-7)
;	   Carry <- zu schreibendes Bit
;
owbwri:	mov	r16,r20			;Busnummer kopieren
	tst	r16			;Busnummer=0?
	breq	ow0wri			;ja -> bearbeiten
	dec	r16			;Busnummer=1?
	breq	ow1wri			;ja -> bearbeiten
	dec	r16			;Busnummer=2?
	breq	ow2wri			;ja -> bearbeiten
	dec	r16			;Busnummer=3?
	breq	ow3wri			;ja -> bearbeiten
	dec	r16			;Busnummer=4?
	breq	ow4wri			;ja -> bearbeiten
	dec	r16			;Busnummer=5?
	breq	ow5wri			;ja -> bearbeiten
	dec	r16			;Busnummer=6?
	breq	ow6wri			;ja -> bearbeiten
	rjmp	ow7wri			;sonst Busnummer 7 bearbeiten
;
ow0wri:	sbi	ddrc,0			;Port auf Ausgang Low setzen
	brcs	ow0w10			;1-Bit? ja -> weiter
	rcall	w060us			;60s warten (=Low)
	cbi	ddrc,0			;Port auf Eingang setzen
	rcall	w005us			;4,5s warten (Pause)
	ret
ow0w10:	rcall	w002us			;2s warten (=High)
	cbi	ddrc,0			;Port auf Eingang setzen
	rcall	w060us			;60s warten
	ret
;
ow1wri:	sbi	ddrc,1			;Port auf Ausgang Low setzen
	brcs	ow1w10			;1-Bit? ja -> weiter
	rcall	w060us			;60s warten (=Low)
	cbi	ddrc,1			;Port auf Eingang setzen
	rcall	w005us			;4,5s warten (Pause)
	ret
ow1w10:	rcall	w002us			;2s warten (=High)
	cbi	ddrc,1			;Port auf Eingang setzen
	rcall	w060us			;60s warten
	ret
;
ow2wri:	sbi	ddrc,2			;Port auf Ausgang Low setzen
	brcs	ow2w10			;1-Bit? ja -> weiter
	rcall	w060us			;60s warten (=Low)
	cbi	ddrc,2			;Port auf Eingang setzen
	rcall	w005us			;4,5s warten (Pause)
	ret
ow2w10:	rcall	w002us			;2s warten (=High)
	cbi	ddrc,2			;Port auf Eingang setzen
	rcall	w060us			;60s warten
	ret
;
ow3wri:	sbi	ddrc,3			;Port auf Ausgang Low setzen
	brcs	ow3w10			;1-Bit? ja -> weiter
	rcall	w060us			;60s warten (=Low)
	cbi	ddrc,3			;Port auf Eingang setzen
	rcall	w005us			;4,5s warten (Pause)
	ret
ow3w10:	rcall	w002us			;2s warten (=High)
	cbi	ddrc,3			;Port auf Eingang setzen
	rcall	w060us			;60s warten
	ret
;
ow4wri:	sbi	ddrd,4			;Port auf Ausgang Low setzen
	brcs	ow4w10			;1-Bit? ja -> weiter
	rcall	w060us			;60s warten (=Low)
	cbi	ddrd,4			;Port auf Eingang setzen
	rcall	w005us			;4,5s warten (Pause)
	ret
ow4w10:	rcall	w002us			;2s warten (=High)
	cbi	ddrd,4			;Port auf Eingang setzen
	rcall	w060us			;60s warten
	ret
;
ow5wri:	sbi	ddrd,5			;Port auf Ausgang Low setzen
	brcs	ow5w10			;1-Bit? ja -> weiter
	rcall	w060us			;60s warten (=Low)
	cbi	ddrd,5			;Port auf Eingang setzen
	rcall	w005us			;4,5s warten (Pause)
	ret
ow5w10:	rcall	w002us			;2s warten (=High)
	cbi	ddrd,5			;Port auf Eingang setzen
	rcall	w060us			;60s warten
	ret
;
ow6wri:	sbi	ddrd,6			;Port auf Ausgang Low setzen
	brcs	ow6w10			;1-Bit? ja -> weiter
	rcall	w060us			;60s warten (=Low)
	cbi	ddrd,6			;Port auf Eingang setzen
	rcall	w005us			;4,5s warten (Pause)
	ret
ow6w10:	rcall	w002us			;2s warten (=High)
	cbi	ddrd,6			;Port auf Eingang setzen
	rcall	w060us			;60s warten
	ret
;
ow7wri:	sbi	ddrd,7			;Port auf Ausgang Low setzen
	brcs	ow7w10			;1-Bit? ja -> weiter
	rcall	w060us			;60s warten (=Low)
	cbi	ddrd,7			;Port auf Eingang setzen
	rcall	w005us			;4,5s warten (Pause)
	ret
ow7w10:	rcall	w002us			;2s warten (=High)
	cbi	ddrd,7			;Port auf Eingang setzen
	rcall	w060us			;60s warten
	ret
;
; 1-Wire, Lesen eines Bits
; Register r16,r17
;	   r20 <- Busnummer (0-7)
;	   Carry -> gelesenes Bit
;
owbrea:	mov	r16,r20			;Busnummer kopieren
	tst	r16			;Busnummer=0?
	breq	ow0rea			;ja -> bearbeiten
	dec	r16			;Busnummer=1?
	breq	ow1rea			;ja -> bearbeiten
	dec	r16			;Busnummer=2?
	breq	ow2rea			;ja -> bearbeiten
	dec	r16			;Busnummer=3?
	breq	ow3rea			;ja -> bearbeiten
	dec	r16			;Busnummer=4?
	breq	ow4rea			;ja -> bearbeiten
	dec	r16			;Busnummer=5?
	breq	owbr10			;ja -> bearbeiten
	dec	r16			;Busnummer=6?
	breq	owbr20			;ja -> bearbeiten
	rjmp	ow7rea			;sonst Busnummer 7 bearbeiten
owbr10:	rjmp	ow5rea
owbr20:	rjmp	ow6rea
;
ow0rea:	sbi	ddrc,0			;Port auf Ausgang Low setzen
	rcall	w002us			;2s warten
	cbi	ddrc,0			;Port auf Eingang setzen
	rcall	w005us			;4,5s warten (Pause)
	rcall	w005us			;4,5s warten (Pause)
	in	r17,pinc		;Port lesen
	andi	r17,0x01		;Portpin selektieren
	clc				;Carry lschen (=0)
	breq	ow0r10			;Port=0? ja -> weiter
	sec				;sonst Carry setzen (=1)
ow0r10:	rcall	w050us			;50s warten
	ret
;
ow1rea:	sbi	ddrc,1			;Port auf Ausgang Low setzen
	rcall	w002us			;2s warten
	cbi	ddrc,1			;Port auf Eingang setzen
	rcall	w005us			;4,5s warten (Pause)
	rcall	w005us			;4,5s warten (Pause)
	in	r17,pinc		;Port lesen
	andi	r17,0x02		;Portpin selektieren
	clc				;Carry lschen (=0)
	breq	ow1r10			;Port=0? ja -> weiter
	sec				;sonst Carry setzen (=1)
ow1r10:	rcall	w050us			;50s warten
	ret
;
ow2rea:	sbi	ddrc,2			;Port auf Ausgang Low setzen
	rcall	w002us			;2s warten
	cbi	ddrc,2			;Port auf Eingang setzen
	rcall	w005us			;4,5s warten (Pause)
	rcall	w005us			;4,5s warten (Pause)
	in	r17,pinc		;Port lesen
	andi	r17,0x04		;Portpin selektieren
	clc				;Carry lschen (=0)
	breq	ow2r10			;Port=0? ja -> weiter
	sec				;sonst Carry setzen (=1)
ow2r10:	rcall	w050us			;50s warten
	ret
;
ow3rea:	sbi	ddrc,3			;Port auf Ausgang Low setzen
	rcall	w002us			;2s warten
	cbi	ddrc,3			;Port auf Eingang setzen
	rcall	w005us			;4,5s warten (Pause)
	rcall	w005us			;4,5s warten (Pause)
	in	r17,pinc		;Port lesen
	andi	r17,0x08		;Portpin selektieren
	clc				;Carry lschen (=0)
	breq	ow3r10			;Port=0? ja -> weiter
	sec				;sonst Carry setzen (=1)
ow3r10:	rcall	w050us			;50s warten
	ret
;
ow4rea:	sbi	ddrd,4			;Port auf Ausgang Low setzen
	rcall	w002us			;2s warten
	cbi	ddrd,4			;Port auf Eingang setzen
	rcall	w005us			;4,5s warten (Pause)
	rcall	w005us			;4,5s warten (Pause)
	in	r17,pind		;Port lesen
	andi	r17,0x10		;Portpin selektieren
	clc				;Carry lschen (=0)
	breq	ow4r10			;Port=0? ja -> weiter
	sec				;sonst Carry setzen (=1)
ow4r10:	rcall	w050us			;60s warten
	ret
;
ow5rea:	sbi	ddrd,5			;Port auf Ausgang Low setzen
	rcall	w002us			;2s warten
	cbi	ddrd,5			;Port auf Eingang setzen
	rcall	w005us			;4,5s warten (Pause)
	rcall	w005us			;4,5s warten (Pause)
	in	r17,pind		;Port lesen
	andi	r17,0x20		;Portpin selektieren
	clc				;Carry lschen (=0)
	breq	ow5r10			;Port=0? ja -> weiter
	sec				;sonst Carry setzen (=1)
ow5r10:	rcall	w050us			;60s warten
	ret
;
ow6rea:	sbi	ddrd,6			;Port auf Ausgang Low setzen
	rcall	w002us			;2s warten
	cbi	ddrd,6			;Port auf Eingang setzen
	rcall	w005us			;4,5s warten (Pause)
	rcall	w005us			;4,5s warten (Pause)
	in	r17,pind		;Port lesen
	andi	r17,0x40		;Portpin selektieren
	clc				;Carry lschen (=0)
	breq	ow6r10			;Port=0? ja -> weiter
	sec				;sonst Carry setzen (=1)
ow6r10:	rcall	w050us			;60s warten
	ret
;
ow7rea:	sbi	ddrd,7			;Port auf Ausgang Low setzen
	rcall	w002us			;2s warten
	cbi	ddrd,7			;Port auf Eingang setzen
	rcall	w005us			;4,5s warten (Pause)
	rcall	w005us			;4,5s warten (Pause)
	in	r17,pind		;Port lesen
	andi	r17,0x80		;Portpin selektieren
	clc				;Carry lschen (=0)
	breq	ow7r10			;Port=0? ja -> weiter
	sec				;sonst Carry setzen (=1)
ow7r10:	rcall	w050us			;60s warten
	ret
;
; 1-Wire, Strong Pullup, Bus auf High legen whrend Temperaturmessung
; Register r16
;	   r20 <- Busnummer
;
owstpu:	cpi	r20,0			;Busnummer=0?
	breq	ow0stp			;ja -> bearbeiten
	cpi	r20,1			;Busnummer=1?
	breq	ow1stp			;ja -> bearbeiten
	cpi	r20,2			;Busnummer=2?
	breq	ow2stp			;ja -> bearbeiten
	cpi	r20,3			;Busnummer=3?
	breq	ow3stp			;ja -> bearbeiten
	cpi	r20,4			;Busnummer=4?
	breq	ow4stp			;ja -> bearbeiten
	cpi	r20,5			;Busnummer=5?
	breq	ow5stp			;ja -> bearbeiten
	cpi	r20,6			;Busnummer=6?
	breq	ow6stp			;ja -> bearbeiten
	rjmp	ow7stp			;sonst Busnummer 7 bearbeiten
;
ow0stp:	sbi	portc,0			;Portausgang auf High setzen
	sbi	ddrc,0			;Port auf Ausgang umschalten
	ret
;
ow1stp:	sbi	portc,1			;Portausgang auf High setzen
	sbi	ddrc,1			;Port auf Ausgang umschalten
	ret
;
ow2stp:	sbi	portc,2			;Portausgang auf High setzen
	sbi	ddrc,2			;Port auf Ausgang umschalten
	ret
;
ow3stp:	sbi	portc,3			;Portausgang auf High setzen
	sbi	ddrc,3			;Port auf Ausgang umschalten
	ret
;
ow4stp:	sbi	portd,4			;Portausgang auf High setzen
	sbi	ddrd,4			;Port auf Ausgang umschalten
	ret
;
ow5stp:	sbi	portd,5			;Portausgang auf High setzen
	sbi	ddrd,5			;Port auf Ausgang umschalten
	ret
;
ow6stp:	sbi	portd,6			;Portausgang auf High setzen
	sbi	ddrd,6			;Port auf Ausgang umschalten
	ret
;
ow7stp:	sbi	portd,7			;Portausgang auf High setzen
	sbi	ddrd,7			;Port auf Ausgang umschalten
	ret
;
; CRC-Berechnung fr 1-Wire-Devices fr 8 Byte im 1-Wire-Puffer, nach
; Application Note 27 von Dallas/Maxim
; Register r10,r11,r16,r18,r19,x
;	   r17 -> ermittelter CRC-Wert
;
crc:	ldi	xl,low(owbuff+8)
	ldi	xh,high(owbuff+8)	;Zeiger auf 1-Wire-Puffer-Ende
	ldi	r19,8			;8 Bytes berechnen
	ldi	r16,0x18		;Konstante fr Berechnung
	mov	r10,r16			;bereitlegen
	clr	r17			;CRC-Startwert=0 setzen
;
crc100:	ld	r18,-x			;Byte (LSB) aus Puffer holen
	mov	r11,r18			;Byte zwischenspeichern
	ldi	r16,8			;8 Verschiebungen
crc200:	eor	r18,r17			;CRC berechnen
	ror	r18			;Ergebnis in Carry schieben
	mov	r18,r17			;letzten CRC-Wert holen
	brcc	crc300			;war Ergebnis=0? ja -> weiter
	eor	r18,r10			;sonst CRC-Wert updaten
crc300:	ror	r18			;neuen CRC-Wert positionieren
	mov	r17,r18			;und speichern
	mov	r18,r11			;restliche Bits holen
	bst	r18,0			;LSB sichern, nchste Bitpos.,
	ror	r18			;gesichertes Bit als MSB ein-
	bld	r18,7			;setzen (Ersatz fr RR-Befehl)
	mov	r11,r18			;und Wert zwischenspeichern
	dec	r16			;alle Verschiebungen erledigt?
	brne	crc200			;nein -> Schleife
	dec	r19			;alle Bytes bearbeitet?
	brne	crc100			;nein -> Schleife
	ret
;
; Ausgabe des LCD-Text-Puffers
; Register: r16,r17,r18,r19,r20,x
;
lctout:	rcall	twista			;TWI Start Condition
	ldi	r17,0x70		;LCD-Adresse und W-Bit
	rcall	twisnd			;Byte senden
	ldi	r17,0xe0		;LCD Device Select 0
	rcall	twisnd			;Byte senden
	ldi	r17,0			;LCD Data Pointer, Adresse 0
	rcall	twisnd			;Byte senden
	ldi	r20,8			;8 Zeichen ausgeben
	ldi	xl,low(lcdbuf)
	ldi	xh,high(lcdbuf)		;Zeiger auf LCD-Puffer
out010:	ld	r18,x+			;ASCII-Zeichen holen
	rcall	asclcd			;in 16-Segmentcode wandeln
	mov	r17,r18			;Byte 1 holen
	rcall	twisnd			;auf LCD ausgeben
	mov	r17,r19			;Byte 2 holen
	rcall	twisnd			;auf LCD ausgeben
	dec	r20			;alle Zeichen ausgegeben?
	brne	out010			;nein -> Schleife
	rcall	twisto			;TWI Stop Condition
	ret
;
; Alarm-Ausgabe als Sonderzeichen auf dem LCD (Zeichen 1-4), auerdem
; Anzeige des Speichermodus (Stereo-Zeichen) und Anzeige des Spezial-
; modus (3 Feldstrke-Balken)
; Register: r16,r17,x
;
lcaout:	rcall	twista			;TWI Start Condition
	ldi	r17,0x70		;LCD-Adresse und W-Bit
	rcall	twisnd			;Byte senden
	ldi	r17,0xe0		;LCD Device Select 0
	rcall	twisnd			;Byte senden
	ldi	r17,32			;LCD Data Pointer, Adresse 32
	rcall	twisnd			;Byte senden
	clr	r17			;alle Sonderz.-Segmente aussch.
	bst	flags,3			;Speichermodus-Flag holen
	bld	r17,7			;und als "Stereo"-Symbol anz.
	lds	r16,jflags		;Jumper-Flags holen
	bst	r16,1			;Spezial-Jumper-Status holen
	bld	r17,2			;und als "Balken 3" anzeigen
	bld	r17,0			;"Balken 2" anzeigen
	rcall	twisnd			;Byte 1 auf LCD ausgeben
	clr	r17			;alle Sonderz.-Segmente aussch.
	bld	r17,4			;"Balken 1" anzeigen
	ldi	xl,low(alstat)
	ldi	xh,high(alstat)		;Zeiger auf Alarmstatus
	ld	r16,x+			;Alarmstatus 1 holen
	bst	r16,1			;letzten stabilen Status lesen
	bld	r17,0			;und als Sonderzeichen einfgen
	ld	r16,x+			;Alarmstatus 2 holen
	bst	r16,1			;letzten stabilen Status lesen
	bld	r17,2			;und als Sonderzeichen einfgen
	ld	r16,x+			;Alarmstatus 3 holen
	bst	r16,1			;letzten stabilen Status lesen
	bld	r17,1			;und als Sonderzeichen einfgen
	ld	r16,x+			;Alarmstatus 4 holen
	bst	r16,1			;letzten stabilen Status lesen
	bld	r17,3			;und als Sonderzeichen einfgen
	rcall	twisnd			;Byte 2 auf LCD ausgeben
	rcall	twisto			;TWI Stop Condition
	ret
;
; Wandlung eines ASCII-Zeichens in 16-Segmentcode fr LCD
; Register r0,r16
;	   r18 <- ASCII-Zeichen
;	   r18 -> 16-Segmentcode Byte 1
;	   r19 -> 16-Segmentcode Byte 2
;
asclcd:	cpi	r18,' '			;Zeichencode<20h?
	brcs	asc010			;ja -> in Leerzeichen wandeln
	cpi	r18,0x60		;Zeichencode>=60h?
	brcs	asc020			;nein -> weiter
	cpi	r18,0x80		;Zeichencode>=80h?
	brcc	asc010			;ja -> in Leerzeichen wandeln
	subi	r18,0x20		;sonst in Grobuchstaben wand.
	rjmp	asc020
asc010:	ldi	r18,' '			;sonst in Leerzeichen wandeln
asc020:	subi	r18,0x20		;Zeichencode -> Tabellenoffset
	add	r18,r18			;Tabellenoffset verdoppeln
	clr	r16			;H-Byte fr Tabellenoffset
	ldi	zl,low(2*lcdtab)
	ldi	zh,high(2*lcdtab)	;Zeiger auf LCD-Segmenttabelle
	add	zl,r18			;Offset L addieren
	adc	zh,r16			;Offset H addieren
	lpm	r18,z+			;16-Segmentcode Byte 1 holen
	lpm	r19,z			;16-Segmentcode Byte 2 holen
	ret
;
; Warteschleife ca. 50ms
; Register r14,r15
;
wait50:	clr	r14
	clr	r15			;Startwerte fr ca. 50ms
wai010:	dec	r15
	brne	wai010			;innere Schleife bearbeiten
	dec	r14
	brne	wai010			;uere Schleife bearbeiten
	ret
;
; Warteschleife 500s
; Register r15,r16
;
w500us:	ldi	r16,3			;Zeitkonstante uere Schleife
	mov	r15,r16			;kopieren
	ldi	r16,50			;Zeitkonstante innere Schleife
w500xx:	dec	r16
	brne	w500xx			;innere Schleife bearbeiten
	dec	r15
	brne	w500xx			;uere Schleife bearbeiten
	ret	
;
; Warteschleife 240s
; Register r15,r16
;
w240us:	ldi	r16,1			;Zeitkonstante uere Schleife
	mov	r15,r16			;kopieren
	ldi	r16,254			;Zeitkonstante innere Schleife
	rjmp	w500xx
;
; Warteschleife 60s
; Register r16
;
w060us:	ldi	r16,76			;Zeitkonstante laden
w060xx:	dec	r16
	brne	w060xx			;Schleife bearbeiten
	ret
;
; Warteschleife 50s
; Register r16
;
w050us:	ldi	r16,65			;Zeitkonstante laden
w050xx:	dec	r16
	brne	w050xx			;Schleife bearbeiten
	ret
;
; Warteschleife 4,5s
; Register r16
;
w005us:	ldi	r16,4			;Zeitkonstante laden
w005xx:	dec	r16
	brne	w005xx			;Schleife bearbeiten
	ret
;
; Warteschleife 2s
; Register r16
;
w002us:	nop
	ret
;
; Alarmausgabe ber RS-232
; Register r16
;	   r19 <- Alarmwert (0/1)
;	   r20 <- Alarmnummer (0-3)
;
rsaout:	mov	r16,r20			;aktuelle Alarmnummer holen
	inc	r16			;Nummer korrigieren
	ori	r16,0x40		;in ASCII-Zeichen wandeln
	rcall	rssend			;Zeichen senden
	ldi	r16,':'			;Doppelpunkt laden
	rcall	rssend			;Zeichen senden
	ldi	r16,'0'			;Zeichen '0' laden
	add	r16,r19			;Alarmwert addieren
	rcall	rssend			;Zeichen senden
	ldi	r16,cr			;CR-Zeichen laden
	rcall	rssend			;Zeichen senden
	ret
;
; USART Senden (RS-232)
; Register r16 <- zu sendendes Zeichen
;
rssend:	sbis	ucsra,udre		;Sendepuffer leer?
	rjmp	rssend			;nein -> warten
	out	udr,r16			;sonst Byte senden
	ret
;
; TWI Start Condition
; Register r16
;
twista:	ldi	r16,(1<<twint)|(1<<twsta)|(1<<twen)
	out	twcr,r16		;Start Condition senden
twi100:	in	r16,twcr		;Status holen
	sbrs	r16,twint		;Kommando ausgefhrt?
	rjmp	twi100			;nein -> warten
	ret
;
; TWI Senden
; Register r16
;	   r17 <- zu sendendes Byte
;
twisnd:	out	twdr,r17		;Byte in Ausgaberegister
	ldi	r16,(1<<twint)|(1<<twen)
	out	twcr,r16		;Byte senden
	rjmp	twi100			;Warten bis ausgefhrt
;
; TWI Stop Condition
; Register r16
;
twisto:	ldi	r16,(1<<twint)|(1<<twsto)|(1<<twen)
	out	twcr,r16		;Stop Condition senden
	ret
;
; Lesen eines Bytes aus dem EEPROM
; Register r16 <- EEPROM-Adresse L
;	   r17 <- EEPROM-Adresse H
;	   r18 -> EEPROM-Daten
;
eeread:	sbic	eecr,eewe		;luft Schreibzyklus?
	rjmp	eeread			;ja - warten
	out	eearl,r16		;EEPROM-Adresse L setzen
	out	eearh,r17		;EEPROM-Adresse H setzen
	sbi	eecr,eere		;EEPROM lesen aktivieren
	in	r18,eedr		;Daten lesen
	ret
;
; Schreiben eines Bytes in den EEPROM
; Register r16 <- EEPROM-Adresse L
;	   r17 <- EEPROM-Adresse H
;	   r18 <- EEPROM-Daten
;
eewrit:	sbic	eecr,eewe		;luft Schreibzyklus?
	rjmp	eewrit			;ja -> warten
	out	eearh,r17		;EEPROM-Adresse H setzen
	out	eearl,r16		;EEPROM-Adresse L setzen
	out	eedr,r18		;EEPROM-Datenbyte setzen
	sbi	eecr,eemwe		;Masterwrite-Bit setzen
	sbi	eecr,eewe		;Write-Bit setzen
	ret
;
; Timer1 Interrupt, wird alle 100ms aufgerufen
; Abfrage der Alarmeingnge
; Register isreg,itemp,y
;
timer1:	in	isreg,sreg		;SREG sichern
	ldi	yl,low(alstat)
	ldi	yh,high(alstat)		;Zeiger auf Alarmstatus
	ld	itemp2,y		;Alarmstatus holen
	in	itemp1,pinb		;Alarmeingnge 1 und 2 lesen
	com	itemp1			;und invertieren
	bst	itemp1,0		;Alarmbit 1 holen
	bld	itemp2,0		;und in Status einsetzen
	st	y+,itemp2		;Alarmstatus speichern
	ld	itemp2,y		;nchsten Alarmstatus holen
	bst	itemp1,1		;Alarmbit 2 holen
	bld	itemp2,0		;und in Status einsetzen
	st	y+,itemp2		;Alarmstatus speichern
	ld	itemp2,y		;nchsten Alarmstatus holen
	in	itemp1,pind		;Alarmeingnge 3 und 4 lesen
	com	itemp1			;und invertieren
	bst	itemp1,2		;Alarmbit 3 holen
	bld	itemp2,0		;und in Status einsetzen
	st	y+,itemp2		;Alarmstatus speichern
	ld	itemp2,y		;nchsten Alarmstatus holen
	bst	itemp1,3		;Alarmbit 2 holen
	bld	itemp2,0		;und in Status einsetzen
	st	y+,itemp2		;Alarmstatus speichern
;
; Steuerung der Alarme (Flags, Alarmzhler)
;
	ldi	yl,low(alstat)
	ldi	yh,high(alstat)		;Zeiger auf Alarmstatus
	ldi	itemp2,almtio		;Alarm-Timeout-Wert laden
	mov	itemp1,itemp2		;und vorbereiten
tim100:	ld	itemp2,y		;Alarmstatus holen
	andi	itemp2,0b11		;nur Statusbits 0/1 bewerten
	cpi	itemp2,0b00		;Status gleich geblieben?
	breq	tim130			;ja -> berspringen
	cpi	itemp2,0b11		;Status gleich geblieben?
	breq	tim130			;ja -> berspringen
	ld	itemp2,y		;Alarmstatus nochmals holen
	bst	itemp2,2		;Bearbeitungs-Bit gesetzt?
	brts	tim110			;ja -> weiter
	sbr	itemp2,0b100		;sonst Bearbeitungs-Flag setzen
	st	y,itemp2		;Alarmstatus wieder speichern
	std	y+4,itemp1		;Alarmzhler neu laden
tim110:	ldd	itemp2,y+4		;Alarmzhler holen
	cpi	itemp2,0		;Zhler abgelaufen?
	brne	tim120			;nein -> berspringen
	ld	itemp2,y		;Alarmstatus nochmals holen
	bst	itemp2,0		;aktuelles Statusbit holen
	bld	itemp2,1		;und als neuen Status setzen
	cbr	itemp2,0b100		;Bearbeitungs-Flag lschen
	sbr	itemp2,0b1000		;Melde-Flag setzen
	st	y,itemp2		;neuen Alarmstatus speichern
	rjmp	tim200
tim120:	dec	itemp2			;sonst Zhler vermindern
	std	y+4,itemp2		;und wieder speichern
	rjmp	tim200
tim130:	ld	itemp2,y		;Alarmstatus nochmals holen
	cbr	itemp2,0b100		;Bearbeitungs-Flag lschen
	st	y,itemp2		;neuen Alarmstatus speichern
tim200:	adiw	yl,1			;Zeiger erhhen
	cpi	yl,low(alstat+4)	;alle Werte bearbeitet?
	brne	tim100			;nein -> Schleife
;
; Interrupt-Zhler bearbeiten, bei Zhlerstand 0 (entspricht Sekunde 0)
; wird das Temperaturmess-Flag gesetzt, bei Sekunde 1 das Messende-Flag
; und im Normalmodus bei Sekunde 2,3,4,5,..,17 das Sensor-Flag; im
; Spezialmodus wird das Spezial-Flag bei Sekunde 1.2 gesetzt
;
	lds	itemp2,icoun2		;Vorteiler holen
	inc	itemp2			;erhhen
	sts	icoun2,itemp2		;und wieder speichern
	cpi	itemp2,10		;Endwert 10 erreicht?
	brcs	tim400			;nein -> Ende
	clr	itemp2
	sts	icoun2,itemp2		;Vorteiler zurcksetzen
	lds	itemp2,icoval		;Interrupt-Zhler-Endwert holen
	mov	itemp1,itemp2		;und kopieren
	lds	itemp2,icount		;Interrupt-Zhler holen
	inc	itemp2			;Zhler erhhen
	cp	itemp2,itemp1		;Zhler-Endwert erreicht?
	breq	tim300			;ja -> Zhler zurcksetzen

	cpi	itemp2,2		;Zhler=2?
	brne	tim305			;nein -> weiter
	lds	itemp1,jflags		;sonst Jumper-Flags holen
	bst	itemp1,1		;Spezial-Jumper gesetzt?
	brtc	tim305			;nein -> weiter

tim300:	clr	itemp2			;sonst Zhler zurcksetzen
tim305:	sts	icount,itemp2		;Interrupt-Zhler speichern
	cpi	itemp2,0		;Temperaturmessung fllig?
	brne	tim310			;nein -> weiter
	sbr	flags,0b001		;sonst TempMess-Flag setzen
	rjmp	tim500			;Ende
tim310:	cpi	itemp2,1		;Temperaturmessung beenden?
	brne	tim320			;nein -> weiter
	sbr	flags,0b010		;sonst Messende-Flag setzen
	rjmp	tim500			;Ende
tim320:	tst	itemp2			;Interrupt-Zhler negativ?
	brmi	tim500			;ja -> Ende
	sbr	flags,0b100		;sonst Sensor-Flag setzen
	rjmp	tim500			;Ende
;
tim400:	cpi	itemp2,2		;Vorteiler=1 (200ms)?
	brne	tim500			;nein -> Ende
	lds	itemp2,icount		;Interrupt-Zhler holen
	cpi	itemp2,1		;Zhler=1 (mit icoun2= 1.2s)?
	brne	tim500			;nein -> Ende
	lds	itemp2,jflags		;Jumper-Flags holen
	bst	itemp2,1		;Spezial-Jumper gesetzt?
	brtc	tim500			;nein -> Ende
	sbr	flags,0x10		;sonst Spezial-Flag setzen
;
tim500:	out	sreg,isreg		;SREG wieder herstellen
	reti
;
;
; ------------------
; Tabellen und Texte
; ------------------
;
; LCD-Segmenttabelle
;
lcdtab:	.db	0x00,0x00	;20h (32) Leerzeichen
	.db	0x04,0x20	;21h (33) !
	.db	0x80,0x20	;22h (34) "
	.db	0xa8,0x8c	;23h (35) # ->  (Grad-Zeichen)
	.db	0xad,0xb5	;24h (36) $
	.db	0xec,0x37	;25h (37) %
	.db	0x3b,0x71	;26h (38) &
	.db	0x00,0x02	;27h (39) '
	.db	0x00,0x42	;28h (40) (
	.db	0x42,0x00	;29h (41) )
	.db	0x66,0x66	;2ah (42) *
	.db	0x24,0x24	;2bh (43) +
	.db	0x40,0x00	;2ch (44) ,
	.db	0x00,0x04	;2dh (45) -
	.db	0x01,0x00	;2eh (46) .
	.db	0x40,0x02	;2fh (47) /
	.db	0x99,0x99	;30h (48) 0
	.db	0x00,0x0b	;31h (49) 1
	.db	0x39,0x9c	;32h (50) 2
	.db	0x29,0x9d	;33h (51) 3
	.db	0xa0,0x0d	;34h (52) 4
	.db	0xa9,0x95	;35h (53) 5
	.db	0xb9,0x95	;36h (54) 6
	.db	0x08,0x89	;37h (55) 7
	.db	0xb9,0x9d	;38h (56) 8
	.db	0xa9,0x9d	;39h (57) 9
	.db	0x21,0x00	;3ah (58) :
	.db	0x21,0x00	;3bh (59) ;
	.db	0x00,0x42	;3ch (60) <
	.db	0x21,0x00	;3dh (61) =
	.db	0x42,0x00	;3eh (62) >
	.db	0x8c,0x8c	;3fh (63) ?
	.db	0x3d,0x99	;40h (64) @
	.db	0xb8,0x8d	;41h (65) A
	.db	0x0d,0xbd	;42h (66) B
	.db	0x99,0x90	;43h (67) C
	.db	0x0d,0xb9	;44h (68) D
	.db	0xb9,0x94	;45h (69) E
	.db	0xb8,0x84	;46h (70) F
	.db	0x99,0x95	;47h (71) G
	.db	0xb0,0x0d	;48h (72) H
	.db	0x0d,0xb0	;49h (73) I
	.db	0x11,0x19	;4ah (74) J
	.db	0xb0,0x42	;4bh (75) K
	.db	0x91,0x10	;4ch (76) L
	.db	0x92,0x0b	;4dh (77) M
	.db	0x92,0x49	;4eh (78) N
	.db	0x99,0x99	;4fh (79) O
	.db	0xb8,0x8c	;50h (80) P
	.db	0x99,0xd9	;51h (81) Q
	.db	0xb8,0xcc	;52h (82) R
	.db	0xa9,0x95	;53h (83) S
	.db	0x0c,0xa0	;54h (84) T
	.db	0x91,0x19	;55h (85) U
	.db	0xd0,0x02	;56h (86) V
	.db	0xd0,0x49	;57h (87) W
	.db	0x42,0x42	;58h (88) X
	.db	0xa1,0x1d	;59h (89) Y
	.db	0x49,0x92	;5ah (90) Z
	.db	0x99,0x00	;5bh (91) [
	.db	0x02,0x40	;5ch (92) \
	.db	0x00,0x99	;5dh (93) ]
	.db	0x40,0x40	;5eh (94) ^
	.db	0x01,0x10	;5fh (95) _
;
verstx:	.db	"SFH V107"	;Versionsnummer
meastx:	.db	"MESSUNG "
sentxt:	.db	"SENDEN  "
senstx:	.db	"K.SENSOR"
errtxt:	.db	"FEHLER"
;
;
; ------------
; EEPROM-Daten
; ------------
;
.eseg
;
dummy:	.byte	1	;Byte 0 wird nicht genutzt
;
eprflg:	.byte	1	;EEPROM-Flag, wenn Wert=0, dann wird keine
			;1-Wire-Suche durchgefhrt und stattdessen die
			;folgende Sensortabelle verwendet
;
esenid:	.byte	16*8	;8-Byte-ROM-IDs der 16 Temperatursensoren
esentb:	.byte	16	;Sensor-Tabelle, enthlt fr jeden der 16 Sen-
			;soren die Nummer des 1-Wire-Busses (0-7), an
			;dem er angeschlossen ist, fr nicht vorhandene
			;Sensoren wird 0xff eingetragen
;