; ====================================================
; Anzeigemodul 3 fr Temperaturmesssystem mit ATmega16
; von Scott-Falk Hhn (s.huehn@soemtron.de)
; Version 2.00, letzte Bearbeitung am 20.01.2013
; ====================================================
;
; --------------------------------------------
; Programmierung der Fusebits (Atmel Studio 6)
; --------------------------------------------
;
; OCDEN =     [ ]
; JTAGEN =    [ ]
; SPIEN =     [X]
; EESAVE =    [X]
; BOOTSZ =    1024W_1C00
; BOOTRST =   [ ]
; CKOPT =     [ ]
; BODLEVEL =  4V0
; BODEN =     [X]
; SUT_CKSEL = EXTHIFXTALRES_16KCK_64MS
;
; HIGH = 0xD1
; LOW =  0x3F
;
; ----------------------
; Belegung der I/O-Ports
; ----------------------
;
; A0: Eingang Taster (Konfiguration), L-aktiv, Pull-Up
; A1: Eingang Jumper (LED-Invertierung), L-aktiv, Pull-Up
; A2: Eingang, frei, Pull-Up
; A3: Eingang, frei, Pull-Up
; A4: Eingang, frei, Pull-Up
; A5: Eingang, frei, Pull-Up
; A6: Eingang, frei, Pull-Up
; A7: Eingang Lichtsensor, analog
;
; B0: Ausgang Anodentreiber 0 (LED 0+4, erste Stelle)
; B1: Ausgang Anodentreiber 1 (LED 1+5, zweite Stelle)
; B2: Ausgang Anodentreiber 2 (LED 2+6, dritte Stelle)
; B3: Ausgang Anodentreiber 3 (LED 3+7, vierte Stelle)
; B4: Ausgang Segmente 1A (obere Anzeige einer Multiplex-Gruppe)
; B5: Eingang MOSI (ISP), Pull-Up
; B6: Eingang MISO (ISP), Pull-Up
; B7: Eingang SCK (ISP), Pull-Up
;
; C0: Ausgang Segmente 2A (untere Anzeige einer Multiplex-Gruppe)
; C1: Ausgang Segmente 2B (untere Anzeige einer Multiplex-Gruppe)
; C2: Ausgang Segmente 2C (untere Anzeige einer Multiplex-Gruppe)
; C3: Ausgang Segmente 2D (untere Anzeige einer Multiplex-Gruppe)
; C4: Ausgang Segmente 2E (untere Anzeige einer Multiplex-Gruppe)
; C5: Ausgang Segmente 2F (untere Anzeige einer Multiplex-Gruppe)
; C6: Ausgang Segmente 2G (untere Anzeige einer Multiplex-Gruppe)
; C7: Ausgang Segmente 2P (untere Anzeige einer Multiplex-Gruppe)
;
; D0: Eingang RS-232-RXD, Pull-Up
; D1: Ausgang Segmente 1B (obere Anzeige einer Multiplex-Gruppe)
; D2: Ausgang Segmente 1C (obere Anzeige einer Multiplex-Gruppe)
; D3: Ausgang Segmente 1D (obere Anzeige einer Multiplex-Gruppe)
; D4: Ausgang Segmente 1E (obere Anzeige einer Multiplex-Gruppe)
; D5: Ausgang Segmente 1F (obere Anzeige einer Multiplex-Gruppe)
; D6: Ausgang Segmente 1G (obere Anzeige einer Multiplex-Gruppe)
; D7: Ausgang Segmente 1P (obere Anzeige einer Multiplex-Gruppe)
;
; ------------------
; Belegung der Timer
; ------------------
;
; Timer1: Haupttimer fr Multiplex-Anzeige, Taster-Entprellung und Auslesen des analogen Lichtsensorwertes, 800Hz / 1,25ms
; Timer2: Timer fr PWM-Erzeugung zur Helligkeitssteuerung der Anzeige, wird bei der Multiplex-Ausgabe auf den berechneten PWM-Wert gesetzt und mit 125kHz / 8s bis zum Overflow
;	  gezhlt
;
.nolist
.include "m16def.inc"
.list
;
; --------------------
; Konstanten festlegen
; --------------------
;
.equ	clock=	4000000			;Controller Taktfrequenz= 4,0MHz
.equ	tm1frq=	800			;Timer1 Frequenz= 800Hz (1,25s)
.equ	tm1ocr=	clock / tm1frq - 1	;Timer1 Teilerwert fr OCR1a
.equ	baudr=	9600			;RS-232 Baudrate= 9600
.equ	lf=	10			;Zeichencode fr Line Feed
.equ	cr=	13			;Zeichencode Cariage Return (Enter)
.equ	keytio=	50			;Langer Tastendruck= 1s (20ms-Einheiten)
.equ	vertio=	200			;Versionsnummer-Zeit= 4s (20ms-Einheiten)
.equ	maptio=	200			;Sensorbelegung-Zeit= 4s (20ms-Einheiten)
.equ	settio=	3000			;Einstellungs-Men-Zeit= 60s (20ms-Einheiten)
.equ	re1tio=	60			;Timeout fr Datenempfang= 60s (Erstwert)
.equ	re2tio=	20			;Timeout fr Datenempfang= 20s (Normalwert)
.equ	sentio=	60			;Sensor-Timeout (Sensorausfall)= 60s
.equ	almtio=	60			;Alarmanzeige-Zeit= 60s
;
; -------------------
; Register definieren
; -------------------
;
.def	isreg=	r2			;Zwischenspeicher fr SREG bei Interruptbearbeitung
.def	rxpoi1=	r3			;UART Empfangspuffer-Pointer1 (Schreibposition)
.def	rxpoi2=	r4			;UART Empfangspuffer-Pointer2 (Leseposition)
.def	rspoin=	r5			;Zeiger auf RS-232 Textpuffer
.def	invlpb=	r6			;LED-Invertierung fr Port B
					;0x00: LED-Ansteuerung direkt
					;0x1f: LED-Ansteuerung ber Treiber
.def	invlpd=	r7			;LED-Invertierung fr Port D
					;0x00: LED-Ansteuerung direkt
					;0xfe: LED-Ansteuerung ber Treiber
.def	invlpc=	r8			;LED-Invertierung fr Port C
					;0x00: LED-Ansteuerung direkt
					;0xff: LED-Ansteuerung ber Treiber
.def	itemp3=	r9			;Zwischenspeicher 3 fr Interruptroutine
.def	itemp1=	r22			;Zwischenspeicher 1 fr Interruptroutine
.def	itemp2=	r23			;Zwischenspeicher 2 fr Interruptroutine
.def	flags=	r24			;Verschiedene Flags
					;Bit0: Sensorwert 1 gltig
					;Bit1: Sensorwert 2 gltig
					;Bit2: Datenempfang ok
					;Bit3: Tastenflag 1 - vorheriger Tasterstatus, wird vom Timer1-Interrupt verwaltet, enthlt Taster-Status vom vorherigen Timer1-Interrupt
					;Bit4: Tastenflag 2 - kurzer Tastendruck, wird vom Interrupt gesetzt und gelscht
					;Bit5: Tastenflag 3 - kurzer Tastendruck quittiert, wird vom Hauptprogramm gesetzt und vom Interrupt wieder gelscht
					;Bit6: Tastenflag 4 - langer Tastendruck, wird vom Interrupt gesetzt und gelscht
					;Bit7: Tastenflag 5 - langer Tastendruck quittiert, wird vom Hauptprogramm gesetzt und vom Interrupt wieder gelscht
.def	flags2=	r25			;Verschiedene Flags 2
					;Bit0: Sensorwert 3 gltig
					;Bit1: Sensorwert 4 gltig
					;Bit7: Merker fr kurzen Tastendruck, wird fr die Einstellungs-Modi bentigt, um das ffnen des Tasters zu erkennen
;
; -----------------------
; Speicheradressen im RAM
; -----------------------
;
.dseg
rstext:	.byte	8			;RS-232-Textpuffer fr empfangene Datenpakete
disply:	.byte	8			;Segmentbelegung der 8 Anzeigestellen
senmap:	.byte	8			;empfangene Sensorbelegung
mpxcnt:	.byte	1			;Multiplexzhler (0-3)
tm1cnt:	.byte	1			;Timerzhler 1 (0-15), 20ms fr Taster und ADC
tm2cnt:	.byte	1			;Timerzhler 2 (0-49), 1000ms fr Timeout-Zhler
keycnt:	.byte	1			;Tasterzhler, wird bei Tastendruck auf	<keytio> gesetzt und auf 0 gezhlt, dient zur Erkennung des langen Tastendrucks
al_set:	.byte	4			;Sollzustnde der 4 Alarme
					;0: Alarmkontakt geffnet
					;1: Alarmkontakt geschlossen
alstat:	.byte	4			;letzter empfangener Alarm-Status
					;0: Alarmkontakt geffnet
					;1: Alarmkontakt geschlossen
al_tio:	.byte	4			;Alarm-Timeout-Zhler der 4 Alarme
alflag:	.byte	1			;Alarm-Flags
					;Bit0: Alarm-Anzeige ist eingeschaltet
					;Bit1: mindestens 1 Alarm ist aktiv
almnum:	.byte	1			;Alarm-Nummer fr Anzeige, wird bei Alarm anstatt Sensorwert 3 ausgegeben, wird von Interrupt gesteuert
adclst:	.byte	4			;letzte 4 ADC-Werte, fr Mittelwertbildung
adcval:	.byte	1			;ermittelter ADC-Mittelwert (Fototransistor)
pwmval:	.byte	1			;ermittelter PWM-Wert fr Timer2 (Helligkeit)
sensr1:	.byte	1			;Sensornummer 1 (0-30, fr obere Zeile)
sensr2:	.byte	1			;Sensornummer 2 (0-30, fr untere Zeile)
sensr3:	.byte	1			;Sensornummer 3 (0-30, 0xff wenn ungenutzt, fr obere Zeile, Wechselanzeige mit Sensornummer 1)
sensr4:	.byte	1			;Sensornummer 4 (0-30, 0xff wenn ungenutzt, fr untere Zeile, Wechselanzeige mit Sensornummer 2)
senva1:	.byte	4			;Sensorwert 1 (fr obere Zeile)
senva2:	.byte	4			;Sensorwert 2 (fr untere Zeile)
senva3:	.byte	4			;Sensorwert 3 (fr obere Zeile, Wechselanzeige mit Sensorwert 1)
senva4:	.byte	4			;Sensorwert 4 (fr untere Zeile, Wechselanzeige mit Sensorwert 2)
se1tio:	.byte	1			;Sensor 1 Timeout-Zhler
se2tio:	.byte	1			;Sensor 2 Timeout-Zhler
se3tio:	.byte	1			;Sensor 3 Timeout-Zhler
se4tio:	.byte	1			;Sensor 4 Timeout-Zhler
rectio:	.byte	1			;Datenempfang Timeout-Zhler
dimode:	.byte	1			;Anzeigemodus
					; 0: Sensoranzeige
					; 1: Alarmanzeige (wird nicht mehr verwendet)
					; 2: Sensorbelegung 1-4
					; 3: Sensorbelegung 5-8
					; 4: Versionsnummer
					; 5: Sensornummer 1 einstellen
					; 6: Sensornummer 2 einstellen
					; 7: Sensornummer 3 einstellen
					; 8: Sensornummer 4 einstellen
					; 9: Alarmanzeige ein-/ausschalten
					;10: Wechselzeit einstellen
distio:	.byte	2			;Anzeige-Timeout in 20ms Einheiten, 16Bit, fr Rckkehr zur normalen Sensor-Anzeige
ed_val:	.byte	1			;Editor-Wert in den Anzeige-Modi 5-7
change:	.byte	1			;Zeitdauer fr Wechselanzeige (1-8 Sekunden)
chgcnt:	.byte	1			;Wechsel-Zhler, zhlt die Sekunden bis zum nchsten Anzeigewechsel
chgflg:	.byte	1			;Wechsel-Flag, wechselt nach Ablauf des Wechsel-Zhlers zwischen 0 und 1
					;0: Anzeige von Sensorwert 1+2
					;1: Anzeige von Sensorwert 3+4
;
.org	0x100				;nicht verschieben!
rxbuff:	.byte	128			;RS-232 Empfangspuffer (Ringpuffer)
;
; ----------------------------------
; Programmbeginn, Interrupt-Vektoren
; ----------------------------------
;
.cseg
	rjmp	start
;
.org	OVF2addr			;Timer fr Helligkeitssteuerung
	rjmp	timer2			;zhlt alle 8s (125kHz)
;
.org	OC1Aaddr			;zyklischer Timer fr Anzeige + Taster
	rjmp	timer1			;zhlt alle 1,25ms (800Hz)
;
.org	URXCaddr
	rjmp	readrx			;RS-232 Empfang
;
.org	INT_VECTORS_SIZE
;
; -------------------------------
; Programmbeginn, Initialisierung
; -------------------------------
;
start:	ldi	xl, low(sram_start)
	ldi	xh, high(sram_start)	;Anfangsadresse RAM
	ldi	r16, low(ramend)
	ldi	r17, high(ramend)	;Endadresse RAM
	out	spl, r16
	out	sph, r17		;Stackpointer setzen
	clr	r18			;Nullbytes schreiben
;
ramclr:	st	x+, r18			;Speicherzelle lschen
	cp	r16, xl			;Endadresse L erreicht?
	brne	ramclr			;nein -> Schleife
	cp	r17, xh			;Endadresse H erreicht?
	brne	ramclr			;nein -> Schleife
;
; I/O-Ports initialisieren
;
	ldi	r16, 0			;PortA alles Eingnge
	out	ddra, r16
	ldi	r16, 0x7f		;PortA1-A7 Pull-Up einschalten
	out	porta, r16
;
	clr	r16			;Warteschleife, Zhler lschen
wait:	dec	r16			;Zhler-1, Ende erreicht?
	brne	wait			;nein -> Schleife
	rcall	invset			;LED-Invertierung setzen
;
	ldi	r16, 0x1f		;PortB0-B4 auf Ausgang setzen
	out	ddrb, r16
	ldi	r16, 0xe0		;alle Ausgnge auf Low (=aus)
	eor	r16, invlpb		;gegebenenfalls invertieren
	out	portb, r16
;
	ldi	r16, 0xff		;PortC alles Ausgnge
	out	ddrc, r16
	ldi	r16, 0x00		;alle Ausgnge auf Low (=aus)
	eor	r16, invlpc		;gegebenenfalls invertieren
	out	portc, r16
;
	ldi	r16, 0xfe		;PortD1-D7 auf Ausgang setzen
	out	ddrd, r16
	ldi	r16, 0x01		;alle Ausgnge auf Low (=aus)
	eor	r16, invlpd		;gegebenenfalls invertieren
	out	portd, r16
;
; RS-232 Empfang initialisieren
;
	ldi	r16, low((clock / (16 * baudr)) - 1)
	ldi	r17, high((clock / (16 * baudr)) - 1)
	out	ubrrl, r16
	out	ubrrh, r17		;Baudrate auf 9600 einstellen
	ldi	r16, 0
	out	ucsra, r16
	ldi	r16, (1 << ursel) | (3 << ucsz0)
	out	ucsrc, r16		;Datenformat 8N1 setzen
	ldi	r16, (1 << rxcie) | (1 << rxen)
	out	ucsrb, r16		;RX mit Interrupt aktivieren
;
; A/D Konverter initialisieren
;
	ldi	r16, (1 << adlar) | (1 << mux2) | (1 << mux1) | (1 << mux0)
	out	admux, r16		;ADC7 auswhlen, linksbndig
	ldi	r16, (1 << aden) | (1 << adps2) | (1 << adps1)
	out	adcsra, r16		;ADC aktivieren, Vorteiler = 64
;
; Timer initialisieren
;
	clr	r16			;Timer1
	out	tcnt1h, r16
	out	tcnt1l, r16		;Zhlerwert lschen
	ldi	r16, low(tm1ocr)
	ldi	r17, high(tm1ocr)	;Teilerwert laden
	out	ocr1ah, r17
	out	ocr1al, r16		;Teilerwert speichern
	ldi	r16, 0			;Timer1, keine OCx-Ausgnge
	out	tccr1a, r16		;Normal-CTC-Modus
	ldi	r16, (1 << wgm12) | (1 << cs10)
	out	tccr1b, r16		;CTC-Modus, Vorteiler = 1
;
	clr	r16			;Timer2
	out	tcnt2, r16		;Zhlerwert lschen
	ldi	r16, (1 << cs21) | (1 << cs20)
	out	tccr2,r16		;Normal-Modus, Vorteiler = 32
;
	ldi	r16, (1 << toie2) | (1 << ocie1a)
	out	timsk, r16		;OCR1A- und TOIE2-Interrupt aktivieren
;
	ldi	r16, low(rxbuff)
	mov	rxpoi1, r16		;Empfangspuffer-Pointer1 setzen
	mov	rxpoi2, r16		;Empfangspuffer-Pointer2 setzen
	ldi	r16, low(rstext)
	mov	rspoin, r16		;Zeiger auf Textpuffer setzen
;
; Daten aus EEPROM lesen
;
	ldi	r16, low(esens1)
	ldi	r17, high(esens1)	;EEPROM-Adresse Sensornummer 1
	rcall	eeread			;EEPROM lesen
	cpi	r18, 31			;gltige Nummer (0-30)?
	brcs	ini010			;ja -> weiter
	clr	r18			;sonst Sensor = 0 setzen
ini010:	sts	sensr1, r18		;Sensornummer 1 speichern
	ldi	r16, low(esens2)
	ldi	r17, high(esens2)	;EEPROM-Adresse Sensornummer 2
	rcall	eeread			;EEPROM lesen
	cpi	r18, 31			;gltige Nummer (0-30)?
	brcs	ini020			;ja -> weiter
	clr	r18			;sonst Sensor = 0 setzen
ini020:	sts	sensr2, r18		;Sensornummer 2 speichern
;
	ldi	r16, low(esens3)
	ldi	r17, high(esens3)	;EEPROM-Adresse Sensornummer 3
	rcall	eeread			;EEPROM lesen
	cpi	r18, 31			;gltige Nummer (0-30)?
	brcs	ini030			;ja -> weiter
	ser	r18			;sonst Sensor = 0xff setzen (ausschalten)
ini030:	sts	sensr3, r18		;Sensornummer 1 speichern
;
	ldi	r16, low(esens4)
	ldi	r17, high(esens4)	;EEPROM-Adresse Sensornummer 4
	rcall	eeread			;EEPROM lesen
	cpi	r18, 31			;gltige Nummer (0-30)?
	brcs	ini040			;ja -> weiter
	ser	r18			;sonst Sensor = 0xff setzen (ausschalten)
ini040:	sts	sensr4, r18		;Sensornummer 1 speichern
;
	ldi	r16, low(ealset)
	ldi	r17, high(ealset)	;EEPROM-Adresse Alarm-Sollzustnde
	ldi	xl, low(al_set)
	ldi	xh, high(al_set)	;Zeiger auf Alarm-Sollzustnde
	ldi	zl, low(alstat)
	ldi	zh, high(alstat)	;Zeiger auf Alarm-Status
	ldi	r19, 4			;4 Bytes lesen
ini050:	rcall	eeread			;EEPROM lesen
	cpi	r18, 0xff		;Wert = 0xff (noch nicht initialisiert)?
	brne	ini060			;nein -> weiter
	clr	r18			;sonst Alarm-Sollzustand = 0 setzen
ini060:	andi	r18, 0x01		;Alarmzustand begrenzen (0-1)
	st	x+, r18			;Alarm-Sollzustand speichern
	st	z+, r18			;auerdem als aktuellen Status speichern
	dec	r19			;alle Daten gelesen?
	brne	ini050			;nein -> Schleife
;
	ldi	r16, low(ealflg)
	ldi	r17, high(ealflg)	;EEPROM-Adresse Alarm-Flag
	rcall	eeread			;EEPROM lesen
	cpi	r18, 0xff		;Wert = 0xff (noch nicht initialisiert)?
	brne	ini070			;nein -> weiter
	clr	r18			;sonst Alarm-Flag = 0 setzen
ini070:	andi	r18, 0x01		;nur Bit0 verwenden
	sts	alflag, r18		;Alarm-Flags speichern
;
	ldi	r16, low(echang)
	ldi	r17, high(echang)	;EEPROM-Adresse Wechselzeit
	rcall	eeread			;EEPROM lesen
	dec	r18			;korrigieren (1-8 -> 0-7)
	cpi	r18, 8			;gltiger Wert (0-7)?
	brcs	ini080			;ja -> weiter
	ldi	r18, 1			;Wechselzeit = 1 setzen (2s)
ini080:	inc	r18			;Wechselzeit wiederherstellen
	sts	change, r18		;und speichern
;
; Taster abfragen, Anzeigetest
;
	sei
	in	r16, pina		;Taster-Port lesen
	bst	r16, 0			;Taster gedrckt?
	brts	ver010			;nein -> zum Hauptprogramm
;
	ldi	r18, 20
	rcall	wait50			;20x 50ms (1s) warten
	ldi	zl, low(disply)
	ldi	zh, high(disply)	;Zeiger auf Display-Speicher
	ldi	r21, 8			;8 Stellen bearbeiten
;
ditest:	ldi	r20, 1			;Zahl = 1 setzen
dit010:	rcall	invset			;LED-Invertierung setzen
	mov	r18, r20		;Zahl kopieren
	rcall	h7segm			;in 7-Segment-Code wandeln
	st	z, r18			;und speichern
	ldi	r18, 10
	rcall	wait50			;10x 50ms (0,5s) warten
	inc	r20			;Zahl erhhen
	cpi	r20, 9			;Ende (8) berschritten?
	brcs	dit010			;nein -> Schleife
;
	ld	r16, z			;sonst Segmentbelegung holen
	ori	r16, 0x80		;Dezimalpunkt setzen
	st	z+, r16			;Segmentbelegung speichern
	ldi	r18, 10
	rcall	wait50			;10x 50ms (0,5s) warten
;
	dec	r21			;alle Stellen bearbeitet?
	brne	ditest			;nein -> Schleife
	ldi	r18, 90
	rcall	wait50			;90x 50ms (4,5s) warten
	cli				;Interrupts sperren
	rjmp	start			;Neustart
;
; Versionsnummer fr 3 Sekunden anzeigen
;
ver010:	ldi	r16, 4			;Anzeige-Modus 4 (Versionsnummer)
	sts	dimode, r16		;setzen
	ldi	r16, low(vertio)	;Anzeigezeit L
	ldi	r17, high(vertio)	;Anzeigezeit H
	sts	distio, r16
	sts	distio + 1, r17		;Zeit setzen
;
	ldi	r16, re1tio		;Datenempfang Timeout-Zhler
	sts	rectio, r16		;auf Anfangswert setzen
	ldi	flags, 0b00000100	;Flags auf Anfangswert setzen
	clr	flags2			;Flags2 auf Anfangswert setzen
;
	wdr				;Watchdog-Timer zurcksetzen
	ldi	r16, 1 << wde		;Watchdog aktivieren,
	out	wdtcr, r16		;Timeout ~16ms
	sei				;Interrupts freigeben
;
;
; -------------
; Hauptprogramm
; -------------
;
; Alarmstatus prfen und Flags setzen
;
main:	rcall	invset			;LED-Invertierung setzen
	lds	r16, alflag		;Alarm-Flags holen
	bst	r16, 0			;Alarm-Anzeige eingeschaltet?
	brtc	ma1000			;nein -> berspringen
;
	ldi	xl, low(al_set)
	ldi	xh, high(al_set)	;Zeiger auf Alarm-Sollzustnde
	ldi	zl, low(alstat)
	ldi	zh, high(alstat)	;Zeiger auf Alarmstatus
	ldi	r20, low(al_tio)
	ldi	r21, high(al_tio)	;Zeiger auf Alarm-Timeout-Zhler
	ldi	r19, 4			;4 Alarme prfen
;
ma0010:	ld	r16, x+			;Soll-Zustand holen
	ld	r17, z+			;Ist-Zustand holen
	movw	r14, zl			;Alarmstatus-Zeiger sichern
	movw	zl, r20			;Zeiger auf Alarm-Timeout-Zhler holen
	eor	r16, r17		;Alarm aktiv?
	breq	ma0020			;nein -> weiter
	ldi	r16, almtio		;sonst Alarm-Anzeigezeit holen
	st	z, r16			;Alarm-Timeout-Zhler neu setzen
ma0020:	adiw	zl, 1			;Zeiger auf nchsten Timeout-Zhler
	movw	r20, zl			;Zeiger auf Alarm-Timeout-Zhler speichern
	movw	zl, r14			;Alarmstatus-Zeiger holen
	dec	r19			;alle Alarme bearbeitet?
	brne	ma0010			;nein -> Schleife
;
; Mittelwert aus den letzten 4 ADC-Werten bilden
;
ma1000:	ldi	xl, low(adclst)
	ldi	xh, high(adclst)	;Zeiger auf Werteliste
	clr	r16			;Summenwert L lschen
	clr	r17			;Summenwert H lschen
	clr	r19			;Summand H lschen
	ldi	r20, 4			;4 Werte addieren
ma1010:	ld	r18, x+			;Wert aus Liste holen
	add	r16, r18		;L-Wert addieren
	adc	r17, r19		;bertrag addieren
	dec	r20			;alle Werte bearbeitet?
	brne	ma1010			;nein -> Schleife
;
	ror	r17
	ror	r16			;Ergebnis / 2
	ror	r17
	ror	r16			;Ergebnis nochmals / 2
	sts	adcval,r16		;Mittelwert speichern
;
; PWM-Wert als quadratische Funktion vom ADC-Mittelwert (Helligkeit):
; pwm = ((adcm / 256) * 151) / 256
; 151 ist eine Konstante fr den nutzbaren PWM-Bereich
;
	mul	r16, r16		;Mittelwert quadrieren
	ldi	r17, 151		;Faktor fr PWM-Wert laden
	mul	r1, r17			;MW-Quadrat H-Byte * PWM-Faktor
	mov	r16, r1			;H-Byte = PWM-Wert
	com	r16			;in negativen Wert wandeln
	subi	r16, 4			;Offset subtrahieren
	sts	pwmval, r16		;PWM-Wert fr Counter speichern
;
; Sensor Timeout-Zhler auswerten
;
ma2000:	lds	r16, se1tio		;Sensor 1 Timeout-Zhler holen
	tst	r16			;Zhler abgelaufen?
	brne	ma2010			;nein -> weiter
	dec	r16			;sonst Stopp-Status setzen
	sts	se1tio, r16		;und Zhler wieder speichern
	cbr	flags, 0b00000001	;Sensorwert 1 ist ungltig
;
ma2010:	lds	r16, se2tio		;Sensor 2 Timeout-Zhler holen
	tst	r16			;Zhler abgelaufen?
	brne	ma2020			;nein -> weiter
	dec	r16			;sonst Stopp-Status setzen
	sts	se2tio, r16		;und Zhler wieder speichern
	cbr	flags, 0b00000010	;Sensorwert 2 ist ungltig
;
ma2020:	lds	r16, se3tio		;Sensor 3 Timeout-Zhler holen
	tst	r16			;Zhler abgelaufen?
	brne	ma2030			;nein -> weiter
	dec	r16			;sonst Stopp-Status setzen
	sts	se3tio, r16		;und Zhler wieder speichern
	cbr	flags2, 0b00000001	;Sensorwert 3 ist ungltig
;
ma2030:	lds	r16, se4tio		;Sensor 4 Timeout-Zhler holen
	tst	r16			;Zhler abgelaufen?
	brne	ma2050			;nein -> weiter
	dec	r16			;sonst Stopp-Status setzen
	sts	se4tio, r16		;und Zhler wieder speichern
	cbr	flags2, 0b00000010	;Sensorwert 4 ist ungltig
;
; Alarm Timeout-Zhler auswerten
;
ma2050:	ldi	xl, low(al_tio)
	ldi	xh, high(al_tio)	;Zeiger auf Alarm-Timeout-Zhler
	ldi	r17, 4			;4 Zhler bearbeiten
ma2060:	ld	r16, x			;Timeout-Zhler holen
	tst	r16			;Zhler abgelaufen?
	brne	ma2070			;nein -> weiter
	dec	r16			;sonst Stopp-Status setzen
ma2070:	st	x+, r16			;und Zhler wieder speichern
	dec	r17			;alle Zhler bearbeitet?
	brne	ma2060			;nein -> Schleife
;
; Anzeige Timeout-Zhler auswerten
;
	lds	r16, distio		;Anzeige Timeout-Zhler L holen
	lds	r17, distio + 1		;Anzeige Timeout-Zhler H holen
	or	r16, r17		;verknpfen, Zhler abgelaufen?
	brne	ma2500			;nein -> weiter
	dec	r17			;sonst Stopp-Status setzen
	sts	distio + 1, r17		;und Zhler H wieder speichern
;
	lds	r16, dimode		;Anzeige-Modus holen
	cpi	r16, 4			;Anzeige der Versionsnummer?
	brne	ma2120			;nein -> weiter testen
ma2110:	clr	r16			;Anzeige-Modus auf normale
	sts	dimode, r16		;Sensoranzeige setzen
	rjmp	ma2500
;
ma2120:	cpi	r16, 2			;Anzeige Sensorbelegung 1-4?
	brne	ma2130			;nein -> weiter testen
	inc	r16			;sonst auf Sensorbelegung 5-8
	sts	dimode, r16		;setzen
	ldi	r16, low(maptio)	;Anzeige-Zeit fr restliche
	ldi	r17, high(maptio)	;Sensorbelegung neu setzen
	sts	distio, r16
	sts	distio + 1, r17		;Zeit setzen
	rjmp	ma2500
;
ma2130:	cpi	r16, 3			;Anzeige Sensorbelegung 5-8?
	breq	ma2110			;ja -> zurck zur Sensoranzeige
;
; Datenempfang Timeout-Zhler auswerten
;
ma2500:	lds	r16, rectio		;Empfang Timeout-Zhler holen
	tst	r16			;Zhler abgelaufen?
	brne	ma3000			;nein -> weiter
	dec	r16			;sonst Stopp-Status setzen
	sts	rectio, r16		;und Zhler wieder speichern
	cbr	flags, 0b00000100	;Flag fr Datenempfang lschen
;
; Daten auf Displayzeile 1 ausgeben, vorher Anzeigemodus prfen
;
ma3000:	ldi	xl, low(disply)
	ldi	xh, high(disply)	;Zeiger auf Display-Speicher
	ldi	r20, 4			;4 Anzeigestellen bearbeiten
;
	lds	r16, dimode		;Anzeige-Modus holen
	tst	r16			;normale Sensoranzeige?
	breq	ma3100			;ja -> Sensor 1+3 anzeigen
;
	cpi	r16, 2			;Sensorbelegung 1-4?
	brne	ma3010			;nein -> weiter testen
	rjmp	ma3400			;sonst Sensorbelegung anzeigen
;
ma3010:	cpi	r16, 3			;Sensorbelegung 5-8?
	brne	ma3020			;nein -> weiter testen
	rjmp	ma3500			;sonst Sensorbelegung anzeigen
;
ma3020:	cpi	r16, 4			;Versionsnummer?
	brne	ma3030			;nein -> weiter testen
	rjmp	ma3600			;sonst Versionsnummer anzeigen
;
ma3030:	cpi	r16, 5			;Sensornummer 1 einstellen?
	brne	ma3040			;nein -> weiter testen
	rjmp	ma3700			;sonst Sensornummer 1 einstellen
;
ma3040:	cpi	r16, 6			;Sensornummer 2 einstellen?
	brne	ma3050			;nein -> weiter testen
	rjmp	ma3800			;sonst Sensornummer 2 einstellen
;
ma3050:	cpi	r16, 7			;Sensornummer 3 einstellen?
	brne	ma3060			;nein -> weiter testen
	rjmp	ma3850			;sonst Sensornummer 3 einstellen
;
ma3060:	cpi	r16, 8			;Sensornummer 4 einstellen?
	brne	ma3070			;nein -> weiter testen
	rjmp	ma3900			;sonst Sensornummer 4 einstellen
;
ma3070:	cpi	r16, 9			;Alarme einstellen?
	brne	ma3080			;nein -> weiter testen
	rjmp	ma3950			;sonst Alarme einstellen
;
ma3080:	cpi	r16, 10			;Wechselzeit einstellen?
	brne	ma3090			;nein -> weiter testen
	rjmp	ma3970			;sonst Wechselzeit einstellen
;
ma3090:	rjmp	ma4000			;Zeile 2 ausgeben
;
; Daten von Sensor 1+3 ausgeben
;
ma3100:	bst	flags, 2		;Datenempfang ok?
	brtc	ma3200			;nein -> Ausfall anzeigen
;
ma3110:	ldi	zl, low(senva1)
	ldi	zh, high(senva1)	;Zeiger auf Sensorwert 1
	lds	r16, sensr1		;Sensornummer 1 holen
	lds	r18, chgflg		;Wechsel-Flag holen
	tst	r18			;Sensor 1 anzeigen?
	breq	ma3120			;ja -> weiter
;
	lds	r17, alflag		;sonst Alarm-Flags holen
	andi	r17, 0b00000011		;relevante Flags filter
	cpi	r17, 0b00000011		;Alarme ein und mindestens 1 Alarm aktiv?
	breq	ma3300			;ja -> Alarm anzeigen
;
	lds	r17, sensr3		;sonst Sensornummer 3 holen
	cpi	r17, 0xff		;Sensornummer gltig?
	breq	ma3120			;nein -> weiter
	ldi	zl, low(senva3)		;sonst
	ldi	zh, high(senva3)	;Zeiger auf Sensorwert 3
	lds	r16, sensr3		;Sensornummer 3 holen
;
ma3120:	ld	r18, z+			;ASCII-Ziffer holen
	rcall	aschex			;in Hexcode wandeln
	rcall	h7segm			;in 7-Segment-Code wandeln
	cpi	r16, 27			;Temperatursensor?
	brcc	ma3130			;nein -> weiter
	cpi	r20, 2			;Einerstelle bei Temperatur?
	brne	ma3150			;nein -> weiter
	rjmp	ma3140			;sonst Dezimalpunkt setzen
ma3130:	cpi	r20, 1			;Einerstelle bei Luftdruck/Luftfeuchtigkeit?
	brne	ma3150			;nein -> weiter
ma3140:	ori	r18, 0x80		;sonst Dezimalpunkt setzen
ma3150:	bst	flags, 0		;Sensorwert 1 gltig?
	lds	r17, chgflg		;Wechsel-Flag holen
	tst	r17			;Sensor 1 anzeigen?
	breq	ma3160			;ja -> weiter
	lds	r17, sensr3		;Sensornummer 3 holen
	cpi	r17, 0xff		;Sensornummer 3 gltig?
	breq	ma3160			;nein -> weiter
	bst	flags2, 0		;sonst Sensorwert 3 gltig?
ma3160:	brts	ma3170			;ja -> weiter
	ldi	r18, 0x40		;sonst Minuszeichen
ma3170:	st	x+, r18			;speichern
	dec	r20			;alle Ziffern bearbeitet?
	brne	ma3120			;nein -> Schleife
	rjmp	ma4000			;Zeile 2 ausgeben
;
; Ausfall des Datenempfangs (Fehler) anzeigen
;
ma3200:	ldi	zl, low(2 * errtxt)
	ldi	zh, high(2 * errtxt)	;Zeiger auf Fehlertext
ma3210:	lpm	r18, z+			;ASCII-Zeichen holen
	rcall	aschex			;in Hexcode wandeln
	rcall	h7segm			;in 7-Segment-Code wandeln
	st	x+, r18			;speichern
	dec	r20			;alle Ziffern bearbeitet?
	brne	ma3210			;nein -> Schleife
	rjmp	ma4000			;Zeile 2 ausgeben
;
; Alarme anzeigen
;
ma3300:	ldi	zl, low(2 * almtxt)
	ldi	zh, high(2 * almtxt)	;Zeiger auf Alarmtext
	ldi	r20, 3			;nur 3 Zeichen ausgeben
ma3310:	lpm	r18, z+			;ASCII-Zeichen holen
	rcall	aschex			;in Hexcode wandeln
	rcall	h7segm			;in 7-Segment-Code wandeln
	st	x+, r18			;speichern
	dec	r20			;alle Zeichen bearbeitet?
	brne	ma3310			;nein -> Schleife
;
	lds	r18, almnum		;Alarm-Nummer holen
	inc	r18			;fr Anzeige korrigieren
	rcall	h7segm			;in 7-Segment-Code wandeln
	st	x+, r18			;speichern
	rjmp	ma4000			;Zeile 2 ausgeben
;
; Sensorbelegung 1-4 anzeigen
;
ma3400:	ldi	zl, low(senmap)
	ldi	zh, high(senmap)	;Zeiger auf Sensorbelegung 1-4
ma3410:	ld	r18, z+			;ASCII-Ziffer holen
	cpi	r18, 'H' + 1		;Wert > 'H'?
	brcs	ma3420			;nein -> weiter
	ldi	r18, '-'		;sonst durch '-' ersetzen
ma3420:	rcall	aschex			;in Hexcode wandeln
	rcall	h7segm			;in 7-Segment-Code wandeln
	ori	r18, 0x80		;sonst Dezimalpunkt setzen
	st	x+, r18			;speichern
	dec	r20			;alle Stellen bearbeitet?
	brne	ma3410			;nein -> Schleife
	rjmp	ma4000			;Zeile 2 ausgeben
;
; Sensorbelegung 5-8 anzeigen
;
ma3500:	ldi	zl, low(senmap + 4)
	ldi	zh, high(senmap + 4)	;Zeiger auf Sensorbelegung 5-8
ma3510:	ld	r18, z+			;ASCII-Ziffer holen
	cpi	r18, 'H' + 1		;Wert > 'H'?
	brcs	ma3520			;nein -> weiter
	ldi	r18, '-'		;sonst durch '-' ersetzen
ma3520:	rcall	aschex			;in Hexcode wandeln
	rcall	h7segm			;in 7-Segment-Code wandeln
	st	x+, r18			;speichern
	dec	r20			;alle Stellen bearbeitet?
	brne	ma3510			;nein -> Schleife
	rjmp	ma4000			;Zeile 2 ausgeben
;
; Versionsnummer anzeigen
;
ma3600:	ldi	zl, low(2 * versnr)
	ldi	zh, high(2 * versnr)	;Zeiger auf Versionsnummer
ma3610:	lpm	r18, z+			;ASCII-Ziffer holen
	rcall	aschex			;in Hexcode wandeln
	rcall	h7segm			;in 7-Segment-Code wandeln
	cpi	r20, 4			;Einerstelle der Versionsnummer?
	brne	ma3620			;nein - > weiter
	ori	r18, 0x80		;sonst Dezimalpunkt setzen
ma3620:	st	x+, r18			;speichern
	dec	r20			;alle Ziffern bearbeitet?
	brne	ma3610			;nein -> Schleife
	rjmp	ma4000			;Zeile 2 ausgeben
;
; Einstellung von Sensornummer 1 anzeigen
;
ma3700:	ldi	zl, low(2 * se1txt)
	ldi	zh, high(2 * se1txt)	;Zeiger auf Sensor 1 Text
ma3710:	ldi	r20, 2			;nur 2 Zeichen ausgeben
ma3720:	lpm	r18, z+			;ASCII-Zeichen holen
	rcall	aschex			;in Hexcode wandeln
	rcall	h7segm			;in 7-Segment-Code wandeln
	cpi	r20, 1			;zweite Anzeigestelle?
	brne	ma3730			;ja -> weiter
	ori	r18, 0x80		;sonst Dezimalpunkt setzen
ma3730:	st	x+, r18			;speichern
	dec	r20			;alle Zeichen bearbeitet?
	brne	ma3720			;nein -> Schleife
;
	lds	r16, tm2cnt		;Timer-Zhler 2 holen
	cpi	r16, 25			;Zhler < 25 (halbe Sekunde)?
	brcs	ma3740			;ja -> weiter
	subi	r16, 25			;sonst Zhlerwert halbieren
ma3740:	cpi	r16, 3			;Zhler >= 3 (>= 60ms)?
	brcc	ma3750			;ja -> Editor-Wert ausgeben
;
	clr	r18			;sonst Editor-Wert ausblenden
	st	x+, r18			;Anzeigestelle 3
	st	x+, r18			;Anzeigestelle 4
	rjmp	ma4000			;Zeile 2 ausgeben
;
ma3750:	lds	r20, ed_val		;Editor-Wert holen
	clr	r18			;Zehnerstelle = 0 setzen
ma3760:	cpi	r20, 10			;Editor-Wert > 10?
	brcs	ma3770			;nein -> weiter
	subi	r20, 10			;sonst Byte - 10
	inc	r18			;Zehnerstelle erhhen
	rjmp	ma3760			;Schleife
ma3770:	tst	r18			;Zehnerstelle = 0?
	brne	ma3780			;nein -> weiter
	ldi	r18, 0x15		;sonst Zehnerstelle = ' ' setzen
ma3780:	rcall	h7segm			;in 7-Segment-Code wandeln
	st	x+, r18			;speichern
	mov	r18, r20		;Editor-Wert Einerstelle holen
	rcall	h7segm			;in 7-Segment-Code wandeln
	st	x+, r18			;speichern
	rjmp	ma4000			;Zeile 2 ausgeben
;
; Einstellung von Sensornummer 2 anzeigen
;
ma3800:	ldi	zl, low(2 * se2txt)
	ldi	zh, high(2 * se2txt)	;Zeiger auf Sensor 2 Text
	rjmp	ma3710			;ausgeben
;
; Einstellung von Sensornummer 3 anzeigen
;
ma3850:	ldi	zl, low(2 * se3txt)
	ldi	zh, high(2 * se3txt)	;Zeiger auf Sensor 3 Text
	rjmp	ma3710			;ausgeben
;
; Einstellung von Sensornummer 4 anzeigen
;
ma3900:	ldi	zl, low(2 * se4txt)
	ldi	zh, high(2 * se4txt)	;Zeiger auf Sensor 4 Text
	rjmp	ma3710			;ausgeben
;
; Einstellung der Alarme anzeigen
;
ma3950:	ldi	zl, low(2 * alstxt)
	ldi	zh, high(2 * alstxt)	;Zeiger auf Alarmtext
	rjmp	ma3710			;ausgeben
;
; Einstellung der Wechselzeit anzeigen
;
ma3970:	ldi	zl, low(2 * chgtxt)
	ldi	zh, high(2 * chgtxt)	;Zeiger auf Wechselzeit-Text
	rjmp	ma3710
;
; Daten von Sensor 2+4 immer auf Zeile 2 ausgeben
;
ma4000:	ldi	xl, low(disply + 4)
	ldi	xh, high(disply + 4)	;Zeiger auf Display-Speicher
	ldi	r20, 4			;4 Anzeigestellen bearbeiten
	ldi	zl, low(senva2)
	ldi	zh, high(senva2)	;Zeiger auf Sensorwert 2
	lds	r16, sensr2		;Sensornummer 2 holen
	lds	r18, chgflg		;Wechsel-Flag holen
	tst	r18			;Sensor 2 anzeigen?
	breq	ma4010			;ja -> weiter
;
	lds	r17, sensr4		;sonst Sensornummer 4 holen
	cpi	r17, 0xff		;Sensornummer gltig?
	breq	ma4010			;nein -> weiter
	ldi	zl, low(senva4)		;sonst
	ldi	zh, high(senva4)	;Zeiger auf Sensorwert 4
	lds	r16, sensr4		;Sensornummer 4 holen
;
ma4010:	ld	r18, z+			;ASCII-Ziffer holen
	rcall	aschex			;in Hexcode wandeln
	rcall	h7segm			;in 7-Segment-Code wandeln
	cpi	r16, 27			;Temperatursensor?
	brcc	ma4020			;nein -> weiter
	cpi	r20, 2			;Einerstelle bei Temperatur?
	brne	ma4040			;nein -> weiter
	rjmp	ma4030			;sonst Dezimalpunkt setzen
ma4020:	cpi	r20, 1			;Einerstelle bei Luftdruck/Luftfeuchtigkeit?
	brne	ma4040			;nein -> weiter
ma4030:	ori	r18, 0x80		;sonst Dezimalpunkt setzen
ma4040:	bst	flags, 1		;Sensorwert 2 gltig?
	lds	r17, chgflg		;Wechsel-Flag holen
	tst	r17			;Sensor 2 anzeigen?
	breq	ma4050			;ja -> weiter
	lds	r17, sensr4		;Sensornummer 4 holen
	cpi	r17, 0xff		;Sensornummer 4 gltig?
	breq	ma4050			;nein -> weiter
	bst	flags2, 1		;sonst Sensorwert 4 gltig?
ma4050:	brts	ma4060			;ja -> weiter
	ldi	r18, 0x40		;sonst Minuszeichen
ma4060:	st	x+, r18			;speichern
	dec	r20			;alle Ziffern bearbeitet?
	brne	ma4010			;nein -> Schleife
;
; Empfangspuffer prfen und Zeichen auslesen
;
ma5000:	rcall	getchr			;Zeichen aus Empfangspuffer
	tst	r17			;holen, neues Zeichen?
	brne	ma5010			;nein -> Ende
	cpi	r16, lf			;LF (10) empfangen?
	breq	ma5010			;ja -> ignorieren
	cpi	r16, cr			;CR (13) empfangen?
	breq	ma5100			;ja -> Puffer auswerten
;
	ldi	xh, high(rstext)
	mov	xl, rspoin		;Zeiger auf Textpuffer
	cpi	xl, rstext + 8		;Puffer voll?
	breq	ma5010			;ja -> Zeichen verwerfen
	st	x+, r16			;Zeichen im Puffer ablegen
	mov	rspoin, xl		;Zeiger wieder speichern
ma5010:	rjmp	ma9000			;Ende
;
ma5100:	mov	r16, rspoin		;Textpuffer-Zeiger holen
	cpi	r16, rstext + 7		;Sensor-Daten empfangen?
	breq	ma6000			;ja -> bearbeiten
	cpi	r16, rstext + 3		;Alarm-Daten empfangen?
	brne	ma5110			;nein -> weiter testen
	rjmp	ma7000			;sonst Alarm-Daten bearbeiten
ma5110:	cpi	r16, rstext + 8		;Sensorbelegung empfangen?
	brne	ma5200			;nein -> Daten verwerfen
	rjmp	ma8000			;sonst Sensorbelegung bearbeiten
;
ma5200:	ldi	r16, rstext		;sonst Pointer fr Textpuffer
	mov	rspoin, r16		;wieder auf Anfang setzen
	rjmp	ma9000			;Ende
;
; Sensorkennung auf Plausibilitt prfen und Sensortyp ermitteln
;
ma6000:	ldi	xl, low(rstext)
	ldi	xh, high(rstext)	;Zeiger auf Textpuffer
	ld	r16, x+			;erstes Zeichen holen
	cpi	r16, '1'		;Zeichen < "1"?
	brcs	ma5200			;ja -> Fehler
	cpi	r16, 'z'+1		;Zeichen > "z"?
	brcc	ma5200			;ja -> Fehler
	cpi	r16, '9'		;Zeichen < "9"?
	brcs	ma6010			;ja -> weiter (Sensor 1-8)
	cpi	r16, 'a'		;Zeichen < "a"?
	brcs	ma5200			;ja -> Fehler
	cpi	r16, 't'		;Zeichen < "t"?
	brcs	ma6010			;ja -> weiter (Sensor a-s)
	cpi	r16, 'w'		;Zeichen < "w"?
	brcs	ma5200			;ja -> Fehler
	rjmp	ma6200			;sonst Luftdruck- oder Luftfeuchtigkeitssensor
;
; Temperaturdaten auf Plausibilitt prfen
;
ma6010:	ld	r16, x+			;zweites Zeichen holen
	cpi	r16, ':'		;Doppelpunkt?
	brne	ma5200			;nein -> Fehler
	ld	r16, x+			;drittes Zeichen holen
	cpi	r16, ' '		;Leerzeichen?
	breq	ma6020			;ja -> weiter
	cpi	r16, '-'		;Minus-Zeichen?
	breq	ma6020			;ja -> weiter
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma5200			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcc	ma5200			;ja -> Fehler
ma6020:	ld	r16, x+			;viertes Zeichen holen
	cpi	r16, ' '		;Leerzeichen?
	breq	ma6030			;ja -> weiter
	cpi	r16, '-'		;Minus-Zeichen?
	breq	ma6030			;ja -> weiter
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma5200			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcc	ma5200			;ja -> Fehler
ma6030:	ld	r16, x+			;fnftes Zeichen holen
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma5200			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcc	ma5200			;ja -> Fehler
	ld	r16, x+			;sechstes Zeichen holen
	cpi	r16, '.'		;Punkt?
	breq	ma6040			;ja -> weiter
	cpi	r16, ','		;Komma?
	brne	ma5200			;nein -> Fehler
ma6040:	ld	r16, x+			;siebtes Zeichen holen
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma5200			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcs	ma6400			;nein -> Sensornummer prfen
ma6100:	rjmp	ma5200			;sonst Fehler
;
; Luftdruckwert auf Plausibilitt prfen
;
ma6200:	cpi	r16, 'w'		;Luftdrucksensor?
	brne	ma6300			;nein -> weiter mit Luftfeuchtigkeitssensor
;
	ld	r16, x+			;zweites Zeichen holen
	cpi	r16, ':'		;Doppelpunkt?
	brne	ma6100			;nein -> Fehler
	ld	r16, x+			;drittes Zeichen holen
	cpi	r16, ' '		;Leerzeichen?
	brne	ma6100			;nein -> Fehler
	ld	r16, x+			;viertes Zeichen holen
	cpi	r16, ' '		;Leerzeichen?
	breq	ma6210			;ja -> weiter
	cpi	r16, '1'		;Zeichen = "1"?
	brne	ma6100			;nein -> Fehler
ma6210:	ld	r16, x+			;fnftes Zeichen holen
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma6100			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcc	ma6100			;ja -> Fehler
	ld	r16, x+			;sechstes Zeichen holen
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma6100			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcc	ma6100			;ja -> Fehler
	ld	r16, x+			;siebtes Zeichen holen
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma6100			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcc	ma6100			;ja -> Fehler
	rjmp	ma6400			;sonst Sensornummer prfen
;
; Luftfeuchtigkeitswert auf Plausibilitt prfen
;
ma6300: ld	r16, x+			;zweites Zeichen holen
	cpi	r16, ':'		;Doppelpunkt?
	brne	ma6100			;nein -> Fehler
	ld	r16, x+			;drittes Zeichen holen
	cpi	r16, ' '		;Leerzeichen?
	brne	ma6100			;nein -> Fehler
	ld	r16, x+			;viertes Zeichen holen
	cpi	r16, ' '		;Leerzeichen?
	brne	ma6100			;nein -> Fehler
	ld	r16, x+			;fnftes Zeichen holen
	cpi	r16, ' '		;Leerzeichen?
	breq	ma6310			;ja -> weiter
	cpi	r16, '1'		;Zeichen = "1"?
	brne	ma6100			;nein -> Fehler
ma6310:	ld	r16, x+			;sechstes Zeichen holen
	cpi	r16, ' '		;Leerzeichen?
	breq	ma6320			;ja -> weiter
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma6100			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcc	ma6100			;ja -> Fehler
ma6320:	ld	r16, x+			;siebtes Zeichen holen
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma6100			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcc	ma6100			;ja -> Fehler
;
; Sensornummer prfen und Daten speichern
;
ma6400:	ldi	r18, sentio		;Sensor-Timeout laden
	lds	r16, rstext		;erstes Zeichen aus Textpuffer holen
	cpi	r16, '9'		;Sensorgruppe 1-8?
	brcc	ma6410			;nein -> weiter
	subi	r16, '1'		;in Sensornummer 0-7 umwandeln
	rjmp	ma6420
ma6410:	subi	r16, 'a'-8		;in Sensornummer 8-26 umwandeln
	cpi	r16, 27			;Sensornummer > 26 (w-z)
	brcs	ma6420			;nein -> weiter
	subi	r16, 3			;sonst Sensornummer korrigieren
;
ma6420:	ldi	zl, low(senva1)
	ldi	zh, high(senva1)	;Zeiger auf Sensorwert 1
	lds	r17, sensr1		;Sensornummer 1 holen
	cp	r16, r17		;wurde Sensornummer 1 empfangen?
	brne	ma6430			;nein -> weiter
	rcall	savsen			;sonst Daten speichern
	sbr	flags, 0b00000101	;Sensorwert 1 ist gltig
	sts	se1tio, r18		;Sensor 1 Timeout neu setzen
;
ma6430:	ldi	zl, low(senva2)
	ldi	zh, high(senva2)	;Zeiger auf Sensorwert 2
	lds	r17, sensr2		;Sensornummer 2 holen
	cp	r16, r17		;wurde Sensornummer 2 empfangen?
	brne	ma6440			;nein -> weiter
	rcall	savsen			;sonst Daten speichern
	sbr	flags, 0b00000110	;Sensorwert 2 ist gltig
	sts	se2tio, r18		;Sensor 2 Timeout neu setzen
;
ma6440:	ldi	zl, low(senva3)
	ldi	zh, high(senva3)	;Zeiger auf Sensorwert 3
	lds	r17, sensr3		;Sensornummer 3 holen
	cp	r16, r17		;wurde Sensornummer 3 empfangen?
	brne	ma6450			;nein -> weiter
	rcall	savsen			;sonst Daten speichern
	sbr	flags, 0b00000100	;Datenempfang ok
	sbr	flags2, 0b00000001	;Sensorwert 3 ist gltig
	sts	se3tio, r18		;Sensor 3 Timeout neu setzen
;
ma6450:	ldi	zl, low(senva4)
	ldi	zh, high(senva4)	;Zeiger auf Sensorwert 4
	lds	r17, sensr4		;Sensornummer 4 holen
	cp	r16, r17		;wurde Sensornummer 4 empfangen?
	brne	ma6500			;nein -> weiter
	rcall	savsen			;sonst Daten speichern
	sbr	flags, 0b00000100	;Datenempfang ok
	sbr	flags2, 0b00000010	;Sensorwert 4 ist gltig
	sts	se4tio, r18		;Sensor 4 Timeout neu setzen
;
ma6500:	rjmp	ma5200			;Zeiger zurcksetzen, Ende
;
; Alarmdaten auf Plausibilitt prfen
;
ma7000:	ldi	xl, low(rstext)
	ldi	xh, high(rstext)	;Zeiger auf Textpuffer
	ld	r16, x+			;erstes Zeichen holen
	cpi	r16, 'A'		;Zeichen < "A"?
	brcs	ma6500			;ja -> Fehler
	cpi	r16, 'D' + 1		;Zeichen > "D"?
	brcc	ma6500			;ja -> Fehler
	ld	r16, x+			;zweites Zeichen holen
	cpi	r16, ':'		;Doppelpunkt?
	brne	ma6500			;nein -> Fehler
	ld	r16, x+			;drittes Zeichen holen
	cpi	r16, '0'		;Zeichen = "0"?
	breq	ma7010			;ja -> bearbeiten
	cpi	r16, '1'		;Zeichen = "1"?
	brne	ma6500			;nein -> Fehler
;
; Alarm-Daten sind ok, speichern
;
ma7010:	lds	r16, rstext		;erstes Zeichen aus Datenpuffer
	andi	r16, 0x07		;in Ziffernwert wandeln
	dec	r16			;korrigieren (= Alarmummer)
	clr	r17
	ldi	xl, low(alstat)
	ldi	xh, high(alstat)	;Zeiger auf Alarmstatus
	add	xl, r16			;Alarmnummer als Offset
	adc	xh, r17			;addieren
	lds	r18, rstext + 2		;Alarmwert holen
	andi	r18, 0x01		;in Binrwert wandeln
	st	x, r18			;und speichern
;
ma7020:	sbr	flags, 0b00000100	;Datenempfang ok
	ldi	r19, re2tio		;Empfangs-Timeout laden
	sts	rectio, r19		;Empfangs-Timeout neu setzen
;
	rjmp	ma5200			;Zeiger zurcksetzen, Ende
;
; Sensorbelegung auf Plausibilitt prfen
;
ma8000:	ldi	xl, low(rstext)
	ldi	xh, high(rstext)	;Zeiger auf Textpuffer
	ldi	r17, 8			;8 Zeichen prfen
ma8010:	ld	r16, x+			;Zeichen aus Puffer holen
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma8100			;ja -> Fehler
	cpi	r16, 'R' + 1		;Zeichen > "R"?
	brcc	ma8100			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen =< "9"?
	brcs	ma8020			;ja -> weiter
	cpi	r16, 'A'		;Zeichen < "A"?
	brcs	ma8100			;ja -> Fehler
ma8020:	dec	r17			;alle Zeichen bearbeitet?
	brne	ma8010			;nein -> Schleife
;
; Sensorbelegung ist ok, Daten speichern
;
	ldi	xl, low(rstext)
	ldi	xh, high(rstext)	;Zeiger auf Textpuffer
	ldi	zl, low(senmap)
	ldi	zh, high(senmap)	;Zeiger auf Sensorbelegung
	ldi	r17, 8			;8 Zeichen kopieren
ma8030:	ld	r16, x+			;Byte aus RX-Datenpuffer holen
	st	z+, r16			;in Sensorbelegung speichern
	dec	r17			;alle Daten kopiert?
	brne	ma8030			;nein -> Schleife
;
	lds	r16, dimode		;Anzeige-Modus holen
	cpi	r16, 5			;Einstellungs-Modus aktiv?
	brcc	ma8040			;ja -> nicht unterbrechen
;
	ldi	r16, 2			;Anzeige-Modus auf
	sts	dimode, r16		;Sensorbelegung 1-4 setzen
	ldi	r16, low(maptio)	;Anzeige-Zeit fr
	ldi	r17, high(maptio)	;Sensorbelegung setzen
	sts	distio, r16
	sts	distio + 1, r17		;Zeit setzen
;
ma8040:	ldi	r16, re2tio		;Empfangs-Timeout-Wert laden
	sts	rectio, r16		;Empfangs-Timeout neu setzen
	sbr	flags, 0b00000100	;Datenempfang ok
;
ma8100:	rjmp	ma5200			;Zeiger zurcksetzen, Ende
;
; Taster im Normalmodus abfragen
;
ma9000:	lds	r16, dimode		;Anzeigemodus holen
	cpi	r16, 5			;Modus >= 5 (Einstellungen)?
	brcc	ma9100			;ja -> Einstellungsmodus bearbeiten
;
	cli				;sonst Interrupts sperren
	bst	flags, 4		;kurzer Tastendruck?
	brtc	ma9010			;nein -> Ende
	bst	flags, 5		;kurzer Tastendruck quittiert?
	brts	ma9010			;ja -> Ende
;
	sbr	flags, 0b10100000	;kurzen + langen Tastendruck quittieren
	sei				;Interrupts wieder freigeben
	ldi	r16, 5			;Anzeigemodus auf Einstellung
	sts	dimode, r16		;fr Sensornummer 1 setzen
	lds	r16,sensr1		;aktuelle Sensornummer 1 holen
	inc	r16			;korrigieren
	sts	ed_val, r16		;und als Einstellwert laden
	rjmp	ma9180			;Timeout-Zhler setzen
;
ma9010:	sei				;Interrupts wieder freigeben
	rjmp	main			;Ende
;
; Taster: Einstellungs-Modus, langen Tastendruck abfragen und bearbeiten, Editor-Wert erhhen oder umschalten, Daten im EEPROM speichern
;
ma9100:	cli				;Interrupts sperren
	bst	flags, 6		;langer Tastendruck?
	brts	ma9110			;ja -> Quittung prfen
	rjmp	ma9300			;sonst kurzen Tastendruck prfen
;
ma9110:	bst	flags, 7		;langer Tastendruck quittiert?
	brts	ma9010			;ja -> Ende
	sbr	flags, 0b10000000	;sonst langen Tastendruck quittieren
	sei				;Interrupts wieder freigeben
	cbr	flags2, 0b10000000	;Merker fr kurzen Tastendruck lschen
;
	lds	r18,ed_val		;Editor-Wert holen
	inc	r18			;erhhen
	cpi	r16, 5			;Anzeigemodus = 5 (Sensor 1)?
	breq	ma9190			;ja -> bearbeiten
	cpi	r16, 6			;Anzeigemodus = 6 (Sensor 2)?
	breq	ma9210			;ja -> bearbeiten
	cpi	r16, 7			;Anzeigemodus = 7 (Sensor 3)?
	brne	ma9120			;nein -> weiter
	rjmp	ma9230			;sonst Sensor 3 bearbeiten
ma9120:	cpi	r16, 8			;Anzeigemodus = 8 (Sensor 4)?
	brne	ma9130			;nein -> weiter
	rjmp	ma9250			;sonst Sensor 4 bearbeiten
ma9130:	cpi	r16, 9			;Anzeigemodus = 9 (Alarme)?
	breq	ma9140			;ja -> bearbeiten
	rjmp	ma9270			;sonst Wechselzeit bearbeiten
;
ma9140:	andi	r18, 0x01		;Endwert begrenzen
	sts	ed_val, r18		;und wieder speichern
	sts	alflag, r18		;Alarm-Flags neu setzen
	breq	ma9170			;Alarme aus? ja -> weiter
;
	ldi	xl, low(alstat)		;sonst
	ldi	xh, high(alstat)	;Zeiger auf Alarm-Status
	ldi	zl, low(al_set)
	ldi	zh, high(al_set)	;Zeiger auf Alarm-Sollzustnde
	ldi	r17, 4			;4 Werte kopieren
ma9150:	ld	r16, x+			;Alarm-Status holen und
	st	z+, r16			;als Sollzustand speichern
	dec	r17			;alle Werte kopiert?
	brne	ma9150			;nein -> Schleife
;
	ldi	r16, low(ealset)
	ldi	r17, high(ealset)	;EEPROM-Adresse Alarm-Sollzustnde
	ldi	xl, low(al_set)
	ldi	xh, high(al_set)	;Zeiger auf Alarm-Sollzustnde
	ldi	r19, 4			;4 Bytes schreiben
ma9160:	ld	r18, x+			;Alarm-Sollzustand holen
	rcall	eewrit			;in EEPROM schreiben
	dec	r19			;alle Daten geschrieben?
	brne	ma9160			;nein -> Schleife
;
ma9170:	ldi	r16, low(ealflg)
	ldi	r17, high(ealflg)	;EEPROM-Adresse Alarm-Flag
	rcall	eewrit			;speichern
;
ma9180:	ldi	r16, low(settio)	;Einstellungs Anzeige-Timeout
	ldi	r17, high(settio)	;setzen
	sts	distio, r16
	sts	distio + 1, r17		;Zeit setzen
	rjmp	main
;
ma9190:	cpi	r18, 32			;Endwert berschritten?
	brcs	ma9200			;nein -> weiter
	ldi	r18, 1			;sonst auf Anfangswert setzen
ma9200:	sts	ed_val, r18		;und wieder speichern
	dec	r18			;Wert korrigieren
	sts	sensr1, r18		;als neuen Sensor 1 speichern
	ldi	r16, low(esens1)
	ldi	r17, high(esens1)	;EEPROM-Adresse Sensornummer 1
	rcall	eewrit			;speichern
	rjmp	ma9180			;Timeout neu setzen, Ende
;
ma9210:	cpi	r18, 32			;Endwert berschritten?
	brcs	ma9220			;nein -> weiter
	ldi	r18, 1			;sonst auf Anfangswert setzen
ma9220:	sts	ed_val, r18		;und wieder speichern
	dec	r18			;Wert korrigieren
	sts	sensr2, r18		;als neuen Sensor 2 speichern
	ldi	r16, low(esens2)
	ldi	r17, high(esens2)	;EEPROM-Adresse Sensornummer 2
	rcall	eewrit			;speichern
	rjmp	ma9180			;Timeout neu setzen, Ende
;
ma9230:	cpi	r18, 32			;Endwert berschritten?
	brcs	ma9240			;nein -> weiter
	clr	r18			;sonst auf Anfangswert setzen
ma9240:	sts	ed_val, r18		;und wieder speichern
	dec	r18			;Wert korrigieren
	sts	sensr3, r18		;als neuen Sensor 3 speichern
	ldi	r16, low(esens3)
	ldi	r17, high(esens3)	;EEPROM-Adresse Sensornummer 3
	rcall	eewrit			;speichern
	rjmp	ma9180			;Timeout neu setzen, Ende
;
ma9250:	cpi	r18, 32			;Endwert berschritten?
	brcs	ma9260			;nein -> weiter
	clr	r18			;sonst auf Anfangswert setzen
ma9260:	sts	ed_val, r18		;und wieder speichern
	dec	r18			;Wert korrigieren
	sts	sensr4, r18		;als neuen Sensor 4 speichern
	ldi	r16, low(esens4)
	ldi	r17, high(esens4)	;EEPROM-Adresse Sensornummer 4
	rcall	eewrit			;speichern
	rjmp	ma9180			;Timeout neu setzen, Ende
;
ma9270:	cpi	r18, 9			;Endwert berschritten?
	brcs	ma9280			;nein -> weiter
	ldi	r18, 1			;sonst auf Anfangswert setzen
ma9280:	sts	ed_val, r18		;und wieder speichern
	sts	change, r18		;als neue Wechselzeit speichern
	ldi	r16, low(echang)
	ldi	r17, high(echang)	;EEPROM-Adresse Wechselzeit
	rcall	eewrit			;speichern
	rjmp	ma9180			;Timeout neu setzen, Ende
;
; Taster: Einstellungs-Modus, kurzen Tastendruck abfragen und bearbeiten; Aktion wird erst nach Loslassen des Tasters ausgelst
;
ma9300:	bst	flags, 4		;kurzer Tastendruck?
	brtc	ma9310			;nein -> weiter
	bst	flags, 5		;kurzer Tastendruck quittiert?
	brts	ma9320			;ja -> Ende
;
	sbr	flags, 0b00100000	;Tastendruck quittieren
	sei				;Interrupts wieder freigeben
	sbr	flags2, 0b10000000	;Merker fr kurzen Tastendruck setzen
	rjmp	main			;Ende
;
ma9310:	sei				;Interrupts wieder freigeben
	bst	flags2, 7		;Merker fr kurzen Tastendruck gesetzt?
	brts	ma9330			;ja -> weiter
ma9320:	sei				;Interrupts wieder freigeben
	rjmp	main			;Ende
;
ma9330:	cbr	flags2, 0b10000000	;Merker fr kurzen Tastendruck lschen
	lds	r16, dimode		;Anzeigemodus holen
	inc	r16			;Modus erhhen
	cpi	r16, 11			;Bereich berschritten?
	brcs	ma9340			;nein -> weiter
	clr	r16			;sonst Anzeigemodus = 0 setzen
ma9340:	sts	dimode, r16		;Anzeigemodus speichern
	tst	r16			;Anzeigemodus = 0?
	brne	ma9350			;nein -> weiter
	rjmp	main			;sonst Ende
;
ma9350:	cpi	r16, 6			;Anzeigemodus = 6 (Sensor 2)?
	brne	ma9360			;nein -> weiter
	lds	r16, sensr2		;aktuelle Sensornummer 2 holen
	inc	r16			;korrigieren
	rjmp	ma9400
;
ma9360:	cpi	r16, 7			;Anzeigemodus = 7 (Sensor 3)?
	brne	ma9370			;nein -> weiter
	lds	r16, sensr3		;aktuelle Sensornummer 3 holen
	inc	r16			;korrigieren
	rjmp	ma9400
;
ma9370:	cpi	r16, 8			;Anzeigemodus = 8 (Sensor 4)?
	brne	ma9380			;nein -> weiter
	lds	r16, sensr4		;aktuelle Sensornummer 4 holen
	inc	r16			;korrigieren
	rjmp	ma9400
;
ma9380:	cpi	r16, 9			;Anzeigemodus = 9 (Alarme)?
	brne	ma9390			;nein -> weiter
	lds	r16, alflag		;Alarm-Flags holen
	andi	r16, 0x01		;Alarm-aktiv-Flag selektieren
	rjmp	ma9400
;
ma9390:	lds	r16, change		;Wechselzeit holen
ma9400:	sts	ed_val, r16		;und als Einstellwert laden
	rjmp	ma9180			;Timeout-Zhler neu setzen
;
;
; Unterprogramm zum bertragen eines Sensorwertes vom Textpuffer in den Datenspeicher
; Register: r17, x
;	    r16 <- Sensornummer (0-30)
;
savsen:	ldi	r17, re2tio		;Empfangs-Timeout laden
	sts	rectio, r17		;Empfangs-Timeout neu setzen
	ldi	xl, low(rstext + 2)
	ldi	xh, high(rstext + 2)	;Zeiger auf Sensorwert im Textpuffer
	cpi	r16, 27			;Temperatursensor?
	brcc	savs10			;nein -> weiter
;
	ld	r17, x+			;hchstwertige Stelle holen
	st	z+, r17			;speichern
	ld	r17, x+			;nchste Stelle holen
	st	z+, r17			;speichern
	ld	r17, x+			;nchste Stelle holen
	st	z+, r17			;speichern
	ld	r17, x+			;Punkt/Komma berspringen
	ld	r17, x+			;letzte Stelle holen
	st	z+, r17			;speichern
	ret
;
savs10:	ld	r17, x+			;hchstwertige Stelle berspringen
	ld	r17, x+			;nchste Stelle holen
	st	z+, r17			;speichern
	ld	r17, x+			;nchste Stelle holen
	st	z+, r17			;speichern
	ld	r17, x+			;nchste Stelle holen
	st	z+, r17			;speichern
	ld	r17, x+			;letzte Stelle holen
	st	z+, r17			;speichern
	ret
;
; Unterprogramm zur Wandlung von Hexcode in 7-Segment-Code
; Register: r19
;           r18 <- Zeichen
;	    r18 -> 7-Segment-Code
;
h7segm:	push	zl
	push	zh			;Register sichern
	ldi	zl, low(segtab * 2)
	ldi	zh, high(segtab * 2)	;Zeiger auf Segment-Tabelle
	clr	r19			;Offset H-Byte = 0
	add	zl, r18			;Zeichencode addieren
	adc	zh, r19			;bertrag addieren
	lpm	r18, z			;7-Segment-Code holen
	pop	zh
	pop	zl			;Register wiederherstellen
	ret
;
; Unterprogramm zur Wandlung einer ASCII-Ziffer in Hexcode
; Register: r18 <- ASCII-Zeichen
;	    r18 -> Hexcode
;
aschex:	cpi	r18, '-'		;Minus-Zeichen?
	breq	asc030			;ja -> direkt konvertieren
	cpi	r18, 'L'		;Zeichen = 'L'?
	breq	asc040			;ja -> direkt konvertieren
	cpi	r18, 'T'		;Zeichen = 'T'?
	breq	asc050			;ja -> direkt konvertieren
	cpi	r18, '0'		;Zeichen < '0'?
	brcs	asc020			;ja -> nicht darstellbar
	cpi	r18, 'H' + 1		;Zeichen > 'H'?
	brcc	asc020			;ja -> nicht darstellbar
	cpi	r18, '9' + 1		;Zeichen <= '9'?
	brcs	asc010			;ja -> konvertieren
	cpi	r18, 'A'		;Zeichen < 'A'?
	brcs	asc020			;ja -> nicht darstellbar
	subi	r18, 'A' - 10		;sonst Hexcode ermitteln
	ret
;
asc010:	subi	r18, '0'		;sonst Hexcode ermitteln
	ret
asc020:	ldi	r18, 0x15		;Hexcode fr Leerzeichen
	ret
asc030:	ldi	r18, 0x13		;Hexcode fr Minus-Zeichen
	ret
asc040:	ldi	r18, 0x12		;Hexcode fr 'L'
	ret
asc050:	ldi	r18, 0x14		;Hexcode fr 'T'
	ret
;
; Unterprogramm zur Wandlung eines Bytes in 3 Dezimalzahlen
; Register: r16 <- Byte
;	    r18, r17, r16 -> Dezimalstellen (Hunderter, Zehner, Einer)
;
bytdez:	clr	r18			;Hunderter lschen
	clr	r17			;Zehner lschen
byt010:	cpi	r16, 100		;Wert <= 100?
	brcs	byt020			;ja -> Zehnerstelle bearbeiten
	subi	r16, 100		;sonst 100 subtrahieren
	inc	r18			;Hunderter erhhen
	rjmp	byt010			;Schleife
;
byt020:	cpi	r16, 10			;Wert <= 10?
	brcs	byt030			;ja -> Ende
	subi	r16, 10			;sonst 10 subtrahieren
	inc	r17			;Zehner erhhen
	rjmp	byt020			;Schleife
byt030:	ret
;
; Unterprogramm zum Auslesen eines Zeichens aus dem RX-Empfangspuffer
; Register: x
;	    r16 -> gelesenes Zeichen
;	    r17 -> 0 = Zeichen gltig, 0xff = kein Zeichen im Puffer
;
getchr:	cp	rxpoi1, rxpoi2		;sind Zeichen im Puffer?
	brne	get010			;ja -> weiter
	ser	r17			;sonst kein Zeichen im Puffer
	ret
get010:	cli				;Interrupt vorbergehend sperren
	ldi	xh, high(rxbuff)	;Pointer H-Byte setzen
	mov	xl, rxpoi2		;Empfangspuffer-Pointer2 laden
	ld	r16, x+			;Byte laden, Pointer erhhen
	andi	xl, 0x7f		;bertrag korrigieren
	mov	rxpoi2, xl		;Pointer2 wieder speichern
	sei				;Interrupt wieder freigeben
	clr	r17			;Zeichen ist gltig
	ret
;
; Unterprogramm zum Lesen des Invers-Jumpers und Setzen der LED-Invertierung
; Register: r16, r17, r18, r19
;	    invlpb <- J1 offen: 0, J1 geschlossen: 0x1f
;	    invlpc <- J1 offen: 0, J1 geschlossen: 0xff
;	    invlpd <- J1 offen: 0, J1 geschlossen: 0xfe
;
invset:	clr	r17			;Anfangswert PortB setzen
	clr	r18			;Anfangswert PortD setzen
	clr	r19			;Anfangswert PortC setzen
	in	r16, pina		;Jumper lesen
	bst	r16, 1			;Jumper gesteckt?
	brtc	inv010			;ja -> weiter
	ldi	r17, 0x1f		;Invertierung PortB einschalten
	ldi	r18, 0xfe		;Invertierung PortD einschalten
	ldi	r19, 0xff		;Invertierung PortC einschalten
inv010:	mov	invlpb, r17		;Invertierung PortB speichern
	mov	invlpd, r18		;Invertierung PortD speichern
	mov	invlpc, r19		;Invertierung PortC speichern
	ret
;
; Unterprogramm Warteschleife ca. 50ms
; Register: r16, r17
;	    r18 <- Anzahl der Warteschleifen
;
wait50:	clr	r17			;H-Byte lschen
	clr	r16			;L-Byte lschen
wai010:	dec	r16			;L-Byte -1
	brne	wai010			;Ende? nein -> Schleife
	dec	r17			;sonst H-Byte -1
	brne	wai010			;Ende? nein -> Schleife
	dec	r18			;Schleifenanzahl -1
	brne	wai010			;Ende? nein -> Schleife
	ret
;
; Unterprogramm zum Lesen eines Byte aus dem EEPROM
; Register: r16 <- EEPROM-Adresse L \ wird nach dem Lesen
;	    r17 <- EEPROM-Adresse H / um 1 erhht
;	    r18 -> EEPROM-Daten
;
eeread:	sbic	eecr, eewe		;luft ein Schreibzyklus?
	rjmp	eeread			;ja -> warten
	out	eearl, r16		;EEPROM-Adresse L setzen
	out	eearh, r17		;EEPROM-Adresse H setzen
	sbi	eecr, eere		;EEPROM lesen aktivieren
	in	r18, eedr		;Daten lesen
;
eer010:	inc	r16			;Adresse L erhhen
	brne	eer020			;bertrag? nein -> weiter
	inc	r17			;sonst Adresse H erhhen
eer020:	ret
;
; Unterprogramm zum Schreiben eines Byte in das EEPROM
; Register: r16 <- EEPROM-Adresse L \ wird nach dem Schreiben
;	    r17 <- EEPROM-Adresse H / um 1 erhht
;	    r18 <- EEPROM-Daten
;
eewrit:	sbic	eecr, eewe		;luft ein Schreibzyklus?
	rjmp	eewrit			;ja -> warten
	out	eearl, r16		;EEPROM-Adresse L setzen
	out	eearh, r17		;EEPROM-Adresse H setzen
	out	eedr, r18		;EEPROM-Daten setzen
	sbi	eecr, eemwe		;EEPROM schreiben freigeben
	sbi	eecr, eewe		;EEPROM schreiben aktivieren
	rjmp	eer010			;Adresse erhhen

; Interrupt-Routine Timer1, wird alle 1,25ms ausgefhrt (800Hz)
; Register: isreg, itemp1, itemp2, itemp3, yl, yh
;
timer1:	in	isreg, sreg		;Status-Register sichern
	wdr				;Watchdog-Timer zurcksetzen
;
; Multiplex-Anzeige bedienen
;
	ldi	itemp2, 0xe0		;Grundzustand PortB laden
	eor	itemp2, invlpb		;gegebenenfalls invertieren
	out	portb, itemp2		;Anodentreiber + 1A ausschalten
	ldi	itemp2, 0x01		;Grundzustand PortD laden
	eor	itemp2, invlpd		;gegebenenfalls invertieren
	out	portd, itemp2		;Segmente 0B-0P ausschalten
	ldi	itemp2, 0x00		;Grundzustand PortC laden
	eor	itemp2, invlpc		;gegebenenfalls invertieren
	out	portc, itemp2		;Segmente 1A-1P ausschalten
;
	lds	itemp2, mpxcnt		;Multiplexzhler holen
	inc	itemp2			;erhhen
	andi	itemp2, 0x03		;bertrag lschen
	sts	mpxcnt, itemp2		;Zhler wieder speichern
	ldi	yl, low(disply)
	ldi	yh, high(disply)	;Zeiger auf Segmentbelegungen
	clr	itemp1			;H-Byte lschen
	add	yl, itemp2		;Offset addieren
	adc	yh, itemp1		;bertrag addieren
;
	ld	itemp2, y		;Segmente fr Zeile 1 holen
	bst	itemp2, 0		;Segment 1A sichern
	ori	itemp2, 0x01		;Pull-up fr RXD einschalten
	eor	itemp2, invlpd		;gegebenenfalls invertieren
	out	portd, itemp2		;Segmente 0B-0P ausgeben
;
	ldd	itemp2, y + 4		;Segmente fr Zeile 2 holen
	eor	itemp2, invlpc		;gegebenenfalls invertieren
	out	portc, itemp2		;Segmente 1A-1P ausgeben
;
	ldi	itemp2, 0x01		;Anfangswert fr Anodentreiber
	lds	itemp1, mpxcnt		;Multiplexzhler holen
tim100:	tst	itemp1			;Zhler = 0?
	breq	tim110			;ja -> Anodentreiber ok
	lsl	itemp2			;Position verschieben
	dec	itemp1			;Zhler vermindern
	rjmp	tim100			;Schleife
;
tim110:	ori	itemp2, 0xe0		;Pull-ups fr Ports 5-7 einschalten
	bld	itemp2, 4		;Segment 1A einsetzen
	eor	itemp2, invlpb		;gegebenenfalls invertieren
	out	portb, itemp2		;Anodentreiber + 1A ausgeben
;
	in	itemp2, tifr		;Timer Int-Flag Register holen
	sbr	itemp2, 1 << tov2	;Overflow-Interrupt lschen
	out	tifr, itemp2		;Register wieder speichern
	lds	itemp2, pwmval		;PWM-Helligkeitswert holen
	out	tcnt2, itemp2		;Zhler setzen
;
; Timerzhler 1 bearbeiten (Teilung auf 20ms / 50Hz)
;
	lds	itemp2, tm1cnt		;Timerzhler 1 holen
	inc	itemp2			;erhhen
	andi	itemp2, 0x0f		;bertrag lschen
	sts	tm1cnt, itemp2		;Zhler wieder speichern
	breq	tim200			;Zhler = 0? ja -> weiter
	rjmp	tim1en			;sonst Ende
;
; ADC lesen und Mittelwert bilden (wird alle 20ms ausgefhrt)
;
tim200:	ldi	yl, low(adclst + 3)
	ldi	yh, high(adclst + 3)	;Zeiger auf ADC-Werte (Ende)
tim210:	ld	itemp2, -y		;Wert aus Liste holen
	std	y + 1, itemp2		;in nchste Position speichern
	cpi	yl, low(adclst)		;alle Werte verschoben?
	brne	tim210			;nein -> Schleife
;
	in	itemp2, adcl		;ADC-Wert L verwerfen
	in	itemp2, adch		;ADC Wert H auslesen
	st	y, itemp2		;und in Liste speichern
	sbi	adcsra, adsc		;neue Conversion starten
;
; Taster lesen und entprellen (wird alle 20ms ausgefhrt)
;
	bst	flags, 3		;letzten Taster-Status holen
	bld	itemp2, 0		;in Arbeitsregister legen
	in	itemp1, pina		;Tastereingang lesen
	com	itemp1			;invertieren
	bst	itemp1, 0		;Taster-Bit kopieren
	bld	flags, 3		;als neuen Status speichern
	eor	itemp2, itemp1		;Statusnderung?
	andi	itemp2, 0x01		;Ergebnis filtern
	brne	tim350			;Statusnderung? ja -> Ende
	bld	flags, 4		;sonst Entprellt-Status setzen
	lds	itemp2, keycnt		;Taster-Timeout-Zhler holen
	brts	tim300			;Gedrckt-Status? ja -> weiter
	cbr	flags, 0b11110000	;sonst alle Taster-Flags lschen
	ldi	itemp2, keytio		;und Timeout-Zhler neu setzen
	rjmp	tim320
tim300:	tst	itemp2			;Timeout-Zhler = 0?
	brne	tim310			;nein -> weiter, sonst
	sbr	flags, 0b01000000	;Flag: Taster lang gedrckt
	rjmp	tim320
tim310:	dec	itemp2			;Timeout-Zhler vermindern
tim320:	sts	keycnt, itemp2		;neuen Zhlerstand speichern
;
; Anzeige Timeout-Zhler fr Versionsnummer, Sensorbelegung und Einstellung (wird alle 20ms ausgefhrt)
;
tim350:	lds	itemp2, distio + 1	;Anzeige Timeout-Zhler H holen
	cpi	itemp2, 0xff		;Zhler gestoppt (H = 0xff)?
	breq	tim400			;ja -> Ende
	lds	itemp1, distio		;Anzeige Timeout-Zhler L holen
	or	itemp2, itemp1		;L + H verknpfen, Zhlerstand = 0?
	breq	tim400			;ja -> Ende
	subi	itemp1, 1		;Anzeige Timeout-Zhler L - 1
	sts	distio, itemp1		;Ergebnis L speichern
	lds	itemp2, distio + 1	;Anzeige Timeout-Zhler H holen
	sbci	itemp2, 0		;bertrag subtrahieren
	sts	distio + 1, itemp2	;Timeout-Zhler H speichern
;
; Timerzhler 2 bearbeiten (Teilung auf 1s / 1Hz)
;
tim400:	lds	itemp2, tm2cnt		;Timerzhler 2 holen
	inc	itemp2			;erhhen
	sts	tm2cnt, itemp2		;und wieder speichern
	cpi	itemp2, 50		;Endwert erreicht?
	brcc	tim410			;ja -> weiter
	rjmp	tim1en			;sonst Ende
tim410:	clr	itemp2			;Zhler zurcksetzen
	sts	tm2cnt, itemp2		;und wieder speichern
;
; Sensor-Timeout-Zhler bearbeiten (wird alle 1s ausgefhrt)
;
	lds	itemp2, se1tio		;Sensor 1 Timeout-Zhler holen
	tst	itemp2			;Zhler bereits abgelaufen?
	breq	tim500			;ja -> weiter
	cpi	itemp2, 0xff		;Zhler bereits gestoppt?
	breq	tim500			;ja -> weiter
	dec	itemp2			;sonst Zhler - 1
	sts	se1tio, itemp2		;und wieder speichern
;
tim500:	lds	itemp2, se2tio		;Sensor 2 Timeout-Zhler holen
	tst	itemp2			;Zhler bereits abgelaufen?
	breq	tim510			;ja -> weiter
	cpi	itemp2, 0xff		;Zhler bereits gestoppt?
	breq	tim510			;ja -> weiter
	dec	itemp2			;sonst Zhler - 1
	sts	se2tio, itemp2		;und wieder speichern
;
tim510:	lds	itemp2, se3tio		;Sensor 3 Timeout-Zhler holen
	tst	itemp2			;Zhler bereits abgelaufen?
	breq	tim520			;ja -> weiter
	cpi	itemp2, 0xff		;Zhler bereits gestoppt?
	breq	tim520			;ja -> weiter
	dec	itemp2			;sonst Zhler - 1
	sts	se3tio, itemp2		;und wieder speichern
;
tim520:	lds	itemp2, se4tio		;Sensor 4 Timeout-Zhler holen
	tst	itemp2			;Zhler bereits abgelaufen?
	breq	tim550			;ja -> weiter
	cpi	itemp2, 0xff		;Zhler bereits gestoppt?
	breq	tim550			;ja -> weiter
	dec	itemp2			;sonst Zhler - 1
	sts	se4tio, itemp2		;und wieder speichern
;
; Alarm-Timeout-Zhler bearbeiten (wird alle 1s ausgefhrt)
;
tim550:	ldi	yl, low(al_tio)
	ldi	yh, high(al_tio)	;Zeiger auf Alarm-Timeout-Zhler
	ldi	itemp1, 4		;4 Zhler bearbeiten
tim560:	ld	itemp2, y		;Alarm-Timeout-Zhler holen
	tst	itemp2			;Zhler bereits abgelaufen?
	breq	tim570			;ja -> weiter
	cpi	itemp2, 0xff		;Zhler bereits gestoppt?
	breq	tim570			;ja -> weiter
	dec	itemp2			;sonst Zhler - 1
tim570:	st	y+, itemp2		;und wieder speichern
	dec	itemp1			;alle Zhler bearbeitet?
	brne	tim560			;nein -> Schleife
;
; Datenempfang Timeout-Zhler bearbeiten (wird alle 1s ausgefhrt)
;
tim600:	lds	itemp2, rectio		;Empfang Timeout-Zhler holen
	tst	itemp2			;Zhler bereits abgelaufen?
	breq	tim700			;ja -> weiter
	cpi	itemp2, 0xff		;Zhler bereits gestoppt?
	breq	tim700			;ja -> weiter
	dec	itemp2			;sonst Zhler - 1
	sts	rectio, itemp2		;und wieder speichern
;
; Zhler und Flag fr Wechselanzeige bearbeiten (wird alle 1s ausgefhrt)
;
tim700:	lds	itemp2, chgcnt		;Wechsel-Zhler holen
	inc	itemp2			;erhhen
	sts	chgcnt, itemp2		;und wieder speichern
	lds	itemp1, change		;Wechselzeit holen
	cp	itemp2, itemp1		;Wechselzeit erreicht?
	brcc	tim710			;ja -> weiter
	rjmp	tim1en			;sonst Ende
;
tim710:	clr	itemp2			;sonst Zhler zurcksetzen
	sts	chgcnt, itemp2		;und wieder speichern
	lds	itemp2, chgflg		;Wechsel-Flag holen
	inc	itemp2			;erhhen
	andi	itemp2, 0x01		;bertrag lschen
	sts	chgflg, itemp2		;und wieder speichern
;
; Zhler fr Alarm-Anzeige abhngig von den anstehenden Alarmen setzen
;
	breq	tim1en			;Ende, wenn Wechselanzeige auf Sensor 1+2
	lds	itemp2, alflag		;sonst Alarm-Flags holen
	bst	itemp2, 0		;Alarm-Anzeige aktiv?
	brtc	tim1en			;nein -> Ende
;
	ldi	yl, low(al_tio)		;sonst
	ldi	yh, high(al_tio)	;Zeiger auf Alarm Timeout-Zhler
	ldi	itemp1, 4		;4 Timeout-Zhler prfen
	clr	itemp3			;Alarm-Zhler lschen
tim800:	ld	itemp2, y+		;Alarm Timeout-Zhler holen
	tst	itemp2			;Zhler bereits abgelaufen
	breq	tim810			;ja -> weiter
	cpi	itemp2, 0xff		;Zhler bereits gestoppt?
	breq	tim810			;ja -> weiter
	inc	itemp3			;sonst Alarm-Zhler erhhen
tim810:	dec	itemp1			;alle Zhler bearbeitet?
	brne	tim800			;nein -> Schleife
;
	tst	itemp3			;sind Alarme aktiv?
	brne	tim820			;ja -> weiter
	sts	almnum, itemp3		;sonst Alarmnummer lschen
	lds	itemp1, alflag		;Alarm-Flags holen
	cbr	itemp1, 0b00000010	;kein Alarm ist aktiv
	sts	alflag, itemp1		;Alarm-Flags wieder speichern
	rjmp	tim1en			;Ende
;
tim820:	lds	itemp1, alflag		;Alarm-Flags holen
	sbr	itemp1, 0b00000010	;mindestens 1 Alarm ist aktiv
	sts	alflag, itemp1		;Alarm-Flags wieder speichern
;
	ldi	yl, low(al_tio)
	ldi	yh, high(al_tio)	;Zeiger auf Alarm Timeout-Zhler
	ldi	itemp1, 4		;4 Timeout-Zhler prfen
	lds	itemp2, almnum		;Alarm-Nummer holen
	clr	itemp3			;H-Byte = 0
	add	yl, itemp2		;Zeiger fr Timeout-Zhler
	adc	yh, itemp3		;berechnen, bertrag
;
tim830:	adiw	yl, 1			;Zeiger auf nchsten Timeout-Zhler
	cpi	yl, low(al_tio + 4)	;Zeigeradresse berschritten?
	brne	tim840			;nein -> weiter
	ldi	yl, low(al_tio)		;sonst
	ldi	yh, high(al_tio)	;Zeiger wieder auf Anfang setzen
tim840:	ld	itemp2, y		;Timeout-Zhler holen
	tst	itemp2			;Zhler bereits abgelaufen
	breq	tim850			;ja -> weiter
	cpi	itemp2, 0xff		;Zhler bereits gestoppt?
	breq	tim850			;ja -> weiter
	subi	yl, low(al_tio)		;Alarmnummer ermitteln (0-3)
	sts	almnum, yl		;als nchste Alarm-Nummer speichern
	rjmp	tim1en
;
tim850:	dec	itemp1			;alle Timeout-Zhler geprft?
	brne	tim830			;nein -> Schleife
	sts	almnum, itemp1		;sonst Alarmnummer lschen
	lds	itemp1, alflag		;Alarm-Flags holen
	cbr	itemp1, 0b00000010	;kein Alarm ist aktiv
	sts	alflag, itemp1		;Alarm-Flags wieder speichern
;
tim1en:	out	sreg, isreg		;Status-Register zurckholen
	reti
;
; Interrupt-Routine Timer2, wird bei berlauf ausgefhrt
; Register: isreg, itemp2
;
timer2:	in	isreg, sreg		;Status-Register sichern
	in	itemp2, portb		;PortB einlesen
	andi	itemp2, 0xf0		;alle Anodentreiber ausschalten
	mov	itemp1, invlpb		;LED-Invertierung PortB holen
	andi	itemp1, 0x0f		;Anodentreiber filtern
	eor	itemp2, itemp1		;gegebenenfalls invertieren
	out	portb, itemp2		;PortB setzen
	out	sreg, isreg		;Status-Register zurckholen
	reti
;
; Interrupt-Routine UART RX Complete, empfangenes Zeichen in RX-Puffer ablegen
; Register: isreg, itemp2, yl, yh
;
readrx:	in	isreg, sreg		;Status-Register sichern
	in	itemp2, udr		;empfangenes Byte lesen
	ldi	yh, high(rxbuff)	;Pointer H-Byte setzen
	mov	yl, rxpoi1		;Empfangspuffer-Pointer1 laden
	st	y+, itemp2		;Byte speichern, Pointer + 1
	andi	yl, 0x7f		;bertrag korrigieren
	mov	rxpoi1, yl		;Pointer1 wieder speichern
	out	sreg, isreg		;Status-Register zurckholen
	reti
;
; Segment-Tabelle fr 7-Segment-Anzeigen
;
segtab:	.db	0b00111111, 0b00000110	;Code 00+01: "0" + "1"
	.db	0b01011011, 0b01001111	;Code 02+03: "2" + "3"
	.db	0b01100110, 0b01101101	;Code 04+05: "4" + "5"
	.db	0b01111101, 0b00000111	;Code 06+07: "6" + "7"
	.db	0b01111111, 0b01101111	;Code 08+09: "8" + "9"
	.db	0b01110111, 0b01111100	;Code 0A+0B: "A" + "B"
	.db	0b00111001, 0b01011110	;Code 0C+0D: "C" + "D"
	.db	0b01111001, 0b01110001	;Code 0E+0F: "E" + "F"
	.db	0b00111101, 0b01110110	;Code 10+11: "G" + "H"
	.db	0b00111000, 0b01000000	;Code 12+13: "L" + "-"
	.db	0b01111000, 0b00000000	;Code 14+15: "T" + " "
;
; Diverse Texte
;
errtxt:	.db	"FEHL"		;Fehlertext
almtxt:	.db	"AL- "		;Alarmtext
versnr:	.db	"200 "		;Versionsnummer
se1txt:	.db	"51  "		;Einstellungstext Sensor 1
se2txt:	.db	"52  "		;Einstellungstext Sensor 2
se3txt:	.db	"53  "		;Einstellungstext Sensor 3
se4txt:	.db	"54  "		;Einstellungstext Sensor 4
alstxt:	.db	"AL  "		;Einstellungstext Alarme
chgtxt:	.db	" T  "		;Einstellungstext Wechselzeit
;
;
.eseg
;
; EEPROM-Bereich
;
esens1:	.byte	1	;Sensornummer 1 (fr obere Zeile)
esens2:	.byte	1	;Sensornummer 2 (fr untere Zeile)
ealset:	.byte	4	;Sollzustnde der 4 Alarme
ealflg:	.byte	1	;Alarm-Flag (Bit0: Alarm-Anzeige ist aktiv)
esens3:	.byte	1	;Sensornummer 3 (fr obere Zeile)
esens4:	.byte	1	;Sensornummer 4 (fr untere Zeile)
echang:	.byte	1	;Zeitdauer fr Wechselanzeige
;
