; =====================================================
; Anzeigemodul 2 fr Temperaturmesssystem mit ATmega128
; von Scott-Falk Hhn (s.huehn@soemtron.de)
; Version 2.020, letzte Bearbeitung am 04.03.2007
; =====================================================
;
;
; Wichtig! Diese Software ist nur fr den privaten Gebrauch bestimmt.
; Eine kommerzielle Nutzung ohne Genehmigung des Autors ist verboten.
;
; Important! This software is only meant for private use. A commercial
; use without permission of the author is forbidden.
;
;
; ---------------------------
; Programmierung der Fusebits
; ---------------------------
;
; M103C=1, WDTON=1, OCDEN=0, JTAGEN=0, SPIEN=0, EESAVE=1
; BOOTSZ1=0, BOOTSZ0=0, BOOTRST=1, BODLEVEL=0, BODEN=0, CKOPT=0
; CKSEL3=1, CKSEL2=1, CKSEL1=1, CKSEL0=1, SUT1=0, SUT0=0
;
; ----------------------
; Belegung der I/O-Ports
; ----------------------
;
; A0: Datenleitung D0 fr LCD (Ein-/Ausgang)
; A1: Datenleitung D1 fr LCD (Ein-/Ausgang)
; A2: Datenleitung D2 fr LCD (Ein-/Ausgang)
; A3: Datenleitung D3 fr LCD (Ein-/Ausgang)
; A4: Datenleitung D4 fr LCD (Ein-/Ausgang)
; A5: Datenleitung D5 fr LCD (Ein-/Ausgang)
; A6: Datenleitung D6 fr LCD (Ein-/Ausgang)
; A7: Datenleitung D7 fr LCD (Ein-/Ausgang)
;
; B0: Ausgang LCD-Betriebsspannung (L-aktiv)
; B1: Eingang SCK/ISP (Pull-Up)
; B2: Ein/Ausgang SCL2 \ TWI/I2C-Bus fr
; B3: Ein/Ausgang SDA2 / RTC und RAM
; B4: Ausgang Tonfrequenz (H-aktiv)
; B5: Ausgang Bluetooth (H-aktiv)
; B6: Ausgang PWM Lautstrke (H-aktiv)
; B7: Ausgang PWM LCD-Backlight (H-aktiv)
;
; C0: Datenleitung /WR fr LCD (Ausgang)
; C1: Datenleitung /RD fr LCD (Ausgang)
; C2: Datenleitung /CE fr LCD (Ausgang)
; C3: Datenleitung C/D fr LCD (Ausgang)
; C4: Eingang Encoder-A (Pull-Up)
; C5: Eingang Encoder-B (Pull-Up)
; C6: Eingang Taster 1 (Pull-Up)
; C7: Eingang Taster 2 (Pull-Up)
;
; D0: Ein/Ausgang SCL1 \ TWI/I2C-Bus fr
; D1: Ein/Ausgang SDA1 / Externes EEPROM
; D2: Eingang RXD1 Daten vom Sensormodul
; D3: Ausgang TXD1 Datenausgang fr sptere Erweiterungen
; D4: Ausgang LED 1 rot
; D5: Ausgang LED 2 gelb
; D6: Ausgang Test-LED 1
; D7: Ausgang Test-LED 2
;
; E0: Eingang RXD0 und MOSI/ISP
; E1: Ausgang TXD0 und MISO/ISP
; E2: Eingang DCF77-Empfnger (Pull-up)
; E3: Eingang (frei, Pull-Up)
; E4: Eingang (frei, Pull-Up)
; E5: Eingang (frei, Pull-Up)
; E6: Eingang (frei, Pull-Up)
; E7: Eingang (frei, Pull-Up)
;
; F0: Eingang Lichtsensor (analog)
; F1: Eingang Batteriespannung (analog)
; F2: Eingang (frei, Pull-Up)
; F3: Eingang (frei, Pull-Up)
; F4: Eingang (Pull-Up) JTAG-TCK \
; F5: Eingang (Pull-Up) JTAG-TMS  \ AVR-JTAG
; F6: Eingang (Pull-Up) JTAG-TDO  / Interface
; F7: Eingang (Pull-Up) JTAG-TMI /
;
; G0: Eingang (frei, Pull-Up)
; G1: Eingang (frei, Pull-Up)
; G2: Eingang (frei, Pull-Up)
; G3: Eingang (frei, Pull-Up)
; G4: Eingang (frei, Pull-Up)
;
; ------------------
; Belegung der Timer
; ------------------
;
; Timer0:  Tonerzeugung fr akustischen Alarm
; Timer1a: Haupttimer fr Uhr, Encoder und DCF-Empfang
;	   luft als Interrupt-Routine mit 1600Hz / 625s
; Timer1b: PWM fr Lautstrkeeinstellung, Frequenz 1600Hz
; Timer2:  PWM fr LCD-Beleuchtung, Frequenz ca. 339Hz
; TImer3a: Steuertimer fr Taster, Timeout-Zhler, LEDs und Tonerzeu-
;	   gung, luft als Interrupt-Routine mit 50Hz / 20ms
;
; --------------------------------------------------
; Belegung des externen EERPOMs (2 x 24C512 = 128kB)
; --------------------------------------------------
;
; Adresse 00000-179FF: Temperaturdaten, 36x24x4 Datenstze mit jeweils
;		       16 Temperaturwerten (28 Byte) = 96768 Byte
; Adresse 17A00-1F0BF: Temperaturdaten, 400 Datenstze mit jeweils
;		       3x16 Temperaturwerten (76 Byte) = 30400 Byte
; Adresse 1F0C0-1FC77: Meldungsdaten, 500 Datenstze mit jeweils einer
;		       Ereignismeldung (6 Byte) = 3000 Byte
; Adresse 1FC78-1FFFF: Speicher fr Konfigurations-Backup = 904 Byte
;
.nolist
.include "m128def.inc"
.list
;
; --------------------
; Konstanten festlegen
; --------------------
;
; System-Daten
;
.equ	clock=	11059200;System-Taktfrequenz
.equ	cltim1=	1600	;Taktfrequenz Timer1a (1600Hz/625s)
.equ	cltim3=	50	;Taktfrequenz Timer3a (50Hz/20ms)
.equ	baud1=	9600	;Baudrate RS232-1 (9600 Baud)
.equ	cr=	13	;Zeichencode Cariage Return (Enter)
.equ	lf=	10	;Zeichencode fr Line Feed
.equ	year2=	'0'	;Jahr Hunderter (ASCII 0)
.equ	year3=	'2'	;Jahr Tausender (ASCII 2)
.equ	yearx=	2000	;Jahr Basiswert fr manuelle Zeiteingabe
.equ	eeprom=	1	;erste genutzte interne EEPROM-Adresse
.equ	txbu0l=	256	;Lnge des RS232-0 Ring-Sendepuffers
.equ	rxbu0l=	256	;Lnge des RS232-0 Ring-Empfangspuffers
.equ	rxbu1l=	256	;Lnge des RS232-1 Ring-Empfangspuffers
.equ	rxda0l=	48	;Lnge des RS232-0 Empfangsdatenpuffers
.equ	rxda1l=	8	;Lnge des RS232-1 Empfangsdatenpuffers
.equ	sigbul=	8	;Lnge des Signalpuffers, Anzahl der Wertepaare
.equ	timtmp=	60	;Timeout-Wert fr Temperaturwerte in Sekunden
			;aktuell eingestellt auf 60 Sekunden
.equ	timmsg=	20	;Timeout-Wert fr alle Meldungen in Sekunden
			;aktuell eingestellt auf 20 Sekunden
.equ	timmss=	45	;Timeout-Wert fr alle Meldungen in Sekunden
			;Wert beim Systemstart 45 Sekunden
.equ	timdcf=	24	;Timeout-Wert fr DCF77-Empfang in Stunden
			;aktuell eingestellt auf 24 Stunden
.equ	timmen=	180	;Timeout-Wert fr automatische Rckkehr zur
			;Temperaturanzeige in Sekunden (180s)
.equ	timlsp=	3	;Timeout-Wert fr kurzzeitige Lautsprecher-
			;Abschaltung in Sekunden (3s)
.equ	timlcd=	30	;Timeout-Wert fr LCD-Abschaltung (30min)
.equ	timerr=	100	;Timeout-Wert fr zyklische Fehleranzeige (2s)
.equ	batlim=	280	;Batterieschwellwert, entspricht 2,8V
.equ	minasc=	0x21	;erstes mit Drehencoder whlbares Zeichen
.equ	maxasc=	0x86	;letztes mit Drehencoder whlbares Zeichen
.equ	mintmp=	-550	;kleinste mit Drehencoder einstellbare Temper.
.equ	maxtmp=	1250	;grte mit Drehencoder einstellbare Temperatur
.equ	minfrq=	675	;minimale mit Drehencoder einstellbare Frequenz
.equ	maxfrq=	5959	;maximale mit Drehencoder einstellbare Frequenz
;
; RTC-Daten
;
.equ	rtcadr=	0xa0	;I2C Basisadresse des RTC/RAM (PCF8583)
;
.equ	rtflag=	0x10	;RTC-Adresse: Flagbyte, 0x5a = Zeit gltig
.equ	rtyear=	0x11	;RTC-Adresse: Jahr Zehner/Einer (BCD)
.equ	rtxt1p=	0x12	;RTC-Adresse: Ext-EEPROM-Pointer Temp-Daten 1
			;	      3 Bytes + 1 Flag (0x12-0x15)
.equ	rtxt2p=	0x16	;RTC-Adresse: Ext-EEPROM-Pointer Temp-Daten 2
			;	      3 Bytes + 1 Flag (0x16-0x19)
.equ	rtxmsp=	0x1a	;RTC-Adresse: Ext-EEPROM-Pointer Meldungsdaten
			;	      3 Bytes + 1 Flag (0x1a-0x1d)
;
;			;0x1e, 0x1f nicht genutzt
;
.equ	rtbval=	0x20	;RTC-Adresse: Backup Temperaturwerte (208 Byte)
			;	      Bereich 0x20-0xef
.equ	rtbdat=	0xf0	;RTC-Adresse: Datum des letzen Backups (4 Byte)
			;	      Tag-Z, Tag-E, Monat-Z, Monat-E
.equ	rtbflg=	0xf4	;RTC-Adresse: Flag fr Backup, 0x5a = gltig
;
; EEPROM-Daten (extern)
;
.equ	xeeadr=	0xa0	;I2C Basisadresse des externen EEPROMs (2x64kB)
			;B0: R/W-Bit (0=Write, 1=Read)
			;B1: Select-Bit (0=EEPROM1, 1=EEPROM2)
.equ	xt1sta=	0		;Ext-EEPROM: Beginn Temp-Daten 1
.equ	xeeend=	0x2000		;Ext-EEPROM Ende
.equ	xt1len=	36*24*4*(4+24)	;Ext-EEPROM: Lnge der Temp-Daten 1
				;(36 Tage, 4 Messwerte pro Stunde)
.equ	xt2sta=	xt1sta+xt1len	;Ext-EEPROM: Beginn Temp-Daten 2
.equ	xt2len=	400*(4+(3*24))	;Ext-EEPROM: Lnge der Temp-Daten 2
				;(400 Tage, Min/Max/Mittel pro Tag)
.equ	xmssta=	xt2sta+xt2len	;Ext-EEPROM: Beginn Meldungsdaten
.equ	xmslen=	500*6		;Ext-EEPROM: Lnge der Meldungsd.
				;(500 Ereignismeldungen mit je 6 Byte)
.equ	cfbsta=	xmssta+xmslen	;Ext-EEPROM: Beginn des Konfig-Backups
.equ	cfblen=	xeeend-cfbsta	;Ext-EEPROM: Lnge des Konfig-Backups
				;(restlicher Speicher, 904 Byte)
;
; -----------------------
; Registernamen festlegen
; -----------------------
;
.def	insreg=	r2	;Zwischenspeicher fr SREG whrend Interrupt
.def	txflag=	r3	;Textflags, enthlt Textattribute
			;B0: Textausgabe invers
			;B1: Textausgabe mit Unterstreichung
			;B2: Textausgabe mit kleinem Font (Hhe 5 Pix.)
;
; ------------------------
; Speicheradressen im SRAM
; ------------------------
;
.dseg
.org	0x100
;
txbuf0:	.byte	txbu0l	;RS232-0 Ring-Sendepuffer
rxbuf0:	.byte	rxbu0l	;RS232-0 Ring-Empfangspuffer
rxbuf1:	.byte	rxbu1l	;RS232-1 Ring-Empfangspuffer
rxdat0:	.byte	rxda0l	;RS232-0 Empfangsdaten
rxdat1:	.byte	rxda1l	;RS232-1 Empfangsdaten
sigbuf:	.byte	sigbul*2;Signal-Puffer, enthlt die Warteschlange fr
			;die auszugebenden akustischen Signale
			;(Signalnummer und Signaldauer)
;
txp0wr:	.byte	2	;RS232-0 Ring-Sendepuffer Schreibposition
txp0rd:	.byte	2	;RS232-0 Ring-Sendepuffer Leseposition
rxp0wr:	.byte	2	;RS232-0 Ring-Empfangspuffer Schreibposition
rxp0rd:	.byte	2	;RS232-0 Ring-Empfangspuffer Leseposition
rxp1wr:	.byte	2	;RS232-1 Ring-Empfangspuffer Schreibposition
rxp1rd:	.byte	2	;RS232-1 Ring-Empfangspuffer Leseposition
rxdap0:	.byte	2	;RS232-0 Empfangsdatenpuffer Position
rxdap1:	.byte	2	;RS232-1 Empfangsdatenpuffer Position
sigbwr:	.byte	2	;Signal-Puffer Schreibposition
sigbrd:	.byte	2	;Signal-Puffer Leseposition
;
temmap:	.byte	16	;Temperatur-Anzeige-Mapping-Tabelle, enthlt
			;die LCD-Positionen der 16 Temperaturwerte
			;(werden aus EERPOM gelesen)
logmap:	.byte	16	;Temperatur-Log-Mapping-Tabelle, enthlt die
			;Listenpositionen der 16 Temperaturwerte
			;(werden aus EERPOM gelesen)
tembak:	.byte	16	;Kopie der Mapping-Tabelle fr Positionseingabe
temflg:	.byte	16	;Flags fr jeden der 16 Temperaturwerte
			;B0: Temperaturwert neu ausgeben
			;B1: Sensorname neu ausgeben
			;B2: Temperaturberschreitung
			;B3: Temperaturunterschreitung
			;B4: Erinnerung nach Temperaturberschreitung
			;B5: Erinnerung nach Temperaturunterschreitung
			;B6: Sensorausfall
			;B7: Temperatursensor nicht genutzt
temtio:	.byte	16	;Timeout-Zhler fr jede Temperatur; wird bei
			;Empfang eines Datensatzes auf 60s gesetzt und
			;vom Timer-Interrupt auf 0 gezhlt; Hauptpro-
			;gramm signalisiert bei 0 den Temperaturausfall
			;und setzt den Wert auf 0xff (Stopp)
temval:	.byte	16*2	;Temperaturwerte der 16 Sensoren im Zweier-
			;komplement 16bit, 0x8000 = Temp. ungltig
temlil:	.byte	16*2	;untere Temperaturlimits der 16 Sensoren im
			;Zweierkomplement 16bit, werden aus dem EEPROM
			;gelesen, 0x8000 = Limit nicht aktiv
			;Hex, wrden aus EEPROM gelesen)
temlih:	.byte	16*2	;obere Temperaturlimits der 16 Sensoren im
			;Zweierkomplement 16bit, werden aus dem EEPROM
			;gelesen, 0x8000 = Limit nicht aktiv
temin1:	.byte	16*2	;ermittelte Minimalwerte der 16 Sensoren im
			;Zweierkomplement 16bit, Werte fr Anzeige
temax1:	.byte	16*2	;ermittelte Maximalwerte der 16 Sensoren im
			;Zweierkomplement 16bit, Werte fr Anzeige
temin2:	.byte	16*2	;ermittelte Minimalwerte der 16 Sensoren im
			;Zweierkomplement 16bit, Werte fr EEPROM-Log
temax2:	.byte	16*2	;ermittelte Maximalwerte der 16 Sensoren im
			;Zweierkomplement 16bit, Werte fr EEPROM-Log
temavs:	.byte	16*3	;Summen fr Mittelwertberechnung im Zweierkom-
			;plement 24bit, Werte fr EEPROM-Log, jeder
			;empfangene Temperaturwert wird addiert
temavc:	.byte	16*2	;Zhler fr Mittelwertberechnung im Zweierkom-
			;plement 16bit, Werte fr EEPROM-Log, wird bei
			;jedem empfangenen Temperaturwert um 1 erhht
almmap:	.byte	4	;Alarm-Mapping-Tabelle, enthlt die LCD-
			;Positionen der 4 Alarmeingnge
almbak:	.byte	4	;Kopie der Mapping-Tabelle fr Positionseingabe
alminv:	.byte	4	;Alarm-Invertierungs-Tabelle; bestimmt, ob die
			;Alarmmeldung invertiert ausgewertet wird:
			;0= normal
			;1= invertiert
almflg:	.byte	4	;Flags fr jeden der 4 Alarmeingnge
			;B0: Alarmname neu ausgeben
			;B1: Alarm aktiv
			;B2: Erinnerung nach Alarmauslsung
			;B7: Alarmeingang nicht genutzt
cfgflg:	.byte	8	;Konfiguration-Flags, werden fr Konfiguration-
			;Dialoge verwendet (fr jede Bildschirmzeile):
			;B0: Name neu ausgeben
			;B1: Daten neu ausgeben
senmap:	.byte	8	;empfangene Sensorbelegung (8 Zeichen)
dflags:	.byte	1	;Anzeige-Flags
			;B0: Normalanzeige (Temperatur+Alarme)
			;B1: Anzeige lschen und neu ausgeben
			;B2: Infozeile neu ausgeben (obere Zeile)
			;B3: Menzeile neu ausgeben (untere Zeile)
			;B4: DCF77-Empfang ok (Synchron. innerhalb 24h)
			;B5: Lautsprecher ist aktiviert (Symbolanzeige)
			;B6: Hintergrundbeleuchtung einschalten
			;B7: Anzeige ist abgeschaltet (fr DCF-Empfang)
			;    wird durch Routinen dcfon/dcfoff gesteuert
xflags:	.byte	1	;Verschiedene Flags 1
			;B0: Signalausgabe aktiv
			;B1: Erinnerungen sind gesetzt
			;B2: Minimalwerte anstatt aktuelle Temperaturen
			;B3: Maximalwerte anstatt aktuelle Temperaturen
			;B4: Status des Lichtsensors (1=hell)
			;B5: Status der Nachtschaltung (1=Nachtzeit)
			;B6: Externes EEPROM 1 (24C512) vorhanden
			;B7: Externes EEPROM 2 (24C512) vorhanden
xflag2:	.byte	1	;Verschiedene Flags 2
			;B0: aktuelle Lautstrke>0 (Lautsprecher ein)
			;B1: Ext-EEPROM-Pointer-Fehler Temp-Daten 1
			;B2: Ext-EEPROM-Pointer-Fehler Temp-Daten 2
			;B3: Ext-EEPROM-Pointer-Fehler Meldungs-Daten
			;B4: Datenfelder der Statusanzeige neu ausgeben
			;B5: Anzeigerichtung fr Meldungen:
			;    0= nchste Seite, 1= vorherige Seite
			;B6: Listenende bei Meldungsanzeige (nchste S)
			;B7: Listenende bei Meldungsanzeige (vorher. S)
xflag3:	.byte	1	;Verschiedene Flags 3
			;B0: Konfiguration im EEPROM speichern
			;B1: Grafikanzeige, Daten aus EERPOM einlesen
			;B2: Grafikanzeige, Grafik auf LCD ausgeben
			;B3: Grafikanzeige, Anzeigerichtung:
			;    0= nchste Seite, 1= vorherige Seite
			;B4: Grafikanzeige, letzter Datensatz erreicht
			;B5: Grafikanzeige, erster Datensatz erreicht
xflag4:	.byte	1	;Verschiedene Flags 4
			;B0: Konfig/Grenzwerte/Unteren, GW ndern
			;B1: Konfig/Grenzwerte/Oberen, GW ndern
			;B2: Konfig/Signale/Zuordnung, Nummer ndern
			;B3: Konfig/Signale/Zuordnung, Dauer ndern
ledflg:	.byte	1	;LED-Flags
			;B0: LED1 (rot) stndig einschalten
			;B1: LED2 (gelb) stndig einschalten
			;B2: LED1 (rot) Blinken mit 3Hz
			;B3: LED2 (gelb) Blinken mit 3Hz
			;B4: LED1 (rot) Blitzen mit 0,5Hz (Erinnerung)
			;B5: LED2 (gelb) Blitzen mit 0,5Hz (Erinnerung)
confg1:	.byte	1	;Konfigurationsbyte 1
			;B0: Sprache (0=englisch, 1=deutsch)
			;B1: Trennzeichen (0=Punkt, 1=Komma)
			;B2: Temperaturalarme immer signalisieren
			;B3: Nachtschaltung fr LCD-Beleuchtung aktiv
			;B4: Auto-Reset fr Min/Max-Werte aktiv
			;B5: Nachtschaltung fr Signal-Lautstrke aktiv
			;B6: LCD abschalten fr Uhr-Synchronisation
			;B7: 0=Normal, 1=Abwesend (Licht und Ton aus)
confg2:	.byte	1	;Konfigurationsbyte 2
			;B0: LED-Modus (0=normal, 1=debug [DCF,RS-232])
			;B1: LCD-Beleuchtung weich schalten
			;B2: Nachtschaltung fr Bluetooth-Modul aktiv
bigtmp:	.byte	1	;Sensornummer fr Gro-Anzeige (0-15)
errflg:	.byte	1	;Fehler-Flags, sind alle Bits auf 0, dann wird
			;in der Infozeile das Datum normal angezeigt,
			;sonst erscheint invers die Fehlermeldung:
			;B0: Uhr nicht eingestellt
			;B1: kein Datenempfang
			;B2: Konfigurationsfehler (internes EEPROM)
			;B3: Speicherfehler (externes EEPROM)
			;B4: RTC-Fehler
			;B5: Pointer-Fehler
			;B7: Fehlererinnerung (Fehler war aufgetreten)
errcpy:	.byte	1	;Fehler-Flags-Kopie, wird fr die zyklische
			;Fehleranzeige bentigt
errmsk:	.byte	1	;Fehler-Maske, ist normalerweise 0 und enthlt
			;(falls mehrere Alarme aktiv sind) das Alarmbit
			;fr den nchsten Anzeigezyklus
errtio:	.byte	1	;Timeout-Zhler fr zyklische Fehleranzeige;
			;wird beim Auftreten mehrerer Fehler vom Haupt-
			;programm auf 2s gesetzt und vom Timer-Inter-
			;rupt auf 0 gezhlt
msgtio:	.byte	1	;Timeout-Zhler fr alle Meldungen; wird bei
			;Empfang eines Datensatzes auf 20s gesetzt und
			;vom Timer-Interrupt auf 0 gezhlt; Hauptpro-
			;gramm signalisiert bei 0 den Temperaturausfall
			;und setzt den Wert auf 0xff (Stopp)
dcftio:	.byte	1	;Timeout-Zhler fr DCF77-Empfang; wird bei
			;erfolgreicher Synchronisation auf 24h gesetzt
			;und vom Timer-Interrupt auf 0 gezhlt; Haupt-
			;programm lscht dann Bit 4 in dflags
signum:	.byte	8	;Signalnummer-Tabelle, enthlt fr jedes
			;Ereignis die Nummer des Signal-Sounds (0-7)
			;Byte 0: Signalnummer fr Alarmeingang 1
			;Byte 1: Signalnummer fr Alarmeingang 2
			;Byte 2: Signalnummer fr Alarmeingang 3
			;Byte 3: Signalnummer fr Alarmeingang 4
			;Byte 4: Signalnummer fr Temperaturberschr.
			;Byte 5: Signalnummer fr Temperaturunterschr.
			;Byte 6: Signalnummer fr Fehlermeldungen
			;Byte 7: Signalnummer fr Sensorausflle
sigdur:	.byte	8	;Signaltabelle, enthlt fr jedes Ereignis die
			;Dauer des akustischen Signals in Sekunden,
			;0 entspricht kein Signal
			;Byte 0: Signaldauer fr Alarmeingang 1
			;Byte 1: Signaldauer fr Alarmeingang 2
			;Byte 2: Signaldauer fr Alarmeingang 3
			;Byte 3: Signaldauer fr Alarmeingang 4
			;Byte 4: Signaldauer fr Temperaturberschr.
			;Byte 5: Signaldauer fr Temperaturunterschr.
			;Byte 6: Signaldauer fr Fehlermeldungen
			;Byte 7: Signaldauer fr Sensorausflle
sigsnd:	.byte	1	;zu erzeugendes akustisches Signal (0-7)
sigtim:	.byte	1	;Dauer der akustischen Signalisierung in Sek.
sigcnt:	.byte	1	;Signalzhler, zhlt in 20ms-Schritten bis 50
sigvo1:	.byte	1	;Signal-Lautstrke 1 Tag (0-15)
sigvo2:	.byte	1	;Signal-Lautstrke 2 Nacht (0-15)
lsptio:	.byte	1	;Timeout-Zhler fr kurzzeitige Lautsprecher-
			;Abschaltung; wird bei Tastendruck 3s gesetzt
			;und vom Timer-Interrupt auf 0 gezhlt; Haupt-
			;programm aktiviert bei 0 den Lautsprecher
t1cnt1:	.byte	1	;Timer1 Interrupt-Zhler 1, zhlt die Inter-
			;rupts von Timer 1 (bis 32; 625s * 32 = 20ms)
t1cnt2:	.byte	1	;Timer1 Interrupt-Zhler 2, zhlt die Zyklen
			;von Zhler1 (bis 50; 20ms * 50 = 1s)
t3cnt1:	.byte	1	;Timer3 Interrupt-Zhler 1, zhlt die Inter-
			;rupts von Timer3 (bis 50; 20ms * 50 = 1s)
t3cnt2:	.byte	1	;Timer3 Interrupt-Zhler 2, zhlt die Zyklen
			;von Zhler1 (bis 60; 1s * 60 = 1min)
t3cnt3:	.byte	1	;Timer3 Interrupt-Zhler 3, zhlt die Zyklen
			;von Zhler2 (bis 60; 1min * 60 = 1h)
ensta1:	.byte	1	;Dreh-Encoder Status beim letzten Interrupt
			;B2: Zustand Encoder-Ausgang A \ nur fr Inter-
			;B3: Zustand Encoder-Ausgang B / rupt-Routine
ensta2:	.byte	1	;letzter stabiler Dreh-Encoder Status
			;B2: Zustand Encoder-Ausgang A \ nur fr Inter-
			;B3: Zustand Encoder-Ausgang B / rupt-Routine
encpos:	.byte	1	;aktuelle Encoder-Position (8bit Zweierkompl.)
encmem:	.byte	1	;letzte gespeicherte Encoder-Position
encflg:	.byte	1	;Encoder-Flags
			;B0: Encoderposition als Menoption anzeigen
			;B1: Anzeige dezimal 3-stellig (0-199)
			;B2: Anzeige Temperaturwert (-55.0 - 125.0)
			;B3: Anzeige in ASCII (minasc-maxasc)
			;B4: Anzeige dezimal 4-stellig (0-9999)
			;B5: Anzeige im Zeitformat (0:00-23:59)
			;B7: Encoderwert hat sich gendert
encval:	.byte	1	;aktueller Encoderwert L
encvah:	.byte	1	;aktueller Encoderwert H
encmin:	.byte	1	;Encoder-Minimalwert (bei 3-stelliger Anzeige)
encmih:	.byte	1	;Encoder-Minimalwert H (bei 4-stelliger Anz.)
encmax:	.byte	1	;Encoder-Maximalwert (bei 3-stelliger Anzeige)
encmah:	.byte	1	;Encoder-Maximalwert H (bei 4-stelliger Anz.)
keyst1:	.byte	1	;Status der beiden Taster beim letzt. Interrupt
			;B6: Zustand von Taster1
			;B7: Zustand von Taster2
keyst2:	.byte	1	;Status der beiden Taster nach Entprellung
			;B6: Taster1 \ wird von Interrupt gesetzt und
			;B7: Taster2 / wieder gelscht
keyst3:	.byte	1	;Quittierungs-Status der beiden Taster
			;B6: Taster1 \ wird vom Hauptprogramm gesetzt
			;B7: Taster2 / und von Interrupt wied. gelscht
dcftab:	.byte	6	;DCF77-Tabelle, enthlt empfangene Bits
			;eingehende Bits werden in Byte 5, Bit 7 ein-
			;geschoben und alle weiteren Bits um eine
			;Position weitergerckt, Byte/Bit-Positionen
			;nach einem kompletten Minutenzyklus:
			;0/6-7: Zeitzone (6=1 Sommer, 7=1 Winter)
			;1/2-5: Minute Einer
			;1/6-7: Minute Zehner LSB
			;2/0:   Minute Zehner MSB
			;2/1:   Prfbit 1 (Minuten)
			;2/2-5: Stunde Einer
			;2/6-7: Stunde Zehner
			;3/0:   Prfbit 2 (Stunden)
			;3/1-4: Kalendertag Einer
			;3/5-6: Kalendertag Zehner
			;3/7:   Wochentag LSB
			;4/0-1: Wochentag MSB
			;4/2-5: Monat Einer
			;4/6:   Monat Zehner
			;4/7:   Jahr Einer LSB
			;5/0-2: Jahr Einer MSB
			;5/3-6: Jahr Zehner
			;5/7:   Prfbit 3 (Datum)
dcflag:	.byte	1	;DCF77-Flags
			;B0: aktueller DCF77-Zustand
			;B1: vorheriger DCF77-Zustand
			;B2: Synchronisations-Flag, wird bei DCF-Sync.
			;    gesetzt und zur vollen Stunde gelscht
			;B3: Beginn einer neuen Minute
dcfcnt:	.byte	1	;DCF77-Zhler zur Ermittlung der Impulslnge
dcfimp:	.byte	1	;DCF77-Zhler zur Ermittlung der Impulsanzahl
dcbuf1:	.byte	12	;DCF-Puffer 1, aktuelle decodierte Uhrzeit
			;Byte  0: Zeitzone
			;Byte  1: Minute Einer
			;Byte  2: Minute Zehner
			;Byte  3: Stunde Einer
			;Byte  4: Stunde Zehner
			;Byte  5: Kalendertag Einer
			;Byte  6: Kalendertag Zehner
			;Byte  7: Wochentag
			;Byte  8: Monat Einer
			;Byte  9: Monat Zehner
			;Byte 10: Jahr Einer
			;Byte 11: Jahr Zehner
dcbuf2:	.byte	12	;DCF-Puffer 2, vorherige decodierte Uhrzeit
			;+1 Minute, Aufbau wie dcbuf1
dcbuf3:	.byte	12	;DCF-Puffer 3, vorherige decodierte Uhrzeit
			;+2 Minuten, Aufbau wie dcbuf1
;
			;aktuelle Uhrzeit
secndl:	.byte	1	;Sekunde Einer (0-9)
secndh:	.byte	1	;Sekunde Zehner (0-5)
minutl:	.byte	1	;Minute Einer (0-9)
minuth:	.byte	1	;Minute Zehner (0-5)
hourl:	.byte	1	;Stunde Einer (0-9)  \ zhlen zusammen
hourh:	.byte	1	;Stunde Zehner (0-2) / bis 24
dayw:	.byte	1	;Wochentag (0-6)
dayl:	.byte	1	;Kalendertag Einer (0-9)
dayh:	.byte	1	;Kalendertag Zehner (0-3)
monthl:	.byte	1	;Monat Einer (0-9)
monthh:	.byte	1	;Monat Zehner (0-1)
yearl:	.byte	1	;Jahr Einer (0-9)
yearh:	.byte	1	;Jahr Zehner (0-9)
tzone:	.byte	1	;Zeitzone (0=Winterzeit, 1=Sommerzeit)
;
			;Zeitkopie fr bertragung RAM<->RTC
ct1cnt:	.byte	1	;Timer1 Interrupt-Zhler 2, zhlt die Zyklen
csecdl:	.byte	1	;Sekunde Einer (0-9)
csecdh:	.byte	1	;Sekunde Zehner (0-5)
cminul:	.byte	1	;Minute Einer (0-9)
cminuh:	.byte	1	;Minute Zehner (0-5)
chourl:	.byte	1	;Stunde Einer (0-9)  \ zhlen zusammen
chourh:	.byte	1	;Stunde Zehner (0-2) / bis 24
cdayw:	.byte	1	;Wochentag (0-6)
cdayl:	.byte	1	;Kalendertag Einer (0-9)
cdayh:	.byte	1	;Kalendertag Zehner (0-3)
cmontl:	.byte	1	;Monat Einer (0-9)
cmonth:	.byte	1	;Monat Zehner (0-1)
cyearl:	.byte	1	;Jahr Einer (0-9)
cyearh:	.byte	1	;Jahr Zehner (0-9)
;
cptiml:	.byte	1	;Editor-Kopie, Uhrzeit (Minutenwert L)
cptimh:	.byte	1	;Editor-Kopie, Uhrzeit (Minutenwert H)
cpday:	.byte	1	;Editor-Kopie, Kalendertag (1-31)
cpmon:	.byte	1	;Editor-Kopie, Monat (1-12)
cpyear:	.byte	1	;Editor-Kopie, Jahr Zehner/Einer (0-99)
cpdayw:	.byte	1	;Editor-Kopie, Wochentag (0-6)
;
mennum:	.byte	1	;aktuelle Mennummer
menopt:	.byte	1	;aktuelle Menoption
mentio:	.byte	1	;Timeout-Zhler fr Men; wird bei Bettigung
			;eines Tasters oder des Drehencoders auf 180s
			;gesetzt und vom Timer-Interrupt auf 0 gezhlt;
			;das Hauptprogramm kehrt dann zur Temperatur-
			;anzeige zurck
menext:	.byte	1	;Mentext-Erweiterung; enthlt Stringnummer des
			;zustzlich anzuzeigenden Mentextes oder 0
menpos:	.byte	1	;ausgewhlte Men-Position
lcdtio:	.byte	1	;Timeout-Zhler fr LCD-Abschaltung; wird beim
			;Abschalten des LCD auf 30min gesetzt und vom
			;Hauptprogramm auf 0 gezhlt; das LCD wird dann
			;wieder eingeschaltet
minfl1:	.byte	1	;Minuten-Flag zur Erkennung des Minutenwechsels
			;fr Neuausgabe der Infozeile
minfl2:	.byte	1	;Minuten-Flag zur Erkennung des Minutenwechsels
			;fr Nachtschaltungen und Min/Max-Autoreset
minfl3:	.byte	1	;Minuten-Flag zur Erkennung des Minutenwechsels
			;fr Log-Aufzeichnungen
houflg:	.byte	1	;Stunden-Flag, Wert des letzten Stunden-Einers
			;(Erkennung des Stundenwechsels fr RTC-Sync.)
signig:	.byte	4	;Nachtschaltung fr akustische Signalisierung
			;Byte 0/1: Startzeit in Minuten (L/H)
			;Byte 2/3: Endzeit in Minuten (L/H)
nighbl:	.byte	4	;Nachtschaltung LCD-Hintergrundbeleuchtung
			;Byte 0/1: Startzeit in Minuten (L/H)
			;Byte 2/3: Endzeit in Minuten (L/H)
nighbt:	.byte	4	;Nachtschaltung fr Bluetooth-Modul
			;Byte 0/1: Startzeit in Minuten (L/H)
			;Byte 2/3: Endzeit in Minuten (L/H)
aresmm:	.byte	4	;Zeiten fr Auto-Reset der Min/Max-Werte
			;Byte 0/1: Auto-Reset-Zeit fr Minimalwerte
			;Byte 2/3: Auto-Reset-Zeit fr Maximalwerte
light:	.byte	1	;Schwellwert fr die Steuerung der Hintergrund-
			;beleuchtung, wird aus EEPROM gelesen (5-250)
adclig:	.byte	2	;aktueller 10-Bit-Wert von ADC0 (Lichtsensor)
adcbat:	.byte	8	;aktueller 10-Bit-Wert von ADC1 (RTC-Batterie)
			;und 3 Werte von vorangegangenen Messungen
adcref:	.byte	8	;aktueller 10-Bit-Wert der ADC-Referenz (1,23V)
			;und 3 Werte von vorangegangenen Messungen
;
xewbuf:	.byte	4+(24*3);Schreibpuffer fr externes EEPROM, enthlt
			;einen Datensatz fr Temperaturen od. Meldungen
			;
			;Byte 0-3: komprimierte Zeit in 4 Bytes
			;  Byte 3     Byte 2     Byte 1     Byte 0
			;[--------] [--------] [--------] [--------]
			; \_____/\_____/\___/   \______/   \______/
			;  Jahr   Monat  Tag     Stunde     Minute
			;
			;weitere Bytes bei Temperaturdaten:
			;
			;Byte 4-27: Temperaturdaten (24 Byte) [Minimal]
			;  Byte 6     Byte 5     Byte 4   \
			;[--------] [--------] [--------]  \ Temperatur
			; \__/\__/   \______/   \______/   / Daten
			;  W2H W1H    Wert2L     Wert1L	  /
			;
			;Byte 28-51: Temperaturdaten (24 Byte) [Maxim.]
			;Byte 52-75: Temperaturdaten (24 Byte) [Mittel]
			;
			;weitere Bytes bei Meldungsdaten:
			;
			;Byte 4-5: Sekunden und Meldungsdaten (2 Byte)
			;  Byte 4     Byte 5   \  MNr= Meldungsnummer
			;[--------] [--------]  \ (0x10-0xd0)
			; \______/   \__/\__/   / SNr= Sensor/Alarm
			;  Sekunde    MNr SNr  /  (0-15) oder (0-3)
;
xerbuf:	.byte	4+(24*3);Lesepuffer fr externes EEPROM fr PC-Daten-
			;ausgabe; gleicher Aufbau wie xewbuf
;
xedbuf:	.byte	4+(24*3);Lesepuffer fr externes EEPROM fr Ausgabe auf
			;dem LCD; gleicher Aufbau wie xewbuf
;
xetemp:	.byte	3	;Zwischenspeicher fr die Suchroutine fr die
			;Schreibposition, enthlt Endadresse
xet1po:	.byte	3	;Zeiger (17 Bit) auf aktuelle Schreibposition
			;fr Temp-Daten 1 im externen EEPROM
xet2po:	.byte	3	;Zeiger (17 Bit) auf aktuelle Schreibposition
			;fr Temp-Daten 2 im externen EEPROM
xemspo:	.byte	3	;Zeiger (17 Bit) auf aktuelle Schreibposition
			;fr Meldungsdaten im externen EEPROM
baud0:	.byte	1	;Baudrate fr RS232-0, 0=9600, 1=19200, 2=
			;38400, 3=57600, 4=115200, 5=230400, 6=460800
pccmd:	.byte	1	;PC-Schnittstelle, auszufhrendes Kommando
pccpnt:	.byte	3	;Zeiger auf aktuelle Position im externen
			;EEPROM whrend Kommando-Bearbeitung
pcccnt:	.byte	2	;Datensatzzhler whrend Kommando-Bearbeitung
pcpara:	.byte	9	;PC-Schnittstelle, Parameterspeicher, Byte 0-7
			;enthalten den empfangenen Datumswert, Byte 8
			;speichert ein empfangenes Minuszeichen
ascbuf:	.byte	128	;Zwischenspeicher fr EEPROM-Datenkonvertie-
			;rung, enthlt die PC-Sendedaten
ascibf:	.byte	16	;ASCII-Eingabepuffer fr Nameneingabe
ascibp:	.byte	1	;ASCII-Eingabepuffer, aktuelle Eingabeposition
ascibm:	.byte	1	;ASCII-Eingabepuffer, maximale Zeichenanzahl
dimode:	.byte	1	;alternativer Anzeigemodus; wird ausgewertet,
			;wenn dflags/Bit0=0 ist, mgliche Werte:
			; 1= Meldungsdaten anzeigen
			; 2= Status anzeigen
			; 3= Konfig/Temperaturen, Position auswhlen
			; 4= Konfig/Temperaturen, Position verschieben
			; 5= Konfig/Temperaturen, Name eingeben
			; 6= Konfig/Temperaturen, Grenzwerte, Auswhlen
			; 7= Konfig/Temperaturen, Grenzwerte, ndern
			; 8= Konfig/Alarme, Position auswhlen
			; 9= Konfig/Alarme, Position verschieben
			;10= Konfig/Alarme, Name eingeben
			;11= Konfig/Sig/Zuordnung, Position auswhlen
			;12= Konfig/Sig/Zuordnung, Nummer/Dauer eingeb.
			;13= Konfig/Signale/Audio, Position auswhlen
			;14= Konfig/Signale/Audio, Daten ndern
			;15= Konfig/Z/Schaltzeiten, Position auswhlen
			;16= Konfig/Z/Schaltzeiten, Zeit ndern
			;17= Konfig/Z/Uhr stellen, Position auswhlen
			;18= Konfig/Z/Uhr stellen, Daten ndern
			;19= Konfig/Sonstiges, Position auswhlen
			;20= Konfig/Sonstiges, Daten ndern
			;21= Konfig/Temp/LogZuord, Position auswhlen
			;22= Konfig/Temp/LogZuord, Position verschieben
			;23= Grafik anzeigen
			;24= Grafik, Fadenkreuz anzeigen und bearbeiten
			;25= Konfig/Grafik-Presets, Position auswhlen
			;26= Konfig/Grafik-Presets, Daten ndern
msgfil:	.byte	1	;Meldungsfilter, steuert die Menanzeige und
			;die Datenausgabe, folgende Werte sind mglich:
			;0= alles anzeigen
			;1= nur Alarme anzeigen
			;2= nur Temperaturen anzeigen
			;3= nur Fehler anzeigen
msgpo1:	.byte	3	;(17 Bit) zeigt auf die aktuelle Position im
			;EEPROM der gerade angezeigten Meldung
msgpo2:	.byte	3	;(17 Bit) zeigt auf die Position im EEPROM der
			;Meldung in der ersten Anzeige-Position; fr
			;die Anzeige der vorherigen Meldungsseite
rcvflg:	.byte	1	;wird bei fehlerfreiem RS-232 Empfang auf 0
			;gesetzt und von der Status-Anzeige ausgewertet
grpres:	.byte	1	;ausgewhltes Grafik-Preset (0-23), bei Auswahl
			;werden die folgenden 5 Bytes eingelesen:
grsen1:	.byte	1	;Grafikanzeige, Grafik 1 Sensornummern:
			;Bits 3-0: Sensornummer (0-15)
			;Bits 7-4: Differenz-Sensornummer (0-15)
grsen2:	.byte	1	;Grafikanzeige, Grafik 2 Sensornummern:
			;Bits 3-0: Sensornummer (0-15)
			;Bits 7-4: Differenz-Sensornummer (0-15)
grmod1:	.byte	1	;Grafikanzeige, Anzeigemodus 1 (0-3):
			;0= 15-min-Werte
			;1= 24-h-Minimalwerte
			;2= 24-h-Maximalwerte
			;3= 24-h-Mittelwerte
grmod2:	.byte	1	;Grafikanzeige, Anzeigemodus 1 (0-3), siehe
			;grmod1, wird nur bei grmod1>0 ausgewertet
grflag:	.byte	1	;Grafikanzeige, Flags fr spezielle Funktionen:
			;B0= Sensornamen in Grafik einblenden
			;B1= Temperaturskala angleich. (Doppeldiagramm)
			;B2= Differenzmodus fr Diagramm 1
			;B3= Differenzmodus fr Diagramm 2
			;B4= Doppeldiagramm
grpoin:	.byte	1	;Grafikanzeige, aktuelle Pos. im Datenpuffer
			;und aktuelle Fadenkreuzposition (0-221)
grlacr:	.byte	1	;letzte Fadenkreuzposition (0-221)
grbuff:	.byte	222*8	;Grafikanzeige, Datenpuffer fr 222 Werte:
			;Byte 0-3: komprimiertes Zeitformat wie xewbuf
			;Byte 4-5: Temperaturwert 1 im Zweierkomplement
			;	   16bit, 0x8000 = kein Wert vorhanden
			;Byte 6-7: Temperaturwert 2 im Zweierkomplement
			;	   16bit, 0x8000 = kein Wert vorhanden
grcdat:	.byte	4	;Grafikanzeige, Zwischenspeicher fr Datum/Zeit
			;komprimiertes Format, siehe xewbuf, Byte 0-3
grpsta:	.byte	3	;Zeiger (17 Bit) im externen EEPROM, enthlt
			;Adresse des ersten Wertes in der Grafik
grpend:	.byte	3	;Zeiger (17 Bit) im externen EEPROM, enthlt
			;Adresse des letzten Wertes in der Grafik
grptem:	.byte	3	;Zeiger (17 Bit) im externen EEPROM, enthlt
			;Adresse des aktuellen Wertes in der Grafik
grsca1:	.byte	1	;Grafikanzeige, Y-Skalierungsfaktor Diagramm 1
grsca2:	.byte	1	;Grafikanzeige, Y-Skalierungsfaktor Diagramm 2
grcen1:	.byte	2	;Grafikanzeige, Y-Mittelpunktwert f. Diagramm 1
grcen2:	.byte	2	;Grafikanzeige, Y-Mittelpunktwert f. Diagramm 2
grysva:	.byte	2	;Grafikanzeige, Y-Skale, aktueller Skalenwert
grlax1:	.byte	1	;Grafikanzeige, \ letzte Koordinat. von Pixel 1
grlay1:	.byte	1	;Grafikanzeige, / dient zum Verbinden der Pixel
grlax2:	.byte	1	;Grafikanzeige, \ letzte Koordinat. von Pixel 2
grlay2:	.byte	1	;Grafikanzeige, / dient zum Verbinden der Pixel
grrbu0:	.byte	44	;Grafikanzeige, Fadenkreuz, LCD-Daten Y
grrbu1:	.byte	28	;Grafikanzeige, Fadenkreuz, LCD-Daten X1
grrbu2:	.byte	28	;Grafikanzeige, Fadenkreuz, LCD-Daten X2
grry1p:	.byte	1	;Grafikanzeige, Fadenkreuz, Y-Pixelposition 1
grry2p:	.byte	1	;Grafikanzeige, Fadenkreuz, Y-Pixelposition 2
;
;
; ----------------------------------
; Programmbeginn, Interrupt-Vektoren
; ----------------------------------
;
.cseg
.org	0x0000
	jmp	start			;Programmstart
;
.org	OC1Aaddr			;Timer1 OutputCompareA
	jmp	t1comp			;wird alle 625s ausgelst
;
.org	URXC0addr			;USART0 RX Complete, Zeichen
	jmp	rxcpl0			;an RS232-0 empfangen
;
.org	UDRE0addr			;USART0 Data Register Empty,
	jmp	txdre0			;Zeichen an RS232-0 senden
;
.org	OC3Aaddr			;Timer3 OutputCompareA
	jmp	t3comp			;wird alle 20ms ausgelst
;
.org	URXC1addr			;USART1 RX Complete, Zeichen
	jmp	rxcpl1			;an RS232-1 empfangen
;
.org	SPMRaddr			;letzter INT-Tabelleneintrag
	reti				;ggf. Int-Routine beenden,
;
.org	SPMRaddr+2			;Hauptprogramm beginnt hinter
					;der Vektortabelle
;
; Stackpointer setzen, RAM lschen
;
start:	cli				;Interrupts sperren
	ldi	r16,low(ramend)		;Stackpointer L
	out	spl,r16			;setzen
	ldi	r16,high(ramend)	;Stackpointer H
	out	sph,r16			;setzen
	clr	r16
	ldi	xl,low(0x100)		;Beginn des RAM-Bereiches Low
	ldi	xh,high(0x100)		;Beginn des RAM-Bereiches High
ramclr:	st	x+,r16			;Speicherzelle lschen
	cpi	xl,low(ramend+1)	;RAM-Ende (Low) erreicht?
	brne	ramclr			;nein -> Schleife
	cpi	xh,high(ramend+1)	;RAM-Ende (High) erreicht?
	brne	ramclr			;nein -> Schleife
;
; I/O-Ports, ADC und TWI (I2C) initialisieren
;
	ldi	r16,0xff
	out	porta,r16		;PortA konfigurieren
	clr	r16
	out	ddra,r16		;PortA konfigurieren
	ldi	r16,0x02
	out	portb,r16		;PortB konfigurieren
	ldi	r16,0xf1
	out	ddrb,r16		;PortB konfigurieren
	ldi	r16,0xff
	out	portc,r16		;PortC konfigurieren
	ldi	r16,0x0f
	out	ddrc,r16		;PortC konfigurieren
	ldi	r16,0xc8
	out	portd,r16		;PortD konfigurieren
	ldi	r16,0xf8
	out	ddrd,r16		;PortD konfigurieren
	ldi	r16,0xfe
	out	porte,r16		;PortE konfigurieren
	ldi	r16,0x02
	out	ddre,r16		;PortE konfigurieren
	ldi	r16,0xfc
	sts	portf,r16		;PortF konfigurieren
	ldi	r16,0
	sts	ddrf,r16		;PortF konfigurieren
	ldi	r16,0x1f
	sts	portg,r16		;PortG konfigurieren
	ldi	r16,0
	sts	ddrg,r16		;PortG konfigurieren
;
	ldi	r16,0			;Referenz an AREF,
	out	admux,r16		;ADC0 auswhlen
	ldi	r16,(1<<aden|7<<adps0)	;ADC ein, Vorteiler= 128,
	out	adcsr,r16		;ADC-Frequenz= 86400 Hz
;
	ldi	r16,10			;kleinster mglicher Wert als
	sts	twbr,r16		;TWI-Master (307200 kHz)
	clr	r16
	sts	twsr,r16		;Vorteiler=1 setzen
	ldi	r16,1<<twen
	sts	twcr,r16		;TWI aktivieren
;
; Pointer der RS232-Schnittstellen und Puffer initialisieren
;
	ldi	xl,low(txbuf0)
	ldi	xh,high(txbuf0)		;Pointer fr RS232-1 S-Puffer
	sts	txp0wr,xl		;Schreibpointer setzen L
	sts	txp0wr+1,xh		;Schreibpointer setzen H
	sts	txp0rd,xl		;Lesepointer setzen L
	sts	txp0rd+1,xh		;Lesepointer setzen H
;
	ldi	xl,low(rxbuf0)
	ldi	xh,high(rxbuf0)		;Pointer fr RS232-0 E-Puffer
	sts	rxp0wr,xl		;Schreibpointer setzen L
	sts	rxp0wr+1,xh		;Schreibpointer setzen H
	sts	rxp0rd,xl		;Lesepointer setzen L
	sts	rxp0rd+1,xh		;Lesepointer setzen H
;
	ldi	xl,low(rxbuf1)
	ldi	xh,high(rxbuf1)		;Pointer fr RS232-1 E-Puffer
	sts	rxp1wr,xl		;Schreibpointer setzen L
	sts	rxp1wr+1,xh		;Schreibpointer setzen H
	sts	rxp1rd,xl		;Lesepointer setzen L
	sts	rxp1rd+1,xh		;Lesepointer setzen H
;
	ldi	xl,low(rxdat0)
	ldi	xh,high(rxdat0)		;Pointer fr RS232-0 D-Puffer
	sts	rxdap0,xl		;Position setzen L
	sts	rxdap0+1,xh		;Position setzen H
	ldi	xl,low(rxdat1)
	ldi	xh,high(rxdat1)		;Pointer fr RS232-1 D-Puffer
	sts	rxdap1,xl		;Position setzen L
	sts	rxdap1+1,xh		;Position setzen H
;
	ldi	xl,low(sigbuf)
	ldi	xh,high(sigbuf)		;Pointer fr Signalpuffer
	sts	sigbwr,xl		;Schreibpointer setzen L
	sts	sigbwr+1,xh		;Schreibpointer setzen H
	sts	sigbrd,xl		;Lesepointer setzen L
	sts	sigbrd+1,xh		;Lesepointer setzen H
;
; LCD einschalten und initialisieren
;
	call	lcdon			;LCD einschalten/initialisieren
;
; Sensor/Temperatur-Anzeige-Mapping-Tabelle aus EEPROM laden, Tempera-
; turflags initialisieren
;
	ldi	r19,16			;Mapping Tabelle, 16 Werte
	ldi	r16,low(etemap)
	ldi	r17,high(etemap)	;EEPROM-Adresse der Tabelle
	ldi	xl,low(temmap)
	ldi	xh,high(temmap)		;Zeiger auf RAM-Adresse
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
ldtmap:	call	eeread			;Byte aus EERPOM lesen
	bst	r18,7			;Inaktivitts-Flag holen
	ld	r20,y			;Temperatur-Flags holen
	bld	r20,7			;Flag fr Inaktivitt setzen
	st	y+,r20			;Flag wieder speichern
	andi	r18,0x0f		;Anzeigeposition filtern
	st	x+,r18			;und im RAM speichern
	dec	r19			;alle Werte kopiert?
	brne	ldtmap			;nein -> Schleife
;
; Sensor/Temperatur-Log-Mapping-Tabelle aus EEPROM laden
;
	ldi	r19,16			;Mapping Tabelle, 16 Werte
	ldi	r16,low(elomap)
	ldi	r17,high(elomap)	;EEPROM-Adresse der Tabelle
	ldi	xl,low(logmap)
	ldi	xh,high(logmap)		;Zeiger auf RAM-Adresse
ldlmap:	call	eeread			;Byte aus EERPOM lesen
	andi	r18,0x0f		;Listenposition filtern
	st	x+,r18			;und im RAM speichern
	dec	r19			;alle Werte kopiert?
	brne	ldlmap			;nein -> Schleife
;
; Temperaturwerte initialisieren (0x8000 setzen) und
; Timeout-Zhler auf Startwert setzen
;
	ldi	r17,16			;16 Temperaturwerte
	ldi	yl,low(temval)
	ldi	yh,high(temval)		;Zeiger auf Temperaturwerte
	ldi	xl,low(temtio)
	ldi	xh,high(temtio)		;Zeiger auf Timeout-Zhler
	ldi	r16,0x80		;Wert fr "Temp. ungltig"
	ldi	r18,timtmp		;Startwert fr Timeout-Zhler
initem:	st	x+,r18			;Timeout-Zhler setzen
	std	y+1,r16			;Wert in H-Byte speichern
	adiw	yl,2			;Zeiger auf nchsten Wert
	dec	r17			;alle Werte bearbeitet?
	brne	initem			;nein -> Schleife
	call	rsmin1			;Minimal-Werte lschen (Anz)
	call	rsmax1			;Maximal-Werte lschen (Anz)
	call	rsmin2			;Minimal-Werte lschen (Log)
	call	rsmax2			;Maximal-Werte lschen (Log)
;
; obere Temperaturlimits aus EERPOM laden
;
	ldi	r19,16*2		;16 Werte laden (16*2 Bytes)
	ldi	r16,low(etelih)
	ldi	r17,high(etelih)	;Zeiger auf EERPOM Daten
	ldi	xl,low(temlih)
	ldi	xh,high(temlih)		;Zeiger auf obere Limits
ldtlih:	call	eeread			;Byte aus EERPOM lesen
	st	x+,r18			;und im RAM speichern
	dec	r19			;alle Werte kopiert?
	brne	ldtlih			;nein -> Schleife
;
; untere Temperaturlimits aus EERPOM laden
;
	ldi	r19,16*2		;16 Werte laden (16*2 Bytes)
	ldi	r16,low(etelil)
	ldi	r17,high(etelil)	;Zeiger auf EERPOM Daten
	ldi	xl,low(temlil)
	ldi	xh,high(temlil)		;Zeiger auf untere Limits
ldtlil:	call	eeread			;Byte aus EERPOM lesen
	st	x+,r18			;und im RAM speichern
	dec	r19			;alle Werte kopiert?
	brne	ldtlil			;nein -> Schleife
;
; Alarm-Mapping-Tabelle aus EEPROM laden, Alarmflags initialisieren
;
	ldi	r19,4			;Mapping Tabelle, 4 Werte
	ldi	r16,low(ealmap)
	ldi	r17,high(ealmap)	;EEPROM-Adresse der Tabelle
	ldi	xl,low(almmap)
	ldi	xh,high(almmap)		;Zeiger auf RAM-Adresse
	ldi	yl,low(almflg)
	ldi	yh,high(almflg)		;Zeiger auf Alarmflags
ldamap:	call	eeread			;Byte aus EERPOM lesen
	bst	r18,7			;Inaktivitts-Flag holen
	ld	r20,y			;Alarmflags holen
	bld	r20,7			;Flag fr Inaktivitt setzen
	st	y+,r20			;Flag wieder speichern
	andi	r18,0x03		;Anzeigeposition filtern
	st	x+,r18			;und im RAM speichern
	dec	r19			;alle Werte kopiert?
	brne	ldamap			;nein -> Schleife
;
; Alarm-Invertierungs-Tabelle aus EEPROM laden
;
	ldi	r19,4			;Invertierungs-Tabelle, 4 Werte
	ldi	r16,low(ealinv)
	ldi	r17,high(ealinv)	;EEPROM-Adresse der Tabelle
	ldi	xl,low(alminv)
	ldi	xh,high(alminv)		;Zeiger auf RAM-Adresse
ldainv:	call	eeread			;Byte aus EERPOM lesen
	cpi	r18,2			;Wert>1?
	brcs	ldai10			;nein -> ok
	clr	r18			;sonst Invertierung aus
ldai10:	st	x+,r18			;und im RAM speichern
	dec	r19			;alle Werte kopiert?
	brne	ldainv			;nein -> Schleife
;
; Sensornummer fr Groanzeige aus EEPROM laden
;
	ldi	r16,low(ebigtm)
	ldi	r17,high(ebigtm)	;EEPROM-Adresse Groanzeige
	call	eeread			;Byte aus EERPOM lesen
	cpi	r18,16			;Sensornummer>15?
	brcs	ldbigt			;nein -> ok
	clr	r18			;sonst Sensor 1 whlen
ldbigt:	sts	bigtmp,r18		;und speichern
;
; Konfigurationsbyte 1 aus EEPROM laden
;
	ldi	r16,low(econf1)
	ldi	r17,high(econf1)	;EEPROM-Adresse Konfig-Byte 1
	call	eeread			;Byte aus EERPOM lesen
	sts	confg1,r18		;und speichern
;
; Konfigurationsbyte 2 aus EEPROM laden
;
	ldi	r16,low(econf2)
	ldi	r17,high(econf2)	;EEPROM-Adresse Konfig-Byte 2
	call	eeread			;Byte aus EERPOM lesen
	sts	confg2,r18		;und speichern
;
; Signalnummern-Tabelle aus EEPROM laden
;
	ldi	r19,8			;Signalnummern-Tabelle, 8 Werte
	ldi	r16,low(esignr)
	ldi	r17,high(esignr)	;EEPROM-Adresse der Tabelle
	ldi	xl,low(signum)
	ldi	xh,high(signum)		;Zeiger auf RAM-Adresse
ldsnum:	call	eeread			;Byte aus EERPOM lesen
	cpi	r18,16			;Signalnummer>15?
	brcs	ldsn10			;nein -> ok
	clr	r18			;sonst Signalnummer 0 whlen
ldsn10:	st	x+,r18			;und im RAM speichern
	dec	r19			;alle Werte kopiert?
	brne	ldsnum			;nein -> Schleife
;
; Signaldauer-Tabelle aus EEPROM laden
;
	ldi	r19,8			;Signaldauer-Tabelle, 8 Werte
	ldi	r16,low(esigdu)
	ldi	r17,high(esigdu)	;EEPROM-Adresse der Tabelle
	ldi	xl,low(sigdur)
	ldi	xh,high(sigdur)		;Zeiger auf RAM-Adresse
ldsdur:	call	eeread			;Byte aus EERPOM lesen
	cpi	r18,31			;Signaldauer>30?
	brcs	ldsd10			;nein -> ok
	clr	r18			;sonst Signalnummer 0 whlen
ldsd10:	st	x+,r18			;und im RAM speichern
	dec	r19			;alle Werte kopiert?
	brne	ldsdur			;nein -> Schleife
;
; Timer-Wert zur Erzeugung der Signalfrequenz aus EEPROM laden
;
	ldi	r16,low(esigtm)
	ldi	r17,high(esigtm)	;EEPROM-Adresse Signal-Frequenz
	call	eeread			;Byte aus EERPOM lesen
	cpi	r18,28			;Timerwert>=28?
	brcc	ldtim0			;ja -> ok
	ldi	r18,28			;sonst hchste Frequenz whlen
ldtim0:	out	ocr0,r18		;holen und Timer0 einstellen
;
; Lautstrke-Wert 1 (Tag) aus EEPROM laden und einstellen
;
	ldi	r16,low(esigv1)
	ldi	r17,high(esigv1)	;EEPROM-Adresse Lautstrke 1
	call	eeread			;Byte aus EERPOM lesen
	cpi	r18,16			;Timerwert>15?
	brcs	ldtv10			;nein -> ok
	ldi	r18,15			;sonst hchste Lautstrke
ldtv10:	sts	sigvo1,r18		;Lautstrke-Wert speichern
;
; Lautstrke-Wert 2 (Nacht) aus EEPROM laden
;
	ldi	r16,low(esigv2)
	ldi	r17,high(esigv2)	;EEPROM-Adresse Lautstrke 2
	call	eeread			;Byte aus EERPOM lesen
	cpi	r18,16			;Timerwert>15?
	brcs	ldtv30			;nein -> ok
	ldi	r18,15			;sonst hchste Lautstrke
ldtv30:	sts	sigvo2,r18		;Lautstrke-Wert speichern
;
; Nachtzeiten fr akustische Signalisierung aus EEPROM laden
;
	ldi	r19,4			;Nachtschaltung, 4 Werte
	ldi	r16,low(esigni)
	ldi	r17,high(esigni)	;EERPOM-Adresse Startzeit
	ldi	xl,low(signig)
	ldi	xh,high(signig)		;Zeiger auf RAM-Adresse
ldsign:	call	eeread			;Byte aus EERPOM lesen
	st	x+,r18			;und im RAM speichern
	dec	r19			;alle Werte kopiert?
	brne	ldsign			;nein -> Schleife
;
; Nachtzeiten fr die LCD-Hintergrundbeleuchtung aus EEPROM laden
;
	ldi	r19,4			;Nachtschaltung, 4 Werte
	ldi	r16,low(enigbl)
	ldi	r17,high(enigbl)	;EERPOM-Adresse Startzeit
	ldi	xl,low(nighbl)
	ldi	xh,high(nighbl)		;Zeiger auf RAM-Adresse
ldnibl:	call	eeread			;Byte aus EERPOM lesen
	st	x+,r18			;und im RAM speichern
	dec	r19			;alle Werte kopiert?
	brne	ldnibl			;nein -> Schleife
;
; Nachtzeiten fr das Bluetooth-Modul aus EEPROM laden
;
	ldi	r19,4			;Nachtschaltung, 4 Werte
	ldi	r16,low(enigbt)
	ldi	r17,high(enigbt)	;EERPOM-Adresse Startzeit
	ldi	xl,low(nighbt)
	ldi	xh,high(nighbt)		;Zeiger auf RAM-Adresse
ldnibt:	call	eeread			;Byte aus EERPOM lesen
	st	x+,r18			;und im RAM speichern
	dec	r19			;alle Werte kopiert?
	brne	ldnibt			;nein -> Schleife
;
; Auto-Reset-Zeiten fr Minimal- und Maximalwerte aus EEPROM laden
;
	ldi	r19,4			;Reset-Zeiten, 2x2 Werte
	ldi	r16,low(earesm)
	ldi	r17,high(earesm)	;EERPOM-Adresse Startzeit
	ldi	xl,low(aresmm)
	ldi	xh,high(aresmm)		;Zeiger auf RAM-Adresse
ldares:	call	eeread			;Byte aus EERPOM lesen
	st	x+,r18			;und im RAM speichern
	dec	r19			;alle Werte kopiert?
	brne	ldares			;nein -> Schleife
;
; Schwellwert fr die Steuerung der Hintergrundbeleuchtung lesen
;
	ldi	r16,low(elight)
	ldi	r17,high(elight)	;EEPROM-Adresse Lichtschwelle
	call	eeread			;Byte aus EERPOM lesen
	cpi	r18,5			;Schwellwert<5?
	brcc	ldli10			;nein -> weiter
	ldi	r18,5			;sonst Schwellwert=5 setzen
ldli10:	cpi	r18,251			;Schwellwert>250?
	brcs	ldli20			;nein -> weiter
	ldi	r18,250			;sonst Schwellwert=250 setzen
ldli20:	sts	light,r18		;Lichtschwellwert speichern
;
; Baudrate fr die PC-Verbindung (RS232-0) lesen
;
	ldi	r16,low(ebaud0)
	ldi	r17,high(ebaud0)	;EEPROM-Adr. Baudrate RS232-0
	call	eeread			;Byte aus EERPOM lesen
	cpi	r18,7			;gltiger Wert<7?
	brcs	ldba10			;ja -> weiter
	ldi	r18,4			;sonst Wert=4 setzen (115200)
ldba10:	sts	baud0,r18		;Baudrate RS232-0 speichern
;
; Meldungsfilter lesen
;
	ldi	r16,low(emsgfi)
	ldi	r17,high(emsgfi)	;EEPROM-Adresse Meldungsfilter
	call	eeread			;Byte aus EERPOM lesen
	cpi	r18,4			;gltiger Wert<4?
	brcs	ldmf10			;ja -> weiter
	clr	r18			;sonst Wert=0 setzen (kein Fi.)
ldmf10:	sts	msgfil,r18		;Meldungsfilter speichern
;
; Prfen, ob RTC vorhanden ist
;
	ldi	r16,rtcadr		;Slave-Adresse RTC + W-Bit
	mov	r15,r16			;kopieren
	call	i2ctio			;zum RTC senden (15ms)
	brcs	rtch10			;ACK empfangen? nein -> Fehler
	clr	r16			;Adresse=0
	call	i2csen			;senden
	brcs	rtch10			;ACK empfangen? nein -> Fehler
	ldi	r16,rtcadr+1		;Slave-Adresse RTC + R-Bit
	call	i2crst			;I2C Repeat Start
	brcs	rtch10			;ACK empfangen? nein -> Fehler
	sec				;kein ACK senden
	call	i2crcv			;Byte lesen
	andi	r16,0x7c		;Bits 7/1/0 lschen, Rest=0?
	brne	rtch10			;nein -> Fehler
	call	i2csto			;I2C Stop Condition
	rjmp	xechck			;weiter mit EEPROM-Test
;
rtch10:	call	i2csto			;I2C Stop Condition
	call	i2cerr			;RTC-Fehler setzen
;
; Prfen, ob beide externen EEPROM ICs vorhanden sind
;
xechck:	lds	r20,xflags		;verschiedene Flags holen
	ldi	r16,xeeadr		;Slave-Adresse EEPROM1 + W-Bit
	mov	r15,r16			;kopieren
	call	twitio			;zum Ext-EEPROM senden (15ms)
	brcs	xech10			;ACK empfangen? nein -> Fehler
	clr	r16			;Adresse H=0
	call	twisnd			;senden
	cpi	r17,0x28		;ACK empfangen?
	brne	xech10			;nein -> Fehler
	clr	r16			;Adresse L=0
	call	twisnd			;senden
	cpi	r17,0x28		;ACK empfangen?
	brne	xech10			;nein -> Fehler
	call	twista			;TWI Start Condition
	cpi	r17,0x10		;TWI Start ok?
	brne	xech10			;nein -> Fehler
	ldi	r16,xeeadr+1		;Slave-Adresse EEPROM1 + R-Bit
	call	twisnd			;senden
	cpi	r17,0x40		;ACK empfangen?
	brne	xech10			;nein -> Fehler
	clt				;kein ACK senden
	call	twircv			;Byte lesen
	cpi	r17,0x58		;Lesen ok?
	brne	xech10			;nein -> Fehler
	sbr	r20,0x40		;Flag fr Ext-EEPROM 1 setzen
;
xech10:	call	twisto			;TWI Stop Condition
	ldi	r16,xeeadr+2		;Slave-Adresse EEPROM2 + W-Bit
	mov	r15,r16			;kopieren
	call	twitio			;zum Ext-EEPROM senden (15ms)
	brcs	xech20			;ACK empfangen? nein -> Fehler
	clr	r16			;Adresse H=0
	call	twisnd			;senden
	cpi	r17,0x28		;ACK empfangen?
	brne	xech20			;nein -> Fehler
	clr	r16			;Adresse L=0
	call	twisnd			;senden
	cpi	r17,0x28		;ACK empfangen?
	brne	xech20			;nein -> Fehler
	call	twista			;TWI Start Condition
	cpi	r17,0x10		;TWI Start ok?
	brne	xech20			;nein -> Fehler
	ldi	r16,xeeadr+3		;Slave-Adresse EEPROM2 + R-Bit
	call	twisnd			;senden
	cpi	r17,0x40		;ACK empfangen?
	brne	xech20			;nein -> Fehler
	clt				;kein ACK senden
	call	twircv			;Byte lesen
	cpi	r17,0x58		;Lesen ok?
	brne	xech20			;nein -> Fehler
	sbr	r20,0x80		;Flag fr Ext-EEPROM 2 setzen
;
xech20:	call	twisto			;TWI Stop Condition
	sts	xflags,r20		;Flags wieder speichern
	andi	r20,0xc0		;Ext-EEPROM-Flags filtern
	cpi	r20,0xc0		;beide EEPROMs vorhanden?
	breq	xesrch			;ja -> weiter
	rjmp	xes300			;sonst Fehler
;
; Ermitteln der nchsten EEPROM-Schreibposition fr Temperaturdaten 1
;
xesrch:	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,4			;RTC-Fehler?
	brts	xes020			;ja -> Position suchen
	ldi	xl,low(xet1po)
	ldi	xh,high(xet1po)		;Zeigeradresse Temp-Daten 1
	ldi	r18,rtxt1p		;RTC-Adresse Pointer Temp-Dat1
	call	i2prcv			;Pointer aus RTC laden
	brcs	xes010			;Fehler? ja -> Status setzen
	cpi	r16,0x5a		;Flag ok?
	brne	xes020			;nein -> Position suchen
	rjmp	xes100			;sonst nchster Pointer
;
xes010:	call	i2cerr			;RTC-Fehlerstatus setzen
xes020:	lds	r16,xflag2		;verschiedene Flags 2 holen
	sbr	r16,0x02		;Pointer-Fehler-Flag 1 setzen
	sts	xflag2,r16		;Flags wieder speichern
	call	poierr			;Pointer-Fehlerstatus setzen
	ldi	r20,low(xt1sta)
	ldi	r21,high(xt1sta)
	ldi	r22,byte3(xt1sta)	;Anfangsadresse Temp1-Daten
	ldi	r23,low(xt1sta+xt1len-28)
	ldi	r24,high(xt1sta+xt1len-28);Adresse des letzt mglichen
	ldi	r25,byte3(xt1sta+xt1len-28);Datensatzbeginns laden
	sts	xetemp,r23
	sts	xetemp+1,r24		;Adresse in Zwischenspeicher
	sts	xetemp+2,r25		;ablegen
	ldi	r19,28			;Datensatz-Offset 28 Bytes
	call	xswpos			;Suche beginnen (Temp1-Daten)
	brcc	xes030			;Suche ok? ja -> weiter
	rjmp	xes300			;sonst Fehler
;
xes030:	sts	xet1po,r12
	sts	xet1po+1,r13		;Zeiger auf restaurierte
	sts	xet1po+2,r14		;Schreibposition setzen
	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,4			;RTC-Fehler?
	brts	xes100			;ja -> berspringen
	ldi	r18,rtxt1p		;RTC-Adresse Temperatur-Daten 1
	ldi	xl,low(xet1po)
	ldi	xh,high(xet1po)		;Zeiger auf Temp-Daten1 Pointer
	call	i2psen			;neuen Pointer in RTC schreiben
	brcc	xes100			;Fehler? nein -> weiter
	call	i2cerr			;sonst RTC-Fehlerstatus setzen
;
; Ermitteln der nchsten EEPROM-Schreibposition fr Temperaturdaten 2
;
xes100:	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,4			;RTC-Fehler?
	brts	xes120			;ja -> Position suchen
	ldi	xl,low(xet2po)
	ldi	xh,high(xet2po)		;Zeigeradresse Temp-Daten 2
	ldi	r18,rtxt2p		;RTC-Adresse Pointer Temp-Dat2
	call	i2prcv			;Pointer aus RTC laden
	brcs	xes110			;Fehler? ja -> Status setzen
	cpi	r16,0x5a		;Flag ok?
	brne	xes120			;nein -> Position suchen
	rjmp	xes200			;sonst nchster Pointer
;
xes110:	call	i2cerr			;RTC-Fehlerstatus setzen
xes120:	lds	r16,xflag2		;verschiedene Flags 2 holen
	sbr	r16,0x04		;Pointer-Fehler-Flag 2 setzen
	sts	xflag2,r16		;Flags wieder speichern
	call	poierr			;Pointer-Fehlerstatus setzen
	ldi	r20,low(xt2sta)
	ldi	r21,high(xt2sta)
	ldi	r22,byte3(xt2sta)	;Anfangsadresse Temp2-Daten
	ldi	r23,low(xt2sta+xt2len-76)
	ldi	r24,high(xt2sta+xt2len-76);Adresse des letzt mglichen
	ldi	r25,byte3(xt2sta+xt2len-76);Datensatzbeginns laden
	sts	xetemp,r23
	sts	xetemp+1,r24		;Adresse in Zwischenspeicher
	sts	xetemp+2,r25		;ablegen
	ldi	r19,76			;Datensatz-Offset 76 Bytes
	call	xswpos			;Suche beginnen (Temp2-Daten)
	brcc	xes130			;Suche ok? ja -> weiter
	rjmp	xes300			;sonst Fehler
;
xes130:	sts	xet2po,r12
	sts	xet2po+1,r13		;Zeiger auf restaurierte
	sts	xet2po+2,r14		;Schreibposition setzen
	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,4			;RTC-Fehler?
	brts	xes200			;ja -> berspringen
	ldi	r18,rtxt2p		;RTC-Adresse Temperatur-Daten 2
	ldi	xl,low(xet2po)
	ldi	xh,high(xet2po)		;Zeiger auf Temp-Daten2 Pointer
	call	i2psen			;neuen Pointer in RTC schreiben
	brcc	xes200			;Fehler? nein -> weiter
	call	i2cerr			;sonst RTC-Fehlerstatus setzen
;
; Ermitteln der nchsten EEPROM-Schreibposition fr Meldungsdaten
;
xes200:	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,4			;RTC-Fehler?
	brts	xes220			;ja -> Position suchen
	ldi	xl,low(xemspo)
	ldi	xh,high(xemspo)		;Zeigeradresse Meldungs-Daten
	ldi	r18,rtxmsp		;RTC-Adresse Pointer Meld-Daten
	call	i2prcv			;Pointer aus RTC laden
	brcs	xes210			;Fehler? ja -> Status setzen
	cpi	r16,0x5a		;Flag ok?
	brne	xes220			;nein -> Position suchen
	rjmp	initim			;sonst nchster Abschnitt
;
xes210:	call	i2cerr			;RTC-Fehlerstatus setzen
xes220:	lds	r16,xflag2		;verschiedene Flags 2 holen
	sbr	r16,0x08		;Pointer-Fehler-Flag 3 setzen
	sts	xflag2,r16		;Flags wieder speichern
	call	poierr			;Pointer-Fehlerstatus setzen
	ldi	r20,low(xmssta)
	ldi	r21,high(xmssta)
	ldi	r22,byte3(xmssta)	;Anfangsadresse Meldungs-Daten
	ldi	r23,low(xmssta+xmslen-6)
	ldi	r24,high(xmssta+xmslen-6);Adresse des letzt mglichen
	ldi	r25,byte3(xmssta+xmslen-6);Datensatzbeginns laden
	sts	xetemp,r23
	sts	xetemp+1,r24		;Adresse in Zwischenspeicher
	sts	xetemp+2,r25		;ablegen
	ldi	r19,6			;Datensatz-Offset 6 Bytes
	call	xswpos			;Suche beginnen (Meldungsdaten)
	brcs	xes300			;Suche ok? nein -> Fehler
;
	sts	xemspo,r12
	sts	xemspo+1,r13		;Zeiger auf restaurierte
	sts	xemspo+2,r14		;Schreibposition setzen
	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,4			;RTC-Fehler?
	brts	initim			;ja -> berspringen
	ldi	r18,rtxmsp		;RTC-Adresse Pointer Meld-Daten
	ldi	xl,low(xemspo)
	ldi	xh,high(xemspo)		;Zeigeradresse Meldungsdaten
	call	i2psen			;neuen Pointer in RTC schreiben
	brcc	initim			;Fehler? nein -> weiter
	call	i2cerr			;sonst RTC-Fehlerstatus setzen
	rjmp	initim			;weiter im nchsten Abschnitt
;
xes300:	lds	r16,errflg		;Fehler-Flags holen
	sbr	r16,0x08		;Speicherfehler-Flag setzen
	sts	errflg,r16		;Flags wieder speichern
;
; Zeit aus RTC lesen und setzen
;
initim:	lds	r16,errflg		;Fehler-Flags holen
	sbr	r16,0x01		;zunchst Uhren-Fehler setzen
	sts	errflg,r16		;Flags wieder speichern
	bst	r16,4			;RTC-Fehler?
	brts	initrs			;ja -> berspringen
	call	tirrtc			;sonst Zeit aus RTC holen
	cli				;Interrupts wieder sperren
	brcs	init10			;Fehler? ja -> weiter
	cpi	r16,0x5a		;Flag-Byte=0x5a?
	brne	initrs			;nein -> berspringen
	lds	r16,errflg		;sonst Fehler-Flags holen
	cbr	r16,0x01		;Uhren-Fehler wieder lschen
	sts	errflg,r16		;Flags wieder speichern
	rjmp	initrs			;weiter im nchsten Abschnitt
;
init10:	call	i2cerr			;RTC-Fehler setzen
;
; serielle Schnittstellen initialisieren
;
initrs:	ldi	r16,low((clock/(16*baud1))-1)  ;RS232-1 auf 9600 Baud
	ldi	r17,high((clock/(16*baud1))-1) ;einstellen
	sts	ubrr1l,r16		;Baudratenwert L setzen
	sts	ubrr1h,r17		;Baudratenwert H setzen
	ldi	r16,(3<<ucsz10)		;Datenformat 8N1
	sts	ucsr1c,r16		;Datenformat setzen
	ldi	r16,(1<<rxcie1)|(1<<rxen1);RX mit Interrupt aktivieren
	sts	ucsr1b,r16		;Einstellungen setzen
;
	call	setbd0			;Baudrate fr RS232-0 setzen
	ldi	r16,(3<<ucsz00)		;Datenformat 8N1
	sts	ucsr0c,r16		;Datenformat setzen
	ldi	r16,(1<<rxcie0)|(1<<rxen0)|(1<<txen0); RX und TX akti-
	out	ucsr0b,r16		;vieren, RX Int einschalten
;
; Timer initialisieren
;
	ldi	r16,(1<<wgm01|3<<cs00)	;Timer0
	out	tccr0,r16		;Mode Normal+CTC, Vorteiler=32
;
	ldi	r16,low((clock/cltim1)-1)
	ldi	r17,high((clock/cltim1)-1);Teilerwert fr Timer1a laden
	out	ocr1ah,r17
	out	ocr1al,r16		;und setzen (H zuerst)
	clr	r16
	ldi	r17,1			;Timer1 auf 256 voreinstellen
	out	tcnt1h,r17		;damit der Interrupt von Timer1
	out	tcnt1l,r16		;vor Timer3 ausgelst wird
	ldi	r16,(1<<com1b1)|(3<<wgm30);Timer1 Fast PWM, OC1B aktiv
	out	tccr1a,r16		;setzen
	ldi	r16,(3<<wgm12|1<<cs10)	;Timer1 Fast PWM, Vorteiler=1
	out	tccr1b,r16		;setzen
	ldi	r16,1<<ocie1a		;Interrupt bei OutputCompareA
	out	timsk,r16		;fr Timer1 aktivieren
;
	ldi	r16,(1<<wgm20)|(1<<com21)|(3<<cs20); Timer2
	out	tccr2,r16		;PhaseCorrect PWM, Vorteiler=64
;
	ldi	r16,low((clock/(8*cltim3))-1)  ;Teilerwert fr Timer3a
	ldi	r17,high((clock/(8*cltim3))-1) ;laden
	sts	ocr3ah,r17
	sts	ocr3al,r16		;und setzen (H zuerst)
	clr	r16			;Timer3 Mode Normal+CTC, keine
	sts	tccr3a,r16		;OC-Ausgnge aktivieren
	ldi	r16,(1<<wgm32|1<<cs31)	;Timer3 Mode CTC, Vorteiler=8
	sts	tccr3b,r16		;setzen
	ldi	r16,1<<ocie3a		;Interrupt bei OutputCompareA
	sts	etimsk,r16		;fr Timer3 aktivieren
;
	sei				;Interrupts aktivieren
;
; Prfen, ob ein aktuelles Backup fr Min-, Max- und Mittelwerte im
; RTC-RAM vorhanden ist
;
	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,4			;RTC-Fehler?
	brts	chkcrc			;ja -> berspringen
;
	ldi	r16,rtcadr		;I2C Basisadresse RTC + Write
	call	i2csta			;I2C Start Condition + Adresse
	brcs	rdberr			;ACK empfangen? nein -> Fehler
	ldi	r16,rtbdat		;Adresse des Backup-Datums
	call	i2csen			;senden
	brcs	rdberr			;ACK empfangen? nein -> Fehler
	ldi	r16,rtcadr+1		;I2C Basisadresse RTC + Read
	call	i2crst			;I2C Repeat Start Condition
	brcs	rdberr			;ACK empfangen? nein -> Fehler
;
	clc				;ACK zum Slave senden
	call	i2crcv			;Byte empfangen
	mov	r20,r16			;Kalendertag Einer kopieren
	clc				;ACK zum Slave senden
	call	i2crcv			;Byte empfangen
	mov	r21,r16			;Kalendertag Zehner kopieren
	clc				;ACK zum Slave senden
	call	i2crcv			;Byte empfangen
	mov	r22,r16			;Monat Einer kopieren
	clc				;ACK zum Slave senden
	call	i2crcv			;Byte empfangen
	mov	r23,r16			;Monat Zehner kopieren
	sec				;kein ACK zum Slave senden
	call	i2crcv			;Byte empfangen
	call	i2csto			;I2C Stop Condition
;
	cpi	r16,0x5a		;Backup-Flag ok?
	brne	chkcrc			;nein -> Restore berspringen
	lds	r16,dayl		;Kalendertag Einer holen
	cp	r16,r20			;Kalendertag Einer ok?
	brne	chkcrc			;nein -> Restore berspringen
	lds	r16,dayh		;Kalendertag Zehner holen
	cp	r16,r21			;Kalendertag Einer ok?
	brne	chkcrc			;nein -> Restore berspringen
	lds	r16,monthl		;Monat Einer holen
	cp	r16,r22			;Monat Einer ok?
	brne	chkcrc			;nein -> Restore berspringen
	lds	r16,monthh		;Monat Zehner holen
	cp	r16,r23			;Monat Zehner ok?
	brne	chkcrc			;nein -> Restore berspringen
	call	i2rest			;Werte aus RTC-RAM zurckholen
	brcc	chkcrc			;Fehler? nein -> weiter
;
rdberr:	call	i2cerr			;RTC-Fehler setzen
;
; EEPROM CRC-Wert prfen und gegebenenfalls Systemfehler aktivieren
;
chkcrc:	call	calcrc			;EEPROM CRC-Berechnung
	call	eeread			;gespeicherten CRC Wert holen
	cp	r18,r20			;CRC ok?
	breq	chkerr			;ja -> weiter
	lds	r16,errflg		;sonst Fehler-Flags holen
	sbr	r16,0x04		;EEPROM-CRC-Fehler setzen
	sts	errflg,r16		;Fehler-Flags wieder speichern
	lds	r18,signum+6		;Signalnummer fr Fehlermeldun.
	lds	r19,sigdur+6		;Signaldauer fr Fehlermeldung.
	call	signal			;Signal ausgeben
;
; Fehler-Flag prfen und Meldungen ins Log schreiben
;
chkerr:	lds	r24,errflg		;Fehler-Flags holen
	bst	r24,3			;Speicherfehler?
	brts	initmn			;ja -> keine Meldungen schreib.
	bst	r24,2			;Konfigurations-Fehler?
	brtc	chke10			;nein -> weiter testen
	ldi	r19,0x20		;Meldung "Konfigurat.-Fehler"
	call	wrimsg			;Fehlermeldung schreiben
chke10:	bst	r24,4			;RTC-Fehler?
	brtc	chke20			;nein -> weiter testen
	ldi	r19,0x40		;Meldung "RTC-Fehler" setzen
	call	wrimsg			;Fehlermeldung schreiben
chke20:	bst	r24,5			;Pointer-Fehler?
	brtc	initmn			;nein -> weiter
	ldi	r19,0x50		;Meldung "Pointer-Fehler"
	call	wrimsg			;Fehlermeldung schreiben
; 
; Hauptschleife initialisieren
;
initmn:	ldi	r16,timmss		;Meldungs-Timeout zunchst
	sts	msgtio,r16		;auf 45s setzen
	lds	r16,minutl		;Minuten Einer holen
	sts	minfl1,r16		;als Minutenflag1 voreinstellen
	sts	minfl3,r16		;als Minutenflag3 voreinstellen
	ser	r16			;0xff laden
	sts	minfl2,r16		;als Minutenflag2 voreinstellen
	lds	r16,hourl		;Stunden Einer holen
	sts	houflg,r16		;als Stundenflag voreinstellen
	clr	r17			;zunchst Lautstrke= 0 setzen
	call	sicalc			;PWM berechnen, Lautst. setzen
	ldi	r16,0xff		;letzte Fadenkreuzposition
	sts	grlacr,r16		;lschen
;
;
; ----------------------------
; Beginn Hauptprogrammschleife
; ----------------------------
;
; Anzeigesteuerung
;
main:	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,1			;Anzeige neu ausgeben?
	brtc	m12000			;nein -> berspringen
	sbr	r16,0x0c		;Flags fr Info/Menzeile setz.
	sts	dflags,r16		;und Flags wieder speichern
;
; Basisanzeige neu ausgeben: LCD lschen, Rahmen zeichnen
;
	call	clrlcd			;LCD lschen
	clr	txflag			;Textflags lschen
	ldi	r18,0			;X=0
m11100:	ldi	r19,0|0x80		;Y=0, oberer Rahmen
	call	pixlcd			;waagerechte Linie zeichnen
	ldi	r19,14|0x80		;Y=14, Trennlinie Kopfzeile
	call	pixlcd			;waagerechte Linie zeichnen
	ldi	r19,112|0x80		;Y=112, Trennlinie Fuzeile
	call	pixlcd			;waagerechte Linie zeichnen
	ldi	r19,127|0x80		;Y=127, unterer Rahmen
	call	pixlcd			;waagerechte Linie zeichnen
	inc	r18			;X-Koordinate erhhen
	cpi	r18,240			;alle Pixel ausgegeben?
	brne	m11100			;nein -> Schleife
;
	ldi	r19,0|0x80		;Y=0
m11110:	ldi	r18,0			;X=0, linker Rahmen
	call	pixlcd			;senkrechte Linie zeichnen
	ldi	r18,239			;X=239, rechter Rahmen
	call	pixlcd			;senkrechte Linie zeichnen
	inc	r19			;Y-Koordinate erhhen
	brne	m11110			;nein -> Schleife
;
	ldi	r19,1|0x80		;Y=1
m11120:	ldi	r18,174			;X=174, Trenlinie fr Uhrzeit
	call	pixlcd			;Linie zeichnen
	ldi	r18,205			;X=205, Trenlinie fr Antenne
	call	pixlcd			;Linie zeichnen
	ldi	r18,222			;X=222, Trennlinie fr Lautspr.
	call	pixlcd			;Linie zeichnen
	inc	r19			;Y-Koordinate erhhen
	cpi	r19,14|0x80		;alle Pixel ausgegeben?
	brne	m11120			;nein -> Schleife
;
	ldi	r21,2			;Y=2
	ldi	r20,224			;X=224
	ldi	r22,0x9d		;Lautsprecher-Symbol
	call	asclcd			;ausgeben
;
; Infozeile ausgeben: DCF77- und Lautsprecher-Status
;
m12000:	clr	txflag			;Textflags lschen
	lds	r25,dflags		;Anzeige-Flags holen
	bst	r25,2			;Neuausgabe der Infozeile?
	brts	m12010			;ja -> bearbeiten
	rjmp	m13000			;sonst berspringen
;
m12010:	cbr	r25,0x04		;Infozeilen-Flag lschen
	sts	dflags,r25		;und Flags wieder speichern
;
	ldi	r20,206			;X=206
	ldi	r21,2			;Y=2
	ldi	r22,0x9c		;Antennen-Symbol
	in	r16,portb		;PortB-Status holen
	bst	r16,5			;Bluetooth aktiv?
	brtc	m12020			;nein -> weiter
	ldi	r22,0x96		;sonst Antennensymbol+Bluetooth
m12020:	call	asclcd			;ausgeben
;
	ldi	r20,214			;X=214
	lds	r16,pccmd		;laufendes PC-Kommando holen
	tst	r16			;PC-bertragung aktiv?
	breq	m12030			;nein -> weiter
	ldi	r22,0x9b		;sonst Blitzsymbol laden
	rjmp	m12040
;
m12030:	bst	r25,4			;DCF77-Signalstatus holen
	ldi	r22,0x9e		;Zeichen fr 'aus' laden
	bld	r22,0			;DCF77-Signalstatus einsetzen
m12040:	call	asclcd			;DCF77-Signalstatus ausgeben
;
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,7			;Status Abwesend?
	brtc	m12050			;nein -> weiter
	ldi	r22,0xad		;sonst Schloss-Symbol laden
	rjmp	m12060			;ausgeben
m12050:	bst	r25,5			;Lautsprecher-Status holen
	ldi	r22,0x9e		;Zeichen fr 'aus' laden
	bld	r22,0			;Lautsprecher-Status einsetzen
m12060:	ldi	r20,231			;X=231
	call	asclcd			;Lautsprecher-Status ausgeben
;
; Infozeile ausgeben: Uhrzeit
;
	ldi	r20,176			;X=176
	ldi	r21,2			;Y=2, Position Uhrzeit
	lds	r22,hourh		;Stunden Zehner holen
	ori	r22,0x30		;in ASCII wandeln
	cpi	r22,0x30		;Stunden Zehner='0'?
	brne	m12100			;nein -> ausgeben
	ldi	r22,' '			;sonst Leerzeichen
m12100:	call	asclcd			;ausgeben
	lds	r22,hourl		;Stunden Einer holen
	ori	r22,0x30		;in ASCII wandeln
	call	asclcd			;ausgeben
	ldi	r22,':'			;Doppelpunkt als Trennzeichen
	call	asclcd			;ausgeben
	lds	r22,minuth		;Minuten Zehner holen
	ori	r22,0x30		;in ASCII wandeln
	call	asclcd			;ausgeben
	lds	r22,minutl		;Minuten Zehner holen
	ori	r22,0x30		;in ASCII wandeln
	call	asclcd			;ausgeben
;
; Infozeile ausgeben: Fehlertext
;
	lds	r16,mennum		;aktuelle Mennummer holen
	tst	r16			;Hauptmen?
	breq	m12110			;ja -> Fehler/Datum ausgeben
	rjmp	m12300			;sonst berspringen
;
m12110:	ldi	r20,2			;X=2, Y=2, Position fr Datum
	ldi	r21,2			;bzw. Fehlertext
	lds	r16,errflg		;Fehler-Flags holen
	lds	r17,errmsk		;Fehlermaske holen
	tst	r17			;Maske vorhanden?
	breq	m12120			;nein -> berspringen
	and	r16,r17			;sonst Maske ber Alarme legen
m12120:	ldi	r22,0xf5		;Fehlertext "Pointer-Fehler"
	bst	r16,5			;Pointer-Flag?
	brts	m12130			;ja -> ausgeben
	ldi	r22,0xf4		;Fehlertext "Echtzeituhr-Fehl."
	bst	r16,4			;Echtzeituhrfehler-Flag?
	brts	m12130			;ja -> ausgeben
	ldi	r22,0xf3		;Fehlertext "Datenspeicherfeh."
	bst	r16,3			;Datenspeicherfehler-Flag?
	brts	m12130			;ja -> ausgeben
	ldi	r22,0xf2		;Fehlertext "Konfigurationsfe."
	bst	r16,2			;Konfigurationsfehler-Flag?
	brts	m12130			;ja -> ausgeben
	ldi	r22,0xf1		;Fehlertext "kein Datenempfang"
	bst	r16,1			;Datenempfangsfehler-Flag?
	brts	m12130			;ja -> ausgeben
	ldi	r22,0xf0		;Fehlertext "Uhr nicht gest."
	bst	r16,0			;Uhreneinstellungs-Flag?
	brtc	m12200			;nein -> Datum ausgeben
;
m12130:	ldi	r17,0x01		;Inversschrift einstellen
	mov	txflag,r17		;Textflag setzen
	call	strlcd			;Fehlertext ausgeben
	rjmp	m12240			;Zeile auffllen
;
; Infozeile ausgeben: Wochentag und Datum
;
m12200:	bst	r16,7			;Fehlererinnerung gesetzt?
	brtc	m12210			;nein -> weiter
	ldi	r17,0x02		;sonst Unterstreichung ein-
	mov	txflag,r17		;stellen und Textflag setzen
m12210:	lds	r22,dayw		;Wochentag holen
	call	strlcd			;Wochtag ausgeben
	ldi	r22,','			;Komma
	call	asclcd			;ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
	lds	r22,dayh		;Kalendertag Zehner holen
	cpi	r22,0			;Kalendertag Zehner=0?
	breq	m12220			;ja -> keine Ausgabe
	ori	r22,0x30		;sonst in ASCII wandeln
	call	asclcd			;ausgeben
m12220:	lds	r22,dayl		;Kalendertag Einer holen
	ori	r22,0x30		;in ASCII wandeln
	call	asclcd			;ausgeben
	ldi	r22,'.'			;Punkt
	call	asclcd			;ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
	lds	r23,monthh		;Monat Zehner holen
	tst	r23			;Monat Zehner=0?
	breq	m12230			;ja -> weiter
	ldi	r23,10			;sonst auf 10 setzen
m12230:	lds	r22,monthl		;Monat Einer holen
	add	r22,r23			;Zehner addieren
	dec	r22			;korrigieren
	ori	r22,0x10		;Textnummer generieren
	call	strlcd			;Monatsname ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
	ldi	r22,year3		;Jahr Tausender holen
	call	asclcd			;ausgeben
	ldi	r22,year2		;Jahr Hunderter holen
	call	asclcd			;ausgeben
	lds	r22,yearh		;Jahr Zehner holen
	ori	r22,0x30		;in ASCII wandeln
	call	asclcd			;ausgeben
	lds	r22,yearl		;Jahr Einer holen
	ori	r22,0x30		;in ASCII wandeln
	call	asclcd			;ausgeben
	ldi	r22,0x7f		;Endezeichen
	call	asclcd			;ausgeben
;
m12240:	clr	txflag			;Textflag lschen
	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,174			;Endwert X Datumbereich setzen
	ldi	r22,11			;zum Endwert Y 11 Pixel
	add	r21,r22			;addieren
	call	clrwnd			;Bereich lschen
	rjmp	m13000
;
; Infozeile ausgeben: Men-berschrift
;
m12300:	call	men1rd			;Mentabellenheader lesen
	clr	txflag			;Textflag lschen
	ldi	r20,2			;X=2, Y=2, Position fr
	ldi	r21,2			;Men-berschrift
	ldi	r22,0xb3		;Hinweis-Dreieck (Pfeil rechts)
	call	asclcd			;ausgeben
	ldi	r22,0x7f		;Ende-Zeichen (2 Pixel Abstand
	call	asclcd			;zum Text)
	mov	r22,r12			;Mentext-Nummer holen
	cpi	r22,10			;String-Nr=10 (Grafik-Preset)?
	brne	m12310			;nein -> normal ausgeben
;
	ldi	r22,0xa0		;sonst Zeichenkette "Preset"
	call	strlcd			;ausgeben
	lds	r18,grpres		;Grafik-Preset-Nummer holen
	inc	r18			;korrigieren
	call	bytasc			;in ASCII wandeln
	mov	r25,r16			;Einer sichern
	mov	r22,r17			;Zehner holen
	call	asclcd			;Zeichen ausgeben
	mov	r22,r25			;gesicherte Einer holen
	call	asclcd			;Zeichen ausgeben
	ldi	r22,':'			;Trennzeichen laden
	call	asclcd			;Zeichen ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen laden
	call	asclcd			;Zeichen ausgeben
	ldi	r16,low(egrprt)
	ldi	r17,high(egrprt)	;Adresse Preset-Texte im EEPROM
	ldi	r19,16			;Textlnge 16 Zeichen
	lds	r18,grpres		;Grafik-Preset-Nummer holen
	mul	r19,r18			;Offset fr Textposition berech.
	add	r16,r0
	adc	r17,r1			;EEPROM-Adresse berechnen
	call	eeplcd			;EEPROM-Text ausgeben
	rjmp	m12240
;
m12310:	call	strlcd			;Mentext ausgeben
	lds	r16,menext		;Men-Erweiterung laden
	tst	r16			;zustzlichen Text ausgeben?
	breq	m12240			;nein -> Zeile auffllen
	ldi	r22,0xa0		;sonst schmales Leerzeichen
	call	asclcd			;ausgeben
	ldi	r22,'-'			;Trennzeichen
	call	asclcd			;ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
	lds	r22,menext		;Men-Erweiterung laden
	call	strlcd			;zustzlichen Text ausgeben
	rjmp	m12240			;Zeile auffllen
;
; Menzeile ausgeben: Text fr Taster 2 (links)
;
m13000:	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,3			;Menzeile neu ausgeben?
	brts	m13010			;ja -> bearbeiten
	rjmp	m14000			;sonst berspringen
;
m13010:	cbr	r16,0x08		;sonst Neuausgabe-Flag lschen
	sts	dflags,r16		;und Flags wieder speichern
;
	clr	txflag			;Textflag lschen
	ldi	r20,2			;X=2, Y=115, Position fr
	ldi	r21,115			;Menzeile, linker Rand
	lds	r16,mennum		;aktuelle Mennummer holen
	tst	r16			;Hauptmen (0)?
	breq	m13020			;ja -> bearbeiten
	ldi	r22,0x1f		;sonst Text "Zurck"
	rjmp	m13040			;ausgeben
;
m13020:	lds	r16,xflags		;verschiedene Flags holen
	bst	r16,0			;sind Signale aktiv?
	brts	m13030			;ja -> Menzeile setzen
	bst	r16,1			;sind Erinnerungen gesetzt?
	brtc	m13050			;nein -> Menzeile setzen
	ldi	r22,0x1d		;sonst Text "Erinnerungen
	rjmp	m13040			;lschen" ausgeben
m13030:	ldi	r22,0x1e		;Text "Signal ausschalten"
m13040:	call	strlcd			;ausgeben
m13050:	mov	r13,r20			;aktuelle X-Position speichern
;
; Menzeile ausgeben: Text fr Taster 1 (rechts)
;
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brtc	m13060			;nein -> normale Men-Option
	rjmp	m13200			;sonst als Encoderwert ausgeben
;
m13060:	call	men1rd			;sonst Mentabellenheader lesen
	call	men2rd			;Mentabellenoption lesen
	lds	r10,menopt		;aktuelle Menoption holen
	elpm	r12,z+			;String-Nr fr Mentext holen
	mov	r22,r12			;String-Nr kopieren
	cpi	r22,10			;String-Nr=10 (Grafik-Preset)?
	brne	m13100			;nein -> normal ausgeben
;
; Menzeile ausgeben: Grafik-Preset Namen ausgeben
;
	ldi	r16,low(egrprt)
	ldi	r17,high(egrprt)	;Adresse Preset-Texte im EEPROM
	ldi	r19,16			;Textlnge 16 Zeichen
	mul	r19,r10			;Offset fr Textposition
	add	r16,r0
	adc	r17,r1			;EEPROM-Adresse berechnen
	movw	r8,r16			;EEPROM-Adresse sichern
	call	eeplen			;Lnge des Textes ermitteln
	ldi	r18,19			;Lnge fr Zusatz "01: "
	add	r18,r20			;Gesamtlnge ermitteln
;
	ldi	r20,222			;Endposition der Menzeile
	sub	r20,r18			;Text-Anfang berechnen, bis
	ldi	r21,126			;dahin alle Pixel lschen
	mov	r18,r13			;gesicherte Anfangsposit. holen
	ldi	r19,115			;Y-Anfangsposition setzen
	call	clrwnd			;Text-Zwischenraum lschen
	ldi	r21,115			;Y-Position fr Text setzen
;
	mov	r18,r10			;Men-Option holen
	inc	r18			;korrigieren
	call	bytasc			;in ASCII wandeln
	mov	r25,r16			;Einer sichern
	mov	r22,r17			;Zehner holen
	call	asclcd			;Zeichen ausgeben
	mov	r22,r25			;gesicherte Einer holen
	call	asclcd			;Zeichen ausgeben
	ldi	r22,':'			;Trennzeichen laden
	call	asclcd			;Zeichen ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen laden
	call	asclcd			;Zeichen ausgeben
	movw	r16,r8			;gesicherte EEPROM-Adresse holen
	ldi	r19,16			;Preset-Textlnge im EEPROM
	call	eeplcd			;EEPROM-Text ausgeben
	rjmp	m13110
;
; Menzeile ausgeben: Men-Option ausgeben
;
m13100:	call	strlen			;Stringlnge ermitteln
	ldi	r20,222			;Endposition der Menzeile
	sub	r20,r18			;Text-Anfang berechnen, bis
	ldi	r21,126			;dahin alle Pixel lschen
	mov	r18,r13			;gesicherte Anfangsposit. holen
	ldi	r19,115			;Y-Anfangsposition setzen
	call	clrwnd			;Text-Zwischenraum lschen
	ldi	r21,115			;Y-Position fr Text setzen
	mov	r22,r12			;gesicherte String-Nr holen
	call	strlcd			;String ausgeben
;
m13110:	ldi	r20,224			;X-Position fr Pfeil links
	ldi	r22,0xb0		;Pfeil links hohl laden
	tst	r10			;erste Option (0)?
	breq	m13120			;ja -> berspringen
	ldi	r22,0xb1		;sonst Pfeil links voll laden
m13120:	call	asclcd			;ausgeben
	ldi	r20,231			;X-Position fr Pfeil rechts
	ldi	r22,0xb2		;Pfeil rechts hohl laden
	mov	r16,r10			;aktuelle Menoption holen
	inc	r16			;um 1 vermindern
	cp	r16,r11			;letzte Option?
	breq	m13130			;ja -> berspringen
	ldi	r22,0xb3		;sonst Pfeil rechts voll laden
m13130:	call	asclcd			;ausgeben
	rjmp	m14000
;
; Menzeile ausgeben: Encoderwert 3- oder 4-stellig dezimal ausgeben
;
m13200:	bst	r16,1			;Ausgabe 3-stellig dezimal?
	brts	m13230			;ja -> bearbeiten
	bst	r16,4			;Ausgabe 4-stellig dezimal?
	brts	m13210			;ja -> bearbeiten
	rjmp	m13600			;sonst weiter testen
;
m13210:	lds	r18,dimode		;Anzeigemodus holen
	cpi	r18,24			;Modus Grafik/Fadenkreuz?
	brne	m13220			;nein -> Wert normal ausgeben
	rjmp	m13400			;sonst Temperaturwerte ausgeben
;
m13220:	lds	r17,encvah		;Encoderwert H holen
	rjmp	m13240
m13230:	clr	r17			;H-Byte lschen
m13240:	lds	r16,encval		;Encoderwert holen
	call	bytas4			;Wert in 4 Byte ASCII wandeln
	clr	r18			;Anfangswert fr Stringlnge
	ser	r19			;Flag fr Vornullenlschung
;
	mov	r16,r4			;erste Stelle holen
	cpi	r16,0x30		;erste Stelle= '0'?
	breq	m13250			;ja -> weiter
	clr	r19			;sonst Vornull-Flag lschen
	call	asclen			;Zeichenlnge ermitteln
	add	r18,r16			;und zur Gesamtlnge addieren
m13250:	mov	r16,r5			;zweite Stelle holen
	tst	r19			;Vornullen lschen?
	breq	m13260			;nein -> Nulltest berspringen
	cpi	r16,0x30		;zweite Stelle= '0'?
	breq	m13270			;ja -> berspringen
	clr	r19			;sonst Vornull-Flag lschen
m13260:	call	asclen			;Zeichenlnge ermitteln
	add	r18,r16			;und zur Gesamtlnge addieren
m13270:	mov	r16,r6			;dritte Stelle holen
	tst	r19			;Vornullen lschen?
	breq	m13280			;nein -> Nulltest berspringen
	cpi	r16,0x30		;dritte Stelle= '0'?
	breq	m13290			;ja -> berspringen
	clr	r19			;sonst Vornull-Flag lschen
m13280:	call	asclen			;Zeichenlnge ermitteln
	add	r18,r16			;und zur Gesamtlnge addieren
m13290:	mov	r16,r7			;vierte Stelle holen
	call	asclen			;Zeichenlnge ermitteln
	add	r18,r16			;und zur Gesamtlnge addieren
;
	ldi	r20,222			;Endposition der Menzeile
	sub	r20,r18			;Text-Anfang berechnen, bis
	ldi	r21,126			;dahin alle Pixel lschen
	mov	r18,r13			;gesicherte Anfangsposit. holen
	ldi	r19,115			;Y-Anfangsposition setzen
	call	clrwnd			;Text-Zwischenraum lschen
	ldi	r21,115			;Y-Position fr Textausgabe
;
	ser	r25			;Flag fr Vornullenlschung
	mov	r22,r4			;erste Stelle holen
	cpi	r22,0x30		;erste Stelle= '0'?
	breq	m13300			;ja -> weiter
	clr	r25			;sonst Vornull-Flag lschen
	call	asclcd			;erste Stelle ausgeben
m13300:	mov	r22,r5			;zweite Stelle holen
	tst	r25			;Vornullen lschen?
	breq	m13310			;nein -> Nulltest berspringen
	cpi	r22,0x30		;zweite Stelle= '0'?
	breq	m13320			;ja -> berspringen
	clr	r25			;sonst Vornull-Flag lschen
m13310:	call	asclcd			;zweite Stelle ausgeben
m13320:	mov	r22,r6			;dritte Stelle holen
	tst	r25			;Vornullen lschen?
	breq	m13330			;nein -> Nulltest berspringen
	cpi	r22,0x30		;dritte Stelle= '0'?
	breq	m13340			;ja -> berspringen
m13330:	call	asclcd			;sonst zweite Stelle ausgeben
m13340:	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;vierte Stelle ausgeben
;
m13350:	ldi	r20,224			;X-Position fr Pfeil links
	ldi	r22,0xb0		;Pfeil links hohl laden
	lds	r16,encval		;Encoderwert holen
	lds	r17,encmin		;Encoder Minimalwert holen
	cp	r16,r17			;Minimalwert erreicht?
	breq	m13360			;ja -> berspringen
	ldi	r22,0xb1		;sonst Pfeil links voll laden
m13360:	call	asclcd			;ausgeben
	ldi	r20,231			;X-Position fr Pfeil rechts
	ldi	r22,0xb2		;Pfeil rechts hohl laden
	lds	r16,encval		;Encoderwert holen
	lds	r17,encmax		;Encoder Maximalwert holen
	cp	r16,r17			;Maximalwert erreicht?
	breq	m13370			;ja -> berspringen
	ldi	r22,0xb3		;sonst Pfeil rechts voll laden
m13370:	call	asclcd			;ausgeben
	rjmp	m14000
;
; Menzeile ausgeben: Datum, Zeit und Temperaturwerte anstatt des
; 4-stelligen Encoderwertes ausgeben (fr Grafik/Fadenkreuz)
;
m13400:	ldi	r17,127			;Position 15min/Einzeldiagamm
	lds	r16,grflag		;Grafik-Flags holen
	bst	r16,4			;Doppeldiagramm?
	brtc	m13410			;nein -> Position korrekt
	ldi	r17,93			;sonst Position Doppeldiagramm
m13410:	lds	r16,grmod1		;Grafikmodus Diagramm 1 holen
	tst	r16			;15min-Werte?
	breq	m13420			;ja -> Position korrekt
	ldi	r16,31			;sonst Position korrigieren,
	add	r17,r16			;nur Datum anzeigen
m13420:	mov	r20,r17			;ermittelte Position kopieren
	ldi	r21,126			;bis dahin alle Pixel lschen
	mov	r18,r13			;gesicherte Anfangsposit. holen
	ldi	r19,115			;Y-Anfangsposition setzen
	call	clrwnd			;Text-Zwischenraum lschen
;
	ldi	xl,low(grbuff)
	ldi	xh,high(grbuff)		;Basisadresse Grafikdatenpuffer
	lds	r16,encval		;neue Fadenkreuzposition holen
	ldi	r17,8
	mul	r17,r16			;Puffer-Offset berechnen
	add	xl,r0			;Pufferadresse ermitteln
	adc	xh,r1			;bertrag
	ld	r8,x+
	ld	r9,x+
	ld	r10,x+
	ld	r11,x+			;Zeitdaten holen
;
	ldi	r21,115			;Y-Position fr Textausgabe
	mov	r16,r8
	and	r16,r9
	and	r16,r10
	and	r16,r11			;Zeitdaten verknpfen
	cpi	r16,0xff		;Zeit gltig?
	brne	m13430			;ja -> Daten ausgeben
;
	ldi	r22,0xaa		;String "keine Daten"
	call	strlen			;Stringlnge ermitteln
	ldi	r20,222			;Endposition der Menzeile
	sub	r20,r18			;Text-Anfang berechnen, bis
	ldi	r21,126			;dahin alle Pixel lschen
	mov	r18,r13			;gesicherte Anfangsposit. holen
	ldi	r19,115			;Y-Anfangsposition setzen
	call	clrwnd			;Text-Zwischenraum lschen
	ldi	r21,115			;Y-Position fr Text setzen
	ldi	r22,0xaa		;String "keine Daten"
	call	strlcd			;String ausgeben
	rjmp	m13350			;Richtungspfeile ausgeben
;
m13430:	call	grelcd			;Datum ausgeben
	ldi	r22,'.'			;zustzlichen Punkt
	call	asclcd			;ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
;
	lds	r16,grmod1		;Grafikmodus Diagramm 1 holen
	tst	r16			;15min-Werte?
	brne	m13440			;nein -> keine Zeit ausgeben
	call	grelc2			;sonst Zeit ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
;
m13440:	ld	r16,x+
	ld	r17,x+			;Temperaturwert 1 holen
	cpi	r17,0x80		;gltiger Wert?
	brne	m13450			;ja -> ausgeben
	ldi	r16,' '			;sonst Leerzeichen
	mov	r4,r16			;als Hunderter setzen
	mov	r5,r16			;als Zehner setzen
	ldi	r16,'-'			;Strich
	mov	r6,r16			;als Einer setzen
	mov	r7,r16			;als Zehntel setzen
	rjmp	m13460			;ausgeben
;
m13450:	call	temasc			;in ASCII wandeln
m13460:	mov	r22,r4			;erste Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r5			;zweite Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,'.'			;Trennzeichen Punkt
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,1			;Trennzeichen Punkt?
	brtc	m13470			;ja -> weiter
	ldi	r22,','			;sonst Trennzeichen Komma
m13470:	call	asclcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,0xa5		;Grad-Zeichen
	call	asclcd			;ausgeben
;
	lds	r16,grflag		;Grafik-Flags holen
	bst	r16,4			;Doppeldiagramm?
	brtc	m13510			;nein -> Endezeichen ausgeben
	ldi	r22,0xa0		;sonst schmales Leerzeichen
	call	asclcd			;ausgeben
;
	ld	r16,x+
	ld	r17,x+			;Temperaturwert 2 holen
	cpi	r17,0x80		;gltiger Wert?
	brne	m13480			;ja -> ausgeben
	ldi	r16,' '			;sonst Leerzeichen
	mov	r4,r16			;als Hunderter setzen
	mov	r5,r16			;als Zehner setzen
	ldi	r16,'-'			;Strich
	mov	r6,r16			;als Einer setzen
	mov	r7,r16			;als Zehntel setzen
	rjmp	m13490			;ausgeben
;
m13480:	call	temasc			;in ASCII wandeln
m13490:	mov	r22,r4			;erste Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r5			;zweite Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,'.'			;Trennzeichen Punkt
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,1			;Trennzeichen Punkt?
	brtc	m13500			;ja -> weiter
	ldi	r22,','			;sonst Trennzeichen Komma
m13500:	call	asclcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,0xa5		;Grad-Zeichen
	call	asclcd			;ausgeben
m13510:	ldi	r22,0x7f		;Endezeichen
	call	asclcd			;ausgeben
	rjmp	m13350			;Richtungspfeile ausgeben
;
; Menzeile ausgeben: Encoderwert als Temperaturwert ausgeben
;
m13600:	bst	r16,2			;Ausgabe als Temperaturwert?
	brts	m13610			;ja -> bearbeiten
	rjmp	m13700			;sonst weiter testen
;
m13610:	lds	r16,encval		;Encoderwert L holen
	lds	r17,encvah		;Encoderwert H holen
	mov	r12,r13			;X-Anfangsposition sichern
	call	temasc			;Wert in ASCII wandeln
	mov	r16,r4			;erste Stelle holen
	call	asclen			;Zeichenlnge ermitteln
	mov	r18,r16			;Lnge speichern
	mov	r16,r5			;zweite Stelle holen
	call	asclen			;Zeichenlnge ermitteln
	add	r18,r16			;und zur Gesamtlnge addieren
	mov	r16,r6			;dritte Stelle holen
	call	asclen			;Zeichenlnge ermitteln
	add	r18,r16			;und zur Gesamtlnge addieren
	ldi	r16,'.'			;Trennzeichen (Punkt) laden
	lds	r17,confg1		;Konfigurationsbyte 1 holen
	bst	r17,1			;Trennzeichen Punkt?
	brtc	m13620			;ja -> weiter, sonst
	ldi	r16,','			;Trennzeichen (Komma) laden
m13620:	call	asclen			;Zeichenlnge ermitteln
	add	r18,r16			;und zur Gesamtlnge addieren
	mov	r16,r7			;vierte Stelle holen
	call	asclen			;Zeichenlnge ermitteln
	add	r18,r16			;und zur Gesamtlnge addieren
;
	ldi	r20,222			;Endposition der Menzeile
	sub	r20,r18			;Text-Anfang berechnen, bis
	ldi	r21,126			;dahin alle Pixel lschen
	mov	r18,r12			;gesicherte Anfangsposit. holen
	ldi	r19,115			;Y-Anfangsposition setzen
	call	clrwnd			;Text-Zwischenraum lschen
	ldi	r21,115			;Y-Position fr Textausgabe
;
	mov	r22,r4			;erste Stelle holen
	call	asclcd			;erste Stelle ausgeben
	mov	r22,r5			;zweite Stelle holen
	call	asclcd			;zweite Stelle ausgeben
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;dritte Stelle ausgeben
	ldi	r22,'.'			;Trennzeichen (Punkt) laden
	lds	r17,confg1		;Konfigurationsbyte 1 holen
	bst	r17,1			;Trennzeichen Punkt?
	brtc	m13630			;ja -> weiter, sonst
	ldi	r22,','			;Trennzeichen (Komma) laden
m13630:	call	asclcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;vierte Stelle ausgeben
;
	ldi	r20,224			;X-Position fr Pfeil links
	ldi	r22,0xb0		;Pfeil links hohl laden
	lds	r16,encval		;Encoderwert L holen
	lds	r17,encvah		;Encoderwert H holen
	ldi	r18,low(mintmp)		;Minimaltemperatur L laden
	ldi	r19,high(mintmp)	;Minimaltemperatur H laden
	sub	r18,r16
	sbc	r19,r17			;Minimalwert erreicht?
	breq	m13640			;ja -> berspringen
	ldi	r22,0xb1		;sonst Pfeil links voll laden
m13640:	call	asclcd			;ausgeben
	ldi	r20,231			;X-Position fr Pfeil rechts
	ldi	r22,0xb2		;Pfeil rechts hohl laden
	lds	r16,encval		;Encoderwert L holen
	lds	r17,encvah		;Encoderwert H holen
	ldi	r18,low(maxtmp)		;Maximaltemperatur L laden
	ldi	r19,high(maxtmp)	;Maximaltemperatur H laden
	sub	r18,r16
	sbc	r19,r17			;Maximalwert erreicht?
	breq	m13650			;ja -> berspringen
	ldi	r22,0xb3		;sonst Pfeil rechts voll laden
m13650:	call	asclcd			;ausgeben
	rjmp	m14000			;sonst weiter testen
;
; Menzeile ausgeben: Encoderwert als ASCII-Zeichen ausgeben
;
m13700:	bst	r16,3			;Ausgabe als ASCII-Zeichen?
	brts	m13710			;ja -> bearbeiten
	rjmp	m13800			;sonst weiter testen
;
m13710:	lds	r16,encval		;Encoderwert holen
	cpi	r16,maxasc+1		;ASCII Maximalwert+1? (Del)
	brne	m13720			;nein -> weiter
	ldi	r16,0xaf		;durch Lschzeichen ersetzen
m13720:	cpi	r16,maxasc+2		;ASCII Maximalwert+2? (Ok)
	brne	m13730			;nein -> weiter
	ldi	r16,0xae		;durch Ok-Zeichen ersetzen
m13730:	call	asclen			;Zeichenlnge ermitteln
;
	ldi	r20,222			;Endposition der Menzeile
	sub	r20,r16			;Text-Anfang berechnen, bis
	ldi	r21,126			;dahin alle Pixel lschen
	mov	r18,r13			;gesicherte Anfangsposit. holen
	ldi	r19,115			;Y-Anfangsposition setzen
	call	clrwnd			;Text-Zwischenraum lschen
	ldi	r21,115			;Y-Position fr Textausgabe
;
	lds	r22,encval		;Encoderwert holen
	cpi	r22,maxasc+1		;ASCII Maximalwert+1? (Del)
	brne	m13740			;nein -> weiter
	ldi	r22,0xaf		;durch Lschzeichen ersetzen
m13740:	cpi	r22,maxasc+2		;ASCII Maximalwert+2? (Ok)
	brne	m13750			;nein -> weiter
	ldi	r22,0xae		;durch Ok-Zeichen ersetzen
m13750:	call	asclcd			;ausgeben
;
	ldi	r20,224			;X-Position fr Pfeil links
	ldi	r22,0xb0		;Pfeil links hohl laden
	lds	r16,encval		;Encoderwert holen
	cpi	r16,minasc		;ASCII Minimalwert erreicht?
	breq	m13760			;ja -> berspringen
	ldi	r22,0xb1		;sonst Pfeil links voll laden
m13760:	call	asclcd			;ausgeben
	ldi	r20,231			;X-Position fr Pfeil rechts
	ldi	r22,0xb2		;Pfeil rechts hohl laden
	lds	r16,encval		;Encoderwert holen
	cpi	r16,maxasc+2		;ASCII Maximalwert erreicht?
	breq	m13770			;ja -> berspringen
	ldi	r22,0xb3		;sonst Pfeil rechts voll laden
m13770:	call	asclcd			;ausgeben
;
; Menzeile ausgeben: Encoderwert im Zeitformat ausgeben
;
m13800:	bst	r16,5			;Ausgabe im Zeitformat?
	brts	m13810			;ja -> bearbeiten
	rjmp	m14000			;sonst Ende
;
m13810:	lds	r16,encval		;Encoderwert L holen
	lds	r17,encvah		;Encoderwert H holen
	call	timasc			;Wert in ASCII wandeln
	mov	r16,r4			;erste Stelle holen
	call	asclen			;Zeichenlnge ermitteln
	mov	r18,r16			;Lnge speichern
	mov	r16,r5			;zweite Stelle holen
	call	asclen			;Zeichenlnge ermitteln
	add	r18,r16			;und zur Gesamtlnge addieren
	ldi	r16,':'			;Doppelpunkt laden
	call	asclen			;Zeichenlnge ermitteln
	add	r18,r16			;und zur Gesamtlnge addieren
	mov	r16,r6			;dritte Stelle holen
	call	asclen			;Zeichenlnge ermitteln
	add	r18,r16			;und zur Gesamtlnge addieren
	mov	r16,r7			;vierte Stelle holen
	call	asclen			;Zeichenlnge ermitteln
	add	r18,r16			;und zur Gesamtlnge addieren
;
	ldi	r20,222			;Endposition der Menzeile
	sub	r20,r18			;Text-Anfang berechnen, bis
	ldi	r21,126			;dahin alle Pixel lschen
	mov	r18,r13			;gesicherte Anfangsposit. holen
	ldi	r19,115			;Y-Anfangsposition setzen
	call	clrwnd			;Text-Zwischenraum lschen
	ldi	r21,115			;Y-Position fr Textausgabe
;
	mov	r22,r4			;erste Stelle holen
	call	asclcd			;erste Stelle ausgeben
	mov	r22,r5			;zweite Stelle holen
	call	asclcd			;zweite Stelle ausgeben
	ldi	r22,':'			;Trennzeichen (Doppelpunkt)
	call	asclcd			;ausgeben
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;dritte Stelle ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;vierte Stelle ausgeben
;
	ldi	r20,224			;X-Position fr Pfeil links
	ldi	r22,0xb1		;Pfeil links voll laden
	call	asclcd			;ausgeben
	ldi	r20,231			;X-Position fr Pfeil rechts
	ldi	r22,0xb3		;Pfeil rechts voll laden
	call	asclcd			;ausgeben
;
; Temperaturanzeige neu ausgeben: zustzliche Linien zeichnen, Flags
; setzen
;
m14000:	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,0			;Temperatur-Anzeige?
	brts	m14010			;ja -> bearbeiten
	rjmp	m15000			;sonst weiter testen
m14010:	bst	r16,1			;Anzeige neu ausgeben?
	brtc	m14100			;nein -> berspringen
	cbr	r16,0x02		;sonst Neuausgabe-Flag lschen
	sts	dflags,r16		;und Flags wieder speichern
;
	ldi	r19,15|0x80		;Y=15
m14020:	ldi	r18,87			;X=87, erste Trennlinie
	call	pixlcd			;senkrechte Linie zeichnen
	ldi	r18,174			;X=174, zweite Trennlinie
	call	pixlcd			;senkrechte Linie zeichnen
	inc	r19			;Y-Koordinate erhhen
	cpi	r19,112|0x80		;alle Pixel ausgegeben?
	brne	m14020			;nein -> Schleife
;
	ldi	r18,175			;X=175
	ldi	r19,64|0x80		;Y=64, Trennlinie fr Groanz.
m14030:	call	pixlcd			;Linie zeichnen
	inc	r18			;X-Koordinate erhhen
	cpi	r18,239			;alle Pixel ausgegeben?
	brne	m14030			;nein -> Schleife
;
	ldi	xl,low(temflg)
	ldi	xh,high(temflg)		;Zeiger auf Temperaturflags
	ldi	r17,16			;16 Flags bearbeiten
m14050:	ld	r16,x			;Temperaturflag holen
	bst	r16,7			;Sensor inaktiv?
	brts	m14060			;ja -> Flags nicht ndern
	sbr	r16,0x03		;Flag fr Neuausgabe setzen
m14060:	st	x+,r16			;Temperaturflag wieder speich.
	dec	r17			;alle Flags bearbeitet?
	brne	m14050			;nein -> Schleife
;
	ldi	xl,low(almflg)
	ldi	xh,high(almflg)		;Zeiger auf Alarmflags
	ldi	r17,4			;4 Flags bearbeiten
m14070:	ld	r16,x			;Alarmflag holen
	bst	r16,7			;Alarm inaktiv?
	brts	m14080			;ja -> Flags nicht ndern
	sbr	r16,0x01		;Flag fr Neuausgabe setzen
m14080:	st	x+,r16			;Alarmflag wieder speichern
	dec	r17			;alle Flags bearbeitet?
	brne	m14070			;nein -> Schleife
;
; Temperaturanzeige: Sensornamen aus EEPROM ausgeben
;
m14100:	ldi	r25,0			;Sensornummerzhler = 0
	ldi	xl,low(temmap)
	ldi	xh,high(temmap)		;Zeiger auf Mapping Tabelle
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
m14110:	ld	r16,y			;Flags holen
	bst	r16,7			;Sensor inaktiv?
	brtc	m14120			;nein -> weiter testen
	rjmp	m14270			;sonst Ausgabe berspringen
m14120:	bst	r16,1			;Neuausgabe erforderlich?
	brts	m14130			;ja -> ausgeben
	rjmp	m14270			;sonst Ausgabe berspringen
m14130:	cbr	r16,0x02		;Ausgabe-Flag lschen
	st	y,r16			;Flags wieder speichern
	clr	txflag			;Textflags lschen
	andi	r16,0x0c		;ber/Unterschreitung?
	breq	m14140			;nein -> weiter testen
	ldi	r16,0x01		;sonst invers darstellen
	rjmp	m14150
m14140:	ld	r16,y			;Flags nochmals holen
	andi	r16,0x30		;Erinnerungs-Flag(s) gesetzt?
	breq	m14160			;nein -> normal ausgeben
	ldi	r16,0x02		;sonst unterstrichen darstellen
m14150:	mov	txflag,r16		;Textflags setzen
;
m14160:	mov	r16,r25			;Sensornummerzhler holen
	add	r16,r16			;verdoppeln und spter als
	clr	r17			;Offset verwenden
	lds	r18,xflags		;verschiedene Flags holen
	bst	r18,2			;Minimal-Werte anzeigen?
	brtc	m14170			;nein -> weiter testen
;
	clr	txflag			;zunchst Textflags lschen
	ldi	zl,low(temlil)
	ldi	zh,high(temlil)		;Zeiger auf untere Temp-Limits
	add	zl,r16
	adc	zh,r17			;Tabellenplatz berechnen
	ld	r18,z			;unteres Temp-Limit L holen
	ldd	r19,z+1			;unteres Temp-Limit H holen
	cpi	r19,0x80		;gltiger Wert?
	breq	m14180			;nein -> Ende
	ldi	zl,low(temin1)
	ldi	zh,high(temin1)		;Zeiger auf Minimalwerte (Anz)
	add	zl,r16
	adc	zh,r17			;Tabellenplatz berechnen
	ld	r16,z			;Minimalwert L holen
	ldd	r17,z+1			;Minimalwert H holen
	cpi	r17,0x80		;gltiger Wert?
	breq	m14180			;nein -> Ende
;
	sub	r16,r18			;Minimalwert - Limit berechnen
	sbc	r17,r19			;Limit unterschritten?
	brpl	m14180			;nein -> weiter
	ldi	r16,1			;sonst Text invers darstellen
	mov	txflag,r16		;Textflag wieder speichern
	rjmp	m14180
;
m14170:	bst	r18,3			;Maximal-Werte anzeigen?
	brtc	m14180			;nein -> weiter
;
	clr	txflag			;zunchst Textflags lschen
	ldi	zl,low(temlih)
	ldi	zh,high(temlih)		;Zeiger auf obere Temp-Limits
	add	zl,r16
	adc	zh,r17			;Tabellenplatz berechnen
	ld	r18,z			;oberes Temp-Limit L holen
	ldd	r19,z+1			;oberes Temp-Limit H holen
	cpi	r19,0x80		;gltiger Wert?
	breq	m14180			;nein -> Ende
	ldi	zl,low(temax1)
	ldi	zh,high(temax1)		;Zeiger auf Maximalwerte (Anz)
	add	zl,r16
	adc	zh,r17			;Tabellenplatz berechnen
	ld	r16,z			;Maximalwert L holen
	ldd	r17,z+1			;Maximalwert H holen
	cpi	r17,0x80		;gltiger Wert?
	breq	m14180			;nein -> Ende
;
	sub	r18,r16			;Limit - Maximalwert berechnen
	sbc	r19,r17			;Limit unterschritten?
	brpl	m14180			;nein -> weiter
	ldi	r16,1			;sonst Text invers darstellen
	mov	txflag,r16		;Textflag wieder speichern
;
m14180:	ld	r18,x			;Anzeigeposition holen
	ldi	r20,2			;X-Startposition Spalte 1
	ldi	r21,16			;Y-Startposition
	cpi	r18,8			;Temperatur in Spalte 1?
	brcs	m14190			;ja -> weiter
	ldi	r20,89			;sonst X-Startposition Spalte 2
m14190:	mov	r16,r18			;Anzeigeposition holen
	andi	r16,0x07		;Zeilenposition filtern
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r16,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
	ldi	r19,9			;Sensornamenlnge im EEPROM
	mul	r25,r19			;Tabellenoffset ermitteln
	ldi	r16,low(etemtx)		;EEPROM-Adresse Sensornamen L
	ldi	r17,high(etemtx)	;EEPROM-Adresse Sensornamen H
	add	r16,r0			;Offset zur Adresse L addieren
	adc	r17,r1			;Offset zur Adresse H addieren
	movw	r10,r16			;EEPROM-Adresse sichern
	call	eeplcd			;EEPROM-Text ausgeben
	ldi	r22,0x7f		;Ende-Zeichen
	call	asclcd			;Zeichen ausgeben
;
	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,52			;Endwert X erste Spalte setzen
	cpi	r18,87			;zweite Spalte?
	brcs	m14230			;nein -> weiter
	ldi	r20,139			;sonst Endwert X zweite Spalte
m14230:	ldi	r22,11			;zum Endwert Y 11 Pixel
	add	r21,r22			;addieren
	bst	txflag,0		;Invers-Darstellung?
	bld	r19,7			;sonst Invers-Bit in den
	bld	r21,7			;Koordinaten setzen
	call	clrwnd			;Bereich lschen oder setzen
;
	lds	r16,bigtmp		;Sensornummer Groanzeige holen
	cp	r16,r25			;entspricht aktuellem Sensor?
	brne	m14270			;nein -> keine Ausgabe
	clr	txflag			;Textflags lschen
	movw	r16,r10			;sonst EEPROM-Adresse holen
	ldi	r19,9			;Sensornamenlnge im EPROM
	ldi	r20,176			;X-Koordinate Groanzeige
	ldi	r21,66			;Y-Koordinate Groanzeige
	call	eeplcd			;EEPROM-Text ausgeben
	ldi	r22,0x7f		;Ende-Zeichen
	call	asclcd			;Zeichen ausgeben
	ldi	r16,0xa6		;Zeichen "Grad Celsius" laden
	call	asclen			;Zeichenlnge bestimmen
	ldi	r20,237			;Position rechtsbndig
	sub	r20,r16			;berechnen
	ldi	r22,0xa6		;Zeichen "Grad Celsius" laden
	call	asclcd			;Zeichen ausgeben
;
m14270:	adiw	xl,1			;Mapping Tabellenzeiger erhhen
	adiw	yl,1			;Flag-Zeiger erhhen
	inc	r25			;nchste Sensornummer
	cpi	r25,16			;alle Sensoren bearbeitet?
	brcc	m14400			;ja -> weiter
	rjmp	m14110			;sonst Schleife
;
; Temperaturanzeige: Temperaturwerte ausgeben
;
m14400:	ldi	r25,0			;Sensornummerzhler = 0
	ldi	xl,low(temmap)
	ldi	xh,high(temmap)		;Zeiger auf Mapping-Tabelle
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
m14410:	ld	r16,y			;Flags holen
	bst	r16,7			;Sensor inaktiv?
	brtc	m14420			;nein -> Daten ausgeben
	rjmp	m145z0			;sonst keine Ausgabe
m14420:	bst	r16,0			;Temperaturnderung?
	brts	m14430			;ja -> Daten ausgeben
	rjmp	m145z0			;sonst keine Ausgabe
m14430:	cbr	r16,0x01		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
	clr	txflag			;Textflags lschen
	andi	r16,0x0c		;ber/Unterschreitung?
	breq	m14440			;nein -> weiter testen
	ldi	r16,0x01		;sonst invers darstellen
	rjmp	m14450
m14440:	ld	r16,y			;Flags nochmals holen
	andi	r16,0x30		;Erinnerungs-Flag(s) gesetzt?
	breq	m14460			;nein -> normal ausgeben
	ldi	r16,0x02		;sonst unterstrichen ausgeben
m14450:	mov	txflag,r16		;Textflags setzen
m14460:	ld	r18,x			;Anzeigeposition holen
	ldi	r20,52			;X-Startposition Spalte 1
	ldi	r21,16			;Y-Startposition
	cpi	r18,8			;Temperatur in Spalte 1?
	brcs	m14470			;ja -> weiter
	ldi	r20,139			;sonst X-Startposition Spalte 2
m14470:	mov	r16,r18			;Anzeigeposition holen
	andi	r16,0x07		;Zeilenposition filtern
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r16,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
	ldi	r19,2			;Temperaturwert 2 Byte
	mul	r25,r19			;Tabellenoffset ermitteln
	movw	r8,yl			;Flag-Zeiger sichern
	ldi	yl,low(temval)
	ldi	yh,high(temval)		;Zeiger auf Temperaturwerte
;
	lds	r16,xflags		;verschiedene Flags holen
	bst	r16,2			;Minimal-Werte anzeigen?
	brtc	m14480			;nein -> weiter testen
	ldi	yl,low(temin1)
	ldi	yh,high(temin1)		;sonst Zeiger auf Minimalwerte
	clr	txflag			;Textflags lschen
;
	ldi	zl,low(temlil)
	ldi	zh,high(temlil)		;Zeiger auf untere Temp-Limits
	add	zl,r0
	adc	zh,r1			;Tabellenplatz berechnen
	ld	r18,z			;unteres Temp-Limit L holen
	ldd	r19,z+1			;unteres Temp-Limit H holen
	cpi	r19,0x80		;gltiger Wert?
	breq	m14490			;nein -> Ende
	ldi	zl,low(temin1)
	ldi	zh,high(temin1)		;Zeiger auf Minimalwerte (Anz)
	add	zl,r0
	adc	zh,r1			;Tabellenplatz berechnen
	ld	r16,z			;Minimalwert L holen
	ldd	r17,z+1			;Minimalwert H holen
	cpi	r17,0x80		;gltiger Wert?
	breq	m14490			;nein -> Ende
;
	sub	r16,r18			;Minimalwert - Limit berechnen
	sbc	r17,r19			;Limit unterschritten?
	brpl	m14490			;nein -> weiter
	ldi	r16,1			;sonst Text invers darstellen
	mov	txflag,r16		;Textflag wieder speichern
	rjmp	m14490
;
m14480:	bst	r16,3			;Minimal-Werte anzeigen?
	brtc	m14490			;nein -> weiter testen
	ldi	yl,low(temax1)
	ldi	yh,high(temax1)		;sonst Zeiger auf Maximalwerte
	clr	txflag			;Textflags lschen
;
	ldi	zl,low(temlih)
	ldi	zh,high(temlih)		;Zeiger auf obere Temp-Limits
	add	zl,r0
	adc	zh,r1			;Tabellenplatz berechnen
	ld	r18,z			;oberes Temp-Limit L holen
	ldd	r19,z+1			;oberes Temp-Limit H holen
	cpi	r19,0x80		;gltiger Wert?
	breq	m14490			;nein -> Ende
	ldi	zl,low(temax1)
	ldi	zh,high(temax1)		;Zeiger auf Maximalwerte (Anz)
	add	zl,r0
	adc	zh,r1			;Tabellenplatz berechnen
	ld	r16,z			;Maximalwert L holen
	ldd	r17,z+1			;Maximalwert H holen
	cpi	r17,0x80		;gltiger Wert?
	breq	m14490			;nein -> Ende
;
	sub	r18,r16			;Limit - Maximalwert berechnen
	sbc	r19,r17			;Limit unterschritten?
	brpl	m14490			;nein -> weiter
	ldi	r16,1			;sonst Text invers darstellen
	mov	txflag,r16		;Textflag wieder speichern
;
m14490:	add	yl,r0			;Tabellenoffset L addieren
	adc	yh,r1			;Tabellenoffset H addieren
	ld	r16,y			;Temperaturwert L holen
	ldd	r17,y+1			;Temperaturwert H holen
	cpi	r17,0x80		;Temperaturwert gltig?
	brne	m14520			;ja -> ausgeben
;
	lds	r16,xflags		;verschiedene Flags holen
	andi	r16,0x0c		;Min/Max-Werte anzeigen?
	breq	m14500			;nein -> weiter
	rjmp	m145y0			;sonst Ausgabe berspringen
m14500:	movw	yl,r8			;Flag-Zeiger zurckholen
	ld	r16,y			;Temperaturflags holen
	bst	r16,6			;Sensorausfall?
	brts	m14510			;ja -> Ausfallzeichen ausgeben
	rjmp	m145y0			;sonst Ausgabe berspringen
m14510:	ldi	r16,31
	add	r20,r16			;X-Position auf Ausfallzeichen
	ldi	r22,'!'			;Ausfallzeichen
	call	asclcd			;ausgeben
	ldi	r22,0x7f		;Endezeichen
	call	asclcd			;ausgeben
	rjmp	m145y0			;keine weiteren Ausgaben
;
m14520:	movw	r10,r16			;Temperaturwert sichern
	call	temasc			;Wert in ASCII wandeln
;
	mov	r12,txflag		;Textflags sichern
	mov	r22,r4			;erste Stelle holen
	cpi	r22,' '			;Leerzeichen?
	brne	m14530			;nein -> weiter
	ldi	r16,0x01		;sonst Unterstreichungs-Flag
	and	txflag,r16		;lschen
m14530:	call	asclcd			;Zeichen ausgeben
	mov	txflag,r12		;Textflags wiederherstellen
	mov	r22,r5			;zweite Stelle holen
	cpi	r22,' '			;Leerzeichen?
	brne	m14540			;nein -> weiter
	ldi	r16,0x01		;sonst Unterstreichungs-Flag
	and	txflag,r16		;lschen
m14540:	call	asclcd			;Zeichen ausgeben
	mov	txflag,r12		;Textflags wiederherstellen
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,'.'			;Trennzeichen Punkt
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,1			;Trennzeichen Punkt?
	brtc	m14550			;ja -> weiter
	ldi	r22,','			;sonst Trennzeichen Komma
m14550:	call	asclcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
;
	ldi	r22,0xa7		;Grad-Zeichen
	movw	yl,r8			;Flag-Zeiger zurckholen
	lds	r16,xflags		;verschiedene Flags holen
	andi	r16,0x0c		;Min/Max-Werte anzeigen?
	brne	m14580			;ja -> kein Sonderzeichen
	ld	r16,y			;Flags holen
	andi	r16,0x14		;berschreitung/Erinnerung?
	breq	m14560			;nein -> weiter
;
	ldi	r22,0xa8		;Grad + Pfeil nach oben
m14560:	ld	r16,y			;Flags holen
	andi	r16,0x28		;Unterschreitung/Erinnerung?
	breq	m14570			;nein -> weiter
	ldi	r22,0xa9		;Grad + Pfeil nach unten
m14570:	ld	r16,y			;Flags holen
	bst	r16,6			;Ausfall-Flag gesetzt?
	brtc	m14580			;nein -> weiter
	ldi	r22,0xaa		;sonst Ausfall-Symbol holen
m14580:	call	asclcd			;Zeichen ausgeben
	ldi	r22,0x7f		;Endezeichen
	call	asclcd			;ausgeben
;
	lds	r16,bigtmp		;Sensornummer Groanzeige holen
	cp	r16,r25			;entspricht aktuellem Sensor?
	breq	m14590			;ja -> anzeigen
	rjmp	m145y0			;sonst keine Ausgabe
;
; Groanzeige berbeiten, Prfen ob ganzzahlig angezeigt werden muss
;
m14590:	ldi	r16,low(999)		;3-stellige Groanzeige nur
	ldi	r17,high(999)		;bis 99,9 Grad
	sub	r16,r10
	sbc	r17,r11			;Temperaturwert >99,9?
	brmi	m145b0			;ja -> ganzzahlig anzeigen
	ldi	r16,low(-99)		;3-stellige Groanzeige nur
	ldi	r17,high(-99)		;ab -9,9 Grad
	movw	r18,r10			;Temperaturwert kopieren
	sub	r18,r16
	sbc	r19,r17			;Temperaturwert <-9,9?
	brpl	m145a0			;nein -> normal ausgeben
;
	ldi	r16,low(-990)		;ganzzahlige Groanzeige nur
	ldi	r17,high(-990)		;ab -99,0 Grad
	movw	r18,r10			;Temperaturwert kopieren
	sub	r18,r16
	sbc	r19,r17			;Temperaturwert <-99,0?
	brpl	m145b0			;nein -> ganzzahlig ausgeben
	movw	r10,r16			;sonst immer Wert -99
	rjmp	m145b0			;ganzzahlig anzeigen
;
; normale Groanzeige im Bereich -9,9 bis 99,9 Grad
;
m145a0:	movw	r16,r10			;Temperaturwert holen
	call	temasc			;und in ASCII wandeln
;
	ldi	r18,194			;Zwischenraum 1 lschen
	ldi	r19,77			;X/Y Anfangs-Koordinaten
	ldi	r20,197
	ldi	r21,110			;X/Y End-Koordinaten
	call	clrwnd			;ausgeben
	ldi	r18,213			;Zwischenraum 2 lschen
	ldi	r19,77			;X/Y Anfangs-Koordinaten
	ldi	r20,219
	ldi	r21,110			;X/Y End-Koordinaten
	call	clrwnd			;ausgeben
;
	mov	r22,r5			;zweite Stelle holen
	ldi	r20,178			;X Koordinate fr 2. Stelle
	ldi	r21,77			;Y Koordinate fr 2. Stelle
	call	biglcd			;ausgeben
	mov	r22,r6			;dritte Stelle holen
	ldi	r20,197			;X Koordinate fr 3. Stelle
	ldi	r21,77			;Y Koordinate fr 3. Stelle
	call	biglcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	ldi	r20,219			;X Koordinate fr 4. Stelle
	ldi	r21,77			;Y Koordinate fr 4. Stelle
	call	biglcd			;ausgeben
;
	ldi	r18,214			;X Koordinate Dezimalpunkt
	ldi	r19,104|0x80		;Y Koordinate Dezimalpunkt
	ldi	r20,218
	ldi	r21,108|0x80		;Dezimalpunkt als 4x4 Block
	call	clrwnd			;ausgeben
	rjmp	m145y0
;
; gerundete Groanzeige im Bereich -99 bis -10 und 100 bis 200 Grad
;
m145b0:	ldi	r18,4			;Wert 4 laden fr Rundung
	clr	r19
	movw	r16,r10			;Temperaturwert kopieren
	bst	r17,7			;Wert negativ?
	brts	m145c0			;ja -> weiter
	add	r16,r18			;4 addieren fr Rundung
	adc	r17,r19			;bertrag
	rjmp	m145d0
;
m145c0:	sub	r16,r18
	sbc	r17,r19			;4 subtrahieren fr Rundung
m145d0:	call	temasc			;in ASCII wandeln
;
	ldi	r18,178			;Zwischenraum 1 lschen
	ldi	r19,77			;X/Y Anfangs-Koordinaten
	ldi	r20,180
	ldi	r21,110			;X/Y End-Koordinaten
	call	clrwnd			;ausgeben
	ldi	r18,196			;Zwischenraum 2 lschen
	ldi	r19,77			;X/Y Anfangs-Koordinaten
	ldi	r20,199
	ldi	r21,110			;X/Y End-Koordinaten
	call	clrwnd			;ausgeben
	ldi	r18,215			;Zwischenraum 3 lschen
	ldi	r19,77			;X/Y Anfangs-Koordinaten
	ldi	r20,218
	ldi	r21,110			;X/Y End-Koordinaten
	call	clrwnd			;ausgeben
	ldi	r18,234			;Zwischenraum 4 lschen
	ldi	r19,77			;X/Y Anfangs-Koordinaten
	ldi	r20,235
	ldi	r21,110			;X/Y End-Koordinaten
	call	clrwnd			;ausgeben
;
	mov	r22,r4			;erste Stelle holen
	ldi	r20,180			;X Koordinate fr 1. Stelle
	ldi	r21,77			;Y Koordinate fr 1. Stelle
	call	biglcd			;ausgeben
	mov	r22,r5			;zweite Stelle holen
	ldi	r20,199			;X Koordinate fr 2. Stelle
	ldi	r21,77			;Y Koordinate fr 2. Stelle
	call	biglcd			;ausgeben
	mov	r22,r6			;dritte Stelle holen
	ldi	r20,218			;X Koordinate fr 3. Stelle
	ldi	r21,77			;Y Koordinate fr 3. Stelle
	call	biglcd			;ausgeben
;
m145y0:	movw	yl,r8			;Flag-Zeiger zurckholen
m145z0:	adiw	yl,1			;nchstes Temperaturflag
	adiw	xl,1			;Mapping Zeiger erhhen
	inc	r25			;nchste Sensornummer
	cpi	r25,16			;alle Sensoren bearbeitet?
	brcc	m14600			;ja -> weiter
	rjmp	m14410			;sonst Schleife
;
; Alarmanzeige: Alarmnamen aus EEPROM ausgeben
;
m14600:	ldi	r25,0			;Alarmnummerzhler = 0
	ldi	xl,low(almmap)
	ldi	xh,high(almmap)		;Zeiger auf Mapping Tabelle
	ldi	yl,low(almflg)
	ldi	yh,high(almflg)		;Zeiger auf Alarmflags
m14610:	clr	txflag			;Textflags lschen
	ld	r16,y			;Flags holen
	bst	r16,7			;Alarm inaktiv?
	brtc	m14620			;nein -> weiter
	rjmp	m14710			;sonst Ausgabe berspringen
m14620:	bst	r16,0			;Neuausgabe erforderlich?
	brts	m14630			;ja -> ausgeben
	rjmp	m14710			;sonst Ausgabe berspringen
m14630:	cbr	r16,0x01		;sonst Ausgabe-Flag lschen
	st	y,r16			;Flags wieder speichern
	bst	r16,1			;Alarm aktiv?
	brtc	m14640			;nein -> weiter testen
	ldi	r16,0x01		;sonst invers darstellen
	rjmp	m14650
m14640:	bst	r16,2			;Erinnerungs-Flag gesetzt?
	brtc	m14660			;nein -> normal ausgeben
	ldi	r16,0x02		;sonst unterstrichen darstellen
m14650:	mov	txflag,r16
m14660:	ld	r18,x			;Anzeigeposition holen
	ldi	r20,176			;X-Startposition
	ldi	r21,16			;Y-Startposition
	mov	r16,r18			;Anzeigeposition holen
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r16,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
	ldi	r19,11			;Alarmnamenlnge im EPROM
	mul	r25,r19			;Tabellenoffset ermitteln
	ldi	r16,low(ealmtx)		;EEPROM-Adresse Alarmnamen L
	ldi	r17,high(ealmtx)	;EEPROM-Adresse Alarmnamen H
	add	r16,r0			;Offset zur Adresse L addieren
	adc	r17,r1			;Offset zur Adresse H addieren
m14670:	call	eeread			;Zeichen aus EEPROM holen
	cpi	r18,0x20		;Leerzeichen (=Ende)?
	breq	m14700			;ja -> Endezeichen ausgeben
	mov	r22,r18			;Zeichen kopieren
	push	r16
	push	r17			;EEPROM-Adresse sichern
	push	r19			;Zhler sichern
	subi	r18,0x20		;Korrektur fr Zeichentabelle
	cpi	r18,0x80		;ASCII-Bereich berschritten?
	brcc	m14680			;ja -> keine Ausgabe
	ldi	r17,12			;ASCII-Tabellen-Offset
	mul	r18,r17			;Tabellenplatz berechnen
;
	ldi	zl,low(chrset*2)	;Zeiger auf Zeichentabelle L
	ldi	zh,high(chrset*2)	;Zeiger auf Zeichentabelle H
	ldi	r16,byte3(chrset*2)	;Zeiger auf Zeichentabelle B3
	clr	r17
	add	zl,r0			;Tabellenadresse L berechnen
	adc	zh,r1			;Tabellenadresse H berechnen
	adc	r16,r17			;Tabellenadresse B3 berechnen
	out	rampz,r16
	elpm	r16,z			;Zeichenbreite holen
	inc	r16			;Zeichenbreite korrigieren
	add	r16,r20			;und X-Koordinate addieren
	cpi	r16,238			;passt Zeichen noch auf Zeile?
	brcc	m14680			;nein -> keine Ausgabe
	call	asclcd			;Zeichen ausgeben
m14680:	pop	r19			;Zhler wieder herstellen
	pop	r17
	pop	r16			;EERPOM-Adresse restaurieren
	dec	r19			;alle Zeichen ausgegeben?
	brne	m14670			;nein -> Schleife
m14700:	ldi	r22,0x7f		;Ende-Zeichen
	call	asclcd			;Zeichen ausgeben
;
	cpi	r20,238			;Zeile auffllen erforderlich?
	brcc	m14710			;nein -> berspringen
	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,238			;Endwert X setzen
	ldi	r22,11			;zum Endwert Y 11 Pixel
	add	r21,r22			;addieren
	bst	txflag,0		;Invers-Darstellung?
	bld	r19,7			;sonst Invers-Bit in den
	bld	r21,7			;Koordinaten setzen
	call	clrwnd			;Bereich lschen oder setzen
;
m14710:	adiw	xl,1			;Alarm-Mappingzeiger erhhen
	adiw	yl,1			;Flag-Zeiger erhhen
	inc	r25			;nchste Alarmnummer
	cpi	r25,4			;alle Alarme bearbeitet?
	brcc	m14800			;ja -> weiter
	rjmp	m14610			;sonst Schleife
;
m14800:	jmp	m20000
;
; Anzeigemodus Meldungen: Bildschirmseite neu ausgeben
;
m15000:	lds	r16,dimode		;Anzeigemodus holen
	cpi	r16,1			;Anzeigemodus Meldungen?
	breq	m15010			;ja -> weiter
	rjmp	m16000			;sonst weiter testen
;
m15010:	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,1			;Anzeige neu ausgeben?
	brts	m15020			;ja -> weiter
	jmp	m20000			;sonst Ende
;
m15020:	cbr	r16,0x02		;Neuausgabe-Flag lschen
	sts	dflags,r16		;und Flags wieder speichern
	clr	txflag			;Text-Flag lschen
	ldi	r25,0			;Zeilenzhler auf 0 setzen
;
m15100:	lds	r19,xflag2		;verschiedene Flags2 holen
	bst	r19,5			;nchste Seite?
	brts	m15250			;nein -> vorherige Seite
;
; Anzeigerichtung nchste Seite (ltere Daten)
;
	bst	r19,6			;Listenende erreicht?
	brtc	m15200			;nein -> weiter
	rjmp	m15600			;sonst Ende
m15200:	lds	r20,msgpo1
	lds	r21,msgpo1+1		;Zeiger 1 auf die aktuelle
	lds	r22,msgpo1+2		;Anzeigeposition holen
	ldi	r16,low(xmssta)
	ldi	r17,high(xmssta)	;Anfangsadresse Meldungs-Daten
	ldi	r18,byte3(xmssta)	;laden
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;Anfangsadresse erreicht?
	brne	m15210			;nein -> weiter
	ldi	r20,low(xmssta+xmslen)	;sonst
	ldi	r21,high(xmssta+xmslen)	;End-Adresse Meldungs-Daten
	ldi	r22,byte3(xmssta+xmslen);laden
m15210:	subi	r20,6
	sbci	r21,0			;Zeiger auf nchste Position
	sbci	r22,0			;setzen (zeitlich davor)
	sts	msgpo1,r20
	sts	msgpo1+1,r21		;neuen Zeiger 1 auf die aktu-
	sts	msgpo1+2,r22		;elle Anzeigeposition speichern
;
	lds	r16,xemspo		;aktuellen EEPROM-Zeiger auf
	lds	r17,xemspo+1		;Meldungs-Daten holen, zeigt
	lds	r18,xemspo+2		;auf letzten Listeneintrag
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;Listenende erreicht?
	brne	m15220			;nein -> weiter
	lds	r19,xflag2		;sonst verschiedene Flags holen
	sbr	r19,1<<6		;Listenende-Flag setzen
	sts	xflag2,r19		;Flags wieder speichern
m15220:	rjmp	m15300
;
; Anzeigerichtung vorherige Seite (neuere Daten)
;
m15250:	bst	r19,7			;Listenende erreicht?
	brtc	m15260			;nein -> weiter
	rjmp	m15600			;sonst Ende
m15260:	lds	r20,msgpo2
	lds	r21,msgpo2+1		;Zeiger 2 auf die aktuelle
	lds	r22,msgpo2+2		;Anzeigeposition holen
	lds	r16,xemspo		;aktuellen EEPROM-Zeiger auf
	lds	r17,xemspo+1		;Meldungs-Daten holen, zeigt
	lds	r18,xemspo+2		;auf letzten Listeneintrag
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;Listenende erreicht?
	brne	m15270			;nein -> weiter
	lds	r19,xflag2		;sonst verschiedene Flags holen
	sbr	r19,1<<7		;Listenende-Flag setzen
	sts	xflag2,r19		;Flags wieder speichern
	rjmp	m15600
;
m15270:	ldi	r16,low(xmssta+xmslen)
	ldi	r17,high(xmssta+xmslen)	;End-Adresse Meldungs-Daten
	ldi	r18,byte3(xmssta+xmslen);laden
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;Endadresse erreicht?
	brne	m15280			;nein -> weiter
	ldi	r20,low(xmssta)		;sonst
	ldi	r21,high(xmssta)	;Anfangsadresse Meldungs-Daten
	ldi	r22,byte3(xmssta)	;laden
;
m15280:	movw	r12,r20			;Zeiger 2 auf die aktuelle
	mov	r14,r22			;Anzeigeposition sichern
	ldi	r16,6
	clr	r17
	add	r20,r16
	adc	r21,r17			;Zeiger auf nchsten Eintrag
	adc	r22,r17			;setzen (zeitlich danach)
	sts	msgpo2,r20
	sts	msgpo2+1,r21		;neuen Zeiger 2 auf die aktu-
	sts	msgpo2+2,r22		;elle Anzeigeposition speichern
;
	lds	r16,xemspo		;aktuellen EEPROM-Zeiger auf
	lds	r17,xemspo+1		;Meldungs-Daten holen, zeigt
	lds	r18,xemspo+2		;auf letzten Listeneintrag
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;Listenende erreicht?
	brne	m15290			;nein -> weiter
	lds	r19,xflag2		;sonst verschiedene Flags holen
	sbr	r19,1<<7		;Listenende-Flag setzen
	sts	xflag2,r19		;Flags wieder speichern
m15290:	movw	r20,r12			;Zeiger 2 auf die aktuelle
	mov	r22,r14			;Anzeigeposition zurckholen
;
; Datensatz lesen und Meldungskategorie feststellen
;
m15300:	ldi	r23,6			;6 Bytes lesen (1 Datensatz)
	call	xdread			;Datensatz aus Ext-EEPROM lesen
	brcc	m15310			;Fehler? nein -> weiter
	call	twierr			;sonst Speicherfehler setzen
	jmp	m20000			;Ausgabe beenden
;
m15310:	lds	r16,xedbuf		;Zeitbyte 0 holen
	lds	r17,xedbuf+1		;Zeitbyte 1 holen
	and	r16,r17			;verknpfen
	lds	r17,xedbuf+2		;Zeitbyte 2 holen
	and	r16,r17			;verknpfen
	lds	r17,xedbuf+3		;Zeitbyte 3 holen
	and	r16,r17			;verknpfen
	cpi	r16,0xff		;gltiger Datenssatz?
	brne	m15330			;ja -> weiter
m15320:	rjmp	m15100			;sonst nchster Datensatz
;
m15330:	lds	r16,xedbuf+5		;Meldungscode holen
	andi	r16,0xf0		;Meldungsnummer filtern
	clr	r22			;Meldungskategorie lschen
	cpi	r16,0xe0		;Meldungsnummer > 0xd0?
	brcc	m15340			;ja -> keine Kategorie zuordnen
	ldi	r22,3			;Meldungskategorie 3 (Fehler)
	cpi	r16,0x70		;Meldungsnummer < 0x70?
	brcs	m15340			;ja -> Kategorie ok, weiter
	ldi	r22,1			;Meldungskategorie 1 (Alarme)
	cpi	r16,0x90		;Meldungsnummer < 0x90?
	brcs	m15340			;ja -> Kategorie ok, weiter
	ldi	r22,3			;Meldungskategorie 3 (Fehler)
	cpi	r16,0xb0		;Meldungsnummer < 0xb0?
	brcs	m15340			;ja -> Kategorie ok, weiter
	ldi	r22,2			;sonst Meldungskat. 3 (Temp.)
;
; Kategorie filtern und Daten ausgeben
;
m15340:	lds	r16,msgfil		;sonst Meldungsfilter holen
	tst	r16			;Meldungsfilter aktiviert?
	breq	m15350			;nein -> weiter
	cp	r16,r22			;stimmt Kategorie berein?
	brne	m15320			;nein -> nchster Datensatz
;
m15350:	ldi	r20,2			;X-Position setzen
	ldi	r21,16			;Y-Position erste Zeile setzen
	ldi	r16,12			;Zeilenabstand setzen
;
	mov	r17,r25			;Zeilenzhler kopieren
	lds	r18,xflag2		;verschiedene Flags2 holen
	bst	r18,5			;nchste Seite?
	brtc	m15360			;ja -> weiter
	com	r17			;sonst Einerkomplement bilden
	andi	r17,0x07		;Zeile von unten beschreiben
;
m15360:	mul	r16,r17			;Y-Position der Zeile berechnen
	add	r21,r0			;zur ersten Position addieren
	tst	r22			;gltige Kategorie?
	breq	m15370			;nein -> kein Symbol ausgeben
	ori	r22,0xa0		;Symbol generieren
	call	asclcd			;ausgeben
m15370:	ldi	r20,11			;X-Position fr Text setzen
	call	xtdtim			;Zeit extrahieren und ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
;
	lds	r16,xedbuf+5		;Meldungscode holen
	swap	r16			;Meldungsnummer an Bit3-Bit0
	ori	r16,0xf0		;Stringnummer generieren
	mov	r13,r16			;Meldungsnummer sichern
	cpi	r16,0xfe		;gltige Meldungsnummer?
	brcc	m15400			;nein -> Zeilenende
	mov	r22,r13			;gesicherte Meldungsnr. holen
	call	strlcd			;Meldungstext ausgeben
	mov	r16,r13			;gesicherte Meldungsnr. holen
	cpi	r16,0xf7		;Parameter ausgeben?
	brcs	m15400			;nein -> Zeilenende
;
	ldi	r22,','			;Komma
	call	asclcd			;ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
	lds	r18,xedbuf+5		;Meldungscode nochmals holen
	andi	r18,0x0f		;Sensor-/Alarmnummer filtern
	mov	r16,r13			;gesicherte Meldungsnr. holen
	cpi	r16,0xf9		;Alarmnummer ausgeben?
	brcs	m15380			;ja -> weiter
	ldi	r19,9			;9 Zeichen Sensortext
	ldi	r16,low(etemtx)		;Zeiger auf EEPROM-Adresse
	ldi	r17,high(etemtx)	;der Sensortexte
	rjmp	m15390			;EEPROM-Text ausgeben
;
m15380:	cpi	r18,4			;Alarmnummer>3?
	brcc	m15400			;ja -> Zeilenende
	ldi	r19,11			;11 Zeichen Alarmtext
	ldi	r16,low(ealmtx)		;Zeiger auf EEPROM-Adresse
	ldi	r17,high(ealmtx)	;der Alarmtexte
;
m15390:	mul	r18,r19			;Adress-Offset ermitteln
	add	r16,r0
	adc	r17,r1			;Tabellenplatz ermitteln
	call	eeplcd			;EEPROM-Text ausgeben
;
m15400:	ldi	r22,0x7f		;Ende-Zeichen
	call	asclcd			;ausgeben
;
; nchste Zeile ausgeben
;
m15500:	inc	r25			;Zeilenzhler erhhen
	cpi	r25,8			;letzte Zeile erreicht?
	breq	m15620			;ja -> Ende
	rjmp	m15100			;sonst nchste Zeile bearbeiten
;
m15600:	tst	r25			;Listenende, leerer Bildschirm?
	breq	m15610			;ja -> weiter
	jmp	m20000			;sonst Ende
;
m15610:	ldi	r20,2			;X-Position fr Ende-Text
	ldi	r21,16			;Y-Position fr Ende-Text
	ldi	r22,0xb2		;Pfeil rechts (hohl)
	call	asclcd			;ausgeben
	inc	r20			;2 Pixel Zwischenraum
	ldi	r22,0x42		;Text "Listenende"
	call	strlcd			;ausgeben
m15620:	jmp	m20000
;
; Anzeigemodus Status: Anzeige komplett neu ausgeben
;
m16000:	cpi	r16,2			;Anzeigemodus Status?
	breq	m16010			;ja -> weiter
	rjmp	m17000			;sonst weiter testen
;
m16010:	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,1			;Anzeige neu ausgeben?
	brts	m16020			;ja -> weiter
	rjmp	m16200			;sonst Datenfelder ausgeben
;
m16020:	cbr	r16,0x02		;Neuausgabe-Flag lschen
	sts	dflags,r16		;und Flags wieder speichern
	clr	txflag			;Text-Flag lschen
;
	ldi	r18,0			;X=1
m16030:	ldi	r19,43|0x80		;Y=43, oberer Trennlinie
	call	pixlcd			;waagerechte Linie zeichnen
	ldi	r19,95|0x80		;Y=95, untere Trennlinie
	call	pixlcd			;waagerechte Linie zeichnen
	inc	r18			;X-Koordinate erhhen
	cpi	r18,239			;alle Pixel ausgegeben?
	brne	m16030			;nein -> Schleife
;
	ldi	r20,2			;X-Position Versionsnummer
	ldi	r21,18			;Y-Position Versionsnummer
	ldi	r22,8			;Versionsnummer-Text
	call	strlcd			;ausgeben
	ldi	r20,2			;X-Position Sensorbelegung
	ldi	r21,30			;Y-Position Sensorbelegung
	ldi	r22,0x39		;Text "Sensorbelegung:"
	call	strlcd			;ausgeben
;
	ldi	r20,2			;X-Position EEPROM1
	ldi	r21,46			;Y-Position EEPROM1
	ldi	r22,0x30		;Text "EEPROM1:"
	call	strlcd			;ausgeben
	ldi	r20,2			;X-Position EEPROM2
	ldi	r21,58			;Y-Position EEPROM2
	ldi	r22,0x31		;Text "EEPROM2:"
	call	strlcd			;ausgeben
	ldi	r20,2			;X-Position Speicher
	ldi	r21,70			;Y-Position Speicher
	ldi	r22,0x32		;Text "Speicher:"
	call	strlcd			;ausgeben
	ldi	r20,2			;X-Position RTC
	ldi	r21,82			;Y-Position RTC
	ldi	r22,0x36		;Text "RTC:"
	call	strlcd			;ausgeben
;
	ldi	r20,100			;X-Position Pointer1
	ldi	r21,46			;Y-Position Pointer1
	ldi	r22,0x33		;Text "Pointer1:"
	call	strlcd			;ausgeben
	ldi	r20,100			;X-Position Pointer2
	ldi	r21,58			;Y-Position Pointer2
	ldi	r22,0x34		;Text "Pointer2:"
	call	strlcd			;ausgeben
	ldi	r20,100			;X-Position Pointer3
	ldi	r21,70			;Y-Position Pointer3
	ldi	r22,0x35		;Text "Pointer3:"
	call	strlcd			;ausgeben
	ldi	r20,100			;X-Position Batterie
	ldi	r21,82			;Y-Position Batterie
	ldi	r22,0x37		;Text "Batterie:"
	call	strlcd			;ausgeben
;
	ldi	r20,2			;X-Position fr LED-Text
	ldi	r21,99			;Y-Position fr LED-Text
	ldi	r22,0x38		;Text "LED1/LED2"
	call	strlcd			;ausgeben
;
; Anzeigemodus Status: Datenfelder neu ausgeben
;
m16200:	lds	r16,xflag2		;verschiedene Flags 2 holen
	bst	r16,4			;Datenfelder neu ausgeben?
	brts	m16210			;ja -> weiter
	jmp	m20000			;sonst Ende
;
m16210:	cbr	r16,1<<4		;Datenfelder-Flag lschen
	sts	xflag2,r16		;Flags wieder speichern
	clr	txflag			;Text-Flag lschen
;
	ldi	r20,100			;X-Position Sensorbelegung
	ldi	r21,30			;Y-Position Sensorbelegung
	ldi	xl,low(senmap)
	ldi	xh,high(senmap)		;Zeiger auf Sensorbelegung
	ld	r22,x+			;erstes Zeichen holen
	tst	r22			;Sensorbelegung gltig?
	breq	m16230			;nein -> Meldung ausgeben
	ldi	r25,8			;sonst 8 Zeichen ausgeben
m16220:	call	asclcd			;Zeichen ausgeben
	dec	r25			;alle Zeichen gesendet
	breq	m16240			;ja -> Ende
	ld	r22,x+			;sonst nchstes Zeichen holen
	rjmp	m16220			;Schleife
m16230:	ldi	r22,0x3a		;Text "nicht vorhanden"
	call	strlcd			;ausgeben
m16240:	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,200			;Endwert X setzen
	ldi	r21,41			;Endwert Y setzen
	call	clrwnd			;Bereich lschen
;
	ldi	r20,60			;X-Position EEPROM1
	ldi	r21,46			;Y-Position EEPROM1
	ldi	r22,0xae		;Ok-Symbol
	lds	r16,xflags		;verschiedene Flags holen
	bst	r16,6			;EEPROM1 vorhanden?
	brts	m16300			;ja -> weiter
	ldi	r22,0xa4		;sonst Nicht-Ok-Symbol
m16300:	call	asclcd			;ausgeben
;
	ldi	r20,60			;X-Position EEPROM2
	ldi	r21,58			;Y-Position EEPROM2
	ldi	r22,0xae		;Ok-Symbol
	lds	r16,xflags		;verschiedene Flags holen
	bst	r16,7			;EEPROM2 vorhanden?
	brts	m16350			;ja -> weiter
	ldi	r22,0xa4		;sonst Nicht-Ok-Symbol
m16350:	call	asclcd			;ausgeben
;
	ldi	r20,60			;X-Position Speicher
	ldi	r21,70			;Y-Position Speicher
	ldi	r22,0xae		;Ok-Symbol
	lds	r16,errflg		;verschiedene Flags holen
	bst	r16,3			;Speicherfehler?
	brtc	m16400			;nein -> weiter
	ldi	r22,0xa4		;sonst Nicht-Ok-Symbol
m16400:	call	asclcd			;ausgeben
;
	ldi	r20,60			;X-Position RTC
	ldi	r21,82			;Y-Position RTC
	ldi	r22,0xae		;Ok-Symbol
	lds	r16,errflg		;verschiedene Flags holen
	bst	r16,4			;RTC-Fehler?
	brtc	m16450			;nein -> weiter
	ldi	r22,0xa4		;sonst Nicht-Ok-Symbol
m16450:	call	asclcd			;ausgeben
;
	ldi	r20,158			;X-Position Pointer1 Wert
	ldi	r21,46			;Y-Position Pointer1 Wert
	ldi	yl,low(xet1po)
	ldi	yh,high(xet1po)		;Zeiger auf Pointer1
	call	poilcd			;Pointer1 ausgeben
	ldi	r20,200			;X-Position Pointer1 Status
	ldi	r22,0xae		;Ok-Symbol
	lds	r16,xflag2		;verschiedene Flags 2 holen
	bst	r16,1			;Pointer1 Fehler?
	brtc	m16500			;nein -> weiter
	ldi	r22,0xa4		;sonst Nicht-Ok-Symbol
m16500:	call	asclcd			;ausgeben
;
	ldi	r20,158			;X-Position Pointer2 Wert
	ldi	r21,58			;Y-Position Pointer2 Wert
	ldi	yl,low(xet2po)
	ldi	yh,high(xet2po)		;Zeiger auf Pointer2
	call	poilcd			;Pointer2 ausgeben
	ldi	r20,200			;X-Position Pointer2 Status
	ldi	r22,0xae		;Ok-Symbol
	lds	r16,xflag2		;verschiedene Flags 2 holen
	bst	r16,2			;Pointer2 Fehler?
	brtc	m16550			;nein -> weiter
	ldi	r22,0xa4		;sonst Nicht-Ok-Symbol
m16550:	call	asclcd			;ausgeben
;
	ldi	r20,158			;X-Position Pointer3 Wert
	ldi	r21,70			;Y-Position Pointer3 Wert
	ldi	yl,low(xemspo)
	ldi	yh,high(xemspo)		;Zeiger auf Pointer3
	call	poilcd			;Pointer3 ausgeben
	ldi	r20,200			;X-Position Pointer3 Status
	ldi	r22,0xae		;Ok-Symbol
	lds	r16,xflag2		;verschiedene Flags 2 holen
	bst	r16,3			;Pointer3 Fehler?
	brtc	m16600			;nein -> weiter
	ldi	r22,0xa4		;sonst Nicht-Ok-Symbol
m16600:	call	asclcd			;ausgeben
;
; Berechnung der Batteriespannung nach folgender Formel:
; U= Batteriewert * 1,23 / Referenzwert; zur Reduzierung von Schwan-
; kungen werden die letzten 4 Messwerte (Batt und Ref) addiert
;
	lds	r16,adcbat
	lds	r17,adcbat+1		;Batterie-Wert 0 holen
	lds	r18,adcbat+2
	lds	r19,adcbat+3		;Batterie-Wert 1 holen
	add	r16,r18
	adc	r17,r19			;addieren
	lds	r18,adcbat+4
	lds	r19,adcbat+5		;Batterie-Wert 2 holen
	add	r16,r18
	adc	r17,r19			;addieren
	lds	r18,adcbat+6
	lds	r19,adcbat+7		;Batterie-Wert 3 holen
	add	r16,r18			;addieren
	adc	r17,r19			;Ergebnis= 4*Batterie-Wert
;
	ldi	r18,low(123)
	ldi	r19,high(123)		;Referenzspannung (1,23V) laden
	call	multip			;mit 4*Batterie-Wert multipliz.
	movw	r16,r18
	mov	r18,r20			;Ergebnis kopieren
;
	lds	r20,adcref
	lds	r21,adcref+1		;Referenz-Wert 0 holen
	lds	r22,adcref+2
	lds	r23,adcref+3		;Referenz-Wert 1 holen
	add	r20,r22
	adc	r21,r23			;addieren
	lds	r22,adcref+4
	lds	r23,adcref+5		;Referenz-Wert 2 holen
	add	r20,r22
	adc	r21,r23			;addieren
	lds	r22,adcref+6
	lds	r23,adcref+7		;Referenz-Wert 3 holen
	add	r20,r22			;addieren
	adc	r21,r23			;Ergebnis= 4*Referenz-Wert
	call	divide			;Ergebnis= Spannung in 1/100V
;
	clr	r25			;Merker fr Batterieschwellwert
	ldi	r18,low(batlim)
	ldi	r19,high(batlim)	;Batterieschwellwert laden
	sub	r18,r16
	sbc	r19,r17			;Schwellwert unterschritten?
	brcs	m16700			;nein -> weiter
	breq	m16700			;nein -> weiter
	inc	r25			;sonst Merker setzen
m16700:	call	temasc			;Hexwert in ASCII wandeln
	ldi	r20,158			;X-Position Batteriespannung
	ldi	r21,82			;Y-Position Batteriespannung
	mov	r22,r5			;Einerstelle kopieren
	call	asclcd			;ausgeben
	ldi	r22,','			;Komma laden
	call	asclcd			;ausgeben
	mov	r22,r6			;Zehntelstelle kopieren
	call	asclcd			;ausgeben
	mov	r22,r7			;Hundertstelstelle kopieren
	call	asclcd			;ausgeben
	ldi	r22,'V'			;'V' laden
	call	asclcd			;ausgeben
;
	ldi	r20,200			;X-Position Batterie Status
	ldi	r22,0xae		;Ok-Symbol
	tst	r25			;Batterieschwelle unterschr.?
	breq	m16750			;nein -> weiter
	ldi	r22,0xa4		;sonst Nicht-Ok-Symbol
m16750:	call	asclcd			;ausgeben
;
	ldi	r20,100			;X-Position LED-Modus
	ldi	r21,99			;Y-Position LED-Modus
	ldi	r22,0x40		;Text fr LED (normal)
	lds	r16,confg2		;Konfigurationsbyte 2 holen
	bst	r16,0			;LED-Modus normal?
	brtc	m16760			;ja -> weiter
	ldi	r22,0x41		;sonst Text fr LED (Debug)
m16760:	call	strlcd			;ausgeben
	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,200			;Endwert X setzen
	ldi	r21,110			;Endwert Y setzen
	call	clrwnd			;Bereich lschen
	jmp	m20000
;
; Anzeigemodus Konfig/Temp/Sensoren: Anzeige komplett neu ausgeben
;
m17000:	cpi	r16,3			;Anzeigemodus Temp/Pos/Auswhl?
	breq	m17010			;ja -> weiter
	cpi	r16,4			;Anzeigemodus Temp/Pos/Versch.?
	breq	m17010			;ja -> weiter
	cpi	r16,5			;Anzeigemodus Temp/Pos/Namen?
	breq	m17010			;ja -> weiter
	rjmp	m17500			;sonst weiter testen
;
m17010:	clr	txflag			;Text-Flag lschen
	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,1			;Anzeige neu ausgeben?
	brtc	m17100			;nein -> weiter
	cbr	r16,0x02		;sonst Neuausgabe-Flag lschen
	sts	dflags,r16		;und Flags wieder speichern
;
	ldi	r19,15|0x80		;Y=15
m17030:	ldi	r18,119			;X=119, Trennlinie
	call	pixlcd			;senkrechte Linie zeichnen
	inc	r19			;Y-Koordinate erhhen
	cpi	r19,112|0x80		;alle Pixel ausgegeben?
	brne	m17030			;nein -> Schleife
;
	call	newtem			;alle Namen/Werte neu ausgeben
;
; Anzeigemodus Konfig/Temp/Sensoren: Sensornamen ausgeben
;
m17100:	ldi	r25,0			;Sensornummerzhler = 0
	ldi	xl,low(tembak)
	ldi	xh,high(tembak)		;Zeiger auf Mapping Tabelle
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
m17110:	ld	r16,y			;Temperaturflags holen
	bst	r16,1			;Name neu ausgeben?
	brts	m17120			;ja -> weiter
	rjmp	m17195			;sonst berspringen
;
m17120:	cbr	r16,0x02		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
	ld	r18,x			;Anzeigeposition holen
	mov	r11,r18			;Anzeigeposition sichern
	ldi	r20,2			;X-Startposition Spalte 1
	ldi	r21,16			;Y-Startposition
	cpi	r18,8			;Position in Spalte 1?
	brcs	m17130			;ja -> weiter
	ldi	r20,121			;sonst X-Startposition Spalte 2
m17130:	mov	r16,r18			;Anzeigeposition holen
	andi	r16,0x07		;Zeilenposition filtern
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r16,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
	movw	r12,r20			;X/Y-Startpositionen sichern
;
	lds	r19,menpos		;ausgewhlte Position holen
	dec	r19			;korrigieren
	cp	r18,r19			;Position identisch?
	brne	m17140			;nein -> weiter
	ldi	r16,0x01		;sonst
	mov	txflag,r16		;Textflag invers setzen
m17140:	ldi	r22,'S'			;Zeichen 'S' (Sensor)
	call	asclcd			;ausgeben
	mov	r16,r25			;Sensornummer kopieren
	inc	r16			;fr Anzeige korrigieren
	clr	r17			;H-Byte lschen
	call	bytas4			;Wert in 4 Byte ASCII wandeln
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,0x7f		;Endezeichen
	call	asclcd			;Zeichen ausgeben
	clr	txflag
;
	movw	r20,r12			;X/Y-Koordinaten restaurieren
	ldi	r16,22			;Position fr Checkbox
	add	r20,r16			;berechnen
	ldi	r22,0xb4		;Zeichen "Checkbox1 aus"
	ld	r16,y			;Temperaturflags holen
	bst	r16,7			;Sensor inaktiv?
	brts	m17150			;ja -> weiter
	ldi	r22,0xb5		;sonst Zeichen "Checkbox1 ein"
m17150:	lds	r16,bigtmp		;Sensornummer Groanzeige holen
	cp	r16,r25			;stimmt die Nummer berein?
	brne	m17160			;nein -> weiter
	ldi	r16,2			;sonst
	add	r22,r16			;Text Checkbox2 benutzen
m17160:	call	asclcd			;Zeichen ausgeben
;
	movw	r20,r12			;X/Y-Koordinaten restaurieren
	ldi	r16,32			;Position fr Sensorname
	add	r20,r16			;berechnen
	lds	r16,dimode		;Anzeigemodus holen
	cpi	r16,5			;Anzeigemodus Temp/Pos/Namen?
	brne	m17170			;nein -> normal ausgeben
	lds	r16,menpos		;sonst ausgew. Position holen
	dec	r16			;korrigieren
	cp	r16,r11			;Position identisch?
	brne	m17170			;nein -> normal ausgeben
	call	txblcd			;sonst Texteingabepuffer
	rjmp	m17180			;ausgeben
;
m17170:	ldi	r19,9			;Sensornamenlnge im EEPROM
	mul	r25,r19			;Tabellenoffset ermitteln
	ldi	r16,low(etemtx)		;EEPROM-Adresse Sensornamen L
	ldi	r17,high(etemtx)	;EEPROM-Adresse Sensornamen H
	add	r16,r0			;Offset zur Adresse L addieren
	adc	r17,r1			;Offset zur Adresse H addieren
	call	eeplcd			;EEPROM-Text ausgeben
;
m17180:	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,86			;X-Endwert Spalte 1
	ld	r22,x			;Anzeigeposition holen
	cpi	r22,8			;Position in Spalte 1?
	brcs	m17190			;ja -> weiter
	ldi	r20,206			;sonst X-Endwert Spalte 2
m17190:	ldi	r22,11			;zum Y-Endwert 11 Pixel
	add	r21,r22			;addieren
	call	clrwnd			;Bereich lschen
;
m17195:	adiw	xl,1			;Mapping Tabellenzeiger erhhen
	adiw	yl,1			;Flag Tabellenzeiger erhhen
	inc	r25			;nchste Sensornummer
	cpi	r25,16			;alle Sensoren bearbeitet?
	brcc	m17200			;ja -> weiter
	rjmp	m17110			;sonst Schleife
;
; Anzeigemodus Konfig/Temp/Sensoren: Temperaturwerte ausgeben
;
m17200:	ldi	r25,0			;Sensornummerzhler = 0
	ldi	xl,low(tembak)
	ldi	xh,high(tembak)		;Zeiger auf Mapping-Tabelle
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
m17210:	ld	r16,y			;Flags holen
	bst	r16,0			;Temperaturnderung?
	brtc	m17270			;nein -> keine Ausgabe
	cbr	r16,0x01		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
	ld	r18,x			;Anzeigeposition holen
	ldi	r20,87			;X-Startposition Spalte 1
	ldi	r21,16			;Y-Startposition
	cpi	r18,8			;Temperatur in Spalte 1?
	brcs	m17220			;ja -> weiter
	ldi	r20,207			;sonst X-Startposition Spalte 2
m17220:	mov	r16,r18			;Anzeigeposition holen
	andi	r16,0x07		;Zeilenposition filtern
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r16,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
	ldi	r19,2			;Temperaturwert 2 Byte
	mul	r25,r19			;Tabellenoffset ermitteln
	movw	r8,yl			;Flag-Zeiger sichern
;
	ldi	yl,low(temval)
	ldi	yh,high(temval)		;Zeiger auf Temperaturwerte
	add	yl,r0			;Tabellenoffset L addieren
	adc	yh,r1			;Tabellenoffset H addieren
	ld	r16,y			;Temperaturwert L holen
	ldd	r17,y+1			;Temperaturwert H holen
	cpi	r17,0x80		;Temperaturwert gltig?
	breq	m17240			;nein -> Feld lschen
;
	call	temasc			;Wert in ASCII wandeln
	mov	r22,r4			;erste Stelle holen
	call	asclcd			;Zeichen ausgeben
	mov	r22,r5			;zweite Stelle holen
	call	asclcd			;Zeichen ausgeben
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,'.'			;Trennzeichen Punkt
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,1			;Trennzeichen Punkt?
	brtc	m17230			;ja -> weiter
	ldi	r22,','			;sonst Trennzeichen Komma
m17230:	call	asclcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,0xa5		;schmales Grad-Zeichen
	call	asclcd			;Zeichen ausgeben
	rjmp	m17260
;
m17240:	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,119			;X-Endwert Spalte 1
	ld	r22,x			;Anzeigeposition holen
	cpi	r22,8			;Position in Spalte 1?
	brcs	m17250			;ja -> weiter
	ldi	r20,238			;sonst X-Startposition Spalte 2
m17250:	ldi	r22,11			;zum Endwert Y 11 Pixel
	add	r21,r22			;addieren
	call	clrwnd			;Bereich lschen
;
m17260:	movw	yl,r8			;Flag-Zeiger zurckholen
m17270:	adiw	yl,1			;nchstes Temperaturflag
	adiw	xl,1			;Mapping Zeiger erhhen
	inc	r25			;nchste Sensornummer
	cpi	r25,16			;alle Sensoren bearbeitet?
	brcc	m17300			;ja -> weiter
	rjmp	m17210			;sonst Schleife
;
; Anzeigemodus Konfig/Temp/Sensoren: Auswahl bearbeiten
;
m17300:	lds	r16,dimode		;Anzeigemodus holen?
	cpi	r16,3			;Anzeigemodus Temp/Pos/Auswhl?
	brne	m17350			;nein -> weiter
	lds	r16,menpos		;ausgewhlte Position holen
	lds	r17,encval		;Encoderwert holen
	cp	r16,r17			;nderung der Position?
	breq	m17310			;nein -> weiter
;
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
	movw	zl,yl			;Zeiger kopieren
	call	stebak			;Sensornummer suchen
	clr	r19			;H-Byte=0
	add	yl,r18			;Flag-Position ermitteln
	adc	yh,r19			;bertrag
	ld	r16,y			;Temperatur-Flags holen
	sbr	r16,0x02		;Name neu ausgeben
	st	y,r16			;Flags wieder speichern
	lds	r16,encval		;Encoderwert holen
	call	stebak			;Sensornummer suchen
	add	zl,r18			;Flag-Position ermitteln
	adc	zh,r19			;bertrag
	ld	r16,z			;Temperatur-Flags holen
	sbr	r16,0x02		;Name neu ausgeben
	st	z,r16			;Flags wieder speichern
	lds	r16,encval		;Encoder-Wert holen
	sts	menpos,r16		;ausgewhlte Position setzen
m17310:	jmp	m20000
;
; Anzeigemodus Konfig/Temp/Sensoren: Verschiebung bearbeiten
;
m17350:	cpi	r16,4			;Anzeigemodus Temp/Pos/Versch.?
	brne	m17400			;nein -> weiter
	lds	r16,menpos		;ausgewhlte Position holen
	lds	r17,encval		;Encoderwert holen
	cp	r16,r17			;nderung der Position?
	breq	m17400			;nein -> weiter
;
m17360:	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
	movw	zl,yl			;Zeiger kopieren
	call	stebak			;Sensornummer suchen
	mov	r20,r18			;Sensornummer 1 sichern
	clr	r19			;H-Byte=0
	add	yl,r18			;Flag-Position ermitteln
	adc	yh,r19			;bertrag
	ld	r16,y			;Temperatur-Flags holen
	cbr	r16,0x7c		;alle Temperaturalarme lschen
	sbr	r16,0x03		;Name und Tempwert neu ausgeben
	st	y,r16			;Flags wieder speichern
	lds	r16,encval		;Encoderwert holen
	call	stebak			;Sensornummer suchen
	mov	r21,r18			;Sensornummer 2 sichern
	add	zl,r18			;Flag-Position ermitteln
	adc	zh,r19			;bertrag
	ld	r16,z			;Temperatur-Flags holen
	cbr	r16,0x7c		;alle Temperaturalarme lschen
	sbr	r16,0x03		;Name und Tempwert neu ausgeben
	st	z,r16			;Flags wieder speichern
;
	ldi	yl,low(tembak)
	ldi	yh,high(tembak)		;Zeiger auf Mapping-Backup
	movw	zl,yl			;Zeiger kopieren
	mov	r18,r20			;Sensornummer 1 holen
	add	yl,r18			;Tabellenplatz ermitteln
	adc	yh,r19			;bertrag
	mov	r18,r21			;Sensornummer 2 holen
	add	zl,r18			;Tabellenplatz ermitteln
	adc	yh,r19			;bertrag
	ld	r16,y			;Anzeigeposition 1 holen
	ld	r17,z			;Anzeigeposition 2 holen
	st	y,r17			;Positionen tauschen und
	st	z,r16			;speichern
	lds	r16,encval		;Encoder-Wert holen
	sts	menpos,r16		;ausgewhlte Position setzen
m17400:	jmp	m20000
;
; Anzeigemodus Konfig/Temp/Grenzwerte: Anzeige komplett neu ausgeben
;
m17500:	cpi	r16,6			;Anzeigemodus Temp/Pos/Grenz/M?
	breq	m17510			;ja -> weiter
	cpi	r16,7			;Anzeigemodus Temp/Pos/Grenz/?
	breq	m17510			;ja -> weiter
	rjmp	m18000			;sonst weiter testen
;
m17510:	clr	txflag			;Text-Flag lschen
	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,1			;Anzeige neu ausgeben?
	brtc	m17600			;nein -> weiter
	cbr	r16,0x02		;sonst Neuausgabe-Flag lschen
	sts	dflags,r16		;und Flags wieder speichern
;
	ldi	r19,15|0x80		;Y=15
m17520:	ldi	r18,119			;X=119, Trennlinie
	call	pixlcd			;senkrechte Linie zeichnen
	inc	r19			;Y-Koordinate erhhen
	cpi	r19,112|0x80		;alle Pixel ausgegeben?
	brne	m17520			;nein -> Schleife
;
	call	newtem			;alle Namen/Werte neu ausgeben
;
; Anzeigemodus Konfig/Temp/Grenzwerte: Sensornamen ausgeben
;
m17600:	ldi	r25,0			;Sensornummerzhler = 0
	ldi	xl,low(temmap)
	ldi	xh,high(temmap)		;Zeiger auf Mapping Tabelle
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
;
m17610:	ld	r16,y			;Temperaturflags holen
	bst	r16,1			;Name neu ausgeben?
	brtc	m17640			;nein -> berspringen
	cbr	r16,0x02		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
	ld	r18,x			;Anzeigeposition holen
	ldi	r20,2			;X-Startposition Spalte 1
	ldi	r21,16			;Y-Startposition
	cpi	r18,8			;Position in Spalte 1?
	brcs	m17620			;ja -> weiter
	ldi	r20,121			;sonst X-Startposition Spalte 2
m17620:	mov	r16,r18			;Anzeigeposition holen
	andi	r16,0x07		;Zeilenposition filtern
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r16,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
;
	lds	r19,menpos		;ausgewhlte Position holen
	dec	r19			;korrigieren
	cp	r18,r19			;Position identisch?
	brne	m17630			;nein -> weiter
	ldi	r16,0x01		;sonst
	mov	txflag,r16		;Textflag invers setzen
m17630:	ldi	r19,9			;Sensornamenlnge im EEPROM
	mul	r25,r19			;Tabellenoffset ermitteln
	ldi	r16,low(etemtx)		;EEPROM-Adresse Sensornamen L
	ldi	r17,high(etemtx)	;EEPROM-Adresse Sensornamen H
	add	r16,r0			;Offset zur Adresse L addieren
	adc	r17,r1			;Offset zur Adresse H addieren
	call	eeplcd			;EEPROM-Text ausgeben
	ldi	r22,0x7f		;Ende-Zeichen
	call	asclcd			;Zeichen ausgeben
	clr	txflag			;Textflags lschen
;
m17640:	adiw	xl,1			;Mapping Tabellenzeiger erhhen
	adiw	yl,1			;Flag Tabellenzeiger erhhen
	inc	r25			;nchste Sensornummer
	cpi	r25,16			;alle Sensoren bearbeitet?
	brcs	m17610			;nein -> Schleife
;
; Anzeigemodus Konfig/Temp/Grenzwerte: Grenzwerte ausgeben
;
m17700:	ldi	r25,0			;Sensornummerzhler = 0
	ldi	xl,low(temmap)
	ldi	xh,high(temmap)		;Zeiger auf Mapping-Tabelle
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
;
m17710:	ld	r18,x			;Anzeigeposition holen
	lds	r16,menpos		;Menposition holen
	dec	r16			;korrigieren
	cp	r16,r18			;Anzeigeposition=Menposition?
	brne	m17720			;nein -> nderungsflag prfen
	lds	r16,xflag4		;verschiedene Flags 4 holen
	andi	r16,0x03		;UG oder OG nderung?
	breq	m17720			;nein -> nderungsflag prfen
	lds	r16,encflg		;sonst Encoder-Flags holen
	bst	r16,7			;Encoder-nderung?
	brts	m17740			;ja -> Wert ausgeben
;
m17720:	ld	r16,y			;Flags holen
	bst	r16,0			;Temperaturnderung?
	brts	m17730			;ja -> ausgeben
	rjmp	m17880			;sonst keine Ausgabe
;
m17730:	cbr	r16,0x01		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
m17740:	ldi	r20,52			;X-Startposition Spalte 1
	ldi	r21,16			;Y-Startposition
	cpi	r18,8			;Temperatur in Spalte 1?
	brcs	m17750			;ja -> weiter
	ldi	r20,171			;sonst X-Startposition Spalte 2
m17750:	mov	r16,r18			;Anzeigeposition holen
	andi	r16,0x07		;Zeilenposition filtern
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r16,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
	movw	r10,r20			;X/Y-Startpositionen sichern
	movw	r8,yl			;Flag-Zeiger sichern
;
	lds	r16,xflag4		;verschiedene Flags 4 holen
	bst	r16,0			;Unteren Grenzwert ndern?
	brtc	m17760			;nein -> Wert normal ausgeben
	lds	r16,encflg		;sonst Encoder-Flags holen
	bst	r16,7			;Encoder-nderung?
	brtc	m17760			;nein -> Wert normal ausgeben
	cbr	r16,1<<7		;sonst nderungs-Flag lschen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	lds	r16,encval		;Encoder-Wert L holen
	lds	r17,encvah		;Encoder-Wert H holen
	rjmp	m17770			;Encoder-Wert ausgeben
;
m17760:	ldi	r19,2			;Temperaturwert 2 Byte
	mul	r25,r19			;Tabellenoffset ermitteln
	ldi	yl,low(temlil)
	ldi	yh,high(temlil)		;Zeiger auf untere Grenzwerte
	add	yl,r0			;Tabellenoffset L addieren
	adc	yh,r1			;Tabellenoffset H addieren
	ld	r16,y			;Temperaturwert L holen
	ldd	r17,y+1			;Temperaturwert H holen
m17770:	cpi	r17,0x80		;Temperaturwert gltig?
	breq	m17790			;nein -> Feld lschen
	call	temasc			;Wert in ASCII wandeln
	mov	r22,r4			;erste Stelle holen
	call	asclcd			;Zeichen ausgeben
	mov	r22,r5			;zweite Stelle holen
	call	asclcd			;Zeichen ausgeben
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,'.'			;Trennzeichen Punkt
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,1			;Trennzeichen Punkt?
	brtc	m17780			;ja -> weiter
	ldi	r22,','			;sonst Trennzeichen Komma
m17780:	call	asclcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,0xab		;schmales Grad-Zeichen
	call	asclcd			;Zeichen ausgeben
	rjmp	m17810
;
m17790:	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,86			;X-Endwert Spalte 1
	ld	r22,x			;Anzeigeposition holen
	cpi	r22,8			;Position in Spalte 1?
	brcs	m17800			;ja -> weiter
	ldi	r20,205			;sonst X-Startposition Spalte 2
m17800:	ldi	r22,11			;zum Endwert Y 11 Pixel
	add	r21,r22			;addieren
	call	clrwnd			;Bereich lschen
;
m17810:	movw	r20,r10			;X/Y-Koordinaten restaurieren
	ldi	r16,34			;X-Position fr obere Limits
	add	r20,r16			;berechnen
	lds	r16,xflag4		;verschiedene Flags 4 holen
	bst	r16,1			;Oberen Grenzwert ndern?
	brtc	m17820			;nein -> Wert normal ausgeben
	lds	r16,encflg		;sonst Encoder-Flags holen
	bst	r16,7			;Encoder-nderung?
	brtc	m17820			;nein -> Wert normal ausgeben
	cbr	r16,1<<7		;sonst nderungs-Flag lschen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	lds	r16,encval		;Encoder-Wert L holen
	lds	r17,encvah		;Encoder-Wert H holen
	rjmp	m17830			;Encoder-Wert ausgeben
;
m17820:	ldi	r19,2			;Temperaturwert 2 Byte
	mul	r25,r19			;Tabellenoffset ermitteln
	ldi	yl,low(temlih)
	ldi	yh,high(temlih)		;Zeiger auf obere Grenzwerte
	add	yl,r0			;Tabellenoffset L addieren
	adc	yh,r1			;Tabellenoffset H addieren
	ld	r16,y			;Temperaturwert L holen
	ldd	r17,y+1			;Temperaturwert H holen
m17830:	cpi	r17,0x80		;Temperaturwert gltig?
	breq	m17850			;nein -> Feld lschen
	call	temasc			;Wert in ASCII wandeln
	mov	r22,r4			;erste Stelle holen
	call	asclcd			;Zeichen ausgeben
	mov	r22,r5			;zweite Stelle holen
	call	asclcd			;Zeichen ausgeben
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,'.'			;Trennzeichen Punkt
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,1			;Trennzeichen Punkt?
	brtc	m17840			;ja -> weiter
	ldi	r22,','			;sonst Trennzeichen Komma
m17840:	call	asclcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,0xab		;schmales Grad-Zeichen
	call	asclcd			;Zeichen ausgeben
	rjmp	m17870
;
m17850:	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,118			;X-Endwert Spalte 1
	ld	r22,x			;Anzeigeposition holen
	cpi	r22,8			;Position in Spalte 1?
	brcs	m17860			;ja -> weiter
	ldi	r20,238			;sonst X-Startposition Spalte 2
m17860:	ldi	r22,11			;zum Endwert Y 11 Pixel
	add	r21,r22			;addieren
	call	clrwnd			;Bereich lschen
;
m17870:	movw	yl,r8			;Flag-Zeiger zurckholen
m17880:	adiw	yl,1			;nchstes Temperaturflag
	adiw	xl,1			;Mapping Zeiger erhhen
	inc	r25			;nchste Sensornummer
	cpi	r25,16			;alle Sensoren bearbeitet?
	brcc	m17900			;ja -> weiter
	rjmp	m17710			;sonst Schleife
;
; Anzeigemodus Konfig/Temp/Grenzwerte: Auswahl bearbeiten
;
m17900:	lds	r16,dimode		;Anzeigemodus holen?
	cpi	r16,6			;Anzeigemodus Temp/Grenz/Ausw.?
	brne	m17910			;nein -> weiter
	lds	r16,menpos		;ausgewhlte Position holen
	lds	r17,encval		;Encoderwert holen
	cp	r16,r17			;nderung der Position?
	breq	m17910			;nein -> weiter
;
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
	movw	zl,yl			;Zeiger kopieren
	call	stemap			;Sensornummer suchen
	clr	r19			;H-Byte=0
	add	yl,r18			;Flag-Position ermitteln
	adc	yh,r19			;bertrag
	ld	r16,y			;Temperatur-Flags holen
	sbr	r16,0x02		;Name neu ausgeben
	st	y,r16			;Flags wieder speichern
	lds	r16,encval		;Encoderwert holen
	call	stemap			;Seonsornummer suchen
	add	zl,r18			;Flag-Position ermitteln
	adc	zh,r19			;bertrag
	ld	r16,z			;Temperatur-Flags holen
	sbr	r16,0x02		;Name neu ausgeben
	st	z,r16			;Flags wieder speichern
	lds	r16,encval		;Encoder-Wert holen
	sts	menpos,r16		;ausgewhlte Position setzen
m17910:	jmp	m20000
;
; Anzeigemodus Konfig/Alarme: Anzeige komplett neu ausgeben
;
m18000:	cpi	r16,8			;Anzeigemodus Alarme/Auswhlen?
	breq	m18010			;ja -> weiter
	cpi	r16,9			;Anzeigemodus Alarme/Verschie.?
	breq	m18010			;ja -> weiter
	cpi	r16,10			;Anzeigemodus Alarme/Namen?
	breq	m18010			;ja -> weiter
	rjmp	m18500			;sonst weiter testen
;
m18010:	clr	txflag			;Text-Flag lschen
	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,1			;Anzeige neu ausgeben?
	brtc	m18100			;nein -> weiter
	cbr	r16,0x02		;sonst Neuausgabe-Flag lschen
	sts	dflags,r16		;und Flags wieder speichern
	call	newalm			;alle Alarmdaten neu ausgeben
;
; Anzeigemodus Konfig/Alarme: Alarmdaten ausgeben
;
m18100:	ldi	r25,0			;Alarmnummerzhler = 0
	ldi	xl,low(almbak)
	ldi	xh,high(almbak)		;Zeiger auf Mapping Tabelle
	ldi	yl,low(almflg)
	ldi	yh,high(almflg)		;Zeiger auf Alarmflags
m18110:	ld	r16,y			;Alarmflags holen
	bst	r16,0			;Daten neu ausgeben?
	brts	m18120			;ja -> weiter
	rjmp	m18180			;sonst berspringen
;
m18120:	cbr	r16,0x01		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
	ld	r18,x			;Anzeigeposition holen
	mov	r11,r18			;Anzeigeposition sichern
	ldi	r20,2			;X-Startposition
	ldi	r21,16			;Y-Startposition
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r18,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
	movw	r12,r20			;X/Y-Startpositionen sichern
;
	lds	r19,menpos		;ausgewhlte Position holen
	dec	r19			;korrigieren
	cp	r18,r19			;Position identisch?
	brne	m18130			;nein -> weiter
	ldi	r16,0x01		;sonst
	mov	txflag,r16		;Textflag invers setzen
m18130:	ldi	r22,'A'			;Zeichen 'A' (Alarm)
	call	asclcd			;ausgeben
	ldi	r22,'L'			;Zeichen 'L' (Alarm)
	call	asclcd			;ausgeben
	mov	r16,r25			;Alarmnummer kopieren
	inc	r16			;fr Anzeige korrigieren
	clr	r17			;H-Byte lschen
	call	bytas4			;Wert in 4 Byte ASCII wandeln
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,0x7f		;Endezeichen
	call	asclcd			;Zeichen ausgeben
	clr	txflag
;
	movw	r20,r12			;X/Y-Koordinaten restaurieren
	ldi	r16,22			;Position fr Checkbox
	add	r20,r16			;berechnen
	ldi	r22,0xb4		;Zeichen "Checkbox1 aus"
	ld	r16,y			;Alarmflags holen
	bst	r16,7			;Alarm inaktiv?
	brts	m18140			;ja -> weiter
	ldi	r22,0xb5		;sonst Zeichen "Checkbox1 ein"
m18140:	call	asclcd			;Zeichen ausgeben
;
	movw	r20,r12			;X/Y-Koordinaten restaurieren
	ldi	r16,33			;Position fr Alarmname
	add	r20,r16			;berechnen
	lds	r16,dimode		;Anzeigemodus holen
	cpi	r16,10			;Anzeigemodus Alarme/Namen?
	brne	m18150			;nein -> normal ausgeben
	lds	r16,menpos		;sonst ausgew. Position holen
	dec	r16			;korrigieren
	cp	r16,r11			;Position identisch?
	brne	m18150			;nein -> normal ausgeben
	call	txblcd			;sonst Texteingabepuffer
	rjmp	m18160			;ausgeben
;
m18150:	ld	r16,y			;Alarmflags holen
	bst	r16,1			;Alarm aktiv?
	brtc	m18155			;nein -> weiter
	ldi	r16,1
	mov	txflag,r16		;sonst Text invers darstellen
m18155:	ldi	r19,11			;Alarmnamenlnge im EEPROM
	mul	r25,r19			;Tabellenoffset ermitteln
	ldi	r16,low(ealmtx)		;EEPROM-Adresse Sensornamen L
	ldi	r17,high(ealmtx)	;EEPROM-Adresse Sensornamen H
	add	r16,r0			;Offset zur Adresse L addieren
	adc	r17,r1			;Offset zur Adresse H addieren
	call	eeplcd			;EEPROM-Text ausgeben
	ldi	r22,0x7f		;Endezeichen
	call	asclcd			;Zeichen ausgeben
	clr	txflag
;
m18160:	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,99			;X-Endwert
	ldi	r22,11			;zum Endwert Y 11 Pixel
	add	r21,r22			;addieren
	call	clrwnd			;Bereich lschen
;
	movw	r20,r12			;X/Y-Koordinaten restaurieren
	ldi	r16,99			;Position fr Alarminvertierung
	add	r20,r16			;berechnen
	ldi	zl,low(alminv)
	ldi	zh,high(alminv)		;Zeiger auf Alarm-Invertierung
	clr	r16
	add	zl,r25			;Tabellenplatz berechnen
	adc	zh,r16			;bertrag
	ldi	r22,0x62		;Text "normal"
	ld	r16,z			;Alarm-Invertierung holen
	tst	r16			;Alarm invertiert?
	breq	m18170			;nein -> weiter
	ldi	r22,0x63		;sonst Text "invertiert"
m18170:	call	strlcd			;ausgeben
;
	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,200			;X-Endwert
	ldi	r22,11			;zum Endwert Y 11 Pixel
	add	r21,r22			;addieren
	call	clrwnd			;Bereich lschen
;
m18180:	adiw	xl,1			;Mapping Tabellenzeiger erhhen
	adiw	yl,1			;Flag Tabellenzeiger erhhen
	inc	r25			;nchste Alarmnummer
	cpi	r25,4			;alle Alarme bearbeitet?
	brcc	m18300			;ja -> weiter
	rjmp	m18110			;sonst Schleife
;
; Anzeigemodus Konfig/Alarme: Auswahl bearbeiten
;
m18300:	lds	r16,dimode		;Anzeigemodus holen?
	cpi	r16,8			;Anzeigemodus Alarme/Auswhlen?
	brne	m18350			;nein -> weiter
	lds	r16,menpos		;ausgewhlte Position holen
	lds	r17,encval		;Encoderwert holen
	cp	r16,r17			;nderung der Position?
	breq	m18310			;nein -> weiter
;
	ldi	yl,low(almflg)
	ldi	yh,high(almflg)		;Zeiger auf Alarmflags
	movw	zl,yl			;Zeiger kopieren
	call	salbak			;Alarmnummer suchen
	clr	r19			;H-Byte=0
	add	yl,r18			;Flag-Position ermitteln
	adc	yh,r19			;bertrag
	ld	r16,y			;Alarm-Flags holen
	sbr	r16,0x01		;Name neu ausgeben
	st	y,r16			;Flags wieder speichern
	lds	r16,encval		;Encoderwert holen
	call	salbak			;Alarmnummer suchen
	add	zl,r18			;Flag-Position ermitteln
	adc	zh,r19			;bertrag
	ld	r16,z			;Alarm-Flags holen
	sbr	r16,0x01		;Name neu ausgeben
	st	z,r16			;Flags wieder speichern
	lds	r16,encval		;Encoder-Wert holen
	sts	menpos,r16		;ausgewhlte Position setzen
m18310:	jmp	m20000
;
; Anzeigemodus Konfig/Alarme: Verschiebung bearbeiten
;
m18350:	cpi	r16,9			;Anzeigemodus Alarme/Verschie.?
	brne	m18400			;nein -> weiter
	lds	r16,menpos		;ausgewhlte Position holen
	lds	r17,encval		;Encoderwert holen
	cp	r16,r17			;nderung der Position?
	breq	m18400			;nein -> weiter
;
m18360:	ldi	yl,low(almflg)
	ldi	yh,high(almflg)		;Zeiger auf Alarmflags
	movw	zl,yl			;Zeiger kopieren
	call	salbak			;Alarmnummer suchen
	mov	r20,r18			;Alarmnummer 1 sichern
	clr	r19			;H-Byte=0
	add	yl,r18			;Flag-Position ermitteln
	adc	yh,r19			;bertrag
	ld	r16,y			;Alarm-Flags holen
	sbr	r16,0x01		;Name neu ausgeben
	st	y,r16			;Flags wieder speichern
	lds	r16,encval		;Encoderwert holen
	call	salbak			;Alarmnummer suchen
	mov	r21,r18			;Alarmnummer 2 sichern
	add	zl,r18			;Flag-Position ermitteln
	adc	zh,r19			;bertrag
	ld	r16,z			;Alarm-Flags holen
	sbr	r16,0x01		;Name neu ausgeben
	st	z,r16			;Flags wieder speichern
;
	ldi	yl,low(almbak)
	ldi	yh,high(almbak)		;Zeiger auf Mapping-Backup
	movw	zl,yl			;Zeiger kopieren
	mov	r18,r20			;Alarmnummer 1 holen
	add	yl,r18			;Tabellenplatz ermitteln
	adc	yh,r19			;bertrag
	mov	r18,r21			;Alarmnummer 2 holen
	add	zl,r18			;Tabellenplatz ermitteln
	adc	yh,r19			;bertrag
	ld	r16,y			;Anzeigeposition 1 holen
	ld	r17,z			;Anzeigeposition 2 holen
	st	y,r17			;Positionen tauschen und
	st	z,r16			;speichern
	lds	r16,encval		;Encoder-Wert holen
	sts	menpos,r16		;ausgewhlte Position setzen
m18400:	jmp	m20000
;
; Anzeigemodus Konfig/Signale/Zuordnung: Anzeige komplett neu ausgeben
;
m18500:	cpi	r16,11			;Anzeigemodus Signale/Auswhl.?
	breq	m18510			;ja -> weiter
	cpi	r16,12
	breq	m18510
	rjmp	m19000			;sonst weiter testen
;
m18510:	clr	txflag			;Text-Flag lschen
	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,1			;Anzeige neu ausgeben?
	brtc	m18600			;nein -> weiter
	cbr	r16,0x02		;sonst Neuausgabe-Flag lschen
	sts	dflags,r16		;und Flags wieder speichern
	call	newcfg			;alle Konfigzeilen neu ausgeben
;
; Anzeigemodus Konfig/Signale/Zuordnung: Namen ausgeben
;
m18600:	ldi	r25,0			;Signal-Zhler = 0
	ldi	yl,low(cfgflg)
	ldi	yh,high(cfgflg)		;Zeiger auf Konfigurationsflags
m18610:	ldi	r20,2			;X-Startposition
	ldi	r21,16			;Y-Startposition
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r25,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
	movw	r12,r20			;X/Y-Startpositionen sichern
;
	ld	r16,y			;Konfigurationflags holen
	bst	r16,0			;Name neu ausgeben?
	brtc	m18650			;nein -> berspringen
	cbr	r16,0x01		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
;
	lds	r19,menpos		;ausgewhlte Position holen
	dec	r19			;korrigieren
	cp	r25,r19			;Position identisch?
	brne	m18620			;nein -> weiter
	ldi	r16,0x01		;sonst
	mov	txflag,r16		;Textflag invers setzen
m18620:	ldi	r22,0x65		;Text "Alarm 1"
	cpi	r25,0			;Zhler auf Alarm 1?
	breq	m18630			;ja -> weiter
	ldi	r22,0x66		;Text "Alarm 2"
	cpi	r25,1			;Zhler auf Alarm 2?
	breq	m18630			;ja -> weiter
	ldi	r22,0x67		;Text "Alarm 3"
	cpi	r25,2			;Zhler auf Alarm 3?
	breq	m18630			;ja -> weiter
	ldi	r22,0x68		;Text "Alarm 4"
	cpi	r25,3			;Zhler auf Alarm 4?
	breq	m18630			;ja -> weiter
	ldi	r22,0xfb		;Text "Temp zu hoch"
	cpi	r25,4			;Zhler auf bertemperatur?
	breq	m18630			;ja -> weiter
	ldi	r22,0xfc		;Text "Temp zu niedrig"
	cpi	r25,5			;Zhler auf Untertemperatur?
	breq	m18630			;ja -> weiter
	ldi	r22,0x69		;Text "Systemfehler"
	cpi	r25,6			;Zhler auf Systemfehler?
	breq	m18630			;ja -> weiter
	ldi	r22,0xf9		;sonst Text "Sensorausfall"
m18630:	call	strlcd			;Text ausgeben
	clr	txflag
;
; Anzeigemodus Konfig/Signale/Zuordnung: Nummer und Dauer ausgeben
;
m18650:	lds	r16,menpos		;Menposition holen
	dec	r16			;korrigieren
	cp	r16,r25			;Anzeigeposition=Menposition?
	brne	m18660			;nein -> nderungsflag prfen
	lds	r16,xflag4		;verschiedene Flags 4 holen
	andi	r16,0x0c		;Nummer oder Dauer ndern?
	breq	m18660			;nein -> nderungsflag prfen
	lds	r16,encflg		;sonst Encoder-Flags holen
	bst	r16,7			;Encoder-nderung?
	brts	m18680			;ja -> Wert ausgeben
;
m18660:	ld	r16,y			;Konfigurationflags holen
	bst	r16,1			;Daten neu ausgeben?
	brts	m18670			;ja -> weiter
	rjmp	m18780			;sonst berspringen
;
m18670:	cbr	r16,0x02		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
m18680:	movw	r20,r12			;X/Y-Koordinaten restaurieren
	ldi	r20,84			;Position fr Signalnummer
	ldi	r22,'N'			;Zeichen "N" (Nummer)
	call	asclcd			;ausgeben
	ldi	r22,':'			;Doppelpunkt
	call	asclcd			;ausgeben
;
	lds	r16,xflag4		;verschiedene Flags 4 holen
	bst	r16,2			;Signalnummer ndern?
	brtc	m18690			;nein -> Wert normal ausgeben
	lds	r16,encflg		;sonst Encoder-Flags holen
	bst	r16,7			;Encoder-nderung?
	brtc	m18690			;nein -> Wert normal ausgeben
	cbr	r16,1<<7		;sonst nderungs-Flag lschen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	lds	r16,encval		;Encoder-Wert holen
	dec	r16			;korrigieren
	rjmp	m18700			;Encoder-Wert ausgeben
;
m18690:	ldi	xl,low(signum)
	ldi	xh,high(signum)		;Zeiger auf Signalnummer-Tab.
	clr	r17
	add	xl,r25			;Tabellenplatz berechnen
	adc	xh,r17			;bertrag
	ld	r16,x			;Signalnummer holen
m18700:	mov	r11,r16			;Signalnummer sichern
	inc	r16			;fr Anzeige korrigieren
	clr	r17			;H-Byte lschen
	call	bytas4			;Wert in 4 Byte ASCII wandeln
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
;
	movw	r20,r12			;X/Y-Koordinaten restaurieren
	ldi	r20,110			;Position fr Signaldauer
	ldi	r22,'D'			;Zeichen "D" (Dauer)
	call	asclcd			;ausgeben
	ldi	r22,':'			;Doppelpunkt
	call	asclcd			;ausgeben
;
	lds	r16,xflag4		;verschiedene Flags 4 holen
	bst	r16,3			;Signaldauer ndern?
	brtc	m18710			;nein -> Wert normal ausgeben
	lds	r16,encflg		;sonst Encoder-Flags holen
	bst	r16,7			;Encoder-nderung?
	brtc	m18710			;nein -> Wert normal ausgeben
	cbr	r16,1<<7		;sonst nderungs-Flag lschen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	lds	r16,encval		;Encoder-Wert holen
	rjmp	m18720			;Encoder-Wert ausgeben
;
m18710:	ldi	xl,low(sigdur)
	ldi	xh,high(sigdur)		;Zeiger auf Signaldauer-Tabelle
	clr	r17
	add	xl,r25			;Tabellenplatz berechnen
	adc	xh,r17			;bertrag
	ld	r16,x			;Signaldauer holen
m18720:	clr	r17			;H-Byte lschen
	call	bytas4			;Wert in 3 Byte ASCII wandeln
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
;
; Anzeigemodus Konfig/Signale/Zuordnung: Signaldiagramm ausgeben
;
	ldi	r18,136			;X-Startposition fr Grafik
	mov	r19,r13			;Y-Startposition holen
	ldi	r16,3			;Grafik beginnt 3 Pixel tiefer
	add	r19,r16			;Y-Startposition berechnen
	ldi	r20,237			;X-Endposition fr Grafik
	mov	r21,r13			;Y-Startposition holen
	ldi	r16,9			;Grafik endet 9 Pixel tiefer
	add	r21,r16			;Y-Endposition berechnen
	call	clrwnd			;Bildausschnitt lschen
;
	ldi	r18,136			;Position fr Signaldiagramm
	mov	r16,r11			;gesicherte Signalnummer holen
	add	r16,r16
	add	r16,r16
	add	r16,r16			;Tabellenoffset berechnen (x8)
	clr	r17
	ldi	zl,low(sndtab*2)
	ldi	zh,high(sndtab*2)	;Zeiger auf Signaltabelle L/H
	add	zl,r16			;Tabellenplatz ermitteln
	adc	zh,r17			;bertrag
	ldi	r16,byte3(sndtab*2)	;Zeiger auf Signaltabelle B3
	adc	r16,r17
	out	rampz,r16
;
	clr	r24			;Zeitmarkenzhler= 0
	clr	r23			;Tabellenzhler= 0
	ldi	r22,8			;Pegelwert= Low
	mov	r19,r13			;gesicherte Y-Position holen
	ldi	r16,8			;erstes Pixel hat Low-Pegel
	add	r19,r16			;addieren
	ori	r19,0x80		;Pixel setzen
	call	pixlcd			;ausgeben
	inc	r18			;nchste X-Position
;
m18730:	elpm	r21,z			;Eintrag aus Signaltabelle hol.
	cp	r24,r21			;Zeitmarke=Eintrag?
	brne	m18770			;nein -> letzten Zustand zeich.
	adiw	zl,1			;sonst nchster Tabelleneintrag
	cpi	r22,8			;Pegelwert=Low?
	breq	m18740			;ja -> auf High setzen
	ldi	r22,8			;sonst auf Low setzen
	rjmp	m18750
m18740:	ldi	r22,3			;Pegelwert auf High setzen
m18750:	ldi	r20,3			;Flanke zeichnen; Y-Offset
m18760:	mov	r19,r13			;gesicherte Y-Position holen
	add	r19,r20			;Offset addieren
	ori	r19,0x80		;Pixel setzen
	call	pixlcd			;ausgeben
	inc	r20			;nchster Y-Wert
	cpi	r20,9			;Endwert erreicht?
	brne	m18760			;nein -> Schleife
;
m18770:	mov	r19,r13			;gesicherte Y-Position holen
	add	r19,r22			;Pegel addieren
	ori	r19,0x80		;Pixel setzen
	call	pixlcd			;ausgeben
	inc	r18			;nchste X-Position
	call	pixlcd			;ausgeben
	inc	r18			;nchste X-Position
	inc	r24			;nchste Zeitmarke
	cpi	r24,50			;letzte Zeitmarke erreicht?
	brcs	m18730			;nein -> Schleife
;
m18780:	adiw	yl,1			;Flag Tabellenzeiger erhhen
	inc	r25			;nchste Alarmnummer
	cpi	r25,8			;alle Signale bearbeitet?
	brcc	m18800			;ja -> weiter
	rjmp	m18610			;sonst Schleife
;
; Anzeigemodus Konfig/Signale/Zuordnung: Auswahl bearbeiten
;
m18800:	lds	r16,dimode		;Anzeigemodus holen?
	cpi	r16,11			;Anzeigemodus Signale/Auswhl.?
	brne	m18810			;nein -> weiter
	lds	r16,menpos		;ausgewhlte Position holen
	lds	r17,encval		;Encoderwert holen
	cp	r16,r17			;nderung der Position?
	breq	m18810			;nein -> weiter
;
	ldi	xl,low(cfgflg)
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	movw	yl,xl			;Zeiger kopieren
	dec	r16			;Positionswert korrigieren
	clr	r19
	add	xl,r16			;Flag-Position ermitteln
	adc	xh,r19			;bertrag
	ld	r16,x			;Alarm-Flags holen
	sbr	r16,0x01		;Name neu ausgeben
	st	x,r16			;Flags wieder speichern
	dec	r17			;Encoderwert korrigieren
	add	yl,r17			;Flag-Position ermitteln
	adc	yh,r19			;bertrag
	ld	r16,y			;Alarm-Flags holen
	sbr	r16,0x01		;Name neu ausgeben
	st	y,r16			;Flags wieder speichern
	lds	r16,encval		;Encoder-Wert holen
	sts	menpos,r16		;ausgewhlte Position setzen
m18810:	jmp	m20000
;
; Anzeigemodus Konfig/Signale/Audio: Anzeige komplett neu ausgeben
;
m19000:	cpi	r16,13			;Anzeigemodus Signale/Audio?
	breq	m19010			;ja -> weiter
	rjmp	m19400			;sonst weiter testen
;
m19010:	clr	txflag			;Text-Flag lschen
	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,1			;Anzeige neu ausgeben?
	brtc	m19100			;nein -> weiter
	cbr	r16,0x02		;sonst Neuausgabe-Flag lschen
	sts	dflags,r16		;und Flags wieder speichern
	call	newcfg			;alle Konfigzeilen neu ausgeben
;
; Anzeigemodus Konfig/Signale/Audio: Namen ausgeben
;
m19100:	ldi	r25,0			;Zeilen-Zhler = 0
	ldi	yl,low(cfgflg)
	ldi	yh,high(cfgflg)		;Zeiger auf Konfigurationsflags
m19110:	ldi	r20,2			;X-Startposition
	ldi	r21,16			;Y-Startposition
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r25,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
	movw	r12,r20			;X/Y-Startpositionen sichern
;
	ld	r16,y			;Konfigurationflags holen
	bst	r16,0			;Name neu ausgeben?
	brtc	m19150			;nein -> berspringen
	cbr	r16,0x01		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
;
	lds	r19,menpos		;ausgewhlte Position holen
	dec	r19			;korrigieren
	cp	r25,r19			;Position identisch?
	brne	m19120			;nein -> weiter
	ldi	r16,0x01		;sonst
	mov	txflag,r16		;Textflag invers setzen
m19120:	ldi	r22,0x6f		;Text "Signalfrequenz"
	add	r22,r25			;Zhler addieren -> Textnummer
	call	strlcd			;Text ausgeben
	clr	txflag
;
; Anzeigemodus Konfig/Signale/Audio: Daten ausgeben
;
m19150:	ld	r16,y			;Konfigurationflags holen
	bst	r16,1			;Daten neu ausgeben?
	brts	m19160			;ja -> weiter
	rjmp	m19210			;sonst berspringen
;
m19160:	cbr	r16,0x02		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
	movw	r20,r12			;X/Y-Koordinaten restaurieren
	ldi	r20,110			;Position fr Daten
	cpi	r25,0			;Zhler auf Signalfrequenz?
	brne	m19180			;nein -> weiter testen
	movw	r12,r20			;X/Y-Koordinaten sichern
	call	timfrq			;sonst Timer0-Wert in Frequenz
	movw	r20,r12			;X/Y-Koordinaten restaurieren
	call	bytas4			;umrechnen und in ASCII wandeln
	mov	r22,r4			;erste Stelle holen
	cpi	r22,'0'			;erste Stelle = 0?
	breq	m19170			;ja -> berspringen
	call	asclcd			;sonst ausgeben
m19170:	mov	r22,r5			;zweite Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
	ldi	r22,'H'			;Zeichen 'H'
	call	asclcd			;ausgeben
	ldi	r22,'z'			;Zeichen 'z' (Hz)
	call	asclcd			;ausgeben
	ldi	r22,' '			;Leerzeichen
	call	asclcd			;ausgeben
	rjmp	m19210
;
m19180:	lds	r16,sigvo1		;Signal Lautstrke Tag holen
	cpi	r25,1			;Zhler auf Lautstrke Tag?
	breq	m19190			;ja -> weiter
	lds	r16,sigvo2		;sonst Lautstrke Nacht holen
m19190:	clr	r17			;H-Byte lschen
	call	bytas4			;in ASCII wandeln
	mov	r22,r6			;dritte Stelle holen
	cpi	r22,'0'			;dritte Stelle = 0?
	breq	m19200			;ja -> berspringen
	call	asclcd			;sonst ausgeben
m19200:	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,' '			;Leerzeichen
	call	asclcd			;ausgeben
;
m19210:	adiw	yl,1			;Flag Tabellenzeiger erhhen
	inc	r25			;nchste Zeile
	cpi	r25,3			;alle Zeilen bearbeitet?
	brcc	m19300			;ja -> weiter
	rjmp	m19110			;sonst Schleife
;
; Anzeigemodus Konfig/Signale/Audio: Auswahl bearbeiten
;
m19300:	lds	r16,menpos		;ausgewhlte Position holen
	lds	r17,encval		;Encoderwert holen
	cp	r16,r17			;nderung der Position?
	breq	m19310			;nein -> weiter
;
	ldi	xl,low(cfgflg)
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	movw	yl,xl			;Zeiger kopieren
	dec	r16			;Positionswert korrigieren
	clr	r19
	add	xl,r16			;Flag-Position ermitteln
	adc	xh,r19			;bertrag
	ld	r16,x			;Zeilen-Flags holen
	sbr	r16,0x01		;Name neu ausgeben
	st	x,r16			;Flags wieder speichern
	dec	r17			;Encoderwert korrigieren
	add	yl,r17			;Flag-Position ermitteln
	adc	yh,r19			;bertrag
	ld	r16,y			;Zeilen-Flags holen
	sbr	r16,0x01		;Name neu ausgeben
	st	y,r16			;Flags wieder speichern
	lds	r16,encval		;Encoder-Wert holen
	sts	menpos,r16		;ausgewhlte Position setzen
m19310:	jmp	m20000
;
; Anzeigemodus Konfig/Signale/Audio: ndern bearbeiten
;
m19400:	cpi	r16,14			;Anzeigemodus Sig/Audio/ndern?
	breq	m19410			;ja -> weiter
	rjmp	m19500			;sonst weiter testen
;
m19410:	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,7			;Wert gendert?
	brts	m19420			;ja -> ausgeben
	rjmp	m19470			;sonst Ende
;
m19420:	cbr	r16,1<<7		;Encoder nderungs-Flag lschen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	lds	r16,menpos		;ausgewhlte Position holen
	dec	r16			;Positionswert korrigieren
	ldi	r20,110			;X-Startposition
	ldi	r21,16			;Y-Startposition
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r16,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
;
	cpi	r16,0			;Position auf Signalfrequenz?
	brne	m19440			;nein -> Lautstrke bearbeiten
	lds	r16,encval		;Encoder-Wert L holen
	lds	r17,encvah		;Encoder-Wert H holen
	call	bytas4			;in ASCII wandeln
	mov	r22,r4			;erste Stelle holen
	cpi	r22,'0'			;erste Stelle = 0?
	breq	m19430			;ja -> berspringen
	call	asclcd			;sonst ausgeben
m19430:	mov	r22,r5			;zweite Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
	ldi	r22,'H'			;Zeichen 'H'
	call	asclcd			;ausgeben
	ldi	r22,'z'			;Zeichen 'z' (Hz)
	call	asclcd			;ausgeben
	ldi	r22,' '			;Leerzeichen
	call	asclcd			;ausgeben
	rjmp	m19460			;Zeilenende lschen
;
m19440:	lds	r16,encval		;Encoder-Wert holen
	clr	r17			;H-Byte lschen
	call	bytas4			;in ASCII wandeln
	mov	r22,r6			;dritte Stelle holen
	cpi	r22,'0'			;dritte Stelle = 0?
	breq	m19450			;ja -> berspringen
	call	asclcd			;sonst ausgeben
m19450:	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,' '			;Leerzeichen
	call	asclcd			;ausgeben
;
m19460:	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,237			;X-Endwert
	ldi	r22,11			;zum Endwert Y 11 Pixel
	add	r21,r22			;addieren
	call	clrwnd			;Bereich lschen
m19470:	jmp	m20000
;
; Anzeigemodus Konfig/Zeiten/Schaltzeiten: Anzeige komplett neu ausgeb.
;
m19500:	cpi	r16,15			;Anzeigemodus Konfig/Zeiten?
	breq	m19510			;ja -> weiter
	rjmp	m19900			;sonst weiter testen
;
m19510:	clr	txflag			;Text-Flag lschen
	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,1			;Anzeige neu ausgeben?
	brtc	m19600			;nein -> weiter
	cbr	r16,0x02		;sonst Neuausgabe-Flag lschen
	sts	dflags,r16		;und Flags wieder speichern
	call	newcfg			;alle Konfigzeilen neu ausgeben
;
; Anzeigemodus Konfig/Zeiten/Schaltzeiten: Namen ausgeben
;
m19600:	ldi	r25,0			;Zeilen-Zhler = 0
	ldi	yl,low(cfgflg)
	ldi	yh,high(cfgflg)		;Zeiger auf Konfigurationsflags
m19610:	ldi	r20,2			;X-Startposition
	ldi	r21,16			;Y-Startposition
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r25,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
	movw	r12,r20			;X/Y-Startpositionen sichern
;
	ld	r16,y			;Konfigurationflags holen
	bst	r16,0			;Name neu ausgeben?
	brtc	m19700			;nein -> berspringen
	cbr	r16,0x01		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
;
	lds	r19,menpos		;ausgewhlte Position holen
	dec	r19			;korrigieren
	cp	r25,r19			;Position identisch?
	brne	m19620			;nein -> weiter
	ldi	r16,0x01		;sonst
	mov	txflag,r16		;Textflag invers setzen
m19620:	cpi	r25,4			;Zhler > 3?
	brcc	m19630			;ja -> weiter
	ldi	r22,0x74		;Text "Signalmodus Nacht von"
	rjmp	m19650			;Text ausgeben
m19630: cpi	r25,6			;Zhler > 5?
	brcc	m19640			;ja -> weiter
	ldi	r22,0xb1-4		;Text "Bluetooth aus von"
	rjmp	m19650			;Text ausgeben
m19640:	ldi	r22,0x78-6		;Text "Auto-Reset Min-Werte"
m19650:	add	r22,r25			;Zhler addieren -> Textnummer
	call	strlcd			;Text ausgeben
	clr	txflag
;
; Anzeigemodus Konfig/Zeiten/Schaltzeiten: Daten ausgeben
;
m19700:	ld	r16,y			;Konfigurationflags holen
	bst	r16,1			;Daten neu ausgeben?
	brts	m19710			;ja -> weiter
	rjmp	m19790			;sonst berspringen
;
m19710:	cbr	r16,0x02		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
	movw	r20,r12			;X/Y-Koordinaten restaurieren
	ldi	r20,198			;Position fr Checkbox
	ldi	r22,0xb4		;Zeichen "Checkbox1 aus"
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	lds	r17,confg2		;Konfigurationsbyte 2 holen
	bst	r16,5			;Flag Nachtschaltung Signale
	cpi	r25,2			;Zeile 0/1 Nachtsch. Signale?
	brcs	m19720			;ja -> bearbeiten
	bst	r16,3			;Flag Nachtschaltung Licht
	cpi	r25,4			;Zeile 2/3 Nachtschalt. Licht?
	brcs	m19720			;ja -> bearbeiten
	bst	r17,2			;Flag Nachtschaltung Bluetooth
	cpi	r25,6			;Zeile 4/5 Nachtsch. Bluetooth?
	brcs	m19720			;ja -> bearbeiten
	bst	r16,4			;Flag Auto-Reset Min/Max-Werte?
m19720:	brtc	m19730			;Flag gelscht? ja -> weiter
	ldi	r22,0xb5		;sonst Zeichen "Checkbox1 ein"
m19730:	call	asclcd			;Zeichen ausgeben
;
	movw	r20,r12			;X/Y-Koordinaten restaurieren
	ldi	r20,210			;Position fr Daten
	lds	r16,signig
	lds	r17,signig+1		;Zeit Signalmodus Nacht 1 laden
	cpi	r25,0			;Zhler Signalmodus Nacht 1?
	breq	m19740			;ja -> bearbeiten
	lds	r16,signig+2
	lds	r17,signig+3		;Zeit Signalmodus Nacht 2 laden
	cpi	r25,1			;Zhler Signalmodus Nacht 2?
	breq	m19740			;ja -> bearbeiten
	lds	r16,nighbl
	lds	r17,nighbl+1		;Zeit LCD Beleuchtung 1 laden
	cpi	r25,2			;Zhler LCD Beleuchtung 1?
	breq	m19740			;ja -> bearbeiten
	lds	r16,nighbl+2
	lds	r17,nighbl+3		;Zeit LCD Beleuchtung 2 laden
	cpi	r25,3			;Zhler LCD Beleuchtung 2?
	breq	m19740			;ja -> bearbeiten
	lds	r16,nighbt
	lds	r17,nighbt+1		;Zeit Bluetooth 1 laden
	cpi	r25,4			;Zhler Bluetooth Zeit 1?
	breq	m19740			;ja -> bearbeiten
	lds	r16,nighbt+2
	lds	r17,nighbt+3		;Zeit Bluetooth 2 laden
	cpi	r25,5			;Zhler Bluetooth Zeit 2?
	breq	m19740			;ja -> bearbeiten
	lds	r16,aresmm
	lds	r17,aresmm+1		;Zeit Reset Minimalwerte laden
	cpi	r25,6			;Zhler Reset Minimalwerte?
	breq	m19740			;ja -> bearbeiten
	lds	r16,aresmm+2		;sonst
	lds	r17,aresmm+3		;Zeit Reset Maximalwerte laden
;
m19740:	call	timasc			;Zeit in ASCII wandeln
	mov	r22,r4			;erste Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r5			;zweite Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,':'			;Trennzeichen laden
	call	asclcd			;ausgeben
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;sonst ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
;
m19790:	adiw	yl,1			;Flag Tabellenzeiger erhhen
	inc	r25			;nchste Zeile
	cpi	r25,8			;alle Zeilen bearbeitet?
	brcc	m19800			;ja -> weiter
	rjmp	m19610			;sonst Schleife
;
; Anzeigemodus Konfig/Zeiten/Schaltzeiten: Auswahl bearbeiten
;
m19800:	lds	r16,menpos		;ausgewhlte Position holen
	lds	r17,encval		;Encoderwert holen
	cp	r16,r17			;nderung der Position?
	breq	m19810			;nein -> weiter
;
	ldi	xl,low(cfgflg)
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	movw	yl,xl			;Zeiger kopieren
	dec	r16			;Positionswert korrigieren
	clr	r19
	add	xl,r16			;Flag-Position ermitteln
	adc	xh,r19			;bertrag
	ld	r16,x			;Zeilen-Flags holen
	sbr	r16,0x01		;Name neu ausgeben
	st	x,r16			;Flags wieder speichern
	dec	r17			;Encoderwert korrigieren
	add	yl,r17			;Flag-Position ermitteln
	adc	yh,r19			;bertrag
	ld	r16,y			;Zeilen-Flags holen
	sbr	r16,0x01		;Name neu ausgeben
	st	y,r16			;Flags wieder speichern
	lds	r16,encval		;Encoder-Wert holen
	sts	menpos,r16		;ausgewhlte Position setzen
m19810:	jmp	m20000
;
; Anzeigemodus Konfig/Zeiten/Schaltzeiten: ndern bearbeiten
;
m19900:	cpi	r16,16			;Anzeigemodus Konf/Zeit/ndern?
	breq	m19910			;ja -> weiter
	rjmp	m1a000			;sonst weiter testen
;
m19910:	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,7			;Wert gendert?
	brtc	m19920			;nein -> Ende
;
	cbr	r16,1<<7		;Encoder nderungs-Flag lschen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	lds	r16,menpos		;ausgewhlte Position holen
	dec	r16			;Positionswert korrigieren
	ldi	r20,210			;X-Startposition
	ldi	r21,16			;Y-Startposition
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r16,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
;
	lds	r16,encval		;Encoder-Wert L holen
	lds	r17,encvah		;Encoder-Wert H holen
	call	timasc			;Zeit in ASCII wandeln
	mov	r22,r4			;erste Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r5			;zweite Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,':'			;Trennzeichen laden
	call	asclcd			;ausgeben
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;sonst ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
m19920:	jmp	m20000
;
; Anzeigemodus Konfig/Zeiten/Uhr stellen: Anzeige komplett neu ausgeben
;
m1a000:	cpi	r16,17			;Anzeigemodus Zeiten/Uhrzeit?
	breq	m1a010			;ja -> weiter
	rjmp	m1a400			;sonst weiter testen
;
m1a010:	clr	txflag			;Text-Flag lschen
	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,1			;Anzeige neu ausgeben?
	brtc	m1a100			;nein -> weiter
	cbr	r16,0x02		;sonst Neuausgabe-Flag lschen
	sts	dflags,r16		;und Flags wieder speichern
	call	newcfg			;alle Konfigzeilen neu ausgeben
;
; Anzeigemodus Konfig/Zeiten/Uhr stellen: Namen ausgeben
;
m1a100:	ldi	r25,0			;Zeilen-Zhler = 0
	ldi	yl,low(cfgflg)
	ldi	yh,high(cfgflg)		;Zeiger auf Konfigurationsflags
m1a110:	ldi	r20,2			;X-Startposition
	ldi	r21,16			;Y-Startposition
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r25,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
	movw	r12,r20			;X/Y-Startpositionen sichern
;
	ld	r16,y			;Konfigurationflags holen
	bst	r16,0			;Name neu ausgeben?
	brtc	m1a150			;nein -> berspringen
	cbr	r16,0x01		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
;
	lds	r19,menpos		;ausgewhlte Position holen
	dec	r19			;korrigieren
	cp	r25,r19			;Position identisch?
	brne	m1a120			;nein -> weiter
	ldi	r16,0x01		;sonst
	mov	txflag,r16		;Textflag invers setzen
m1a120:	ldi	r22,0x7a		;Text "Uhrzeit"
	add	r22,r25			;Zhler addieren -> Textnummer
	call	strlcd			;Text ausgeben
	clr	txflag
;
; Anzeigemodus Konfig/Zeiten/Uhr stellen: Daten ausgeben
;
m1a150:	ld	r16,y			;Konfigurationflags holen
	bst	r16,1			;Daten neu ausgeben?
	brts	m1a160			;ja -> weiter
	rjmp	m1a250			;sonst berspringen
;
m1a160:	cbr	r16,0x02		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
	movw	r20,r12			;X/Y-Koordinaten restaurieren
	ldi	r20,110			;Position fr Daten
;
	cpi	r25,0			;Zhler auf Zeit?
	brne	m1a180			;nein -> weiter testen
	lds	r16,cptiml		;sonst
	lds	r17,cptimh		;Zeit laden
	call	timasc			;Zeit in ASCII wandeln
	mov	r22,r4			;erste Stelle holen
	cpi	r22,' '			;Leerzeichen?
	breq	m1a170			;ja -> nicht ausgeben
	call	asclcd			;ausgeben
m1a170:	mov	r22,r5			;zweite Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,':'			;Trennzeichen laden
	call	asclcd			;ausgeben
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;sonst ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	rjmp	m1a240
;
m1a180:	cpi	r25,1			;Zhler auf Tag?
	brne	m1a200			;nein -> weiter testen
	lds	r16,cpday		;sonst Tag laden
	clr	r17
	call	bytas4			;Wert in 4 Byte ASCII wandeln
	mov	r22,r6			;dritte Stelle holen
	cpi	r22,'0'			;dritte Stelle =0?
	breq	m1a190			;ja -> berspringen
	call	asclcd			;ausgeben
m1a190:	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	rjmp	m1a240
;
m1a200:	cpi	r25,2			;Zhler auf Monat?
	brne	m1a210			;nein -> weiter testen
	lds	r22,cpmon		;sonst Monat laden
	ldi	r16,0x10-1		;Stringnummer Monate -1
	add	r22,r16			;Stringnummer addieren
	call	strlcd			;ausgeben
	rjmp	m1a240
;
m1a210:	cpi	r25,3			;Zhler auf Jahr?
	brne	m1a220			;nein -> weiter testen
	ldi	r22,year3		;Jahr Tausender holen
	call	asclcd			;ausgeben
	ldi	r22,year2		;Jahr Hunderter holen
	call	asclcd			;ausgeben
	lds	r16,cpyear		;Jahr laden
	clr	r17
	call	bytas4			;Wert in 4 Byte ASCII wandeln
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	rjmp	m1a240
;
m1a220:	cpi	r25,4			;Zhler auf Wochentag?
	brne	m1a250			;nein -> Ende
	lds	r22,cpdayw		;sonst Wochentag laden
	call	strlcd			;ausgeben
;
m1a240:	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,210			;X-Endwert
	ldi	r22,11			;zum Endwert Y 11 Pixel
	add	r21,r22			;addieren
	call	clrwnd			;Bereich lschen
;
m1a250:	adiw	yl,1			;Flag Tabellenzeiger erhhen
	inc	r25			;nchste Zeile
	cpi	r25,6			;alle Zeilen bearbeitet?
	brcc	m1a300			;ja -> weiter
	rjmp	m1a110			;sonst Schleife
;
; Anzeigemodus Konfig/Zeiten/Uhr stellen: Auswahl bearbeiten
;
m1a300:	lds	r16,menpos		;ausgewhlte Position holen
	lds	r17,encval		;Encoderwert holen
	cp	r16,r17			;nderung der Position?
	breq	m1a310			;nein -> weiter
;
	ldi	xl,low(cfgflg)
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	movw	yl,xl			;Zeiger kopieren
	dec	r16			;Positionswert korrigieren
	clr	r19
	add	xl,r16			;Flag-Position ermitteln
	adc	xh,r19			;bertrag
	ld	r16,x			;Zeilen-Flags holen
	sbr	r16,0x01		;Name neu ausgeben
	st	x,r16			;Flags wieder speichern
	dec	r17			;Encoderwert korrigieren
	add	yl,r17			;Flag-Position ermitteln
	adc	yh,r19			;bertrag
	ld	r16,y			;Zeilen-Flags holen
	sbr	r16,0x01		;Name neu ausgeben
	st	y,r16			;Flags wieder speichern
	lds	r16,encval		;Encoder-Wert holen
	sts	menpos,r16		;ausgewhlte Position setzen
m1a310:	jmp	m20000
;
; Anzeigemodus Konfig/Zeiten/Uhr stellen: ndern bearbeiten
;
m1a400:	cpi	r16,18			;Anzeigemodus Zeiten/Uhr/nd.?
	breq	m1a410			;ja -> weiter
	rjmp	m1a500			;sonst weiter testen
;
m1a410:	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,7			;Wert gendert?
	brts	m1a420			;ja -> ausgeben
	rjmp	m1a4a0			;sonst Ende
;
m1a420:	cbr	r16,1<<7		;Encoder nderungs-Flag lschen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	lds	r16,menpos		;ausgewhlte Position holen
	dec	r16			;Positionswert korrigieren
	ldi	r20,110			;X-Startposition
	ldi	r21,16			;Y-Startposition
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r16,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
;
	cpi	r16,0			;Position auf Zeit?
	brne	m1a440			;nein -> weiter testen
	lds	r16,encval		;Encoder-Wert L holen
	lds	r17,encvah		;Encoder-Wert H holen
	call	timasc			;Zeit in ASCII wandeln
	mov	r22,r4			;erste Stelle holen
	cpi	r22,' '			;Leerzeichen?
	breq	m1a430			;ja -> nicht ausgeben
	call	asclcd			;ausgeben
m1a430:	mov	r22,r5			;zweite Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,':'			;Trennzeichen laden
	call	asclcd			;ausgeben
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;sonst ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	rjmp	m1a490			;Zeilenende lschen
;
m1a440:	cpi	r16,1			;Position auf Tag?
	brne	m1a460			;nein -> weiter testen
	lds	r16,encval		;Encoder-Wert holen
	clr	r17
	call	bytas4			;Wert in 4 Byte ASCII wandeln
	mov	r22,r6			;dritte Stelle holen
	cpi	r22,'0'			;dritte Stelle =0?
	breq	m1a450			;ja -> berspringen
	call	asclcd			;ausgeben
m1a450:	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	rjmp	m1a490			;Zeilenende lschen
;
m1a460:	cpi	r16,2			;Position auf Monat?
	brne	m1a470			;nein -> weiter testen
	lds	r22,encval		;Encoder-Wert holen
	ldi	r16,0x10-1		;Stringnummer Monate -1
	add	r22,r16			;Stringnummer addieren
	call	strlcd			;ausgeben
	rjmp	m1a490			;Zeilenende lschen
;
m1a470:	cpi	r16,3			;Position auf Jahr?
	brne	m1a480			;nein -> Wochentag bearbeiten
	lds	r16,encval		;Encoder-Wert L holen
	lds	r17,encvah		;Encoder-Wert H holen
	call	bytas4			;Wert in 4 Byte ASCII wandeln
	mov	r22,r4			;erste Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r5			;zweite Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	rjmp	m1a490			;Zeilenende lschen
;
m1a480:	lds	r22,encval		;Encoder-Wert holen (Wochentag)
	dec	r22			;korrigieren
	call	strlcd			;ausgeben
;
m1a490:	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,237			;X-Endwert
	ldi	r22,11			;zum Endwert Y 11 Pixel
	add	r21,r22			;addieren
	call	clrwnd			;Bereich lschen
m1a4a0:	jmp	m20000
;
; Anzeigemodus Konfig/Sonstiges: Anzeige komplett neu ausgeben
;
m1a500:	cpi	r16,19			;Anzeigemodus Konfig/Sonstiges?
	breq	m1a510			;ja -> weiter
	rjmp	m1a900			;sonst weiter testen
;
m1a510:	clr	txflag			;Text-Flag lschen
	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,1			;Anzeige neu ausgeben?
	brtc	m1a600			;nein -> weiter
	cbr	r16,0x02		;sonst Neuausgabe-Flag lschen
	sts	dflags,r16		;und Flags wieder speichern
	call	newcfg			;alle Konfigzeilen neu ausgeben
;
; Anzeigemodus Konfig/Sonstiges: Namen ausgeben
;
m1a600:	ldi	r25,0			;Zeilen-Zhler = 0
	ldi	yl,low(cfgflg)
	ldi	yh,high(cfgflg)		;Zeiger auf Konfigurationsflags
m1a610:	ldi	r20,2			;X-Startposition
	ldi	r21,16			;Y-Startposition
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r25,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
	movw	r12,r20			;X/Y-Startpositionen sichern
;
	ld	r16,y			;Konfigurationflags holen
	bst	r16,0			;Name neu ausgeben?
	brtc	m1a650			;nein -> berspringen
	cbr	r16,0x01		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
;
	lds	r19,menpos		;ausgewhlte Position holen
	dec	r19			;korrigieren
	cp	r25,r19			;Position identisch?
	brne	m1a620			;nein -> weiter
	ldi	r16,0x01		;sonst
	mov	txflag,r16		;Textflag invers setzen
m1a620:	ldi	r22,0x9b		;Text "weiche Beleuchtung"
	cpi	r25,4			;Zhler=4?
	breq	m1a630			;ja -> ausgeben
	ldi	r22,0x80		;Text "Sprache"
	add	r22,r25			;Zhler addieren -> Textnummer
	cpi	r25,4			;Zhler<4?
	brcs	m1a630			;ja -> ausgeben
	dec	r22			;sonst Textnummer korrigieren
m1a630:	call	strlcd			;Text ausgeben
	clr	txflag
;
; Anzeigemodus Konfig/Sonstiges: Daten ausgeben
;
m1a650:	ld	r16,y			;Konfigurationflags holen
	bst	r16,1			;Daten neu ausgeben?
	brts	m1a660			;ja -> weiter
	rjmp	m1a810			;sonst berspringen
;
m1a660:	cbr	r16,0x02		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
	movw	r20,r12			;X/Y-Koordinaten restaurieren
	ldi	r20,180			;Position fr Daten
;
	cpi	r25,0			;Zhler auf Sprache?
	brne	m1a670			;nein -> weiter testen
	ldi	r22,9			;Sprache laden
	call	strlcd			;ausgeben
	rjmp	m1a800
;
m1a670:	cpi	r25,1			;Zhler auf Trennzeichen?
	brne	m1a690			;nein -> weiter testen
	ldi	r22,'.'			;Trennzeichen Punkt laden
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,1			;Trennzeichen Punkt?
	brtc	m1a680			;ja -> weiter
	ldi	r22,','			;Trennzeichen Komma laden
m1a680:	call	asclcd			;ausgeben
	rjmp	m1a800
;
m1a690:	cpi	r25,2			;Zhler auf Temp-Alarm-Modus?
	brne	m1a710			;nein -> weiter testen
	ldi	r22,0xb4		;Zeichen "Checkbox1 aus"
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,2			;Temp-Alarme immer melden?
	brtc	m1a700			;nein -> weiter
	ldi	r22,0xb5		;Zeichen "Checkbox1 ein"
m1a700:	call	asclcd			;ausgeben
	rjmp	m1a800
;
m1a710:	cpi	r25,3			;Zhler auf LCD-Abschaltung?
	brne	m1a730			;nein -> weiter testen
	ldi	r22,0xb4		;Zeichen "Checkbox1 aus"
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,6			;LCD Abschaltung 3:00?
	brtc	m1a720			;nein -> weiter
	ldi	r22,0xb5		;Zeichen "Checkbox1 ein"
m1a720:	call	asclcd			;ausgeben
	rjmp	m1a800
;
m1a730:	cpi	r25,4			;Zhler auf LCD-Backlight?
	brne	m1a750			;nein -> weiter testen
	ldi	r22,0xb4		;Zeichen "Checkbox1 aus"
	lds	r16,confg2		;Konfigurationsbyte 2 holen
	bst	r16,1			;weiches LCD Backlight?
	brtc	m1a740			;nein -> weiter
	ldi	r22,0xb5		;Zeichen "Checkbox1 ein"
m1a740:	call	asclcd			;ausgeben
	rjmp	m1a800
;
m1a750:	cpi	r25,5			;Zhler auf Schwellwert LCD?
	brne	m1a790			;nein -> weiter testen
	lds	r16,light		;Schwellwert LCD-Beleuchtung
	clr	r17
	call	bytas4			;Wert in 4 Byte ASCII wandeln
	mov	r22,r5			;zweite Stelle holen
	cpi	r22,'0'			;zweite Stelle =0?
	breq	m1a760			;ja -> berspringen
	call	asclcd			;sonst ausgeben
	mov	r22,r6			;dritte Stelle holen
	rjmp	m1a770			;sofort ausgeben
m1a760:	mov	r22,r6			;dritte Stelle holen
	cpi	r22,'0'			;dritte Stelle =0?
	breq	m1a780			;ja -> berspringen
m1a770:	call	asclcd			;ausgeben
m1a780:	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	rjmp	m1a800
;
m1a790:	ldi	r22,0x86		;Text "9600"
	lds	r16,baud0		;Baudrate PC-Schnittstelle
	add	r22,r16			;Baudratentext generieren
	call	strlcd			;ausgeben
;
m1a800:	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,237			;X-Endwert
	ldi	r22,11			;zum Endwert Y 11 Pixel
	add	r21,r22			;addieren
	call	clrwnd			;Bereich lschen
;
m1a810:	adiw	yl,1			;Flag Tabellenzeiger erhhen
	inc	r25			;nchste Zeile
	cpi	r25,7			;alle Zeilen bearbeitet?
	brcc	m1a850			;ja -> weiter
	rjmp	m1a610			;sonst Schleife
;
; Anzeigemodus Konfig/Sonstiges: Auswahl bearbeiten
;
m1a850:	lds	r16,menpos		;ausgewhlte Position holen
	lds	r17,encval		;Encoderwert holen
	cp	r16,r17			;nderung der Position?
	breq	m1a860			;nein -> weiter
;
	ldi	xl,low(cfgflg)
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	movw	yl,xl			;Zeiger kopieren
	dec	r16			;Positionswert korrigieren
	clr	r19
	add	xl,r16			;Flag-Position ermitteln
	adc	xh,r19			;bertrag
	ld	r16,x			;Zeilen-Flags holen
	sbr	r16,0x01		;Name neu ausgeben
	st	x,r16			;Flags wieder speichern
	dec	r17			;Encoderwert korrigieren
	add	yl,r17			;Flag-Position ermitteln
	adc	yh,r19			;bertrag
	ld	r16,y			;Zeilen-Flags holen
	sbr	r16,0x01		;Name neu ausgeben
	st	y,r16			;Flags wieder speichern
	lds	r16,encval		;Encoder-Wert holen
	sts	menpos,r16		;ausgewhlte Position setzen
m1a860:	rjmp	m20000
;
; Anzeigemodus Konfig/Sonstiges: ndern bearbeiten
;
m1a900:	cpi	r16,20			;Anzeigemodus Konfig/Sonst/nd?
	breq	m1a910			;ja -> weiter
	rjmp	m1b000			;sonst weiter testen
;
m1a910:	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,7			;Wert gendert?
	brtc	m1a950			;nein -> Ende
;
	cbr	r16,1<<7		;Encoder nderungs-Flag lschen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	lds	r16,menpos		;ausgewhlte Position holen
	dec	r16			;Positionswert korrigieren
	ldi	r20,180			;X-Startposition
	ldi	r21,16			;Y-Startposition
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r16,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
;
	cpi	r16,5			;Position auf Lichtschwellwert?
	brne	m1a950			;nein -> Ende
	lds	r16,encval		;Encoder-Wert holen
	clr	r17			;H-Byte=0
	call	bytas4			;Wert in 4 Byte ASCII wandeln
	mov	r22,r5			;zweite Stelle holen
	cpi	r22,'0'			;zweite Stelle =0?
	breq	m1a920			;ja -> berspringen
	call	asclcd			;sonst ausgeben
	mov	r22,r6			;dritte Stelle holen
	rjmp	m1a930			;sofort ausgeben
m1a920:	mov	r22,r6			;dritte Stelle holen
	cpi	r22,'0'			;dritte Stelle =0?
	breq	m1a940			;ja -> berspringen
m1a930:	call	asclcd			;ausgeben
m1a940:	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
;
	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,237			;X-Endwert
	ldi	r22,11			;zum Endwert Y 11 Pixel
	add	r21,r22			;addieren
	call	clrwnd			;Bereich lschen
m1a950:	rjmp	m20000
;
; Anzeigemodus Konfig/Temp/Log-Zuordnung: Anzeige komplett neu ausgeben
;
m1b000:	cpi	r16,21			;Anzeigemodus Temp/Log/Auswhl?
	breq	m1b010			;ja -> weiter
	cpi	r16,22			;Anzeigemodus Temp/Log/Versch.?
	breq	m1b010			;ja -> weiter
	rjmp	m1c000			;sonst weiter testen
;
m1b010:	clr	txflag			;Text-Flag lschen
	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,1			;Anzeige neu ausgeben?
	brtc	m1b100			;nein -> weiter
	cbr	r16,0x02		;sonst Neuausgabe-Flag lschen
	sts	dflags,r16		;und Flags wieder speichern
;
	ldi	r19,15|0x80		;Y=15
m1b020:	ldi	r18,119			;X=119, Trennlinie
	call	pixlcd			;senkrechte Linie zeichnen
	inc	r19			;Y-Koordinate erhhen
	cpi	r19,112|0x80		;alle Pixel ausgegeben?
	brne	m1b020			;nein -> Schleife
;
	call	newtem			;alle Namen/Werte neu ausgeben
;
; Anzeigemodus Konfig/Temp/Log-Zuordnung: Sensornamen ausgeben
;
m1b100:	ldi	r25,0			;Log-Positionszhler = 0
	ldi	xl,low(tembak)
	ldi	xh,high(tembak)		;Zeiger auf Mapping Tabelle
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
m1b110:	ld	r16,y			;Temperaturflags holen
	bst	r16,1			;Name neu ausgeben?
	brts	m1b120			;ja -> weiter
	rjmp	m1b160			;sonst berspringen
;
m1b120:	cbr	r16,0x02		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
	ld	r18,x			;Log-Position holen
	mov	r11,r18			;Log-Position sichern
	ldi	r20,2			;X-Startposition Spalte 1
	ldi	r21,16			;Y-Startposition
	cpi	r18,8			;Position in Spalte 1?
	brcs	m1b130			;ja -> weiter
	ldi	r20,121			;sonst X-Startposition Spalte 2
m1b130:	mov	r16,r18			;Anzeigeposition holen
	andi	r16,0x07		;Zeilenposition filtern
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r16,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
	movw	r12,r20			;X/Y-Startpositionen sichern
;
	ldi	r22,0x98		;String "Log"
	call	strlcd			;ausgeben
	mov	r16,r11			;Log-Position holen
	inc	r16			;fr Anzeige korrigieren
	clr	r17			;H-Byte lschen
	call	bytas4			;Wert in 4 Byte ASCII wandeln
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	ldi	r16,6			;Position fr Sensornummer
	add	r20,r16			;berechnen
;
	lds	r19,menpos		;ausgewhlte Position holen
	dec	r19			;korrigieren
	cp	r11,r19			;Position identisch?
	brne	m1b140			;nein -> weiter
	ldi	r16,0x01		;sonst
	mov	txflag,r16		;Textflag invers setzen
m1b140:	ldi	r22,'S'			;Zeichen 'S' (Sensor)
	call	asclcd			;ausgeben
	mov	r16,r25			;Sensornummer kopieren
	inc	r16			;fr Anzeige korrigieren
	clr	r17			;H-Byte lschen
	call	bytas4			;Wert in 4 Byte ASCII wandeln
	mov	r22,r6			;dritte Stelle holen
	call	asclcd			;ausgeben
	mov	r22,r7			;vierte Stelle holen
	call	asclcd			;ausgeben
	ldi	r22,0x7f		;Endezeichen
	call	asclcd			;Zeichen ausgeben
	clr	txflag
;
	ldi	r16,6			;Position fr Sensorname
	add	r20,r16			;berechnen
	ldi	r19,9			;Sensornamenlnge im EEPROM
	mul	r25,r19			;Tabellenoffset ermitteln
	ldi	r16,low(etemtx)		;EEPROM-Adresse Sensornamen L
	ldi	r17,high(etemtx)	;EEPROM-Adresse Sensornamen H
	add	r16,r0			;Offset zur Adresse L addieren
	adc	r17,r1			;Offset zur Adresse H addieren
	call	eeplcd			;EEPROM-Text ausgeben
;
	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,117			;X-Endwert Spalte 1
	ld	r22,x			;Anzeigeposition holen
	cpi	r22,8			;Position in Spalte 1?
	brcs	m1b150			;ja -> weiter
	ldi	r20,237			;sonst X-Startposition Spalte 2
m1b150:	ldi	r22,11			;zum Endwert Y 11 Pixel
	add	r21,r22			;addieren
	call	clrwnd			;Bereich lschen
;
m1b160:	adiw	xl,1			;Mapping Tabellenzeiger erhhen
	adiw	yl,1			;Flag Tabellenzeiger erhhen
	inc	r25			;nchste Sensornummer
	cpi	r25,16			;alle Sensoren bearbeitet?
	brcc	m1b200			;ja -> weiter
	rjmp	m1b110			;sonst Schleife
;
; Anzeigemodus Konfig/Temp/Sensoren: Auswahl bearbeiten
;
m1b200:	lds	r16,dimode		;Anzeigemodus holen?
	cpi	r16,21			;Anzeigemodus Temp/Log/Auswhl?
	brne	m1b250			;nein -> weiter
	lds	r16,menpos		;ausgewhlte Position holen
	lds	r17,encval		;Encoderwert holen
	cp	r16,r17			;nderung der Position?
	breq	m1b210			;nein -> weiter
;
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
	movw	zl,yl			;Zeiger kopieren
	call	stebak			;Sensornummer suchen
	clr	r19			;H-Byte=0
	add	yl,r18			;Flag-Position ermitteln
	adc	yh,r19			;bertrag
	ld	r16,y			;Temperatur-Flags holen
	sbr	r16,0x02		;Name neu ausgeben
	st	y,r16			;Flags wieder speichern
	lds	r16,encval		;Encoderwert holen
	call	stebak			;Sensornummer suchen
	add	zl,r18			;Flag-Position ermitteln
	adc	zh,r19			;bertrag
	ld	r16,z			;Temperatur-Flags holen
	sbr	r16,0x02		;Name neu ausgeben
	st	z,r16			;Flags wieder speichern
	lds	r16,encval		;Encoder-Wert holen
	sts	menpos,r16		;ausgewhlte Position setzen
m1b210:	rjmp	m20000
;
; Anzeigemodus Konfig/Temp/Sensoren: Verschiebung bearbeiten
;
m1b250:	cpi	r16,22			;Anzeigemodus Temp/Log/Versch.?
	brne	m1b300			;nein -> weiter
	lds	r16,menpos		;ausgewhlte Position holen
	lds	r17,encval		;Encoderwert holen
	cp	r16,r17			;nderung der Position?
	breq	m1b300			;nein -> weiter
;
m1b260:	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
	movw	zl,yl			;Zeiger kopieren
	call	stebak			;Sensornummer suchen
	mov	r20,r18			;Sensornummer 1 sichern
	clr	r19			;H-Byte=0
	add	yl,r18			;Flag-Position ermitteln
	adc	yh,r19			;bertrag
	ld	r16,y			;Temperatur-Flags holen
	sbr	r16,0x02		;Name neu ausgeben
	st	y,r16			;Flags wieder speichern
	lds	r16,encval		;Encoderwert holen
	call	stebak			;Sensornummer suchen
	mov	r21,r18			;Sensornummer 2 sichern
	add	zl,r18			;Flag-Position ermitteln
	adc	zh,r19			;bertrag
	ld	r16,z			;Temperatur-Flags holen
	sbr	r16,0x02		;Name neu ausgeben
	st	z,r16			;Flags wieder speichern
;
	ldi	yl,low(tembak)
	ldi	yh,high(tembak)		;Zeiger auf Mapping-Backup
	movw	zl,yl			;Zeiger kopieren
	mov	r18,r20			;Sensornummer 1 holen
	add	yl,r18			;Tabellenplatz ermitteln
	adc	yh,r19			;bertrag
	mov	r18,r21			;Sensornummer 2 holen
	add	zl,r18			;Tabellenplatz ermitteln
	adc	yh,r19			;bertrag
	ld	r16,y			;Anzeigeposition 1 holen
	ld	r17,z			;Anzeigeposition 2 holen
	st	y,r17			;Positionen tauschen und
	st	z,r16			;speichern
	lds	r16,encval		;Encoder-Wert holen
	sts	menpos,r16		;ausgewhlte Position setzen
m1b300:	rjmp	m20000
;
; Anzeigemodus Grafik: Anzeige komplett neu ausgeben
;
m1c000:	cpi	r16,23			;Anzeigemodus Grafik?
	breq	m1c010			;ja -> weiter
	rjmp	m1f000			;sonst weiter testen
;
m1c010:	clr	txflag			;Text-Flag lschen
	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,1			;Anzeige neu ausgeben?
	brtc	m1c100			;nein -> weiter testen
	cbr	r16,0x02		;sonst Neuausgabe-Flag lschen
	sts	dflags,r16		;und Flags wieder speichern
;
	ldi	r18,15			;X=15 \ Anfangspunkt Y-Achse
	ldi	r19,16|0x80		;Y=16 / fr beide Diagramme
m1c020:	call	pixlcd			;senkrechte Linie zeichnen
	inc	r19			;Y-Koordinate erhhen
	cpi	r19,104|0x80		;alle Pixel ausgegeben?
	brne	m1c020			;nein -> Schleife
;
	ldi	r18,16			;X=16  \ Anfangspunkt X-Achse
	ldi	r19,103|0x80		;Y=103 / fr Diagramm 1
m1c030:	call	pixlcd			;waagerechte Linie zeichnen
	inc	r18			;X-Koordinate erhhen
	cpi	r18,238			;alle Pixel ausgegeben?
	brne	m1c030			;nein -> Schleife
;
	lds	r16,grflag		;Grafik-Flags holen
	bst	r16,4			;Doppeldiagramm?
	brtc	m1c050			;nein -> nur ein Diagramm
;
	ldi	r18,16			;X=16  \ Anfangspunkt X-Achse
	ldi	r19,59|0x80		;Y=103 / fr Diagramm 2
m1c040:	call	pixlcd			;waagerechte Linie zeichnen
	inc	r18			;X-Koordinate erhhen
	cpi	r18,238			;alle Pixel ausgegeben?
	brne	m1c040			;nein -> Schleife
;
m1c050:	call	grclrb			;Grafikpuffer/Pointer initialis.
;
m1c100:	lds	r16,xflag3		;verschiedene Flags 3 holen
	bst	r16,1			;Daten aus EEPROM einlesen?
	brts	m1c110			;ja -> Daten einlesen
	rjmp	m1d000			;sonst weiter testen
;
; Anzeigemodus Grafik: Daten aus dem Externen EERPOM einlesen und im
; Datenpuffer speichern
;
m1c110:	lds	r17,errflg		;Fehler-Flags holen
	bst	r17,3			;Speicherfehler?
	brtc	m1c210			;nein -> weiter
;
m1c200:	lds	r16,xflag3		;sonst verschiedene Flags holen
	cbr	r16,1<<1		;Daten einlesen beenden und im
	sbr	r16,1<<2		;Grafikausgabe im nchsten Zyk.
	sts	xflag3,r16		;Flags wieder speichern
;
	lds	r16,grptem
	lds	r17,grptem+1
	lds	r18,grptem+2		;Adresse aktueller Datensatz
	sts	grpsta,r16
	sts	grpsta+1,r17
	sts	grpsta+2,r18		;als Anfangsadresse speichern
	rjmp	m20000			;nchster Abschnitt
;
m1c210:	bst	r16,3			;vorherige Seite anzeigen?
	brts	m1c220			;ja - bearbeiten
	rjmp	m1c500			;sonst nchste Seite
;
; Anzeigemodus Grafik: Einlesen der Daten rckwrts
;
m1c220:	lds	r16,xflag3		;verschiedene Flags 3 holen
	bst	r16,5			;erster Datensatz erreicht?
	brts	m1c200			;ja -> Daten einlesen beenden
;
	lds	r20,grptem
	lds	r21,grptem+1		;Zeiger auf die aktuelle Posi-
	lds	r22,grptem+2		;tion im Externen EEPROM holen
	ldi	r23,28			;Datensatzlnge 15-min-Werte
	lds	r16,grmod1		;Anzeigemodus 1 holen
	tst	r16			;15-Minuten-Werte anzeigen?
	breq	m1c230			;ja -> weiter
	ldi	r23,76			;sonst Datensatzlnge 24h-Werte
m1c230:	call	xdread			;Datensatz aus EEPROM lesen
	brcc	m1c240			;Fehler? nein -> weiter
	call	twierr			;sonst Speicherfehler setzen
	rjmp	m1c200			;Daten einlesen beenden
;
m1c240:	lds	r16,xedbuf		;Zeitbyte 0 holen
	lds	r17,xedbuf+1		;Zeitbyte 1 holen
	lds	r18,xedbuf+2		;Zeitbyte 2 holen
	lds	r19,xedbuf+3		;Zeitbyte 3 holen
	mov	r20,r16			;Zeitbyte 0 kopieren
	and	r20,r17			;mit Zeitbyte 1 verknpfen
	and	r20,r18			;mit Zeitbyte 2 verknpfen
	and	r20,r19			;mit Zeitbyte 3 verknpfen
	cpi	r20,0xff		;gltiger Datenssatz?
	breq	m1c200			;nein -> Ende
;
	lds	r20,grpoin		;Grafikpuffer-Zeiger holen
	cpi	r20,221			;Zeiger auf letzten Platz?
	brne	m1c250			;nein -> weiter
	sts	grcdat,r16
	sts	grcdat+1,r17
	sts	grcdat+2,r18		;sonst Zeitdaten in Zwischen-
	sts	grcdat+3,r19		;speicher legen
	rjmp	m1c310
;
m1c250:	lds	r20,grcdat
	lds	r21,grcdat+1
	lds	r22,grcdat+2		;Zeitdaten aus Zwischenspeicher
	lds	r23,grcdat+3		;holen
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22
	sbc	r19,r23			;stimmt Zeit berein?
	breq	m1c310			;ja -> Daten in Grafikpuffer
	brcc	m1c300			;Zeit grer? ja -> weiter
;
	ldi	r16,0			;sonst
	ldi	r17,0x80		;Temperaturwert ungltig setzen
	call	grbts1			;Zeit/Tempwert 1 in Puffer
	ldi	r16,0
	ldi	r17,0x80		;Temperaturwert ungltig setzen
	call	grbts2			;Temperaturwert 2 in Puffer
	call	grprev			;vorherigen Zeitpunkt ermitteln
	lds	r16,grpoin		;Datenpuffer-Zeiger holen
	cpi	r16,0			;Pufferanfang erreicht?
	brne	m1c290			;nein -> weiter
	rjmp	m1c200			;sonst Daten einlesen beenden
;
m1c290:	dec	r16			;Zeiger auf vorherigen Platz
	sts	grpoin,r16		;Datenpuffer-Zeiger speichern
	rjmp	m1c240			;Datensatz erneut prfen
;
m1c300:	call	grdeca			;Adresse vorheriger Datensatz
	rjmp	m1c220			;nchsten Datensatz einlesen
;
m1c310:	lds	r16,grsen1		;Grafik 1 Sensornummern holen
	andi	r16,0x0f		;unteres Nibble filtern
	call	grslgm			;Log-Position ermitteln
	ldi	yl,low(xedbuf+4)
	ldi	yh,high(xedbuf+4)	;Zeiger auf Temperaturdaten
	lds	r17,grmod1		;Anzeigemodus 1 holen
	tst	r17			;15-Minuten-Werte anzeigen?
	breq	m1c320			;ja -> weiter
	dec	r17			;sonst Anzeigemodus korrigieren
	ldi	r18,24			;Offset fr Wertebereich
	mul	r17,r18			;Adressoffset berechnen
	add	yl,r0
	adc	yh,r1			;Zeiger auf Min/Max/Mittelwerte
m1c320:	call	grgete			;Temperaturwert holen
;
	lds	r18,grflag		;Grafik-Flags holen
	bst	r18,2			;Differenz-Modus aktiv?
	brtc	m1c340			;nein -> Wert speichern
	cpi	r17,0x80		;Wert ungltig?
	breq	m1c340			;ja -> Wert speichern
;
	movw	r20,r16			;Temperaturwert kopieren
	lds	r16,grsen1		;Grafik 1 Sensornummern holen
	swap	r16			;Differenzsensor holen
	andi	r16,0x0f		;unteres Nibble filtern
	call	grslgm			;Log-Position ermitteln
	ldi	yl,low(xedbuf+4)
	ldi	yh,high(xedbuf+4)	;Zeiger auf Temperaturdaten
	lds	r17,grmod1		;Anzeigemodus 1 holen
	tst	r17			;15-Minuten-Werte anzeigen?
	breq	m1c330			;ja -> weiter
	dec	r17			;sonst Anzeigemodus korrigieren
	ldi	r18,24			;Offset fr Wertebereich
	mul	r17,r18			;Adressoffset berechnen
	add	yl,r0
	adc	yh,r1			;Zeiger auf Min/Max/Mittelwerte
m1c330:	call	grgete			;Temperaturwert holen
	cpi	r17,0x80		;Wert ungltig?
	breq	m1c340			;ja -> Wert speichern
	sub	r20,r16			;sonst Werte subtrahieren
	sbc	r21,r17			;bertrag
	movw	r16,r20			;Ergebnis kopieren
m1c340:	call	grbts1			;Zeit/Tempwert 1 in Puffer
;
	lds	r16,grflag		;Grafik-Flags holen
	bst	r16,4			;Doppeldiagramm?
	brtc	m1c380			;nein -> nur einen Wert lesen
	lds	r16,grsen2		;Grafik 2 Sensornummern holen
	andi	r16,0x0f		;unteres Nibble filtern
	call	grslgm			;sonst Log-Position ermitteln
	ldi	yl,low(xedbuf+4)
	ldi	yh,high(xedbuf+4)	;Zeiger auf Temperaturdaten
	lds	r17,grmod2		;Anzeigemodus 2 holen
	tst	r17			;15-Minuten-Werte anzeigen?
	breq	m1c350			;ja -> weiter
	dec	r17			;sonst Anzeigemodus korrigieren
	ldi	r18,24			;Offset fr Wertebereich
	mul	r17,r18			;Adressoffset berechnen
	add	yl,r0
	adc	yh,r1			;Zeiger auf Min/Max/Mittelwerte
m1c350:	call	grgete			;Temperaturwert holen
;
	lds	r18,grflag		;Grafik-Flags holen
	bst	r18,3			;Differenz-Modus aktiv?
	brtc	m1c370			;nein -> Wert speichern
	cpi	r17,0x80		;Wert ungltig?
	breq	m1c370			;ja -> Wert speichern
;
	movw	r20,r16			;Temperaturwert kopieren
	lds	r16,grsen2		;Grafik 2 Sensornummern holen
	swap	r16			;Differenzsensor holen
	andi	r16,0x0f		;unteres Nibble filtern
	call	grslgm			;Log-Position ermitteln
	ldi	yl,low(xedbuf+4)
	ldi	yh,high(xedbuf+4)	;Zeiger auf Temperaturdaten
	lds	r17,grmod2		;Anzeigemodus 2 holen
	tst	r17			;15-Minuten-Werte anzeigen?
	breq	m1c360			;ja -> weiter
	dec	r17			;sonst Anzeigemodus korrigieren
	ldi	r18,24			;Offset fr Wertebereich
	mul	r17,r18			;Adressoffset berechnen
	add	yl,r0
	adc	yh,r1			;Zeiger auf Min/Max/Mittelwerte
m1c360:	call	grgete			;Temperaturwert holen
	cpi	r17,0x80		;Wert ungltig?
	breq	m1c370			;ja -> Wert speichern
	sub	r20,r16			;sonst Werte subtrahieren
	sbc	r21,r17			;bertrag
	movw	r16,r20			;Ergebnis kopieren
m1c370:	call	grbts2			;Temperaturwert 2 in Puffer
;
m1c380:	call	grprev			;vorherigen Zeitpunkt ermitteln
	lds	r16,grpoin		;Datenpuffer-Zeiger holen
	cpi	r16,0			;Pufferanfang erreicht?
	brne	m1c390			;nein -> weiter
	rjmp	m1c200			;sonst Daten einlesen beenden
;
m1c390:	dec	r16			;Zeiger auf vorherigen Platz
	sts	grpoin,r16		;Datenpuffer-Zeiger speichern
	call	grdeca			;Adresse vorheriger Datensatz
	rjmp	m1c220			;sonst weiter Daten einlesen
;
; Anzeigemodus Grafik: Einlesen der Daten vorwrts
;
m1c500:	lds	r16,xflag3		;verschiedene Flags 3 holen
	bst	r16,4			;letzter Datensatz erreicht?
	brtc	m1c520			;nein -> Daten einlesen
;
m1c510:	lds	r16,xflag3		;sonst verschiedene Flags holen
	cbr	r16,1<<1		;Daten einlesen beenden und im
	sbr	r16,1<<2		;Grafikausgabe im nchsten Zyk.
	sts	xflag3,r16		;Flags wieder speichern
;
	lds	r16,grptem
	lds	r17,grptem+1
	lds	r18,grptem+2		;Datensatz-Adresse holen
	sts	grpend,r16
	sts	grpend+1,r17
	sts	grpend+2,r18		;als Endadresse speichern
	rjmp	m20000			;nchster Abschnitt
;
m1c520:	lds	r20,grptem
	lds	r21,grptem+1		;Zeiger auf die aktuelle Posi-
	lds	r22,grptem+2		;tion im Externen EEPROM holen
	ldi	r23,28
	lds	r16,grmod1		;Anzeigemodus 1 holen
	tst	r16			;15-Minuten-Werte anzeigen?
	breq	m1c530			;ja -> weiter
	ldi	r23,76			;sonst Datensatzlnge 24h-Werte
m1c530:	call	xdread			;Datensatz aus EEPROM lesen
	brcc	m1c540			;Fehler? nein -> weiter
	call	twierr			;sonst Speicherfehler setzen
	rjmp	m1c510			;Daten einlesen beenden
;
m1c540:	lds	r16,xedbuf		;Zeitbyte 0 holen
	lds	r17,xedbuf+1		;Zeitbyte 1 holen
	lds	r18,xedbuf+2		;Zeitbyte 2 holen
	lds	r19,xedbuf+3		;Zeitbyte 3 holen
	mov	r20,r16			;Zeitbyte 0 kopieren
	and	r20,r17			;mit Zeitbyte 1 verknpfen
	and	r20,r18			;mit Zeitbyte 2 verknpfen
	and	r20,r19			;mit Zeitbyte 3 verknpfen
	cpi	r20,0xff		;gltiger Datenssatz?
	breq	m1c510			;nein -> Ende
;
	lds	r20,grpoin		;Grafikpuffer-Zeiger holen
	cpi	r20,0			;Zeiger auf ersten Platz?
	brne	m1c550			;nein -> weiter
	sts	grcdat,r16
	sts	grcdat+1,r17
	sts	grcdat+2,r18		;sonst Zeitdaten in Zwischen-
	sts	grcdat+3,r19		;speichern legen
	rjmp	m1c600
;
m1c550:	lds	r20,grcdat
	lds	r21,grcdat+1
	lds	r22,grcdat+2		;Zeitdaten aus Zwischenspeicher
	lds	r23,grcdat+3		;holen
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22
	sbc	r19,r23			;stimmt Zeit berein?
	breq	m1c600			;ja -> Daten in Grafikpuffer
	brcc	m1c570			;Zeit grer? ja -> weiter
	call	grinca			;sonst Adresse nchster Datens.
	rjmp	m1c500			;nchsten Datensatz einlesen
;
m1c570:	ldi	r16,0
	ldi	r17,0x80		;Temperaturwert ungltig setzen
	call	grbts1			;Zeit/Tempwert 1 in Puffer
	ldi	r16,0
	ldi	r17,0x80		;Temperaturwert ungltig setzen
	call	grbts2			;Temperaturwert 2 in Puffer
	call	grnext			;nchsten Zeitpunkt ermitteln
	lds	r16,grpoin		;Datenpuffer-Zeiger holen
	cpi	r16,221			;Pufferende erreicht?
	brne	m1c580			;nein -> weiter
	rjmp	m1c510			;sonst Daten einlesen beenden
;
m1c580:	inc	r16			;Datenpufferzeiger erhhen
	sts	grpoin,r16		;und wieder speichern
	rjmp	m1c540			;Datensatz erneut prfen
;
m1c600:	lds	r16,grsen1		;Grafik 1 Sensornummern holen
	andi	r16,0x0f		;unteres Nibble filtern
	call	grslgm			;Log-Position ermitteln
	ldi	yl,low(xedbuf+4)
	ldi	yh,high(xedbuf+4)	;Zeiger auf Temperaturdaten
	lds	r17,grmod1		;Anzeigemodus 1 holen
	tst	r17			;15-Minuten-Werte anzeigen?
	breq	m1c610			;ja -> weiter
	dec	r17			;sonst Anzeigemodus korrigieren
	ldi	r18,24			;Offset fr Wertebereich
	mul	r17,r18			;Adressoffset berechnen
	add	yl,r0
	adc	yh,r1			;Zeiger auf Min/Max/Mittelwerte
m1c610:	call	grgete			;Temperaturwert holen
;
	lds	r18,grflag		;Grafik-Flags holen
	bst	r18,2			;Differenz-Modus aktiv?
	brtc	m1c630			;nein -> Wert speichern
	cpi	r17,0x80		;Wert ungltig?
	breq	m1c630			;ja -> Wert speichern
;
	movw	r20,r16			;Temperaturwert kopieren
	lds	r16,grsen1		;Grafik 1 Sensornummern holen
	swap	r16			;Differenzsensor holen
	andi	r16,0x0f		;unteres Nibble filtern
	call	grslgm			;Log-Position ermitteln
	ldi	yl,low(xedbuf+4)
	ldi	yh,high(xedbuf+4)	;Zeiger auf Temperaturdaten
	lds	r17,grmod1		;Anzeigemodus 1 holen
	tst	r17			;15-Minuten-Werte anzeigen?
	breq	m1c620			;ja -> weiter
	dec	r17			;sonst Anzeigemodus korrigieren
	ldi	r18,24			;Offset fr Wertebereich
	mul	r17,r18			;Adressoffset berechnen
	add	yl,r0
	adc	yh,r1			;Zeiger auf Min/Max/Mittelwerte
m1c620:	call	grgete			;Temperaturwert holen
	cpi	r17,0x80		;Wert ungltig?
	breq	m1c630			;ja -> Wert speichern
	sub	r20,r16			;sonst Werte subtrahieren
	sbc	r21,r17			;bertrag
	movw	r16,r20			;Ergebnis kopieren
m1c630:	call	grbts1			;Zeit/Tempwert 1 in Puffer
;
	lds	r16,grflag		;Grafik-Flags holen
	bst	r16,4			;Doppeldiagramm?
	brtc	m1c670			;nein -> nur einen Wert lesen
	lds	r16,grsen2		;Grafik 2 Sensornummern holen
	andi	r16,0x0f		;unteres Nibble filtern
	call	grslgm			;sonst Log-Position ermitteln
	ldi	yl,low(xedbuf+4)
	ldi	yh,high(xedbuf+4)	;Zeiger auf Temperaturdaten
	lds	r17,grmod2		;Anzeigemodus 2 holen
	tst	r17			;15-Minuten-Werte anzeigen?
	breq	m1c640			;ja -> weiter
	dec	r17			;sonst Anzeigemodus korrigieren
	ldi	r18,24			;Offset fr Wertebereich
	mul	r17,r18			;Adressoffset berechnen
	add	yl,r0
	adc	yh,r1			;Zeiger auf Min/Max/Mittelwerte
m1c640:	call	grgete			;Temperaturwert holen
;
	lds	r18,grflag		;Grafik-Flags holen
	bst	r18,3			;Differenz-Modus aktiv?
	brtc	m1c660			;nein -> Wert speichern
	cpi	r17,0x80		;Wert ungltig?
	breq	m1c660			;ja -> Wert speichern
;
	movw	r20,r16			;Temperaturwert kopieren
	lds	r16,grsen2		;Grafik 2 Sensornummern holen
	swap	r16			;Differenzsensor holen
	andi	r16,0x0f		;unteres Nibble filtern
	call	grslgm			;Log-Position ermitteln
	ldi	yl,low(xedbuf+4)
	ldi	yh,high(xedbuf+4)	;Zeiger auf Temperaturdaten
	lds	r17,grmod2		;Anzeigemodus 2 holen
	tst	r17			;15-Minuten-Werte anzeigen?
	breq	m1c650			;ja -> weiter
	dec	r17			;sonst Anzeigemodus korrigieren
	ldi	r18,24			;Offset fr Wertebereich
	mul	r17,r18			;Adressoffset berechnen
	add	yl,r0
	adc	yh,r1			;Zeiger auf Min/Max/Mittelwerte
m1c650:	call	grgete			;Temperaturwert holen
	cpi	r17,0x80		;Wert ungltig?
	breq	m1c660			;ja -> Wert speichern
	sub	r20,r16			;sonst Werte subtrahieren
	sbc	r21,r17			;bertrag
	movw	r16,r20			;Ergebnis kopieren
m1c660:	call	grbts2			;Temperaturwert 2 in Puffer
;
m1c670:	call	grnext			;nchsten Zeitpunkt ermitteln
	lds	r16,grpoin		;Datenpuffer-Zeiger holen
	cpi	r16,221			;Pufferende erreicht?
	brne	m1c680			;nein -> weiter
	rjmp	m1c510			;sonst Daten einlesen beenden
;
m1c680:	inc	r16			;Datenpufferzeiger erhhen
	sts	grpoin,r16		;und wieder speichern
	call	grinca			;Adresse nchster Datensatz
	rjmp	m1c500			;nchsten Datensatz lesen
;
; Anzeigemodus Grafik: Daten auf LCD ausgeben
;
m1d000:	bst	r16,2			;Daten auf LCD ausgeben?
	brts	m1d010			;ja -> Daten ausgeben
	rjmp	m20000			;sonst zum nchsten Abschnitt
;
; Anzeigemodus Grafik: Minimal- und Maximalwerte im Puffer ermitteln
;
m1d010:	ldi	zl,low(grbuff)
	ldi	zh,high(grbuff)		;Basisadresse Grafikdatenpuffer
	ldi	r20,0			;Anfangswert Pufferzhler
	ldi	r16,low(-1000)
	ldi	r17,high(-1000)		;Anfangswert Maximum (-100,0)
	movw	r8,r16			;Maximalwert fr Sensornummer 1
	movw	r12,r16			;Maximalwert fr Sensornummer 2
	ldi	r16,low(2000)
	ldi	r17,high(2000)		;Anfangswert Minimum (200,0)
	movw	r10,r16			;Minimalwert fr Sensornummer 1
	movw	r14,r16			;Minimalwert fr Sensornummer 2
;
m1d020:	adiw	zl,4			;Datum/Zeit berspringen
	ld	r16,z+			;Temperaturwert 1 L holen
	ld	r17,z+			;Temperaturwert 1 H holen
	cpi	r17,0x80		;Wert ungltig?
	breq	m1d040			;ja -> nicht auswerten
	movw	r18,r8			;Maximalwert kopieren
	sub	r18,r16
	sbc	r19,r17			;Neuer Wert > Maximalwert?
	brpl	m1d030			;nein -> weiter
	movw	r8,r16			;sonst neuen Maximalwert setzen
m1d030:	movw	r18,r16			;aktuellen Wert kopieren
	sub	r18,r10
	sbc	r19,r11			;Neuer Wert < Minimalwert?
	brpl	m1d040			;nein -> weiter
	movw	r10,r16			;sonst neuen Maximalwert setzen
;
m1d040:	ld	r16,z+			;Temperaturwert 2 L holen
	ld	r17,z+			;Temperaturwert 2 H holen
	lds	r18,grflag		;Grafik-Flags holen
	bst	r18,4			;Doppeldiagramm?
	brtc	m1d060			;nein -> Tempwert 2 ignorieren
	cpi	r17,0x80		;Wert ungltig?
	breq	m1d060			;ja -> nicht auswerten
	movw	r18,r12			;Maximalwert kopieren
	sub	r18,r16
	sbc	r19,r17			;Neuer Wert > Maximalwert?
	brpl	m1d050			;nein -> weiter
	movw	r12,r16			;sonst neuen Maximalwert setzen
m1d050:	movw	r18,r16			;aktuellen Wert kopieren
	sub	r18,r14
	sbc	r19,r15			;Neuer Wert < Minimalwert?
	brpl	m1d060			;nein -> weiter
	movw	r14,r16			;sonst neuen Maximalwert setzen
;
m1d060:	inc	r20			;Pufferposition erhhen
	cpi	r20,222			;letzter Y-Wert?
	brne	m1d020			;nein -> Schleife
;
; Anzeigemodus Grafik: Skalierungen und Y-Mittelpunktwerte berechnen
;
	movw	r16,r8			;Maxwert Sensornummer 1 holen
	cpi	r17,high(-1000)		;wurde Anfangswert H verndert?
	brne	m1d200			;ja -> weiter
	cpi	r16,low(-1000)		;wurde Anfangswert L verndert?
	brne	m1d200			;ja -> weiter
	rjmp	m1d260			;sonst Berechnung berspringen
;
m1d200:	sub	r8,r10			;Differenz
	sbc	r9,r11			;Maximalwert - Minimalwert
	movw	r16,r8			;Differenz kopieren
	asr	r17
	ror	r16			;Differenz halbieren
	brcc	m1d210			;Runden? nein -> weiter
	inc	r16			;sonst aufrunden
	brne	m1d210			;bertrag? nein -> weiter
	inc	r17			;sonst bertrag addieren
m1d210:	add	r16,r10
	adc	r17,r11			;zum Minimalwert addieren
	sts	grcen1,r16
	sts	grcen1+1,r17		;Y-Mittelpunktwert 1 speichern
;
	ldi	r16,86			;Y-Bereich fr Einzeldiagramm
	ldi	r17,0			;86 Pixel
	lds	r20,grflag		;Grafik-Flags holen
	bst	r20,4			;Doppeldiagramm?
	brtc	m1d220			;nein -> Y-Bereich ok
	ldi	r16,41			;sonst Y-Bereich nur 41 Pixel
m1d220:	ldi	r20,1			;Skalierungsfaktor= 1 setzen
m1d230:	movw	r18,r16			;Bereichswert kopieren
	sub	r18,r8
	sbc	r19,r9			;Bereich berschritten?
	brcc	m1d250			;nein -> weiter
	asr	r9			;sonst
	ror	r8			;Differenz halbieren
	brcc	m1d240			;Runden? nein -> weiter
	inc	r8			;sonst aufrunden
	brne	m1d240			;bertrag? nein -> weiter
	inc	r9			;sonst bertrag addieren
m1d240:	add	r20,r20			;Skalierungsfaktor verdoppeln
	rjmp	m1d230			;Schleife
m1d250:	sts	grsca1,r20		;Skalierungsfaktor 1 speichern
;
m1d260:	movw	r16,r12			;Maxwert Sensornummer 2 holen
	cpi	r17,high(-1000)		;wurde Anfangswert H verndert?
	brne	m1d270			;ja -> weiter
	cpi	r16,low(-1000)		;wurde Anfangswert L verndert?
	brne	m1d270			;ja -> weiter
	rjmp	m1d400			;sonst Berechnung berspringen
;
m1d270:	sub	r12,r14			;Differenz
	sbc	r13,r15			;Maximalwert - Minimalwert
	movw	r16,r12			;Differenz kopieren
	asr	r17
	ror	r16			;Differenz halbieren
	brcc	m1d280			;Runden? nein -> weiter
	inc	r16			;sonst aufrunden
	brne	m1d280			;bertrag? nein -> weiter
	inc	r17			;sonst bertrag addieren
m1d280:	add	r16,r14
	adc	r17,r15			;zum Minimalwert addieren
	sts	grcen2,r16
	sts	grcen2+1,r17		;Y-Mittelpunktwert 2 speichern
;
	ldi	r16,41			;Y-Bereich fr Diagramm 2
	ldi	r17,0			;41 Pixel
	ldi	r20,1			;Skalierungsfaktor= 1 setzen
m1d290:	movw	r18,r16			;Bereichswert kopieren
	sub	r18,r12
	sbc	r19,r13			;Bereich berschritten?
	brcc	m1d310			;nein -> weiter
	asr	r13			;sonst
	ror	r12			;Differenz halbieren
	brcc	m1d300			;Runden? nein -> weiter
	inc	r12			;sonst aufrunden
	brne	m1d300			;bertrag? nein -> weiter
	inc	r13			;sonst bertrag addieren
m1d300:	add	r20,r20			;Skalierungsfaktor verdoppeln
	rjmp	m1d290			;Schleife
;
m1d310:	sts	grsca2,r20		;Skalierungsfaktor 2 speichern
	lds	r16,grflag		;Grafik-Flags holen
	bst	r16,1			;Temperaturskala angleichen?
	brtc	m1d400			;nein -> berspringen
;
	lds	r19,grsca1		;sonst Skalierungsfaktor 1 hol.
	cp	r19,r20			;Skalierungsfaktoren vergleich.
	breq	m1d400			;Faktoren gleich? ja -> Ende
	brcc	m1d320			;Faktor 1 grer? ja -> weiter
	mov	r19,r20			;sonst Faktor 2 verwenden
	rjmp	m1d330
m1d320:	mov	r20,r19			;Faktor 1 verwenden
m1d330:	sts	grsca1,r19		;Skalierungsfaktor 1 speichern
	sts	grsca2,r20		;Skalierungsfaktor 2 speichern
;
; Anzeigemodus Grafik: Grafiken zeichnen, Y-Pixelposition berechnen
; Y = Y-Mittelposition - ((Tempwert - Y-Mittelpunktwert) / Skalierung)
;
m1d400:	clr	r16
	sts	grlax1,r16		;letzte
	sts	grlay1,r16		;Pixelkoordinaten 1 lschen
	sts	grlax2,r16		;letzte
	sts	grlay2,r16		;Pixelkoordinaten 2 lschen
	ldi	zl,low(grbuff)
	ldi	zh,high(grbuff)		;Basisadresse Grafikdatenpuffer
	ldi	r20,0			;Anfangswert Pufferzhler
	ldi	r16,59			;Y-Position 1 Mittelwert=59
	lds	r18,grflag		;Grafik-Flags holen
	bst	r18,4			;Doppeldiagramm?
	brtc	m1d410			;nein -> Y-Position ok
	ldi	r16,37			;sonst Y-Position Mittelwert=37
m1d410:	ldi	r17,81			;Y-Position 2 Mittelwert=81
	movw	r8,r16			;Y-Positionen kopieren
	lds	r10,grsca1		;Skalierungsfaktor 1 holen
	lds	r11,grsca2		;Skalierungsfaktor 2 holen
	lds	r12,grcen1
	lds	r13,grcen1+1		;Y-Mittelpunktwert 1 holen
	lds	r14,grcen2
	lds	r15,grcen2+1		;Y-Mittelpunktwert 2 holen
;
; Anzeigemodus Grafik: Pixel von Messwert 1 berechnen
;
m1d420:	adiw	zl,4			;Datum/Zeit berspringen
	ld	r16,z+			;Temperaturwert 1 L holen
	ld	r17,z+			;Temperaturwert 1 H holen
	cpi	r17,0x80		;Wert ungltig?
	brne	m1d430			;nein -> bearbeiten
	rjmp	m1d550			;sonst keine Ausgabe
;
m1d430:	sub	r16,r12
	sbc	r17,r13			;Temperaturwert - Mittelpunktw.
	mov	r18,r10			;Skalierungsfaktor 1 holen
m1d440:	lsr	r18			;Halbierung notwendig?
	breq	m1d450			;nein -> weiter
	asr	r17
	ror	r16			;sonst Anzeigewert halbieren
	brcc	m1d440			;Runden? nein -> Schleife
	inc	r16			;sonst aufrunden
	brne	m1d440			;bertrag? nein -> Schleife
	inc	r17			;sonst bertrag addieren
	rjmp	m1d440			;Schleife
;
m1d450:	mov	r19,r8			;Y-Position Mittelwert kopieren
	sub	r19,r16			;Y-Pixel-Position ermitteln
	sbr	r19,0x80		;Bit7 setzen (Pixel setzen)
	ldi	r18,16			;X Position Diagrammanfang
	add	r18,r20			;Pufferposition addieren
	call	pixlcd			;Pixel ausgeben
;
; Anzeigemodus Grafik: Pixel von Grafik 1 verbinden
;
	lds	r16,grlax1		;sonst letzte X-Koordinate hol.
	lds	r17,grlay1		;letzte Y-Koordinate holen
	sts	grlax1,r18		;aktuelle X-Koordinate speich.
	sts	grlay1,r19		;aktuelle Y-Koordinate speich.
	bst	r17,7			;Koordinate gltig?
	brtc	m1d550			;nein -> berspringen
	sub	r18,r16			;X-Differenz berechnen
	sub	r19,r17			;Y-Differenz berechnen
	mov	r21,r19			;Y-Differenz kopieren
	mov	r22,r19			;Y-Differenz nochmals kopieren
	brcc	m1d460			;Differenz positiv? ja -> weit.
	neg	r22			;in positiven Wert wandeln
m1d460:	mov	r23,r22			;positive Y-Differenz kopieren
	or	r23,r18			;mit X-Differenz verknpfen
	cpi	r23,2			;Differenz<2?
	brcs	m1d550			;ja -> keine Pixel zu verbinden
	mov	r23,r18			;X-Differenz kopieren
	movw	r18,r16			;Ursprungskoordinaten kopieren
	ldi	r24,1			;Zykluszhler Startwert
	cp	r23,r22			;Y-Differenz>X-Differenz?
	brcs	m1d500			;ja -> bearbeiten
;
	mov	r25,r23			;X-Differenz kopieren
	lsr	r25			;halbieren -> Startwert Y-Zhl.
m1d470:	inc	r18			;X-Koordinate erhhen
	sub	r25,r22			;Y-Zhler - Y-Differenz
	brcc	m1d490			;Y-Zhler negativ? n. -> weiter
	add	r25,r23			;sonst X-Differenz addieren
	bst	r21,7			;Y-Differenz negativ?
	brts	m1d480			;ja -> Y-Koordinate vermindern
	inc	r19			;sonst Y-Koordinate erhhen
	rjmp	m1d490
m1d480:	dec	r19			;Y-Koordinate vermindern
m1d490:	call	pixlcd			;Pixel ausgeben
	inc	r24			;nchster Zyklus
	cp	r24,r23			;alle Pixel ausgegeben?
	brne	m1d470			;nein -> Schleife
	rjmp	m1d550
;
m1d500:	mov	r25,r22			;Y-Differenz kopieren
	lsr	r25			;halbieren -> Startwert X-Zhl.
m1d510:	bst	r21,7			;Y-Differenz negativ?
	brts	m1d520			;ja -> Y-Koordinate vermindern
	inc	r19			;sonst Y-Koordinate erhhen
	rjmp	m1d530
m1d520:	dec	r19			;Y-Koordinate vermindern
m1d530:	sub	r25,r23			;X-Zhler - X-Differenz
	brcc	m1d540			;Y-Zhler negativ? n. -> weiter
	add	r25,r22			;sonst Y-Differenz addieren
	inc	r18			;X-Koordinate erhhen
m1d540:	call	pixlcd			;Pixel ausgeben
	inc	r24			;nchster Zyklus
	cp	r24,r22			;alle Pixel ausgegeben?
	brne	m1d510			;nein -> Schleife
;
; Anzeigemodus Grafik: Pixel von Messwert 2 berechnen
;
m1d550:	ld	r16,z+			;Temperaturwert 2 L holen
	ld	r17,z+			;Temperaturwert 2 H holen
	cpi	r17,0x80		;Wert ungltig?
	brne	m1d560			;nein -> bearbeiten
	rjmp	m1d680			;sonst keine Ausgabe
;
m1d560:	sub	r16,r14
	sbc	r17,r15			;Temperaturwert - Mittelpunktw.
	mov	r18,r11			;Skalierungsfaktor 2 holen
m1d570:	lsr	r18			;Halbierung notwendig?
	breq	m1d580			;nein -> weiter
	asr	r17
	ror	r16			;sonst Anzeigewert halbieren
	brcc	m1d570			;Runden? nein -> Schleife
	inc	r16			;sonst aufrunden
	brne	m1d570			;bertrag? nein -> Schleife
	inc	r17			;sonst bertrag addieren
	rjmp	m1d570			;Schleife
;
m1d580:	mov	r19,r9			;Y-Position Mittelwert kopieren
	sub	r19,r16			;Y-Pixel-Position ermitteln
	sbr	r19,0x80		;Bit7 setzen (Pixel setzen)
	ldi	r18,16			;X Position Diagrammanfang
	add	r18,r20			;Pufferposition addieren
	call	pixlcd			;Pixel ausgeben
;
; Anzeigemodus Grafik: Pixel von Grafik 2 verbinden
;
	lds	r16,grlax2		;sonst letzte X-Koordinate hol.
	lds	r17,grlay2		;letzte Y-Koordinate holen
	sts	grlax2,r18		;aktuelle X-Koordinate speich.
	sts	grlay2,r19		;aktuelle Y-Koordinate speich.
	bst	r17,7			;Koordinate gltig?
	brtc	m1d680			;nein -> berspringen
	sub	r18,r16			;X-Differenz berechnen
	sub	r19,r17			;Y-Differenz berechnen
	mov	r21,r19			;Y-Differenz kopieren
	mov	r22,r19			;Y-Differenz nochmals kopieren
	brcc	m1d590			;Differenz positiv? ja -> weit
	neg	r22			;in positiven Wert wandeln
m1d590:	mov	r23,r22			;positive Y-Differenz kopieren
	or	r23,r18			;mit X-Differenz verknpfen
	cpi	r23,2			;Differenz<2?
	brcs	m1d680			;ja -> keine Pixel zu verbinden
	mov	r23,r18			;X-Differenz kopieren
	movw	r18,r16			;Ursprungskoordinaten kopieren
	ldi	r24,1			;Zykluszhler Startwert
	cp	r23,r22			;Y-Differenz>X-Differenz?
	brcs	m1d630			;ja -> bearbeiten
;
	mov	r25,r23			;X-Differenz kopieren
	lsr	r25			;halbieren -> Startwert Y-Zhl.
m1d600:	inc	r18			;X-Koordinate erhhen
	sub	r25,r22			;Y-Zhler - Y-Differenz
	brcc	m1d620			;Y-Zhler negativ? n. -> weiter
	add	r25,r23			;sonst X-Differenz addieren
	bst	r21,7			;Y-Differenz negativ?
	brts	m1d610			;ja -> Y-Koordinate vermindern
	inc	r19			;sonst Y-Koordinate erhhen
	rjmp	m1d620
m1d610:	dec	r19			;Y-Koordinate vermindern
m1d620:	call	pixlcd			;Pixel ausgeben
	inc	r24			;nchster Zyklus
	cp	r24,r23			;alle Pixel ausgegeben?
	brne	m1d600			;nein -> Schleife
	rjmp	m1d680
;
m1d630:	mov	r25,r22			;Y-Differenz kopieren
	lsr	r25			;halbieren -> Startwert X-Zhl.
m1d640:	bst	r21,7			;Y-Differenz negativ?
	brts	m1d650			;ja -> Y-Koordinate vermindern
	inc	r19			;sonst Y-Koordinate erhhen
	rjmp	m1d660
m1d650:	dec	r19			;Y-Koordinate vermindern
m1d660:	sub	r25,r23			;X-Zhler - X-Differenz
	brcc	m1d670			;Y-Zhler negativ? n. -> weiter
	add	r25,r22			;sonst Y-Differenz addieren
	inc	r18			;X-Koordinate erhhen
m1d670:	call	pixlcd			;Pixel ausgeben
	inc	r24			;nchster Zyklus
	cp	r24,r22			;alle Pixel ausgegeben?
	brne	m1d640			;nein -> Schleife
;
m1d680:	inc	r20			;Pufferposition erhhen
	cpi	r20,222			;letzter Y-Wert?
	breq	m1e000			;ja -> Ende
	rjmp	m1d420			;sonst Schleife
;
; Anzeigemodus Grafik: Beschriftung der Y-Achse von Diagramm 1
;
m1e000:	ldi	r16,4
	mov	txflag,r16		;Mini-Schrift einstellen
	lds	r16,grsca1		;Skalierungsfaktor 1 holen
	tst	r16			;gltiger Wert?
	brne	m1e010			;ja -> bearbeiten
	rjmp	m1e200			;sonst weiter zu Diagramm 2
;
m1e010:	ldi	r25,102			;untere Y-Position= 102
	lds	r16,grflag		;Grafik-Flags holen
	bst	r16,4			;Doppeldiagramm?
	brtc	m1e020			;nein -> untere Position ok
	ldi	r25,58			;sonst untere Y-Position= 58
;
m1e020:	mov	r16,r25			;untere Position kopieren
	sub	r16,r8			;Y-Position Mittelwert subtrah.
	mul	r16,r10			;mit Skalierung 1 multiplizier.
	movw	r16,r12			;Y-Mittelpunktwert 1 holen
	sub	r16,r0			;Ergebnis vom Mittelpunktwert
	sbc	r17,r1			;subtrahieren = kleinster Wert
;
m1e030:	lds	r21,grsca1		;Skalierungsfaktor 1 holen
	call	grchek			;Wert passend zur Skalierung?
	breq	m1e040			;ja -> weiter
	inc	r16			;sonst Wert +1
	brne	m1e030			;bertrag? nein -> weiter
	inc	r17			;sonst H-Byte erhhen
	rjmp	m1e030			;Schleife
;
m1e040:	sts	grysva,r16
	sts	grysva+1,r17		;Y-Skalenwert speichern
	lds	r18,grcen1
	lds	r19,grcen1+1		;Mittelpunktwert holen
	sub	r16,r18
	sbc	r17,r19			;Temperaturwert - Mittelpunktw.
	lds	r18,grsca1		;Skalierungsfaktor 1 holen
m1e050:	lsr	r18			;Halbierung notwendig?
	breq	m1e060			;nein -> weiter
	asr	r17
	ror	r16			;sonst Anzeigewert halbieren
	brcc	m1e050			;Runden? nein -> Schleife
	inc	r16			;sonst aufrunden
	brne	m1e050			;bertrag? nein -> Schleife
	inc	r17			;sonst bertrag addieren
	rjmp	m1e050			;Schleife
;
m1e060:	mov	r19,r8			;Y-Position Mittelwert kopieren
	sub	r19,r16			;Y-Pixel-Position ermitteln
	cpi	r19,16			;Y-Bereich berschritten?
	brcs	m1e200			;ja -> Ende
	mov	r21,r19			;neue Y-Skalenposition kopieren
	sbr	r19,0x80		;Bit7 setzen (Pixel setzen)
	ldi	r18,14			;X Position Gradmarkierung
	call	pixlcd			;Pixel ausgeben
	subi	r21,2			;Y-Position von Skalenwert
	cpi	r21,16			;Y-Bereich berschritten?
	brcs	m1e200			;ja -> Ende
;
	mov	r20,r25			;untere Y-Position kopieren
	cpi	r20,102			;groes Diagramm?
	breq	m1e070			;ja -> weiter
	dec	r20			;Korrektur fr oberes Diagramm
m1e070:	subi	r20,2			;fr Skalentext korrigieren
	cp	r21,r20			;unterer Y-Bereich berschrit.?
	brcc	m1e080			;ja -> keinen Skalentext ausg.
;
	lds	r16,grysva		;sonst
	lds	r17,grysva+1		;letzten Y-Skalenwert holen
	ldi	r18,low(-1000)
	ldi	r19,high(-1000)		;Vergleichswert -100 Grad laden
	sub	r18,r16			;Wert noch darstellbar (> -100
	sbc	r19,r17			;Grad)?
	brpl	m1e080			;nein -> Ausgabe berspringen
;
	call	temasc			;in ASCII wandeln
	ldi	r20,1			;X-Position
	mov	r22,r4			;Hunderter holen
	call	asclcd			;ausgeben
	mov	r22,r5			;Zehner holen
	call	asclcd			;ausgeben
	mov	r22,r6			;Einer holen
	call	asclcd			;ausgeben
;
m1e080:	lds	r16,grysva
	lds	r17,grysva+1		;letzten Y-Skalenwert holen
	lds	r21,grsca1		;Skalierungsfaktor 1 holen
	call	grysca			;nchsten Skalenwert ermitteln
	rjmp	m1e040
;
; Anzeigemodus Grafik: Beschriftung der Y-Achse von Diagramm 2
;
m1e200:	lds	r16,grsca2		;Skalierungsfaktor 2 holen
	tst	r16			;gltiger Wert?
	brne	m1e210			;ja -> bearbeiten
	rjmp	m1e400			;sonst Ende
;
m1e210:	lds	r16,grflag		;Grafik-Flags holen
	bst	r16,4			;Doppelgrafik?
	brts	m1e220			;ja -> bearbeiten
	rjmp	m1e400			;sonst Ende
;
m1e220:	ldi	r16,102			;untere Y-Position= 102
	sub	r16,r9			;Y-Position Mittelwert subtrah.
	mul	r16,r11			;mit Skalierung 2 multiplizier.
	lds	r16,grcen2
	lds	r17,grcen2+1		;Y-Mittelpunktwert 2 holen
	sub	r16,r0			;Ergebnis vom Mittelpunktwert
	sbc	r17,r1			;subtrahieren = kleinster Wert
;
m1e230:	lds	r21,grsca2		;Skalierungsfaktor 2 holen
	call	grchek			;Wert passend zur Skalierung?
	breq	m1e240			;ja -> weiter
	inc	r16			;sonst Wert +1
	brne	m1e230			;bertrag? nein -> weiter
	inc	r17			;sonst H-Byte erhhen
	rjmp	m1e230			;Schleife
;
m1e240:	sts	grysva,r16
	sts	grysva+1,r17		;Y-Skalenwert speichern
	lds	r18,grcen2
	lds	r19,grcen2+1		;Mittelpunktwert holen
	sub	r16,r18
	sbc	r17,r19			;Temperaturwert - Mittelpunktw.
	lds	r18,grsca2		;Skalierungsfaktor 2 holen
m1e250:	lsr	r18			;Halbierung notwendig?
	breq	m1e260			;nein -> weiter
	asr	r17
	ror	r16			;sonst Anzeigewert halbieren
	brcc	m1e250			;Runden? nein -> Schleife
	inc	r16			;sonst aufrunden
	brne	m1e250			;bertrag? nein -> Schleife
	inc	r17			;sonst bertrag addieren
	rjmp	m1e250			;Schleife
;
m1e260:	mov	r19,r9			;Y-Position Mittelwert kopieren
	sub	r19,r16			;Y-Pixel-Position ermitteln
	cpi	r19,60			;Y-Bereich berschritten?
	brcs	m1e400			;ja -> Ende
	mov	r21,r19			;neue Y-Skalenposition kopieren
	sbr	r19,0x80		;Bit7 setzen (Pixel setzen)
	ldi	r18,14			;X-Position Gradmarkierung
	call	pixlcd			;Pixel ausgeben
	subi	r21,2			;Y-Position von Skalenwert
	cpi	r21,60			;Y-Bereich berschritten?
	brcs	m1e400			;ja -> Ende
;
	cpi	r21,100			;unterer Y-Bereich berschrit.?
	brcc	m1e280			;ja -> keinen Skalentext ausg.
	lds	r16,grysva		;sonst
	lds	r17,grysva+1		;letzten Y-Skalenwert holen
	ldi	r18,low(-1000)
	ldi	r19,high(-1000)		;Vergleichswert -100 Grad laden
	sub	r18,r16			;Wert noch darstellbar (> -100
	sbc	r19,r17			;Grad)?
	brpl	m1e280			;nein -> Ausgabe berspringen

	call	temasc			;in ASCII wandeln
	ldi	r20,1			;X-Position
	mov	r22,r4			;Hunderter holen
	call	asclcd			;ausgeben
	mov	r22,r5			;Zehner holen
	call	asclcd			;ausgeben
	mov	r22,r6			;Einer holen
	call	asclcd			;ausgeben
;
m1e280:	lds	r16,grysva
	lds	r17,grysva+1		;letzten Y-Skalenwert holen
	lds	r21,grsca2		;Skalierungsfaktor 2 holen
	call	grysca			;nchsten Skalenwert ermitteln
	rjmp	m1e240
;
; Anzeigemodus Grafik: Beschriftung der X-Achse, 15min-Werte
;
m1e400:	lds	r16,grsca1		;Skalierungsfaktor 1 holen
	lds	r17,grsca2		;Skalierungsfaktor 2 holen
	or	r16,r17			;mindestens 1 Wert gltig?
	brne	m1e405			;ja -> weiter
	rjmp	m1e800			;sonst Ende
;
m1e405:	ldi	yl,low(grbuff)
	ldi	yh,high(grbuff)		;Basisadresse Grafikdatenpuffer
	ldi	r25,0			;Anfangswert Pufferzhler
	lds	r16,grmod1		;Anzeigemodus Sensornummer 1
	tst	r16			;15-Minuten-Werte anzeigen?
	breq	m1e410			;ja -> bearbeiten
	rjmp	m1e600			;sonst 24h-Werte bearbeiten
;
m1e410:	ld	r8,y+
	ld	r9,y+
	ld	r10,y+
	ld	r11,y+			;Zeitdaten holen
	mov	r16,r8			;Minutenwert kopieren
	or	r16,r9			;mit Stundenwert verknpfen
	brne	m1e420			;Tageswechsel? nein -> weiter
;
	ldi	r18,16			;X-Anfangsposition
	add	r18,r25			;Pufferposition addieren
	ldi	r19,104|0x80		;Y-Position Skalenmarkierung
	call	pixlcd			;Skalenmarkierung setzen
	mov	r20,r18			;X-Position holen
	subi	r20,9			;Position fr Datumtext
	cpi	r20,221			;X-Endposition erreicht?
	brcc	m1e450			;ja -> Datumausgabe berspring.
	ldi	r21,106			;sonst Y-Position fr Datumtext
	call	grelcd			;Datum ausgeben
	rjmp	m1e450
;
m1e420:	mov	r16,r8			;Minutenwert kopieren
	tst	r16			;Minutenwert=0?
	brne	m1e450			;nein -> keine Zeit ausgeben
	mov	r16,r9			;sonst Stundenwert kopieren
	andi	r16,0x03		;Stundenwert /4 teilbar?
	brne	m1e450			;nein -> keine Zeit ausgeben
;
	ldi	r18,16			;X-Anfangsposition
	add	r18,r25			;Pufferposition addieren
	ldi	r19,104|0x80		;Y-Position Skalenmarkierung
	call	pixlcd			;Skalenmarkierung setzen
	mov	r20,r18			;X-Position holen
	subi	r20,2			;X-Position fr Zeittext-Einer
	ldi	r21,106			;Y-Position fr Zeittext
;
	mov	r18,r9			;Stundenwert holen
	call	bytasc			;in ASCII wandeln
	cpi	r17,'0'			;Zehner=0?
	brne	m1e430			;nein -> weiter
	cpi	r20,235			;X-Endposition erreicht?
	brcc	m1e450			;ja -> Zeitausgabe berspringen
	mov	r22,r16			;sonst Einer holen
	rjmp	m1e440			;ausgeben
m1e430:	subi	r20,2			;X-Position fr Zeittext-Zehner
	cpi	r20,231			;X-Endposition erreicht?
	brcc	m1e450			;ja -> Zeitausgabe berspringen
	mov	r13,r16			;Einer sichern
	mov	r22,r17			;Zehner holen
	call	asclcd			;ausgeben
	mov	r22,r13			;gesicherte Einer holen
m1e440:	call	asclcd			;ausgeben
;
m1e450:	adiw	yl,4			;Datenbytes berspringen
	inc	r25			;Pufferzhler erhhen
	cpi	r25,222			;letzte Position erreicht?
	breq	m1e800			;ja -> Ende
	rjmp	m1e410			;sonst Schleife
;
; Anzeigemodus Grafik: Beschriftung der X-Achse, 24h-Werte
;
m1e600:	adiw	yl,2			;Stunden/Minuten berspringen
	ld	r10,y+
	ld	r11,y+			;Zeitdaten Byte 2+3 holen
	mov	r16,r10			;Zeitbyte 2 kopieren
	andi	r16,0x1f		;Tageswert filtern
	cpi	r16,1			;Tag 1 (Monatsanfang)?
	brne	m1e610			;nein -> weiter
;
	ldi	r18,16			;sonst X-Anfangsposition
	add	r18,r25			;Pufferposition addieren
	ldi	r19,104|0x80		;Y-Position Skalenmarkierung
	call	pixlcd			;Skalenmarkierung setzen
	mov	r20,r18			;X-Position holen
	subi	r20,9			;Position fr Datumtext
	cpi	r20,222			;X-Endposition erreicht?
	brcc	m1e620			;ja -> Datumausgabe berspring.
	ldi	r21,106			;sonst Y-Position fr Datumtext
	call	grelcd			;Datum ausgeben
	rjmp	m1e620
;
m1e610:	cpi	r16,15			;Tag 15 (Monatsmitte)?
	brne	m1e620			;nein -> Ende
	ldi	r18,16			;X-Anfangsposition
	add	r18,r25			;Pufferposition addieren
	ldi	r19,104|0x80		;Y-Position Skalenmarkierung
	call	pixlcd			;Skalenmarkierung setzen
;
m1e620:	adiw	yl,4			;Datenbytes berspringen
	inc	r25			;Pufferzhler erhhen
	cpi	r25,222			;letzte Position erreicht?
	brne	m1e600			;nein -> Schleife
;
; Anzeigemodus Grafik: Beschriftung der Diagramme mit Sensornamen; es
; wird ermittelt, ob die ersten 60 Grafikpuffer-Werte schwerpunktmig
; ber oder unter dem Mittelpunktwert liegen; die Ausgabe der Texte
; erfolgt davon abhngig oben oder unten
;
m1e800:	clr	txflag			;Normalschrift einstellen
	lds	r16,grflag		;Grafik-Flags holen
	bst	r16,0			;Sensornamen anzeigen?
	brts	m1e810			;ja -> bearbeiten
	rjmp	m1e940			;sonst Ende
;
m1e810:	ldi	zl,low(grbuff)
	ldi	zh,high(grbuff)		;Basisadresse Grafikdatenpuffer
	lds	r8,grcen1
	lds	r9,grcen1+1		;Y-Mittelpunktwert 1 holen
	lds	r10,grcen2
	lds	r11,grcen2+1		;Y-Mittelpunktwert 2 holen
	clr	r12
	clr	r13			;Mittelwert 1 lschen
	clr	r14
	clr	r15			;Mittelwert 2 lschen
	ldi	r20,0			;Anfangswert Pufferzhler
;
m1e820:	adiw	zl,4			;Datum/Zeit berspringen
	ld	r16,z+			;Temperaturwert 1 L holen
	ld	r17,z+			;Temperaturwert 1 H holen
	cpi	r17,0x80		;Wert ungltig?
	breq	m1e830			;ja -> nicht auswerten
	sub	r16,r8
	sbc	r17,r9			;Mittelpunktwert 1 subtrahieren
	add	r12,r16
	adc	r13,r17			;zum Mittelwert 1 addieren
;
m1e830:	ld	r16,z+			;Temperaturwert 2 L holen
	ld	r17,z+			;Temperaturwert 2 H holen
	cpi	r17,0x80		;Wert ungltig?
	breq	m1e840			;ja -> nicht auswerten
	sub	r16,r10
	sbc	r17,r11			;Mittelpunktwert 2 subtrahieren
	add	r14,r16
	adc	r15,r17			;zum Mittelwert 2 addieren
;
m1e840:	inc	r20			;Pufferposition erhhen
	cpi	r20,60			;letzter relevanter Y-Wert?
	brne	m1e820			;nein -> Schleife
;
	mov	r25,r15			;Mittelwert 2 H-Byte sichern
	ldi	r20,17			;X-Position fr Sensorname 1
	bst	r13,7			;Mittelwert 1 negativ?
	brtc	m1e850			;nein -> Sensorname nach unten
	ldi	r21,16			;sonst obere Y-Position=16
	rjmp	m1e860			;Sensorname ausgeben
;
m1e850:	ldi	r21,91			;untere Y-Position=91
	lds	r16,grflag		;Grafik-Flags holen
	bst	r16,4			;Doppeldiagramm?
	brtc	m1e860			;nein -> Y-Position ok
	ldi	r21,47			;sonst untere Y-Position=47

m1e860:	lds	r18,grflag		;Grafik-Flags holen
	bst	r18,2			;Differenz-Modus aktiv?
	brtc	m1e870			;nein -> weiter
	ldi	r22,0x97		;sonst Delta-Zeichen laden
	call	asclcd			;ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
;
m1e870:	lds	r18,grsen1		;Grafik 1 Sensornummern holen
	andi	r18,0x0f		;unteres Nibble filtern
	ldi	r19,9			;Sensornamenlnge im EEPROM
	mul	r18,r19			;Tabellenoffset ermitteln
	ldi	r16,low(etemtx)		;EEPROM-Adresse Sensornamen L
	ldi	r17,high(etemtx)	;EEPROM-Adresse Sensornamen H
	add	r16,r0			;Offset zur Adresse L addieren
	adc	r17,r1			;Offset zur Adresse H addieren
	call	eeplcd			;EEPROM-Text ausgeben
;
	lds	r18,grflag		;Grafik-Flags holen
	bst	r18,2			;Differenz-Modus aktiv?
	brtc	m1e880			;nein -> weiter
	ldi	r22,'/'			;sonst Schrgstrich laden
	call	asclcd			;ausgeben
	lds	r18,grsen1		;Grafik 1 Sensornummern holen
	swap	r18			;Differenzsensor holen
	andi	r18,0x0f		;unteres Nibble filtern
	ldi	r19,9			;Sensornamenlnge im EEPROM
	mul	r18,r19			;Tabellenoffset ermitteln
	ldi	r16,low(etemtx)		;EEPROM-Adresse Sensornamen L
	ldi	r17,high(etemtx)	;EEPROM-Adresse Sensornamen H
	add	r16,r0			;Offset zur Adresse L addieren
	adc	r17,r1			;Offset zur Adresse H addieren
	call	eeplcd			;EEPROM-Text ausgeben
;
m1e880:	lds	r16,grmod1		;Anzeigemodus Sensornummer 1
	tst	r16			;15-Minuten-Werte anzeigen?
	breq	m1e890			;ja -> weiter
	ldi	r22,0xa0		;sonst schmales Leerzeichen
	call	asclcd			;ausgeben
	lds	r16,grmod1		;Anzeigemodus Sensornummer 1
	ldi	r22,0x98-1		;Zeichen fr Minimalwert laden
	add	r22,r16			;Anzeigemodus addieren; Min,
	call	asclcd			;Max oder Avg-Zeichen ausgeben
m1e890:	ldi	r22,0x7f		;Ende-Zeichen
	call	asclcd			;ausgeben
;
	lds	r18,grflag		;Grafik-Flags holen
	bst	r18,4			;Doppelgrafik?
	brtc	m1e940			;nein -> Ende
;
	ldi	r20,17			;X-Position fr Sensorname 2
	ldi	r21,91			;untere Y-Position=91
	bst	r25,7			;Mittelwert 2 negativ?
	brtc	m1e900			;nein -> Sensorname nach unten
	ldi	r21,61			;sonst obere Y-Position=61
m1e900:	lds	r18,grflag		;Grafik-Flags holen
	bst	r18,3			;Differenz-Modus aktiv?
	brtc	m1e910			;nein -> weiter
	ldi	r22,0x97		;sonst Delta-Zeichen laden
	call	asclcd			;ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
;
m1e910:	lds	r18,grsen2		;Grafik 2 Sensornummern
	andi	r18,0x0f		;unteres Nibble filtern
	ldi	r19,9			;Sensornamenlnge im EEPROM
	mul	r18,r19			;Tabellenoffset ermitteln
	ldi	r16,low(etemtx)		;EEPROM-Adresse Sensornamen L
	ldi	r17,high(etemtx)	;EEPROM-Adresse Sensornamen H
	add	r16,r0			;Offset zur Adresse L addieren
	adc	r17,r1			;Offset zur Adresse H addieren
	call	eeplcd			;EEPROM-Text ausgeben
;
	lds	r18,grflag		;Grafik-Flags holen
	bst	r18,3			;Differenz-Modus aktiv?
	brtc	m1e920			;nein -> weiter
	ldi	r22,'/'			;sonst Schrgstrich laden
	call	asclcd			;ausgeben
	lds	r18,grsen2		;Grafik 2 Sensornummern holen
	swap	r18			;Differenzsensor holen
	andi	r18,0x0f		;unteres Nibble filtern
	ldi	r19,9			;Sensornamenlnge im EEPROM
	mul	r18,r19			;Tabellenoffset ermitteln
	ldi	r16,low(etemtx)		;EEPROM-Adresse Sensornamen L
	ldi	r17,high(etemtx)	;EEPROM-Adresse Sensornamen H
	add	r16,r0			;Offset zur Adresse L addieren
	adc	r17,r1			;Offset zur Adresse H addieren
	call	eeplcd			;EEPROM-Text ausgeben
;
m1e920:	lds	r16,grmod2		;Anzeigemodus Sensornummer 2
	tst	r16			;15-Minuten-Werte anzeigen?
	breq	m1e930			;ja -> weiter
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
	lds	r16,grmod2		;Anzeigemodus Sensornummer 2
	ldi	r22,0x98-1		;Zeichen fr Minimalwert laden
	add	r22,r16			;Anzeigemodus addieren; Min,
	call	asclcd			;Max oder Avg-Zeichen ausgeben
m1e930:	ldi	r22,0x7f		;Ende-Zeichen
	call	asclcd			;Zeichen ausgeben
;
m1e940:	lds	r16,xflag3		;verschiedene Flags 3 holen
	cbr	r16,1<<2		;Grafikausgabe beendet
	sts	xflag3,r16		;Flags wieder speichern
	rjmp	m20000
;
; Anzeigemodus Grafik: Fadenkreuz neu ausgeben
;
m1f000:	cpi	r16,24			;Anzeigemodus Grafik/Fadenkr.?
	breq	m1f010			;ja -> weiter
	rjmp	m1g000			;sonst weiter testen
;
m1f010:	lds	r18,grpoin		;aktuelle Fadenkreuzpos. holen
	lds	r19,encval		;aktuellen Encoderwert holen
	cp	r18,r19			;Positionsnderung?
	breq	m1f100			;nein -> keine Ausgabe
	sts	grlacr,r19		;Fadenkreuzposition sichern
	sts	grpoin,r19		;neue Fadenkreuzposition setzen
	call	grscrr			;Grafik wieder herstellen
	lds	r18,grpoin		;aktuelle Fadenkreuzpos. holen
	call	grscxp			;X-Koordinaten fr Fadenkreuz
	lds	r18,grpoin		;aktuelle Fadenkreuzpos. holen
	call	grscrb			;Grafik sichern
	lds	r18,grpoin		;aktuelle Fadenkreuzpos. holen
	call	grcros			;Fadenkreuz ausgeben
m1f100:	rjmp	m20000
;
; Anzeigemodus Konfig/Grafik-Presets: Anzeige komplett neu ausgeben
;
m1g000:	cpi	r16,25			;Anzeigemodus Konfig/Grafik?
	breq	m1g010			;ja -> weiter
	rjmp	m1g500			;sonst weiter testen
;
m1g010:	clr	txflag			;Text-Flag lschen
	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,1			;Anzeige neu ausgeben?
	brtc	m1g100			;nein -> weiter
	cbr	r16,0x02		;sonst Neuausgabe-Flag lschen
	sts	dflags,r16		;und Flags wieder speichern
	call	newcfg			;alle Konfigzeilen neu ausgeben
;
; Anzeigemodus Konfig/Grafik-Presets: Namen ausgeben
;
m1g100:	ldi	r25,0			;Zeilen-Zhler = 0
	ldi	yl,low(cfgflg)
	ldi	yh,high(cfgflg)		;Zeiger auf Konfigurationsflags
m1g110:	ldi	r20,2			;X-Startposition
	ldi	r21,16			;Y-Startposition
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r25,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
	movw	r12,r20			;X/Y-Startpositionen sichern
;
	ld	r16,y			;Konfigurationflags holen
	bst	r16,0			;Name neu ausgeben?
	brtc	m1g140			;nein -> berspringen
	cbr	r16,0x01		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
;
	lds	r19,menpos		;ausgewhlte Position holen
	dec	r19			;korrigieren
	cp	r25,r19			;Position identisch?
	brne	m1g120			;nein -> weiter
	ldi	r16,0x01		;sonst
	mov	txflag,r16		;Textflag invers setzen
m1g120:	ldi	r22,0xa0		;Text "Preset"
	cpi	r25,0			;Zhler auf Position 1?
	breq	m1g130			;ja -> Text ausgeben
	ldi	r22,0x5c		;Text "Name"
	cpi	r25,1			;Zhler auf Position 2?
	breq	m1g130			;ja -> Text ausgeben
	ldi	r22,0xa2		;sonst Basiswert laden
	add	r22,r25			;Position addieren -> Text-Nr.
m1g130:	call	strlcd			;Text ausgeben
	clr	txflag
;
; Anzeigemodus Konfig/Grafik-Presets: Daten ausgeben
;
m1g140:	ld	r16,y			;Konfigurationflags holen
	bst	r16,1			;Daten neu ausgeben?
	brts	m1g150			;ja -> weiter
	rjmp	m1g290			;sonst berspringen
;
m1g150:	cbr	r16,0x02		;sonst nderungs-Flag lschen
	st	y,r16			;Flags wieder speichern
	movw	r20,r12			;X/Y-Koordinaten restaurieren
	ldi	r20,128			;X-Position fr Daten
;
	cpi	r25,0			;Zhler auf Preset-Nummer?
	brne	m1g160			;nein -> weiter testen
	lds	r18,grpres		;aktuelle Preset-Nummer holen
	inc	r18			;korrigieren
	call	bytlcd			;auf LCD ausgeben
	rjmp	m1g280			;Bereich hinter Text lschen
;
m1g160:	cpi	r25,1			;Zhler auf Preset-Name?
	brne	m1g170			;nein -> weiter testen
	ldi	r16,low(egrprt)
	ldi	r17,high(egrprt)	;EEPROM-Adresse Preset-Namen
	lds	r18,grpres		;aktuelle Preset-Nummer holen
	ldi	r19,16			;Offset/Lnge 16 Bytes
	mul	r18,r19			;Adressoffset berechnen
	add	r16,r0
	adc	r17,r1			;EEPROM-Adresse ermitteln
	call	eeplcd			;Preset-Name ausgeben
	rjmp	m1g280			;Bereich hinter Text lschen
;
m1g170:	cpi	r25,2			;Zhler auf Grafik 1 Sensor?
	brne	m1g190			;nein -> weiter testen
	ldi	r22,'S'			;sonst Zeichen 'S'
	call	asclcd			;ausgeben
	lds	r18,grsen1		;Grafik 1 Sensornummern holen
	andi	r18,0x0f		;unteres Nibble filtern
	inc	r18			;korrigieren
	call	bytlcd			;auf LCD ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
;
	lds	r18,grsen1		;Grafik 1 Sensornummern holen
m1g180:	andi	r18,0x0f		;unteres Nibble filtern
	ldi	r19,9			;Offset/Lnge 9 Bytes
	mul	r18,r19			;Adressoffset berechnen
	ldi	r16,low(etemtx)
	ldi	r17,high(etemtx)	;EEPROM-Adresse Sensor-Namen
	add	r16,r0
	adc	r17,r1			;EEPROM-Adresse ermitteln
	call	eeplcd			;Preset-Name ausgeben
	rjmp	m1g280			;Bereich hinter Text lschen
;
m1g190:	cpi	r25,3			;Zhler auf Grafik 1 Diff-Sen.?
	brne	m1g200			;nein -> weiter testen
	lds	r16,grflag		;sonst Grafik-Flags holen
	bst	r16,2			;Differenz-Modus aktiv?
	brtc	m1g220			;nein -> Text "aus" ausgeben
;
	ldi	r22,'S'			;Zeichen 'S'
	call	asclcd			;ausgeben
	lds	r18,grsen1		;Grafik 1 Sensornummern holen
	swap	r18			;Nibbles tauschen
	andi	r18,0x0f		;unteres Nibble filtern
	inc	r18			;korrigieren
	call	bytlcd			;auf LCD ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
	lds	r18,grsen1		;Grafik 1 Sensornummern holen
	swap	r18			;Nibbles tauschen
	rjmp	m1g180			;Name ausgeben
;
m1g200:	cpi	r25,4			;Zhler auf Grafik 1 Modus?
	brne	m1g210			;nein -> weiter testen
	lds	r16,grmod1		;sonst Grafik 1 Modus holen
	rjmp	m1g260			;Text ermitteln und ausgeben
;
m1g210:	cpi	r25,5			;Zhler auf Grafik 2 Sensor?
	brne	m1g240			;nein -> weiter testen
	lds	r16,grflag		;sonst Grafik-Flags holen
	bst	r16,4			;Doppelgrafik?
	brts	m1g230			;ja -> bearbeiten
m1g220:	ldi	r22,0xb0		;sonst Text "aus"
	rjmp	m1g270			;ausgeben
;
m1g230:	ldi	r22,'S'			;sonst Zeichen 'S'
	call	asclcd			;ausgeben
	lds	r18,grsen2		;Grafik 2 Sensornummern holen
	andi	r18,0x0f		;unteres Nibble filtern
	inc	r18			;korrigieren
	call	bytlcd			;auf LCD ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
	lds	r18,grsen2		;Grafik 2 Sensornummern holen
	rjmp	m1g180			;Name ausgeben
;
m1g240:	cpi	r25,6			;Zhler auf Grafik 2 Diff-Sen.?
	brne	m1g250			;nein -> weiter testen
	lds	r16,grflag		;sonst Grafik-Flags holen
	bst	r16,4			;Doppelgrafik?
	brtc	m1g220			;nein -> "aus" ausgeben
	bst	r16,3			;Differenz-Modus aktiv?
	brtc	m1g220			;nein -> "aus" ausgeben
;
	ldi	r22,'S'			;sonst Zeichen 'S'
	call	asclcd			;ausgeben
	lds	r18,grsen2		;Grafik 2 Sensornummern holen
	swap	r18			;Nibbles tauschen
	andi	r18,0x0f		;unteres Nibble filtern
	inc	r18			;korrigieren
	call	bytlcd			;auf LCD ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
	lds	r18,grsen2		;Grafik 2 Sensornummern holen
	swap	r18			;Nibbles tauschen
	rjmp	m1g180			;Name ausgeben
;
m1g250:	lds	r16,grflag		;Grafik-Flags holen
	bst	r16,4			;Doppelgrafik?
	brtc	m1g220			;nein -> "aus" ausgeben
	lds	r16,grmod2		;sonst Grafik-Modus 2 holen
m1g260:	ldi	r22,0xab		;Text "15 min Werte"
	add	r22,r16			;Grafik-Modus addieren
m1g270:	call	strlcd			;Text ausgeben
;
m1g280:	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,237			;X-Endwert
	ldi	r22,11			;zum Endwert Y 11 Pixel
	add	r21,r22			;addieren
	call	clrwnd			;Bereich lschen
;
m1g290:	adiw	yl,1			;Flag Tabellenzeiger erhhen
	inc	r25			;nchste Zeile
	cpi	r25,8			;alle Zeilen bearbeitet?
	brcc	m1g400			;ja -> weiter
	rjmp	m1g110			;sonst Schleife
;
; Anzeigemodus Konfig/Grafik-Presets: Auswahl bearbeiten
;
m1g400:	lds	r16,menpos		;ausgewhlte Position holen
	lds	r17,encval		;Encoderwert holen
	cp	r16,r17			;nderung der Position?
	breq	m1g410			;nein -> weiter
;
	ldi	xl,low(cfgflg)
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	movw	yl,xl			;Zeiger kopieren
	dec	r16			;Positionswert korrigieren
	clr	r19
	add	xl,r16			;Flag-Position ermitteln
	adc	xh,r19			;bertrag
	ld	r16,x			;Zeilen-Flags holen
	sbr	r16,0x01		;Name neu ausgeben
	st	x,r16			;Flags wieder speichern
	dec	r17			;Encoderwert korrigieren
	add	yl,r17			;Flag-Position ermitteln
	adc	yh,r19			;bertrag
	ld	r16,y			;Zeilen-Flags holen
	sbr	r16,0x01		;Name neu ausgeben
	st	y,r16			;Flags wieder speichern
	lds	r16,encval		;Encoder-Wert holen
	sts	menpos,r16		;ausgewhlte Position setzen
m1g410:	rjmp	m20000
;
; Anzeigemodus Konfig/Grafik-Presets: ndern bearbeiten
;
m1g500:	cpi	r16,26			;Anzeigemodus Konfig/Grafik/.?
	breq	m1g510			;ja -> weiter
	rjmp	m1h000			;sonst weiter testen
;
m1g510:	lds	r19,encflg		;Encoder-Flags holen
	bst	r19,7			;Wert gendert?
	brts	m1g530			;ja -> bearbeiten
	ldi	xl,low(cfgflg)
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	lds	r16,menpos		;ausgewhlte Position holen
	dec	r16			;Positionswert korrigieren
	clr	r17
	add	xl,r16			;Flag-Position ermitteln
	adc	xh,r17			;bertrag
	ld	r18,x			;Zeilen-Flags holen
	bst	r18,1			;Daten neu ausgeben?
	brts	m1g520			;ja -> weiter
	rjmp	m1g590			;sonst Ende
;
m1g520:	cbr	r18,0x02		;Neuausgabe-Flag lschen
	st	x,r18			;Flags wieder speichern
m1g530:	cbr	r19,1<<7		;Encoder nderungs-Flag lschen
	sts	encflg,r19		;Encoder-Flags wieder speichern
	lds	r16,menpos		;ausgewhlte Position holen
	dec	r16			;Positionswert korrigieren
	ldi	r20,128			;X-Startposition
	ldi	r21,16			;Y-Startposition
	ldi	r17,12			;Zeilenabstand festlegen
	mul	r16,r17			;neue Zeilenposition ermitteln
	add	r21,r0			;zur Startposition addieren
;
	cpi	r16,0			;Position auf Preset-Nummer?
	brne	m1g540			;nein -> weiter testen
	lds	r18,encval		;aktuellen Encoderwert holen
	call	bytlcd			;Wert ausgeben
	rjmp	m1g580			;Zeilenende lschen
;
m1g540:	cpi	r16,1			;Position auf Namen-Eingabe?
	brne	m1g550			;nein -> weiter testen
	call	txblcd			;sonst Eingabepuffer ausgeben
	rjmp	m1g580			;Zeilenende lschen
;
m1g550:	cpi	r16,4			;Position auf Grafik 1 Modus?
	breq	m1g570			;ja -> Modus ausgeben
	cpi	r16,7			;Position auf Grafik 1 Modus?
	breq	m1g570			;ja -> Modus ausgeben
;
	lds	r18,encval		;aktuellen Encoderwert holen
	tst	r18			;Encoderwert=0?
	brne	m1g560			;nein -> Sensordaten ausgeben
	ldi	r22,0xb0		;sonst Text "aus"
	call	strlcd			;ausgeben
	rjmp	m1g580			;Zeilenende lschen
;
m1g560:	ldi	r22,'S'			;Zeichen 'S'
	call	asclcd			;ausgeben
	lds	r18,encval		;aktuellen Encoderwert holen
	call	bytlcd			;auf LCD ausgeben
	ldi	r22,0xa0		;schmales Leerzeichen
	call	asclcd			;ausgeben
	lds	r18,encval		;aktuellen Encoderwert holen
	dec	r18			;korrigieren
	ldi	r19,9			;Offset/Lnge 9 Bytes
	mul	r18,r19			;Adressoffset berechnen
	ldi	r16,low(etemtx)
	ldi	r17,high(etemtx)	;EEPROM-Adresse Sensor-Namen
	add	r16,r0
	adc	r17,r1			;EEPROM-Adresse ermitteln
	call	eeplcd			;Preset-Name ausgeben
	rjmp	m1g580			;Zeilenende lschen
;
m1g570:	lds	r16,encval		;sonst Encoderwert holen
	ldi	r22,0xab		;Text "15min Werte"
	add	r22,r16			;Grafik-Modus addieren
	call	strlcd			;Text ausgeben
;
m1g580:	movw	r18,r20			;X/Y Koordinaten kopieren
	ldi	r20,237			;X-Endwert
	ldi	r22,11			;zum Endwert Y 11 Pixel
	add	r21,r22			;addieren
	call	clrwnd			;Bereich lschen
m1g590:	rjmp	m20000
;
m1h000:	rjmp	m20000			;eventuell weitere Anzeigemodi
;
; Empfangspuffer RS232-1 lesen (Datenempfang vom Sensormodul)
;
m20000:	call	get1ch			;Zeichen aus Empfangspuffer
	tst	r17			;holen, neues Zeichen?
	brne	m20010			;nein -> Ende
	cpi	r16,lf			;LF (10) empfangen?
	breq	m20010			;ja -> ignorieren
	cpi	r16,cr			;CR (13) empfangen?
	breq	m20100			;ja -> Puffer auswerten
;
	lds	xl,rxdap1
	lds	xh,rxdap1+1		;Zeiger auf RX-Datenpuffer 1
	ldi	yl,low(rxdat1+rxda1l)
	ldi	yh,high(rxdat1+rxda1l)	;Lnge des RX-Datenpuffers 1
	sub	yl,xl
	sbc	yh,xh			;Pufferende erreicht (voll)?
	breq	m20010			;ja -> Zeichen verwerfen
	st	x+,r16			;Zeichen im Puffer ablegen
	sts	rxdap1,xl
	sts	rxdap1+1,xh		;Zeiger wieder speichern
m20010:	rjmp	m30000			;weiter zum nchsten Abschnitt
;
m20100:	lds	xl,rxdap1
	lds	xh,rxdap1+1		;Zeiger auf RX-Datenpuffer 1
	ldi	yl,low(rxdat1)
	ldi	yh,high(rxdat1)		;Adresse des RX-Datenpuffers 1
	sub	xl,yl
	sbc	xh,yh			;Pufferposition berechnen
	tst	xh			;gltige Position?
	brne	m20200			;nein -> Ende
	cpi	xl,7			;Temperatur-Daten empfangen?
	breq	m21000			;ja -> bearbeiten
	cpi	xl,3			;Alarm-Daten empfangen?
	brne	m20110			;nein -> weiter testen
	rjmp	m22000			;sonst Alarm bearbeiten
m20110:	cpi	xl,8			;Sensorbelegung empfangen?	
	brne	m20200			;nein -> Ende
	rjmp	m23000			;sonst Sensorbelegung bearbeit.
;
m20200:	ldi	yl,low(rxdat1)		;sonst Zeiger fr Datenpuffer 1
	ldi	yh,high(rxdat1)		;wieder auf Anfangswert setzen
	sts	rxdap1,yl
	sts	rxdap1+1,yh		;Zeiger speichern
	rjmp	m30000			;weiter zum nchsten Abschnitt
;
; empfangene Zeile (Temperaturwert) auf Plausibilitt prfen
;
m21000:	ld	r16,y+			;erstes Zeichen holen
	cpi	r16,'1'			;Zeichen<"1"?
	brcs	m20200			;ja -> Fehler
	cpi	r16,'h'+1		;Zeichen>"h"?
	brcc	m20200			;ja -> Fehler
	cpi	r16,'9'			;Zeichen<"9"?
	brcs	m21010			;ja -> weiter (Sensor 1-8)
	cpi	r16,'a'			;Zeichen<"a"?
	brcs	m20200			;ja -> Fehler
m21010:	ld	r16,y+			;zweites Zeichen holen
	cpi	r16,':'			;Doppelpunkt?
	brne	m20200			;nein -> Fehler
	ld	r16,y+			;drittes Zeichen holen
	cpi	r16,' '			;Leerzeichen?
	breq	m21020			;ja -> weiter
	cpi	r16,'-'			;Minus-Zeichen?
	breq	m21020			;ja -> weiter
	cpi	r16,'1'			;Zeichen="1"?
	brne	m20200			;nein -> Fehler
m21020:	ld	r16,y+			;viertes Zeichen holen
	cpi	r16,' '			;Leerzeichen?
	breq	m21030			;ja -> weiter
	cpi	r16,'-'			;Minus-Zeichen?
	breq	m21030			;ja -> weiter
	cpi	r16,'0'			;Zeichen<"0"?
	brcs	m20200			;ja -> Fehler
	cpi	r16,'9'+1		;Zeichen>"9"?
	brcc	m20200			;ja -> Fehler
m21030:	ld	r16,y+			;fnftes Zeichen holen
	cpi	r16,'0'			;Zeichen<"0"?
	brcs	m20200			;ja -> Fehler
	cpi	r16,'9'+1		;Zeichen>"9"?
	brcc	m20200			;ja -> Fehler
	ld	r16,y+			;sechstes Zeichen holen
	cpi	r16,'.'			;Punkt?
	breq	m21040			;ja -> weiter
	cpi	r16,','			;Komma?
	brne	m20200			;nein -> Fehler
m21040:	ld	r16,y+			;siebtes Zeichen holen
	cpi	r16,'0'			;Zeichen<"0"?
	brcs	m20200			;ja -> Fehler
	cpi	r16,'9'+1		;Zeichen>"9"?
	brcc	m20200			;ja -> Fehler
;
; Temperatur-Daten sind ok, in SRAM-Tabelle speichern
;
	clr	r16			;Flag fr fehlerfreien Empfang
	sts	rcvflg,r16		;setzen (fr Status-Anzeige)
	lds	r16,rxdat1		;erstes Zeichen aus Datenpuffer
	cpi	r16,'9'			;Sensorgruppe 1-8?
	brcc	m21100			;nein -> weiter
	subi	r16,'1'			;in Sensornummer 0-7 umwandeln
	rjmp	m21110
m21100:	subi	r16,'a'-8		;in Sensornummer 8-15 umwandeln
m21110:	mov	r11,r16			;Sensornr. fr Meldung sichern
	mov	r20,r16			;Sensornummer zwischenspeichern
	add	r20,r20			;Sensornummer verdoppeln
	clr	r21
	ldi	yl,low(temval)
	ldi	yh,high(temval)		;Zeiger auf Temperaturwerte
	add	yl,r20			;Temperaturwert-Adresse berech.
	adc	yh,r21			;bertrag addieren
	ldi	xl,low(temflg)
	ldi	xh,high(temflg)		;Zeiger auf Temperaturflags
	add	xl,r16			;Flag-Adresse berechnen
	adc	xh,r21			;bertrag addieren
	call	temhex			;Temperaturwert in Hex wandeln
	ld	r18,y			;alten L-Wert aus Tabelle holen
	cp	r16,r18			;hat sich der Wert gendert?
	brne	m21120			;ja -> Flag setzen
	ldd	r18,y+1			;alten H-Wert aus Tabelle holen
	cp	r17,r18			;hat sich der Wert gendert?
	brne	m21120			;ja -> speichern und prfen
	rjmp	m21300			;Limit und Min/Max berspringen
m21120:	st	y,r16			;neuen Wert L speichern
	std	y+1,r17			;neuen Wert H speichern
	ld	r22,x			;sonst Temperaturflags holen
	sbr	r22,0x01		;nderungs-Flag setzen
;
; Temperatur auf Limit ber- oder Unterschreitung prfen und Flags
; setzen
;
	ldi	zl,low(temlih)		;Zeiger auf obere Temperatur-
	ldi	zh,high(temlih)		;Limits (Schwellwerte)
	add	zl,r20			;obere Limit-Adresse berechnen
	adc	zh,r21			;bertrag addieren
	ldd	r19,z+1			;oberes Limit H holen
	cpi	r19,0x80		;Alarmberwachung aktiv?
	breq	m21150			;nein -> Prfung berspringen
	bst	r22,7			;Sensor inaktiv?
	brts	m21150			;ja -> Prfung berspringen
;
	ld	r18,z			;sonst oberes Limit L holen
	sub	r18,r16
	sbc	r19,r17			;oberes Limit berschritten?
	brpl	m21140			;nein -> Flags prfen
	bst	r22,2			;berschreitungs-Flag gesetzt?
	brts	m21150			;ja -> weiter
	cbr	r22,0x28		;Unterschr./Erinnerung-L lsch.
	sbr	r22,0x06		;berschreitung/Sensorname neu
;
	mov	r18,r11			;gesicherte Sensornummer holen
	ori	r18,0xb0		;Fehlercode "berschreitung"
	mov	r11,r18			;Meldungscode wieder speichern
;
	lds	r18,confg1		;Konfigurationsbyte 1 holen
	bst	r18,2			;jede berschreitung melden?
	brts	m21130			;ja -> Signal ausgeben
	bst	r22,4			;berschreitungserinnerung?
	brts	m21150			;ja -> kein weiteres Signal
m21130:	lds	r18,signum+4		;Signalnummer fr berschreit.
	lds	r19,sigdur+4		;Signaldauer fr berschreit.
	call	signal			;Signal ausgeben
	rjmp	m21150
;
m21140:	bst	r22,2			;berschreitungs-Flag gesetzt?
	brtc	m21150			;nein -> weiter
	cbr	r22,0x24		;berschr./Erinnerung-L lschen
	sbr	r22,0x12		;Erinnerung-H/Sensorname setzen
;
	mov	r18,r11			;gesicherte Sensornummer holen
	ori	r18,0xd0		;Fehlercode "Temperatur normal"
	mov	r11,r18			;Meldungscode wieder speichern
;
m21150:	ldi	zl,low(temlil)		;Zeiger auf untere Temperatur-
	ldi	zh,high(temlil)		;Limits (Schwellwerte)
	add	zl,r20			;untere Limit-Adresse berechnen
	adc	zh,r21			;bertrag addieren
	ldd	r19,z+1			;unteres Limit H holen
	cpi	r19,0x80		;Alarmberwachung aktiv?
	breq	m21180			;nein -> Prfung berspringen
	bst	r22,7			;Sensor inaktiv?
	brts	m21180			;ja -> Prfung berspringen
;
	ld	r18,z			;sonst unteres Limit L holen
	sub	r16,r18
	sbc	r17,r19			;unteres Limit unterschritten?
	brpl	m21170			;nein -> Flags prfen
	bst	r22,3			;Unterschreitungs-Flag gesetzt?
	brts	m21180			;ja -> weiter
	cbr	r22,0x14		;berschr./Erinnerung-H lschen
	sbr	r22,0x0a		;Unterschreitung/Sensorname neu
;
	mov	r18,r11			;gesicherte Sensornummer holen
	ori	r18,0xc0		;Fehlercode "Unterschreitung"
	mov	r11,r18			;Meldungscode wieder speichern
;
	lds	r18,confg1		;Konfigurationsbyte 1 holen
	bst	r18,2			;jede Unterschreitung melden?
	brts	m21160			;ja -> Signal ausgeben
	bst	r22,5			;Unterschreitungserinnerung?
	brts	m21180			;ja -> kein weiteres Signal
m21160:	lds	r18,signum+5		;Signalnummer fr Unterschreit.
	lds	r19,sigdur+5		;Signaldauer fr Unterschreit.
	call	signal			;Signal ausgeben
	rjmp	m21180
;
m21170:	bst	r22,3			;Unterschreitungs-Flag gesetzt?
	brtc	m21180			;nein -> weiter
	cbr	r22,0x18		;Unterschr/Erinnerung-H lschen
	sbr	r22,0x22		;Erinnerung + Sensorname setzen
;
	mov	r18,r11			;gesicherte Sensornummer holen
	ori	r18,0xd0		;Fehlercode "Temperatur normal"
	mov	r11,r18			;Meldungscode wieder speichern
;
m21180:	st	x,r22			;Flags wieder speichern
;
; Maximal- und Minimal-Werte ermitteln und speichern (Anzeige)
;
	ld	r16,y
	ldd	r17,y+1			;Temperaturwert holen
	ldi	zl,low(temax1)
	ldi	zh,high(temax1)		;Zeiger auf Maximalwerte (Anz)
	add	zl,r20			;Maximalwert-Adresse berechnen
	adc	zh,r21			;bertrag addieren
	ldd	r19,z+1			;Maximalwert H holen
	cpi	r19,0x80		;Maximalwert gltig
	breq	m21200			;nein -> aktuellen Wert setzen
	ld	r18,z			;Maximalwert L holen
	sub	r18,r16			;bisheriger Maximalwert
	sbc	r19,r17			;berschritten?
	brpl	m21210			;nein -> weiter
m21200:	st	z,r16			;sonst neuen Maximalwert
	std	z+1,r17			;speichern
;
m21210:	ldi	zl,low(temin1)
	ldi	zh,high(temin1)		;Zeiger auf Minimalwerte (Anz)
	add	zl,r20			;Minimalwert-Adresse berechnen
	adc	zh,r21			;bertrag addieren
	ldd	r19,z+1			;Minimalwert H holen
	cpi	r19,0x80		;Minimalwert gltig?
	breq	m21220			;nein -> aktuellen Wert setzen
	ld	r18,z			;Minimalwert L holen
	sub	r16,r18			;bisheriger Minimalwert
	sbc	r17,r19			;unterschritten?
	brpl	m21230			;nein -> weiter
	ld	r16,y			;sonst nochmals aktuellen
	ldd	r17,y+1			;Temperaturwert holen
m21220:	st	z,r16			;und als neuen Minimalwert
	std	z+1,r17			;speichern
;
; Maximal- und Minimal-Werte ermitteln und speichern (EEPROM-Log)
;
m21230:	ld	r16,y
	ldd	r17,y+1			;Temperaturwert holen
	ldi	zl,low(temax2)
	ldi	zh,high(temax2)		;Zeiger auf Maximalwerte (Log)
	add	zl,r20			;Maximalwert-Adresse berechnen
	adc	zh,r21			;bertrag addieren
	ldd	r19,z+1			;Maximalwert H holen
	cpi	r19,0x80		;Maximalwert gltig
	breq	m21240			;nein -> aktuellen Wert setzen
	ld	r18,z			;Maximalwert L holen
	sub	r18,r16			;bisheriger Maximalwert
	sbc	r19,r17			;berschritten?
	brpl	m21250			;nein -> weiter
m21240:	st	z,r16			;sonst neuen Maximalwert
	std	z+1,r17			;speichern
;
m21250:	ldi	zl,low(temin2)
	ldi	zh,high(temin2)		;Zeiger auf Minimalwerte (Log)
	add	zl,r20			;Minimalwert-Adresse berechnen
	adc	zh,r21			;bertrag addieren
	ldd	r19,z+1			;Minimalwert H holen
	cpi	r19,0x80		;Minimalwert gltig?
	breq	m21260			;nein -> aktuellen Wert setzen
	ld	r18,z			;Minimalwert L holen
	sub	r16,r18			;bisheriger Minimalwert
	sbc	r17,r19			;unterschritten?
	brpl	m21300			;nein -> weiter
	ld	r16,y			;sonst nochmals aktuellen
	ldd	r17,y+1			;Temperaturwert holen
m21260:	st	z,r16			;und als neuen Minimalwert
	std	z+1,r17			;speichern
;
; Messwert zur Mittelwertsumme addieren und Mittelwertzhler erhhen
;
m21300:	ldi	zl,low(temavc)
	ldi	zh,high(temavc)		;Zeiger auf Mittelwertzhler
	add	zl,r20			;Zhler-Adresse berechnen
	adc	zh,r21			;bertrag addieren
	ld	r16,z			;Zhlerwert L holen
	ldd	r17,z+1			;Zhlerwert H holen
	inc	r16			;L-Wert erhhen
	brne	m21310			;bertrag? nein -> weiter
	inc	r17			;sonst H-Wert erhhen
m21310:	st	z,r16			;Zhlerwert L wieder speichern
	std	z+1,r17			;Zhlerwert H wieder speichern
;
	ldi	zl,low(temavs)
	ldi	zh,high(temavs)		;Zeiger auf Mittelwertsummen
	mov	r23,r20			;Adress-Offset kopieren
	lsr	r23			;halbieren und addieren wegen
	add	r20,r23			;Tabellen-Offset von 3 Byte
	add	zl,r20			;Zhler-Adresse berechnen
	adc	zh,r21			;bertrag addieren
	ld	r16,z			;Summenwert L holen
	ldd	r17,z+1			;Summenwert H holen
	ldd	r18,z+2			;Summenwert B3 holen
	ld	r13,y			;Temperaturwert L holen
	ldd	r14,y+1			;Temperaturwert H holen
	clr	r15			;Temperaturwert B3=0
	bst	r14,7			;Temperaturwert negativ?
	brtc	m21320			;nein -> weiter
	dec	r15			;sonst B3=0xff
m21320:	add	r16,r13
	adc	r17,r14
	adc	r18,r15			;Temperatur zur Summe addieren
	st	z,r16			;neuen Summenwert L speichern
	std	z+1,r17			;neuen Summenwert H speichern
	std	z+2,r18			;neuen Summenwert B3 speichern
;
; Timeout-Zhler und einige Flags neu setzen
;
	ldi	zl,low(temtio)
	ldi	zh,high(temtio)		;Zeiger auf Timeout-Zhler
	add	zl,r23			;Zhler-Adresse berechnen
	adc	zh,r21			;bertrag addieren
	ldi	r16,timtmp		;Timeout-Wert laden
	st	z,r16			;und als Timeout setzen
	ldi	r16,timmsg		;Meldungs-Timeout-Wert laden
	sts	msgtio,r16		;und als Timeout neu setzen
	ld	r22,x			;Temperatur-Flags holen
	bst	r22,6			;war Ausfall-Flag gesetzt?
	brtc	m21500			;nein -> berspringen
	cbr	r22,0x40		;sonst Ausfall-Flag lschen
	sbr	r22,0x01		;und nderungs-Flag setzen
	st	x,r22			;Flags wieder speichern
;
	mov	r19,r11			;Fehlercode holen
	andi	r19,0x0f		;Meldungsnummer lschen
	ori	r19,0xa0		;Meldung "Sensor ok" generieren
	call	wrimsg			;Fehlermeldung schreiben
;
m21500:	mov	r16,r11			;Fehlercode holen
	andi	r16,0xf0		;Fehlermeldung vorhanden?
	breq	m21600			;nein -> weiter
	mov	r19,r11			;Fehlercode holen
	call	wrimsg			;Fehlermeldung schreiben
;
m21600:	ldi	r16,timmsg		;Meldungs-Timeout-Wert laden
	sts	msgtio,r16		;und als Timeout neu setzen
	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,1			;Flag "Datenempfangsfehler"?
	brtc	m21610			;nein -> weiter
	cbr	r16,0x02		;Datenempfangsfehler lschen
	sbr	r16,0x80		;Fehlererinnerung setzen
	sts	errflg,r16		;Flags wieder speichern
;
	ldi	r19,0x60		;Meldung "Datenempfang ok"
	call	wrimsg			;Fehlermeldung schreiben
	lds	r16,dflags		;Anzeige-Flags holen
	sbr	r16,0x04		;Infozeile neu ausgeben
	sts	dflags,r16		;Flags wieder speichern
m21610:	rjmp	m20200			;Zeiger zurcksetzen, Ende
;
; empfangene Zeile (Alarmmeldung) auf Plausibilitt prfen
;
m22000:	ld	r16,y+			;erstes Zeichen holen
	cpi	r16,'A'			;Zeichen<"A"?
	brcs	m21610			;ja -> Fehler
	cpi	r16,'D'+1		;Zeichen>"D"?
	brcc	m21610			;ja -> Fehler
m22010:	ld	r16,y+			;zweites Zeichen holen
	cpi	r16,':'			;Doppelpunkt?
	brne	m22200			;nein -> Fehler
	ld	r16,y+			;drittes Zeichen holen
	cpi	r16,'0'			;Zeichen="0"?
	breq	m22100			;ja -> bearbeiten
	cpi	r16,'1'			;Zeichen="1"?
	brne	m22200			;nein -> Fehler
;
; Alarm-Daten sind ok, in SRAM-Tabelle speichern
;
m22100:	clr	r16			;Flag fr fehlerfreien Empfang
	sts	rcvflg,r16		;setzen (fr Status-Anzeige)
	lds	r16,rxdat1		;erstes Zeichen aus Datenpuffer
	andi	r16,0x07		;in Ziffernwert wandeln
	dec	r16			;korrigieren (=Alarmummer)
	clr	r17
	mov	r11,r16			;Alarmnr. fr Meldung sichern
	ldi	xl,low(almflg)
	ldi	xh,high(almflg)		;Zeiger auf Alarmflags
	add	xl,r16			;Tabellenplatz berechnen
	adc	xh,r17			;bertrag
	ldi	yl,low(alminv)
	ldi	yh,high(alminv)		;Zeiger auf Alarm-Invertierung
	add	yl,r16			;Tabellenplatz berechnen
	adc	yh,r17			;bertrag
	lds	r18,rxdat1+2		;Alarmwert holen
	andi	r18,0x01		;in Binrwert wandeln
	ld	r19,y			;Invertierungswert holen
	eor	r18,r19			;Alarmwert invertieren
	ld	r19,x			;Alarmflags holen
	lsr	r19			;Aktiv-Flag an Alarmposition
	andi	r19,0x01		;Aktiv-Flag filtern
	cp	r18,r19			;hat sich der Wert gendert?
	breq	m22120			;nein -> weiter
	ld	r19,x			;sonst Alarmflags holen
	bst	r19,7			;wird Alarm genutzt?
	brts	m22120			;nein -> nicht signalisieren
;
	bst	r18,0			;neuen Alarmwert holen und
	bld	r19,1			;als Aktiv-Flag setzen
	sbr	r19,0x01		;nderungs-Flag setzen
	st	x,r19			;Flags wieder speichern
	brts	m22110			;Alarm ein? ja -> Signal
	sbr	r19,0x04		;sonst Erinnerung setzen
	st	x,r19			;Flags wieder speichern
;
	mov	r18,r11			;gesicherte Alarmnummer holen
	ori	r18,0x80		;Fehlercode "Alarm aus" setzen
	mov	r11,r18			;Meldungscode wieder speichern
	rjmp	m22120
;
m22110:	ldi	xl,low(signum)
	ldi	xh,high(signum)		;Zeiger auf Signalnummern
	add	xl,r16			;Tabellenplatz berechnen
	adc	xh,r17			;bertrag
	ldi	yl,low(sigdur)
	ldi	yh,high(sigdur)		;Zeiger auf Signaldauern
	add	yl,r16			;Tabellenplatz berechnen
	adc	yh,r17			;bertrag
	ld	r18,x			;zugeordnete Signalnummer holen
	ld	r19,y			;zugeordnete Signaldauer holen
	call	signal			;Signal ausgeben
;
	mov	r18,r11			;gesicherte Alarmnummer holen
	ori	r18,0x70		;Fehlercode "Alarm ein" setzen
	mov	r11,r18			;Meldungscode wieder speichern
;
m22120:	rjmp	m21500			;Empfangsfehler-Flag testen
m22200:	rjmp	m20200			;Zeiger zurcksetzen, Ende
;
; empfangene Zeile (Sensorbelegung) auf Plausibilitt prfen
;
m23000:	ldi	r17,8			;8 Zeichen prfen
m23010:	ld	r16,y+			;Zeichen aus Puffer holen
	cpi	r16,'0'			;Zeichen<"0"?
	brcs	m23200			;ja -> Fehler
	cpi	r16,'G'+1		;Zeichen>"G"?
	brcc	m23200			;ja -> Fehler
	cpi	r16,'9'+1		;Zeichen=<"9"?
	brcs	m23020			;ja -> weiter
	cpi	r16,'A'			;Zeichen<"A"?
	brcs	m23200			;ja -> Fehler
m23020:	dec	r17			;alle Zeichen bearbeitet?
	brne	m23010			;nein -> Schleife
;
; Sensorbelegung ist ok, Daten speichern
;
	clr	r16			;Flag fr fehlerfreien Empfang
	sts	rcvflg,r16		;setzen (fr Status-Anzeige)
	ldi	r17,8			;8 Zeichen kopieren
	ldi	yl,low(rxdat1)
	ldi	yh,high(rxdat1)		;Zeiger auf RX Datenpuffer 1
	ldi	xl,low(senmap)
	ldi	xh,high(senmap)		;Zeiger auf Sensorbelegung
m23030:	ld	r16,y+			;Byte aus RX-Datenpuffer holen
	st	x+,r16			;in Sensorbelegung speichern
	dec	r17			;alle Daten kopiert?
	brne	m23030			;nein -> Schleife
	rjmp	m21600			;Empfangsfehler-Flag testen
m23200:	rjmp	m20200			;Zeiger zurcksetzen, Ende
;
; Empfangspuffer RS232-0 lesen (Datenempfang von PC-Schnittstelle)
;
m30000:	call	get0ch			;Zeichen aus Empfangspuffer
	tst	r17			;holen, neues Zeichen?
	brne	m30010			;nein -> Ende
	cpi	r16,lf			;LF (10) empfangen?
	breq	m30010			;ja -> ignorieren
	cpi	r16,cr			;CR (13) empfangen?
	breq	m30100			;ja -> Puffer auswerten
;
	lds	xl,rxdap0
	lds	xh,rxdap0+1		;Zeiger auf RX-Datenpuffer 0
	ldi	yl,low(rxdat0+rxda0l)
	ldi	yh,high(rxdat0+rxda0l)	;Lnge des RX-Datenpuffers 0
	sub	yl,xl
	sbc	yh,xh			;Pufferende erreicht (voll)?
	breq	m30010			;ja -> Zeichen verwerfen
	st	x+,r16			;Zeichen im Puffer ablegen
	sts	rxdap0,xl
	sts	rxdap0+1,xh		;Zeiger wieder speichern
m30010:	rjmp	m40000			;weiter zum nchsten Abschnitt
;
m30100:	lds	r16,rxdap0		;Zeiger auf aktuelle Position
	lds	r17,rxdap0+1		;im RX-Datenpuffer 0
	ldi	yl,low(rxdat0)		;Startadresse des RX-
	ldi	yh,high(rxdat0)		;Datenpuffers 0
	sub	r16,yl			;Anzahl der Zeichen im Puffer
	sbc	r17,yh			;berechnen
	tst	r17			;Zeichenanzahl<256
	breq	m30120			;ja -> weiter
m30110:	rjmp	m30300			;sonst Ende
m30120:	tst	r16			;Zeichenanzahl=0?
	breq	m30110			;ja -> Ende
;
; Empfangsdatenpuffer lesen, Kommando und Parameter ermitteln
;
	mov	r21,r16			;Zeichenanzahl kopieren
	clr	r19			;Zhler fr Kommandolnge=0
m30130:	ld	r16,y+			;Zeichen aus Puffer holen
	cpi	r16,' '			;Kommando-Ende?
	breq	m30140			;ja -> weiter
	inc	r19			;sonst Zhler erhhen
	cp	r19,r21			;letztes Zeichen im Puffer?
	brne	m30130			;nein -> Schleife
;
m30140:	ldi	xl,low(pccmdl*2)
	ldi	xh,high(pccmdl*2)
	ldi	r22,byte3(pccmdl*2)	;Zeiger auf Kommandoliste
	clr	r20			;Kommandonummer=0 setzen
m30150:	inc	r20			;Kommandonummer erhhen
	movw	zl,xl			;Listenzeiger L/H kopieren
	out	rampz,r22
	elpm	r16,z			;erstes Zeichen holen
	cpi	r16,0xff		;Listen-Ende?
	breq	m30300			;ja -> Kommando nicht gefunden
	clr	r18			;Kommandolnge=0 setzen
m30160:	elpm	r16,z+			;Zeichen aus Liste holen
	cpi	r16,' '			;Kommando-Ende?
	breq	m30170			;ja -> weiter
	inc	r18			;Komandolnge +1
	cpi	r18,8			;alle 8 Zeichen gelesen?
	brcs	m30160			;nein -> Schleife
m30170:	cp	r18,r19			;Zeichenlnge identisch?
	breq	m30190			;ja -> weiter
m30180:	adiw	xl,8			;sonst Zeiger auf nchstes Kom.
	clr	r16
	adc	r22,r16			;bertrag in Byte3
	rjmp	m30150			;Schleife
;
m30190:	movw	zl,xl			;Listenzeiger L/H kopieren
	out	rampz,r22
	ldi	yl,low(rxdat0)
	ldi	yh,high(rxdat0)		;Adresse des RX-Datenpuffers 0
m30200:	elpm	r16,z+			;Zeichen aus Liste holen
	ld	r17,y+			;Zeichen aus Datenpuffer holen
	ori	r17,0x20		;in Kleinbuchstaben wandeln
	cp	r16,r17			;bereinstimmung?
	brne	m30180			;nein -> nchstes Kommando
	dec	r18			;alle Zeichen verglichen?
	brne	m30200			;nein -> Schleife
	cpi	r20,1			;Ende/Abbruch-Kommando (1)?
	breq	m30210			;ja -> speichern
	lds	r16,pccmd		;sonst aktuelles Kommando holen
	tst	r16			;wird gerade Kommando bearb.?
	brne	m30300			;ja -> neues Kommando ignorier.
m30210:	sts	pccmd,r20		;sonst Kommandonummer speichern
	lds	r16,dflags		;Anzeige-Flags holen
	sbr	r16,0x04		;Info-Zeile neu ausgeben
	sts	dflags,r16		;Anzeige-Flags wieder speichern
;
	call	clpara			;Parameterspeicher lschen
	cp	r19,r21			;wurde ein Parameter empfangen?
	breq	m30300			;nein -> Ende
	ldi	xl,low(pcpara)		;sonst
	ldi	xh,high(pcpara)		;Zeiger auf Parameterspeicher
	ldi	r17,8			;Parameterlnge 8 Zeichen
m30230:	ld	r16,y+			;Zeichen aus Puffer holen
	cpi	r16,' '			;Leerzeichen?
	breq	m30250			;ja -> berspringen
	cpi	r16,'-'			;Minuszeichen (Bindestrich)?
	brne	m30240			;nein -> weiter
	sts	pcpara+8,r16		;sonst Minuszeichen speichern
	rjmp	m30300			;Ende
;
m30240:	tst	r17			;Parameterspeicher voll?
	breq	m30250			;ja -> berspringen, sonst
	st	x+,r16			;Zeichen in Parameterspeicher
	dec	r17			;Zhler-1
m30250:	inc	r19			;Zeichenzhler+1
	cp	r19,r21			;Pufferende erreicht?
	brne	m30230			;nein -> nchstes Zeichen
;
m30300:	ldi	yl,low(rxdat0)		;sonst Zeiger fr Datenpuffer 0
	ldi	yh,high(rxdat0)		;wieder auf Anfangswert setzen
	sts	rxdap0,yl
	sts	rxdap0+1,yh		;Zeiger speichern
	rjmp	m40000			;weiter zum nchsten Abschnitt
;
; Steuerung RS232-0 TX
;
m40000:	lds	r16,pccmd		;aktuelles Kommando holen
	cpi	r16,0			;wird ein Kommando ausgefhrt?
	brne	m40100			;ja -> weiter
	rjmp	m50000			;sonst zum nchsten Abschnitt
;
; Kommando 1 bearbeiten (Ende/Abbruch)
;
m40100:	cpi	r16,1			;Kommando 1 (Ende/Abbruch)?
	brne	m40500			;nein -> nchstes Kommando
m40200:	clr	r16			;sonst
	sts	pccmd,r16		;aktuelles Kommando beenden
	sts	pccpnt,r16
	sts	pccpnt+1,r16
	sts	pccpnt+2,r16		;EEPROM-Zeigeradresse lschen
	sts	pcccnt,r16
	sts	pcccnt+1,r16		;Datensatz-Zhler lschen
	lds	r16,dflags		;Anzeige-Flags holen
	sbr	r16,0x04		;Info-Zeile neu ausgeben
	sts	dflags,r16		;Anzeige-Flags wieder speichern
	rjmp	m50000
;
; Kommando 2 bearbeiten (Info)
;
m40500:	cpi	r16,2			;Kommando 2 (Info)?
	brne	m41000			;nein -> nchstes Kommando
;
	lds	xl,txp0wr
	lds	xh,txp0wr+1		;Sendepuffer Schreibzeiger hol.
	lds	yl,txp0rd
	lds	yh,txp0rd+1		;Sendepuffer Lesezeiger holen
	sub	xl,yl
	sbc	xh,yh			;Sendepuffer leer?
	breq	m40510			;ja -> weiter
	rjmp	m50000			;sonst Ende
;
m40510:	ldi	r17,8			;String 0x08 (Versionsnummer)
	ldi	xl,low(ascbuf)
	ldi	xh,high(ascbuf)		;Zeiger auf ASCII-Puffer
	call	xtrfst			;String in ASCII-Puffer legen
	call	xtbyte			;Anzahl der Bytes ermitteln
	call	senasc			;ASCII-Puffer senden
	rjmp	m40200			;Kommando beenden
;
; Kommando 3 bearbeiten (Temperaturdaten 1 senden, 15-min-Werte)
;
m41000:	cpi	r16,3			;Kommando 3 (Temp-Daten 1)?
	breq	m41010			;ja -> ausfhren
	rjmp	m42000			;sonst nchstes Kommando
;
m41010:	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,3			;Speicher-Fehler?
	brtc	m41020			;nein -> weiter
	rjmp	m40200			;sonst Kommando beenden
m41020:	lds	r20,pccpnt
	lds	r21,pccpnt+1
	lds	r22,pccpnt+2		;EEPROM-Zeigeradresse holen
	lds	r24,pcccnt
	lds	r25,pcccnt+1		;Datensatz-Zhler holen
	mov	r16,r24
	or	r16,r25			;Datensatz-Zhler=0?
	brne	m41100			;nein -> nchster Datensatz
	lds	r20,xet1po		;sonst aktuellen Zeiger der
	lds	r21,xet1po+1		;Temp-Daten 1 (15min) holen,
	lds	r22,xet1po+2		;zeigt auf ltesten Datensatz
	sts	pccpnt,r20
	sts	pccpnt+1,r21
	sts	pccpnt+2,r22		;EEPROM-Zeigeradresse setzen
;
m41100:	lds	xl,txp0wr
	lds	xh,txp0wr+1		;Sendepuffer Schreibzeiger hol.
	lds	yl,txp0rd
	lds	yh,txp0rd+1		;Sendepuffer Lesezeiger holen
	sub	xl,yl
	sbc	xh,yh			;Sendepuffer leer?
	breq	m41110			;ja -> weiter
	rjmp	m50000			;sonst Ende
;
m41110:	ldi	r23,28			;28 Bytes lesen (1 Datensatz)
	call	xread			;Datensatz aus Ext-EEPROM lesen
	brcc	m41120			;Fehler? nein -> weiter
	call	twierr			;sonst Speicherfehler setzen
	rjmp	m40200			;Kommando beenden
m41120:	sts	pccpnt,r20
	sts	pccpnt+1,r21		;neue EEPROM-Zeigeradresse
	sts	pccpnt+2,r22		;speichern
	adiw	r24,1			;Datensatz-Zhler erhhen
	sts	pcccnt,r24
	sts	pcccnt+1,r25		;Datensatz-Zhler speichern
;
; Daten aufbereiten und ausgeben
;
	lds	r16,xerbuf		;Zeitbyte 0 holen
	lds	r17,xerbuf+1		;Zeitbyte 1 holen
	and	r16,r17			;verknpfen
	lds	r17,xerbuf+2		;Zeitbyte 2 holen
	and	r16,r17			;verknpfen
	lds	r17,xerbuf+3		;Zeitbyte 3 holen
	and	r16,r17			;verknpfen
	cpi	r16,0xff		;gltiger Datenssatz?
	breq	m41300			;nein -> Ausgabe berspringen
;
	call	xtrtim			;sonst Zeit extrahieren
	ldi	yl,low(pcpara)
	ldi	yh,high(pcpara)		;Zeiger auf Parameterspeicher
	ld	r16,y			;erstes Zeichen vom Parameter
	tst	r16			;Parameter gltig?
	breq	m41250			;nein -> Datensatz ausgeben
;
	ldi	r18,8			;8 Zeichen vergleichen
	lds	r16,pcpara+8		;Parameter-Zusatz holen
	cpi	r16,'-'			;Minuszeichen?
	breq	m41210			;ja -> bearbeiten
;
	ldi	zl,low(ascbuf)		;sonst
	ldi	zh,high(ascbuf)		;Zeiger auf ASCII-Puffer
m41200:	ld	r16,y+			;Zeichen aus Parametersp. holen
	ld	r17,z+			;Zeichen aus ASCII-Puffer holen
	tst	r16			;Parameter-Ende?
	breq	m41250			;ja -> Datensatz ausgeben
	cp	r16,r17			;Zeichen identisch?
	brne	m41300			;nein -> Ausgabe berspringen
	dec	r18			;alle Zeichen verglichen?
	brne	m41200			;nein -> Schleife
	rjmp	m41250
;
m41210:	ldi	zl,low(ascbuf+8)	;Zeiger auf ASCII-Puffer
	ldi	zh,high(ascbuf+8)	;(Tag-Einer)
	ldi	yl,low(pcpara+8)	;Zeiger auf Parameterspeicher
	ldi	yh,high(pcpara+8)	;(Ende)
	clc				;bertrag lschen
m41220:	ld	r17,-z			;Zeichen aus ASCII-Puffer holen
	ld	r16,-y			;Zeichen aus Parametersp. holen
	tst	r16			;gltiges Zeichen?
	brne	m41240			;ja -> weiter
	dec	r18			;sonst alle Zeichen bearbeitet?
	brne	m41220			;nein -> Schleife
	rjmp	m41250			;sonst Datensatz ausgeben
;
m41230:	ld	r17,-z			;Zeichen aus ASCII-Puffer holen
	ld	r16,-y			;Zeichen aus Parametersp. holen
m41240:	sbc	r17,r16			;Pufferposition - Parameterpos.
	dec	r18			;alle Zeichen verglichen?
	brne	m41230			;nein -> Schleife
	brcs	m41300			;Param > Puffer? -> keine Ausg.
;
m41250:	call	xtrse0			;Sekunde 00 ergnzen
	ldi	yl,low(xerbuf+4)
	ldi	yh,high(xerbuf+4)	;Zeiger auf Datensatz (Daten)
	call	xtrdat			;Datensatz extrahieren
	call	senasc			;ASCII-Puffer senden
;
m41300:	ldi	r16,low(xt1sta+xt1len)
	ldi	r17,high(xt1sta+xt1len)	;End-Adresse Datenbereich 1
	ldi	r18,byte3(xt1sta+xt1len);laden (15-min-Werte)
	sub	r20,r16
	sbc	r21,r17			;aktuelle Adresse noch im
	sbc	r22,r18			;gltigen Speicherbereich?
	brcs	m41310			;ja -> weiter
	ldi	r16,low(xt1sta)
	ldi	r17,high(xt1sta)	;sonst Adresse wieder auf
	ldi	r18,byte3(xt1sta)	;Anfang setzen
	sts	pccpnt,r16
	sts	pccpnt+1,r17		;neue EEPROM-Zeigeradresse
	sts	pccpnt+2,r18		;speichern
;
m41310:	lds	r20,pccpnt
	lds	r21,pccpnt+1
	lds	r22,pccpnt+2		;EEPROM-Zeigeradresse holen
	lds	r16,xet1po		;aktuellen EEPROM-Zeiger auf
	lds	r17,xet1po+1		;15-min-Daten holen = End-
	lds	r18,xet1po+2		;Adresse fr Datenausgabe
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;Endadresse erreicht?
	brne	m41400			;nein -> weiter
	rjmp	m40200			;sonst Kommando beenden
;
m41400:	rjmp	m50000			;Ende
;
; Kommando 4 bearbeiten (Temperaturdaten 2 senden, Minimal-Werte)
;
m42000:	cpi	r16,4			;Kommando 4 (Minmal-Werte)?
	breq	m42010			;ja -> ausfhren
	rjmp	m43000			;sonst nchstes Kommando
;
m42010:	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,3			;Speicher-Fehler?
	brtc	m42020			;nein -> weiter
	rjmp	m40200			;sonst Kommando beenden
m42020:	lds	r20,pccpnt
	lds	r21,pccpnt+1
	lds	r22,pccpnt+2		;EEPROM-Zeigeradresse holen
	lds	r24,pcccnt
	lds	r25,pcccnt+1		;Datensatz-Zhler holen
	mov	r16,r24
	or	r16,r25			;Datensatz-Zhler=0?
	brne	m42100			;nein -> nchster Datensatz
	lds	r20,xet2po		;sonst aktuellen Zeiger der
	lds	r21,xet2po+1		;Temp-Daten 2 (24h) holen,
	lds	r22,xet2po+2		;zeigt auf ltesten Datensatz
	sts	pccpnt,r20
	sts	pccpnt+1,r21
	sts	pccpnt+2,r22		;EEPROM-Zeigeradresse setzen
;
m42100:	lds	xl,txp0wr
	lds	xh,txp0wr+1		;Sendepuffer Schreibzeiger hol.
	lds	yl,txp0rd
	lds	yh,txp0rd+1		;Sendepuffer Lesezeiger holen
	sub	xl,yl
	sbc	xh,yh			;Sendepuffer leer?
	breq	m42110			;ja -> weiter
	rjmp	m50000			;sonst Ende
;
m42110:	ldi	r23,76			;76 Bytes lesen (1 Datensatz)
	call	xread			;Datensatz aus Ext-EEPROM lesen
	brcc	m42120			;Fehler? nein -> weiter
	call	twierr			;sonst Speicherfehler setzen
	rjmp	m40200			;Kommando beenden
m42120:	sts	pccpnt,r20
	sts	pccpnt+1,r21		;neue EEPROM-Zeigeradresse
	sts	pccpnt+2,r22		;speichern
	adiw	r24,1			;Datensatz-Zhler erhhen
	sts	pcccnt,r24
	sts	pcccnt+1,r25		;Datensatz-Zhler speichern
;
; Daten aufbereiten und ausgeben
;
	lds	r16,xerbuf		;Zeitbyte 0 holen
	lds	r17,xerbuf+1		;Zeitbyte 1 holen
	and	r16,r17			;verknpfen
	lds	r17,xerbuf+2		;Zeitbyte 2 holen
	and	r16,r17			;verknpfen
	lds	r17,xerbuf+3		;Zeitbyte 3 holen
	and	r16,r17			;verknpfen
	cpi	r16,0xff		;gltiger Datenssatz?
	breq	m42300			;nein -> Ausgabe berspringen
;
	call	xtrtim			;sonst Zeit extrahieren
	ldi	yl,low(pcpara)
	ldi	yh,high(pcpara)		;Zeiger auf Parameterspeicher
	ld	r16,y			;erstes Zeichen vom Parameter
	tst	r16			;Parameter gltig?
	breq	m42250			;nein -> Datensatz ausgeben
;
	ldi	r18,8			;8 Zeichen vergleichen
	lds	r16,pcpara+8		;Parameter-Zusatz holen
	cpi	r16,'-'			;Minuszeichen?
	breq	m42210			;ja -> bearbeiten
;
	ldi	zl,low(ascbuf)
	ldi	zh,high(ascbuf)		;Zeiger auf ASCII-Puffer
m42200:	ld	r16,y+			;Zeichen aus Parametersp. holen
	ld	r17,z+			;Zeichen aus ASCII-Puffer holen
	tst	r16			;Parameter-Ende?
	breq	m42250			;ja -> Datensatz ausgeben
	cp	r16,r17			;Zeichen identisch?
	brne	m42300			;nein -> Ausgabe berspringen
	dec	r18			;alle Zeichen verglichen?
	brne	m42200			;nein -> Schleife
	rjmp	m42250			;sonst Datensatz ausgeben
;
m42210:	ldi	zl,low(ascbuf+8)	;Zeiger auf ASCII-Puffer
	ldi	zh,high(ascbuf+8)	;(Tag-Einer)
	ldi	yl,low(pcpara+8)	;Zeiger auf Parameterspeicher
	ldi	yh,high(pcpara+8)	;(Ende)
	clc				;bertrag lschen
m42220:	ld	r17,-z			;Zeichen aus ASCII-Puffer holen
	ld	r16,-y			;Zeichen aus Parametersp. holen
	tst	r16			;gltiges Zeichen?
	brne	m42240			;ja -> weiter
	dec	r18			;sonst alle Zeichen bearbeitet?
	brne	m42220			;nein -> Schleife
	rjmp	m42250			;sonst Datensatz ausgeben
;
m42230:	ld	r17,-z			;Zeichen aus ASCII-Puffer holen
	ld	r16,-y			;Zeichen aus Parametersp. holen
m42240:	sbc	r17,r16			;Pufferposition - Parameterpos.
	dec	r18			;alle Zeichen verglichen?
	brne	m42230			;nein -> Schleife
	brcs	m42300			;Param > Puffer? -> keine Ausg.
;
m42250:	call	xtrse0			;Sekunde 00 ergnzen
	ldi	yl,low(xerbuf+4)	;Zeiger auf Datensatz (Daten)
	ldi	yh,high(xerbuf+4)	;(Minimal-Werte)
	call	xtrdat			;Datensatz extrahieren
	call	senasc			;ASCII-Puffer senden
;
m42300:	ldi	r16,low(xt2sta+xt2len)
	ldi	r17,high(xt2sta+xt2len)	;End-Adresse Datenbereich 2
	ldi	r18,byte3(xt2sta+xt2len);laden (Min/Max/Mittel-Werte)
	sub	r20,r16
	sbc	r21,r17			;aktuelle Adresse noch im
	sbc	r22,r18			;gltigen Speicherbereich?
	brcs	m42310			;ja -> weiter
	ldi	r16,low(xt2sta)
	ldi	r17,high(xt2sta)	;sonst Adresse wieder auf
	ldi	r18,byte3(xt2sta)	;Anfang setzen
	sts	pccpnt,r16
	sts	pccpnt+1,r17		;neue EEPROM-Zeigeradresse
	sts	pccpnt+2,r18		;speichern
;
m42310:	lds	r20,pccpnt
	lds	r21,pccpnt+1
	lds	r22,pccpnt+2		;EEPROM-Zeigeradresse holen
	lds	r16,xet2po		;aktuellen EEPROM-Zeiger auf
	lds	r17,xet2po+1		;Tages-Daten holen = End-
	lds	r18,xet2po+2		;Adresse fr Datenausgabe
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;Endadresse erreicht?
	brne	m42400			;nein -> weiter
	rjmp	m40200			;sonst Kommando beenden
;
m42400:	rjmp	m50000			;Ende
;
; Kommando 5 bearbeiten (Temperaturdaten 2 senden, Maximal-Werte)
;
m43000:	cpi	r16,5			;Kommando 5 (Maximal-Werte)?
	breq	m43010			;ja -> ausfhren
	rjmp	m44000			;sonst nchstes Kommando
;
m43010:	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,3			;Speicher-Fehler?
	brtc	m43020			;nein -> weiter
	rjmp	m40200			;sonst Kommando beenden
m43020:	lds	r20,pccpnt
	lds	r21,pccpnt+1
	lds	r22,pccpnt+2		;EEPROM-Zeigeradresse holen
	lds	r24,pcccnt
	lds	r25,pcccnt+1		;Datensatz-Zhler holen
	mov	r16,r24
	or	r16,r25			;Datensatz-Zhler=0?
	brne	m43100			;nein -> nchster Datensatz
	lds	r20,xet2po		;sonst aktuellen Zeiger der
	lds	r21,xet2po+1		;Temp-Daten 2 (24h) holen,
	lds	r22,xet2po+2		;zeigt auf ltesten Datensatz
	sts	pccpnt,r20
	sts	pccpnt+1,r21
	sts	pccpnt+2,r22		;EEPROM-Zeigeradresse setzen
;
m43100:	lds	xl,txp0wr
	lds	xh,txp0wr+1		;Sendepuffer Schreibzeiger hol.
	lds	yl,txp0rd
	lds	yh,txp0rd+1		;Sendepuffer Lesezeiger holen
	sub	xl,yl
	sbc	xh,yh			;Sendepuffer leer?
	breq	m43110			;ja -> weiter
	rjmp	m50000			;sonst Ende
;
m43110:	ldi	r23,76			;76 Bytes lesen (1 Datensatz)
	call	xread			;Datensatz aus Ext-EEPROM lesen
	brcc	m43120			;Fehler? nein -> weiter
	call	twierr			;sonst Speicherfehler setzen
	rjmp	m40200			;Kommando beenden
m43120:	sts	pccpnt,r20
	sts	pccpnt+1,r21		;neue EEPROM-Zeigeradresse
	sts	pccpnt+2,r22		;speichern
	adiw	r24,1			;Datensatz-Zhler erhhen
	sts	pcccnt,r24
	sts	pcccnt+1,r25		;Datensatz-Zhler speichern
;
; Daten aufbereiten und ausgeben
;
	lds	r16,xerbuf		;Zeitbyte 0 holen
	lds	r17,xerbuf+1		;Zeitbyte 1 holen
	and	r16,r17			;verknpfen
	lds	r17,xerbuf+2		;Zeitbyte 2 holen
	and	r16,r17			;verknpfen
	lds	r17,xerbuf+3		;Zeitbyte 3 holen
	and	r16,r17			;verknpfen
	cpi	r16,0xff		;gltiger Datenssatz?
	breq	m43300			;nein -> Ausgabe berspringen
;
	call	xtrtim			;sonst Zeit extrahieren
	ldi	yl,low(pcpara)
	ldi	yh,high(pcpara)		;Zeiger auf Parameterspeicher
	ld	r16,y			;erstes Zeichen vom Parameter
	tst	r16			;Parameter gltig?
	breq	m43250			;nein -> Datensatz ausgeben
;
	ldi	r18,8			;8 Zeichen vergleichen
	lds	r16,pcpara+8		;Parameter-Zusatz holen
	cpi	r16,'-'			;Minuszeichen?
	breq	m43210			;ja -> bearbeiten
;
	ldi	zl,low(ascbuf)		;sonst
	ldi	zh,high(ascbuf)		;Zeiger auf ASCII-Puffer
m43200:	ld	r16,y+			;Zeichen aus Parametersp. holen
	ld	r17,z+			;Zeichen aus ASCII-Puffer holen
	tst	r16			;Parameter-Ende?
	breq	m43250			;ja -> Datensatz ausgeben
	cp	r16,r17			;Zeichen identisch?
	brne	m43300			;nein -> Ausgabe berspringen
	dec	r18			;alle Zeichen verglichen?
	brne	m43200			;nein -> Schleife
	rjmp	m43250
;
m43210:	ldi	zl,low(ascbuf+8)	;Zeiger auf ASCII-Puffer
	ldi	zh,high(ascbuf+8)	;(Tag-Einer)
	ldi	yl,low(pcpara+8)	;Zeiger auf Parameterspeicher
	ldi	yh,high(pcpara+8)	;(Ende)
	clc				;bertrag lschen
m43220:	ld	r17,-z			;Zeichen aus ASCII-Puffer holen
	ld	r16,-y			;Zeichen aus Parametersp. holen
	tst	r16			;gltiges Zeichen?
	brne	m43240			;ja -> weiter
	dec	r18			;sonst alle Zeichen bearbeitet?
	brne	m43220			;nein -> Schleife
	rjmp	m43250			;sonst Datensatz ausgeben
;
m43230:	ld	r17,-z			;Zeichen aus ASCII-Puffer holen
	ld	r16,-y			;Zeichen aus Parametersp. holen
m43240:	sbc	r17,r16			;Pufferposition - Parameterpos.
	dec	r18			;alle Zeichen verglichen?
	brne	m43230			;nein -> Schleife
	brcs	m43300			;Param > Puffer? -> keine Ausg.
;
m43250:	call	xtrse0			;Sekunde 00 ergnzen
	ldi	yl,low(xerbuf+4+24)	;Zeiger auf Datensatz (Daten)
	ldi	yh,high(xerbuf+4+24)	;(Maximal-Werte)
	call	xtrdat			;Datensatz extrahieren
	call	senasc			;ASCII-Puffer senden
;
m43300:	ldi	r16,low(xt2sta+xt2len)
	ldi	r17,high(xt2sta+xt2len)	;End-Adresse Datenbereich 2
	ldi	r18,byte3(xt2sta+xt2len);laden (Min/Max/Mittel-Werte)
	sub	r20,r16
	sbc	r21,r17			;aktuelle Adresse noch im
	sbc	r22,r18			;gltigen Speicherbereich?
	brcs	m43310			;ja -> weiter
	ldi	r16,low(xt2sta)
	ldi	r17,high(xt2sta)	;sonst Adresse wieder auf
	ldi	r18,byte3(xt2sta)	;Anfang setzen
	sts	pccpnt,r16
	sts	pccpnt+1,r17		;neue EEPROM-Zeigeradresse
	sts	pccpnt+2,r18		;speichern
;
m43310:	lds	r20,pccpnt
	lds	r21,pccpnt+1
	lds	r22,pccpnt+2		;EEPROM-Zeigeradresse holen
	lds	r16,xet2po		;aktuellen EEPROM-Zeiger auf
	lds	r17,xet2po+1		;Tages-Daten holen = End-
	lds	r18,xet2po+2		;Adresse fr Datenausgabe
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;Endadresse erreicht?
	brne	m43400			;nein -> weiter
	rjmp	m40200			;sonst Kommando beenden
;
m43400:	rjmp	m50000			;Ende
;
; Kommando 6 bearbeiten (Temperaturdaten 2 senden, Mittel-Werte)
;
m44000:	cpi	r16,6			;Kommando 6 (Mittel-Werte)?
	breq	m44010			;ja -> ausfhren
	rjmp	m45000			;sonst nchstes Kommando
;
m44010:	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,3			;Speicher-Fehler?
	brtc	m44020			;nein -> weiter
	rjmp	m40200			;sonst Kommando beenden
m44020:	lds	r20,pccpnt
	lds	r21,pccpnt+1
	lds	r22,pccpnt+2		;EEPROM-Zeigeradresse holen
	lds	r24,pcccnt
	lds	r25,pcccnt+1		;Datensatz-Zhler holen
	mov	r16,r24
	or	r16,r25			;Datensatz-Zhler=0?
	brne	m44100			;nein -> nchster Datensatz
	lds	r20,xet2po		;sonst aktuellen Zeiger der
	lds	r21,xet2po+1		;Temp-Daten 2 (24h) holen,
	lds	r22,xet2po+2		;zeigt auf ltesten Datensatz
	sts	pccpnt,r20
	sts	pccpnt+1,r21
	sts	pccpnt+2,r22		;EEPROM-Zeigeradresse setzen
;
m44100:	lds	xl,txp0wr
	lds	xh,txp0wr+1		;Sendepuffer Schreibzeiger hol.
	lds	yl,txp0rd
	lds	yh,txp0rd+1		;Sendepuffer Lesezeiger holen
	sub	xl,yl
	sbc	xh,yh			;Sendepuffer leer?
	breq	m44110			;ja -> weiter
	rjmp	m50000			;sonst Ende
;
m44110:	ldi	r23,76			;76 Bytes lesen (1 Datensatz)
	call	xread			;Datensatz aus Ext-EEPROM lesen
	brcc	m44120			;Fehler? nein -> weiter
	call	twierr			;sonst Speicherfehler setzen
	rjmp	m40200			;Kommando beenden
m44120:	sts	pccpnt,r20
	sts	pccpnt+1,r21		;neue EEPROM-Zeigeradresse
	sts	pccpnt+2,r22		;speichern
	adiw	r24,1			;Datensatz-Zhler erhhen
	sts	pcccnt,r24
	sts	pcccnt+1,r25		;Datensatz-Zhler speichern
;
; Daten aufbereiten und ausgeben
;
	lds	r16,xerbuf		;Zeitbyte 0 holen
	lds	r17,xerbuf+1		;Zeitbyte 1 holen
	and	r16,r17			;verknpfen
	lds	r17,xerbuf+2		;Zeitbyte 2 holen
	and	r16,r17			;verknpfen
	lds	r17,xerbuf+3		;Zeitbyte 3 holen
	and	r16,r17			;verknpfen
	cpi	r16,0xff		;gltiger Datenssatz?
	breq	m44300			;nein -> Ausgabe berspringen
;
	call	xtrtim			;sonst Zeit extrahieren
	ldi	yl,low(pcpara)
	ldi	yh,high(pcpara)		;Zeiger auf Parameterspeicher
	ld	r16,y			;erstes Zeichen vom Parameter
	tst	r16			;Parameter gltig?
	breq	m44250			;nein -> Datensatz ausgeben
;
	ldi	r18,8			;8 Zeichen vergleichen
	lds	r16,pcpara+8		;Parameter-Zusatz holen
	cpi	r16,'-'			;Minuszeichen?
	breq	m44210			;ja -> bearbeiten
;
	ldi	zl,low(ascbuf)		;sonst
	ldi	zh,high(ascbuf)		;Zeiger auf ASCII-Puffer
m44200:	ld	r16,y+			;Zeichen aus Parametersp. holen
	ld	r17,z+			;Zeichen aus ASCII-Puffer holen
	tst	r16			;Parameter-Ende?
	breq	m44250			;ja -> Datensatz ausgeben
	cp	r16,r17			;Zeichen identisch?
	brne	m44300			;nein -> Ausgabe berspringen
	dec	r18			;alle Zeichen verglichen?
	brne	m44200			;nein -> Schleife
	rjmp	m44250
;
m44210:	ldi	zl,low(ascbuf+8)	;Zeiger auf ASCII-Puffer
	ldi	zh,high(ascbuf+8)	;(Tag-Einer)
	ldi	yl,low(pcpara+8)	;Zeiger auf Parameterspeicher
	ldi	yh,high(pcpara+8)	;(Ende)
	clc				;bertrag lschen
m44220:	ld	r17,-z			;Zeichen aus ASCII-Puffer holen
	ld	r16,-y			;Zeichen aus Parametersp. holen
	tst	r16			;gltiges Zeichen?
	brne	m44240			;ja -> weiter
	dec	r18			;sonst alle Zeichen bearbeitet?
	brne	m44220			;nein -> Schleife
	rjmp	m44250			;sonst Datensatz ausgeben
;
m44230:	ld	r17,-z			;Zeichen aus ASCII-Puffer holen
	ld	r16,-y			;Zeichen aus Parametersp. holen
m44240:	sbc	r17,r16			;Pufferposition - Parameterpos.
	dec	r18			;alle Zeichen verglichen?
	brne	m44230			;nein -> Schleife
	brcs	m44300			;Param > Puffer? -> keine Ausg.
;
m44250:	call	xtrse0			;Sekunde 00 ergnzen
	ldi	yl,low(xerbuf+4+24+24)	;Zeiger auf Datensatz (Daten)
	ldi	yh,high(xerbuf+4+24+24)	;(Mittel-Werte)
	call	xtrdat			;Datensatz extrahieren
	call	senasc			;ASCII-Puffer senden
;
m44300:	ldi	r16,low(xt2sta+xt2len)
	ldi	r17,high(xt2sta+xt2len)	;End-Adresse Datenbereich 2
	ldi	r18,byte3(xt2sta+xt2len);laden (Min/Max/Mittel-Werte)
	sub	r20,r16
	sbc	r21,r17			;aktuelle Adresse noch im
	sbc	r22,r18			;gltigen Speicherbereich?
	brcs	m44310			;ja -> weiter
	ldi	r16,low(xt2sta)
	ldi	r17,high(xt2sta)	;sonst Adresse wieder auf
	ldi	r18,byte3(xt2sta)	;Anfang setzen
	sts	pccpnt,r16
	sts	pccpnt+1,r17		;neue EEPROM-Zeigeradresse
	sts	pccpnt+2,r18		;speichern
;
m44310:	lds	r20,pccpnt
	lds	r21,pccpnt+1
	lds	r22,pccpnt+2		;EEPROM-Zeigeradresse holen
	lds	r16,xet2po		;aktuellen EEPROM-Zeiger auf
	lds	r17,xet2po+1		;Tages-Daten holen = End-
	lds	r18,xet2po+2		;Adresse fr Datenausgabe
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;Endadresse erreicht?
	brne	m44400			;nein -> weiter
	rjmp	m40200			;sonst Kommando beenden
;
m44400:	rjmp	m50000			;Ende
;
; Kommando 7 bearbeiten (Meldungsdaten senden)
;
m45000:	cpi	r16,7			;Kommando 7 (Meldungs-Daten)?
	breq	m45010			;ja -> ausfhren
	rjmp	m46000			;sonst nchstes Kommando
;
m45010:	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,3			;Speicher-Fehler?
	brtc	m45020			;nein -> weiter
	rjmp	m40200			;sonst Kommando beenden
m45020:	lds	r20,pccpnt
	lds	r21,pccpnt+1
	lds	r22,pccpnt+2		;EEPROM-Zeigeradresse holen
	lds	r24,pcccnt
	lds	r25,pcccnt+1		;Datensatz-Zhler holen
	mov	r16,r24
	or	r16,r25			;Datensatz-Zhler=0?
	brne	m45100			;nein -> nchster Datensatz
	lds	r20,xemspo		;sonst aktuellen Zeiger der
	lds	r21,xemspo+1		;Meldungs-Daten holen, zeigt
	lds	r22,xemspo+2		;auf ltesten Datensatz
	sts	pccpnt,r20
	sts	pccpnt+1,r21
	sts	pccpnt+2,r22		;EEPROM-Zeigeradresse setzen
;
m45100:	lds	xl,txp0wr
	lds	xh,txp0wr+1		;Sendepuffer Schreibzeiger hol.
	lds	yl,txp0rd
	lds	yh,txp0rd+1		;Sendepuffer Lesezeiger holen
	sub	xl,yl
	sbc	xh,yh			;Sendepuffer leer?
	breq	m45110			;ja -> weiter
	rjmp	m50000			;sonst Ende
;
m45110:	ldi	r23,6			;6 Bytes lesen (1 Datensatz)
	call	xread			;Datensatz aus Ext-EEPROM lesen
	brcc	m45120			;Fehler? nein -> weiter
	call	twierr			;sonst Speicherfehler setzen
	rjmp	m40200			;Kommando beenden
m45120:	sts	pccpnt,r20
	sts	pccpnt+1,r21		;neue EEPROM-Zeigeradresse
	sts	pccpnt+2,r22		;speichern
	adiw	r24,1			;Datensatz-Zhler erhhen
	sts	pcccnt,r24
	sts	pcccnt+1,r25		;Datensatz-Zhler speichern
;
; Daten aufbereiten und ausgeben
;
	lds	r16,xerbuf		;Zeitbyte 0 holen
	lds	r17,xerbuf+1		;Zeitbyte 1 holen
	and	r16,r17			;verknpfen
	lds	r17,xerbuf+2		;Zeitbyte 2 holen
	and	r16,r17			;verknpfen
	lds	r17,xerbuf+3		;Zeitbyte 3 holen
	and	r16,r17			;verknpfen
	cpi	r16,0xff		;gltiger Datenssatz?
	brne	m45190			;ja -> ausgeben
	rjmp	m45300			;sonst Ausgabe berspringen
;
m45190:	call	xtrtim			;sonst Zeit extrahieren
	ldi	yl,low(pcpara)
	ldi	yh,high(pcpara)		;Zeiger auf Parameterspeicher
	ld	r16,y			;erstes Zeichen vom Parameter
	tst	r16			;Parameter gltig?
	breq	m45250			;nein -> Datensatz ausgeben
;
	ldi	r18,8			;8 Zeichen vergleichen
	lds	r16,pcpara+8		;Parameter-Zusatz holen
	cpi	r16,'-'			;Minuszeichen?
	breq	m45210			;ja -> bearbeiten
;
	ldi	zl,low(ascbuf)		;sonst
	ldi	zh,high(ascbuf)		;Zeiger auf ASCII-Puffer
m45200:	ld	r16,y+			;Zeichen aus Parametersp. holen
	ld	r17,z+			;Zeichen aus ASCII-Puffer holen
	tst	r16			;Parameter-Ende?
	breq	m45250			;ja -> Datensatz ausgeben
	cp	r16,r17			;Zeichen identisch?
	brne	m45300			;nein -> Ausgabe berspringen
	dec	r18			;alle Zeichen verglichen?
	brne	m45200			;nein -> Schleife
	rjmp	m45250
;
m45210:	ldi	zl,low(ascbuf+8)	;Zeiger auf ASCII-Puffer
	ldi	zh,high(ascbuf+8)	;(Tag-Einer)
	ldi	yl,low(pcpara+8)	;Zeiger auf Parameterspeicher
	ldi	yh,high(pcpara+8)	;(Ende)
	clc				;bertrag lschen
m45220:	ld	r17,-z			;Zeichen aus ASCII-Puffer holen
	ld	r16,-y			;Zeichen aus Parametersp. holen
	tst	r16			;gltiges Zeichen?
	brne	m45240			;ja -> weiter
	dec	r18			;sonst alle Zeichen bearbeitet?
	brne	m45220			;nein -> Schleife
	rjmp	m45250			;sonst Datensatz ausgeben
;
m45230:	ld	r17,-z			;Zeichen aus ASCII-Puffer holen
	ld	r16,-y			;Zeichen aus Parametersp. holen
m45240:	sbc	r17,r16			;Pufferposition - Parameterpos.
	dec	r18			;alle Zeichen verglichen?
	brne	m45230			;nein -> Schleife
	brcs	m45300			;Param > Puffer? -> keine Ausg.
;
m45250:	call	xtrsec			;Sekunden ergnzen
	lds	r17,xerbuf+5		;Meldungscode holen
	swap	r17			;Meldungsnummer an Bit3-Bit0
	ori	r17,0xf0		;Stringnummer generieren
	cpi	r17,0xfe		;gltige Stringnummer?
	brcc	m45280			;nein -> keinen String ausgeben
	call	xtrfst			;Meldungstext in Puffer legen
	ldi	r16,';'			;Listentrennzeichen laden
	st	x+,r16			;Zeichen in Puffer legen
	cpi	r17,0xf7		;Parameter ausgeben?
	brcs	m45280			;nein -> Daten senden
;
	lds	r18,xerbuf+5		;Meldungscode nochmals holen
	andi	r18,0x0f		;Sensor-/Alarmnummer filtern
	cpi	r17,0xf9		;Alarmnummer ausgeben?
	brcs	m45260			;ja -> weiter
;
	ldi	r19,9			;9 Zeichen Sensortext
	ldi	r16,low(etemtx)		;Zeiger auf EEPROM-Adresse
	ldi	r17,high(etemtx)	;der Sensortexte
	rjmp	m45270			;EEPROM-Text ausgeben
;
m45260:	cpi	r18,4			;Alarmnummer>3?
	brcc	m45280			;ja -> berspringen
	ldi	r19,11			;11 Zeichen Alarmtext
	ldi	r16,low(ealmtx)		;Zeiger auf EEPROM-Adresse
	ldi	r17,high(ealmtx)	;der Alarmtexte
;
m45270:	mul	r18,r19			;Adress-Offset ermitteln
	add	r16,r0
	adc	r17,r1			;Tabellenplatz ermitteln
	call	xtrest			;Sensor-/Alarmtext in Puffer
m45280:	call	xtbyte			;Anzahl der Bytes ermitteln
	call	senasc			;ASCII-Puffer senden
;
m45300:	ldi	r16,low(xmssta+xmslen)
	ldi	r17,high(xmssta+xmslen)	;End-Adresse Datenbereich 3
	ldi	r18,byte3(xmssta+xmslen);laden (Meldungs-Daten)
	sub	r20,r16
	sbc	r21,r17			;aktuelle Adresse noch im
	sbc	r22,r18			;gltigen Speicherbereich?
	brcs	m45310			;ja -> weiter
	ldi	r16,low(xmssta)
	ldi	r17,high(xmssta)	;sonst Adresse wieder auf
	ldi	r18,byte3(xmssta)	;Anfang setzen
	sts	pccpnt,r16
	sts	pccpnt+1,r17		;neue EEPROM-Zeigeradresse
	sts	pccpnt+2,r18		;speichern
;
m45310:	lds	r20,pccpnt
	lds	r21,pccpnt+1
	lds	r22,pccpnt+2		;EEPROM-Zeigeradresse holen
	lds	r16,xemspo		;aktuellen EEPROM-Zeiger auf
	lds	r17,xemspo+1		;Meldungs-Daten holen = End-
	lds	r18,xemspo+2		;Adresse fr Datenausgabe
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;Endadresse erreicht?
	brne	m45400			;nein -> weiter
	rjmp	m40200			;sonst Kommando beenden
;
m45400:	rjmp	m50000			;Ende
;
; Kommando 8 bearbeiten (Konfiguration senden)
;
m46000:	cpi	r16,8			;Kommando 8 (Konfiguration)?
	breq	m46010			;ja -> ausfhren
	rjmp	m47000			;sonst nchstes Kommando
;
m46010:	lds	xl,txp0wr
	lds	xh,txp0wr+1		;Sendepuffer Schreibzeiger hol.
	lds	yl,txp0rd
	lds	yh,txp0rd+1		;Sendepuffer Lesezeiger holen
	sub	xl,yl
	sbc	xh,yh			;Sendepuffer leer?
	breq	m46100			;ja -> weiter
	rjmp	m50000			;sonst Ende
;
; Daten aufbereiten und ausgeben
;
m46100:	ldi	xl,low(ascbuf)		;sonst
	ldi	xh,high(ascbuf)		;Zeiger auf ASCII-Puffer
	lds	r24,pcccnt		;Datensatz-Zhler L holen
	cpi	r24,16			;Sensordaten senden (>15)?
	brcc	m46200			;ja -> weiter
;
	ldi	r16,'L'			;Zeichen 'L'
	st	x+,r16			;in ASCII-Puffer speichern
	mov	r18,r24			;aktuelle Sensornummer kopieren
	inc	r18			;Sensornummer korrigieren
	call	bytasc			;in zweist. Dezimalzahl wandeln
	st	x+,r17			;Zehner in ASCII-Puffer
	st	x+,r16			;Einer in ACSII-Puffer
	ldi	r16,':'			;Doppelpunkt
	st	x+,r16			;in ASCII-Puffer speichern
;
	ldi	yl,low(logmap)
	ldi	yh,high(logmap)		;Zeiger auf Log-Tabelle
	ldi	r18,1			;Zhler=1 setzen
m46110:	ld	r17,y+			;Tabelleneintrag holen
	cp	r17,r24			;Sensornummer gefunden?
	breq	m46120			;ja -> Zhler ausgeben
	inc	r18			;nchster Tabelleneintrag
	cpi	r18,17			;letzter Eintrag erreicht?
	brne	m46110			;nein -> Schleife
	clr	r18			;sonst Zhler=0 setzen
m46120:	call	bytasc			;in zweist. Dezimalzahl wandeln
	st	x+,r17			;Zehner in ASCII-Puffer
	st	x+,r16			;Einer in ACSII-Puffer
	rjmp	m46410			;Daten ausgeben
;
m46200:	subi	r24,16			;Datensatz-Zhler korrigieren
	cpi	r24,16			;Alarmdaten senden (>15)?
	brcc	m46300			;ja -> weiter
;
	ldi	r16,'S'			;Zeichen 'S'
	st	x+,r16			;in ASCII-Puffer speichern
	mov	r18,r24			;aktuelle Sensornummer kopieren
	inc	r18			;Sensornummer korrigieren
	call	bytasc			;in zweist. Dezimalzahl wandeln
	st	x+,r17			;Zehner in ASCII-Puffer
	st	x+,r16			;Einer in ACSII-Puffer
	ldi	r16,':'			;Doppelpunkt
	st	x+,r16			;in ASCII-Puffer speichern
;
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperatur-Flags
	clr	r25
	add	yl,r24
	adc	yh,r25			;Offset addieren
	ld	r16,y			;Temperatur-Flags holen
	bst	r16,7			;Sensor inaktiv?
	brts	m46410			;ja -> keinen Text ausgeben
;
	ldi	r19,9			;9 Zeichen Sensortext
	ldi	r16,low(etemtx)		;Zeiger auf EEPROM-Adresse
	ldi	r17,high(etemtx)	;der Sensortexte
	rjmp	m46400			;Text ausgeben
;
m46300:	ldi	r16,'A'			;Zeichen 'A'
	st	x+,r16			;in ASCII-Puffer speichern
	ldi	r16,'L'			;Zeichen 'L'
	st	x+,r16			;in ASCII-Puffer speichern
	subi	r24,16			;Alarmnummer ermitteln
	mov	r16,r24			;Alarmnummer kopieren
	inc	r16			;Alarmnummer korrigieren
	ori	r16,0x30		;in ASCII wandeln
	st	x+,r16			;in ACSII-Puffer speichern
	ldi	r16,':'			;Doppelpunkt
	st	x+,r16			;in ASCII-Puffer speichern
;
	ldi	yl,low(almflg)
	ldi	yh,high(almflg)		;Zeiger auf Alarm-Flags
	clr	r25
	add	yl,r24
	adc	yh,r25			;Offset addieren
	ld	r16,y			;Alarm-Flags holen
	bst	r16,7			;Alarm inaktiv?
	brts	m46410			;ja -> keinen Text ausgeben
;
	ldi	r19,11			;11 Zeichen Alarmtext
	ldi	r16,low(ealmtx)		;Zeiger auf EEPROM-Adresse
	ldi	r17,high(ealmtx)	;der Alarmtexte
;
m46400:	mul	r24,r19			;Adress-Offset ermitteln
	add	r16,r0
	adc	r17,r1			;Tabellenplatz ermitteln
	call	xtrest			;Sensor-/Alarmtext in Puffer
m46410:	call	xtbyte			;Anzahl der Bytes ermitteln
	call	senasc			;ASCII-Puffer senden
;
	lds	r24,pcccnt		;Datensatz-Zhler L holen
	inc	r24			;Datensatz-Zhler L erhhen
	sts	pcccnt,r24		;Datensatz-Zhler L speichern
	cpi	r24,36			;Log/Sensor/Alarmdat. gesendet?
	brcs	m46500			;nein -> weiter
	rjmp	m40200			;sonst Kommando beenden
m46500:	rjmp	m50000			;Ende
;
m47000:					;weitere Kommandos
;
; DCF77-Auswertung in mehreren Stufen
;
m50000:	lds	r16,dcflag		;sonst DCF77-Flags holen
	bst	r16,3			;Beginn einer neuen Minute?
	brts	m50010			;ja -> auswerten
	rjmp	m60000			;sonst zum nchsten Abschnitt
m50010:	cbr	r16,0x08		;Minuten-Flag lschen
	sts	dcflag,r16		;DCF77-Flags wieder speichern
;
; DCF77-Auswertung: Parittsprfung fr Minuten, Stunden und Datum
;
	clr	r19			;Fehlerzhler lschen
	clr	r21			;fr Parittsberechnung
	lds	r22,dcfimp		;DCF77-Impulszhler holen
	sts	dcfimp,r21		;DCF77-Impulszhler lschen
	cpi	r22,59			;sind 59 Impulse eingegangen?
	breq	m50100			;ja -> weiter
	inc	r19			;sonst Fehlerzhler erhhen
	rjmp	m50500			;Auswertung vorzeitig beenden
;
m50100:	clr	r22			;Zhler fr Parittsberechnung
	lds	r23,dcftab+1		;Byte 1 aus Tabelle holen
	ldi	r20,6			;6 Minuten-Bits auswerten
m50110:	rol	r23			;relevantes Bit in Carry
	adc	r22,r21			;Carry addieren (wenn Carry=1)
	dec	r20			;alle Bits ausgewertet?
	brne	m50110			;nein -> Schleife
	lds	r23,dcftab+2		;Byte 2 aus Tabelle holen
	ror	r23			;relevantes Bit in Carry
	adc	r22,r21			;Carry addieren (wenn Carry=1)
	ror	r23			;relevantes Bit in Carry
	adc	r22,r21			;Carry addieren (wenn Carry=1)
	bst	r22,0			;war das Ergebnis geradzahlig?
	brtc	m50120			;ja -> weiter
	inc	r19			;sonst Fehlerzhler erhhen
;
m50120:	clr	r22			;Zhler fr Parittsberechnung
	ldi	r20,6			;6 Stunden-Bits auswerten
m50130:	ror	r23			;relevantes Bit in Carry
	adc	r22,r21			;Carry addieren (wenn Carry=1)
	dec	r20			;alle Bits ausgewertet?
	brne	m50130			;nein -> Schleife
	lds	r23,dcftab+3		;Byte 3 aus Tabelle holen
	ror	r23			;relevantes Bit in Carry
	adc	r22,r21			;Carry addieren (wenn Carry=1)
	bst	r22,0			;war das Ergebnis geradzahlig?
	brtc	m50140			;ja -> weiter
	inc	r19			;sonst Fehlerzhler erhhen
;
m50140:	clr	r22			;Zhler fr Parittsberechnung
	ldi	r20,7			;7 Datum-Bits auswerten
m50150:	ror	r23			;relevantes Bit in Carry
	adc	r22,r21			;Carry addieren (wenn Carry=1)
	dec	r20			;alle Bits ausgewertet?
	brne	m50150			;nein -> Schleife
	lds	r23,dcftab+4		;Byte 4 aus Tabelle holen
	ldi	r20,8			;weitere 8 Datum-Bits auswerten
m50160:	ror	r23			;relevantes Bit in Carry
	adc	r22,r21			;Carry addieren (wenn Carry=1)
	dec	r20			;alle Bits ausgewertet?
	brne	m50160			;nein -> Schleife
	lds	r23,dcftab+5		;Byte 5 aus Tabelle holen
	ldi	r20,8			;weitere 8 Datum-Bits auswerten
m50170:	ror	r23			;relevantes Bit in Carry
	adc	r22,r21			;Carry addieren (wenn Carry=1)
	dec	r20			;alle Bits ausgewertet?
	brne	m50170			;nein -> Schleife
	bst	r22,0			;war das Ergebnis geradzahlig?
	brtc	m50180			;ja -> weiter
	inc	r19			;sonst Fehlerzhler erhhen
m50180:	tst	r19			;Parittsfehler aufgetreten?
	breq	m50200			;nein -> weiter, sonst
	rjmp	m50500			;Auswertung vorzeitig beenden
;
; DCF77-Auswertung: Ablegen der Zeitinformation in Puffer 1
;
m50200:	ldi	zl,low(dcbuf1)
	ldi	zh,high(dcbuf1)		;Zeiger auf DCF77-Puffer 1
	lds	r23,dcftab		;Byte 0 aus Tabelle holen
	rol	r23
	rol	r23			;Bit 6 auf Bitposition 0
	rol	r23			;schieben
	andi	r23,0x01		;relevantes Bit filtern und
	st	z+,r23			;als Zeitzone speichern
	lds	r23,dcftab+1		;Byte 1 aus Tabelle holen
	ror	r23			;Bits 2-5 auf Bitposition 0-3
	ror	r23			;schieben
	andi	r23,0x0f		;relevante Bits filtern und
	st	z+,r23			;als Minuten Einer speichern
	lds	r23,dcftab+1		;Byte 1 aus Tabelle holen
	lds	r22,dcftab+2		;Byte 2 aus Tabelle holen
	rol	r23
	rol	r22			;Bits 6-7 aus r23 und Bit 0
	rol	r23			;aus r22 auf Bitposition 0-2
	rol	r22			;von r22 schieben
	andi	r22,0x07		;relevante Bits filtern und
	st	z+,r22			;als Minuten Zehner speichern
	lds	r23,dcftab+2		;Byte 2 aus Tabelle holen
	ror	r23			;Bits 2-5 auf Bitposition 0-3
	ror	r23			;schieben
	andi	r23,0x0f		;relevante Bits filtern und
	st	z+,r23			;als Stunden Einer speichern
	lds	r23,dcftab+2		;Byte 2 aus Tabelle holen
	rol	r23
	rol	r23			;Bits 6-7 auf Bitposition 0-1
	rol	r23			;schieben
	andi	r23,0x03		;relevante Bits filtern und
	st	z+,r23			;als Stunden Zehner speichern
	lds	r23,dcftab+3		;Byte 3 aus Tabelle holen
	ror	r23			;Bits 1-4 auf Bits 0-3 schieben
	andi	r23,0x0f		;relevante Bits filtern und als
	st	z+,r23			;Kalendertag Einer speichern
	lds	r23,dcftab+3		;Byte 3 aus Tabelle holen
	ror	r23			;Bits 5-6 auf Bits 4-5 schieben
	swap	r23			;Nibbles tauschen
	andi	r23,0x03		;relevante Bits filtern und als
	st	z+,r23			;Kalendertag Zehner speichern
	lds	r23,dcftab+3		;Byte 3 aus Tabelle holen
	rol	r23			;Bit 7 in Carry schieben
	lds	r23,dcftab+4		;Byte 4 aus Tabelle holen
	rol	r23			;Carry in Bit 0 schieben
	andi	r23,0x07		;relevante Bits filtern und
	st	z+,r23			;als Wochentag speichern
	lds	r23,dcftab+4		;Byte 4 aus Tabelle holen
	ror	r23			;Bits 2-5 auf Bitposition 0-3
	ror	r23			;schieben
	andi	r23,0x0f		;relevante Bits filtern und
	st	z+,r23			;als Monat Einer speichern
	lds	r23,dcftab+4		;Byte 4 aus Tabelle holen
	rol	r23
	rol	r23			;Bit 6 auf Bitposition 0
	rol	r23			;schieben
	andi	r23,0x01		;relevantes Bit filtern und
	st	z+,r23			;als Monat Zehner speichern
	lds	r23,dcftab+4		;Byte 4 aus Tabelle holen
	rol	r23			;Bit 7 in Carry schieben
	lds	r23,dcftab+5		;Byte 5 aus Tabelle holen
	rol	r23			;Carry in Bit 0 schieben
	andi	r23,0x0f		;relevante Bits filtern und
	st	z+,r23			;als Jahr Einer speichern
	lds	r23,dcftab+5		;Byte 5 aus Tabelle holen
	rol	r23			;Bits 3-6 in Bits 4-7 schieben
	swap	r23			;Nibbles tauschen
	andi	r23,0x0f		;relevante Bits filtern und
	st	z+,r23			;als Jahr Zehner speichern
;
; DCF77-Auswertung: Vergleich der aktuellen Zeit mit der in den beiden
; vorherigen Zyklen ermittelten Zeit, gleichzeitig Puffer 2 nach Puffer
; 3 und Puffer 1 nach Puffer 2 kopieren
;
	ldi	yl,low(dcbuf1)
	ldi	yh,high(dcbuf1)		;Zeiger auf DCF77-Puffer 1
	ldi	zl,low(dcbuf2)
	ldi	zh,high(dcbuf2)		;Zeiger auf DCF77-Puffer 2
	ldi	xl,low(dcbuf3)
	ldi	xh,high(dcbuf3)		;Zeiger auf DCF77-Puffer 3
	clr	r18			;Nullbyte-Detektor lschen
	ldi	r23,12			;12 Bytes vergleichen
m50300:	ld	r22,y+			;Puffer 1 Inhalt holen
	or	r18,r22			;Nullbyte-Detektor bearbeiten
	ld	r21,z			;Puffer 2 Inhalt holen
	ld	r20,x			;Puffer 3 Inhalt holen
	st	x+,r21			;Puffer 2 nach 3 kopieren
	st	z+,r22			;Puffer 1 nach 2 kopieren
	cp	r21,r22			;Inhalt P1/P2 identisch?
	brne	m50310			;nein -> Fehler
	cp	r20,r21			;Inhalt P2/P3 identisch?
	breq	m50320			;ja -> weiter
m50310:	inc	r19			;sonst Fehlerzhler erhhen
m50320:	dec	r23			;alle Bytes bearbeitet?
	brne	m50300			;nein -> Schleife
	tst	r18			;alles Nullbytes im Puffer?
	brne	m50400			;nein -> weiter
	inc	r19			;sonst Fehlerzhler erhhen
;
; DCF77-Auswertung: Puffer 2 und 3 um eine Minute erhhen
;
m50400:	ldi	zl,low(dcbuf2)
	ldi	zh,high(dcbuf2)		;Zeiger auf DCF77-Puffer 2
	call	incdcb			;um eine Minute erhhen
	ldi	zl,low(dcbuf3)
	ldi	zh,high(dcbuf3)		;Zeiger auf DCF77-Puffer 3
	call	incdcb			;um eine Minute erhhen
;
; DCF77-Auswertung: DCF77-Tabelle lschen
;
m50500:	ldi	zl,low(dcftab)		;Zeiger auf DCF77-Tabelle
	ldi	zh,high(dcftab)
	ldi	r23,6			;6 Bytes bearbeiten
	clr	r22			;Nullwert
m50510:	st	z+,r22			;Tabellenplatz lschen
	dec	r23			;alle Pltze bearbeitet?
	brne	m50510			;nein -> Schleife
	tst	r19			;DCF77 ok?
	breq	m50600			;ja -> Uhr stellen
	rjmp	m60000			;sonst zum nchsten Abschnitt
;
; DCF77-Auswertung: aktuelle Zeit in den Uhrenbereich kopieren und Uhr
; synchronisieren
;
m50600:	cli				;Int vorbergehend sperren
	lds	r23,dcbuf1		;Zeitzone holen
	sts	tzone,r23		;und speichern
	lds	r23,dcbuf1+5		;Kalendertag Einer holen
	sts	dayl,r23		;und speichern
	lds	r23,dcbuf1+6		;Kalendertag Zehner holen
	sts	dayh,r23		;und speichern
	lds	r23,dcbuf1+7		;Wochentag holen
	dec	r23			;korrigieren (1-7 -> 0-6)
	sts	dayw,r23		;und speichern
	lds	r23,dcbuf1+8		;Monat Einer holen
	sts	monthl,r23		;und speichern
	lds	r23,dcbuf1+9		;Monat Zehner holen
	sts	monthh,r23		;und speichern
	lds	r23,dcbuf1+10		;Jahr Einer holen
	sts	yearl,r23		;und speichern
	lds	r23,dcbuf1+11		;Jahr Zehner holen
	sts	yearh,r23		;und speichern
	lds	r23,dcbuf1+1		;Minuten Einer holen
	sts	minutl,r23		;und speichern
	lds	r23,dcbuf1+2		;Minuten Zehner holen
	sts	minuth,r23		;und speichern
	lds	r23,dcbuf1+3		;Stunden Einer holen
	sts	hourl,r23		;und speichern
	lds	r23,dcbuf1+4		;Stunden Zehner holen
	sts	hourh,r23		;und speichern
	clr	r23
	sts	secndl,r23		;Sekunden Einer lschen
	sts	secndh,r23		;Sekunden Zehner lschen
	sts	t1cnt2,r23		;Timer1 Interruptzhler lschen
	lds	r16,dcflag		;DCF77-Flags holen
	sbr	r16,0x04		;Synchronisations-Flag setzen
	sts	dcflag,r16		;und wieder speichern
	ldi	r16,timdcf		;DCF77-Timeout-Wert laden
	sts	dcftio,r16		;und als Timeout neu setzen
	sei				;Interrupts wieder aktivieren
;
	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,0			;Uhreinstellungsfehler?
	brtc	m50610			;nein -> berspringen
	cbr	r16,0x01		;sonst Uhren-Fehler lschen
	sbr	r16,0x80		;Fehlererinnerung setzen
	sts	errflg,r16		;Flags wieder speichern
	lds	r17,minutl		;Minuten Einer holen
	sts	minfl1,r17		;als Minutenflag1 voreinstellen
	sts	minfl3,r17		;als Minutenflag3 voreinstellen
	ser	r17			;0xff laden
	sts	minfl2,r17		;als Minutenflag2 voreinstellen
	lds	r17,hourl		;Stunden Einer holen
	sts	houflg,r17		;als Stundenflag voreinstellen
	bst	r16,4			;RTC-Fehler?
	brts	m50610			;ja -> berspringen
	call	tiwrtc			;sonst Zeit in RTC bertragen
	brcc	m50610			;Fehler? nein -> weiter
	call	i2cerr			;RTC-Fehler setzen
	ldi	r19,0x40		;Meldung "RTC-Fehler" setzen
	call	wrimsg			;Fehlermeldung schreiben
;
m50610:	lds	r16,dflags		;Anzeige-Flags holen
	sbr	r16,0x14		;Infozeile neu ausgeben, DCF ok
	sts	dflags,r16		;Anzeige-Flags wieder speichern
	bst	r16,7			;LCD wegen Sync. abgeschaltet?
	brtc	m60000			;nein -> weiter
	call	lcdon			;sonst LCD wieder einschalten
;
; Timeouts fr Temperaturen bearbeiten
;
m60000:	ldi	xl,low(temtio)
	ldi	xh,high(temtio)		;Zeiger auf Timeout-Zhler
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperatur-Flags
	clr	r18			;Sensornummern-Zhler
m60010:	ld	r17,y			;Temperatur-Flags holen
	bst	r17,7			;Temperatursensor aktiv?
	brts	m60030			;nein -> berspringen
	ld	r16,x			;sonst Timeout-Zhler holen
	tst	r16			;Zhler=0?
	brne	m60030			;nein -> berspringen, sonst
	sbr	r17,0x41		;nd.-Flag+Temp-Ausfall setzen
	st	y,r17			;und Flags wieder speichern
	dec	r16			;Timeout-Zhler auf 0xff setzen
	st	x,r16			;(Stopp) und wieder speichern
;
	lds	r19,errflg		;Fehler-Flags holen
	bst	r19,1			;Datenempfangsfehler?
	brts	m60020			;ja -> keine Sensorausflle
	push	r18			;akustisch melden, sonst
	lds	r18,signum+7		;Signalnummer fr Sensorausf.
	lds	r19,sigdur+7		;Signaldauer fr Sensorausf.
	call	signal			;Signal ausgeben
	pop	r18			;Register wieder herstellen
;
m60020:	push	r18
	push	xl
	push	xh
	push	yl
	push	yh			;Register sichern
	mov	r19,r18			;Sensornummer kopieren
	ori	r19,0x90		;Fehler "Sensorausfall" setzen
	call	wrimsg			;Fehlermeldung schreiben
	pop	yh
	pop	yl
	pop	xh
	pop	xl
	pop	r18			;Register wieder herstellen
;
m60030:	adiw	xl,1			;Zeiger auf nchsten Tabellen-
	adiw	yl,1			;Platz setzen
	inc	r18			;nchsten Sensor bearbeiten
	cpi	r18,16			;alle Sensoren bearbeitet?
	brne	m60010			;nein -> Schleife
;
; Timeout fr Meldungen bearbeiten
;
	lds	r16,msgtio		;Meldungs-Timeout-Zhler holen
	tst	r16			;Zhler=0?
	brne	m60040			;nein -> berspringen
	dec	r16			;Timeout-Zhler auf 0xff setzen
	sts	msgtio,r16		;(Stopp) und wieder speichern
	lds	r16,errflg		;Fehler-Flags holen
	sbr	r16,0x02		;Datenempfangsfehler setzen
	sts	errflg,r16		;Fehler-Flags wieder speichern
	lds	r16,dflags		;Anzeige-Flags holen
	sbr	r16,0x04		;Infozeile neu ausgeben
	sts	dflags,r16		;Anzeige-Flags wieder speichern
	lds	r18,signum+6		;Signalnummer fr Fehlermeldun.
	lds	r19,sigdur+6		;Signaldauer fr Fehlermeldung.
	call	signal			;Signal ausgeben
	ldi	r19,0x10		;Fehlercode "kein Datenempfang"
	call	wrimsg			;Fehlermeldung schreiben
;
; Timeout fr DCF77-Empfang bearbeiten
;
m60040:	lds	r16,dcftio		;DCF77-Timeout-Zhler holen
	tst	r16			;Zhler=0?
	brne	m60050			;nein -> berspringen
	dec	r16			;Timeout-Zhler auf 0xff setzen
	sts	dcftio,r16		;(Stopp) und wieder speichern
	lds	r16,dflags		;Anzeige-Flags holen
	cbr	r16,0x10		;DCF77-Empfangs-Flag lschen
	sbr	r16,0x04		;Infozeile neu ausgeben
	sts	dflags,r16		;Anzeige-Flags wieder speichern
;
; Timeout fr Men-Rckkehr bearbeiten
;
m60050:	lds	r16,mentio		;Men-Timeout-Zhler holen
	tst	r16			;Zhler=0?
	brne	m60080			;nein -> berspringen
	dec	r16			;Timeout-Zhler auf 0xff setzen
	sts	mentio,r16		;(Stopp) und wieder speichern
	clr	r18			;neue Menebene und Menoption
	lds	r17,dflags		;Anzeigeflags holen
	lds	r16,menopt		;derzeitige Menoption holen
	cp	r16,r18			;nderung der Menoption?
	breq	m60060			;nein -> berspringen
	sts	menopt,r18		;sonst Menoption 0 setzen
	sbr	r17,0x08		;Menzeile neu ausgeben
m60060:	lds	r16,mennum		;derzeitige Mennummer holen
	cp	r16,r18			;nderung der Menebene?
	breq	m60070			;nein -> berspringen
	sts	mennum,r18		;sonst Mennummer 0 setzen
	lds	r16,xflags		;verschiedene Flags holen
	cbr	r16,0x0c		;Min/Max-Anzeige abschalten
	sts	xflags,r16		;Flags wieder speichern
	clr	r16
	sts	encflg,r16		;Encoder-Flags lschen
	sbr	r17,0x0f		;Temp-Anzeige neu ausgeben
m60070:	sts	dflags,r17		;Anzeigeflags wieder speichern
;
; Timeout fr kurzzeitige Lautsprecher-Abschaltung bearbeiten
;
m60080:	lds	r16,lsptio		;Lautspr.-Timeout-Zhler holen
	tst	r16			;Zhler=0?
	brne	m60090			;nein -> berspringen
	dec	r16			;Timeout-Zhler auf 0xff setzen
	sts	lsptio,r16		;(Stopp) und wieder speichern
	lds	r17,xflag2		;verschiedene Flags 2 holen
	lds	r16,dflags		;Anzeigeflags holen
	bst	r17,0			;Lautsprecher-Status holen
	bld	r16,5			;und neu setzen
	sbr	r16,0x04		;Infozeile neu ausgeben
	sts	dflags,r16		;Anzeigeflags wieder speichern
;
; Timeout fr LCD-Abschaltung bearbeiten
;
m60090:	lds	r16,lcdtio		;LCD-Timeout-Zhler holen
	tst	r16			;Zhler=0?
	brne	m61000			;nein -> berspringen
	dec	r16			;Timeout-Zhler auf 0xff setzen
	sts	lcdtio,r16		;(Stopp) und wieder speichern
	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,7			;LCD wegen Sync. abgeschaltet?
	brtc	m61000			;nein -> weiter
	call	lcdon			;LCD wieder einschalten
;
; Timeout fr zyklische Fehleranzeige bearbeiten
;
m61000:	lds	r16,errflg		;Fehler-Flags holen
	lds	r17,errcpy		;Fehler-Flags Kopie holen
	sts	errcpy,r16		;neue Kopie erzeugen
	cp	r16,r17			;Fehler-nderungen?
	brne	m61010			;ja -> bearbeiten
	lds	r17,errtio		;Fehleranzeige-Timeout holen
	cpi	r17,0xff		;Zhler=0xff (gestoppt)?
	breq	m61010			;ja -> Fehler prfen
	tst	r17			;Zhler=0?
	brne	m62000			;nein -> berspringen
	dec	r17			;Timeout-Zhler auf 0xff setzen
	sts	errtio,r17		;(Stopp) und wieder speichern
;
m61010:	lds	r17,errmsk		;Fehler-Maske holen
	mov	r19,r17			;aktuelle Maske kopieren
	andi	r16,0x7f		;Fehler-Erinnerung ausblenden
	breq	m61030			;Fehler? nein -> weiter
;
	ldi	r20,9			;9 Maskenverschiebungen
	clc				;Carry lschen
	tst	r17			;Maske vorhanden?
	brne	m61020			;ja -> weiter
	sec				;sonst Carry setzen
m61020:	rol	r17			;Maske auf nchsten Fehler
	mov	r18,r16			;Fehler-Flags kopieren
	and	r18,r17			;Maske ber Fehler-Flags legen
	brne	m61040			;Fehler gefunden? ja -> weiter
	dec	r20			;alle Verschiebungen bearbeit.?
	brne	m61020			;ja -> Schleife
;
m61030:	clr	r17			;sonst Maske lschen
m61040:	sts	errmsk,r17		;und speichern
	cp	r17,r19			;wurde Maske gendert?
	breq	m62000			;nein -> weiter
	lds	r16,dflags		;Anzeige-Flags holen
	sbr	r16,0x04		;Infozeile neu ausgeben
	sts	dflags,r16		;Anzeige-Flags wieder speichern
	tst	r17			;Maske vorhanden?
	breq	m62000			;nein -> Ende
	ldi	r16,timerr		;sonst Timeout-Wert holen
	sts	errtio,r16		;und Timeout-Zhler neu laden
;
; Optische Signalisierung von Alarmen ber LED1 und LED2:
; 1. Systemfehler:	LED1 (rot) leuchtet stndig
; 2. Sensorausfall:	LED2 (gelb) leuchtet stndig
; 3. Alarmmeldung:	LED1 (rot) blinkt
; 4. Temperaturalarm:	LED2 (gelb) blinkt
;
m62000:	lds	r17,ledflg		;LED-Flags holen
	cbr	r17,0x0f		;zunchst alle LEDs ausschalten
	lds	r16,errflg		;Fehlerflags holen
	andi	r16,0x7f		;Fehlererinnerung ausblenden
	breq	m62100			;Fehler? nein -> weiter
	sbr	r17,0x01		;sonst LED1 (rot) einschalten
;
m62100:	ldi	xl,low(temflg)
	ldi	xh,high(temflg)		;Zeiger auf Temperaturflags
	ldi	r20,16			;16 Temperaturen bearbeiten
	clr	r19			;Alarmspeicher lschen
m62110:	ld	r18,x+			;Temperatur-Flags holen
	andi	r18,0x40		;Sensorausfall-Flag filtern
	or	r19,r18			;Flag speichern wenn gesetzt
	dec	r20			;alle Flags bearbeitet?
	brne	m62110			;nein -> Schleife
	tst	r19			;Sensorausflle vorhanden?
	breq	m62200			;nein -> weiter
	sbr	r17,0x02		;sonst LED2 (gelb) einschalten
;
m62200:	ldi	xl,low(almflg)
	ldi	xh,high(almflg)		;Zeiger auf Alarmflags
	ldi	r20,4			;4 Alarme bearbeiten
	clr	r19			;Alarmspeicher lschen
m62210:	ld	r18,x+			;Alarmflag holen
	andi	r18,0x02		;Flag fr aktiven Alarm filtern
	or	r19,r18			;Flag verknpfen
	dec	r20			;alle Alarme bearbeitet?
	brne	m62210			;nein -> Schleife
	tst	r19			;aktive Alarme vorhanden?
	breq	m62300			;nein -> weiter
	sbr	r17,0x04		;sonst LED1 (rot) blinken
;
m62300:	ldi	xl,low(temflg)
	ldi	xh,high(temflg)		;Zeiger auf Temperaturflags
	ldi	r20,16			;16 Temperaturen bearbeiten
	clr	r19			;Alarmspeicher lschen
m62310:	ld	r18,x+			;Temperaturflag holen
	andi	r18,0x0c		;Temp-Alarmflags filtern
	or	r19,r18			;Flag speichern wenn gesetzt
	dec	r20			;alle Flags bearbeitet?
	brne	m62310			;nein -> Schleife
	tst	r19			;aktive Temp-Alarme vorhanden?
	breq	m62400			;nein -> weiter
	sbr	r17,0x08		;sonst LED2 (gelb) blinken
;
m62400:	sts	ledflg,r17		;LED-Flags wieder speichern
;
; Akustische Signalisierung steuern, Warteschlange auslesen
;
m64000:	lds	xl,sigbwr
	lds	xh,sigbwr+1		;Schreibzeiger holen
	lds	yl,sigbrd
	lds	yh,sigbrd+1		;Lesezeiger holen
	sub	xl,yl
	sbc	xh,yh			;Differenz berechnen
	or	xh,xl			;sind Signaldaten im Puffer?
	breq	m65000			;nein -> Ende
	lds	r16,sigtim		;Status des Signaltimers holen
	tst	r16			;werden Signal ausgegeben?
	brne	m65000			;ja -> berspringen
	ld	r16,y+			;Signalnummer holen, Pointer+1
	ld	r17,y+			;Signaldauer holen, Pointer+1
	lds	r18,dflags		;Anzeige-Flags holen
	bst	r18,5			;Lautsprecher eingeschaltet?
	brtc	m64010			;nein -> keine Signalausgabe
	cli				;Interrupts sperren
	sts	sigsnd,r16		;Signalnummer setzen
	sts	sigtim,r17		;Signaldauer setzen
	sei				;Interrupts freigeben
m64010:	ldi	xl,low(sigbuf+(sigbul*2))
	ldi	xh,high(sigbuf+(sigbul*2));Adresse Pufferende laden
	sub	xl,yl
	sbc	xh,yh			;Differenz berechnen
	or	xh,xl			;Endadresse erreicht?
	brne	m64020			;nein -> weiter
	ldi	yl,low(sigbuf)
	ldi	yh,high(sigbuf)		;sonst Adresse auf Pufferanfang
m64020:	sts	sigbrd,yl
	sts	sigbrd+1,yh		;Schreibzeiger speichern
;
; Signalausgabe prfen und Signal-Flag setzen
;
m65000:	clr	r18			;Signalmerker lschen
	lds	r16,sigtim		;Signaldauer holen?
	tst	r16			;wird ein Signal ausgegeben?
	brne	m65010			;ja -> merken
	lds	xl,sigbwr		;Signalpuffer Schreibzeiger
	lds	xh,sigbwr+1		;holen
	lds	yl,sigbrd		;Signalpuffer Lesezeiger
	lds	yh,sigbrd+1		;holen
	sub	xl,yl
	sbc	xh,yh			;Differenz berechnen
	or	xh,xl			;sind Signaldaten im Puffer?
	breq	m65020			;nein -> weiter
m65010:	ldi	r18,0x01		;Signalmerker setzen
m65020:	lds	r16,xflags		;verschiedene Flags holen
	andi	r16,0x01		;Signalflag filtern
	cp	r16,r18			;Statusnderung?
	breq	m65030			;nein -> berspringen
	lds	r17,mennum		;aktuelle Mennummer holen
	tst	r17			;Hauptmen (0)?
	brne	m65030			;nein -> berspringen
	lds	r17,dflags		;sonst Anzeige-Flags holen
	sbr	r17,0x08		;Menzeile neu ausgeben
	sts	dflags,r17		;Anzeige-Flags wieder speichern
m65030:	lds	r16,xflags		;verschiedene Flags holen
	bst	r18,0			;ermittelten Signalstatus
	bld	r16,0			;holen und neu setzen
	sts	xflags,r16		;Flags wieder speichern
;
; alle Erinnerungen prfen und allgemeines Erinnerungs-Flag setzen,
; auerdem Erinnerungsflags fr LED-Signalisierung setzen
;
m66000: clr	r18			;Erinnerungsmerker lschen
	lds	r19,ledflg		;LED-Flags holen
	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,7			;Erinnerung gesetzt?
	brtc	m66010			;nein -> weiter
	ldi	r18,0x02		;sonst Erinnerung merken
	sbr	r19,0x10		;LED1 Erinnerung setzen
;
m66010:	ldi	r17,16			;16 Temperaturflags bearbeiten
	ldi	xl,low(temflg)
	ldi	xh,high(temflg)		;Zeiger auf Temperaturflags
m66020:	ld	r16,x+			;Flags holen
	bst	r16,7			;Sensor inaktiv?
	brts	m66030			;ja -> berspringen
	andi	r16,0x30		;Erinnerungen gesetzt?
	breq	m66030			;nein -> weiter
	ldi	r18,0x02		;sonst Erinnerung merken
	sbr	r19,0x20		;LED2 Erinnerung setzen
m66030:	dec	r17			;alle Sensoren bearbeitet?
	brne	m66020			;nein -> Schleife
;
	ldi	r17,4			;4 Alarmflags bearbeiten
	ldi	xl,low(almflg)
	ldi	xh,high(almflg)		;Zeiger auf Alarmflags
m66040:	ld	r16,x+			;Flags holen
	bst	r16,7			;Alarmeingang inaktiv?
	brts	m66050			;ja -> berspringen
	bst	r16,2			;Erinnerung gesetzt?
	brtc	m66050			;nein -> weiter
	ldi	r18,0x02		;sonst Erinnerung merken
	sbr	r19,0x10		;LED1 Erinnerung setzen
m66050:	dec	r17			;alle Alarme bearbeitet?
	brne	m66040			;nein -> Schleife
	sts	ledflg,r19		;LED-Flags wieder speichern
;
	lds	r16,xflags		;verschiedene Flags holen
	andi	r16,0x02		;allgemeine Erinnerung filtern
	cp	r16,r18			;Statusnderung?
	breq	m66060			;nein -> berspringen
	lds	r17,mennum		;aktuelle Mennummer holen
	tst	r17			;Hauptmen (0)?
	brne	m66060			;nein -> berspringen
	lds	r17,dflags		;sonst Anzeige-Flags holen
	sbr	r17,0x08		;Menzeile neu ausgeben
	sts	dflags,r17		;Anzeige-Flags wieder speichern
m66060:	lds	r16,xflags		;verschiedene Flags holen
	bst	r18,1			;ermittelten Erinnerungsstatus
	bld	r16,1			;holen und neu setzen
	sts	xflags,r16		;Flags wieder speichern
;
; LCD-Hintergrundbeleuchtung steuern
;
	lds	r16,adclig
	lds	r17,adclig+1		;ADC0-Wert (Lichtsensor) holen
	lsr	r17
	ror	r16
	lsr	r17			;10-Bit-Wert in 8-Bit-Wert
	ror	r16			;umwandeln
	lds	r18,light		;Schwellwert Lichtsensor holen
	tst	r18			;Wert>127?
	brpl	m66900			;nein -> berspringen
	neg	r18			;sonst Zweierkomplement bilden
m66900:	swap	r18			;oberes Nibble holen und
	andi	r18,0x0f		;filtern, =Hysterese fr Licht
	brne	m66910			;Hysterese=0? nein -> weiter
	ldi	r18,1			;sonst Hysterese=1 setzen
m66910:	lds	r17,light		;Schwellwert Lichtsensor holen
	lds	r19,xflags		;verschiedene Flags holen
	bst	r19,4			;war Lichtsensor hell?
	brtc	m66950			;nein -> weiter
	sub	r17,r18			;Schwellwert - Hysterese
	sub	r16,r17			;Sensorwert<Schwellwert?
	brcc	m66980			;nein -> weiter
	cbr	r19,0x10		;sonst Lichtsensor dunkel
	rjmp	m66980
m66950:	add	r17,r18			;Schwellwert + Hysterese
	sub	r17,r16			;Sensorwert>Schwellwert?
	brcc	m66980			;nein -> weiter
	sbr	r19,0x10		;sonst Lichtsensor hell
m66980:	sts	xflags,r19
;
	lds	r17,dflags		;Anzeigeflags holen
	bst	r19,5			;Nachtschaltung aktiv?
	brtc	m67010			;nein -> bearbeiten
m67000:	lds	r16,mentio		;sonst Men-Timeout-Zhler hol.
	cpi	r16,0xff		;luft Timeout-Zhler?
	brne	m67020			;ja -> LCD-Beleuchtung ein
	cbr	r17,0x40		;sonst LCD-Beleuchtung aus
	rjmp	m67030			;Flags wieder speichern
;
m67010:	bst	r19,4			;Lichtsensor hell?
	brts	m67000			;ja -> LCD-Beleuchtung aus
m67020:	sbr	r17,0x40		;sonst LCD-Beleuchtung ein
m67030:	sts	dflags,r17		;Flags wieder speichern
;
; Lausprecherstatus setzen
;
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,7			;Status Abwesend?
	brts	m67100			;ja -> berspringen
	lds	r16,lsptio		;Lautsprecher-Timeout-Z. holen
	cpi	r16,0xff		;Zhler aktiv?
	brne	m67100			;ja -> berspringen
	lds	r16,xflag2		;sonst versch. Flags 2 holen
	lds	r17,dflags		;Anzeigeflags holen
	mov	r18,r17			;und kopieren
	bst	r16,0			;Lautsprecherstatus holen
	bld	r17,5			;und neu setzen
	cp	r17,r18			;hat sich der Status gendert?
	breq	m67040			;nein -> weiter
	sbr	r17,0x04		;sonst Infozeile neu ausgeben
m67040:	sts	dflags,r17		;Anzeigeflags wieder speichern
;
; Konfiguration nach nderung im EEPROM speichern
;
m67100:	lds	r16,xflag3		;verschiedene Flags 3 holen
	bst	r16,0			;Konfig im EEPROM speichern?
	brtc	m68000			;nein -> berspringen
	cbr	r16,0x01		;sonst Konfig-Flag lschen
	sts	xflag3,r16		;Flags wieder speichern
	call	strwai			;Warte-Text ausgeben
	call	eewcfg			;sonst Konfiguration speichern
	lds	r16,dflags		;Anzeige-Flags holen
	sbr	r16,1<<3		;Menzeile neu ausgeben
	sts	dflags,r16		;Flags wieder speichern
;
; Zeitabhngige Vorgnge steuern - Minutenwechsel prfen, Infozeile neu
; ausgeben
;
m68000:	cli				;Interrupts sperren
	lds	r16,minutl		;aktuellen Minuten-Einer holen
	lds	r17,minuth		;aktuellen Minuten-Zehner holen
	lds	r18,hourl		;aktuellen Stunden-Einer holen
	lds	r19,hourh		;aktuellen Stunden-Zehner holen
	sei				;Interrupts wieder freigeben
	lds	r20,minfl1		;Minuten-Flag 1 holen
	cp	r16,r20			;Minuten-Wechsel?
	breq	m68010			;nein -> berspringen
	sts	minfl1,r16		;neues Minuten-Flag 1 speichern
	lds	r20,dflags		;Anzeigeflags holen
	sbr	r20,0x04		;Infozeile neu ausgeben
	sts	dflags,r20		;Flags wieder speichern
;
; Stunden- und Tageswechsel prfen; bei DCF77-Synchronisation in der
; vergangenen Stunde wird die RTC neu gesetzt, sonst werden die Zeit-
; daten von der RTC geholt; zum Tageswechsel erfolgt keine Synchronisa-
; sation mit der RTC
;
m68010:	lds	r20,errflg		;Fehler-Flags holen
	bst	r20,4			;RTC-Fehler?
	brts	m68100			;ja -> berspringen
	lds	r20,houflg		;sonst Stunden-Flag holen
	cp	r18,r20			;Stunden-Wechsel?
	breq	m68100			;nein -> berspringen
	sts	houflg,r18		;neues Stunden-Flag speichern
;
	cpi	r16,9			;Minuten-Einer=9?
	brne	m68020			;nein -> 0:00 Uhr testen
	cpi	r17,5			;Minuten-Zehner=5?
	brne	m68020			;nein -> 0:00 Uhr testen
	cpi	r18,3			;Stunden-Einer=3?
	brne	m68020			;nein -> 0:00 Uhr testen
	cpi	r19,2			;Stunden-Zehner=2?
	breq	m68100			;ja -> kein Uhrenabgleich
m68020:	or	r16,r17
	or	r16,r18			;Stunden- und Minuten-Werte
	or	r16,r19			;verknpfen, Zeit 0:00 Uhr?
	breq	m68100			;ja -> kein Uhrenabgleich
;
	lds	r16,dcflag		;DCF77-Flags holen
	bst	r16,2			;Synchronisations-Flag gesetzt?
	brtc	m68030			;nein -> Zeit aus RTC holen
	cbr	r16,1<<2		;Synchronisations-Flag lschen
	sts	dcflag,r16		;Flags wieder speichern
	call	tiwrtc			;Uhrzeit in RTC bertragen
	rjmp	m68040
m68030:	call	tirrtc			;sonst Uhrzeit aus RTC holen
m68040:	brcc	m68100			;I2C-Fehler? nein -> weiter
	call	i2cerr			;sonst Fehler melden
	ldi	r19,0x40		;Meldung "RTC-Fehler" setzen
	call	wrimsg			;Fehlermeldung schreiben
;
; Weitere Aktionen nur bei synchronisierter Uhr ausfhren
;
m68100:	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,0			;ist Uhr eingestellt?
	brtc	m68110			;ja -> bearbeiten
	rjmp	m70000			;sonst berspringen
;
m68110:	lds	r16,minutl		;aktuellen Minuten-Einer holen
	lds	r17,minfl2		;Minuten-Flag 2 holen
	cp	r16,r17			;Minuten-Wechsel?
	brne	m68120			;ja -> weiter
	rjmp	m69000			;sonst berspringen
m68120:	sts	minfl2,r16		;sonst Minuten-Flag 2 speichern
;
; Aktuelle Uhrzeit zu einem 16-Bit-Minutenwert zusammenfassen
;
	ldi	r16,10			;Multiplikator 10 laden
	lds	r15,hourh		;Stunden Zehner holen
	mul	r15,r16			;verzehnfachen
	lds	r15,hourl		;Stunden Einer holen
	add	r15,r0			;Zehner addieren
	lds	r14,minuth		;Minuten Zehner holen
	mul	r14,r16			;verzehnfachen
	lds	r14,minutl		;Minuten Einer holen
	add	r14,r0			;Zehner addieren
	ldi	r16,60			;Multiplikator 60 laden
	mul	r15,r16			;Stunden in Minuten umrechnen
	clr	r16
	add	r0,r14			;Minuten und bertrag addieren
	adc	r1,r16			;Minutenwert in r0/r1 (L/H)
;
; Nachtschaltung der LCD-Hintergrundbeleuchtung steuern
;
	lds	r20,xflags		;verschiedene Flags holen
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,7			;Status Abwesend?
	brts	m68210			;ja -> Nachtmodus aktivieren
	bst	r16,3			;LCD-Nachtschaltung aktiv?
	brtc	m68250			;nein -> berspringen
;
	lds	r16,nighbl		;Nacht-Startzeit L holen
	lds	r17,nighbl+1		;Nacht-Startzeit H holen
	lds	r18,nighbl+2		;Nacht-Endezeit L holen
	lds	r19,nighbl+3		;Nacht-Endezeit H holen
	movw	r14,r18			;Endezeit kopieren
	sub	r14,r16			;Endezeit - Startzeit
	sbc	r15,r17			;Startzeit > Endezeit?
	brcs	m68200			;ja -> bearbeiten
;
	movw	r14,r0			;aktuelle Zeit kopieren
	sub	r14,r16
	sbc	r15,r17			;Startzeit erreicht?
	brcs	m68220			;nein -> Nachtmodus aus
	movw	r14,r0			;aktuelle Zeit kopieren
	sub	r14,r18
	sbc	r15,r19			;Endezeit erreicht?
	brcs	m68210			;nein -> Nachtmodus ein
	rjmp	m68220			;sonst Nachtmodus aus
;
m68200:	movw	r14,r0			;aktuelle Zeit kopieren
	sub	r14,r18
	sbc	r15,r19			;Endezeit erreicht?
	brcs	m68210			;nein -> Nachtmodus ein
	movw	r14,r0			;aktuelle Zeit kopieren
	sub	r14,r16
	sbc	r15,r17			;Startzeit erreicht?
	brcs	m68220			;nein -> Nachtmodus aus
;
m68210:	sbr	r20,0x20		;Nachtmodus aktivieren
	rjmp	m68230
m68220:	cbr	r20,0x20		;Nachtmodus deaktivieren
m68230:	sts	xflags,r20		;Flags wieder speichern
;
; Nachtschaltung des Bluetooth-Moduls steuern
;
m68250:	lds	r20,xflags		;verschiedene Flags holen
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,7			;Status Abwesend?
	brts	m68270			;ja -> Bluetooth ausschalten
	lds	r16,confg2		;Konfigurationsbyte 2 holen
	bst	r16,2			;Bluetooth-Nachtsch. aktiv?
	brtc	m68280			;nein -> Bluetooth einschalten
;
	lds	r16,nighbt		;Nacht-Startzeit L holen
	lds	r17,nighbt+1		;Nacht-Startzeit H holen
	lds	r18,nighbt+2		;Nacht-Endezeit L holen
	lds	r19,nighbt+3		;Nacht-Endezeit H holen
	movw	r14,r18			;Endezeit kopieren
	sub	r14,r16			;Endezeit - Startzeit
	sbc	r15,r17			;Startzeit > Endezeit?
	brcs	m68260			;ja -> bearbeiten
;
	movw	r14,r0			;aktuelle Zeit kopieren
	sub	r14,r16
	sbc	r15,r17			;Startzeit erreicht?
	brcs	m68280			;nein -> Nachtmodus aus
	movw	r14,r0			;aktuelle Zeit kopieren
	sub	r14,r18
	sbc	r15,r19			;Endezeit erreicht?
	brcs	m68270			;nein -> Nachtmodus ein
	rjmp	m68280			;sonst Nachtmodus aus
;
m68260:	movw	r14,r0			;aktuelle Zeit kopieren
	sub	r14,r18
	sbc	r15,r19			;Endezeit erreicht?
	brcs	m68270			;nein -> Nachtmodus ein
	movw	r14,r0			;aktuelle Zeit kopieren
	sub	r14,r16
	sbc	r15,r17			;Startzeit erreicht?
	brcs	m68280			;nein -> Nachtmodus aus
;
m68270:	cbi	portb,5			;Bluetooth ausschalten
	rjmp	m68300
m68280:	sbi	portb,5			;Bluetooth einschalten
;
; Automatisches Rcksetzen der Minimal- und Maximalwerte steuern
;
m68300:	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,4			;Auto-Reset Min/Maxwerte aktiv?
	brtc	m68400			;nein -> berspringen
;
	lds	r16,aresmm		;Auto-Reset Min-Werte L holen
	lds	r17,aresmm+1		;Auto-Reset Min-Werte H holen
	sub	r16,r0
	sbc	r17,r1			;Auto-Reset-Zeit erreicht?
	brne	m68310			;nein -> weiter
	call	rsmin1			;sonst Minimal-Werte lschen
;
m68310:	lds	r16,aresmm+2		;Auto-Reset Max-Werte L holen
	lds	r17,aresmm+3		;Auto-Reset Max-Werte H holen
	sub	r16,r0
	sbc	r17,r1			;Auto-Reset-Zeit erreicht?
	brne	m68400			;nein -> weiter
	call	rsmax1			;sonst Maximal-Werte lschen
;
; Nachtschaltung der Signal-Lautstrke steuern
;
m68400:	lds	r16,dimode		;alternativen Anzeigemodus hol.
	cpi	r16,13			;Modus Konf/Sig/Audio/Auswhl?
	breq	m68500			;ja -> berspringen
	cpi	r16,14			;Modus Konf/Sig/Audio/ndern?
	breq	m68500			;ja -> berspringen
;
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,5			;Signal-Nachtschaltung aktiv?
	brtc	m68430			;nein -> Taglautstrke setzen
;
	lds	r16,signig		;Nacht-Startzeit L holen
	lds	r17,signig+1		;Nacht-Startzeit H holen
	lds	r18,signig+2		;Nacht-Endezeit L holen
	lds	r19,signig+3		;Nacht-Endezeit H holen
	movw	r14,r18			;Endezeit kopieren
	sub	r14,r16			;Endezeit - Startzeit
	sbc	r15,r17			;Startzeit > Endezeit?
	brcs	m68410			;ja -> bearbeiten
;
	movw	r14,r0			;aktuelle Zeit kopieren
	sub	r14,r16
	sbc	r15,r17			;Startzeit erreicht?
	brcs	m68430			;nein -> Nachtmodus aus
	movw	r14,r0			;aktuelle Zeit kopieren
	sub	r14,r18
	sbc	r15,r19			;Endezeit erreicht?
	brcs	m68420			;nein -> Nachtmodus ein
	rjmp	m68430			;sonst Nachtmodus aus
;
m68410:	movw	r14,r0			;aktuelle Zeit kopieren
	sub	r14,r18
	sbc	r15,r19			;Endezeit erreicht?
	brcs	m68420			;nein -> Nachtmodus ein
	movw	r14,r0			;aktuelle Zeit kopieren
	sub	r14,r16
	sbc	r15,r17			;Startzeit erreicht?
	brcs	m68430			;nein -> Nachtmodus aus
;
m68420:	lds	r17,sigvo2		;Nachtmodus ein, Nachtlaut-
	rjmp	m68440			;strke holen
m68430:	lds	r17,sigvo1		;Nachtmodus aus, Taglautstrke
m68440:	call	sicalc			;umrechnen und PWM einstellen
;
; LCD zum Synchronisieren der Uhr abschalten
;
m68500:	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,6			;LCD-Abschaltung fr DCF-Sync.?
	brtc	m69000			;nein -> berspringen
	lds	r16,mentio		;Men-Timeout-Zhler holen
	cpi	r16,0xff		;Zhler aktiv?
	brne	m69000			;ja -> berspringen
	ldi	r16,low(3*60)
	ldi	r17,high(3*60)		;Minutenwert fr 3:00 Uhr laden
	sub	r16,r0
	sbc	r17,r1			;Zeitpunkt erreicht?
	brne	m69000			;nein -> berspringen
	call	lcdoff			;sonst LCD ausschalten
	ldi	r16,timlcd		;Timeout-Zhler fr LCD-
	sts	lcdtio,r16		;Abschaltung neu setzen
;
; Przisen Minutenwechsel fr Log-Funktion prfen
;
m69000:	lds	r16,minutl		;aktuellen Minuten-Einer holen
	lds	r17,minfl3		;Minuten-Flag 3 holen
	inc	r17			;Minuten-Flag erhhen
	cpi	r17,10			;Flag>9?
	brcs	m69010			;nein -> weiter
	clr	r17			;sonst Flag=0 setzen
m69010:	cp	r16,r17			;erwartete Minute erreicht?
	breq	m69020			;ja -> weiter
	rjmp	m70000			;sonst berspringen
;
; Zeitdaten fr weitere Verabeitung komprimieren und im EEPROM Schreib-
; Puffer ablegen
;
m69020:	sts	minfl3,r16		;Minuten-Flag 3 neu setzen
	call	cpstim			;Zeitdaten komprimieren
;
; Zur Minute 0, 15, 30 und 45 einen Temperaturdatensatz ins externe
; EEPROM schreiben
;
	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,3			;Speicherfehler?
	brtc	m69100			;nein -> weiter
	rjmp	m70000			;sonst berspringen
m69100:	lds	r16,xewbuf		;aktuelle Zeit (Minuten) holen
	cpi	r16,0			;Minuten=0?
	breq	m69200			;ja -> bearbeiten
	cpi	r16,15			;Minuten=15?
	breq	m69200			;ja -> bearbeiten
	cpi	r16,30			;Minuten=30?
	breq	m69200			;ja -> bearbeiten
	cpi	r16,45			;Minuten=45?
	breq	m69200			;ja -> bearbeiten
	rjmp	m69300			;sonst berspringen
;
; Temperaturwerte als 12-Bit Daten in 24 Byte im externen EEPROM Puffer
; speichern
;
m69200:	ldi	yl,low(xewbuf+4)	;Zeiger auf Ext-EEPROM Schreib-
	ldi	yh,high(xewbuf+4)	;Puffer setzen
	ldi	r18,8			;8 Wertepaare mit je 3 Byte
	ldi	r19,0			;Zhler fr Log-Position
;
m69210:	call	slgmp1			;Sensornummer suchen
	ld	r16,z			;Temperaturflag holen
	bst	r16,6			;Sensorausfall?
	brts	m69220			;ja -> Wert nicht verarbeiten
	bst	r16,7			;Sensor inaktiv?
	brtc	m69230			;nein -> Wert verarbeiten
m69220:	clr	r16
	st	y+,r16			;1L in Ext-EEPROM-Puffer legen
	ldi	r17,0x08		;sonst Wert ungltig setzen
	rjmp	m69240			;H-Byte bearbeiten
;
m69230:	ld	r16,x+			;Temperaturwert 1 L holen
	st	y+,r16			;und in Ext-EEPROM-Puffer legen
	ld	r17,x			;Temperaturwert 1 H holen
	cpi	r17,0x80		;Temperaturwert gltig?
	brne	m69240			;ja -> weiter
	swap	r17			;sonst Kennung= 0x08 setzen
m69240:	andi	r17,0x0f		;nur unteres Nibble verwenden
;
	call	slgmp1			;Sensornummer suchen
	ld	r16,z			;Temperaturflag holen
	bst	r16,6			;Sensorausfall?
	brts	m69250			;ja -> Wert nicht verarbeiten
	bst	r16,7			;Sensor inaktiv?
	brtc	m69260			;nein -> Wert verarbeiten
m69250:	clr	r16
	st	y+,r16			;2L in Ext-EEPROM-Puffer legen
	ldi	r16,0x80		;sonst Wert ungltig setzen
	rjmp	m69270			;H-Byte bearbeiten
;
m69260:	ld	r16,x+			;Temperaturwert 2 L holen
	st	y+,r16			;und in Ext-EEPROM-Puffer legen
	ld	r16,x			;Temperaturwert 2 H holen
	cpi	r16,0x80		;Temperaturwert gltig?
	breq	m69270			;nein -> weiter
	swap	r16			;sonst in oberes Nibble legen
m69270:	andi	r16,0xf0		;nur oberes Nibble verwenden
	or	r16,r17			;mit Wert 1 H verknpfen
	st	y+,r16			;verknpfte H-Werte speichern
	dec	r18			;alle Werte verarbeitet?
	brne	m69210			;nein -> Schleife
;
	lds	r20,xet1po
	lds	r21,xet1po+1		;Zeiger auf aktuelle Position
	lds	r22,xet1po+2		;im externen EEPROM holen
	ldi	r23,28			;28 Bytes zum Ext-EEPROM senden
	call	xwrite			;Puffer zum Ext-EEPROM senden
	brcc	m69280			;Fehler? nein -> weiter
	call	twierr			;sonst Speicherfehler setzen
	rjmp	m69300			;Adressberechnung berspringen
;
m69280:	ldi	r16,low(xt1sta+xt1len-28)
	ldi	r17,high(xt1sta+xt1len-28);Adresse des letzten mgli-
	ldi	r18,byte3(xt1sta+xt1len-28);chen Datensatzbeginns laden
	sub	r16,r20
	sbc	r17,r21			;aktuelle Adresse noch im
	sbc	r18,r22			;gltigen Speicherbereich?
	brcc	m69290			;ja -> weiter
	ldi	r20,low(xt1sta)
	ldi	r21,high(xt1sta)	;sonst Adresse wieder auf
	ldi	r22,byte3(xt1sta)	;Anfang setzen
m69290:	sts	xet1po,r20
	sts	xet1po+1,r21		;Zeiger fr Position im
	sts	xet1po+2,r22		;externen EEPROM speichern
;
	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,4			;RTC-Fehler?
	brts	m69300			;ja -> berspringen
	ldi	r18,rtxt1p		;RTC-Adresse Temperatur-Daten 1
	ldi	xl,low(xet1po)
	ldi	xh,high(xet1po)		;Zeiger auf Temp-Daten1 Pointer
	call	i2psen			;Pointer in RTC schreiben
	brcc	m69300			;Fehler? nein -> weiter
	call	i2cerr			;sonst RTC Fehlerstatus setzen
	ldi	r19,0x40		;Meldung "RTC-Fehler" setzen
	call	wrimsg			;Fehlermeldung schreiben
;
; Zeit 23:59 Uhr prfen und Tageswerte ins Externe EEPROM schreiben
;
m69300:	lds	r16,xewbuf		;Minutenwert holen
	cpi	r16,59			;Minute=59?
	brne	m69310			;nein -> berspringen
	lds	r16,xewbuf+1		;Stundenwert holen
	cpi	r16,23			;Stunde=23?
	breq	m69320			;ja -> Tageswerte schreiben
m69310:	rjmp	m69800			;sonst berspringen
;
; Minimal-Temperaturen als 12-Bit Daten in 24 Byte im Puffer fr das
; externe EEPROM speichern (ab Position 4)
;
m69320:	ldi	yl,low(xewbuf+4)	;Zeiger auf Ext-EEPROM Schreib-
	ldi	yh,high(xewbuf+4)	;Puffer setzen (Position 4)
	ldi	r18,8			;8 Wertepaare mit je 3 Byte
	ldi	r19,0			;Zhler fr Log-Position
;
m69330:	call	slgmp2			;Sensornummer suchen
	ldi	xl,low(temin2)
	ldi	xh,high(temin2)		;Zeiger auf Temperaturwerte
	add	xl,r20			;Tabellenoffset addieren
	adc	xh,r21			;bertrag
	ld	r16,x+			;Temperaturwert 1 L holen
	st	y+,r16			;und in Ext-EEPROM-Puffer legen
	ld	r17,x			;Temperaturwert 1 H holen
	cpi	r17,0x80		;Temperaturwert gltig?
	brne	m69340			;nein -> weiter
	swap	r17			;sonst Kennung= 0x08 setzen
m69340:	andi	r17,0x0f		;nur unteres Nibble verwenden
;
	call	slgmp2			;Sensornummer suchen
	ldi	xl,low(temin2)
	ldi	xh,high(temin2)		;Zeiger auf Temperaturwerte
	add	xl,r20			;Tabellenoffset addieren
	adc	xh,r21			;bertrag
	ld	r16,x+			;Temperaturwert 2 L holen
	st	y+,r16			;und in Ext-EEPROM-Puffer legen
	ld	r16,x			;Temperaturwert 2 H holen
	cpi	r16,0x80		;Temperaturwert gltig?
	breq	m69350			;ja -> weiter
	swap	r16			;sonst in oberes Nibble legen
m69350:	andi	r16,0xf0		;nur oberes Nibble verwenden
	or	r16,r17			;mit Wert 1 H verknpfen
	st	y+,r16			;verknpfte H-Werte speichern
	dec	r18			;alle Werte verarbeitet?
	brne	m69330			;nein -> Schleife
;
; Maximal-Temperaturen als 12-Bit Daten in 24 Byte im Puffer fr das
; externe EEPROM speichern (ab Position 28)
;
	ldi	yl,low(xewbuf+4+24)	;Zeiger auf Ext-EEPROM Schreib-
	ldi	yh,high(xewbuf+4+24)	;Puffer setzen (Position 28)
	ldi	r18,8			;8 Wertepaare mit je 3 Byte
	ldi	r19,0			;Zhler fr Log-Position
;
m69400:	call	slgmp2			;Sensornummer suchen
	ldi	xl,low(temax2)
	ldi	xh,high(temax2)		;Zeiger auf Temperaturwerte
	add	xl,r20			;Tabellenoffset addieren
	adc	xh,r21			;bertrag
	ld	r16,x+			;Temperaturwert 1 L holen
	st	y+,r16			;und in Ext-EEPROM-Puffer legen
	ld	r17,x			;Temperaturwert 1 H holen
	cpi	r17,0x80		;Temperaturwert gltig?
	brne	m69410			;nein -> weiter
	swap	r17			;sonst Kennung= 0x08 setzen
m69410:	andi	r17,0x0f		;nur unteres Nibble verwenden
;
	call	slgmp2			;Sensornummer suchen
	ldi	xl,low(temax2)
	ldi	xh,high(temax2)		;Zeiger auf Temperaturwerte
	add	xl,r20			;Tabellenoffset addieren
	adc	xh,r21			;bertrag
	ld	r16,x+			;Temperaturwert 2 L holen
	st	y+,r16			;und in Ext-EEPROM-Puffer legen
	ld	r16,x			;Temperaturwert 2 H holen
	cpi	r16,0x80		;Temperaturwert gltig?
	breq	m69420			;ja -> weiter
	swap	r16			;sonst in oberes Nibble legen
m69420:	andi	r16,0xf0		;nur oberes Nibble verwenden
	or	r16,r17			;mit Wert 1 H verknpfen
	st	y+,r16			;verknpfte H-Werte speichern
	dec	r18			;alle Werte verarbeitet?
	brne	m69400			;nein -> Schleife
;
; Mittelwerte berechnen und Ergebnisse als 12-Bit Daten in 24 Byte im
; Puffer fr das externe EEPROM speichern (ab Position 52)
;
	ldi	zl,low(xewbuf+4+48)	;Zeiger auf Ext-EEPROM Schreib-
	ldi	zh,high(xewbuf+4+48)	;Puffer setzen (Position 52)
	ldi	r24,8			;8 Wertepaare berechnen
	clr	r8			;Zhler fr Log-Position
;
m69500:	ldi	r23,2			;je 2 Mittelwerte berechnen
m69510:	ldi	xl,low(logmap)
	ldi	xh,high(logmap)		;Zeiger auf Log-Map
	clr	r10			;Suchzhler fr Log-Position
m69520:	ld	r11,x+			;Tabelleneintrag holen
	cp	r8,r11			;Log-Position gefunden?
	breq	m69530			;ja -> Ende
	inc	r10			;Suchzhler erhhen
	mov	r16,r10			;Suchzhler kopieren
	cpi	r16,16			;letzte Position erreicht?
	brcs	m69520			;nein -> Schleife
m69530:	inc	r8			;nchste Log-Position
	clr	r11			;H-Byte fr Tabellenoffset
	mov	r9,r10			;Offset kopieren
	add	r10,r9			;Offset verdoppeln
;
	ldi	yl,low(temavc)
	ldi	yh,high(temavc)		;Zeiger auf Mittelwertzhler
	add	yl,r10			;Tabellenoffset addieren
	adc	yh,r11			;bertrag
	ldi	xl,low(temavs)
	ldi	xh,high(temavs)		;Zeiger auf Mittelwertsummen
	add	r10,r9			;Offset verdreifachen
	add	xl,r10			;Tabellenoffset addieren
	adc	xh,r11			;bertrag
;
	ld	r16,x+
	ld	r17,x+
	ld	r18,x			;Mittelwertsumme holen
	bst	r18,7			;Wert negativ?
	brtc	m69540			;nein -> weiter
	clr	r19
	clr	r20
	clr	r21			;sonst Zweierkomplement bilden
	sub	r19,r16
	sbc	r20,r17
	sbc	r21,r18			;Wert = 0 - Wert
	mov	r16,r19
	mov	r17,r20
	mov	r18,r21			;Ergebnis kopieren
m69540:	lsl	r16
	rol	r17			;Wert verdoppeln zur Erhhung
	rol	r18			;der Genauigkeit
	ld	r20,y+
	ld	r21,y			;Mittelwertzhler holen
	mov	r22,r20			;L kopieren
	or	r22,r21			;H verknpfen
	brne	m69550			;Zhler=0? nein -> dividieren
	clr	r16
	ldi	r17,0x80		;sonst Wert=0x8000 setzen
	rjmp	m69570
;
m69550:	call	divide			;Mittelwert berechnen
	lsr	r17
	ror	r16			;Wert halbieren
	brcc	m69560			;Nachkommastelle 0? n -> weiter
	inc	r16			;sonst Ergebnis aufrunden
	brne	m69560			;bertrag? nein -> weiter
	inc	r17			;sonst H-Byte erhhen
m69560:	brtc	m69570			;war Mittelwertsumme negativ?
	clr	r18			;nein -> weiter
	clr	r19			;sonst Zweierkomplement bilden
	sub	r18,r16
	sbc	r19,r17			;Wert = 0-Wert
	movw	r16,r18			;Ergebnis kopieren
m69570:	cpi	r23,2			;erster Wert?
	brne	m69580			;nein -> weiter
	movw	r12,r16			;sonst Wert sichern
m69580:	dec	r23			;alle Werte berechnet?
	breq	m69590			;ja -> weiter
	rjmp	m69510			;sonst Schleife
;
m69590:	movw	r18,r12			;ersten Wert holen
	st	z+,r18			;Temp 1 L in Ext-EEPROM-Puffer
	cpi	r19,0x80		;Temperaturwert 1 gltig?
	brne	m69600			;ja -> weiter
	swap	r19			;sonst Kennung= 0x08 setzen
m69600:	andi	r19,0x0f		;nur unteres Nibble verwenden
	st	z+,r16			;Temp 2 L in Ext-EEPROM-Puffer
	cpi	r17,0x80		;Temperaturwert 2 gltig?
	breq	m69610			;nein -> weiter
	swap	r17			;sonst in oberes Nibble legen
m69610:	andi	r17,0xf0		;nur oberes Nibble verwenden
	or	r17,r19			;mit Wert 1 H verknpfen
	st	z+,r17			;verknpfte H-Werte speichern
	dec	r24			;alle Werte verarbeitet?
	breq	m69620			;ja -> weiter
	rjmp	m69500			;sonst Schleife
;
m69620:	lds	r20,xet2po
	lds	r21,xet2po+1		;Zeiger auf aktuelle Position
	lds	r22,xet2po+2		;im externen EEPROM holen
	ldi	r23,76			;76 Bytes zum Ext-EEPROM senden
	call	xwrite			;Puffer zum Ext-EEPROM senden
	brcc	m69630			;Fehler? nein -> weiter
	call	twierr			;sonst Speicherfehler setzen
	rjmp	m69700			;Adressberechnung berspringen
;
m69630:	ldi	r16,low(xt2sta+xt2len-76)
	ldi	r17,high(xt2sta+xt2len-76);Adresse des letzten mgli-
	ldi	r18,byte3(xt2sta+xt2len-76);chen Datensatzbeginns laden
	sub	r16,r20
	sbc	r17,r21			;aktuelle Adresse noch im
	sbc	r18,r22			;gltigen Speicherbereich?
	brcc	m69640			;ja -> weiter
	ldi	r20,low(xt2sta)
	ldi	r21,high(xt2sta)	;sonst Adresse wieder auf
	ldi	r22,byte3(xt2sta)	;Anfang setzen
m69640:	sts	xet2po,r20
	sts	xet2po+1,r21		;Zeiger fr Position im
	sts	xet2po+2,r22		;externen EEPROM speichern
;
	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,4			;RTC-Fehler?
	brts	m69700			;ja -> berspringen
	ldi	r18,rtxt2p		;RTC-Adresse Temperatur-Daten 2
	ldi	xl,low(xet2po)
	ldi	xh,high(xet2po)		;Zeiger auf Temp-Daten2 Pointer
	call	i2psen			;Pointer in RTC schreiben
	brcc	m69700			;Fehler? nein -> weiter
	call	i2cerr			;sonst RTC Fehlerstatus setzen
	ldi	r19,0x40		;Meldung "RTC-Fehler" setzen
	call	wrimsg			;Fehlermeldung schreiben
;
m69700:	call	rsmin2			;Minimalwerte (Log) lschen
	call	rsmax2			;Maximalwerte (Log) lschen
	call	resavg			;Mittelwerte (Log) lschen
;
; Zur Minute 14, 29, 44 und 59 (auer 23:59) ein Backup der Min-, Max-
; und Mittelwerte in das RTC-RAM schreiben
;
m69800:	lds	r16,xewbuf		;aktuelle Zeit (Minuten) holen
	cpi	r16,14			;Minuten=14?
	breq	m69810			;ja -> bearbeiten
	cpi	r16,29			;Minuten=29?
	breq	m69810			;ja -> bearbeiten
	cpi	r16,44			;Minuten=44?
	breq	m69810			;ja -> bearbeiten
	cpi	r16,59			;Minuten=59?
	brne	m70000			;nein -> berspringen
	lds	r16,xewbuf+1		;aktuelle Zeit (Stunden) holen
	cpi	r16,23			;Stunde=23 (Zeit 23:59)?
	breq	m70000			;ja -> berspringren
;
m69810:	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,4			;RTC-Fehler?
	brts	m70000			;ja -> berspringen
	call	i2back			;Werte im RTC-RAM sichern
	brcc	m70000			;Fehler? nein -> weiter
	call	i2cerr			;sonst RTC-Fehlerstatus setzen
	ldi	r19,0x40		;Meldung "RTC-Fehler" setzen
	call	wrimsg			;Fehlermeldung schreiben
;
;
; Mensteuerung: Auswertung der Taster
;
m70000:	lds	r16,keyst2		;Tastenstatus holen
	lds	r17,keyst3		;Quittierungsstatus holen
	bst	r16,6			;wurde Taster 1 gedrckt?
	brtc	m70010			;nein -> weiter testen
	bst	r17,6			;wurde Taster 1 quittiert?
	brts	m70010			;ja -> Taster 1 ignorieren
	rjmp	m72000			;nein -> Taster 1 bearbeiten
m70010:	bst	r16,7			;wurde Taster 2 gedrckt?
	brtc	m70020			;nein -> weiter testen
	bst	r17,7			;wurde Taster 2 quittiert?
	brts	m70020			;ja -> Taster 2 ignorieren
	rjmp	m73000			;sonst Taster 2 bearbeiten
m70020:	lds	r16,encpos		;aktuelle Encoder-Position
	lds	r17,encmem		;gespeicherte Encoder-Position
	cp	r16,r17			;hat sich der Wert gendert?
	brne	m71000			;ja -> bearbeiten
	jmp	m99400
;
; Mensteuerung: Bearbeiten einer Encoder-Vernderung (Mentext)
;
m71000:	lds	r18,dflags		;sonst Anzeige-Flags holen
	bst	r18,7			;LCD wegen Sync. abgeschaltet?
	brtc	m71010			;nein -> weiter
	sts	encpos,r17		;aktuelle Encoder-Pos. lschen
	call	lcdon			;LCD wieder einschalten
	jmp	m99400			;Ende
;
m71010:	sts	encmem,r16		;neue Encoder-Position speich.
	sub	r16,r17			;Anzahl der Schritte ermitteln
	lds	r18,encflg		;Encoder-Flags holen
	bst	r18,0			;Encoderwert im Men anzeigen?
	brts	m71100			;ja -> bearbeiten
	lds	r10,menopt		;aktuelle Menoption holen
	add	r10,r16			;Schrittanzahl addieren
	tst	r10			;ermittelte Menoption<0?
	brpl	m71020			;nein -> weiter testen
	clr	r10			;sonst Menuoption 0 setzen
	rjmp	m71030			;neue Option speichern
;
m71020:	call	men1rd			;Mentabellenheader lesen
	cp	r10,r11			;Option>Maximalwert?
	brcs	m71030			;nein -> neue Option speichern
	mov	r10,r11			;sonst auf letzte Option setzen
	dec	r10			;und korrigieren
m71030:	lds	r16,menopt		;bisherige Menoption holen
	cp	r10,r16			;nderung?
	breq	m71050			;nein -> weiter
	sts	menopt,r10		;sonst neue Option speichern
m71040:	lds	r18,dflags		;Anzeigeflags holen
	sbr	r18,0x08		;Menzeile neu ausgeben
	sts	dflags,r18		;Anzeigeflags wieder speichern
m71050:	jmp	m99300
;
; Mensteuerung: Bearbeiten einer Encoder-Vernderung (3-stellig dez.)
;
m71100:	sbr	r18,1<<7		;nderungs-Flag setzen
	sts	encflg,r18		;Encoder-Flags wieder speichern
	bst	r18,1			;Anzeige 3-stellig dezimal?
	brtc	m71200			;nein -> weiter testen
	lds	r17,encval		;Encoderwert holen
	add	r16,r17			;Schrittanzahl addieren
	cpi	r16,225			;Wert>225 (negativer berlauf)?
	brcs	m71110			;nein -> weiter
	clr	r16			;sonst Wert=0 setzen
m71110:	cpi	r16,200			;Wert>200 (positiver berlauf)?
	brcs	m71120			;nein -> weiter
	ldi	r16,199			;sonst Wert=199 setzen
m71120:	lds	r17,encmax		;Encoder Maximalwert holen
	cp	r17,r16			;Maximalwert berschritten?
	brcc	m71130			;nein -> weiter
	mov	r16,r17			;sonst auf Maximalwert setzen
m71130:	lds	r17,encmin		;Encoder Minimalwert holen
	cp	r16,r17			;Minimalwert unterschritten?
	brcc	m71140			;nein -> weiter
	mov	r16,r17			;sonst auf Minimalwert setzen
m71140:	sts	encval,r16		;neuen Encoderwert speichern
	rjmp	m71040			;Menzeile neu ausgeben
;
; Mensteuerung: Bearbeiten einer Encoder-Vernderung (Temperaturwert)
;
m71200:	bst	r18,2			;Anzeige Temperaturwert?
	brtc	m71300			;nein -> weiter testen
	mov	r18,r16			;Encoder-Schrittanzahl kopieren
	tst	r18			;Wert negativ?
	brpl	m71210			;nein -> weiter
	neg	r18			;sonst in posit. Wert wandeln
m71210:	cpi	r18,2			;Schrittwert<2?
	brcs	m71220			;ja -> weiter
	dec	r18			;sonst um 1 vermindern und
m71220:	muls	r16,r18			;neue Schrittanzahl berechnen
;
	lds	r18,encval		;Encoderwert L holen
	lds	r19,encvah		;Encoderwert H holen
	add	r18,r0			;Schrittanzahl addieren
	adc	r19,r1			;bertrag addieren
	ldi	r16,low(maxtmp)
	ldi	r17,high(maxtmp)	;Temperaturmaximalwert laden
	sub	r16,r18
	sbc	r17,r19			;Maximalwert berschritten?
	brpl	m71230			;nein -> weiter
	ldi	r18,low(maxtmp)		;sonst
	ldi	r19,high(maxtmp)	;Temperaturmaximalwert laden
m71230:	movw	r20,r18			;Encoderwert sichern
	ldi	r16,low(mintmp)
	ldi	r17,high(mintmp)	;Temperaturminimalwert laden
	sub	r18,r16
	sbc	r19,r17			;Minimalwert unterschritten?
	movw	r18,r20			;Encoderwert restaurieren
	brpl	m71240			;nein -> weiter
	ldi	r18,low(mintmp)		;sonst
	ldi	r19,high(mintmp)	;Temperaturminimalwert laden
m71240:	sts	encval,r18		;neuen Encoderwert L speichern
	sts	encvah,r19		;neuen Encoderwert H speichern
	rjmp	m71040			;Menzeile neu ausgeben
;
; Mensteuerung: Bearbeiten einer Encoder-Vernderung (ASCII-Zeichen)
;
m71300:	bst	r18,3			;Anzeige ASCII-Zeichen?
	brtc	m71400			;nein -> weiter testen
	mov	r18,r16			;Schrittanzahl sichern
	lds	r17,encval		;Encoderwert holen
	add	r16,r17			;Schrittanzahl addieren
	ldi	r17,maxasc+2		;letztes Zeichen (+2) laden
	cp	r17,r16			;letztes Zeichen berschritten?
	brcc	m71310			;nein -> weiter
	mov	r16,r17			;sonst letztes Zeichen setzen
m71310:	cpi	r16,minasc		;erstes Zeichen unterschritten?
	brcc	m71320			;nein -> weiter
	ldi	r16,minasc		;sonst erstes Zeichen setzen
m71320:	cpi	r16,0x7f		;Ende-Zeichen?
	brne	m71340			;nein -> Zeichen speichern
	tst	r18			;Schrittanzahl negativ?
	brmi	m71330			;ja -> bearbeiten
	inc	r16			;sonst nchstes Zeichen
	rjmp	m71340			;Zeichen speichern
m71330:	dec	r16			;vorheriges Zeichen
m71340:	sts	encval,r16		;neuen Encoderwert speichern
	rjmp	m71040			;Menzeile neu ausgeben
;
; Mensteuerung: Bearbeiten einer Encoder-Vernderung (4-stellig dez.)
;
m71400:	bst	r18,4			;Anzeige 4-stellig dezimal?
	brtc	m71500			;nein -> weiter testen
	mov	r18,r16			;Encoder-Schrittanzahl kopieren
	tst	r18			;Wert negativ?
	brpl	m71410			;nein -> weiter
	neg	r18			;sonst in posit. Wert wandeln
m71410:	ldi	r17,1			;Inc/Decrement auf 1 setzen
	cpi	r18,3			;mehr als 2 Schritte?
	brcs	m71420			;nein -> berspringen, sonst
	ldi	r17,8			;Inc/Decrement auf 8 setzen
	cpi	r18,5			;mehr als 4 Schritte?
	brcs	m71420			;nein -> berspringen, sonst
	ldi	r17,20			;Inc/Decrement auf 20 setzen
m71420:	muls	r16,r17			;mit Schrittanzahl multipliz.
;
	lds	r18,encval		;Encoderwert L holen
	lds	r19,encvah		;Encoderwert H holen
	add	r18,r0			;Schrittanzahl addieren
	adc	r19,r1			;bertrag addieren
	lds	r16,encmax
	lds	r17,encmah		;Maximalwert laden
	sub	r16,r18
	sbc	r17,r19			;Maximalwert berschritten?
	brpl	m71430			;nein -> weiter
	lds	r18,encmax		;sonst
	lds	r19,encmah		;Maximalwert laden
m71430:	movw	r20,r18			;Encoderwert sichern
	lds	r16,encmin
	lds	r17,encmih		;Minimalwert laden
	sub	r18,r16
	sbc	r19,r17			;Minimalwert unterschritten?
	movw	r18,r20			;Encoderwert restaurieren
	brpl	m71440			;nein -> weiter
	lds	r18,encmin		;sonst
	lds	r19,encmih		;Temperaturminimalwert laden
m71440:	sts	encval,r18		;neuen Encoderwert L speichern
	sts	encvah,r19		;neuen Encoderwert H speichern
	rjmp	m71040			;Menzeile neu ausgeben
;
; Mensteuerung: Bearbeiten einer Encoder-Vernderung (Zeitwert)
;
m71500:	bst	r18,5			;Anzeige Temperaturwert?
	brtc	m71600			;nein -> weiter testen
	mov	r18,r16			;Encoder-Schrittanzahl kopieren
	tst	r18			;Wert negativ?
	brpl	m71510			;nein -> weiter
	neg	r18			;sonst in posit. Wert wandeln
m71510:	ldi	r17,1			;Inc/Decrement auf 1 setzen
	cpi	r18,3			;mehr als 2 Schritte?
	brcs	m71520			;nein -> berspringen, sonst
	ldi	r17,8			;Inc/Decrement auf 8 setzen
m71520:	muls	r16,r17			;mit Schrittanzahl multipliz.
;
	lds	r18,encval		;Encoderwert L holen
	lds	r19,encvah		;Encoderwert H holen
	add	r18,r0			;Schrittanzahl addieren
	adc	r19,r1			;bertrag addieren
	ldi	r16,low(23*60+59)
	ldi	r17,high(23*60+59)	;Maximalwert (23:59) laden
	sub	r16,r18
	sbc	r17,r19			;Maximalwert berschritten?
	brpl	m71530			;nein -> weiter
	clr	r18			;sonst
	clr	r19			;wieder bei 0 beginnen
m71530:	movw	r20,r18			;Encoderwert sichern
	clr	r16
	clr	r17			;Minimalwert (0) laden
	sub	r18,r16
	sbc	r19,r17			;Minimalwert unterschritten?
	movw	r18,r20			;Encoderwert restaurieren
	brpl	m71540			;nein -> weiter
	ldi	r18,low(23*60+59)	;sonst wieder beim
	ldi	r19,high(23*60+59)	;Maximalwert (23:59) beginnen
m71540:	sts	encval,r18		;neuen Encoderwert L speichern
	sts	encvah,r19		;neuen Encoderwert H speichern
m71600:	rjmp	m71040			;Menzeile neu ausgeben
;
; Mensteuerung: Bearbeiten der Bettigung von Taster 1 (rechts)
;
m72000:	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,7			;LCD wegen Sync. abgeschaltet?
	brtc	m72010			;nein -> weiter
	call	lcdon			;sonst LCD wieder einschalten
	jmp	m99400			;Ende
;
m72010:	cli				;Interrupts sperren
	lds	r16,keyst3		;Tasterquittungs-Flags holen
	sbr	r16,0x40		;Taster 1 quittieren
	sts	keyst3,r16		;Flags wieder speichern
	sei				;Interrupts wieder freigeben
	clr	r16			;zunchst
	sts	menext,r16		;Menerweiterung lschen
	call	men1rd			;Mentabellenheader lesen
	call	men2rd			;Mentabellenoption lesen
	elpm	r16,z+			;Infotextstring lesen (dummy)
	elpm	r16,z+			;Untermen-Nummer lesen
	cpi	r16,0xff		;gltiges Untermen?
	breq	m72020			;nein -> Men nicht ndern
	sts	mennum,r16		;neue Mennummer setzen
	clr	r16
	sts	menopt,r16		;neue Menoption 0 setzen
m72020:	elpm	r16,z+			;Programmadresse L holen
	elpm	r17,z			;Programmadresse H holen
	tst	r17			;gltige Programmadresse?
	brne	m72030			;ja -> Programm aufrufen
	lds	r16,dflags		;sonst Anzeigeflags holen
	sbr	r16,0x0c		;Info/Menzeile neu ausgeben
	sts	dflags,r16		;Anzeigeflags wieder speichern
	jmp	m99300
m72030:	movw	zl,r16			;Adresse kopieren
	ijmp				;Menprogramm starten
;
; Mensteuerung: Bearbeiten der Bettigung von Taster 2 (links)
;
m73000:	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,7			;LCD wegen Sync. abgeschaltet?
	brtc	m73010			;nein -> weiter
	call	lcdon			;sonst LCD wieder einschalten
	jmp	m99400			;Ende
;
m73010:	cli				;Interrupts sperren
	lds	r16,keyst3		;Tasterquittungs-Flags holen
	sbr	r16,0x80		;Taster 2 quittieren
	sts	keyst3,r16		;Flags wieder speichern
	sei				;Interrupts wieder freigeben
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert anzeigen?
	brts	m73120			;ja -> bearbeiten
;
m73020:	lds	r16,mennum		;aktuelle Mennummer holen
	tst	r16			;Hauptmen (0)?
	breq	m73100			;ja -> bearbeiten
	call	men1rd			;Mentabellenheader lesen
	elpm	r16,z+			;bergeordnete Mennummer holen
	elpm	r17,z			;bergeordnete Menoption holen
	bst	r17,7			;Neuausgabe-Flag sichern
	andi	r17,0x7f		;bergeordn. Menoption filtern
	tst	r17			;Menoption=0?
	breq	m73030			;ja -> weiter
	dec	r17			;sonst korrigieren
m73030:	sts	mennum,r16		;neue Mennummer setzen
	sts	menopt,r17		;neue Menoption setzen
	lds	r18,dflags		;Anzeigeflags holen
	sbr	r18,0x0c		;Info/Menzeile neu ausgeben
	brtc	m73040			;Neuausgabe? nein -> weiter
	sbr	r18,0x03		;sonst Temp-Anzeige neu ausgeb.
m73040:	sts	dflags,r18		;Anzeigeflags wieder speichern
	lds	r16,xflags		;verschiedene Flags holen
	cbr	r16,0x0c		;Min/Max-Anzeige-Flag lschen
	sts	xflags,r16		;Flags wieder speichern
m73050:	clr	r16
	sts	menext,r16		;Menerweiterung lschen
	jmp	m99300
;
m73100:	lds	r16,xflags		;verschiedene Flags holen
	bst	r16,0			;sind Signale aktiv?
	brtc	m73105			;nein -> weiter
	rjmp	m73200			;sonst Tonausgabe stoppen
;
m73105:	bst	r16,1			;sind Erinnerungen gesetzt?
	brts	m73110			;ja -> Erinnerungen lschen
	jmp	m99300			;sonst Ende
m73110:	rjmp	m73300			;Erinnerungen lschen
;
m73120:	clr	r16			;Encoder-Flags zurcksetzen
	sts	encflg,r16		;und speichern
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	lds	r18,dflags		;Anzeigeflags holen
	sbr	r18,0x0c		;Info/Menzeile neu ausgeben
	sts	dflags,r18		;Anzeigeflags wieder speichern
	lds	r16,dimode		;Anzeigemodus holen
	cpi	r16,4			;Modus "Temp/Pos/Verschieben"?
	brne	m73130			;nein -> weiter testen
	call	cmapbk			;sonst Mapping-Tabelle kopieren
	call	newtem			;und Daten neu ausgeben
	rjmp	m73050
;
m73130:	cpi	r16,5			;Modus "Temp/Pos/Namen"?
	brne	m73140			;nein -> weiter testen
	ldi	r16,3			;sonst
	sts	dimode,r16		;Anzeigemodus 3 setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	call	newtem			;Daten neu ausgeben
	rjmp	m73050
;
m73140:	cpi	r16,7			;Modus "Temp/Grenzw./ndern"?
	brne	m73150			;nein -> weiter testen
	ldi	r16,6			;sonst
	sts	dimode,r16		;Anzeigemodus 6 setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	lds	r16,xflag4		;verschiedene Flags 4 holen
	cbr	r16,3			;Flags fr "UG/OG ndern" lsch
	sts	xflag4,r16		;Flags wieder speichern
	call	newtem			;Daten neu ausgeben
	rjmp	m73050
;
m73150:	cpi	r16,9			;Modus "Alarme/Verschieben"?
	brne	m73160			;nein -> weiter testen
	call	calmbk			;sonst Mapping-Tabelle kopieren
	call	newalm			;und Daten neu ausgeben
	rjmp	m73050
;
m73160:	cpi	r16,10			;Modus "Alarme/Namen"?
	brne	m73170			;nein -> weiter testen
	ldi	r16,8			;sonst
	sts	dimode,r16		;Anzeigemodus 8 setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	call	newalm			;Daten neu ausgeben
	rjmp	m73050
;
m73170:	cpi	r16,12			;Modus "Signale/Zuordnung/nd"?
	brne	m73180			;nein -> weiter testen
	ldi	r16,11			;sonst
	sts	dimode,r16		;Anzeigemodus 11 setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	lds	r16,xflag4		;verschiedene Flags 4 holen
	cbr	r16,1<<2|1<<3		;Flags "Nr/Dauer ndern" lsch.
	sts	xflag4,r16		;Flags wieder speichern
	call	newcf2			;Daten neu ausgeben
	rjmp	m73050
;
m73180:	cpi	r16,14			;Modus "Signale/Audio/ndern"?
	brne	m73190			;nein -> weiter testen
	ldi	r16,13			;sonst
	sts	dimode,r16		;Anzeigemodus 13 setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	call	newcf2			;Daten neu ausgeben
	rjmp	m73050
;
m73190:	cpi	r16,16			;Modus "Zeiten/Schalt./ndern"?
	brne	m731a0			;nein -> weiter testen
	ldi	r16,15			;sonst
	sts	dimode,r16		;Anzeigemodus 15 setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	call	newcf2			;Daten neu ausgeben
	rjmp	m73050
;
m731a0:	cpi	r16,18			;Modus "Zeiten/Uhr/ndern"?
	brne	m731b0			;nein -> weiter testen
	ldi	r16,17			;sonst
	sts	dimode,r16		;Anzeigemodus 17 setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	call	newcf2			;Daten neu ausgeben
	rjmp	m73050
;
m731b0:	cpi	r16,20			;Modus "Sonstiges/ndern"?
	brne	m731c0			;nein -> weiter testen
	ldi	r16,19			;sonst
	sts	dimode,r16		;Anzeigemodus 19 setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	call	newcf2			;Daten neu ausgeben
	rjmp	m73050
;
m731c0:	cpi	r16,24			;Modus "Grafik/Fadenkreuz"?
	brne	m731d0			;nein -> weiter testen
	ldi	r16,23			;sonst
	sts	dimode,r16		;Anzeigemodus 23 setzen
	lds	r18,grpoin		;aktuelle Fadenkreuzpos. holen
	call	grscrr			;Grafik wieder herstellen
	rjmp	m73020			;eine Menebene zurck
;
m731d0:	cpi	r16,26			;Modus "Konfig/Grafik/ndern"?
	brne	m731e0			;nein -> weiter testen
	ldi	r16,25			;sonst
	sts	dimode,r16		;Anzeigemodus 25 setzen
	call	newcf2			;Datenfelder neu ausgeben
m731e0:	rjmp	m73050			;Menerweiterung lschen
;
; Mensteuerung Taster 2: Signal ausschalten
;
m73200:	lds	r16,dflags		;Anzeigeflags holen
	cbr	r16,0x20		;Lautsprecher abschalten
	sbr	r16,0x04		;Infozeile neu ausgeben
	sts	dflags,r16		;Anzeigeflags wieder speichern
	ldi	r16,timlsp		;Timeout-Wert Lsp.-Abschaltung
	sts	lsptio,r16		;neu setzen
	jmp	m99300
;
; Mensteuerung Taster 2: Erinnerungen lschen
;
m73300:	lds	r16,errflg		;Fehler-Flags holen
	cbr	r16,0x80		;Erinnerung lschen
	sts	errflg,r16		;Fehler-Flags wieder speichern
;
	ldi	r17,16			;16 Temperaturflags bearbeiten
	ldi	xl,low(temflg)
	ldi	xh,high(temflg)		;Zeiger auf Temperaturflags
m73310:	ld	r16,x			;Flags holen
	bst	r16,7			;Sensor inaktiv?
	brts	m73320			;ja -> berspringen
	andi	r16,0x30		;Erinnerungen gesetzt?
	breq	m73320			;nein -> weiter
	ld	r16,x			;sonst Flags nochmals holen
	cbr	r16,0x30		;Erinnerungen lschen
	sbr	r16,0x03		;Name und Wert neu ausgeben
	st	x,r16			;Flags wieder speichern
m73320:	adiw	xl,1			;Zeiger auf nchsten Sensor
	dec	r17			;alle Sensoren bearbeitet?
	brne	m73310			;nein -> Schleife
;
	ldi	r17,4			;4 Alarmflags bearbeiten
	ldi	xl,low(almflg)
	ldi	xh,high(almflg)		;Zeiger auf Alarmflags
m73330:	ld	r16,x			;Flags holen
	bst	r16,7			;Alarmeingang inaktiv?
	brts	m73340			;ja -> berspringen
	bst	r16,2			;Erinnerung gesetzt?
	brtc	m73340			;nein -> weiter
	cbr	r16,0x04		;Erinnerung lschen
	sbr	r16,0x01		;Alarmname neu ausgeben
m73340:	st	x+,r16			;Flags wieder speichern
	dec	r17			;alle Alarme bearbeitet?
	brne	m73330			;nein -> Schleife
;
	lds	r16,ledflg		;LED-Flags holen
	cbr	r16,0x30		;Erinnerungen lschen
	sts	ledflg,r16		;LED-Flags wieder speichern
	lds	r16,xflags		;diverse Flags holen
	cbr	r16,0x02		;allg. Erinnerungsflag lschen
	sts	xflags,r16		;Flags wieder speichern
	lds	r16,dflags		;Anzeigeflags holen
	sbr	r16,0x0c		;Info/Menzeile neu ausgeben
	sts	dflags,r16		;Anzeigeflags wieder speichern
	jmp	m99300
;
; Menprogramm: Anzeige der Minimalwerte
;
m80000:	lds	r16,xflags		;verschiedene Flags holen
	sbr	r16,0x04		;Flag fr Minimalanzeige setzen
	sts	xflags,r16		;Flags wieder speichern
	jmp	m99000			;Temperaturanzeige neu ausgeben
;
m80100:	call	rsmin1			;Minimalwerte zurcksetzen
	call	okbeep			;Tonsignal ausgeben
	jmp	m99200			;Temperaturanzeige neu ausgeben
;
; Menprogramm: Anzeige der Maximalwerte
;
m80200:	lds	r16,xflags		;verschiedene Flags holen
	sbr	r16,0x08		;Flag fr Maximalanzeige setzen
	sts	xflags,r16		;Flags wieder speichern
	jmp	m99000			;Temperaturanzeige neu ausgeben
;
m80300:	call	rsmax1			;Maximalwerte zurcksetzen
	call	okbeep			;Tonsignal ausgeben
	jmp	m99200			;Temperaturanzeige neu ausgeben
;
; Menprogramm: Anzeige der Meldungen, aktuelle zuerst zeigen
;
m81000:	lds	r16,xemspo
	lds	r17,xemspo+1		;aktuelle Meldungsposition
	lds	r18,xemspo+2		;holen
	sts	msgpo1,r16
	sts	msgpo1+1,r17		;und als Zeiger fr die Anzeige
	sts	msgpo1+2,r18		;speichern
	sts	msgpo2,r16
	sts	msgpo2+1,r17		;zustzlich als Zeiger fr die
	sts	msgpo2+2,r18		;vorherige Seite speichern
	lds	r16,dflags		;Anzeige-Flags holen
	cbr	r16,0x01		;Temperaturanzeige ausschalten
	sts	dflags,r16		;Flags wieder speichern
	ldi	r16,1			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Meldungsanzeige setzen
	lds	r16,xflag2		;Verschiedene Flags 2 holen
	cbr	r16,1<<5|1<<6		;nchste Seite, kein Listenende
	sbr	r16,1<<7		;Listenende fr vorherige Seite
m81010:	sts	xflag2,r16		;Flags wieder speichern
	lds	r16,msgfil		;Meldungsfilter holen
	ldi	r17,0x2a		;Stringnummer fr Filtertext
	add	r16,r17			;Stringnummer berechnen
	sts	menext,r16		;Menerweiterung setzen
	jmp	m99100			;Ende, Anzeige neu ausgeben
;
; Menprogramm: Anzeige der Meldungen, nchste Seite anzeigen
;
m81100:	lds	r16,xflag2		;Verschiedene Flags 2 holen
	bst	r16,5			;Richtungsnderung?
	brts	m81120			;ja -> weiter
	bst	r16,6			;Listenende erreicht?
	brtc	m81120			;nein -> weiter
m81110:	lds	r16,msgfil		;sonst Meldungsfilter holen
	ldi	r17,0x2a		;Stringnummer fr Filtertext
	add	r16,r17			;Stringnummer berechnen
	sts	menext,r16		;Menerweiterung setzen
	jmp	m99300			;Ende ohne Neuausgabe
m81120:	lds	r20,msgpo1
	lds	r21,msgpo1+1		;aktuelle Meldungsposition
	lds	r22,msgpo1+2		;fr nchste Seite holen
	sts	msgpo2,r20
	sts	msgpo2+1,r21		;und als Position fr die vor-
	sts	msgpo2+2,r22		;herige Seite speichern
	cbr	r16,1<<5|1<<6		;nchste Seite, kein Listenende
	rjmp	m81010			;Flags speichern, Anzeige ausg.
;
; Menprogramm: Anzeige der Meldungen, vorherige Seite anzeigen
;
m81200:	lds	r16,xflag2		;Verschiedene Flags 2 holen
	bst	r16,5			;Richtungsnderung?
	brtc	m81210			;ja -> weiter
	bst	r16,7			;Listenende erreicht?
	brts	m81110			;ja -> Ende ohne Neuausgabe
m81210:	lds	r20,msgpo2
	lds	r21,msgpo2+1		;aktuelle Meldungsposition
	lds	r22,msgpo2+2		;fr vorherige Seite holen
	sts	msgpo1,r20
	sts	msgpo1+1,r21		;und als Position fr die
	sts	msgpo1+2,r22		;nchste Seite speichern
	sbr	r16,1<<5		;Seite zurckblttern
	cbr	r16,1<<7		;Listenende-Flags lschen
	rjmp	m81010			;Flags speichern, Anzeige ausg.
;
; Menprogramm: Anzeige der Meldungen, Filter umschalten
;
m81300:	lds	r16,msgfil		;Meldungsfilter holen
	inc	r16			;nchstes Filter setzen
	cpi	r16,4			;Endwert berschritten?
	brcs	m81310			;nein -> weiter
	clr	r16			;sonst erstes Filter setzen
m81310:	sts	msgfil,r16		;und wieder speichern
	call	okbeep			;Tonsignal ausgeben
	call	eewcon			;Konfig im EEPROM sichern
	rjmp	m81000			;Liste neu anzeigen
;
; Menprogramm: Status-Anzeige, Anzeige neu ausgeben
;
m82000:	lds	r16,dflags		;Anzeige-Flags holen
	cbr	r16,0x01		;Temperaturanzeige ausschalten
	sts	dflags,r16		;Flags wieder speichern
	ldi	r16,2			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Statusanzeige setzen
	ldi	r16,0x2e		;Stringnummer "Normal"
	lds	r17,confg1		;Konfigurationsbyte 1 holen
	bst	r17,7			;Status Abwesend gesetzt?
	brtc	m82020			;nein -> weiter
	ldi	r16,0x2f		;sonst Stringnummer "Abwesend"
m82020:	sts	menext,r16		;Menerweiterung setzen
	jmp	m99100			;Ende, Anzeige neu ausgeben
;
; Menprogramm: Status-Anzeige, Normal/Abwesend umschalten
;
m82400:	lds	r18,dflags		;Anzeige-Flags holen
	lds	r17,confg1		;Konfigurationsbyte 1 holen
	bst	r17,7			;Status Abwesend gesetzt?
	brtc	m82410			;nein -> weiter
	sbr	r18,0x20		;sonst Lautsprecher einschalten
	cbr	r17,0x80		;Status Normal setzen
	ldi	r16,0x2e		;Stringnummer "Normal"
	rjmp	m82420
m82410:	ser	r16
	sts	lsptio,r16		;Lautsprecher-Timeout stopppen
	cbr	r18,0x20		;Lautsprecher ausschalten
	sbr	r17,0x80		;Status Abwesend setzen
	ldi	r16,0x2f		;Stringnummer "Abwesend"
m82420:	sts	dflags,r18		;Anzeige-Flags wieder speichern
	sts	confg1,r17		;Konfigurationsbyte 1 speichern
	sts	menext,r16		;Menerweiterung setzen
	call	okbeep			;Tonsignal ausgeben
	call	eewcon			;Konfig im EEPROM sichern
	jmp	m99200			;Ende, Menzeile neu ausgeben
;
; Menprogramm: Status-Anzeige, LED umschalten
;
m82500:	lds	r17,confg2		;Konfigurationsbyte 2 holen
	bst	r17,0			;LED in Debug-Modus?
	brtc	m82510			;nein -> weiter
	cbr	r17,1			;sonst in Normal-Modus setzen
	rjmp	m82520
m82510:	ser	r16
	sts	rcvflg,r16		;RS-232 Empfangsflag lschen
	sbr	r17,1			;LED in Debug-Modus setzen
m82520:	sts	confg2,r17		;Konfigurationsbyte speichern
	call	eewcon			;Konfig im EEPROM sichern
	rjmp	m82810			;Menerweiterung setzen
;
; Menprogramm: Status-Anzeige, Speicherfehler lschen
;
m82600:	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,3			;Speicherfehler?
	brtc	m82820			;nein -> Ende
	lds	r17,xflags		;verschiedene Flags holen
	andi	r17,0xc0		;EEPROM-Bits filtern
	cpi	r17,0xc0		;sind beide EEPROMs vorhanden?
	brne	m82820			;nein -> Ende
	cbr	r16,1<<3		;sonst Speicherfehler lschen
	sts	errflg,r16		;Fehler-Flags wieder speichern
	rjmp	m82810			;Tonsignal ausgeben, Men-Erw.
;
; Menprogramm: Status-Anzeige, RTC-Fehler lschen
;
m82700:	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,4			;RTC-Fehler?
	brtc	m82820			;nein -> Ende
	cbr	r16,1<<4		;sonst RTC-Fehler lschen
	sts	errflg,r16		;Fehler-Flags wieder speichern
	rjmp	m82810			;Tonsignal ausgeben, Men-Erw.
;
; Menprogramm: Status-Anzeige, Pointer-Fehler lschen
;
m82800:	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,5			;Pointer-Fehler?
	brtc	m82820			;nein -> Ende
	cbr	r16,1<<5		;sonst Pointer-Fehler lschen
	sts	errflg,r16		;Fehler-Flags wieder speichern
	lds	r16,xflag2		;verschiedene Flags 2 holen
	cbr	r16,0x0e		;Pointer-Flags lschen
	sts	xflag2,r16		;Flags wieder speichern
;
m82810:	call	okbeep			;Tonsignal ausgeben
m82820:	ldi	r16,0x2e		;Stringnummer "Normal"
	lds	r17,confg1		;Konfigurationsbyte 1 holen
	bst	r17,7			;Status Abwesend gesetzt?
	brtc	m82830			;nein -> weiter
	ldi	r16,0x2f		;sonst Stringnummer "Abwesend"
m82830:	sts	menext,r16		;Menerweiterung setzen
	jmp	m99300			;Ende
;
; Menprogramm: Konfiguration/Temperaturen/Sensoren
;
m83000:	call	cmapbk			;Mapping-Tabelle kopieren
	lds	r16,dflags		;Anzeige-Flags holen
	cbr	r16,0x01		;Temperaturanzeige ausschalten
	sts	dflags,r16		;Flags wieder speichern
	ldi	r16,3			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temperaturkonfig. setzen
	lds	r16,menpos		;ausgewhlte Position holen
	cpi	r16,17			;Wert >16?
	brcs	m83010			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m83010:	tst	r16			;Position=0?
	brne	m83020			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m83020:	sts	menpos,r16		;ausgewhlte Position speichern
	sts	encval,r16		;und als Encoderwert benutzen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,16
	sts	encmax,r16		;Encoder-Maximum= 16 setzen
	jmp	m99100			;Ende, Anzeige neu ausgeben
;
; Menprogramm: Konfig/Temp/Sensoren, Auswhlen
;
m83100:	ldi	r16,0x5a		;Stringnummer "Auswhlen"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m83150			;ja -> bearbeiten
	sbr	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	ldi	r16,3			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Konfig/Ausw. setzen
	jmp	m99200
;
m83150:	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	jmp	m99200
;
; Menprogramm: Konfig/Temp/Sensoren, Ein/Ausschalten
;
m83200:	ldi	r16,3			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Konfig/Ausw. setzen
	lds	r16,menpos		;ausgewhlte Position holen
	call	stemap			;Sensornummer suchen
	clr	r19
	ldi	xl,low(temflg)
	ldi	xh,high(temflg)		;Zeiger auf Temperaturflags
	add	xl,r18			;Position berechnen
	adc	xh,r19			;bertrag
	ld	r16,x			;Temperaturflags holen
	bst	r16,7			;Sensor inaktiv?
	brts	m83210			;ja -> bearbeiten
;
	ldi	r16,0x83		;sonst inaktiv, Neuausgabe
	ldi	yl,low(temval)
	ldi	yh,high(temval)		;Zeiger auf Temperaturwerte
	add	r18,r18			;Offset verdoppeln
	add	yl,r18			;Position berechnen
	adc	yh,r19			;bertrag
	clr	r20			;Temperaturwert lschen L
	ldi	r21,0x80		;Temperaturwert lschen H
	st	y,r20			;Temperaturwert L speichern
	std	y+1,r21			;Temperaturwert H speichern
	rjmp	m83220
;
m83210:	ldi	r16,0x03		;aktiv setzen, Neuausgabe
	ldi	yl,low(temtio)
	ldi	yh,high(temtio)		;Zeiger auf Timeout-Zhler
	add	yl,r18			;Position berechnen
	adc	yh,r19			;bertrag
	ldi	r17,timtmp		;Timeoutwert holen
	st	y,r17			;Timeout-Zhler neu setzen
;
m83220:	st	x,r16			;Temperaturflags speichern
	call	eewcon			;Konfig im EEPROM sichern
	call	okbeep			;Tonsignal ausgeben
	jmp	m99300
;
; Menprogramm: Konfig/Temp/Sensoren, Namen
;
m83300:	ldi	r16,0x5c		;Stringnummer "Namen"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,menpos		;ausgewhlte Position holen
	call	stemap			;Sensornummer suchen
	clr	r19
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
	add	yl,r18			;Position berechnen
	adc	yh,r19			;bertrag
	ld	r17,y			;Temperaturflags holen
	sbr	r17,0x02		;Name neu ausgeben
	st	y,r17			;Flags wieder speichern
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m83350			;ja -> bearbeiten
;
	ldi	r19,9			;Sensornamenlnge im EEPROM
	ldi	r16,low(etemtx)		;EEPROM-Adresse Sensornamen L
	ldi	r17,high(etemtx)	;EEPROM-Adresse Sensornamen H
	call	cpeeab			;EEPROM-Text in Eingabepuffer
	ldi	r16,0x09		;Encoderwert als ASCII anzeigen
	sts	encflg,r16		;Encoder-Flags setzen
	ldi	r16,'a'			;Zeichen "a" voreinstellen
	sts	encval,r16		;und speichern
	ldi	r16,5			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Konfig/Namen setzen
	jmp	m99200
;
m83350:	lds	r20,ascibp		;aktuelle Pufferposition holen
	lds	r16,encval		;aktuellen Encoderwert holen
	cpi	r16,maxasc+2		;Ok-Zeichen?
	breq	m83390			;ja -> String speichern
	cpi	r16,maxasc+1		;Lschzeichen?
	brne	m83380			;nein -> weiter
	tst	r20			;aktuelle Position=0?
	breq	m83370			;ja -> berspringen
	dec	r20			;sonst eine Position zurck
m83360:	sts	ascibp,r20		;neue Position speichern	
m83370:	jmp	m99200
;
m83380:	cpi	r20,9			;letzte Position?
	breq	m83370			;ja -> Zeichen ignorieren
	mov	r13,r16			;Zeichen sichern
	call	txblen			;Pufferlnge in Pixel ermitteln
	mov	r16,r13			;Zeichen neu holen
	call	asclen			;Zeichenlnge ermitteln
	add	r18,r16			;zur Gesamtlnge addieren
	cpi	r18,49			;maximale Lnge berschritten?
	brcc	m83370			;ja -> Zeichen ignorieren
;
	mov	r16,r13			;Zeichen neu holen
	ldi	xl,low(ascibf)		;sonst
	ldi	xh,high(ascibf)		;Zeiger auf Texteingabepuffer
	clr	r21
	add	xl,r20			;Pufferadresse ermitteln
	adc	xh,r21			;bertrag
	st	x,r16			;Zeichen speichern
	inc	r20			;aktuelle Position erhhen
	rjmp	m83360			;speichern
;
m83390:	ldi	r19,9			;Sensornamenlnge im EEPROM
	ldi	r16,low(etemtx)		;EEPROM-Adresse Sensornamen L
	ldi	r17,high(etemtx)	;EEPROM-Adresse Sensornamen H
	call	cpabee			;Texteingabepuffer in EEPROM
	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	ldi	r16,3			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Konfig/Ausw. setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	call	eewcrc			;EERPOM-CRC berechnen/speichern
	call	okbeep			;Tonsignal ausgeben
	jmp	m99200
;
; Menprogramm: Konfig/Temp/Sensoren, Verschieben
;
m83400:	ldi	r16,0x5d		;Stringnummer "Verschieben"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m83450			;ja -> bearbeiten
	sbr	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	ldi	r16,4			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Konfig/Versch. setzen
	jmp	m99200
;
m83450:	call	cbkmap			;Mapping-Tabelle aus Kopie
	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	call	eewcon			;Konfig im EEPROM sichern
	call	okbeep			;Tonsignal ausgeben
	jmp	m99200
;
; Menprogramm: Konfig/Temp/Sensoren, Groanzeige
;
m83500:	ldi	r16,3			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Konfig/Ausw. setzen
	lds	r16,menpos		;ausgewhlte Position holen
	call	stemap			;Sensornummer suchen
	lds	r17,bigtmp		;aktuelle Groanzeige holen
	sts	bigtmp,r18		;neue Groanzeige setzen
	clr	r19
	ldi	xl,low(temflg)
	ldi	xh,high(temflg)		;Zeiger auf Temperaturflags
	add	xl,r18			;Position berechnen (neu)
	adc	xh,r19			;bertrag
	ld	r16,x			;Temperaturflags holen
	sbr	r16,0x02		;Name neu ausgeben
	st	x,r16			;Flags wieder speichern
	ldi	xl,low(temflg)
	ldi	xh,high(temflg)		;Zeiger auf Temperaturflags
	add	xl,r17			;Position berechnen (alt)
	adc	xh,r19			;bertrag
	ld	r16,x			;Temperaturflags holen
	sbr	r16,0x02		;Name neu ausgeben
	st	x,r16			;Flags wieder speichern
	call	eewcon			;Konfig im EEPROM sichern
	call	okbeep			;Tonsignal ausgeben
	jmp	m99300
;
; Menprogramm: Konfiguration/Temperaturen/Grenzwerte
;
m84000:	lds	r16,dflags		;Anzeige-Flags holen
	cbr	r16,0x01		;Temperaturanzeige ausschalten
	sts	dflags,r16		;Flags wieder speichern
	ldi	r16,6			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Sensoren/Grenzwerte setzen
	lds	r16,menpos		;ausgewhlte Position holen
	cpi	r16,17			;Wert >16?
	brcs	m84010			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m84010:	tst	r16			;Position=0?
	brne	m84020			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m84020:	sts	menpos,r16		;ausgewhlte Position speichern
	sts	encval,r16		;und als Encoderwert benutzen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,16
	sts	encmax,r16		;Encoder-Maximum= 16 setzen
	jmp	m99100			;Ende, Anzeige neu ausgeben
;
; Menprogramm: Konfiguration/Temperaturen/Grenzwerte, Auswhlen
;
m84100:	ldi	r16,0x5a		;Stringnummer "Auswhlen"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m84150			;ja -> bearbeiten
	sbr	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	ldi	r16,6			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Konfig/Grenzw. setzen
	jmp	m99200
;
m84150:	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	jmp	m99200
;
; Menprogramm: Konfiguration/Temperaturen/Grenzwerte, UG ndern
;
m84200:	ldi	r16,0x60		;Stringnummer "ndern"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,menpos		;ausgewhlte Position holen
	call	stemap			;Sensornummer suchen
	clr	r19
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
	add	yl,r18			;Position berechnen
	adc	yh,r19			;bertrag
	ld	r17,y			;Temperaturflags holen
	sbr	r17,0x01		;Werte neu ausgeben
	st	y,r17			;Flags wieder speichern
	add	r18,r18			;Sensornummer verdoppeln
	clr	r19			;(16 Bit Werte)
	ldi	yl,low(temlil)
	ldi	yh,high(temlil)		;Zeiger auf untere Temp-Limits
	add	yl,r18			;Position berechnen
	adc	yh,r19			;bertrag
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m84250			;ja -> bearbeiten
;
	ld	r16,y
	ldd	r17,y+1			;unteres Temperaturlimit holen
	cpi	r17,0x80		;Wert gltig?
	brne	m84210			;ja -> weiter
	clr	r16
	clr	r17			;sonst Anfangswert= 0.0 Grad
m84210:	sts	encval,r16		;Encoderwert L setzen
	sts	encvah,r17		;Encoderwert H setzen
	ldi	r16,0x05		;Encoderwert als Temp anzeigen
	sts	encflg,r16		;Encoder-Flags setzen
	ldi	r16,7			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Konfig/Grenzw. setzen
	lds	r16,xflag4		;verschiedene Flags 4 holen
	sbr	r16,1			;Flag fr "UG ndern" setzen
	sts	xflag4,r16		;Flags wieder speichern
	jmp	m99200
;
m84250:	lds	r16,encval		;Encoderwert L holen
	lds	r17,encvah		;Encoderwert H holen
	st	y,r16
	std	y+1,r17			;als Temp-Limit speichern
	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	ldi	r16,6			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Konfig/Grenzw. setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	lds	r16,xflag4		;verschiedene Flags 4 holen
	cbr	r16,1			;Flag fr "UG ndern" lschen
	sts	xflag4,r16		;Flags wieder speichern
	call	eewcon			;Konfig im EEPROM sichern
	call	okbeep			;Tonsignal ausgeben
	jmp	m99200
;
; Menprogramm: Konfiguration/Temperaturen/Grenzwerte, UG lschen
;
m84300:	lds	r16,menpos		;ausgewhlte Position holen
	call	stemap			;Sensornummer suchen
	clr	r19
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
	add	yl,r18			;Position berechnen
	adc	yh,r19			;bertrag
	ld	r17,y			;Temperaturflags holen
	sbr	r17,0x01		;Werte neu ausgeben
	st	y,r17			;Flags wieder speichern
	add	r18,r18			;Sensornummer verdoppeln
	clr	r19			;(16 Bit Werte)
	ldi	yl,low(temlil)
	ldi	yh,high(temlil)		;Zeiger auf untere Temp-Limits
	add	yl,r18			;Position berechnen
	adc	yh,r19			;bertrag
	ldi	r16,0
	ldi	r17,0x80		;Temperaturwert lschen
	st	y,r16
	std	y+1,r17			;als Temp-Limit speichern
	ldi	r16,6			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Konfig/Grenzw. setzen
	call	eewcon			;Konfig im EEPROM sichern
	call	okbeep			;Tonsignal ausgeben
	jmp	m99300
;
; Menprogramm: Konfiguration/Temperaturen/Grenzwerte, OG ndern
;
m84400:	ldi	r16,0x60		;Stringnummer "ndern"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,menpos		;ausgewhlte Position holen
	call	stemap			;Sensornummer suchen
	clr	r19
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
	add	yl,r18			;Position berechnen
	adc	yh,r19			;bertrag
	ld	r17,y			;Temperaturflags holen
	sbr	r17,0x01		;Werte neu ausgeben
	st	y,r17			;Flags wieder speichern
	add	r18,r18			;Sensornummer verdoppeln
	clr	r19			;(16 Bit Werte)
	ldi	yl,low(temlih)
	ldi	yh,high(temlih)		;Zeiger auf obere Temp-Limits
	add	yl,r18			;Position berechnen
	adc	yh,r19			;bertrag
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m84450			;ja -> bearbeiten
;
	ld	r16,y
	ldd	r17,y+1			;oberes Temperaturlimit holen
	cpi	r17,0x80		;Wert gltig?
	brne	m84410			;ja -> weiter
	clr	r16
	clr	r17			;sonst Anfangswert= 0.0 Grad
m84410:	sts	encval,r16		;Encoderwert L setzen
	sts	encvah,r17		;Encoderwert H setzen
	ldi	r16,0x05		;Encoderwert als Temp anzeigen
	sts	encflg,r16		;Encoder-Flags setzen
	ldi	r16,7			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Konfig/Grenzw. setzen
	lds	r16,xflag4		;verschiedene Flags 4 holen
	sbr	r16,1<<1		;Flag fr "OG ndern" setzen
	sts	xflag4,r16		;Flags wieder speichern
	jmp	m99200
;
m84450:	lds	r16,encval		;Encoderwert L holen
	lds	r17,encvah		;Encoderwert H holen
	st	y,r16
	std	y+1,r17			;als Temp-Limit speichern
	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	ldi	r16,6			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Konfig/Grenzw. setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	lds	r16,xflag4		;verschiedene Flags 4 holen
	cbr	r16,1<<1		;Flag fr "OG ndern" lschen
	sts	xflag4,r16		;Flags wieder speichern
	call	eewcon			;Konfig im EEPROM sichern
	call	okbeep			;Tonsignal ausgeben
	jmp	m99200
;
; Menprogramm: Konfiguration/Temperaturen/Grenzwerte, OG lschen
;
m84500:	lds	r16,menpos		;ausgewhlte Position holen
	call	stemap			;Sensornummer suchen
	clr	r19
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
	add	yl,r18			;Position berechnen
	adc	yh,r19			;bertrag
	ld	r17,y			;Temperaturflags holen
	sbr	r17,0x01		;Werte neu ausgeben
	st	y,r17			;Flags wieder speichern
	add	r18,r18			;Sensornummer verdoppeln
	clr	r19			;(16 Bit Werte)
	ldi	yl,low(temlih)
	ldi	yh,high(temlih)		;Zeiger auf obere Temp-Limits
	add	yl,r18			;Position berechnen
	adc	yh,r19			;bertrag
	ldi	r16,0
	ldi	r17,0x80		;Temperaturwert lschen
	st	y,r16
	std	y+1,r17			;als Temp-Limit speichern
	ldi	r16,6			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Konfig/Grenzw. setzen
	call	eewcon			;Konfig im EEPROM sichern
	call	okbeep			;Tonsignal ausgeben
	jmp	m99300
;
; Menprogramm: Konfiguration/Temperaturen/Log-Zuordnung
;
m84600:	call	clmpbk			;Log-Mapping-Tabelle kopieren
	lds	r16,dflags		;Anzeige-Flags holen
	cbr	r16,0x01		;Temperaturanzeige ausschalten
	sts	dflags,r16		;Flags wieder speichern
	ldi	r16,21			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Log-Zuordnung setzen
	lds	r16,menpos		;ausgewhlte Position holen
	cpi	r16,17			;Wert >16?
	brcs	m84610			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m84610:	tst	r16			;Position=0?
	brne	m84620			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m84620:	sts	menpos,r16		;ausgewhlte Position speichern
	sts	encval,r16		;und als Encoderwert benutzen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,16
	sts	encmax,r16		;Encoder-Maximum= 16 setzen
	jmp	m99100			;Ende, Anzeige neu ausgeben
;
; Menprogramm: Konfiguration/Temperaturen/Log-Zuordnung, Auswhlen
;
m84700:	ldi	r16,0x5a		;Stringnummer "Auswhlen"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m84750			;ja -> bearbeiten
	sbr	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	ldi	r16,21			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Log-Zuordnung setzen
	jmp	m99200
;
m84750:	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	jmp	m99200
;
; Menprogramm: Konfiguration/Temperaturen/Log-Zuordnung, Verschieben
;
m84800:	ldi	r16,0x5d		;Stringnummer "Verschieben"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m84850			;ja -> bearbeiten
	sbr	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	ldi	r16,22			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Log-Zuordnung setzen
	jmp	m99200
;
m84850:	call	cbklmp			;Log-Mapping-Tabelle aus Kopie
	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	call	eewcon			;Konfig im EEPROM sichern
	call	okbeep			;Tonsignal ausgeben
	jmp	m99200
;
; Menprogramm: Konfiguration/Alarme
;
m85000:	call	calmbk			;Mapping-Tabelle kopieren
	lds	r16,dflags		;Anzeige-Flags holen
	cbr	r16,0x01		;Temperaturanzeige ausschalten
	sts	dflags,r16		;Flags wieder speichern
	ldi	r16,8			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Alarmkonfiguration setzen
	lds	r16,menpos		;ausgewhlte Position holen
	cpi	r16,5			;Wert >4?
	brcs	m85010			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m85010:	tst	r16			;Position=0?
	brne	m85020			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m85020:	sts	menpos,r16		;ausgewhlte Position speichern
	sts	encval,r16		;und als Encoderwert benutzen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,4
	sts	encmax,r16		;Encoder-Maximum= 4 setzen
	jmp	m99100			;Ende, Anzeige neu ausgeben
;
; Menprogramm: Konfiguration/Alarme, Auswhlen
;
m85100:	ldi	r16,0x5a		;Stringnummer "Auswhlen"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m85150			;ja -> bearbeiten
	sbr	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	ldi	r16,8			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Alarm/Auswhlen setzen
	jmp	m99200
;
m85150:	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	jmp	m99200
;
; Menprogramm: Konfiguration/Alarme, Ein/Ausschalten
;
m85200:	ldi	r16,8			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Alarme/Auswhlen setzen
	lds	r16,menpos		;ausgewhlte Position holen
	call	salmap			;Alarmnummer suchen
	clr	r19
	ldi	xl,low(almflg)
	ldi	xh,high(almflg)		;Zeiger auf Alarmflags
	add	xl,r18			;Position berechnen
	adc	xh,r19			;bertrag
	ld	r16,x			;Alarmflags holen
	bst	r16,7			;Sensor inaktiv?
	brts	m85210			;ja -> bearbeiten
	ldi	r16,0x81		;sonst inaktiv, Neuausgabe
	rjmp	m85220
m85210:	ldi	r16,0x01		;aktiv setzen, Neuausgabe
m85220:	st	x,r16			;Alarmflags speichern
	call	eewcon			;Konfig im EEPROM sichern
	call	okbeep			;Tonsignal ausgeben
	jmp	m99300
;
; Menprogramm: Konfiguration/Alarme, Namen
;
m85300:	ldi	r16,0x5c		;Stringnummer "Namen"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,menpos		;ausgewhlte Position holen
	call	salmap			;Alarmnummer suchen
	clr	r19
	ldi	yl,low(almflg)
	ldi	yh,high(almflg)		;Zeiger auf Alarmflags
	add	yl,r18			;Position berechnen
	adc	yh,r19			;bertrag
	ld	r17,y			;Alarmflags holen
	sbr	r17,0x01		;Name neu ausgeben
	st	y,r17			;Flags wieder speichern
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m85350			;ja -> bearbeiten
;
	ldi	r19,11			;Alarmnamenlnge im EEPROM
	ldi	r16,low(ealmtx)		;EEPROM-Adresse Alarmnamen L
	ldi	r17,high(ealmtx)	;EEPROM-Adresse Alarmnamen H
	call	cpeeab			;EEPROM-Text in Eingabepuffer
	ldi	r16,0x09		;Encoderwert als ASCII anzeigen
	sts	encflg,r16		;Encoder-Flags setzen
	ldi	r16,'a'			;Zeichen "a" voreinstellen
	sts	encval,r16		;und speichern
	ldi	r16,10			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Alarme/Namen setzen
	jmp	m99200
;
m85350:	lds	r20,ascibp		;aktuelle Pufferposition holen
	lds	r16,encval		;aktuellen Encoderwert holen
	cpi	r16,maxasc+2		;Ok-Zeichen?
	breq	m85390			;ja -> String speichern
	cpi	r16,maxasc+1		;Lschzeichen?
	brne	m85380			;nein -> weiter
	tst	r20			;aktuelle Position=0?
	breq	m85370			;ja -> berspringen
	dec	r20			;sonst eine Position zurck
m85360:	sts	ascibp,r20		;neue Position speichern	
m85370:	jmp	m99200
;
m85380:	cpi	r20,11			;letzte Position?
	breq	m85370			;ja -> Zeichen ignorieren
	mov	r13,r16			;Zeichen sichern
	call	txblen			;Pufferlnge in Pixel ermitteln
	mov	r16,r13			;Zeichen neu holen
	call	asclen			;Zeichenlnge ermitteln
	add	r18,r16			;zur Gesamtlnge addieren
	cpi	r18,62			;maximale Lnge berschritten?
	brcc	m85370			;ja -> Zeichen ignorieren
;
	mov	r16,r13			;Zeichen neu holen
	ldi	xl,low(ascibf)		;sonst
	ldi	xh,high(ascibf)		;Zeiger auf Texteingabepuffer
	clr	r21
	add	xl,r20			;Pufferadresse ermitteln
	adc	xh,r21			;bertrag
	st	x,r16			;Zeichen speichern
	inc	r20			;aktuelle Position erhhen
	rjmp	m85360			;speichern
;
m85390:	ldi	r19,11			;Alarmnamenlnge im EEPROM
	ldi	r16,low(ealmtx)		;EEPROM-Adresse Alarmnamen L
	ldi	r17,high(ealmtx)	;EEPROM-Adresse Alarmnamen H
	call	cpabee			;Texteingabepuffer in EEPROM
	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	ldi	r16,8			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Alarme/Auswhlen setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	call	eewcrc			;EERPOM-CRC berechnen/speichern
	call	okbeep			;Tonsignal ausgeben
	jmp	m99200
;
; Menprogramm: Konfiguration/Alarme, Invertieren
;
m85400:	ldi	r16,8			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Alarme/Auswhlen setzen
	lds	r16,menpos		;ausgewhlte Position holen
	call	salmap			;Alarmnummer suchen
	clr	r19
	ldi	yl,low(almflg)
	ldi	yh,high(almflg)		;Zeiger auf Alarmflags
	add	yl,r18			;Position berechnen
	adc	yh,r19			;bertrag
	ld	r17,y			;Alarmflags holen
	sbr	r17,0x01		;Name neu ausgeben
	st	y,r17			;Flags wieder speichern
;
	ldi	xl,low(alminv)
	ldi	xh,high(alminv)		;Zeiger auf Alarminvertierung
	add	xl,r18			;Position berechnen
	adc	xh,r19			;bertrag
	ld	r16,x			;Invertierung holen
	ldi	r17,0x01
	eor	r16,r17			;Alarm Invertieren
	st	x,r16			;Invertierung speichern
	call	eewcon			;Konfig im EEPROM sichern
	call	okbeep			;Tonsignal ausgeben
	jmp	m99300
;
; Menprogramm: Konfiguration/Alarme, Verschieben
;
m85500:	ldi	r16,0x5d		;Stringnummer "Verschieben"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m85550			;ja -> bearbeiten
	sbr	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	ldi	r16,9			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Alarme/Verschieben setzen
	jmp	m99200
;
m85550:	call	cbkalm			;Mapping-Tabelle aus Kopie
	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	call	eewcon			;Konfig im EEPROM sichern
	call	okbeep			;Tonsignal ausgeben
	rjmp	m99200
;
; Menprogramm: Konfiguration/Signale/Zuordnung
;
m86000:	lds	r16,dflags		;Anzeige-Flags holen
	cbr	r16,0x01		;Temperaturanzeige ausschalten
	sts	dflags,r16		;Flags wieder speichern
	ldi	r16,11			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Signalzuordnung setzen
	lds	r16,menpos		;ausgewhlte Position holen
	cpi	r16,9			;Wert >9?
	brcs	m86010			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m86010:	tst	r16			;Position=0?
	brne	m86020			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m86020:	sts	menpos,r16		;ausgewhlte Position speichern
	sts	encval,r16		;und als Encoderwert benutzen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,8
	sts	encmax,r16		;Encoder-Maximum= 8 setzen
	rjmp	m99100			;Ende, Anzeige neu ausgeben
;
; Menprogramm: Konfiguration/Signale/Zuordnung, Auswhlen
;
m86100:	ldi	r16,0x5a		;Stringnummer "Auswhlen"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m86150			;ja -> bearbeiten
	sbr	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,8
	sts	encmax,r16		;Encoder-Maximum= 8 setzen
	ldi	r16,11			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Signale/Auswhlen setzen
	rjmp	m99200
;
m86150:	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	rjmp	m99200
;
; Menprogramm: Konfiguration/Signale/Zuordnung, Nummer
;
m86200:	ldi	r16,0x6c		;Stringnummer "Nummer"
	sts	menext,r16		;Menerweiterung setzen
	lds	r18,menpos		;ausgewhlte Position holen
	dec	r18			;korrigieren
	clr	r19
	ldi	xl,low(cfgflg)
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	add	xl,r18			;Position berechnen
	adc	xh,r19			;bertrag
	ld	r16,x			;Konfigurationsflags holen
	sbr	r16,0x02		;Daten neu ausgeben
	st	x,r16			;Flags wieder speichern
	ldi	xl,low(signum)
	ldi	xh,high(signum)		;Zeiger auf Signalnummern
	add	xl,r18			;Position berechnen
	adc	xh,r19			;bertrag
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m86250			;ja -> bearbeiten
;
	ld	r16,x			;Signalnummer holen
	inc	r16			;fr Anzeige korrigieren
	sts	encval,r16		;Encoderwert L setzen
	ldi	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags setzen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 0 setzen
	ldi	r16,15
	sts	encmax,r16		;Encoder-Maximum= 15 setzen
	ldi	r16,12			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Signale/Nummer setzen
	lds	r16,xflag4		;verschiedene Flags 4 holen
	sbr	r16,1<<2		;Flag fr "Num. ndern" setzen
	sts	xflag4,r16		;Flags wieder speichern
	rjmp	m99200
;
m86250:	lds	r16,encval		;Encoderwert L holen
	dec	r16			;korrigieren und
	st	x,r16			;als Signalnummer speichern
	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,8
	sts	encmax,r16		;Encoder-Maximum= 8 setzen
	ldi	r16,11			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Signale/Auswhlen setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	lds	r16,xflag4		;verschiedene Flags 4 holen
	cbr	r16,1<<2		;Flag fr "Num. ndern" lschen
	sts	xflag4,r16		;Flags wieder speichern
	call	eewcon			;Konfig im EEPROM sichern
	call	okbeep			;Tonsignal ausgeben
	rjmp	m99200
;
; Menprogramm: Konfiguration/Signale/Zuordnung, Dauer
;
m86300:	ldi	r16,0x6d		;Stringnummer "Dauer"
	sts	menext,r16		;Menerweiterung setzen
	lds	r18,menpos		;ausgewhlte Position holen
	dec	r18			;korrigieren
	clr	r19
	ldi	xl,low(cfgflg)
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	add	xl,r18			;Position berechnen
	adc	xh,r19			;bertrag
	ld	r16,x			;Konfigurationsflags holen
	sbr	r16,0x02		;Daten neu ausgeben
	st	x,r16			;Flags wieder speichern
	ldi	xl,low(sigdur)
	ldi	xh,high(sigdur)		;Zeiger auf Signaldauer
	add	xl,r18			;Position berechnen
	adc	xh,r19			;bertrag
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m86350			;ja -> bearbeiten
;
	ld	r16,x			;Signalnummer holen
	sts	encval,r16		;Encoderwert L setzen
	ldi	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags setzen
	ldi	r16,0
	sts	encmin,r16		;Encoder-Minimum= 0 setzen
	ldi	r16,30
	sts	encmax,r16		;Encoder-Maximum= 30 setzen
	ldi	r16,12			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Signale/Dauer setzen
	lds	r16,xflag4		;verschiedene Flags 4 holen
	sbr	r16,1<<3		;Flag fr "Dauer ndern" setzen
	sts	xflag4,r16		;Flags wieder speichern
	rjmp	m99200
;
m86350:	lds	r16,encval		;Encoderwert L holen
	st	x,r16			;als Signaldauer speichern
	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,8
	sts	encmax,r16		;Encoder-Maximum= 8 setzen
	ldi	r16,11			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Signale/Auswhlen setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	lds	r16,xflag4		;verschiedene Flags 4 holen
	cbr	r16,1<<3		;Flag fr "Dauer ndern" lsch.
	sts	xflag4,r16		;Flags wieder speichern
	call	eewcon			;Konfig im EEPROM sichern
	call	okbeep			;Tonsignal ausgeben
	rjmp	m99200
;
; Menprogramm: Konfiguration/Signale/Zuordnung, Abspielen
;
m86400:	lds	r16,menpos		;ausgewhlte Position holen
	dec	r16			;korrigieren
	clr	r17
	ldi	xl,low(signum)
	ldi	xh,high(signum)		;Zeiger auf Signalnamen
	add	xl,r16			;Position berechnen
	adc	xh,r17			;bertrag
	ldi	yl,low(sigdur)
	ldi	yh,high(sigdur)		;Zeiger auf Signaldauer
	add	yl,r16			;Position berechnen
	adc	yh,r17			;bertrag
	ld	r18,x			;Signalnummer holen
	ld	r19,y			;Signaldauer holen
	call	signal			;Signal ausgeben
	rjmp	m99300
;
; Menprogramm: Konfiguration/Signale/Audio
;
m87000:	lds	r16,dflags		;Anzeige-Flags holen
	cbr	r16,0x01		;Temperaturanzeige ausschalten
	sts	dflags,r16		;Flags wieder speichern
	ldi	r16,13			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Signale/Audio setzen
	lds	r16,menpos		;ausgewhlte Position holen
	cpi	r16,4			;Wert >3?
	brcs	m87010			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m87010:	tst	r16			;Position=0?
	brne	m87020			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m87020:	sts	menpos,r16		;ausgewhlte Position speichern
	sts	encval,r16		;und als Encoderwert benutzen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,3
	sts	encmax,r16		;Encoder-Maximum= 3 setzen
	rjmp	m99100			;Ende, Anzeige neu ausgeben
;
; Menprogramm: Konfiguration/Signale/Audio, Auswhlen
;
m87100:	ldi	r16,0x5a		;Stringnummer "Auswhlen"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m87150			;ja -> bearbeiten
	sbr	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,3
	sts	encmax,r16		;Encoder-Maximum= 3 setzen
	ldi	r16,13			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Signale/Audio setzen
	rjmp	m99200
;
m87150:	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	rjmp	m99200
;
; Menprogramm: Konfiguration/Signale/Audio, ndern
;
m87200:	ldi	r16,0x60		;Stringnummer "ndern"
	sts	menext,r16		;Menerweiterung setzen
	lds	r18,menpos		;ausgewhlte Position holen
	dec	r18			;korrigieren
	clr	r19
	ldi	xl,low(cfgflg)
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	add	xl,r18			;Position berechnen
	adc	xh,r19			;bertrag
	ld	r16,x			;Konfigurationsflags holen
	sbr	r16,0x02		;Daten neu ausgeben
	st	x,r16			;Flags wieder speichern
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m87250			;ja -> bearbeiten
;	
	lds	r16,menpos		;ausgewhlte Position holen
	cpi	r16,1			;Position auf Signalfrequenz?
	brne	m87210			;nein -> weiter testen
	call	timfrq			;sonst Timer0-Wert in Frequenz
	sts	encval,r16		;Encoderwert L setzen
	sts	encvah,r17		;Encoderwert H setzen
	ldi	r16,low(minfrq)
	ldi	r17,high(minfrq)	;minimale Frequenz laden
	sts	encmin,r16
	sts	encmih,r17		;als Minimalwert setzen
	ldi	r16,low(maxfrq)
	ldi	r17,high(maxfrq)	;maximale Frequenz laden
	sts	encmax,r16
	sts	encmah,r17		;als Maximalwert setzen
	ldi	r16,0x11		;Encoderwert dezimal4 anzeigen
	sts	encflg,r16		;Encoder-Flags setzen
	ldi	r16,14			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Signale/Audio/nd. setzen
	rjmp	m99200
;
m87210:	lds	r17,sigvo1		;Signal-Lautstrke Tag holen
	cpi	r16,2			;Position auf Lautstrke Tag?
	breq	m87220			;ja -> weiter
	lds	r17,sigvo2		;sonst Lautstrke Nacht holen
m87220:	sts	encval,r17		;Encoderwert setzen
	clr	r16
	sts	encmin,r16		;Encoder-Minimum= 0 setzen
	ldi	r16,15
	sts	encmax,r16		;Encoder-Maximum= 15 setzen
	ldi	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags setzen
	ldi	r16,14			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Signale/Audio/nd. setzen
	rjmp	m99200
;
m87250:	lds	r16,menpos		;ausgewhlte Position holen
	cpi	r16,1			;Position auf Signalfrequenz?
	brne	m87260			;nein -> weiter testen
	lds	r20,encval		;Encoderwert L holen
	lds	r21,encvah		;Encoderwert H holen
	call	frqtim			;Frequenz in Timer0-Wert
	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	ldi	r16,13			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Signale/Audio setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	call	eewcon			;Konfig im EEPROM sichern
	call	okbeep			;Tonsignal ausgeben
	rjmp	m99200
;
m87260:	lds	r17,encval		;Encoderwert L holen
	cpi	r16,2			;Position auf Lautstrke Tag?
	brne	m87270			;nein -> weiter
	sts	sigvo1,r17		;als Lautstrke Tag speichern
	rjmp	m87280
m87270:	sts	sigvo2,r17		;als Lautstrke Nacht speichern
m87280:	call	sicalc			;Lautstrke einstellen
	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,3
	sts	encmax,r16		;Encoder-Maximum= 3 setzen
	ldi	r16,13			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Signale/Audio setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	call	eewcon			;Konfig im EEPROM sichern
	call	okbeep			;Tonsignal ausgeben
	rjmp	m99200
;
; Menprogramm: Konfiguration/Signale/Audio, Abspielen
;
m87300:	ldi	r18,1			;Signalnummer 1 setzen
	ldi	r19,3			;Signaldauer 3s setzen
	call	signal			;Signal ausgeben
	rjmp	m99300
;
; Menprogramm: Konfiguration/Zeiten
;
m87500:	lds	r16,dflags		;Anzeige-Flags holen
	cbr	r16,0x01		;Temperaturanzeige ausschalten
	sts	dflags,r16		;Flags wieder speichern
	ldi	r16,15			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Konfig/Zeiten setzen
	lds	r16,menpos		;ausgewhlte Position holen
	cpi	r16,9			;Wert >8?
	brcs	m87510			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m87510:	tst	r16			;Position=0?
	brne	m87520			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m87520:	sts	menpos,r16		;ausgewhlte Position speichern
	sts	encval,r16		;und als Encoderwert benutzen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,8
	sts	encmax,r16		;Encoder-Maximum= 8 setzen
	rjmp	m99100			;Ende, Anzeige neu ausgeben
;
; Menprogramm: Konfiguration/Zeiten, Auswhlen
;
m87600:	ldi	r16,0x5a		;Stringnummer "Auswhlen"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m87650			;ja -> bearbeiten
	sbr	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,8
	sts	encmax,r16		;Encoder-Maximum= 8 setzen
	ldi	r16,15			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Zeiten/Schaltzeiten setzen
	rjmp	m99200
;
m87650:	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	rjmp	m99200
;
; Menprogramm: Konfiguration/Zeiten, Ein/Ausschalten
;
m87700:	ldi	r16,15			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Zeiten/Schaltzeiten setzen
	lds	r18,menpos		;ausgewhlte Position holen
	dec	r18			;korrigieren
	mov	r20,r18			;korrigierte Position sichern
	cbr	r18,0x01		;Bit 0 lschen, erstes Zeile
	clr	r19			;eines Paares neu ausgeben
	ldi	xl,low(cfgflg)
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	add	xl,r18			;Position berechnen
	adc	xh,r19			;bertrag
	ld	r16,x			;Konfigurationsflags holen
	sbr	r16,0x02		;Daten neu ausgeben
	st	x,r16			;Flags wieder speichern
;
	sbr	r18,0x01		;Bit 0 setzen, zweites Zeile
	ldi	xl,low(cfgflg)		;des Paares neu ausgeben
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	add	xl,r18			;Position berechnen
	adc	xh,r19			;bertrag
	ld	r16,x			;Konfigurationsflags holen
	sbr	r16,0x02		;Daten neu ausgeben
	st	x,r16			;Flags wieder speichern
;
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	lds	r18,confg2		;Konfigurationsbyte 2 holen
	clr	r17			;Maskenbyte 1 lschen
	clr	r19			;Maskenbyte 2 lschen
	cpi	r20,2			;Zeile 0/1 (Signalmodus Nacht)?
	brcc	m87710			;nein -> weiter testen
	ldi	r17,1<<5		;Maske fr Signalmodus Nacht
	rjmp	m87740
m87710:	cpi	r20,4			;Zeile 2/3 (LCD-Beleuchtung)?
	brcc	m87720			;nein -> weiter testen
	ldi	r17,1<<3		;Maske fr LCD-Beleuchtung
	rjmp	m87740
m87720:	cpi	r20,6			;Zeile 4/5 (Bluetooth)?
	brcc	m87730			;nein -> weiter testen
	ldi	r19,1<<2		;Maske fr Bluetooth
	rjmp	m87740
m87730:	ldi	r17,1<<4		;Maske fr Min/Max-Reset
m87740:	eor	r16,r17			;Flag invertieren
	eor	r18,r19			;Flag invertieren
	sts	confg1,r16		;Konfigurationsbyte 1 speichern 
	sts	confg2,r18		;Konfigurationsbyte 2 speichern 
	call	eewcon			;Konfig im EEPROM sichern
	rcall	okbeep			;Tonsignal ausgeben
	rjmp	m99300
;
; Menprogramm: Konfiguration/Zeiten, ndern
;
m87800:	ldi	r16,0x60		;Stringnummer "ndern"
	sts	menext,r16		;Menerweiterung setzen
	lds	r18,menpos		;ausgewhlte Position holen
	dec	r18			;korrigieren
	clr	r19
	ldi	xl,low(cfgflg)
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	add	xl,r18			;Position berechnen
	adc	xh,r19			;bertrag
	ld	r16,x			;Konfigurationsflags holen
	sbr	r16,0x02		;Daten neu ausgeben
	st	x,r16			;Flags wieder speichern
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m87850			;ja -> bearbeiten
;	
	lds	r16,signig
	lds	r17,signig+1		;Zeit Signalmodus Nacht 1 laden
	cpi	r18,0			;Position Signalmodus Nacht 1?
	breq	m87810			;ja -> bearbeiten
	lds	r16,signig+2
	lds	r17,signig+3		;Zeit Signalmodus Nacht 2 laden
	cpi	r18,1			;Position Signalmodus Nacht 2?
	breq	m87810			;ja -> bearbeiten
	lds	r16,nighbl
	lds	r17,nighbl+1		;Zeit LCD Beleuchtung 1 laden
	cpi	r18,2			;Position LCD Beleuchtung 1?
	breq	m87810			;ja -> bearbeiten
	lds	r16,nighbl+2
	lds	r17,nighbl+3		;Zeit LCD Beleuchtung 2 laden
	cpi	r18,3			;Position LCD Beleuchtung 2?
	breq	m87810			;ja -> bearbeiten
	lds	r16,nighbt
	lds	r17,nighbt+1		;Zeit Bluetooth 1 laden
	cpi	r18,4			;Position Bluetooth Zeit 1?
	breq	m87810			;ja -> bearbeiten
	lds	r16,nighbt+2
	lds	r17,nighbt+3		;Zeit Bluetooth 2 laden
	cpi	r18,5			;Position Bluetooth Zeit 2?
	breq	m87810			;ja -> bearbeiten
	lds	r16,aresmm
	lds	r17,aresmm+1		;Zeit Reset Minimalwerte laden
	cpi	r18,6			;Position Reset Minimalwerte?
	breq	m87810			;ja -> bearbeiten
	lds	r16,aresmm+2		;sonst
	lds	r17,aresmm+3		;Zeit Reset Maximalwerte laden
m87810:	sts	encval,r16		;Encoderwert L setzen
	sts	encvah,r17		;Encoderwert H setzen
	ldi	r16,0x21		;Encoderwert als Zeit anzeigen
	sts	encflg,r16		;Encoder-Flags setzen
	ldi	r16,16			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Schaltzeiten/ndern setzen
	rjmp	m99200
;
m87850:	lds	r16,encval		;Encoderwert L holen
	lds	r17,encvah		;Encoderwert H holen
	lds	r18,menpos		;ausgewhlte Position holen
	cpi	r18,1			;Position Signalmodus Nacht 1?
	brne	m87860			;nein -> weiter testen
	sts	signig,r16
	sts	signig+1,r17		;Signalmodus Nacht 1 speichern
	rjmp	m87950
m87860:	cpi	r18,2			;Position Signalmodus Nacht 2?
	brne	m87870			;nein -> weiter testen
	sts	signig+2,r16
	sts	signig+3,r17		;Signalmodus Nacht 2 speichern
	rjmp	m87950
m87870:	cpi	r18,3			;Position LCD Beleuchtung 1?
	brne	m87880			;nein -> weiter testen
	sts	nighbl,r16
	sts	nighbl+1,r17		;LCD Beleuchtung 1 speichern
	rjmp	m87950
m87880:	cpi	r18,4			;Position LCD Beleuchtung 2?
	brne	m87890			;nein -> weiter testen
	sts	nighbl+2,r16
	sts	nighbl+3,r17		;LCD Beleuchtung 2 speichern
	rjmp	m87950
m87890:	cpi	r18,5			;Position Bluetooth Zeit 1?
	brne	m87900			;nein -> weiter testen
	sts	nighbt,r16
	sts	nighbt+1,r17		;Bluetooth Zeit 1 speichern
	rjmp	m87950
m87900:	cpi	r18,6			;Position Bluetooth Zeit 2?
	brne	m87910			;nein -> weiter testen
	sts	nighbt+2,r16
	sts	nighbt+3,r17		;Bluetooth Zeit 2 speichern
	rjmp	m87950
m87910:	cpi	r18,7			;Position Reset Minimalwerte?
	brne	m87920			;nein -> weiter
	sts	aresmm,r16
	sts	aresmm+1,r17		;Reset Minimalwerte speichern
	rjmp	m87950
m87920:	sts	aresmm+2,r16		;sonst
	sts	aresmm+3,r17		;Reset Maximalwerte speichern
;
m87950:	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	ldi	r16,15			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Zeiten/Schaltzeiten setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	call	eewcon			;Konfig im EEPROM sichern
	rcall	okbeep			;Tonsignal ausgeben
	rjmp	m99200
;
; Menprogramm: Konfiguration/Zeiten/Uhr stellen
;
m88000:	lds	r16,dflags		;Anzeige-Flags holen
	cbr	r16,0x01		;Temperaturanzeige ausschalten
	sts	dflags,r16		;Flags wieder speichern
	ldi	r16,17			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Zeiten/Uhr stellen setzen
	lds	r16,menpos		;ausgewhlte Position holen
	cpi	r16,6			;Wert >5?
	brcs	m88010			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m88010:	tst	r16			;Position=0?
	brne	m88020			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m88020:	sts	menpos,r16		;ausgewhlte Position speichern
	sts	encval,r16		;und als Encoderwert benutzen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,6
	sts	encmax,r16		;Encoder-Maximum= 6 setzen
	rcall	cpytim			;aktuelle Zeitdaten kopieren
	rjmp	m99100			;Ende, Anzeige neu ausgeben
;
; Menprogramm: Konfiguration/Zeiten/Uhr stellen, Auswhlen
;
m88100:	ldi	r16,0x5a		;Stringnummer "Auswhlen"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m88150			;ja -> bearbeiten
	sbr	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,6
	sts	encmax,r16		;Encoder-Maximum= 6 setzen
	ldi	r16,17			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Zeiten/Uhr stellen setzen
	rjmp	m99200
;
m88150:	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	lds	r16,encval		;Encoderwert holen
	cpi	r16,6
	brne	m88170
;
	rcall	cftime			;Zeit aus Zwischensp. kopieren
	lds	r16,errflg		;Fehler-Flags holen
	cbr	r16,0x01		;Uhren-Fehler lschen
	sts	errflg,r16		;Flags wieder speichern
	lds	r17,minutl		;Minuten Einer holen
	sts	minfl1,r17		;als Minutenflag1 voreinstellen
	sts	minfl3,r17		;als Minutenflag3 voreinstellen
	ser	r17			;0xff laden
	sts	minfl2,r17		;als Minutenflag2 voreinstellen
	lds	r17,hourl		;Stunden Einer holen
	sts	houflg,r17		;als Stundenflag voreinstellen
	bst	r16,4			;RTC-Fehler?
	brts	m88160			;ja -> berspringen
	call	tiwrtc			;sonst Zeit in RTC bertragen
	brcc	m88160			;Fehler? nein -> weiter
	call	i2cerr			;RTC-Fehler setzen
	ldi	r19,0x40		;Meldung "RTC-Fehler" setzen
	call	wrimsg			;Fehlermeldung schreiben
m88160:	rcall	okbeep
m88170:	rjmp	m99200
;
; Menprogramm: Konfiguration/Zeiten/Uhr stellen, ndern
;
m88200:	lds	r18,menpos		;ausgewhlte Position holen
	dec	r18			;korrigieren
	cpi	r18,5			;Position auf "Zeit bernehm."?
	brne	m88210			;nein -> weiter
	rjmp	m99200			;sonst keine Aktion
;
m88210:	ldi	r16,0x60		;Stringnummer "ndern"
	sts	menext,r16		;Menerweiterung setzen
;
	clr	r19			;H-Byte fr Offset
	ldi	xl,low(cfgflg)
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	add	xl,r18			;Position berechnen
	adc	xh,r19			;bertrag
	ld	r16,x			;Konfigurationsflags holen
	sbr	r16,0x02		;Daten neu ausgeben
	st	x,r16			;Flags wieder speichern
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brtc	m88220			;nein -> ndern vorbereiten
	rjmp	m88300			;sonst neuen Wert speichern
;	
m88220:	cpi	r18,0			;Position auf Uhrzeit?
	brne	m88230			;nein -> weiter testen
	lds	r16,cptiml
	lds	r17,cptimh		;Uhrzeit laden
	sts	encval,r16		;Encoderwert L setzen
	sts	encvah,r17		;Encoderwert H setzen
	ldi	r16,0x21		;Encoderwert als Zeit anzeigen
	rjmp	m88280
;
m88230:	cpi	r18,1			;Position auf Kalendertag?
	brne	m88240			;nein -> weiter testen
	lds	r16,cpday		;sonst Tag laden
	sts	encval,r16		;als Encoderwert setzen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,31
	sts	encmax,r16		;Encoder-Maximum= 31 setzen
	ldi	r16,0x03		;Encoderwert dezimal3 anzeigen
	rjmp	m88280
;
m88240:	cpi	r18,2			;Position auf Monat?
	brne	m88250			;nein -> weiter testen
	lds	r16,cpmon		;sonst Monat laden
	sts	encval,r16		;als Encoderwert setzen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,12
	sts	encmax,r16		;Encoder-Maximum= 12 setzen
	ldi	r16,0x03		;Encoderwert dezimal3 anzeigen
	rjmp	m88280
;
m88250:	cpi	r18,3			;Position auf Jahr?
	brne	m88260			;nein -> weiter testen
	lds	r16,cpyear		;sonst Jahr laden
	clr	r17
	ldi	r18,low(yearx)
	ldi	r19,high(yearx)		;Jahr Basiswert laden
	add	r16,r18
	adc	r17,r19			;zum Jahreswert addieren
	sts	encval,r16
	sts	encvah,r17		;als Encoderwert setzen
	ldi	r16,low(2006)
	ldi	r17,high(2006)
	sts	encmin,r16
	sts	encmih,r17		;Encoder-Minimum= 2006 setzen
	ldi	r16,low(2099)
	ldi	r17,high(2099)
	sts	encmax,r16
	sts	encmah,r17		;Encoder-Maximum= 2099 setzen
	ldi	r16,0x11		;Encoderwert dezimal4 anzeigen
	rjmp	m88280
;
m88260:	cpi	r18,4			;Position auf Wochentag?
	brne	m88290			;nein -> Ende
	lds	r16,cpdayw		;sonst Wochentag laden
	inc	r16			;korrigieren
	sts	encval,r16		;als Encoderwert setzen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,7
	sts	encmax,r16		;Encoder-Maximum= 7 setzen
	ldi	r16,0x03		;Encoderwert dezimal3 anzeigen
;
m88280:	sts	encflg,r16		;Encoder-Flags wieder speichern
	ldi	r16,18			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Schaltzeiten/ndern setzen
m88290:	rjmp	m99200
;
m88300:	cpi	r18,0			;Position auf Uhrzeit?
	brne	m88310			;nein -> weiter testen
	lds	r16,encval		;Encoderwert L holen
	lds	r17,encvah		;Encoderwert H holen
	sts	cptiml,r16
	sts	cptimh,r17		;Uhrzeit speichern
	rjmp	m88380
;
m88310:	cpi	r18,1			;Position auf Kalendertag?
	brne	m88320			;nein -> weiter testen
	lds	r16,encval		;Encoderwert holen
	sts	cpday,r16		;Kalendertag speichern
	rjmp	m88380
;
m88320:	cpi	r18,2			;Position auf Monat?
	brne	m88330			;nein -> weiter testen
	lds	r16,encval		;Encoderwert holen
	sts	cpmon,r16		;Monat speichern
	rjmp	m88380
;
m88330:	cpi	r18,3			;Position auf Jahr?
	brne	m88340			;nein -> weiter testen
	lds	r16,encval
	lds	r17,encvah		;Encoderwert holen
	ldi	r18,low(yearx)
	ldi	r19,high(yearx)		;Jahr Basiswert laden
	sub	r16,r18
	sbc	r17,r19			;vom Encoderwert abziehen
	sts	cpyear,r16		;Jahr speichern
	rjmp	m88380
;
m88340:	cpi	r18,4			;Position auf Wochentag?
	brne	m88390			;nein -> Ende
	lds	r16,encval		;Encoderwert holen
	dec	r16			;korrigieren
	sts	cpdayw,r16		;Wochentag speichern
;
m88380:	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	ldi	r16,17			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Zeiten/Schaltzeiten setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
m88390:	rjmp	m99200
;
; Menprogramm: Konfiguration/Sonstiges
;
m88500:	lds	r16,dflags		;Anzeige-Flags holen
	cbr	r16,0x01		;Temperaturanzeige ausschalten
	sts	dflags,r16		;Flags wieder speichern
	ldi	r16,19			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Konfig/Sonstiges setzen
	lds	r16,menpos		;ausgewhlte Position holen
	cpi	r16,8			;Wert >7?
	brcs	m88510			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m88510:	tst	r16			;Position=0?
	brne	m88520			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m88520:	sts	menpos,r16		;ausgewhlte Position speichern
	sts	encval,r16		;und als Encoderwert benutzen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,7
	sts	encmax,r16		;Encoder-Maximum= 7 setzen
	rjmp	m99100			;Ende, Anzeige neu ausgeben
;
; Menprogramm: Konfiguration/Sonstiges, Auswhlen
;
m88600:	ldi	r16,0x5a		;Stringnummer "Auswhlen"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m88650			;ja -> bearbeiten
	sbr	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,7
	sts	encmax,r16		;Encoder-Maximum= 7 setzen
	ldi	r16,19			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Konfig/Sonstiges setzen
	rjmp	m99200
;
m88650:	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	rjmp	m99200
;
; Menprogramm: Konfiguration/Sonstiges, ndern
;
m88700:	lds	r18,menpos		;ausgewhlte Position holen
	dec	r18			;korrigieren
	clr	r19
	ldi	xl,low(cfgflg)
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	add	xl,r18			;Position berechnen
	adc	xh,r19			;bertrag
	ld	r16,x			;Konfigurationsflags holen
	sbr	r16,0x02		;Daten neu ausgeben
	st	x,r16			;Flags wieder speichern
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brtc	m88710			;nein -> ndern vorbereiten
	rjmp	m88800			;sonst neuen Wert speichern
;	
m88710:	cpi	r18,0			;Position auf Sprache?
	brne	m88720			;nein -> weiter testen
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	ldi	r17,0x01		;Maske fr Sprachauswahl laden
	eor	r16,r17			;Sprache umschalten
	sts	confg1,r16		;Konfigurationsbyte 1 speichern
	call	eewcon			;Konfig im EEPROM sichern
	rcall	okbeep			;Tonsignal ausgeben
	rjmp	m99100
;
m88720:	cpi	r18,1			;Position auf Trennzeichen?
	brne	m88730			;nein -> weiter testen
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	ldi	r17,0x02		;Maske fr Trennzeichen laden
	eor	r16,r17			;Trennzeichen umschalten
	sts	confg1,r16		;Konfigurationsbyte 1 speichern
	rjmp	m88790			;Konfiguration sichern
;
m88730:	cpi	r18,2			;Position auf Temp-Signalis.?
	brne	m88740			;nein -> weiter testen
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	ldi	r17,0x04		;Maske fr Temp-Signalis. laden
	eor	r16,r17			;Temp-Signalisierung umschalten
	sts	confg1,r16		;Konfigurationsbyte 1 speichern
	rjmp	m88790			;Konfiguration sichern
;
m88740:	cpi	r18,3			;Position auf LCD-Abschaltung?
	brne	m88750			;nein -> weiter testen
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	ldi	r17,0x40		;Maske fr LCD-Abschalt. laden
	eor	r16,r17			;LCD-Abschaltung umschalten
	sts	confg1,r16		;Konfigurationsbyte 1 speichern
	rjmp	m88790			;Konfiguration sichern
;
m88750:	cpi	r18,4			;Position auf LCD-Backlight?
	brne	m88760			;nein -> weiter testen
	lds	r16,confg2		;Konfigurationsbyte 2 holen
	ldi	r17,0x02		;Maske fr LCD-Backlight laden
	eor	r16,r17			;weiches Backlight umschalten
	sts	confg2,r16		;Konfigurationsbyte 2 speichern
	rjmp	m88790			;Konfiguration sichern
;
m88760:	cpi	r18,5			;Position auf Lichtschwellwert?
	brne	m88770			;nein -> weiter testen
	ldi	r16,0x60		;Stringnummer "ndern"
	sts	menext,r16		;Menerweiterung setzen
	clr	r17
	lds	r16,light		;Schwellwert Lichtsensor holen
	sts	encval,r16		;Encoderwert L setzen
	sts	encvah,r17		;Encoderwert H setzen
	ldi	r16,5
	sts	encmin,r16
	sts	encmih,r17		;Minimalwert 5 setzen
	ldi	r16,250
	sts	encmax,r16
	sts	encmah,r17		;Maximalwert 250 setzen
	ldi	r16,0x11		;Encoderwert dezimal4 anzeigen
	sts	encflg,r16		;Encoder-Flags setzen
	ldi	r16,20			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Konfig/Sonst/ndern setzen
	rjmp	m99200
;
m88770:	lds	r16,baud0		;Baudrate PC-Schnittstelle
	inc	r16			;Wert erhhen
	cpi	r16,7			;Endwert berschritten?
	brcs	m88780			;nein -> weiter
	clr	r16			;sonst auf Anfangswert setzen
m88780:	sts	baud0,r16		;Baudrate wieder speichern
	call	setbd0			;neue Baudrate setzen
m88790:	call	eewcon			;Konfig im EEPROM sichern
	rcall	okbeep			;Tonsignal ausgeben
	rjmp	m99200
;
m88800:	cpi	r18,5			;Position auf Lichtschwellwert?
	brne	m88810			;nein -> Ende
	lds	r16,encval		;Encoderwert L holen
	sts	light,r16		;neuen Schwellwert speichern
	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	ldi	r16,19			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Konfig/Sonstiges setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	ser	r16			;Zwangseinschaltung der
	sts	mentio,r16		;LCD Beleuchtung deaktivieren
	lds	r16,dflags		;Anzeigeflags holen
	sbr	r16,0x0c		;Info/Menzeile neu ausgeben
	sts	dflags,r16		;Anzeigeflags wieder speichern
	call	eewcon			;Konfig im EEPROM sichern
	rcall	okbeep			;Tonsignal ausgeben
	jmp	main			;Lichtautomatik berspringen
;
m88810:	rjmp	m99200
;
; Menprogramm: Konfiguration/Backup, Sichern
;
m89000:	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,3			;Speicherfehler?
	brts	m89010			;ja -> berspringen
	bst	r16,2			;Konfigurationsfehler?
	brts	m89010			;ja -> berspringen
	call	xwconf			;sonst Konfiguration sichern
	brcs	m89020			;Fehler? ja -> melden
	call	okbeep			;sonst Tonsignal ausgeben
m89010:	rjmp	m99200
;
m89020:	lds	r16,errflg		;Fehler-Flags holen
	sbr	r16,0x08		;Speicherfehler setzen
	sts	errflg,r16		;Flags wieder speichern
	rjmp	m99200
;
; Menprogramm: Konfiguration/Backup, Laden/Neustart
;
m89100:	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,3			;Speicherfehler?
	brts	m89010			;ja -> berspringen
	call	strwai			;Warte-Text ausgeben
	call	xrconf			;sonst Konfiguration laden
	brcs	m89020			;Fehler? ja -> melden
m89110:	cli				;sonst Interrupts sperren
	call	wait50			;50ms warten
	jmp	start			;System neu starten
;
; Menprogramm: Konfiguration/Grundeinstellung, Basiskonfiguration lad.
;
m89500:	call	strwai			;Warte-Text ausgeben
	call	eedcfg			;Default-Konfiguration laden
	rjmp	m89110			;System neu starten
;
; Menprogramm: Konfiguration/Grundeinstellung, 15min Werte lschen
;
m89600:	lds	r16,errflg		;Fehler-Flags holen
	andi	r16,0x18		;Ext-EEPROM- oder RTC-Fehler?
	brne	m89630			;ja - berspringen
	call	strwai			;Warte-Text ausgeben
	ldi	r20,low(xt1sta)
	ldi	r21,high(xt1sta)	;Anfangsadresse von EEPROM-
	ldi	r22,byte3(xt1sta)	;Speicherbereich 1
	ldi	r23,low(xt1sta+xt1len)
	ldi	r24,high(xt1sta+xt1len)	;Endadresse von EEPROM-
	ldi	r25,byte3(xt1sta+xt1len);Speicherbereich 1
	call	eraepr			;EEPROM-Speicherbereich lschen
	brcc	m89610			;Fehler? nein -> weiter
	call	twierr			;sonst Speicherfehler setzen
m89610:	ldi	r16,low(xt1sta)
	ldi	r17,high(xt1sta)	;Anfangsadresse von EEPROM-
	ldi	r18,byte3(xt1sta)	;Speicherbereich 1
	sts	xet1po,r16
	sts	xet1po+1,r17		;als neuen Pointerwert 1
	sts	xet1po+2,r18		;speichern
	ldi	xl,low(xet1po)
	ldi	xh,high(xet1po)		;Zeiger auf Pointer 1 setzen
	ldi	r18,rtxt1p		;RTC-Adresse von Pointer 1
	call	i2psen			;Pointer in RTC schreiben
	brcc	m89620			;Fehler? nein -> weiter
	call	i2cerr			;sonst RTC-Fehler setzen
	ldi	r19,0x40		;Meldung "RTC-Fehler" setzen
	rcall	wrimsg			;Fehlermeldung schreiben
m89620:	rcall	okbeep			;Tonsignal ausgeben
m89630:	rjmp	m99200			;Ende, Men-Rckkehr
;
; Menprogramm: Konfiguration/Grundeinstellung, 24h Werte lschen
;
m89700:	lds	r16,errflg		;Fehler-Flags holen
	andi	r16,0x18		;Ext-EEPROM- oder RTC-Fehler?
	brne	m89730			;ja - berspringen
	call	strwai			;Warte-Text ausgeben
	ldi	r20,low(xt2sta)
	ldi	r21,high(xt2sta)	;Anfangsadresse von EEPROM-
	ldi	r22,byte3(xt2sta)	;Speicherbereich 2
	ldi	r23,low(xt2sta+xt2len)
	ldi	r24,high(xt2sta+xt2len)	;Endadresse von EEPROM-
	ldi	r25,byte3(xt2sta+xt2len);Speicherbereich 2
	call	eraepr			;EEPROM-Speicherbereich lschen
	brcc	m89710			;Fehler? nein -> weiter
	call	twierr			;sonst Speicherfehler setzen
m89710:	ldi	r16,low(xt2sta)
	ldi	r17,high(xt2sta)	;Anfangsadresse von EEPROM-
	ldi	r18,byte3(xt2sta)	;Speicherbereich 2
	sts	xet2po,r16
	sts	xet2po+1,r17		;als neuen Pointerwert 2
	sts	xet2po+2,r18		;speichern
	ldi	xl,low(xet2po)
	ldi	xh,high(xet2po)		;Zeiger auf Pointer 2 setzen
	ldi	r18,rtxt2p		;RTC-Adresse von Pointer 2
	call	i2psen			;Pointer in RTC schreiben
	brcc	m89720			;Fehler? nein -> weiter
	call	i2cerr			;sonst RTC-Fehler setzen
	ldi	r19,0x40		;Meldung "RTC-Fehler" setzen
	rcall	wrimsg			;Fehlermeldung schreiben
m89720:	rcall	okbeep			;Tonsignal ausgeben
m89730:	rjmp	m99200			;Ende, Men-Rckkehr
;
; Menprogramm: Konfiguration/Grundeinstellung, Meldungen lschen
;
m89800:	lds	r16,errflg		;Fehler-Flags holen
	andi	r16,0x18		;Ext-EEPROM- oder RTC-Fehler?
	brne	m89830			;ja - berspringen
	call	strwai			;Warte-Text ausgeben
	ldi	r20,low(xmssta)
	ldi	r21,high(xmssta)	;Anfangsadresse von EEPROM-
	ldi	r22,byte3(xmssta)	;Speicherbereich 3
	ldi	r23,low(xmssta+xmslen)
	ldi	r24,high(xmssta+xmslen)	;Endadresse von EEPROM-
	ldi	r25,byte3(xmssta+xmslen);Speicherbereich 3
	call	eraepr			;EEPROM-Speicherbereich lschen
	brcc	m89810			;Fehler? nein -> weiter
	call	twierr			;sonst Speicherfehler setzen
m89810:	ldi	r16,low(xmssta)
	ldi	r17,high(xmssta)	;Anfangsadresse von EEPROM-
	ldi	r18,byte3(xmssta)	;Speicherbereich 3
	sts	xemspo,r16
	sts	xemspo+1,r17		;als neuen Pointerwert 3
	sts	xemspo+2,r18		;speichern
	ldi	xl,low(xemspo)
	ldi	xh,high(xemspo)		;Zeiger auf Pointer 3 setzen
	ldi	r18,rtxmsp		;RTC-Adresse von Pointer 3
	call	i2psen			;Pointer in RTC schreiben
	brcc	m89820			;Fehler? nein -> weiter
	call	i2cerr			;sonst RTC-Fehler setzen
	ldi	r19,0x40		;Meldung "RTC-Fehler" setzen
	rcall	wrimsg			;Fehlermeldung schreiben
m89820:	rcall	okbeep			;Tonsignal ausgeben
m89830:	rjmp	m99200			;Ende, Men-Rckkehr
;
; Menprogramm: Grafik/Preset-01
;
m8a000:	ldi	r16,0			;Preset-01 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-02
;
m8a010:	ldi	r16,1			;Preset-02 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-03
;
m8a020:	ldi	r16,2			;Preset-03 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-04
;
m8a030:	ldi	r16,3			;Preset-04 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-05
;
m8a040:	ldi	r16,4			;Preset-05 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-06
;
m8a050:	ldi	r16,5			;Preset-06 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-07
;
m8a060:	ldi	r16,6			;Preset-07 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-08
;
m8a070:	ldi	r16,7			;Preset-08 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-09
;
m8a080:	ldi	r16,8			;Preset-09 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-10
;
m8a090:	ldi	r16,9			;Preset-10 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-11
;
m8a100:	ldi	r16,10			;Preset-11 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-12
;
m8a110:	ldi	r16,11			;Preset-12 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-13
;
m8a120:	ldi	r16,12			;Preset-13 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-14
;
m8a130:	ldi	r16,13			;Preset-14 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-15
;
m8a140:	ldi	r16,14			;Preset-15 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-16
;
m8a150:	ldi	r16,15			;Preset-16 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-17
;
m8a160:	ldi	r16,16			;Preset-17 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-18
;
m8a170:	ldi	r16,17			;Preset-18 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-19
;
m8a180:	ldi	r16,18			;Preset-19 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-20
;
m8a190:	ldi	r16,19			;Preset-20 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-21
;
m8a200:	ldi	r16,20			;Preset-21 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-22
;
m8a210:	ldi	r16,21			;Preset-22 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-23
;
m8a220:	ldi	r16,22			;Preset-23 auswhlen
	rjmp	m8a250
;
; Menprogramm: Grafik/Preset-24
;
m8a230:	ldi	r16,23			;Preset-24 auswhlen
;
m8a250:	sts	grpres,r16		;Grafik-Preset speichern
	ldi	r17,5			;Grafik-Presets jeweils 5 Bytes
	mul	r16,r17			;Tabellenoffset ermitteln
	ldi	r16,low(egrpre)		;EEPROM-Adresse Presets L
	ldi	r17,high(egrpre)	;EEPROM-Adresse Presets H
	add	r16,r0			;Offset zur Adresse L addieren
	adc	r17,r1			;Offset zur Adresse H addieren
	call	eeread			;Zeichen aus EEPROM holen
	sts	grsen1,r18		;als Sensornummer 1 speichern
	call	eeread			;Zeichen aus EEPROM holen
	sts	grsen2,r18		;als Sensornummer 2 speichern
	call	eeread			;Zeichen aus EEPROM holen
	andi	r18,0x03		;relevante Bits filtern
	sts	grmod1,r18		;als Anzeigemodus 1 speichern
	call	eeread			;Zeichen aus EEPROM holen
	andi	r18,0x03		;relevante Bits filtern
	sts	grmod2,r18		;als Anzeigemodus 2 speichern
	call	eeread			;Zeichen aus EEPROM holen
	sts	grflag,r18		;als Grafik-Flags speichern
;
	lds	r20,dflags		;Anzeige-Flags holen
	cbr	r20,0x01		;Temperaturanzeige ausschalten
	sts	dflags,r20		;Flags wieder speichern
	ldi	r20,23			;alternativen Anzeigemodus
	sts	dimode,r20		;auf Grafikanzeige setzen
	lds	r20,xflag3		;verschiedene Flags 3 holen
	sbr	r20,1<<3|1<<1		;Grafikdaten aus EEPROM lesen
	cbr	r20,1<<5|1<<4|1<<2	;Grafik noch nicht anzeigen
	sts	xflag3,r20		;Flags wieder speichern
	lds	r16,grmod1		;Anzeigemodus 1 holen
	tst	r16			;Modus 15-min-Anzeige?
	brne	m8a260			;nein -> weiter
;
	lds	r20,xet1po		;sonst
	lds	r21,xet1po+1		;aktuelle Position fr 15-min
	lds	r22,xet1po+2		;Temperaturdaten holen
	rjmp	m8a270
;
m8a260:	lds	r20,xet2po
	lds	r21,xet2po+1		;aktuelle Position fr 24-h
	lds	r22,xet2po+2		;Temperaturdaten holen
;
m8a270:	call	grdecx			;Zeiger auf letzten Datensatz
	sts	grpend,r20
	sts	grpend+1,r21		;und Zeiger fr letzten Wert
	sts	grpend+2,r22		;der Grafikanzeige speichern
	rjmp	m99100			;Ende, Anzeige neu ausgeben
;
; Menprogramm: Grafik/Preset, Rckwrts blttern
;
m8a300:	lds	r19,xflag3		;verschiedene Flags 3 holen
	bst	r19,5			;erster Datensatz erreicht?
	brtc	m8a310			;nein -> weiter
	rjmp	m99200			;sonst keine Neuausgabe
;
m8a310:	lds	r19,xflag3		;verschiedene Flags 3 holen
	sbr	r19,1<<3|1<<1		;Grafikdaten aus EEPROM lesen
	cbr	r19,1<<4|1<<2		;Grafik noch nicht anzeigen
	sts	xflag3,r19		;Flags wieder speichern
;
	lds	r16,grpsta
	lds	r17,grpsta+1		;Position vom ersten Grafik-
	lds	r18,grpsta+2		;pufferwert holen
	sts	grptem,r16		;auerdem
	sts	grptem+1,r17		;als Zeiger fr aktuellen Wert
	sts	grptem+2,r18		;der Grafikanzeige speichern
	call	grdeca			;Adresse vorheriger Datensatz
	sts	grpend,r20
	sts	grpend+1,r21		;und Zeiger fr letzten Wert
	sts	grpend+2,r22		;der Grafikanzeige speichern
	rjmp	m99100			;Ende, Anzeige neu ausgeben
;
; Menprogramm: Grafik/Preset, Vorwrts blttern
;
m8a400:	lds	r19,xflag3		;verschiedene Flags 3 holen
	bst	r19,4			;letzter Datensatz erreicht?
	brtc	m8a410			;nein -> weiter
	rjmp	m99200			;sonst keine Neuausgabe
;
m8a410:	lds	r16,grpend
	lds	r17,grpend+1		;Position vom letzten Grafik-
	lds	r18,grpend+2		;pufferwert holen
	sts	grptem,r16
	sts	grptem+1,r17		;als Zeiger fr aktuellen Wert
	sts	grptem+2,r18		;der Grafikanzeige speichern
	call	grinca			;Adresse nchster Datensatz
	lds	r16,grmod1		;Anzeigemodus 1 holen
	tst	r16			;Modus 15-min-Anzeige?
	brne	m8a420			;nein -> weiter
;
	lds	r16,xet1po		;sonst
	lds	r17,xet1po+1		;aktuelle Position fr 15-min
	lds	r18,xet1po+2		;Temperaturdaten holen
	rjmp	m8a430
;
m8a420:	lds	r16,xet2po
	lds	r17,xet2po+1		;aktuelle Position fr 24-h
	lds	r18,xet2po+2		;Temperaturdaten holen
;
m8a430:	sub	r16,r20
	sbc	r17,r21			;zeigt die Grafik die letzten
	sbc	r18,r22			;Werte?
	brne	m8a440			;nein -> weiter
	rjmp	m99200			;sonst keine Neuausgabe
;
m8a440:	lds	r19,xflag3		;verschiedene Flags 3 holen
	sbr	r19,1<<1		;Grafikdaten aus EEPROM lesen
	cbr	r19,1<<5|1<<3|1<<2	;Grafik noch nicht anzeigen
	sts	xflag3,r19		;Flags wieder speichern
;
	sts	grpsta,r20
	sts	grpsta+1,r21		;und als Zeiger fr ersten Wert
	sts	grpsta+2,r22		;der Grafikanzeige speichern
	sts	grptem,r20		;auerdem
	sts	grptem+1,r21		;als Zeiger fr aktuellen Wert
	sts	grptem+2,r22		;der Grafikanzeige speichern
	rjmp	m99100
;
; Menprogramm: Grafik/Preset, Fadenkreuz aktivieren
;
m8a500:	clr	r17			;H-Byte=0
	lds	r16,grlacr		;letzte Fadenkreuzposition
	cpi	r16,222			;gltige Position?
	brcs	m8a510			;ja -> bernehmen, sonst
	ldi	r16,110			;Fadenkreuzposition auf Mitte
m8a510:	sts	encval,r16		;Encoderwert L setzen
	sts	encvah,r17		;Encoderwert H setzen
	clr	r16
	sts	encmin,r16
	sts	encmih,r17		;Minimalwert 0 setzen
	ldi	r16,221
	sts	encmax,r16
	sts	encmah,r17		;Maximalwert 221 setzen
	ldi	r16,0x11		;Encoderwert dezimal4 anzeigen
	sts	encflg,r16		;Encoder-Flags setzen
	ldi	r16,24			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Grafikanzeige/Fadenkreuz
	ldi	r18,110			;Fadenkreuzposition auf Mitte
	sts	grpoin,r18		;als Fadenkreuzposition setzen
	call	grscxp			;X-Koordinaten fr Fadenkreuz
	lds	r18,grpoin		;aktuelle Fadenkreuzpos. holen
	call	grscrb			;LCD-Daten Fadenkreuz sichern
	lds	r18,grpoin		;aktuelle Fadenkreuzpos. holen
	call	grcros			;Fadenkreuz zeichnen
	rjmp	m99200
;
; Menprogramm: Grafik/Preset, Sensornamen anzeigen
;
m8a600:	lds	r16,grflag		;Grafik-Flags holen
	bst	r16,0			;Option "Sensornamen" aktiv?
	brts	m8a630			;ja -> bearbeiten
	sbr	r16,1<<0		;sonst Option einschalten
	sts	grflag,r16		;Grafik-Flags wieder speichern

m8a610:	lds	r16,xflag3		;verschiedene Flags 3 holen
	sbr	r16,1<<2		;Grafik neu ausgeben
	sts	xflag3,r16		;Flags wieder speichern
;
m8a620:	ldi	r16,low(egrpre)
	ldi	r17,high(egrpre)	;EEPROM-Adresse Preset-Daten
	lds	r18,grpres		;aktuelle Grafik-Preset-Nummer
	ldi	r19,5			;Offset 5 Bytes
	mul	r18,r19			;Adressoffset Preset-Daten
	add	r16,r0
	adc	r17,r1			;EEPROM-Adresse ermitteln
	ldi	r18,4			;Offset fr Grafik-Flags
	clr	r19
	add	r16,r18
	adc	r17,r19			;EEPROM-Adresse Grafik-Flags
	lds	r18,grflag		;Grafik-Flags holen
	call	eewrit			;im EEPROM speichern
	call	eewcrc			;EERPOM-CRC berechnen/speichern
	rcall	okbeep			;Tonsignal ausgeben
	rjmp	m99200
;
m8a630:	cbr	r16,1<<0		;Option ausschalten
	sts	grflag,r16		;Grafik-Flags wieder speichern
	ldi	r18,16
	ldi	r19,16			;Start-Koordinaten
	ldi	r20,238
	ldi	r21,103			;End-Koordinaten
	lds	r16,grflag		;Grafik-Flags holen
	bst	r16,4			;Doppeldiagramm?
	brtc	m8a640			;nein -> ein Diagramm lschen
	ldi	r21,59			;sonst Endkoordinate Diagramm 1
	call	clrwnd			;Grafikbereich lschen
	ldi	r18,16
	ldi	r19,60			;Start-Koordinaten
	ldi	r20,238
	ldi	r21,103			;End-Koordinaten
m8a640:	call	clrwnd			;Grafikbereich lschen
	rjmp	m8a610			;und neu ausgeben
;
; Menprogramm: Grafik/Preset, Temperaturskala angleichen
;
m8a700:	lds	r16,grflag		;Grafik-Flag holen
	bst	r16,4			;Doppelgrafik?
	brtc	m8a720			;nein -> Option ignorieren
;
	lds	r16,grflag		;Grafik-Flags holen
	bst	r16,1			;Option "Temperatursk." aktiv?
	brts	m8a730			;ja -> bearbeiten
	sbr	r16,1<<1		;sonst Option einschalten
m8a710:	sts	grflag,r16		;Grafik-Flags wieder speichern
;
	ldi	r18,1
	ldi	r19,16			;Start-Koordinaten Diagramm 1
	ldi	r20,15
	ldi	r21,59			;End-Koordinaten Diagramm 1
	call	clrwnd			;Grafikbereich lschen
	ldi	r18,16
	ldi	r19,16			;Start-Koordinaten Diagramm 1
	ldi	r20,238
	ldi	r21,59			;End-Koordinaten Diagramm 1
	call	clrwnd			;Grafikbereich lschen
	ldi	r18,1
	ldi	r19,60			;Start-Koordinaten Diagramm 1
	ldi	r20,15
	ldi	r21,104			;End-Koordinaten Diagramm 1
	call	clrwnd			;Grafikbereich lschen
	ldi	r18,16
	ldi	r19,60			;Start-Koordinaten Diagramm 2
	ldi	r20,238
	ldi	r21,103			;End-Koordinaten Diagramm 2
	call	clrwnd			;Grafikbereich lschen
;
	lds	r16,xflag3		;verschiedene Flags 3 holen
	sbr	r16,1<<2		;Grafik neu ausgeben
	sts	xflag3,r16		;Flags wieder speichern
	rjmp	m8a620			;nderung im EEPROM speichern
;
m8a720:	rjmp	m99200
;
m8a730:	cbr	r16,1<<1		;Option ausschalten
	rjmp	m8a710
;
; Menprogramm: Konfiguration/Grafik-Presets
;
m8b000:	lds	r16,dflags		;Anzeige-Flags holen
	cbr	r16,0x01		;Temperaturanzeige ausschalten
	sts	dflags,r16		;Flags wieder speichern
	ldi	r16,25			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Konfig/Grafik-Pres. setzen
	lds	r16,menpos		;ausgewhlte Position holen
	cpi	r16,9			;Wert >8?
	brcs	m8b010			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m8b010:	tst	r16			;Position=0?
	brne	m8b020			;nein -> weiter
	ldi	r16,1			;sonst Position= 1 setzen
m8b020:	sts	menpos,r16		;ausgewhlte Position speichern
	sts	encval,r16		;und als Encoderwert benutzen
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,8
	sts	encmax,r16		;Encoder-Maximum= 8 setzen
;
m8b030:	lds	r16,grpres		;aktuelles Grafik-Preset holen
	ldi	r17,5			;Grafik-Presets jeweils 5 Bytes
	mul	r16,r17			;Tabellenoffset ermitteln
	ldi	r16,low(egrpre)		;EEPROM-Adresse Presets L
	ldi	r17,high(egrpre)	;EEPROM-Adresse Presets H
	add	r16,r0			;Offset zur Adresse L addieren
	adc	r17,r1			;Offset zur Adresse H addieren
	call	eeread			;Zeichen aus EEPROM holen
	sts	grsen1,r18		;als Sensornummer 1 speichern
	call	eeread			;Zeichen aus EEPROM holen
	sts	grsen2,r18		;als Sensornummer 2 speichern
	call	eeread			;Zeichen aus EEPROM holen
	andi	r18,0x03		;relevante Bits filtern
	sts	grmod1,r18		;als Anzeigemodus 1 speichern
	call	eeread			;Zeichen aus EEPROM holen
	andi	r18,0x03		;relevante Bits filtern
	sts	grmod2,r18		;als Anzeigemodus 2 speichern
	call	eeread			;Zeichen aus EEPROM holen
	sts	grflag,r18		;als Grafik-Flags speichern
	rjmp	m99100			;Ende, Anzeige neu ausgeben
; 
; Menprogramm: Konfiguration/Grafik-Presets/Auswhlen
;
m8b100:	ldi	r16,0x5a		;Stringnummer "Auswhlen"
	sts	menext,r16		;Menerweiterung setzen
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brts	m8b150			;ja -> bearbeiten
	sbr	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags wieder speichern
	ldi	r16,1
	sts	encmin,r16		;Encoder-Minimum= 1 setzen
	ldi	r16,8
	sts	encmax,r16		;Encoder-Maximum= 8 setzen
	ldi	r16,25			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Konfig/Grafik-Presets
	rjmp	m99200
;
m8b150:	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	rjmp	m99200
; 
; Menprogramm: Konfiguration/Grafik-Presets/ndern
;
m8b200:	lds	r18,menpos		;ausgewhlte Position holen
	dec	r18			;korrigieren
	clr	r19
	ldi	xl,low(cfgflg)
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	add	xl,r18			;Position berechnen
	adc	xh,r19			;bertrag
	ld	r16,x			;Konfigurationsflags holen
	sbr	r16,0x02		;Daten neu ausgeben
	st	x,r16			;Flags wieder speichern
	lds	r16,encflg		;Encoder-Flags holen
	bst	r16,0			;Encoderwert als Men-Option?
	brtc	m8b210			;nein -> ndern vorbereiten
	rjmp	m8b400			;sonst neuen Wert speichern
;
; nderung vorbereiten
;
m8b210:	lds	r17,grflag		;Grafik-Flags holen
	cpi	r18,0			;Position auf Preset-Nummer?
	brne	m8b220			;nein -> weiter testen
	lds	r16,grpres		;aktuelle Preset-Nummer holen
	inc	r16			;korrigieren
	ldi	r18,1			;Minimalwert 1 setzen
	ldi	r19,24			;Maximalwert 24 setzen
	rjmp	m8b320			;Encoderparameter setzen
;
m8b220:	cpi	r18,1			;Position auf Preset-Name?
	brne	m8b230			;nein -> weiter testen
	lds	r18,grpres		;Grafik-Preset-Nummer holen
	ldi	r19,16			;Sensornamenlnge im EEPROM
	ldi	r16,low(egrprt)		;EEPROM-Adresse Sensornamen L
	ldi	r17,high(egrprt)	;EEPROM-Adresse Sensornamen H
	call	cpeeab			;EEPROM-Text in Eingabepuffer
	ldi	r16,0x09		;Encoderwert als ASCII anzeigen
	sts	encflg,r16		;Encoder-Flags setzen
	ldi	r16,'a'			;Zeichen "a" voreinstellen
	sts	encval,r16		;und speichern
	rjmp	m8b330			;auf ndern umschalten
;
m8b230:	cpi	r18,2			;Position auf Grafik 1 Sensor?
	brne	m8b240			;nein -> weiter testen
	lds	r16,grsen1		;Grafik 1 Sensornummern holen
	andi	r16,0x0f		;Hauptsensornummer filtern
	inc	r16			;korrigieren
	ldi	r18,1			;Minimalwert 1 setzen
	ldi	r19,16			;Maximalwert 16 setzen
	rjmp	m8b320			;Encoderparameter setzen
;
m8b240:	cpi	r18,3			;Position auf Grafik 1 Diff-S.?
	brne	m8b250			;nein -> weiter testen
	clr	r16			;Sensornummer=0 setzen (aus)
	bst	r17,2			;Differenzmodus aktiv?
	brtc	m8b280			;nein -> "aus" voreinstellen
	lds	r16,grsen1		;Grafik 1 Sensornummern holen
	swap	r16			;Nibbles tauschen
	andi	r16,0x0f		;Differenzsensornummer filtern
	inc	r16			;korrigieren
	rjmp	m8b280			;Encoderparameter setzen
;
m8b250:	cpi	r18,4			;Position auf Grafik 1 Modus?
	brne	m8b260			;nein -> weiter testen
	lds	r16,grmod1		;Grafik 1 Modus holen
	ldi	r18,0			;Minimalwert 0 setzen
	ldi	r19,3			;Maximalwert 3 setzen
	rjmp	m8b320			;Encoderparameter setzen
;
m8b260:	cpi	r18,5			;Position auf Grafik 2 Sensor?
	brne	m8b270			;nein -> weiter testen
	clr	r16			;Wert fr "aus" voreinstellen
	bst	r17,4			;Doppelgrafik?
	brtc	m8b280			;nein -> "aus" voreinstellen
	lds	r16,grsen2		;Grafik 2 Sensornummern holen
	andi	r16,0x0f		;Hauptsensornummer filtern
	inc	r16			;korrigieren
	rjmp	m8b280			;Encoderparameter setzen
;
m8b270:	cpi	r18,6			;Position auf Grafik 2 Diff-S.?
	brne	m8b290			;nein -> Grafik 2 Modus
	bst	r17,4			;Doppelgrafik?
	brtc	m8b350			;nein -> keine Eingabe mglich
	clr	r16			;Wert fr "aus" voreinstellen
	bst	r17,3			;Differenzmodus aktiv?
	brtc	m8b280			;nein -> "aus" voreinstellen
	lds	r16,grsen2		;Grafik 2 Sensornummern holen
	swap	r16			;Nibbles tauschen
	andi	r16,0x0f		;Differenzsensornummer filtern
	inc	r16			;korrigieren
m8b280:	ldi	r18,0			;Minimalwert 0 setzen
	ldi	r19,16			;Maximalwert 16 setzen
	rjmp	m8b320			;Encoderparameter setzen
;
m8b290:	bst	r17,4			;Doppelgrafik?
	brtc	m8b350			;nein -> keine Eingabe mglich
	lds	r16,grmod1		;Grafik 1 Modus holen
	lds	r17,grmod2		;Grafik 2 Modus holen
	cpi	r16,0			;Grafik 1 Modus 0?
	brne	m8b300			;nein -> weiter
	mov	r17,r16			;sonst Grafik Modus kopieren
	mov	r18,r16			;Minimalwert 0 setzen
	mov	r19,r16			;Maximalwert 0 setzen
	rjmp	m8b320			;Encoderparameter setzen
;
m8b300:	cpi	r17,0			;Grafik 2 Modus 0?
	brne	m8b310			;nein -> Wert ok
	mov	r17,r16			;sonst Grafik 1 Modus kopieren
m8b310:	ldi	r18,1			;Minimalwert 1 setzen
	ldi	r19,3			;Maximalwert 3 setzen
m8b320:	sts	encval,r16		;Encoderwert setzen
	sts	encmin,r18		;Minimalwert setzen
	sts	encmax,r19		;Maximalwert setzen
	ldi	r16,0x03		;Encoderwert dezimal3 anzeigen
	sts	encflg,r16		;Encoder-Flags setzen
m8b330:	ldi	r16,26			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Konfig/Grafik/ndern setz.
m8b340:	ldi	r16,0x60		;Stringnummer "ndern"
	sts	menext,r16		;Menerweiterung setzen
m8b350:	rjmp	m99200
;
; nderung durchfhren
;
m8b400:	lds	r16,encval		;Encoderwert holen
	cpi	r18,0			;Position auf Preset-Nummer?
	brne	m8b410			;nein -> weiter testen
	dec	r16			;korrigieren
	sts	grpres,r16		;Preset-Nummer speichern
	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	ldi	r16,25			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Konfig/Grafik setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	rjmp	m8b030			;Preset-Daten ausgeben
;
m8b410:	cpi	r18,1			;Position auf Preset-Name?
	brne	m8b460			;nein -> weiter testen
	lds	r20,ascibp		;aktuelle Pufferposition holen
	cpi	r16,maxasc+2		;Ok-Zeichen?
	breq	m8b450			;ja -> String speichern
	cpi	r16,maxasc+1		;Lschzeichen?
	brne	m8b440			;nein -> weiter
	tst	r20			;aktuelle Position=0?
	breq	m8b340			;ja -> berspringen
	dec	r20			;sonst eine Position zurck
m8b420:	sts	ascibp,r20		;neue Position speichern	
	rjmp	m8b340			;Menerweiterung neu setzen
;
m8b440:	cpi	r20,16			;letzte Position?
	breq	m8b340			;ja -> Zeichen ignorieren
	ldi	xl,low(ascibf)		;sonst
	ldi	xh,high(ascibf)		;Zeiger auf Texteingabepuffer
	clr	r21
	add	xl,r20			;Pufferadresse ermitteln
	adc	xh,r21			;bertrag
	st	x,r16			;Zeichen speichern
	inc	r20			;aktuelle Position erhhen
	rjmp	m8b420			;speichern
;
m8b450:	lds	r18,grpres		;Grafik-Preset-Nummer holen
	ldi	r19,16			;Sensornamenlnge im EEPROM
	ldi	r16,low(egrprt)		;EEPROM-Adresse Sensornamen L
	ldi	r17,high(egrprt)	;EEPROM-Adresse Sensornamen H
	call	cpabee			;Texteingabepuffer in EEPROM
	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	ldi	r16,25			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Temp/Grafik-Presets setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
	rjmp	m8b610			;CRC berechnen und speichern
;
m8b460:	cpi	r18,2			;Position auf Grafik 1 Sensor?
	brne	m8b470			;nein -> weiter testen
	dec	r16			;korrigieren
	lds	r17,grsen1		;Grafik 1 Sensoren holen
	andi	r17,0xf0		;Hauptsensor-Bits lschen
	or	r17,r16			;mit neuem Sensor verknpfen
	sts	grsen1,r17		;Grafik 1 Sensoren speichern
	rjmp	m8b600			;Daten im EEPROM speichern
;
m8b470:	cpi	r18,3			;Position auf Grafik 1 Diff-S.?
	brne	m8b500			;nein -> weiter testen
	lds	r17,grflag		;Grafik-Flags holen
	tst	r16			;Encoderwert=0 (aus)?
	brne	m8b480			;nein -> weiter
	cbr	r17,1<<2		;Differenzmodus ausschalten
	rjmp	m8b490
;
m8b480:	sbr	r17,1<<2		;Differenzmodus einschalten
	dec	r16			;Encoderwert korrigieren
	swap	r16			;ins oberes Nibble legen
m8b490:	sts	grflag,r17		;Grafik-Flags wieder speichern
	lds	r17,grsen1		;Grafik 1 Sensoren holen
	andi	r17,0x0f		;Differenzsensor-Bits lschen
	or	r17,r16			;mit neuem Sensor verknpfen
	sts	grsen1,r17		;Grafik 1 Sensoren speichern
	rjmp	m8b600			;Daten im EEPROM speichern
;
m8b500:	cpi	r18,4			;Position auf Grafik 1 Modus?
	brne	m8b510			;nein -> weiter testen
	sts	grmod1,r16		;Grafik 1 Modus speichern
	rjmp	m8b540			;Modus prfen, Daten speichern
;
m8b510:	cpi	r18,5			;Position auf Grafik 2 Sensor?
	brne	m8b560			;nein -> weiter testen
	lds	r17,grflag		;Grafik-Flags holen
	tst	r16			;Encoderwert=0 (aus)?
	brne	m8b520			;nein -> weiter
	cbr	r17,1<<4|1<<3		;sonst Doppelgrafik/Diff. aus
	rjmp	m8b530
;
m8b520:	sbr	r17,1<<4		;Doppelgrafik einschalten
	dec	r16			;Encoderwert korrigieren
m8b530:	sts	grflag,r17		;Grafik-Flags wieder speichern
	lds	r17,grsen2		;Grafik 2 Sensoren holen
	andi	r17,0xf0		;Hauptsensor-Bits lschen
	or	r17,r16			;mit neuem Sensor verknpfen
	sts	grsen2,r17		;Grafik 2 Sensoren speichern
m8b540:	lds	r16,grmod1		;Grafik 1 Modus holen
	lds	r17,grmod2		;Grafik 2 Modus holen
	tst	r16			;Grafik 1 Modus=0?
	breq	m8b550			;ja -> in Grafik 2 kopieren
	tst	r17			;sonst Grafik 2 Modus=0?
	brne	m8b600			;nein -> Daten im EEPROM spei.
m8b550:	sts	grmod2,r16		;sonst Grafik Modus kopieren
	rjmp	m8b600			;Daten im EEPROM speichern
;
m8b560:	cpi	r18,6			;Position auf Grafik 2 Diff-S.?
	brne	m8b590			;nein -> Grafik 2 Modus
	lds	r17,grflag		;Grafik-Flags holen
	tst	r16			;Encoderwert=0 (aus)?
	brne	m8b570			;nein -> weiter
	cbr	r17,1<<3		;Differenzmodus ausschalten
	rjmp	m8b580
;
m8b570:	sbr	r17,1<<3		;Differenzmodus einschalten
	dec	r16			;Encoderwert korrigieren
	swap	r16			;ins oberes Nibble legen
m8b580:	sts	grflag,r17		;Grafik-Flags wieder speichern
	lds	r17,grsen2		;Grafik 2 Sensoren holen
	andi	r17,0x0f		;Differenzsensor-Bits lschen
	or	r17,r16			;mit neuem Sensor verknpfen
	sts	grsen2,r17		;Grafik 2 Sensoren speichern
	rjmp	m8b600			;Daten im EEPROM speichern
;
m8b590:	sts	grmod2,r16		;Grafik 2 Modus speichern
	rjmp	m8b540			;Modus prfen, Daten speichern
;
m8b600:	clr	r16
	sts	encflg,r16		;Encoder-Flags zurcksetzen
	sts	menext,r16		;Menerweiterung lschen
	ldi	r16,25			;alternativen Anzeigemodus
	sts	dimode,r16		;auf Konfig/Grafik setzen
	lds	r16,menpos		;ausgewhlte Position holen
	sts	encval,r16		;als Encoderwert setzen
;
	ldi	r16,low(egrpre)
	ldi	r17,high(egrpre)	;EEPROM-Adresse Grafik-Presets
	lds	r18,grpres		;aktuelles Grafik-Preset holen
	ldi	r19,5			;Adress-Offset 5 Bytes
	mul	r18,r19			;Adress-Offset ermitteln
	add	r16,r0
	adc	r17,r1			;EEPROM-Adresse ermitteln
	lds	r18,grsen1		;Grafik 1 Sensoren holen
	call	eewrit			;im EERPOM speichern
	lds	r18,grsen2		;Grafik 2 Sensoren holen
	call	eewrit			;im EERPOM speichern
	lds	r18,grmod1		;Grafik 1 Modus holen
	call	eewrit			;im EERPOM speichern
	lds	r18,grmod2		;Grafik 2 Modus holen
	call	eewrit			;im EERPOM speichern
	lds	r18,grflag		;Grafik-Flags holen
	call	eewrit			;im EERPOM speichern
;
m8b610:	call	eewcrc			;EERPOM-CRC berechnen/speichern
	call	okbeep			;Tonsignal ausgeben
	rcall	newcf2			;Datenfelder neu ausgeben
	rjmp	m99200
;
; Menprogramm: Konfiguration/Grundeinstellung, Grafik-Presets zurcks.
;
m8c000:	rcall	strwai			;Warte-Text ausgeben
	call	eedgrp			;Default-Grafik-Presets laden
	call	eewcrc			;EERPOM-CRC berechnen/speichern
	call	okbeep			;Tonsignal ausgeben
	rjmp	m99200
;
; Ende des Menprogramms, Anzeige wieder herstellen
;
m99000:	clr	r16			;alternativen Anzeigemodus
	sts	dimode,r16		;lschen
	lds	r16,dflags		;Anzeigeflags holen
	sbr	r16,0x0f		;Temperaturanzeige neu ausgeben
	rjmp	m99210			;Flags wieder speichern
m99100:	lds	r16,dflags		;Anzeigeflags holen
	sbr	r16,0x0e		;Anzeige komplett neu ausgeben
	rjmp	m99210			;Flags wieder speichern
m99200:	lds	r16,dflags		;Anzeigeflags holen
	sbr	r16,0x0c		;Info/Menzeile neu ausgeben
m99210:	sts	dflags,r16		;Anzeigeflags wieder speichern
m99300:	ldi	r16,timmen		;Timeout-Wert fr Men-Rckkehr
	sts	mentio,r16		;neu setzen
	lds	r16,dflags		;Anzeigeflags holen
	sbr	r16,0x40		;Hintergrundbeleuchtung ein
	sts	dflags,r16		;Flags wieder speichern
	ser	r16			;PWM-Wert fr LCD-Hintergrund
	out	ocr2,r16		;setzen
m99400:	jmp	main
;
; Ende des Menprogramms, Rckkehr ins bergeordnete Men
;
m99500:	rcall	men1rd			;Mentabellenheader lesen
	elpm	r16,z+			;bergeordnete Mennummer holen
	elpm	r17,z			;bergeordnete Menoption holen
	andi	r17,0x7f		;bergeordn. Menoption filtern
	tst	r17			;Menoption=0?
	breq	m99510			;ja -> weiter
	dec	r17			;sonst korrigieren
m99510:	sts	mennum,r16		;neue Mennummer setzen
	sts	menopt,r17		;neue Menoption setzen
	rjmp	m99000			;Anzeige neu ausgeben
;
;
; Unterprogramm 1 zum Auslesen der Daten aus den Mentabellen
; Register: r16,r17,r18,r19
;	    r11 -> Anzahl der Menoptionen
;	    r12 -> Menstring
;	    z   -> Zeiger auf Menkopfdaten Byte 2 (bergeordn. Men)
;
men1rd:	ldi	zl,low(menidx*2)
	ldi	zh,high(menidx*2)
	ldi	r18,byte3(menidx*2)	;Zeiger auf Menindex
	lds	r16,mennum		;aktuelle Mennummer holen
	add	r16,r16			;verdoppeln
	clr	r17			;H-Byte=0
	add	zl,r16
	adc	zh,r17
	adc	r18,r17			;Men-Indexposition ermitteln
	out	rampz,r18
	elpm	r18,z+			;Adresse von Mentabelle
	elpm	r19,z			;holen
	movw	zl,r18			;Adresse kopieren
	clr	r18
	add	zl,zl
	adc	zh,zh
	adc	r18,r18			;Men-Adresse verdoppeln
	out	rampz,r18
	elpm	r11,z+			;Anzahl der Menoptionen holen
	elpm	r12,z+			;Menstring holen
	ret
;
; Unterprogramm 2 zum Auslesen der Daten aus den Mentabellen (muss
; immer direkt nach Unterprogramm men1rd aufgerufen werden
; Register: wie men1rd, zustzlich:
;	    z   -> Zeiger auf aktuelle Optionsdaten
;
men2rd:	elpm	r16,z+
	elpm	r16,z+			;Zeiger auf Optionsdaten setzen
	lds	r16,menopt		;aktuelle Menoption holen
	add	r16,r16			;zweimal verdoppeln
	add	r16,r16			;Offset= 4 Bytes
	in	r18,rampz
	add	zl,r16
	adc	zh,r17			;Optionsdaten-Adresse ermitteln
	adc	r18,r17
	out	rampz,r18
	ret
;
; Unterprogramm zum Kopieren der Sensor-Map in den Backup-Speicher
; Register: r16,r17,x,y
;
cmapbk:	ldi	xl,low(temmap)		;Mapping-Tabelle kopieren
	ldi	xh,high(temmap)		;Zeiger auf Mapping-Tabelle
	ldi	yl,low(tembak)
	ldi	yh,high(tembak)		;Zeiger auf Mapping-Backup
	ldi	r17,16			;16 Werte bearbeiten
cmap10:	ld	r16,x+			;Wert aus Tabelle holen
	st	y+,r16			;und in Kopie speichern
	dec	r17			;alle Werte bearbeitet?
	brne	cmap10			;nein -> Schleife
	ret
;
; Unterprogramm zum Kopieren des Backup-Speichers in die Sensor-Map
; Register: r16,r17,x,y
;
cbkmap:	ldi	xl,low(temmap)		;Mapping-Tabelle aus Kopie
	ldi	xh,high(temmap)		;Zeiger auf Mapping-Tabelle
	ldi	yl,low(tembak)
	ldi	yh,high(tembak)		;Zeiger auf Mapping-Backup
	ldi	r17,16			;16 Werte bearbeiten
cbkm10:	ld	r16,y+			;Wert aus Kopie holen
	st	x+,r16			;und in Tabelle speichern
	dec	r17			;alle Werte bearbeitet?
	brne	cbkm10			;nein -> Schleife
	ret
;
; Unterprogramm zum Suchen der Sensornummer in der Sensor-Map
; Register: r17,x
;	    r16 <- Anzeigeposition (1-16)
;	    r18 -> Sensornummer (0-15)
;
stemap:	ldi	xl,low(temmap)
	ldi	xh,high(temmap)		;Zeiger auf Mapping-Tabelle
	rjmp	ste010
;
; Unterprogramm zum Suchen der Sensornummer in der Sensor-Map-Kopie
; Register: wie stemap
;
stebak:	ldi	xl,low(tembak)
	ldi	xh,high(tembak)		;Zeiger auf Mapping-Kopie
ste010:	dec	r16			;Anzeigeposition korrigieren
	clr	r18			;Sensornummer Anfangswert
ste020:	ld	r17,x+			;Anzeigeposition holen
	cp	r16,r17			;Anzeigeposition gefunden?
	brne	ste030			;nein -> weiter
	ret				;sonst Ende
ste030:	inc	r18			;Sensornummer erhhen
	cpi	r18,16			;Endwert erreicht?
	brcs	ste020			;nein -> Schleife
	clr	r18			;sonst Sensornummer 0 setzen
	ret
;
; Unterprogramm zum Kopieren der Log-Map in den Backup-Speicher
; Register: r16,r17,x,y
;
clmpbk:	ldi	xl,low(logmap)		;Mapping-Tabelle kopieren
	ldi	xh,high(logmap)		;Zeiger auf Mapping-Tabelle
	ldi	yl,low(tembak)
	ldi	yh,high(tembak)		;Zeiger auf Mapping-Backup
	ldi	r17,16			;16 Werte bearbeiten
clmp10:	ld	r16,x+			;Wert aus Tabelle holen
	st	y+,r16			;und in Kopie speichern
	dec	r17			;alle Werte bearbeitet?
	brne	clmp10			;nein -> Schleife
	ret
;
; Unterprogramm zum Kopieren des Backup-Speichers in die Log-Map
; Register: r16,r17,x,y
;
cbklmp:	ldi	xl,low(logmap)		;Mapping-Tabelle aus Kopie
	ldi	xh,high(logmap)		;Zeiger auf Mapping-Tabelle
	ldi	yl,low(tembak)
	ldi	yh,high(tembak)		;Zeiger auf Mapping-Backup
	ldi	r17,16			;16 Werte bearbeiten
cbkl10:	ld	r16,y+			;Wert aus Kopie holen
	st	x+,r16			;und in Tabelle speichern
	dec	r17			;alle Werte bearbeitet?
	brne	cbkl10			;nein -> Schleife
	ret
;
; Unterprogramm zum Suchen einer Sensornummer in der Log-Map und Ermit-
; teln der Adresse des Temperaturwertes und der Flags, fr 15min-Werte
; Register: r20,r21
;	    r19 <- gesuchte Log-Position (wird am Ende um 1 erhht)
;	    x   -> Zeiger auf ermittelten Temperaturwert
;	    z   -> Zeiger auf ermitteltes Temperaturflag
;
slgmp1:	ldi	xl,low(logmap)
	ldi	xh,high(logmap)		;Zeiger auf Log-Map
	ldi	r20,0			;Suchzhler fr Log-Position
slg010:	ld	r21,x+			;Tabelleneintrag holen
	cp	r19,r21			;Log-Position gefunden?
	breq	slg020			;ja -> Ende
	inc	r20			;Suchzhler erhhen
	cpi	r20,16			;letzte Position erreicht?
	brcs	slg010			;nein -> Schleife
slg020:	clr	r21			;H-Byte fr Tabellenoffset
	ldi	zl,low(temflg)
	ldi	zh,high(temflg)		;Zeiger auf Temperaturflags
	add	zl,r20			;Tabellenplatz berechnen
	adc	zh,r21			;bertrag
	ldi	xl,low(temval)
	ldi	xh,high(temval)		;Zeiger auf Temperaturwert
	add	r20,r20			;Offset verdoppeln
	add	xl,r20			;Tabellenplatz berechnen
	adc	xh,r21			;bertrag
	inc	r19			;nchste Log-Position
	ret
;
; Unterprogramm zum Suchen einer Sensornummer in der Log-Map und Ermit-
; teln des Adressoffset, fr Minimal- und Maximalwerte
; Register: x
;	    r19     <- gesuchte Log-Position (wird am Ende um 1 erhht)
;	    r20,r21 -> Offset fr Tabelle
;
slgmp2:	ldi	xl,low(logmap)
	ldi	xh,high(logmap)		;Zeiger auf Log-Map
	ldi	r20,0			;Suchzhler fr Log-Position
slg050:	ld	r21,x+			;Tabelleneintrag holen
	cp	r19,r21			;Log-Position gefunden?
	breq	slg060			;ja -> Ende
	inc	r20			;Suchzhler erhhen
	cpi	r20,16			;letzte Position erreicht?
	brcs	slg050			;nein -> Schleife
slg060:	clr	r21			;H-Byte fr Tabellenoffset
	add	r20,r20			;Offset verdoppeln
	inc	r19			;nchste Log-Position
	ret
;
; Unterprogramm zum Kopieren der Alarm-Mapping-Tabelle in den Backup-
; Speicher
; Register: r16,r17,x,y
;
calmbk:	ldi	xl,low(almmap)		;Mapping-Tabelle kopieren
	ldi	xh,high(almmap)		;Zeiger auf Mapping-Tabelle
	ldi	yl,low(almbak)
	ldi	yh,high(almbak)		;Zeiger auf Mapping-Backup
	ldi	r17,4			;4 Werte bearbeiten
calm10:	ld	r16,x+			;Wert aus Tabelle holen
	st	y+,r16			;und in Kopie speichern
	dec	r17			;alle Werte bearbeitet?
	brne	calm10			;nein -> Schleife
	ret
;
; Unterprogramm zum Kopieren des Alarm-Backup-Speichers in die Mapping-
; Tabelle
; Register: r16,r17,x,y
;
cbkalm:	ldi	xl,low(almmap)		;Mapping-Tabelle aus Kopie
	ldi	xh,high(almmap)		;Zeiger auf Mapping-Tabelle
	ldi	yl,low(almbak)
	ldi	yh,high(almbak)		;Zeiger auf Mapping-Backup
	ldi	r17,4			;4 Werte bearbeiten
cbka10:	ld	r16,y+			;Wert aus Kopie holen
	st	x+,r16			;und in Tabelle speichern
	dec	r17			;alle Werte bearbeitet?
	brne	cbka10			;nein -> Schleife
	ret
;
; Unterprogramm zum Suchen der Alarmnummer in der Alarm-Mapping-Tabelle
; Register: r17,x
;	    r16 <- Anzeigeposition (1-4)
;	    r18 -> Alarmnummer (0-3)
;
salmap:	ldi	xl,low(almmap)
	ldi	xh,high(almmap)		;Zeiger auf Mapping-Tabelle
	rjmp	sal010
;
; Unterprogramm zum Suchen der Alarmnummer in der Mapping-Kopie
; Register: wie stemap
;
salbak:	ldi	xl,low(almbak)
	ldi	xh,high(almbak)		;Zeiger auf Mapping-Kopie
sal010:	dec	r16			;Anzeigeposition korrigieren
	clr	r18			;Alarmnummer Anfangswert
sal020:	ld	r17,x+			;Anzeigeposition holen
	cp	r16,r17			;Anzeigeposition gefunden?
	brne	sal030			;nein -> weiter
	ret				;sonst Ende
sal030:	inc	r18			;Alarmnummer erhhen
	cpi	r18,4			;Endwert erreicht?
	brcs	sal020			;nein -> Schleife
	clr	r18			;sonst Alarmnummer 0 setzen
	ret
;
; Unterprogramm zur Neuausgabe aller Temperaturwerte (Flags setzen)
; Register: r16,r17,x
;
newtem:	ldi	xl,low(temflg)
	ldi	xh,high(temflg)		;Zeiger auf Temperaturflags
	ldi	r17,16			;16 Flags bearbeiten
newt10:	ld	r16,x			;Temperaturflag holen
	sbr	r16,0x03		;Flag fr Neuausgabe setzen
	st	x+,r16			;Temperaturflag wieder speich.
	dec	r17			;alle Flags bearbeitet?
	brne	newt10			;nein -> Schleife
	ret
;
; Unterprogramm zur Neuausgabe aller Alarmwerte (Flags setzen)
; Register: r16,r17,x
;
newalm:	ldi	xl,low(almflg)
	ldi	xh,high(almflg)		;Zeiger auf Alarmflags
	ldi	r17,4			;4 Flags bearbeiten
newa10:	ld	r16,x			;Alarmflag holen
	sbr	r16,0x01		;Flag fr Neuausgabe setzen
	st	x+,r16			;Alarmflag wieder speichern
	dec	r17			;alle Flags bearbeitet?
	brne	newa10			;nein -> Schleife
	ret
;
; Unterprogramm zur kompletten Neuausgabe aller Konfigurations-Zeilen
; (Flags setzen)
; Register: r16,r17,r18,x
;
newcfg:	ldi	r18,0x03		;Flag-Maske fr Namen und Daten
	rjmp	newc10
;
; Unterprogramm zur Neuausgabe der Datenfelder aller Konfigurations-
; Zeilen (Flags setzen)
; Register: r16,r17,r18,x
;
newcf2:	ldi	r18,0x02		;Flag-Maske fr Datenfelder
newc10:	ldi	xl,low(cfgflg)
	ldi	xh,high(cfgflg)		;Zeiger auf Konfigurationsflags
	ldi	r17,8			;8 Flags bearbeiten
newc20:	ld	r16,x			;Konfigurationsflag holen
	or	r16,r18			;Flags fr Neuausgabe setzen
	st	x+,r16			;Konfig-Flag wieder speichern
	dec	r17			;alle Flags bearbeitet?
	brne	newc20			;nein -> Schleife
	ret
;
; Unterprogramm zur CRC8-Berechnung fr das interne EEPROM, basierend
; auf dem Algorithmus der 1-Wire Devices nach Application Note 27 von
; Dallas/Maxim
; Register: r14,r15,r16,r17,r18,r19
;	    r20 -> ermittelter CRC-Wert
;
calcrc:	ldi	r16,low(eeprom)
	ldi	r17,high(eeprom)	;Zeiger auf EEPROM-Beginn
	ldi	r19,0x18		;Konstante fr Berechnung
	mov	r14,r19			;bereitlegen
	ser	r20			;CRC-Startwert= 0xff setzen
calc10:	call	eeread			;Byte aus EEPROM holen
	mov	r15,r18			;Byte zwischenspeichern
	ldi	r19,8			;8 Verschiebungen
calc20:	eor	r18,r20			;CRC berechnen
	ror	r18			;Ergebnis in Carry schieben
	mov	r18,r20			;letzten CRC-Wert holen
	brcc	calc30			;war Ergebnis=0? ja -> weiter
	eor	r18,r14			;sonst CRC-Wert updaten
calc30:	ror	r18			;neuen CRC-Wert positionieren
	mov	r20,r18			;und speichern
	mov	r18,r15			;restliche Bits holen
	bst	r18,0			;LSB sichern, nchste Bitpos.,
	ror	r18			;gesichertes Bit als MSB ein-
	bld	r18,7			;setzen (Ersatz fr RR-Befehl)
	mov	r15,r18			;und Wert zwischenspeichern
	dec	r19			;alle Verschiebungen erledigt?
	brne	calc20			;nein -> Schleife
	cpi	r16,low(eprcrc)		;End-Adresse L erreicht?
	brne	calc10			;nein -> Schleife
	cpi	r17,high(eprcrc)	;End-Adresse H erreicht?
	brne	calc10			;nein -> Schleife
	ret				;sonst Ende
;
; Unterprogramm zum Kopieren der aktuellen Zeit in einen temporren
; Zwischenspeicher zum Einstellen der Zeitdaten
; Register: r0,r1,r16,r17,r18,r19,r20
;
cpytim:	cli				;Interrupts sperren
	lds	r16,minutl
	lds	r17,minuth
	lds	r18,hourl
	lds	r19,hourh		;aktuelle Zeit holen
	sei				;Interrupts wieder freigeben
	ldi	r20,10
	mul	r20,r19			;Stunden Zehner x10
	add	r18,r0			;Einer addieren
	ldi	r20,60
	mul	r20,r18			;Gesamtstunden x60
	movw	r18,r0			;Ergebnis kopieren
	ldi	r20,10
	mul	r20,r17			;Minuten Einer x10
	add	r18,r0
	adc	r19,r1			;zum Gesamtwert addieren
	clr	r17
	add	r18,r16
	adc	r19,r17			;Minuten Einer addieren
	sts	cptiml,r18
	sts	cptimh,r19		;Minutenwert speichern
;
	cli				;Interrupts sperren
	lds	r16,dayl
	lds	r17,dayh		;aktuellen Kalendertag holen
	sei				;Interrupts wieder freigeben
	ldi	r20,10
	mul	r20,r17			;Kalendertag Zehner x10
	add	r0,r16			;Einer addieren
	sts	cpday,r0		;Kalendertag speichern
;
	cli				;Interrupts sperren
	lds	r16,monthl
	lds	r17,monthh		;aktuellen Monat holen
	sei				;Interrupts wieder freigeben
	ldi	r20,10
	mul	r20,r17			;Monat Zehner x10
	add	r0,r16			;Einer addieren
	sts	cpmon,r0		;Monat speichern
;
	cli				;Interrupts sperren
	lds	r16,yearl
	lds	r17,yearh		;aktuelles Jahr holen
	sei				;Interrupts wieder freigeben
	ldi	r20,10
	mul	r20,r17			;Monat Zehner x10
	add	r0,r16			;Einer addieren
	sts	cpyear,r0		;Monat speichern
;
	lds	r16,dayw		;aktuellen Wochentag holen
	sts	cpdayw,r16		;Wochentag speichern
	ret
;
; Unterprogramm zum bernehmen der Zeit aus dem temporren Zwischen-
; speicher
; Register: r16,r17,r18,r19,r20
;
cftime:	lds	r16,cptiml		;Zeit aus Zwischenspeicher
	lds	r17,cptimh		;holen
	ldi	r18,60
	clr	r19			;Subtrahend 60 laden
	clr	r20			;Stundenzhler lschen
cft010:	sub	r16,r18			;60 subtrahieren
	sbc	r17,r19			;zuviel subtrahiert?
	brmi	cft020			;ja -> weiter
	inc	r20			;sonst Stundenzhler erhhen
	rjmp	cft010			;Schleife
cft020:	add	r16,r18			;Wert wieder herstellen
	adc	r17,r19			;bertrag
;
cft030:	cpi	r16,10			;Minuten >9?
	brcs	cft040			;ja -> weiter
	subi	r16,10			;sonst 10 subtrahieren
	inc	r17			;Minutenzehnerzhler erhhen
	rjmp	cft030			;Schleife
;
cft040:	cli				;Interrupts sperren
	clr	r18
	sts	secndl,r18		;Sekunden Einer lschen
	sts	secndh,r18		;Sekunden Zehner lschen
	sts	t1cnt2,r18		;Timer1 Interruptzhler lschen
	sts	minutl,r16		;Minuten Einer speichern
	sts	minuth,r17		;Minuten Zehner speichern
	sei				;Interrupts wieder freigeben
;
	clr	r17			;Stundenzehnerzhler lschen
cft050:	cpi	r20,10			;Stunden <10?
	brcs	cft060			;ja -> weiter
	subi	r20,10			;sonst 10 subtrahieren
	inc	r17			;Stundenzehnerzhler erhhen
	rjmp	cft050			;Schleife
cft060:	sts	hourl,r20		;Stunden Einer speichern
	sts	hourh,r17		;Stunden Zehner speichern
;
	lds	r16,cpday		;Kalendertag holen
	clr	r17			;Tagzehnerzhler lschen
cft070:	cpi	r16,10			;Kalendertag <10?
	brcs	cft080			;ja -> weiter
	subi	r16,10			;sonst 10 subtrahieren
	inc	r17			;Tagzehnerzhler erhhen
	rjmp	cft070			;Schleife
cft080:	sts	dayl,r16		;Kalendertag Einer speichern
	sts	dayh,r17		;Kalendertag Zehner speichern
;
	lds	r16,cpmon		;Monat holen
	clr	r17			;Monatzehnerzhler lschen
	cpi	r16,10			;Monat <10?
	brcs	cft090			;ja -> weiter
	subi	r16,10			;sonst 10 subtrahieren
	inc	r17			;Monatzehnerzhler erhhen
cft090:	sts	monthl,r16		;Monat Einer speichern
	sts	monthh,r17		;Monat Zehner speichern
;
	lds	r16,cpyear		;Jahr holen
	clr	r17			;Jahrzehnerzhler lschen
cft100:	cpi	r16,10			;Jahr <10?
	brcs	cft110			;ja -> weiter
	subi	r16,10			;sonst 10 subtrahieren
	inc	r17			;Tagzehnerzhler erhhen
	rjmp	cft100			;Schleife
cft110:	sts	yearl,r16		;Jahr Einer speichern
	sts	yearh,r17		;Jahr Zehner speichern
;
	lds	r16,cpdayw		;Wochentag holen
	sts	dayw,r16		;Wochentag speichern
	ret
;
; Unterprogramm zum Erhhen des DCF77-Pufferinhaltes um eine Minute,
; es werden nur Minuten und Stunden bercksichtigt
; Register: r22,r23
;           z <- Zeiger auf DCF77-Puffer
;
incdcb:	ldd	r23,z+1			;Minuten Einer holen
	inc	r23			;erhhen
	std	z+1,r23			;und wieder speichern
	cpi	r23,10			;bertrag?
	brcs	incd20			;nein -> weiter
	clr	r23			;sonst zurcksetzen
	std	z+1,r23			;und speichern
	ldd	r23,z+2			;Minuten Zehner holen
	inc	r23			;erhhen
	std	z+2,r23			;und wieder speichern
	cpi	r23,6			;bertrag?
	brcs	incd20			;nein -> weiter
	clr	r23			;sonst zurcksetzen
	std	z+2,r23			;und speichern
	ldd	r22,z+4			;Stunden Zehner holen
	ldd	r23,z+3			;Stunden Einer holen
	inc	r23			;erhhen
	std	z+3,r23			;und wieder speichern
	cpi	r23,10			;bertrag?
	brcs	incd10			;nein -> Test auf "24"
	clr	r23			;sonst zurcksetzen
	std	z+3,r23			;und speichern
	inc	r22			;Stunden Zehner erhhen
	std	z+4,r22			;und speichern
incd10:	cpi	r22,2			;Stunden Zehner=2?
	brne	incd20			;nein -> weiter
	cpi	r23,4			;Stunden Einer=4 (=24) ?
	brne	incd20			;nein -> weiter
	clr	r23			;sonst Stunden zurcksetzen
	std	z+3,r23			;Stunden Einer speichern
	std	z+4,r23			;Stunden Zehner speichern
incd20:	ret
;
; Unterprogramm zum Erzeugen eines Tonsignals als Tastenquittung
; Register: r18,r19
;
okbeep:	ldi	r18,15			;Signalnummer fr Tasten-Beep
	ldi	r19,1			;Signaldauer fr Tasten-Beep
;
; Unterprogramm: neues Signal-Ereignis in die Warteschlange einreihen
; Register: r18 <- Signalnummer
;	    r19 <- Signaldauer
;
signal:	push	xl
	push	xh
	push	yl
	push	yh			;Zeigerregister sichern
	lds	xl,sigbwr
	lds	xh,sigbwr+1		;Schreibzeiger holen
	st	x+,r18			;Signalnummer speichern
	st	x+,r19			;Signaldauer speichern
	ldi	yl,low(sigbuf+(sigbul*2))
	ldi	yh,high(sigbuf+(sigbul*2));Adresse Pufferende laden
	sub	yl,xl
	sbc	yh,xh			;Differenz berechnen
	or	yh,yl			;Endadresse erreicht?
	brne	sig010			;nein -> weiter
	ldi	xl,low(sigbuf)
	ldi	xh,high(sigbuf)		;sonst Adresse auf Anfangswert
sig010:	sts	sigbwr,xl
	sts	sigbwr+1,xh		;Schreibzeiger speichern
	pop	yh
	pop	yl
	pop	xh
	pop	xl			;Zeigerregister wieder herst.
	ret
;
; Unterprogramm zum Umrechnen des Lautstrkewertes in den PWM-Wert fr
; Timer1b und Einstellen des PWM-Wertes
; Register: r16
;	    r17 <- Lautstrkewert (0-15)
;
sicalc:	lds	r16,xflag2		;verschiedene Flags 2 holen
	cbr	r16,0x01		;Lautsprecher-Flag lschen
	tst	r17			;Lautstrke-Wert=0?
	breq	sica10			;ja -> weiter, sonst
	sbr	r16,0x01		;Lautsprecher-Flag setzen
	dec	r17			;Wertebereich auf 0-14 setzen
sica10:	sts	xflag2,r16		;Flags wieder speichern
	add	r17,r17			;Wert verdoppeln, Bereich 0-28
	ldi	r16,247			;Multiplikator 247 setzen
	push	r0
	push	r1			;Register sichern
	mul	r16,r17			;Ergebnis im Bereich 0-6916
	out	ocr1bh,r1		;PWM-Wert H setzen
	out	ocr1bl,r0		;PWM-Wert L setzen
	pop	r1
	pop	r0			;Register wiederherstellen
	ret
;
; Unterprogramm zum Umrechnen des Timerwertes (Timer0) in die Frequenz
; Formel: f = (Clock/32) / (1+ocr0), Ergebnis hat den doppelten Wert,
; LSB wird zum Runden benutzt
; Register: r14,r15,r18,r19,r20,r21
;	    r16,r17 -> Frequenz (2 Byte)
;
timfrq:	ldi	r16,low(clock/32)
	ldi	r17,high(clock/32)	;Divident laden
	ldi	r18,byte3(clock/32)	;Taktfrequenz/32
	in	r20,ocr0		;Timer0-Wert holen
	clr	r21			;H-Byte lschen
	inc	r20			;um 1 erhhen
	brne	timf10			;bertrag? nein -> weiter
	inc	r21			;sonst bertrag auf H-Byte
timf10:	rcall	divide			;Division 24bit/16bit
	asr	r17			;Ergebnis H halbieren
	ror	r16			;Ergebnis L halbieren
	brcc	timf20			;Runden? nein -> Ende
	inc	r16			;sonst aufrunden
	brne	timf20			;bertrag? nein -> weiter
	inc	r17			;sonst bertrag auf H-Byte
timf20:	ret
;
; Unterprogramm zum Umrechnen der Frequenz in den Timerwert (Timer0)
; Formel: ocr0 = ((Clock/32) / f) - 1, Ergebnis hat den doppelten Wert,
; LSB wird zum Runden benutzt
; Register: r14,r15,r16,r17,r18,r19
;	    r20,r21 <- Frequenz (2 Byte)
;
frqtim:	ldi	r16,low(clock/32)
	ldi	r17,high(clock/32)
	ldi	r18,byte3(clock/32)
	rcall	divide			;Division 24bit/16bit
	asr	r17			;Ergebnis H halbieren
	ror	r16			;Ergebnis L halbieren
	brcc	frqt10			;Runden? nein -> weiter
	inc	r16			;sonst aufrunden
	brne	frqt10			;bertrag? nein -> weiter
	inc	r17			;sonst bertrag auf H-Byte
frqt10:	subi	r16,1			;um 1 vermindern
	sbci	r17,0			;bertrag
	tst	r17			;H-Byte=0?
	breq	frqt20			;ja -> weiter
	ldi	r16,0xff		;sonst auf Maxwert 255 setzen
frqt20:	cpi	r16,28			;Timerwert>=28?
	brcc	frqt30			;ja -> weiter
	ldi	r16,28			;sonst auf Minwert 28 setzen
frqt30:	out	ocr0,r16		;Timer0 setzen
	ret
;
; Unterprogramm zur Multiplikation 16bit*16bit ohne Vorzeichen (nach
; Application Note AVR200)
; Register: r22
;	    r16,r17 <- Faktor1 (2 Byte)
;	    r18,r19 <- Faktor2 (2 Byte)
;	    r18-r21 -> Ergebnis (4 Byte)
;
multip:	clr	r20
	clr	r21			;obere Ergebnis-Bytes lschen
	ldi	r22,16			;16 Schleifen
	lsr	r19
	ror	r18			;Faktor2 LSB in Carry schieben
mult10:	brcc	mult20			;LSB gesetzt? nein -> weiter
	add	r20,r16			;Faktor1L + Ergebnis-Byte 2
	adc	r21,r17			;Faktor1H + Ergebnis-Byte 3
mult20:	ror	r21			;Ergebnis-Byte 3 verschieben
	ror	r20			;Ergebnis-Byte 2 verschieben
	ror	r19			;Ergebnis-Byte 1 + Faktor2H
	ror	r18			;Ergebnis-Byte 0 + Faktor2L
	dec	r22			;alle Schleifen ausgefhrt?
	brne	mult10			;nein -> Schleife
	ret
;
; Unterprogramm zur Division 24bit/16bit ohne Vorzeichen (nach
; Application Note AVR200, etwas modifiziert)
; Register: r19
;	    r16,r17,r18 <- Dividend (3 Byte)
;	    r20,r21     <- Divisor (2 Byte)
;	    r16,r17     -> Ergebnis (2 Byte)
;	    r14,r15     -> Rest (2 Byte)
;
divide:	clr	r14			;Rest L lschen
	sub	r15,r15			;Rest H und Carry lschen
	ldi	r19,25			;Schleifenzhler setzen
divu10:	rol	r16			;Dividend nach links verschie-
	rol	r17			;ben, hchstwertiges Bit in
	rol	r18			;Carry
	dec	r19			;alle Schleifen durchlaufen?
	brne	divu20			;nein -> weiter
	ret				;sonst Ende
divu20:	rol	r14			;Dividend in Rest schieben
	rol	r15
	sub	r14,r20			;Rest = Rest - Divisor
	sbc	r15,r21
	brcc	divu30			;wenn Ergebnis negativ ->
	add	r14,r20			;Rest wieder herstellen
	adc	r15,r21
	clc				;Carry lschen und ins Ergebnis
	rjmp	divu10			;einschieben, oder
divu30:	sec				;Carry setzen und ins Ergebnis
	rjmp	divu10			;einschieben
;
; Unterprogramm zum Abschalten des LCD (fr fehlerfreien DCF77-Empfang)
; Register: r16
;
lcdoff:	lds	r16,dflags		;Anzeige-Flags holen
	sbr	r16,0x80		;Flag fr Abschaltung setzen
	sts	dflags,r16		;und Flags wieder speichern
	clr	r16
	out	porta,r16		;PortA Pull-Ups abschalten
	in	r16,portc		;PortC Status holen
	cbr	r16,0x0f		;alle LCD-Steuerleitungen auf
	out	portc,r16		;Low setzen
	sbi	portb,0			;LCD-Spannung abschalten
	ret
;
; Unterprogramm zum Einschalten und Initialisieren des LCD
; Register: r16
;
lcdon:	cbi	portb,0			;LCD-Spannung einschalten
	in	r16,portc		;PortC Status holen
	sbr	r16,0x0f		;alle LCD-Steuerleitungen auf
	out	portc,r16		;High setzen
	ldi	r16,0xff
	out	porta,r16		;PortA Pull-Ups einschalten
	lds	r16,dflags		;Anzeige-Flags holen
	cbr	r16,0x80		;Abschaltungs-Flag lschen
	sbr	r16,0x0f		;Flags fr Neuausgabe der
	sts	dflags,r16		;Anzeige setzen, speichern
;
; LCD im Grafikmode initialisieren und lschen
;
	call	wait50			;50ms auf LCD warten
	clr	r17			;LCD Grafik-Adresse=0
	clt				;Data senden
	rcall	outlcd			;ausgeben Low Address
	rcall	outlcd			;ausgeben High Address
	ldi	r17,0x42		;Graphic Home Address Set
	set				;Command senden
	rcall	outlcd			;ausgeben
	ldi	r17,30			;LCD Area=0
	clt				;Data senden
	rcall	outlcd			;ausgeben
	clr	r17			;Datenbyte2=0
	rcall	outlcd			;ausgeben
	ldi	r17,0x43		;Graphic Area Set
	set				;Command senden
	rcall	outlcd			;ausgeben
	rcall	clrlcd			;alle Pixel lschen
	ldi	r17,0x98		;LCD in Grafikmode
	set				;Command senden
	rcall	outlcd			;ausgeben
	ldi	r17,0x80		;Text/Grafik in OR-Mode
	set				;Command senden
	rcall	outlcd			;ausgeben
	ret
;
; Unterprogramm zum Rcksetzen der Minimalwerte (fr Anzeige) durch
; Kopieren der aktuellen Temperaturwerte
; Register: r16,r17,r18,r19,r20,x,y,z
;
rsmin1:	lds	r19,xflags		;verschiedene Flags holen
	ldi	r20,16			;16 Minimalwerte bearbeiten
	ldi	xl,low(temflg)
	ldi	xh,high(temflg)		;Zeiger auf Temperaturflags
	ldi	yl,low(temval)
	ldi	yh,high(temval)		;Zeiger auf Temperaturwerte
	ldi	zl,low(temin1)
	ldi	zh,high(temin1)		;Zeiger auf Minimalwerte (Anz)
rsmi10:	ld	r18,x			;Temperaturflags holen
	bst	r18,7			;Sensor inaktiv?
	brtc	rsmi20			;nein -> weiter
	clr	r16
	ldi	r17,0x80		;sonst Wert lschen (0x8000)
	rjmp	rsmi40			;Wert speichern
rsmi20:	bst	r19,2			;Minimalanzeige gerade aktiv?
	brtc	rsmi30			;nein -> berspringen
	sbr	r18,0x01		;sonst Temp-Wert neu ausgeben
	st	x,r18			;Flags wieder speichern
rsmi30:	ld	r16,y			;Temperaturwert L holen
	ldd	r17,y+1			;Temperaturwert H holen
rsmi40:	st	z,r16			;als Minimalwert L speichern
	std	z+1,r17			;als Minimalwert H speichern
	adiw	zl,2			;nchster Minimalwert
	adiw	yl,2			;nchster Temperaturwert
	adiw	xl,1			;nchstes Temperaturflag
	dec	r20			;alle Werte bearbeitet?
	brne	rsmi10			;nein -> Schleife
	ret
;
; Unterprogramm zum Rcksetzen der Maximalwerte (fr Anzeige) durch
; Kopieren der aktuellen Temperaturwerte
; Register: r16,r17,r18,r19,r20,x,y,z
;
rsmax1:	lds	r19,xflags		;verschiedene Flags holen
	ldi	r20,16			;16 Maximalwerte bearbeiten
	ldi	xl,low(temflg)
	ldi	xh,high(temflg)		;Zeiger auf Temperaturflags
	ldi	yl,low(temval)
	ldi	yh,high(temval)		;Zeiger auf Temperaturwerte
	ldi	zl,low(temax1)
	ldi	zh,high(temax1)		;Zeiger auf Maximalwerte (Anz)
rsma10:	ld	r18,x			;Temperaturflags holen
	bst	r18,7			;Sensor inaktiv?
	brtc	rsma20			;nein -> weiter
	clr	r16
	ldi	r17,0x80		;sonst Wert lschen (0x8000)
	rjmp	rsma40			;Wert speichern
rsma20:	bst	r19,3			;Maximalanzeige gerade aktiv?
	brtc	rsma30			;nein -> berspringen
	sbr	r18,0x01		;sonst Temp-Wert neu ausgeben
	st	x,r18			;Flags wieder speichern
rsma30:	ld	r16,y			;Temperaturwert L holen
	ldd	r17,y+1			;Temperaturwert H holen
rsma40:	st	z,r16			;als Maximalwert L speichern
	std	z+1,r17			;als Maximalwert H speichern
	adiw	zl,2			;nchster Maximalwert
	adiw	yl,2			;nchster Temperaturwert
	adiw	xl,1			;nchstes Temperaturflag
	dec	r20			;alle Werte bearbeitet?
	brne	rsma10			;nein -> Schleife
	ret
;
; Unterprogramm zum Rcksetzen der Minimalwerte (fr EEPROM-Log) durch
; Kopieren der aktuellen Temperaturwerte
; Register: r16,r17,r18,r20,x,y,z
;
rsmin2:	ldi	r20,16			;16 Minimalwerte bearbeiten
	ldi	xl,low(temflg)
	ldi	xh,high(temflg)		;Zeiger auf Temperaturflags
	ldi	yl,low(temval)
	ldi	yh,high(temval)		;Zeiger auf Temperaturwerte
	ldi	zl,low(temin2)
	ldi	zh,high(temin2)		;Zeiger auf Minimalwerte (Log)
rsmi50:	ld	r18,x			;Temperaturflags holen
	bst	r18,7			;Sensor inaktiv?
	brtc	rsmi60			;nein -> weiter
	clr	r16
	ldi	r17,0x80		;sonst Wert lschen (0x8000)
	rjmp	rsmi70			;Wert speichern
rsmi60:	ld	r16,y			;Temperaturwert L holen
	ldd	r17,y+1			;Temperaturwert H holen
rsmi70:	st	z,r16			;als Minimalwert L speichern
	std	z+1,r17			;als Minimalwert H speichern
	adiw	zl,2			;nchster Minimalwert
	adiw	yl,2			;nchster Temperaturwert
	adiw	xl,1			;nchstes Temperaturflag
	dec	r20			;alle Werte bearbeitet?
	brne	rsmi50			;nein -> Schleife
	ret
;
; Unterprogramm zum Rcksetzen der Maximalwerte (fr EEPROM-Log) durch
; Kopieren der aktuellen Temperaturwerte
; Register: r16,r17,r18,r20,x,y,z
;
rsmax2:	ldi	r20,16			;16 Maximalwerte bearbeiten
	ldi	xl,low(temflg)
	ldi	xh,high(temflg)		;Zeiger auf Temperaturflags
	ldi	yl,low(temval)
	ldi	yh,high(temval)		;Zeiger auf Temperaturwerte
	ldi	zl,low(temax2)
	ldi	zh,high(temax2)		;Zeiger auf Maximalwerte (Log)
rsma50:	ld	r18,x			;Temperaturflags holen
	bst	r18,7			;Sensor inaktiv?
	brtc	rsma60			;nein -> weiter
	clr	r16
	ldi	r17,0x80		;sonst Wert lschen (0x8000)
	rjmp	rsma70			;Wert speichern
rsma60:	ld	r16,y			;Temperaturwert L holen
	ldd	r17,y+1			;Temperaturwert H holen
rsma70:	st	z,r16			;als Maximalwert L speichern
	std	z+1,r17			;als Maximalwert H speichern
	adiw	zl,2			;nchster Maximalwert
	adiw	yl,2			;nchster Temperaturwert
	adiw	xl,1			;nchstes Temperaturflag
	dec	r20			;alle Werte bearbeitet?
	brne	rsma50			;nein -> Schleife
	ret
;
; Unterprogramm zum Rcksetzen der Mittelwerte (Summe und Zhler auf 0
; setzen)
; Register: r16,r17,y
;
resavg:	ldi	r17,16*3		;16 Summenwerte mit je 24bit
	ldi	yl,low(temavs)		;bearbeiten
	ldi	yh,high(temavs)		;Zeiger auf Summenwerte
	clr	r16
resa10:	st	y+,r16			;Byte lschen
	dec	r17			;alle Werte bearbeitet?
	brne	resa10			;nein -> Schleife
;
	ldi	r17,16*2		;16 Zhlerwerte mit je 16bit
	ldi	yl,low(temavc)		;bearbeiten
	ldi	yh,high(temavc)		;Zeiger auf Zhlerwerte
resa20:	st	y+,r16			;Byte lschen
	dec	r17			;alle Werte bearbeitet?
	brne	resa20			;nein -> Schleife
	ret
;
; Unterprogramm zum Lschen des Parameterspeichers (PC-Schnittstelle)
; Register: r16,r17
;
clpara:	push	xl
	push	xh			;Register sichern
	ldi	xl,low(pcpara)
	ldi	xh,high(pcpara)		;Zeiger auf Parameterspeicher
	ldi	r17,9			;9 Zeichen lschen
	clr	r16
clpa10:	st	x+,r16			;Parameterzeichen lschen
	dec	r17			;alle Zeichen bearbeitet?
	brne	clpa10			;nein -> Schleife
	pop	xh
	pop	xl			;Register wieder herstellen
	ret
;
; Unterprogramm zum Komprimieren einer Meldung, Ablegen im EEPROM
; Schreibpuffer und Schreiben in das externe EEPROM
; Register: r0,r1,r16,r17,r18,r20,r21,r22,r23,x,y
;	    r19 <- Meldungscode (B7-B4: enthlt Meldungsnummer,
;				B3-B0: enthlt Sensor-/Alarmnummer)
;
wrimsg:	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,3			;Speicherfehler?
	brts	wrim30			;ja -> Ende
	bst	r16,0			;Uhr schon eingestellt?
	brts	wrim30			;nein -> Ende
	rcall	cpstim			;sonst Zeitdaten komprimieren
	lds	r17,secndh		;Sekunden Zehner holen
	mul	r17,r16			;verzehnfachen (r16=10)
	lds	r16,secndl		;Sekunden Einer holen
	add	r16,r0			;Zehner addieren
	sts	xewbuf+4,r16		;Sekunden speichern
	sts	xewbuf+5,r19		;Meldungscode speichern
;
	lds	r20,xemspo
	lds	r21,xemspo+1		;Zeiger auf aktuelle Position
	lds	r22,xemspo+2		;im externen EEPROM holen
	ldi	r23,6			;6 Bytes zum Ext-EEPROM senden
	call	xwrite			;Puffer zum Ext-EEPROM senden
	brcc	wrim10			;Fehler? nein -> weiter
	call	twierr			;sonst Speicherfehler setzen
	rjmp	wrim30			;Adressberechnung berspringen
;
wrim10:	ldi	r16,low(xmssta+xmslen-6)
	ldi	r17,high(xmssta+xmslen-6);Adresse des letzten mgli-
	ldi	r18,byte3(xmssta+xmslen-6);chen Datensatzbeginns laden
	sub	r16,r20
	sbc	r17,r21			;aktuelle Adresse noch im
	sbc	r18,r22			;gltigen Speicherbereich?
	brcc	wrim20			;ja -> weiter
	ldi	r20,low(xmssta)
	ldi	r21,high(xmssta)	;sonst Adresse wieder auf
	ldi	r22,byte3(xmssta)	;Anfang setzen
wrim20:	sts	xemspo,r20
	sts	xemspo+1,r21		;Zeiger fr Position im
	sts	xemspo+2,r22		;externen EEPROM speichern
;
	lds	r16,errflg		;Fehler-Flags holen
	bst	r16,4			;RTC-Fehler?
	brts	wrim30			;ja -> Ende
	ldi	r18,rtxmsp		;RTC-Adresse Meldungs-Daten
	ldi	xl,low(xemspo)
	ldi	xh,high(xemspo)		;Zeiger auf Meld-Daten Pointer
	call	i2psen			;Pointer in RTC schreiben
	brcc	wrim30			;Fehler? nein -> weiter
	call	i2cerr			;sonst RTC Fehlerstatus setzen
wrim30:	ret
;
; Unterprogramm zum Komprimieren der Zeitdaten und Ablegen im EEPROM
; Schreibpuffer
; Register: r0,r1,r16,r17,r18
;
cpstim:	ldi	r16,10			;Zehner-Multiplikator laden
	lds	r17,dayh		;Tag Zehner holen
	mul	r17,r16			;verzehnfachen
	lds	r17,dayl		;Tag Einer holen
	add	r17,r0			;Zehner addieren
	mov	r18,r17			;Ergebnis zwischenspeichern
	lds	r17,monthh		;Monat Zehner holen
	mul	r17,r16			;verzehnfachen
	lds	r17,monthl		;Monat Einer holen
	add	r17,r0			;Zehner addieren
	swap	r17			;Nibbles tauschen
	rol	r17			;Wert verschieben, MSB in Carry
	andi	r17,0xe0		;relevante Bits filtern und
	or	r18,r17			;mit Tag verknpfen
	sts	xewbuf+2,r18		;kompakte Daten speichern
	rol	r18			;MSB vom Monat aus Carry holen
	andi	r18,0x01		;und filtern
	lds	r17,yearh		;Jahr Zehner holen
	mul	r17,r16			;verzehnfachen
	lds	r17,yearl		;Jahr Einer holen
	add	r17,r0			;Zehner addieren
	rol	r17			;Wert verschieben
	andi	r17,0xfe		;relevante filtern und
	or	r18,r17			;mit MSB vom Monat verknpfen
	sts	xewbuf+3,r18		;kompakte Daten speichern
;
	lds	r17,hourh		;Stunden Zehner holen
	mul	r17,r16			;verzehnfachen
	lds	r17,hourl		;Stunden Einer holen
	add	r17,r0			;Zehner addieren
	sts	xewbuf+1,r17		;kompakten Stundenwert speich.
	lds	r17,minuth		;Minuten Zehner holen
	mul	r17,r16			;verzehnfachen
	lds	r17,minutl		;Minuten Einer holen
	add	r17,r0			;Zehner addieren
	sts	xewbuf,r17		;kompakten Minutenwert speich.
	ret
;
; Unterprogramm zum Extrahieren der Zeitdaten aus einem EEPROM-Daten-
; satz in den ASCII-Puffer fr die Datenausgabe zum PC
; Register: r16,r17,r18
;
xtrtim:	ldi	xl,low(ascbuf)
	ldi	xh,high(ascbuf)		;Zeiger auf ASCII-Puffer
	ldi	r16,year3		;Jahr Tausender holen
	st	x+,r16			;in Puffer speichern
	ldi	r16,year2		;Jahr Hunderter holen
	st	x+,r16			;in Puffer speichern
	lds	r18,xerbuf+3		;komprimierte Zeit Byte 3 holen
	bst	r18,0			;LSB sichern, gehrt zum Monat
	lsr	r18			;Jahr an Position schieben
	rcall	bytasc			;in ASCII-Wert wandeln
	st	x+,r17			;Jahr Zehner speichern
	st	x+,r16			;Jahr Einer speichern
	lds	r18,xerbuf+2		;komprimierte Zeit Byte 2 holen
	ror	r18			;Monat nach rechts schieben
	bld	r18,7			;LSB von Byte 3 holen, Monat an
	swap	r18			;richtige Position schieben
	andi	r18,0x0f		;Nicht relevante Bits filtern
	rcall	bytasc			;in ASCII-Wert wandeln
	st	x+,r17			;Monat Zehner speichern
	st	x+,r16			;Monat Einer speichern
	lds	r18,xerbuf+2		;komprimierte Zeit Byte 2 holen
	andi	r18,0x1f		;Nicht relevante Bits filtern
	rcall	bytasc			;in ASCII-Wert wandeln
	st	x+,r17			;Tag Zehner speichern
	st	x+,r16			;Tag Einer speichern
	ldi	r16,';'			;Feldtrennzeichen holen
	st	x+,r16			;Feldtrennzeichen speichern
	lds	r18,xerbuf+1		;komprimierte Zeit Byte 1 holen
	rcall	bytasc			;in ASCII-Wert wandeln
	st	x+,r17			;Stunde Zehner speichern
	st	x+,r16			;Stunde Einer speichern
	ldi	r16,':'			;Trennzeichen holen
	st	x+,r16			;Trennzeichen speichern
	lds	r18,xerbuf		;komprimierte Zeit Byte 0 holen
	rcall	bytasc			;in ASCII-Wert wandeln
	st	x+,r17			;Minute Zehner speichern
	st	x+,r16			;Minute Einer speichern
	ldi	r16,':'			;Trennzeichen holen
	st	x+,r16			;Trennzeichen speichern
	ret
;
; Unterprgramm zum Ergnzen der 00-Sekunde bei Temperaturdaten
; Register: r16
;	    x <- Zeiger auf ASCII-Puffer
;
xtrse0:	ldi	r16,'0'			;Sekunden = 00
	st	x+,r16			;Sekunden Zehner speichern
	st	x+,r16			;Sekunden Einer speichern
xtrse1:	ldi	r16,';'			;Feldtrennzeichen holen
	st	x+,r16			;Feldtrennzeichen speichern
	ret
;
; Unterprogramm zum Ergnzen der Sekunden bei Meldungsdaten
; Register: r16,r17,r18
;	    x <- Zeiger auf ASCII-Puffer
;
xtrsec:	lds	r18,xerbuf+4		;komprimierte Sekunden holen
	rcall	bytasc			;in ASCII-Wert wandeln
	st	x+,r17			;Sekunde Zehner speichern
	st	x+,r16			;Sekunde Einer speichern
	rjmp	xtrse1			;Feldtrennzeichen ausgeben
;
; Unterprogramm zum Extrahieren der Temperaturdaten aus einem EEPROM-
; Datensatz in den ASCII-Puffer fr die Datenausgabe zum PC
; Register: r13,r14,r15,r16,r17,r18,r19,r23
;	    x	<- Zeiger auf ASCII-Puffer
;	    y	<- Zeiger auf Datensatz im RAM-Puffer
;	    r19	-> Anzahl der Bytes im Puffer
;
xtrdat:	ldi	r23,8			;8 Wertepaare bearbeiten
xtrd10:	ld	r16,y+			;L-Byte von Tempwert 1 holen
	ld	r19,y+			;L-Byte von Tempwert 2 holen
	ld	r17,y			;H-Bytes von Tempwert 1/2 holen
	andi	r17,0x0f		;Bits von Tempwert 1H filtern
	cpi	r17,0x08		;gltiger Wert?
	breq	xtrd30			;nein -> Ausgabe berspringen
	brcs	xtrd20			;Wert negativ? nein -> weiter
	ori	r17,0xf0		;sonst Vorzeichen setzen
xtrd20:	rcall	temasc			;Temperatur in ASCII wandeln
	st	x+,r4			;Hunderter ausgeben
	st	x+,r5			;Zehner ausgeben
	st	x+,r6			;Einer ausgeben
	ldi	r16,','			;Komma holen
	st	x+,r16			;Komma ausgeben
	st	x+,r7			;Zehntel ausgeben
xtrd30:	ldi	r16,';'			;Feldtrennzeichen holen
	st	x+,r16			;Feldtrennzeichen ausgeben
	mov	r16,r19			;L-Byte von Tempwert 2 holen
	ld	r17,y+			;H-Bytes von Tempwert 1/2 holen
	swap	r17			;H-Byte an richtige Position
	andi	r17,0x0f		;Bits von Tempwert 2H filtern
	cpi	r17,0x08		;gltiger Wert?
	breq	xtrd50			;nein -> Ausgabe berspringen
	brcs	xtrd40			;Wert negativ? nein -> weiter
	ori	r17,0xf0		;sonst Vorzeichen setzen
xtrd40:	rcall	temasc			;Temperatur in ASCII wandeln
	st	x+,r4			;Hunderter ausgeben
	st	x+,r5			;Zehner ausgeben
	st	x+,r6			;Einer ausgeben
	ldi	r16,','			;Komma holen
	st	x+,r16			;Komma ausgeben
	st	x+,r7			;Zehntel ausgeben
xtrd50:	cpi	r23,1			;letztes Wertepaar?
	breq	xtrd60			;ja -> kein Feldtrennzeichen
	ldi	r16,';'			;Feldtrennzeichen holen
	st	x+,r16			;Feldtrennzeichen ausgeben
xtrd60:	dec	r23			;alle Wertepaare bearbeitet?
	brne	xtrd10			;nein -> Schleife
;
; Unterprogramm zum Ermitteln der Byteanzahl im ASCII-Puffer
; Register: r16,r17
;	    x   <- Zeiger auf aktuelle Position im ASCII-Puffer
;	    r19 -> Anzahl der Bytes im Puffer
;
xtbyte:	ldi	r16,low(ascbuf)
	ldi	r17,high(ascbuf)	;Anfangsadresse ASCII-Puffer
	sub	xl,r16			;Differenz zur aktuellen
	sbc	xh,r17			;Adresse berechnen = Anzahl
	mov	r19,xl			;Byteanzahl kopieren
	ret
;
; Unterprogramm zum Extrahieren einer Zeichenkette aus dem Flash-ROM in
; den ASCII-Puffer fr die Datenausgabe zum PC
; Register: r16,z
;	    r17 <- Stringnummer (0x00-0xfd)
;	    x   <- Zeiger auf ASCII-Puffer
;
xtrfst:	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,0			;Sprache englisch?
	brts	xtrf10			;nein -> weiter
	ldi	zl,byte3(stengl*2)	;sonst
	out	rampz,zl
	ldi	zl,low(stengl*2)
	ldi	zh,high(stengl*2)	;Zeiger auf englische Texte
	rjmp	xtrf20
;
xtrf10:	ldi	zl,byte3(stgerm*2)
	out	rampz,zl
	ldi	zl,low(stgerm*2)
	ldi	zh,high(stgerm*2)	;Zeiger auf deutsche Texte
;
xtrf20:	elpm	r16,z+			;Zeichen holen
	cpi	r16,0xfe		;String-Kennung gefunden?
	brne	xtrf20			;nein -> weiter suchen
	elpm	r16,z+			;Zeichen holen
	cp	r16,r17			;Zeichenkette gefunden?
	breq	xtrf40			;ja -> ausgeben
	cpi	r16,0xfe		;Endezeichen gefunden?
	brne	xtrf20			;nein -> weitersuchen
xtrf30:	ret				;Ende
;
xtrf40:	elpm	r16,z+			;Zeichen holen
	cpi	r16,0xff		;Endezeichen?
	breq	xtrf30			;ja -> Ende
	cpi	r16,0x80		;Zeichencode>0x7f?
	brcs	xtrf50			;nein -> Zeichen in Puffer
	cpi	r16,0x90		;Zeichencode>0x8f?
	brcc	xtrf40			;ja -> berspringen
	rcall	xtruml			;sonst Umlaute konvertieren
xtrf50:	st	x+,r16			;Zeichen in Puffer legen
	rjmp	xtrf40			;Schleife
;
; Unterprogramm zum Extrahieren einer Zeichenkette aus dem EEPROM in
; den ASCII-Puffer fr die Datenausgabe zum PC
; Register: r18
;	    r16,r17 <- EEPROM-Adresse der Texte (etemtx, ealmtx)
;	    r19	   <- Lnge der Zeichenkette
;	    x	   <- Zeiger auf ASCII-Puffer
;
xtrest:	call	eeread			;Zeichen aus EEPROM holen
	cpi	r18,0x20		;Leerzeichen (=Ende)?
	brne	xtre10			;nein -> weiter
	ret				;sonst Ende
;
xtre10:	cpi	r18,0x80		;Zeichencode>0x7f?
	brcs	xtre20			;nein -> Zeichen in Puffer
	cpi	r18,0x90		;Zeichencode>0x8f?
	brcc	xtre30			;ja -> berspringen
	push	r16			;sonst EEPROM-Adresse sichern
	mov	r16,r18			;Zeichencode kopieren
	rcall	xtruml			;Umlaute konvertieren
	mov	r18,r16			;Zeichen kopieren
	pop	r16			;EEPROM-Adresse restaurieren
xtre20:	st	x+,r18			;Zeichen in Puffer legen
xtre30:	dec	r19			;alle Zeichen ausgegeben?
	brne	xtrest			;nein -> Schleife
	ret
;
; Unterprogramm zum Konvertieren der Umlaute in 2 Zeichen fr die
; Datenausgabe zum PC
; Register: r4
;	    r16 <- Zeichencode
;	    x   <- Zeiger auf ASCII-Puffer
;
xtruml:	push	zl
	push	zh
	in	zl,rampz
	push	zl			;Register sichern
	ldi	zl,byte3(umltab*2)
	mov	r4,zl			;B3 Wert zwischenspeichern
	ldi	zl,low(umltab*2)
	ldi	zh,high(umltab*2)	;Zeiger auf Umlaut-Tabelle
	subi	r16,0x80		;Zeichencode als Offset
	add	r16,r16			;Offset verdoppeln (2 Zeichen)
	add	zl,r16			;zur Tabellenadresse addieren
	clr	r16
	adc	zh,r16			;bertrag addieren (H)
	adc	r4,r16			;bertrag addieren (B3)
	out	rampz,r4
	elpm	r16,z+			;erstes Zeichen holen
	st	x+,r16			;Zeichen in Puffer legen
	elpm	r16,z+			;zweites Zeichen holen
	pop	zl
	out	rampz,zl
	pop	zh
	pop	zl			;Register wieder herstellen
	ret
;
; Unterprogramm zum Extrahieren der Zeitdaten aus einem EEPROM-Daten-
; satz und Ausgabe auf dem LCD
; Register: r0,r1,r13,r14,r15,r16,r17,r18,r19,r23,r24,zl,zh,t
;	    r20 <- X-Koordinate (0-239)
;	    r21 <- Y-Koordinate (0-127)
;
xtdtim:	lds	r18,xedbuf+2		;komprimierte Zeit Byte 2 holen
	andi	r18,0x1f		;nicht relevante Bits filtern
	rcall	bytasc			;in ASCII-Wert wandeln
	mov	r13,r16			;Tag Einer sichern
	mov	r22,r17			;Tag Zehner kopieren
	rcall	asclcd			;ausgeben
	mov	r22,r13			;gesicherten Tag Einer holen
	rcall	asclcd			;ausgeben
;
	ldi	r22,'.'			;Trennzeichen
	rcall	asclcd			;ausgeben
	lds	r18,xedbuf+3		;komprimierte Zeit Byte 3 holen
	bst	r18,0			;LSB sichern, gehrt zum Monat
	lds	r18,xedbuf+2		;komprimierte Zeit Byte 2 holen
	ror	r18			;Monat nach rechts schieben
	bld	r18,7			;LSB von Byte 3 holen, Monat an
	swap	r18			;richtige Position schieben
	andi	r18,0x0f		;Nicht relevante Bits filtern
	rcall	bytasc			;in ASCII-Wert wandeln
	mov	r13,r16			;Monat Einer sichern
	mov	r22,r17			;Monat Zehner kopieren
	rcall	asclcd			;ausgeben
	mov	r22,r13			;gesicherten Monat Einer holen
	rcall	asclcd			;ausgeben
;
	ldi	r22,'.'			;Trennzeichen
	rcall	asclcd			;ausgeben
	lds	r18,xedbuf+3		;komprimierte Zeit Byte 3 holen
	lsr	r18			;Jahr an Position schieben
	rcall	bytasc			;in ASCII-Wert wandeln
	mov	r13,r16			;Jahr Einer sichern
	mov	r22,r17			;Jahr Zehner kopieren
	rcall	asclcd			;ausgeben
	mov	r22,r13			;gesicherten Jahr Einer holen
	rcall	asclcd			;ausgeben
;
	ldi	r22,0xa0		;schmales Leerzeichen
	rcall	asclcd			;ausgeben
	lds	r18,xedbuf+1		;komprimierte Zeit Byte 1 holen
	rcall	bytasc			;in ASCII-Wert wandeln
	mov	r13,r16			;Stunde Einer sichern
	mov	r22,r17			;Stunde Zehner kopieren
	rcall	asclcd			;ausgeben
	mov	r22,r13			;gesicherten Stunde Einer holen
	rcall	asclcd			;ausgeben
	ldi	r22,':'			;Trennzeichen holen
	rcall	asclcd			;ausgeben
	lds	r18,xedbuf		;komprimierte Zeit Byte 0 holen
	rcall	bytasc			;in ASCII-Wert wandeln
	mov	r13,r16			;Minute Einer sichern
	mov	r22,r17			;Minute Zehner kopieren
	rcall	asclcd			;ausgeben
	mov	r22,r13			;gesicherten Minute Einer holen
	rcall	asclcd			;ausgeben
	ldi	r22,':'			;Trennzeichen holen
	rcall	asclcd			;ausgeben
;
	lds	r18,xedbuf+4		;komprimierte Sekunden holen
	rcall	bytasc			;in ASCII-Wert wandeln
	mov	r13,r16			;Sekunde Einer sichern
	mov	r22,r17			;Sekunde Zehner kopieren
	rcall	asclcd			;ausgeben
	mov	r22,r13			;gesicherten Sekunde Einer hol.
	rcall	asclcd			;ausgeben
	ret
;
; Unterprogramm zur Ausgabe einer Zeichenkette aus dem EEPROM zum LCD
; Register: r0,r1,r14,r15,r18,r19,r23,r24,zl,zh,t
;	    r16,r17 <- EEPROM-Adresse der Zeichenkette
;	    r19     <- Lnge der Zeichenkette
;	    r20     <- X-Koordinate (0-239)
;	    r21     <- Y-Koordinate (0-127)
;
eeplcd:	call	eeread			;Zeichen aus EEPROM holen
	cpi	r18,0x21		;Zeichen <= Leerzeichen?
	brcs	eepl10			;ja -> Ende
	mov	r22,r18			;sonst Zeichen kopieren
	push	r16
	push	r17			;EEPROM-Adresse sichern
	push	r19			;Zhler sichern
	call	asclcd			;Zeichen ausgeben
	pop	r19			;Zhler wieder herstellen
	pop	r17
	pop	r16			;EERPOM-Adresse restaurieren
	dec	r19			;alle Zeichen ausgegeben?
	brne	eeplcd			;nein -> Schleife
eepl10:	ret
;
; Unterprogramm zur Ermittlung der Pixel-Lnge einer EEPROM-Zeichen-
; kette
; Register: r0,r1,r18,z
;	    r16,r17 <- EEPROM-Adresse der Zeichenkette
;	    r19     <- Lnge der Zeichenkette
;	    r20     -> Stringlnge in Pixel
;
eeplen:	clr	r20			;Lnge Anfangswert=0 setzen
eepl20:	call	eeread			;Zeichen aus EEPROM holen
	cpi	r18,0x20		;Leerzeichen gefunden?
	brne	eepl30			;nein -> weiter
	ret				;sonst Ende
;
eepl30:	subi	r18,0x20		;Korrektur fr Tabelle
	cpi	r18,0xe0		;Bereich berschritten?
	brcs	eepl40			;nein -> weiter
	ret				;sonst Ende (Fehler)
;
eepl40:	push	r16
	push	r17			;EEPROM-Adresse sichern
	ldi	r16,12			;Tabellen-Offset
	mul	r18,r16			;Tabellenplatz berechnen
	ldi	zl,low(chrset*2)	;Zeiger auf Zeichentabelle L
	ldi	zh,high(chrset*2)	;Zeiger auf Zeichentabelle H
	ldi	r16,byte3(chrset*2)	;Zeiger auf Zeichentabelle B3
	clr	r17
	add	zl,r0			;Tabellenadresse L berechnen
	adc	zh,r1			;Tabellenadresse H berechnen
	adc	r16,r17			;Tabellenadresse B3 berechnen
	out	rampz,r16
	elpm	r16,z			;Zeichenbreite holen
	inc	r16			;Zeichenbreite korrigieren
	add	r20,r16			;zur Gesamtlnge addieren
	pop	r17
	pop	r16			;EEPROM-Adresse holen
	dec	r19			;alle Zeichen bearbeitet?
	brne	eepl20			;nein -> Schleife
	ret				;sonst Ende
;
; Unterprogramm zum Kopieren eines EEPROM-Textes in den Texteingabe-
; puffer
; Register: r0,r1,r20,x
;	    r16 <- EEPROM-Adresse L
;	    r17 <- EEPROM-Adresse H
;	    r18 <- Textnummer in Tabelle (Sensorn./Alarmn./Preset-N.)
;	    r19 <- Maximale Textlnge (8, 11 oder 16 Zeichen)
;
cpeeab:	sts	ascibm,r19		;maximale Zeichenanzahl setzen
	mul	r18,r19			;Tabellenoffset ermitteln
	add	r16,r0			;Offset zur Adresse L addieren
	adc	r17,r1			;Offset zur Adresse H addieren
	ldi	xl,low(ascibf)
	ldi	xh,high(ascibf)		;Zeiger auf Texteingabepuffer
	clr	r20			;aktuelle Position lschen
cpee10:	call	eeread			;Zeichen aus EEPROM holen
	cpi	r18,0x20		;String-Ende?
	breq	cpee20			;ja -> Ende, sonst Zeichen im
	st	x+,r18			;Texteingabepuffer speichern
	inc	r20			;Position erhhen
	cp	r20,r19			;letzte Position erreicht?
	brne	cpee10			;nein -> Schleife
cpee20:	sts	ascibp,r20		;aktuelle Position speichern
	ret
;
; Unterprogramm zum Kopieren des Texteingabepuffers in das EEPROM
; Register: r0,r1,r20,r21,x
;	    r16 <- EEPROM-Adresse L
;	    r17 <- EEPROM-Adresse H
;	    r18 <- Textnummer in Tabelle (Sensorn./Alarmn./Preset-N.)
;	    r19 <- Maximale Textlnge (8, 11 oder 16 Zeichen)
;
cpabee:	mul	r18,r19			;Tabellenoffset ermitteln
	add	r16,r0			;Offset zur Adresse L addieren
	adc	r17,r1			;Offset zur Adresse H addieren
	ldi	xl,low(ascibf)
	ldi	xh,high(ascibf)		;Zeiger auf Texteingabepuffer
	lds	r21,ascibp		;aktuelle Pufferposition holen
	clr	r20			;Positionszhler lschen
cpab10:	cp	r20,r21			;aktuelle Position erreicht?
	breq	cpab20			;ja -> weiter
	ld	r18,x+			;Zeichen aus Eingabepuffer hol.
	call	eewrit			;im EEPROM sichern
	inc	r20			;Positionszhler erhhen
	rjmp	cpab10			;Schleife
cpab20:	cp	r20,r19			;letzte Position erreicht?
	brcc	cpab30			;ja -> Ende
	ldi	r18,' '			;sonst Leerzeichen
	call	eewrit			;im EEPROM sichern
	inc	r20			;Positionszhler erhhen
	rjmp	cpab20			;Schleife
cpab30:	ret
;
; Unterprogramm zur Ausgabe des Texteingabepuffers auf dem LCD
; Register: r0,r1,r14,r15,r16,r17,r18,r19,r23,r24,zl,zh,t
;	    r20 <- X-Koordinate (0-239)
;	    r21 <- Y-Koordinate (0-127)
;
txblcd:	ldi	zl,low(ascibf)
	ldi	zh,high(ascibf)		;Zeiger auf Texteingabepuffer
	lds	r17,ascibp		;aktuelle Eingabeposition holen
	tst	r17			;Puffer leer?
	breq	txbl20			;ja -> berspringen
	clr	r16			;Positionszhler lschen
txbl10:	ld	r22,z+			;Zeichen holen
	push	r16			;Positionszhler sichern
	push	zl
	push	zh			;Zeiger sichern
	rcall	asclcd			;Zeichen ausgeben
	pop	zh
	pop	zl			;Zeiger restaurieren
	pop	r16			;Positionszhler restaurieren
	inc	r16			;Position erhhen
	lds	r17,ascibp		;aktuelle Eingabeposition holen
	cp	r16,r17			;Eingabeposition erreicht?
	brne	txbl10			;nein -> Schleife
	lds	r17,ascibm		;maximale Zeichenanzahl holen
	cp	r16,r17			;letzte Position erreicht?
	brcc	txbl30			;ja -> senkrechter Cursor
txbl20:	ldi	r22,0xac		;sonst Cursorsymbol
	rcall	asclcd			;ausgeben
	ret
txbl30:	ldi	r22,0x7f		;Endezeichen laden
	ldi	r16,0x01
	mov	txflag,r16		;Textflag invers setzen
	rcall	asclcd			;ausgeben
	clr	txflag			;Textflag lschen
	ret
;
; Unterprogramm zur Ermittlung der Lnge des Texteingabepuffers
; Register: r0,r1,r16,r17,r19,zl,zh
;	    r18 -> Pufferlnge (Pixel)
;
txblen:	ldi	zl,low(ascibf)
	ldi	zh,high(ascibf)		;Zeiger auf Texteingabepuffer
	clr	r19			;Positionszhler lschen
	clr	r18			;Lngenzhler lschen
	lds	r16,ascibp		;aktuelle Eingabeposition holen
	tst	r16			;Puffer leer?
	brne	txbl50			;nein -> weiter
	ret				;sonst Ende
;
txbl50:	ld	r16,z+			;Zeichen holen
	subi	r16,0x20		;Korrektur fr Tabelle
	cpi	r16,0xe0		;Bereich berschritten?
	brcs	txbl60			;nein -> weiter
	ret				;sonst Ende (Fehler)
;
txbl60:	ldi	r17,12			;Tabellen-Offset
	mul	r16,r17			;Tabellenplatz berechnen
	push	zl
	push	zh			;Puffer-Zeiger sichern
	clr	r17
	ldi	zl,low(chrset*2)	;Zeiger auf Zeichentabelle L
	ldi	zh,high(chrset*2)	;Zeiger auf Zeichentabelle H
	ldi	r16,byte3(chrset*2)	;Zeiger auf Zeichenzabelle B3
	add	zl,r0			;Tabellenadresse L berechnen
	adc	zh,r1			;Tabellenadresse H berechnen
	adc	r16,r17
	out	rampz,r16
	elpm	r16,z			;Zeichenbreite holen
	pop	zh
	pop	zl			;Puffer-Zeiger restaurieren
	inc	r16			;Zeichenbreite korrigieren
	add	r18,r16			;zur Gesamtlnge addieren
;
	inc	r19			;Position erhhen
	lds	r16,ascibp		;aktuelle Eingabeposition holen
	cp	r16,r19			;Eingabeposition erreicht?
	brne	txbl50			;nein -> Schleife
	ret
;
; Unterprogramm zum Senden des ASCII-Pufferinhaltes + CR/LF zum PC
; Register: r16,z
;	    r19 <- Anzahl der zu sendenden Bytes
;
senasc:	tst	r19			;sind Zeichen zu senden?
	breq	sencrl			;nein -> nur CR/LF senden
	ldi	zl,low(ascbuf)
	ldi	zh,high(ascbuf)		;Zeiger auf ASCII-Puffer
sena10:	ld	r16,z+			;Zeichen aus Puffer holen
	call	put0ch			;Zeichen senden
	dec	r19			;alle Zeichen gesendet?
	brne	sena10			;nein -> Schleife
;
; Unterprogramm zum Senden von CR/LF zum PC
; Register: r16
;
sencrl:	ldi	r16,cr			;CR laden
	call	put0ch			;CR ausgeben
	ldi	r16,lf			;LF laden
	call	put0ch			;LF ausgeben
	ret
;
; Unterprogramm zum Umwandeln eines Bytes in 2 Dezimalstellen
; Register: r16 -> Ergebnis L
;	    r17 -> Ergebnis H
;	    r18 <- Byte
;
bytasc:	cpi	r18,100			;Wert<100?
	brcs	byta10			;ja -> weiter
	subi	r18,100			;sonst 100 subtrahieren
	rjmp	bytasc
byta10:	clr	r17			;H-Byte Anfangswert
byta20:	cpi	r18,10			;Wert<10?
	brcs	byta30			;ja -> weiter
	subi	r18,10			;sonst 10 subtrahieren
	inc	r17			;H-Wert erhhen
	rjmp	byta20
byta30:	mov	r16,r18			;L-Wert kopieren
	ori	r17,0x30		;H-Wert in ASCII wandeln
	ori	r16,0x30		;L-Wert in ASCII wandeln
	ret
;
; Unterprogramm zum Umwandeln von 2 Bytes in 4 Dezimalstellen (0-9999)
; Register: r4  -> Ergebnis Stelle 1
;	    r5  -> Ergebnis Stelle 2
;	    r6  -> Ergebnis Stelle 3
;	    r7  -> Ergebnis Stelle 4
;	    r16 <- Byte L
;	    r17 <- Byte H
;
bytas4:	clr	r4			;Stelle 1 Anfangswert
	movw	r6,r16			;Zahl kopieren
	ldi	r16,low(9999)
	ldi	r17,high(9999)		;Vergleichswert laden
	sub	r16,r6
	sbc	r17,r7			;Zahl<10000?
	brcc	byta50			;ja -> weiter
	clr	r4
	clr	r5
	clr	r6
	clr	r7			;sonst alle Stellen lschen
	rjmp	byta85			;in ASCII wandeln
;
byta50:	clr	r4			;Stelle 1 lschen
byta55:	ldi	r16,low(999)
	ldi	r17,high(999)		;Vergleichswert laden
	sub	r16,r6
	sbc	r17,r7			;Zahl<1000?
	brcc	byta60			;ja -> weiter
	ldi	r16,low(1000)
	ldi	r17,high(1000)		;sonst 1000
	sub	r6,r16
	sbc	r7,r17			;subtrahieren
	inc	r4			;Stelle 1 erhhen
	rjmp	byta55			;Schleife
;
byta60:	clr	r5			;Stelle 2 lschen
byta65:	ldi	r16,99
	clr	r17			;Vergleichswert laden
	sub	r16,r6
	sbc	r17,r7			;Zahl<100?
	brcc	byta70			;ja -> weiter
	ldi	r16,100
	clr	r17			;sonst 100
	sub	r6,r16
	sbc	r7,r17			;subtrahieren
	inc	r5			;Stelle 2 erhhen
	rjmp	byta65			;Schleife
;
byta70:	mov	r16,r6			;Restwert holen
	clr	r6			;Stelle 3 lschen
byta75:	cpi	r16,10			;Zahl<10?
	brcs	byta80			;ja -> weiter
	subi	r16,10			;sonst 10 subtrahieren
	inc	r6			;Stelle 3 erhhen
	rjmp	byta75			;Schleife
;
byta80:	mov	r7,r16			;Stelle 4 kopieren
byta85:	ldi	r16,0x30
	or	r4,r16			;Stelle 1 in ASCII wandeln
	or	r5,r16			;Stelle 2 in ASCII wandeln
	or	r6,r16			;Stelle 3 in ASCII wandeln
	or	r7,r16			;Stelle 4 in ASCII wandeln
	ret
;
; Unterprogramm zum Umwandeln eines Bytes in 2 Hexadezimalstellen
; Register: r16 -> Ergebnis L
;	    r17 -> Ergebnis H
;	    r18 <- Byte
;
hexasc:	mov	r16,r18			;Byte in L-Wert kopieren
	mov	r17,r18			;Byte in H-Wert kopieren
	ldi	r18,7			;fr Umrechnung in Hex-Wert
	andi	r17,0xf0		;oberes Nibble filtern
	swap	r17			;Nibbles tauschen
	ori	r17,0x30		;in ASCII wandeln
	cpi	r17,0x3a		;Wert>"9"?
	brcs	hexa10			;nein -> weiter
	add	r17,r18			;sonst in Buchstaben wandeln
hexa10:	andi	r16,0x0f		;unteres Nibble filtern
	ori	r16,0x30		;in ASCII wandeln
	cpi	r16,0x3a		;Wert>"9"?
	brcs	hexa20			;nein -> weiter
	add	r16,r18			;sonst in Buchstaben wandeln
hexa20:	ret
;
; Unterprogramm zur Wandlung eines Temperaturwertes ASCII->Hex
; Register: r14,r15,r18
;	    rxdat1 <- ASCII-Wert
;	    r16    -> Ergebnis Hexwert L \ Ergebnis in
;	    r17    -> Ergebnis Hexwert H / Zehntel Grad
;
temhex:	clr	r14
	clr	r15			;Anfangswert lschen
	lds	r18,rxdat1+2		;Hunderterstelle holen
	cpi	r18,'1'			;Hunderterstelle belegt?
	brne	temh10			;nein -> weiter
	ldi	r16,low(1000)		;sonst 1000 addieren
	ldi	r17,high(1000)		;(wegen Zehntel)
	add	r14,r16			;addieren
	adc	r15,r17			;bertrag addieren
temh10:	lds	r18,rxdat1+3		;Zehnerstelle holen
	cpi	r18,'-'			;Minuszeichen?
	breq	temh30			;ja -> weiter
	cpi	r18,' '			;Leerzeichen?
	breq	temh30			;ja -> weiter
	andi	r18,0x0f		;sonst Wandlung in Ziffer
	ldi	r16,100			;sonst 100 addieren
	clr	r17			;(wegen Zehntel)
temh20:	tst	r18			;Zhler=0?
	breq	temh30			;ja -> weiter
	add	r14,r16			;sonst addieren
	adc	r15,r17			;bertrag addieren
	dec	r18			;Zhler vermindern
	rjmp	temh20			;Schleife
temh30:	lds	r18,rxdat1+4		;Einerstelle holen
	andi	r18,0x0f		;Wandlung in Ziffer
	ldi	r16,10			;10 addieren
	clr	r17			;(wegen Zehntel)
temh40:	tst	r18			;Zhler=0?
	breq	temh50			;ja -> weiter
	add	r14,r16			;sonst addieren
	adc	r15,r17			;bertrag addieren
	dec	r18			;Zhler vermindern
	rjmp	temh40			;Schleife
temh50:	lds	r18,rxdat1+6		;Zehntelstelle holen
	andi	r18,0x0f		;Wandlung in Ziffer
	ldi	r16,1			;1 addieren
	clr	r17			;(wegen Zehntel)
temh60:	tst	r18			;Zhler=0?
	breq	temh70			;ja -> weiter
	add	r14,r16			;sonst addieren
	adc	r15,r17			;bertrag addieren
	dec	r18			;Zhler vermindern
	rjmp	temh60			;Schleife
temh70:	lds	r18,rxdat1+2		;Hunderterstelle holen
	cpi	r18,'-'			;Minuszeichen?
	breq	temh80			;ja -> Zweierkomplement bilden
	lds	r18,rxdat1+3		;Zehnerstelle holen
	cpi	r18,'-'			;Minuszeichen?
	breq	temh80			;ja -> Zweierkomplement bilden
	movw	r16,r14			;sonst Ergebnis kopieren
	ret
temh80:	clr	r16			;Umwandlung ins 16-Bit-
	clr	r17			;Zweierkomplement
	sub	r16,r14			;0x10000 - Ergebnis
	sbc	r17,r15			;bertrag
	ret
;
; Unterprogramm zur Wandlung eines Temperaturwertes Hex->ASCII
; Register: r13,r14,r15,r18
;	    r16   <- Hexwert L \ Temperaturwert
;	    r17   <- Hexwert H / in Zehntel Grad
;	    r4-r7 -> Ergebnis in ASCII (r4=MSB, r7=LSB)
;
temasc:	ldi	r18,' '			;Ausgabedaten initialisieren
	mov	r5,r18
	mov	r6,r18
	mov	r7,r18
	ldi	r18,'0'			;Tausender initialisieren
	mov	r13,r17			;Vorzeichen sichern
	bst	r17,7			;Wert negativ?
	brtc	tema10			;nein -> bearbeiten
	movw	r14,r16			;Eingabewert kopieren
	clr	r16			;Umwandlung ins 16-Bit-
	clr	r17			;Zweierkomplement
	sub	r16,r14			;0x10000 - Ergebnis
	sbc	r17,r15			;bertrag
tema10:	movw	r14,r16			;Eingabewert kopieren
tema20:	ldi	r16,low(1000)		;Hunderterstelle ermitteln
	ldi	r17,high(1000)		;(1000 wegen Zehntel benutzen)
	sub	r14,r16			;subtrahieren
	sbc	r15,r17			;bertrag subtrahieren
	brcs	tema30			;zuviel subtrahiert? ja->weiter
	inc	r18			;sonst Tausender erhhen
	rjmp	tema20			;Schleife
tema30:	cpi	r18,'0'			;Tausender=0?
	brne	tema40			;nein -> weiter
	ldi	r18,' '			;sonst durch Leerzei. ersetzen
tema40:	mov	r4,r18			;an Tausenderstelle kopieren
	add	r14,r16			;Originalwert
	adc	r15,r17			;wieder herstellen
	ldi	r16,100			;Zehnerstelle ermitteln
	clr	r17			;(100 wegen Zehntel benutzen)
	clr	r18			;Zhler lschen
tema50:	sub	r14,r16			;100 subtrahieren
	sbc	r15,r17			;bertrag subtrahieren
	brcs	tema60			;zuviel subtrahiert? ja->weiter
	inc	r18			;sonst Zhler erhhen und
	rjmp	tema50			;nochmals 100 subtrahieren
tema60:	add	r14,r16			;Originalwert
	adc	r15,r17			;wieder herstellen
	ori	r18,0x30		;Zhler in ASCII wandeln und
	mov	r5,r18			;als ASCII Zehner speichern
	cpi	r18,'0'			;war Zehner=0?
	brne	tema70			;nein -> weiter zum Einer
	mov	r18,r4			;ASCII Hunderter nochmals holen
	cpi	r18,' '			;war Hunderter ebenfalls=0?
	brne	tema70			;nein -> weiter zum Einer
	ldi	r18,' '			;sonst Leerzeichen laden
	mov	r5,r18			;und an ASCII Zehner setzen
tema70:	ldi	r16,10			;Einerstelle ermitteln
	clr	r18			;Zhler lschen
tema80:	cp	r14,r16			;Einerstelle erreicht?
	brcs	tema90			;ja -> weiter
	sub	r14,r16			;sonst 10 subtrahieren
	inc	r18			;Zhler erhhen
	rjmp	tema80			;Schleife
tema90:	ori	r18,0x30		;Zhler in ASCII wandeln und
	mov	r6,r18			;als ASCII Einer speichern
	ldi	r16,1			;Zehntelstelle ermitteln
	clr	r18			;Zhler lschen
tem100:	cp	r14,r16			;Zehntelstelle erreicht?
	brcs	tem110			;ja -> weiter
	sub	r14,r16			;sonst 1 subtrahieren
	inc	r18			;Zhler erhhen
	rjmp	tem100			;Schleife
tem110:	ori	r18,0x30		;Zhler in ASCII wandeln und
	mov	r7,r18			;als ASCII Zehntel speichern
	bst	r13,7			;war Hexwert negativ?
	brtc	tem120			;nein -> Ende
	ldi	r16,'-'			;Minuszeichen laden
	mov	r17,r5			;Zehner holen
	cpi	r17,' '			;Platz fr Minuszeichen?
	brne	tem130			;nein -> weiter
	mov	r5,r16			;Minuszeichen anstatt Zehner
tem120:	ret				;einsetzen
tem130:	mov	r4,r16			;Minuszeichen anstatt Hunderter
	ret				;einsetzen
;
; Unterprogramm zur Wandlung eines Zeitwertes Hex->ASCII
; Register: r14,r15,r18
;	    r16   <- Hexwert L \ Zeit als
;	    r17   <- Hexwert H / Minutenwert
;	    r4-r7 -> Ergebnis in ASCII (r4=MSB, r7=LSB)
;
timasc:	movw	r14,r16			;Eingabewert kopieren
	ldi	r16,low(60)
	ldi	r17,high(60)		;Stunden und Minuten trennen
	clr	r18			;Zhler lschen
tima10:	sub	r14,r16			;60 subtrahieren
	sbc	r15,r17			;bertrag subtrahieren
	brcs	tima20			;zuviel subtrahiert? ja->weiter
	inc	r18			;sonst Zhler erhhen und
	rjmp	tima10			;nochmals 60 subtrahieren
tima20:	add	r14,r16			;Originalwert
	adc	r15,r17			;wieder herstellen
	mov	r6,r14			;Minuten zwischenspeichern
	rcall	bytasc			;Byte in ASCII Zeichen wandeln
	cpi	r17,'0'			;Stunden Zehner= 0?
	brne	tima30			;nein -> weiter
	ldi	r17,' '			;sonst Leerzeichen benutzen
tima30:	mov	r4,r17			;Stunden Zehner speichern
	mov	r5,r16			;Stunden Einer speichern
	mov	r18,r6			;Minutenwert holen
	rcall	bytasc			;Byte in ASCII Zeichen wandeln
	mov	r6,r17			;Minuten Zehner speichern
	mov	r7,r16			;Minuten Einer speichern
	ret
;
; Unterprogramm zur Ausgabe eines 3-Byte EEPROM-Pointers auf das LCD
; Register: r0,r1,r14,r15,r16,r17,r18,r19,r23,r24,zl,zh,t
;	    r20 <- X-Koordinate (0-239)
;	    r21 <- Y-Koordinate (0-127)
;	    y   <- Zeiger auf Pointer im RAM
;
poilcd:	ldd	r18,y+2			;Pointer1 Byte3 holen
	rcall	hexasc			;in ASCII-Zeichen wandeln
	mov	r22,r16			;erste Stelle kopieren
	rcall	asclcd			;ausgeben
	ldd	r18,y+1			;Pointer1 H-Byte holen
	rcall	hexasc			;in ASCII-Zeichen wandeln
	mov	r13,r16			;dritte Stelle sichern
	mov	r22,r17			;zweite Stelle kopieren
	rcall	asclcd			;ausgeben
	mov	r22,r13			;gesicherte dritte Stelle holen
	rcall	asclcd			;ausgeben
	ld	r18,y			;Pointer1 L-Byte holen
	rcall	hexasc			;in ASCII-Zeichen wandeln
	mov	r13,r16			;fnfte Stelle sichern
	mov	r22,r17			;vierte Stelle kopieren
	rcall	asclcd			;ausgeben
	mov	r22,r13			;gesicherte fnfte Stelle holen
	rcall	asclcd			;ausgeben
	ret
;
; Unterprogramm zur Ausgabe der Zeichenkette "Bitte warten" in der
; Menzeile (zentriert)
; Register: r0,r1,r14-r24,zl,zh,t
;
strwai:	ldi	r22,0x43		;String "Bitte warten"
	rcall	strlen			;Lnge ermitteln
	asr	r18			;Lnge halbieren
	ldi	r20,119			;X-Mittenposition laden
	sub	r20,r18			;Text-Anfangsposition ermitteln
	ldi	r18,2			;X-Anfangsposition Menzeile
	ldi	r19,115			;Y-Anfangsposition Menzeile
	ldi	r21,126			;Y-Endposition zum Lschen
	rcall	clrwnd			;Bereich vor dem Text lschen
	ldi	r21,115			;Y-Textposition
	ldi	r22,0x43		;String "Bitte warten"
	rcall	strlcd			;ausgeben
	mov	r18,r20			;X-Position Textende kopieren
	ldi	r19,115			;Y-Position Textende
	ldi	r20,238			;X-Endposition Menzeile
	ldi	r21,126			;Y-Endposition Menzeile
	rcall	clrwnd			;Bereich hinter dem Text lsch.
	ret
;
; Unterprogramm zur Ausgabe einer Zeichenkette aus dem Flash-ROM
; Register: r0,r1,r14,r15,r16,r17,r18,r19,r23,r24,zl,zh,t
;	    r20 <- X-Koordinate (0-239)
;	    r21 <- Y-Koordinate (0-127)
;	    r22 <- Stringnummer (0x00-0xfd)
;
strlcd:	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,0			;Sprache englisch?
	brts	str010			;nein -> weiter
	ldi	zl,byte3(stengl*2)	;sonst
	out	rampz,zl
	ldi	zl,low(stengl*2)
	ldi	zh,high(stengl*2)	;Zeiger auf englische Texte
	rjmp	str020
;
str010:	ldi	zl,byte3(stgerm*2)
	out	rampz,zl
	ldi	zl,low(stgerm*2)
	ldi	zh,high(stgerm*2)	;Zeiger auf deutsche Texte
;
str020:	elpm	r16,z+			;Zeichen holen
	cpi	r16,0xfe		;String-Kennung gefunden?
	brne	str020			;nein -> weiter suchen
	elpm	r16,z+			;Zeichen holen
	cp	r16,r22			;Zeichenkette gefunden?
	breq	str030			;ja -> ausgeben
	cpi	r16,0xfe		;Endezeichen gefunden?
	brne	str020			;nein -> weiter suchen
	ret				;sonst Ende
;
str030:	elpm	r22,z+			;Zeichen holen
	cpi	r22,0xff		;Endezeichen?
	brne	str040			;nein -> weiter
	ldi	r22,0x7f		;sonst Endezeichen
	rcall	asclcd			;Zeichen ausgeben
	ret
str040:	cpi	r22,' '			;Leerzeichen?
	brne	str050			;nein -> ausgeben
	ldi	r22,0xa0		;sonst schmales Leerzeichen
str050:	push	zl
	push	zh
	in	zl,rampz
	push	zl			;Flash-Zeiger sichern
	rcall	asclcd			;Zeichen ausgeben
	pop	zl
	out	rampz,zl
	pop	zh
	pop	zl			;Flashzeiger wieder herstellen
	rjmp	str030			;nchstes Zeichen bearbeiten
;
; Unterprogramm zur Ermittlung der Pixel-Lnge einer Zeichenkette
; Register: r0,r1,r16,r17,z
;	    r22 <- Stringnummer  (0x00-0xfc)
;	    r18 -> Stringlnge in Pixel
;
strlen:	clr	r18			;Lnge Anfangswert=0 setzen
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,0			;Sprache englisch?
	brts	str110			;nein -> weiter
	ldi	zl,byte3(stengl*2)	;sonst
	out	rampz,zl
	ldi	zl,low(stengl*2)
	ldi	zh,high(stengl*2)	;Zeiger auf englische Texte
	rjmp	str120
;
str110:	ldi	zl,byte3(stgerm*2)
	out	rampz,zl
	ldi	zl,low(stgerm*2)
	ldi	zh,high(stgerm*2)	;Zeiger auf deutsche Texte
;
str120:	elpm	r16,z+			;Zeichen holen
	cpi	r16,0xfe		;String-Kennung gefunden?
	brne	str120			;nein -> weiter suchen
	elpm	r16,z+			;Zeichen holen
	cp	r16,r22			;Zeichenkette gefunden?
	breq	str130			;ja -> Lnge ermitteln
	cpi	r16,0xfe		;Endezeichen gefunden?
	brne	str120			;nein -> weitersuchen
	ret				;sonst Ende
;
str130:	elpm	r16,z+			;Zeichen holen
	cpi	r16,0xff		;Endezeichen?
	brne	str140			;nein -> weiter
	ret				;sonst Ende
str140:	cpi	r16,' '			;Leerzeichen?
	brne	str150			;nein -> weiter
	ldi	r16,0xa0		;sonst schmales Leerzeichen
str150:	subi	r16,0x20		;Korrektur fr Tabelle
	cpi	r16,0xe0		;Bereich berschritten?
	brcs	str160			;nein -> weiter
	ret				;sonst Ende (Fehler)
str160:	ldi	r17,12			;Tabellen-Offset
	mul	r16,r17			;Tabellenplatz berechnen
	push	zl
	push	zh
	in	zl,rampz
	push	zl			;String-Zeiger sichern
;
	ldi	zl,low(chrset*2)	;Zeiger auf Zeichentabelle L
	ldi	zh,high(chrset*2)	;Zeiger auf Zeichentabelle H
	ldi	r16,byte3(chrset*2)	;Zeiger auf Zeichentabelle B3
	clr	r17
	add	zl,r0			;Tabellenadresse L berechnen
	adc	zh,r1			;Tabellenadresse H berechnen
	adc	r16,r17			;Tabellenadresse B3 berechnen
	out	rampz,r16
	elpm	r16,z			;Zeichenbreite holen
;
	pop	zl
	out	rampz,zl
	pop	zh
	pop	zl			;String-Zeiger restaurieren
	inc	r16			;Zeichenbreite korrigieren
	add	r18,r16			;zur Gesamtlnge addieren
	rjmp	str130			;Schleife
;
; Unterprogramm zur Ausgabe eines ASCII-Zeichens auf dem LCD, die
; Darstellung ist abhngig vom Textflag (invers, unterstrichen, Mini)
; Register: r0,r1,r14,r15,r16,r17,r18,r19,r23,r24,zl,zh,t
;	    r20 <- X-Koordinate (0-239)
;	    r21 <- Y-Koordinate (0-127)
;	    r22 <- ASCII-Zeichencode
;
asclcd:	mov	r16,r22			;ASCII-Zeichen kopieren
	subi	r16,0x20		;Korrektur fr Tabelle
	cpi	r16,0xe0		;Bereich berschritten?
	brcs	asc010			;nein -> weiter
	ret				;sonst Ende (Fehler)
;
asc010:	bst	txflag,2		;Mini-Zeichensatz?
	brts	asc020			;ja -> weiter
	ldi	r17,12			;Tabellen-Offset Normal-Font
	mul	r16,r17			;Tabellenplatz berechnen
	clr	r17
	ldi	zl,low(chrset*2)	;Zeiger auf Zeichentabelle L
	ldi	zh,high(chrset*2)	;Zeiger auf Zeichentabelle H
	ldi	r16,byte3(chrset*2)	;Zeiger auf Zeichentabelle B3
	rjmp	asc030
;
asc020:	ldi	r17,6			;Tabellen-Offset Mini-Font
	mul	r16,r17			;Tabellenplatz berechnen
	clr	r17
	ldi	zl,low(chrsmi*2)	;Zeiger auf Zeichentabelle L
	ldi	zh,high(chrsmi*2)	;Zeiger auf Zeichentabelle H
	ldi	r16,byte3(chrsmi*2)	;Zeiger auf Zeichentabelle B3
;
asc030:	add	zl,r0			;Tabellenadresse L
	adc	zh,r1			;Tabellenadresse H
	adc	r16,r17			;Tabellenadresse B3
	out	rampz,r16
	elpm	r14,z+			;Zeichenbreite holen
	ldi	r23,11			;11 Zeilen ausgeben
	brtc	asc040			;Mini-Font? nein -> weiter
	ldi	r23,5			;sonst nur 5 Zeilen ausgeben
asc040:	mov	r19,r21			;Y-Koordinate kopieren
asc050:	mov	r18,r20			;X-Koordinate kopieren
	bst	txflag,0		;Inversflag holen
	bld	r19,7			;und kopieren
	cpi	r23,2			;vorletzte Zeile?
	brne	asc070			;nein -> normale Ausgabe
;
	bst	txflag,1		;Unterstreichung?
	brtc	asc070			;nein -> normale Ausgabe
	mov	r16,txflag		;Textflags kopieren und
	com	r16			;invertieren (fr Unterstrich)
	bst	r16,0			;Inversflag holen
	bld	r19,7			;und in Y-Koordinate einsetzen
	mov	r24,r14			;Zeichenbreite kopieren
	inc	r24			;und korrigieren
asc060:	rcall	pixlcd			;Pixel ausgeben
	inc	r18			;nchste X-Koordinate setzen
	dec	r24			;alle Pixel ausgegeben?
	brne	asc060			;nein -> Schleife
	elpm	r0,z+			;nchste Zeichenzeile
	rjmp	asc110			;weiter mit normaler Ausgabe
;
asc070:	rcall	pixlcd			;1 Pixel lschen (vor Zeichen)
	inc	r18			;nchste X-Koordinate setzen
	elpm	r15,z+			;eine Zeichenzeile holen
	bst	txflag,0		;Invers-Darstellung?
	brtc	asc080			;nein -> weiter
	com	r15			;sonst alle Pixel invertieren
asc080:	ldi	r24,8			;Zhler fr Bitverschiebung
asc090:	cp	r24,r14			;Zhler = Zeichenbreite?
	breq	asc100			;ja -> Ausgabe beginnen
	rol	r15			;sonst Zeichenzeile verschieben
	dec	r24			;komplettes Byte verschoben?
	brne	asc090			;nein -> Schleife
	rjmp	asc110			;sonst keine weiteren Pixel
asc100:	bst	r15,7			;Bit aus Zeichenzeile holen
	bld	r19,7			;und in Y-Koordinate einsetzen
	rcall	pixlcd			;Zeichenpixel ausgeben
	rol	r15			;nchstes Pixel bearbeiten
	inc	r18			;nchste X-Koordinate setzen
	dec	r24			;alle Pixel ausgegeben?
	brne	asc100			;nein -> nchstes Pixel
asc110:	andi	r19,0x7f		;Y-Koordinate filtern
	inc	r19			;nchste Y-Koordinate setzen
	dec	r23			;alle Zeilen ausgegeben?
	brne	asc050			;nein -> nchste Zeile
	add	r20,r14			;X-Koordinate fr nchstes
	inc	r20			;Zeichen ermitteln
	ret
;
; Unterprogramm zur Ausgabe eines Bytes als 2 ASCII-Zeichen auf dem LCD
; Register: r0,r1,r14,r15,r16,r17,r18,r19,r23,r24,zl,zh,t
;	    r18 <- Byte (0-99)
;	    r20 <- X-Koordinate (0-239)
;	    r21 <- Y-Koordinate (0-127)
;
bytlcd:	call	bytasc			;Byte in ASCII wandeln
	push	r16			;Einer sichern
	mov	r22,r17			;Zehner holen und
	call	asclcd			;ausgeben
	pop	r22			;gesicherte Einer holen und
	call	asclcd			;ausgeben
	ret
;
; Unterprogramm zur Ermittlung der Pixel-Lnge eines Zeichens
; Register: r0,r1,r17,z
;	    r16 <- ASCII-Zeichencode (0x20-0x9f)
;	    r16 -> Zeichenlnge in Pixel
;
asclen:	subi	r16,0x20		;Korrektur fr Tabelle
	cpi	r16,0xe0		;Bereich berschritten?
	brcs	asc120			;nein -> weiter
	ret				;sonst Ende (Fehler)
;
asc120:	bst	txflag,2		;Mini-Zeichensatz?
	brts	asc130			;ja -> weiter
	ldi	r17,12			;Tabellen-Offset Normal-Font
	mul	r16,r17			;Tabellenplatz berechnen
	clr	r17
	ldi	zl,low(chrset*2)	;Zeiger auf Zeichentabelle L
	ldi	zh,high(chrset*2)	;Zeiger auf Zeichentabelle H
	ldi	r16,byte3(chrset*2)	;Zeiger auf Zeichentabelle B3
	rjmp	asc140
;
asc130:	ldi	r17,6			;Tabellen-Offset Mini-Font
	mul	r16,r17			;Tabellenplatz berechnen
	clr	r17
	ldi	zl,low(chrsmi*2)	;Zeiger auf Zeichentabelle L
	ldi	zh,high(chrsmi*2)	;Zeiger auf Zeichentabelle H
	ldi	r16,byte3(chrsmi*2)	;Zeiger auf Zeichentabelle B3
;
asc140:	add	zl,r0			;Tabellenadresse L berechnen
	adc	zh,r1			;Tabellenadresse H berechnen
	adc	r16,r17			;Tabellenadresse B3 berechnen
	out	rampz,r16
	elpm	r16,z			;Zeichenbreite holen
	inc	r16			;Zeichenbreite korrigieren
	ret
;
; Unterprogramm zur Ausgabe eines ASCII-Zeichens auf dem LCD im Gro-
; Format (2x6 normale Zeichen = 16x33 Pixel), mglich sind die Zeichen
; 0-9, Minus und Leerzeichen; das Textflag wird nicht ausgewertet
; Register: r0,r1,r14,r15,r16,r17,r18,r19,r23,r24,zl,zh,t
;	    r20 <- X-Koordinate (0-239)
;	    r21 <- Y-Koordinate (0-127)
;	    r22 <- ASCII-Zeichencode
;
biglcd:	cpi	r22,'-'			;Minus-Zeichen?
	brne	bigl10			;nein -> weiter testen
	ldi	r22,10			;sonst Zeichencode fr Minus
	rjmp	bigl30			;setzen und ausgeben
bigl10:	cpi	r22,' '			;Leerzeichen?
	brne	bigl20			;nein -> ausgeben
	ldi	r22,11			;sonst Zeichencode fr Leer-
	rjmp	bigl30			;zeichen setzen und ausgeben
bigl20:	andi	r22,0x0f		;Ziffernwert ermitteln
bigl30:	ldi	r16,6			;Offset fr Zeichentabelle
	mul	r22,r16			;Tabellenposition ermitteln
	ldi	r22,0x98		;ASCII-Wert fr Grozeichen
	add	r22,r0			;ermitteln
	push	r20			;X Koordinate sichern
	rcall	ascfrg			;Fragment oben links ausgeben
	inc	r22			;ASCII-Wert nchstes Fragment
	rcall	ascfrg			;Fragment oben rechts ausgeben
	pop	r20			;X Koordinate wieder herstellen
	ldi	r16,11			;Y Koordinate fr mittlere
	add	r21,r16			;Fragmente ermitteln
	inc	r22			;ASCII-Wert nchstes Fragment
	push	r20			;X Koordinate sichern
	rcall	ascfrg			;Fragment Mitte links ausgeben
	inc	r22			;ASCII-Wert nchstes Fragment
	rcall	ascfrg			;Fragment Mitte rechts ausgeben
	pop	r20			;X Koordinate wieder herstellen
	ldi	r16,11			;Y Koordinate fr untere
	add	r21,r16			;Fragmente ermitteln
	inc	r22			;ASCII-Wert nchstes Fragment
	rcall	ascfrg			;Fragment unten links ausgeben
	inc	r22			;ASCII-Wert nchstes Fragment
	rcall	ascfrg			;Fragment unten rechts ausgeben
	ret
;
; Unterprogramm zur Ausgabe eines Grozeichen-Fragments
;
ascfrg:	mov	r16,r22			;Zeichencode kopieren
	ldi	r17,12			;Tabellen-Offset
	mul	r16,r17			;Tabellenplatz berechnen
	clr	r17
	ldi	zl,low(chrset*2)	;Zeiger auf Zeichentabelle L
	ldi	zh,high(chrset*2)	;Zeiger auf Zeichentabelle H
	ldi	r16,byte3(chrset*2)	;Zeiger auf Zeichentabelle B3
	add	zl,r0			;Tabellenadresse L
	adc	zh,r1			;Tabellenadresse H
	adc	r16,r17			;Tabellenadresse B3
	out	rampz,r16
	elpm	r14,z+			;Zeichenbreite holen
	ldi	r23,11			;11 Zeilen ausgeben
	mov	r19,r21			;Y-Koordinate kopieren
ascf10:	mov	r18,r20			;X-Koordinate kopieren
	elpm	r15,z+			;eine Zeichenzeile holen
	ldi	r24,8			;Zhler fr Bitverschiebung
ascf20:	cp	r24,r14			;Zhler = Zeichenbreite?
	breq	ascf30			;ja -> Ausgabe beginnen
	rol	r15			;sonst Zeichenzeile verschieben
	dec	r24			;komplettes Byte verschoben?
	brne	ascf20			;nein -> Schleife
	rjmp	ascf40			;sonst keine weiteren Pixel
ascf30:	bst	r15,7			;Bit aus Zeichenzeile holen
	bld	r19,7			;und in Y-Koordinate einsetzen
	rcall	pixlcd			;Zeichenpixel ausgeben
	rol	r15			;nchstes Pixel bearbeiten
	inc	r18			;nchste X-Koordinate setzen
	dec	r24			;alle Pixel ausgegeben?
	brne	ascf30			;nein -> nchstes Pixel
ascf40:	andi	r19,0x7f		;Y-Koordinate filtern
	inc	r19			;nchste Y-Koordinate setzen
	dec	r23			;alle Zeilen ausgegeben?
	brne	ascf10			;nein -> nchste Zeile
	add	r20,r14			;X-Koordinate fr nchstes
	ret				;Zeichen ermitteln, Ende
;
; Unterprogramm zum Lschen eines Pixelfensters auf dem LCD
; Register: r0,r1,r15,r16,r17
;	    r18 <- X-Koordinate Startpunkt (0-239)
;           r19 <- Y-Koordinate Startpunkt (0-127, Bit7=1 zum Setzen
;	    r20 <- X-Koordinate Endpunkt (0-239)	     der Pixel)
;	    r21 <- Y-Koordinate Endpunkt (0-127, Bit7=1 zum Setzen der
;							         Pixel)
;
clrwnd:	mov	r15,r18			;X-Startkoordinate sichern
clrw10:	mov	r18,r15			;X-Startkoordinate holen
clrw20:	rcall	pixlcd			;Pixel lschen
	inc	r18			;nchstes Pixel, gleiche Zeile
	cp	r18,r20			;X-Endwert erreicht?
	brcs	clrw20			;nein -> Schleife
	inc	r19			;sonst nchste Zeile
	cp	r19,r21			;Y-Endwert erreicht?
	brcs	clrw10			;nein -> Schleife
	ret
;
; Unterprogramm zum Zeichnen oder Lschen eines Pixels auf dem LCD
; Register: r0,r1,r16,r17,t
;	    r18 <- X-Koordinate (0-239)
;	    r19 <- Y-Koordinate (0-127, Bit7=1 setzen, Bit7=0 lschen)
;
pixlcd:	mov	r16,r19			;Y-Koordinate kopieren
	andi	r16,0x7f		;Koordinatenbits filtern
	ldi	r17,30			;mit Anzahl der Byte pro Zeile
	mul	r16,r17			;multiplizieren
	mov	r16,r18			;X-Koordinate kopieren
	ror	r16
	ror	r16
	ror	r16			;Wert durch 8 dividieren
	andi	r16,0x1f		;relevante Bits filtern
	clr	r17			;H-Byte fr 16-Bit-Addition
	add	r0,r16			;Spaltenoffset addieren
	adc	r1,r17			;bertrag addieren
;
	mov	r17,r0			;berechnete Adresse Low
	clt				;LCD in Data Mode
	rcall	outlcd			;Byte ausgeben
	mov	r17,r1			;berechnete Adresse High
	rcall	outlcd			;Byte ausgeben
	ldi	r17,0x24		;Address Pointer Set
	set				;LCD in Command Mode
	rcall	outlcd			;Byte ausgeben
	mov	r17,r18			;X-Koordinate kopieren
	com	r17			;Bitposition fr LCD ermitteln
	andi	r17,0x07		;Pixel-Bitnummer filtern
	ori	r17,0xf0		;Command-Bits ergnzen
	bst	r19,7			;Setzen/Lschen-Bit holen
	bld	r17,3			;und in Command einsetzen
	set				;LCD in Command Mode
	rcall	outlcd			;Byte ausgeben
	ret
;
; Unterprogramm zur Ausgabe eines Bytes auf das LCD; vorher Flag
; prfen, ob LCD eingeschaltet ist
; Register: r16
;	    r17    <- Datenbyte
;	    t-flag <- 0=Data, 1=Command
;
outlcd:	lds	r16,dflags		;Anzeige-Flags holen
	andi	r16,0x80		;Anzeige abgeschaltet?
	breq	outl10			;nein -> Byte ausgeben
	ret				;sonst Ende
;
outl10:	cbi	portd,7			;Testausgabe
;
	cbi	portc,2			;/CE auf Low
	cbi	portc,1			;/RD auf Low
	nop
	nop				;warten (mindestens 150ns)
outl20:	in	r16,pina		;LCD-Datenbyte lesen
	andi	r16,0x03		;STA0 und STA1 filtern
	cpi	r16,0x03		;STA0 und STA1 gesetzt?
	brne	outl20			;nein -> Abfrage wiederholen
	sbi	portc,1			;/RD wieder auf High
	brts	outl30			;Command? ja -> weiter
	cbi	portc,3			;sonst C/D auf Low (=Data)
outl30:	out	porta,r17		;Byte auf LCD-Port ausgeben
	ser	r16
	out	ddra,r16		;LCD-Datenport auf Ausgang
	cbi	portc,0			;/WR auf Low
	sbi	portc,0			;/WR wieder auf High
	out	porta,r16		;Pull-up einschalten
	clr	r16
	out	ddra,r16		;LCD-Datenport auf Eingang
	sbi	portc,2			;/CE wieder auf High
	sbi	portc,3			;C/D wieder auf High
;
	sbi	portd,7			;Testausgabe
;
	ret
;
; Unterprogramm zum Einlesen eines Datenbytes aus dem LCD; vorher Flag
; prfen, ob LCD eingeschaltet ist
; Register: r16
;	    r17    -> Datenbyte
;
in_lcd:	lds	r16,dflags		;Anzeige-Flags holen
	andi	r16,0x80		;Anzeige abgeschaltet?
	breq	in_l10			;nein -> Byte einlesen
	ret				;sonst Ende
;
in_l10:	cbi	portd,7			;Testausgabe
;
	cbi	portc,2			;/CE auf Low
	cbi	portc,1			;/RD auf Low
	nop
	nop				;warten (mindestens 150ns)
in_l20:	in	r16,pina		;LCD-Datenbyte lesen
	andi	r16,0x03		;STA0 und STA1 filtern
	cpi	r16,0x03		;STA0 und STA1 gesetzt?
	brne	in_l20			;nein -> Abfrage wiederholen
	sbi	portc,1			;sonst /RD wieder auf High
	cbi	portc,3			;C/D auf Low (=Data)
	nop				;90ns warten
	cbi	portc,1			;/RD auf Low
	nop
	nop				;warten (mindestens 150ns)
	in	r17,pina		;LCD-Datenbyte lesen
	sbi	portc,1			;/RD wieder auf High
	sbi	portc,2			;/CE wieder auf High
	sbi	portc,3			;C/D wieder auf High
;
	sbi	portd,7			;Testausgabe
;
	ret
;
; Unterprogramm zum Lschen des LCD Grafikspeichers
; Register: r16,r17,r18,r19,t
;
clrlcd:	ldi	r17,0			;Pointer auf Anfangsposition
	clt				;LCD in Data Mode
	rcall	outlcd			;Byte ausgeben
	rcall	outlcd			;Byte ausgeben
	ldi	r17,0x24		;Address Pointer Set
	set				;LCD in Command Mode
	rcall	outlcd			;Byte ausgeben
	ldi	r19,128			;Zeilenzhler setzen
clrl10:	ldi	r18,30			;Spaltenzhler setzen
clrl20:	clr	r17			;0-Werte schreiben
	clt				;LCD in Data Mode
	rcall	outlcd			;Byte ausgeben
	ldi	r17,0xc0		;Datawrite Auto Increment
	set				;LCD in Command Mode
	rcall	outlcd			;Byte ausgeben
	dec	r18			;alle Spalten ausgegeben?
	brne	clrl20			;nein -> Schleife
	dec	r19			;alle Zeilen ausgegeben?
	brne	clrl10			;nein -> Schleife
	ret
;
; Unterprogramm (Grafikanzeige) zum Lschen des Grafik-Datenpuffers,
; Initialisieren des Pointers, des Zeit-Zwischenspeichers und einiger
; Variablen
; Register: r16,r17,r18,r19,x
;
grclrb:	ldi	xl,low(grbuff)
	ldi	xh,high(grbuff)		;Zeiger auf Grafik-Datenpuffer
	ldi	r16,0
	ldi	r17,0x80		;ungltiger Temperaturwert
	ldi	r18,0xff		;Wert fr ungltige Zeitdaten
	ldi	r19,222			;222 Datenstze initialisieren
grcl10:	st	x+,r18
	st	x+,r18
	st	x+,r18			;Datum/Zeit ungltig machen
	st	x+,r18			;(alle Bytes 0xff)
	st	x+,r16
	st	x+,r17			;Tempwert 1 lschen (0x8000)
	st	x+,r16
	st	x+,r17			;Tempwert 2 lschen (0x8000)
	dec	r19			;alle Datenstze bearbeitet?
	brne	grcl10			;nein -> Schleife
;
	sts	grcdat,r18
	sts	grcdat+1,r18
	sts	grcdat+2,r18		;Zeit-Zwischenspeicher
	sts	grcdat+3,r18		;initialisieren
	sts	grsca1,r16		;Y-Skalierung 1 initialisieren
	sts	grsca2,r16		;Y-Skalierung 2 initialisieren
	sts	grcen1,r16		;Y-Mittelpunktwert
	sts	grcen1+1,r16		;fr Diagramm 1 initialisieren
	sts	grcen2,r16		;Y-Mittelpunktwert
	sts	grcen2+1,r16		;fr Diagramm 2 initialisieren
;
	lds	r17,xflag3		;verschiedene Flags 3 holen
	bst	r17,3			;nchste Seite anzeigen?
	brtc	grcl20			;ja -> Zeiger ok
	ldi	r16,221			;sonst Zeiger auf Puffer-Ende
grcl20:	sts	grpoin,r16		;Grafikpuffer-Zeiger speichern
	ret
;
; Unterprogramm (Grafikanzeige) zum Ermitteln der Adresse des vorheri-
; gen Log-Eintrages, ohne Test auf Erreichen des Log-Anfangs
; Register: r16,r17,r18,r19
;	    r20,r21,r22 <- EEPROM-Zeiger fr Grafikpuffer
;
grdecx:	lds	r19,grmod1		;Grafikanzeige-Modus holen
	tst	r19			;15-Minuten-Werte anzeigen?
	breq	grde20			;ja -> bearbeiten
	rjmp	grde60			;sonst 24h-Werte bearbeiten

; Unterprogramm (Grafikanzeige) zum Ermitteln der Adresse des vorheri-
; gen Log-Eintrages, vorher Test auf Erreichen des Log-Anfangs
; Register: r16,r17,r18,r19,r20,r21,r22
;
grdeca:	lds	r20,grptem
	lds	r21,grptem+1		;EEPROM-Zeiger fr Grafikpuffer
	lds	r22,grptem+2		;holen
	lds	r19,grmod1		;Grafikanzeige-Modus holen
	tst	r19			;15-Minuten-Werte anzeigen?
	brne	grde50			;nein -> weiter
;
	lds	r16,xet1po		;aktuellen EEPROM-Zeiger auf
	lds	r17,xet1po+1		;Temperaturdaten 1 holen, zeigt
	lds	r18,xet1po+2		;auf ltesten Eintrag
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;erster Datensatz erreicht?
	brne	grde20			;nein -> weiter
grde10:	lds	r19,xflag3		;sonst verschiedene Flags holen
	sbr	r19,1<<5		;erster Datensatz erreicht
	sts	xflag3,r19		;Flags wieder speichern
	ret
;
grde20:	ldi	r16,low(xt1sta)
	ldi	r17,high(xt1sta)	;Anfangsadresse Temperaturdaten
	ldi	r18,byte3(xt1sta)	;15-min laden
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;Anfangsadresse erreicht?
	brne	grde30			;nein -> weiter
	ldi	r20,low(xt1sta+xt1len)	;sonst
	ldi	r21,high(xt1sta+xt1len)	;End-Adresse Temperatur-Daten
	ldi	r22,byte3(xt1sta+xt1len);15-min laden
grde30:	subi	r20,28
	sbci	r21,0			;Zeiger auf vorherige Position
	sbci	r22,0			;setzen (zeitlich davor)
	sts	grptem,r20
	sts	grptem+1,r21		;neuen EEPROM-Zeiger fr den
	sts	grptem+2,r22		;Grafikpuffer speichern
	ret
;
grde50:	lds	r16,xet2po		;aktuellen EEPROM-Zeiger auf
	lds	r17,xet2po+1		;Temperatur-Daten 1 holen,
	lds	r18,xet2po+2		;zeigt auf ltesten Eintrag
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;erster Datensatz erreicht?
	breq	grde10			;ja -> Ende
;
grde60:	ldi	r16,low(xt2sta)
	ldi	r17,high(xt2sta)	;Anfangsadresse Temperaturdaten
	ldi	r18,byte3(xt2sta)	;24-h laden
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;Anfangsadresse erreicht?
	brne	grde70			;nein -> weiter
	ldi	r20,low(xt2sta+xt2len)	;sonst
	ldi	r21,high(xt2sta+xt2len)	;End-Adresse Temperatur-Daten
	ldi	r22,byte3(xt2sta+xt2len);24-h laden
grde70:	subi	r20,76
	sbci	r21,0			;Zeiger auf vorherige Position
	sbci	r22,0			;setzen (zeitlich davor)
	sts	grptem,r20
	sts	grptem+1,r21		;neuen EEPROM-Zeiger fr den
	sts	grptem+2,r22		;Grafikpuffer speichern
	ret
;
; Unterprogramm (Grafikanzeige) zum Ermitteln der Adresse des nchsten
; Log-Eintrages
; Register: r16,r17,r18,r19,r20,r21,r22
;
grinca:	lds	r20,grptem
	lds	r21,grptem+1		;EEPROM-Zeiger fr Grafikpuffer
	lds	r22,grptem+2		;holen
	clr	r17			;Offset zum nchst. Datensatz H
	lds	r19,grmod1		;Grafikanzeige-Modus holen
	tst	r19			;15-Minuten-Werte anzeigen?
	brne	grin50			;nein -> weiter
;
	ldi	r16,28			;Offset zum nchst. Datensatz L
	add	r20,r16
	adc	r21,r17			;Zeiger auf nchsten Eintrag
	adc	r22,r17			;setzen (zeitlich danach)
	sts	grptem,r20
	sts	grptem+1,r21		;neuen EEPROM-Zeiger fr
	sts	grptem+2,r22		;Grafikpuffer speichern
;
	ldi	r16,low(xt1sta+xt1len)
	ldi	r17,high(xt1sta+xt1len)	;Endadresse Temperaturdaten
	ldi	r18,byte3(xt1sta+xt1len);15-min laden
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;Endadresse erreicht?
	brne	grin10			;nein -> weiter
	ldi	r20,low(xt1sta)		;sonst
	ldi	r21,high(xt1sta)	;Anfangsadresse Temperaturdaten
	ldi	r22,byte3(xt1sta)	;15-min laden
	sts	grptem,r20
	sts	grptem+1,r21		;neuen EEPROM-Zeiger fr
	sts	grptem+2,r22		;Grafikpuffer speichern
;
grin10:	lds	r16,xet1po		;aktuellen EEPROM-Zeiger auf
	lds	r17,xet1po+1		;Temperaturdaten 1 holen, zeigt
	lds	r18,xet1po+2		;auf letzten Listeneintrag
grin20:	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;letzter Datensatz erreicht?
	brne	grin30			;nein -> weiter
	lds	r19,xflag3		;sonst verschiedene Flags holen
	sbr	r19,1<<4		;letzter Datensatz erreicht
	sts	xflag3,r19		;Flags wieder speichern
grin30:	ret
;
grin50:	ldi	r16,76			;Offset zum nchst. Datensatz L
	add	r20,r16
	adc	r21,r17			;Zeiger auf nchsten Eintrag
	adc	r22,r17			;setzen (zeitlich danach)
	sts	grptem,r20
	sts	grptem+1,r21		;neuen EEPROM-Zeiger fr
	sts	grptem+2,r22		;Grafikpuffer speichern
;
	ldi	r16,low(xt2sta+xt2len)
	ldi	r17,high(xt2sta+xt2len)	;Endadresse Temperaturdaten 24h
	ldi	r18,byte3(xt2sta+xt2len);laden
	sub	r16,r20
	sbc	r17,r21
	sbc	r18,r22			;Endadresse erreicht?
	brne	grin60			;nein -> weiter
	ldi	r20,low(xt2sta)		;sonst
	ldi	r21,high(xt2sta)	;Anfangsadresse Temperaturdaten
	ldi	r22,byte3(xt2sta)	;24h laden
	sts	grptem,r20
	sts	grptem+1,r21		;neuen EEPROM-Zeiger fr
	sts	grptem+2,r22		;Grafikpuffer speichern
;
grin60:	lds	r16,xet2po		;aktuellen EEPROM-Zeiger auf
	lds	r17,xet2po+1		;Temperaturdaten 2 holen, zeigt
	lds	r18,xet2po+2		;auf letzten Listeneintrag
	rjmp	grin20
;
; Unterprogramm (Grafikanzeige) zum Ermitteln des vorherigen Log-Zeit-
; punktes
; Register: r16,r17,r18,r19,r20,r21
;
grprev:	rcall	grexti			;Zeitdaten extrahieren
	lds	r21,grmod1		;Grafikanzeige-Modus holen
	tst	r21			;15-Minuten-Werte anzeigen?
	brne	grpr10			;nein -> weiter
;
	subi	r16,15			;Minute -15
	brcc	grpr30			;bertrag? nein -> Ende
	ldi	r16,45			;sonst Minute= 45 setzen
	subi	r17,1			;Stunde -1
	brcc	grpr30			;bertrag? nein -> Ende
	ldi	r17,23			;sonst Stunde= 23 setzen
;
grpr10:	dec	r18			;Tag -1
	brne	grpr30			;bertrag? nein -> Ende
	dec	r19			;sonst Monat -1
;
	ldi	r18,31			;Tag= 31 setzen
	cpi	r19,1			;Monat Januar?
	breq	grpr20			;ja -> Tag ok
	cpi	r19,3			;Monat Mrz?
	breq	grpr20			;ja -> Tag ok
	cpi	r19,5			;Monat Mai?
	breq	grpr20			;ja -> Tag ok
	cpi	r19,7			;Monat Juli?
	breq	grpr20			;ja -> Tag ok
	cpi	r19,8			;Monat August?
	breq	grpr20			;ja -> Tag ok
	cpi	r19,10			;Monat Oktober?
	breq	grpr20			;ja -> Tag ok
	cpi	r19,0			;Monat Dezember?
	breq	grpr20			;ja -> Tag ok
;
	ldi	r18,30			;Tag= 30 setzen
	cpi	r19,4			;Monat April?
	breq	grpr20			;ja -> Tag ok
	cpi	r19,6			;Monat Juni?
	breq	grpr20			;ja -> Tag ok
	cpi	r19,9			;Monat September?
	breq	grpr20			;ja -> Tag ok
	cpi	r19,11			;Monat November?
	breq	grpr20			;ja -> Tag ok
;
	ldi	r18,29			;Tag= 29 setzen
	mov	r21,r20			;sonst Jahr kopieren
	andi	r21,0x03		;Jahr durch 4 teilbar?
	breq	grpr20			;ja -> Tag ok
	ldi	r18,28			;sonst Tag= 28 setzen
;
grpr20:	cpi	r19,0			;Monat=0 (bertrag)?
	brne	grpr30			;nein -> Ende
	ldi	r19,12			;sonst Monat= 12 setzen
	dec	r20			;Jahr -1
;
grpr30:	rcall	grcpti			;Zeitdaten wieder komprimieren
	ret
;
; Unterprogramm (Grafikanzeige) zum Ermitteln des nchsten Log-Zeit-
; punktes
; Register: r16,r17,r18,r19,r20,r21
;
grnext:	rcall	grexti			;Zeitdaten extrahieren
	lds	r21,grmod1		;Grafikanzeige-Modus holen
	tst	r21			;15-Minuten-Werte anzeigen?
	brne	grne10			;nein -> weiter
;
	ldi	r21,15			;Minutenabstand= 15
	add	r16,r21			;Minute +15
	cpi	r16,60			;bertrag?
	brcs	grne70			;nein -> Ende
	ldi	r16,0			;sonst Minute= 0 setzen
	inc	r17			;Stunde +1
	cpi	r17,24			;bertrag?
	brcs	grne70			;bertrag? nein -> Ende
	ldi	r17,0			;sonst Stunde= 0 setzen
;
grne10:	inc	r18			;Tag +1
	cpi	r19,1			;Monat Januar?
	breq	grne20			;ja -> weiter (31 Tage)
	cpi	r19,2			;Monat Februar?
	breq	grne40			;ja -> weiter (28/29 Tage)
	cpi	r19,3			;Monat Mrz?
	breq	grne20			;ja -> weiter (31 Tage)
	cpi	r19,4			;Monat April?
	breq	grne30			;ja -> weiter (30 Tage)
	cpi	r19,5			;Monat Mai?
	breq	grne20			;ja -> weiter (31 Tage)
	cpi	r19,6			;Monat Juni?
	breq	grne30			;ja -> weiter (30 Tage)
	cpi	r19,7			;Monat Juli?
	breq	grne20			;ja -> weiter (31 Tage)
	cpi	r19,8			;Monat August?
	breq	grne20			;ja -> weiter (31 Tage)
	cpi	r19,9			;Monat September?
	breq	grne30			;ja -> weiter (30 Tage)
	cpi	r19,10			;Monat Oktober?
	breq	grne20			;ja -> weiter (31 Tage)
	cpi	r19,11			;Monat November?
	breq	grne30			;ja -> weiter (30 Tage)
;
grne20:	cpi	r18,32			;bertrag (Tag=32)?
	rjmp	grne60			;Test auf Monatswechsel
;
grne30:	cpi	r18,31			;bertrag (Tag=31)?
	rjmp	grne60			;Test auf Monatswechsel
;
grne40:	mov	r21,r20			;Jahr kopieren
	andi	r21,0x03		;Jahr durch 4 teilbar?
	breq	grne50			;ja -> weiter
	cpi	r18,29			;berlauf (Tag=29)?
	rjmp	grne60			;Test auf Monatswechsel
;
grne50:	cpi	r18,30			;berlauf (Tag=30)?
grne60:	brcs	grne70			;berlauf? nein -> Ende
	ldi	r18,1			;sonst Tag=1 setzen
	inc	r19			;Monat +1
	cpi	r19,13			;berlauf?
	brcs	grne70			;nein -> Ende	
	ldi	r19,1			;sonst Monat=1 setzen
	inc	r20			;Jahr +1
grne70:	rcall	grcpti			;Zeitdaten wieder komprimieren
	ret
;
; Unterprogramm (Grafikanzeige) zum Extrahieren der Zeitdaten
; Register: r16,r17,r18,r19,r20 -> extrahierte Zeit (min,h,d,mon,y)
;
grexti:	lds	r20,grcdat+3		;komprimierte Zeit Byte 3 holen
	bst	r20,0			;LSB sichern, gehrt zum Monat
	lsr	r20			;Jahr an Position schieben
	lds	r19,grcdat+2		;komprimierte Zeit Byte 2 holen
	ror	r19			;Monat nach rechts schieben
	bld	r19,7			;LSB von Byte 3 holen, Monat an
	swap	r19			;richtige Position schieben
	andi	r19,0x0f		;nicht relevante Bits filtern
	lds	r18,grcdat+2		;komprimierte Zeit Byte 2 holen
	andi	r18,0x1f		;nicht relevante Bits filtern
	lds	r17,grcdat+1		;Stunde speichern
	lds	r16,grcdat		;Minute speichern
	ret
;
; Unterprogramm (Grafikanzeige) zum Komprimieren der Zeitdaten
; Register: r16,r17,r18,r19,r20 <- komprimierte Zeit (min,h,d,mon,y)
;
grcpti:	sts	grcdat,r16		;Minuten speichern
	sts	grcdat+1,r17		;Stunde speichern
	swap	r19			;Monat, Nibbles tauschen
	bst	r19,7			;MSB sichern, kommt in Byte 3
	lsl	r19			;Wert verschieben und
	or	r19,r18			;mit Tag verknpfen
	sts	grcdat+2,r19		;Byte 2 speichern
	lsl	r20			;Jahr, Wert verschieben
	bld	r20,0			;MSB vom Monat ergnzen
	sts	grcdat+3,r20		;Byte 3 speichern
	ret
;
; Unterprogramm (Grafikanzeige) zum Suchen der Log-Position eines
; Sensors in der Log-Map
; Register: r17,x
;	    r16 <- Sensornummer
;	    r16 -> gesuchte Log-Position
;
grslgm:	ldi	xl,low(logmap)
	ldi	xh,high(logmap)		;Zeiger auf Log-Map
	clr	r17			;H-Byte fr Berechnung
	add	xl,r16
	adc	xh,r17			;Tabellenplatz berechnen
	ld	r16,x			;Log-Position holen
	ret
;
; Unterprogramm (Grafikanzeige) zum Extrahieren eines Temperaturwertes
; aus einem Log-Datensatz
; Register: r0,r1,t
;	    r16     <- Nummer des Log-Eintrages (0-15)
;	    y       <- Zeiger auf Temperaturwerte im Log-Datensatz
;	    r16,r17 -> Temperaturwert im Zweierkomplement
;
grgete:	bst	r16,0			;LSB der Nummer sichern
	lsr	r16			;Nummer halbieren
	ldi	r17,3			;Wertepaare jeweils 3 Byte
	mul	r16,r17			;Offset vom Wertepaar berechnen
	add	yl,r0
	adc	yh,r1			;Adresse vom Wertepaar ermitt.
	ld	r16,y			;L-Byte von Wert 1 holen
	ldd	r17,y+2			;H-Byte von beiden Werten holen
	brtc	grge10			;erster Wert? ja -> weiter
	ldd	r16,y+1			;sonst L-Byte von Wert 2 holen
	swap	r17			;H-Byte an richt. Stelle legen
grge10:	andi	r17,0x0f		;relevante Bits filtern
	cpi	r17,0x08		;Wert ungltig?
	brne	grge20			;nein -> weiter
	swap	r17			;sonst H-Byte auf 0x80 setzen
	ret
grge20:	brcs	grge30			;Wert positiv? ja -> weiter
	ori	r17,0xf0		;sonst H-Byte korrigieren
grge30:	ret
;
; Unterprogramm (Grafikanzeige) zum Ablegen eines gelesenen Temperatur-
; Wertes in den Grafik-Datenpuffer als Temperaturwert 1
; Register: r0,r1,r18,r19,z
;	    r16,r17 <- Temperaturwert im Zweierkomplement
;
grbts1:	lds	r18,grpoin		;Zeiger holen
	ldi	r19,8			;Pufferplatz 8 Bytes gro
	mul	r18,r19			;Puffer-Offset berechnen
	ldi	zl,low(grbuff)
	ldi	zh,high(grbuff)		;Basisadresse Grafikdatenpuffer
	add	zl,r0
	adc	zh,r1			;Datenpuffer-Adresse berechnen
	std	z+4,r16
	std	z+5,r17			;Temperaturwert 1 speichern
	lds	r16,grcdat
	lds	r17,grcdat+1
	lds	r18,grcdat+2
	lds	r19,grcdat+3		;Zeit aus Zwischenspeicher hol.
	st	z,r16
	std	z+1,r17
	std	z+2,r18
	std	z+3,r19			;Zeit in Datenpuffer speichern
	ret
;
; Unterprogramm (Grafikanzeige) zum Ablegen eines gelesenen Temperatur-
; Wertes in den Grafik-Datenpuffer als Temperaturwert 2
; Register: r0,r1,r18,r19,z
;	    r16,r17 <- Temperaturwert im Zweierkomplement
;
grbts2:	lds	r18,grpoin		;Zeiger holen
	ldi	r19,8			;Pufferplatz 8 Bytes gro
	mul	r18,r19			;Puffer-Offset berechnen
	ldi	zl,low(grbuff)
	ldi	zh,high(grbuff)		;Basisadresse Grafikdatenpuffer
	add	zl,r0
	adc	zh,r1			;Datenpuffer-Adresse berechnen
	std	z+6,r16
	std	z+7,r17			;Temperaturwert 2 speichern
	ret				;Zeitdaten bereits im Puffer
;
; Unterprogramm (Grafikanzeige) zum Ermitteln eines passenden Tempera-
; turwertes entsprechend des Skalierungsfaktors
; Register: r18,r19,r20
;	    r16,r17 <- Temperaturwert im Zweierkomplement
;	    r21     <- Skalierungsfaktor
;	    Z-Flag  -> gesetzt, wenn Temperaturwert ganzzahlig
;
grchek:	movw	r18,r16			;Temperaturwert kopieren
	bst	r19,7			;Wert negativ?
	brtc	grch10			;nein -> weiter
	clr	r18
	clr	r19
	sub	r18,r16
	sbc	r19,r17			;in positiven Wert wandeln
grch10:	mov	r20,r18
	or	r20,r19			;Ergebnis=0?
	breq	grch20			;ja -> Ende
	cpi	r21,1			;Skalierungsfaktor 1?
	brne	grch30			;nein -> weiter testen
	subi	r18,10			;sonst 10 subtrahieren
	sbci	r19,0			;bertrag, Ergebnis positiv?
	brcc	grch10			;ja -> Schleife
grch20:	ret				;sonst Ende
;
grch30:	cpi	r21,2			;Skalierungsfaktor 2?
	brne	grch40			;nein -> weiter testen
	subi	r18,20			;sonst 20 subtrahieren
	sbci	r19,0			;bertrag, Ergebnis positiv?
	brcc	grch10			;ja -> Schleife
	ret				;sonst Ende
;
grch40:	cpi	r21,4			;Skalierungsfaktor 4?
	brne	grch50			;nein -> weiter testen
	subi	r18,30			;sonst 30 subtrahieren
	sbci	r19,0			;bertrag, Ergebnis positiv?
	brcc	grch10			;ja -> Schleife
	ret				;sonst Ende
;
grch50:	cpi	r21,8			;Skalierungsfaktor 8?
	brne	grch60			;nein -> weiter testen
	subi	r18,50			;sonst 50 subtrahieren
	sbci	r19,0			;bertrag, Ergebnis positiv?
	brcc	grch10			;ja -> Schleife
	ret				;sonst Ende
;
grch60:	cpi	r21,16			;Skalierungsfaktor 16?
	brne	grch70			;nein -> weiter testen
	subi	r18,100			;sonst 100 subtrahieren
	sbci	r19,0			;bertrag, Ergebnis positiv?
	brcc	grch10			;ja -> Schleife
	ret				;sonst Ende
;
grch70:	cpi	r21,32			;Skalierungsfaktor 32?
	brne	grch80			;nein -> weiter testen
	subi	r18,200			;sonst 200 subtrahieren
	sbci	r19,0			;bertrag, Ergebnis positiv?
	brcc	grch10			;ja -> Schleife
	ret				;sonst Ende
;
grch80:	cpi	r21,64			;Skalierungsfaktor 64?
	brne	grch90			;nein -> weiter
	subi	r18,low(500)		;sonst 500 subtrahieren
	sbci	r19,high(500)		;bertrag, Ergebnis positiv?
	brcc	grch10			;ja -> Schleife
	ret				;sonst Ende
;
grch90:	subi	r18,low(1000)		;sonst 1000 subtrahieren
	sbci	r19,high(1000)		;bertrag, Ergebnis positiv?
	brcc	grch10			;ja -> Schleife
	ret				;sonst Ende
;
; Unterprogramm (Grafikanzeige) zum Ermitteln des nchsten Skalenwertes
; Register: r18,r19
;	    r16,r17 <- Temperaturwert im Zweierkomplement
;	    r21     <- Skalierungsfaktor
;
grysca:	clr	r19			;H-Byte lschen
	cpi	r21,1			;Skalierungsfaktor 1?
	brne	grys10			;nein -> weiter testen
	ldi	r18,10			;sonst
	ldi	r19,0			;Wert 10 laden
	rjmp	grys80
;
grys10:	cpi	r21,2			;Skalierungsfaktor 2?
	brne	grys20			;nein -> weiter testen
	ldi	r18,20			;sonst Wert 20 laden
	rjmp	grys80
;
grys20:	cpi	r21,4			;Skalierungsfaktor 4?
	brne	grys30			;nein -> weiter testen
	ldi	r18,30			;sonst Wert 30 laden
	rjmp	grys80
;
grys30:	cpi	r21,8			;Skalierungsfaktor 8?
	brne	grys40			;nein -> weiter testen
	ldi	r18,50			;sonst Wert 50 laden
	rjmp	grys80
;
grys40:	cpi	r21,16			;Skalierungsfaktor 16?
	brne	grys50			;nein -> weiter testen
	ldi	r18,100			;sonst Wert 100 laden
	rjmp	grys80
;
grys50:	cpi	r21,32			;Skalierungsfaktor 32?
	brne	grys60			;nein -> weiter testen
	ldi	r18,200			;sonst Wert 200 laden
	rjmp	grys80
;
grys60:	cpi	r21,64			;Skalierungsfaktor 64?
	brne	grys70			;nein -> weiter
	ldi	r18,low(500)		;sonst
	ldi	r19,high(500)		;Wert 500 laden
	rjmp	grys80
;
grys70:	ldi	r18,low(1000)
	ldi	r19,high(1000)		;Wert 1000 laden
grys80:	add	r16,r18
	adc	r17,r19			;Y-Skalenwert +10
	ret
;
; Unterprogramm (Grafikanzeige) zum Extrahieren des Datums aus einem
; EEPROM-Datensatz und Ausgabe auf dem LCD
; Register: r0,r1,r13,r14,r15,r16,r17,r18,r19,r23,r24,zl,zh,t
;	    r8-r11 <- Zeitdaten im Format eines EEPROM-Datensatzes
;	    r20	   <- X-Koordinate (0-239)
;	    r21    <- Y-Koordinate (0-127)
;
grelcd:	mov	r18,r10			;komprimierte Zeit Byte 2 holen
	andi	r18,0x1f		;nicht relevante Bits filtern
	rcall	bytasc			;in ASCII-Wert wandeln
	mov	r13,r16			;Tag Einer sichern
	mov	r22,r17			;Tag Zehner kopieren
	rcall	asclcd			;ausgeben
	mov	r22,r13			;gesicherten Tag Einer holen
	rcall	asclcd			;ausgeben
	ldi	r22,'.'			;Trennzeichen
	rcall	asclcd			;ausgeben
	mov	r18,r11			;komprimierte Zeit Byte 3 holen
	bst	r18,0			;LSB sichern, gehrt zum Monat
	mov	r18,r10			;komprimierte Zeit Byte 2 holen
	ror	r18			;Monat nach rechts schieben
	bld	r18,7			;LSB von Byte 3 holen, Monat an
	swap	r18			;richtige Position schieben
	andi	r18,0x0f		;Nicht relevante Bits filtern
	rcall	bytasc			;in ASCII-Wert wandeln
	mov	r13,r16			;Monat Einer sichern
	mov	r22,r17			;Monat Zehner kopieren
	rcall	asclcd			;ausgeben
	mov	r22,r13			;gesicherten Monat Einer holen
	rcall	asclcd			;ausgeben
	ret
;
; Unterprogramm (Grafikanzeige) zum Extrahieren der Zeit aus einem
; EEPROM-Datensatz und Ausgabe auf dem LCD
; Register: r0,r1,r13,r14,r15,r16,r17,r18,r19,r23,r24,zl,zh,t
;	    r8-r11 <- Zeitdaten im Format eines EEPROM-Datensatzes
;	    r20	   <- X-Koordinate (0-239)
;	    r21    <- Y-Koordinate (0-127)
;
grelc2:	mov	r18,r9			;Stundenwert holen
	call	bytasc			;in ASCII wandeln
	cpi	r17,'0'			;Zehner=0?
	brne	gre010			;nein -> weiter
	ldi	r17,' '			;sonst durch Leerz. ersetzen
gre010:	mov	r13,r16			;Einer sichern
	mov	r22,r17			;Zehner holen
	call	asclcd			;ausgeben
	mov	r22,r13			;gesicherte Einer holen
	call	asclcd			;ausgeben
	ldi	r22,':'			;Doppelpunkt
	call	asclcd			;ausgeben
	mov	r18,r8			;Minutenwert holen
	call	bytasc			;in ASCII wandeln
	mov	r13,r16			;Einer sichern
	mov	r22,r17			;Zehner holen
	call	asclcd			;ausgeben
	mov	r22,r13			;gesicherte Einer holen
	call	asclcd			;ausgeben
	ret
;
; Unterprogramm (Grafikanzeige) zum Berechnen der X-Koordinaten fr das
; Fadenkreuz
; Register: r16,r17,r19,x
;	    r18	<- Fadenkreuz-Position (0-221)
;
grscxp:	clr	r16
	sts	grry1p,r16		;Y-Position 1 lschen
	sts	grry2p,r16		;Y-Position 2 lschen
	ldi	xl,low(grbuff)
	ldi	xh,high(grbuff)		;Basisadresse Grafikdatenpuffer
	ldi	r19,8			;Offset 8 Bytes
	mul	r18,r19			;Puffer-Offset berechnen
	add	xl,r0			;Pufferadresse ermitteln
	adc	xh,r1			;bertrag
	adiw	xl,4			;Adresse auf Temperaturwert 1
;
	ld	r16,x+
	ld	r17,x+			;Temperaturwert 1 holen
	cpi	r17,0x80		;Temperaturwert 1 ungltig?
	breq	grsc40			;ja -> berspringen
	lds	r18,grcen1
	lds	r19,grcen1+1		;Mittelpunktwert 1 holen
	sub	r16,r18
	sbc	r17,r19			;Temperaturwert - Mittelpunktw.
	lds	r18,grsca1		;Skalierungsfaktor 1 holen
grsc10:	lsr	r18			;Halbierung notwendig?
	breq	grsc20			;nein -> weiter
	asr	r17
	ror	r16			;sonst Anzeigewert halbieren
	brcc	grsc10			;Runden? nein -> Schleife
	inc	r16			;sonst aufrunden
	brne	grsc10			;bertrag? nein -> Schleife
	inc	r17			;sonst bertrag addieren
	rjmp	grsc10			;Schleife
;
grsc20:	ldi	r19,59			;Y-Position 1 Mittelwert=59
	lds	r18,grflag		;Grafik-Flags holen
	bst	r18,4			;Doppelgrafik?
	brtc	grsc30			;nein -> Y-Position ok
	ldi	r19,37			;sonst Y-Position Mittelwert=37
grsc30:	sub	r19,r16			;Y-Pixel-Position ermitteln
	sts	grry1p,r19		;Y-Position 1 speichern
;
grsc40:	lds	r18,grflag		;Grafik-Flags holen
	bst	r18,4			;Doppelgrafik?
	brtc	grsc70			;nein -> nur ein Diagramm
	ld	r16,x+			;sonst
	ld	r17,x+			;Temperaturwert 2 holen
	cpi	r17,0x80		;Temperaturwert 2 ungltig?
	breq	grsc70			;ja -> berspringen
	lds	r18,grcen2
	lds	r19,grcen2+1		;Mittelpunktwert 2 holen
	sub	r16,r18
	sbc	r17,r19			;Temperaturwert - Mittelpunktw.
	lds	r18,grsca2		;Skalierungsfaktor 2 holen
grsc50:	lsr	r18			;Halbierung notwendig?
	breq	grsc60			;nein -> weiter
	asr	r17
	ror	r16			;sonst Anzeigewert halbieren
	brcc	grsc50			;Runden? nein -> Schleife
	inc	r16			;sonst aufrunden
	brne	grsc50			;bertrag? nein -> Schleife
	inc	r17			;sonst bertrag addieren
	rjmp	grsc50			;Schleife
;
grsc60:	ldi	r19,81			;Y-Position 2 Mittelwert=81
	sub	r19,r16			;Y-Pixel-Position ermitteln
	sts	grry2p,r19		;Y-Position 1 speichern
grsc70:	ret

; Unterprogramm (Grafikanzeige) zum Sichern der LCD-Daten im Faden-
; kreuzbereich
; Register: r16,r17,r19,r20,x
;	    r18	<- Fadenkreuz-Position (0-221)
;
grscrb:	ldi	r16,16			;Anfangsposition der Diagramme
	add	r18,r16			;Fadenkreuz X-Position ermitt.
	lsr	r18
	lsr	r18
	lsr	r18			;Byte-Adresse ermitteln
	clr	r19			;H-Byte=0
	ldi	r16,low(16*30)		;Adresse von erster Diagramm-
	ldi	r17,high(16*30)		;zeile laden
	add	r18,r16			;Adresse von oberster Faden-
	adc	r19,r17			;kreuzzeile ermitteln
;
	ldi	xl,low(grrbu0)		;Zeiger auf LCD-Datenpuffer fr
	ldi	xh,high(grrbu0)		;Fadenkreuz
	ldi	r20,87			;87 Pixel in Y-Richtung
;
grsb10:	bst	r20,0			;Zhler geradzahlig?
	brtc	grsb20			;ja -> Ausgabe berspringen
	clt				;Datenbyte an LCD ausgeben
	mov	r17,r18			;Adresse L-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	mov	r17,r19			;Adresse H-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	set				;Kommando an LCD ausgeben
	ldi	r17,0x24		;Kommando "Adress Pointer Set"
	rcall	outlcd			;Byte an LCD ausgeben
	ldi	r17,0xc5		;Kommando "Data Read"
	rcall	outlcd			;Byte an LCD ausgeben
	rcall	in_lcd			;Datenbyte aus LCD einlesen
	st	x+,r17			;Datenbyte in Fadenkreuzpuffer
;
grsb20:	ldi	r16,30			;Adress-Offset zur nchsten
	clr	r17			;LCD-Zeile
	add	r18,r16			;Adresse von nchster Faden-
	adc	r19,r17			;kreuzzeile ermitteln
	dec	r20			;alle Datenbytes gelesen?
	brne	grsb10			;nein -> Schleife
;
; Fadenkreuz-Zeile Diagramm 1 sichern
;
	lds	r16,grry1p		;Y-Position von Wert 1 holen
	tst	r16			;Y-Position gltig?
	breq	grsb40			;nein -> keine Sicherung
	ldi	r18,2
	ldi	r19,0			;Adress-Offset Diagrammdaten
	ldi	r17,30			;Adress-Offset nchste Zeile
	mul	r16,r17			;Zeilenadresse berechnen
	add	r18,r0
	adc	r19,r1			;Adresse Messwertzeile ermitt.
	ldi	xl,low(grrbu1)		;Zeiger auf LCD-Datenpuffer fr
	ldi	xh,high(grrbu1)		;Fadenkreuzlinie 1
	ldi	r20,28			;28 Bytes aus LCD lesen
;
	clt				;Datenbyte an LCD ausgeben
	mov	r17,r18			;Adresse L-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	mov	r17,r19			;Adresse H-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	set				;Kommando an LCD ausgeben
	ldi	r17,0x24		;Kommando "Adress Pointer Set"
	rcall	outlcd			;Byte an LCD ausgeben
grsb30:	set				;Kommando an LCD ausgeben
	ldi	r17,0xc1		;Kommando "Data Read Increment"
	rcall	outlcd			;Byte an LCD ausgeben
	rcall	in_lcd			;Datenbyte aus LCD einlesen
	st	x+,r17			;Datenbyte in Fadenkreuzpuffer
	dec	r20			;alle Datenbytes gelesen?
	brne	grsb30			;nein -> Schleife
;
; Fadenkreuz-Zeile Diagramm 2 sichern
;
grsb40:	lds	r18,grflag		;Grafik-Flags holen
	bst	r18,4			;Doppelgrafik?
	brtc	grsb60			;nein -> nur ein Diagramm
	lds	r16,grry2p		;Y-Position von Wert 2 holen
	tst	r16			;Y-Position gltig?
	breq	grsb60			;nein -> keine Sicherung
	ldi	r18,2
	ldi	r19,0			;Adress-Offset Diagrammdaten
	ldi	r17,30			;Adress-Offset nchste Zeile
	mul	r16,r17			;Zeilenadresse berechnen
	add	r18,r0
	adc	r19,r1			;Adresse Messwertzeile ermitt.
	ldi	xl,low(grrbu2)		;Zeiger auf LCD-Datenpuffer fr
	ldi	xh,high(grrbu2)		;Fadenkreuzlinie 2
	ldi	r20,28			;28 Bytes aus LCD lesen
;
	clt				;Datenbyte an LCD ausgeben
	mov	r17,r18			;Adresse L-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	mov	r17,r19			;Adresse H-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	set				;Kommando an LCD ausgeben
	ldi	r17,0x24		;Kommando "Adress Pointer Set"
	rcall	outlcd			;Byte an LCD ausgeben
grsb50:	set				;Kommando an LCD ausgeben
	ldi	r17,0xc1		;Kommando "Data Read Increment"
	rcall	outlcd			;Byte an LCD ausgeben
	rcall	in_lcd			;Datenbyte aus LCD einlesen
	st	x+,r17			;Datenbyte in Fadenkreuzpuffer
	dec	r20			;alle Datenbytes gelesen?
	brne	grsb50			;nein -> Schleife
grsb60:	ret
;
; Unterprogramm (Grafikanzeige) zum Wiederherstellen der LCD-Daten im
; Fadenkreuzbereich
; Register: r16,r17,r19,r20,x
;	    r18	<- Fadenkreuz-Position (0-221)
;
grscrr:	ldi	r16,16			;Anfangsposition der Diagramme
	add	r18,r16			;Fadenkreuz X-Position ermitt.
	lsr	r18
	lsr	r18
	lsr	r18			;Byte-Adresse ermitteln
	clr	r19			;H-Byte=0
	ldi	r16,low(16*30)		;Adresse von erster Diagramm-
	ldi	r17,high(16*30)		;zeile laden
	add	r18,r16			;Adresse von oberster Faden-
	adc	r19,r17			;kreuzzeile ermitteln
;
	ldi	xl,low(grrbu0)		;Zeiger auf LCD-Datenpuffer fr
	ldi	xh,high(grrbu0)		;Fadenkreuz
	ldi	r20,87			;87 Pixel in Y-Richtung
;
grsr10:	bst	r20,0			;Zhler geradzahlig?
	brtc	grsr20			;ja -> Ausgabe berspringen
	clt				;Datenbyte an LCD ausgeben
	mov	r17,r18			;Adresse L-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	mov	r17,r19			;Adresse H-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	set				;Kommando an LCD ausgeben
	ldi	r17,0x24		;Kommando "Adress Pointer Set"
	rcall	outlcd			;Byte an LCD ausgeben
	clt				;Datenbyte an LCD ausgeben
	ld	r17,x+			;Datenbyte aus Fadenkreuzpuffer
	rcall	outlcd			;Byte an LCD ausgeben
	set				;Kommando an LCD ausgeben
	ldi	r17,0xc4		;Kommando "Data Write"
	rcall	outlcd			;Byte an LCD ausgeben
;
grsr20:	ldi	r16,30			;Adress-Offset zur nchsten
	clr	r17			;LCD-Zeile
	add	r18,r16			;Adresse von nchster Faden-
	adc	r19,r17			;kreuzzeile ermitteln
	dec	r20			;alle Datenbytes gelesen?
	brne	grsr10			;nein -> Schleife
;
; Fadenkreuz-Zeile Diagramm 1 wiederherstellen
;
	lds	r16,grry1p		;Y-Position von Wert 1 holen
	tst	r16			;Y-Position gltig?
	breq	grsr40			;nein -> nicht wiederherstellen
	ldi	r18,2
	ldi	r19,0			;Adress-Offset Diagrammdaten
	ldi	r17,30			;Adress-Offset nchste Zeile
	mul	r16,r17			;Zeilenadresse berechnen
	add	r18,r0
	adc	r19,r1			;Adresse Wertzeile ermitteln
	ldi	xl,low(grrbu1)		;Zeiger auf LCD-Datenpuffer fr
	ldi	xh,high(grrbu1)		;Fadenkreuzlinie 1
	ldi	r20,28			;28 Bytes an LCD ausgeben
;
	clt				;Datenbyte an LCD ausgeben
	mov	r17,r18			;Adresse L-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	mov	r17,r19			;Adresse H-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	set				;Kommando an LCD ausgeben
	ldi	r17,0x24		;Kommando "Adress Pointer Set"
	rcall	outlcd			;Byte an LCD ausgeben
grsr30:	clt				;Datenbyte an LCD ausgeben
	ld	r17,x+			;Datenbyte aus Fadenkreuzpuffer
	rcall	outlcd			;Byte an LCD ausgeben
	set				;Kommando an LCD ausgeben
	ldi	r17,0xc0		;Kommando "Data Write Increm."
	rcall	outlcd			;Byte an LCD ausgeben
	dec	r20			;alle Datenbytes gelesen?
	brne	grsr30			;nein -> Schleife
;
; Fadenkreuz-Zeile Diagramm 2 wiederherstellen
;
grsr40:	lds	r18,grflag		;Grafik-Flags holen
	bst	r18,4			;Doppelgrafik?
	brtc	grsr60			;nein -> nur ein Diagramm
	lds	r16,grry2p		;Y-Position von Wert 2 holen
	tst	r16			;Y-Position gltig?
	breq	grsr60			;nein -> nicht wiederherstellen
	ldi	r18,2
	ldi	r19,0			;Adress-Offset Diagrammdaten
	ldi	r17,30			;Adress-Offset nchste Zeile
	mul	r16,r17			;Zeilenadresse berechnen
	add	r18,r0
	adc	r19,r1			;Adresse Wertzeile ermitteln
	ldi	xl,low(grrbu2)		;Zeiger auf LCD-Datenpuffer fr
	ldi	xh,high(grrbu2)		;Fadenkreuzlinie 2
	ldi	r20,28			;28 Bytes an LCD ausgeben
;
	clt				;Datenbyte an LCD ausgeben
	mov	r17,r18			;Adresse L-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	mov	r17,r19			;Adresse H-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	set				;Kommando an LCD ausgeben
	ldi	r17,0x24		;Kommando "Adress Pointer Set"
	rcall	outlcd			;Byte an LCD ausgeben
grsr50:	clt				;Datenbyte an LCD ausgeben
	ld	r17,x+			;Datenbyte aus Fadenkreuzpuffer
	rcall	outlcd			;Byte an LCD ausgeben
	set				;Kommando an LCD ausgeben
	ldi	r17,0xc0		;Kommando "Data Write Increm."
	rcall	outlcd			;Byte an LCD ausgeben
	dec	r20			;alle Datenbytes gelesen?
	brne	grsr50			;nein -> Schleife
grsr60:	ret
;
; Unterprogramm (Grafikanzeige) zum Zeichnen des Fadenkreuzes, es
; werden die gesicherten LCD-Daten ausgegeben, Pixel an der Fadenkreuz-
; Position werden gesetzt, an den gespeicherten Y-Positionen wird eine
; Rasterlinie gezeichnet
; Register: r16,r17,r19,r20,r21,r22,x
;	    r18	<- Fadenkreuz-Position (0-221)
;
grcros:	mov	r22,r18			;Fadenkreuz-Position sichern
	mov	r20,r18			;Fadenkreuz-Position kopieren
	andi	r20,7			;Pixelposition filtern
	ldi	r21,0x80		;Maske Anfangswert
grcr10:	tst	r20			;Maske auf richtiger Position?
	breq	grcr20			;ja -> weiter
	lsr	r21			;Maske verschieben
	dec	r20			;Positionszhler vermindern
	rjmp	grcr10			;Schleife
;
grcr20:	ldi	r16,16			;Anfangsposition der Diagramme
	add	r18,r16			;Fadenkreuz X-Position ermitt.
	lsr	r18
	lsr	r18
	lsr	r18			;Byte-Adresse ermitteln
	clr	r19			;H-Byte=0
	ldi	r16,low(16*30)		;Adresse erste Diagrammzeile
	ldi	r17,high(16*30)		;laden
	add	r18,r16			;Adresse von oberster Faden-
	adc	r19,r17			;kreuzzeile ermitteln
;
	ldi	xl,low(grrbu0)		;Zeiger auf LCD-Datenpuffer fr
	ldi	xh,high(grrbu0)		;Fadenkreuz
	ldi	r20,87			;87 Pixel in Y-Richtung
;
grcr30:	bst	r20,0			;Zhler geradzahlig?
	brtc	grcr40			;ja -> Ausgabe berspringen
	clt				;Datenbyte an LCD ausgeben
	mov	r17,r18			;Adresse L-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	mov	r17,r19			;Adresse H-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	set				;Kommando an LCD ausgeben
	ldi	r17,0x24		;Kommando "Adress Pointer Set"
	rcall	outlcd			;Byte an LCD ausgeben
	ld	r17,x+			;Datenbyte aus Fadenkreuzpuffer
	or	r17,r21			;Fadenkreuz-Pixel setzen
	clt				;Datenbyte an LCD ausgeben
	rcall	outlcd			;Byte an LCD ausgeben
	set				;Kommando an LCD ausgeben
	ldi	r17,0xc4		;Kommando "Data Write"
	rcall	outlcd			;Byte an LCD ausgeben
;
grcr40:	ldi	r16,30			;Adress-Offset zur nchsten
	clr	r17			;LCD-Zeile
	add	r18,r16			;Adresse von nchster Faden-
	adc	r19,r17			;kreuzzeile ermitteln
	dec	r20			;alle Datenbytes gelesen?
	brne	grcr30			;nein -> Schleife
;
; Fadenkreuz-Zeile Diagramm 1 zeichnen
;
	lds	r16,grry1p		;Y-Position von Wert 1 holen
	tst	r16			;Y-Position gltig?
	breq	grcr60			;nein -> keine Linie zeichnen
	ldi	r18,2
	ldi	r19,0			;Adress-Offset Diagrammdaten
	ldi	r17,30			;Adress-Offset nchste Zeile
	mul	r16,r17			;Zeilenadresse berechnen
	add	r18,r0
	adc	r19,r1			;Adresse Wertzeile ermitteln
	ldi	xl,low(grrbu1)		;Zeiger auf LCD-Datenpuffer fr
	ldi	xh,high(grrbu1)		;Fadenkreuzlinie 1
	ldi	r20,28			;28 Bytes an LCD ausgeben
;
	clt				;Datenbyte an LCD ausgeben
	mov	r17,r18			;Adresse L-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	mov	r17,r19			;Adresse H-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	set				;Kommando an LCD ausgeben
	ldi	r17,0x24		;Kommando "Adress Pointer Set"
	rcall	outlcd			;Byte an LCD ausgeben
grcr50:	clt				;Datenbyte an LCD ausgeben
	ld	r17,x+			;Datenbyte aus Fadenkreuzpuffer
	ori	r17,0x55		;Rasterlinie berlagern
	rcall	outlcd			;Byte an LCD ausgeben
	set				;Kommando an LCD ausgeben
	ldi	r17,0xc0		;Kommando "Data Write Increm."
	rcall	outlcd			;Byte an LCD ausgeben
	dec	r20			;alle Datenbytes gelesen?
	brne	grcr50			;nein -> Schleife
;
; Fadenkreuz-Zeile Diagramm 2 zeichnen
;
grcr60:	lds	r18,grflag		;Grafik-Flags holen
	bst	r18,4			;Doppelgrafik?
	brtc	grcr80			;nein -> nur ein Diagramm
	lds	r16,grry2p		;Y-Position von Wert 2 holen
	tst	r16			;Y-Position gltig?
	breq	grcr80			;nein -> keine Linie zeichnen
	ldi	r18,2
	ldi	r19,0			;Adress-Offset Diagrammdaten
	ldi	r17,30			;Adress-Offset nchste Zeile
	mul	r16,r17			;Zeilenadresse berechnen
	add	r18,r0
	adc	r19,r1			;Adresse Wertzeile ermitteln
	ldi	xl,low(grrbu2)		;Zeiger auf LCD-Datenpuffer fr
	ldi	xh,high(grrbu2)		;Fadenkreuzlinie 2
	ldi	r20,28			;28 Bytes an LCD ausgeben
;
	clt				;Datenbyte an LCD ausgeben
	mov	r17,r18			;Adresse L-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	mov	r17,r19			;Adresse H-Byte kopieren
	rcall	outlcd			;Byte an LCD ausgeben
	set				;Kommando an LCD ausgeben
	ldi	r17,0x24		;Kommando "Adress Pointer Set"
	rcall	outlcd			;Byte an LCD ausgeben
grcr70:	clt				;Datenbyte an LCD ausgeben
	ld	r17,x+			;Datenbyte aus Fadenkreuzpuffer
	ori	r17,0x55		;Rasterlinie berlagern
	rcall	outlcd			;Byte an LCD ausgeben
	set				;Kommando an LCD ausgeben
	ldi	r17,0xc0		;Kommando "Data Write Increm."
	rcall	outlcd			;Byte an LCD ausgeben
	dec	r20			;alle Datenbytes gelesen?
	brne	grcr70			;nein -> Schleife
grcr80:	ret
;
; Unterprogramm zum Schreiben von Default-Werten und einer CRC8
; Checksumme ins EEPROM
; Register: r14,r15,r16,r17,r18,r19,r20,z
;
; Defaultwerte fr Sensor/Temperatur-Mapping-Tabelle im EEPROM speich.
;
eedcfg:	ldi	r20,16			;Mapping Tabelle, 16 Werte
	ldi	r16,low(etemap)
	ldi	r17,high(etemap)	;EEPROM-Adresse der Tabelle
	ldi	zl,byte3(dtemap*2)
	out	rampz,zl
	ldi	zl,low(dtemap*2)
	ldi	zh,high(dtemap*2)	;Zeiger auf Default-Daten
eedtma:	elpm	r18,z+			;Anzeigeposition holen
	rcall	eewrit			;Daten im EEPROM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eedtma			;nein -> Schleife
;
; Defaultwerte fr Temperatur-Log-Mapping-Tabelle im EEPROM speichern
;
	ldi	r20,16			;Mapping Tabelle, 16 Werte
	ldi	r16,low(elomap)
	ldi	r17,high(elomap)	;EEPROM-Adresse der Tabelle
	ldi	zl,byte3(dtemap*2)
	out	rampz,zl
	ldi	zl,low(dtemap*2)
	ldi	zh,high(dtemap*2)	;Zeiger auf Default-Daten
eedlma:	elpm	r18,z+			;Anzeigeposition holen
	rcall	eewrit			;Daten im EEPROM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eedlma			;nein -> Schleife
;
; Defaultwerte fr Temperaturtexte im EERPOM speichern
;
	ldi	r20,16			;16 Texte
	ldi	r16,low(etemtx)
	ldi	r17,high(etemtx)	;EEPROM-Adresse der Texte
	ldi	zl,byte3(dtemtx*2)
	out	rampz,zl
	ldi	zl,low(dtemtx*2)
	ldi	zh,high(dtemtx*2)	;Zeiger auf Default-Daten
eedtt1:	ldi	r19,9			;jeweils 9 Zeichen Lnge
eedtt2:	elpm	r18,z+			;Textzeichen holen
	rcall	eewrit			;Daten im EEPROM speichern
	dec	r19			;Text komplett?
	brne	eedtt2			;nein -> Schleife
	dec	r20			;alle Texte gespeichert?
	brne	eedtt1			;nein -> Schleife
;
; Defaultwerte fr obere Temperaturlimits im EERPOM speichern
;
	ldi	r20,16*2		;16 Werte (16*2 Bytes)
	ldi	r16,low(etelih)
	ldi	r17,high(etelih)	;Zeiger auf EERPOM Daten
	ldi	zl,byte3(dtelih*2)
	out	rampz,zl
	ldi	zl,low(dtelih*2)
	ldi	zh,high(dtelih*2)	;Zeiger auf Default-Daten
eedlih:	elpm	r18,z+			;Byte holen
	rcall	eewrit			;Daten im EEPROM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eedlih			;nein -> Schleife
;
; Defaultwerte fr untere Temperaturlimits im EERPOM speichern
;
	ldi	r20,16*2		;16 Werte (16*2 Bytes)
	ldi	r16,low(etelil)
	ldi	r17,high(etelil)	;Zeiger auf EERPOM Daten
	ldi	zl,byte3(dtelil*2)
	out	rampz,zl
	ldi	zl,low(dtelil*2)
	ldi	zh,high(dtelil*2)	;Zeiger auf Default-Daten
eedlil:	elpm	r18,z+			;Byte holen
	rcall	eewrit			;Daten im EEPROM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eedlil			;nein -> Schleife
;
; Defaultwerte fr Alarm-Mapping-Tabelle im EEPROM speichern
;
	ldi	r20,4			;Mapping Tabelle, 4 Werte
	ldi	r16,low(ealmap)
	ldi	r17,high(ealmap)	;EEPROM-Adresse der Tabelle
	ldi	zl,byte3(dalmap*2)
	out	rampz,zl
	ldi	zl,low(dalmap*2)
	ldi	zh,high(dalmap*2)	;Zeiger auf Default-Daten
eedama:	elpm	r18,z+			;Anzeigeposition holen
	rcall	eewrit			;Daten im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eedama			;nein -> Schleife
;
; Defaultwerte fr Alarm-Invertierungs-Tabelle im EEPROM speichern
;
	ldi	r20,4			;Invertierungs-Tabelle, 4 Werte
	ldi	r16,low(ealinv)
	ldi	r17,high(ealinv)	;EEPROM-Adresse der Tabelle
	ldi	zl,byte3(dalinv*2)
	out	rampz,zl
	ldi	zl,low(dalinv*2)
	ldi	zh,high(dalinv*2)	;Zeiger auf Default-Daten
eedinv:	elpm	r18,z+			;Invertierung holen
	rcall	eewrit			;Daten im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eedinv			;nein -> Schleife
;
; Defaultwerte fr Alarmtexte im EERPOM speichern
;
	ldi	r20,4			;4 Texte
	ldi	r16,low(ealmtx)
	ldi	r17,high(ealmtx)	;EEPROM-Adresse der Texte
	ldi	zl,byte3(dalmtx*2)
	out	rampz,zl
	ldi	zl,low(dalmtx*2)
	ldi	zh,high(dalmtx*2)	;Zeiger auf Default-Daten
eedat1:	ldi	r19,11			;jeweils 11 Zeichen Lnge
eedat2:	elpm	r18,z+			;Textzeichen holen
	rcall	eewrit			;Daten im EEPROM speichern
	dec	r19			;Text komplett?
	brne	eedat2			;nein -> Schleife
	adiw	zl,1			;Adresse korrigieren
	dec	r20			;alle Texte gespeichert?
	brne	eedat1			;nein -> Schleife
;
; Defaultwert fr Sensornummer fr Groanzeige im EEPROM speichern
;
	ldi	r16,low(ebigtm)
	ldi	r17,high(ebigtm)	;EEPROM-Adresse Groanzeige
	ldi	zl,byte3(dbigtm*2)
	out	rampz,zl
	ldi	zl,low(dbigtm*2)
	ldi	zh,high(dbigtm*2)	;Zeiger auf Default-Daten
	elpm	r18,z			;Byte holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Defaultwerte fr Signalnummern-Tabelle im EEPROM speichern
;
	ldi	r20,8			;Signalnummern-Tabelle, 8 Werte
	ldi	r16,low(esignr)
	ldi	r17,high(esignr)	;EEPROM-Adresse der Tabelle
	ldi	zl,byte3(dsignr*2)
	out	rampz,zl
	ldi	zl,low(dsignr*2)
	ldi	zh,high(dsignr*2)	;Zeiger auf Default-Daten
eedsnu:	elpm	r18,z+			;Signalnummer holen
	rcall	eewrit			;Daten im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eedsnu			;nein -> Schleife
;
; Defaultwerte fr Signaldauer-Tabelle im EEPROM speichern
;
	ldi	r20,8			;Signaldauer-Tabelle, 8 Werte
	ldi	r16,low(esigdu)
	ldi	r17,high(esigdu)	;EEPROM-Adresse der Tabelle
	ldi	zl,byte3(dsigdu*2)
	out	rampz,zl
	ldi	zl,low(dsigdu*2)
	ldi	zh,high(dsigdu*2)	;Zeiger auf Default-Daten
eedsdu:	elpm	r18,z+			;Signaldauer holen
	rcall	eewrit			;Daten im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eedsdu			;nein -> Schleife
;
; Default-Timer-Wert zur Erzeugung der Signalfrequenz im EEPROM speich.
;
	ldi	r16,low(esigtm)
	ldi	r17,high(esigtm)	;EEPROM-Adresse Signal-Frequenz
	ldi	zl,byte3(dsigtm*2)
	out	rampz,zl
	ldi	zl,low(dsigtm*2)
	ldi	zh,high(dsigtm*2)	;Zeiger auf Default-Daten
	elpm	r18,z			;Byte holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Default-Lautstrke-Wert 1 (Tag) im EEPROM speichern
;
	ldi	r16,low(esigv1)
	ldi	r17,high(esigv1)	;EEPROM-Adresse Lautstrke 1
	ldi	zl,byte3(dsigv1*2)
	out	rampz,zl
	ldi	zl,low(dsigv1*2)
	ldi	zh,high(dsigv1*2)	;Zeiger auf Default-Daten
	elpm	r18,z			;Byte holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Default-Lautstrke-Wert 2 (Nacht) im EEPROM speichern
;
	ldi	r16,low(esigv2)
	ldi	r17,high(esigv2)	;EEPROM-Adresse Lautstrke 2
	ldi	zl,byte3(dsigv2*2)
	out	rampz,zl
	ldi	zl,low(dsigv2*2)
	ldi	zh,high(dsigv2*2)	;Zeiger auf Default-Daten
	elpm	r18,z			;Byte holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Default-Nachtzeiten fr akustische Signalisierung im EEPROM speichern
;
	ldi	r20,4			;Nachtschaltung, 2x2 Werte
	ldi	r16,low(esigni)
	ldi	r17,high(esigni)	;EERPOM-Adresse Startzeit
	ldi	zl,byte3(dsigni*2)
	out	rampz,zl
	ldi	zl,low(dsigni*2)
	ldi	zh,high(dsigni*2)	;Zeiger auf Default-Daten
eedsig:	elpm	r18,z+			;Wert holen
	rcall	eewrit			;Byte im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eedsig			;nein -> Schleife
;
; Default-Nachtzeiten fr die LCD-Hintergrundbeleuchtung im EEPROM sp.
;
	ldi	r20,4			;Nachtschaltung, 2x2 Werte
	ldi	r16,low(enigbl)
	ldi	r17,high(enigbl)	;EERPOM-Adresse Startzeit
	ldi	zl,byte3(dnigbl*2)
	out	rampz,zl
	ldi	zl,low(dnigbl*2)
	ldi	zh,high(dnigbl*2)	;Zeiger auf Default-Daten
eednib:	elpm	r18,z+			;Wert holen
	rcall	eewrit			;Byte im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eednib			;nein -> Schleife
;
; Default-Auto-Reset-Zeiten fr Min- und Max-Werte im EEPROM speichern
;
	ldi	r20,4			;Reset-Zeiten, 2x2 Werte
	ldi	r16,low(earesm)
	ldi	r17,high(earesm)	;EERPOM-Adresse Startzeit
	ldi	zl,byte3(daresm*2)
	out	rampz,zl
	ldi	zl,low(daresm*2)
	ldi	zh,high(daresm*2)	;Zeiger auf Default-Daten
eedare:	elpm	r18,z+			;Wert holen
	rcall	eewrit			;Byte im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eedare			;nein -> Schleife
;
; Defaultwert fr Konfigurationsbyte 1 im EEPROM speichern
;
	ldi	r16,low(econf1)
	ldi	r17,high(econf1)	;EEPROM-Adresse Konfig-Byte 1
	ldi	zl,byte3(dconf1*2)
	out	rampz,zl
	ldi	zl,low(dconf1*2)
	ldi	zh,high(dconf1*2)	;Zeiger auf Default-Daten
	elpm	r18,z			;Byte holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Defaultwert fr Konfigurationsbyte 2 im EEPROM speichern
;
	ldi	r16,low(econf2)
	ldi	r17,high(econf2)	;EEPROM-Adresse Konfig-Byte 2
	ldi	zl,byte3(dconf2*2)
	out	rampz,zl
	ldi	zl,low(dconf2*2)
	ldi	zh,high(dconf2*2)	;Zeiger auf Default-Daten
	elpm	r18,z			;Byte holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Default-Schwellwert fr die Steuerung der Hintergrundbeleuchtung sp.
;
	ldi	r16,low(elight)
	ldi	r17,high(elight)	;EEPROM-Adresse Lichtschwelle
	ldi	zl,byte3(dlight*2)
	out	rampz,zl
	ldi	zl,low(dlight*2)
	ldi	zh,high(dlight*2)	;Zeiger auf Default-Daten
	elpm	r18,z			;Byte holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Default-Baudrate fr die PC-Verbindung (RS232-0) im EEPROM speichern
;
	ldi	r16,low(ebaud0)
	ldi	r17,high(ebaud0)	;EEPROM-Adr. Baudrate RS232-0
	ldi	zl,byte3(dbaud0*2)
	out	rampz,zl
	ldi	zl,low(dbaud0*2)
	ldi	zh,high(dbaud0*2)	;Zeiger auf Default-Daten
	elpm	r18,z			;Byte holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Defaultwert fr Meldungsfilter im EEPROM speichern
;
	ldi	r16,low(emsgfi)
	ldi	r17,high(emsgfi)	;EEPROM-Adresse Meldungsfilter
	ldi	zl,byte3(dmsgfi*2)
	out	rampz,zl
	ldi	zl,low(dmsgfi*2)
	ldi	zh,high(dmsgfi*2)	;Zeiger auf Default-Daten
	elpm	r18,z			;Byte holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Default-Grafik-Presets 1-24 im EEPROM speichern
;
eedgrp:	ldi	r20,24*5		;Grafik-Presets, 24x5 Werte
	ldi	r16,low(egrpre)
	ldi	r17,high(egrpre)	;EERPOM-Adresse Presets
	ldi	zl,byte3(dgrpre*2)
	out	rampz,zl
	ldi	zl,low(dgrpre*2)
	ldi	zh,high(dgrpre*2)	;Zeiger auf Default-Daten
eedg10:	elpm	r18,z+			;Wert holen
	rcall	eewrit			;Byte im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eedg10			;nein -> Schleife
;
; Default-Grafik-Preset-Texte 1-24 im EERPOM speichern
;
	ldi	r20,24			;24 Texte
	ldi	r16,low(egrprt)
	ldi	r17,high(egrprt)	;EEPROM-Adresse der Texte
	ldi	zl,byte3(dgrprt*2)
	out	rampz,zl
	ldi	zl,low(dgrprt*2)
	ldi	zh,high(dgrprt*2)	;Zeiger auf Default-Daten
eedg20:	ldi	r19,16			;jeweils 16 Zeichen Lnge
eedg30:	elpm	r18,z+			;Textzeichen holen
	rcall	eewrit			;Daten im EEPROM speichern
	dec	r19			;Text komplett?
	brne	eedg30			;nein -> Schleife
	dec	r20			;alle Texte gespeichert?
	brne	eedg20			;nein -> Schleife
;
; Default-Nachtzeiten fr das Bluetooth-Modul im EEPROM speichern
;
	ldi	r20,4			;Nachtschaltung, 2x2 Werte
	ldi	r16,low(enigbt)
	ldi	r17,high(enigbt)	;EERPOM-Adresse Startzeit
	ldi	zl,byte3(dnigbt*2)
	out	rampz,zl
	ldi	zl,low(dnigbt*2)
	ldi	zh,high(dnigbt*2)	;Zeiger auf Default-Daten
eednbt:	elpm	r18,z+			;Wert holen
	rcall	eewrit			;Byte im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eednbt			;nein -> Schleife
;
; ermittelten CRC-Wert im EEPROM speichern
;
	call	calcrc			;EEPROM CRC-Berechnung
	mov	r18,r20			;ermittelten CRC-Wert holen
	rcall	eewrit			;Byte im EERPOM speichern
	ret
;
; Unterprogramm zum Aktivieren der Konfigurationsspeicherung im EEPROM
; Register: r16
;
eewcon:	lds	r16,xflag3		;verschiedene Flags 3 holen
	sbr	r16,0x01		;Flag fr Konfigschreiben setz.
	sts	xflag3,r16		;Flags wieder speichern
	ret
;
; Unterprogramm zum Schreiben aller Konfigurationssdaten und einer CRC8
; Checksumme ins EEPROM
; Register: r14,r15,r16,r17,r18,r19,r20,x,y
;
; Sensor/Temperatur-Anzeige-Mapping-Tabelle im EEPROM speichern
;
eewcfg:	ldi	r20,16			;Mapping Tabelle, 16 Werte
	ldi	r16,low(etemap)
	ldi	r17,high(etemap)	;EEPROM-Adresse der Tabelle
	ldi	xl,low(temmap)
	ldi	xh,high(temmap)		;Zeiger auf RAM-Adresse
	ldi	yl,low(temflg)
	ldi	yh,high(temflg)		;Zeiger auf Temperaturflags
eewtma:	ld	r18,x+			;Anzeigeposition holen
	ld	r19,y+			;Temperatur-Flags holen
	bst	r19,7			;Inaktivitts-Flag holen
	bld	r18,7			;in Anzeigeposition einsetzen
	rcall	eewrit			;Daten im EEPROM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eewtma			;nein -> Schleife
;
; Sensor/Temperatur-Log-Mapping-Tabelle im EEPROM speichern
;
	ldi	r20,16			;Mapping Tabelle, 16 Werte
	ldi	r16,low(elomap)
	ldi	r17,high(elomap)	;EEPROM-Adresse der Tabelle
	ldi	xl,low(logmap)
	ldi	xh,high(logmap)		;Zeiger auf RAM-Adresse
eewlma:	ld	r18,x+			;Log-Position holen
	rcall	eewrit			;Daten im EEPROM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eewlma			;nein -> Schleife
;
; obere Temperaturlimits im EERPOM speichern
;
	ldi	r20,16*2		;16 Werte (16*2 Bytes) sichern
	ldi	r16,low(etelih)
	ldi	r17,high(etelih)	;Zeiger auf EERPOM Daten
	ldi	xl,low(temlih)
	ldi	xh,high(temlih)		;Zeiger auf obere Limits
eewlih:	ld	r18,x+			;Byte aus RAM holen
	rcall	eewrit			;Daten im EEPROM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eewlih			;nein -> Schleife
;
; untere Temperaturlimits im EERPOM speichern
;
	ldi	r20,16*2		;16 Werte (16*2 Bytes) sichern
	ldi	r16,low(etelil)
	ldi	r17,high(etelil)	;Zeiger auf EERPOM Daten
	ldi	xl,low(temlil)
	ldi	xh,high(temlil)		;Zeiger auf untere Limits
eewlil:	ld	r18,x+			;Byte aus RAM holen
	rcall	eewrit			;Daten im EEPROM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eewlil			;nein -> Schleife
;
; Alarm-Mapping-Tabelle im EEPROM speichern
;
	ldi	r20,4			;Mapping Tabelle, 4 Werte
	ldi	r16,low(ealmap)
	ldi	r17,high(ealmap)	;EEPROM-Adresse der Tabelle
	ldi	xl,low(almmap)
	ldi	xh,high(almmap)		;Zeiger auf RAM-Adresse
	ldi	yl,low(almflg)
	ldi	yh,high(almflg)		;Zeiger auf Alarmflags
eewama:	ld	r18,x+			;Anzeigeposition holen
	ld	r19,y+			;Alarmflags holen
	bst	r19,7			;Inaktivitts-Flag holen
	bld	r18,7			;in Anzeigeposition einsetzen
	rcall	eewrit			;Daten im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eewama			;nein -> Schleife
;
; Alarm-Invertierungs-Tabelle im EEPROM speichern
;
	ldi	r20,4			;Invertierungs-Tabelle, 4 Werte
	ldi	r16,low(ealinv)
	ldi	r17,high(ealinv)	;EEPROM-Adresse der Tabelle
	ldi	xl,low(alminv)
	ldi	xh,high(alminv)		;Zeiger auf RAM-Adresse
eewinv:	ld	r18,x+			;Invertierung holen
	rcall	eewrit			;Daten im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eewinv			;nein -> Schleife
;
; Sensornummer fr Groanzeige im EEPROM speichern
;
	ldi	r16,low(ebigtm)
	ldi	r17,high(ebigtm)	;EEPROM-Adresse Groanzeige
	lds	r18,bigtmp		;Sensornummer holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Signalnummern-Tabelle im EEPROM speichern
;
	ldi	r20,8			;Signalnummern-Tabelle, 8 Werte
	ldi	r16,low(esignr)
	ldi	r17,high(esignr)	;EEPROM-Adresse der Tabelle
	ldi	xl,low(signum)
	ldi	xh,high(signum)		;Zeiger auf RAM-Adresse
eewsnu:	ld	r18,x+			;Signalnummer holen
	rcall	eewrit			;Daten im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eewsnu			;nein -> Schleife
;
; Signaldauer-Tabelle im EEPROM speichern
;
	ldi	r20,8			;Signaldauer-Tabelle, 8 Werte
	ldi	r16,low(esigdu)
	ldi	r17,high(esigdu)	;EEPROM-Adresse der Tabelle
	ldi	xl,low(sigdur)
	ldi	xh,high(sigdur)		;Zeiger auf RAM-Adresse
eewsdu:	ld	r18,x+			;Signaldauer holen
	rcall	eewrit			;Daten im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eewsdu			;nein -> Schleife
;
; Timer-Wert zur Erzeugung der Signalfrequenz im EEPROM speichern
;
	ldi	r16,low(esigtm)
	ldi	r17,high(esigtm)	;EEPROM-Adresse Signal-Frequenz
	in	r18,ocr0		;Timer0-Wert holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Lautstrke-Wert 1 (Tag) im EEPROM speichern
;
	ldi	r16,low(esigv1)
	ldi	r17,high(esigv1)	;EEPROM-Adresse Lautstrke 1
	lds	r18,sigvo1		;Lautstrke-Wert 1 holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Lautstrke-Wert 2 (Nacht) im EEPROM speichern
;
	ldi	r16,low(esigv2)
	ldi	r17,high(esigv2)	;EEPROM-Adresse Lautstrke 2
	lds	r18,sigvo2		;Lautstrke-Wert 2 holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Nachtzeiten fr akustische Signalisierung im EEPROM speichern
;
	ldi	r20,4			;Nachtschaltung, 2x2 Werte
	ldi	r16,low(esigni)
	ldi	r17,high(esigni)	;EERPOM-Adresse Startzeit
	ldi	xl,low(signig)
	ldi	xh,high(signig)		;Zeiger auf RAM-Adresse
eewsig:	ld	r18,x+			;Wert aus RAM holen
	rcall	eewrit			;Byte im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eewsig			;nein -> Schleife
;
; Nachtzeiten fr die LCD-Hintergrundbeleuchtung im EEPROM speichern
;
	ldi	r20,4			;Nachtschaltung, 2x2 Werte
	ldi	r16,low(enigbl)
	ldi	r17,high(enigbl)	;EERPOM-Adresse Startzeit
	ldi	xl,low(nighbl)
	ldi	xh,high(nighbl)		;Zeiger auf RAM-Adresse
eewnib:	ld	r18,x+			;Wert aus RAM holen
	rcall	eewrit			;Byte im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eewnib			;nein -> Schleife
;
; Nachtzeiten fr das Bluetooth-Modul im EEPROM speichern
;
	ldi	r20,4			;Nachtschaltung, 2x2 Werte
	ldi	r16,low(enigbt)
	ldi	r17,high(enigbt)	;EERPOM-Adresse Startzeit
	ldi	xl,low(nighbt)
	ldi	xh,high(nighbt)		;Zeiger auf RAM-Adresse
eewnbt:	ld	r18,x+			;Wert aus RAM holen
	rcall	eewrit			;Byte im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eewnbt			;nein -> Schleife
;
; Auto-Reset-Zeiten fr Minimal- und Maximalwerte im EEPROM speichern
;
	ldi	r20,4			;Reset-Zeiten, 2x2 Werte
	ldi	r16,low(earesm)
	ldi	r17,high(earesm)	;EERPOM-Adresse Startzeit
	ldi	xl,low(aresmm)
	ldi	xh,high(aresmm)		;Zeiger auf RAM-Adresse
eeware:	ld	r18,x+			;Wert aus RAM holen
	rcall	eewrit			;Byte im EERPOM speichern
	dec	r20			;alle Werte gespeichert?
	brne	eeware			;nein -> Schleife
;
; Konfigurationsbyte 1 im EEPROM speichern
;
	ldi	r16,low(econf1)
	ldi	r17,high(econf1)	;EEPROM-Adresse Konfig-Byte 1
	lds	r18,confg1		;Konfig-Byte 1 holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Konfigurationsbyte 2 im EEPROM speichern
;
	ldi	r16,low(econf2)
	ldi	r17,high(econf2)	;EEPROM-Adresse Konfig-Byte 2
	lds	r18,confg2		;Konfig-Byte 2 holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Schwellwert fr die Steuerung der Hintergrundbeleuchtung speichern
;
	ldi	r16,low(elight)
	ldi	r17,high(elight)	;EEPROM-Adresse Lichtschwelle
	lds	r18,light		;Lichtschwellwert holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Baudrate fr die PC-Verbindung (RS232-0) im EEPROM speichern
;
	ldi	r16,low(ebaud0)
	ldi	r17,high(ebaud0)	;EEPROM-Adr. Baudrate RS232-0
	lds	r18,baud0		;Baudrate RS232-0 holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; Meldungsfilter im EEPROM speichern
;
	ldi	r16,low(emsgfi)
	ldi	r17,high(emsgfi)	;EEPROM-Adresse Meldungsfilter
	lds	r18,msgfil		;Meldungsfilter holen
	rcall	eewrit			;Byte im EERPOM speichern
;
; ermittelten CRC-Wert im EEPROM speichern, Konfig-Fehler lschen
;
eewcrc:	call	calcrc			;EEPROM CRC-Berechnung
	mov	r18,r20			;ermittelten CRC-Wert holen
	rcall	eewrit			;Byte im EERPOM speichern
	lds	r16,errflg		;Fehler-Flags holen
	cbr	r16,1<<2		;Konfigurationsfehler lschen
	sts	errflg,r16		;Fehler-Flags wieder speichern
	ret
;
; Unterprogramm zum Lesen eines Byte aus dem EEPROM, nach dem Lesen
; wird die EEPROM-Adresse um 1 erhht
; Register: r16 <- EEPROM-Adresse L
;	    r17 <- EEPROM-Adresse H
;	    r18 -> EEPROM-Daten
;
eeread:	sbic	eecr,eewe		;luft ein Schreibzyklus?
	rjmp	eeread			;ja -> warten
	out	eearl,r16		;EEPROM-Adresse L setzen
	out	eearh,r17		;EEPROM-Adresse H setzen
	sbi	eecr,eere		;EEPROM lesen aktivieren
	in	r18,eedr		;Daten lesen
	inc	r16			;nchste EEPROM-Adresse
	brne	eere10			;bertrag? nein -> weiter
	inc	r17			;sonst H-Byte erhhen
eere10:	ret
;
; Unterprogramm zum Schreiben eines Byte in das EEPROM, nach dem
; Schreiben wird die EEPROM-Adresse um 1 erhht
; Register: r16 <- EEPROM-Adresse L
;	    r17 <- EEPROM-Adresse H
;	    r18 <- EEPROM-Daten
;
eewrit:	sbic	eecr,eewe		;luft ein Schreibzyklus?
	rjmp	eewrit			;ja -> warten
	cli				;Interrupts sperren
	out	eearl,r16		;EEPROM-Adresse L setzen
	out	eearh,r17		;EEPROM-Adresse H setzen
	out	eedr,r18		;EEPROM-Daten setzen
	sbi	eecr,eemwe		;EEPROM schreiben freigeben
	sbi	eecr,eewe		;EEPROM schreiben aktivieren
	sei				;Interrupts wieder freigeben
	inc	r16			;nchste EEPROM-Adresse
	brne	eewr10			;bertrag? nein -> weiter
	inc	r17			;sonst H-Byte erhhen
eewr10:	ret
;
; Unterprogramm Warteschleife ca. 50ms
; Register: r14,r15,r16
;
wait50:	ldi	r16,206
	mov	r15,r16			;Zhler setzen (M)
	ldi	r16,3			;Zhler setzen (H)
	clr	r14			;Zhler setzen (L)
wait51:	dec	r14			;L-Zhler abgelaufen?
	brne	wait51			;nein -> Schleife
	dec	r15			;M-Zhler abgelaufen?
	brne	wait51			;nein -> Schleife
	dec	r16			;H-Zhler abgelaufen?
	brne	wait51			;nein -> Schleife
	ret
;
; Unterprogramm bei Fehler in den TWI-Routinen
; Register: r16
;
twierr:	lds	r16,errflg		;Fehler-Flags holen
	sbr	r16,0x08		;Speicherfehler-Flag setzen
	sts	errflg,r16		;Flags wieder speichern
	lds	r16,dflags		;Anzeige-Flags holen
	sbr	r16,0x04		;Infozeile neu ausgeben
	sts	dflags,r16		;Anzeige-Flags wieder speichern
	ret
;
; Unterprogramm zum Lschen eines Speicherbereiches im Externen EEPROM
; Register: r15,r16,r17,r18,r19
;	    r20-r22 <- Anfangsadresse
;	    r23-r25 <- Endadresse
;	    Carry   -> 0=ok, 1=Fehler
;
eraepr:	ldi	r16,xeeadr		;Slave-Adresse + Write-Bit
	bst	r22,0			;Adress-Bit 16 holen und als
	bld	r16,1			;EEPROM-Select-Bit einsetzen
	mov	r15,r16			;Slave-Adresse kopieren
	call	twitio			;zum Ext-EEPROM senden (15ms)
	brcs	era040			;ACK empfangen? nein -> Fehler
	mov	r16,r21			;Adresse H-Byte holen
	call	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x28		;ACK empfangen?
	brne	era040			;nein -> Fehler
	mov	r16,r20			;Adresse L-Byte holen
	call	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x28		;ACK empfangen?
	brne	era040			;nein -> Fehler
;
era010:	ldi	r16,0xff		;Leer-Byte holen
	call	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x28		;ACK empfangen?
	brne	era040			;nein -> Fehler
	inc	r20			;Adresse L erhhen
	brne	era020			;bertrag? nein -> weiter
	inc	r21			;sonst Adresse H erhhen
	brne	era020			;bertrag? nein -> weiter
	inc	r22			;sonst Adresse B3 erhhen
era020:	cp	r20,r23			;Endadresse L erreicht?
	brne	era030			;nein -> weiter
	cp	r21,r24			;Endadresse H erreicht?
	brne	era030			;nein -> weiter
	cp	r22,r25			;Endadresse B3 erreicht?
	brne	era030			;nein -> weiter
	rcall	twisto			;sonst TWI Stop Condition
	clc				;Carry=0, Schreiben ok
	ret
;
era030:	mov	r16,r20			;L-Byte holen
	andi	r16,0x7f		;Bits 6-0 filtern, EEPROM-Page-
	brne	era010			;Wechsel? nein -> weiter
	rcall	twisto			;sonst TWI Stop Condition
	rjmp	eraepr			;sonst neuer Schreibzyklus
;
era040:	sec				;Carry=1, Fehler
	ret
;
; Unterprogramm zum Sichern der Konfigurationsdaten im Externen EEPROM
; Register: r12,r13,r15,r16,r17,r18,r20,r21,r22,r24,r25
;	    Carry -> 0=ok, 1=Fehler
;
xwconf:	ldi	r20,low(cfbsta)
	ldi	r21,high(cfbsta)
	ldi	r22,byte3(cfbsta)	;Anfangsadresse Backup-Bereich
	ldi	r16,low(eeprom)
	ldi	r17,high(eeprom)	;Beginn Konfig-Daten im EEPROM
	ldi	r24,low(eprcrc-eeprom+1)
	ldi	r25,high(eprcrc-eeprom+1);Anzahl Konfig-Daten im EEPROM
	movw	r12,r16			;EEPROM-Adresse sichern
;
xwc010:	ldi	r16,xeeadr		;Slave-Adresse + Write-Bit
	bst	r22,0			;Adress-Bit 16 holen und als
	bld	r16,1			;EEPROM-Select-Bit einsetzen
	mov	r15,r16			;kopieren
	rcall	twitio			;zum Ext-EEPROM senden (15ms)
	brcs	xwc060			;ACK empfangen? nein -> Fehler
	mov	r16,r21			;Adresse H-Byte holen
	rcall	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x28		;ACK empfangen?
	brne	xwc060			;nein -> Fehler
	mov	r16,r20			;Adresse L-Byte holen
	rcall	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x28		;ACK empfangen?
	brne	xwc060			;nein -> Fehler
;
xwc020:	movw	r16,r12			;EEPROM-Adresse restaurieren
	rcall	eeread			;Byte aus EEPROM lesen
	movw	r12,r16			;EEPROM-Adresse wieder sichern
	mov	r16,r18			;gelesenes Byte kopieren
	rcall	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x28		;ACK empfangen?
	brne	xwc060			;nein -> Fehler
	inc	r20			;Adresse L erhhen
	brne	xwc030			;bertrag? nein -> weiter
	inc	r21			;sonst Adresse H erhhen
	brne	xwc030			;bertrag? nein -> weiter
	inc	r22			;sonst Adresse B3 erhhen
xwc030:	mov	r16,r20			;L-Byte holen
	andi	r16,0x7f		;Bits 6-0 filtern, EEPROM-Page-
	brne	xwc040			;Wechsel? nein -> weiter
	rcall	twisto			;sonst TWI Stop Condition
	sbiw	r24,1			;alle Bytes kopiert?
	breq	xwc050			;ja -> Ende
	rjmp	xwc010			;sonst neuer Schreibzyklus
;
xwc040:	sbiw	r24,1			;alle Bytes kopiert?
	brne	xwc020			;nein -> Schleife
	rcall	twisto			;sonst TWI Stop Condition
xwc050:	clc				;Carry=0, Schreiben ok
	ret
xwc060:	sec				;Carry=1, Fehler
	ret
;
; Unterprogramm zum Laden der Konfigurationsdaten aus dem Externen
; EEPROM
; Register: r12,r13,r15,r16,r17,r18,r20,r21,r22,r24,r25
;	    Carry -> 0=ok, 1=Fehler
;
xrconf:	ldi	r20,low(cfbsta)
	ldi	r21,high(cfbsta)
	ldi	r22,byte3(cfbsta)	;Anfangsadresse Backup-Bereich
	ldi	r16,low(eeprom)
	ldi	r17,high(eeprom)	;Beginn Konfig-Daten im EEPROM
	ldi	r24,low(eprcrc-eeprom+1)
	ldi	r25,high(eprcrc-eeprom+1);Anzahl Konfig-Daten im EEPROM
	movw	r12,r16			;EEPROM-Adresse sichern
;
xrc010:	ldi	r16,xeeadr		;Slave-Adresse + Write-Bit
	bst	r22,0			;Adress-Bit 16 holen und als
	bld	r16,1			;EEPROM-Select-Bit einsetzen
	mov	r15,r16			;kopieren
	rcall	twitio			;zum Ext-EEPROM senden (15ms)
	brcs	xrc070			;ACK empfangen? nein -> Fehler
	mov	r16,r21			;Adresse H-Byte holen
	rcall	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x28		;ACK empfangen?
	brne	xrc070			;nein -> Fehler
	mov	r16,r20			;Adresse L-Byte holen
	rcall	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x28		;ACK empfangen?
	brne	xrc070			;nein -> Fehler
;
	rcall	twista			;TWI Start Condition
	cpi	r17,0x10		;Repeat Start ok?
	brne	xrc070			;nein -> Fehler
	ldi	r16,xeeadr+1		;Slave-Adresse + Read-Bit		
	bst	r22,0			;Adress-Bit 16 holen und als
	bld	r16,1			;EEPROM-Select-Bit einsetzen
	rcall	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x40		;ACK empfangen?
	brne	xrc070			;nein -> Fehler
;
xrc020:	set				;ACK zum Slave senden
	cpi	r20,0xff		;wird bertrag L erwartet?
	brne	xrc030			;nein -> weiter
	cpi	r21,0xff		;wird bertrag H erwartet?
	brne	xrc030			;nein -> weiter
	clt				;sonst kein ACK zum Slave send.
xrc030:	cpi	r25,0			;Zhler H-Byte=0?
	brne	xrc040			;nein -> weiter
	cpi	r24,1			;letztes Byte?
	brne	xrc040			;nein -> weiter
	clt				;sonst kein ACK zum Slave send.
xrc040:	rcall	twircv			;Byte vom Ext-EEPROM lesen
	andi	r17,0xf0		;ACK-Status ausblenden
	cpi	r17,0x50		;Empfang ok?
	brne	xrc070			;nein -> Fehler
	mov	r18,r16			;empfangenes Byte kopieren
	movw	r16,r12			;EEPROM-Adresse restaurieren
	rcall	eewrit			;Byte in EEPROM schreiben
	movw	r12,r16			;EEPROM-Adresse wieder sichern
;
	inc	r20			;Adresse L erhhen
	brne	xrc050			;bertrag? nein -> weiter
	inc	r21			;Adresse H erhhen
	brne	xrc050			;bertrag? nein -> weiter
	inc	r22			;Adresse B3 erhhen
	sbiw	r24,1			;alle Bytes gelesen?
	breq	xrc060			;ja -> weiter
	rcall	twisto			;sonst TWI Stop Condition
	rjmp	xrc010			;Schleife
;
xrc050:	sbiw	r24,1			;alle Bytes gelesen?
	brne	xrc020			;nein -> Schleife
xrc060:	rcall	twisto			;TWI Stop Condition
	clc				;Carry=0, Lesen ok
	ret
xrc070:	sec				;Carry=1, Fehler
	ret
;
; Unterprogramm zum Schreiben des Pufferinhaltes in das Externe EEPROM
; Register: r16,r17,r18,r19,yl,yh
;	    r20-r22 <- aktuelle Adresse im externen EEPROM
;	    r23	   <- Anzahl der zu sendenden Bytes (maximal 76)
;	    Carry   -> 0=ok, 1=Fehler
;
xwrite:	ldi	yl,low(xewbuf)		;Zeiger auf Ext-EEPROM Schreib-
	ldi	yh,high(xewbuf)		;Puffer setzen
xwr010:	ldi	r16,xeeadr		;Slave-Adresse + Write-Bit
	bst	r22,0			;Adress-Bit 16 holen und als
	bld	r16,1			;EEPROM-Select-Bit einsetzen
	mov	r15,r16			;kopieren
	rcall	twitio			;zum Ext-EEPROM senden (15ms)
	brcs	xwr060			;ACK empfangen? nein -> Fehler
	mov	r16,r21			;Adresse H-Byte holen
	rcall	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x28		;ACK empfangen?
	brne	xwr060			;nein -> Fehler
	mov	r16,r20			;Adresse L-Byte holen
	rcall	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x28		;ACK empfangen?
	brne	xwr060			;nein -> Fehler
;
xwr020:	ld	r16,y+			;Byte aus EEPROM-Puffer holen
	rcall	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x28		;ACK empfangen?
	brne	xwr060			;nein -> Fehler
	inc	r20			;Adresse L erhhen
	brne	xwr030			;bertrag? nein -> weiter
	inc	r21			;sonst Adresse H erhhen
	brne	xwr030			;bertrag? nein -> weiter
	inc	r22			;sonst Adresse B3 erhhen
xwr030:	mov	r16,r20			;L-Byte holen
	andi	r16,0x7f		;Bits 6-0 filtern, EEPROM-Page-
	brne	xwr040			;Wechsel? nein -> weiter
	rcall	twisto			;sonst TWI Stop Condition
	dec	r23			;alle Bytes ausgegeben?
	breq	xwr050			;ja -> Ende
	rjmp	xwr010			;sonst neuer Schreibzyklus
;
xwr040:	dec	r23			;alle Bytes ausgegeben?
	brne	xwr020			;nein -> Schleife
	rcall	twisto			;sonst TWI Stop Condition
xwr050:	clc				;Carry=0, Schreiben ok
	ret
xwr060:	sec				;Carry=1, Fehler
	ret
;
; Unterprogramm zum Lesen des Pufferinhaltes aus dem Externen EEPROM
; in den Lesepuffer fr die PC-Datenausgabe
; Register: r16,r17,r18,r19,yl,yh
;	    r20-r22 <- aktuelle Adresse im externen EEPROM
;	    r23	   <- Anzahl der zu lesenden Bytes (maximal 76)
;	    Carry   -> 0=ok, 1=Fehler
;
xread:	ldi	yl,low(xerbuf)		;Zeiger auf Ext-EEPROM Lese-
	ldi	yh,high(xerbuf)		;Puffer setzen
xre010:	ldi	r16,xeeadr		;Slave-Adresse + Write-Bit
	bst	r22,0			;Adress-Bit 16 holen und als
	bld	r16,1			;EEPROM-Select-Bit einsetzen
	mov	r15,r16			;kopieren
	rcall	twitio			;zum Ext-EEPROM senden (15ms)
	brcs	xre070			;ACK empfangen? nein -> Fehler
	mov	r16,r21			;Adresse H-Byte holen
	rcall	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x28		;ACK empfangen?
	brne	xre070			;nein -> Fehler
	mov	r16,r20			;Adresse L-Byte holen
	rcall	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x28		;ACK empfangen?
	brne	xre070			;nein -> Fehler
;
	rcall	twista			;TWI Start Condition
	cpi	r17,0x10		;Repeat Start ok?
	brne	xre070			;nein -> Fehler
	ldi	r16,xeeadr+1		;Slave-Adresse + Read-Bit		
	bst	r22,0			;Adress-Bit 16 holen und als
	bld	r16,1			;EEPROM-Select-Bit einsetzen
	rcall	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x40		;ACK empfangen?
	brne	xre070			;nein -> Fehler
;
xre020:	set				;ACK zum Slave senden
	cpi	r20,0xff		;wird bertrag L erwartet?
	brne	xre030			;nein -> weiter
	cpi	r21,0xff		;wird bertrag H erwartet?
	brne	xre030			;nein -> weiter
	clt				;sonst kein ACK zum Slave send.
xre030:	cpi	r23,1			;letztes Byte?
	brne	xre040			;nein -> weiter
	clt				;sonst kein ACK zum Slave send.
xre040:	rcall	twircv			;Byte vom Ext-EEPROM lesen
	andi	r17,0xf0		;ACK-Status ausblenden
	cpi	r17,0x50		;Empfang ok?
	brne	xre070			;nein -> Fehler
	st	y+,r16			;Byte im Puffer speichern
	inc	r20			;Adresse L erhhen
	brne	xre050			;bertrag? nein -> weiter
	inc	r21			;Adresse H erhhen
	brne	xre050			;bertrag? nein -> weiter
	inc	r22			;Adresse B3 erhhen
	dec	r23			;alle Bytes gelesen?
	breq	xre060			;ja -> weiter
	rcall	twisto			;sonst TWI Stop Condition
	rjmp	xre010			;Schleife
;
xre050:	dec	r23			;alle Bytes gelesen?
	brne	xre020			;nein -> Schleife
xre060:	rcall	twisto			;TWI Stop Condition
	clc				;Carry=0, Lesen ok
	ret
xre070:	sec				;Carry=1, Fehler
	ret
;
; Unterprogramm zum Lesen des Pufferinhaltes aus dem Externen EEPROM
; in den Lesepuffer fr die Anzeige auf dem LCD
; Register: r16,r17,r18,r19,yl,yh
;	    r20-r22 <- aktuelle Adresse im externen EEPROM
;	    r23	   <- Anzahl der zu lesenden Bytes (maximal 76)
;
xdread:	ldi	yl,low(xedbuf)		;Zeiger auf Ext-EEPROM Lese-
	ldi	yh,high(xedbuf)		;Puffer fr LCD setzen
	rjmp	xre010
;
; Unterprogramm zum Durchsuchen des externen EEPROMs nach der aktuellen
; Schreibposition fr den nchsten Eintrag
; Register: r15,r16,r17,r18,r23,r24,r25,yl,yh
;	    r4-r7      Zeit-Zwischenspeicher des ltesten Datensatzes
;	    r8-r11     Rechenregister fr Zeiten-Vergleich
;	    r12-r14 -> ermittelte Position des ltesten Eintrages
;	    r19     <- Datensatz-Offset
;	    r20-r22 <- aktuelle Adresse im EEPROM-Bereich
;	    Carry   -> 0=ok, 1=Fehler
;
xswpos:	clr	r4			;Zeit-Zwischenspeicher fr
	clr	r5			;niedrigsten gelesenen Wert
	movw	r6,r4			;initialisieren
	movw	r12,r20
	mov	r14,r22			;Adress-Zeiger initialisieren
xsw010:	ldi	yl,low(xerbuf)		;Zeiger auf Ext-EEPROM Lese-
	ldi	yh,high(xerbuf)		;Puffer setzen
	ldi	r18,4			;4 Bytes lesen
	mov	r23,r20
	mov	r24,r21			;aktuelle Adresse zwischen-
	mov	r25,r22			;speichern
xsw020:	rcall	xesadr			;Adresse an Ext-EEPROM senden
	brcc	xsw040			;ok? ja -> weiter
xsw030:	rcall	twisto			;TWI bertragung beenden
	sec				;Carry=1 Fehler
	ret
;
xsw040:	set				;ACK zum Slave senden
	cpi	r20,0xff		;wird bertrag L erwartet?
	brne	xsw050			;nein -> weiter
	cpi	r21,0xff		;wird bertrag H erwartet?
	brne	xsw050			;nein -> weiter
	clt				;sonst kein ACK zum Slave send.
xsw050:	cpi	r18,1			;letztes Byte?
	brne	xsw060			;nein -> weiter
	clt				;sonst kein ACK zum Slave send.
xsw060:	rcall	twircv			;Byte vom Ext-EEPROM lesen
	andi	r17,0xf0		;ACK-Status ausblenden
	cpi	r17,0x50		;Empfang ok?
	brne	xsw030			;nein -> Ende
	st	y+,r16			;Byte im Puffer speichern
	inc	r20			;Adresse L erhhen
	brne	xsw070			;bertrag? nein -> weiter
	inc	r21			;Adresse H erhhen
	brne	xsw070			;bertrag? nein -> weiter
	inc	r22			;Adresse B3 erhhen
	dec	r18			;alle Bytes gelesen?
	breq	xsw080			;ja -> weiter
	rcall	twisto			;sonst TWI Stop Condition
	rjmp	xsw020			;Schleife
;
xsw070:	dec	r18			;alle Bytes gelesen?
	brne	xsw040			;nein -> Schleife
xsw080:	rcall	twisto			;TWI Stop Condition
	mov	r20,r23
	mov	r21,r24			;zwischengespeicherte Adresse
	mov	r22,r25			;holen
	ld	r11,-y
	ld	r10,-y
	ld	r9,-y			;gelesene Zeitdaten holen
	ld	r8,-y			;(insgesamt 4 Bytes)
	mov	r16,r8
	and	r16,r9
	and	r16,r10
	and	r16,r11			;alles Bytes= 0xff? (noch nicht
	cpi	r16,0xff		;genutzter Speicher)
	brne	xsw110			;nein -> Zeitwert vergleichen
xsw090:	movw	r12,r20			;sonst Speicheradresse verwen-
	mov	r14,r22			;den (insgesamt 3 Bytes)
xsw100:	clc				;Carry=0 normales Ende
	ret
;
xsw110:	sub	r8,r4
	sbc	r9,r5			;ist der gelesene Zeitwert
	sbc	r10,r6			;niedriger (lter) als der
	sbc	r11,r7			;gespeicherte Zeitwert?
	brcs	xsw090			;ja -> Position gefunden
	ld	r4,y+
	ld	r5,y+
	ld	r6,y+			;sonst gelesene Zeitdaten als
	ld	r7,y+			;neuen Wert speichern
;
	clr	r16			;H-Byte=0 fr Berechnung
	add	r20,r19			;Offset zur Adresse addieren
	adc	r21,r16			;bertrag addieren
	adc	r22,r16			;bertrag addieren
	lds	r16,xetemp
	lds	r17,xetemp+1		;Endadresse holen
	lds	r18,xetemp+2		;(insgesamt 3 Bytes)
	sub	r16,r20
	sbc	r17,r21			;berechnete Adresse noch im
	sbc	r18,r22			;gltigen Speicherbereich?
	brcs	xsw100			;nein -> Ende
	rjmp	xsw010			;sonst nchsten Datensatz lesen
;
; Unterprogramm zur Ausgabe der Adresse an das Externe EEPROM zum Lesen
; Register: r16,r17
;	    r20-r22 <- aktuelle Adresse im EEPROM-Bereich
;	    Carry   -> 0=ok, 1=Fehler
;
xesadr:	rcall	twista			;TWI Start Condition
	cpi	r17,0x08		;Start ok?
	brne	xesa10			;nein -> Fehler
	ldi	r16,xeeadr		;Slave-Adresse + Write-Bit
	bst	r22,0			;Adress-Bit 16 holen und als
	bld	r16,1			;EEPROM-Select-Bit einsetzen
	rcall	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x18		;ACK empfangen?
	brne	xesa10			;nein -> Fehler
	mov	r16,r21			;Adresse H-Byte holen
	rcall	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x28		;ACK empfangen?
	brne	xesa10			;nein -> Fehler
	mov	r16,r20			;Adresse L-Byte holen
	rcall	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x28		;ACK empfangen?
	brne	xesa10			;nein -> Fehler
;
	rcall	twista			;TWI Start Condition
	cpi	r17,0x10		;Repeat Start ok?
	brne	xesa10			;nein -> Fehler
	ldi	r16,xeeadr+1		;Slave-Adresse + Read-Bit		
	bst	r22,0			;Adress-Bit 16 holen und als
	bld	r16,1			;EEPROM-Select-Bit einsetzen
	rcall	twisnd			;zum Ext-EEPROM senden
	cpi	r17,0x40		;ACK empfangen?
	brne	xesa10			;nein -> Fehler
	clc				;Carry=0 alles ok
	ret
;
xesa10:	sec				;Carry=1 Fehler
	ret
;
; Unterprogramm TWI (I2C) Start, Slave-Adresse senden, Timeout 15ms
; Register: r16,r17,r18,r19
;	    r15 <- Slave-Adresse
;	    Carry -> 0= ok, 1= Fehler
;
twitio:	ldi	r19,high(600)
	ldi	r18,low(600)		;Zeitkonstante ca. 15ms
twit10:	rcall	twista			;TWI Start Condition
	mov	r16,r15			;Slave-Adresse kopieren
	rcall	twisnd			;senden
	cpi	r17,0x18		;ACK empfangen?
	brne	twit20			;nein -> weiter
	clc				;sonst Carry lschen (ok)
	ret
twit20:	rcall	twisto			;TWI Stop Condition
	dec	r18			;Schleifenzhler L vermindern
	brne	twit10			;Ende? nein -> wiederholen
	dec	r19			;Schleifenzhler H vermindern
	brne	twit10			;Ende? nein -> wiederholen
	sec				;sonst Carry setzen (Fehler)
	ret
;
; Unterprogramm TWI (I2C) Start Condition
; Register: r16
;
twista:	ldi	r16,(1<<twint)|(1<<twsta)|(1<<twen)
	sts	twcr,r16		;Start Condition senden
	rjmp	twis10			;warten bis ausgefhrt
;
; Unterprogramm TWI (I2C) Senden
; Register: r16 <- zu sendendes Byte
;	    r17 -> Status
;
twisnd:	sts	twdr,r16		;Byte in Ausgaberegister
	ldi	r16,(1<<twint)|(1<<twen)
	sts	twcr,r16		;Byte senden
twis10:	lds	r16,twcr		;Status holen
	sbrs	r16,twint		;Kommando ausgefhrt?
	rjmp	twis10			;nein -> warten
	lds	r17,twsr		;sonst Statusregister lesen
	ret
;
; Unterprogramm TWI (I2C) Empfangen
; Register: T   <- Acknowledge (1), Not Acknowledge (0)
;	    r16 -> empfangenes Byte
;	    r17 -> Status
;
twircv:	ldi	r16,(1<<twint)|(1<<twen)
	bld	r16,twea
	sts	twcr,r16		;Byte senden
twir10:	lds	r16,twcr		;Status holen
	sbrs	r16,twint		;Kommando ausgefhrt?
	rjmp	twir10			;nein -> warten
	lds	r16,twdr		;sonst empfangene Daten holen
	lds	r17,twsr		;Statusregister lesen
	ret
;
; Unterprogramm TWI (I2C) Stop Condition
; Register: r16
;
twisto:	ldi	r16,(1<<twint)|(1<<twsto)|(1<<twen)
	sts	twcr,r16		;Stop Condition senden
	ret
;
; Unterprogramm Zeitdaten vom SRAM in die RTC bertragen
; Register: r0,r1,r16,r17,r18
;	    Carry -> 0= ok, 1= Fehler
;
tiwrtc:	cli				;Interrupts sperren
	lds	r16,t1cnt2
	sts	ct1cnt,r16		;Timer1 Int-Zhler 2 kopieren
	lds	r16,secndl
	sts	csecdl,r16		;Sekunde Einer kopieren
	lds	r16,secndh
	sts	csecdh,r16		;Sekunde Zehner kopieren
	lds	r16,minutl
	sts	cminul,r16		;Minute Einer kopieren
	lds	r16,minuth
	sts	cminuh,r16		;Minute Zehner kopieren
	lds	r16,hourl
	sts	chourl,r16		;Stunde Einer kopieren
	lds	r16,hourh
	sts	chourh,r16		;Stunde Zehner kopieren
	lds	r16,dayw
	sts	cdayw,r16		;Wochentag kopieren
	lds	r16,dayl
	sts	cdayl,r16		;Kalendertag Einer kopieren
	lds	r16,dayh
	sts	cdayh,r16		;Kalendertag Zehner kopieren
	lds	r16,monthl
	sts	cmontl,r16		;Monat Einer kopieren
	lds	r16,monthh
	sts	cmonth,r16		;Monat Zehner kopieren
	lds	r16,yearl
	sts	cyearl,r16		;Jahr Einer kopieren
	lds	r16,yearh
	sts	cyearh,r16		;Jahr Zehner kopieren
	sei				;Interrupts wieder freigeben
;
	ldi	r16,rtcadr		;I2C Basisadresse RTC + Write
	rcall	i2csta			;I2C Start Condition + Adresse
	brcs	tiw100			;ACK empfangen? nein -> Fehler
	ldi	r16,rtflag		;RTC-RAM, Flag-Adresse
	rcall	i2csen			;senden
	brcs	tiw100			;ACK empfangen? nein -> Fehler
	clr	r16			;Flag=0
	rcall	i2csen			;senden
	brcs	tiw100			;ACK empfangen? nein -> Fehler
;
	ldi	r16,rtcadr		;I2C Basisadresse RTC + Write
	rcall	i2crst			;I2C Repeat Start Condition
	brcs	tiw100			;ACK empfangen? nein -> Fehler
	clr	r16			;RTC-RAM, Adresse=0
	rcall	i2csen			;senden
	brcs	tiw100			;ACK empfangen? nein -> Fehler
	ldi	r16,0x80		;RTC-Adr.=0, Zhler stoppen
	rcall	i2csen			;senden
	brcs	tiw100			;ACK empfangen? nein -> Fehler
;
	lds	r18,ct1cnt		;Vorteiler 2 holen (1/50s)
	clr	r16			;Anfangswert=0, Umrechnen in
	ldi	r17,16			;Zehntel-Sekunden Increment
tiw010:	subi	r18,5			;5 Zehntel subtrahieren
	brcs	tiw020			;berlauf? ja -> weiter
	add	r16,r17			;sonst Zehntel-Ergebnis erhhen
	rjmp	tiw010			;Schleife
tiw020:	rcall	i2csen			;Zehntel im oberen Nibble send.
	brcs	tiw100			;ACK empfangen? nein -> Fehler
;
	lds	r17,csecdh		;Sekunden Zehner holen
	swap	r17			;in oberes Nibble legen
	lds	r16,csecdl		;Sekunden Einer holen
	or	r16,r17			;Zehner und Einer verknpfen
	rcall	i2csen			;senden
	brcs	tiw100			;ACK empfangen? nein -> Fehler
;
	lds	r17,cminuh		;Minuten Zehner holen
	swap	r17			;in oberes Nibble legen
	lds	r16,cminul		;Minuten Einer holen
	or	r16,r17			;Zehner und Einer verknpfen
	rcall	i2csen			;senden
	brcs	tiw100			;ACK empfangen? nein -> Fehler
;
	lds	r17,chourh		;Stunden Zehner holen
	swap	r17			;in oberes Nibble legen
	lds	r16,chourl		;Stunden Einer holen
	or	r16,r17			;Zehner und Einer verknpfen
	rcall	i2csen			;senden
	brcc	tiw110			;ACK empfangen? ja -> weiter
tiw100:	rjmp	tiwerr			;sonst Fehler
;
tiw110:	lds	r16,cyearh		;Jahr Zehner holen
	ldi	r17,10			;mit 10 multiplizieren
	mul	r16,r17			;Umrechnung in Binrwert
	lds	r16,cyearl		;Jahr Einer holen
	add	r16,r0			;Zehner und Einer addieren
	ror	r16
	ror	r16			;Bits 1,0 an Position 7,6
	ror	r16			;schieben
	andi	r16,0xc0		;und filtern
	lds	r17,cdayh		;Tag Zehner holen
	swap	r17			;in oberes Nibble legen
	or	r17,r16			;Jahr und Tag Zehner verknpfen
	lds	r16,cdayl		;Tag Einer holen
	or	r16,r17			;Zehner und Einer verknpfen
	rcall	i2csen			;senden
	brcs	tiwerr			;ACK empfangen? nein -> Fehler
;
	lds	r16,cdayw		;Wochentag holen
	lsl	r16			;in Bits 3-1 schieben
	lds	r17,cmonth		;Monat Zehner holen
	or	r17,r16			;Monat und Wochentag verknpfen
	swap	r17			;in oberes Nibble legen
	lds	r16,cmontl		;Monat Einer holen
	or	r16,r17			;Zehner und Einer verknpfen
	rcall	i2csen			;senden
	brcs	tiwerr			;ACK empfangen? nein -> Fehler
;
	ldi	r16,rtcadr		;I2C Basisadresse RTC + Write
	rcall	i2crst			;I2C Repeat Start Condition
	brcs	tiwerr			;ACK empfangen? nein -> Fehler
	clr	r16			;RTC-RAM, Adresse=0
	rcall	i2csen			;senden
	brcs	tiwerr			;ACK empfangen? nein -> Fehler
	clr	r16			;RTC-Adr.=0, Zhler starten
	rcall	i2csen			;senden
	brcs	tiwerr			;ACK empfangen? nein -> Fehler
;
	ldi	r16,rtcadr		;I2C Basisadresse RTC + Write
	rcall	i2crst			;I2C Repeat Start Condition
	brcs	tiwerr			;ACK empfangen? nein -> Fehler
	ldi	r16,rtyear		;RTC-RAM, Jahr-Adresse
	rcall	i2csen			;senden
	brcs	tiwerr			;ACK empfangen? nein -> Fehler
	lds	r17,cyearh		;Jahr Zehner holen
	swap	r17			;in oberes Nibble legen
	lds	r16,cyearl		;Jahr Einer holen
	or	r16,r17			;Zehner und Einer verknpfen
	rcall	i2csen			;senden
	brcs	tiwerr			;ACK empfangen? nein -> Fehler
;
	ldi	r16,rtcadr		;I2C Basisadresse RTC + Write
	rcall	i2crst			;I2C Repeat Start Condition
	brcs	tiwerr			;ACK empfangen? nein -> Fehler
	ldi	r16,rtflag		;RTC-RAM, Flag-Adresse
	rcall	i2csen			;senden
	brcs	tiwerr			;ACK empfangen? nein -> Fehler
	ldi	r16,0x5a		;Flag= Uhrzeit gltig
	rcall	i2csen			;senden
	brcs	tiwerr			;ACK empfangen? nein -> Fehler
;
	rcall	i2csto			;I2C Stop Condition
	clc				;Carry lschen (ok)
	ret
;
tiwerr:	rcall	i2csto			;I2C Stop Condition
	sec				;Carry setzen (Fehler)
	ret
;
; Unterprogramm Zeitdaten von der RTC in das SRAM bertragen
; Register: r0,r1,r15,r17,r18,r19
;	    r16 -> Inhalt das Flagbytes (wenn Carry=0)
;	    Carry -> 0= ok, 1= Fehler
;
tirrtc:	ldi	r16,rtcadr		;I2C Basisadresse RTC + Write
	rcall	i2csta			;I2C Start Condition + Adresse
	brcs	tir100			;ACK empfangen? nein -> Fehler
	ldi	r16,1			;RTC-RAM, Adresse 1
	rcall	i2csen			;senden
	brcs	tir100			;ACK empfangen? nein -> Fehler
	ldi	r16,rtcadr+1		;I2C Basisadresse RTC + Read
	rcall	i2crst			;I2C Repeat Start Condition
	brcc	tir110			;ACK empfangen? ja -> weiter
tir100:	rjmp	tirerr			;sonst Fehler
;
tir110:	clc				;ACK zum Slave senden
	rcall	i2crcv			;Zehntel/Hundertstel empfangen
	swap	r16			;Zehntel in unteres Nibble
	andi	r16,0x0f		;legen und filtern
	ldi	r17,5			;mit 5
	mul	r16,r17			;multiplizieren
	sts	ct1cnt,r0		;und als 1/50s speichern
;
	clc				;ACK zum Slave senden
	rcall	i2crcv			;Sekunden empfangen
	mov	r17,r16			;Wert kopieren
	swap	r17			;Zehner in unteres Nibble
	andi	r17,0x0f		;legen und filtern
	sts	csecdh,r17		;Sekunden Zehner speichern
	andi	r16,0x0f		;Einer filtern
	sts	csecdl,r16		;Sekunden Einer speichern
;
	clc				;ACK zum Slave senden
	rcall	i2crcv			;Minuten empfangen
	mov	r17,r16			;Wert kopieren
	swap	r17			;Zehner in unteres Nibble
	andi	r17,0x0f		;legen und filtern
	sts	cminuh,r17		;Minuten Zehner speichern
	andi	r16,0x0f		;Einer filtern
	sts	cminul,r16		;Minuten Einer speichern
;
	clc				;ACK zum Slave senden
	rcall	i2crcv			;Stunden empfangen
	mov	r17,r16			;Wert kopieren
	swap	r17			;Zehner in unteres Nibble
	andi	r17,0x03		;legen und filtern
	sts	chourh,r17		;Stunden Zehner speichern
	andi	r16,0x0f		;Einer filtern
	sts	chourl,r16		;Stunden Einer speichern
;
	clc				;ACK zum Slave senden
	rcall	i2crcv			;Tage empfangen
	mov	r17,r16			;Wert kopieren
	swap	r17			;Zehner in unteres Nibble
	mov	r19,r17			;Wert kopieren
	lsr	r19
	lsr	r19			;Jahr an Position 1,0 schieben
	andi	r19,0x01		;LSB filtern
	andi	r17,0x03		;legen und filtern
	sts	cdayh,r17		;Tag Zehner speichern
	andi	r16,0x0f		;Einer filtern
	sts	cdayl,r16		;Tag Einer speichern
;
	sec				;kein ACK zum Slave senden
	rcall	i2crcv			;Monate/Wochentag empfangen
	mov	r17,r16			;Wert kopieren
	swap	r17			;Zehner/Tag in unteres Nibble
	mov	r18,r17			;Wert kopieren
	lsr	r18			;Tag an Position 2-0 schieben
	andi	r18,0x07		;Wochentag filtern
	sts	cdayw,r18		;Wochentag speichern
	andi	r17,0x01		;Monat Zehner filtern
	sts	cmonth,r17		;Monat Zehner speichern
	andi	r16,0x0f		;Einer filtern
	sts	cmontl,r16		;Monat Einer speichern
;
	ldi	r16,rtcadr		;I2C Basisadresse RTC + Write
	rcall	i2crst			;I2C Repeat Start Condition
	brcs	tirerr			;ACK empfangen? nein -> Fehler
	ldi	r16,rtyear		;RTC-RAM, Jahr-Adresse
	rcall	i2csen			;senden
	brcs	tirerr			;ACK empfangen? nein -> Fehler
	ldi	r16,rtcadr+1		;I2C Basisadresse RTC + Read
	rcall	i2crst			;I2C Repeat Start Condition
	brcs	tirerr			;ACK empfangen? nein -> Fehler
	sec				;kein ACK zum Slave senden
	rcall	i2crcv			;Jahr empfangen
	mov	r17,r16			;Wert kopieren
	swap	r17			;Zehner in unteres Nibble
	andi	r17,0x0f		;legen und filtern
	sts	cyearh,r17		;Jahr Zehner speichern
	andi	r16,0x0f		;Einer filtern
	sts	cyearl,r16		;Jahr Einer speichern
	mov	r18,r16			;Einer kopieren
	andi	r18,0x01		;Einer LSB filtern
	eor	r18,r19			;LSB identisch?
	breq	tir300			;ja -> Jahr ok, weiter
;
	inc	r16			;sonst Jahr Einer erhhen
	sts	cyearl,r16		;Jahr Einer speichern
	cpi	r16,10			;bertrag?
	brcs	tir200			;nein -> weiter
	clr	r16			;sonst Jahr Einer zurcksetzen
	sts	cyearl,r16		;Jahr Einer speichern
	inc	r17			;Jahr Zehner erhhen
	sts	cyearh,r17		;Jahr Zehner speichern
	cpi	r17,10			;bertrag?
	brcs	tir200			;nein -> weiter
	clr	r17			;sonst Jahr Zehner zurcksetzen
	sts	cyearh,r17		;Jahr Zehner speichern
tir200:	swap	r17			;in oberes Nibble legen
	or	r16,r17			;Zehner und Einer verknpfen
	mov	r18,r16			;Jahr kopieren
;
	ldi	r16,rtcadr		;I2C Basisadresse RTC + Write
	rcall	i2crst			;I2C Repeat Start Condition
	brcs	tirerr			;ACK empfangen? nein -> Fehler
	ldi	r16,rtyear		;RTC-RAM, Jahr-Adresse
	rcall	i2csen			;senden
	brcs	tirerr			;ACK empfangen? nein -> Fehler
	mov	r16,r18			;Jahr holen
	rcall	i2csen			;senden
	brcs	tirerr			;ACK empfangen? nein -> Fehler
;
tir300:	ldi	r16,rtcadr		;I2C Basisadresse RTC + Write
	rcall	i2crst			;I2C Repeat Start Condition
	brcs	tirerr			;ACK empfangen? nein -> Fehler
	ldi	r16,rtflag		;RTC-RAM, Flag-Adresse
	rcall	i2csen			;senden
	brcs	tirerr			;ACK empfangen? nein -> Fehler
	ldi	r16,rtcadr+1		;I2C Basisadresse RTC + Read
	rcall	i2crst			;I2C Repeat Start Condition
	brcc	tirend			;ACK empfangen? ja -> weiter
;
tirerr:	rcall	i2csto			;sonst I2C Stop Condition
	sec				;Carry setzen (Fehler)
	ret
;
tirend:	sec				;kein ACK zum Slave senden
	rcall	i2crcv			;Flag-Byte empfangen
	rcall	i2csto			;I2C Stop Condition
	cli				;Interrupts sperren
	lds	r17,ct1cnt
	sts	t1cnt2,r17		;Timer1 Int-Zhler 2 kopieren
	lds	r17,csecdl
	sts	secndl,r17		;Sekunde Einer kopieren
	lds	r17,csecdh
	sts	secndh,r17		;Sekunde Zehner kopieren
	lds	r17,cminul
	sts	minutl,r17		;Minute Einer kopieren
	lds	r17,cminuh
	sts	minuth,r17		;Minute Zehner kopieren
	lds	r17,chourl
	sts	hourl,r17		;Stunde Einer kopieren
	lds	r17,chourh
	sts	hourh,r17		;Stunde Zehner kopieren
	lds	r17,cdayw
	sts	dayw,r17		;Wochentag kopieren
	lds	r17,cdayl
	sts	dayl,r17		;Kalendertag Einer kopieren
	lds	r17,cdayh
	sts	dayh,r17		;Kalendertag Zehner kopieren
	lds	r17,cmontl
	sts	monthl,r17		;Monat Einer kopieren
	lds	r17,cmonth
	sts	monthh,r17		;Monat Zehner kopieren
	lds	r17,cyearl
	sts	yearl,r17		;Jahr Einer kopieren
	lds	r17,cyearh
	sts	yearh,r17		;Jahr Zehner kopieren
	sei				;Interrupts wieder freigeben
	clc				;Carry lschen (ok)
	ret
;
; Unterprogramm zum Schreiben eines Backups (Min/Max/Mittelwerte) in
; das RTC-RAM
; Register: r16,r17,r18,x
;	    Carry -> 0= ok, 1= Fehler
;
i2back:	ldi	r16,rtcadr		;I2C Basisadresse RTC + Write
	rcall	i2csta			;I2C Start Condition + Adresse
	brcc	i2b010			;ACK empfangen? ja -> weiter
	rjmp	i2berr			;sonst Fehler
i2b010:	ldi	r16,rtbflg		;RTC-RAM, Adresse Backup-Flag
	rcall	i2csen			;senden
	brcc	i2b020			;ACK empfangen? ja -> weiter
	rjmp	i2berr			;sont Fehler
i2b020:	clr	r16			;Flag=0 setzen, Backup ungltig
	rcall	i2csen			;senden
	brcc	i2b030			;ACK empfangen? ja -> weiter
	rjmp	i2berr			;sonst Fehler
;
i2b030:	ldi	r16,rtcadr		;I2C Basisadresse RTC + Write
	rcall	i2crst			;I2C Repeat Start Con + Adresse
	brcc	i2b040			;ACK empfangen? ja -> weiter
	rjmp	i2berr			;sonst Fehler
i2b040:	ldi	r16,rtbval		;Adresse der Temperaturwerte
	rcall	i2csen			;senden
	brcc	i2b050			;ACK empfangen? ja -> weiter
	rjmp	i2berr			;sonst Fehler
;
i2b050:	ldi	r18,32			;32 Datenbytes senden (temin1)
	ldi	xl,low(temin1)
	ldi	xh,high(temin1)		;Zeiger auf Minimal-Werte (Anz)
i2b060:	ld	r16,x+			;Byte aus RAM holen
	rcall	i2csen			;senden
	brcs	i2berr			;ACK empfangen? nein -> Fehler
	dec	r18			;alle Bytes gesendet?
	brne	i2b060			;nein -> Schleife
;
	ldi	r18,32			;32 Datenbytes senden (temax1)
	ldi	xl,low(temax1)
	ldi	xh,high(temax1)		;Zeiger auf Maximal-Werte (Anz)
i2b070:	ld	r16,x+			;Byte aus RAM holen
	rcall	i2csen			;senden
	brcs	i2berr			;ACK empfangen? nein -> Fehler
	dec	r18			;alle Bytes gesendet?
	brne	i2b070			;nein -> Schleife
;
	ldi	r18,32			;32 Datenbytes senden (temin2)
	ldi	xl,low(temin2)
	ldi	xh,high(temin2)		;Zeiger auf Minimal-Werte (Log)
i2b080:	ld	r16,x+			;Byte aus RAM holen
	rcall	i2csen			;senden
	brcs	i2berr			;ACK empfangen? nein -> Fehler
	dec	r18			;alle Bytes gesendet?
	brne	i2b080			;nein -> Schleife
;
	ldi	r18,32			;32 Datenbytes senden (temax2)
	ldi	xl,low(temax2)
	ldi	xh,high(temax2)		;Zeiger auf Maximal-Werte (Log)
i2b090:	ld	r16,x+			;Byte aus RAM holen
	rcall	i2csen			;senden
	brcs	i2berr			;ACK empfangen? nein -> Fehler
	dec	r18			;alle Bytes gesendet?
	brne	i2b090			;nein -> Schleife
;
	ldi	r18,48			;48 Datenbytes senden (temavs)
	ldi	xl,low(temavs)
	ldi	xh,high(temavs)		;Zeiger auf Mittelwert-Summen
i2b100:	ld	r16,x+			;Byte aus RAM holen
	rcall	i2csen			;senden
	brcs	i2berr			;ACK empfangen? nein -> Fehler
	dec	r18			;alle Bytes gesendet?
	brne	i2b100			;nein -> Schleife
;
	ldi	r18,32			;32 Datenbytes senden (temavc)
	ldi	xl,low(temavc)
	ldi	xh,high(temavc)		;Zeiger auf Mittelwert-Zhler
i2b110:	ld	r16,x+			;Byte aus RAM holen
	rcall	i2csen			;senden
	brcs	i2berr			;ACK empfangen? nein -> Fehler
	dec	r18			;alle Bytes gesendet?
	brne	i2b110			;nein -> Schleife
;
	lds	r16,dayl		;Kalendertag Einer holen
	rcall	i2csen			;senden
	brcs	i2berr			;ACK empfangen? nein -> Fehler
	lds	r16,dayh		;Kalendertag Zehner holen
	rcall	i2csen			;senden
	brcs	i2berr			;ACK empfangen? nein -> Fehler
	lds	r16,monthl		;Monat Einer holen
	rcall	i2csen			;senden
	brcs	i2berr			;ACK empfangen? nein -> Fehler
	lds	r16,monthh		;Monat Zehner holen
	rcall	i2csen			;senden
	brcs	i2berr			;ACK empfangen? nein -> Fehler
	ldi	r16,0x5a		;Flag fr Backup ok
	rcall	i2csen			;senden
;
i2berr:	rcall	i2csto			;I2C Stop Condition
	ret
;
; Unterprogramm zum Zurcklesen eines Backups (Min/Max/Mittelwerte) aus
; dem RTC-RAM
; Register: r16,r17,r18,x
;	    Carry -> 0= ok, 1= Fehler
;
i2rest:	ldi	r16,rtcadr		;I2C Basisadresse RTC + Write
	rcall	i2csta			;I2C Start Condition + Adresse
	brcs	i2reer			;ACK empfangen? nein -> Fehler
	ldi	r16,rtbval		;Adresse der Temperaturwerte
	rcall	i2csen			;senden
	brcs	i2reer			;ACK empfangen? nein -> Fehler
	ldi	r16,rtcadr+1		;I2C Basisadresse RTC + Read
	rcall	i2crst			;I2C Repeat Start Condition
	brcs	i2reer			;ACK empfangen? nein -> Fehler

	ldi	r18,32			;32 Datenbytes lesen (temin1)
	ldi	xl,low(temin1)
	ldi	xh,high(temin1)		;Zeiger auf temin1-Werte
i2re10:	clc				;ACK zum Slave senden
	rcall	i2crcv			;Byte empfangen
	st	x+,r16			;Byte in Puffer speichern
	dec	r18			;alle Bytes empfangen?
	brne	i2re10			;nein -> Schleife
;
	ldi	r18,32			;32 Datenbytes lesen (temax1)
	ldi	xl,low(temax1)
	ldi	xh,high(temax1)		;Zeiger auf temax1-Werte
i2re20:	clc				;ACK zum Slave senden
	rcall	i2crcv			;Byte empfangen
	st	x+,r16			;Byte in Puffer speichern
	dec	r18			;alle Bytes empfangen?
	brne	i2re20			;nein -> Schleife
;
	ldi	r18,32			;32 Datenbytes lesen (temin2)
	ldi	xl,low(temin2)
	ldi	xh,high(temin2)		;Zeiger auf temin2-Werte
i2re30:	clc				;ACK zum Slave senden
	rcall	i2crcv			;Byte empfangen
	st	x+,r16			;Byte in Puffer speichern
	dec	r18			;alle Bytes empfangen?
	brne	i2re30			;nein -> Schleife
;
	ldi	r18,32			;32 Datenbytes lesen (temax2)
	ldi	xl,low(temax2)
	ldi	xh,high(temax2)		;Zeiger auf temax2-Werte
i2re40:	clc				;ACK zum Slave senden
	rcall	i2crcv			;Byte empfangen
	st	x+,r16			;Byte in Puffer speichern
	dec	r18			;alle Bytes empfangen?
	brne	i2re40			;nein -> Schleife
;
	ldi	r18,48			;48 Datenbytes lesen (temavs)
	ldi	xl,low(temavs)
	ldi	xh,high(temavs)		;Zeiger auf temavs-Werte
i2re50:	clc				;ACK zum Slave senden
	rcall	i2crcv			;Byte empfangen
	st	x+,r16			;Byte in Puffer speichern
	dec	r18			;alle Bytes empfangen?
	brne	i2re50			;nein -> Schleife
;
	ldi	r18,32			;32 Datenbytes lesen (temavc)
	ldi	xl,low(temavc)
	ldi	xh,high(temavc)		;Zeiger auf temavc-Werte
i2re60:	clc				;ACK zum Slave senden
	cpi	r18,1			;letztes Byte?
	brne	i2re70			;nein -> weiter
	sec				;sonst kein ACK zum Slave send.
i2re70:	rcall	i2crcv			;Byte empfangen
	st	x+,r16			;Byte in Puffer speichern
	dec	r18			;alle Bytes empfangen?
	brne	i2re60			;nein -> Schleife
;
	clc				;Carry lschen (alles ok)
i2reer:	rcall	i2csto			;I2C Stop Condition
	ret
;
; Unterprogramm zum Senden eines Ext-EEPROM-Pointers zum RTC-RAM
; Register: r16,r17,r19
;	    x   <- Zeiger auf Pointer im SRAM
;	    r18 <- RTC Pointer-Adresse
;	    Carry -> 0= ok, 1= Fehler
;
i2psen:	ldi	r16,rtcadr		;I2C Basisadresse RTC + Write
	rcall	i2csta			;I2C Start Condition + Adresse
	brcs	i2pser			;ACK empfangen? nein -> Fehler
	ldi	r16,3			;Offset fr Flagbyte laden
	add	r16,r18			;RTC-RAM, Pointer-Adresse add.
	rcall	i2csen			;senden
	brcs	i2pser			;ACK empfangen? nein -> Fehler
	clr	r16			;Flag=0 setzen
	rcall	i2csen			;senden
	brcs	i2pser			;ACK empfangen? nein -> Fehler
;
	ldi	r16,rtcadr		;I2C Basisadresse RTC + Write
	rcall	i2crst			;I2C Start Condition + Adresse
	brcs	i2pser			;ACK empfangen? nein -> Fehler
	mov	r16,r18			;RTC-RAM, Pointer-Adresse
	rcall	i2csen			;senden
	brcs	i2pser			;ACK empfangen? nein -> Fehler
	ldi	r19,3			;3 Pointer-Bytes senden
i2ps10:	ld	r16,x+			;Byte aus Puffer holen
	rcall	i2csen			;senden
	brcs	i2pser			;ACK empfangen? nein -> Fehler
	dec	r19			;alle Bytes gesendet?
	brne	i2ps10			;nein -> Schleife
	ldi	r16,0x5a		;Flag-Byte laden
	rcall	i2csen			;senden
i2pser:	rcall	i2csto			;I2C Stop Condition
	ret
;
; Unterprogramm zum Empfangen eines Ext-EEPROM-Pointers vom RTC-RAM
; Register: r17
;	    x   <- Zeiger auf Datenblock
;	    r18 <- Startadresse
;	    r16 -> Flag-Byte
;	    Carry -> 0= ok, 1= Fehler
;
i2prcv:	ldi	r16,rtcadr		;I2C Basisadresse RTC + Write
	rcall	i2csta			;I2C Start Condition + Adresse
	brcs	i2prer			;ACK empfangen? nein -> Fehler
	mov	r16,r18			;RTC-RAM, Block Start-Adresse
	rcall	i2csen			;senden
	brcs	i2prer			;ACK empfangen? nein -> Fehler
	ldi	r16,rtcadr+1		;I2C Basisadresse RTC + Read
	rcall	i2crst			;I2C Repeat Start Condition
	brcs	i2prer			;ACK empfangen? nein -> Fehler
	ldi	r19,3			;3 Pointer-Bytes empfangen
i2pr10:	clc				;ACK zum Slave senden
	rcall	i2crcv			;Byte empfangen
	st	x+,r16			;Byte in Puffer speichern
	dec	r19			;alle Bytes empfangen?
	brne	i2pr10			;nein -> Schleife
	sec				;kein ACK zum Slave senden
	rcall	i2crcv			;Flag-Byte empfangen
	clc				;Carry lschen (alles ok)
i2prer:	rcall	i2csto			;I2C Stop Condition
	ret
;
; I2C Implementierung nach Application Note AVR300, modifiziert und
; eingestellt auf PortB2 (SCL) + PortB3 (SDA), Frequenz ca. 100 KHz
;
; Unterprogramm I2C Start Condition, Slave-Adresse senden, Timeout 15ms
; Register: r16,r17
;	    r15 <- I2C Slave-Adresse
;	    Carry -> 0= ok, 1= Fehler
;
i2ctio:	ldi	r18,120			;Zeitkonstante ca. 15ms
i2ct10:	mov	r16,r15			;Slave-Adresse kopieren
	rcall	i2csta			;I2C Start Condition, senden
	brcs	i2ct20			;ACK empfangen? nein -> weiter
	ret				;sonst Ende (ok)
i2ct20:	rcall	i2csto			;I2C Stop Condition
	dec	r18			;Schleifenzhler vermindern
	brne	i2ct10			;Ende? nein -> wiederholen
	sec				;sonst Carry setzen (Fehler)
	ret
;
; Unterprogramm I2C Repeat Start Condition und Slave-Adresse senden
; Register: r17
;	    r16 <- I2C Slave-Adresse (Bit0=0: Write, Bit0=1: Read)
;	    Carry -> ACK vom Slave
;
i2crst:	sbi	ddrb,2			;SCL low
	cbi	ddrb,3			;SDA freigeben
	rcall	i2hdel			;halbe Periode warten
	cbi	ddrb,2			;SCL freigeben
	rcall	i2hdel			;halbe Periode warten
;
; Unterprogramm I2C Start Condition und Slave-Adresse senden
; Register: r17
;	    r16 <- I2C Slave-Adresse (Bit0=0: Write, Bit0=1: Read)
;	    Carry -> ACK vom Slave
;
i2csta:	sbi	ddrb,3			;SDA low
	rcall	i2hdel			;halbe Periode warten
;
; Unterprogramm I2C ein Byte senden
; Register: r17
;	    r16 <- zu sendendes Byte
;	    Carry -> ACK vom Slave
;
i2csen:	sec				;Carry setzen
	rol	r16			;hchstes Bit in Carry schieben
	rjmp	i2sbfi			;erstes Bit ausgeben
i2sbit:	lsl	r16			;nchstes Bit, alles ausgegeb.?
i2sbfi:	breq	i2sack			;ja -> ACK holen
	sbi	ddrb,2			;SCL low
	brcc	i2slow			;0-Bit? ja -> ausgeben
	nop				;Zyklen ausgleichen
	cbi	ddrb,3			;SDA freigeben
	rjmp	i2shig			;1-Bit ausgeben
i2slow:	sbi	ddrb,3			;SDA low
	rjmp	i2shig			;Zyklen ausgleichen
i2shig:	rcall	i2hdel			;halbe Periode warten
	cbi	ddrb,2			;SCL freigeben
	rcall	i2hdel			;halbe Periode warten
	rjmp	i2sbit			;Schleife
;
i2sack:	sbi	ddrb,2			;ACK lesen, SCL low
	cbi	ddrb,3			;SDA freigeben
	rcall	i2hdel			;halbe Periode warten
	cbi	ddrb,2			;SCL freigeben
i2sawa:	sbis	pinb,2			;auf SCL high warten, falls
	rjmp	i2sawa			;Wait-States eingefgt wurden
	clc				;Carry lschen
	sbic	pinb,3			;SDA high? nein -> weiter
	sec				;sonst Carry setzen
	rcall	i2hdel			;halbe Periode warten
	ret
;
; Unterprogramm I2C ein Byte empfangen
; Register: r15,r17
;	    r16 -> gelesenes Byte
;	    Carry <- ACK zum Slave
;
i2crcv:	rol	r15			;ACK speichern
	ldi	r16,0x01		;Daten=0x01 setzen
i2rbit:	sbi	ddrb,2			;SCL low
	cbi	ddrb,3			;SDA freigeben (vorheriges ACK)
	rcall	i2hdel			;halbe Periode warten
	cbi	ddrb,2			;SCL freigeben
	rcall	i2hdel			;halbe Periode warten
	clc				;Carry lschen
	sbic	pinb,3			;SDA high? nein -> weiter
	sec				;sonst Carry setzen
	rol	r16			;Datenbit speichern
	brcc	i2rbit			;alle Bits empfangen? -> Schl.
;
	sbi	ddrb,2			;SCL low
	ror	r15			;Status Bit holen
	brcc	i2ralo			;Bit=0? ja -> ausgeben
	cbi	ddrb,3			;sonst SDA freigeben
	rjmp	i2rahi			;und High ausgeben
i2ralo:	sbi	ddrb,3			;SDA low
i2rahi:	rcall	i2hdel			;halbe Periode warten
	cbi	ddrb,2			;SCL freigeben
i2rawa:	sbis	pinb,2			;auf SCL high warten
	rjmp	i2rawa			;Schleife
	rcall	i2hdel			;halbe Periode warten
	ret
;
; Unterprogramm I2C Stop Condition
; Register: r17
;
i2csto:	sbi	ddrb,2			;SCL low
	sbi	ddrb,3			;SDA low
	rcall	i2hdel			;halbe Periode warten
	cbi	ddrb,2			;SCL freigeben
	rcall	i2hdel			;halbe Periode warten
	cbi	ddrb,3			;SDA freigeben
	rcall	i2hdel			;halbe Periode warten
	ret
;
; Unterprogramm I2C Warteschleife fr halbe Periode
; Register: r17
;
i2hdel:	ldi	r17,16			;Wartezeit ca. 5,0s
i2hd10:	dec	r17
	brne	i2hd10
	ret
;
; Unterprogramm bei Fehler in den I2C-Routinen, Fehlerstatus setzen und
; Infozeile neu ausgeben
; Register: r16
;
i2cerr:	lds	r16,errflg		;Fehler-Flags holen
	sbr	r16,0x10		;RTC-Fehler-Flag setzen
	sts	errflg,r16		;Flags wieder speichern
	lds	r16,dflags		;Anzeige-Flags holen
	sbr	r16,0x04		;Infozeile neu ausgeben
	sts	dflags,r16		;Anzeige-Flags wieder speichern
	ret
;
; Unterprogramm bei Pointerfehler, Fehlerstatus setzen
; Infozeile neu ausgeben
;
poierr:	lds	r16,errflg		;Fehler-Flags holen
	sbr	r16,0x20		;Pointer-Fehler-Flag setzen
	sts	errflg,r16		;Flags wieder speichern
	lds	r16,dflags		;Anzeige-Flags holen
	sbr	r16,0x04		;Infozeile neu ausgeben
	sts	dflags,r16		;Anzeige-Flags wieder speichern
	ret
;
; Unterprogramm zum Setzen der Baudrate der PC-Schnittstelle (RS232-0)
; Register: r16,r17,r18
;
setbd0:	lds	r18,baud0		;Baudratenwert RS232-0 holen
	cpi	r18,0			;9600 Baud?
	brne	setb10			;nein -> weiter testen
	ldi	r16,low((clock/(16*9600))-1)  ;RS232-0 auf 9600 Baud
	ldi	r17,high((clock/(16*9600))-1) ;einstellen
	rjmp	setb70
setb10:	cpi	r18,1			;19200 Baud?
	brne	setb20			;nein -> weiter testen
	ldi	r16,low((clock/(16*19200))-1)  ;RS232-0 auf 19200 Baud
	ldi	r17,high((clock/(16*19200))-1) ;einstellen
	rjmp	setb70
setb20:	cpi	r18,2			;38400 Baud?
	brne	setb30			;nein -> weiter testen
	ldi	r16,low((clock/(16*38400))-1)  ;RS232-0 auf 38400 Baud
	ldi	r17,high((clock/(16*38400))-1) ;einstellen
	rjmp	setb70
setb30:	cpi	r18,3			;57600 Baud?
	brne	setb40			;nein -> weiter testen
	ldi	r16,low((clock/(16*57600))-1)  ;RS232-0 auf 57600 Baud
	ldi	r17,high((clock/(16*57600))-1) ;einstellen
	rjmp	setb70
setb40:	cpi	r18,4			;115200 Baud?
	brne	setb50			;nein -> weiter testen
	ldi	r16,low((clock/(16*115200))-1)  ;sonst RS232-0 auf
	ldi	r17,high((clock/(16*115200))-1) ;115200 Baud einstellen
	rjmp	setb70
setb50:	cpi	r18,5			;230400 Baud?
	brne	setb60			;nein -> weiter testen
	ldi	r16,low((clock/(16*230400))-1)  ;sonst RS232-0 auf
	ldi	r17,high((clock/(16*230400))-1) ;230400 Baud einstellen
	rjmp	setb70
setb60:	cpi	r18,6			;460800 Baud?
	brne	setb90			;nein -> keine nderung
	ldi	r16,low((clock/(8*460800))-1)  ;sonst RS232-0 auf
	ldi	r17,high((clock/(8*460800))-1) ;460800 Baud einstellen
	ldi	r18,1<<u2x0		;USART Taktverdopplung setzen
	rjmp	setb80
setb70:	ldi	r18,0<<u2x0		;USART Taktverdopplung lschen
setb80:	out	ucsr0a,r18		;Taktverdopplung einstellen
	out	ubrr0l,r16		;Baudratenwert L setzen
	sts	ubrr0h,r17		;Baudratenwert H setzen
setb90:	ret
;
; Unterprogramm zum Schreiben eines Zeichens in den RS232-0 Sendepuffer
; Register: x,y
;	    r16 <- zu sendendes Zeichen
;
put0ch:	cli				;Int vorbergehend sperren
	lds	xl,txp0wr
	lds	xh,txp0wr+1		;Schreibzeiger holen
	st	x+,r16			;Byte speichern, Zeiger+1
	ldi	yl,low(txbuf0+txbu0l)
	ldi	yh,high(txbuf0+txbu0l)	;Adresse Pufferende laden
	sub	yl,xl
	sbc	yh,xh			;Differenz berechnen
	or	yh,yl			;Endadresse erreicht?
	brne	put010			;nein -> weiter
	ldi	xl,low(txbuf0)
	ldi	xh,high(txbuf0)		;sonst Adresse auf Anfangswert
put010:	sts	txp0wr,xl
	sts	txp0wr+1,xh		;Schreibzeiger speichern
	sbi	ucsr0b,udrie0		;Sendeinterrupt aktivieren
	sei				;Interrupts wieder freigeben
	ret
;
; Unterprogramm zum Auslesen eines Zeichens aus dem RS232-0
; Empfangspuffer
; Register: x,y
;	    r16 -> gelesenes Zeichen
;	    r17 -> 0= Zeichen gltig, 0xff= kein Zeichen im Puffer
;
get0ch:	cli				;Int vorbergehend sperren
	lds	xl,rxp0wr
	lds	xh,rxp0wr+1		;Schreibzeiger holen
	sei				;Int wieder freigeben
	lds	yl,rxp0rd
	lds	yh,rxp0rd+1		;Lesezeiger holen
	sub	xl,yl
	sbc	xh,yh			;Differenz berechnen
	or	xh,xl			;sind Zeichen im Puffer?
	brne	get010			;ja -> weiter
	ser	r17			;kein Zeichen im Puffer
	ret
get010:	ld	r16,y+			;Zeichen holen, Pointer+1
	ldi	xl,low(rxbuf0+rxbu0l)
	ldi	xh,high(rxbuf0+rxbu0l)	;Adresse Pufferende laden
	sub	xl,yl
	sbc	xh,yh			;Differenz berechnen
	or	xh,xl			;Endadresse erreicht?
	brne	get020			;nein -> weiter
	ldi	yl,low(rxbuf0)
	ldi	yh,high(rxbuf0)		;sonst Adresse auf Pufferanfang
get020:	sts	rxp0rd,yl
	sts	rxp0rd+1,yh		;Schreibzeiger speichern
	clr	r17			;Zeichen ist gltig
	ret
;
; Unterprogramm zum Auslesen eines Zeichens aus dem RS232-1
; Empfangspuffer
; Register: x,y
;	    r16 -> gelesenes Zeichen
;	    r17 -> 0= Zeichen gltig, 0xff= kein Zeichen im Puffer
;
get1ch:	cli				;Int vorbergehend sperren
	lds	xl,rxp1wr
	lds	xh,rxp1wr+1		;Schreibzeiger holen
	sei				;Int wieder freigeben
	lds	yl,rxp1rd
	lds	yh,rxp1rd+1		;Lesezeiger holen
	sub	xl,yl
	sbc	xh,yh			;Differenz berechnen
	or	xh,xl			;sind Zeichen im Puffer?
	brne	get110			;ja -> weiter
	ser	r17			;kein Zeichen im Puffer
	ret
get110:	ld	r16,y+			;Zeichen holen, Pointer+1
	ldi	xl,low(rxbuf1+rxbu1l)
	ldi	xh,high(rxbuf1+rxbu1l)	;Adresse Pufferende laden
	sub	xl,yl
	sbc	xh,yh			;Differenz berechnen
	or	xh,xl			;Endadresse erreicht?
	brne	get120			;nein -> weiter
	ldi	yl,low(rxbuf1)
	ldi	yh,high(rxbuf1)		;sonst Adresse auf Pufferanfang
get120:	sts	rxp1rd,yl
	sts	rxp1rd+1,yh		;Schreibzeiger speichern
	clr	r17			;Zeichen ist gltig
	ret
;
; Interrupt-Routinen -----
;
; Interrupt-Routine fr Timer1, wird alle 625s (=1600Hz) aufgerufen
;
t1comp:	in	insreg,sreg		;SREG sichern
	push	r16			;diverse Register sichern
	push	r17
	push	r18
	push	yl
	push	yh
	push	r0
	push	r1
;
; Teil 1 luft alle 625s und bedient den Dreh-Encoder
;
	in	r16,pinc		;Encoder lesen
	andi	r16,0x30		;Encoder-Bits filtern
	lds	r17,ensta1		;letzten Encoder-Status holen
	sts	ensta1,r16		;neuen Encoder-Status speichern
	cp	r16,r17			;Statusnderung?
	brne	t1c100			;ja -> Wert unstabil, Ende
	lds	r17,ensta2		;letzten stabilen Status holen
	sts	ensta2,r16		;neuen stabilen Status speich.
	cp	r16,r17			;Statusnderung?
	breq	t1c100			;nein -> Ende
;
	cpi	r17,0			;war alter Status=0?
	brne	t1c020			;nein -> weiter testen
	lds	r17,encpos		;sonst Encoderposition holen
	cpi	r16,0x10		;neuer Status=0x10 (rechts)?
	brne	t1c010			;nein -> weiter testen
	inc	r17			;sonst Encoderposition erhhen
	rjmp	t1c040			;und speichern
t1c010:	cpi	r16,0x20		;neuer Status=0x20 (links)?
	brne	t1c100			;nein -> Ende
	dec	r17			;sonst Encoderposit. vermindern
	rjmp	t1c040			;und speichern
;
t1c020:	cpi	r17,0x30		;war alter Status=0x30?
	brne	t1c100			;nein -> Ende
	lds	r17,encpos		;sonst Encoderposition holen
	cpi	r16,0x20		;neuer Status=0x20 (rechts)?
	brne	t1c030			;nein -> weiter testen
	inc	r17			;sonst Encoderposition erhhen
	rjmp	t1c040			;und speichern
t1c030:	cpi	r16,0x10		;neuer Status=0x10 (links)?
	brne	t1c100			;nein -> Ende
	dec	r17			;sonst Encoderposit. vrmindern
t1c040:	sts	encpos,r17		;neuen Encoderposit. speichern
;
; Teil 2 luft alle 20ms (Software-Teilung durch 32) und bedient den
; DCF77-Empfang
;
t1c100:	lds	r16,t1cnt1		;Timer1 Int-Zhler1 holen
	inc	r16			;erhhen
	sts	t1cnt1,r16		;und wieder speichern
	cpi	r16,32			;Endwert erreicht?
	brcc	t1c110			;ja -> weiter
	rjmp	t1cend			;sonst Ende
;
t1c110:	clr	r16			;Timer1 Int-Zhler1 lschen
	sts	t1cnt1,r16		;und wieder speichern
;
	lds	r16,dcfcnt		;DCF77-Zhler holen
	inc	r16			;Zhler erhhen
	sts	dcfcnt,r16		;und wieder speichern
	in	r16,pine		;aktuellen DCF77-Status lesen
	bst	r16,2			;DCF77-Status prfen
;
	in	r16,portd
	bld	r16,6			;Testausgabe
	out	portd,r16
;
	lds	r16,dcflag		;DCF77-Flags holen
	bld	r16,0			;und aktuellen Status speichern
	brts	t1c130			;ist DCF77 High? ja -> weiter
	bst	r16,1			;vorherigen DCF77-Status prfen
	brtc	t1c150			;DCF77 vorher H? nein -> weiter
	lds	r17,dcfcnt		;DCF77-Zhler holen
	cpi	r17,75			;Zhler<75 (<1,5s)?
	brcs	t1c120			;nein -> weiter
	sbr	r16,0x08		;neuen Minutenbeginn setzen
t1c120:	cpi	r17,45			;Zhler<45 (900ms) ?
	brcs	t1c150			;ja -> ignorieren
	clr	r17			;sonst
	sts	dcfcnt,r17		;DCF77-Zhler lschen
	rjmp	t1c150			;Ende
t1c130:	bst	r16,1			;sonst vorh. DCF-Status prfen
	brts	t1c150			;war DCF77 vorher H? ja -> Ende
	lds	r17,dcfcnt		;sonst DCF77-Zhler holen
	cpi	r17,13			;DCF-Zhler<13 (260ms)?
	brcc	t1c150			;nein -> ignorieren
	cpi	r17,3			;DCF-Zhler<3 (60ms)?
	brcs	t1c150			;ja -> ignorieren
	lds	r18,dcfimp		;sonst DCF-Impulsanzahl holen
	inc	r18			;Zhler erhhen
	sts	dcfimp,r18		;und wieder speichern
	cpi	r17,7			;Zhler<7 (<140ms)? -> Carry
	ldi	r17,6			;6 Pufferbytes bearbeiten
	ldi	yl,low(dcftab+6)
	ldi	yh,high(dcftab+6)	;Zeiger auf Pufferende setzen
	rol	r18			;Carry in Register legen
	com	r18			;alle Bits invertieren
	ror	r18			;wieder in Carry schieben
t1c140:	ld	r18,-y			;Pufferbyte holen
	ror	r18			;neues Bit einschieben als MSB
	st	y,r18			;LSB in Carry und speichern
	dec	r17			;alle Pufferbytes bearbeitet?
	brne	t1c140			;nein -> Schleife
t1c150:	bst	r16,0			;aktuellen DCF77-Status holen
	bld	r16,1			;und als neuen Status speichern
	sts	dcflag,r16		;DCF77-Flags speichern
;
; Teil 3 luft einmal pro Sekunde (Software-Teilung durch 50), steuert
; die Uhr und andere Zhler
;
t1c400:	lds	r16,t1cnt2		;Timer1 Int-Zhler2 holen
	inc	r16			;erhhen
	sts	t1cnt2,r16		;und wieder speichern
	cpi	r16,50			;Endwert erreicht?
	brcc	t1c410			;ja -> weiter
	rjmp	t1cend			;sonst Ende
;
t1c410:	clr	r16			;Timer1 Int-Zhler2 lschen
	sts	t1cnt2,r16		;und wieder speichern
;
; Sekunden-Timer
;
	lds	r16,secndl		;Sekunden Einer holen
	inc	r16			;erhhen
	sts	secndl,r16		;und wieder speichern
	cpi	r16,10			;Endwert erreicht?
	brcc	t1c420			;ja -> weiter
	rjmp	t1cend			;sonst Ende
;
t1c420:	clr	r16			;Sekunden Einer lschen
	sts	secndl,r16		;und wieder speichern
	lds	r16,secndh		;Sekunden Zehner holen
	inc	r16			;erhhen
	sts	secndh,r16		;und wieder speichern
	cpi	r16,6			;Endwert erreicht?
	brcc	t1c430			;ja -> weiter
	rjmp	t1cend			;sonst Ende
t1c430:	clr	r16			;Sekunden Zehner lschen
	sts	secndh,r16		;und wieder speichern
;
; Minuten-Timer
;
	lds	r16,minutl		;Minuten Einer holen
	inc	r16			;erhhen
	sts	minutl,r16		;und wieder speichern
	cpi	r16,10			;Endwert erreicht?
	brcc	t1c440			;ja -> weiter
	rjmp	t1cend			;sonst Ende
t1c440:	clr	r16			;Minuten Einer lschen
	sts	minutl,r16		;und wieder speichern
	lds	r16,minuth		;Minuten Zehner holen
	inc	r16			;erhhen
	sts	minuth,r16		;und wieder speichern
	cpi	r16,6			;Endwert erreicht?
	brcc	t1c450			;ja -> weiter
	rjmp	t1cend			;sonst Ende
t1c450:	clr	r16			;Minuten Zehner lschen
	sts	minuth,r16		;und wieder speichern
;
; Stunden-Timer
;
	lds	r16,hourl		;Stunden Einer holen
	inc	r16			;erhhen
	sts	hourl,r16		;und wieder speichern
	cpi	r16,10			;Endwert erreicht?
	brcs	t1c460			;nein -> Test auf '24'
	clr	r16			;sonst zurcksetzen
	sts	hourl,r16		;und wieder speichern
	lds	r16,hourh		;Stunden Zehner holen
	inc	r16			;erhhen
	sts	hourh,r16		;und wieder speichern
	rjmp	t1cend			;Ende
;
t1c460:	lds	r16,hourh		;Stunden Zehner holen
	cpi	r16,2			;Stunden Zehner=2?
	breq	t1c470			;ja -> weiter
	rjmp	t1cend			;sonst Ende
t1c470:	lds	r16,hourl		;nochmals Stunden Einer holen
	cpi	r16,4			;Stunden Einer=4 (=24)?
	breq	t1c480			;ja -> weiter
	rjmp	t1cend			;sonst Ende
t1c480:	clr	r16			;sonst Stunden zurcksetzen
	sts	hourl,r16		;Stunden Einer speichern
	sts	hourh,r16		;Stunden Zehner speichern
;
; Tages-Timer
;
	lds	r16,dayw		;Wochentag holen
	inc	r16			;erhhen
	sts	dayw,r16		;und wieder speichern
	cpi	r16,7			;Endwert erreicht?
	brcs	t1c490			;nein -> weiter
	clr	r16			;sonst Wochentag lschen
	sts	dayw,r16		;und wieder speichern
;
t1c490:	lds	r16,dayl		;Tag Einer holen
	inc	r16			;erhhen
	sts	dayl,r16		;und wieder speichern
	cpi	r16,10			;Endwert erreicht?
	brcs	t1c500			;nein -> Monatsende testen
	clr	r16			;sonst Tag Einer lschen
	sts	dayl,r16		;und wieder speichern
	lds	r16,dayh		;Tag Zehner holen
	inc	r16			;erhhen
	sts	dayh,r16		;und wieder speichern
;
t1c500:	lds	r16,monthh		;Monat Zehner holen
	ldi	r17,10			;mit 10
	mul	r16,r17			;multiplizieren
	lds	r18,monthl		;Monat Einer holen
	add	r18,r0			;mit Zehner verknpfen
	lds	r16,dayh		;Tag Zehner holen
	mul	r16,r17			;mit 10 multiplizieren
	lds	r16,dayl		;Tag Einer holen
	add	r16,r0			;mit Zehner verknpfen
;
	cpi	r18,1			;Monat Januar?
	brne	t1c510			;nein -> weiter testen
	cpi	r16,32			;Tag>31?
	brcc	t1c515			;ja -> Monatswechsel
	rjmp	t1cend			;sonst Ende
;
t1c510:	cpi	r18,2			;Monat Februar?
	brne	t1c530			;nein -> weiter testen
	lds	r18,yearh		;Jahr Zehner holen
	mul	r18,r17			;mit 10 multiplizieren
	lds	r18,yearl		;Jahr Einer holen
	add	r18,r0			;mit Zehner verknpfen
	andi	r18,0x03		;Schaltjahr?
	breq	t1c520			;ja -> 29 Tage
	cpi	r16,29			;sonst Tag>28?
t1c515:	brcc	t1c700			;ja -> Monatswechsel
	rjmp	t1cend			;sonst Ende
t1c520:	cpi	r16,30			;Tag>29?
	brcc	t1c700			;ja -> Monatswechsel
	rjmp	t1cend			;sonst Ende
;
t1c530:	cpi	r18,3			;Monat Mrz?
	brne	t1c540			;nein -> weiter testen
	cpi	r16,32			;Tag>31?
	brcc	t1c700			;ja -> Monatswechsel
	rjmp	t1cend			;sonst Ende
;
t1c540:	cpi	r18,4			;Monat April?
	brne	t1c550			;nein -> weiter testen
	cpi	r16,31			;Tag>30?
	brcc	t1c700			;ja -> Monatswechsel
	rjmp	t1cend			;sonst Ende
;
t1c550:	cpi	r18,5			;Monat Mai?
	brne	t1c560			;nein -> weiter testen
	cpi	r16,32			;Tag>31?
	brcc	t1c700			;ja -> Monatswechsel
	rjmp	t1cend			;sonst Ende
;
t1c560:	cpi	r18,6			;Monat Juni?
	brne	t1c570			;nein -> weiter testen
	cpi	r16,31			;Tag>30?
	brcc	t1c700			;ja -> Monatswechsel
	rjmp	t1cend			;sonst Ende
;
t1c570:	cpi	r18,7			;Monat Juli?
	brne	t1c580			;nein -> weiter testen
	cpi	r16,32			;Tag>31?
	brcc	t1c700			;ja -> Monatswechsel
	rjmp	t1cend			;sonst Ende
;
t1c580:	cpi	r18,8			;Monat August?
	brne	t1c590			;nein -> weiter testen
	cpi	r16,32			;Tag>31?
	brcc	t1c700			;ja -> Monatswechsel
	rjmp	t1cend			;sonst Ende
;
t1c590:	cpi	r18,9			;Monat September?
	brne	t1c600			;nein -> weiter testen
	cpi	r16,31			;Tag>30?
	brcc	t1c700			;ja -> Monatswechsel
	rjmp	t1cend			;sonst Ende
;
t1c600:	cpi	r18,10			;Monat Oktober?
	brne	t1c610			;nein -> weiter testen
	cpi	r16,32			;Tag>31?
	brcc	t1c700			;ja -> Monatswechsel
	rjmp	t1cend			;sonst Ende
;
t1c610:	cpi	r18,11			;Monat November?
	brne	t1c620			;nein -> Monat Dezember
	cpi	r16,31			;Tag>30?
	brcc	t1c700			;ja -> Monatswechsel
	rjmp	t1cend			;sonst Ende
;
t1c620:	cpi	r16,32			;Tag>31?
	brcs	t1cend			;nein -> Ende
;
t1c700:	ldi	r16,1			;Tag Einer
	sts	dayl,r16		;lschen
	clr	r16			;Tag Zehner
	sts	dayh,r16		;lschen
;
; Monats-Timer
;
	lds	r17,monthh		;Monat Zehner holen
	lds	r16,monthl		;Monat Einer holen
	inc	r16			;erhhen
	sts	monthl,r16		;und wieder speichern
	cpi	r16,10			;Endwert erreicht?
	brcs	t1c800			;nein -> Jahrende testen
	clr	r16			;sonst Monat Einer lschen
	sts	monthl,r16		;und wieder speichern
	inc	r17			;erhhen
	sts	monthh,r17		;und wieder speichern
;
t1c800:	cpi	r17,1			;Monat Zehner=1?
	brne	t1cend			;nein -> Ende
	cpi	r16,3			;Monat Einer=3?
	brne	t1cend			;nein -> Ende
	sts	monthl,r17		;sonst Monat Einer=1 setzen
	clr	r17
	sts	monthh,r17		;Monat Zehner lschen
;
	lds	r16,yearl		;Jahr Einer holen
	inc	r16			;erhhen
	sts	yearl,r16		;und wieder speichern
	cpi	r16,10			;Endwert erreicht?
	brcs	t1cend			;nein -> Ende
	clr	r16			;sonst Jahr Einer lschen
	sts	yearl,r16		;und wieder speichern
	lds	r16,yearh		;Jahr Zehner holen
	inc	r16			;erhhen
	sts	yearh,r16		;und wieder speichern
;
t1cend:	pop	r1
	pop	r0
	pop	yh
	pop	yl
	pop	r18
	pop	r17
	pop	r16			;diverse Register restaurieren
	out	sreg,insreg		;SREG wiederherstellen
	reti
;
; Interrupt-Routine fr Timer3, wird alle 20ms (=50Hz) aufgerufen
;
t3comp:	in	insreg,sreg		;SREG sichern
	push	r16			;diverse Register sichern
	push	r17
	push	r18
	push	zl
	push	zh
	in	zl,rampz
	push	zl
;
; Taster lesen und entprellen
;
	in	r16,pinc		;Taster lesen
	com	r16			;invertieren
	andi	r16,0xc0		;Taster-Bits filtern
	mov	r17,r16			;und kopieren
	lds	r18,keyst1		;letzten Tastenstatus holen
	sts	keyst1,r16		;neuen Tastenstatus speichern
	eor	r17,r18			;Statusnderungs-Maske erzeugen
	lds	r18,keyst2		;gltige Tasten holen
	and	r18,r17			;unvernderte Tasten filtern
	com	r17			;Maske invertieren
	and	r16,r17			;vernderte Tasten filtern
	or	r18,r16			;gltige Tasten kombinieren
	sts	keyst2,r18		;und speichern
	lds	r17,keyst3		;Bearbeitungs-Flags holen
	and	r17,r18			;losgelassene Tasten lschen
	sts	keyst3,r17		;Bearbeitungs-Flags speichern
;
; Ein/Ausschalten der Hintergrundbeleuchtung steuern
;
	in	r16,ocr2		;aktuellen PWM-Wert holen
	lds	r17,dflags		;Anzeige-Flags holen
	bst	r17,7			;LCD ausgeschaltet?
	brtc	t3c010			;nein -> weiter
	clr	r16			;sonst PWM-Wert=0 setzen
	rjmp	t3c080			;Wert speichern
;
t3c010:	lds	r18,confg2		;Konfigurationsbyte 2 holen
	bst	r18,1			;Backlight weich schalten?
	brts	t3c020			;ja -> bearbeiten
	bst	r17,6			;Backlight einschalten?
	brts	t3c070			;ja -> PWM-Wert= 0xff setzen
	rjmp	t3c040			;sonst PWM-Wert= 0 setzen
;
t3c020:	bst	r17,6			;Backlight einschalten?
	brts	t3c050			;ja -> weiter
	tst	r16			;unterer Endwert erreicht?
	breq	t3c080			;ja -> keine nderung
	mov	r18,r16			;sonst Wert kopieren und die
	swap	r18			;oberen 3 Bit als Decrement
	lsr	r18			;verwenden
	andi	r18,0x07		;relevante Bits filtern
	brne	t3c030			;Wert ok? ja -> weiter
	inc	r18			;sonst Decrement auf 1 setzen
t3c030:	sub	r16,r18			;PWM-Wert vermindern 
	brcc	t3c080			;Wert ok? ja -> speichern
t3c040:	clr	r16			;sonst unteren Endwert setzen
	rjmp	t3c080			;und speichern
;
t3c050:	cpi	r16,0xff		;oberer Endwert erreicht?
	breq	t3c080			;ja -> keine nderung
	mov	r18,r16			;sonst Wert kopieren und die
	swap	r18			;oberen 3 Bit als Increment
	lsr	r18			;verwenden
	andi	r18,0x07		;relevante Bits filtern
	brne	t3c060			;Wert ok? ja -> weiter
	inc	r18			;sonst Increment auf 1 setzen
t3c060:	add	r16,r18			;PWM-Wert erhhen
	brcc	t3c080			;Wert ok? ja -> speichern
t3c070:	ldi	r16,0xff		;sonst oberen Endwert setzen
t3c080:	out	ocr2,r16		;neuen PWM-Wert speichern
;
; LED-Debug-Modus: LED1 als DCF77 und LED2 als RS-232 Anzeige
;
	in	r18,portd		;Status PortD holen (LEDs)
	lds	r16,confg1		;Konfigurationsbyte 1 holen
	bst	r16,7			;Abwesenheit aktiviert?
	brtc	t3c090			;nein -> weiter testen
	sbr	r18,0x30		;sonst LEDs auschalten
	rjmp	t3c260			;neuen LED1/2-Status setzen
;
t3c090:	lds	r16,confg2		;Konfigurationsbyte 2 holen
	bst	r16,0			;LEDs im Normal-Modus?
	brtc	t3c100			;ja -> normale LED-Ausgabe
	in	r17,pine		;Eingnge PortD lesen
	bst	r17,2			;DCF77-Signal holen und
	bld	r18,4			;als neuen LED1-Status setzen
	lds	r17,rcvflg		;RS-232 Empfangsflag holen
	bst	r17,0			;LSB holen und
	bld	r18,5			;als neuen LED2-Status setzen
	sec				;Carry setzen
	ror	r17			;in MSB einschieben
	sts	rcvflg,r17		;Empfangsflag wieder speichern
	rjmp	t3c260			;neuen LED1/2-Status setzen
;
; Blinken und Dauerlicht von LED1 (rot) und LED2 (gelb) steuern
;
t3c100:	lds	r16,ledflg		;LED-Flags holen
	lds	r17,t3cnt2		;Timer3 Sekundenzhler holen
	andi	r16,0x05		;LED1 (rot) ansteuern?
	brne	t3c170			;ja -> bearbeiten
	lds	r16,ledflg		;sonst LED-Flags nochmals holen
	bst	r16,4			;Erinnerungs-Flag gesetzt?
	brts	t3c150			;ja -> bearbeiten
	sbr	r18,0x10		;sonst LED1 auschalten
	rjmp	t3c200			;weiter mit LED2
;
t3c150:	bst	r17,0			;Sekunde geradzahlig?
	brtc	t3c160			;ja -> LED1 ausschalten
	lds	r16,t3cnt1		;sonst Timer3 Int-Zhler holen
	tst	r16			;Zhlerstand=0?
	brne	t3c160			;nein -> LED1 ausschalten
	cbr	r18,0x10		;sonst LED1 einschalten
	rjmp	t3c200			;weiter mit LED2
t3c160:	sbr	r18,0x10		;sonst LED1 ausschalten
	rjmp	t3c200			;weiter mit LED2
;
t3c170:	bst	r16,0			;LED1 Dauerlicht?
	brtc	t3c180			;nein -> berspringen
	cbr	r18,0x10		;sonst LED1 einschalten
t3c180:	bst	r16,2			;LED1 Blinken?
	brtc	t3c200			;nein -> berspringen
	bst	r16,0			;zustzlich LED1 Dauerlicht?
	brtc	t3c190			;nein -> direkt zum Blinken
	bst	r17,0			;Sekunde geradzahlig?
	brtc	t3c200			;ja -> berspringen
t3c190:	lds	r16,t3cnt1		;Timer3 Interrupt-Zhler holen
	sbr	r18,0x10		;LED1 ausschalten
	cpi	r16,9			;Zhlerstand<9?
	brcs	t3c200			;ja -> LED1-Zustand ok
	cbr	r18,0x10		;sonst LED1 einschalten
	cpi	r16,17			;Zhlerstand<17?
	brcs	t3c200			;ja -> LED1-Zustand ok
	sbr	r18,0x10		;sonst LED1 ausschalten
	cpi	r16,25			;Zhlerstand<25?
	brcs	t3c200			;ja -> LED1-Zustand ok
	cbr	r18,0x10		;sonst LED1 einschalten
	cpi	r16,33			;Zhlerstand<33?
	brcs	t3c200			;ja -> LED1-Zustand ok
	sbr	r18,0x10		;sonst LED1 ausschalten
	cpi	r16,42			;Zhlerstand<42?
	brcs	t3c200			;ja -> LED1-Zustand ok
	cbr	r18,0x10		;sonst LED1 einschalten
;
t3c200:	lds	r16,ledflg		;LED-Flags holen
	andi	r16,0x0a		;LED2 (gelb) ansteuern?
	brne	t3c230			;ja -> bearbeiten
	lds	r16,ledflg		;sonst LED-Flags nochmals holen
	bst	r16,5			;Erinnerungs-Flag gesetzt?
	brts	t3c210			;ja -> bearbeiten
	sbr	r18,0x20		;sonst LED2 auschalten
	rjmp	t3c260			;Ende
;
t3c210:	bst	r17,0			;Sekunde geradzahlig?
	brtc	t3c220			;ja -> LED2 ausschalten
	lds	r16,t3cnt1		;sonst Timer3 Int-Zhler holen
	tst	r16			;Zhlerstand=0?
	brne	t3c220			;nein -> LED2 ausschalten
	cbr	r18,0x20		;sonst LED2 einschalten
	rjmp	t3c260			;Ende
t3c220:	sbr	r18,0x20		;sonst LED2 ausschalten
	rjmp	t3c260			;Ende
;
t3c230:	bst	r16,1			;LED2 Dauerlicht?
	brtc	t3c240			;nein -> berspringen
	cbr	r18,0x20		;sonst LED2 einschalten
t3c240:	bst	r16,3			;LED2 Blinken?
	brtc	t3c260			;nein -> berspringen
	bst	r16,1			;zustzlich LED2 Dauerlicht?
	brtc	t3c250			;nein -> direkt zum Blinken
	bst	r17,0			;Sekunde geradzahlig?
	brtc	t3c260			;ja -> berspringen
t3c250:	lds	r16,t3cnt1		;Timer3 Interrupt-Zhler holen
	sbr	r18,0x20		;LED2 ausschalten
	cpi	r16,9			;Zhlerstand<9?
	brcs	t3c260			;ja -> LED2-Zustand ok
	cbr	r18,0x20		;sonst LED2 einschalten
	cpi	r16,17			;Zhlerstand<17?
	brcs	t3c260			;ja -> LED2-Zustand ok
	sbr	r18,0x20		;sonst LED2 ausschalten
	cpi	r16,25			;Zhlerstand<25?
	brcs	t3c260			;ja -> LED2-Zustand ok
	cbr	r18,0x20		;sonst LED2 einschalten
	cpi	r16,33			;Zhlerstand<33?
	brcs	t3c260			;ja -> LED2-Zustand ok
	sbr	r18,0x20		;sonst LED2 ausschalten
	cpi	r16,42			;Zhlerstand<42?
	brcs	t3c260			;ja -> LED2-Zustand ok
	cbr	r18,0x20		;sonst LED2 einschalten
t3c260:	out	portd,r18		;neuen LED1/2-Status setzen
;
; Tonausgabe steuern
;
	lds	r16,sigtim		;Signaldauer holen
	tst	r16			;Signal ausgeben?
	brne	t3c270			;ja -> bearbeiten
	rjmp	t3c310			;sonst berspringen

t3c270:	lds	r16,dflags		;Anzeige-Flags holen
	bst	r16,5			;Lautsprecher eingeschaltet?
	brts	t3c280			;ja -> normale Signalausgabe
	ldi	r16,49			;sonst Signalzhler auf Endwert
	sts	sigcnt,r16		;Signalzhler speichern
	ldi	r16,1			;Signaldauer auf 1s setzen
	sts	sigtim,r16		;Signaldauer speichern

t3c280:	ldi	zl,low(sndtab*2)
	ldi	zh,high(sndtab*2)
	ldi	r18,byte3(sndtab*2)	;Zeiger auf Sound-Tabelle
	clr	r17			;H-Byte fr Adressberechnung
	lds	r16,sigsnd		;Signalsound-Nummer holen
	add	r16,r16
	add	r16,r16
	add	r16,r16			;Offset x8 fr Tabellenzugriff
	add	zl,r16
	adc	zh,r17
	adc	r18,r17			;Tabellenplatz berechnen
	out	rampz,r18
;
	lds	r16,sigcnt		;Signalzhler holen
	in	r18,tccr0		;Timer0 Steuerregister holen
	sbr	r18,1<<com00		;Ton einschalten
	elpm	r17,z+			;1. Ton-Einschaltzeit ignorier.
	elpm	r17,z+			;1. Ton-Ausschaltzeit holen
	cp	r16,r17			;Zhlerstand<Ausschaltzeit?
	brcs	t3c290			;ja -> Tonzustand ok
	cbr	r18,1<<com00		;Ton ausschalten
	elpm	r17,z+			;2. Ton-Einschaltzeit holen
	cp	r16,r17			;Zhlerstand<Ausschaltzeit?
	brcs	t3c290			;ja -> Tonzustand ok
	sbr	r18,1<<com00		;Ton einschalten
	elpm	r17,z+			;2. Ton-Ausschaltzeit holen
	cp	r16,r17			;Zhlerstand<Ausschaltzeit?
	brcs	t3c290			;ja -> Tonzustand ok
	cbr	r18,1<<com00		;Ton ausschalten
	elpm	r17,z+			;3. Ton-Einschaltzeit holen
	cp	r16,r17			;Zhlerstand<Ausschaltzeit?
	brcs	t3c290			;ja -> Tonzustand ok
	sbr	r18,1<<com00		;Ton einschalten
	elpm	r17,z+			;3. Ton-Ausschaltzeit holen
	cp	r16,r17			;Zhlerstand<Ausschaltzeit?
	brcs	t3c290			;ja -> Tonzustand ok
	cbr	r18,1<<com00		;Ton ausschalten
	elpm	r17,z+			;4. Ton-Einschaltzeit holen
	cp	r16,r17			;Zhlerstand<Ausschaltzeit?
	brcs	t3c290			;ja -> Tonzustand ok
	sbr	r18,1<<com00		;Ton einschalten
	elpm	r17,z+			;4. Ton-Ausschaltzeit holen
	cp	r16,r17			;Zhlerstand<Ausschaltzeit?
	brcs	t3c290			;ja -> Tonzustand ok
	cbr	r18,1<<com00		;sonst Ton immer ausschalten
t3c290:	out	tccr0,r18		;Timer0 Steueregister speichern
;
	inc	r16			;Signalzhler erhhen
	sts	sigcnt,r16		;Signalzhler wieder speichern
	cpi	r16,50			;Endwert erreicht?
	brcs	t3c310			;nein -> weiter
	clr	r16			;sonst Signalzhler lschen
	sts	sigcnt,r16		;Signalzhler wieder speichern
	lds	r16,sigtim		;Signaldauer holen
	dec	r16			;Wert um 1s vermindern
	sts	sigtim,r16		;Signaldauer wieder speichern
;
; ADC nach folgendem Muster steuern:
;   0ms: ADC-Ref-Wert lesen (fr Spannungsberechnung), ADC0 auswhlen
; 100ms: Wandlung starten
; 200ms: ADC0-Wert lesen (Lichtsensor), ADC1 auswhlen
; 300ms: Wandlung starten
; 400ms: ADC1-Wert lesen (Batterie-Spannung), ADC-Ref auswhlen
; 500ms: Wandlung starten
;
t3c310:	lds	r18,t3cnt1		;Timer1 Int-Zhler2 holen
	cpi	r18,0			;Zeitpunkt 0ms?
	brne	t3c320			;nein -> weiter testen
	lds	r16,adcref+4
	lds	r17,adcref+5		;Wert 2 holen
	sts	adcref+6,r16
	sts	adcref+7,r17		;und in Wert 3 kopieren
	lds	r16,adcref+2
	lds	r17,adcref+3		;Wert 1 holen
	sts	adcref+4,r16
	sts	adcref+5,r17		;und in Wert 2 kopieren
	lds	r16,adcref
	lds	r17,adcref+1		;Wert 0 holen
	sts	adcref+2,r16
	sts	adcref+3,r17		;und in Wert 1 kopieren
	in	r16,adcl
	in	r17,adch		;ADC-Wert holen
	sts	adcref,r16
	sts	adcref+1,r17		;und als Wert 0 speichern
	ldi	r16,0
	out	admux,r16		;Multiplexer auf ADC0 setzen
	rjmp	t3c400
t3c320:	cpi	r18,5			;Zeitpunkt 100ms?
	brne	t3c330			;nein -> weiter testen
	ldi	r17,(1<<aden|1<<adsc|7<<adps0) ;ADC ein, Vorteiler=128,
	out	adcsr,r17		;ADC Start Conversion
	rjmp	t3c400
t3c330:	cpi	r18,10			;Zeitpunkt 200ms?
	brne	t3c340			;nein -> weiter testen
	in	r16,adcl
	in	r17,adch		;sonst ADC-Wert holen
	sts	adclig,r16
	sts	adclig+1,r17		;und als ADC0-Wert speichern
	ldi	r17,1<<mux0
	out	admux,r17		;Multiplexer auf ADC1 setzen
	rjmp	t3c400
t3c340:	cpi	r18,15			;Zeitpunkt 300ms?
	brne	t3c350			;nein -> weiter testen
	ldi	r17,(1<<aden|1<<adsc|7<<adps0) ;ADC ein, Vorteiler=128,
	out	adcsr,r17		;ADC Start Conversion
	rjmp	t3c400
t3c350:	cpi	r18,20			;Zeitpunkt 400ms?
	brne	t3c360			;nein -> weiter testen
	lds	r16,adcbat+4
	lds	r17,adcbat+5		;Wert 2 holen
	sts	adcbat+6,r16
	sts	adcbat+7,r17		;und in Wert 3 kopieren
	lds	r16,adcbat+2
	lds	r17,adcbat+3		;Wert 1 holen
	sts	adcbat+4,r16
	sts	adcbat+5,r17		;und in Wert 2 kopieren
	lds	r16,adcbat
	lds	r17,adcbat+1		;Wert 0 holen
	sts	adcbat+2,r16
	sts	adcbat+3,r17		;und in Wert 1 kopieren
	in	r16,adcl
	in	r17,adch		;ADC-Wert holen
	sts	adcbat,r16
	sts	adcbat+1,r17		;und als Wert 0 speichern
	ldi	r17,0x1e<<mux0
	out	admux,r17		;Multiplexer auf ADC-Ref setzen
	rjmp	t3c400
t3c360:	cpi	r18,25			;Zeitpunkt 500ms?
	brne	t3c400			;nein -> weiter testen
	ldi	r17,(1<<aden|1<<adsc|7<<adps0) ;ADC ein, Vorteiler=128,
	out	adcsr,r17		;ADC Start Conversion
;
; whrend Statusanzeige zum Zeitpunkt 500ms Flag fr Neuausgabe setzen
;
	lds	r16,dimode		;Anzeigemodus holen
	cpi	r16,2			;Anzeigemodus Status?
	brne	t3c400			;nein -> berspringen
	lds	r16,xflag2		;sonst verschied. Flags 2 holen
	sbr	r16,1<<4		;Datenfelder neu ausgeben
	sts	xflag2,r16		;Flags wieder speichern
;
; Rckwrtszhlung des Timeout-Zhlers fr die zyklische Fehleranzeige
;
t3c400:	lds	r16,errtio		;Fehler-Timeout holen
	cpi	r16,0			;Timeout-Zhler=0 (Ende)?
	breq	t3c500			;ja -> nicht bearbeiten
	cpi	r16,0xff		;Timeout-Zhler=0xff (Stopp)?
	breq	t3c500			;ja -> nicht bearbeiten
	dec	r16			;sonst Zhler vermindern
	sts	errtio,r16		;Zhler wieder speichern
;
; Sekunden-Timer (Software-Teilung durch 50)
;
t3c500:	lds	r16,t3cnt1		;Timer1 Int-Zhler2 holen
	inc	r16			;erhhen
	sts	t3cnt1,r16		;und wieder speichern
	cpi	r16,50			;Endwert erreicht?
	brcc	t3c510			;ja -> weiter
	rjmp	t3cend			;sonst Ende
;
t3c510:	clr	r16			;Timer1 Int-Zhler2 lschen
	sts	t3cnt1,r16		;und wieder speichern
;
; Rckwrtszhlung der Timeout-Zhler fr die 16 Temperaturen, den
; Meldungs-Timeout, den Men-Timeout und den Lautsprecher-Timeout
;
	ldi	zl,low(temtio)
	ldi	zh,high(temtio)		;Zeiger auf Temperatur-Timeouts
	ldi	r17,16			;16 Werte bearbeiten
t3c520:	ld	r16,z			;Timeout-Zhler holen
	cpi	r16,0			;Timeout-Zhler=0 (Ende)?
	breq	t3c530			;ja -> nicht bearbeiten
	cpi	r16,0xff		;Timeout-Zhler=0xff (Stopp)?
	breq	t3c530			;ja -> nicht bearbeiten
	dec	r16			;sonst Zhler vermindern
t3c530:	st	z+,r16			;Zhler wieder speichern
	dec	r17			;alle Zhler bearbeitet?
	brne	t3c520			;nein -> Schleife
;
	lds	r16,msgtio		;Meldungs-Timeout holen
	cpi	r16,0			;Timeout-Zhler=0 (Ende)?
	breq	t3c540			;ja -> nicht bearbeiten
	cpi	r16,0xff		;Timeout-Zhler=0xff (Stopp)?
	breq	t3c540			;ja -> nicht bearbeiten
	dec	r16			;sonst Zhler vermindern
	sts	msgtio,r16		;Zhler wieder speichern
;
t3c540:	lds	r16,mentio		;Men-Timeout holen
	cpi	r16,0			;Timeout-Zhler=0 (Ende)?
	breq	t3c550			;ja -> nicht bearbeiten
	cpi	r16,0xff		;Timeout-Zhler=0xff (Stopp)?
	breq	t3c550			;ja -> nicht bearbeiten
	dec	r16			;sonst Zhler vermindern
	sts	mentio,r16		;Zhler wieder speichern
;
t3c550:	lds	r16,lsptio		;Lautsprecher-Timeout holen
	cpi	r16,0			;Timeout-Zhler=0 (Ende)?
	breq	t3c600			;ja -> nicht bearbeiten
	cpi	r16,0xff		;Timeout-Zhler=0xff (Stopp)?
	breq	t3c600			;ja -> nicht bearbeiten
	dec	r16			;sonst Zhler vermindern
	sts	lsptio,r16		;Zhler wieder speichern
;
; Minuten-Timer (Softwareteilung durch 60)
;
t3c600:	lds	r16,t3cnt2		;Timer3 Sekunden-Zhler holen
	inc	r16			;erhhen
	sts	t3cnt2,r16		;und wieder speichern
	cpi	r16,60			;Endwert erreicht?
	brcc	t3c610			;ja -> weiter
	rjmp	t3cend			;sonst Ende
;
t3c610:	clr	r16			;Timer3 Sekunden-Zhler lschen
	sts	t3cnt2,r16		;und wieder speichern
;
; Rckwrtszhlung des Timeout-Zhler fr die LCD-Abschaltung
;
	lds	r16,lcdtio		;LCD-Timeout holen
	cpi	r16,0			;Timeout-Zhler=0 (Ende)?
	breq	t3c800			;ja -> nicht bearbeiten
	cpi	r16,0xff		;Timeout-Zhler=0xff (Stopp)?
	breq	t3c800			;ja -> nicht bearbeiten
	dec	r16			;sonst Zhler vermindern
	sts	lcdtio,r16		;Zhler wieder speichern
;
; Stunden-Timer (Softwarteilung durch 60)
;
t3c800:	lds	r16,t3cnt3		;Timer3 Minuten-Zhler holen
	inc	r16			;erhhen
	sts	t3cnt3,r16		;und wieder speichern
	cpi	r16,60			;Endwert erreicht?
	brcc	t3c810			;ja -> weiter
	rjmp	t3cend			;sonst Ende
;
t3c810:	clr	r16			;Timer3 Minuten-Zhler lschen
	sts	t3cnt3,r16		;und wieder speichern
;
; Rckwrtszhlung des DCF77-Timeout-Zhlers
;
	lds	r16,dcftio		;DCF77-Timeout holen
	cpi	r16,0			;Timeout-Zhler=0 (Ende)?
	breq	t3cend			;ja -> nicht bearbeiten
	cpi	r16,0xff		;Timeout-Zhler=0xff (Stopp)?
	breq	t3cend			;ja -> nicht bearbeiten
	dec	r16			;sonst Zhler vermindern
	sts	dcftio,r16		;Zhler wieder speichern
;
t3cend:	pop	zl
	out	rampz,zl
	pop	zh
	pop	zl
	pop	r18
	pop	r17
	pop	r16			;diverse Register restaurieren
	out	sreg,insreg		;SREG wiederherstellen
	reti
;
; Interrupt-Routine "USART0 Daten-Register leer"
; nchstes Byte aus dem Sende-Ringpuffer0 holen und senden
;
txdre0:	in	insreg,sreg		;SREG sichern
	push	r16
	push	xl
	push	xh
	push	yl
	push	yh			;diverse Register sichern
	lds	xl,txp0wr
	lds	xh,txp0wr+1		;Schreibzeiger holen
	lds	yl,txp0rd
	lds	yh,txp0rd+1		;Lesezeiger holen
	sub	xl,yl
	sbc	xh,yh			;Differenz berechnen
	or	xh,xl			;sind Zeichen im Puffer?
	breq	txd030			;nein -> Sendeint abschalten
	ld	r16,y+			;sonst nchstes Byte holen
	out	udr0,r16		;und senden
	ldi	xl,low(txbuf0+txbu0l)
	ldi	xh,high(txbuf0+txbu0l)	;Adresse Pufferende laden
	sub	xl,yl
	sbc	xh,yh			;Differenz berechnen
	or	xh,xl			;Endadresse erreicht?
	brne	txd010			;nein -> weiter
	ldi	yl,low(txbuf0)
	ldi	yh,high(txbuf0)		;sonst Adresse auf Pufferanfang
txd010:	sts	txp0rd,yl
	sts	txp0rd+1,yh		;Lesezeiger wieder speichern
txd020:	pop	yh
	pop	yl
	pop	xh
	pop	xl
	pop	r16			;diverse Register restaurieren
	out	sreg,insreg		;SREG wiederherstellen
	reti
;
txd030:	cbi	ucsr0b,udrie0		;Sendeinterrupt abschalten
	rjmp	txd020			;Ende
;
; Interrupt-Routine "USART0 Empfang komplett"
; empfangenes Byte im Empfangs-Ringpuffer0 speichern
;
rxcpl0:	in	insreg,sreg		;SREG sichern
	push	r16
	push	xl
	push	xh
	push	yl
	push	yh			;diverse Register sichern
	lds	xl,rxp0wr
	lds	xh,rxp0wr+1		;Schreibzeiger holen
	in	r16,udr0		;empfangenes Byte lesen
	st	x+,r16			;Byte speichern, Zeiger+1
	ldi	yl,low(rxbuf0+rxbu0l)
	ldi	yh,high(rxbuf0+rxbu0l)	;Adresse Pufferende laden
	sub	yl,xl
	sbc	yh,xh			;Differenz berechnen
	or	yh,yl			;Endadresse erreicht?
	brne	rxc010			;nein -> weiter
	ldi	xl,low(rxbuf0)
	ldi	xh,high(rxbuf0)		;sonst Adresse auf Anfangswert
rxc010:	sts	rxp0wr,xl
	sts	rxp0wr+1,xh		;Schreibzeiger speichern
	pop	yh
	pop	yl
	pop	xh
	pop	xl
	pop	r16			;diverse Register restaurieren
	out	sreg,insreg		;SREG wiederherstellen
	reti
;
; Interrupt-Routine "USART1 Empfang komplett"
; empfangenes Byte im Empfangs-Ringpuffer1 speichern
;
rxcpl1:	in	insreg,sreg		;SREG sichern
	push	r16
	push	xl
	push	xh
	push	yl
	push	yh			;diverse Register sichern
	lds	xl,rxp1wr
	lds	xh,rxp1wr+1		;Schreibzeiger holen
	lds	r16,udr1		;empfangenes Byte lesen
	st	x+,r16			;Byte speichern, Zeiger+1
	ldi	yl,low(rxbuf1+rxbu1l)
	ldi	yh,high(rxbuf1+rxbu1l)	;Adresse Pufferende laden
	sub	yl,xl
	sbc	yh,xh			;Differenz berechnen
	or	yh,yl			;Endadresse erreicht?
	brne	rxc110			;nein -> weiter
	ldi	xl,low(rxbuf1)
	ldi	xh,high(rxbuf1)		;sonst Adresse auf Anfangswert
rxc110:	sts	rxp1wr,xl
	sts	rxp1wr+1,xh		;Schreibzeiger speichern
	pop	yh
	pop	yl
	pop	xh
	pop	xl
	pop	r16			;diverse Register restaurieren
	out	sreg,insreg		;SREG wiederherstellen
	reti
;
;
.cseg
;
; ASCII Zeichentabelle, jedes Zeichen wird mit 12 Byte definiert
; Byte 0 enthlt die Zeichenbreite (0-8), die restlichen Bytes ent-
; halten die Bitmap des Zeichens, Beispiel Prozentzeichen:
;
; Byte  0= 6 (Zeichen ist 6 Pixel breit)
; Byte  1= 0		. . . . . . . .
; Byte  2= 0x10		. . . o . . . .
; Byte  3= 0x29		. . o . o . . o
; Byte  4= 0x12		. . . o . . o .
; Byte  5= 4		. . . . . o . .
; Byte  6= 8		. . . . o . . .
; Byte  7= 0x12		. . . o . . o .
; Byte  8= 0x25		. . o . . o . o
; Byte  9= 2		. . . . . . o .
; Byte 10= 0		. . . . . . . .		(fr Unterlngen)
; Byte 11= 0		. . . . . . . .		(fr Unterlngen)
;
;
; ASCII 0x20 - 0x2f
;
chrset:	.db 5,0,0,0,0,0,0,0,0,0,0,0			    ;0x20 ' '
	.db 1,0,1,1,1,1,1,1,0,1,0,0			    ;0x21 '!'
	.db 5,0,0x1b,0x1b,9,0x12,0,0,0,0,0,0		    ;0x22 '"'
	.db 5,0,0,0x0a,0x0a,0x1f,0x0a,0x1f,0x0a,0x0a,0,0    ;0x23 '#'
	.db 5,0,4,0x0e,0x14,0x0e,5,0x15,0x0e,4,0,0	    ;0x24 '$'
	.db 6,0,0x10,0x29,0x12,4,8,0x12,0x25,2,0,0	    ;0x25 '%'
	.db 5,0,0x0c,0x12,0x12,0x14,8,0x15,0x12,0x0d,0,0    ;0x26 '&'
	.db 2,0,3,3,1,2,0,0,0,0,0,0			    ;0x27 '''
	.db 3,0,1,2,4,4,4,4,2,1,0,0			    ;0x28 '('
	.db 3,0,4,2,1,1,1,1,2,4,0,0			    ;0x29 ')'
	.db 5,0,0,0,0x0a,4,0x1f,4,0x0a,0,0,0		    ;0x2a '*'
	.db 5,0,0,0,4,4,0x1f,4,4,0,0,0			    ;0x2b '+'
	.db 2,0,0,0,0,0,0,0,3,3,1,2			    ;0x2c ','
	.db 5,0,0,0,0,0,0x1f,0,0,0,0,0			    ;0x2d '-'
	.db 2,0,0,0,0,0,0,0,3,3,0,0			    ;0x2e '.'
	.db 4,0,1,1,2,2,4,4,8,8,0,0			    ;0x2f '/'
;
; ASCII 0x30 - 0x3f
;
	.db 5,0,0x0e,0x11,0x11,0x11,0x11,0x11,0x11,0x0e,0,0 ;0x30 '0'
	.db 5,0,2,6,0x0a,2,2,2,2,2,0,0			    ;0x31 '1'
	.db 5,0,0x0e,0x11,1,2,4,8,0x10,0x1f,0,0		    ;0x32 '2'
	.db 5,0,0x0e,0x11,1,6,1,1,0x11,0x0e,0,0		    ;0x33 '3'
	.db 5,0,2,6,0x0a,0x12,0x1f,2,2,2,0,0		    ;0x34 '4'
	.db 5,0,0x1f,0x10,0x10,0x1e,1,1,0x11,0x0e,0,0	    ;0x35 '5'
	.db 5,0,0x0e,0x11,0x10,0x1e,0x11,0x11,0x11,0x0e,0,0 ;0x36 '6'
	.db 5,0,0x1f,1,1,2,2,4,4,4,0,0			    ;0x37 '7'
	.db 5,0,0x0e,0x11,0x11,0x0e,0x11,0x11,0x11,0x0e,0,0 ;0x38 '8'
	.db 5,0,0x0e,0x11,0x11,0x11,0x0f,1,0x11,0x0e,0,0    ;0x39 '9'
	.db 2,0,0,0,0,3,3,0,3,3,0,0			    ;0x3a ':'
	.db 2,0,0,0,0,3,3,0,3,3,1,2			    ;0x3b ';'
	.db 4,0,0,1,2,4,8,4,2,1,0,0			    ;0x3c '<'
	.db 5,0,0,0,0,0x1f,0,0x1f,0,0,0,0		    ;0x3d '='
	.db 4,0,0,8,4,2,1,2,4,8,0,0			    ;0x3e '>'
	.db 5,0,0x0e,0x11,1,2,4,4,0,4,0,0		    ;0x3f '?'
;
; ASCII 0x40 - 0x4f
;
	.db 7,0,0x1e,0x21,0x4d,0x55,0x55,0x4e,0x20,0x1e,0,0 ;0x40 '@'
	.db 5,0,0x04,0x0a,0x11,0x11,0x1f,0x11,0x11,0x11,0,0 ;0x41 'A'
	.db 5,0,0x1e,0x11,0x11,0x1e,0x11,0x11,0x11,0x1e,0,0 ;0x42 'B'
	.db 5,0,0x0e,0x11,0x10,0x10,0x10,0x10,0x11,0x0e,0,0 ;0x43 'C'
	.db 5,0,0x1c,0x12,0x11,0x11,0x11,0x11,0x12,0x1c,0,0 ;0x44 'D'
	.db 5,0,0x1f,0x10,0x10,0x1e,0x10,0x10,0x10,0x1f,0,0 ;0x45 'E'
	.db 5,0,0x1f,0x10,0x10,0x1e,0x10,0x10,0x10,0x10,0,0 ;0x46 'F'
	.db 5,0,0x0e,0x11,0x10,0x10,0x13,0x11,0x11,0x0e,0,0 ;0x47 'G'
	.db 5,0,0x11,0x11,0x11,0x1f,0x11,0x11,0x11,0x11,0,0 ;0x48 'H'
	.db 1,0,1,1,1,1,1,1,1,1,0,0			    ;0x49 'I'
	.db 4,0,1,1,1,1,1,1,9,6,0,0			    ;0x4a 'J'
	.db 5,0,0x11,0x12,0x14,0x18,0x18,0x14,0x12,0x11,0,0 ;0x4b 'K'
	.db 5,0,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x1f,0,0 ;0x4c 'L'
	.db 5,0,0x11,0x1b,0x15,0x15,0x11,0x11,0x11,0x11,0,0 ;0x4d 'M'
	.db 5,0,0x11,0x19,0x19,0x15,0x15,0x13,0x13,0x11,0,0 ;0x4e 'N'
	.db 5,0,0x0e,0x11,0x11,0x11,0x11,0x11,0x11,0x0e,0,0 ;0x4f '0'
;
; ASCII 0x50 - 0x5f
;
	.db 5,0,0x1e,0x11,0x11,0x11,0x1e,0x10,0x10,0x10,0,0 ;0x50 'P'
	.db 5,0,0x0e,0x11,0x11,0x11,0x11,0x15,0x12,0x0d,0,0 ;0x51 'Q'
	.db 5,0,0x1e,0x11,0x11,0x11,0x1e,0x14,0x12,0x11,0,0 ;0x52 'R'
	.db 5,0,0x0e,0x11,0x10,0x0e,1,1,0x11,0x0e,0,0	    ;0x53 'S'
	.db 5,0,0x1f,4,4,4,4,4,4,4,0,0			    ;0x54 'T'
	.db 5,0,0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x0e,0,0 ;0x55 'U'
	.db 5,0,0x11,0x11,0x11,0x11,0x0a,0x0a,4,4,0,0	    ;0x56 'V'
	.db 5,0,0x11,0x11,0x11,0x11,0x15,0x15,0x15,0x0a,0,0 ;0x57 'W'
	.db 5,0,0x11,0x11,0x0a,4,0x0a,0x11,0x11,0x11,0,0    ;0x58 'X'
	.db 5,0,0x11,0x11,0x11,0x0a,4,4,4,4,0,0		    ;0x59 'Y'
	.db 6,0,0x3f,1,2,4,8,0x10,0x20,0x3f,0,0		    ;0x5a 'Z'
	.db 3,0,7,4,4,4,4,4,4,7,0,0			    ;0x5b '['
	.db 4,0,8,8,4,4,2,2,1,1,0,0			    ;0x5c '\'
	.db 3,0,7,1,1,1,1,1,1,7,0,0			    ;0x5d ']'
	.db 5,0,4,0x0a,0x11,0,0,0,0,0,0,0		    ;0x5e '^'
	.db 5,0,0,0,0,0,0,0,0,0x1f,0,0			    ;0x5f '_'
;
; ASCII 0x60 - 0x6f
;
	.db 3,0,4,2,1,0,0,0,0,0,0,0			    ;0x60 '`'
	.db 5,0,0,0,0,0x0e,1,0x0f,0x11,0x0f,0,0		    ;0x61 'a'
	.db 5,0,0x10,0x10,0x10,0x1e,0x11,0x11,0x11,0x1e,0,0 ;0x62 'b'
	.db 4,0,0,0,0,7,8,8,8,7,0,0			    ;0x63 'c'
	.db 5,0,1,1,1,0x0f,0x11,0x11,0x11,0x0f,0,0	    ;0x64 'd'
	.db 5,0,0,0,0,0x0e,0x11,0x1f,0x10,0x0e,0,0	    ;0x65 'e'
	.db 3,0,1,2,2,7,2,2,2,2,0,0			    ;0x66 'f'
	.db 5,0,0,0,0,0x0f,0x11,0x11,0x11,0x0f,1,0x0e	    ;0x67 'g'
	.db 5,0,0x10,0x10,0x10,0x1e,0x11,0x11,0x11,0x11,0,0 ;0x68 'h'
	.db 1,0,0,1,0,1,1,1,1,1,0,0			    ;0x69 'i'
	.db 2,0,0,1,0,1,1,1,1,1,1,2			    ;0x6a 'j'
	.db 4,0,8,8,8,9,0x0a,0x0c,0x0a,9,0,0		    ;0x6b 'k'
	.db 2,0,2,2,2,2,2,2,2,1,0,0			    ;0x6c 'l'
	.db 5,0,0,0,0,0x1a,0x15,0x15,0x15,0x15,0,0	    ;0x6d 'm'
	.db 5,0,0,0,0,0x16,0x19,0x11,0x11,0x11,0,0	    ;0x6e 'n'
	.db 5,0,0,0,0,0x0e,0x11,0x11,0x11,0x0e,0,0	    ;0x6f 'o'
;
; ASCII 0x70 - 0x7f
;
	.db 5,0,0,0,0,0x1e,0x11,0x11,0x11,0x1e,0x10,0x10    ;0x70 'p'
	.db 5,0,0,0,0,0x0f,0x11,0x11,0x11,0x0f,1,1	    ;0x71 'q'
	.db 4,0,0,0,0,0x0a,0x0d,8,8,8,0,0		    ;0x72 'r'
	.db 5,0,0,0,0,0x0e,0x10,0x0e,1,0x1e,0,0		    ;0x73 's'
	.db 3,0,2,2,2,7,2,2,2,1,0,0			    ;0x74 't'
	.db 5,0,0,0,0,0x11,0x11,0x11,0x11,0x0e,0,0	    ;0x75 'u'
	.db 5,0,0,0,0,0x11,0x11,0x11,0x0a,4,0,0		    ;0x76 'v'
	.db 5,0,0,0,0,0x11,0x11,0x15,0x15,0x0a,0,0	    ;0x77 'w'
	.db 5,0,0,0,0,0x11,0x0a,4,0x0a,0x11,0,0		    ;0x78 'x'
	.db 5,0,0,0,0,0x11,0x11,0x11,0x11,0x0f,1,0x0e	    ;0x79 'y'
	.db 5,0,0,0,0,0x1f,2,4,8,0x1f,0,0		    ;0x7a 'z'
	.db 3,0,1,2,2,4,2,2,2,1,0,0			    ;0x7b '{'
	.db 1,0,1,1,1,0,1,1,1,1,0,0			    ;0x7c '|'
	.db 3,0,4,2,2,1,2,2,2,4,0,0			    ;0x7d '}'
	.db 5,0,8,0x15,2,0,0,0,0,0,0,0			    ;0x7e '~'
	.db 0,0,0,0,0,0,0,0,0,0,0,0		    ;0x7f Endezeichen
;
; ASCII 0x80 - 0x8f (fr Umlaute und Sonderzeichen)
;
	.db 5,0x11,4,0x0a,0x11,0x11,0x1f,0x11,0x11,0x11,0,0 ;0x80 ''
	.db 5,0x0a,0,0x0e,0x11,0x11,0x11,0x11,0x11,0x0e,0,0 ;0x81 ''
	.db 5,0x0a,0,0x11,0x11,0x11,0x11,0x11,0x11,0x0e,0,0 ;0x82 ''
	.db 5,0,0,0x0a,0,0x0e,1,0x0f,0x11,0x0f,0,0	    ;0x83 ''
	.db 5,0,0,0x0a,0,0x0e,0x11,0x11,0x11,0x0e,0,0	    ;0x84 ''
	.db 5,0,0,0x0a,0,0x11,0x11,0x11,0x11,0x0e,0,0	    ;0x85 ''
	.db 5,0,0x0e,0x11,0x11,0x16,0x11,0x11,0x11,0x16,0,0 ;0x86 ''
	.db 0,0,0,0,0,0,0,0,0,0,0,0			    ;0x87 frei
	.db 0,0,0,0,0,0,0,0,0,0,0,0			    ;0x88 frei
	.db 0,0,0,0,0,0,0,0,0,0,0,0			    ;0x89 frei
	.db 0,0,0,0,0,0,0,0,0,0,0,0			    ;0x8a frei
	.db 0,0,0,0,0,0,0,0,0,0,0,0			    ;0x8b frei
	.db 0,0,0,0,0,0,0,0,0,0,0,0			    ;0x8c frei
	.db 0,0,0,0,0,0,0,0,0,0,0,0			    ;0x8d frei
	.db 0,0,0,0,0,0,0,0,0,0,0,0			    ;0x8e frei
	.db 0,0,0,0,0,0,0,0,0,0,0,0			    ;0x8f frei
;
; ASCII 0x90 - 0x95 (fr Umlaute und Sonderzeichen)
;
	.db 0,0,0,0,0,0,0,0,0,0,0,0			    ;0x90 frei
	.db 0,0,0,0,0,0,0,0,0,0,0,0			    ;0x91 frei
	.db 0,0,0,0,0,0,0,0,0,0,0,0			    ;0x92 frei
	.db 0,0,0,0,0,0,0,0,0,0,0,0			    ;0x93 frei
	.db 0,0,0,0,0,0,0,0,0,0,0,0			    ;0x94 frei
	.db 0,0,0,0,0,0,0,0,0,0,0,0			    ;0x95 frei
;
; ^ bis an diese Stelle knnen eigene Zeichen definiert werden,
; die Konstante maxasc muss den Code des letzten Zeichen enthalten
;
; ASCII 0x96 - 0x9f
;
	.db 7,0x60,0x51,0x62,0x56,0x64,0x1e,0x14,0x16,0x12,0x11,0x38
						   ;0x96 Antennensy.+BT
	.db 7,0,8,8,0x14,0x14,0x22,0x22,0x41,0x7f,0,0;0x97 Delta-Symbol
	.db 6,0,0,0,0x20,0x11,9,5,3,0x1f,0,0	   ;0x98 Minimum-Symbol
	.db 6,0,0,0,0x1f,3,5,9,0x11,0x20,0,0	   ;0x99 Maximum-Symbol
	.db 6,0,0,0,0x0d,0x12,0x25,0x29,0x12,0x2c,0,0 ;0x9a Mittelwert
	.db 5,0,1,2,4,8,0x0f,1,0x0a,0x0c,0x0e,0	      ;0x9b Blitzsymbol
	.db 7,0,1,2,6,4,0x1e,0x14,0x16,0x12,0x11,0x38;0x9c Antennensym.
	.db 6,1,3,7,0x3d,0x25,0x25,0x25,0x3d,7,3,1   ;0x9d Lautsprecher
	.db 5,0,0,0,0x11,0x0a,4,0x0a,0x11,0,0,0	   ;0x9e Ant/Lsp 'aus'
	.db 5,0,4,2,9,5,0x15,5,9,2,4,0		   ;0x9f Ant/Lsp 'ein'
;
; ASCII 0xa0 - 0xaf
;
	.db 3,0,0,0,0,0,0,0,0,0,0,0		    ;0xa0 schmales ' '
	.db 7,8,0x1c,0x1c,0x3e,0x3e,0x3e,0x7f,0x10,0x30,0x30,0
						    ;0xa1 Alarmsymbol
	.db 7,8,0x14,0x14,0x14,0x14,0x1c,0x1c,0x3e,0x3e,0x1c,0
						    ;0xa2 Temp-Symbol
	.db 7,0x0e,0x0e,0x0e,0x0c,0x1c,0x18,0x18,0,0x30,0x30,0
						    ;0xa3 Fehlersymbol
	.db 7,0x41,0x63,0x22,0x36,0x1c,0x1c,0x36,0x22,0x63,0x41,0
						    ;0xa4 Nicht-Ok-Sym.
	.db 2,0,3,3,0,0,0,0,0,0,0,0			    ;0xa5 ''
	.db 8,0,0xce,0xd1,0x10,0x10,0x10,0x10,0x11,0x0e,0,0 ;0xa6 'C'
	.db 5,0,0x18,0x18,0,0,0,0,0,0,0,0		    ;0xa7 ' '
	.db 5,0,0x18,0x18,0,0x0f,3,5,9,0x10,0,0		    ;0xa8 '^'
	.db 5,0,0x18,0x18,0,0x10,9,5,3,0x0f,0,0		    ;0xa9 'v'
	.db 5,0,0x19,0x19,1,1,1,1,0,1,0,0		    ;0xaa '!'
	.db 2,0,3,3,0,0,0,0,0,0,0,0		    ;0xab schmales ''
	.db 5,0,0,0,0,0,0,0,0,0x11,0x1f,0	    ;0xac Cursorsymbol
	.db 5,0,4,0x0a,0x0a,0x0a,0x1f,0x1b,0x1b,0x1f,0,0
						    ;0xad Schlosssymbol
	.db 7,1,1,3,3,6,4,0x4c,0x68,0x38,0x10,0	    ;0xae Ok-Symbol
	.db 8,0,0,0x13,0x23,0x43,0xff,0x43,0x23,0x13,0,0
						    ;0xaf Lschsymbol
;
; ASCII 0xb0 - 0xb7
;
	.db 5,0,1,3,5,9,0x11,9,5,3,1,0		;0xb0 Pfeil links hohl
	.db 5,0,1,3,7,0x0f,0x1f,0x0f,7,3,1,0	;0xb1 Pfeil links voll
	.db 5,0,0x10,0x18,0x14,0x12,0x11,0x12,0x14,0x18,0x10,0
						;0xb2 Pfeil rechts hohl
	.db 5,0,0x10,0x18,0x1c,0x1e,0x1f,0x1e,0x1c,0x18,0x10,0
						;0xb3 Pfeil rechts voll
	.db 6,0,0,0,0x3f,0x21,0x21,0x21,0x21,0x3f,0,0 ;0xb4 Checkbox1a
	.db 6,0,0,0,0x3f,0x3f,0x3f,0x3f,0x3f,0x3f,0,0 ;0xb5 Checkbox1b
	.db 6,0,0,0,0x0c,0x12,0x21,0x21,0x12,0x0c,0,0 ;0xb6 Checkbox2a
	.db 6,0,0,0,0x0c,0x1e,0x3f,0x3f,0x1e,0x0c,0,0 ;0xb7 Checkbox2b
;
;
; Ziffern 0-9, sowie '-' und Leerzeichen im Groformat; besteht aus
; jeweils 2 Blcken im weiter oben beschriebenen 12-Byte-Format fr den
; oberen Teil, 2 fr den mittleren Teil und 2 fr den unteren Teil
;
; 0xb8 - 0xbd '0'
;
	.db 8,0,0,7,0x1f,0x3f,0x3f,0x7c,0x78,0x78,0xf8,0xf0
	.db 8,0,0,0xe0,0xf8,0xfc,0xfc,0x3e,0x1e,0x1e,0x1f,0x0f
	.db 8,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0,0xf0
	.db 8,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f
	.db 8,0xf0,0xf8,0x78,0x78,0x7c,0x3f,0x3f,0x1f,7,0,0
	.db 8,0x0f,0x1f,0x1e,0x1e,0x3e,0xfc,0xfc,0xf8,0xe0,0,0
;
; 0xbe - 0xc3 '1'
;
	.db 8,0,0,0,0,0,0,1,3,7,0x0f,0x1f
	.db 8,0,0,0x78,0x78,0xf8,0xf8,0xf8,0xf8,0xf8,0xf8,0x78
	.db 8,0x1e,0x1c,0,0,0,0,0,0,0,0,0
	.db 8,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78
	.db 8,0,0,0,0,0,0,0,0,0,0,0
	.db 8,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0,0
;
; 0xc4 - 0xc9 '2'
;
	.db 8,0,0,7,0x1f,0x3f,0x7f,0x7c,0xf8,0xf0,0,0
	.db 8,0,0,0xe0,0xf8,0xfc,0xfe,0x3e,0x1f,0x0f,0x0f,0x0f
	.db 8,0,0,0,0,0,0,0,1,3,7,0x0f
	.db 8,0x0f,0x1f,0x1e,0x3e,0x3c,0x7c,0xf8,0xf0,0xe0,0xc0,0x80
	.db 8,0x1f,0x3e,0x7c,0x78,0xf8,0xff,0xff,0xff,0xff,0,0
	.db 8,0,0,0,0,0,0xff,0xff,0xff,0xff,0,0
;
; 0xca - 0xcf '3'
;
	.db 8,0,0,3,0x0f,0x1f,0x3f,0x3e,0x7c,0x78,0,0
	.db 8,0,0,0xc0,0xf0,0xf8,0xfc,0x7c,0x3e,0x1e,0x1e,0x1e
	.db 8,0,0,0,7,7,7,7,0,0,0,0
	.db 8,0x1e,0x3e,0x7c,0xfc,0xf8,0xf8,0xfc,0x7e,0x3e,0x1f,0x0f
	.db 8,0,0,0xf0,0xf8,0x7c,0x7f,0x3f,0x1f,7,0,0
	.db 8,0x0f,0x0f,0x0f,0x1f,0x3e,0xfe,0xfc,0xf8,0xe0,0,0
;
; 0xd0 - 0xd5 '4'
;
	.db 8,0,0,0,0,0,0,1,1,3,3,7
	.db 8,0,0,0x78,0x78,0xf8,0xf8,0xf8,0xf8,0xf8,0xf8,0xf8
	.db 8,7,0x0f,0x0f,0x1e,0x1e,0x3c,0x3c,0x78,0x78,0xff,0xff
	.db 8,0xf8,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0xff,0xff
	.db 8,0xff,0xff,0,0,0,0,0,0,0,0,0
	.db 8,0xff,0xff,0x78,0x78,0x78,0x78,0x78,0x78,0x78,0,0
;
; 0xd6 - 0xdb '5'
;
	.db 8,0,0,0x1f,0x1f,0x1f,0x1f,0x1e,0x3e,0x3e,0x3c,0x3c
	.db 8,0,0,0xfe,0xfe,0xfe,0xfe,0,0,0,0,0
	.db 8,0x3c,0x7c,0x7f,0x7f,0x7f,0x7f,0x78,0,0,0,0
	.db 8,0,0,0xe0,0xf8,0xfc,0xfe,0x3e,0x1f,0x0f,0x0f,0x0f
	.db 8,0,0,0xf0,0xf8,0x7c,0x7f,0x3f,0x1f,7,0,0
	.db 8,0x0f,0x0f,0x0f,0x1f,0x3e,0xfe,0xfc,0xf8,0xe0,0,0
;
; 0xdc - 0xe1 '6'
;
	.db 8,0,0,7,0x1f,0x3f,0x3f,0x7c,0x78,0x78,0xf8,0xf0
	.db 8,0,0,0xe0,0xf8,0xfc,0xfc,0x3e,0x1e,0x1e,0,0
	.db 8,0xf0,0xf0,0xf7,0xff,0xff,0xff,0xfc,0xf8,0xf0,0xf0,0xf0
	.db 8,0,0,0xe0,0xf8,0xfc,0xfe,0x3e,0x1f,0x0f,0x0f,0x0f
	.db 8,0xf0,0xf0,0xf0,0xf8,0x7c,0x7f,0x3f,0x1f,7,0,0
	.db 8,0x0f,0x0f,0x0f,0x1f,0x3e,0xfe,0xfc,0xf8,0xe0,0,0
;
; 0xe2 - 0xe7 '7'
;
	.db 8,0,0,0xff,0xff,0xff,0xff,0,0,0,0,0
	.db 8,0,0,0xff,0xff,0xff,0xff,0x0f,0x1f,0x1e,0x3e,0x3c
	.db 8,0,0,0,0,1,1,1,3,3,3,3
	.db 8,0x7c,0x78,0xf8,0xf0,0xf0,0xe0,0xe0,0xe0,0xc0,0xc0,0xc0
	.db 8,7,7,7,7,7,7,7,7,7,0,0
	.db 8,0xc0,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0,0
;
; 0xe8 - 0xed '8'
;
	.db 8,0,0,3,0x0f,0x1f,0x3f,0x3e,0x7c,0x78,0x78,0x78
	.db 8,0,0,0xc0,0xf0,0xf8,0xfc,0x7c,0x3e,0x1e,0x1e,0x1e
	.db 8,0x78,0x7c,0x3e,0x3f,0x1f,0x3f,0x7f,0x7c,0xf8,0xf0,0xf0
	.db 8,0x1e,0x3e,0x7c,0xfc,0xf8,0xfc,0xfe,0x3e,0x1f,0x0f,0x0f
	.db 8,0xf0,0xf0,0xf0,0xf8,0x7c,0x7f,0x3f,0x1f,7,0,0
	.db 8,0x0f,0x0f,0x0f,0x1f,0x3e,0xfe,0xfc,0xf8,0xe0,0,0
;
; 0xee - 0xf3 '9'
;
	.db 8,0,0,7,0x1f,0x3f,0x7f,0x7c,0xf8,0xf0,0xf0,0xf0
	.db 8,0,0,0xe0,0xf8,0xfc,0xfe,0x3e,0x1f,0x0f,0x0f,0x0f
	.db 8,0xf0,0xf0,0xf0,0xf8,0x7c,0x7f,0x3f,0x1f,7,0,0
	.db 8,0x0f,0x0f,0x0f,0x1f,0x3f,0xff,0xff,0xff,0xef,0x0f,0x0f
	.db 8,0,0,0x78,0x78,0x7c,0x3f,0x3f,0x1f,7,0,0
	.db 8,0x0f,0x1f,0x1e,0x1e,0x3e,0xfc,0xfc,0xf8,0xe0,0,0
;
; 0xf4 - 0xf9 '-' (10)
;
	.db 8,0,0,0,0,0,0,0,0,0,0,0
	.db 8,0,0,0,0,0,0,0,0,0,0,0
	.db 8,0,0,0,0x3f,0x7f,0x7f,0x3f,0,0,0,0
	.db 8,0,0,0,0xfc,0xfe,0xfe,0xfc,0,0,0,0
	.db 8,0,0,0,0,0,0,0,0,0,0,0
	.db 8,0,0,0,0,0,0,0,0,0,0,0
;
; 0xfa - 0xff ' ' (11)
;
	.db 8,0,0,0,0,0,0,0,0,0,0,0
	.db 8,0,0,0,0,0,0,0,0,0,0,0
	.db 8,0,0,0,0,0,0,0,0,0,0,0
	.db 8,0,0,0,0,0,0,0,0,0,0,0
	.db 8,0,0,0,0,0,0,0,0,0,0,0
	.db 8,0,0,0,0,0,0,0,0,0,0,0
;
; ASCII Mini-Zeichentabelle fr Grafik-Beschriftung (3x5 Pixel), jedes
; Zeichen wird mit 6 Byte definiert: Byte 0 enthlt die Zeichenbreite
; (0-8), die restlichen Bytes enthalten die Bitmap des Zeichens,
; Beispiel Prozentzeichen:
;
; Byte  0= 3 (Zeichen ist 3 Pixel breit)
; Byte  1= 5		. . . . . o . o
; Byte  2= 1		. . . . . . . o
; Byte  3= 2		. . . . . . o .
; Byte  4= 4		. . . . . o . .
; Byte  5= 5		. . . . . o . o
;
;
; ASCII 0x20 - 0x2f
;
chrsmi:	.db 3,0,0,0,0,0					;0x20 ' '
	.db 1,1,1,1,0,1					;0x21 '!'
	.db 3,5,0,0,0,0					;0x22 '"'
	.db 5,0x0a,0x1f,0x0a,0x1f,0x0a			;0x23 '#'
	.db 5,0x1f,0x14,0x1f,5,0x1f			;0x24 '$'
	.db 3,5,1,2,4,5					;0x25 '%'
	.db 4,4,0x0a,4,0x0a,0x0d			;0x26 '&'
	.db 1,1,0,0,0,0					;0x27 '''
	.db 2,1,2,2,2,1					;0x28 '('
	.db 2,2,1,1,1,2					;0x29 ')'
	.db 5,0x0a,4,0x1f,4,0x0a			;0x2a '*'
	.db 3,0,2,7,2,0					;0x2b '+'
	.db 2,0,0,0,1,2					;0x2c ','
	.db 3,0,0,7,0,0					;0x2d '-'
	.db 1,0,0,0,0,1					;0x2e '.'
	.db 3,1,1,2,4,4					;0x2f '/'
;
; ASCII 0x30 - 0x3f
;
	.db 3,7,5,5,5,7					;0x30 '0'
	.db 3,1,3,1,1,1					;0x31 '1'
	.db 3,7,1,7,4,7					;0x32 '2'
	.db 3,7,1,3,1,7					;0x33 '3'
	.db 3,5,5,7,1,1					;0x34 '4'
	.db 3,7,4,7,1,7					;0x35 '5'
	.db 3,2,4,7,5,7					;0x36 '6'
	.db 3,7,1,2,2,2					;0x37 '7'
	.db 3,7,5,7,5,7					;0x38 '8'
	.db 3,7,5,7,1,2					;0x39 '9'
	.db 1,0,1,0,1,0					;0x3a ':'
	.db 2,0,1,0,1,2					;0x3b ';'
	.db 3,1,2,4,2,1					;0x3c '<'
	.db 3,0,7,0,7,0					;0x3d '='
	.db 3,4,2,1,2,4					;0x3e '>'
	.db 3,7,1,2,0,2					;0x3f '?'
;
; ASCII 0x40 - 0x4f
;
	.db 5,0x0e,0x11,0x17,0x15,0x17			;0x40 '@'
	.db 3,2,5,5,7,5					;0x41 'A'
	.db 3,6,5,6,5,6					;0x42 'B'
	.db 3,3,4,4,4,3					;0x43 'C'
	.db 3,6,5,5,5,6					;0x44 'D'
	.db 3,7,4,6,4,7					;0x45 'E'
	.db 3,7,4,6,4,4					;0x46 'F'
	.db 4,6,8,0x0b,9,6				;0x47 'G'
	.db 3,5,5,7,5,5					;0x48 'H'
	.db 1,1,1,1,1,1					;0x49 'I'
	.db 3,3,1,1,5,2					;0x4a 'J'
	.db 4,9,0x0a,0x0c,0x0a,9			;0x4b 'K'
	.db 3,4,4,4,4,7					;0x4c 'L'
	.db 4,0x0f,0x0f,9,9,9				;0x4d 'M'
	.db 4,9,0x0d,0x0b,9,9				;0x4e 'N'
	.db 3,7,5,5,5,7					;0x4f '0'
;
; ASCII 0x50 - 0x5f
;
	.db 3,7,5,7,4,4					;0x50 'P'
	.db 3,7,5,5,7,7					;0x51 'Q'
	.db 3,7,5,6,5,5					;0x52 'R'
	.db 3,7,4,7,1,7					;0x53 'S'
	.db 3,7,2,2,2,2					;0x54 'T'
	.db 3,5,5,5,5,7					;0x55 'U'
	.db 3,5,5,5,5,2					;0x56 'V'
	.db 4,9,9,9,0x0f,6				;0x57 'W'
	.db 3,5,5,2,5,5					;0x58 'X'
	.db 3,5,5,7,2,2					;0x59 'Y'
	.db 3,7,1,2,4,7					;0x5a 'Z'
	.db 2,3,2,2,2,3					;0x5b '['
	.db 3,4,4,2,1,1					;0x5c '\'
	.db 2,3,1,1,1,3					;0x5d ']'
	.db 3,2,5,0,0,0					;0x5e '^'
	.db 3,0,0,0,0,7					;0x5f '_'
;
; ASCII 0x60 - 0x6f
;
	.db 1,1,0,0,0,0					;0x60 '`'
	.db 3,2,5,5,7,5					;0x61 'a'
	.db 3,6,5,6,5,6					;0x62 'b'
	.db 3,3,4,4,4,3					;0x63 'c'
	.db 3,6,5,5,5,6					;0x64 'd'
	.db 3,7,4,6,4,7					;0x65 'e'
	.db 3,7,4,6,4,4					;0x66 'f'
	.db 4,6,8,0x0b,9,6				;0x67 'g'
	.db 3,5,5,7,5,5					;0x68 'h'
	.db 1,1,1,1,1,1					;0x69 'i'
	.db 3,3,1,1,5,2					;0x6a 'j'
	.db 4,9,0x0a,0x0c,0x0a,9			;0x6b 'k'
	.db 3,4,4,4,4,7					;0x6c 'l'
	.db 4,0x0f,0x0f,9,9,9				;0x6d 'm'
	.db 4,9,0x0d,0x0b,9,9				;0x6e 'n'
	.db 3,7,5,5,5,7					;0x6f 'o'
;
; ASCII 0x70 - 0x7f
;
	.db 3,7,5,7,4,4					;0x70 'p'
	.db 3,7,5,5,7,7					;0x71 'q'
	.db 3,7,5,6,5,5					;0x72 'r'
	.db 3,7,4,7,1,7					;0x73 's'
	.db 3,7,2,2,2,2					;0x74 't'
	.db 3,5,5,5,5,7					;0x75 'u'
	.db 3,5,5,5,5,2					;0x76 'v'
	.db 4,9,9,9,0x0f,6				;0x77 'w'
	.db 3,5,5,2,5,5					;0x78 'x'
	.db 3,5,5,7,2,2					;0x79 'y'
	.db 3,7,1,2,4,7					;0x7a 'z'
	.db 3,3,2,4,2,3					;0x7b '{'
	.db 1,1,1,1,1,1					;0x7c '|'
	.db 3,6,2,1,2,6					;0x7d '}'
	.db 4,5,0x0a,0,0,0				;0x7e '~'
	.db 0,0,0,0,0,0				      ;0x7f Endezeichen
;
; ASCII 0x80 - 0x8f (reserviert fr Umlaute und Sonderzeichen)
;
	.db 3,5,2,5,7,5					;0x80 ''
	.db 3,5,0,7,5,7					;0x81 ''
	.db 3,5,0,5,5,7					;0x82 ''
	.db 3,5,2,5,7,5					;0x83 ''
	.db 3,5,0,7,5,7					;0x84 ''
	.db 3,5,0,5,5,7					;0x85 ''
	.db 3,6,5,7,7,4					;0x86 ''
	.db 0,0,0,0,0,0					;0x87 frei
	.db 0,0,0,0,0,0					;0x88 frei
	.db 0,0,0,0,0,0					;0x89 frei
	.db 0,0,0,0,0,0					;0x8a frei
	.db 0,0,0,0,0,0					;0x8b frei
	.db 0,0,0,0,0,0					;0x8c frei
	.db 0,0,0,0,0,0					;0x8d frei
	.db 0,0,0,0,0,0					;0x8e frei
	.db 0,0,0,0,0,0					;0x8f frei
;

;
; Umlaut-Wandlungs-Tabelle, wird zur Datenbertragung zum PC bentigt,
; alle Umlaute im Code-Bereich 0x80-0x8f werden bei der Datenausgabe
; zum PC durch zwei Zeichen ersetzt:
;
umltab:	.db	"Ae"	;Zeichencode 0x80,  wird zu Ae
	.db	"Oe"	;Zeichencode 0x81,  wird zu Oe
	.db	"Ue"	;Zeichencode 0x82,  wird zu Ue
	.db	"ae"	;Zeichencode 0x83,  wird zu ae
	.db	"oe"	;Zeichencode 0x84,  wird zu oe
	.db	"ue"	;Zeichencode 0x85,  wird zu ue
	.db	"ss"	;Zeichencode 0x86,  wird zu ss
	.db	"  "	;Zeichencode 0x87, frei
	.db	"  "	;Zeichencode 0x88, frei
	.db	"  "	;Zeichencode 0x89, frei
	.db	"  "	;Zeichencode 0x8a, frei
	.db	"  "	;Zeichencode 0x8b, frei
	.db	"  "	;Zeichencode 0x8c, frei
	.db	"  "	;Zeichencode 0x8d, frei
	.db	"  "	;Zeichencode 0x8e, frei
	.db	"  "	;Zeichencode 0x8f, frei
;
;
; Sound-Tabelle, enthlt 16 vordefinierte Tonfolgen (Tonfolge 16 ist
; als Tasten-Quittungston reserviert); jede Folge besteht aus 8 Bytes,
; welche den zeitlichen Verlauf im 20ms-Raster von maximal 4 Tnen
; festlegen; die gesamte Dauer darf 1s (Maximalwert 49) nicht ber-
; schreiten; Byte 0 ist immer 0, Beispiel Tonfolge 13:
;
; Byte 0=  0	Zeitpunkt  0x 20ms =   0ms, Beginn von Ton 1 (immer 0)
; Byte 1=  4	Zeitpunkt  4x 20ms =  80ms, Ende von Ton 1
; Byte 2=  8	Zeitpunkt  8x 20ms = 160ms, Beginn von Ton 2
; Byte 3= 20	Zeitpunkt 20x 20ms = 400ms, Ende von Ton 2
; Byte 4=  0	Beginn von Ton 3 \ hier nicht genutzt, in diesem
; Byte 5=  0	Ende von Ton 3   / Fall beide Werte= 0 setzen
; Byte 6=  0	Beginn von Ton 4 \ hier nicht genutzt, in diesem
; Byte 7=  0	Ende von Ton 4   / Fall beide Werte= 0 setzen
;
sndtab:	.db	0,4,8,12,16,20,24,28	; 1. 4x 80ms/80ms
	.db	0,4,8,12,16,20,0,0	; 2. 3x 80ms/80ms
	.db	0,4,8,12,0,0,0,0	; 3. 2x 80ms/80ms
	.db	0,4,0,0,0,0,0,0		; 4. 1x 80ms
	.db	0,2,4,6,8,10,12,14	; 5. 4x 40ms/40ms
	.db	0,2,4,6,8,10,0,0	; 6. 3x 40ms/40ms
	.db	0,2,4,6,0,0,0,0		; 7. 2x 40ms/40ms
	.db	0,2,0,0,0,0,0,0		; 8. 1x 40ms
	.db	0,2,25,27,0,0,0,0	; 9. 2x 40ms/460ms
	.db	0,25,0,0,0,0,0,0	;10. 1x 500ms/500ms
	.db	0,12,25,37,0,0,0,0	;11. 2x 240ms/260ms
	.db	0,6,12,18,25,31,37,43	;12. 4x	120ms/120ms
	.db	0,4,8,20,0,0,0,0	;13. 1x 80ms/80ms + 240ms
	.db	0,12,16,20,0,0,0,0	;14. 1x 240ms/80ms + 80ms
	.db	0,8,17,25,34,42,0,0	;15. 3x 160ms/180ms
	.db	0,3,0,0,0,0,0,0		;16. 1x 60ms (reserviert)
;
;
; Kommando-Liste fr die PC-Datenkommunikation, jedes Kommando darf
; maximal 8 Zeichen lang sein (der Rest wird mit Leerzeichen aufge-
; fllt), das Tabellenende wird mit 0xff gekennzeichnet
;
pccmdl:	.db	"end     "	;Kommando 1 (Ende/Abbruch)
	.db	"info    "	;Kommando 2 (Infozeile senden)
	.db	"gett15  "	;Kommando 3 (15min-Werte senden)
	.db	"getmin  "	;Kommando 4 (Minimal-Werte senden)
	.db	"getmax  "	;Kommando 5 (Maximal-Werte senden)
	.db	"getavg  "	;Kommando 6 (Mittel-Werte senden)
	.db	"getmsg  "	;Kommando 7 (Meldungsliste senden)
	.db	"getcfg  "	;Kommando 8 (Konfiguration senden)
	.db	0xff,0xff	;Listenende
;
;
; Various strings - English
;
.include "AnzModul2-eng.asm"
;
;
; Diverse Texte - Deutsch
;
.include "AnzModul2-deu.asm"
;
;
; Mentabellen, enthalten die komplette Menstruktur in einer Index-
; Tabelle und weiteren Tabellen fr jedes Untermen
;
; Die Index-Tabelle enthlt die Speicheradressen der einzelnen Men-
; tabellen
;
menidx:	.dw	menu00		;Men 00, Hauptmen
	.dw	menu01		;Men 01, Min-Werte
	.dw	menu02		;Men 02, Max-Werte
	.dw	menu03		;Men 03, Meldungen
	.dw	menu04		;Men 04, Status
	.dw	menu05		;Men 05, Konfiguration
	.dw	menu06		;Men 06, Konfig./Temperaturen
	.dw	menu07		;Men 07, Konfig./Temp./Positionen
	.dw	menu08		;Men 08, Konfig./Temp./Grenzwerte
	.dw	menu09		;Men 09, Konfig./Temp./Unt. Grenzwerte
	.dw	menu10		;Men 10, Konfig./Temp./Ob. Grenzwerte
	.dw	menu11		;Men 11, Konfiguration/Alarme
	.dw	menu12		;Men 12, Konfiguration/Signale
	.dw	menu13		;Men 13, Konfiguration/Sig./Zuordnung
	.dw	menu14		;Men 14, Konfiguration/Signale/Audio
	.dw	menu15		;Men 15, Konfiguration/Zeiten
	.dw	menu16		;Men 16, Konfiguration/Zeiten/Schaltz.
	.dw	menu17		;Men 17, Konfiguration/Zeiten/Uhr st.
	.dw	menu18		;Men 18, Konfiguration/Sonstiges
	.dw	menu19		;Men 19, Konfiguration/Backup
	.dw	menu20		;Men 20, Konfiguration/Grundeinstell.
	.dw	menu21		;Men 21, Konfig./Grund./Konfig zurcks
	.dw	menu22		;Men 22, Konfig./Grund./15min Werte l
	.dw	menu23		;Men 23, Konfig./Grund./24h Werte ls.
	.dw	menu24		;Men 24, Konfig./Grund./Meldungen ls.
	.dw	menu25		;Men 25, Konfig./Temp./Log-Zuordnung
	.dw	menu26		;Men 26, Grafik
	.dw	menu27		;Men 27, Grafik/Preset-01
	.dw	menu28		;Men 28, Grafik/Preset-02
	.dw	menu29		;Men 29, Grafik/Preset-03
	.dw	menu30		;Men 30, Grafik/Preset-04
	.dw	menu31		;Men 31, Grafik/Preset-05
	.dw	menu32		;Men 32, Grafik/Preset-06
	.dw	menu33		;Men 33, Grafik/Preset-07
	.dw	menu34		;Men 34, Grafik/Preset-08
	.dw	menu35		;Men 35, Grafik/Preset-09
	.dw	menu36		;Men 36, Grafik/Preset-10
	.dw	menu37		;Men 37, Grafik/Preset-11
	.dw	menu38		;Men 38, Grafik/Preset-12
	.dw	menu39		;Men 39, Grafik/Preset-13
	.dw	menu40		;Men 40, Grafik/Preset-14
	.dw	menu41		;Men 41, Grafik/Preset-15
	.dw	menu42		;Men 42, Grafik/Preset-16
	.dw	menu43		;Men 43, Grafik/Preset-17
	.dw	menu44		;Men 44, Grafik/Preset-18
	.dw	menu45		;Men 45, Grafik/Preset-19
	.dw	menu46		;Men 46, Grafik/Preset-20
	.dw	menu47		;Men 47, Grafik/Preset-21
	.dw	menu48		;Men 48, Grafik/Preset-22
	.dw	menu49		;Men 49, Grafik/Preset-23
	.dw	menu50		;Men 50, Grafik/Preset-24
	.dw	menu51		;Men 51, Grafik/Preset-01/Fadenkreuz
	.dw	menu52		;Men 52, Grafik/Preset-02/Fadenkreuz
	.dw	menu53		;Men 53, Grafik/Preset-03/Fadenkreuz
	.dw	menu54		;Men 54, Grafik/Preset-04/Fadenkreuz
	.dw	menu55		;Men 55, Grafik/Preset-05/Fadenkreuz
	.dw	menu56		;Men 56, Grafik/Preset-06/Fadenkreuz
	.dw	menu57		;Men 57, Grafik/Preset-07/Fadenkreuz
	.dw	menu58		;Men 58, Grafik/Preset-08/Fadenkreuz
	.dw	menu59		;Men 59, Grafik/Preset-09/Fadenkreuz
	.dw	menu60		;Men 60, Grafik/Preset-10/Fadenkreuz
	.dw	menu61		;Men 61, Grafik/Preset-11/Fadenkreuz
	.dw	menu62		;Men 62, Grafik/Preset-12/Fadenkreuz
	.dw	menu63		;Men 63, Grafik/Preset-13/Fadenkreuz
	.dw	menu64		;Men 64, Grafik/Preset-14/Fadenkreuz
	.dw	menu65		;Men 65, Grafik/Preset-15/Fadenkreuz
	.dw	menu66		;Men 66, Grafik/Preset-16/Fadenkreuz
	.dw	menu67		;Men 67, Grafik/Preset-17/Fadenkreuz
	.dw	menu68		;Men 68, Grafik/Preset-18/Fadenkreuz
	.dw	menu69		;Men 69, Grafik/Preset-19/Fadenkreuz
	.dw	menu70		;Men 70, Grafik/Preset-20/Fadenkreuz
	.dw	menu71		;Men 71, Grafik/Preset-21/Fadenkreuz
	.dw	menu72		;Men 72, Grafik/Preset-22/Fadenkreuz
	.dw	menu73		;Men 73, Grafik/Preset-23/Fadenkreuz
	.dw	menu74		;Men 74, Grafik/Preset-24/Fadenkreuz
	.dw	menu75		;Men 75, Konfiguration/Grafik
	.dw	menu76		;Men 76, Konfig./Grund./Grafik zurcks
;
; Eine Mentabelle besteht aus 4 Startbytes und jeweils 4 Bytes fr
; jedes Untermen:
;
; Byte 0= Anzahl der Untermens
; Byte 1= Nummer der Zeichenkette fr die Infoanzeige
; Byte 2= Nummer des bergeordneten Mens
; Byte 3= Nummer der bergeordneten Menoption,
;	  Bit7: 1= Flag fr Neuausgabe der Temperaturanzeige
; Byte 4= Nummer der Zeichenkette fr die Menanzeige \
; Byte 5= Nummer des Untermens oder 0xff	       \ Daten fr
; Byte 6= \ Startadresse fr das Menprogramm	       / Menoption 1
; Byte 7= / oder 0 (wenn keine Aktion)		      /
; Byte 8= Nummer der Zeichenkette fr die Menanzeige \
; Byte 9= Nummer des Untermens oder 0xff	       \ Daten fr
; Byte10= \ Startadresse fr das Menprogramm	       / Menoption 2
; Byte11= / oder 0 (wenn keine Aktion)		      /
; Byte12= weitere Optionen...
;
; Men 00 - Hauptmen
;
menu00:	.db	6,0		;6 Untermens, keine Infoanzeige
	.db	0,0		;kein bergeordnetes Men/Menoption
	.db	0x20,1		;Option 1: "Min-Werte", Men 1
	.dw	m80000		;	   Programmadresse
	.db	0x21,2		;Option 2: "Max-Werte", Men 2
	.dw	m80200		;	   Programmadresse
	.db	0x23,26		;Option 3: "Grafik", Men 26
	.dw	0		;	   keine Aktion durchfhren
	.db	0x22,3		;Option 4: "Meldungen", Men 3
	.dw	m81000		;	   Programmadresse
	.db	0x24,4		;Option 5: "Status", Men 4
	.dw	m82000		;	   Programmadresse
	.db	0x25,5		;Option 6: "Konfiguration", Men 5
	.dw	0		;	   keine Aktion durchfhren
;
; Men 01 - Min-Werte
;
menu01:	.db	1,0x20		;1 Untermen, Infotext "Min-Werte"
	.db	0,1|0x80	;bergeordnetes Men 0, Option 1
	.db	0x26,0xff	;Option 1: "Werte lschen", k.Untermen
	.dw	m80100		;	   Programmadresse
;
; Men 02 - Max-Werte
;
menu02:	.db	1,0x21		;1 Untermen, Infotext "Max-Werte"
	.db	0,2|0x80	;bergeordnetes Men 0, Option 2
	.db	0x26,0xff	;Option 1: "Werte lschen"
	.dw	m80300		;	   Programmadresse
;
; Men 03 - Meldungen
;
menu03:	.db	3,0x22		;3 Untermens, Infotext "Meldungen"
	.db	0,4|0x80	;bergeordnetes Men 0, Option 4
	.db	0x27,0xff	;Option 1: "nchste Seite"
	.dw	m81100		;	   Programmadresse
	.db	0x28,0xff	;Option 2: "vorher. Seite"
	.dw	m81200		;	   Programmadresse
	.db	0x29,0xff	;Option 3: "Filter"
	.dw	m81300		;	   Programmadresse
;
; Men 04 - Status
;
menu04:	.db	5,0x24		;5 Untermens, Infotext "Status"
	.db	0,5|0x80	;bergeordnetes Men 0, Option 5
	.db	0x3b,0xff	;Option 1: "Normal/Abwesend"
	.dw	m82400		;	   Programmadresse
	.db	0x3c,0xff	;Option 2: "LED umschalten"
	.dw	m82500		;	   Programmadresse
	.db	0x3d,0xff	;Option 3: "Speicherfehler lschen"
	.dw	m82600		;	   Programmadresse
	.db	0x3e,0xff	;Option 4: "RTC-Fehler lschen"
	.dw	m82700		;	   Programmadresse
	.db	0x3f,0xff	;Option 5: "Pointer-Fehler lschen"
	.dw	m82800		;	   Programmadresse
;
; Men 05 - Konfiguration
;
menu05:	.db	8,0x25		;8 Untermens, Infotext "Konfiguration"
	.db	0,6		;bergeordnetes Men 0, Option 6
	.db	0x50,6		;Option 1: "Temperaturen", Men 6
	.dw	0		;	   keine Aktion durchfhren
	.db	0x51,11		;Option 2: "Alarme", Men 11
	.dw	m85000		;	   Programmadresse
	.db	0x52,12		;Option 3: "Signale", Men 12
	.dw	0		;	   keine Aktion durchfhren
	.db	0x53,15		;Option 4: "Zeiten", Men 15
	.dw	0		;	   Programmadresse
	.db	0x54,18		;Option 5: "Sonstiges", Men 18
	.dw	m88500		;	   keine Aktion durchfhren
	.db	0xaf,75		;Option 6: "Grafik-Presets", Men 75
	.dw	m8b000		;	   Programmadresse
	.db	0x55,19		;Option 7: "Backup", Men 19
	.dw	0		;	   keine Aktion durchfhren
	.db	0x56,20		;Option 8: "Grundeinstellung", Men 20
	.dw	0		;	   keine Aktion durchfhren
;
; Men 06 - Konfiguration - Temperaturen
;
menu06:	.db	3,0x50		;3 Untermens, Infotext "Temperaturen"
	.db	5,1		;bergeordnetes Men 5, Option 1
	.db	0x57,7		;Option 1: "Sensoren"
	.dw	m83000		;	   Programmadresse
	.db	0x5f,8		;Option 2: "Grenzwerte"
	.dw	m84000		;	   Programmadresse
	.db	0x97,25		;Option 3: "Log-Zuordnung"
	.dw	m84600		;	   Programmadresse
;
; Men 07 - Konfiguration - Temperaturen - Positionen
;
menu07:	.db	5,0x57		;5 Untermens, Infotext "Sensoren"
	.db	6,1|0x80	;bergeordnetes Men 6, Option 1
	.db	0x5a,0xff	;Option 1: "Auswhlen"
	.dw	m83100		;	   Programmadresse
	.db	0x5b,0xff	;Option 2: "Ein/Ausschalten"
	.dw	m83200		;	   Programmadresse
	.db	0x5c,0xff	;Option 3: "Namen"
	.dw	m83300		;	   Programmadresse
	.db	0x5d,0xff	;Option 4: "Verschieben"
	.dw	m83400		;	   Programmadresse
	.db	0x5e,0xff	;Option 5: "Groanzeige"
	.dw	m83500		;	   Programmadresse
;
; Men 08 - Konfiguration - Temperaturen - Grenzwerte
;
menu08:	.db	3,0x5f		;3 Untermens, Infotext "Grenzwerte"
	.db	6,2|0x80	;bergeordnetes Men 6, Option 2
	.db	0x5a,0xff	;Option 1: "Auswhlen"
	.dw	m84100		;	   Programmadresse
	.db	0x58,9		;Option 2: "Unterer Grenzwert", Men 9
	.dw	0		;	   keine Aktion durchfhren
	.db	0x59,10		;Option 3: "Oberer Grenzwert", Men 10
	.dw	0		;	   keine Aktion durchfhren
;
; Men 09 - Konfiguration - Temperaturen - Grenzwerte - Unterer
;
menu09:	.db	2,0x58		;2 Untermens, Infotext "Unterer Gren."
	.db	8,2		;bergeordnetes Men 8, Option 2
	.db	0x60,0xff	;Option 1: "ndern"
	.dw	m84200		;	   Programmadresse
	.db	0x61,0xff	;Option 2: "Lschen"
	.dw	m84300		;	   Programmadresse
;
; Men 10 - Konfiguration - Temperaturen - Grenzwerte - Oberer
;
menu10:	.db	2,0x59		;2 Untermens, Infotext "Oberer Grenz."
	.db	8,3		;bergeordnetes Men 8, Option 3
	.db	0x60,0xff	;Option 1: "ndern"
	.dw	m84400		;	   Programmadresse
	.db	0x61,0xff	;Option 2: "Lschen"
	.dw	m84500		;	   Programmadresse
;
; Men 11 - Konfiguration - Alarme
;
menu11:	.db	5,0x51		;5 Untermens, Infotext "Alarme"
	.db	5,2|0x80	;bergeordnetes Men 5, Option 2
	.db	0x5a,0xff	;Option 1: "Auswhlen"
	.dw	m85100		;	   Programmadresse
	.db	0x5b,0xff	;Option 2: "Ein/Ausschalten"
	.dw	m85200		;	   Programmadresse
	.db	0x5c,0xff	;Option 3: "Namen"
	.dw	m85300		;	   Programmadresse
	.db	0x64,0xff	;Option 4: "Invertieren"
	.dw	m85400		;	   Programmadresse
	.db	0x5d,0xff	;Option 5: "Verschieben"
	.dw	m85500		;	   Programmadresse
;
; Men 12 - Konfiguration - Signale
;
menu12:	.db	2,0x52		;2 Untermens, Infotext "Signale"
	.db	5,3		;bergeordnetes Men 5, Option 3
	.db	0x6a,13		;Option 1: "Zuordnung", Men 13
	.dw	m86000		;	   Programmadresse
	.db	0x6b,14		;Option 2: "Audio", Men 14
	.dw	m87000		;	   Programmadresse
;
; Men 13 - Konfiguration - Signale - Zuordnung
;
menu13:	.db	4,0x6a		;4 Untermens, Infotext "Zuordnung"
	.db	12,1|0x80	;bergeordnetes Men 12, Option 1
	.db	0x5a,0xff	;Option 1: "Auswhlen"
	.dw	m86100		;	   Programmadresse
	.db	0x6c,0xff	;Option 2: "Nummer"
	.dw	m86200		;	   Programmadresse
	.db	0x6d,0xff	;Option 3: "Dauer"
	.dw	m86300		;	   Programmadresse
	.db	0x6e,0xff	;Option 3: "Abspielen"
	.dw	m86400		;	   Programmadresse
;
; Men 14 - Konfiguration - Signale - Audio
;
menu14:	.db	3,0x6b		;3 Untermens, Infotext "Audio"
	.db	12,2|0x80	;bergeordnetes Men 12, Option 2
	.db	0x5a,0xff	;Option 1: "Auswhlen"
	.dw	m87100		;	   Programmadresse
	.db	0x60,0xff	;Option 2: "ndern"
	.dw	m87200		;	   Programmadresse
	.db	0x6e,0xff	;Option 3: "Abspielen"
	.dw	m87300		;	   Programmadresse
;
; Men 15 - Konfiguration - Zeiten
;
menu15:	.db	2,0x53		;2 Untermens, Infotext "Zeiten"
	.db	5,4		;bergeordnetes Men 5, Option 4
	.db	0x72,16		;Option 1: "Schaltzeiten", Men 16
	.dw	m87500		;	   Programmadresse
	.db	0x73,17		;Option 1: "Uhr stellen", Men 17
	.dw	m88000		;	   Programmadresse
;
; Men 16 - Konfiguration - Zeiten - Schaltzeiten
;
menu16:	.db	3,0x72		;3 Untermens, Infotext "Schaltzeiten"
	.db	15,1|0x80	;bergeordnetes Men 15, Option 1
	.db	0x5a,0xff	;Option 1: "Auswhlen"
	.dw	m87600		;	   Programmadresse
	.db	0x5b,0xff	;Option 2: "Ein/Ausschalten"
	.dw	m87700		;	   Programmadresse
	.db	0x60,0xff	;Option 2: "ndern"
	.dw	m87800		;	   Programmadresse
;
; Men 17 - Konfiguration - Zeiten - Uhr stellen
;
menu17:	.db	2,0x73		;2 Untermens, Infotext "Uhr stellen"
	.db	15,2|0x80	;bergeordnetes Men 15, Option 2
	.db	0x5a,0xff	;Option 1: "Auswhlen"
	.dw	m88100		;	   Programmadresse
	.db	0x60,0xff	;Option 2: "ndern"
	.dw	m88200		;	   Programmadresse
;
; Men 18 - Konfiguration - Sonstiges
;
menu18:	.db	2,0x54		;2 Untermens, Infotext "Sonstiges"
	.db	5,5|0x80	;bergeordnetes Men 5, Option 5
	.db	0x5a,0xff	;Option 1: "Auswhlen"
	.dw	m88600		;	   Programmadresse
	.db	0x60,0xff	;Option 2: "ndern"
	.dw	m88700		;	   Programmadresse
;
; Men 19 - Konfiguration - Backup
;
menu19:	.db	2,0x55		;2 Untermens, Infotext "Backup"
	.db	5,7		;bergeordnetes Men 5, Option 7
	.db	0x90,0xff	;Option 1: "Sichern"
	.dw	m89000		;	   Programmadresse
	.db	0x91,0xff	;Option 2: "Laden/Neustart"
	.dw	m89100		;	   Programmadresse
;
; Men 20 - Konfiguration - Grundeinstellung
;
menu20:	.db	5,0x56		;4 Untermens, Infotext "Grundeinstell"
	.db	5,8		;bergeordnetes Men 5, Option 8
	.db	0x92,21		;Option 1: "Basiskonfigur. laden", M.21
	.dw	0		;	   keine Aktion durchfhren
	.db	0x9a,76		;Option 2: "Grafik-Presets rck.", M.76
	.dw	0		;	   keine Aktion durchfhren
	.db	0x93,22		;Option 3: "15min Werte lschen", M.22
	.dw	0		;	   keine Aktion durchfhren
	.db	0x94,23		;Option 4: "24h Werte lschen", Men 23
	.dw	0		;	   keine Aktion durchfhren
	.db	0x95,24		;Option 5: "Meldungen lschen", Men 24
	.dw	0		;	   keine Aktion durchfhren
;
; Men 21 - Konfiguration - Grundeinstellung - Basiskonfiguration laden
;
menu21:	.db	1,0x92		;1 Untermen, Infotext "Basiskonfigur."
	.db	20,1		;bergeordnetes Men 20, Option 1
	.db	0x96,0xff	;Option 1: "Sicher?"
	.dw	m89500		;	   Programmadresse
;
; Men 22 - Konfiguration - Grundeinstellung - 15min Werte lschen
;
menu22:	.db	1,0x93		;1 Untermen, Infotext "15min Werte l"
	.db	20,3		;bergeordnetes Men 20, Option 3
	.db	0x96,0xff	;Option 1: "Sicher?"
	.dw	m89600		;	   Programmadresse
;
; Men 23 - Konfiguration - Grundeinstellung - 24h Werte lschen
;
menu23:	.db	1,0x94		;1 Untermen, Infotext "24h Werte ls."
	.db	20,4		;bergeordnetes Men 20, Option 4
	.db	0x96,0xff	;Option 1: "Sicher?"
	.dw	m89700		;	   Programmadresse
;
; Men 24 - Konfiguration - Grundeinstellung - Meldungen lschen
;
menu24:	.db	1,0x95		;1 Untermen, Infotext "Meldungen ls."
	.db	20,5		;bergeordnetes Men 20, Option 5
	.db	0x96,0xff	;Option 1: "Sicher?"
	.dw	m89800		;	   Programmadresse
;
; Men 25 - Konfiguration - Temperaturen - Log-Positionen
;
menu25:	.db	2,0x97		;2 Untermens, Infotext "Log-Zuordnung"
	.db	6,3|0x80	;bergeordnetes Men 6, Option 3
	.db	0x5a,0xff	;Option 1: "Auswhlen"
	.dw	m84700		;	   Programmadresse
	.db	0x5d,0xff	;Option 2: "Verschieben"
	.dw	m84800		;	   Programmadresse
;
; Men 26 - Grafik
;
menu26:	.db	24,0x23		;24 Untermens, Infotext "Grafik"
	.db	0,3		;bergeordnetes Men 0, Option 3
	.db	10,27		;Option 01: "Preset 01", Men 27
	.dw	m8a000		;	    Programmadresse
	.db	10,28		;Option 02: "Preset 02", Men 28
	.dw	m8a010		;	    Programmadresse
	.db	10,29		;Option 03: "Preset 03", Men 29
	.dw	m8a020		;	    Programmadresse
	.db	10,30		;Option 04: "Preset 04", Men 30
	.dw	m8a030		;	    Programmadresse
	.db	10,31		;Option 05: "Preset 05", Men 31
	.dw	m8a040		;	    Programmadresse
	.db	10,32		;Option 06: "Preset 06", Men 32
	.dw	m8a050		;	    Programmadresse
	.db	10,33		;Option 07: "Preset 07", Men 33
	.dw	m8a060		;	    Programmadresse
	.db	10,34		;Option 08: "Preset 08", Men 34
	.dw	m8a070		;	    Programmadresse
	.db	10,35		;Option 09: "Preset 09", Men 35
	.dw	m8a080		;	    Programmadresse
	.db	10,36		;Option 10: "Preset 10", Men 36
	.dw	m8a090		;	    Programmadresse
	.db	10,37		;Option 11: "Preset 11", Men 37
	.dw	m8a100		;	    Programmadresse
	.db	10,38		;Option 12: "Preset 12", Men 38
	.dw	m8a110		;	    Programmadresse
	.db	10,39		;Option 13: "Preset 13", Men 39
	.dw	m8a120		;	    Programmadresse
	.db	10,40		;Option 14: "Preset 14", Men 40
	.dw	m8a130		;	    Programmadresse
	.db	10,41		;Option 15: "Preset 15", Men 41
	.dw	m8a140		;	    Programmadresse
	.db	10,42		;Option 16: "Preset 16", Men 42
	.dw	m8a150		;	    Programmadresse
	.db	10,43		;Option 17: "Preset 17", Men 43
	.dw	m8a160		;	    Programmadresse
	.db	10,44		;Option 18: "Preset 18", Men 44
	.dw	m8a170		;	    Programmadresse
	.db	10,45		;Option 19: "Preset 19", Men 45
	.dw	m8a180		;	    Programmadresse
	.db	10,46		;Option 20: "Preset 20", Men 46
	.dw	m8a190		;	    Programmadresse
	.db	10,47		;Option 21: "Preset 21", Men 47
	.dw	m8a200		;	    Programmadresse
	.db	10,48		;Option 22: "Preset 22", Men 48
	.dw	m8a210		;	    Programmadresse
	.db	10,49		;Option 23: "Preset 23", Men 49
	.dw	m8a220		;	    Programmadresse
	.db	10,50		;Option 24: "Preset 24", Men 50
	.dw	m8a230		;	    Programmadresse
;
; Men 27 - Grafik - Preset-01
;
menu27:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,1|0x80	;bergeordnetes Men 26, Option 1
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,51		;Option 3: "Fadenkreuz", Men 51
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 28 - Grafik - Preset-02
;
menu28:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,2|0x80	;bergeordnetes Men 26, Option 2
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,52		;Option 3: "Fadenkreuz", Men 52
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 29 - Grafik - Preset-03
;
menu29:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,3|0x80	;bergeordnetes Men 26, Option 3
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,53		;Option 3: "Fadenkreuz", Men 53
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 30 - Grafik - Preset-04
;
menu30:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,4|0x80	;bergeordnetes Men 26, Option 4
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,54		;Option 3: "Fadenkreuz", Men 54
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 31 - Grafik - Preset-05
;
menu31:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,5|0x80	;bergeordnetes Men 26, Option 5
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,55		;Option 3: "Fadenkreuz", Men 55
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 32 - Grafik - Preset-06
;
menu32:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,6|0x80	;bergeordnetes Men 26, Option 6
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,56		;Option 3: "Fadenkreuz", Men 56
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 33 - Grafik - Preset-07
;
menu33:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,7|0x80	;bergeordnetes Men 26, Option 7
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,57		;Option 3: "Fadenkreuz", Men 57
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 34 - Grafik - Preset-08
;
menu34:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,8|0x80	;bergeordnetes Men 26, Option 8
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,58		;Option 3: "Fadenkreuz", Men 58
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 35 - Grafik - Preset-09
;
menu35:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,9|0x80	;bergeordnetes Men 26, Option 9
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,59		;Option 3: "Fadenkreuz", Men 59
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 36 - Grafik - Preset-10
;
menu36:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,10|0x80	;bergeordnetes Men 26, Option 10
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,60		;Option 3: "Fadenkreuz", Men 60
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 37 - Grafik - Preset-11
;
menu37:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,11|0x80	;bergeordnetes Men 26, Option 11
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,61		;Option 3: "Fadenkreuz", Men 61
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 38 - Grafik - Preset-12
;
menu38:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,12|0x80	;bergeordnetes Men 26, Option 12
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,62		;Option 3: "Fadenkreuz", Men 62
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 39 - Grafik - Preset-13
;
menu39:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,13|0x80	;bergeordnetes Men 26, Option 13
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,63		;Option 3: "Fadenkreuz", Men 63
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 40 - Grafik - Preset-14
;
menu40:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,14|0x80	;bergeordnetes Men 26, Option 14
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,64		;Option 3: "Fadenkreuz", Men 64
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 41 - Grafik - Preset-15
;
menu41:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,15|0x80	;bergeordnetes Men 26, Option 15
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,65		;Option 3: "Fadenkreuz", Men 65
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 42 - Grafik - Preset-16
;
menu42:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,16|0x80	;bergeordnetes Men 26, Option 16
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,66		;Option 3: "Fadenkreuz", Men 66
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 43 - Grafik - Preset-17
;
menu43:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,17|0x80	;bergeordnetes Men 26, Option 17
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,67		;Option 3: "Fadenkreuz", Men 67
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 44 - Grafik - Preset-18
;
menu44:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,18|0x80	;bergeordnetes Men 26, Option 18
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,68		;Option 3: "Fadenkreuz", Men 68
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 45 - Grafik - Preset-19
;
menu45:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,19|0x80	;bergeordnetes Men 26, Option 19
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,69		;Option 3: "Fadenkreuz", Men 69
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 46 - Grafik - Preset-20
;
menu46:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,20|0x80	;bergeordnetes Men 26, Option 20
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,70		;Option 3: "Fadenkreuz", Men 70
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 47 - Grafik - Preset-21
;
menu47:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,21|0x80	;bergeordnetes Men 26, Option 21
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,71		;Option 3: "Fadenkreuz", Men 71
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 48 - Grafik - Preset-22
;
menu48:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,22|0x80	;bergeordnetes Men 26, Option 22
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,72		;Option 3: "Fadenkreuz", Men 72
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 49 - Grafik - Preset-23
;
menu49:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,23|0x80	;bergeordnetes Men 26, Option 23
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,73		;Option 3: "Fadenkreuz", Men 73
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 50 - Grafik - Preset-24
;
menu50:	.db	5,10		;5 Untermens, Infotext "Preset"
	.db	26,24|0x80	;bergeordnetes Men 26, Option 24
	.db	0x28,0xff	;Option 1: "Vorherige Seite"
	.dw	m8a300		;	   Programmadresse
	.db	0x27,0xff	;Option 2: Nchste Seite"
	.dw	m8a400		;	   Programmadresse
	.db	0xa1,74		;Option 3: "Fadenkreuz", Men 74
	.dw	m8a500		;	   Programmadresse
	.db	0xa2,0xff	;Option 4: "Sensornamen anzeigen"
	.dw	m8a600		;	   Programmadresse
	.db	0xa3,0xff	;Option 5: "Temperaturskala angleichen"
	.dw	m8a700		;	   Programmadresse
;
; Men 51 - Grafik - Preset-01 - Fadenkreuz
;
menu51:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	27,3		;bergeordnetes Men 27, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 52 - Grafik - Preset-02 - Fadenkreuz
;
menu52:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	28,3		;bergeordnetes Men 28, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 53 - Grafik - Preset-03 - Fadenkreuz
;
menu53:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	29,3		;bergeordnetes Men 29, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 44 - Grafik - Preset-04 - Fadenkreuz
;
menu54:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	30,3		;bergeordnetes Men 30, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 55 - Grafik - Preset-05 - Fadenkreuz
;
menu55:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	31,3		;bergeordnetes Men 31, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 56 - Grafik - Preset-06 - Fadenkreuz
;
menu56:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	32,3		;bergeordnetes Men 32, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 57 - Grafik - Preset-07 - Fadenkreuz
;
menu57:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	33,3		;bergeordnetes Men 33, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 58 - Grafik - Preset-08 - Fadenkreuz
;
menu58:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	34,3		;bergeordnetes Men 34, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 59 - Grafik - Preset-09 - Fadenkreuz
;
menu59:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	35,3		;bergeordnetes Men 35, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 60 - Grafik - Preset-10 - Fadenkreuz
;
menu60:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	36,3		;bergeordnetes Men 36, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 61 - Grafik - Preset-11 - Fadenkreuz
;
menu61:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	37,3		;bergeordnetes Men 37, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 62 - Grafik - Preset-12 - Fadenkreuz
;
menu62:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	38,3		;bergeordnetes Men 38, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 63 - Grafik - Preset-13 - Fadenkreuz
;
menu63:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	39,3		;bergeordnetes Men 39, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 64 - Grafik - Preset-14 - Fadenkreuz
;
menu64:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	40,3		;bergeordnetes Men 40, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 65 - Grafik - Preset-15 - Fadenkreuz
;
menu65:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	41,3		;bergeordnetes Men 41, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 66 - Grafik - Preset-16 - Fadenkreuz
;
menu66:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	42,3		;bergeordnetes Men 42, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 67 - Grafik - Preset-17 - Fadenkreuz
;
menu67:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	43,3		;bergeordnetes Men 43, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 68 - Grafik - Preset-18 - Fadenkreuz
;
menu68:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	44,3		;bergeordnetes Men 44, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 69 - Grafik - Preset-19 - Fadenkreuz
;
menu69:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	45,3		;bergeordnetes Men 45, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 70 - Grafik - Preset-20 - Fadenkreuz
;
menu70:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	46,3		;bergeordnetes Men 46, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 71 - Grafik - Preset-21 - Fadenkreuz
;
menu71:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	47,3		;bergeordnetes Men 47, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 72 - Grafik - Preset-22 - Fadenkreuz
;
menu72:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	48,3		;bergeordnetes Men 48, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 73 - Grafik - Preset-23 - Fadenkreuz
;
menu73:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	49,3		;bergeordnetes Men 49, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 74 - Grafik - Preset-24 - Fadenkreuz
;
menu74:	.db	1,10		;1 Untermen, Infotext "Preset"
	.db	50,3		;bergeordnetes Men 50, Option 3
	.db	11,0xff		;Option 1: Datenanzeige
	.dw	0		;	   keine Aktion durchfhren
;
; Men 75 - Konfiguration - Grafik-Presets
;
menu75:	.db	2,0xaf		;2 Untermens,Infotext "Grafik-Presets"
	.db	5,6|0x80	;bergeordnetes Men 5, Option 6
	.db	0x5a,0xff	;Option 1: "Auswhlen"
	.dw	m8b100		;	   Programmadresse
	.db	0x60,0xff	;Option 2: "ndern"
	.dw	m8b200		;	   Programmadresse
;
; Men 76 - Konfiguration - Grundeinstellung - Grafik-Presets rcksetz.
;
menu76:	.db	1,0x9a		;1 Untermen, Infotext "Grafik rcks."
	.db	20,2		;bergeordnetes Men 20, Option 2
	.db	0x96,0xff	;Option 1: "Sicher?"
	.dw	m8c000		;	   Programmadresse
;
;
; -------------------------------------------------------------
; Default-Konfiguration, wird bei Bedarf ins EEPROM geschrieben
; -------------------------------------------------------------
;
; Temperatur Mapping Tabelle: Hier werden die Anzeigepositionen der
; 16 Temperatursensoren festgelegt (0-15), wenn Bit7=1: Temperatursen-
; sor inaktiv
;
dtemap:	.db	0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15
;
; Temperaturtexte: werden an der von der Mapping Tabelle festgelegten
; Position auf dem LCD angezeigt, Lnge=9 Zeichen
;
dtemtx:	.db	"S-01     ","S-02     "	;Text fr Sensor 01/02 (1/2)
	.db	"S-03     ","S-04     "	;Text fr Sensor 03/04 (3/4)
	.db	"S-05     ","S-06     "	;Text fr Sensor 05/06 (5/6)
	.db	"S-07     ","S-08     "	;Text fr Sensor 07/08 (7/8)
	.db	"S-09     ","S-10     "	;Text fr Sensor 09/10 (a/b)
	.db	"S-11     ","S-12     "	;Text fr Sensor 11/12 (c/d)
	.db	"S-13     ","S-14     "	;Text fr Sensor 13/14 (e/f)
	.db	"S-15     ","S-16     "	;Text fr Sensor 15/16 (g/h)
;
; Obere Temperaturlimits in Zehntel Grad (16-Bit Hex),
; 0x8000=Limitberwachung ausgeschaltet
;
dtelih:	.dw	0x8000,0x8000,0x8000,0x8000
	.dw	0x8000,0x8000,0x8000,0x8000
	.dw	0x8000,0x8000,0x8000,0x8000
	.dw	0x8000,0x8000,0x8000,0x8000
;
; Untere Temperaturlimits in Zehntel Grad (16-Bit Hex),
; 0x8000=Limitberwachung ausgeschaltet
;
dtelil:	.dw	0x8000,0x8000,0x8000,0x8000
	.dw	0x8000,0x8000,0x8000,0x8000
	.dw	0x8000,0x8000,0x8000,0x8000
	.dw	0x8000,0x8000,0x8000,0x8000
;
; Alarm Mapping Tabelle: hier werden die Anzeigepositionen der Alarm-
; daten festgelegt (0-3), wenn Bit7=1: Alarm nicht genutzt
;
dalmap:	.db	0,1,2,3
;
; Alarm Invertierungs Mapping Tabelle: hier wird festgelegt, ob der
; entsprechende Alarm invertiert ausgewertet werden soll
; 0=normal, 1=invertiert
;
dalinv:	.db	0,0,0,0
;
; Alarmtexte: werden an der von der Mapping Tabelle festgelegten
; Position auf dem LCD angezeigt, Lnge=11 Zeichen
;
dalmtx:	.db	"Al-1       ",0		;Alarmeingang 1
	.db	"Al-2       ",0		;Alarmeingang 2
	.db	"Al-3       ",0		;Alarmeingang 3
	.db	"Al-4       ",0		;Alarmeingang 4
;
; Nummer des Sensors, der als Groanzeige erscheint
;
dbigtm:	.db	0,0	;logische Sensornummer (0-15)
;
; Signaltabelle, enthlt die Nummer des akustischen Signals (0-15) fr
; 8 verschiedene Ereignisse
;
dsignr:	.db	0,1	;Signal fr Alarmeingang 1 / Alarmeingang 2
	.db	2,3	;Signal fr Alarmeingang 3 / Alarmeingang 4
	.db	12,8	;Signal fr Temp-berschreit/Temp-Unterschreit.
	.db	9,10	;Signal fr Fehlermeldungen / Sensorausflle
;
; Signaltabelle, enthlt die Dauer des akustischen Signals in Sekunden
; fr 8 verschiedene Ereignisse (1-30s), 0= akustisches Signal aus
;
dsigdu:	.db	4,4	;Dauer fr Alarmeingang 1 / Alarmeingang 2
	.db	4,4	;Dauer fr Alarmeingang 3 / Alarmeingang 4
	.db	3,3	;Dauer fr Temp-berschreit./Temp-Unterschreit.
	.db	10,3	;Dauer fr Fehlermeldungen / Sensorausflle
;
; Timerwert fr die Signalfrequenz
;
dsigtm:	.db	122,0	;Timerwert (28-255), Beispiel-Wert fr 1405Hz
;
; Standard-Signallautstrke
;
dsigv1:	.db	10,0	;Lautstrkewert (0-15)
;
; Signallautstrke whrend der Nachtzeit
;
dsigv2:	.db	1,0	;Lautstrkewert (0-15)
;
; Start- und Endzeit fr die Nachtschaltung der Signalisierung, zur
; Startzeit wird auf Nacht-Lautstrke umgeschaltet und zur Endzeit auf
; Standard-Lautstrke; Wert in Minuten (16-Bit, 0-1439)
;
dsigni:	.dw	21*60	;Startzeit (21:00)
	.dw	7*60+30	;Endzeit (7:30)
;
; Start- und Endzeit fr die Nachtschaltung der LCD-Hintergrundbeleuch-
; tung, zur Startzeit wird die Beleuchtung abgeschaltet und zur Endzeit
; wieder eingeschaltet; Wert in Minuten (16-Bit, 0-1439)
;
dnigbl:	.dw	23*60	;Startzeit (23:00)
	.dw	7*60	;Endzeit (7:00)
;
; Zeiten fr automatisches Rcksetzen der Minimal- und Maximalwerte
; Wert in Minuten (16-Bit, 0-1439)
;
daresm:	.dw	20*60	;Auto-Reset-Zeit fr Minimalwerte (20:00)
	.dw	8*60	;Auto-Reset-Zeit fr Maximalwerte (8:00)
;
; Konfigurationsbyte 1
;
dconf1:	.db 0b01111011,0;B0: Sprache (0=englisch, 1=deutsch)
			;B1: Trennzeichen (0=Punkt, 1=Komma)
			;B2: Temperaturalarme immer signalisieren
			;B3: Nachtschaltung fr LCD-Beleuchtung aktiv
			;B4: Auto-Reset fr Min/Max-Werte aktiv
			;B5: Nachtschaltung fr Signal-Lautstrke aktiv
			;B6: LCD abschalten fr Uhr-Synchronisation
			;B7: 0=Normal, 1=Abwesend (Licht und Ton aus)
;
; Konfigurationsbyte 2
;
dconf2:	.db 0b00000100,0;B0: LED-Modus (0=normal, 1=debug [DCF,RS-232])
			;B1: LCD-Beleuchtung weich schalten
			;B2: Nachtschaltung fr Bluetooth-Modul aktiv
;
; Schwellwert fr die Steuerung der Hintergrundbeleuchtung, wird mit
; dem Wert des Lichtsensors verglichen (ADC0)
;
dlight:	.db	150,0	;Schwellwert fr Hintergrundbeleuchtung
;
; Baudrate fr die PC-Verbindung (RS232-0)
; 0=9600, 1=19200, 2=38400, 3=57600, 4=115200, 5=230400, 6=460800
;
dbaud0:	.db	4,0	;Baudrate USART0 (115200 Baud)
;
; Meldungsfilter, steuert die Menanzeige und die Datenausgabe, fol-
; gende Werte sind mglich: 0= alles anzeigen, 1= nur Alarme anzeigen,
; 2= nur Temperaturen anzeigen, 3= nur Fehler anzeigen
;
dmsgfi:	.db	0,0	;Meldungsfilter (alles anzeigen)
;
; Grafik-Presets, 16 Voreinstellungen fr die Grafikanzeige mit jeweils
; 5 Werten:
; Byte 0: Grafik 1, Sensornummer 1 und 2
; Byte 1: Grafik 2, Sensornummer 1 und 2
; Byte 2: Grafik 1, Anzeigemodus
; Byte 3: Grafik 2, Anzeigemodus
; Byte 4: Optionen fr die Grafikanzeige
;
; Presets 1-16 zeigen Einzeldiagramme mit 15min-Werten
; Presets 17-24 zeigen Doppeldiagramme mit 24h-Mittelwerten
;
dgrpre:	.db	0,0,0,0,1,1,0,0,0,1		;Preset 01, Preset 02
	.db	2,0,0,0,1,3,0,0,0,1		;Preset 03, Preset 04
	.db	4,0,0,0,1,5,0,0,0,1		;Preset 05, Preset 06
	.db	6,0,0,0,1,7,0,0,0,1		;Preset 07, Preset 08
	.db	8,0,0,0,1,9,0,0,0,1		;Preset 09, Preset 10
	.db	10,0,0,0,1,11,0,0,0,1		;Preset 11, Preset 12
	.db	12,0,0,0,1,13,0,0,0,1		;Preset 13, Preset 14
	.db	14,0,0,0,1,15,0,0,0,1		;Preset 15, Preset 16
;
	.db	0,1,3,3,0x11,2,3,3,3,0x11	;Preset 17, Preset 18
	.db	4,5,3,3,0x11,6,7,3,3,0x11	;Preset 19, Preset 20
	.db	8,9,3,3,0x11,10,11,3,3,0x11	;Preset 21, Preset 22
	.db	12,13,3,3,0x11,14,15,3,3,0x11	;Preset 23, Preset 24
;
; Grafik-Preset-Texte, Lnge=16 Zeichen
;
dgrprt:	.db	"Sensor01-15min  "	;Text fr Preset 01
	.db	"Sensor02-15min  "	;Text fr Preset 02
	.db	"Sensor03-15min  "	;Text fr Preset 03
	.db	"Sensor04-15min  "	;Text fr Preset 04
	.db	"Sensor05-15min  "	;Text fr Preset 05
	.db	"Sensor06-15min  "	;Text fr Preset 06
	.db	"Sensor07-15min  "	;Text fr Preset 07
	.db	"Sensor08-15min  "	;Text fr Preset 08
	.db	"Sensor09-15min  "	;Text fr Preset 09
	.db	"Sensor10-15min  "	;Text fr Preset 10
	.db	"Sensor11-15min  "	;Text fr Preset 11
	.db	"Sensor12-15min  "	;Text fr Preset 12
	.db	"Sensor13-15min  "	;Text fr Preset 13
	.db	"Sensor14-15min  "	;Text fr Preset 14
	.db	"Sensor15-15min  "	;Text fr Preset 15
	.db	"Sensor16-15min  "	;Text fr Preset 16
;
	.db	"Sen01/02-24h-avg"	;Text fr Preset 17
	.db	"Sen03/04-24h-avg"	;Text fr Preset 18
	.db	"Sen05/06-24h-avg"	;Text fr Preset 19
	.db	"Sen07/08-24h-avg"	;Text fr Preset 20
	.db	"Sen09/10-24h-avg"	;Text fr Preset 21
	.db	"Sen11/12-24h-avg"	;Text fr Preset 22
	.db	"Sen13/14-24h-avg"	;Text fr Preset 23
	.db	"Sen15/16-24h-avg"	;Text fr Preset 24
;
; Start- und Endzeit fr die Nachtschaltung des Bluetooth-Moduls, zur
; Startzeit wird das Modul abgeschaltet und zur Endzeit wieder einge-
; schaltet; Wert in Minuten (16-Bit, 0-1439)
;
dnigbt:	.dw	22*60	;Startzeit (22:00)
	.dw	9*60	;Endzeit (9:00)
;
;
; -----------------------------
; Belegung des internen EEPROMs
; -----------------------------
;
.eseg
;
	.byte	1	;Byte 0 wird nicht genutzt
;
etemap:	.byte	16	;Temperatur Anzeige Mapping Tabelle: Hier wer-
			;den die Anzeigepositionen der 16 Temperatur-
			;sensoren festgelegt (0-15), wenn Bit7=1: Tem-
			;peratursensor inaktiv
elomap:	.byte	16	;Temperatur Log Mapping Tabelle: Hier werden
			;die Listenpositionen der 16 Temperatursensoren
			;festgelegt (0-15)
etemtx:	.byte	16*9	;16 Temperaturtexte: werden an der von der
			;Mapping Tabelle festgelegten Position auf dem
			;LCD angezeigt, Lnge=9 Zeichen
etelih:	.byte	16*2	;Obere Temperaturlimits in Zehntel Grad (16-Bit
			;Hex), 0x8000=Limitberwachung ausgeschaltet
etelil:	.byte	16*2	;Untere Temperaturlimits in Zehntel Grad (16-
			;Bit Hex),0x8000=Limitberwachung ausgeschaltet
ealmap:	.byte	4	;Alarm Mapping Tabelle: hier werden die An-
			;zeigepositionen der Alarmdaten festgelegt
			;(0-3), wenn Bit7=1: Alarm nicht genutzt
ealinv:	.byte	4	;Alarm Invertierungs Mapping Tabelle: hier wird
			;festgelegt, ob der entsprechende Alarm inver-
			;tiert ausgewertet werden soll; 0=normal,
			;1=invertiert
ealmtx:	.byte	4*11	;Alarmtexte: werden an der von der Mapping
			;Tabelle festgelegten Position auf dem LCD
			;angezeigt, Lnge=11 Zeichen
ebigtm:	.byte	1	;logische Sensornummer (0-15)
esignr:	.byte	8	;Signaltabelle, enthlt die Nummer des akusti-
			;schen Signals (0-15) fr 8 verschiedene Ereig-
			;nisse
esigdu:	.byte	8	;Signaltabelle, enthlt die Dauer des akusti-
			;schen Signals in Sekunden fr 8 verschiedene
			;Ereignisse (1-30s), 0= akustisches Signal aus
esigtm:	.byte	1	;Timerwert fr die Signalfrequenz
esigv1:	.byte	1	;Standard-Signallautstrke
esigv2:	.byte	1	;Signallautstrke whrend der Nachtzeit
esigni:	.byte	2*2	;Start- und Endzeit fr die Nachtschaltung der
			;Signalisierung, zur Startzeit wird auf Nacht-
			;Lautstrke umgeschaltet und zur Endzeit auf
			;Standard-Lautstrke; Wert in Minuten (16-Bit,
			;0-1439)
enigbl:	.byte	2*2	;Start- und Endzeit fr die Nachtschaltung der
			;LCD-Hintergrundbeleuchtung, zur Startzeit wird
			;die Beleuchtung abgeschaltet und zur Endzeit
			;wieder eingeschaltet; Wert in Minuten (16-Bit,
			;0-1439)
earesm:	.byte	2*2	;Zeiten fr automatisches Rcksetzen der Mini-
			;mal- und Maximalwerte; Wert in Minuten (16-
			;Bit, 0-1439)
econf1:	.byte	1	;Konfigurationsbyte 1
			;B0: Sprache (0=englisch, 1=deutsch)
			;B1: Trennzeichen (0=Punkt, 1=Komma)
			;B2: Temperaturalarme immer signalisieren
			;B3: Nachtschaltung fr LCD-Beleuchtung aktiv
			;B4: Auto-Reset fr Min/Max-Werte aktiv
			;B5: Nachtschaltung fr Signal-Lautstrke aktiv
			;B6: LCD abschalten fr Uhr-Synchronisation
			;B7: 0=Normal, 1=Abwesend (Licht und Ton aus)
econf2:	.byte	1	;Konfigurationsbyte 2
			;B0: LED-Modus (0=normal, 1=debug [DCF,RS-232])
			;B1: LCD-Beleuchtung weich schalten
			;B2: Nachtschaltung fr Bluetooth-Modul aktiv
elight:	.byte	1	;Schwellwert fr die Steuerung der Hintergrund-
			;beleuchtung, wird mit dem Wert des Lichtsen-
			;sors verglichen (ADC0)
ebaud0:	.byte	1	;Baudrate fr die PC-Verbindung (RS232-0)
			;0=9600, 1=19200, 2=38400, 3=57600, 4=115200,
			;5=230400, 6=460800
emsgfi:	.byte	1	;Meldungsfilter, steuert die Menanzeige und
			;die Datenausgabe, folgende Werte sind mglich:
			;0= alles anzeigen
			;1= nur Alarme anzeigen
			;2= nur Temperaturen anzeigen
			;3= nur Fehler anzeigen
egrpre:	.byte	24*5	;24 Grafik-Presets mit jeweils 5 Bytes:
			;Byte 0: Grafik 1, Sensornummer 1 und 2
			;Byte 1: Grafik 2, Sensornummer 1 und 2
			;Byte 2: Mode 1 (0-3): 0=15min, 1=24h-Minimal,
			;	 2=24h-Maximalwerte, 3=24h-Mittelwerte
			;Byte 3: Mode 2 (0-3): 0=15min, 1=24h-Minimal,
			;	 2=24h-Maximalwerte, 3=24h-Mittelwerte
			;Byte 4: Verschiedene Grafik-Optionen:
			;	 B0= Sensornamen in Grafik einblenden
			;	 B1= Temperaturskala angleichen
			;	 B2= Differenzmodus fr Diagramm 1
			;	 B3= Differenzmodus fr Diagramm 2
			;	 B4= Doppeldiagramm
egrprt:	.byte	24*16	;24 Grafik-Preset-Texte mit jeweils 16 Zeichen
enigbt:	.byte	2*2	;Start- und Endzeit fr die Nachtschaltung des
			;Bluetooth-Moduls, zur Startzeit wird das Modul
			;abgeschaltet und zur Endzeit wieder einge-
			;schaltet; Wert in Minuten (16-Bit, 0-1439)
eprcrc:	.byte	1	;8-Bit-CRC, enthlt die Prfsumme ber den
			;genutzten EEPROM-Bereich vom Byte 1 beginnend
			;CRC wird bei jedem Schreibvorgang neu gesetzt
			;und beim Lesen geprft
;
