; =====================================================
; Anzeigemodul 1 fr Temperaturmessystem mit ATtiny4313
; von Scott-Falk Hhn (s.huehn@soemtron.de)
; Version 2.00, letzte Bearbeitung am 21.03.2013
; =====================================================
;
;.include "Praxis-Beispiel 4 Sensoren.inc"
.include "Praxis-Beispiel 8 Sensoren.inc"
;.include "Beispiel-01-08.inc"
;.include "Beispiel-09-16.inc"
;.include "Beispiel-17-24.inc"
;.include "Beispiel-25-31.inc"
;
; --------------------------------------------
; Programmierung der Fusebits (Atmel Studio 6)
; --------------------------------------------
;
; SELFPRGEN = [ ]
; DWEN =      [ ]
; EESAVE =    [X]
; SPIEN =     [X]
; WDTON =     [ ]
; BODLEVEL =  2V7
; RSTDISBL =  [ ]
; CKDIV8 =    [ ]
; CKOUT =     [ ]
; SUT_CKSEL = EXTXOSC_8MHZ_XX_14CK_65MS
;
; EXTENDED = 0xFF
; HIGH = 0x9B
; LOW = 0xFF
;
; ----------------------
; Belegung der I/O-Ports
; ----------------------
;
; PortA0: (nicht initialisiert, nicht genutzt)
; PortA1: (nicht initialisiert, nicht genutzt)
; PortA2: (nicht initialisiert, nicht genutzt)
;
; PortB0: Ausgang LCD-Anschluss D0 \
; PortB1: Ausgang LCD-Anschluss D1  \
; PortB2: Ausgang LCD-Anschluss D2   \
; PortB3: Ausgang LCD-Anschluss D3    \ Standard-LCD Anschluss mit
; PortB4: Ausgang LCD-Anschluss D4    / HD44780 oder KS0066 Controller
; PortB5: Ausgang LCD-Anschluss D5   /
; PortB6: Ausgang LCD-Anschluss D6  /
; PortB7: Ausgang LCD-Anschluss D7 /
;
; PortD0: Eingang mit Pull-up, RS232-RXD, Empfangsdaten, Baudrate 9600
; PortD1: Eingang mit Pull-up, Taster
; PortD2: Ausgang LCD-Anschluss RS
; PortD3: Ausgang LCD-Anschluss E
; PortD4: Ausgang LED (Low = ein)
; PortD5: Ausgang Signalgeber (High = ein)
; PortD6: Eingang mit Pull-up (nicht genutzt)
;
; ------------------
; Belegung der Timer
; ------------------
;
; Timer0: nicht genutzt
; Timer1: generiert Interrupts alle 20 ms zur Steuerung der Timeouts und zur Tastenentprellung
;
; --------------------
; Konstanten festlegen
; --------------------
;
.equ	clock=	4000000	;Taktfrequenz = 4,0 MHz
.equ	timval=	10000	;Timer1-Wert fr 50Hz / 20ms bei Vorteiler 8
.equ	baud=	9600	;Baudrate = 9600
.equ	timsen=	60	;Timeout-Wert fr Sensorwerte in Sekunden (eingestellt auf 60 Sekunden)
.equ	timms1= 60	;Timeout-Wert fr Alarmmeldungen in Sekunden bei Systemstart (eingestellt auf 60 Sekunden)
.equ	timmsg=	20	;Timeout-Wert fr Alarmmeldungen in Sekunden im laufenden Betrieb (eingestellt auf 20 Sekunden)
.equ	timsem=	10	;Timeout-Wert fr Sensorbelegungs-Anzeige in Sekunden (eingestellt auf 10 Sekunden)
.equ	timbuz=	150	;Timeout-Wert fr Buzzer in 1/50 Sekunden, sollte ein Vielfaches von 50 betragen (eingestellt auf 3,0 Sekunden)
.equ	timalm=	60	;Timeout-Wert fr Alarmanzeige-Zeit (eingestellt auf 60s)
.equ	cr=	13	;Zeichencode Cariage Return (Enter)
.equ	lf=	10	;Zeichencode fr Line Feed
.equ	led=	4	;LED-Bit von PortD
.equ	buzzer=	5	;Piezo-Summer-Bit von PortD
;
; -------------------
; Register definieren
; -------------------
;
.def	insreg=	r1	;Zwischenspeicher fr SREG whrend Interrupt
.def	rxpoi1=	r2	;UART Empfangspuffer-Pointer1 (Schreibposition)
.def	rxpoi2=	r3	;UART Empfangspuffer-Pointer2 (Leseposition)
.def	delayc=	r4	;Zeitschleifenzhler
.def	rspoin=	r5	;Zeiger auf RS-232 Textpuffer
.def	precnt=	r6	;Interrupt-Vorteiler (20 ms -> 1 sek)
.def	buzcn1=	r7	;Timeout-Zhler fr Buzzer, wird vom Hauptprogramm auf einen Wert gesetzt und vom Timer-Interrupt im Takt von 20ms auf 0 gezhlt
.def	buzcn2=	r8	;wie r7, fr Temperatur-Alarm
.def	seccnt=	r9	;Sekundenzhler, wird fr blinkende Symbole und Alarm-Wechselanzeige bentigt
.def	extcnt=	r10	;zustzlicher Zeitzhler, wird nach jedem berlauf von <seccnt> erhht und bei 14 wieder zurckgesetzt (entspricht 3584s / ca. 1h)
.def	msgtio=	r11	;Timeout-Zhler fr Meldungen; wird bei Empfang eines Datensatzes auf 20s gesetzt und vom Timer-Interrupt auf 0 gezhlt; Hauptprogramm signalisiert bei 0 "Kein Datenempfang"
			;und setzt den Wert auf 0xff (Stopp)
.def	distio=	r12	;Timeout-Zhler fr die Rckkehr zur normalen Sensoranzeige nach dem Empfang einer Sensorbelegung
.def	lastal=	r13	;Alarmnummer des zuletzt ausgelsten Alarms, wird fr Alarmton bentigt
.def	xflags=	r14	;Sonstige Flags
			;Bit 0 = Fehler-Quittung, wird bei der Anzeige eines Verbindungsausfalls gesetzt und verhindert ein Flackern der Anzeige
			;Bit 1 = Stunden-Flag, wird von Timer-Interrupt nach 3584s gesetzt, Hauptprogramm speichert den aktuellen Luftdruckwert im Puffer und lscht das Flag wieder
.def	flags=	r23	;Verschiedene Flags
			;Bit 0 = nderungsflag, wird bei nderungen von Sensorwerten oder Alarmen gesetzt und nach der Aktualisierung des LCD wieder gelscht
			;Bit 1 = Tasten-Flag 1 (gedrckt), wird vom Timer-Interrupt gesetzt und gelscht
			;Bit 2 = Tasten-Flag 2 (entprellt), wird vom Timer-Interrupt gesetzt und gelscht; dient als Flag fr das Hauptprogramm
			;Bit 3 = Tasten-Flag 3 (quittiert), wird vom Hauptprogramm gesetzt und vom Timer-Interrupt gelscht
			;Bit 4 = Fehler-Flag, wird nach Ablauf des Meldungs-Timeout gesetzt; anstatt der Sensoranzeige erscheint ein Fehlertext
			;Bit 5 = Alarm-Flag, wird gesetzt, wenn mindestens ein Alarm-Timeout-Zhler aktiv ist; wird durch Interrupt-Routine geseteuert
			;Bit 6 = Alarm-Anzeige unterdrckt, kann mit Taster gesetzt bzw. gelscht werden und wird bei Alarm-Auslsung immer gelscht
			;Bit 7 = Anzeige der Sensorbelegung
.def	itemp1=	r24	;Zwischenspeicher 1 whrend Interruptbearbeitung
.def	itemp2=	r25	;Zwischenspeicher 2 whrend Interruptbearbeitung
;
; -----------------------
; Speicheradressen im RAM
; -----------------------
;
.dseg
;
rstext:	.byte	8	;Text-Empfangspuffer; wird vom Hauptprogramm aus UART-Puffer kopiert und verarbeitet
sentab:	.byte	8	;Sensor-Tabelle, wird aus EEPROM gelesen und enthlt die Nummern der 8 Sensoren in der Reihenfolge der Anzeigepositionen
sentio:	.byte	8	;Timeout-Zhler fr Sensor 1-8; wird bei Empfang eines Datensatzes auf 60s gesetzt und von Timer-Interrupt auf 0 gezhlt; Hauptprogramm lscht bei 0 den Sensorwert auf dem LCD
			;und setzt den Wert auf 0xff (Stopp)
alminv:	.byte	4	;Alarm-Invertierungs-Tabelle, wird aus EEPROM gelesen und bestimmt, ob die empfangene Alarmmeldung invertiert wird oder nicht:
			;0 = normal
			;1 = invertiert
almbuf:	.byte	4	;Speicher fr 4 Alarme, jeder Alarm kann folgende Zustnde annehmen:
			;0 = kein Alarm
			;1 = Alarm aktiv
almtio:	.byte	4	;Timeout-Zhler fr 4 Alarme, wird bei aktivem Alarm auf <timalm> gehalten und vom Timer-Interrupt nach Alarm-Ende auf 0 gezhlt
senbuf:	.byte	8 * 5	;Speicher fr die empfangenen Sensorwerte
temlim:	.byte	8	;Flags fr Temperaturlimit-berschreitungen:
			;Bit 0 = Temperatur-Unterschreitung
			;Bit 1 = Temperatur-berschreitung
prsbuf:	.byte	16 * 2	;Luftdruckwerte-Puffer, speichert jede Stunde (alle 3584s) den aktuellen Luftdruck als 16-Bit-Binrwert an der 1. Position, alle anderen Werte werden dabei um eine Position
			;verschoben; der erste Wert wird als Startwert auf alle Positionen geschrieben
semode:	.byte	1	;Sensor-Modus:
			;0 = 4-Sensor-Modus, Anzeige von 4 Sensorwerten, Sensornamen sind 14 Zeichen lang
			;1 = 8-Sensor-Modus, Anzeige von 8 Sensorwerten, Sensornamen sind 4 Zeichen lang
almnum:	.byte	1	;Alarm-Nummer fr Anzeige, wird bei Alarm auf Zeile 4 ausgegeben, wird von Interrupt gesteuert
prtend:	.byte	1	;Luftdruck-Tendenz:
			;0 = keine Tendenz erkannt
			;1 = Tendenz fallend
			;2 = Tendenz steigend
prs_th:	.byte	1	;Schwellwert fr Luftdruck-Tendenz-Anzeige, wird aus EEPROM gelesen; der aktuelle Luftdruckwert wird mit dem gesamten Luftdruckwerte-Puffer vergleichen und bei ber- oder
			;Unterschreitung des Schwellwertes wird eine fallende oder steigende Tendenz angezeigt
;
.org	0x100		;UART-Empfangspuffer muss fest auf 0x100 liegen
rxbuff:	.byte	64	;UART Empfangspuffer (64 Bytes)
;
; ----------------------------------
; Programmbeginn, Interrupt-Vektoren
; ----------------------------------
;
.cseg
	rjmp	start			;Programmstart
;
.org	OC1Aaddr
	rjmp	timer1			;Timer1, Compare Match A, wird alle 20ms ausgelst
;
.org	URXCaddr
	rjmp	rxcint			;UART Empfang komplett
;
.org	INT_VECTORS_SIZE
;
; ----------------------------------------------------------
; Programmbeginn, Initialisierung von I/O, RAM und Registern
; ----------------------------------------------------------
;
start:	cli				;Interrupts sperren
	wdr				;Watchdog zurcksetzen
	ldi	r16, (1 << WDCE) | (1 << WDE)
	out	WDTCSR, r16		;Watchdog-nderung freigeben
	ldi	r16, (1 << WDE) | (1 << WDP3)
	out	WDTCSR, r16		;Watchdog aktivieren (4,0s)
;
	ldi	zl, low(SRAM_START)
	ldi	zh, high(SRAM_START)	;Zeiger auf RAM-Anfang setzen
	clr	r16			;Nullwert laden
sta010:	st	z+, r16					;Speicherzelle lschen
	cpi	zl, low(SRAM_START + SRAM_SIZE)		;Ende erreicht?
	brne	sta010					;nein -> Schleife
	cpi	zh, high(SRAM_START + SRAM_SIZE)	;Ende erreicht?
	brne	sta010					;nein -> Schleife
	ldi	r16, low(RAMEND)
	ldi	r17, high(RAMEND)	;Stackpointer-Adresse laden
	out	SPL, r16
	out	SPH, r17		;Stackpointer setzen
;
	clr	r16
	out	PORTB, r16		;PortB alle Ausgnge auf 0 setzen
	ldi	r16, (1 << PORTD6) | (1 << PORTD1) | (1 << PORTD0)
	out	PORTD, r16		;PortD Ausgnge auf 0 setzen, bei Eingngen Pull-up einschalten
	ser	r16
	out	DDRB, r16		;PortB komplett als Ausgnge setzen
	ldi	r16, (1 << PORTD5) | (1 << PORTD4) | (1 << PORTD3) | (1 << PORTD2)
	out	DDRD, r16		;PortD2 - PortD5 als Ausgnge setzen, Rest als Eingang
;
	ldi	r16, low(rxbuff)	;Pointer fr Empfangspuffer laden (Anfangswert)
	mov	rxpoi1, r16		;Pointer 1 setzen (Schreibposition)
	mov	rxpoi2, r16		;Pointer 2 setzen (Leseposition)
	ldi	yh, high(rstext)	;Pointer High-Byte fr Variablen setzen
	ldi	r16, low(rstext)	;Pointer fr Textpuffer laden (Anfangswert)
	mov	rspoin,	r16		;Pointer setzen
	clr	precnt			;Interrupt-Vorteiler lschen
	clr	buzcn1			;Buzzer-Timeout-Zhler 1 lschen
	clr	buzcn2			;Buzzer-Timeout-Zhler 2 lschen
	clr	flags			;Flags lschen
	ldi	r16, 1 << 1		;Stunden-Flag setzen (ersten Luftdruckwert speichern)
	mov	xflags, r16		;Sonstige Flags voreinstellen
	clr	distio			;Anzeige-Timeout-Zhler lschen
	clr	lastal			;letzte Alarmnummer lschen
;
	ldi	yl, low(senbuf)		;Zeiger auf Sensorwerte
	ldi	r17, 8 * 5		;8 Werte mit je 5 Zeichen
	ldi	r16, ' '		;mit Leerzeichen fllen
sta020:	st	y+, r16			;Speicherzelle lschen
	dec	r17			;alle Zellen gelscht?
	brne	sta020			;nein -> Schleife
;
	ldi	r16, timms1
	mov	msgtio, r16		;Anfangswert fr Meldungs-Timeout setzen
;
	ldi	r16, (clock / (16 * baud)) - 1
	clr	r17
	out	UBRRL, r16
	out	UBRRH, r17		;Baudrate RS-232 auf 9600 Baud setzen
	clr	r16
	out	UCSRA, r16		;normale Geschwindigkeit, kein Multi-Prozessor-Modus
	ldi	r16, (1 << RXCIE) | (1 << RXEN)
	out	UCSRB, r16		;RX mit Interrupt aktivieren
	ldi	r16, (1 << UCSZ1) | (1 << UCSZ0)
	out	UCSRC, r16		;Asynchron, keine Paritt, 1 Stopp-Bit, 8 Daten-Bits
;
	clr	r16
	out	TCCR1A, r16		;keine Output Compare Pins verwenden, CTC-Modus
	ldi	r16, (1 << WGM12) | (1 << CS11)
	out	TCCR1B, r16		;CTC-Modus, Vorteiler = 8 setzen
	clr	r16
	out	TCCR1C, r16		;keine Force Output Compare verwenden
	ldi	r16, low(timval)
	ldi	r17, high(timval)	;Timer1-Compare-Wert setzen
	out	OCR1AH, r17
	out	OCR1AL, r16		;ausgeben
	ldi	r16, (1 << OCIE1A)
	out	TIMSK, r16		;Timer1 Output Compare Match A Interrupt aktivieren
;
; LCD initialisieren und Sonderzeichen programmieren
;
	rcall	w50ms			;Warteschleife 50ms
	clr	r17			;LCD in Befehlsmodus setzen
	ldi	r16, 0x30		;Initialisierung
	rcall	lcdout			;ausgeben
	rcall	w04ms			;Warteschleife 4,1ms
	rcall	lcdout			;Init nochmals ausgeben
	rcall	w100my			;Warteschleife 100s
	rcall	lcdw40			;Init nochmals ausgeben
	ldi	r16, 0x38		;8-Bit-Interface, 2-Line, 5x7 Font
	rcall	lcdw40			;ausgeben
	ldi	r16, 0x0c		;Display ein, kein Cursor
	rcall	lcdw40			;ausgeben
	ldi	r16, 0x06		;Increment, kein Scolling
	rcall	lcdw40			;ausgeben
;
	ldi	r16, 0x40		;LCD-Adresse der Sonderzeichen
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdw40			;Adresse setzen
	ldi	r19, 8 * 8		;8 Sonderzeichen mit jeweils 8 Bytes
	ldi	r17, 1			;LCD in Datenmodus setzen
	ldi	zl, low(2 * chars)
	ldi	zh, high(2 * chars)	;Zeiger auf Zeichen-Bitmaps
sta100:	lpm	r16, z+			;Datenbyte aus Programmspeicher holen
	rcall	lcdw40			;an LCD ausgeben
	dec	r19			;alle Zeilen ausgegeben?
	brne	sta100			;nein -> Schleife
;
	ldi	r16, 0x01		;Befehl: Anzeige lschen
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdout			;ausgeben
	rcall	w15ms			;Warteschleife 15,2ms
;
; Begrungstext und Versionsnummer ausgeben
;
	clr	r19			;Zeichenzhler = 0 setzen
	ldi	zl, low(2 * sttext)
	ldi	zh, high(2 * sttext)	;Zeiger auf Text setzen
sta200:	cpi	r19, 20			;Beginn Zeile 2?
	breq	sta220			;ja -> LCD-Adresse setzen
	cpi	r19, 40			;Beginn Zeile 3?
	breq	sta230			;ja -> LCD-Adresse setzen
	cpi	r19, 60			;Beginn Zeile 4?
	breq	sta240			;ja -> LCD-Adresse setzen
	cpi	r19, 80			;Textende erreicht?
	breq	sta260			;ja -> Ende
sta210:	lpm	r16, z+			;Zeichen aus Programmspeicher holen
	ldi	r17, 1			;LCD in Datenmodus setzen
	rcall	lcdw40			;Zeichen ausgeben
	inc	r19			;Zeichenzhler erhhen
	rjmp	sta200			;Schleife
;
sta220:	movw	yl, zl			;Textzeiger sichern
	ldi	r16, 1
	rcall	lcdadr			;LCD-Adresse von Zeile 2 ermitteln
	movw	zl, yl			;Textzeiger wiederherstellen
	rjmp	sta250
sta230:	movw	yl, zl			;Textzeiger sichern
	ldi	r16, 2
	rcall	lcdadr			;LCD-Adresse von Zeile 3 ermitteln
	movw	zl, yl			;Textzeiger wiederherstellen
	rjmp	sta250
sta240:	movw	yl, zl			;Textzeiger sichern
	ldi	r16, 3
	rcall	lcdadr			;LCD-Adresse von Zeile 4 ermitteln
	movw	zl, yl			;Textzeiger wiederherstellen
sta250:	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdw40			;Zeichen ausgeben
	rjmp	sta210			;Schleife
;
sta260:	ldi	r19, 200		;Warteschleife 200 * 50ms (10s)
sta270:	rcall	w50ms
	dec	r19
	wdr				;Watchdog zurcksetzen
	brne	sta270
;
	clr	r17			;LCD in Befehlsmodus setzen
	ldi	r16, 0x01		;Befehl: Anzeige lschen
	rcall	lcdout			;ausgeben
	rcall	w15ms			;Warteschleife 15,2ms
;
; Sensor-Tabelle aus EEPROM lesen und Werte prfen
;
	ldi	r16, esentb		;EEPROM-Adresse der Sensor-Tabelle
	ldi	r18, 8			;8 Werte auslesen
	ldi	yl, low(sentab)		;Zeiger auf Sensor-Tabelle
sta300:	rcall	eeread			;EEPROM lesen
	cpi	r17, 32			;Sensornummer > 31?
	brcc	sta900			;ja -> Fehler
	st	y+, r17			;sonst Sensornummer speichern
	inc	r16			;EEPROM-Adresse erhhen
	dec	r18			;alle Zeichen gelesen?
	brne	sta300			;nein -> Schleife
;
; Sensor-Modus bestimmen (Eintrge 5-8 der Sensor-Tabelle prfen)
;
	ldi	yl, low(sentab + 4)	;Zeiger auf Sensor-Tabelle (Sensor 5)
	ldi	r18, 4			;4 Werte prfen
	clr	r16			;Summenzhler lschen
sta310:	ld	r17, y+			;Sensornummer aus Tabelle holen
	add	r16, r17		;Sensornummer addieren
	dec	r18			;alle Tabellen-Eintrge gelesen?
	brne	sta310			;nein -> Schleife
	tst	r16			;Summenzhler = 0?
	breq	sta320			;ja -> weiter (4-Sensor-Modus)
	ldi	r16, 1			;sonst 8-Sensor-Modus setzen
sta320:	sts	semode, r16		;Sensor-Modus speichern
;
; Alarm-Invertierungs-Tabelle aus EEPROM lesen und Werte prfen
;
	ldi	r16, ealmin		;EEPROM-Adresse der Invertierungs-Tabelle
	ldi	r18, 4			;4 Werte auslesen
	ldi	yl, low(alminv)		;Zeiger auf Werte-Tabelle
sta400:	rcall	eeread			;EEPROM lesen
	cpi	r17, 2			;Wert > 1?
	brcc	sta900			;ja -> EEPROM-Fehler
	st	y+, r17			;sonst Zeichen speichern
	inc	r16			;EEPROM-Adresse erhhen
	dec	r18			;alle Zeichen gelesen?
	brne	sta400			;nein -> Schleife
;
; Alarmnamen auf Plausibilitt prfen
;
	ldi	r16, ealmtx		;EEPROM-Adresse der Alarmnamen
	ldi	r18, 4 * 12		;4 Alarme mit jeweils 12 Zeichen prfen
sta500:	rcall	eeread			;Zeichen aus EEPROM holen
	cpi	r17, ' '		;Zeichen < Leerzeichen?
	brcs	sta900			;ja -> EEPROM-Fehler
	cpi	r17, 255		;Zeichencode = 255?
	breq	sta900			;ja -> EEPROM-Fehler
	inc	r16			;EEPROM-Adresse erhhen
	dec	r18			;alle Texte geprft?
	brne	sta500			;nein -> Schleife
;
; Sensornamen auf Plausibilitt prfen
;
	ldi	r16, esentx		;EEPROM-Adresse der Sensornamen
	ldi	r18, 8 * 14		;8 Sensoren mit jeweils 14 Zeichen prfen
sta600:	rcall	eeread			;Zeichen aus EEPROM holen
	cpi	r17, ' '		;Zeichen < Leerzeichen?
	brcs	sta900			;ja -> EEPROM-Fehler
	cpi	r17, 255		;Zeichencode = 255?
	breq	sta900			;ja -> EEPROM-Fehler
	inc	r16			;EEPROM-Adresse erhhen
	dec	r18			;alle Texte geprft?
	brne	sta600			;nein -> Schleife
;
; Schwellwert fr Luftdruck-Tendenz-Anzeige und Trennzeichen (Komma) prfen
;
	ldi	r16, eprsth		;EEPROM-Adresse des Schwellwertes
	rcall	eeread			;Wert aus EEPROM holen
	cpi	r17, 2			;Wert < 2?
	brcs	sta900			;ja -> EEPROM-Fehler
	cpi	r17, 10			;Wert >= 10?
	brcc	sta900			;ja -> EEPROM-Fehler
	sts	prs_th, r17		;sonst Schwellwert speichern
;
	ldi	r16, epoint		;EEPROM-Adresse des Trennzeichens (Komma)
	rcall	eeread			;Zeichen aus EEPROM holen
	cpi	r17, 7			;Sonderzeichen 7?
	breq	sta700			;ja -> Systemstart
	cpi	r17, ' '		;Zeichen < ' ' (Steuerzeichen)?
	brcs	sta900			;ja -> EEPROM-Fehler
;
sta700:	sei				;Interrupts aktivieren
	rjmp	main
;
; Fehlertext mit Speicheradresse bei unplausiblen EEPROM-Daten ausgeben, LED bleibt eingeschaltet
;
sta900:	mov	r20, r16		;letzte EEPROM-Adresse sichern
	ldi	r16, 1
	rcall	lcdadr			;LCD-Adresse von Zeile 1 ermitteln
	inc	r16			;Zeichenposition 1 setzen
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdw40			;Ausgabe
	ldi	r19, 14			;14 Zeichen ausgeben
	ldi	zl, low(2 * eftext)
	ldi	zh, high(2 * eftext)	;Zeiger auf Text setzen
sta910:	lpm	r16, z+			;Zeichen aus Programmspeicher holen
	ldi	r17, 1			;LCD in Datenmodus setzen
	rcall	lcdw40			;Zeichen ausgeben
	dec	r19			;alle Zeichen ausgegeben?
	brne	sta910			;nein -> Schleife
;
	ldi	r16, '0'		;Zeichen '0'
	rcall	lcdw40			;ausgeben
	ldi	r16, 'x'		;Zeichen 'x'
	rcall	lcdw40			;ausgeben
	mov	r16, r20		;gesicherte letzte EEPROM-Adresse holen
	andi	r16, 0xf0		;oberes Nibble filtern
	swap	r16			;Nibbles tauschen
	ori	r16, 0x30		;in ASCII wandeln
	cpi	r16, 0x3a		;Wert > "9"?
	brcs	sta920			;nein -> weiter
	subi	r16, -7			;sonst in Buchstabe wandeln
sta920:	rcall	lcdw40			;Zeichen ausgeben
	mov	r16, r20		;gesicherte letzte EEPROM-Adresse holen
	andi	r16, 0x0f		;unteres Nibble filtern
	ori	r16, 0x30		;in ASCII wandeln
	cpi	r16, 0x3a		;Wert > "9"?
	brcs	sta930			;nein -> weiter
	subi	r16, -7			;sonst in Buchstabe wandeln
sta930:	rcall	lcdw40			;Zeichen ausgeben
;
sta940:	wdr				;Watchdog zurcksetzen
	rjmp	sta940			;Endlos-Schleife
;
; ---------------------
; Hauptprogrammschleife
; ---------------------
;
; Alarm-LED bei Fehler oder Alarm blinken lassen
; 40ms aus / 460ms ein (2Hz) bei Alarm
; 40ms ein / 960ms aus (1Hz) bei Fehler oder Temperatur-Alarm
;
main:	ldi	yl, low(almbuf)		;Zeiger auf Alarmwerte
	clr	r16			;Zhler fr Alarme
ma1000:	ld	r17, y+			;Alarmwert holen
	tst	r17			;Alarm aktiv?
	breq	ma1010			;nein -> weiter
	inc	r16			;sonst Alarmzhler erhhen
ma1010:	cpi	yl, low(almbuf + 4)	;alle Werte verarbeitet?
	brne	ma1000			;nein -> Schleife
	tst	r16			;sind Alarme aktiv?
	brne	ma1030			;ja -> LED zeigt Alarm
	bst	flags, 4		;Fehler-Flag gesetzt?
	brts	ma1060			;ja -> LED zeigt Fehler
;
	ldi	yl, low(temlim)		;Zeiger auf Limit-Flags
	clr	r16			;Summenzhler lschen
ma1020:	ld	r17, y+			;Flag holen und
	or	r16, r17		;mit Summenzhler verknpfen
	cpi	yl, low(temlim + 8)	;alle Werte verarbeitet?
	brne	ma1020			;nein -> Schleife
	tst	r16			;Limit-berschreitungen?
	brne	ma1060			;ja -> LED zeigt Fehler
	sbi	PORTD, led		;sonst LED ausschalten
	rjmp	ma1100
;
ma1030:	mov	r16, precnt		;Vorteiler holen
	cpi	r16, 25			;Wert >= 25?
	brcs	ma1040			;nein -> weiter
	subi	r16, 25			;sonst Wert-25
ma1040:	cpi	r16, 2			;Wert > 1? (entspricht 40ms)
	brcc	ma1050			;ja -> LED ein
	sbi	PORTD, led		;sonst LED ausschalten
	rjmp	ma1100
ma1050:	cbi	PORTD, led		;LED einschalten
	rjmp	ma1100
;
ma1060:	mov	r16, precnt		;Vorteiler holen
	cpi	r16, 2			;Wert > 1? (entspricht 40ms)
	brcc	ma1070			;ja -> LED aus
	cbi	PORTD, led		;sonst LED einschalten
	rjmp	ma1100
ma1070:	sbi	PORTD, led		;LED ausschalten
;
; Taster-Abfrage bearbeiten
;
ma1100:	bst	flags, 2		;Taster gedrckt?
	brtc	ma1120			;nein -> weiter
	bst	flags, 3		;wurde Taster schon quittiert?
	brts	ma1120			;ja -> weiter
	sbr	flags, (1 << 3) | (1 << 0)	;Taster quittieren, nderungs-Flag setzen
	bst	flags, 7		;Anzeige der Sensorbelegung?
	brtc	ma1110			;nein -> weiter
	cbr	flags, 1 << 7		;sonst auf normale Sensoranzeige zurckschalten
	ser	r16			;Anzeigetimeout inaktiv setzen
	mov	distio, r16		;Timeout-Zhler speichern
	rjmp	ma1120
;
ma1110:	ldi	r16, 1 << 6		;Maske fr Alarm-Unterdrckungs-Flag laden
	eor	flags, r16		;Alarm-Unterdrckungs-Flag umschalten
;
ma1120:	bst	flags, 0		;nderungs-Flag gesetzt?
	brtc	ma1130			;nein -> keine neue Anzeige
	cbr	flags, 1 << 0		;sonst nderungs-Flag lschen
	bst	flags, 7		;Anzeige der Sensorbelegung?
	brtc	ma1200			;nein -> normale Sensoranzeige
ma1130:	rjmp	ma1600			;sonst Anzeige berspringen
;
; Anzeige der Sensor- und Alarmdaten auf dem LCD
;
ma1200:	bst	flags, 4		;Fehler-Flag gesetzt?
	brtc	ma1210			;nein -> weiter
	rjmp	ma1510			;sonst Fehlertext anzeigen
;
ma1210:	clr	r20			;Positions-Zhler = 0
ma1220:	mov	r16, r20		;Positions-Zhler kopieren
	rcall	lcdadr			;LCD-Adresse des aktuellen Sensors ermitteln
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdw40			;Adresse ausgeben
;
	bst	flags, 6		;Alarm-Unterdrckung aktiv?
	brts	ma1400			;ja -> normale Sensorausgabe
	bst	flags, 5		;Alarm-Flag gesetzt?
	brtc	ma1400			;nein -> normale Sensorausgabe
	mov	r16, seccnt		;Sekundenzhler holen
	andi	r16, 0b00000010		;Sekunde 0 und 1 im 4s-Intervall?
	brne	ma1400			;nein -> normale Sensorausgabe
	cpi	r20, 7			;Anzeige-Position 7 in Bearbeitung?
	brne	ma1230			;nein -> weiter testen
	rjmp	ma1480			;sonst Ausgabe berspringen
ma1230:	cpi	r20, 3			;Anzeige-Position 3 in Bearbeitung?
	brne	ma1400			;nein -> normale Sensorausgabe
;
; Ausgabe eines Alarms
;
	ldi	zl, low(2 * almtxt)
	ldi	zh, high(2 * almtxt)	;Zeiger auf Alarm-Text
	ldi	r19, 8			;8 Zeichen ausgeben
	ldi	r17, 1			;LCD in Datenmodus setzen
ma1300:	lpm	r16, z+			;Textzeichen holen
	rcall	lcdw40			;Zeichen ausgeben
	dec	r19			;alle Zeichen ausgegeben?
	brne	ma1300			;nein -> Schleife
;
	lds	r16, almnum		;aktuelle Alarmnummer holen
	ldi	r17, 12			;Lnger der Alarmnamen 12 Zeichen
	rcall	mul8x8			;Position des Alarmnamens ermitteln
	ldi	r21, ealmtx		;Zeiger auf Alarmnamen im EEPROM
	add	r21, r17		;ermittelte Position addieren
	ldi	r19, 12			;12 Zeichen ausgeben
ma1310:	mov	r16, r21		;EEPROM-Adresse setzen
	rcall	eeread			;EEPROM lesen
	mov	r16, r17		;Zeichen kopieren
	ldi	r17, 1			;LCD in Datenmodus setzen
	rcall	lcdw40			;Zeichen ausgeben
	inc	r21			;nchste EEPROM-Adresse
	dec	r19			;alle Zeichen ausgegeben?
	brne	ma1310			;nein -> Schleife
	rjmp	ma1480
;
; Ausgabe eines Sensorwertes
;
ma1400:	mov	r16, r20		;Sensorzhler holen
	ldi	r17, 5			;mit 5 multiplizieren
	rcall	mul8x8			;Offset fr Sensorwert berechnen
	mov	r21, r17		;Offset kopieren
	ldi	r17, 14			;mit 14 multiplizieren
	rcall	mul8x8			;Offset fr EEPROM-Adresse berechnen
	ldi	r16, esentx		;EEPROM-Adresse Sensortexte
	add	r16, r17		;Adresse berechnen
	ldi	r19, 4			;4 Zeichen lesen
	lds	r18, semode		;Sensor-Modus holen
	tst	r18			;8-Sensor-Modus?
	brne	ma1410			;ja -> weiter
	ldi	r19, 14			;sonst 14 Zeichen lesen
;
ma1410:	rcall	eeread			;EEPROM lesen
	push	r16			;EEPROM-Adresse sichern
	mov	r16, r17		;EEPROM-Daten kopieren
	ldi	r17, 1			;LCD in Datenmodus setzen
	rcall	lcdw40			;Zeichen ausgeben
	pop	r16			;EEPROM-Adresse zurckholen
	inc	r16			;nchste Adresse
	dec	r19			;alle Zeichen ausgegeben?
	brne	ma1410			;nein -> Schleife
;
	ldi	r19, 5			;5 Zeichen ausgeben (Sensorwert)
	ldi	yl, low(senbuf)		;Zeiger auf Sensorwerte
	add	yl, r21			;Tabellenoffset addieren
ma1420:	ld	r16, y+			;Zeichen holen, Adresse erhhen
	cpi	r16, '.'		;Trennzeichen?
	brne	ma1430			;nein -> weiter
	ldi	r16, epoint		;sonst EEPROM-Adresse Trennzeichen
	rcall	eeread			;Trennzeichen lesen
	mov	r16, r17		;und kopieren
	ldi	r17, 1			;LCD in Datenmodus setzen
ma1430:	rcall	lcdw40			;Zeichen ausgeben
	dec	r19			;alle Zeichen ausgegeben?
	brne	ma1420			;nein -> Schleife
	cpi	r16, ' '		;letztes Zeichen = Leerzeichen (kein Sensorwert)?
	breq	ma1470			;ja -> kein Sonderzeichen ausgeben
;
	ldi	yl, low(sentab)		;sonst Zeiger auf Sensor-Tabelle
	add	yl, r20			;Sensorzhler addieren
	ld	r16, y			;Sensornummer holen
	cpi	r16, 28			;Temperatursensor?
	brcs	ma1460			;ja -> passendes Grad-Symbol ausgeben
	breq	ma1440			;Luftdrucksensor? ja -> Tendenz-Symbol ausgeben
	ldi	r16, 3			;sonst Prozentzeichen fr Luftfeuchtigkeitssensor
	rjmp	ma1470			;ausgeben
;
ma1440:	ldi	r16, ' '		;kein Tendenz-Symbol voreinstellen
	lds	r18, prtend		;Luftdruck-Tendenz holen
	cpi	r18, 1			;Tendenz fallend?
	brne	ma1450			;nein -> weiter testen
	ldi	r16, 4			;sonst Symbol fr fallende Tendenz laden
	rjmp	ma1470			;ausgeben
;
ma1450:	cpi	r18, 2			;Tendenz steigend?
	brne	ma1470			;nein -> kein Symbol ausgeben
	ldi	r16, 5			;sonst Symbol fr steigende Tendenz laden
	rjmp	ma1470			;ausgeben
;
ma1460:	clr	r16			;Sonderzeichen 0 (Grad)
	ldi	yl, low(temlim)		;Zeiger auf Limit-Tabelle
	add	yl, r20			;Tabellenplatz berechnen
	ld	r15, y			;Limit-Flag holen
	tst	r15			;Limit-Unter/berschreitung?
	breq	ma1470			;nein -> berspringen
	mov	r16, r15		;sonst Limit-Flags holen
	andi	r16, 3			;Flag entspricht Sonderzeichen-Code
	mov	r18, seccnt		;Sekundenzhler holen
	andi	r18, 0x01		;Sekunde geradzahlig?
	breq	ma1470			;ja -> Limit-Sonderzeichen ausgeben
	clr	r16			;sonst nur Grad-Zeichen ausgeben
ma1470:	rcall	lcdw40			;ausgeben
;
ma1480:	inc	r20			;Zhler auf nchsten Sensor
	ldi	r16, 8			;Schleife fr 8 Sensoren
	lds	r17, semode		;Sensor-Modus holen
	tst	r17			;8-Sensor-Modus?
	brne	ma1490			;ja -> weiter
	ldi	r16, 4			;sonst Schleife fr 4 Sensoren
ma1490:	cp	r20, r16		;alle Tabellenpltze ausgegeben?
	breq	ma1500			;ja -> Ende
	rjmp	ma1220			;sonst Schleife
ma1500:	cbr	flags, 1 << 0		;nderungs-Flag lschen
	rjmp	ma1600
;
ma1510:	bst	xflags, 0		;Fehler-Quittungs-Flag gesetzt?
	brts	ma1600			;ja -> Fehlertext nicht neu ausgeben
	set
	bld	xflags, 0		;sonst Fehler-Quittungs-Flag setzen
	ldi	r16, 0x01		;Befehl: Anzeige lschen
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdout			;ausgeben
	rcall	w15ms			;Warteschleife 15,2ms
	ldi	r16, 1
	rcall	lcdadr			;LCD-Adresse der Zeile 2 ermitteln
	subi	r16, -1			;Anzeigeposition 1 setzen
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdw40			;Adresse ausgeben
	ldi	r19, 18			;18 Zeichen ausgeben
	ldi	zl, low(2 * msgerr)
	ldi	zh, high(2 * msgerr)	;Zeiger auf Text setzen
ma1520:	lpm	r16, z+			;Zeichen aus Programmspeicher holen
	ldi	r17, 1			;LCD in Datenmodus setzen
	rcall	lcdw40			;Zeichen ausgeben
	dec	r19			;alle Zeichen ausgegeben?
	brne	ma1520			;nein -> Schleife
;
; Timeouts fr Sensoren bearbeiten
;
ma1600:	ldi	yl, low(sentio)		;Zeiger auf Timeout-Zhler
	clr	r18			;aktuelle Sensornummer
ma1610:	ld	r19, y			;Timeout-Zhler holen
	tst	r19			;Zhler = 0?
	brne	ma1630			;nein -> berspringen
	ldi	r17, 5			;sonst 5 Leerzeichen ausgeben
	ldi	zl, senbuf
	clr	zh			;Zeiger auf Sensorwerte
	mov	r16, r18		;Sensornummer holen
	add	r16, r16		;verdoppeln
	add	r16, r16		;nochmals verdoppeln (x 4)
	add	r16, r18		;insgesamt x 5 als Offset
	add	zl, r16			;fr Sensor-Tabelle addieren
	ldi	r16, ' '		;Leerzeichen laden
ma1620:	st	z+, r16			;und in Werte-Tabelle speichern
	dec	r17			;alle Zeichen ausgegeben?
	brne	ma1620			;nein -> Schleife
	dec	r19			;Timeout-Zhler auf 255 setzen
	sbr	flags, 1 << 0		;nderungs-Flag setzen
	st	y, r19			;Timeout-Zhler wieder speichern
ma1630:	inc	yl			;Zeiger auf nchsten Tabellenplatz
	inc	r18			;nchsten Sensor bearbeiten
	cpi	r18, 8			;alle Sensoren bearbeitet?
	brne	ma1610			;nein -> Schleife
;
; Timeouts fr Alarme bearbeiten
;
	ldi	yl, low(almbuf)
	ldi	yh, high(almbuf)	;Zeiger auf Alarm-Puffer
	ldi	zl, low(almtio)
	ldi	zh, high(almtio)	;Zeiger auf Alarm-Timeout-Zhler
	ldi	r17, 4			;4 Alarme prfen
;
ma1700:	ld	r16, y+			;Alarm-Zustand holen
	tst	r16			;Alarm aktiv?
	breq	ma1710			;nein -> weiter
	ldi	r16, timalm		;sonst Alarm-Anzeigezeit holen
	st	z, r16			;Alarm-Timeout-Zhler neu setzen
ma1710:	adiw	zl, 1			;Zeiger auf nchsten Timeout-Zhler
	dec	r17			;alle Alarme bearbeitet?
	brne	ma1700			;nein -> Schleife
;
; Timeout fr Meldungen bearbeiten
;
	tst	msgtio			;Meldungs-Timeout-Zhler = 0?
	brne	ma1720			;nein -> berspringen
	dec	msgtio			;sonst Meldungs-Timeout inaktiv setzen
	sbr	flags, (1 << 4) | (1 << 0)	;Fehler- und nderungs-Flag setzen
	clt
	bld	xflags, 0		;Fehler-Quittungs-Flag lschen
;
; Timeout fr Anzeige-Rckstellung auf normale Sensoranzeige bearbeiten
;
ma1720:	mov	r16, distio		;sonst Anzeige-Timeout-Zhler holen
	tst	distio			;Anzeige-Timeout-Zhler = 0?
	brne	ma1800			;nein -> berspringen
	dec	distio			;sonst Anzeige-Timeout inaktiv setzen
	cbr	flags, 1 << 7		;Anzeige der Sensorbelegung abschalten
	sbr	flags, 1 << 0		;nderungs-Flag setzen
;
; Buzzer entsprechend der Alarmnummer steuern:
; 4x 80ms Tonfolge pro Sekunde (Alarm 1)
; 3x 80ms Tonfolge pro Sekunde (Alarm 2)
; 2x 80ms Tonfolge pro Sekunde (Alarm 3)
; 1x 80ms Tonfolge pro Sekunde (Alarm 4)
; 2x 20ms Tonfolge pro Sekunde (Temperatur-Alarm)
;
ma1800:	mov	r16, buzcn1		;Buzzer-Timeout-Zhler 1 holen
	cpi	r16, 0xff		;Timeout-Zhler aktiv?
	breq	ma1870			;nein -> berspringen
	cpi	r16, 0			;Zhler = 0?
	brne	ma1810			;nein -> bearbeiten
	dec	buzcn1			;sonst Timeout-Zhler auf 255 setzen
	cbi	PORTD, buzzer		;Buzzer abschalten
	rjmp	ma2000
;
ma1810:	mov	r19, lastal		;letzte Alarmnummer holen
	ldi	r18, 32			;Tonfolgenlnge setzen (640ms)
ma1820:	tst	r19			;enspricht Alarm = Tonfolge?
	breq	ma1830			;ja -> Tonausgabe
	subi	r18, 8			;sonst Tonfolgenlnge -160ms
	dec	r19			;nchste Alarmnummer
	rjmp	ma1820			;Schleife
;
ma1830:	ldi	r17, timbuz		;Buzzer-Timeout-Wert holen
	sub	r17, r16		;aktuellen Zhlerwert subtrahieren
ma1840:	cpi	r17, 50			;Ergebnis < 50 (< 1 Sekunde)?
	brcs	ma1850			;ja -> Ton ausgeben
	subi	r17, 50			;sonst Wert um 50 vermindern
	rjmp	ma1840			;Schleife
;
ma1850:	cp	r17, r18		;Wert > Tonfolgenlnge?
	brcc	ma1860			;ja -> Buzzer ausschalten
	bst	r17, 2			;Wert 4-7, 12-15, 20-23 usw.?
	brts	ma1860			;ja -> Buzzer ausschalten
	sbi	PORTD, buzzer		;sonst Buzzer einschalten
	rjmp	ma2000
ma1860:	cbi	PORTD, buzzer		;Buzzer ausschalten
	rjmp	ma2000
;
ma1870:	mov	r16, buzcn2		;Buzzer-Timeout-Zhler 2 holen
	cpi	r16, 0xff		;Timeout-Zhler aktiv?
	breq	ma2000			;nein -> berspringen
	cpi	r16, 0			;Zhler = 0?
	brne	ma1880			;nein -> bearbeiten
	dec	buzcn2			;sonst Timeout-Zhler auf 255 setzen
	cbi	PORTD, buzzer		;Buzzer abschalten
	rjmp	ma2000
;
ma1880:	ldi	r17, timbuz		;Buzzer-Timeout-Wert holen
	sub	r17, r16		;aktuellen Zhlerwert subtrahieren
ma1890:	cpi	r17, 25			;Ergebnis < 25 (< 0,5 Sekunden)?
	brcs	ma1900			;ja -> Ton ausgeben
	subi	r17, 25			;sonst Wert um 25 vermindern
	rjmp	ma1890			;Schleife
;
ma1900:	cpi	r17, 0			;Wert = 0 (Beginn einer 1/2 Sek)?
	brne	ma1910			;nein -> Buzzer ausschalten
	sbi	PORTD, buzzer		;sonst Buzzer einschalten
	rjmp	ma2000
ma1910:	cbi	PORTD, buzzer		;Buzzer ausschalten
;
; Empfangspuffer lesen
;
ma2000:	rcall	getchr			;Zeichen aus Empfangspuffer holen
	tst	r17			;neues Zeichen vorhanden?
	brne	ma2010			;nein -> weiter
	cpi	r16, lf			;LF (10) empfangen?
	breq	ma2010			;ja -> ignorieren
	cpi	r16, cr			;CR (13) empfangen?
	breq	ma2020			;ja -> Puffer auswerten
;
	mov	yl, rspoin		;Zeiger auf Textpuffer
	cpi	yl, low(rstext + 8)	;Puffer voll?
	breq	ma2010			;ja -> Zeichen verwerfen
	st	y+, r16			;sonst Zeichen im Puffer ablegen
	mov	rspoin, yl		;Zeiger wieder speichern
ma2010:	rjmp	main			;Schleife
;
ma2020:	mov	r16, rspoin		;Textpuffer-Zeiger holen
	cpi	r16, rstext + 7		;Sensor-Daten empfangen?
	breq	ma2100			;ja -> bearbeiten
	cpi	r16, rstext + 3		;Alarm-Daten empfangen?
	brne	ma2030			;nein -> weiter testen
	rjmp	ma3000			;sonst Alarm bearbeiten
ma2030:	cpi	r16, rstext + 8		;Sensorbelegung empfangen?	
	brne	ma2040			;nein -> Ende
	rjmp	ma4000			;sonst Sensorbelegung bearbeiten
;
ma2040:	ldi	r16, rstext		;Pointer fr Textpuffer
	mov	rspoin, r16		;wieder auf Anfang setzen
	rjmp	main			;Schleife
;
; Sensorkennung auf Plausibilitt prfen und Sensortyp ermitteln
;
ma2100:	ldi	yl, low(rstext)		;Zeiger auf Textpuffer
	ld	r16, y+			;erstes Zeichen holen
	cpi	r16, '1'		;Zeichen < "1"?
	brcs	ma2040			;ja -> Fehler
	cpi	r16, 'z' + 1		;Zeichen > "h"?
	brcc	ma2040			;ja -> Fehler
	cpi	r16, '9'		;Zeichen < "9"?
	brcs	ma2110			;ja -> weiter (Sensor 1-8)
	cpi	r16, 'a'		;Zeichen < "a"?
	brcs	ma2040			;ja -> Fehler
	cpi	r16, 't'		;Zeichen < "t"?
	brcs	ma2110			;ja -> weiter (Sensor a-s)
	cpi	r16, 'w'		;Zeichen < "w"?
	brcs	ma2040			;ja -> Fehler
	rjmp	ma2200			;sonst Luftdruck- oder Luftfeuchtigkeitssensor
;
; Temperaturdaten auf Plausibilitt prfen
;
ma2110:	ld	r16, y+			;zweites Zeichen holen
	cpi	r16, ':'		;Doppelpunkt?
	brne	ma2040			;nein -> Fehler
	ld	r16, y+			;drittes Zeichen holen
	cpi	r16, ' '		;Leerzeichen?
	breq	ma2120			;ja -> weiter
	cpi	r16, '-'		;Minus-Zeichen?
	breq	ma2120			;ja -> weiter
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma2040			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcc	ma2040			;ja -> Fehler
ma2120:	ld	r16, y+			;viertes Zeichen holen
	cpi	r16, ' '		;Leerzeichen?
	breq	ma2130			;ja -> weiter
	cpi	r16, '-'		;Minus-Zeichen?
	breq	ma2130			;ja -> weiter
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma2040			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcc	ma2040			;ja -> Fehler
ma2130:	ld	r16, y+			;fnftes Zeichen holen
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma2040			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcc	ma2040			;ja -> Fehler
	ld	r16, y+			;sechstes Zeichen holen
	cpi	r16, '.'		;Punkt?
	breq	ma2140			;ja -> weiter
	cpi	r16, ','		;Komma?
	brne	ma2040			;nein -> Fehler
ma2140:	ld	r16, y+			;siebentes Zeichen holen
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma2040			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcs	ma2400			;nein -> Sensornummer prfen
ma2150:	rjmp	ma2040			;ja -> Fehler
;
; Luftdruckwert auf Plausibilitt prfen
;
ma2200:	cpi	r16, 'w'		;Luftdrucksensor?
	brne	ma2300			;nein -> weiter mit Luftfeuchtigkeitssensor
;
	ld	r16, y+			;zweites Zeichen holen
	cpi	r16, ':'		;Doppelpunkt?
	brne	ma2150			;nein -> Fehler
	ld	r16, y+			;drittes Zeichen holen
	cpi	r16, ' '		;Leerzeichen?
	brne	ma2150			;nein -> Fehler
	ld	r16, y+			;viertes Zeichen holen
	cpi	r16, ' '		;Leerzeichen?
	breq	ma2210			;ja -> weiter
	cpi	r16, '1'		;Zeichen = "1"?
	brne	ma2150			;nein -> Fehler
ma2210:	ld	r16, y+			;fnftes Zeichen holen
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma2150			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcc	ma2150			;ja -> Fehler
	ld	r16, y+			;sechstes Zeichen holen
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma2150			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcc	ma2150			;ja -> Fehler
	ld	r16, y+			;siebtes Zeichen holen
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma2150			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcc	ma2150			;ja -> Fehler
	rjmp	ma2400			;sonst Sensornummer prfen
;
; Luftfeuchtigkeitswert auf Plausibilitt prfen
;
ma2300: ld	r16, y+			;zweites Zeichen holen
	cpi	r16, ':'		;Doppelpunkt?
	brne	ma2150			;nein -> Fehler
	ld	r16, y+			;drittes Zeichen holen
	cpi	r16, ' '		;Leerzeichen?
	brne	ma2150			;nein -> Fehler
	ld	r16, y+			;viertes Zeichen holen
	cpi	r16, ' '		;Leerzeichen?
	brne	ma2150			;nein -> Fehler
	ld	r16, y+			;fnftes Zeichen holen
	cpi	r16, ' '		;Leerzeichen?
	breq	ma2310			;ja -> weiter
	cpi	r16, '1'		;Zeichen = "1"?
	brne	ma2150			;nein -> Fehler
ma2310:	ld	r16, y+			;sechstes Zeichen holen
	cpi	r16, ' '		;Leerzeichen?
	breq	ma2320			;ja -> weiter
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma2150			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcc	ma2150			;ja -> Fehler
ma2320:	ld	r16, y+			;siebentes Zeichen holen
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma2150			;ja -> Fehler
	cpi	r16, '9' + 1		;Zeichen > "9"?
	brcc	ma2150			;ja -> Fehler
;
; prfen, ob Sensornummer in Anzeige-Tabelle vorhanden ist
;
ma2400:	ldi	yl, low(rstext)		;Zeiger auf Textpuffer
	ld	r16, y+			;erstes Zeichen holen
	cpi	r16, '9'		;Sensorgruppe 1-8?
	brcc	ma2410			;nein -> weiter
	subi	r16, '1'		;in Sensornummer 0-7 umwandeln
	rjmp	ma2420
ma2410:	subi	r16, 'a' - 8		;in Sensornummer 8-26 umwandeln
	cpi	r16, 27			;Sensornummer > 26 (w-z)?
	brcs	ma2420			;nein -> weiter
	subi	r16, 3			;sonst Sensornummer korrigieren
;
ma2420:	ldi	zl, low(sentab)
	ldi	zh, high(sentab)	;Zeiger auf Anzeige-Tabelle
	clr	r18			;Tabellenplatz-Zhler
ma2430:	ld	r17, z			;Tabelle lesen
	dec	r17			;Wert korrigieren
	cp	r16, r17		;Sensornummer gefunden?
	breq	ma2500			;ja -> weiter
	adiw	zl, 1			;Zeiger auf nchsten Tabellenplatz
	inc	r18			;nchster Tabellenplatz
	cpi	r18, 8			;alle Tabellenpltze gelesen?
	brne	ma2430			;nein -> Schleife
	rjmp	ma2040			;sonst Datenpaket verwerfen

; Sensor-Daten sind ok, in Werte-Puffer speichern

ma2500:	mov	r22, r16		;Sensornummer zwischenspeichern
	mov	r20, r18		;Tabellenplatz zwischenspeichern
	adiw	yl, 1			;Zeiger auf Text setzen
	ldi	r17, 5			;5 Textzeichen kopieren/ausgeben
	ldi	zl, low(senbuf)
	ldi	zh, high(senbuf)	;Zeiger auf Sensorwerte
	add	r18, r18		;Sensornummer verdoppeln
	add	r18, r18		;nochmals verdoppeln (x4)
	add	r18, r20		;insgesamt x5 als Offset
	add	zl, r18			;Platz im Werte-Puffer ermitteln
	mov	r21, zl			;Zeiger auf Sensorwert sichern
ma2510:	ld	r16, y+			;Zeichen aus Textpuffer holen
	ld	r18, z			;alten Wert aus Puffer holen
	cp	r16, r18		;hat sich der Wert gendert?
	breq	ma2520			;nein -> weiter
	sbr	flags, 1 << 0		;sonst nderungs-Flag setzen
ma2520:	st	z+, r16			;neuen Wert speichern
	dec	r17			;alle Zeichen gespeichert?
	brne	ma2510			;nein -> Schleife
;
; Sensornummer prfen, Temperatur und Luftdruck filtern
;
	cpi	r22, 27			;Sensornummer = 27 (Luftdrucksensor)?
	breq	ma2610			;ja -> bearbeiten
	brcc	ma2600			;Sensornummer > 27 (Luftfeuchtigkeit)? ja -> Ende
	rjmp	ma2700			;sonst Temperatursensor, Limits prfen
ma2600:	rjmp	ma2760			;sonst Ende
;
; Inhalt des Luftdruckwerte-Puffers um einen Platz verschieben
;
ma2610:	bst	xflags, 1		;Stunden-Flag gesetzt?
	brtc	ma2600			;nein -> Ende
	clt
	bld	xflags, 1		;sonst Stunden-Flag wieder lschen
;
	mov	zl, r21			;Zeiger auf Luftdruckwert holen
	rcall	prs_hx			;Luftdruckwert in Hex wandeln
	ldi	zl, low(prsbuf + 15 * 2)	;Zeiger auf letzten Platz
	ldi	zh, high(prsbuf + 15 * 2)	;des Luftdruckwerte-Puffers
	clr	r21			;Summen-Zhler lschen
	ld	r16, z			;Pufferwert Low holen
	ldd	r17, z + 1		;Pufferwert High holen
	or	r21, r16		;Low-Wert mit Summenzhler verknpfen
	or	r21, r17		;High-Wert mit Summenzhler verknpfen
	ldi	r22, 15			;15 Pufferpltze bearbeiten
ma2620:	sbiw	zl, 2			;Zeiger auf vorherigen Pufferplatz
	ld	r16, z			;Pufferwert Low holen
	ldd	r17, z + 1		;Pufferwert High holen
	or	r21, r16		;Low-Wert mit Summenzhler verknpfen
	or	r21, r17		;High-Wert mit Summenzhler verknpfen
	std	z + 2, r16		;Low-Wert auf nchsten Pufferplatz legen
	std	z + 3, r17		;High-Wert auf nchsten Pufferplatz legen
	dec	r22			;alle Pufferpltze bearbeitet?
	brne	ma2620			;nein -> Schleife
;
; aktuellen Luftdruckwert in Puffer speichern, gegebenenfalls leere Pufferpltze initialisieren
;
	st	z, r18			;neuen Luftdruckwert Low auf ersten Pufferplatz legen
	std	z + 1, r19		;neuen Luftdruckwert High auf ersten Pufferplatz legen
	tst	r21			;Summenzhler = 0 (Puffer noch leer)?
	brne	ma2640			;nein -> Tendenz ermitteln
	ldi	r22, 15			;sonst Puffer komplett mit neuem Luftdruckwert fllen
ma2630:	adiw	zl, 2			;Zeiger auf nchsten Pufferplatz
	st	z, r18			;neuen Luftdruckwert Low speichern
	std	z + 1, r19		;neuen Luftdruckwert High speichern
	dec	r22			;alle Pufferpltze geschrieben?
	brne	ma2630			;nein -> Schleife
;
; Luftdruck-Tendenz ermitteln
;
ma2640:	ldi	zl, low(prsbuf + 2)	;Zeiger auf zweiten Platz
	ldi	zh, high(prsbuf + 2)	;des Luftdruckwerte-Puffers
	ldi	r22, 15			;15 Pufferpltze bearbeiten
	clr	r16
	sts	prtend, r16		;Luftdruck-Tendenz zunchst lschen
ma2650:	ld	r16, z			;Pufferwert Low holen
	ldd	r17, z + 1		;Pufferwert High holen
	sub	r16, r18
	sbc	r17, r19		;Differenz zum aktuellen Wert ermitteln
	lds	r15, prs_th		;Schwellwert fr Tendenz-Berechnung holen
	clr	r21			;High-Byte fr Vergleich = 0 setzen
	cp	r16, r15
	cpc	r17, r21		;Differenz >= Schwellwert?
	brmi	ma2660			;nein -> weiter testen
;
	ldi	r16, 1
	sts	prtend, r16		;Luftdruck-Tendenz fallend setzen
	rjmp	ma2760
;
ma2660:	neg	r15			;Schwellwert negieren
	inc	r15			;korrigieren
	ser	r21			;H-Byte fr Vergleich = -1 setzen
	cp	r16, r15
	cpc	r17, r21		;Differenz <= Schwellwert?
	brmi	ma2670			;ja -> Tendenz steigend setzen
	adiw	zl, 2			;sonst Zeiger auf nchsten Pufferplatz
	dec	r22			;alle Pufferpltze bearbeitet?
	brne	ma2650			;nein -> Schleife
	rjmp	ma2760
;
ma2670:	ldi	r16, 2
	sts	prtend, r16		;Luftdruck-Tendenz steigend setzen
	rjmp	ma2760
;
; Temperatur auf Limit-ber- oder Unterschreitung prfen
;
ma2700:	bst	flags, 0		;nderungs-Flag gesetzt?
	brtc	ma2760			;nein -> weiter
	ldi	yl, low(temlim)		;Zeiger auf Limit-Flags
	add	yl, r20			;Tabellenplatz addieren
	mov	zl, r21			;Zeiger auf Temperaturwert holen
	rcall	tem_hx			;Temperaturwert in Hex wandeln
;
	ldi	r16, elim_l		;EEPROM-Adresse Minimalwerte
	add	r16, r20		;Tabellenplatz addieren
	rcall	eeread			;EERPOM lesen
	cpi	r17, 128		;Alarmberwachung aktiv?
	breq	ma2710			;nein -> Prfung berspringen
	cp	r18, r17		;Minimalwert unterschritten?
	brmi	ma2730			;ja -> auswerten
;
ma2710:	ldi	r16, elim_h		;EEPROM-Adresse Maximalwerte
	add	r16, r20		;Tabellenplatz addieren
	rcall	eeread			;EERPOM lesen
	cpi	r17, 128		;Alarmberwachung aktiv?
	breq	ma2720			;nein -> Prfung berspringen
	cp	r18, r17		;Maximalwert berschritten?
	brpl	ma2740			;ja -> auswerten
;	
ma2720:	clr	r16			;sonst Limit-Flags lschen
	rjmp	ma2750
;
ma2730:	ld	r16, y			;aktuelle Limit-Flags holen
	bst	r16, 0			;unteres Limit-Flag schon gesetzt?
	brts	ma2760			;ja -> berspringen
	ldi	r17, timbuz		;Timeout-Wert fr Buzzer
	mov	buzcn2, r17		;setzen
	ldi	r16, 1 << 0		;Limit-Flag fr Unterschreitung setzen
	rjmp	ma2750
ma2740:	ld	r16, y			;aktuelle Limit-Flags holen
	bst	r16, 1			;oberes Limit-Flag schon gesetzt?
	brts	ma2760			;ja -> berspringen
	ldi	r17, timbuz		;Timeout-Wert fr Buzzer
	mov	buzcn2, r17		;setzen
	ldi	r16, 1 << 1		;Limit-Flag fr berschreitung setzen
ma2750:	st	y, r16			;Limit-Flags wieder speichern
;
ma2760:	ldi	yl, low(sentio)		;Zeiger auf Timeout-Zhler
	add	yl, r20			;Tabellenplatz addieren
	ldi	r16, timsen		;Timeout-Zhler neu setzen
	st	y, r16			;und speichern
	ldi	r16, timmsg		;Meldungs-Timeout-Zhler neu
	mov	msgtio, r16		;setzen
	cbr	flags, 1 << 4		;Fehler-Flag lschen
	sbr	flags, 1 << 0		;nderungs-Flag setzen
	rjmp	ma2040			;Text-Zeiger zurcksetzen, Ende
;
; Alarmmeldung auf Plausibilitt prfen
;
ma3000:	ldi	yl, low(rstext)		;Zeiger auf Textpuffer
	ld	r16, y+			;erstes Zeichen holen
	cpi	r16, 'A'		;Zeichen < "A"?
	brcs	ma3100			;ja -> Fehler
	cpi	r16, 'D' + 1		;Zeichen > "D"?
	brcc	ma3100			;ja -> Fehler
	ld	r16, y+			;zweites Zeichen holen
	cpi	r16, ':'		;Doppelpunkt?
	brne	ma3100			;nein -> Fehler
	ld	r16, y+			;drittes Zeichen holen
	cpi	r16, '0'		;Zeichen = "0"?
	breq	ma3010			;ja -> bearbeiten
	cpi	r16, '1'		;Zeichen = "1"?
	brne	ma3100			;nein -> Fehler
;
; Alarm-Daten sind ok, in Alarm-Puffer speichern
;
ma3010:	ldi	yl, low(rstext)		;Zeiger auf Textpuffer
	ld	r16, y			;erstes Zeichen holen
	subi	r16, 'A'		;in Alarmnummer wandeln
	adiw	yl, 2			;Zeiger auf Text setzen
	ldi	zl, low(almbuf)
	ldi	zh, high(almbuf)	;Zeiger auf Alarm-Puffer
	add	zl, r16			;Tabellenplatz berechnen
	ld	r17, y			;Zeichen aus Textpuffer holen
	andi	r17, 0x01		;in Ziffernwert wandeln
	ldi	yl, low(alminv)		;Zeiger auf Alarm-Invertierung
	add	yl, r16			;Tabellenplatz berechnen
	ld	r18, y			;Invertierungswert holen
	eor	r17, r18		;bei Bedarf Alarm invertieren
	ld	r18, z			;alten Wert aus Tabelle holen
	cp	r17, r18		;hat sich der Wert gendert?
	breq	ma3020			;nein -> weiter
	cpi	r17, 1			;wurde Alarm ausgelst?
	brne	ma3020			;nein -> weiter
;
	cbr	flags, 1 << 6		;sonst Alarm-Unterdrckung ausschalten
	ldi	r18, timbuz		;Timeout-Wert fr Buzzer
	mov	buzcn1, r18		;setzen
	mov	lastal, r16		;letzte Alarmnummer speichern
;
ma3020:	st	z, r17			;neuen Alarm-Wert speichern
	ldi	r16, timmsg		;Meldungs-Timeout-Zhler neu
	mov	msgtio, r16		;setzen
	cbr	flags, 1 << 4		;Fehler-Flag lschen
	sbr	flags, 1 << 0		;nderungs-Flag setzen
ma3100:	rjmp	ma2040			;Text-Zeiger zurcksetzen, Ende
;
; Sensorbelegung auf Plausibilitt prfen
;
ma4000:	ldi	yl, low(rstext)		;Zeiger auf Textpuffer
	clr	r20			;Sensorzhler = 0 setzen
ma4010:	ld	r16, y+			;Zeichen aus Puffer holen
	cpi	r16, '0'		;Zeichen < "0"?
	brcs	ma3100			;ja -> Fehler
	cpi	r16, 'R' + 1		;Zeichen > "R"?
	brcc	ma3100			;ja -> Fehler

	subi	r16, '0'		;Zeichen in Ziffernwert wandeln
	cpi	r16, 10			;gltiger Wert 0-9?
	brcs	ma4020			;ja -> weiter
	subi	r16, 7			;sonst Buchstabe in Wert wandeln
	cpi	r16, 10			;gltiger Wert (> 9)?
	brcs	ma3100			;nein -> Fehler
ma4020:	add	r20, r16		;Wert zum Sensorzhler addieren
	cpi	yl, low(rstext + 8)	;Pufferende erreicht?
	brne	ma4010			;nein -> Schleife
;
; Sensorbelegung ist ok, Daten sofort auf LCD ausgeben
;
	ldi	r16, 0x01		;Befehl: Anzeige lschen
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdout			;ausgeben
	rcall	w15ms			;Warteschleife 15,2ms
	ldi	r16, 1
	rcall	lcdadr			;LCD-Adresse von Zeile 2 ermitteln
	subi	r16, -3			;Zeichenposition 3 setzen
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdw40			;Adresse ausgeben
	ldi	r19, 14			;14 Zeichen ausgeben
	ldi	zl, low(2 * sentxt)
	ldi	zh, high(2 * sentxt)	;Zeiger auf Text setzen
ma4030:	lpm	r16, z+			;Zeichen aus Programmspeicher holen
	ldi	r17, 1			;LCD in Datenmodus setzen
	rcall	lcdw40			;Zeichen ausgeben
	dec	r19			;alle Zeichen ausgegeben?
	brne	ma4030			;nein -> Schleife
;
	ldi	r16, 2
	rcall	lcdadr			;LCD Adresse von Zeile 3 ermitteln
	subi	r16, -3			;Zeichenposition 3 setzen
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdw40			;Adresse ausgeben
	ldi	yl, low(rstext)		;Zeiger auf Textpuffer
ma4040:	ld	r16, y+			;Zeichen aus Puffer holen
	ldi	r17, 1			;LCD in Datenmodus setzen
	rcall	lcdw40			;Zeichen ausgeben
	cpi	yl, low(rstext + 8)	;Pufferende erreicht?
	brne	ma4040			;nein -> Schleife
;
	ldi	r16, ' '		;Leerzeichen
	rcall	lcdw40			;ausgeben
	rcall	lcdw40			;nochmals ausgeben
	ldi	r16, '('		;"Klammer auf"
	rcall	lcdw40			;ausgeben
	cpi	r20, 28			;mehr als 27 Sensoren?
	brcc	ma4100			;ja -> Fragezeichen ausgeben
;
	ldi	r21, '0'		;Zehner auf "0" voreinstellen
ma4050:	cpi	r20, 10			;mehr als 10 Sensoren?
	brcs	ma4060			;nein -> Einer bearbeiten
	subi	r20, 10			;Sensorzhler - 10
	inc	r21			;Zehner erhhen
	rjmp	ma4050			;Schleife
;
ma4060:	mov	r16, r21		;Zehner kopieren
	rcall	lcdw40			;ausgeben
ma4070:	mov	r16, r20		;Einer kopieren
	ori	r16, '0'		;in ASCII wandeln
	rjmp	ma4110			;ausgeben
;
ma4100:	ldi	r16, '?'		;Fragezeichen
	rcall	lcdw40			;ausgeben
ma4110:	rcall	lcdw40			;nochmals ausgeben
	ldi	r16, ')'		;"Klammer zu"
	rcall	lcdw40			;ausgeben
;
	sbr	flags, 1 << 7		;Anzeige der Sensorbelegung aktivieren
	ldi	r16, timsem		;Timeout-Wert fr Sensorbelegung laden
	mov	distio, r16		;und Zhler neu setzen
	ldi	r16, timmsg
	mov	msgtio, r16		;Meldungs-Timeout-Zhler neu setzen
	cbr	flags, 1 << 4		;Fehler-Flag lschen
	rjmp	ma2040			;Text-Zeiger zurcksetzen, Ende
;
; Unterprogramm Multiplikation 8 Bit * 8 Bit (aus Atmel Application Note AVR200)
; Register: r19
;	    r16 <- Faktor 1
;	    r17 <- Faktor 2
;	    r17 -> Ergebnis Low
;	    r18 -> Ergebnis High
;	    
mul8x8:	clr	r18			;Ergebnis High-Byte lschen
	ldi	r19, 8			;8 Schleifen
	lsr	r17			;Faktor 2 rotieren
;
mul010:	brcc	mul020			;Carry-Flag gesetzt
	add 	r18, r16		;Faktor 1 zum High-Byte addieren
mul020:	ror	r18			;Ergebnis High-Byte nach rechts rotieren
	ror	r17			;Ergebnis Low-Byte nach rechts rotieren
	dec	r19			;alle Schleifen durchlaufen?
	brne	mul010			;nein -> Schleife
	ret
;
; Unterprogramm zum Ermitteln der LCD-Adresse
; Register: r17, z
;	    r16 <- Zeilennummer (0-7)
;	    r16 -> LCD-Adresse
;
lcdadr:	clr	r17			;Dummy fr Adress-Berechnung
	ldi	zl, low(2 * lcdtab)
	ldi	zh, high(2 * lcdtab)	;Zeiger auf LCD-Adress-Tabelle
	add	zl, r16			;Tabellenplatz berechnen
	adc	zh, r17			;bertrag addieren
	lpm	r16, z			;Tabelleneintrag holen
	ret
;
; Unterprogramm zum Auslesen eines Zeichens aus dem RS-232-Empfangspuffer
; Register: y
;	    r16 -> gelesenes Zeichen
;	    r17 -> 0 = Zeichen gltig, 0xff = kein Zeichen im Puffer
;
getchr:	cp	rxpoi1, rxpoi2		;sind Zeichen im Puffer?
	brne	get010			;ja -> weiter
	ser	r17			;sonst kein Zeichen im Puffer
	ret
;
get010:	cli				;Interrupts vorbergehend sperren
	ldi	yh, high(rxbuff)	;Pointer-High-Byte auf Empfangspuffer setzen
	mov	yl, rxpoi2		;Empfangspuffer-Pointer2 laden
	ld	r16, y+			;Zeichen laden, Pointer erhhen
	andi	yl, 0x3f		;bertrag korrigieren
	mov	rxpoi2, yl		;Pointer2 wieder speichern
	ldi	yh, high(rstext)	;Pointer High-Byte fr Variablen setzen
	sei				;Interrupts wieder freigeben
	clr	r17			;Zeichen ist gltig
	ret
;
; Unterprogramm zur LCD-Ausgabe eines Zeichens
; Register: r16 <- LCD-Daten/Befehlsbyte
;	    r17 <- 0 = Befehlsbyte, 1 = Datenbyte
;
lcdout:	out	PORTB, r16		;Daten/Befehlsbyte ausgeben
	tst	r17			;Datenbyte?
	brne	lcd010			;ja -> weiter
	cbi	PORTD, 2		;sonst LCD in Befehlsmodus setzen
	rjmp	lcd020
lcd010:	sbi	PORTD, 2		;LCD in Datenmodus setzen
lcd020:	nop
	sbi	PORTD, 3		;E auf High setzen
	nop
	cbi	PORTD, 3		;E wieder auf Low setzen
	ret				;(Datenbernahme)
;
; Unterprogramm zur LCD-Ausgabe eines Zeichens und anschlieende Warteschleife 40s
; Register: r16 <- LCD-Daten/Befehlsbyte
;	    r17 <- 0 = Befehlsbyte, 1 = Datenbyte
;
lcdw40:	rcall	lcdout			;Zeichen ausgeben und
	rjmp	w040my			;Warteschleife 40s
;
; Unterprogramm Warteschleife ca. 50ms
; Register: r18 <- Zhler High
;
w50ms:	ldi	r18, 0			;High-Zhler setzen
	clr	delayc			;Low-Zhler setzen
w50ms1:	dec	delayc			;Low-Zhler - 1
	brne	w50ms1			;Schleife
	dec	r18			;High-Zhler - 1
	brne	w50ms1			;Schleife
	ret
;
; Unterprogramm Warteschleife ca. 15,2ms
; Register: r18 <- Zhler High
;
w15ms:	ldi	r18, 243
	mov	delayc, r18		;Low-Zhler setzen
	ldi	r18, 79			;High-Zhler setzen
	rjmp	w50ms1
;
; Unterprogramm Warteschleife ca. 4,1ms
; Register: r18 <- Zhler High
;
w04ms:	ldi	r18, 73
	mov	delayc, r18		;Low-Zhler setzen
	ldi	r18, 22			;High-Zhler setzen
	rjmp	w50ms1
;
; Unterprogramm Warteschleife ca. 1.64ms
; Register: r18 <- Zhler High
;
w02ms:	ldi	r18, 130
	mov	delayc, r18		;Low-Zhler setzen
	ldi	r18, 9			;High-Zhler setzen
	rjmp	w50ms1
;
; Unterprogramm Warteschleife ca. 100s
; Register: r18 <- Zhler High
;
w100my:	ldi	r18, 129
	mov	delayc, r18		;Low-Zhler setzen
	ldi	r18, 1			;High-Zhler setzen
	rjmp	w50ms1
;
; Unterprogramm Warteschleife ca. 40s
; Register: r18 <- Zhler High
;
w040my:	ldi	r18, 49
	mov	delayc, r18		;Low-Zhler setzen
	ldi	r18, 1			;High-Zhler setzen
	rjmp	w50ms1
;
; Unterprogramm zum Lesen eines Byte aus dem EEPROM
; Register: r16 <- EEPROM-Adresse
;	    r17 -> EEPROM-Daten
;
eeread:	sbic	EECR, EEPE		;luft ein Schreibzyklus?
	rjmp	eeread			;ja -> warten
	out	EEAR, r16		;EEPROM-Adresse setzen
	sbi	EECR, EERE		;EEPROM lesen aktivieren
	in	r17, EEDR		;Daten lesen
	ret
;
; Unterprogramm zur Konvertierung eines Temperaturwertes von ASCII in einen Hexadezimalwert (ohne Nachkommastelle)
; Register: r16
;	    z   <- Zeiger auf Temperaturwert (ASCII)
;	    r18	-> Temperaturwert in Hex
;
tem_hx:	clr	r18			;Startwert = 0 setzen
	ld	r16, z			;Vorzeichen/Hunderter holen
	cpi	r16, '1'		;Ziffer 1?
	brne	tem010			;nein -> weiter
	ldi	r18, 100		;sonst Startwert = 100 setzen
tem010:	ldd	r16, z + 1		;Zehner holen
	cpi	r16, ' '		;Leerzeichen?
	breq	tem030			;ja -> weiter zum Einer
	cpi	r16, '-'		;Minus-Zeichen?
	breq	tem030			;ja -> weiter zum Einer
	subi	r16, 0x30		;ASCII in Zahl umwandeln
tem020:	tst	r16			;Zahl = 0?
	breq	tem030			;ja -> weiter zum Einer
	subi	r18, -10		;sonst 10 addieren
	dec	r16			;Zhler vermindern
	rjmp	tem020			;Schleife
;
tem030:	ldd	r16, z + 2		;Einer holen
	subi	r16, 0x30		;ASCII in Zahl umwandeln
	add	r18, r16		;Einer addieren
	ld	r16, z			;nochmals Vorzeichen/Hunderter holen
	cpi	r16, '-'		;Minus-Zeichen?
	brne	tem060			;nein -> weiter
tem040:	neg	r18			;sonst Zweierkomplement bilden
	ldd	r16, z + 4		;Zehntel holen
	cpi	r16, '0'		;Zehntel = 0?
	breq	tem050			;ja -> Ergebnis ok
	dec	r18			;sonst Ergebnis korrigieren
tem050:	ret				;Ende
;
tem060:	ldd	r16, z + 1		;nochmals Zehner holen
	cpi	r16, '-'		;Minus-Zeichen?
	breq	tem040			;ja -> Zweierkomplement
	ret				;sonst Ende
;
; Unterprogramm zur Konvertierung eines Luftdruckwertes von ASCII in einen Hexadezimalwert
; Register: r16
;	    z   <- Zeiger auf Luftdruckwert (ASCII)
;	    r18	-> Luftdruckwert in Hex (Low)
;	    r19	-> Luftdruckwert in Hex (High)
;
prs_hx:	clr	r18
	clr	r19			;Startwert = 0 setzen
	ldd	r16, z + 1		;Tausender holen
	cpi	r16, '1'		;Ziffer 1?
	brne	prs010			;nein -> weiter
	ldi	r18, low(1000)
	ldi	r19, high(1000)		;sonst Startwert = 1000 setzen
prs010:	ldd	r16, z + 2		;Hunderter holen
	subi	r16, 0x30		;ASCII in Zahl umwandeln
prs020:	tst	r16			;Zahl = 0?
	breq	prs030			;ja -> weiter zum Zehner
	subi	r18, -100		;sonst 100 addieren
	sbci	r19, -1			;bertrag addieren
	dec	r16			;Zhler vermindern
	rjmp	prs020			;Schleife
;
prs030:	ldd	r16, z + 3		;Zehner holen
	subi	r16, 0x30		;ASCII in Zahl umwandeln
prs040:	tst	r16			;Zahl = 0?
	breq	prs050			;ja -> weiter zum Einer
	subi	r18, -10		;sonst 10 addieren
	sbci	r19, -1			;bertrag addieren
	dec	r16			;Zhler vermindern
	rjmp	prs040			;Schleife
;
prs050:	ldd	r16, z + 4		;Einer holen
	subi	r16, 0x30		;ASCII in Zahl umwandeln
	add	r18, r16		;Einer addieren
	clr	r16
	adc	r19, r16		;bertrag addieren
	ret				;Ende
;
; Interrupt-Routine "UART Empfang komplett"; empfangenes Byte im Ringpuffer speichern
; Register: x
;
rxcint:	in	insreg, SREG		;SREG sichern
	in	itemp1, UDR		;empfangenes Byte lesen
	ldi	xh, high(rxbuff)	;Pointer High-Byte auf Empfangspuffer setzen
	mov	xl, rxpoi1		;Empfangspuffer-Pointer1 laden
	st	x+, itemp1		;Byte speichern, Pointer erhhen
	andi	xl, 0x3f		;bertrag korrigieren
	mov	rxpoi1, xl		;Pointer1 wieder speichern
	out	SREG, insreg		;SREG wiederherstellen
	reti
;
; Interrupt-Routine bei Timer1 Compare Match A, wird alle 20ms ausgefhrt
; Register: x
;
; Taster lesen und Flags setzen
;
timer1:	in	insreg, SREG		;SREG sichern
	in	itemp1, PIND		;Taster lesen
	bst	itemp1, 1		;Taster gedrckt?
	brts	tim020			;nein -> weiter
	bst	flags, 1		;war Taster bereits gedrckt?
	brtc	tim010			;nein -> Tastenflag 1 setzen
	sbr	flags, 1 << 2		;sonst Tastenflag 2 setzen
	rjmp	tim100
tim010:	sbr	flags, 1 << 1		;Tastenflag 1 setzen
	rjmp	tim100
;
tim020:	bst	flags, 1		;war Taster vorher gedrckt?
	brtc	tim030			;nein -> Tastenflag 2 lschen
	cbr	flags, 1 << 1		;sonst Tastenflag 1 lschen
	rjmp	tim100
tim030:	cbr	flags, (1 << 3) | (1 << 2)	;Tastenflags 2 und 3 lschen
;
; Timeout 1 und 2 fr Buzzer steuern
;
tim100:	mov	itemp1, buzcn1		;Timeout-Zhler holen
	tst	itemp1			;Zhlerwert = 0?
	breq	tim110			;ja -> berspringen
	cpi	itemp1, 0xff		;Zhlerwert = 255?
	breq	tim110			;ja -> berspringen
	dec	itemp1			;sonst Zhlerwert - 1
	mov	buzcn1, itemp1		;Zhlerwert wieder speichern
;
tim110:	mov	itemp1, buzcn2		;Timeout-Zhler holen
	tst	itemp1			;Zhlerwert = 0?
	breq	tim120			;ja -> berspringen
	cpi	itemp1, 0xff		;Zhlerwert = 255?
	breq	tim120			;ja -> berspringen
	dec	itemp1			;sonst Zhlerwert - 1
	mov	buzcn2, itemp1		;Zhlerwert wieder speichern
;
; Vorteiler bearbeiten (Teilung auf 1Hz / 1s)
;
tim120:	inc	precnt			;Vorteiler erhhen
	mov	itemp1, precnt		;Vorteiler kopieren
	cpi	itemp1, 50		;Endwert erreicht?
	breq	tim130			;ja -> weiter
	rjmp	timend			;sonst Ende
;
tim130:	clr	precnt			;Vorteiler zurcksetzen
	inc	seccnt			;und Sekundenzhler erhhen
	sbr	flags, 1 << 0		;nderungs-Flag setzen (Blinken und Alarmwechsel)
;
; Timeouts fr Sensoren, Alarme, Meldungen und Anzeige steuern (wird alle 1s ausgefhrt)
;
	ldi	xl, low(sentio)
	ldi	xh, high(sentio)	;Zeiger auf Sensor-Timeout-Werte
tim200:	ld	itemp1, x		;Zhlerwert holen
	tst	itemp1			;Zhlerwert = 0?
	breq	tim210			;ja -> berspringen
	cpi	itemp1, 0xff		;Zhlerwert = 255?
	breq	tim210			;ja -> berspringen
	dec	itemp1			;sonst Zhlerwert - 1
tim210:	st	x+, itemp1		;Zhlerwert wieder speichern
	cpi	xl, low(sentio + 8)	;alle Werte bearbeitet?
	brne	tim200			;nein -> Schleife
;
	ldi	xl, low(almtio)
	ldi	xh, high(almtio)	;Zeiger auf Alarm-Timeout-Zhler
	clr	itemp2			;Zhler fr aktive Alarme lschen
tim220:	ld	itemp1, x		;Zhlerwert holen
	tst	itemp1			;Zhlerwert = 0?
	breq	tim230			;ja -> berspringen
	dec	itemp1			;sonst Zhlerwert - 1
	inc	itemp2			;aktiven Alarm zhlen
tim230:	st	x+, itemp1		;Zhlerwert wieder speichern
	cpi	xl, low(almtio + 4)	;alle Zhler bearbeitet?
	brne	tim220			;nein -> Schleife
;
	mov	itemp1, msgtio		;Meldungs-Timeout-Zhler holen
	tst	itemp1			;Zhlerwert = 0?
	breq	tim240			;ja -> berspringen
	cpi	itemp1, 0xff		;Zhlerwert = 255?
	breq	tim240			;ja -> berspringen
	dec	itemp1			;sonst Zhlerwert - 1
	mov	msgtio, itemp1		;Zhler wieder speichern
;
tim240:	mov	itemp1, distio		;Anzeige-Timeout-Zhler holen
	tst	itemp1			;Zhlerwert = 0?
	breq	tim250			;ja -> berspringen
	cpi	itemp1, 0xff		;Zhlerwert = 255?
	breq	tim250			;ja -> berspringen
	dec	itemp1			;sonst Zhlerwert - 1
	mov	distio, itemp1		;Zhlerwert wieder speichern
tim250:	wdr				;Watchdog zurcksetzen
;
; Teilung auf 4s
;
	mov	itemp1, seccnt		;Sekundenzhler holen
	andi	itemp1, 0x03		;durch 4 teilbar?
	brne	tim400			;nein -> Ende
;
; aktive Alarme prfen und Alarmzhler erhhen (wird alle 4s ausgefhrt)
;
	tst	itemp2			;sind Alarme aktiv?
	brne	tim300			;ja -> weiter
	sts	almnum, itemp2		;sonst Alarmnummer lschen
	cbr	flags, 1 << 5		;Alarm-Flag lschen
	rjmp	tim400			;Ende
;
tim300:	ldi	xl, low(almtio)
	ldi	xh, high(almtio)	;Zeiger auf Alarm-Timeout-Zhler
	ldi	itemp2, 4		;4 Timeout-Zhler prfen
	lds	itemp1, almnum		;Alarm-Nummer holen
	add	xl, itemp1		;Zeiger auf Timeout-Zhler ermitteln
;
tim310:	adiw	xl, 1			;Zeiger auf nchsten Timeout-Zhler
	cpi	xl, low(almtio + 4)	;Zeigeradresse berschritten?
	brne	tim320			;nein -> weiter
	ldi	xl, low(almtio)		;sonst Zeiger wieder auf Anfang setzen
tim320:	ld	itemp1, x		;Timeout-Zhler holen
	tst	itemp1			;Zhler bereits abgelaufen
	breq	tim330			;ja -> weiter
	subi	xl, low(almtio)		;Alarmnummer ermitteln (0-3)
	sts	almnum, xl		;als nchste Alarm-Nummer speichern
	sbr	flags, 1 << 5		;Alarm-Flag setzen
	rjmp	tim400
;
tim330:	dec	itemp2			;alle Timeout-Zhler geprft?
	brne	tim310			;nein -> Schleife
	sts	almnum, itemp2		;sonst Alarmnummer lschen
	cbr	flags, 1 << 5		;und Alarm-Flag lschen
;
; Sekundenzhler prfen und zustzlichen Zhler bearbeiten (Teilung auf 3584s)
;
tim400:	tst	seccnt			;Sekundenzhler abgelaufen?
	brne	timend			;nein -> Ende
	inc	extcnt			;zustzlichen Zhler erhhen
	mov	itemp1, extcnt		;Zhler kopieren
	cpi	itemp1, 14		;Endwert erreicht (256 * 14 = 3584s)?
	brcs	timend			;nein -> Ende
	clr	extcnt			;zustzlichen Zhler wieder lschen
	set
	bld	xflags, 1		;Stunden-Flag setzen
;
timend:	out	SREG, insreg		;SREG wiederherstellen
	reti
;
; Verschiedene Texte und Tabellen
;
sttext:	.db	"Anzeigemodul 1 fuer Temperaturmesssystem"	;Begrungstext 4 x 20 Zeichen
	.db	"v2.00 vom 21.03.2013(c) Scott-Falk Huehn"
;
eftext:	.db	"EEPROM-Fehler "				;Anzeige bei EEPROM-Fehler
;
sentxt:	.db	"Sensorbelegung"				;Anzeige Sensorbelegung
;
msgerr:	.db	"Kein Datenempfang "				;Anzeige bei Daten-Timeout
;
almtxt:	.db	6, "Alarm: "					;Anzeige bei Alarm
;
lcdtab:	.db	0x80, 0xc0, 0x94, 0xd4				;LCD-Adresse fr Werte 1-4 (Zeilenanfang 1-4)
	.db	0x8a, 0xca, 0x9e, 0xde				;LCD-Adresse fr Werte 5-8 (Zeilenmitte 1-4)
;
chars:	.db	0x10, 0, 0, 0, 0, 0, 0, 0			;Bitmap Sonderzeichen 0 (Grad, normale Temperatur)
	.db	0x10, 0, 0x10, 0x0a, 6, 0x0e, 0, 0		;Bitmap Sonderzeichen 1 (Grad + Pfeil schrg nach unten, Temperaturunterschreitung)
	.db	0x10, 0, 0x0e, 6, 0x0a, 0x10, 0, 0		;Bitmap Sonderzeichen 2 (Grad + Pfeil schrg nach oben, Temperaturberschreitung)
	.db	0, 0x10, 4, 8, 0x10, 4, 0, 0			;Bitmap Sonderzeichen 3 (kleines Prozentzeichen, Luftfeuchtigkeit)
	.db	0, 0, 0x10, 0x0a, 6, 0x0e, 0, 0			;Bitmap Sonderzeichen 4 (Pfeil schrg nach unten, Luftdruck-Tendenz fallend)
	.db	0, 0, 0x0e, 6, 0x0a, 0x10, 0, 0			;Bitmap Sonderzeichen 5 (Pfeil schrg nach oben, Luftdruck-Tendenz steigend)
	.db	0x10, 0x18, 0x1c, 0x1e, 0x1c, 0x18, 0x10, 0	;Bitmap Sonderzeichen 6 (Dreieck-Pfeil rechts)
	.db	0, 0, 0, 0, 0, 0x0c, 4, 8			;Bitmap Sonderzeichen 7 (Komma mit Unterlnge)
;
