// ====================================================================================================================
// MultiModul 1 und MultiModul 2 (Akku-berwachung, Lichtsteuerung, Fahrwerksteuerung) mit ATmega88PA
// Version 1.03, 29.02.2012
// ====================================================================================================================

// Definition der LEDs ------------------------------------------------------------------------------------------------

// LED 1 (PD4) - Positionslicht oder Blitzer mglich
// LED 2 (PD1) - Positionslicht oder Blitzer mglich
// LED 3 (PD0) - Positionslicht oder Blitzer mglich
// LED 4 (PB1) - Positionslicht oder Blitzer mglich
// LED 5 (PB0) - Positionslicht oder Blitzer mglich
// LED 6 (PD6) - Positionslicht, Blitzer oder Beacon mglich (weiches Auf- und Abblenden)
// LED 7 (PD5) - Positionslicht, Blitzer oder Beacon mglich (weiches Auf- und Abblenden)
// LED 8 (PB3) - Landelicht (weiches Auf- und Abblenden)

// Belegung der I/O-Ports ---------------------------------------------------------------------------------------------

// B0: Ausgang LED 5 (L-aktiv)
// B1: Ausgang LED 4 (L-aktiv)
// B2: Ausgang Fahrwerk-Servo (OC1B)
// B3: Ausgang LED-Treiber fr Landelicht (OC2B, H-aktiv) und Eingang MOSI
// B4: Eingang MISO (Pull-Up)
// B5: Eingang SCK (Pull-Up)
// B6: Ausgang externer Akku-Alarm (H-aktiv)
// B7: Eingang Taster (Pull-Up)

// C0: Eingang analog fr Spannungsberwachung

// D0: Ausgang LED 3 (L-aktiv)
// D1: Ausgang LED 2 (L-aktiv)
// D2: Eingang Empfnger-Kanal A - Licht (Pull-Up)
// D3: Eingang Empfnger-Kanal B - Fahrwerk (Pull-Up)
// D4: Ausgang LED 1 (L-aktiv)
// D5: Ausgang LED 7, Beacon 2 (OC0B, L-aktiv)
// D6: Ausgang LED 6, Beacon 1 (OC0A, L-aktiv)

// Belegung der Timer -------------------------------------------------------------------------------------------------

// Timer0A: PWM-Steuerung von LED 6 (Beacon 1)
// Timer0B: PWM-Steuerung von LED 7 (Beacon 2)
// Timer1A: Messen der Impulslnge der Empfngerkanle, PWM-Ausgabe der Servoimpulse, Takt fr Lichtprogramme
// Timer2A: PWM-Steuerung des Landelichts

#define mm1                     // Software fr MultiModul 1 erzeugen
//#define mm2                     // Software fr MultiModul 2 erzeugen

.nolist
.include "m88def.inc"
.list

// Konstanten ---------------------------------------------------------------------------------------------------------

.equ    clock=  8000000         // Taktfrequenz= 8,0MHz
.equ    lig1th= 1350            // Schaltschwelle fr die Beleuchtung (PPM-Impulslnge ab 1,35ms)
.equ    lig2th= 1650            // Schaltschwelle fr das Landelicht (PPM-Impulslnge ab 1,65ms)
.equ    lowlim= 750             // unterer Grenzwert fr PPM-Impulslnge (0,75ms)
.equ    higlim= 2250            // oberer Grenzwert fr PPM-Impulslnge (2,25ms)
.equ    keytim= 25              // Zeit in 20ms-Intervallen fr langen Tastendruck (500ms)
.equ    almtim= 64              // Timeout-Wert fr das Landelicht bei Akku-Alarm in 20ms-Einheiten (1,28s)
.equ    settim= 24              // Timeout-Wert fr das automatische Beenden des Setup-Modus in 5,12s-Einheiten (123s)
.equ    margin= 3               // PPM-Toleranzschwelle fr Fahrwerk-Servo-Bewegungen (3s)
.equ    accmax= 933             // hchster Schwellwert fr den Akku-Alarm (entspricht 11,0V)
.equ    acstep= 9               // Akku-Alarm-Stufenwert (entspricht ca. 106mV)
.equ    gearll= 50              // Toleranz-Wert fr Fahrwerk-abhngiges Landelicht; ist die PPM-Impulslnge des Fahr-
                                // werk-Kanals im Bereich von +/-50s von der gespeicherten Ausfahr-Position, dann wird
                                // das Landelicht eingeschaltet (sofern die Beleuchtung eingeschaltet ist)

// Register-Definitionen ----------------------------------------------------------------------------------------------

.def    k1cnt1a= r2             // aktueller Zhlerstand beim Impulsbeginn von Kanal A (Low)
.def    k1cnt1b= r3             // aktueller Zhlerstand beim Impulsbeginn von Kanal A (High)
.def    k1cnt2a= r4             // aktueller Zhlerstand beim Impulsende von Kanal A (Low)
.def    k1cnt2b= r5             // aktueller Zhlerstand beim Impulsende von Kanal A (High)
.def    k2cnt1a= r6             // aktueller Zhlerstand beim Impulsbeginn von Kanal B (Low)
.def    k2cnt1b= r7             // aktueller Zhlerstand beim Impulsbeginn von Kanal B (High)
.def    k2cnt2a= r8             // aktueller Zhlerstand beim Impulsende von Kanal B (Low)
.def    k2cnt2b= r9             // aktueller Zhlerstand beim Impulsende von Kanal B (High)
.def    isreg=  r10             // Zwischenspeicher fr SREG in Interrupt-Routinen
.def    icount= r11             // Zhler fr Timer1-Interrupts, alle 20ms
.def    kstate= r25             // aktueller Status der Empfnger-Kanle:
                                // Bit 0: Kanal A: 0 - normaler Betrieb
                                //        Kanal A: 1 - Pegel undefiniert (nach Reset)
                                // Bit 1: Kanal A: Impulsende-Flag, wird von Int0 beim Impulsende auf 1 gesetzt und
                                //                 vom Hauptprogramm nach der Bearbeitung wieder gelscht
                                // Bit 2: Kanal B: 0 - normaler Betrieb
                                //        Kanal B: 1 - Pegel undefiniert (nach Reset)
                                // Bit 3: Kanal B: Impulsende-Flag, wird von Int0 beim Impulsende auf 1 gesetzt und
                                //                 vom Hauptprogramm nach der Bearbeitung wieder gelscht

// Variablen im RAM ---------------------------------------------------------------------------------------------------

.dseg
.org    0x100

k1ppm:  .byte   5 * 2           // 5x2 Byte  ermittelte PPM-Impulslnge (16 Bit) von Empfnger-Kanal A (Licht);
                                //           Eintrge 0-3 enthalten die letzten 4 Werte; Eintrag 4 enthlt den
                                //           Mittelwert, wenn alle 4 Werte vollstndig sind
k2ppm:  .byte   5 * 2           // 5x2 Byte  ermittelte PPM-Impulslnge (16 Bit) von Empfnger-Kanal B (Gear);
                                //           Eintrge 0-3 enthalten die letzten 4 Werte; Eintrag 4 enthlt den
                                //           Mittelwert, wenn alle 4 Werte vollstndig sind
adcval: .byte   4 * 2           // 4x2 Byte  ermittelte Spannung der Akku-berwachung (16 Bit); Eintrge 0-3
                                //           enthalten die letzten 4 Werte
servo0: .byte   2               // 750-2250  Servo-Position fr das automatische Ausfahren des Fahrwerks, wird beim
                                //           Start aus dem EEPROM gelesen
ccount: .byte   1               // 0-255     Zyklus-Zhler, entspricht Zhlerstand des 20ms-Interrupt-Zhlers
kflags: .byte   1               //           Taster-Flags:
                                //           Bit 0: Status im vorherigen 20ms-Zyklus
                                //           Bit 1: entprellter Taster-Status
                                //           Bit 2: quittierter Taster-Status
                                //           Bit 3: langer Tastendruck (ber 500ms)
                                //           Bit 4: langer Tastendruck quittiert
                                //           Bit 5: Merker fr kurzen Tastendruck, steuert die Bearbeitung des Tasten-
                                //                  drucks nach dem Loslassen des Tasters
keycnt: .byte   1               // 0-24      Zhler zur Erkennung eines langen Tastendrucks
sdelay: .byte   1               // 0-3       Servo-Verzgerung; wird bei Servobewegungen in einen Schrittwert umgerech-
                                //           net (8, 6, 5, 4) und alle 20ms addiert oder subtrahiert
                                //           aktuellen Servo-Werte , bis der Zielwert erreicht ist
acclev: .byte   1               // 0-9       Akku-Alarm-Stufe, dient zur Einstellung des Schwellwertes
accalm: .byte   1               //           Akku-Alarm-Flags:
                                //           Bit 0: wird bei Akku-Alarm direkt gesetzt oder gelscht
                                //           Bit 1: wird bei Akku-Alarm gesetzt und nach Ablauf des Alarm-Timeout-
                                //                  Zhlers wieder gelscht
alm_to: .byte   1               // 0-64      Timeout-Zhler fr den Akku-Alarm in 20ms-Einheiten
geardn: .byte   1               // 0-1       wird gesetzt, wenn Akku-Alarm ausgelst wurde und das automatische Ausfah-
                                //           ren des Fahrwerks aktiviert ist, einmal gesetzes Bit bleibt gesetzt
mode:   .byte   1               // 0-4       Betriebs-Modus:
                                //           0: Startsequenz, LED 1 + LED 2 blinken 2x
                                //           1: normaler Betrieb
                                //           2: alle LEDs ausschalten (wird fr das Setup und Alarm bentigt)
                                //           3: alle LEDs einschalten (wird fr das Setup und Alarm bentigt)
                                //           4: LED-Steuerung bergehen (wird fr das Setup und Alarm bentigt)
prgnum: .byte   7               // 0-17      enthalten die aus dem EEPROM kopierten Licht-Programm-Nummern der LEDs 1-7
ledprg: .byte   3 * 7           // 3x7 Byte  enthalten die Parameter fr die Licht-Programme der LEDs 1-7
                                //           Byte 00-06: aktuelle Licht-Programm-Nummer der LEDs 1-7
                                //           Byte 07-13: nchste Programm-Position der LEDs 1-7
                                //           Byte 14-20: aktueller Zhlerstand fr die Kommando-Dauer der LEDs 1-7
beacfl: .byte   1               //           Beacon-Flags und Landelicht-Flag:
                                //           Bit 0: Beacon-LED 1, 0= weich ausschalten, 1= weich einschalten
                                //           Bit 1: Beacon-LED 2, 0= weich ausschalten, 1= weich einschalten
                                //           Bit 2: Landelicht-Flag, wird beim Ausschalten in der letzten Phase
                                //                  genutzt, um das Raster von 20ms auf 40ms zu vergrern
lights: .byte   1               // 0-2       Licht-Status:
                                //           0: alle Lichter aus
                                //           1: Lichter ein (Position, Blitzer, Beacon)
                                //           2: zuztzlich Landelicht ein
almprg: .byte   1               // 0-7       enthlt die aus dem EEPROM kopierte Alarm-Programm-Nummer:
                                //               Beleuchtung     Landelicht     Fahrwerk
                                //           0:       -              -              -
                                //           1:     blinkt           -              -
                                //           2:       -            blinkt           -
                                //           3:     blinkt         blinkt           -
                                //           4:       -              -          fhrt aus
                                //           5:     blinkt           -          fhrt aus
                                //           6:       -            blinkt       fhrt aus
                                //           7:     blinkt         blinkt       fhrt aus
servfl: .byte   1               // 0-1       Flag fr erfolgreiche Speicherung einer Servo-Position
semode: .byte   1               // 0-10      Setup-Modus:
                                //           0:  Setup aus
                                //           1:  Licht-Programm fr LED 1
                                //           ...
                                //           7:  Licht-Programm fr LED 7
                                //           8:  Akku-Alarm-Programm
                                //           9:  Spannungsschwelle fr Akku-Alarm
                                //           10: Stufe fr Servo-Verzgerung
                                //           11: Servo-Position fr automatisches Fahrwerk
sestep: .byte   1               // 0-9       Setup-Schrittzhler: 
                                //           0: Ende
                                //           1: alle LEDs ausschalten
                                //           2: 1. Blinken (LED ein)
                                //           3: 1. Blinken (LED aus)
                                //           4: 2. Blinken (LED ein)
                                //           5: 2. Blinken (LED aus)
secoun: .byte   1               //           Setup-Wartezhler, wird fr Wartezeiten bei der Eingabe verwendet
senumb: .byte   1               //           Setup-Nummernzhler, wird fr die Blink-Ausgabe eines Wertes verwendet
set_to: .byte   1               // 0-24      Setup-Timeout-Zhler, wird bei jeder Taster-Bettigung auf den maximalen
                                //           Wert (23) gesetzt und alle 5,12s dekrementiert, beim Erreichen von 0 wird
                                //           der Setup-Modus verlassen und in den normalen Betriebs-Modus gewechselt

// Macro-Definitionen -------------------------------------------------------------------------------------------------

#if mm1                                         // --- Beginn MultiModul 1 ---
.macro  led1_on                                 // LED 1 einschalten
        cbi     portd, 4                        // LED-Port auf Low setzen
.endmacro
.macro  led1_off                                // LED 1 ausschalten
        sbi     portd, 4			// LED-Port auf High setzen
.endmacro
.macro  led2_on                                 // LED 2 einschalten
        cbi     portd, 1                        // LED-Port auf Low setzen
.endmacro
.macro  led2_off                                // LED 2 ausschalten
        sbi     portd, 1                        // LED-Port auf High setzen
.endmacro
.macro  led3_on                                 // LED 3 einschalten
        cbi     portd, 0                        // LED-Port auf Low setzen
.endmacro
.macro  led3_off                                // LED 3 ausschalten
        sbi     portd, 0                        // LED-Port auf High setzen
.endmacro
.macro  led4_on                                 // LED 4 einschalten
        cbi     portb, 1                        // LED-Port auf Low setzen
.endmacro
.macro  led4_off                                // LED 4 ausschalten
        sbi     portb, 1                        // LED-Port auf High setzen
.endmacro
.macro  led5_on                                 // LED 5 einschalten
        cbi     portb, 0                        // LED-Port auf Low setzen
.endmacro
.macro  led5_off                                // LED 5 ausschalten
        sbi     portb, 0                        // LED-Port auf High setzen
.endmacro
#endif                                          // --- Ende - MultiModul 1 ---

#if mm2                                         // --- Beginn MultiModul 2 ---
.macro  led1_on                                 // LED 1 einschalten
        sbi     portd, 4                        // LED-Port auf High setzen
.endmacro
.macro  led1_off                                // LED 1 ausschalten
        cbi     portd, 4			// LED-Port auf Low setzen
.endmacro
.macro  led2_on                                 // LED 2 einschalten
        sbi     portd, 1                        // LED-Port auf High setzen
.endmacro
.macro  led2_off                                // LED 2 ausschalten
        cbi     portd, 1                        // LED-Port auf Low setzen
.endmacro
.macro  led3_on                                 // LED 3 einschalten
        sbi     portd, 0                        // LED-Port auf High setzen
.endmacro
.macro  led3_off                                // LED 3 ausschalten
        cbi     portd, 0                        // LED-Port auf Low setzen
.endmacro
.macro  led4_on                                 // LED 4 einschalten
        sbi     portb, 1                        // LED-Port auf High setzen
.endmacro
.macro  led4_off                                // LED 4 ausschalten
        cbi     portb, 1                        // LED-Port auf Low setzen
.endmacro
.macro  led5_on                                 // LED 5 einschalten
        sbi     portb, 0                        // LED-Port auf High setzen
.endmacro
.macro  led5_off                                // LED 5 ausschalten
        cbi     portb, 0                        // LED-Port auf Low setzen
.endmacro
#endif                                          // --- Ende - MultiModul 2 ---

.macro  exalm_on                                // Externen Alarm einschalten
        sbi     portb, 6                        // Alarm-Port auf High setzen
.endmacro
.macro  exalm_off                               // Externen Alarm ausschalten
        cbi     portb, 6                        // Alarm-Port auf Low setzen
.endmacro

// Interrupt-Vektoren -------------------------------------------------------------------------------------------------

.cseg
.org    0
        rjmp    start                           // Programmstart

.org    INT0addr
        rjmp    goint0                          // Externer Interrupt INT0

.org    INT1addr
        rjmp    goint1                          // Externer Interrupt INT1

.org    OC1Aaddr
        rjmp    gooc1a                          // Interrupt bei Erreichen von OCR1A (20ms)

// Interrupt-Service-Routinen =========================================================================================

// Externer Interrupt 0 bei Pegelwechsel an PD2 -----------------------------------------------------------------------

goint0: in      isreg, sreg                     // SREG sichern
        sbis    pind, 2                         // wenn H-Pegel an PD2 -> berspringen
        rjmp    i0_010                          // sonst L-Pegel an PD2 -> bearbeiten
        cbr     kstate, (1 << 1) | (1 << 0)     // undefinierten Pegel-Status und Impulsende-Flag lschen
        lds     k1cnt1a, tcnt1l                 // aktuellen Zhlerstand von Timer1 lesen (L)
        lds     k1cnt1b, tcnt1h                 // aktuellen Zhlerstand von Timer1 lesen (H)
        rjmp    i0_end

i0_010: sbrc    kstate, 0                       // wenn Pegel-Status von Kanal A ok -> berspringen
        rjmp    i0_end                          // sonst Pegel-Status undefiniert -> Ende
        lds     k1cnt2a, tcnt1l                 // aktuellen Zhlerstand von Timer1 lesen (L)
        lds     k1cnt2b, tcnt1h                 // aktuellen Zhlerstand von Timer1 lesen (H)
        sbr     kstate, 1 << 1                  // Impulsende-Flag setzen

i0_end: out     sreg, isreg                     // SREG wiederherstellen
        reti

// Externer Interrupt 1 bei Pegelwechsel an PD3 -----------------------------------------------------------------------

goint1: in      isreg, sreg                     // SREG sichern
        sbis    pind, 3                         // wenn H-Pegel an PD3 -> berspringen
        rjmp    i1_010                          // sonst L-Pegel an PD3 -> bearbeiten
        cbr     kstate, (1 << 3) | (1 << 2)     // undefinierten Pegel-Status und Impulsende-Flag lschen
        lds     k2cnt1a, tcnt1l                 // aktuellen Zhlerstand von Timer1 lesen (L)
        lds     k2cnt1b, tcnt1h                 // aktuellen Zhlerstand von Timer1 lesen (H)
        rjmp    i1_end

i1_010: sbrc    kstate, 2                       // wenn Pegel-Status von Kanal B ok -> berspringen
        rjmp    i1_end                          // sonst Pegel-Status undefiniert -> Ende
        lds     k2cnt2a, tcnt1l                 // aktuellen Zhlerstand von Timer1 lesen (L)
        lds     k2cnt2b, tcnt1h                 // aktuellen Zhlerstand von Timer1 lesen (H)
        sbr     kstate, 1 << 3                  // Impulsende-Flag setzen

i1_end: out     sreg, isreg                     // SREG wiederherstellen
        reti

// Interrupt bei Erreichen des Endwertes OCR1A (alle 20ms) ------------------------------------------------------------

gooc1a: in      isreg, sreg                     // SREG sichern
        inc     icount                          // Interrupt-Zhler erhhen
        out     sreg, isreg                     // SREG wiederherstellen
        reti

// Hauptprogramm - Initialisierung ====================================================================================

start:  cli                                     // Interrupts sperren

        wdr                                     // Watchdog-Timer zurcksetzen
        ldi     r16, (1 << wdce) | (1 << wde)
        sts     wdtcsr, r16                     // Watchdog-nderung freigeben
        ldi     r16, (1 << wde) | (1 << wdp1) | (1 << wdp0)
        sts     wdtcsr, r16                     // Watchdog-Timer auf 125ms setzen

        clr     r1                              // Nullwert laden
        ldi     r16, 255
        mov     r12, r16                        // Wert 255 laden
        ldi     r16, low (ramend)
        ldi     r17, high (ramend)              // Adresse vom RAM-Ende laden
        out     spl, r16
        out     sph, r17                        // Stackpointer auf RAM-Ende setzen

        ldi     xl, 0
        ldi     xh, 1                           // Zeiger auf RAM-Beginn (0x100)
sta010: st      x+, r1                          // Speicherzelle lschen
        cp      xl, r16
        cpc     xh, r17                         // RAM-Ende erreicht?
        brcs    sta010                          // nein -> Schleife

// I/O-Ports initialisieren -------------------------------------------------------------------------------------------

#if mm1                                         // --- Beginn MultiModul 1 ---
        ldi     r16, 0b10110011                 // B0, B1 auf High; B2, B3, B6 auf Low; Rest auf High fr Pull-Up
        out     portb, r16                      // ausgeben
        ldi     r16, 0b01001111                 // B0-B3, B6 auf Ausgang setzen; Rest auf Eingang
        out     ddrb, r16                       // ausgeben
        ldi     r16, 0b11111110                 // C0 auf Low setzen (ohne Pull-up); Rest auf High fr Pull-Up
        out     portc, r16                      // ausgeben
        ldi     r16, 0b00000000                 // alle Ports sind Eingnge
        out     ddrc, r16                       // ausgeben
        ldi     r16, 0b11111111                 // Ausgnge auf High setzen bzw. auf High fr Pull-Up
        out     portd, r16                      // ausgeben
        ldi     r16, 0b01110011                 // D2, D3, D7 auf Eingang setzen; Rest auf Ausgang
        out     ddrd, r16                       // ausgeben
#endif                                          // --- Ende - MultiModul 1 ---

#if mm2                                         // --- Beginn MultiModul 2 ---
        ldi     r16, 0b10110000                 // B0, B1, B2, B3, B6 auf Low; Rest auf High fr Pull-Up
        out     portb, r16                      // ausgeben
        ldi     r16, 0b01001111                 // B0-B3, B6 auf Ausgang setzen; Rest auf Eingang
        out     ddrb, r16                       // ausgeben
        ldi     r16, 0b11111110                 // C0 auf Low setzen (ohne Pull-up); Rest auf High fr Pull-Up
        out     portc, r16                      // ausgeben
        ldi     r16, 0b00000000                 // alle Ports sind Eingnge
        out     ddrc, r16                       // ausgeben
        ldi     r16, 0b10001100                 // D0, D1, D4, D5, D6 auf Low setzen; Rest auf High fr Pull-Up
        out     portd, r16                      // ausgeben
        ldi     r16, 0b01110011                 // D2, D3, D7 auf Eingang setzen; Rest auf Ausgang
        out     ddrd, r16                       // ausgeben
#endif                                          // --- Ende - MultiModul 2 ---

// Timer initialisieren -----------------------------------------------------------------------------------------------

        ldi     r16, 3 << wgm10
        sts     tccr1a, r16                     // Timer1 - Fast PWM mit OCR1A als Endwert
        ldi     r16, (3 << wgm12) | (1 << cs11)
        sts     tccr1b, r16                     // Fast PWM mit OCR1A als Endwert, Vorteiler 8
        ldi     r16, low (clock / 8 / 50 - 1)
        ldi     r17, high (clock / 8 / 50 - 1)  // Endwert fr Timer 1 laden (50Hz / 20ms) festlegen
        sts     ocr1ah, r17
        sts     ocr1al, r16                     // ausgeben
        sts     ocr1bh, r1
        sts     ocr1bl, r1                      // PWM fr Servo zunchst auf kleinsten Wert setzen
        ldi     r16, 1 << ocie1a
        sts     timsk1, r16                     // Interrupt bei Erreichen des Endwertes OCR1A aktivieren

#if mm1                                         // --- Beginn MultiModul 1 ---
        ldi     r16, (3 << com0a0) | (3 << com0b0) | (1 << wgm00)                       // PWM invertiert
#endif                                          // --- Ende - MultiModul 1 ---

#if mm2                                         // --- Beginn MultiModul 2 ---
        ldi     r16, (1 << com0a1) | (1 << com0b1) | (1 << wgm00)                       // PWM normal
#endif                                          // --- Ende - MultiModul 2 ---

        out     tccr0a, r16                     // Phase Correct PWM mit Ausgang OC0A und OC0B
        ldi     r16, 1 << cs01
        out     tccr0b, r16                     // Vorteiler 8 (1MHz -> PWM-Frequenz 3906Hz)
        out     ocr0a, r1                       // PWM fr Beacon 1 zunchst auf kleinsten Wert setzen
        out     ocr0b, r1                       // PWM fr Beacon 2 zunchst auf kleinsten Wert setzen

        ldi     r16, (1 << com2a1) | (1 << wgm20)
        sts     tccr2a, r16                     // Phase Correct PWM mit Ausgang OC2A
        ldi     r16, 1 << cs21
        sts     tccr2b, r16                     // Vorteiler 8 (1MHz -> PWM-Frequenz 3906Hz)
        sts     ocr2a, r1                       // PWM fr Landelicht zunchst auf kleinsten Wert setzen

// Externe Interrupts (fr PWM-Erkennung von Kanal A und B) und ADC initialisieren ------------------------------------

        ldi     r16, (1 << isc10) | (1 << isc00)
        sts     eicra, r16                      // Externer Interrupt 0 und 1 bei Pegelwechsel auslsen
        ldi     r16, (1 << int1) | (1 << int0)
        out     eimsk, r16                      // Externen Interrupt 0 und 1 aktivieren

        sts     admux, r1                       // Referenz an AREF, ADC0 auswhlen
        ldi     r16, (1 << aden) | (1 << adsc) | (1 << adate) | (7 << adps0)
        sts     adcsra, r16                     // ADC einschalten, Auto Trigger Mode, Vorteiler = 128
        sts     adcsrb, r1                      // ADC Free Running Mode

// EEPROM-Daten lesen -------------------------------------------------------------------------------------------------

        ldi     r16, e_effn                     // EEPROM-Adresse der Licht-Programm-Nummern
        ldi     r18, 7                          // 7 Werte lesen
        ldi     xl, low (prgnum)
        ldi     xh, high (prgnum)               // Zeiger auf Speicher-Adresse der Licht-Programm-Nummern
sta100: rcall   eeread                          // LED-Programm-Nummer lesen
        cpi     r17, ligend - ligtab            // Wertebereich berschritten?
        brcs    sta110                          // nein -> weiter
        ldi     r17, 1                          // sonst auf Wert 1 (Programm-Nummer 1) setzen
sta110: tst     r17                             // Wertebereich unterschritten?
        brne    sta120                          // nein -> weiter
        ldi     r17, 1                          // sonst auf Wert 1 (Programm-Nummer 1) setzen
sta120: st      x+, r17                         // Licht-Programm-Nummer speichern
        inc     r16                             // nchste EEPROM-Adresse
        dec     r18                             // alle Werte gelesen?
        brne    sta100                          // nein -> Schleife

        ldi     r16, e_almn                     // EEPROM-Adresse der Alarm-Programm-Nummer
        rcall   eeread                          // Alarm-Programm-Nummer lesen
        cpi     r17, 8                          // Wertebereich berschritten?
        brcs    sta130                          // nein -> weiter
        clr     r17                             // sonst auf Wert 0 (Programm-Nummer 0) setzen
sta130: sts     almprg, r17                     // Alarm-Programm-Nummer speichern

        ldi     r16, e_accl                     // EEPROM-Adresse der Akku-Alarm-Stufe
        rcall   eeread                          // Akku-Alarm-Stufe lesen
        cpi     r17, 15                         // Wertebereich berschritten?
        brcs    sta140                          // nein -> weiter
        clr     r17                             // sonst auf kleinsten Wert setzen
sta140: sts     acclev, r17                     // Akku-Alarm-Stufe speichern

        ldi     r16, e_sdel                     // EEPROM-Adresse der Servo-Verzgerung
        rcall   eeread                          // Geschwindigkeitsstufe lesen
        cpi     r17, 4                          // Wertebereich berschritten?
        brcs    sta150                          // nein -> weiter
        clr     r17                             // sonst auf kleinsten Wert setzen
sta150: sts     sdelay, r17                     // Servo-Verzgerung speichern

        ldi     r16, e_serv                     // EEPROM-Adresse der Fahrwerk-Servo-Position (Low)
        rcall   eeread                          // Servo-Position Low-Byte lesen
        mov     r18, r17                        // Low-Byte kopieren
        ldi     r16, e_serv + 1                 // EEPROM-Adresse der Fahrwerk-Servo-Position (High)
        rcall   eeread                          // Servo-Position HighByte lesen
        mov     r19, r17                        // High-Byte kopieren
        movw    r16, r18                        // 16-Bit-Wert kopieren

        subi    r16, low (lowlim)
        sbci    r17, high (lowlim)              // unteren PPM-Grenzwert subtrahieren
        subi    r16, low (higlim - lowlim)
        sbci    r17, high (higlim - lowlim)     // Differenz der Grenzwerte subtrahieren
        brcs    sta160                          // Differenz im gltigen Bereich? ja -> weiter
        mov     r18, r1                         // sonst:
        mov     r19, r1                         // 16-Bit-Wert auf 0 setzen
sta160: sts     servo0, r18                     // Fahrwerk-Servo-Position (Low) speichern
        sts     servo0 + 1, r19                 // Fahrwerk-Servo-Position (High) speichern

// Register-Variablen und Flags initialisieren ------------------------------------------------------------------------

        clr     k1cnt1a                         // aktuellen Zhlerstand beim Impulsbeginn von Kanal A (Low) lschen
        clr     k1cnt1b                         // aktuellen Zhlerstand beim Impulsbeginn von Kanal A (High) lschen
        clr     k1cnt2a                         // aktuellen Zhlerstand beim Impulsende von Kanal A (Low) lschen
        clr     k1cnt2b                         // aktuellen Zhlerstand beim Impulsende von Kanal A (High) lschen
        clr     k2cnt1a                         // aktuellen Zhlerstand beim Impulsbeginn von Kanal B (Low) lschen
        clr     k2cnt1b                         // aktuellen Zhlerstand beim Impulsbeginn von Kanal B (High) lschen
        clr     k2cnt2a                         // aktuellen Zhlerstand beim Impulsende von Kanal B (Low) lschen
        clr     k2cnt2b                         // aktuellen Zhlerstand beim Impulsende von Kanal B (High) lschen
        ldi     kstate, (1 << 2) | (1 << 0)     // Status von Empfnger-Kanal A und B ist zunchst undefiniert
        clr     icount                          // Interrupt-Zhler zurcksetzen
        sei                                     // Interrupts freigeben

// Hauptprogrammschleife ==============================================================================================

// Impulslnge von Empfnger-Kanal A (Licht) ermitteln, speichern und Mittel aus den letzten 4 Werten berechnen -------

main:   sbrs    kstate, 1                       // wenn Impulsende bei Empfnger-Kanal A -> berspringen
        rjmp    ma0100                          // sonst Empfnger-Kanal B bearbeiten
        cbr     kstate, 1 << 1                  // Impulsende-Flag wieder lschen

        cp      k1cnt2a, k1cnt1a
        cpc     k1cnt2b, k1cnt1b                // Impulsende-Wert >= Impulsbeginn-Wert?
        brcc    ma0010                          // ja -> weiter
        ldi     r16, low (20000)
        ldi     r17, high (20000)               // sonst Wert 20000 laden
        add     r16, k1cnt2a
        adc     r17, k1cnt2b                    // und zum Impulsende-Wert addieren (Zhler-berlauf korrigieren)
        rjmp    ma0020
ma0010: movw    r16, k1cnt2a                    // sonst Impulsende-Wert kopieren

ma0020: sub     r16, k1cnt1a
        sbc     r17, k1cnt1b                    // Impulsbeginn-Wert subtrahieren
        movw    r14, r16                        // Ergebnis kopieren

        subi    r16, low (lowlim)
        sbci    r17, high (lowlim)              // unteren PPM-Grenzwert subtrahieren
        subi    r16, low (higlim - lowlim)
        sbci    r17, high (higlim - lowlim)     // Differenz der Grenzwerte subtrahieren
        brcs    ma0030                          // Differenz im gltigen Bereich? ja -> bearbeiten
        rjmp    ma0100                          // sonst weiter mit Kanal B

ma0030: lds     r18, k1ppm + 4                  // sonst Werte-Tabelle nachrcken
        lds     r19, k1ppm + 5                  // Wert von Position 2 lesen
        sts     k1ppm + 6, r18
        sts     k1ppm + 7, r19                  // auf Position 3 speichern

        lds     r20, k1ppm + 2
        lds     r21, k1ppm + 3                  // Wert von Position 1 lesen
        sts     k1ppm + 4, r20
        sts     k1ppm + 5, r21                  // auf Position 2 speichern

        lds     r22, k1ppm
        lds     r23, k1ppm + 1                  // Wert von Position 0 lesen
        sts     k1ppm + 2, r22
        sts     k1ppm + 3, r23                  // auf Position 1 speichern

        sts     k1ppm, r14
        sts     k1ppm + 1, r15                  // neuen Wert auf Position 0 speichern

        cp      r18, r1
        cpc     r19, r1                         // Wert 3 = 0?
        breq    ma0100                          // ja -> Mittelwert-Berechnung berspringen
        cp      r20, r1
        cpc     r21, r1                         // Wert 2 = 0?
        breq    ma0100                          // ja -> Mittelwert-Berechnung berspringen
        cp      r22, r1
        cpc     r23, r1                         // Wert 1 = 0?
        breq    ma0100                          // ja -> Mittelwert-Berechnung berspringen

        add     r14, r18                        // Mittelwert von Eintrag 0-3 berechnen
        adc     r15, r19                        // Wert 3 addieren
        add     r14, r20
        adc     r15, r21                        // Wert 2 addieren
        add     r14, r22
        adc     r15, r23                        // Wert 1 addieren
        lsr     r15
        ror     r14                             // Summe halbieren
        lsr     r15
        ror     r14                             // Summe nochmals halbieren
        sts     k1ppm + 8, r14
        sts     k1ppm + 9, r15                  // Mittelwert speichern

// Impulslnge von Empfnger-Kanal B (Gear) ermitteln, speichern und Mittel aus den letzten 4 Werten berechnen --------

ma0100: sbrs    kstate, 3                       // wenn Impulsende bei Empfnger-Kanal B -> berspringen
        rjmp    ma1000                          // sonst weiter
        cbr     kstate, 1 << 3                  // Impulsende-Flag wieder lschen

        cp      k2cnt2a, k2cnt1a
        cpc     k2cnt2b, k2cnt1b                // Impulsende-Wert >= Impulsbeginn-Wert?
        brcc    ma0110                          // ja -> weiter
        ldi     r16, low (20000)
        ldi     r17, high (20000)               // sonst Wert 20000 laden
        add     r16, k2cnt2a
        adc     r17, k2cnt2b                    // und zum Impulsende-Wert addieren (Zhler-berlauf korrigieren)
        rjmp    ma0120
ma0110: movw    r16, k2cnt2a                    // sonst Impulsende-Wert kopieren

ma0120: sub     r16, k2cnt1a
        sbc     r17, k2cnt1b                    // Impulsbeginn-Wert subtrahieren
        movw    r14, r16                        // Ergebnis kopieren

        subi    r16, low (lowlim)
        sbci    r17, high (lowlim)              // unteren PPM-Grenzwert subtrahieren
        subi    r16, low (higlim - lowlim)
        sbci    r17, high (higlim - lowlim)     // Differenz der Grenzwerte subtrahieren
        brcs    ma0130                          // Differenz im gltigen Bereich? ja -> bearbeiten
        rjmp    ma1000                          // sonst weiter mit dem nchsten Abschnitt

ma0130: lds     r18, k2ppm + 4                  // sonst Werte-Tabelle nachrcken
        lds     r19, k2ppm + 5                  // Wert von Position 2 lesen
        sts     k2ppm + 6, r18
        sts     k2ppm + 7, r19                  // auf Position 3 speichern

        lds     r20, k2ppm + 2
        lds     r21, k2ppm + 3                  // Wert von Position 1 lesen
        sts     k2ppm + 4, r20
        sts     k2ppm + 5, r21                  // auf Position 2 speichern

        lds     r22, k2ppm
        lds     r23, k2ppm + 1                  // Wert von Position 0 lesen
        sts     k2ppm + 2, r22
        sts     k2ppm + 3, r23                  // auf Position 1 speichern

        sts     k2ppm, r14
        sts     k2ppm + 1, r15                  // neuen Wert auf Position 0 speichern

        cp      r18, r1
        cpc     r19, r1                         // Wert 3 = 0?
        brne    ma0140                          // nein -> weiter testen
        rjmp    ma1000                          // sonst Mittelwert-Berechnung berspringen
ma0140: cp      r20, r1
        cpc     r21, r1                         // Wert 2 = 0?
        breq    ma1000                          // ja -> Mittelwert-Berechnung berspringen
        cp      r22, r1
        cpc     r23, r1                         // Wert 1 = 0?
        breq    ma1000                          // ja -> Mittelwert-Berechnung berspringen

        add     r14, r18                        // Mittelwert von Eintrag 0-3 berechnen
        adc     r15, r19                        // Wert 3 addieren
        add     r14, r20
        adc     r15, r21                        // Wert 2 addieren
        add     r14, r22
        adc     r15, r23                        // Wert 1 addieren
        lsr     r15
        ror     r14                             // Summe halbieren
        lsr     r15
        ror     r14                             // Summe nochmals halbieren
        sts     k2ppm + 8, r14
        sts     k2ppm + 9, r15                  // Mittelwert speichern

// Ersten Mittelwert von Kanal B direkt an das Servo weitergegeben ----------------------------------------------------

        cli                                     // Interrupts beim Zugriff auf Timer1-Register sperren
        lds     r16, ocr1bl
        lds     r17, ocr1bh                     // aktuellen PWM-Wert vom Servo holen
        sei                                     // Interrupts wieder freigeben
        or      r16, r17                        // ist schon ein PWM-Wert (>0) gesetzt?
        brne    ma0150                          // ja -> weiter
        cli                                     // sonst Interrupts beim Zugriff auf Timer1-Register sperren
        sts     ocr1bh, r15
        sts     ocr1bl, r14                     // neuen PWM-Wert direkt auf den Mittelwert von Kanal B setzen
        sei                                     // Interrupts wieder freigeben

// Servo-Ausgang zum richtigen Zeitpunkt aktivieren (wenn Zhlerstand von Timer1 > PWM-Wert) --------------------------

ma0150: lds     r18, tccr1a                     // Einstellungen von Timer1 holen
        sbrc    r18, com1b1                     // ist der Servo-Ausgang bereits aktiviert? ja -> weiter
        rjmp    ma1000                          // sonst Ende
        cli                                     // Interrupts beim Zugriff auf Timer1-Register sperren
        lds     r16, tcnt1l                     // aktuellen Zhlerstand von Timer1 (Low) holen
        lds     r17, tcnt1h                     // aktuellen Zhlerstand von Timer1 (High) holen
        sei                                     // Interrupts wieder freigeben
        cp      r14, r16
        cpc     r15, r17                        // ist der Mittelwert (Sollwert) < Zhlerstand
        brcc    ma1000                          // nein -> weiter (ungnstiger Zeitpunkt)
        sbr     r18, 1 << com1b1                // sonst Ausgang OC1B (Servo-Ausgang) aktivieren
        sts     tccr1a, r18                     // neue Timer1-Einstellungen speichern

// Die folgenden Programmteile werden alle 20ms ausgefhrt ============================================================

ma1000: lds     r16, ccount                     // letzten Stand des 20ms-Interrupt-Zhlers holen
        cp      r16, icount                     // hat sich der Zhlerstand gendert?
        brne    ma1010                          // ja -> weiter
        rjmp    main                            // sonst Ende

ma1010: mov     r16, icount                     // neuen Zhlerstand kopieren
        sts     ccount, r16                     // und als Low-Byte speichern

// Taster abfragen und Flags setzen -----------------------------------------------------------------------------------

ma1100: in      r16, pinb                       // Taster-Port lesen
        com     r16                             // invertieren
        bst     r16, 7                          // Taster-Bit sichern
        bld     r16, 0                          // Taster-Bit an Position 0 laden
        lds     r17, kflags                     // Taster-Flags holen
        eor     r16, r17                        // Status-nderung ermitteln
        bld     r17, 0                          // neuen Taster-Status in Flags laden
        sts     kflags, r17                     // Taster-Flags speichern
        sbrc    r16, 0                          // hat sich der Taster-Status gendert? nein -> weiter
        rjmp    ma1200                          // sonst -> Ende

        bld     r17, 1                          // sonst entprellten Taster-Status setzen
        sts     kflags, r17                     // Taster-Flags speichern
        lds     r16, keycnt                     // Zhler fr langen Tastendruck holen
        brts    ma1110                          // Taster aktuell gedrckt? ja -> weiter
        cbr     r17, 0b00011100                 // sonst kurzen und langen Tastendruck sowie Quittung lschen
        clr     r16                             // Zhler fr langen Tastendruck lschen
        rjmp    ma1120

ma1110: inc     r16                             // Zhler fr langen Tastendruck erhhen
        cpi     r16, keytim                     // Zeit fr langen Tastendruck erreicht?
        brcs    ma1120                          // nein -> weiter
        dec     r16                             // sonst Zhler wieder vermindern (berlauf verhindern)
        sbr     r17, 0b00001000                 // Flag fr langen Tastendruck setzen

ma1120: sts     kflags, r17                     // Flags wieder speichern
        sts     keycnt, r16                     // Zhler fr langen Tastendruck wieder speichern

// ADC lesen, speichern und Mittelwert berechnen ----------------------------------------------------------------------

ma1200: lds     r16, adcval + 4                 // Werte-Tabelle nachrcken
        lds     r17, adcval + 5                 // Wert von Position 2 lesen
        sts     adcval + 6, r16
        sts     adcval + 7, r17                 // auf Position 3 speichern

        lds     r18, adcval + 2
        lds     r19, adcval + 3                 // Wert von Position 1 lesen
        sts     adcval + 4, r18
        sts     adcval + 5, r19                 // auf Position 2 speichern

        lds     r20, adcval
        lds     r21, adcval + 1                 // Wert von Position 0 lesen
        sts     adcval + 2, r20
        sts     adcval + 3, r21                 // auf Position 1 speichern

        cli                                     // Interrupts beim Zugriff auf ADC-Register sperren
        lds     r22, adcl
        lds     r23, adch                       // aktuellen ADC-Wert lesen
        sei                                     // Interrupts wieder freigeben
        sts     adcval, r22
        sts     adcval + 1, r23                 // auf Position 0 speichern

        add     r16, r18                        // Mittelwert von Eintrag 0-3 berechnen
        adc     r17, r19                        // Wert 3 addieren
        add     r16, r20
        adc     r17, r21                        // Wert 2 addieren
        add     r16, r22
        adc     r17, r23                        // Wert 1 addieren
        lsr     r17
        ror     r16                             // Summe halbieren
        lsr     r17
        ror     r16                             // Summe nochmals halbieren -> Mittelwert

// Akku-Spannung prfen, Alarm-Flag und Externen Alarm setzen ---------------------------------------------------------

        lds     r22, accalm                     // Akku-Alarm-Flags holen
        cbr     r22, 1 << 0                     // direktes Alarm-Flag zunchst lschen
        sts     accalm, r22                     // und speichern

        ldi     r18, low (accmax)
        ldi     r19, high (accmax)              // Hchsten Schwellwert fr den Akku-Alarm laden
        ldi     r20, acstep                     // Akku-Alarm-Stufenwert laden
        lds     r21, acclev                     // Akku-Alarm-Stufe holen
        mul     r20, r21                        // Schwellwert-Offset berechnen
        sub     r18, r0                         // berechneten Offset subtrahieren ->
        sbc     r19, r1                         // konfigurierten Schwellwert ermitteln
        sub     r16, r18
        sbc     r17, r19                        // Schwellwert unterschritten?
        brcc    ma1240                          // nein -> weiter

        lds     r21, mode                       // aktuellen Betriebs-Modus holen
        tst     r21                             // Startsequenz aktiv?
        breq    ma1240                          // ja -> keinen Akku-Alarm auslsen
        sbr     r22, 3 << 0                     // sonst beide Alarm-Flags setzn
        sts     accalm, r22                     // und speichern

ma1240: sbrs    r22, 0                          // direktes Alarm-Flag gesetzt? ja -> berspringen
        rjmp    ma1250                          // sonst Externen Alarm ausschalten
        exalm_on                                // Externen Alarm einschalten
        rjmp    ma1260
ma1250: exalm_off                               // Externen Alarm ausschalten

ma1260: lds     r21, alm_to                     // Alarm-Timeout-Zhler holen
        sbrs    r22, 0                          // direktes Alarm-Flag gesetzt? ja -> berspringen
        rjmp    ma1270                          // sonst Timeout-Zhler prfen
        ldi     r21, almtim
        sts     alm_to, r21                     // Alarm-Timeout-Zhler neu starten
ma1270: tst     r21                             // Alarm-Timeout-Zhler abgelaufen?
        breq    ma1280                          // ja -> verzgertes Alarm-Flag lschen
        dec     r21                             // sonst Alarm-Timeout-Zhler vermindern
        sts     alm_to, r21                     // und wieder speichern
        rjmp    ma1300
ma1280: cbr     r22, 1 << 1                     // verzgertes Alarm-Flag lschen
        sts     accalm, r22                     // und speichern

// Verzgerung fr Fahrwerk-Servo steuern -----------------------------------------------------------------------------

ma1300: cli                                     // Interrupts beim Zugriff auf Timer1-Register sperren
        lds     r16, ocr1bl
        lds     r17, ocr1bh                     // aktuellen Servo-Wert lesen
        sei                                     // Interrupts wieder freigeben
        mov     r20, r16
        or      r20, r17                        // Wert = 0 (Servo nicht aktiv)?
        breq    ma1400                          // ja -> berspringen

        lds     r18, k2ppm + 8
        lds     r19, k2ppm + 9                  // Servo-Zielwert lesen
        mov     r20, r18
        or      r20, r19                        // Wert = 0 (Empfnger-Kanal B nicht aktiv)?
        breq    ma1400                          // ja -> berspringen

        lds     r20, geardn                     // Flag fr automatisches Ausfahren des Fahrwerks holen
        tst     r20                             // soll Fahrwerk ausgefahren werden?
        breq    ma1310                          // nein -> normale Funktion
        lds     r20, servo0
        lds     r21, servo0 + 1                 // sonst gespeicherte Fahrwerk-Ausfahr-Position holen
        mov     r22, r20
        or      r22, r21                        // Wert = 0 (keine Fahrwerk-Ausfahr-Position vorhanden)?
        breq    ma1310                          // ja -> normale Funktion
        movw    r18, r20                        // sonst Fahrwerk-Ausfahr-Position als Servo-Zielwert verwenden

ma1310: lds     r23, sdelay                     // Servo-Verzgerung holen
        com     r23                             // Wert negieren
        subi    r23, -8                         // und 8 addieren, Servo-Verzgerung 0-3 ergibt Schrittwert 7-4
        cpi     r23, 7                          // Schrittwert = 7?
        brne    ma1320                          // nein -> weiter
        inc     r23                             // sonst Schrittwert auf 8 setzen
ma1320: movw    r20, r16                        // aktuellen Servo-Wert kopieren
        sub     r20, r18
        sbc     r21, r19                        // Zielwert vom Servo-Wert subtrahieren
        breq    ma1400                          // Zielwert erreicht? ja -> Ende
        brcc    ma1330                          // muss Wert vermindert werden? ja -> weiter

        subi    r20, low (-margin)              // sonst Wert erhhen
        sbci    r21, high (-margin)             // Toleranzschwelle addieren
        brcc    ma1400                          // Zielwert jetzt erreicht? ja -> berspringen
        add     r16, r23                        // sonst Schrittwert addieren
        adc     r17, r1                         // bertrag addieren
        cp      r18, r16
        cpc     r19, r17                        // Zielwert berschritten?
        brcc    ma1350                          // nein -> Ende
        rjmp    ma1340                          // sonst Zielwert setzen

ma1330: subi    r20, low (margin)
        sbci    r21, high (margin)              // Toleranzschwelle subtrahieren
        brcs    ma1400                          // Zielwert jetzt erreicht? ja -> berspringen
        sub     r16, r23                        // Schrittwert subtrahieren
        sbc     r17, r1                         // bertrag subtrahieren
        cp      r16, r18
        cpc     r17, r19                        // Zielwert unterschritten?
        brcc    ma1350                          // nein -> Ende
ma1340: movw    r16, r18                        // sonst Zielwert setzen

ma1350: cli                                     // Interrupts beim Zugriff auf Timer1-Register sperren
        sts     ocr1bh, r17
        sts     ocr1bl, r16                     // neuen Servo-Wert ausgeben
        sei                                     // Interrupts wieder freigeben

// Startsequenz, LED 1 und 2 leuchten zweimal kurz auf: 1s aus, 300ms ein, 300ms aus, 300ms ein -----------------------

ma1400: lds     r16, mode                       // aktuellen Betriebs-Modus holen
        tst     r16                             // Startsequenz aktiv?
        brne    ma1500                          // nein -> Ende (weiter mit Licht-Programm-Steuerung)

        lds     r16, ccount                     // aktuellen Stand des Zyklus-Zhlers holen
        cpi     r16, 100                        // Zhlerstand 100 erreicht (2,0s)?
        brcs    ma1410                          // nein -> weitere Zhlerstnde testen
        ldi     r16, 1                          // sonst
        sts     mode, r16                       // Betriebs-Modus auf normalen Betrieb setzen
        rjmp    ma1500                          // weiter zur Licht-Programm-Steuerung

ma1410: cpi     r16, 85                         // Zhlerstand 85 erreicht (1,7s)?
        brcs    ma1420                          // nein -> weitere Zhlerstnde testen
        led1_off                                // sonst LED 1 ausschalten
        led2_off                                // LED 2 ausschalten
        exalm_off                               // Externen Alarm ausschalten
        rjmp    ma3000                          // Startsequenz noch aktiv, Licht-Programm-Steuerung berspringen

ma1420: cpi     r16, 70                         // Zhlerstand 70 erreicht (1,4s)?
        brcs    ma1430                          // nein -> weitere Zhlerstnde testen
        led1_on                                 // sonst LED 1 einschalten
        led2_on                                 // LED 2 einschalten
        exalm_on                                // Externen Alarm einschalten
        rjmp    ma3000                          // Startsequenz noch aktiv, Licht-Programm-Steuerung berspringen

ma1430: cpi     r16, 55                         // Zhlerstand 55 erreicht (1,1s)?
        brcs    ma1440                          // nein -> weitere Zhlerstnde testen
        led1_off                                // sonst LED 1 ausschalten
        led2_off                                // LED 2 ausschalten
        exalm_off                               // Externen Alarm ausschalten
        rjmp    ma3000                          // Startsequenz noch aktiv, Licht-Programm-Steuerung berspringen

ma1440: cpi     r16, 40                         // Zhlerstand 40 erreicht (0,8s)?
        brcs    ma1450                          // nein -> LED 1 und Externer Alarm bleibt noch aus
        led1_on                                 // sonst LED 1 einschalten
        led2_on                                 // LED 2 einschalten
        exalm_on                                // Externen Alarm einschalten
ma1450: rjmp    ma3000                          // Startsequenz noch aktiv, Licht-Programm-Steuerung berspringen

// Nach Ablauf der Start-Sequenz Licht-Schwellwerte prfen ------------------------------------------------------------

ma1500: lds     r21, lights                     // bisherigen Licht-Status holen
        lds     r16, mode                       // aktuellen Betriebs-Modus holen
        cpi     r16, 1                          // Normaler Betrieb?
        brne    ma1510                          // nein -> weiter testen

        clr     r20                             // neuen Licht-Status zunchst auf "aus" setzen
        lds     r16, k1ppm + 8
        lds     r17, k1ppm + 9                  // Mittelwert von Kanal A holen
        mov     r18, r16
        or      r18, r17                        // gltiger Mittelwert vorhanden (>0)?
        breq    ma1540                          // nein -> Ende

        ldi     r18, low (lig1th)
        ldi     r19, high (lig1th)              // Schaltschwelle fr "Licht ein" laden
        cp      r16, r18
        cpc     r17, r19                        // Schaltschwelle erreicht?
        brcs    ma1540                          // nein -> Ende
        inc     r20                             // sonst Licht-Status auf "ein" setzen
        ldi     r18, low (lig2th)
        ldi     r19, high (lig2th)              // Schaltschwelle fr "Landelicht ein" laden
        cp      r16, r18
        cpc     r17, r19                        // Schaltschwelle erreicht?
        brcs    ma1540                          // nein -> Ende
        inc     r20                             // sonst Licht-Status auf "Landelicht ein" setzen
        rjmp    ma1540

ma1510: cpi     r16, 2                          // alle LEDs ausschalten?
        brne    ma1520                          // sonst alle LEDs einschalten
        clr     r20                             // neuen Lichtstatus auf "aus" setzen
        rjmp    ma1540
ma1520: cpi     r16, 3                          // alle LEDs einschalten?
        breq    ma1530                          // ja -> bearbeiten
        rjmp    ma2000                          // sonst Lichtsteuerung komplett bergehen

ma1530: ldi     r20, 1                          // alle LEDs einschalten

// Bei eingeschalteter Beleuchtung Fahrwerk-Status prfen und gegebenenfalls das Landelicht einschalten

ma1540: cpi     r20, 1                          // neuer Licht-Status "ein"?
        brne    ma1546                          // nein -> Kopplung von Fahrwerk und Landelicht berspringen
        lds     r16, servo0
        lds     r17, servo0 + 1                 // Servo-Position fr das automatische Ausfahren des Fahrwerks holen
        mov     r22, r16                        // Low-Wert kopieren
        or      r22, r17                        // mit High-Wert verknpfen, gltige Servo-Position vorhanden?
        breq    ma1546                          // nein -> Ende

        cli                                     // Interrupts beim Zugriff auf Timer1-Register sperren
        lds     r18, ocr1bl
        lds     r19, ocr1bh                     // aktuellen Servo-Wert lesen
        sei                                     // Interrupts wieder freigeben
        mov     r22, r18                        // Low-Wert kopieren
        or      r22, r19                        // mit High-Wert verknpfen, gltiger Servo-Wert vorhanden?
        breq    ma1546                          // nein -> Ende

        sub     r18, r16                        // Differenz zwischen gespeicherter Fahrwerk-Ausfahr-Position und
        sbc     r19, r17                        // aktueller Servo-Position ermitteln
        ldi     r16, low (gearll)
        ldi     r17, High (gearll)              // Toleranz-Wert laden
        brcc    ma1542                          // Differenz positiv? ja -> weiter
        add     r18, r16
        adc     r19, r17                        // Differenz ist negativ - Toleranz-Wert addieren
        brmi    ma1546                          // Endergebnis immer noch negativ? ja -> Ende
        rjmp    ma1544                          // sonst im Toleranz-Bereich -> Landelicht einschalten

ma1542: sub     r18, r16
        sbc     r19, r17                        // Differenz ist positiv - Toleranz-Wert subtrahieren
        brpl    ma1546                          // Endergebnis immer noch positiv? ja -> Ende
ma1544: inc     r20                             // sonst im Toleranz-Bereich - Landelicht einschalten
ma1546: cp      r20, r21                        // hat sich der Licht-Status gendert?
        breq    ma1580                          // nein -> Ende
        sts     lights, r20                     // neuen Licht-Status speichern

// Bei nderung des Lichtstatus - Licht-Programm-Daten entsprechend einstellen ----------------------------------------

        tst     r20                             // neuer Licht-Status "aus"?
        brne    ma1560                          // nein -> weiter testen

        ldi     r24, 7                          // sonst Programm-Daten der LEDs 1-7 ndern
        ldi     yl, low (ledprg)
        ldi     yh, high (ledprg)               // Zeiger auf Licht-Programm-Daten
ma1550: std     y + 7, r1                       // Programm-Position lschen
        std     y + 14, r1                      // Zhlerstand fr die Kommando-Dauer lschen
        st      y+, r1                          // Licht-Programm-Nummer 0 setzen (LED aus)
        dec     r24                             // alle LEDs bearbeitet?
        brne    ma1550                          // nein -> Schleife
        rjmp    ma1580                          // Ende

ma1560: tst     r21                             // war vorheriger Status "aus"?
        brne    ma1580                          // nein -> Ende

        ldi     r24, 7                          // sonst Programm-Daten der LEDs 1-7 ndern
        ldi     xl, low (prgnum)
        ldi     xh, high (prgnum)               // Zeiger auf Licht-Programm-Nummern
        ldi     yl, low (ledprg)
        ldi     yh, high (ledprg)               // Zeiger auf Licht-Programm-Daten
ma1570: ld      r16, x+                         // gespeicherte Licht-Programm-Nummer holen
        std     y + 7, r1                       // Programm-Position lschen
        std     y + 14, r1                      // Zhlerstand fr die Kommando-Dauer lschen
        st      y+, r16                         // Licht-Programm neu setzen
        dec     r24                             // alle LEDs bearbeitet?
        brne    ma1570                          // nein -> Schleife

// Bei Akku-Alarm und inaktivem Setup - Licht nach Alarm-Programm steuern ---------------------------------------------

ma1580: lds     r16, accalm                     // Akku-Alarm aktiv?
        tst     r16
        breq    ma1600                          // nein -> normale Licht-Funktion
        lds     r16, semode                     // aktuellen Setup-Modus holen
        tst     r16                             // Setup aktiv?
        brne    ma1600                          // ja -> normale Licht-Funktion

        lds     r16, almprg                     // aktuelle Alarm-Programm-Nummer holen
        sbrs    r16, 0                          // Bit 0 gesetzt (Beleuchtung blinkt bei Alarm)? ja -> berspringen
        rjmp    ma1600                          // sonst normale Licht-Funktion
        lds     r16, ccount                     // Zyklus-Zhler holen
        sbrs    r16, 3                          // Bit 3 wechselt alle 160ms, Bit 3 gesetzt? ja -> LEDs ausschalten
        rjmp    ma1590                          // sonst LEDs einschalten

        led1_off                                // LED 1 ausschalten
        led2_off                                // LED 2 ausschalten
        led3_off                                // LED 3 ausschalten
        led4_off                                // LED 4 ausschalten
        led5_off                                // LED 5 ausschalten
        out     ocr0a, r1                       // LED 6 ausschalten
        out     ocr0b, r1                       // LED 7 ausschalten
        rjmp    ma2000

ma1590: led1_on                                 // LED 1 einschalten
        led2_on                                 // LED 2 einschalten
        led3_on                                 // LED 3 einschalten
        led4_on                                 // LED 4 einschalten
        led5_on                                 // LED 5 einschalten
        out     ocr0a, r12                      // LED 6 einschalten
        out     ocr0b, r12                      // LED 7 einschalten
        rjmp    ma2000

// Steuerung der Licht-Programme der LEDs 1-7 -------------------------------------------------------------------------

ma1600: ldi     r24, 0                          // LED-Zhler
        ldi     yl, low (ledprg)
        ldi     yh, high (ledprg)               // Zeiger auf LED-Programm-Daten

ma1610: ldd     r16, y + 14                     // Zhlerstand fr Kommando-Dauer holen
        tst     r16                             // Zhler abgelaufen?
        breq    ma1700                          // ja -> nchstes Kommando ausfhren
        dec     r16                             // sonst Zhler vermindern
        std     y + 14, r16                     // und wieder speichern

        lds     r16, beacfl                     // Beacon-Flags holen
        cpi     r24, 5                          // wird LED 6 (Beacon 1) bearbeitet?
        brne    ma1650                          // nein -> weiter testen
        in      r17, ocr0a                      // PWM-Wert von LED 6 holen
        tst     r17                             // LED ausgeschaltet?
        breq    ma1690                          // ja -> Ende
        cpi     r17, 255                        // LED voll eingeschaltet?
        breq    ma1690                          // ja -> Ende

        mov     r18, r17                        // aktuellen PWM-Wert kopieren
        lsr     r18                             // halbieren
        lsr     r18                             // nochmals halbieren (1/4 des aktuellen PWM-Wertes ermitteln)
        tst     r18                             // Wert = 0 (kann sonst nicht verwendet werden)?
        brne    ma1620                          // nein -> weiter
        inc     r18                             // sonst Wert= 1 setzen

ma1620: sbrs    r16, 0                          // soll LED 6 weich eingeschaltet werden? ja -> weiter
        rjmp    ma1630                          // sonst -> LED 6 weich ausschalten
        add     r17, r18                        // LED weich einschalten: berechneten Wert addieren
        brcc    ma1640                          // berlauf? nein -> neuen PWM-Wert speichern
        ldi     r17, 255                        // sonst neuen PWM-Wert auf Maximum setzen
        rjmp    ma1640                          // neuen PWM-Wert speichern

ma1630: sub     r17, r18                        // LED weich ausschalten: berechneten Wert subtrahieren
        brcc    ma1640                          // berlauf? nein -> neuen PWM-Wert speichern
        clr     r17                             // sonst neuen PWM-Wert auf Minimum setzen
ma1640: out     ocr0a, r17                      // PWM-Wert fr LED 6 speichern
        rjmp    ma1950

ma1650: cpi     r24, 6                          // wird LED 7 (Beacon 2) bearbeitet?
        brne    ma1690                          // nein -> Ende
        in      r17, ocr0b                      // PWM-Wert von LED 7 holen
        tst     r17                             // LED ausgeschaltet?
        breq    ma1690                          // ja -> Ende
        cpi     r17, 255                        // LED voll eingeschaltet?
        breq    ma1690                          // ja -> Ende

        mov     r18, r17                        // aktuellen PWM-Wert kopieren
        lsr     r18                             // halbieren
        lsr     r18                             // nochmals halbieren (1/4 des aktuellen PWM-Wertes ermitteln)
        tst     r18                             // Wert = 0 (kann sonst nicht verwendet werden)?
        brne    ma1660                          // nein -> weiter
        inc     r18                             // sonst Wert= 1 setzen

ma1660: sbrs    r16, 1                          // soll LED 7 weich eingeschaltet werden? ja -> weiter
        rjmp    ma1670                          // sonst -> LED 7 weich ausschalten
        add     r17, r18                        // LED weich einschalten: berechneten Wert addieren
        brcc    ma1680                          // berlauf? nein -> neuen PWM-Wert speichern
        ldi     r17, 255                        // sonst neuen PWM-Wert auf Maximum setzen
        rjmp    ma1680                          // neuen PWM-Wert speichern

ma1670: sub     r17, r18                        // LED weich ausschalten: berechneten Wert subtrahieren
        brcc    ma1680                          // berlauf? nein -> neuen PWM-Wert speichern
        clr     r17                             // sonst neuen PWM-Wert auf Minimum setzen
ma1680: out     ocr0b, r17                      // PWM-Wert fr LED 7 speichern
ma1690: rjmp    ma1950

ma1700: ld      r16, y                          // LED-Programm-Nummer holen
        add     r16, r16                        // verdoppeln wegen 16-Bit-Tabelle
        ldi     zl, low (ligtab * 2)
        ldi     zh, high (ligtab * 2)           // Zeiger auf Licht-Programm-Tabelle setzen
        add     zl, r16                         // Programm-Nummer addieren
        adc     zh, r1                          // bertrag addieren
        lpm     xl, z+                          // Adresse der Licht-Programm-Daten holen (Low)
        lpm     xh, z                           // Adresse der Licht-Programm-Daten holen (High)
        lsl     xl                              // Adresse verdoppeln (low)
        rol     xh                              // Adresse verdoppeln (High)
        movw    zl, xl                          // Adresse kopieren (Startadresse des Programms)
        ldd     r19, y + 7                      // nchste Programm-Position holen
        mov     r18, r19                        // Position kopieren
        add     r18, r18                        // und verdoppeln (Offset-Adresse des nchsten Kommando-Bytes)
        add     zl, r18                         // Programm-Position addieren
        adc     zh, r1                          // bertrag addieren
        lpm     r16, z+                         // nchstes Programm-Kommando holen
        lpm     r17, z                          // nchsten Kommando-Parameter holen
        inc     r19                             // Programm-Position auf nchstes Kommando setzen
        cpi     r16, 4                          // Normales LED-Kommando (0-3)?
        brcs    ma1710                          // ja -> weiter
        movw    zl, xl                          // sonst Ende-Kommando, Startadresse nochmals kopieren
        lpm     r16, z+                         // erstes Programm-Kommando holen
        lpm     r17, z                          // ersten Kommando-Parameter holen
        ldi     r19, 1                          // Programm-Position auf zweites Kommando setzen
ma1710: std     y + 7, r19                      // neue Programm-Position speichern
        dec     r17                             // Parameter korrigieren
        std     y + 14, r17                     // und speichern

        cpi     r24, 0                          // wird LED 1 bearbeitet?
        brne    ma1730                          // nein -> weiter
        sbrs    r16, 0                          // LED-Einschalt-Kommando? ja -> berspringen
        rjmp    ma1720                          // sonst LED 1 auschalten
        led1_on                                 // LED 1 einschalten
        rjmp    ma1730
ma1720: led1_off                                // LED 1 auschalten

ma1730: cpi     r24, 1                          // wird LED 2 bearbeitet?
        brne    ma1750                          // nein -> weiter
        sbrs    r16, 0                          // LED-Einschalt-Kommando? ja -> berspringen
        rjmp    ma1740                          // sonst LED 2 ausschalten
        led2_on                                 // LED 2 einschalten
        rjmp    ma1750
ma1740: led2_off                                // LED 2 ausschalten

ma1750: cpi     r24, 2                          // wird LED 3 bearbeitet?
        brne    ma1770                          // nein -> weiter
        sbrs    r16, 0                          // LED-Einschalt-Kommando? ja -> berspringen
        rjmp    ma1760                          // sonst LED 3 ausschalten
        led3_on                                 // LED 3 einschalten
        rjmp    ma1770
ma1760: led3_off                                // LED 3 ausschalten

ma1770: cpi     r24, 3                          // wird LED 4 bearbeitet?
        brne    ma1790                          // nein -> weiter
        sbrs    r16, 0                          // LED-Einschalt-Kommando? ja -> berspringen
        rjmp    ma1780                          // sonst LED 3 ausschalten
        led4_on                                 // LED 4 einschalten
        rjmp    ma1790
ma1780: led4_off                                // LED 4 ausschalten

ma1790: cpi     r24, 4                          // wird LED 5 bearbeitet?
        brne    ma1810                          // nein -> weiter
        sbrs    r16, 0                          // LED-Einschalt-Kommando? ja -> berspringen
        rjmp    ma1800                          // sonst LED 3 ausschalten
        led5_on                                 // LED 5 einschalten
        rjmp    ma1810
ma1800: led5_off                                // LED 5 ausschalten

ma1810: cpi     r24, 5                          // wird LED 6 bearbeitet?
        brne    ma1860                          // nein -> weiter
        cpi     r16, 0                          // normales LED-Ausschalt-Kommando?
        brne    ma1820                          // nein -> weiter testen
        out     ocr0a, r1                       // PWM auf minimalen Wert setzen (LED ausschalten)
        rjmp    ma1860

ma1820: cpi     r16, 1                          // normales LED-Einschalt-Kommando?
        brne    ma1830                          // nein -> weiter testen
        ldi     r17, 255                        // PWM auf maximalen Wert setzen (LED einschalten)
        rjmp    ma1850

ma1830: cpi     r16, 2                          // weiches LED-Ausschalt-Kommando?
        brne    ma1840                          // nein -> weiter
        lds     r18, beacfl                     // Beacon-Flags holen
        cbr     r18, 1 << 0                     // Flag fr Beacon 1 (LED 6) auf weiches Ausschalten setzen
        sts     beacfl, r18                     // Beacon-Flags wieder speichern
        in      r17, ocr0a                      // aktuellen PWM-Wert holen
        cpi     r17, 254                        // luft bereits ein Ausschalt-Vorgang?
        brcs    ma1860                          // ja -> Ende
        ldi     r17, 254                        // sonst PWM-Startwert fr weiches Ausschalten setzen
        rjmp    ma1850

ma1840: lds     r18, beacfl                     // weiches LED-Einschalt-Kommando, Beacon-Flags holen
        sbr     r18, 1 << 0                     // Flag fr Beacon 1 (LED 6) auf weiches Einschalten setzen
        sts     beacfl, r18                     // Beacon-Flags wieder speichern
        ldi     r17, 1                          // PWM-Startwert fr weiches Einschalten setzen
ma1850: out     ocr0a, r17                      // neuen PWM-Wert speichern

ma1860: cpi     r24, 6                          // wird LED 7 bearbeitet?
        brne    ma1950                          // nein -> Ende
        cpi     r16, 0                          // normales LED-Ausschalt-Kommando?
        brne    ma1870                          // nein -> weiter testen
        out     ocr0b, r1                       // PWM auf minimalen Wert setzen (LED ausschalten)
        rjmp    ma1950

ma1870: cpi     r16, 1                          // normales LED-Einschalt-Kommando?
        brne    ma1880                          // nein -> weiter testen
        ldi     r17, 255                        // PWM auf maximalen Wert setzen (LED einschalten)
        rjmp    ma1900

ma1880: cpi     r16, 2                          // weiches LED-Ausschalt-Kommando?
        brne    ma1890                          // nein -> weiter
        lds     r18, beacfl                     // Beacon-Flags holen
        cbr     r18, 1 << 1                     // Flag fr Beacon 2 (LED 7) auf weiches Ausschalten setzen
        sts     beacfl, r18                     // Beacon-Flags wieder speichern
        in      r17, ocr0b                      // aktuellen PWM-Wert holen
        cpi     r17, 254                        // luft bereits ein Ausschalt-Vorgang?
        brcs    ma1950                          // ja -> Ende
        ldi     r17, 254                        // sonst PWM-Startwert fr weiches Ausschalten setzen
        rjmp    ma1900

ma1890: lds     r18, beacfl                     // weiches LED-Einschalt-Kommando, Beacon-Flags holen
        sbr     r18, 1 << 1                     // Flag fr Beacon 2 (LED 7) auf weiches Einschalten setzen
        sts     beacfl, r18                     // Beacon-Flags wieder speichern
        ldi     r17, 1                          // PWM-Startwert fr weiches Einschalten setzen
ma1900: out     ocr0b, r17                      // Timer0 PWM-Wert B auf hchsten Wert setzen (LED einschalten)
        rjmp    ma1950

ma1950: adiw    yl, 1                           // Zeiger auf Daten der nchsten LED setzen
        inc     r24                             // LED-Zhler erhhen
        cpi     r24, 7                          // alle 7 LEDs bearbeitet?
        brcc    ma2000                          // ja -> Ende
        rjmp    ma1610                          // sonst Schleife

// Steuerung des Landelichts ------------------------------------------------------------------------------------------

ma2000: lds     r16, semode                     // Setup-Modus holen
        tst     r16                             // Setup aktiv?
        breq    ma2010                          // nein -> weiter
        rjmp    ma2500                          // sonst Landelicht-Steuerung berspringen

ma2010: lds     r16, accalm                     // Akku-Alarm holen
        tst     r16                             // Akku-Alarm aktiv?
        breq    ma2100                          // nein -> Akku-Alarm-Behandlung berspringen
        lds     r16, almprg                     // sonst aktuelle Alarm-Programm-Nummer holen
        sbrs    r16, 1                          // Bit 1 gesetzt (Landelicht blinkt bei Alarm)? ja -> berspringen
        rjmp    ma2100                          // sonst Akku-Alarm-Behandlung berspringen
        lds     r16, ccount                     // Zyklus-Zhler holen
        sbrs    r16, 3                          // Bit 3 wechselt alle 160ms, Bit 3 gesetzt? ja -> LED einschalten
        rjmp    ma2020                          // sonst LED einschalten
        sts     ocr2a, r12                      // Landelicht einschalten (invertiert zur Beleuchtung)
        rjmp    ma2500
ma2020: sts     ocr2a, r1                       // Landelicht ausschalten (invertiert zur Beleuchtung)
        rjmp    ma2500

ma2100: lds     r18, lights                     // aktuellen Licht-Status holen
        lds     r16, ocr2a                      // PWM-Wert vom Landelicht holen
        cpi     r18, 2                          // soll Landelicht eingeschaltet werden?
        breq    ma2130                          // ja -> bearbeiten
        tst     r16                             // sonst ausschalten, ist Landelicht bereits ausgeschaltet?
        breq    ma2500                          // ja -> Ende

        mov     r17, r16                        // sonst aktuellen PWM-Wert kopieren
        lsr     r17                             // halbieren
        lsr     r17                             // halbieren
        lsr     r17                             // nochmals halbieren (1/8 des aktuellen PWM-Wertes ermitteln)
        tst     r17                             // ist der ermittelte Wert > 0?
        brne    ma2120                          // ja -> weiter
        ldi     r17, 1                          // sonst Wert auf 1 setzen
        cpi     r16, 8                          // ist der aktuelle PWM-Wert < 8 (letzte Phase)?
        brcc    ma2120                          // nein -> zur Berechnung

        lds     r19, beacfl                     // Beacon- und Landelicht-Flags holen
        sbrc    r19, 2                          // Landelicht-Flag gesetzt? nein -> weiter
        rjmp    ma2110                          // sonst Flag wieder lschen und zur Berechnung
        sbr     r19, 1 << 2                     // Landelicht-Flag setzen
        sts     beacfl, r19                     // Flags wieder speichern
        rjmp    ma2500                          // Berechnung erst im nchsten Zyklus durchfhren

ma2110: cbr     r19, 1 << 2                     // Landelicht-Flag wieder lschen
        sts     beacfl, r19                     // und Flags speichern
ma2120: sub     r16, r17                        // sonst Licht weich ausschalten: berechneten Wert subtrahieren
        brcc    ma2160                          // berlauf? nein -> neuen PWM-Wert speichern
        clr     r16                             // sonst neuen PWM-Wert auf Minimum setzen
        rjmp    ma2160                          // PWM-Wert setzen

ma2130: cpi     r16, 255                        // ist Landelicht bereits eingeschaltet?
        breq    ma2500                          // ja -> Ende
        tst     r16                             // ist Landelicht noch aus?
        brne    ma2140                          // nein -> weiter
        ldi     r16, 1                          // sonst PWM-Anfangswert fr weiches Einschalten setzen
        rjmp    ma2160                          // PWM-Wert setzen

ma2140: lsl     r16                             // PWM-Wert verdoppeln
        tst     r16                             // ist PWM-Wert = 0 (hchstes Bit wurde herausgeschoben)?
        breq    ma2150                          // ja -> PWM auf maximalen Wert setzen
        cpi     r16, 129                        // ist PWM-Wert > 128 (Wert auer der Reihe)?
        brcs    ma2160                          // nein -> PWM-Wert setzen
ma2150: ldi     r16, 255                        // sonst PWM-Wert auf Maximum setzen
ma2160: sts     ocr2a, r16

// Fahrwerk bei Unterschreitung der Akkuspannung ausfahren ------------------------------------------------------------

ma2500: lds     r18, accalm                     // Akku-Alarm-Status holen
        tst     r18                             // Akku-Alarm aktiv?
        breq    ma3000                          // nein -> Ende
        lds     r19, almprg                     // Alarm-Programm-Nummer holen
        andi    r19, 0x04                       // automatisches Ausfahren des Fahrwerks (Programm 4-7)?
        breq    ma3000                          // nein -> Ende
        ldi     r16, 1                          // sonst
        sts     geardn, r16                     // Flag zum Ausfahren des Fahrwerks setzen

// Setup bearbeiten ---------------------------------------------------------------------------------------------------

ma3000: lds     r16, secoun                     // Setup-Wartezhler holen
        tst     r16                             // Zhler abgelaufen?
        breq    ma3010                          // ja -> weiter
        dec     r16                             // sonst Zhler vermindern
        sts     secoun, r16                     // und wieder speichern
        rjmp    ma9000                          // Ende

ma3010: lds     r17, semode                     // Setup-Modus holen
        lds     r16, sestep                     // aktuellen Setup-Schritt holen
        tst     r16                             // sind Schritte zu bearbeiten?
        brne    ma3020                          // ja -> weiter
        rjmp    ma9000                          // sonst Ende

ma3020: cpi     r16, 1                          // Setup-Schritt 1?
        brne    ma3100                          // nein -> weiter testen
        tst     r17                             // Setup-Modus 0?
        brne    ma3030                          // nein -> weiter testen
        clr     r16
        sts     sestep, r16                     // Setup-Schritt = 0 (Setup beendet)
        ldi     r16, 1
        sts     mode, r16                       // Betriebs-Modus auf normal setzen
        rjmp    ma9000

ma3030: inc     r16                             // Setup-Schritt 2 setzen
        sts     sestep, r16                     // speichern
        ldi     r16, 5
        sts     secoun, r16                     // Setup-Wartezhler auf 100ms setzen
        ldi     r16, 4
        sts     mode, r16                       // Betriebs-Modus 4 setzen (LED-Programme bergehen)
        rjmp    ma9000

// Setup-Modus 1 (Programm fr LED 1 einstellen) ----------------------------------------------------------------------

ma3100: cpi     r17, 1                          // Setup-Modus 1 (LED 1)?
        brne    ma3200                          // nein -> weiter testen
        cpi     r16, 2                          // Setup-Schritt 2?
        brne    ma3110                          // nein -> weiter testen
        led1_on                                 // sonst LED 1 einschalten
        rjmp    ma3720                          // nchsten Schritt setzen

ma3110: cpi     r16, 3                          // Setup-Schritt 3?
        breq    ma3120                          // ja -> bearbeiten
        rjmp    ma3730                          // sonst weiter testen
ma3120: led1_off                                // sonst LED 1 ausschalten
        rjmp    ma3720                          // nchsten Schritt setzen

// Setup-Modus 2 (Programm fr LED 2 einstellen) ----------------------------------------------------------------------

ma3200: cpi     r17, 2                          // Setup-Modus 2 (LED 2)?
        brne    ma3300                          // nein -> weiter testen
        cpi     r16, 2                          // Setup-Schritt 2?
        brne    ma3210                          // nein -> weiter testen
        led2_on                                 // sonst LED 2 einschalten
        rjmp    ma3720                          // nchsten Schritt setzen

ma3210: cpi     r16, 3                          // Setup-Schritt 3?
        brne    ma3730                          // nein -> weiter testen
        led2_off                                // sonst LED 2 ausschalten
        rjmp    ma3720                          // nchsten Schritt setzen

// Setup-Modus 3 (Programm fr LED 3 einstellen) ----------------------------------------------------------------------

ma3300: cpi     r17, 3                          // Setup-Modus 3 (LED 3)?
        brne    ma3400                          // nein -> weiter testen
        cpi     r16, 2                          // Setup-Schritt 2?
        brne    ma3310                          // nein -> weiter testen
        led3_on                                 // sonst LED 3 einschalten
        rjmp    ma3720                          // nchsten Schritt setzen

ma3310: cpi     r16, 3                          // Setup-Schritt 3?
        brne    ma3730                          // nein -> weiter testen
        led3_off                                // sonst LED 3 ausschalten
        rjmp    ma3720                          // nchsten Schritt setzen

// Setup-Modus 4 (Programm fr LED 4 einstellen) ----------------------------------------------------------------------

ma3400: cpi     r17, 4                          // Setup-Modus 4 (LED 4)?
        brne    ma3500                          // nein -> weiter testen
        cpi     r16, 2                          // Setup-Schritt 2?
        brne    ma3410                          // nein -> weiter testen
        led4_on                                 // sonst LED 4 einschalten
        rjmp    ma3720                          // nchsten Schritt setzen

ma3410: cpi     r16, 3                          // Setup-Schritt 3?
        brne    ma3730                          // nein -> weiter testen
        led4_off                                // sonst LED 4 ausschalten
        rjmp    ma3720                          // nchsten Schritt setzen

// Setup-Modus 5 (Programm fr LED 5 einstellen) ----------------------------------------------------------------------

ma3500: cpi     r17, 5                          // Setup-Modus 5 (LED 5)?
        brne    ma3600                          // nein -> weiter testen
        cpi     r16, 2                          // Setup-Schritt 2?
        brne    ma3510                          // nein -> weiter testen
        led5_on                                 // sonst LED 5 einschalten
        rjmp    ma3720                          // nchsten Schritt setzen

ma3510: cpi     r16, 3                          // Setup-Schritt 3?
        brne    ma3730                          // nein -> weiter testen
        led5_off                                // sonst LED 5 ausschalten
        rjmp    ma3720                          // nchsten Schritt setzen

// Setup-Modus 6 (Programm fr LED 6 einstellen) ----------------------------------------------------------------------

ma3600: cpi     r17, 6                          // Setup-Modus 6 (LED 6)?
        brne    ma3700                          // nein -> weiter testen
        cpi     r16, 2                          // Setup-Schritt 2?
        brne    ma3610                          // nein -> weiter testen
        out     ocr0a, r12                      // sonst LED 6 einschalten
        rjmp    ma3720                          // nchsten Schritt setzen

ma3610: cpi     r16, 3                          // Setup-Schritt 3?
        brne    ma3730                          // nein -> weiter testen
        out     ocr0a, r1                       // sonst LED 6 ausschalten
        rjmp    ma3720                          // nchsten Schritt setzen

// Setup-Modus 7 (Programm fr LED 7 einstellen) ----------------------------------------------------------------------

ma3700: cpi     r17, 7                          // Setup-Modus 7 (LED 7)?
        brne    ma3800                          // nein -> weiter testen
        cpi     r16, 2                          // Setup-Schritt 2?
        brne    ma3710                          // nein -> weiter testen
        out     ocr0b, r12                      // sonst LED 7 einschalten
        rjmp    ma3720                          // nchsten Schritt setzen

ma3710: cpi     r16, 3                          // Setup-Schritt 3?
        brne    ma3730                          // nein -> weiter testen
        out     ocr0b, r1                       // sonst LED 7 ausschalten

ma3720: inc     r16                             // Schrittzhler erhhen
        sts     sestep, r16                     // und speichern
        ldi     r16, 25
        sts     secoun, r16                     // Setup-Wartezhler auf 500ms setzen
        rjmp    ma9000                          // Ende

ma3730: cpi     r16, 4                          // Setup-Schritt 4?
        brne    ma3740                          // nein -> weiter testen
        inc     r16                             // Schrittzhler erhhen
        sts     sestep, r16                     // und speichern
        ldi     r16, 3
        sts     mode, r16                       // Betriebs-Modus 3 setzen (alle LED-Programme einschalten)
        rjmp    ma9000                          // Ende

ma3740: cpi     r16, 10                         // Setup-Schritt 10 (nderung des Programmes)?
        breq    ma3750                          // ja -> weiter
        rjmp    ma9000                          // sonst alles andere ignorieren
ma3750: ldi     r17, 1                          // sonst
        sts     sestep, r17                     // Setup-Schritt 1 setzen
        sts     secoun, r1                      // keine Wartezeit
        ldi     r17, 2
        sts     mode, r17                       // Betriebs-Modus auf 2 setzen (alle LEDs ausschalten)
        rjmp    ma9000

// Setup-Modus 8 (Akku-Alarm-Programm einstellen) ---------------------------------------------------------------------

ma3800: cpi     r17, 8                          // Setup-Modus 8 (Akku-Alarm-Programm)
        brne    ma3900                          // nein -> weiter testen
        cpi     r16, 2                          // Setup-Schritt 2?
        brne    ma3810                          // nein -> weiter testen
        ldi     r18, 25                         // sonst
        sts     secoun, r18                     // Setup-Wartezhler auf 500ms setzen
        inc     r16                             // Schrittzhler erhhen
        rjmp    ma3850                          // und speichern

ma3810: cpi     r16, 3                          // Setup-Schritt 3?
        brne    ma3840                          // nein -> weiter testen
        lds     r17, senumb                     // sonst Nummernzhler holen
        tst     r17                             // Zhler abgelaufen?
        breq    ma3830                          // ja -> weiter
        dec     r17                             // sonst Nummernzhler vermindern
        sts     senumb, r17                     // und wieder speichern
        ldi     r18, 15
        sts     secoun, r18                     // Setup-Wartezhler auf 300ms setzen

        sbrs    r17, 0                          // Nummernzhler ungeradzahlig? ja -> Landelicht einschalten
        rjmp    ma3820                          // sonst Landelicht ausschalten
        ldi     r17, 20
        sts     ocr2a, r17                      // Landelicht mit niedriger Helligkeit einschalten
        rjmp    ma9000
ma3820: sts     ocr2a, r1                       // Landelicht ausschalten
        rjmp    ma9000

ma3830: ldi     r18, 100
        sts     secoun, r18                     // Setup-Wartezhler auf 3s setzen
        sts     ocr2a, r1                       // Landelicht ausschalten
        rjmp    ma3860                          // Zhler neu setzen

ma3840: cpi     r16, 10                         // Setup-Schritt 10 (nderung des Programmes)?
        brne    ma3870                          // nein -> alles andere ignorieren
        ldi     r18, 5
        sts     secoun, r18                     // Setup-Wartezhler auf 100ms setzen
        ldi     r16, 3                          // Setup-Schritt 3 setzen
ma3850: sts     sestep, r16                     // und speichern
ma3860: lds     r16, almprg                     // Akku-Alarm-Programm holen
        inc     r16                             // erhhen (Anzahl der Blinkzeichen setzen)
        lsl     r16                             // verdoppeln (Anzahl der LED Ein/Aus-Zeiten setzen)
        sts     senumb, r16                     // als Nummernzhler speichern
        led1_on                                 // LED 1 einschalten
        sts     ocr2a, r1                       // Landelicht ausschalten
ma3870: rjmp    ma9000

// Setup-Modus 9 (Akku-Alarm-Stufe einstellen) ------------------------------------------------------------------------

ma3900: cpi     r17, 9                          // Setup-Modus 9 (Akku-Alarm-Stufe)
        brne    ma4000                          // nein -> weiter testen
        cpi     r16, 2                          // Setup-Schritt 2?
        brne    ma3910                          // nein -> weiter testen
        ldi     r18, 25                         // sonst
        sts     secoun, r18                     // Setup-Wartezhler auf 500ms setzen
        inc     r16                             // Schrittzhler erhhen
        rjmp    ma3950                          // und speichern

ma3910: cpi     r16, 3                          // Setup-Schritt 3?
        brne    ma3940                          // nein -> weiter testen
        lds     r17, senumb                     // sonst Nummernzhler holen
        tst     r17                             // Zhler abgelaufen?
        breq    ma3930                          // ja -> weiter
        dec     r17                             // sonst Nummernzhler vermindern
        sts     senumb, r17                     // und wieder speichern
        ldi     r18, 15
        sts     secoun, r18                     // Setup-Wartezhler auf 300ms setzen

        sbrs    r17, 0                          // Nummernzhler ungeradzahlig? ja -> Landelicht einschalten
        rjmp    ma3920                          // sonst Landelicht ausschalten
        ldi     r17, 20
        sts     ocr2a, r17                      // Landelicht mit niedriger Helligkeit einschalten
        rjmp    ma9000
ma3920: sts     ocr2a, r1                       // Landelicht ausschalten
        rjmp    ma9000

ma3930: ldi     r18, 100
        sts     secoun, r18                     // Setup-Wartezhler auf 3s setzen
        sts     ocr2a, r1                       // Landelicht ausschalten
        rjmp    ma3960                          // Zhler neu setzen

ma3940: cpi     r16, 10                         // Setup-Schritt 10 (nderung des Programmes)?
        brne    ma3970                          // nein -> alles andere ignorieren
        ldi     r18, 5
        sts     secoun, r18                     // Setup-Wartezhler auf 100ms setzen
        ldi     r16, 3                          // Setup-Schritt 3 setzen
ma3950: sts     sestep, r16                     // und speichern
ma3960: lds     r16, acclev                     // Akku-Alarm-Stufe holen
        inc     r16                             // erhhen (Anzahl der Blinkzeichen setzen)
        lsl     r16                             // verdoppeln (Anzahl der LED Ein/Aus-Zeiten setzen)
        sts     senumb, r16                     // als Nummernzhler speichern
        led2_on                                 // LED 2 einschalten
        sts     ocr2a, r1                       // Landelicht ausschalten
ma3970: rjmp    ma9000

// Setup-Modus 10 (Servo-Verzgerung einstellen) ----------------------------------------------------------------------

ma4000: cpi     r17, 10                         // Setup-Modus 10 (Servo-Verzgerung)
        brne    ma4100                          // nein -> weiter testen
        cpi     r16, 2                          // Setup-Schritt 2?
        brne    ma4010                          // nein -> weiter testen
        ldi     r18, 25                         // sonst
        sts     secoun, r18                     // Setup-Wartezhler auf 500ms setzen
        inc     r16                             // Schrittzhler erhhen
        rjmp    ma4050                          // und speichern

ma4010: cpi     r16, 3                          // Setup-Schritt 3?
        brne    ma4040                          // nein -> weiter testen
        lds     r17, senumb                     // sonst Nummernzhler holen
        tst     r17                             // Zhler abgelaufen?
        breq    ma4030                          // ja -> weiter
        dec     r17                             // sonst Nummernzhler vermindern
        sts     senumb, r17                     // und wieder speichern
        ldi     r18, 15
        sts     secoun, r18                     // Setup-Wartezhler auf 300ms setzen

        sbrs    r17, 0                          // Nummernzhler ungeradzahlig? ja -> Landelicht einschalten
        rjmp    ma4020                          // sonst Landelicht ausschalten
        ldi     r17, 20
        sts     ocr2a, r17                      // Landelicht mit niedriger Helligkeit einschalten
        rjmp    ma9000
ma4020: sts     ocr2a, r1                       // Landelicht ausschalten
        rjmp    ma9000

ma4030: ldi     r18, 100
        sts     secoun, r18                     // Setup-Wartezhler auf 3s setzen
        sts     ocr2a, r1                       // Landelicht ausschalten
        rjmp    ma4060                          // Zhler neu setzen

ma4040: cpi     r16, 10                         // Setup-Schritt 10 (nderung des Programmes)?
        brne    ma9000                          // nein -> alles andere ignorieren
        ldi     r18, 5
        sts     secoun, r18                     // Setup-Wartezhler auf 100ms setzen
        ldi     r16, 3                          // Setup-Schritt 3 setzen
ma4050: sts     sestep, r16                     // und speichern
ma4060: lds     r16, sdelay                     // Servo-Verzgerung holen
        inc     r16                             // erhhen (Anzahl der Blinkzeichen setzen)
        lsl     r16                             // verdoppeln (Anzahl der LED Ein/Aus-Zeiten setzen)
        sts     senumb, r16                     // als Nummernzhler speichern
        led1_on                                 // LED 1 einschalten
        led2_on                                 // LED 2 einschalten
        sts     ocr2a, r1                       // Landelicht ausschalten
        rjmp    ma9000

// Setup-Modus 11 (Servo-Position fr Fahrwerk-Automatik lernen) ------------------------------------------------------

ma4100: cpi     r17, 11                         // Setup-Modus 11 (Servo-Position fr Fahrwerk-Automatik)
        brne    ma4200                          // nein -> weiter testen
        cpi     r16, 2                          // Setup-Schritt 2?
        brne    ma4110                          // nein -> weiter testen
        sts     secoun, r1                      // sonst Setup-Wartezhler lschen
        inc     r16                             // Schrittzhler erhhen
        sts     sestep, r16                     // und speichern
        sts     ocr2a, r1                       // Landelicht ausschalten
        rjmp    ma9000

ma4110: cpi     r16, 3                          // Setup-Schritt 3?
        brne    ma4140                          // nein -> weiter testen
        lds     r18, ccount                     // sonst Zyklus-Zhler holen
        sbrs    r18, 3                          // Bit 3 wechselt alle 160ms, Bit 3 gesetzt? ja -> LED 1+2 ausschalten
        rjmp    ma4120                          // sonst LED 1 und 2 einschalten
        led1_off                                // LED 1 ausschalten
        led2_off                                // LED 2 ausschalten
        rjmp    ma4130
ma4120: led1_on                                 // LED 1 einschalten
        led2_on                                 // LED 2 einschalten
ma4130: sts     ocr2a, r1                       // Landelicht ausschalten
        rjmp    ma9000

ma4140: cpi     r16, 10                         // Setup-Schritt 10 (nderung des Programmes)?
        brne    ma9000                          // nein -> alles andere ignorieren
        lds     r18, servfl                     // sonst Flag fr Speicherung der Servo-Position holen
        tst     r18                             // war Speicherung erfolgreich?
        breq    ma4150                          // nein -> kein Quittungssignal ausgeben
        sts     servfl, r1                      // sonst Flag wieder lschen
        ldi     r18, 15
        sts     secoun, r18                     // Setup-Wartezhler auf 300ms setzen
        ldi     r17, 20
        sts     ocr2a, r17                      // Landelicht mit niedriger Helligkeit einschalten
ma4150: ldi     r16, 3
        sts     sestep, r16                     // Setup-Schritt 3 setzen
        rjmp    ma9000

ma4200: nop                                     // eventuell weitere Setup-Modi

// Kurzen Tastendruck auswerten ---------------------------------------------------------------------------------------

ma9000: lds     r16, kflags                     // Taster-Flags holen
        sbrs    r16, 1                          // Taster gedrckt?
        rjmp    ma9010                          // nein -> Merker prfen
        sbrc    r16, 2                          // sonst Taster quittiert?
        rjmp    ma9100                          // ja -> langen Tastendruck prfen

        sbr     r16, (1 << 5) | (1 << 2)        // Tastendruck quittieren und Merker setzen
        sts     kflags, r16                     // Taster-Flags wieder speichern
        rjmp    ma9100                          // Ende

ma9010: sbrs    r16, 5                          // Merker gesetzt (Taster wurde kurz gedrckt)? ja -> weiter
        rjmp    ma9100                          // sonst Ende
        cbr     r16, 1 << 5                     // Merker wieder lschen
        sts     kflags, r16                     // Taster-Flags wieder speichern

// Kurzen Tastendruck bearbeiten --------------------------------------------------------------------------------------

        lds     r16, semode                     // Setup-Modus holen
ma9020: inc     r16                             // Setup-Modus erhhen
        sts     semode, r16                     // neuen Setup-Modus speichern
        cpi     r16, 12                         // Bereich berschritten?
        brcs    ma9040                          // nein -> weiter

ma9030: sts     semode, r1                      // sonst Setup-Modus wieder auf 0 setzen (ausschalten)
        sts     mode, r1                        // Betriebs-Modus auf Start-Sequenz setzen
        ldi     r16, 30                         // verkrzte Startsequenz 1,4s
        sts     ccount, r16                     // Zykluszhler voreinstellen
        mov     icount, r16                     // Interruptzhler voreinstellen
        led1_off                                // LED 1 ausschalten
        led2_off                                // LED 2 ausschalten
        led3_off                                // LED 3 ausschalten
        led4_off                                // LED 4 ausschalten
        led5_off                                // LED 5 ausschalten
        out     ocr0a, r1                       // LED 6 ausschalten
        out     ocr0b, r1                       // LED 7 ausschalten
        sts     ocr2a, r1                       // Landelicht ausschalten
        rjmp    ma9200

ma9040: ldi     r16, 1
        sts     sestep, r16                     // Setup-Schritt 1 setzen
        sts     secoun, r1                      // keine Wartezeit
        ldi     r16, 2
        sts     mode, r16                       // Betriebs-Modus auf 2 setzen (alle LEDs ausschalten)
        sts     ocr2a, r1                       // Landelicht direkt ausschalten
        ldi     r16, settim                     // Timeout-Wert fr Setup-Modus holen
        sts     set_to, r16                     // und Timeout-Zhler setzen
        rjmp    ma9200

// Langen Tastendruck auswerten ---------------------------------------------------------------------------------------

ma9100: sbrs    r16, 3                          // Taster lange gedrckt?
        rjmp    ma9200                          // nein -> Ende
        sbrc    r16, 4                          // sonst langer Tastendruck quittiert?
        rjmp    ma9200                          // ja -> Ende

        sbr     r16, 1 << 4                     // langen Tastendruck quittieren
        cbr     r16, 1 << 5                     // gegebenenfalls Merker lschen
        sts     kflags, r16                     // Taster-Flags wieder speichern

// Langen Tastendruck bearbeiten --------------------------------------------------------------------------------------

        ldi     r18, settim                     // Timeout-Wert fr Setup-Modus holen
        sts     set_to, r18                     // und Timeout-Zhler setzen
        lds     r18, semode                     // Setup-Modus holen
        tst     r18                             // Setup-Modus = 0?
        brne    ma9110                          // nein -> Setup-Schritt auf Speichern setzen
        sts     geardn, r1                      // sonst Flag fr automatisches Ausfahren des Fahrwerks lschen
        rjmp    ma9030                          // weiter mit verkrzter Start-Sequenz

ma9110: ldi     r19, 10
        sts     sestep, r19                     // Setup-Schritt auf 10 setzen
        sts     secoun, r1                      // Wartezhler lschen

        cpi     r18, 8                          // Setup-Modus 1-7 (LED 1-7)?
        brcc    ma9130                          // nein -> weiter testen
        dec     r18                             // sonst Tabellen-Offset aus Setup-Modus ermitteln (-1)
        ldi     xl, low (prgnum)
        ldi     xh, high (prgnum)               // Zeiger auf Licht-Programm-Nummern
        add     xl, r18                         // Tabellen-Offset addieren
        adc     xh, r1                          // bertrag addieren
        ld      r17, x                          // aktuelle Licht-Programm-Nummer holen
        inc     r17                             // auf nchste Nummer setzen
        cpi     r17, ligend - ligtab            // Nummernbereich berschritten?
        brcs    ma9120                          // nein -> weiter
        ldi     r17, 1                          // sonst wieder auf Nummer 1 setzen
ma9120: st      x, r17                          // neue Licht-Programm-Nummer speichern
        ldi     r16, e_effn                     // EEPROM-Adresse der Licht-Programm-Nummern laden
        add     r16, r18                        // Tabellen-Offset addieren
        rcall   eewrit                          // neue Licht-Programm-Nummer im EEPROM speichern
        rjmp    ma9200

ma9130: cpi     r18, 8                          // Setup-Modus 8 (Akku-Alarm-Programm)?
        brne    ma9140                          // nein -> weiter testen
        lds     r17, almprg                     // sonst aktuelle Alarm-Programm-Nummer holen
        inc     r17                             // auf nchste Nummer setzen
        andi    r17, 0x07                       // Wertebereich auf 0-7 begrenzen
        sts     almprg, r17                     // neue Alarm-Programm-Nummer speichern
        ldi     r16, e_almn                     // EEPROM-Adresse der Alarm-Programm-Nummer laden
        rcall   eewrit                          // neue Alarm-Programm-Nummer im EEPROM speichern
        rjmp    ma9200

ma9140: cpi     r18, 9                          // Setup-Modus 9 (Akku-Alarm-Stufe)?
        brne    ma9160                          // nein -> weiter testen
        lds     r17, acclev                     // sonst Akku-Alarm-Stufe holen
        inc     r17                             // auf nchste Stufe setzen
        cpi     r17, 15                         // Wertebereich berschritten?
        brcs    ma9150                          // nein -> weiter
        clr     r17                             // sonst wieder auf die kleinste Stufe setzen
ma9150: sts     acclev, r17                     // neue Akku-Alarm-Stufe speichern
        ldi     r16, e_accl                     // EEPROM-Adresse der Akku-Alarm-Stufe laden
        rcall   eewrit                          // neue Akku-Alarm-Stufe im EEPROM speichern
        rjmp    ma9200

ma9160: cpi     r18, 10                         // Setup-Modus 10 (Servo-Verzgerung)?
        brne    ma9170                          // nein -> weiter testen
        lds     r17, sdelay                     // Servo-Verzgerung holen
        inc     r17                             // auf nchste Stufe setzen
        andi    r17, 0x03                       // Wertebereich auf 0-3 begrenzen
        sts     sdelay, r17                     // neue Servo-Verzgerung speichern
        ldi     r16, e_sdel                     // EEPROM-Adresse der Servo-Verzgerung
        rcall   eewrit                          // neue Servo-Verzgerung im EEPROM speichern
        rjmp    ma9200

ma9170: cpi     r18, 11                         // Setup-Modus 11 (Fahrwerk-Position)?
        brne    ma9200                          // nein -> Ende
        lds     r18, k2ppm + 8
        lds     r19, k2ppm + 9                  // Mittelwert von Kanal B holen
        mov     r16, r18                        // Mittelwert Low-Byte kopieren
        or      r16, r19                        // mit Mittelwert High-Byte verknpfen, Mittelwert = 0?
        breq    ma9200                          // ja -> Wert nicht speichern

        sts     servo0, r18                     // sonst neuen Wert (Low) speichern
        sts     servo0 + 1, r19                 // neuen Wert (High) speichern
        ldi     r16, e_serv                     // EEPROM-Adresse der Fahrwerk-Position (Low)
        mov     r17, r18                        // Low-Byte kopieren
        rcall   eewrit                          // neue Fahrwerk-Position (Low) im EEPROM speichern
        ldi     r16, e_serv + 1                 // EEPROM-Adresse der Fahrwerk-Position (High)
        mov     r17, r19                        // High-Byte kopieren
        rcall   eewrit                          // neue Fahrwerk-Position (High) im EEPROM speichern
        ldi     r16, 1
        sts     servfl, r16                     // Flag fr erfolgreiche Speicherung setzen

// Setup-Timeout-Zhler bearbeiten und gegebenenfalls Setup-Modus beenden ---------------------------------------------

ma9200: lds     r16, semode                     // Setup-Modus holen
        tst     r16                             // Setup-Modus aktiv = 0?
        breq    ma9500                          // nein -> Ende
        lds     r16, set_to                     // Setup-Timeout-Zhler holen
        tst     r16                             // Timeout-Zhler abgelaufen?
        brne    ma9210                          // nein -> weiter
        rjmp    ma9030                          // sonst Setup-Modus beenden

ma9210: lds     r17, ccount                     // Zyklus-Zhler holen
        tst     r17                             // Zhler abgelaufen?
        brne    ma9500                          // nein -> Ende
        dec     r16                             // sonst Setup-Timeout-Zhler dekrementieren
        sts     set_to, r16                     // und wieder speichern

// Improvisierte Schutzfunktion ---------------------------------------------------------------------------------------

ma9500: tst     r1                              // hat Register r1 immer noch den Wert 0?
        brne    ma9510                          // nein -> weiter ohne WDR
        mov     r16, r12
        cpi     r16, 255                        // hat Register r12 immer noch den Wert 255?
        brne    ma9510                          // nein -> weiter ohne WDR
        wdr                                     // sonst alles ok
ma9510: rjmp    main                            // Programmschleife ausfhren

// Unterprogramme =====================================================================================================

// Ein Byte aus dem EEPROM lesen --------------------------------------------------------------------------------------
// Register r16 - Adresse (Low)
//          r17 - gelesener Wert

eeread: sbic    eecr, eepe                      // luft gerade ein Schreibzyklus?
        rjmp    eeread                          // ja -> warten
        out     eearh, r1                       // EEPROM-Adresse High setzen (ist immer 0)
        out     eearl, r16                      // EEPROM-Adresse Low setzen
        sbi     eecr, eere                      // EEPROM lesen
        in      r17, eedr                       // Datenbyte holen
	ret

// Ein Byte in das EEPROM schreiben -----------------------------------------------------------------------------------
// Register r16 - Adresse (Low)
//          r17 - zu schreibender Wert

eewrit: sbic    eecr, eepe                      // luft gerade ein Schreibzyklus?
        rjmp    eewrit                          // ja -> warten
        out     eearh, r1                       // EEPROM-Adresse High setzen (ist immer 0)
        out     eearl, r16                      // EEPROM-Adresse Low setzen
        out     eedr, r17                       // EEPROM-Daten setzen
        sbi     eecr, eempe                     // EEPROM schreiben freigeben
        sbi     eecr, eepe                      // EEPROM schreiben aktivieren
        ret

// Licht-Programme ====================================================================================================

// Diese bestehen aus einer Reihe von Kommandos, die aus jeweils zwei Bytes zusammengesetzt werden. Das erste Byte ist
// das Kommando, das zweite Byte der Parameter.

// Kommando-Byte 0 - LED ausschalten, Parameter enthlt die Ausschaltdauer in Einheiten zu 20ms
// Kommando-Byte 1 - LED einschalten, Parameter enthlt die Einschaltdauer in Einheiten zu 20ms
// Kommando-Byte 2 - LED weich ausschalten, Parameter enthlt die Ausschaltdauer in Einheiten zu 20ms (fr Beacon-LEDs)
// Kommando-Byte 3 - LED weich einschalten, Parameter enthlt die Einschaltdauer in Einheiten zu 20ms (fr Beacon-LEDs)
// Kommando-Byte 255 (0xff) - Ende und Beginn von vorn, Parameter wird nicht ausgewertet

// Licht-Programm 00 - Licht aus
ligh00: .db     0, 1                            // Licht ausschalten, 20ms warten (wird nur intern verwendet)
        .db     255, 255

// Licht-Programm 01 - Dauerlicht
ligh01: .db     1, 1                            // Licht einschalten, 20ms warten
        .db     255, 255

// Licht-Programm 02 - Einfach-Blitzlicht 1s
ligh02: .db     1, 3                            // Licht einschalten,  60ms warten
        .db     0, 47                           // Licht ausschalten, 940ms warten
        .db     255, 255

// Licht-Programm 03 - Zweifach-Blitzlicht 1s
ligh03: .db     1, 3                            // Licht einschalten,  60ms warten
        .db     0, 6                            // Licht ausschalten, 120ms warten
        .db     1, 3                            // Licht einschalten,  60ms warten
        .db     0, 38                           // Licht ausschalten, 760ms warten
        .db     255, 255

// Licht-Programm 04 - Dreifach-Blitzlicht 1s
ligh04: .db     1, 3                            // Licht einschalten,  60ms warten
        .db     0, 6                            // Licht ausschalten, 120ms warten
        .db     1, 3                            // Licht einschalten,  60ms warten
        .db     0, 6                            // Licht ausschalten, 120ms warten
        .db     1, 3                            // Licht einschalten,  60ms warten
        .db     0, 29                           // Licht ausschalten, 580ms warten
        .db     255, 255

// Licht-Programm 05 - Einfach-Blitzlicht 1,6s
ligh05: .db     1, 3                            // Licht einschalten,   60ms warten
        .db     0, 77                           // Licht ausschalten, 1540ms warten
        .db     255, 255

// Licht-Programm 06 - Zweifach-Blitzlicht 1,6s
ligh06: .db     1, 3                            // Licht einschalten,   60ms warten
        .db     0, 6                            // Licht ausschalten,  120ms warten
        .db     1, 3                            // Licht einschalten,   60ms warten
        .db     0, 68                           // Licht ausschalten, 1360ms warten
        .db     255, 255

// Licht-Programm 07 - Dreifach-Blitzlicht 1,6s
ligh07: .db     1, 3                            // Licht einschalten,   60ms warten
        .db     0, 6                            // Licht ausschalten,  120ms warten
        .db     1, 3                            // Licht einschalten,   60ms warten
        .db     0, 6                            // Licht ausschalten,  120ms warten
        .db     1, 3                            // Licht einschalten,   60ms warten
        .db     0, 59                           // Licht ausschalten, 1180ms warten
        .db     255, 255

// Licht-Programm 08 - Einfach-Blitzlicht 1,6s zeitversetzt zu Licht-Programm 05
ligh08: .db     0, 40                           // Licht bleibt aus,  800ms warten
        .db     1, 3                            // Licht einschalten,  60ms warten
        .db     0, 37                           // Licht ausschalten, 740ms warten
        .db     255, 255

// Licht-Programm 09 - Zweifach-Blitzlicht 1,6s zeitversetzt zu Licht-Programm 06
ligh09: .db     0, 40                           // Licht bleibt aus,  800ms warten
        .db     1, 3                            // Licht einschalten,  60ms warten
        .db     0, 6                            // Licht ausschalten, 120ms warten
        .db     1, 3                            // Licht einschalten,  60ms warten
        .db     0, 28                           // Licht ausschalten, 560ms warten
        .db     255, 255

// Licht-Programm 10 - Dreifach-Blitzlicht 1,6s zeitversetzt zu Licht-Programm 07
ligh10: .db     0, 40                           // Licht bleibt aus,  800ms warten
        .db     1, 3                            // Licht einschalten,  60ms warten
        .db     0, 6                            // Licht ausschalten, 120ms warten
        .db     1, 3                            // Licht einschalten,  60ms warten
        .db     0, 6                            // Licht ausschalten, 120ms warten
        .db     1, 3                            // Licht einschalten,  60ms warten
        .db     0, 19                           // Licht ausschalten, 380ms warten
        .db     255, 255

// Licht-Programm 11 - Beacon 2,72s

ligh11: .db     3, 65                           // Licht einblenden, 1300ms warten
        .db     2, 71                           // Licht ausblenden, 1420ms warten
        .db     255, 255

// Licht-Programm 12 - Beacon 3,72s
ligh12: .db     3, 65                           // Licht einblenden, 1300ms warten
        .db     2, 121                          // Licht ausblenden, 2420ms warten
        .db     255, 255

// Licht-Programm 13 - Beacon 2,72s zeitversetzt zu Licht-Programm 11
ligh13: .db     2, 68                           // Licht bleibt aus, 1360ms warten
        .db     3, 65                           // Licht einblenden, 1300ms warten
        .db     2, 3                            // Licht ausblenden,   60ms warten
        .db     255, 255

// Licht-Programm 14 - Beacon 3,72s zeitversetzt zu Licht-Programm 12
ligh14: .db     2, 93                           // Licht bleibt aus, 1860ms warten
        .db     3, 65                           // Licht einblenden, 1300ms warten
        .db     2, 28                           // Licht ausblenden,  560ms warten
        .db     255, 255

// Licht-Programm 15 - Blinklicht 1,3s
ligh15: .db     1, 13                           // Licht einschalten,  260ms warten
        .db     0, 52                           // Licht ausschalten, 1040ms warten
        .db     255, 255

// Licht-Programm 16 - Blinklicht 1,7s
ligh16: .db     1, 17                           // Licht einschalten,  340ms warten
        .db     0, 68                           // Licht ausschalten, 1360ms warten
        .db     255, 255

// Licht-Programm 17 - Blinklicht 2,1s
ligh17: .db     1, 21                           // Licht einschalten,  420ms warten
        .db     0, 84                           // Licht ausschalten, 1680ms warten
        .db     255, 255

// Licht-Programm-Tabelle
ligtab: .dw     ligh00                          // Adresse von Licht-Programm 00 (wird nur intern verwendet)
        .dw     ligh01                          // Adresse von Licht-Programm 01
        .dw     ligh02                          // Adresse von Licht-Programm 02
        .dw     ligh03                          // Adresse von Licht-Programm 03
        .dw     ligh04                          // Adresse von Licht-Programm 04
        .dw     ligh05                          // Adresse von Licht-Programm 05
        .dw     ligh06                          // Adresse von Licht-Programm 06
        .dw     ligh07                          // Adresse von Licht-Programm 07
        .dw     ligh08                          // Adresse von Licht-Programm 08
        .dw     ligh09                          // Adresse von Licht-Programm 09
        .dw     ligh10                          // Adresse von Licht-Programm 10
        .dw     ligh11                          // Adresse von Licht-Programm 11
        .dw     ligh12                          // Adresse von Licht-Programm 12
        .dw     ligh13                          // Adresse von Licht-Programm 13
        .dw     ligh14                          // Adresse von Licht-Programm 14
        .dw     ligh15                          // Adresse von Licht-Programm 15
        .dw     ligh16                          // Adresse von Licht-Programm 16
        .dw     ligh17                          // Adresse von Licht-Programm 17

// Ende der Licht-Programm-Tabelle, Label wird zur Ermittlung der Anzahl der Licht-Programme bentigt
ligend:

// EEPROM-Bereich =====================================================================================================

.eseg
e_effn: .byte   7               // 1-17      Programm-Nummer fr LEDs 1-7
e_almn: .byte   1               // 0-7       Programm-Nummer fr den Akku-Alarm
e_accl: .byte   1               // 0-9       Akku-Alarm-Stufe
e_sdel: .byte   1               // 0-3       Servo-Verzgerung
e_serv: .byte   2               // 750-2250  Servo-Position fr das automatische Ausfahren des Fahrwerks
